/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <AGDeviceUserDropDown.h>
#include <AGDesktopInfoPalm.h>
#include <AGConfigResource.h>
#include <stdio.h>
#include <stdlib.h>
#include <commctrl.h>
#include <windowsx.h>

typedef enum {
    cePlatform,
    palmPlatform
} platformType;

extern HINSTANCE g_hInstance;

static AGBool isPlatformInstalled(platformType platform)
{
    AGBool result = FALSE;
    HKEY hkey;

    if (ERROR_SUCCESS == RegOpenKeyEx(HKEY_CURRENT_USER,
        "Software\\Mobile Application Link",
        0,
        KEY_READ,
        &hkey)) {

        DWORD size = sizeof(DWORD);
        DWORD dwType;
        DWORD value = 0;

        RegQueryValueEx(hkey,
            (cePlatform == platform) ? "CEInstalled" : "PalmInstalled",
            0,
            &dwType,
            (LPBYTE)&value,
            &size);

        result = (value != 0);

        RegCloseKey(hkey);

    }

    return result;

}

/* ----------------------------------------------------------------------------
*/
static AGDeviceEntry * makeDeviceEntry(char * username,
                                       AGDeviceType deviceType,
                                       char * basePath)
{
    char * temp;
    TCHAR userDir[MAX_PATH];

    strcpy(userDir, basePath);

    /* Build the path of the config file. */
    temp = (char*)AGSyncCommonGetStringConstant(agMALSubdirectoryName,
        FALSE);
    PathAppend(userDir, temp);
    free(temp);

    /* Make sure the path exists. */
    if (AG_NO_DEVICE_TYPE != deviceType)
        CreateDirectory(userDir, NULL);

    return AGMobileLinkDeviceEntryNew(strdup(username),
        deviceType,
        strdup(userDir));
}

/* ----------------------------------------------------------------------------
*/
static void walkPalmList(HWND deviceCombo, AGArray * array)
{
    int i, n;
    char nameTemplate[MAX_PATH];
    char nameBuffer[MAX_PATH];
    AGDeviceEntry * dle = NULL;
    AGDesktopInfoPalm * desktopInfo;
    
    desktopInfo = AGDesktopInfoPalmNew();
    if (NULL == desktopInfo)
        return;

    LoadString(g_hInstance,
        IDS_PALM_DEVICE_NAME,
        nameTemplate,
        MAX_PATH);

    n = AGDesktopInfoPalmGetUserCount(desktopInfo);

    for (i = 0; i < n; ++i) {

        TCHAR username[MAX_PATH];
        TCHAR tempDir[MAX_PATH];
        short bufSize;
        int   intBufSize; // aaaaaaaaaaaaaarrrrrgh

        bufSize = MAX_PATH;
        AGDesktopInfoPalmGetUsername(desktopInfo, i, username, &bufSize);
        if (0 == strlen(username))
            continue;

        intBufSize = MAX_PATH;
        AGDesktopInfoPalmGetUserDirectory(desktopInfo,
            username,
            tempDir,
            &intBufSize);
        if (0 == strlen(tempDir))
            continue;

        dle = makeDeviceEntry(username, AG_PALM_DEVICE_TYPE, tempDir);
        if (NULL == dle)
            continue;

        sprintf(nameBuffer, nameTemplate, dle->deviceName);

        ComboBox_InsertString(deviceCombo,
            AGArrayCount(array),
            nameBuffer);

        AGArrayAppend(array, (void*)dle);

    }

    AGDesktopInfoPalmFree(desktopInfo);
}

/* ----------------------------------------------------------------------------
*/
static void walkCEList(HWND deviceCombo, AGArray * array)
{
    char nameTemplate[MAX_PATH];
    char nameBuffer[MAX_PATH];
    AGDeviceEntry * dle = NULL;
    HKEY ceRootKey, partnerKey;
    LONG result;
    DWORD dwIndex;

    LoadString(g_hInstance,
        IDS_CE_DEVICE_NAME,
        nameTemplate,
        MAX_PATH);

    result = RegOpenKeyEx(HKEY_CURRENT_USER,
        "Software\\Microsoft\\Windows CE Services\\Partners",
        0,
        KEY_READ,
        &ceRootKey);

    if (ERROR_SUCCESS != result)
        return;

    dwIndex = 0;
    do {
        
        TCHAR subKeyName[MAX_PATH];
        TCHAR path[MAX_PATH];
        DWORD size;
        TCHAR deviceName[MAX_PATH];
        DWORD dwType;
        FILETIME filetime;

        size = MAX_PATH;
        result = RegEnumKeyEx(ceRootKey,
            dwIndex,
            subKeyName,
            &size,
            NULL,
            NULL,
            NULL,
            &filetime);

        ++dwIndex;
        
        if (ERROR_SUCCESS != result)
            continue;
        
        result = RegOpenKeyEx(ceRootKey,
            subKeyName,
            0,
            KEY_READ,
            &partnerKey);

        if (ERROR_SUCCESS != result)
            continue;

        size = MAX_PATH;
        result = RegQueryValueEx(partnerKey,
            "DataFolder",
            0,
            &dwType,
            (LPBYTE)path,
            &size);

        if (ERROR_SUCCESS == result) {

            size = MAX_PATH;
            result = RegQueryValueEx(partnerKey,
                "DisplayName",
                0,
                &dwType,
                (LPBYTE)deviceName,
                &size);

            if (ERROR_SUCCESS == result) {

                dle = makeDeviceEntry(deviceName,
                    AG_CE_DEVICE_TYPE,
                    path);

                if (NULL != dle) {

                    sprintf(nameBuffer, nameTemplate, deviceName);

                    ComboBox_InsertString(deviceCombo,
                        AGArrayCount(array),
                        nameBuffer);
                    AGArrayAppend(array, (void*)dle);

                }

            }

        }

        RegCloseKey(partnerKey);

    } while (ERROR_SUCCESS == result);

    RegCloseKey(ceRootKey);

}

/* ----------------------------------------------------------------------------
*/
AGBool AGDeviceUserDropDownSelectDevice(HWND deviceCombo,
                                        AGArray * array,
                                        char * deviceName,
                                        AGDeviceType deviceType)
{
    AGBool result = FALSE;

    int i, n;
    int which = -1;

    if (NULL != deviceName) {

        n = AGArrayCount(array);

        for (i = 0; i < n; ++i) {

            AGDeviceEntry * devtmp;

            devtmp = (AGDeviceEntry *)AGArrayElementAt(array, i);

            if (0 == strcmp(deviceName, devtmp->deviceName)
                && devtmp->deviceType == deviceType) {
                which = i;
                break;
            }

        }

    }

    /* If we couldn't find the current selection, or if there was no
    current selection, select the first person in the list. */
    if (which < 0) {

        which = 0;
        result = FALSE;

    }
    else
        result = TRUE;

    if (AG_NO_DEVICE_TYPE == deviceType)
        result = FALSE;

    /* Actually update the selection. */
    ComboBox_SetCurSel(deviceCombo, which);

    return result;
}

/* ----------------------------------------------------------------------------
*/
AGBool AGDeviceUserDropDownSelectCurrentDevice(HWND deviceCombo,
                                               AGArray * array)
{
    AGBool result = FALSE;
    AGDeviceEntry * dle = NULL;

    /* Get the current device (according to the registry) and
    iterate through drop-down list to match up with that device. */
    dle = AGMobileLinkGetCurrentDevice();
    if (NULL != dle) {

        result = AGDeviceUserDropDownSelectDevice(deviceCombo,
            array,
            dle->deviceName,
            dle->deviceType);

        AGMobileLinkDeviceEntryFree(dle);

    }
    else
        result = AGDeviceUserDropDownSelectDevice(deviceCombo,
            array,
            "",
            AG_NO_DEVICE_TYPE);

    return result;

}

/* ----------------------------------------------------------------------------
*/
AGArray * AGDeviceUserDropDownBuild(HWND deviceCombo, AGBool * ok)
{
    AGArray * array;
    AGDeviceEntry * dle = NULL;
    AGBool result;

    if (NULL != ok)
        *ok = FALSE;

    array = AGArrayNew(AGUnownedPointerElements, 0);
    if (NULL == array)
        return NULL;

    ComboBox_ResetContent(deviceCombo);

    ShowWindow(deviceCombo, SW_SHOWNORMAL);

    dle = AGMobileLinkGetCurrentDevice();
    if (NULL != dle)
        AGMobileLinkDeviceEntryFree(dle);
    else {

        dle = makeDeviceEntry("", AG_NO_DEVICE_TYPE, "");

        if (NULL != dle) {

            ComboBox_InsertString(deviceCombo,
                0,
                "Select a user");
            AGArrayAppend(array, (void*)dle);

        }

    }

    if (isPlatformInstalled(palmPlatform))
        walkPalmList(deviceCombo, array);

    if (isPlatformInstalled(cePlatform))
        walkCEList(deviceCombo, array);

    result = AGDeviceUserDropDownSelectCurrentDevice(deviceCombo, array);

    if (NULL != ok)
        *ok = result;

    return array;

}

/* ----------------------------------------------------------------------------
*/
void AGDeviceUserDropDownReleaseDeviceList(AGArray * array)
{
    uint32 i, n;

    n = AGArrayCount(array);

    for (i = 0; i < n; ++i) {

        AGDeviceEntry * dle = (AGDeviceEntry*)AGArrayElementAt(array, i);

        if (NULL != dle)
            AGMobileLinkDeviceEntryFree(dle);

    }

    if (n > 0)
        AGArrayRemoveAll(array);

    if (NULL != array)
        AGArrayFree(array);
}

/* ----------------------------------------------------------------------------
*/
AGBool AGDeviceUserDropDownHandleSelection(HWND deviceCombo,
                                           AGArray * array)
{
    AGDeviceEntry * dle;
    int index = ComboBox_GetCurSel(deviceCombo);

    /* If there's no selection, select first guy in list. */
    if (LB_ERR == index || index >= AGArrayCount(array))
        index = 0;

    dle = (AGDeviceEntry*)AGArrayElementAt(array, index);

    /* Something's very wrong if this is NULL (bad installation), but
    better to return something meaningful than to crash. */
    if (NULL != dle) {

        AGMobileLinkSetCurrentDevice(dle);

        return (dle->deviceType != AG_NO_DEVICE_TYPE);

    }
    else
        return FALSE;
}

/* ----------------------------------------------------------------------------
*/
void AGDeviceUserDropDownDoSelectionDropDown(AGArray * array,
                                             HWND hwndDeviceList)
{
    RECT r;
    HDC dc;
    HFONT font;
    TEXTMETRIC tm;
    int height;

    dc = GetDC(hwndDeviceList);
    font = GetWindowFont(hwndDeviceList);
    if (NULL != font)
        SelectObject(dc, font);
    GetTextMetrics(dc, &tm);
    ReleaseDC(hwndDeviceList, dc);
    height = max(ComboBox_GetCount(hwndDeviceList) + 2, 4)
        * (tm.tmHeight + tm.tmExternalLeading);

    ComboBox_GetDroppedControlRect(hwndDeviceList, &r);
    ScreenToClient(GetParent(hwndDeviceList), (LPPOINT)&r.left);
    ScreenToClient(GetParent(hwndDeviceList), (LPPOINT)&r.right);

    MoveWindow(hwndDeviceList, r.left, r.top, r.right - r.left,
        height, TRUE);
}

/* ----------------------------------------------------------------------------
*/
void AGDeviceUserDropDownNoDeviceErrorMessage(HWND hwnd)
{
    char message[MAX_PATH];
    char caption[MAX_PATH];

    LoadString(g_hInstance,
        IDS_NO_DEVICE_ERROR_MESSAGE,
        message,
        MAX_PATH);

    LoadString(g_hInstance,
        IDS_NO_DEVICE_ERROR_CAPTION,
        caption,
        MAX_PATH);

    MessageBox(hwnd, message, caption, MB_OK | MB_ICONEXCLAMATION);
}
