/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <AGDigest.h>
#include <AGProtocol.h>
#include <AGClientProcessor.h>
#include <AGCommandProcessor.h>
#include <AGBufferWriter.h>
#include <AGUserConfig.h>
#include <AGSyncCommon.h>
#include <stdlib.h>
#include <stdio.h>
#include <AGConfigResource.h>
#include <AGMobileLinkNet.h>
#include <windowsx.h>

#define DDE_CLASS_NAME "Mobile Link DDE Server Class"

static const LPTSTR kServerClass = TEXT("MAL.Server");
static const LPTSTR kHandlerClass = TEXT("MAL.DDEHandler");
static const LPTSTR kTopic = TEXT("Subscription");
static const LPTSTR kAvantGoServerURI = TEXT("/sync/sub");
static const char * kAvantGoSubXMLWrapper = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\
<!DOCTYPE avantgo_subs_file [ \
<!ELEMENT sub (#PCDATA)>\
]>\
<sub>%s</sub>";

typedef struct {

    AGBool sentCommand;
    AGBool receivedAck;
    char * buffer;
    long bufferSize;
    AGCommandProcessor *cp;
    AGClientProcessor * clientProcessor;
    AGLocationConfig * locationConfig;
    AGServerConfig * serverConfig;
    AGPlatformCalls * platformCalls;
    HWND hDialog;
    UINT iconTimer;
    DWORD iconTimePeriod;
    int iconDegrees;
    int dIconDegrees;
    POINT iconPosition, dIconPosition, iconSize;

} subState;

LRESULT CALLBACK serverProc(HWND, UINT, WPARAM, LPARAM);

/* ----------------------------------------------------------------------------
*/
LRESULT AGConfigAGSubsFilesDDEInitiate(HWND hwnd,
                                       HWND hwndClient,
                                       LPARAM lParam,
                                       ATOM atom)
{
    HWND hwndServer;
    ATOM aApp, aTop;

    aApp = GlobalAddAtom (kServerClass);
    aTop = GlobalAddAtom (kTopic);

    if ((LOWORD(lParam) == NULL
        || LOWORD(lParam) == aApp)
        && (HIWORD (lParam) == NULL
        || HIWORD (lParam) == aTop)) {

        hwndServer = CreateWindow((LPCTSTR)MAKELONG(atom, 0),
            NULL,
            WS_CHILD,
            CW_USEDEFAULT,
            CW_USEDEFAULT,
            CW_USEDEFAULT,
            CW_USEDEFAULT,
            hwnd,
            NULL,
            g_hInstance,
            NULL);

        SetWindowLong(hwndServer, GWL_USERDATA, (LONG)hwndClient);

        SendMessage(hwndClient,
            WM_DDE_ACK,
            (WPARAM)hwndServer,
            MAKELPARAM(aApp, aTop));

    } else {
        GlobalDeleteAtom(aApp);
        GlobalDeleteAtom(aTop);
    }

    return 0;
}

/* ----------------------------------------------------------------------------
*/
int32 getNextCommand(void *out,
                     int32 *newCommand,
                     int32 *commandLength,
                     void **commandBytes)
{
    subState *state = (subState *)out;

    if(!state->sentCommand) {

        state->sentCommand = TRUE;
        *newCommand = (int32)AG_XMLDATA_CMD;
        *commandLength = state->bufferSize;
        *commandBytes = state->buffer;

        return AGCLIENT_CONTINUE;

    } else
        return AGCLIENT_IDLE;
}


/* ----------------------------------------------------------------------------
*/
static void updateProgressText(HWND hwnd, subState * info)
{
    char winText[MAX_PATH], * newWinText;

    GetWindowText(GetDlgItem(hwnd, IDC_PROGRESS_TEXT), winText, MAX_PATH);

    newWinText = malloc(sizeof(char)
        * (strlen(winText) + 16 + strlen(info->serverConfig->serverName)
        + strlen(info->serverConfig->friendlyName)));

    sprintf(newWinText,
        winText, 
        info->serverConfig->friendlyName,
        info->serverConfig->serverName,
        info->serverConfig->serverPort);

    SetWindowText(GetDlgItem(hwnd, IDC_PROGRESS_TEXT), newWinText);

    free(newWinText);
}

/* ----------------------------------------------------------------------------
*/
static void updateResultText(HWND hwnd,
                             subState * info,
                             char * message,
                             AGBool firstTime)
{
    char winText[MAX_PATH], * newWinText;

    if (NULL == message) {
        if (firstTime)
            LoadString(g_hInstance,
                IDS_SUBS_CONNECTING_MESSAGE,
                winText,
                MAX_PATH);
        else
            LoadString(g_hInstance,
                IDS_SUBS_FAILED_MESSAGE,
                winText,
                MAX_PATH);
        newWinText = strdup(winText);
    } else {

        LoadString(g_hInstance, IDS_SUBS_PROGRESS_MESSAGE, winText, MAX_PATH);

        newWinText = malloc(sizeof(char) *
            (strlen(winText) + 16 + strlen(message)));

        sprintf(newWinText, winText, message);

    }

    SetWindowText(GetDlgItem(hwnd, IDC_PROGRESS_RESULT_TEXT), newWinText);

    free(newWinText);
}

/* ----------------------------------------------------------------------------
*/
static int32 handleGOODBYE(void *out,
                           int32 *returnErrorCode,
                           AGSyncStatus syncStatus,
                           int32 errorCode,
                           char *errorMessage)
{
    subState *state = (subState *)out;

    if (NULL != errorMessage) {
        updateResultText(state->hDialog, out, errorMessage, FALSE);
        state->receivedAck = !strcmp(errorMessage, "Subs file received");
    }

    /* pending(miket):  Until server returns right message, we'll always
    assume success if we got this far. */
    state->receivedAck = TRUE;

    return AGCLIENT_CONTINUE;
}

/* ----------------------------------------------------------------------------
*/
static UINT restartTimer(HWND hwnd, subState * info)
{
    return SetTimer(hwnd,
        0,
        info->iconTimePeriod,
        NULL);
}

/* ----------------------------------------------------------------------------
*/
static BOOL onInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
    subState * info;
    HWND hwndIcon;
    RECT r;

    SetWindowLong(hwnd, GWL_USERDATA, lParam);
    info = (subState *)GetWindowLong(hwnd, GWL_USERDATA);

    updateProgressText(hwnd, info);

    updateResultText(hwnd, info, NULL, TRUE);

    info->iconTimePeriod = 750;
    info->dIconDegrees = 90;

    if (0x8000 & GetAsyncKeyState(VK_CONTROL)) {

        info->iconTimePeriod = 15;
        info->dIconDegrees = 2;

        hwndIcon = GetDlgItem(hwnd, IDC_PROGRESS_ICON);
        GetClientRect(hwndIcon, &r);
        info->iconSize.x = r.right;
        info->iconSize.y = r.bottom;

        GetWindowRect(hwndIcon, &r);

        info->iconPosition.x = r.left;
        info->iconPosition.y = r.top;
        ScreenToClient(hwnd, &info->iconPosition);

        info->dIconPosition.x = (rand() % 7) - 3;
        info->dIconPosition.y = (rand() % 7) - 3;
    }

    info->iconTimer = restartTimer(hwnd, info);

    return TRUE;
}

/* ----------------------------------------------------------------------------
*/
static void onTimer(HWND hwnd, UINT id)
{
    subState * info = (subState *)GetWindowLong(hwnd, GWL_USERDATA);
    int iconID;
    HWND hwndIcon;

    if (!info->receivedAck) {
        info->iconTimer = restartTimer(hwnd, info);

        info->iconDegrees += info->dIconDegrees;
        info->iconDegrees %= 360;

        iconID = -1;
        switch (info->iconDegrees / 90) {
            case 0:
                iconID = IDI_APP_ICON;
                break;
            case 1:
                iconID = IDI_APP_ICON90;
                break;
            case 2:
                iconID = IDI_APP_ICON180;
                break;
            case 3:
                iconID = IDI_APP_ICON270;
                break;
        }
    }
    else
        iconID = IDI_APP_ICON;

    hwndIcon = GetDlgItem(hwnd, IDC_PROGRESS_ICON);
    if (iconID >= 0)
        SendMessage(hwndIcon,
            STM_SETIMAGE,
            IMAGE_ICON,
            (LPARAM)LoadIcon(hinst, MAKEINTRESOURCE(iconID)));

    if (info->dIconPosition.x) {

        RECT r;

        GetClientRect(hwnd, &r);

        info->iconPosition.x += info->dIconPosition.x;
        info->iconPosition.y += info->dIconPosition.y;

        if (info->iconPosition.x + info->iconSize.x > r.right) {
            info->iconPosition.x = r.right - info->iconSize.x;
            info->dIconPosition.x = - info->dIconPosition.x;
        }

        if (info->iconPosition.x < r.left) {
            info->iconPosition.x = r.left;
            info->dIconPosition.x = - info->dIconPosition.x;
        }

        if (info->iconPosition.y + info->iconSize.y > r.bottom) {
            info->iconPosition.y = r.bottom - info->iconSize.y;
            info->dIconPosition.y = - info->dIconPosition.y;
        }

        if (info->iconPosition.y < r.top) {
            info->iconPosition.y = r.top;
            info->dIconPosition.y = - info->dIconPosition.y;
        }

        MoveWindow(hwndIcon,
            info->iconPosition.x,
            info->iconPosition.y,
            info->iconSize.x,
            info->iconSize.y,
            TRUE);

    }
}

/* ----------------------------------------------------------------------------
*/
static void onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    subState * info = (subState *)GetWindowLong(hwnd, GWL_USERDATA);

    switch (codeNotify) {
        case BN_CLICKED:
            switch (id) {
                case IDCANCEL: {
                    PostQuitMessage(0);
                    return;
                }
                default:
                    break;
            }
            break;
        default:
            break;

    }
}

/* ----------------------------------------------------------------------------
*/
static void onDestroy(HWND hwnd)
{
    subState * info;

    info = (subState *)GetWindowLong(hwnd, GWL_USERDATA);

    if (info->iconTimer != 0)
        KillTimer(hwnd, info->iconTimer);
}


/* ----------------------------------------------------------------------------
*/
static BOOL CALLBACK progressDialogProc(HWND hwnd,
                                        UINT message,
                                        WPARAM wParam,
                                        LPARAM lParam)
{
    switch (message) {
        HANDLE_MSG(hwnd, WM_INITDIALOG, onInitDialog);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_TIMER, onTimer);
    }

    return 0;   // except for WM_INITDIALOG, returning zero means
                // we didn't process the message.
}

/* ----------------------------------------------------------------------------
*/
static HWND createProgressDialog(HWND hwnd,
                                 void * out)
{
    return CreateDialogParam(hInstance,
        MAKEINTRESOURCE(IDD_PROGRESS_DIALOG),
        hwnd,
        progressDialogProc,
        (LPARAM)out);
}

/* ----------------------------------------------------------------------------
*/
static void terminateProgressDialog(HWND hwnd)
{
    DestroyWindow(hwnd);
}

/* ----------------------------------------------------------------------------
*/
static AGBool sendBytesToSubsServer(HWND hwnd,
                                    char * bytes,
                                    uint32 fileSize,
                                    AGServerConfig * sc)
{
    subState state;
    uint32 timeToClose;
    char * stringConstant;
    AGBool userCancel;
    AGNetCtx ctx;

    if (NULL == bytes)
        return FALSE;

    bzero(&state, sizeof(subState));

    state.bufferSize = fileSize;

    state.buffer = bytes;

    state.serverConfig = AGServerConfigDup(sc);

    /* Set true to cancel the user. No, not really. */
    userCancel = FALSE;

    AGNetInit(&ctx);

    /* Send bytes to server. */

    stringConstant = AGSyncCommonGetStringConstant(agLocationConfigPath,
        FALSE);
    state.locationConfig = AGReadLocationConfigFromDisk(stringConstant);
    free(stringConstant);

    if (NULL != state.serverConfig->serverUri)
        free(state.serverConfig->serverUri);
    state.serverConfig->serverUri = strdup(kAvantGoServerURI);

    state.cp = AGCommandProcessorNew(state.serverConfig);
    state.cp->commands.out = &state;
    state.cp->commands.performGoodbyeFunc = handleGOODBYE;

    state.platformCalls = malloc(sizeof(AGPlatformCalls));
    if (NULL != state.platformCalls) {

        bzero(state.platformCalls, sizeof(AGPlatformCalls));

        state.platformCalls->out = &state;
        state.platformCalls->nextExpansionCommandFunc =
            getNextCommand;

        state.platformCalls->performCommandOut = state.cp;
        state.platformCalls->performCommandFunc =
            AGCommandProcessorGetPerformFunc(state.cp);

    }

    state.hDialog = createProgressDialog(hInstance, hwnd, &state);

    state.clientProcessor = AGClientProcessorNew(state.serverConfig,
        NULL,
        state.locationConfig,
        state.platformCalls,
        TRUE,
        &ctx);

    if (NULL != state.clientProcessor) {

        do {

            state.serverConfig->sendDeviceInfo = FALSE;

            AGCommandProcessorStart(state.cp);

            AGClientProcessorSetBufferServerCommands(state.clientProcessor,
                TRUE);

            AGClientProcessorSync(state.clientProcessor);

            userCancel =
                (AGMobileLinkNetDoProcess(state.clientProcessor) < 0);

        } while (AGCommandProcessorShouldSyncAgain(state.cp)
            && !userCancel);

        AGClientProcessorFree(state.clientProcessor);

    }

    if (!userCancel) {

        if (!state.receivedAck) {
            if (state.iconTimer != 0)
                KillTimer(state.hDialog, state.iconTimer);
            updateResultText(state.hDialog, &state, NULL, FALSE);
        }

        SetWindowText(GetDlgItem(state.hDialog, IDCANCEL), "&Close");

        timeToClose = GetTickCount() + 5000;
        while (!AGMobileLinkNetDoWindowsMessagePump(TRUE)
            && GetTickCount() < timeToClose)
            ;

    }

    if (IsWindow(state.hDialog))
        DestroyWindow(state.hDialog);

    AGLocationConfigFree(state.locationConfig);
    AGCommandProcessorFree(state.cp);

    free(state.platformCalls);

    AGNetClose(&ctx);

    return (state.receivedAck);
}

/* ----------------------------------------------------------------------------
*/
static AGBool handleSubsFile(HWND hwnd, LPSTR name)
{
    HANDLE hFile;
    AGBool submitResult = FALSE;

    hFile = CreateFile(name,
        GENERIC_READ,
        FILE_SHARE_READ,
        NULL,
        OPEN_EXISTING,
        FILE_ATTRIBUTE_NORMAL, NULL);

    if (hFile != INVALID_HANDLE_VALUE) {

        long fileSize = GetFileSize(hFile, NULL);

        HANDLE hMapping = CreateFileMapping(hFile,
            NULL,
            PAGE_READONLY,
            0,
            0,
            NULL);

        if (NULL != hMapping) {

            char *data = (char *)MapViewOfFile(hMapping,
                FILE_MAP_READ,
                0,
                0,
                0);

            if (NULL != data) {

                char * stringConstant;
                AGUserConfig * userConfig;
                uint32 xml_wrapped_data_size;
                char * xml_wrapped_data = NULL;

                /* 
                    The "-1" is computed as follows: 
                    
                    Subtract two for the "%s" that will be replaced in the
                    sprintf function.

                    Add one for the null terminator.

                    Ergo, -1.

                */
                xml_wrapped_data_size = strlen(kAvantGoSubXMLWrapper)
                    + fileSize - 1;
                xml_wrapped_data = (char*)malloc(xml_wrapped_data_size);

                if (NULL != xml_wrapped_data) {

                    HANDLE f;

                    sprintf(xml_wrapped_data,
                        kAvantGoSubXMLWrapper,
                        data);
                
                    stringConstant = AGSyncCommonGetStringConstant(
                        agPalmPreferencesPath,
                        FALSE);

                    userConfig =
                        AGReadUserConfigFromDiskAtomically(stringConstant,
                        &f);
                    free(stringConstant);

                    if (NULL != userConfig) {

                        AGServerConfig * serverConfig = NULL;
                        AGUserConfigEnumerateState * eState = NULL;
                        AGBool alreadySentSuccessfully = FALSE;

                        /* pending (miket): we will eventually put up a dialog
                        asking the user which server to send to... for now,
                        we'll simply send to the first valid server we find
                        in the user's list, and keep iterating until we send
                        one successfully or run out of servers. */
                        do {

                            serverConfig = AGUserConfigEnumerate(userConfig,
                                &eState);
                            if (alreadySentSuccessfully)
                                continue;
                            if (NULL == serverConfig)
                                continue;
                            if (!AGServerConfigIsValid(serverConfig))
                                continue;
                            submitResult = sendBytesToSubsServer(hwnd,
                                hInstance,
                                xml_wrapped_data,
                                xml_wrapped_data_size,
                                serverConfig);
                            if (submitResult)
                                alreadySentSuccessfully = TRUE;

                        } while (NULL != serverConfig);

                    }
                    else
                        userConfig = AGUserConfigNew();

                    /* We must write the userConfig back to disk because
                    the server will go through a standard login process and
                    update the nonce accordingly. */
                    AGWriteUserConfigToDiskAtomically(userConfig, f);

                    AGUserConfigFree(userConfig);
                    
                    free(xml_wrapped_data);

                }

                UnmapViewOfFile(data);

            }

            CloseHandle(hMapping);

        }

        CloseHandle(hFile);

    }

    return submitResult;
}

/* ----------------------------------------------------------------------------
*/
static LRESULT onDDEExecute(HWND hwnd,
                            HWND hwndClient,
                            GLOBALHANDLE hCommands)
{
    DDEACK DdeAck;
    LPSTR pCommands;
    char title[MAX_PATH];
    char message[MAX_PATH];
    AGBool submitResult;

    pCommands = (LPSTR)GlobalLock(hCommands);

    DdeAck.bAppReturnCode = 0;
    DdeAck.reserved = 0;
    DdeAck.fBusy = FALSE;
    DdeAck.fAck = FALSE;

    submitResult = handleSubsFile(g_hInstance,
        hwnd,
        (LPSTR)pCommands);
    
    GlobalUnlock(hCommands);

    if (!PostMessage(hwndClient, WM_DDE_ACK, (WPARAM)hwnd,
        PackDDElParam(WM_DDE_ACK, (DWORD)&DdeAck, (UINT)hCommands))) {
        GlobalFree (hCommands);
        }

    if (submitResult)
        LoadString(g_hInstance,
            IDS_SUBS_SUCCESS_MESSAGE, message, MAX_PATH);
    else
        LoadString(g_hInstance,
            IDS_SUBS_FAILURE_MESSAGE, message, MAX_PATH);

    LoadString(g_hInstance,
        IDS_SUBS_RESULT_TITLE, title, MAX_PATH);
    
    return 0;
}

/* ----------------------------------------------------------------------------
*/
static LRESULT onDDETerminate(HWND hwnd, HWND hwndClient)
{
    PostMessage(hwnd, WM_DDE_TERMINATE, (WPARAM)hwndClient, 0L) ;
    DestroyWindow(hwnd);
    return 0;
}

/* ----------------------------------------------------------------------------
*/
LRESULT CALLBACK serverProc(HWND hwnd,
                            UINT message,
                            WPARAM wParam,
                            LPARAM lParam)
{
    switch (message) {
        case WM_DDE_EXECUTE:
            return onDDEExecute(hwnd, (HWND)wParam, (GLOBALHANDLE)lParam);
        case WM_DDE_TERMINATE:
            return onDDETerminate(hwnd, (HWND)wParam);
    }
    return DefWindowProc(hwnd, message, wParam, lParam);
}

/* ----------------------------------------------------------------------------
*/
ATOM AGConfigAGSubsFilesInit(void)
{
    WNDCLASSEX wcex;

    wcex.cbSize        = sizeof(WNDCLASSEX);
    wcex.style         = 0;
    wcex.lpfnWndProc   = serverProc;
    wcex.cbClsExtra    = 0;
    wcex.cbWndExtra    = 0;
    wcex.hInstance     = hInst;
    wcex.hIcon         = NULL;
    wcex.hCursor       = LoadCursor (NULL, IDC_ARROW);
    wcex.hbrBackground = NULL;
    wcex.lpszMenuName  = NULL;
    wcex.lpszClassName = DDE_CLASS_NAME;
    wcex.hIconSm       = NULL;

    return RegisterClassEx(&wcex);
}
