/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <AGMLSynchronizeCE.h>
#include <AGClientProcessor.h>
#include <AGMLDLLManager.h>
#include <AGUtil.h>
#include <AGCommandProcessor.h>
#include <AGProtocol.h>
#include <AGBufferReader.h>
#include <commctrl.h>
#include <windowsx.h>

typedef sword (*netInitFunc)(AGNetCtx *ctx);
typedef sword (*netCloseFunc)(AGNetCtx *ctx);
typedef void  (*netToggleFunc)(AGNetCtx *ctx, AGBool useAltNet);
typedef int32 (*netCtxSizeFunc)(void);

typedef struct SyncInfo {
    
    AGCommandProcessor *cp;
    AGRecord * record;
    AGMLDLLManager * dllManager;
    HINSTANCE hInstance;
    HWND hwndParent;
    HWND hwndStatus;
    HWND hwndProgress;
    HWND hwndServerName;
    HWND hwndXofY;
    int progressBarPoint;
    int progressBarMax;

    /* Secure Network Library Stuff */
    AGBool          hasseclib;
    netInitFunc     secnetinit;
    netCloseFunc    secnetclose;
    netToggleFunc   sectoggle;
    netCtxSizeFunc  secctxsize;

} SyncInfo;

static int32 performCommand(void *out, int32 *errCode,
                            AGReader *commandToProcess);
static int32 initAndOpenDatabase(void *out, AGDBConfig *db, 
                                 int32 *errCode);
static int32 getNextModifiedRecord(void * out, AGRecord ** record, 
                                   int32 *errCode);
static int32 getNextRecord(void * out, AGRecord ** record,
                           int32 *errCode);
static int32 nextExpansionCommand(void *out, int32 *newCommand,
                                  int32 *commandLength, void **commandBytes);

/* ----------------------------------------------------------------------------
    void syncInfoFree(SyncInfo * pInfo)

    Free the given SyncInfo structure and everything that it knows about.

*/
void syncInfoFree(SyncInfo * pInfo)
{
    if (NULL != pInfo)
        free(pInfo);
}

/* ----------------------------------------------------------------------------
    SyncInfo * syncInfoNew()

    Allocate SyncInfo structure and whatever else is guaranteed to be needed,
    as well.

*/
SyncInfo * syncInfoNew()
{
    SyncInfo * pInfo;

    /* Allocate the SyncInfo structure. */
    pInfo = (SyncInfo *)malloc(sizeof(SyncInfo));
    if (NULL != pInfo) {

        bzero(pInfo, sizeof(SyncInfo));

        return pInfo;
    }

    syncInfoFree(pInfo);
    return NULL;
}

/* ----------------------------------------------------------------------------
    static AGPlatformCalls * setupPlatformCalls(void * out)
*/
static AGPlatformCalls * setupPlatformCalls(void * out)
{
    AGPlatformCalls * result;
    result = (AGPlatformCalls *)malloc(sizeof(AGPlatformCalls));
    bzero(result, sizeof(AGPlatformCalls));
    if (NULL != result) {
        result->out = out;
        result->nextModifiedRecordFunc = getNextModifiedRecord;
        result->nextRecordFunc = getNextRecord;
        result->openDatabaseFunc = initAndOpenDatabase;
        result->nextExpansionCommandFunc = nextExpansionCommand;

        result->performCommandOut = out;
        result->performCommandFunc = performCommand;
    }
    return result;
}

/* ----------------------------------------------------------------------------
    static BOOL doWindowsMessagePump(BOOL eatQuit)

    returns TRUE only if WM_QUIT message was posted.

    if eatQuit is FALSE, we post another nearly identical quit message after
    seeing one.
    
*/
static BOOL doWindowsMessagePump(BOOL eatQuit)
{
    MSG msg;
    while (PeekMessage(&msg, 0, 0, 0, PM_REMOVE | PM_NOYIELD)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
        if (msg.message == WM_QUIT) {
            if (!eatQuit)
                PostQuitMessage((int)msg.wParam);
            return TRUE;
        }
    }
    return FALSE;
}

/* ----------------------------------------------------------------------------
    static int32 doStartServer(void *out,
                               AGServerConfig * serverConfig,
                               int32 *errCode)

    Prepares ourselves and device for a set of MAL commands pertaining to a
    particular server.

*/
static int32 doStartServer(void *out,
                           AGServerConfig * serverConfig,
                           int32 *errCode)
{
    SyncInfo * pInfo = (SyncInfo *)out;
    return AGMLDLLManagerStartServer(pInfo->dllManager,
        serverConfig);
}

/* ----------------------------------------------------------------------------
    static int32 doEndServer(void *out, int32 *errCode)

    Cleanup after sequence of MAL commands.

*/
static int32 doEndServer(void *out, int32 *errCode)
{
    SyncInfo * pInfo = (SyncInfo *)out;
    return AGMLDLLManagerEndServer(pInfo->dllManager);
}

static AGDeviceInfo * allocateAndReadDeviceInfo(void)
{
    AGDeviceInfo * result = NULL;
    return result;
}

static int32 performCommand(void *out, int32 *errCode,
                            AGReader *r)
{
    int32 result;
    SyncInfo * pInfo = (SyncInfo *)out;
    AGPerformCommandFunc defaultPerformCommand =
        AGCommandProcessorGetPerformFunc(pInfo->cp);
    AGBufferReader rc;

    AGBufferReaderInit(&rc, AGBufferReaderPeek((AGBufferReader *)r));

    result = (*defaultPerformCommand)(pInfo->cp, errCode, (AGReader *)&rc);

    result = AGMLDLLManagerPerformCommand(pInfo->dllManager,
        errCode, r);

    AGBufferReaderFinalize(&rc);

    return result;
}

static int32 performTask(void *out, int32 *returnErrorCode,
                         char *currentTask, AGBool bufferable)
{
    SyncInfo * pInfo = (SyncInfo *)out;

    if (NULL != currentTask) {

        WCHAR message[MAX_PATH];
        mbstowcs(message, currentTask, -1);
        SetWindowText(pInfo->hwndStatus, message);

    }

    return AGCLIENT_CONTINUE;
}

static int32 performItem(void *out, int32 *returnErrorCode,
                         int32 currentItemNumber,
                         int32 totalItemCount,
                         char *currentItem)
{
    SyncInfo * pInfo = (SyncInfo *)out;
    WCHAR buf[32];

    SendMessage(pInfo->hwndProgress,
        PBM_SETPOS,
        (currentItemNumber * pInfo->progressBarMax) / totalItemCount,
        0);

    wsprintf(buf, TEXT("%d of %d"), currentItemNumber, totalItemCount);
    SetWindowText(pInfo->hwndXofY, buf);
    UpdateWindow(pInfo->hwndXofY);

    return AGCLIENT_CONTINUE;
}

static int32 performGoodbye(void *out,
                            int32 *returnErrorCode,
                            AGSyncStatus syncStatus,
                            int32 errorCode,
                            char *errorMessage)
{
    SyncInfo * pInfo = (SyncInfo *)out;

    if (NULL != errorMessage) {

        if (0 != errorCode) {

            WCHAR message[MAX_PATH];
            mbstowcs(message, errorMessage, -1);
            MessageBox(pInfo->hwndParent,
                message,
                TEXT("Message from server"),
                MB_ICONEXCLAMATION | MB_OK);

        }

    }

    return AGCLIENT_CONTINUE;
}

static int32 initAndOpenDatabase(void *out, AGDBConfig *db, 
                                 int32 *errCode)
{
    SyncInfo * pInfo = (SyncInfo *)out;

    pInfo->record = NULL;

    return AGMLDLLManagerOpenDatabase(pInfo->dllManager,
        db,
        errCode);
}

static int32 nextExpansionCommand(void *out, int32 *newCommand,
                                  int32 *commandLength, void **commandBytes)
{
    SyncInfo * pInfo = (SyncInfo *)out;

    pInfo->record = NULL;

    return AGCLIENT_IDLE;
}

static int32 getRecordBase(SyncInfo * pInfo, AGBool modonly,
                           AGRecord ** record, int32 * errCode)
{
    int32 result;

    if (NULL != pInfo->record) {
        AGRecordFree(pInfo->record);
        pInfo->record = NULL;
    }

    /* Send notification to device that this is a getNextRecord or 
    getNextModifiedRecord sequence. */
    if (modonly)
        result = AGMLDLLManagerGetNextModifiedRecord(
            pInfo->dllManager,
            &pInfo->record,
            errCode);
    else
        result = AGMLDLLManagerGetNextRecord(
            pInfo->dllManager,
            &pInfo->record,
            errCode);

    *record = pInfo->record;

    return result;

}

static int32 getNextModifiedRecord(void * out, AGRecord ** record, 
                                   int32 *errCode)
{
    return getRecordBase((SyncInfo *)out, TRUE, record, errCode);
}

static int32 getNextRecord(void * out, AGRecord ** record,
                           int32 *errCode)
{
    return getRecordBase((SyncInfo *)out, FALSE, record, errCode);
}

/* ----------------------------------------------------------------------------
    static AGDeviceInfo * createDeviceInfo(void)

    Create a deviceInfo record and fill it in.

*/
static AGDeviceInfo * createDeviceInfo(void)
{
    AGDeviceInfo * devInfo = NULL;

    devInfo = AGDeviceInfoNew();

    if (NULL != devInfo) {

        STORE_INFORMATION info;
        OSVERSIONINFO osver;
        HDC hdc = GetDC(NULL);
        char osverstringmajor[3];
        char osverstringminor[2];
        char osverstring[6];
        char * minorptr;

        /* Ask device how much memory it has. */
        GetStoreInformation(&info);

        /* Find out version of operating system. */
        osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&osver);
        _itoa(osver.dwMajorVersion, osverstringmajor, 10);
        if (osver.dwMinorVersion < 10) {
            strcpy(osverstringminor, "0");
            minorptr = &osverstringminor[1];
        }
        else
            minorptr = &osverstringminor[0];
        _itoa(osver.dwMinorVersion, minorptr, 10);
        strcpy(osverstring, osverstringmajor);
        strcat(osverstring, ".");
        strcat(osverstring, osverstringminor);

        devInfo->availableBytes   = info.dwFreeSize;
        devInfo->serialNumber     = strdup("");
        devInfo->osName           = strdup("WINCE_OS");
        devInfo->osVersion        = strdup(osverstring);
        devInfo->screenWidth      = GetDeviceCaps(hdc, HORZRES);
        devInfo->screenHeight     = GetDeviceCaps(hdc, VERTRES);
        devInfo->colorDepth       = GetDeviceCaps(hdc, BITSPIXEL);

        ReleaseDC(NULL, hdc);
    }
    return devInfo;
}

static AGBool doClientProcessorLoop(SyncInfo * pInfo,
                                    AGUserConfig * userConfig,
                                    AGLocationConfig * locationConfig,
                                    AGPlatformCalls * platform,
                                    AGNetCtx *ctx,
                                    HINSTANCE hInstance,
                                    HWND hwndParent,
                                    HWND hwndStatus,
                                    HWND hwndProgress,
                                    AGBool * quit)
{
    int32 dummyError;
    int32 cpResult;
    int32 syncCount;
    int32 i, n;
    AGClientProcessor * clientProcessor = NULL;
    AGBool cancelled = FALSE;

    n = AGUserConfigCount(userConfig);

    for (i = 0; i < n; ++i) {

        TCHAR friendlybuf[MAX_PATH];

        AGDeviceInfo * deviceInfo = NULL;
        AGServerConfig * sc = AGUserConfigGetServerByIndex(userConfig, i);

        if (cancelled)
            continue;

        if (NULL == sc)
            continue;

        if (sc->disabled)
            continue;

        if (NULL == sc->serverName || sc->serverPort <= 0)
            continue;

        mbstowcs(friendlybuf, sc->friendlyName, -1);
        SetWindowText(pInfo->hwndServerName, friendlybuf);
        UpdateWindow(pInfo->hwndServerName);

        pInfo->cp = AGCommandProcessorNew(sc);
        pInfo->cp->commands.out = pInfo;
        pInfo->cp->commands.performTaskFunc = performTask;
        pInfo->cp->commands.performItemFunc = performItem;
        pInfo->cp->commands.performGoodbyeFunc = performGoodbye;

        doStartServer(pInfo, sc, &dummyError);

        syncCount = 0;
        do {
            deviceInfo = createDeviceInfo();

            AGCommandProcessorStart(pInfo->cp);
            clientProcessor = AGClientProcessorNew(pInfo->cp->serverConfig,
                                                   deviceInfo, 
                                                   locationConfig,
                                                   platform,
                                                   TRUE,
                                                   ctx);
//            AGClientProcessorSetBufferServerCommands(clientProcessor, TRUE);

            if (NULL == clientProcessor)
                continue;

            AGClientProcessorSync(clientProcessor);
            cpResult = AGCLIENT_CONTINUE;
            while (AGCLIENT_CONTINUE == cpResult && !cancelled) {
                cpResult = AGClientProcessorProcess(clientProcessor);
                if (quit && *quit)
                    cancelled = TRUE;
                if (AGCLIENT_CONTINUE == cpResult 
                    && doWindowsMessagePump(TRUE)) {
                    cancelled = TRUE;
                    cpResult = AGCLIENT_IDLE;
                }
            }

            AGClientProcessorFree(clientProcessor);


            if (NULL != deviceInfo)
                AGDeviceInfoFree(deviceInfo);
        } while (!cancelled &&
                 AGCommandProcessorShouldSyncAgain(pInfo->cp) &&
                 syncCount++ < 2);

        doEndServer(pInfo, &dummyError);

        AGCommandProcessorFree(pInfo->cp);
        pInfo->cp = NULL;
    }

    return TRUE; /* success */

}
/*---------------------------------------------------------------------------*/
static void checkForSecNet(SyncInfo * pInfo) 
{
    
    /* PENDING Must look in registry for dll */
    HINSTANCE h = LoadLibrary(TEXT("secnet.dll"));
    pInfo->hasseclib = 0;

    if (h) {
        pInfo->secnetinit = 
            (netInitFunc)GetProcAddress(h, TEXT("NetInit"));
        pInfo->secnetclose = 
            (netCloseFunc)GetProcAddress(h, TEXT("NetClose"));
        pInfo->sectoggle = 
            (netToggleFunc)GetProcAddress(h, TEXT("NetToggle"));
        pInfo->secctxsize = 
            (netCtxSizeFunc)GetProcAddress(h, TEXT("NetGetCtxSize"));

        if (pInfo->sectoggle && pInfo->secnetclose && 
            pInfo->secnetinit && pInfo->secctxsize)
            pInfo->hasseclib = 1;
    }
}

void AGMLSynchronize(HINSTANCE hInstance,
                     HWND hwndParent,
                     HWND hwndStatus,
                     HWND hwndProgress,
                     HWND hwndServerName,
                     HWND hwndXofY,
                     AGUserConfig * userConfig,
                     AGLocationConfig * locationConfig,
                     AGBool * quit)
{
    SyncInfo * pInfo = NULL;
    AGPlatformCalls * platform = NULL;
    AGNetCtx *ctx;

    pInfo = syncInfoNew();
    if (NULL != pInfo) {

        SetWindowText(hwndStatus,
            TEXT("Establishing connection to network..."));

        pInfo->hInstance = hInstance;
        pInfo->hwndParent = hwndParent;
        pInfo->hwndStatus = hwndStatus;
        pInfo->hwndProgress = hwndProgress;
        pInfo->hwndServerName = hwndServerName;
        pInfo->hwndXofY = hwndXofY;
        pInfo->progressBarPoint = 0;
        pInfo->progressBarMax = 100;
        ShowWindow(hwndProgress, SW_SHOWNORMAL);
        UpdateWindow(hwndProgress);
        ShowWindow(hwndServerName, SW_SHOWNORMAL);
        UpdateWindow(hwndServerName);
        ShowWindow(hwndXofY, SW_SHOWNORMAL);
        UpdateWindow(hwndXofY);

        SendMessage(hwndProgress,
            PBM_SETRANGE,
            0,
            MAKELPARAM(0, pInfo->progressBarMax));
        SendMessage(hwndProgress, PBM_SETPOS, pInfo->progressBarPoint, 0);

        /* Start Mobile Application Link net services. */
        checkForSecNet(pInfo);
        if (pInfo->hasseclib) {
            ctx = (AGNetCtx *)malloc((*pInfo->secctxsize)());
            (*pInfo->secnetinit)(ctx);
        } else {
            ctx = (AGNetCtx *)malloc(sizeof(AGNetCtx));
            AGNetInit(ctx);
        }

        platform = setupPlatformCalls(pInfo);
        if (NULL != platform) {
    
            pInfo->dllManager = AGMLDLLManagerNew();
            if (NULL != pInfo->dllManager) {

                doClientProcessorLoop(pInfo,
                    userConfig,
                    locationConfig,
                    platform,
                    ctx,
                    hInstance,
                    hwndParent,
                    hwndStatus,
                    hwndProgress,
                    quit);

                AGMLDLLManagerFree(pInfo->dllManager);

            }

            free(platform);

        }

        if (pInfo->hasseclib) {
            (*pInfo->secnetclose)(ctx);
        } else { 
            AGNetClose(ctx);
        }

        syncInfoFree(pInfo);

        ShowWindow(hwndProgress, SW_HIDE);
        ShowWindow(hwndServerName, SW_HIDE);
        ShowWindow(hwndXofY, SW_HIDE);

    }
    
}

static BOOL onInitDialog(HWND hwnd, HWND hwndFocus, LPARAM lParam)
{
    PROPSHEETPAGE * psp;

    /* When the PropertySheet function creates the page,
    the dialog box procedure for the page receives a WM_INITDIALOG message.
    The lParam parameter of this message points to the PROPSHEETPAGE
    structure used to create the page. */

    psp = (PROPSHEETPAGE *)lParam;
    SetWindowLong(hwnd, GWL_USERDATA, psp->lParam);

    return TRUE;
}

static BOOL CALLBACK syncDlgProc(HWND hwnd,
                                 UINT message,
                                 WPARAM wParam,
                                 LPARAM lParam)
{
    switch (message) {
        HANDLE_MSG(hwnd, WM_INITDIALOG, onInitDialog);
    }
    return 0;
}

void AGMLStandaloneSynchronize(HINSTANCE hInstance)
{
    //pending(miket):  implement
}
