/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

#include <malclient.h>
#include <AGServerConfig.h>
#include <AGDeviceInfo.h>
#include <AGProtocol.h>
#include <AGBufferReader.h>
#include <AGArray.h>
#include <AGClientProcessor.h>
#include <AGUtil.h>
#include <AGWriter.h> // for AGCompactSize()
#include <AGMsg.h>
#include <AGPalmProtocol.h>
#include <AGCommandProcessor.h>

#include <progresspanel.h>
#include <palmmal.h>
#include <debug.h>

#include <SysEvtMgr.h> //For the EvtResetAutoOffTimer() call

#include <libmal.h>
#include <thunk.h>

extern Word MALLibRefNum;

#ifdef HAVE_SEC_LIB
#include <sslnet.h>
#endif

#define DEFAULT_CARD_NUM ((UInt)0)
typedef struct ClientState {
    AGServerConfig *serverConfig;
    AGDBConfig *dbconfig;
    AGDeviceInfo deviceInfo;
    ProgressPanel *progPanel;

    uint32 currentDbHandle;
    AGBool dbIsResource;
    PalmMalCRawRecordInfo rawRec;   
    AGRecord agRecord;
    char *password;
    
    AGCommandProcessor cp;
} ClientState;

static ClientState *clientStateNew(AGServerConfig *serverConfig);
static void clientStateFree(ClientState *state);

static int32 getNextModifiedRecord(void *out, AGRecord **record, int32 *errCode); 
static int32 getNextRecord(void *out, AGRecord **record, int32 *errCode);
static int32 openDatabaseCallback(void *out, AGDBConfig *db, int32 *errCode);


static int32 cmdTASK(void *out, int32 *returnErrorCode, char *currentTask, 
                     AGBool bufferable);
static int32 cmdITEM(void *out, int32 *returnErrorCode,
                                   int32 currentItemNumber,
                                   int32 totalItemCount,
                                   char *currentItem);
static int32 cmdDELETEDATABASE(void *out, int32 *returnErrorCode,
                                             char *dbname);
static int32 cmdOPENDATABASE(void *out, int32 *returnErrorCode,
                                           char *dbname);
static int32 cmdCLOSEDATABASE(void *out, int32 *returnErrorCode);
static int32 cmdCLEARMODS(void *out, int32 *returnErrorCode);
static int32 cmdGOODBYE(void *out, int32 *returnErrorCode,
                                      AGSyncStatus syncStatus,
                                      int32 errorCode,
                                      char *errorMessage);
static int32 cmdRECORD(void *out, int32 *returnErrorCode,
                                     int32 *newUID,
                                     int32 uid,
                                     AGRecordStatus mod,
                                     int32 recordDataLength,
                                     void *recordData,
                                     int32 platformDataLength,
                                     void *platformData);
//static int32 cmdEXPANSION(void *out, int32 *returnErrorCode,
//                                      int32 expansionCommand, 
//                                      int32 commandLength,
//                                      void *commandBytes);

static long primativeOpenDB(ClientState *state, char *dbname, 
                            AGBool create, AGDBConfig *dbConfig);
static void closeCurrentRecord(ClientState *state);

static int16 doSyncServer(AGServerConfig *serverConfig, ProgressPanel *panel,
                            AGNetCtx *ctx);
static void updateDeviceAvailableBytes(ClientState *state);

ExportFunc int16 MALSyncAllServers(AGUserConfig *userConfig)
{
    int32 count, i;
    ProgressPanel panel;
    AGServerConfig *serverConfig;
    int16 cancelled = 0;
    AGNetCtx* ctx;
    AGBool candosecure;
    
    count = MALLibAGUserConfigCount(MALLibRefNum, userConfig);

    if(count < 1)
        return 1;

    /* Until we resolve the issues with loading the security library as
       a shared library, we will if def it */
    ctx = (AGNetCtx *)malloc(MALLibAGNetGetCtxSize(MALLibRefNum));
    if (!ctx)
        return 1;
    bzero(ctx, MALLibAGNetGetCtxSize(MALLibRefNum));
    if(MALLibAGNetInit(MALLibRefNum, ctx)) {
        free(ctx);
        return 1;
    }
    
    candosecure = MALLibAGNetCanDoSecure(MALLibRefNum, ctx);
    ProgressPanelInit(&panel);
    ProgressPanelShow(&panel);

    for(i = 0; i < count && !cancelled; i++) {
        serverConfig = MALLibAGUserConfigGetServerByIndex(MALLibRefNum, userConfig, i);

        if (candosecure)
            MALLibAGNetToggle(MALLibRefNum, ctx,
                              serverConfig->connectSecurely);
 
        if(!serverConfig->disabled
                && MALLibAGServerConfigIsValid(MALLibRefNum, serverConfig))
            cancelled = doSyncServer(serverConfig, &panel, ctx);
    }
    userConfig->dirty = TRUE;

    ProgressPanelFinalize(&panel);

    MALLibAGNetClose(MALLibRefNum, ctx);

    free(ctx);

    return 0;
}

ExportFunc int16 MALSyncServer(AGServerConfig *serverConfig)
{
    ProgressPanel panel;
    AGNetCtx ctx;

    if(serverConfig->disabled 
        || !MALLibAGServerConfigIsValid(MALLibRefNum, serverConfig)) {
        return 1;
    }

    if(MALLibAGNetInit(MALLibRefNum, &ctx))
        return 1;

    ProgressPanelInit(&panel);
    ProgressPanelShow(&panel);

    doSyncServer(serverConfig, &panel, &ctx);
    ProgressPanelFinalize(&panel);
    MALLibAGNetClose(MALLibRefNum, &ctx);
    return 0;
}

static int16 doSyncServer(AGServerConfig *serverConfig, ProgressPanel *panel,
                          AGNetCtx *ctx)
{
    int8 syncCount = 0;
    AGPlatformCalls calls;
    AGClientProcessor *clientProcessor;
    ClientState *state;
    EventType event;
    AGBool cancel = FALSE;
    int32 result = 0;

    state = clientStateNew(serverConfig);
    state->progPanel = panel;

    bzero(&calls, sizeof(AGPlatformCalls));
    calls.out = state;
    calls.openDatabaseFunc = ThunkMake(openDatabaseCallback);
    calls.nextModifiedRecordFunc = ThunkMake(getNextModifiedRecord);
    calls.nextRecordFunc = ThunkMake(getNextRecord);

    calls.performCommandOut = &state->cp;
    calls.performCommandFunc = MALLibAGCommandProcessorGetPerformFunc(MALLibRefNum, &state->cp);
    do {
        ProgressPanelSetTask(state->progPanel, MALLibAGGetMsg(MALLibRefNum, AGMSGStartingStringId));

        MALLibAGCommandProcessorStart(MALLibRefNum, &state->cp);
        updateDeviceAvailableBytes(state);
        clientProcessor = MALLibAGClientProcessorNew(MALLibRefNum, state->serverConfig, 
                                               &state->deviceInfo, NULL, 
                                               &calls, FALSE, ctx);
        MALLibAGClientProcessorSync(MALLibRefNum, clientProcessor);
        while (!cancel 
            && (result = MALLibAGClientProcessorProcess(MALLibRefNum, clientProcessor)) 
                == AGCLIENT_CONTINUE)
        {
            // This makes sure the device doesn't goto sleep while
            // we're processing data from the modem. This was directly
            // from Palm's FAQ.
            EvtResetAutoOffTimer();
    
            EvtGetEvent(&event, 1);
            cancel = (PROGRESS_CANCELED 
                        == ProgressPanelHandleEvent(state->progPanel, &event));
        }

//PENDING(klobad) replace with alert dialog
        if(result == AGCLIENT_ERR) {
            char *errStr = MALLibAGGetMsg(MALLibRefNum, clientProcessor->errStringId);
            if(errStr != NULL) {
                ProgressPanelSetTask(state->progPanel, errStr);
            } else {
                ProgressPanelSetTask(state->progPanel, 
                                            "Unknown Err during Sync");
            }
            sleep(5);
        }

        MALLibAGClientProcessorFree(MALLibRefNum, clientProcessor);
    } while (!cancel && MALLibAGCommandProcessorShouldSyncAgain(MALLibRefNum, &state->cp) && syncCount++ < 2);
    
    ThunkFree(calls.openDatabaseFunc);
    ThunkFree(calls.nextModifiedRecordFunc);
    ThunkFree(calls.nextRecordFunc);

    clientStateFree(state);
    return cancel;
}

static void updateDeviceAvailableBytes(ClientState *state)
{
    int32 freespace;
    PalmMalAvailableBytes(0, &freespace);
    state->deviceInfo.availableBytes = freespace;
}

static ClientState *clientStateNew(AGServerConfig *serverConfig)
{
    ClientState *state = MemPtrNew(sizeof(ClientState));

    bzero(state, sizeof(ClientState));
    MALLibAGDeviceInfoInit(MALLibRefNum, &state->deviceInfo);

    state->serverConfig = serverConfig;
    //availableBytes filled in during main loop
    PalmMalGetHardwareConfig(&state->deviceInfo.osName,
                                &state->deviceInfo.osVersion,
                                &state->deviceInfo.colorDepth,
                                &state->deviceInfo.screenWidth,
                                &state->deviceInfo.screenHeight,
                                &state->deviceInfo.serialNumber);

    MALLibAGCommandProcessorInit(MALLibRefNum, &state->cp, state->serverConfig);
    state->cp.commands.out = state;
    state->cp.commands.performTaskFunc = ThunkMake(cmdTASK);
    state->cp.commands.performItemFunc = ThunkMake(cmdITEM);
    state->cp.commands.performDeleteDatabaseFunc =ThunkMake(cmdDELETEDATABASE);
    state->cp.commands.performOpenDatabaseFunc = ThunkMake(cmdOPENDATABASE);
    state->cp.commands.performCloseDatabaseFunc = ThunkMake(cmdCLOSEDATABASE);
    state->cp.commands.performClearModsFunc = ThunkMake(cmdCLEARMODS);
    state->cp.commands.performGoodbyeFunc = ThunkMake(cmdGOODBYE);
    state->cp.commands.performRecordFunc = ThunkMake(cmdRECORD);
//  state->cp.commands.performExpansionFunc = ThunkMake(cmdEXPANSION);

    return state;
}

static void clientStateFree(ClientState *state)
{
    // if the user cancelled, we need to close the currently 
    // open database
    if(state->currentDbHandle) {
        PalmMalCloseDB(state->currentDbHandle);
        state->currentDbHandle = NULL;
        state->dbIsResource = FALSE;
    }

    /* clean up thunks */
    ThunkFree(state->cp.commands.performTaskFunc);
    ThunkFree(state->cp.commands.performItemFunc);
    ThunkFree(state->cp.commands.performDeleteDatabaseFunc);
    ThunkFree(state->cp.commands.performOpenDatabaseFunc);
    ThunkFree(state->cp.commands.performCloseDatabaseFunc);
    ThunkFree(state->cp.commands.performClearModsFunc);
    ThunkFree(state->cp.commands.performGoodbyeFunc);
    ThunkFree(state->cp.commands.performRecordFunc);
//  ThunkFree(state->cp.commands.performExpansionFunc);

    MALLibAGDeviceInfoFinalize(MALLibRefNum, &state->deviceInfo);
    MALLibAGCommandProcessorFinalize(MALLibRefNum, &state->cp);

    MemPtrFree(state);
}

// This function is passed into to AGPlatformCalls struct
// when we initialize the AGClientProcessor
//
// The AGClientProcessor calls this record when the 
// AGDBConfig says that the server wants just the mod the records
// from a database. 
static int32 getNextModifiedRecord(void *out, AGRecord **record, int32 *errCode) 
{
    ClientState *state = (ClientState *)out;
    int32 result = 0;
    uint32 lastRecId;

    if(state->rawRec.recId != 0) {
        lastRecId = state->rawRec.recId;
        PalmMalCloseRecord(&state->rawRec);
        state->rawRec.recId = 0;
    } else {
        lastRecId = 0;
    }

    bzero(&state->rawRec, sizeof(PalmMalCRawRecordInfo));
    bzero(&state->agRecord, sizeof(AGRecord));

    state->rawRec.recId = lastRecId;
    state->rawRec.fileHandle = state->currentDbHandle;

    result = PalmMalReadNextModifiedRec(&state->rawRec);
    if (result == PALMMAL_OK) { 
        state->agRecord.uid = state->rawRec.recId;
        state->agRecord.recordDataLength = state->rawRec.recSize;
        state->agRecord.recordData = state->rawRec.pBytes;
        state->agRecord.platformDataLength = 0;
        state->agRecord.platformData = NULL;
        state->agRecord.status = MALLibAGPalmPilotAttribsToMALMod(MALLibRefNum, 
                                                    state->rawRec.attribs);
        *record = &state->agRecord;
        return AGCLIENT_CONTINUE;
    } 

    // There were no more records, end properly
    if(result == PALMMAL_RECORD_NOT_FOUND) {
        *record = NULL;
        *errCode = 0;

        if(state->currentDbHandle != NULL) {
            PalmMalCloseDB(state->currentDbHandle);
            state->currentDbHandle = NULL;
            state->dbIsResource = FALSE;
        }

        return AGCLIENT_IDLE;
    }

    // This should not happen
//  *errCode = ???;
    return AGCLIENT_ERR;
}

// This function is passed into to AGPlatformCalls struct
// when we initialize the AGClientProcessor
//
// The AGClientProcessor calls this record when the 
// AGDBConfig says that the server wants all the records
// from a database. 
static int32 getNextRecord(void *out, AGRecord **record, int32 *errCode) 
{
    ClientState *state = (ClientState *)out;
    int32 result = 0;
    uint32 lastRecIndex = 0;
    int32 count = 0;

    if(state->rawRec.recId != 0) {
        lastRecIndex = state->rawRec.recIndex;
        PalmMalCloseRecord(&state->rawRec);
        state->rawRec.recId = 0;
    } else {
        lastRecIndex = -1;
    }
    bzero(&state->rawRec, sizeof(PalmMalCRawRecordInfo));
    bzero(&state->agRecord, sizeof(AGRecord));

    PalmMalGetDBRecordCount(state->currentDbHandle, &count);
    lastRecIndex++;
    if(count < 1 || count <= lastRecIndex) {
        *record = NULL;
        *errCode = 0;
        return AGCLIENT_IDLE;
    }

    state->rawRec.recIndex = lastRecIndex;
    state->rawRec.fileHandle = state->currentDbHandle;

    result = PalmMalReadRecordByIndex(&state->rawRec);
    if (result == PALMMAL_OK) { 
        state->agRecord.uid = state->rawRec.recId;
        state->agRecord.recordDataLength = state->rawRec.recSize;
        state->agRecord.recordData = state->rawRec.pBytes;
        //PENDING(klobad) check the wantsPlatformData value here
        state->agRecord.platformDataLength = 0;
        state->agRecord.platformData = NULL;
        state->agRecord.status = MALLibAGPalmPilotAttribsToMALMod(MALLibRefNum, 
                                            state->rawRec.attribs);
        *record = &state->agRecord;
        return AGCLIENT_CONTINUE;
    }

    // There were no more records, end properly
    if(result == PALMMAL_RECORD_NOT_FOUND) {
        *record = NULL;
        *errCode = 0;

        if(state->currentDbHandle != NULL) {
            PalmMalCloseDB(state->currentDbHandle);
            state->currentDbHandle = NULL;
            state->dbIsResource = FALSE;
        }
        return AGCLIENT_IDLE;
    }

    // This should not happen
//  *errCode = ???;
    return AGCLIENT_ERR;
}

static int32 openDatabaseCallback(void *out, AGDBConfig *db, int32 *errCode)
{
    int32 result;
    ClientState *state = (ClientState *)out;

    result = primativeOpenDB(state, db->dbname, FALSE, NULL);
    if(result != PALMMAL_OK) {
        *errCode = AGCLIENT_OPEN_ERR;
        return AGCLIENT_ERR;
    }
    return result;
}

static int32 cmdTASK(void *out, int32 *returnErrorCode, char *currentTask, 
                     AGBool bufferable)
{
    closeCurrentRecord((ClientState *)out);
    ProgressPanelSetTask(((ClientState *)out)->progPanel, currentTask);
    return AGCLIENT_CONTINUE;
}

static int32 cmdITEM(void *out, int32 *returnErrorCode,
                                   int32 currentItemNumber,
                                   int32 totalItemCount,
                                   char *currentItem)
{
    closeCurrentRecord((ClientState *)out);
    ProgressPanelSetItem(((ClientState *)out)->progPanel, 
                            currentItem, 
                            currentItemNumber,
                            totalItemCount);
    return AGCLIENT_CONTINUE;
}

static int32 cmdDELETEDATABASE(void *out, int32 *returnErrorCode,
                                             char *dbname)
{
    closeCurrentRecord((ClientState *)out);
    PalmMalDeleteDB(dbname, DEFAULT_CARD_NUM);
    return AGCLIENT_CONTINUE;
}

static int32 cmdOPENDATABASE(void *out, int32 *returnErrorCode,
                                           char *dbname)
{
    ClientState *state = (ClientState *)out;
    closeCurrentRecord(state);
    primativeOpenDB(state, dbname, TRUE, state->dbconfig);
    return AGCLIENT_CONTINUE;
}

static int32 cmdCLOSEDATABASE(void *out, int32 *returnErrorCode)
{
    ClientState *state = (ClientState *)out;

    closeCurrentRecord(state);
    if(state->currentDbHandle) {
        PalmMalCloseDB(state->currentDbHandle);
        state->currentDbHandle = NULL;
        state->dbIsResource = FALSE;
    }
    return AGCLIENT_CONTINUE;
}

static int32 cmdCLEARMODS(void *out, int32 *returnErrorCode)
{
    ClientState *state = (ClientState *)out;

    closeCurrentRecord(state);
    if(state->currentDbHandle && !state->dbIsResource) {
        PalmMalPurgeDeletedRecs(state->currentDbHandle);
        PalmMalResetSyncFlags(state->currentDbHandle);
    }

    return AGCLIENT_CONTINUE;
}

static int32 cmdGOODBYE(void *out, int32 *returnErrorCode,
                                      AGSyncStatus syncStatus,
                                      int32 errorCode,
                                      char *errorMessage)
{
    ClientState *state = (ClientState *)out;

    closeCurrentRecord(state);
    if(errorMessage != NULL) {
        ProgressPanelSetItem(state->progPanel, 
                            NULL, 
                            0,
                            0);
        ProgressPanelSetTask(state->progPanel, errorMessage);

        SysTaskDelay(100); /* sleep for one second */
    }
    return AGCLIENT_CONTINUE;
}

static int32 cmdRECORD(void *out, int32 *returnErrorCode,
                                     int32 *newUID,
                                     int32 uid,
                                     AGRecordStatus mod,
                                     int32 recordDataLength,
                                     void *recordData,
                                     int32 platformDataLength,
                                     void *platformData)
{
    PalmMalCRawRecordInfo rawRec;
    int32 result;
    int16 recIndex = 0;
    ClientState *state = (ClientState *)out;
    AGBufferReader r;

    closeCurrentRecord(state);
    if(mod == AG_RECORD_NEW_TEMPORARY_UID) {
        uid = 0; // reset the uid so the WriteRecord creates a new one
    }

    if(platformDataLength && platformData) {
        MALLibAGBufferReaderInit(MALLibRefNum, &r, platformData);
        MALLibAGPalmReadRecordPlatformData(MALLibRefNum, (AGReader *)&r, &recIndex);
        MALLibAGBufferReaderFinalize(MALLibRefNum, &r);
    }

    if (AG_RECORD_DELETED == mod) {
        PalmMalInit(&rawRec, state->currentDbHandle,
                            uid, recIndex, 0, 0, 0);
        result = PalmMalDeleteRecord(&rawRec);
    } else if (recordDataLength <= 0x0000ffff) {
        PalmMalInit(&rawRec, state->currentDbHandle,
                    uid, recIndex, (uint8)MALLibAGPalmMALModToPilotAttribs(MALLibRefNum, mod),
                    recordDataLength, (uint8*)recordData);
        if(!state->dbIsResource) {
            result = PalmMalWriteRecord(&rawRec);
            *newUID = rawRec.recId;
        } else {
            result = PalmMalWriteResource(&rawRec);
        }
    }
    return AGCLIENT_CONTINUE;
}

static long primativeOpenDB(ClientState *state, char *dbname, 
                            AGBool create, AGDBConfig *dbConfig)
{
    long result = 0;
    uint32 creator = 0, type = 0, flags = 0;
    AGDBConfig *dbconfig;
    AGBufferReader * r = NULL;

    if (NULL == dbname || NULL == state) {
        return result;
    }

    if(state->currentDbHandle) {
        PalmMalCloseDB(state->currentDbHandle);
        state->currentDbHandle = NULL;
        state->dbIsResource = FALSE;
    }
    
    result = PalmMalOpenDB(dbname, DEFAULT_CARD_NUM, 
                        dmModeReadWrite, &state->currentDbHandle);
    if(result == PALMMAL_OK)
        return result;

    if(!create)
        return result;

    // The open failed and they want us to create the database
    if(dbConfig != NULL)
        dbconfig = dbConfig;
    else
        dbconfig = MALLibAGServerConfigGetDBConfigNamed(MALLibRefNum, state->serverConfig, dbname);

    if (NULL == dbconfig)
        return result;

    // platformData is required for the palm device
    if (0 == dbconfig->platformDataLength 
        || NULL == dbconfig->platformData)
        return result;

    r = MALLibAGBufferReaderNew(MALLibRefNum, (uint8*)dbconfig->platformData);
    if (NULL == r)
        return result;

    MALLibAGPalmReadDBConfigPlatformData(MALLibRefNum, (AGReader*)r, &creator, &type, &flags);

    if((flags & dmHdrAttrResDB) > 0)
        state->dbIsResource = true;
    else
        state->dbIsResource = false;

    result = PalmMalCreateDB(creator, flags, DEFAULT_CARD_NUM, 
                    dbconfig->dbname, type, &state->currentDbHandle);
    MALLibAGBufferReaderFree(MALLibRefNum, r);

    return result;
}

static void closeCurrentRecord(ClientState *state)
{
    if(state->rawRec.recId != 0) {
        PalmMalCloseRecord(&state->rawRec);
        bzero(&state->rawRec, sizeof(PalmMalCRawRecordInfo));
    }
}

