/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

#include <debug.h>

#ifdef DEBUG_ON
#include <mobilelinkRsc.h>
#include <unix_stdarg.h>
#include <unix_stdio.h>

#define DEFAULT_CARD_NUM 0
#define DEBUG_ALERT_ID DebugAlert
#define debugDBType 'bugs'
#define debugDBName "DebugLog"

static Boolean useLogFile = false;
static int loggingEnabled = 1;

static void doDebug(char *msg1, char *msg2, char *msg3);
static void writeToLogDatabase(char *msg1, char *msg2, char *msg3);
//static Boolean libcPrcExists();

void debugLogToDatabase(Boolean logToDb)
{
    useLogFile = logToDb; 
}

extern void debugSetLogging(Boolean shouldLog)
{
    if(shouldLog)
        loggingEnabled++;
    else
        loggingEnabled--;
    if(loggingEnabled < 0)
        loggingEnabled = 0;
}


void debugClearLogDatabase()
{
    DmOpenRef oldDb;
    int recordCount;
    LocalID dbId;
 
    oldDb = DmOpenDatabaseByTypeCreator(debugDBType, 'AvGo', 
                                        dmModeReadWrite);
    if (oldDb == NULL) {
        return;
    }

    recordCount = DmNumRecords(oldDb);
    while (recordCount-- > 0) {
        DmRemoveRecord(oldDb, recordCount);
    }

    DmCloseDatabase(oldDb);

    dbId = DmFindDatabase(DEFAULT_CARD_NUM, debugDBName);
    if (dbId == 0)
        return;

    DmDeleteDatabase(DEFAULT_CARD_NUM, dbId);
}

void debugAlert(char *fmt, ...)
{
    va_list args;
    char buf[128];
 
    MemSet(buf, 128, 0);
//    if(libcPrcExists()) {  
        va_start(args, fmt);
        vsprintf(buf,fmt,args);
        va_end(args);
        doDebug(buf, "", "");
//    } else {
//        doDebug("!libc.prc-", fmt, "");
//    }
}

void debugAlertString(char *msg)
{
    doDebug(msg, "", "");
}

void debugAlertStrings(char *msg1, char *msg2, char *msg3)
{
    doDebug(msg1, msg2, msg3);
}

void debugAlertInt(char *msg, int num1)
{
    char number1[20];
    StrIToA(number1, num1);
    doDebug(msg, number1, "");
}

///////////////////////////////////////////////////////////////////////////////
/// Internals
///////////////////////////////////////////////////////////////////////////////
static void doDebug(char *msg1, char *msg2, char *msg3)
{

    if(!loggingEnabled)
        return;

    if(msg1 == NULL || *msg1 == NULL)
        msg1 = " ";
    if(msg2 == NULL || *msg2 == NULL)
        msg2 = " ";
    if(msg3 == NULL || *msg3 == NULL)
        msg3 = " ";

    if(!useLogFile) {
        FrmCustomAlert(DEBUG_ALERT_ID, msg1, msg2, msg3);
        return;
    }

    writeToLogDatabase(msg1, msg2, msg3);

}

static void writeToLogDatabase(char *msg1, char *msg2, char *msg3)
{
    DmOpenRef openDb;
    LocalID dbId;
    Err err;
    VoidHand handle;
    void *ptr;
    int size;
    UInt index = -1;

    dbId = DmFindDatabase(DEFAULT_CARD_NUM, debugDBName);
    if (dbId == 0) {
        err = DmCreateDatabase(DEFAULT_CARD_NUM, debugDBName, 
                                'AvGo', 
                                debugDBType, false);
        if (err) {
            FrmCustomAlert(DEBUG_ALERT_ID, "couldn't write log ", msg1, msg2);
            return;
        }
    }
    openDb = DmOpenDatabaseByTypeCreator(debugDBType, 'AvGo', dmModeReadWrite);

    size = StrLen(msg1) + StrLen(msg2) + StrLen(msg3) + 1;

    handle = DmNewRecord(openDb, &index, size); 
    if (handle == NULL) {
        FrmCustomAlert(DEBUG_ALERT_ID, "couldn't write log ", msg1, msg2);
        return;
    }

    ptr = MemHandleLock(handle);
    if (ptr == NULL) {
        //PENDING(klobad) if the lock fails should we try and whack the record?
        DmDeleteRecord(openDb, index); 
        FrmCustomAlert(DEBUG_ALERT_ID, "couldn't write log ", msg1, msg2);
        return;
    }

    DmSet(ptr, 0, size, 0);
    DmWrite(ptr, 0, msg1, StrLen(msg1));
    DmWrite(ptr, StrLen(msg1), msg2, StrLen(msg2));
    DmWrite(ptr, StrLen(msg1) + StrLen(msg2), msg3, StrLen(msg3));
    MemHandleUnlock(handle);

    DmCloseDatabase(openDb);
}
/*
static Boolean libcPrcExists()
{
    DmOpenRef openDB;

    openDB = DmOpenDatabaseByTypeCreator('GLib', 'Libc', dmModeReadOnly);
    if(openDB == 0)
        return false;
    DmCloseDatabase(openDB);
    return true;
}
*/
#endif // DEBUG_ON
