/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * thunk.c
 *
 * created: djw@avantgo.com, March 18, 1999.
 *
 * Contributor(s):
 */
#include <Pilot.h>
#include <ErrorMgr.h>

#include "thunk.h"

typedef unsigned long register_type;

#if defined(__MWERKS__)

static register_type A4_GET(void) = 0x200C;   /* move.l a5,d0 */
static register_type A5_GET(void) = 0x200D;   /* move.l a5,d0 */
static void A4_SET(register_type x) = 0x2857; /* move.l (sp),a4 */
static void A5_SET(register_type x) = 0x2A57; /* move.l (sp),a5 */

#elif __GNUC__

register register_type reg_a4 asm("%a4");
register register_type reg_a5 asm("%a5");
#define A4_GET() reg_a4
#define A5_GET() reg_a5
#define A4_SET(x) reg_a4 = (register_type)(x)
#define A5_SET(x) reg_a5 = (register_type)(x)

#else

#error "no compiler defined, not sure how to do assembler"

#endif

#if defined(__MWERKS__)

/*
 * Uses A4 in general register allocation, so there's no need to restore A4,
 * but the library caller may be gcc built in which case, A4 must be saved
 * over the thunk call. So save both anyway.
 */
#define SAVE_A5
#define SAVE_A4

#elif __GNUC__

/*
 * Gcc cares about both A4 (globals) and A5 (system).
 */
#define SAVE_A5
#define SAVE_A4

#else

#error "no compiler defined, not sure how to do assembler"

#endif

typedef struct ThunkData
{
    register_type a4;
    register_type a5;
    void* func;
} ThunkData;

typedef struct ThunkGlobals {
    void* ra;
#ifdef SAVE_A5
    register_type a5;
#endif
#ifdef SAVE_A4
    register_type a4;
#endif
#if defined(SAVE_A4) && defined(SAVE_A5)
    void* _pad; /* so indexing doesnt do mul */
#endif
} ThunkGlobals;

static ThunkGlobals thunkStack[THUNK_MAX_RECURSE];
static unsigned     thunkStackIndex;

/* need to be global to avoid gcc warnings */
void* thunkPushGlobals(ThunkData* thunkData, void* ra);
void* thunkPopGlobals();

void*
thunkPushGlobals(ThunkData* thunkData, void* ra)
{
    ThunkGlobals* stack;

#ifdef SAVE_A4
    register_type a4 = A4_GET();
#endif
#ifdef SAVE_A5
    register_type a5 = A5_GET();
#endif

#ifdef SAVE_A4
    A4_SET(thunkData->a4);
#endif
#ifdef SAVE_A5
    A5_SET(thunkData->a5);
#endif

    ErrFatalDisplayIf((thunkStackIndex >= THUNK_MAX_RECURSE),
		      "Thunk stack overflow");

    stack = &thunkStack[thunkStackIndex++];

#ifdef SAVE_A4
    stack->a4 = a4;
#endif
#ifdef SAVE_A5
    stack->a5 = a5;
#endif
    stack->ra = ra;

    return thunkData->func;
}

void*
thunkPopGlobals()
{
    ThunkGlobals* stack;

    ErrFatalDisplayIf((thunkStackIndex == 0), "Thunk stack underflow");

    stack = &thunkStack[--thunkStackIndex];

    /*
     * Note: you have to look at the generated code for this to make sure
     * that a4,a5 are not used after they have been whacked.
     */
#ifdef SAVE_A4
    A4_SET(stack->a4);
#endif
#ifdef SAVE_A5
    A5_SET(stack->a5);
#endif

    return stack->ra;
}

#if defined(__MWERKS__)

/*
 * Stack looks like:
 *       argsn
 *       ...
 *       args1
 *       args0
 *       caller return address
 *       ThunkThunkRts
 * sp ->
 *
 * We need to:
 * save a4,a5, and the caller return address.
 * load thunked a4,a5
 * move the sp up above the caller address.
 * jsr the thunkedFunc.
 * while preserving a0,d0, 
 * restore saved a4,a5.
 * push the caller return address back onto the stack.
 * rts
 */
static void asm
thunkHandler()
{
    /*
     * Stack is now:
     *       caller return address
     *       ThunkThunkRts
     * sp ->
     *
     * all set for: void* thunkPushGlobals(thunkData,ra);
     */
    jsr thunkPushGlobals /* installs new a4,a5,puts thunkedFunc in a0 */
    add.w #8,sp
    jsr (a0)     
    subq.w #4,sp    /* make room for the return address */
    move.l a0,-(sp) /* make sure a0,d0 is preserved. */
    move.l d0,-(sp)
    jsr thunkPopGlobals  /* restores old a4,a5, puts return address in a0 */
    move.l a0,8(sp)
    move.l (sp)+,d0
    move.l (sp)+,a0
    rts
}

#elif __GNUC__

void thunkHandler();

asm("
    .globl thunkHandler
thunkHandler:
    jsr thunkPushGlobals(%pc)
    add.w #8,%sp
    jsr (%a0)     
    subq.w #4,%sp
    move.l %a0,-(%sp)
    move.l %d0,-(%sp)
    jsr thunkPopGlobals(%pc)
    move.l %a0,8(%sp)
    move.l (%sp)+,%d0
    move.l (%sp)+,%a0
    rts
");

#else

#error "no compiler defined, not sure how to do assembler"

#endif

void*
ThunkInitFromA4A5(void* pbuf, void* funcPtr, void* a4, void* a5)
{
    Word*  wbuf = (Word*)pbuf;
    ThunkData* thunkData;

    /*
     * Thunk structure is:
     * move.l #thunkHandler,a0 // 3 words
     * jsr (a0)                // 1 word
     * dc.l a4                 // 2 words each
     * dc.l a5
     * dc.l thunkedFunc
     */
    wbuf[0] = 0x207c; /* move.l #N */
    *(void**)&wbuf[1] = &thunkHandler;
    wbuf[3] = 0x4e90; /* jsr (a0) */
    thunkData = (ThunkData*)&wbuf[4];
    thunkData->a4 = (register_type)a4;
    thunkData->a5 = (register_type)a5;
    thunkData->func = funcPtr;
    
    return pbuf;
}

void*
ThunkInit(void* pbuf, void* funcPtr)
{
    return ThunkInitFromA4A5(pbuf, funcPtr, (void*)A4_GET(), (void*)A5_GET());
}

void*
ThunkMake(void* funcPtr)
{
    void* thunk;

#ifdef DEBUG_djw
    thunk = MemPtrNew(2 + sizeof(ThunkType));
    *(Word*)thunk = 0x4e48; /* trap #8 */
    ThunkInit((char*)thunk+2, funcPtr);
#else
    thunk = MemPtrNew(sizeof(ThunkType));
    ThunkInit((char*)thunk, funcPtr);
#endif

    return thunk;
}

