/*
 *  Copyright (C) 2006  MakeHuman Project
 *
 *  This program is free software; you  can  redistribute  it  and/or
 *  modify  it  under  the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either  version  2  of
 *  the License, or (at your option) any later version.
 *
 *  This  program  is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the  implied  warranty  of
 *  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foun-
 *  dation, Inc., 59 Temple Place, Suite 330, Boston,  MA  02111-1307
 *  USA
 *  
 *  File   : ConsoleListener.cpp
 *  Project: MakeHuman <info@makehuman.org>, http://www.makehuman.org/
 *  App    : makehuman
 *
 *  For individual developers look into the AUTHORS file.
 *   
 */

#include <animorph/ObjExporter.h>
#include <animorph/BodySettings.h>
#include <mhgui/Console.h>
#include <mhgui/CGUtilities.h>
#include <mhgui/Window.h>
#include <mhgui/Component.h>
#include <stdio.h>
#include <iostream>
#include <assert.h>
#include "util.h"
#include "ConsoleCommands.h"
#include "ConsoleListener.h"
#include "Global.h"

#if defined(__APPLE__) && defined(__MACH__)
  #include "FileTools.h" // Move this out when tested on Linux and WIN32 too!
  #include <GLUT/glut.h>
#else
  #ifdef USE_FREEGLUT     
    #include <GL/freeglut.h>
  #else
    #include <GL/glut.h>
  #endif
#endif

using namespace std;
using namespace mhgui;

ConsoleListener::ConsoleListener() 
: AbstractListener() 
{
}

ConsoleListener::~ConsoleListener() 
{
}

bool ConsoleListener::mouseOver (const Point& inMousePos, Component *source)
{
  return true;
}

bool ConsoleListener::mouseOut (const Point& inMousePos, Component *source)
{
  return true;
}

bool ConsoleListener::mouseDragged (const Point& inMousePos, Component *source)
{

  return true;
}

bool ConsoleListener::mousePressed (const Point& inMousePos, int button, Component *source)
{
  return true;
}

bool ConsoleListener::mouseReleased (const Point& inMousePos, int button, Component *source)
{
  return true;
}

bool ConsoleListener::keyType (unsigned char key, Component *source)
{
  Console *console = dynamic_cast<Console *>(source); // req. RTTI!
  assert(console); // Check if this is really a Console object?

  if(console->acceptUserInput()) 
  {                        
    switch (toupper(key)) 
    {
      case '\r':
        parseCommand(*console);
        break;
      case '\b':
        console->removeUserText();
        break;
      case '\t':
        console->addUserText(' ');
        console->addUserText(' ');
        console->addUserText(' ');        
        break;    
      case 27:
        console->close();
        break;       
      default:
        console->addUserText(key);
    }
  }
  else 
  {
    if(toupper(key) == 27)
    {
      console->close();                
    }     
  }
  return true;
}

void ConsoleListener::parseCommand(Console& console)
{
  int i;
  string cmd;
  string arg;
  const string& line = console.getUserText();
  
  if(!line.empty())
  {
    if(console.getStatus() == Console::PROMPT) 
    {
      i = line.find_first_of(' ');
      if(i == -1)
      {
        cmd = line;
        arg = "";   
      }
      else
      {
        cmd = line.substr(0, i);
        arg = line.substr(i + 1, line.length() - i); 
      }
      console.setCommand(cmd);
    }
    else if(console.getStatus() == Console::INPUT)
    {
      cmd = console.getCommand();  
      arg = line;      
    }
    
    if(cmd == kConsoleCommand_Load_Background)
    {
      if(arg.size() == 0)
      {
        console.setStatus(Console::INPUT);
        console.setInputMessage(kConsoleMessage_Load_Background);
      }           
      else
      {
        loadWindowBackground(console, arg);                   
      }
    }
    else if(cmd == kConsoleCommand_Load_Bodysettings)
    {
      if(arg.size() == 0)
      {
        console.setStatus(Console::INPUT);
        console.setInputMessage(kConsoleMessage_Load_Bodysettings);
        console.setUserText(getMyBodysettingsPath());
      }           
      else
      {
        loadBodySettings(console, arg);
      }           
    }
    else if(cmd == kConsoleCommand_Load_Poses)
    {
      if(arg.size() == 0)
      {
        console.setStatus(Console::INPUT);
        console.setInputMessage(kConsoleMessage_Load_Poses);
        console.setUserText(getMyPosesPath());
      }           
      else
      {
        loadPoses(console, arg);
      }           
    }
    else if(cmd == kConsoleCommand_Save_Bodysettings)
    {
      if(arg.size() == 0)
      {
        console.setStatus(Console::INPUT);
        console.setInputMessage(kConsoleMessage_Save_Bodysettings);
        console.setUserText(getMyBodysettingsPath());
      }           
      else
      {
        saveBodySettings(console, arg);
      }           
    }
    else if(cmd == kConsoleCommand_Save_Poses)
    {
      if(arg.size() == 0)
      {
        console.setStatus(Console::INPUT);
        console.setInputMessage(kConsoleMessage_Save_Poses);
        console.setUserText(getMyPosesPath());
      }           
      else
      {
        savePoses(console, arg);
      }           
    }
    else if(cmd == kConsoleCommand_Export_Object)
    {
      if(arg.size() == 0)
      {
        console.setStatus(Console::INPUT);
        console.setInputMessage(kConsoleMessage_Export_Object);
        console.setUserText(getMyObjPath());
      }           
      else
      {
        exportBodySettings(console, arg);
      }           
    }           
    else
    {     
      commandNotFound(console);              
    }  
  }
}

void ConsoleListener::commandNotFound(Console& console)
{
  console.setMessage(kConsoleMessage_Unknown_Command);
  console.setStatus(Console::MESSAGE);       
}

void ConsoleListener::loadWindowBackground(Console& console, const string& filename)
{
  Window& mainWindow(Window::instance());
  if(mainWindow.loadPNG (searchBackgroundFile(filename)))
  {
    mainWindow.show();  
    console.printMessage(kConsoleMessage_Load_Background_Success);
  }
  else
  {
    console.printMessage(kConsoleMessage_Load_Error);
  }
}

void ConsoleListener::loadBodySettings(Console& console, const string& filename)
{
  Global &global = Global::instance ();     
  Mesh *mesh = global.getMesh ();
  assert (mesh);

  BodySettings bodyset;
  bool state = bodyset.load (filename);  

  if (state)
  {
    mesh->doMorph (bodyset);
    console.printMessage(kConsoleMessage_Load_Bodysettings_Success);
  }
  else
  {
    console.printMessage(kConsoleMessage_Load_Error);
  }
}

void ConsoleListener::loadPoses(Console& console, const string& filename)
{
  Global &global = Global::instance ();     
  Mesh *mesh = global.getMesh ();
  assert (mesh);

  BodySettings poses;
  bool state = poses.load (filename);  

  if (state)
  {
    mesh->doPose (poses);
    console.printMessage(kConsoleMessage_Load_Poses_Success);
  }
  else
  {
    console.printMessage(kConsoleMessage_Load_Error);
  }
}

void ConsoleListener::saveBodySettings(Console& console, const string& filename)
{
  Global &global = Global::instance ();
  Mesh *mesh = global.getMesh ();
  assert (mesh);

  BodySettings bodyset = mesh->getBodySettings ();

#if defined(__APPLE__) && defined(__MACH__)

/* This piece of code has been tested on Mac OS X only!
 * On OS X the bodysettings (and all user specific settings as well) will be 
 * are supposed to be saved to the Users Document folder in a directory named 
 * makehuman. 
 *
 * Since this directory does not exists the first time so we'll need to create 
 * one.
 */

  // First create a directory at the given location if it does not exists
  
  string path(FileTools::getFilePath(filename));
  if (!FileTools::fileExists(path))
  {
    bool rc = FileTools::makeDirHier(path);
    assert(rc == true);
  }
#endif // MAC OS X specific code

  bool state = bodyset.save (filename);     

  if (state)
  {
    console.printMessage(kConsoleMessage_Save_Bodysettings_Success);
  }
  else
  {
    console.printMessage(kConsoleMessage_Save_Error);
  }
}

void ConsoleListener::savePoses(Console& console, const string& filename)
{
  Global &global = Global::instance ();
  Mesh *mesh = global.getMesh ();
  assert (mesh);

  BodySettings poses = mesh->getPoses ();

#if defined(__APPLE__) && defined(__MACH__)
/* This piece of code has been tested on Mac OS X only!
 * On OS X the poses (and all user specific settings as well) will be 
 * are supposed to be saved to the Users Document folder in a directory named 
 * makehuman. 
 *
 * Since this directory does not exists the first time so we'll need to create 
 * one.
 */

  // First create a directory at the given location if it does not exists
  
  string path(FileTools::getFilePath(filename));
  if (!FileTools::fileExists(path))
  {
    bool rc = FileTools::makeDirHier(path);
    assert(rc == true);
  }
#endif // MAC OS X specific code

  bool state = poses.save (filename);     

  if (state)
  {
    console.printMessage(kConsoleMessage_Save_Poses_Success);
  }
  else
  {
    console.printMessage(kConsoleMessage_Save_Error);
  }
}

void ConsoleListener::exportBodySettings(Console& console, const string& filename)
{
  Global &global = Global::instance ();
  Mesh *mesh = global.getMesh ();
  assert (mesh);

  ObjExporter obj_export (*mesh);

#if defined(__APPLE__) && defined(__MACH__)
/* This piece of code has been tested on Mac OS X only!
 * On OS X the body settings (and all user specific settings as well) will be 
 * are supposed to be saved to the Users Document folder in a directory named 
 * makehuman. 
 *
 * Since this directory does not exists the first time so we'll need to create 
 * one.
 */

  // First create a directory at the given location if it does not exists
  
  string path(FileTools::getFilePath(filename));
  if (!FileTools::fileExists(path))
  {
    bool rc = FileTools::makeDirHier(path);
    assert(rc == true);
  }
#endif // MAC OS X specific code

  bool state = obj_export.exportFile (filename);
  
  if (state)
  {
    console.printMessage(kConsoleMessage_Export_Object_Success);
  }
  else
  {
    console.printMessage(kConsoleMessage_Save_Error);
  }
}


