/*
    LZMAinfo - Show information about LZMA files

    Copyright (C) 2005 Lasse Collin <lasse.collin@tukaani.org>

    This software is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.
*/

#include <stdio.h>
#include <string.h>
#include "LzmaUtilsCommon.h"

UInt32 Log2 (UInt32 n)
{
	UInt32 e;
	for (e = 0; n > 1; e++, n /= 2);
	return (e);
}

int main (int argc, char **argv)
{
	unsigned char headerBytes[LZMA_HEADER_SIZE];
	CLzmaProperties properties;
	FILE *file;
	int i;
	int isStreamed = 1; /* Assume streamed LZMA file */
	UInt64 uncompressedSize = 0;

	if (argc > 2) {
		printf ("Too many arguments, try 'lzmainfo --help'.\n");
		return (1);
	}

	if (argc > 1 && strcmp ("--help", argv[1]) == 0) {
		printf ("\nLZMAinfo - Show information stored "
			"in the LZMA file header\n\n"
			"Usage: lzmainfo [ --help | filename.lzma ]\n"
			"If no filename is specified lzmainfo reads stdin.\n"
			"The information is always printed to stdout.\n\n"
			LZMA_SDK_COPYRIGHT "\n"
			LZMA_UTILS_COPYRIGHT "\nLZMAinfo is free software "
			"licensed under the GNU LGPL.\n\n");
		return (0);
	}

	if (argc > 1) {
		file = fopen (argv[1], "r");
		if (file == NULL) {
			printf ("Error opening file '%s'.\nTry `%s "
					"--help' for more information.\n",
					argv[1], argv[0]);
			return (1);
		}
	} else {
		file = stdin;
	}

	/* Read the LZMA header */
	if (fread (headerBytes, sizeof (headerBytes[0]), LZMA_HEADER_SIZE, file)
			!= LZMA_HEADER_SIZE)
		return (1); /* Not enough input */

	/* Decode properties (lc, lp, pb) */
	if (LzmaDecodeProperties (&properties, headerBytes,
			LZMA_PROPERTIES_SIZE) != LZMA_RESULT_OK)
		return (1); /* Invalid properties data */

	/* Calculate the decompressed size */
	for (i = 0; i < 8; i++) {
		uncompressedSize += (UInt64)(headerBytes[
				i + LZMA_PROPERTIES_SIZE]) << (i * 8);
		/* Check if it is not streamed */
		if (headerBytes[i + LZMA_PROPERTIES_SIZE] != 0xFF)
			isStreamed = 0;
	}

	printf ("Uncompressed size:             ");
	if (isStreamed)
		printf ("Unknown");
	else
		printf ("%llu MB (%llu bytes)",
				(uncompressedSize + 512 * 1024)
				/ (1024 * 1024), uncompressedSize);
	printf ("\nDictionary size:               %u MB (2^%u bytes)\n"
		"Literal context bits (lc):     %d\n"
		"Literal pos bits (lp):         %d\n"
		"Number of pos bits (pb):       %d\n",
		(properties.DictionarySize + 512 * 1024) / (1024 * 1024),
		Log2 (properties.DictionarySize),
		properties.lc,
		properties.lp,
		properties.pb);

	if (file != stdin)
		fclose (file);

	return (0);
}
