/* This file is part of
 * ====================================================== 
 * 
 *           LyX, The Document Processor 	 
 *	     Copyright 1995 Matthias Ettrich
 *           Copyright 1995-2000 The LyX Team.
 *
 *           This file is Copyright 1996-2000
 *           Lars Gullik Bjnnes
 *
 * ====================================================== 
 */

#include <config.h>

#ifdef __GNUG__
#pragma implementation
#endif
#include <fstream>

#include "support/filetools.h"
#include "LaTeX.h"
#include "support/FileInfo.h"
#include "debug.h"
#include "support/lyxlib.h"
#include "support/syscall.h"
#include "support/syscontr.h"
#include "support/path.h"
#include "support/LRegex.h"
#include "support/LSubstring.h"
#include "bufferlist.h"
#include "minibuffer.h"
#include "gettext.h"
#include "lyx_gui_misc.h"

using std::ifstream;
using std::getline;
using std::endl;

// TODO: in no particular order
// - get rid of the extern BufferList and the call to
//   BufferList::updateIncludedTeXfiles, this should either
//   be done before calling LaTeX::funcs or in a completely
//   different way.
// - the bibtex command options should be supported.
// - the makeindex style files should be taken care of with
//   the dependency mechanism.
// - makeindex commandline options should be supported
// - somewhere support viewing of bibtex and makeindex log files.
// - we should perhaps also scan the bibtex log file
// - we should perhaps also scan the bibtex log file

extern BufferList bufferlist;

/*
 * CLASS TEXERRORS
 */

void TeXErrors::insertError(int line, string const & error_desc,
			    string const & error_text)
{
        Error newerr(line, error_desc, error_text);
	errors.push_back(newerr);
}

/*
 * CLASS LaTeX
 */

LaTeX::LaTeX(string const & latex, string const & f, string const & p)
		: cmd(latex), file(f), path(p)
{
	num_errors = 0;
	depfile = file + ".dep";
	if (prefixIs(cmd, "pdf")) // Do we use pdflatex ?
		depfile += "-pdf";
}


void LaTeX::deleteFilesOnError() const
{
	// currently just a dummy function.

	// What files do we have to delete?

	// This will at least make latex do all the runs
	lyx::unlink(depfile);

	// but the reason for the error might be in a generated file...

	string ofname = OnlyFilename(file);

	// bibtex file
	string bbl = ChangeExtension(ofname, ".bbl");
	lyx::unlink(bbl);

	// makeindex file
	string ind = ChangeExtension(ofname, ".ind");
	lyx::unlink(ind);
	
	// Also remove the aux file
	string aux = ChangeExtension(ofname, ".aux");
	lyx::unlink(aux);
}


int LaTeX::run(TeXErrors & terr, MiniBuffer * minib)
	// We know that this function will only be run if the lyx buffer
	// has been changed. We also know that a newly written .tex file
	// is always different from the previous one because of the date
	// in it. However it seems safe to run latex (at least) on time each
	// time the .tex file changes.
{
	int scanres = LaTeX::NO_ERRORS;
	unsigned int count = 0; // number of times run
	num_errors = 0; // just to make sure.
	const unsigned int MAX_RUN = 6;
	DepTable head; // empty head
	bool rerun = false; // rerun requested
	
	// The class LaTeX does not know the temp path.
	bufferlist.updateIncludedTeXfiles(lyx::getcwd()); //GetCWD());
	
	// Never write the depfile if an error was encountered.
	
	// 0
	// first check if the file dependencies exist:
	//     ->If it does exist
	//             check if any of the files mentioned in it have
	//             changed (done using a checksum).
	//                 -> if changed:
	//                        run latex once and
	//                        remake the dependency file
	//                 -> if not changed:
	//                        just return there is nothing to do for us.
	//     ->if it doesn't exist
	//             make it and
	//             run latex once (we need to run latex once anyway) and
	//             remake the dependency file.
	//
	FileInfo fi(depfile);
	bool run_bibtex = false;
	if (fi.exist()) {
		// Read the dep file:
		head.read(depfile);
		// Update the checksums
		head.update();
		
		lyxerr[Debug::DEPEND] << "Dependency file exists" << endl;
		if (head.sumchange()) {
			++count;
			lyxerr[Debug::DEPEND]
				<< "Dependency file has changed" << endl;
			lyxerr[Debug::LATEX]
				<< "Run #" << count << endl; 
			WriteStatus(minib,
				    string(_("LaTeX run number ")) + tostr(count));
			this->operator()();
			scanres = scanLogFile(terr);
			if (scanres & LaTeX::ERRORS) {
				deleteFilesOnError();
				return scanres; // return on error
			}
			
			run_bibtex = scanAux(head);
			if (run_bibtex)
				lyxerr[Debug::DEPEND]
					<< "Bibtex demands rerun" << endl;
		} else {
			lyxerr[Debug::DEPEND] << "return no_change" << endl;
			return LaTeX::NO_CHANGE;
		}
	} else {
		++count;
		lyxerr[Debug::DEPEND]
			<< "Dependency file does not exist" << endl;
		
		lyxerr[Debug::LATEX]
			<< "Run #" << count << endl;
		head.insert(file, true);
		WriteStatus(minib,
			    string(_("LaTeX run number ")) + tostr(count));
		this->operator()();
		scanres = scanLogFile(terr);
		if (scanres & LaTeX::ERRORS) {
			deleteFilesOnError();
			return scanres; // return on error
		}
		
	}

	// update the dependencies.
	deplog(head); // reads the latex log
	head.update();

	// 0.5
	// At this point we must run external programs if needed.
	// makeindex will be run if a .idx file changed or was generated.
	// And if there were undefined citations or changes in references
	// the .aux file is checked for signs of bibtex. Bibtex is then run
	// if needed.
	
	// run makeindex
	if (head.haschanged(OnlyFilename(ChangeExtension(file, ".idx")))) {
		// no checks for now
		lyxerr[Debug::LATEX] << "Running MakeIndex." << endl;
		WriteStatus(minib, _("Running MakeIndex."));
		rerun = runMakeIndex(OnlyFilename(ChangeExtension(file, ".idx")));
	}

	// run bibtex
	if (scanres & LaTeX::UNDEF_CIT
	    || scanres & LaTeX::RERUN
	    || run_bibtex) {
		// Here we must scan the .aux file and look for
		// "\bibdata" and/or "\bibstyle". If one of those
		// tags is found -> run bibtex and set rerun = true;
		// no checks for now
		lyxerr[Debug::LATEX] << "Running BibTeX." << endl;
		WriteStatus(minib, _("Running BibTeX."));
		rerun = runBibTeX(OnlyFilename(ChangeExtension(file, ".aux")), 
				  head);
	}
	
	// 1
	// we know on this point that latex has been run once (or we just
	// returned) and the question now is to decide if we need to run
	// it any more. This is done by asking if any of the files in the
	// dependency file has changed. (remember that the checksum for
	// a given file is reported to have changed if it just was created)
	//     -> if changed or rerun == true:
	//             run latex once more and
	//             update the dependency structure
	//     -> if not changed:
	//             we does nothing at this point
	//
	if (rerun || head.sumchange()) {
		rerun = false;
		++count;
		lyxerr[Debug::DEPEND]
			<< "Dep. file has changed or rerun requested" << endl;
		lyxerr[Debug::LATEX]
			<< "Run #" << count << endl;
		WriteStatus(minib,
			    string(_("LaTeX run number ")) + tostr(count));
		this->operator()();
		scanres = scanLogFile(terr);
		if (scanres & LaTeX::ERRORS) {
			deleteFilesOnError();
			return scanres; // return on error
		}
		
		// update the depedencies
		deplog(head); // reads the latex log
		head.update();
	} else {
		lyxerr[Debug::DEPEND] << "Dep. file has NOT changed" << endl;
	}

	// 1.5
	// The inclusion of files generated by external programs like
	// makeindex or bibtex might have done changes to pagenumbereing,
	// etc. And because of this we must run the external programs
	// again to make sure everything is redone correctly.
	// Also there should be no need to run the external programs any
	// more after this.
	
	// run makeindex if the <file>.idx has changed or was generated.
	if (head.haschanged(OnlyFilename(ChangeExtension(file, ".idx")))) {
		// no checks for now
		lyxerr[Debug::LATEX] << "Running MakeIndex." << endl;
		WriteStatus(minib, _("Running MakeIndex."));
		rerun = runMakeIndex(OnlyFilename(ChangeExtension(file, ".idx")));
	}
	
	// 2
	// we will only run latex more if the log file asks for it.
	// or if the sumchange() is true.
	//     -> rerun asked for:
	//             run latex and
	//             remake the dependency file
	//             goto 2 or return if max runs are reached.
	//     -> rerun not asked for:
	//             just return (fall out of bottom of func)
	//
	while ((head.sumchange() || rerun || (scanres & LaTeX::RERUN)) 
	       && count < MAX_RUN) {
		// Yes rerun until message goes away, or until
		// MAX_RUNS are reached.
		rerun = false;
		++count;
		lyxerr[Debug::LATEX] << "Run #" << count << endl;
		WriteStatus(minib, string(_("LaTeX run number ")) + tostr(count));
		this->operator()();
		scanres = scanLogFile(terr);
		if (scanres & LaTeX::ERRORS) {
			deleteFilesOnError();
			return scanres; // return on error
		}
		
		// keep this updated
		head.update();
	}

	// Write the dependencies to file.
	head.write(depfile);
	lyxerr[Debug::LATEX] << "Done." << endl;
	return scanres;
}


int LaTeX::operator()()
{
#ifndef __EMX__
	string tmp = cmd + ' ' + QuoteName(file) + " > /dev/null";
#else // cmd.exe (OS/2) causes SYS0003 error at "/dev/null"
	string tmp = cmd + ' ' + file + " > nul";
#endif
        Systemcalls one;
	return one.startscript(Systemcalls::System, tmp);
}


bool LaTeX::runMakeIndex(string const & f)
{
	lyxerr[Debug::LATEX] << "idx file has been made,"
		" running makeindex on file "
			     <<  f << endl;

	// It should be possible to set the switches for makeindex
	// sorting style and such. It would also be very convenient
	// to be able to make style files from within LyX. This has
	// to come for a later time. (0.13 perhaps?)
	string tmp = "makeindex -c -q ";
	tmp += f;
	Systemcalls one;
	one.startscript(Systemcalls::System, tmp);
	return true;
}


bool LaTeX::scanAux(DepTable & dep)
{
	// if any of the bib file has changed we don't have to
	// check the .aux file.
	if (dep.extchanged(".bib")
	    || dep.extchanged(".bst")) return true;
	
	string aux = OnlyFilename(ChangeExtension(file, ".aux"));
	ifstream ifs(aux.c_str());
	string token;
	LRegex reg1("\\\\bibdata\\{([^}]+)\\}");
	LRegex reg2("\\\\bibstyle\\{([^}]+)\\}");
	while (getline(ifs, token)) {
		if (reg1.exact_match(token)) {
			LRegex::SubMatches sub = reg1.exec(token);
			string data = LSubstring(token, sub[1].first,
						 sub[1].second);
			string::size_type b;
			do {
				b = data.find_first_of(',', 0);
				string l;
				if (b == string::npos)
					l = data;
				else {
					l = data.substr( 0, b - 0);
					data.erase(0, b + 1);
				}
				string full_l =
					findtexfile(
						ChangeExtension(l, "bib"), "bib");
				if (!full_l.empty()) {
					if (!dep.exist(full_l))
						return true;
				}
			} while (b != string::npos);
		} else if (reg2.exact_match(token)) {
			LRegex::SubMatches sub = reg2.exec(token);
			string style = LSubstring(token, sub[1].first,
						  sub[1].second);
			// token is now the style file
			// pass it to the helper
			string full_l =
				findtexfile(
					ChangeExtension(style, "bst"),
					"bst");
			if (!full_l.empty()) {
				if (!dep.exist(full_l))
					return true;
			}
		}
	}
	return false;
}


bool LaTeX::runBibTeX(string const & f, DepTable & dep)
{
	// Since a run of Bibtex mandates more latex runs it is ok to
	// remove all ".bib" and ".bst" files, it is also required to
	// discover style and database changes.
	dep.remove_files_with_extension(".bib");
	dep.remove_files_with_extension(".bst");
	ifstream ifs(f.c_str());
	string token;
	bool using_bibtex = false;
	LRegex reg1("\\\\bibdata\\{([^}]+)\\}");
	LRegex reg2("\\\\bibstyle\\{([^}]+)\\}");
	while (getline(ifs, token)) {
		if (reg1.exact_match(token)) {
			using_bibtex = true;
			LRegex::SubMatches const & sub = reg1.exec(token);
			string data = LSubstring(token, sub[1].first,
						 sub[1].second);
			// data is now all the bib files separated by ','
			// get them one by one and pass them to the helper
			string::size_type b;
			do {
				b = data.find_first_of(',', 0);
				string l;
				if (b == string::npos)
					l = data;
				else {
					l = data.substr(0, b - 0);
					data.erase(0, b + 1);
				}
				string full_l = 
					findtexfile(
						ChangeExtension(l, "bib"),
						"bib");
				lyxerr[Debug::LATEX] << "Bibtex database: `"
						     << full_l << "'" << endl;
				if (!full_l.empty()) {
					// add full_l to the dep file.
					dep.insert(full_l, true);
				}
			} while (b != string::npos);
		} else if (reg2.exact_match(token)) {
			using_bibtex = true;
			LRegex::SubMatches const & sub = reg2.exec(token);
			string style = LSubstring(token, sub[1].first,
						  sub[1].second);
			// token is now the style file
			// pass it to the helper
			string full_l = 
				findtexfile(
					ChangeExtension(style, "bst"),
					"bst");
			lyxerr[Debug::LATEX] << "Bibtex style: `"
					     << full_l << "'" << endl;
			if (!full_l.empty()) {
				// add full_l to the dep file.
				dep.insert(full_l, true);
			}
		}
	}
	if (using_bibtex) {
		// run bibtex and
		string tmp = "bibtex ";
		tmp += OnlyFilename(ChangeExtension(file, string()));
		Systemcalls one;
		one.startscript(Systemcalls::System, tmp);
		return true;
	}
	// bibtex was not run.
	return false;
}


int LaTeX::scanLogFile(TeXErrors & terr)
{
	int last_line = -1;
	int line_count = 1;
	int retval = NO_ERRORS;
	string tmp = OnlyFilename(ChangeExtension(file, ".log"));
	lyxerr[Debug::LATEX] << "Log file: " << tmp << endl;
	ifstream ifs(tmp.c_str());

	string token;
	while (getline(ifs, token)) {
		lyxerr[Debug::LATEX] << "Log line: " << token << endl;
		
		if (token.empty())
			continue;

		if (prefixIs(token, "LaTeX Warning:")) {
			// Here shall we handle different
			// types of warnings
			retval |= LATEX_WARNING;
			lyxerr[Debug::LATEX] << "LaTeX Warning." << endl;
			if (contains(token, "Rerun to get cross-references")) {
				retval |= RERUN;
				lyxerr[Debug::LATEX]
					<< "We should rerun." << endl;
			} else if (contains(token, "Citation")
				   && contains(token, "on page")
				   && contains(token, "undefined")) {
				retval |= UNDEF_CIT;
			}
		} else if (prefixIs(token, "Package")) {
			// Package warnings
			retval |= PACKAGE_WARNING;
			if (contains(token, "natbib Warning:")) {
				// Natbib warnings
				if (contains(token, "Citation")
				    && contains(token, "on page")
				    && contains(token, "undefined")) {
					retval |= UNDEF_CIT;
				}
			} else if (contains(token, "Rerun LaTeX.")) {
				// at least longtable.sty might use this.
				retval |= RERUN;
			}
		} else if (prefixIs(token, "! ")) {
			// Ok, we have something that looks like a TeX Error
			// but what do we really have.

			// Just get the error description:
			string desc(token, 2);
			if (contains(token, "LaTeX Error:"))
				retval |= LATEX_ERROR;
			// get the next line
			string tmp;
			int count = 0;
			do {
				if (!getline(ifs, tmp))
					break;
				if (++count > 10)
					break;
			} while (!prefixIs(tmp, "l."));
			if (prefixIs(tmp, "l.")) {
				// we have a latex error
				retval |=  TEX_ERROR;
				// get the line number:
				int line = 0;
				sscanf(tmp.c_str(), "l.%d", &line);
				// get the rest of the message:
				string errstr(tmp, tmp.find(' '));
				errstr += '\n';
				getline(ifs, tmp);
				while (!contains(errstr, "l.")
				       && !tmp.empty()
				       && !prefixIs(tmp, "! ")
				       && !contains(tmp, "(job aborted")) {
					errstr += tmp;
					errstr += "\n";
					getline(ifs, tmp);
				}
				lyxerr[Debug::LATEX]
					<< "line: " << line << '\n'
					<< "Desc: " << desc << '\n'
					<< "Text: " << errstr << endl;
				if (line == last_line)
					++line_count;
				else {
					line_count = 1;
					last_line = line;
				}
				if (line_count <= 5) {
					terr.insertError(line, desc, errstr);
					++num_errors;
				}
			}
		} else {
			// information messages, TeX warnings and other
			// warnings we have not caught earlier.
			if (prefixIs(token, "Overfull ")) {
				retval |= TEX_WARNING;
			} else if (prefixIs(token, "Underfull ")) {
				retval |= TEX_WARNING;
			} else if (contains(token, "Rerun to get citations")) {
				// Natbib seems to use this.
				retval |= RERUN;
			} else if (contains(token, "No pages of output")) {
				// A dvi file was not created
				retval |= NO_OUTPUT;
			} else if (contains(token, "That makes 100 errors")) {
				// More than 100 errors were reprted
				retval |= TOO_MANY_ERRORS;
			}
		}
	}
	lyxerr[Debug::LATEX] << "Log line: " << token << endl;
	return retval;
}


void LaTeX::deplog(DepTable & head)
{
	// This function reads the LaTeX log file end extracts all the external
	// files used by the LaTeX run. The files are then entered into the
	// dependency file.

	string logfile = OnlyFilename(ChangeExtension(file, ".log"));

	LRegex reg1(")* *\\(([^ \\)]+).*");
	LRegex reg2("File: ([^ ]+).*");
	LRegex reg3("No file ([^ ]+)\\..*");
	LRegex reg4("\\\\openout[0-9]+.*=.*`([^ ]+)'\\..*");
	LRegex unwanted("^.*\\.(aux|log|dvi|bbl|ind|glo)$");
	
	ifstream ifs(logfile.c_str());
	while (ifs) {
		// Ok, the scanning of files here is not sufficient.
		// Sometimes files are named by "File:xxx" only
		// So I think we should use some regexps to find files instead.
		// "(\([^ ]+\)"   should match the "(file " variant
		// "File: \([^ ]+\)" should match the "File: file" variant
		string foundfile;
		string token;
		getline(ifs, token);
		if (token.empty()) continue;
		
		if (reg1.exact_match(token)) {
			LRegex::SubMatches const & sub = reg1.exec(token);
			foundfile = LSubstring(token, sub[1].first,
					       sub[1].second);
		} else if (reg2.exact_match(token)) {
			LRegex::SubMatches const & sub = reg2.exec(token);
			foundfile = LSubstring(token, sub[1].first,
					       sub[1].second);
		} else if (reg3.exact_match(token)) {
			LRegex::SubMatches const & sub = reg3.exec(token);
			foundfile = LSubstring(token, sub[1].first,
					       sub[1].second);
		} else if (reg4.exact_match(token)) {
			LRegex::SubMatches const & sub = reg4.exec(token);
			foundfile = LSubstring(token, sub[1].first,
					       sub[1].second);
		} else {
			continue;
		}

		lyxerr[Debug::DEPEND] << "Found file: " 
				      << foundfile << endl;
		
		// Ok now we found a file.
		// Now we should make sure that this is a file that we can
		// access through the normal paths.
		// We will not try any fancy search methods to
		// find the file.
		
		// (1) foundfile is an
		//     absolute path and should
		//     be inserted.
		if (AbsolutePath(foundfile)) {
			lyxerr[Debug::DEPEND] << "AbsolutePath file: " 
					      << foundfile << endl;
			// On initial insert we want to do the update at once
			// since this file can not be a file generated by
			// the latex run.
			head.insert(foundfile, true);
			continue;
		}

		// (2) foundfile is in the tmpdir
		//     insert it into head
		if (FileInfo(OnlyFilename(foundfile)).exist()) {
			if (unwanted.exact_match(foundfile)) {
				lyxerr[Debug::DEPEND]
					<< "We don't want "
					<< OnlyFilename(foundfile)
					<< " in the dep file"
					<< endl;
			} else if (suffixIs(foundfile, ".tex")) {
				// This is a tex file generated by LyX
				// and latex is not likely to change this
				// during its runs.
				lyxerr[Debug::DEPEND]
					<< "Tmpdir TeX file: "
					<< OnlyFilename(foundfile)
					<< endl;
				head.insert(foundfile, true);
			} else {
				lyxerr[Debug::DEPEND]
					<< "In tmpdir file:"
					<< OnlyFilename(foundfile)
					<< endl;
				head.insert(OnlyFilename(foundfile));
			}
			continue;
		}
		lyxerr[Debug::DEPEND]
			<< "Not a file or we are unable to find it."
			<< endl;
	}
}
