-- See Copyright Notice in the file LICENSE

-- This file should contain only test sets that behave identically
-- when being run with pcre or posix regex libraries.

local luatest = require "luatest"
local N = luatest.NT

local function norm(a) return a==nil and N or a end

local function get_gsub (lib)
  return lib.gsub or
    function (subj, pattern, repl, n)
      return lib.new (pattern) : gsub (subj, repl, n)
    end
end

local function set_f_gmatch (lib, flg)
  -- gmatch (s, p, [cf], [ef])
  local function test_gmatch (subj, patt)
    local out, guard = {}, 10
    for a, b in lib.gmatch (subj, patt) do
      table.insert (out, { norm(a), norm(b) })
      guard = guard - 1
      if guard == 0 then break end
    end
    return unpack (out)
  end
  return {
    Name = "Function gmatch",
    Func = test_gmatch,
  --{  subj             patt         results }
    { {("abcd"):rep(3), "(.)b.(d)"}, {{"a","d"},{"a","d"},{"a","d"}} },
    { {"abcd",          ".*" },      {{"abcd",N},{"",N}  } },--zero-length match
  }
end

local function set_f_split (lib, flg)
  -- split (s, p, [cf], [ef])
  local function test_split (subj, patt)
    local out, guard = {}, 10
    for a, b, c in lib.split (subj, patt) do
      table.insert (out, { norm(a), norm(b), norm(c) })
      guard = guard - 1
      if guard == 0 then break end
    end
    return unpack (out)
  end
  return {
    Name = "Function split",
    Func = test_split,
  --{  subj             patt      results }
    { {"ab",            ","},     {{"ab",N,N},                           } },
    { {",",             ","},     {{"",",",N},                           } },
    { {",,",            ","},     {{"",",",N},     {"",",",N},           } },
    { {"a,b",           ","},     {{"a",",",N},    {"b",N,N},            } },
    { {",a,b",          ","},     {{"",",",N},     {"a",",",N}, {"b",N,N}} },
    { {"a,b,",          ","},     {{"a",",",N},    {"b",",",N},          } },
    { {"a,,b",          ","},     {{"a",",",N},    {"",",",N},  {"b",N,N}} },
    { {"ab<78>c", "<(.)(.)>"},    {{"ab","7","8"}, {"c",N,N},            } },
  }
end

local function set_f_find (lib, flg)
  return {
    Name = "Function find",
    Func = lib.find,
  --  {subj, patt, st},         { results }
    { {"abcd", ".+"},           { 1,4 }   },      -- [none]
    { {"abcd", ".+", 2},        { 2,4 }   },      -- positive st
    { {"abcd", ".+", -2},       { 3,4 }   },      -- negative st
    { {"abcd", ".*"},           { 1,4 }   },      -- [none]
    { {"abc",  "bc"},           { 2,3 }   },      -- [none]
    { {"abcd", "(.)b.(d)"},     { 1,4,"a","d" }}, -- [captures]
  }
end

local function set_f_match (lib, flg)
  return {
    Name = "Function match",
    Func = lib.match,
  --  {subj, patt, st},         { results }
    { {"abcd", ".+"},           {"abcd"}  }, -- [none]
    { {"abcd", ".+", 2},        {"bcd"}   }, -- positive st
    { {"abcd", ".+", -2},       {"cd"}    }, -- negative st
    { {"abcd", ".*"},           {"abcd"}  }, -- [none]
    { {"abc",  "bc"},           {"bc"}    }, -- [none]
    { {"abcd", "(.)b.(d)"},     {"a","d"} }, -- [captures]
  }
end

local function set_m_exec (lib, flg)
  return {
    Name = "Method exec",
    Method = "exec",
  --{patt},                 {subj, st}           { results }
    { {".+"},               {"abcd"},            {1,4,{}}  }, -- [none]
    { {".+"},               {"abcd",2},          {2,4,{}}  }, -- positive st
    { {".+"},               {"abcd",-2},         {3,4,{}}  }, -- negative st
    { {".*"},               {"abcd"},            {1,4,{}}  }, -- [none]
    { {"bc"},               {"abc"},             {2,3,{}}  }, -- [none]
    { { "(.)b.(d)"},        {"abcd"},            {1,4,{1,1,4,4}}},--[captures]
  }
end

local function set_m_tfind (lib, flg)
  return {
    Name = "Method tfind",
    Method = "tfind",
  --{patt},                 {subj, st}           { results }
    { {".+"},               {"abcd"},            {1,4,{}}  }, -- [none]
    { {".+"},               {"abcd",2},          {2,4,{}}  }, -- positive st
    { {".+"},               {"abcd",-2},         {3,4,{}}  }, -- negative st
    { {".*"},               {"abcd"},            {1,4,{}}  }, -- [none]
    { {"bc"},               {"abc"},             {2,3,{}}  }, -- [none]
    { { "(.)b.(d)"},        {"abcd"},            {1,4,{"a","d"}}},--[captures]
  }
end

local function set_f_plainfind (lib, flg)
  return {
    Name = "Function plainfind",
    Func = lib.plainfind,
  --{ subj,  patt,  st, ci }      { results }
    { {"abcd", "bc"},             {2,3} }, -- [none]
    { {"abcd", "dc"},             {N}   }, -- [none]
    { {"abcd", "cd"},             {3,4} }, -- positive st
    { {"abcd", "cd", 3},          {3,4} }, -- positive st
    { {"abcd", "cd", 4},          {N}   }, -- failing st
    { {"abcd", "bc", 2},          {2,3} }, -- positive st
    { {"abcd", "bc", -4},         {2,3} }, -- negative st
    { {"abcd", "bc", 3},          {N}   }, -- failing st
    { {"abcd", "BC"},             {N}   }, -- case sensitive
    { {"abcd", "BC", N, true},    {2,3} }, -- case insensitive
    { {"ab\0cd", "b\0c"},         {2,4} }, -- contains nul
  }
end

local function set_f_gsub1 (lib, flg)
  local subj, pat = "abcdef", "[abef]+"
  return {
    Name = "Function gsub, set1",
    Func = get_gsub (lib),
  --{ s,     p,   f,   n,    res1,   res2 },
    { {subj, pat, "",  0},   {subj,    0} }, -- test "n" + empty_replace
    { {subj, pat, "", -1},   {subj,    0} }, -- test "n" + empty_replace
    { {subj, pat, "",  1},   {"cdef",  1} },
    { {subj, pat, "",  2},   {"cd",    2} },
    { {subj, pat, "",  3},   {"cd",    2} },
    { {subj, pat, ""    },   {"cd",    2} },
    { {subj, pat, "#", 0},   {subj,    0} }, -- test "n" + non-empty_replace
    { {subj, pat, "#", 1},   {"#cdef", 1} },
    { {subj, pat, "#", 2},   {"#cd#",  2} },
    { {subj, pat, "#", 3},   {"#cd#",  2} },
    { {subj, pat, "#"   },   {"#cd#",  2} },
  }
end

local function set_f_gsub2 (lib, flg)
  local subj, pat = "abc", "([ac])"
  return {
    Name = "Function gsub, set2",
    Func = get_gsub (lib),
  --{ s,     p,   f,   n,     res1,     res2 },
    { {subj, pat, "<%1>" },   {"<a>b<c>", 2} }, -- test non-escaped chars in f
    { {subj, pat, "%<%1%>" }, {"<a>b<c>", 2} }, -- test escaped chars in f
    { {subj, pat, "" },       {"b",       2} }, -- test empty replace
    { {subj, pat, "1" },      {"1b1",     2} }, -- test odd and even %'s in f
    { {subj, pat, "%1" },     {"abc",     2} },
    { {subj, pat, "%%1" },    {"%1b%1",   2} },
    { {subj, pat, "%%%1" },   {"%ab%c",   2} },
    { {subj, pat, "%%%%1" },  {"%%1b%%1", 2} },
    { {subj, pat, "%%%%%1" }, {"%%ab%%c", 2} },
  }
end

local function set_f_gsub3 (lib, flg)
  return {
    Name = "Function gsub, set3",
    Func = get_gsub (lib),
  --{ s,      p,      f,  n,   res1,  res2 },
    { {"abc", "a",    "%0" }, {"abc", 1} }, -- test (in)valid capture index
    { {"abc", "a",    "%1" }, {"abc", 1} },
    { {"abc", "[ac]", "%1" }, {"abc", 2} },
    { {"abc", "(a)",  "%1" }, {"abc", 1} },
    { {"abc", "(a)",  "%2" }, "invalid capture index" },
  }
end

local function set_f_gsub4 (lib, flg)
  return {
    Name = "Function gsub, set4",
    Func = get_gsub (lib),
  --{ s,           p,              f, n,  res1,       res2 },
    { {"a2c3",     ".",            "#" }, {"####",      4} }, -- test .
    { {"a2c3",     ".+",           "#" }, {"#",         1} }, -- test .+
    { {"a2c3",     ".*",           "#" }, {"##",        2} }, -- test .*
    { {"/* */ */", "\\/\\*(.*)\\*\\/", "#" }, {"#",     1} },
    { {"a2c3",     "[0-9]",        "#" }, {"a#c#",      2} }, -- test %d
    { {"a2c3",     "[^0-9]",       "#" }, {"#2#3",      2} }, -- test %D
    { {"a \t\nb",  "[ \t\n]",      "#" }, {"a###b",     3} }, -- test %s
    { {"a \t\nb",  "[^ \t\n]",     "#" }, {"# \t\n#",   2} }, -- test %S
  }
end

local function set_f_gsub5 (lib, flg)
  local function frep1 () end                       -- returns nothing
  local function frep2 () return "#" end            -- ignores arguments
  local function frep3 (...) return table.concat({...}, ",") end -- "normal"
  local function frep4 () return {} end             -- invalid return type
  local function frep5 () return "7", "a" end       -- 2-nd return is "a"
  local function frep6 () return "7", "break" end   -- 2-nd return is "break"
  local subj = "a2c3"
  return {
    Name = "Function gsub, set5",
    Func = get_gsub (lib),
  --{ s,     p,          f,   n,   res1,      res2 },
    { {subj, "a(.)c(.)", frep1 }, {subj,        1} },
    { {subj, "a(.)c(.)", frep2 }, {"#",         1} },
    { {subj, "a(.)c(.)", frep3 }, {"2,3",       1} },
    { {subj, "a.c.",     frep3 }, {subj,        1} },
    { {subj, "z*",       frep1 }, {subj,        5} },
    { {subj, "z*",       frep2 }, {"#a#2#c#3#", 5} },
    { {subj, "z*",       frep3 }, {subj,        5} },
    { {subj, subj,       frep4 }, "invalid return type" },
    { {"abc",".",        frep5 }, {"777",       3} },
    { {"abc",".",        frep6 }, {"7bc",       1} },
  }
end

local function set_f_gsub6 (lib, flg)
  local tab1, tab2, tab3 = {}, { ["2"] = 56 }, { ["2"] = {} }
  local subj = "a2c3"
  return {
    Name = "Function gsub, set6",
    Func = get_gsub (lib),
  --{ s,     p,          f, n,   res1, res2 },
    { {subj, "a(.)c(.)", tab1 }, {subj,  1} },
    { {subj, "a(.)c(.)", tab2 }, {"56",  1} },
    { {subj, "a(.)c(.)", tab3 }, "invalid replacement type" },
    { {subj, "a.c.",     tab1 }, {subj,  1} },
    { {subj, "a.c.",     tab2 }, {subj,  1} },
    { {subj, "a.c.",     tab3 }, {subj,  1} },
  }
end

return function (libname)
  local lib = require (libname)
  return {
    set_f_gmatch    (lib),
    set_f_split     (lib),
    set_f_find      (lib),
    set_f_match     (lib),
    set_m_exec      (lib),
    set_m_tfind     (lib),
    set_f_gsub1     (lib),
    set_f_gsub2     (lib),
    set_f_gsub3     (lib),
    set_f_gsub4     (lib),
    set_f_gsub5     (lib),
    set_f_gsub6     (lib),
    set_f_plainfind (lib),
  }
end
