/* Copyright (C) 2000-2003 Markus Lausser (sgop@users.sf.net)
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/time.h>
#include <stdlib.h>
#include <sys/types.h>
#include <dirent.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "callbacks.h"
#include "connection.h"
#include "global.h"
#include "search.h"
#include "browse.h"
#include "hotlist.h"
#include "dialog.h"
#include "support.h"
#include "chat.h"
#include "handler.h"
#include "resume.h"
#include "whois.h"
#include "clist_rename.h"
#include "utils.h"

static int hot_description_func(clist_rename_t* cr) {
  int row;
  hot_t* hot;

  if (!cr || !cr->new_text || !cr->data) return 0;
  row = gtk_clist_find_row_from_data(cr->clist, cr->data);
  if (row < 0) return 0;
  
  hot = cr->data;
  if (hot->description) g_free(hot->description);
  hot->description = g_strdup(cr->new_text);
  gtk_clist_set_text(cr->clist, row, cr->col, hot->description);  
  
  return 1;
}

static void
on_enter_description_activate(GtkMenuItem * menuitem ATTR_UNUSED, 
			      gpointer user_data)
{
  hot_t* hot = user_data;
  GtkCList* clist;

  if (!hot) return;
  clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
  clist_rename(clist, 2, hot_description_func, 
	       hot, hot->description);
}

GtkWidget *create_hotlist_popup(hot_t* hot)
{
  GtkWidget *popup;
  GtkWidget *user_popup;
  GtkWidget *item;
  GtkAccelGroup *popup_accels;
  GtkWidget *refresh_users_files;
  GtkWidget *remove_user_from_hotlist;
  GtkWidget *trennlinie6;
  GtkWidget *trennlinie20;
  GtkWidget *customize_list6;

  popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));

  if (hot) {
    refresh_users_files =
      gtk_menu_item_new_with_label("Refresh Users Files");
    gtk_widget_show(refresh_users_files);
    gtk_container_add(GTK_CONTAINER(popup), refresh_users_files);
    gtk_signal_connect(GTK_OBJECT(refresh_users_files), "activate",
		       GTK_SIGNAL_FUNC(on_refresh_users_files_activate),
		       hot);

    remove_user_from_hotlist =
      gtk_menu_item_new_with_label("Remove User From Hotlist");
    gtk_widget_show(remove_user_from_hotlist);
    gtk_container_add(GTK_CONTAINER(popup), remove_user_from_hotlist);
    gtk_signal_connect(GTK_OBJECT(remove_user_from_hotlist), "activate",
		       GTK_SIGNAL_FUNC(on_remove_user_from_hotlist_activate), hot);

    trennlinie6 = gtk_menu_item_new();
    gtk_widget_show(trennlinie6);
    gtk_container_add(GTK_CONTAINER(popup), trennlinie6);
    gtk_widget_set_sensitive(trennlinie6, FALSE);

    remove_user_from_hotlist =
      gtk_menu_item_new_with_label("Enter description");
    gtk_widget_show(remove_user_from_hotlist);
    gtk_container_add(GTK_CONTAINER(popup), remove_user_from_hotlist);
    gtk_signal_connect(GTK_OBJECT(remove_user_from_hotlist), "activate",
		       GTK_SIGNAL_FUNC(on_enter_description_activate),
		       hot);

    trennlinie6 = gtk_menu_item_new();
    gtk_widget_show(trennlinie6);
    gtk_container_add(GTK_CONTAINER(popup), trennlinie6);
    gtk_widget_set_sensitive(trennlinie6, FALSE);

    item = gtk_menu_item_new_with_label("User Menu");
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);

    user_popup = create_user_popup(M_HOTLIST, hot);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), user_popup);

    trennlinie20 = gtk_menu_item_new();
    gtk_widget_show(trennlinie20);
    gtk_container_add(GTK_CONTAINER(popup), trennlinie20);
    gtk_widget_set_sensitive(trennlinie20, FALSE);
  }

  customize_list6 = gtk_menu_item_new_with_label("Customize List");
  gtk_widget_ref(customize_list6);
  gtk_object_set_data_full(GTK_OBJECT(popup), "customize_list6",
			   customize_list6,
			   (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(customize_list6);
  gtk_container_add(GTK_CONTAINER(popup), customize_list6);
  gtk_signal_connect(GTK_OBJECT(customize_list6), "activate",
		     GTK_SIGNAL_FUNC(on_customize_list_activate), NULL);

  return popup;
}

hot_t *hotlist_new(char *user, char *des, int visible) {
  hot_t *result;

  result = (hot_t *) g_malloc(sizeof(hot_t));
  result->user = g_strdup(user);
  if (des) result->description = g_strdup(des);
  else result->description = NULL;
  result->visible = visible;
  result->nets = NULL;
  global.hotlist = g_list_append(global.hotlist, result);
  if (visible && global.status.rc_read) write_rc();

  return result;
}

hot_t *hotlist_search_user(char *user) {
  GList *dlist;
  hot_t *hot;

  for (dlist = global.hotlist; dlist; dlist = dlist->next) {
    hot = dlist->data;
    if (!g_strcasecmp(hot->user, user)) return hot;
  }

  return NULL;
}

hot_t* hotlist_add(char *nick, char *des, int visible) {
  GtkWidget *temp;
  int row;
  hot_t *hot;
  char* text;

  hot = hotlist_search_user(nick);
  if (hot && (!visible || hot->visible)) {
    text = g_strdup_printf("User <%s> is already in hotlist",
			   nick);
    info_dialog(text);
    g_free(text);
    return NULL;
  }

  // !hot || (visible && !hot->visible)

  temp = lookup_widget(global.win, "clist16");
  if (!hot) {
    // !hot
    hot = hotlist_new(nick, des, visible);
    // sending entry to all servers
    command_send(NULL, CMD_ADD_HOTLIST, nick);
    if (visible) {
      strcpy(tstr[0], nick);
      strcpy(tstr[1], "");
      if (hot->description)
	strcpy(tstr[2], hot->description);
      else tstr[2][0] = 0;
      row = gtk_clist_append(GTK_CLIST(temp), list);
      gtk_clist_set_row_data(GTK_CLIST(temp), row, hot);
      gtk_clist_set_pixtext(GTK_CLIST(temp), row, 0,
			    tstr[0], 5,
			    global.pix.user2, global.pix.user2b);
    }
    hot->visible = visible;
  } else {
    // (visible && !hot->visible)
    strcpy(tstr[0], nick);
    if (hot->nets) strcpy(tstr[1], LineSpeed(hot->speed));
    else strcpy(tstr[1], "");
    if (hot->description) g_free(hot->description);
    if (des) hot->description = g_strdup(des);
    else hot->description = NULL;
    if (hot->description)
      strcpy(tstr[2], hot->description);
    else tstr[2][0] = 0;

    row = gtk_clist_append(GTK_CLIST(temp), list);
    gtk_clist_set_row_data(GTK_CLIST(temp), row, hot);

    if (hot->nets) {
      gtk_clist_set_pixtext(GTK_CLIST(temp), row, 0,
			    tstr[0], 5,
			    global.pix.user1, global.pix.user1b);
    } else {
      gtk_clist_set_pixtext(GTK_CLIST(temp), row, 0,
			    tstr[0], 5,
			    global.pix.user2, global.pix.user2b);
    }
    hot->visible = 1;
  }
  return hot;
}

void hotlist_remove(char *nick, int visible) {
  int row;
  GtkCList *clist;
  hot_t *hot;

  hot = hotlist_search_user(nick);
  if (!hot) {
    client_message("Message", "User <%s> is not in hotlist", nick);
    return;
  }

  if (visible) {
    clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
    row = gtk_clist_find_row_from_data(clist, hot);
    if (row >= 0) gtk_clist_remove(clist, row);
    hot->visible = 0;
    if (!chat_page_search(NULL, hot->user, P_PRIVATE, 3)) {
      // remove from list, if user has no private page.
      command_send(NULL, CMD_REMOVE_HOTLIST, hot->user);
      global.hotlist = g_list_remove(global.hotlist, hot);
      if (hot->user) g_free(hot->user);
      if (hot->nets) g_list_free(hot->nets);
      if (hot->description) g_free(hot->description);
      g_free(hot);
    }
  } else if (!hot->visible) {
    // only remove, if this is no visible hotlist entry
    command_send(NULL, CMD_REMOVE_HOTLIST, hot->user);
    global.hotlist = g_list_remove(global.hotlist, hot);
    if (hot->user) g_free(hot->user);
    if (hot->nets) g_list_free(hot->nets);
    if (hot->description) g_free(hot->description);
    g_free(hot);
  }
}

  /*
void hotlist_user_unknown(hot_t * hot)
{
  printf("hotlist_user_unknown not defined\n");
  GtkCList *clist;
  int row;
  chat_page_t *page;

  resume_user_offline(hot->user);
  hot->online = HOT_UNKNOWN;

  if (hot->visible) {
    clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
    row = gtk_clist_find_row_from_data(clist, hot);
    if (row >= 0) {
      gtk_clist_set_pixtext(clist, row, 0, hot->user, 5,
			    global.pix.dummy, global.pix.dummyb);
      gtk_clist_set_text(clist, row, 1, "");
      gtk_clist_set_text(clist, row, 2, "");
      gtk_clist_sort(clist);
    }
  }
  page = chat_page_search(NULL, hot->user, P_PRIVATE, 3);
  if (!page) return;
  chat_page_set_active(page, 0);
}
  */

void hotlist_user_online(net_t* net, hot_t * hot) {
  GtkCList *clist;
  int row;
  chat_page_t *page;
  char* prefix;
  
  if (!hot->nets) {
    if (hot->visible) {
      ext_handle(EVENT_USER_ONLINE, hot->user, hot->description);
      page = chat_page_get_printable();
      chat_print_time_stamp(page, M_PUBLIC);
      prefix = cparse(global.scheme->client_prefix);
      chat_print_colored(page, M_PUBLIC, "message", prefix);
      chat_print_text(page, M_PUBLIC, "user", "<");
      chat_print_nick(page, M_PUBLIC, hot->user, net);
      chat_print_text(page, M_PUBLIC, "user", "> ");
      chat_print_text(page, M_PUBLIC, "message", "is on ");
      chat_print_text(page, M_PUBLIC, "text", net->name);
      if (hot->description && hot->description[0]) {
	chat_print_text(page, M_PUBLIC, "message", " (");
	chat_print_text(page, M_PUBLIC, "message", hot->description);
	chat_print_text(page, M_PUBLIC, "message", ")");
      }
      chat_print_text(page, M_PUBLIC, "message", "\n");
      clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
      row = gtk_clist_find_row_from_data(clist, hot);
      if (row >= 0) {
	gtk_clist_set_pixtext(clist, row, 0, hot->user, 5,
			      global.pix.user1, global.pix.user1b);
	gtk_clist_set_text(clist, row, 1, LineSpeed(hot->speed));
	gtk_clist_sort(clist);
      }
    }
    page = chat_page_search2(hot->user, P_PRIVATE);
    if (page && !page->net) {
      chat_page_update_network(page, net);
      chat_page_set_active(page, 1);
    }
  }

  hot->nets = g_list_append(hot->nets, net);
}

void hotlist_user_offline(net_t* net, hot_t * hot)
{
  GtkCList *clist;
  int row;
  chat_page_t *page;
  int online;
  char* prefix;

  if (!g_list_find(hot->nets, net)) return;

  online = (hot->nets != NULL);
  hot->nets = g_list_remove(hot->nets, net);
  
  page = chat_page_search2(hot->user, P_PRIVATE);
  if (page && page->net == net) {
    if (hot->nets)
      chat_page_update_network(page, hot->nets->data);
    else {
      chat_page_update_network(page, NULL);
      chat_page_set_active(page, 0);
    }
  }
  
  if (hot->nets) return;

  if (hot->visible) {
    if (online) {
      ext_handle(EVENT_USER_OFFLINE, hot->user, hot->description);
      page = chat_page_get_printable();
      chat_print_time_stamp(page, M_PUBLIC);
      prefix = cparse(global.scheme->client_prefix);
      chat_print_colored(page, M_PUBLIC, "message", prefix);
      chat_print_text(page, M_PUBLIC, "user", "<");
      chat_print_nick(page, M_PUBLIC, hot->user, net);
      chat_print_text(page, M_PUBLIC, "user", "> ");
      chat_print_text(page, M_PUBLIC, "message", "is off\n");

      clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
      row = gtk_clist_find_row_from_data(clist, hot);
      if (row >= 0) {
	gtk_clist_set_pixtext(clist, row, 0, hot->user, 5,
			      global.pix.user2, global.pix.user2b);
	gtk_clist_set_text(clist, row, 1, "");
	gtk_clist_sort(clist);
      }
    }
  } else {
    // remove the user from hotlist, if there is no private page.
    if (!page)
      global.hotlist = g_list_remove(global.hotlist, hot);
  }
}
