/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <unistd.h>
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <dirent.h>
#include <sys/stat.h>
#include <time.h>
#include <sys/time.h>
#include <netdb.h>

#include "lopster.h"
#include "connection.h"
#include "global.h"
#include "search.h"
#include "transfer.h"
#include "resume.h"
#include "interface.h"
#include "support.h"
#include "callbacks.h"
#include "log.h"
#include "browse.h"
#include "share.h"
#include "hotlist.h"
#include "commands.h"
#include "chat.h"
#include "dirselect.h"
#include "scheme.h"
#include "handler.h"
#include "whois.h"
#include "exec.h"
#include "string_list.h"
#include "userinfo.h"

void whois_request(char *user, int flag)
{
  whois_t *whois;

  whois = whois_search(user);
  if (whois) {
    // whois already in progress, adding flag;
    whois->flag |= flag;
  } else {
    whois = whois_new(user, flag);
    send_command(CMD_CLIENT_WHOIS, whois->user);
  }
}

whois_t *whois_new(char *user, int flag)
{
  whois_t *result;

  if (!user)
    return NULL;

  result = (whois_t *) l_malloc(sizeof(whois_t));
  result->flag = flag;
  result->received = 0;
  result->online = 0;
  result->advanced = 0;
  result->yourself = 0;
  result->user = l_strdup(user);
  result->level = L_USER;
  result->online_time = 0;
  result->channels = NULL;
  result->status = NULL;
  result->shared = 0;
  result->downloads = 0;
  result->uploads = 0;
  result->link = 0;
  result->client = NULL;
  result->total_down = 0;
  result->total_up = 0;
  result->ip = NULL;
  result->server_port = 0;
  result->data_port = 0;
  result->email = NULL;
  result->server = NULL;

  global.whois_requests = g_list_append(global.whois_requests, result);

  return result;
}

void whois_destroy(whois_t * whois)
{
  if (whois->user)
    l_free(whois->user);
  if (whois->channels)
    l_free(whois->channels);
  if (whois->status)
    l_free(whois->status);
  if (whois->client)
    l_free(whois->client);
  if (whois->ip)
    l_free(whois->ip);
  if (whois->email)
    l_free(whois->email);
  if (whois->server)
    l_free(whois->server);
  l_free(whois);
}

void whois_remove(whois_t * whois)
{
  global.whois_requests = g_list_remove(global.whois_requests, whois);
  whois_destroy(whois);
}

void whois_action(whois_t * whois)
{
  if (!whois)
    return;
  if (!whois->received)
    return;

  if (whois->yourself)
    whois_action_yourself(whois);
  if (whois->flag & WHOIS_SHOW)
    whois_action_show(whois);
  if (whois->flag & WHOIS_TEXT)
    whois_action_text(whois);
  whois_action_online(whois);
  whois_action_userinfo(whois);
  whois_remove(whois);
}

whois_t *whois_search(char *user)
{
  GList *dlist;
  whois_t *whois;

  if (!user || !*user)
    return NULL;

  for (dlist = global.whois_requests; dlist; dlist = dlist->next) {
    whois = (whois_t *) (dlist->data);
    if (!l_strcasecmp(whois->user, user))
      return whois;
  }
  return NULL;
}

whois_t *whois_search_unreceived(char *user)
{
  GList *dlist;
  whois_t *whois;

  if (!user || !*user)
    return NULL;

  for (dlist = global.whois_requests; dlist; dlist = dlist->next) {
    whois = (whois_t *) (dlist->data);
    if (!l_strcasecmp(whois->user, user) && !whois->received)
      return whois;
  }
  return NULL;
}

void whois_eval_whois(char *data)
{
  whois_t *whois;
  char *pos1;

  pos1 = arg(data, 0);

  whois = whois_search_unreceived(pos1);

  if (!whois)
    return;

  l_free(whois->user);
  whois->user = l_strdup(pos1);

  if (!l_strcasecmp(whois->user, SERVER->nick))
    whois->yourself = 1;

  whois->online = 1;
  whois->received = global.current_time;

  pos1 = arg(NULL, 0);
  whois->level = level2int(pos1);

  pos1 = arg(NULL, 0);
  whois->online_time = strtoul(pos1, NULL, 10);

  pos1 = arg(NULL, 0);
  if (whois->channels)
    l_free(whois->channels);
  whois->channels = l_strdup(pos1);

  pos1 = arg(NULL, 0);
  if (whois->status)
    l_free(whois->status);
  whois->status = l_strdup(pos1);

  pos1 = arg(NULL, 0);
  whois->shared = atoi(pos1);

  pos1 = arg(NULL, 0);
  whois->downloads = atoi(pos1);

  pos1 = arg(NULL, 0);
  whois->uploads = atoi(pos1);

  pos1 = arg(NULL, 0);
  whois->link = atoi(pos1);

  pos1 = arg(NULL, 0);
  if (whois->client)
    l_free(whois->client);
  whois->client = l_strdup(pos1);

  pos1 = arg(NULL, 0);
  if (!pos1) {
    whois->advanced = 0;
  } else {
    whois->advanced = 1;
    whois->total_down = atoi(pos1);

    pos1 = arg(NULL, 0);
    whois->total_up = atoi(pos1);

    pos1 = arg(NULL, 0);
    if (whois->ip)
      l_free(whois->ip);
    whois->ip = l_strdup(pos1);

    pos1 = arg(NULL, 0);
    whois->server_port = atoi(pos1);

    pos1 = arg(NULL, 0);
    whois->data_port = atoi(pos1);

    pos1 = arg(NULL, 0);
    if (pos1) {
      if (whois->email) l_free(whois->email);
      whois->email = l_strdup(pos1);
    }

    pos1 = arg(NULL, 0);
    if (pos1) {
      if (whois->server) l_free(whois->server);
      whois->server = l_strdup(pos1);
    }
  }

  whois_action(whois);
}

void whois_eval_whowas(char *data)
{
  whois_t *whois;
  char *pos1;

  pos1 = arg(data, 0);

  whois = whois_search_unreceived(pos1);
  l_free(whois->user);
  whois->user = l_strdup(pos1);

  whois->online = 0;
  whois->advanced = 0;
  whois->received = global.current_time;

  pos1 = arg(NULL, 0);
  whois->level = level2int(pos1);

  pos1 = arg(NULL, 0);
  whois->online_time = strtoul(pos1, NULL, 10);

  whois_action(whois);
}

void whois_eval_not_online(char *data)
{
  whois_t *whois;
  char *pos1;

  whois = whois_search_unreceived(data);
  l_free(whois->user);
  whois->user = l_strdup(data);

  whois->online = 0;
  whois->advanced = 0;
  whois->received = global.current_time;

  pos1 = arg(NULL, 0);
  whois->level = L_USER;

  pos1 = arg(NULL, 0);
  whois->online_time = 0;

  whois_action(whois);
}

// WHOIS_TEXT
void whois_action_text(whois_t * whois)
{
  char *text;
  char str[1024];
  char *prefix;

  if (!whois) return;
  if (!whois->received) return;

  chat_print_time_stamp(global.current_page, M_PUBLIC);
  prefix = cparse(global.scheme->client_prefix);
  chat_print("message", prefix);
  chat_print("message", "User       : ");
  chat_print("message", whois->user);
  if (whois->online && whois->advanced) {
    chat_print("message", " (");
    chat_print("message", whois->ip);
    chat_print("message", ")");
  }
  chat_print("message", "\n");

  chat_print_time_stamp(global.current_page, M_PUBLIC);
  prefix = cparse(global.scheme->client_prefix);
  chat_print("message", prefix);
  if (whois->online) {
    chat_print("message", "Online     : ");
    if (whois->online_time >= 60 * 60 * 24)
      text = l_strdup_printf(_("%ld days %ld hours %ld minutes %ld seconds"),
			     whois->online_time / (24 * 60 * 60),
			     (whois->online_time % (24 * 60 * 60)) / (60 * 60),
			     (whois->online_time % (60 * 60)) / 60,
			     whois->online_time % 60);
    else if (whois->online_time >= 60 * 60)
      text = l_strdup_printf(_("%ld hours %ld minutes %ld seconds"),
			     whois->online_time / (60 * 60),
			     (whois->online_time % (60 * 60)) / 60,
			     whois->online_time % 60);
    else
      text = l_strdup_printf(_("%ld minutes %ld seconds"),
			     whois->online_time / 60,
			     whois->online_time % 60);
    chat_print("message", text);
    chat_print("message", "\n");
    l_free(text);
  } else if (whois->online_time > 0) {
    chat_print("message", "Last Seen  : ");
    text = l_strdup_printf("%s", ctime(&whois->online_time));
    chat_print("message", text);
    l_free(text);
  } else {
    chat_print("message", "Last Seen  : ");
    text = l_strdup(_("Unregistered\n"));
    chat_print("message", text);
    l_free(text);
    return;
  }
  
  chat_print_time_stamp(global.current_page, M_PUBLIC);
  prefix = cparse(global.scheme->client_prefix);
  chat_print("message", prefix);
  chat_print("message", _("Level      : "));
  chat_print("message", LevelShort(whois->level));
  if (whois->online) {
    chat_print("message", " (");
    chat_print("message", whois->status);
    chat_print("message", ")");
  }
  chat_print("message", "\n");
  
  if (!whois->online) return;

  chat_print_time_stamp(global.current_page, M_PUBLIC);
  prefix = cparse(global.scheme->client_prefix);
  chat_print("message", prefix);
  chat_print("message", _("Channels   : "));
  chat_print("message", whois->channels);
  chat_print("message", "\n");
  
  chat_print_time_stamp(global.current_page, M_PUBLIC);
  prefix = cparse(global.scheme->client_prefix);
  chat_print("message", prefix);
  chat_print("message", _("Shared     : "));
  if (whois->shared == 1) sprintf(str, _("1 file"));
  else sprintf(str, _("%d files"), whois->shared);
  chat_print("message", str);
  if (whois->advanced) {
    sprintf(str, _(" on port %d"), whois->data_port);
    chat_print("message", str);
  }
  chat_print("message", "\n");
  
  chat_print_time_stamp(global.current_page, M_PUBLIC);
  prefix = cparse(global.scheme->client_prefix);
  chat_print("message", prefix);
  chat_print("message", _("Transfers  : "));
  if (whois->advanced)
    sprintf(str, "%d(%d) down %d(%d) up", 
	    whois->downloads, whois->total_down,
	    whois->uploads, whois->total_up);
  else
    sprintf(str, "%d down %d up", whois->downloads, whois->uploads);
  chat_print("message", str);
  chat_print("message", "\n");
  
  chat_print_time_stamp(global.current_page, M_PUBLIC);
  prefix = cparse(global.scheme->client_prefix);
  chat_print("message", prefix);
  chat_print("message", _("Link       : "));
  chat_print("message", LineSpeed(whois->link));
  chat_print("message", "\n");
  
  chat_print_time_stamp(global.current_page, M_PUBLIC);
  prefix = cparse(global.scheme->client_prefix);
  chat_print("message", prefix);
  chat_print("message", _("Client     : "));
  chat_print("message", whois->client);
  chat_print("message", "\n");

  if (whois->advanced) {
    chat_print_time_stamp(global.current_page, M_PUBLIC);
    prefix = cparse(global.scheme->client_prefix);
    chat_print("message", prefix);
    chat_print("message", _("Server     : "));
    if (whois->server) {
      sprintf(str, "%s:%d", whois->server, whois->server_port);
    } else {
      sprintf(str, "%d", whois->server_port);
    }
    chat_print("message", str);
    chat_print("message", "\n");

    chat_print_time_stamp(global.current_page, M_PUBLIC);
    prefix = cparse(global.scheme->client_prefix);
    chat_print("message", prefix);
    chat_print("message", _("E-mail     : "));
    chat_print("message", whois->email);
    chat_print("message", "\n");
  }

  if (whois->advanced && whois->ip) {
    sprintf(str, "nslookup %s", whois->ip);
    exec_command(str);
  }
}

void whois_action_show(whois_t * whois)
{
  char *text;
  char *text2;
  char str[1024];
  GtkEntry *entry;
  GtkWidget *temp;

  if (!whois)
    return;
  if (!whois->received)
    return;

  if (!global.whois_win) {
    global.whois_win = create_whois_win();
  } else {
    gdk_window_raise(global.whois_win->window);
  }
  gtk_widget_show(global.whois_win);

  text = l_strdup(ctime(&whois->received));
  text[strlen(text) - 1] = 0;
  text2 = l_strdup_printf("%s [%s]", _("Whois User"), text);
  gtk_window_set_title(GTK_WINDOW(global.whois_win), text2);
  l_free(text);
  l_free(text2);

  temp = lookup_widget(global.whois_win, "label116");
  if (whois->online)
    gtk_label_set_text(GTK_LABEL(temp), _("Online"));
  else
    gtk_label_set_text(GTK_LABEL(temp), _("Last Seen"));

  temp = lookup_widget(global.whois_win, "label527");
  gtk_label_set_text(GTK_LABEL(temp), whois->user);

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry25"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, _("Level: "));
  gtk_entry_append_text(entry, LevelShort(whois->level));

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry26"));
  if (whois->online) {
    if (whois->online_time >= 60 * 60 * 24)
      text =
	  l_strdup_printf(_("%ld days %ld hours %ld minutes %ld seconds"),
			  whois->online_time / (24 * 60 * 60),
			  (whois->online_time % (24 * 60 * 60)) / (60 *
								   60),
			  (whois->online_time % (60 * 60)) / 60,
			  whois->online_time % 60);
    else if (whois->online_time >= 60 * 60)
      text = l_strdup_printf(_("%ld hours %ld minutes %ld seconds"),
			     whois->online_time / (60 * 60),
			     (whois->online_time % (60 * 60)) / 60,
			     whois->online_time % 60);
    else
      text = l_strdup_printf(_("%ld minutes %ld seconds"),
			     whois->online_time / 60,
			     whois->online_time % 60);
  } else if (whois->online_time > 0) {
    text = l_strdup_printf("%s", ctime(&whois->online_time));
  } else {
    text = l_strdup(_("Unregistered"));
  }
  gtk_entry_set_text(entry, text);
  l_free(text);

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry27"));
  if (whois->online) {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
    gtk_entry_set_text(entry, whois->channels);
  } else {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), FALSE);
    gtk_entry_set_text(entry, "");
  }

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry28"));
  if (whois->online) {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
    gtk_entry_set_text(entry, _("Status: "));
    gtk_entry_append_text(entry, whois->status);
  } else {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), FALSE);
    gtk_entry_set_text(entry, "");
  }

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry62"));
  if (whois->online) {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
    gtk_entry_set_text(entry, _("Shared: "));
    sprintf(str, "%d", whois->shared);
    gtk_entry_append_text(entry, str);
  } else {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), FALSE);
    gtk_entry_set_text(entry, "");
  }

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry30"));
  if (whois->online) {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
    sprintf(str, "%d", whois->downloads);
    gtk_entry_set_text(entry, str);
    gtk_entry_append_text(entry, _(" downs"));
  } else {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), FALSE);
    gtk_entry_set_text(entry, "");
  }

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry59"));
  if (whois->online) {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
    sprintf(str, "%d", whois->uploads);
    gtk_entry_set_text(entry, str);
    gtk_entry_append_text(entry, _(" ups"));
  } else {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), FALSE);
    gtk_entry_set_text(entry, "");
  }

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry61"));
  if (whois->online) {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
    gtk_entry_set_text(entry, _("Link: "));
    gtk_entry_append_text(entry, LineSpeed(whois->link));
  } else {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), FALSE);
    gtk_entry_set_text(entry, "");
  }

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry60"));
  if (whois->online) {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
    gtk_entry_set_text(entry, whois->client);
  } else {
    gtk_widget_set_sensitive(GTK_WIDGET(entry), FALSE);
    gtk_entry_set_text(entry, "");
  }

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry34"));
  gtk_entry_set_text(entry, "");

  temp = lookup_widget(global.whois_win, "table18");
  if (!whois->advanced) {
    gtk_widget_hide(temp);
    return;
  } else {
    gtk_widget_show(temp);
  }

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry63"));
  sprintf(str, "%d", whois->total_down);
  gtk_entry_set_text(entry, str);
  gtk_entry_append_text(entry, _(" downs"));

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry64"));
  sprintf(str, "%d", whois->total_up);
  gtk_entry_set_text(entry, str);
  gtk_entry_append_text(entry, _(" ups"));

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry65"));
  gtk_entry_set_text(entry, whois->ip);

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry66"));
  if (whois->server) {
    sprintf(str, "%s: %d", whois->server, whois->server_port);
  } else {
    sprintf(str, "Server: %d", whois->server_port);
  }
  gtk_entry_set_text(entry, str);

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry67"));
  sprintf(str, "%d", whois->data_port);
  gtk_entry_set_text(entry, _("Data: "));
  gtk_entry_append_text(entry, str);

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry68"));
  gtk_entry_set_text(entry, whois->email);
}

void whois_action_yourself(whois_t * whois)
{

  if (!whois)
    return;

  global.user.status = status2int(whois->status);

  if (global.user.level != whois->level)
    set_user_level(whois->level);
}

void whois_action_online(whois_t * whois)
{
  if (!whois->online)
    resume_user_offline(whois->user);
}

void whois_action_userinfo(whois_t * whois)
{
  user_info_t *userinfo;

  userinfo = user_info_search(whois->user, 1);
  if (!userinfo) return;

  if (!whois->online) {
    // make lopster request info if user appears again
    userinfo->timestamp = 1;
    return;
  }

  if (userinfo->linespeed != whois->link) {
    userinfo->linespeed = whois->link;
    upload_update_speed(userinfo);
  }
  userinfo->shares = whois->shared;
  userinfo->timestamp = global.current_time;
  userinfo->last_seen = global.current_time;
  if (userinfo->last_server) l_free(userinfo->last_server);
  if (SERVER->description)
    userinfo->last_server = strdup(SERVER->description);
  else
    userinfo->last_server = strdup(SERVER->address);
  if (!strncasecmp(whois->client, "WinMX", 5))
    userinfo->client = C_WINMX;
  else
    userinfo->client = C_OTHER;
}

void speed_request(char *user, int flag)
{
  speed_t *speed;

  speed = speed_search(user, SPEED_ALL);
  if (speed) {
    speed->flag |= flag;
    if (speed->speed >= 0) speed_action(speed);
    return;
  }
  speed = speed_new(user, flag);
  if (!speed)
    return;
  send_command(CMD_CLIENT_USERSPEED, speed->user);
  speed->sent = global.current_time;
}

speed_t *speed_new(char *user, int flag)
{
  speed_t *speed;

  if (!user || !(*user))
    return NULL;

  speed = (speed_t *) l_malloc(sizeof(speed_t));
  speed->speed = -1;
  speed->sent = 0;
  speed->flag = flag;
  speed->user = l_strdup(user);

  global.speed_requests = g_list_prepend(global.speed_requests, speed);

  return speed;
}

void speed_destroy(speed_t * speed)
{
  if (speed->user)
    l_free(speed->user);
  l_free(speed);
}

void speed_remove(speed_t * speed)
{
  global.speed_requests = g_list_remove(global.speed_requests, speed);
  if (speed->speed < 0) speed_action(speed);
  speed_destroy(speed);
}

void speed_action(speed_t * speed)
{
  if (!speed) return;

  if (speed->flag & SPEED_UPLOAD) {
    speed_action_upload(speed);
  }
  if (speed->flag & SPEED_DOWNLOAD) {
    speed_action_download(speed);
  }
  if (speed->flag & SPEED_RESUME) {
    speed_action_resume(speed);
  }
  if (speed->flag & SPEED_SHOW) {
    speed_action_show(speed);
  }
}

speed_t *speed_search(char *user, int flag)
{
  GList *dlist;
  speed_t *speed;

  if (!user || !*user)
    return NULL;

  for (dlist = global.speed_requests; dlist; dlist = dlist->next) {
    speed = (speed_t *) (dlist->data);
    if (!l_strcasecmp(speed->user, user) && (speed->flag & flag))
      return speed;
  }
  return NULL;
}

void speed_action_resume(speed_t * speed)
{
  if (speed->speed >= 0) {
    resume_user_online(speed->user, speed->speed);
  } else
    resume_user_offline(speed->user);
}

void speed_action_upload(speed_t * speed)
{
  GtkCList *clist;
  GList *s = NULL;
  GList *dlist;
  socket_t *socket;
  transfer_t *transfer;
  int i1;
  user_info_t* userinfo;

  if (speed->speed >= 0) return;

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  userinfo = user_info_search(speed->user, 0);
  if (!userinfo) return;

  gtk_clist_freeze(clist);
  for (i1 = 0; i1 < clist->rows; i1++) {
    socket = (socket_t *) gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t *) (socket->data);
    if (transfer->user_info != userinfo) continue;

    if (transfer->status == S_QUEUED) s = g_list_prepend(s, socket);
  }

  for (dlist = s; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    socket_end(socket, &(SocketStatus[S_DELETE]));
  }
  g_list_free(s);
  gtk_clist_thaw(clist);
}

void speed_action_download(speed_t * speed)
{
  GList *s = NULL;
  GList *dlist;
  socket_t *socket;
  transfer_t *transfer;
  user_info_t* userinfo;

  if (speed->speed >= 0) return;
  
  userinfo = user_info_search(speed->user, 0);
  if (!userinfo) return;

  s = NULL;
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = dlist->data;
    if (!socket) continue;
    transfer = socket->data;
    if (!transfer || transfer->type != T_DOWNLOAD ||
	transfer->user_info != userinfo) continue;
    s = g_list_prepend(s, socket);
  }

  for (dlist = s; dlist; dlist = dlist->next) {
    socket = dlist->data;
    transfer = socket->data;

    // remove socket from resume
    if (transfer->resume) {
      transfer->resume->downloads = 
	g_list_remove(transfer->resume->downloads, socket);
    }
    socket_destroy(socket, &(SocketStatus[S_CANCELED]));
  }
  g_list_free(s);
}

void speed_action_show(speed_t * speed)
{
  if (speed->speed >= 0)
    client_message("Message", _("User <%s> has linespeed [%s]"),
		   speed->user, LineSpeed(speed->speed));
  else {
    client_message("Message", _("User <%s> seems to be offline"), speed->user);
  }
}

