/* Copyright (C) 2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <unistd.h>
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <dirent.h>
#include <sys/stat.h>
#include <time.h>
#include <sys/time.h>
#include <netdb.h>
#include <sys/utsname.h>

#include "lopster.h"
#include "connection.h"
#include "global.h"
#include "search.h"
#include "transfer.h"
#include "resume.h"
#include "interface.h"
#include "support.h"
#include "callbacks.h"
#include "browse.h"
#include "share.h"
#include "hotlist.h"
#include "commands.h"
#include "chat.h"
#include "dirselect.h"
#include "scheme.h"
#include "handler.h"
#include "resume.h"
#include "server.h"
#include "preferences.h"
#include "log.h"
#include "exec.h"
#include "ping.h"

void ping_fire_up(ping_t * ping);
gint get_ping_input(gpointer data, gint source,
		    GdkInputCondition condition);
void ping_update(ping_t * ping);

ping_t *ping_new(unsigned long ip)
{
  ping_t *ping;

  ping = (ping_t *) l_malloc(sizeof(ping_t));
  ping->cnt = 0;
  ping->input = -1;
  ping->pos = 0;
  ping->pid = -1;

  ping->time = 0;
  ping->ip_long = ip;

  global.pings = g_list_append(global.pings, ping);

  return ping;
}

void ping_close(ping_t * ping)
{
  int status;

  if (!ping)
    return;

  waitpid(ping->pid, &status, WNOHANG);
  ping->pid = -1;
  if (ping->fd >= 0)
    close(ping->fd);
  ping->fd = -1;
  if (ping->input >= 0)
    gdk_input_remove(ping->input);
  ping->input = -1;
  ping_update(ping);
}

void ping_destroy(ping_t * ping)
{
  if (!ping)
    return;

  ping_close(ping);
  global.pings = g_list_remove(global.pings, ping);
  if (ping->ip)
    l_free(ping->ip);
  l_free(ping);
}

ping_t *ping_search(unsigned long ip)
{
  GList *dlist;
  ping_t *ping;

  for (dlist = global.pings; dlist; dlist = dlist->next) {
    ping = (ping_t *) (dlist->data);
    if (ping->ip_long == ip)
      return ping;
  }
  return NULL;
}

int ping_up_to_date(ping_t * ping)
{
  if (ping->update + 10 < global.current_time)
    return 0;
  else
    return 1;
}

void ping_update_file(GtkWidget * widget, file_t * file)
{
  int row;
  char str[1024];
  GtkWidget *clist;

  if (!widget)
    return;

  clist = gtk_object_get_data(GTK_OBJECT(widget), "clist");
  row = gtk_clist_find_row_from_data(GTK_CLIST(clist), file);
  if (row == -1)
    return;

  if (file->ping) {
    sprintf(str, "%d ms", file->ping);
  } else {
    sprintf(str, "n/a");
  }
  gtk_clist_set_text(GTK_CLIST(clist), row, 7, str);
}

void ping_update(ping_t * ping)
{
  search_t *search;
  GList *dlist;
  GList *dlist2;
  file_t *file;

  for (dlist = global.searches; dlist; dlist = dlist->next) {
    search = (search_t *) (dlist->data);
    for (dlist2 = search->results; dlist2; dlist2 = dlist2->next) {
      file = (file_t *) (dlist2->data);
      if (file->ip == ping->ip_long) {
	file->ping = ping->time;
	if (!search->resume)
	  ping_update_file(search->link, file);
      }
    }
  }
}

void ping(unsigned long addr)
{
  ping_t *ping;

  ping = ping_search(addr);
  if (!ping) ping = ping_new(addr);

  ping->ip = l_strdup(ntoa(addr));
  ping->update = global.current_time;

  ping_fire_up(ping);
}

void ping2(char *ip)
{
  ping_t *ping;
  unsigned long addr;

  addr = resolv_address(ip);
  ping = ping_new(addr);

  ping->ip = l_strdup(ip);
  ping->update = global.current_time;

  ping_fire_up(ping);
}

void ping_fire_up(ping_t * ping)
{
  int p0[2], p1[2], p2[2], pid;
  int i;
  gchar *argv[20];
  gchar *pos;
  char *pcom;
  int i1;

  p0[0] = p1[0] = p2[0] = -1;
  p0[1] = p1[1] = p2[1] = -1;

  if (pipe(p0) || pipe(p1) || pipe(p2)) {
    client_message(NULL, _("Unable to start new process: %s"),
		   strerror(errno));
    if (p0[0] >= 0)
      close(p0[0]);
    if (p0[0] >= 0)
      close(p0[1]);
    if (p0[0] >= 0)
      close(p1[0]);
    if (p0[0] >= 0)
      close(p1[1]);
    if (p0[0] >= 0)
      close(p2[0]);
    if (p0[0] >= 0)
      close(p2[1]);
    return;
  }

  switch ((pid = fork())) {
  case -1:
    client_message(NULL, _("Couldn't start new process!"));
    break;
  case 0:
    // child
    setsid();
    setuid(getuid());
    setgid(getgid());

    dup2(p1[1], 1);		// copy stdout
    close(p0[0]);
    close(p0[1]);
    close(p1[0]);
    close(p2[0]);
    close(p2[1]);
    p0[1] = p0[0] = p1[0] = p2[0] = p2[1] = -1;
    for (i = 3; i < 256; i++)
      close(i);

    pcom = l_strdup(global.ping_command);
    //    printf("[%s]\n", pcom);
    argv[0] = arg(pcom, 0);
    for (i1 = 1; i1 < 18; i1++) {
      pos = arg(NULL, 0);
      argv[i1] = pos;
      if (!pos)
	break;
      if (!l_strcasecmp(argv[i1], "$IP"))
	argv[i1] = ping->ip;
      //      printf("arg: %s\n", argv[i1]);
    }
    execve(argv[0], argv, environ);
    _exit(-1);
    break;
  default:
    // parent
    close(p0[0]);
    close(p0[1]);
    close(p1[1]);
    close(p2[0]);
    close(p2[1]);
    p0[1] = p0[0] = p1[1] = p2[0] = p2[1] = -1;

    ping->pid = pid;
    ping->fd = p1[0];

    ping->input =
	gdk_input_add(ping->fd, GDK_INPUT_READ,
		      GTK_SIGNAL_FUNC(get_ping_input), ping);

    break;
  }
}

gint get_ping_input(gpointer data, gint source,
		    GdkInputCondition condition)
{
  ping_t *ping = (ping_t *) data;
  int res;
  char *pos1;
  char *pos2;
  char *t1;
  char *t2;

  if (condition != GDK_INPUT_READ) {
    ping_close(ping);
    return 0;
  }
  switch (res = read(source, ping->buf + ping->pos, 2048 - ping->pos - 1)) {
  case -1:
    ping_close(ping);
    return 0;
  case 0:
    ping_close(ping);
    return 0;
  default:
    break;
  }

  ping->buf[res + ping->pos] = 0;

  pos1 = ping->buf;
  while ((pos2 = strchr(pos1, '\n')) != NULL) {
    *pos2 = 0;
    //    printf("p %s: %s\n", ping->ip, pos1);
    if (!strncmp("round-trip", pos1, 10)) {
      //      printf("%s\n", pos1);
      t2 = strchr(pos1, '=');
      if (t2)
	t1 = strchr(t2, '/');
      else
	t1 = NULL;
      if (t1)
	t2 = strchr(t1 + 1, '/');
      else
	t2 = NULL;
      if (t1 && t2) {
	*t2 = 0;
	ping->time = atoi(t1 + 1);
	//      printf("got ping %d\n", ping->time);
      }
    }
    pos1 = pos2 + 1;
  }

  strcpy(ping->buf, pos1);
  ping->pos = strlen(ping->buf);
  return 0;
}
