#include <sys/types.h>
#include <sys/socket.h>
#include <dirent.h>
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "callbacks.h"
#include "connection.h"
#include "global.h"
#include "support.h"
#include "search.h"
#include "browse.h"
#include "share.h"
#include "scheme.h"
#include "md5.h"
#include "handler.h"

#define FRAMES_FLAG     0x0001
#define BYTES_FLAG      0x0002
#define TOC_FLAG        0x0004
#define VBR_SCALE_FLAG  0x0008

/*
  load -------------.
                     >-- commit
  refresh -- remove-
*/

file_t* global_file;
GList* cur_file;

void lib_sort();
file_t* lib_search_file(char* longname);
void lib_insert_file(file_t *file);
void lib_tree_update_file(file_t *file);


unsigned int get_bits(unsigned char* buffer, int start, int no) {
  unsigned int res = 0;

  res |= buffer[0]<<24 | buffer[1]<<16 |
    buffer[2]<<8 | buffer[3];
  
  res <<= start;
  res >>= (32-no);
  return res;
}

static int ExtractI4(unsigned char *buf) {
  int x;
  /* big endian extract */ 
  x = buf[0];
  x <<= 8;
  x |= buf[1];
  x <<= 8;
  x |= buf[2];
  x <<= 8;
  x |= buf[3];
  
  return x;
}

char* read_xing(FILE* fd, int offset) {
  unsigned char buffer[1024];
  unsigned char* buf;

  buf = buffer;
  if (fread(buffer, 1, offset+120, fd) < offset+120) return NULL;

  buf += offset;
  if (buf[0] != 'X') return 0;	/* fail */
  if (buf[1] != 'i') return 0;	/* header not found */
  if (buf[2] != 'n') return 0;
  if (buf[3] != 'g') return 0;
  
  buf += 4;
  return buf;
}

void file_parse_mp3_header(file_t *file) {
  FILE* fd;
  unsigned char buffer[4];  // header
  unsigned char* buf;
  int mpeg_version;
  int mpeg_layer;
  int res;
  int index;
  long totalframes;
  int padding;
  int stereo;
  int protection;
  int lsf;
  double bpf;
  static int bs[4] = {0, 384, 1152, 1152};
  static int s1[4] = {0, 48000, 144000, 144000};
  double tpf;
  int xing_offset;
  int pos;
  short br[2][4][16] = {
    {  // mpeg 2
      { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0 },      // 
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0 },      // 
      { 0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, 0 }  // 
    },
    {  // mpeg1
      { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
      { 0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, 0 },  // layer I
      { 0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384, 0 },    // layer II
      { 0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 0 }     // layer III
    }
  };

  int freq[2][2][4] = {
    {
      {11025, 12000, 8000, 0},   // mpeg 2.5
      {0, 0, 0},                 // reserved
    },
    {
      {22050, 24000, 16000, 0},  // mpeg 2
      {44100, 48000, 32000, 0}   // mpeg 1
    },
  };

  if ((fd = fopen(file->longname, "r")) != NULL) {
    if (fread(buffer, sizeof(char), 4, fd) != 4) {
      printf("could not read header [%s]\n", file->longname);
      return;
    }
    while (get_bits(buffer, 0, 11) != 0x7ff) {
      buffer[0] = buffer[1];
      buffer[1] = buffer[2];
      buffer[2] = buffer[3];
      if (fread(&buffer[3], sizeof(char), 1, fd) != 1) {
	printf("could not read header [%s]\n", file->longname);
	return;
      }
    }
  } else {
    //    printf("could not open file for reading header [%s]\n", file->longname);
    return;
  }

  // sync word
  res = get_bits(buffer, 0, 11);
  
  // sync word
  index = get_bits(buffer, 11, 1);

  // MPEG Version
  mpeg_version = get_bits(buffer, 12, 1);
  if (mpeg_version) lsf = 0x0;
  else lsf = 0x1;

  // MPEG Layer
  mpeg_layer = get_bits(buffer, 13, 2);
  mpeg_layer = 4 - mpeg_layer;
  if (mpeg_layer > 3) return;

  // Protection Bit
  protection = get_bits(buffer, 15, 1);
  
  // Bitrate
  res = get_bits(buffer, 16, 4);
  if (res > 14) return;
  file->bitrate = br[mpeg_version][mpeg_layer][res];

  // Fequency
  res = get_bits(buffer, 20, 2);
  file->frequency = freq[index][mpeg_version][res];
  if ((file->bitrate == 0) || (file->frequency == 0)) {
    //    printf("corrupted header! [%s]\n", file->longname);
    return;
  }

  // Padding Bit
  padding = get_bits(buffer, 22, 1);
  
  // Private Bit
  res = get_bits(buffer, 23, 1);
  
  // Mode
  res = get_bits(buffer, 24, 2);
  if (res == 3) stereo = 0;
  else stereo = 1;

  tpf = (double) bs[mpeg_layer] / (file->frequency << lsf);

  if (mpeg_version) {           /* mpeg1 */
    if (stereo) xing_offset = 32;
    else xing_offset = 17;
  } else {                      /* mpeg2 */
    if (stereo) xing_offset = 17;
    else xing_offset = 9;
  }

  pos = ftell(fd);
  if ((buf = read_xing(fd, xing_offset))) {
    int head_flags;
    int vbr_scale = -1;
    int bytes;

    head_flags = ExtractI4(buf);
    buf += 4;
    if (head_flags & FRAMES_FLAG) {
      totalframes = ExtractI4(buf);
      buf += 4;
    } else totalframes = 1;
    if (head_flags & BYTES_FLAG) {
      bytes = ExtractI4(buf);
      buf += 4;
    } else bytes = 1;
    if (head_flags & TOC_FLAG) {
      buf += 100;
    }
    if (head_flags & VBR_SCALE_FLAG) {
      vbr_scale = ExtractI4(buf);
      buf += 4;
    }
    file->bitrate = 
      (gint) ((bytes * 8) / (tpf * totalframes * 1000));
    bpf = (double)(file->size-pos) / totalframes;
    file->vbr = 1;
  } else {
    bpf = file->bitrate * s1[mpeg_layer] / (file->frequency << lsf);
    totalframes = (int)((file->size-pos) / bpf);
    file->vbr = 0;
  }
  
  file->duration = (time_t) (totalframes * tpf);

#ifdef SHARE_DEBUG
  printf("mpeg %d layer %d, br %d, freq %d bpf %.1f, frames %ld, duration %d, vbr:%d\n",
	 lsf+1, mpeg_layer, file->bitrate, file->frequency, bpf, 
	 totalframes, file->duration, file->vbr);
#endif

  // Mode Extention
  res = get_bits(buffer, 26, 2);
  
  // Copyright
  res = get_bits(buffer, 28, 1);
  
  // Original Home
  res = get_bits(buffer, 29, 1);
  
  // Enphasis
  res = get_bits(buffer, 30, 2);
  
  fclose(fd);
}

void file_parse_header(file_t *file) {
  char* suf;

  suf = strrchr(file->longname, '.');
  if (suf) suf++;
  
  if (suf && !strcasecmp(suf, "MP3")) {
    file_parse_mp3_header(file);
  } else if (suf && !strcasecmp(suf, "OGG")) {
    // for now set dummy values
    file->bitrate = 32;
    file->frequency = 32000;
    file->duration = 60;
  } else {
    file->bitrate = 32;
    file->frequency = 32000;
    file->duration = 60;
  }
}

void file_calc_md5(file_t *file) {
  int i1;
  char str[33];
  char hex[16] = "0123456789abcdef";

  if (file->md5) free(file->md5);
  file->md5 = NULL;
  if (global.options.dummy_md5) {
    for (i1 = 0; i1 < 32; i1++) str[i1] = hex[random()%16];
    str[32] = 0;
    file->md5 = g_strdup_printf("%s-%ld", str, file->size);
  } else {
    file->md5 = MD5File(file->longname, file->md5);
  }
}

suffix_t* check_suffix(char* name, int mime) {
  GList* dlist;
  char* suffix;
  suffix_t* suf;

  suffix = strrchr(name, '.');
  if (!suffix) return NULL;
  else suffix++;

  for (dlist = global.mimetype[mime].suffixes; dlist; dlist=dlist->next) {
    suf = (suffix_t*)(dlist->data);
    if (!strcasecmp(suffix, suf->suffix)) return suf;
  }
  return NULL;
}

void lib_commit_file (file_t* file) {
  char t2[2048];
  char t[1024];
  suffix_t* suffix;

  if (global.status.connection < 2) return;

  if ((file->flags & FLAG_TO_SHARE) &&
      ((file->flags & FLAG_SHARED) == 0)) {
    suffix = get_suffix(file->winname);
    if ((suffix && suffix->as_mp3) || (!opennap_version(0, 37))) {
      strcpy(t2, "\"");
      strcat(t2, file->winname);
      strcat(t2, "\" ");
      strcat(t2, file->md5);
      if (opennap_version(0, 0))
	sprintf(t, " %ld %d %d %d",
		file->size, find_valid_bitrate(file->bitrate),
		file->frequency, file->duration);
      else
	sprintf(t, " %ld %d %d %d", file->size, file->bitrate,
		file->frequency, file->duration);
      strcat(t2, t);
      send_command(CMD_CLIENT_ADD_FILE, t2);
    } else  {
      strcpy(t2, "\"");
      strcat(t2, file->winname);
      strcat(t2, "\"");
      sprintf(t, " %ld %s %s",
	      file->size, file->md5, 
	      MimeNames_((file->mime_type==MIME_NONE)?(MIME_TEXT):(file->mime_type)));
      strcat(t2, t);
      send_command(CMD_CLIENT_SHARE_FILE, t2);
    }
    file->flags &= (0xff^FLAG_TO_SHARE);
    file->flags |= FLAG_SHARED;
    global.user.unshared--;
    lib_tree_update_file(file);
  }
}

gint lib_commit_idle(gpointer data) {
  file_t* file;
  int i1;

#ifdef SHARE_DEBUG
  printf("commit idle %d %d\n", global.status.building,
	 global.status.connection);
#endif
  if (global.status.building > 1) return 0;
  if (global.status.connection < 2) return 0;

  if (!cur_file) cur_file = global.user.files;

  for (i1 = 0; i1 < 100; i1++) {
    if (!cur_file) break;
    file = (file_t*)(cur_file->data);
    lib_commit_file(file);
    cur_file = cur_file->next;
  }
  
  if (cur_file) return 1;

  lib_update_stats();

  return 0;
}

void lib_update_file (file_t* file) {
  struct stat stats;
  unsigned long size;

  if (stat(file->longname, &stats) < 0) {
    size = file->size;
  } else {
    size = stats.st_size;
  }
  if (file->size != size) {
    //    printf("%lu - %lu\n", file->size, size);
    global.user.bytes -= file->size;
    file->size = size;
    global.user.bytes += file->size;
    //    printf("parsing header\n");
    file_parse_header(file);
    file_calc_md5(file);
  }
  lib_tree_update_file(file);
}

int lib_update_idle(gpointer data) {
  file_t* file;
  int i1;

#ifdef SHARE_DEBUG
  printf("update idle\n");
#endif
  if (!cur_file) cur_file = global.user.files;

  for (i1 = 0; i1 < 10; i1++) {
    if (!cur_file) break;
    file = (file_t*)(cur_file->data);
    lib_update_file(file);
    cur_file = cur_file->next;
  }
  
  if (cur_file) return 1;
  
  global.status.building = 0;
  lib_update_stats();

  gtk_idle_add(lib_commit_idle, NULL);
  return 0;
}

void lib_remove_file(file_t* file) {
  char* command;
  GtkCTreeNode* node;
  GtkCTreeNode* node2;
  GtkCTree* ctree;

#ifdef SHARE_DEBUG
  printf("removing [%s]\n", file->longname);
#endif
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  if (file->flags & FLAG_SHARED) {
    // unsharing
    if (global.status.connection > 1) {
      if (opennap_version(0, 37) || !opennap_version(0,0))
	command = g_strdup_printf("\"%s\"", file->winname);
      else
	command = g_strdup_printf("%s", file->winname);
      send_command(CMD_CLIENT_REMOVE_FILE, command);
      g_free(command);
    }
    global.user.unshared++;
  }
  if (file->flags & FLAG_TO_DELETE) {
    unlink(file->longname);
  }

  if ((file->flags & FLAG_TO_DELETE) ||
      (file->flags & FLAG_TO_REMOVE)) {
    global.user.shared--;
    global.user.unshared--;
    global.user.bytes -= file->size;
    global.user.files = g_list_remove(global.user.files, file);
    node = gtk_ctree_find_by_row_data(ctree, NULL, file);
    while (node) {
      node2 = GTK_CTREE_ROW(node)->parent;
      gtk_ctree_remove_node(ctree, node);
      if (!node2 || (GTK_CTREE_ROW(node2)->children != NULL)) break;
      node = node2;
    }
  } else {
    file->flags &= 0xff^(FLAG_SHARED|FLAG_TO_UNSHARE);
    lib_tree_update_file(file);
  }
}

gint lib_remove_idle(gpointer data) {
  file_t* file;
  int i1;
  GtkCList* clist;

  global.status.building = 2;
  if (!cur_file) cur_file = global.user.files;
  
  clist = GTK_CLIST(lookup_widget(global.win, "lib_tree"));
  gtk_clist_freeze(clist);

  for (i1 = 0; i1 < 10; i1++) {
    if (!cur_file) break;
    file = (file_t*)(cur_file->data);
    
    cur_file = cur_file->next;
    if ((file->flags & FLAG_TO_REMOVE) ||
	(file->flags & FLAG_TO_DELETE) ||
	(file->flags & FLAG_TO_UNSHARE))
      lib_remove_file(file);
  }
  gtk_clist_thaw(clist);
  if (cur_file) return 1;

  // now updating stats
  lib_update_stats();

  lib_sort();

  gtk_idle_add(lib_update_idle, NULL);

  return 0;
}

int add_shared_directory(int mime, char* dirname) {
  DIR* dir;
  struct dirent* entry;
  struct stat buf;
  char name[2048];
  file_t* file;
  
  if ((dir = opendir(dirname)) == NULL) {
#ifdef SHARE_DEBUG
    printf("could not open dir [%s] (add_shared)\n", dirname);
#endif
    return 0;
  }

  while ((entry = readdir(dir)) != NULL) {
    if (!strncmp(entry->d_name, ".", 1)) continue;
    if (!strncmp(entry->d_name, "..", 2)) continue;
    
    strcpy(name, dirname);
    strcat(name, "/");
    strcat(name, entry->d_name);
    stat(name, &buf);
    if (buf.st_mode&S_IFDIR) {
      // links are followed
      //      printf("mode : %s:%x:%x\n", name, buf.st_mode, S_IFLNK);
      //      if ((buf.st_mode&S_IFLNK) == 0)
      add_shared_directory(mime, name);
    } else {
      if ((mime == MIME_NONE) || check_suffix(entry->d_name, mime)) {
	file = lib_search_file(name);
	if (!file) {
	  file = file_create_from_local(name);
	  if (file) lib_insert_file(file);
	} else {
#ifdef SHARE_DEBUG
	  printf("updating [%s]\n", file->longname);
#endif
	  file->flags &= 0xff^FLAG_TO_REMOVE;
	}
      }
      while (gtk_events_pending()) gtk_main_iteration();
    }
  }
  closedir(dir);

  return 1;
}

void lib_refresh() {
  int i1 = 0;
  GList* dlist = NULL;
  GtkCList* clist;
  char* text;

#ifdef SHARE_DEBUG
  printf("refresh\n");
#endif
  // already buidling?
  if (global.status.building > 1) return;

  global.status.building = 2;

  lib_set_flag(FLAG_TO_REMOVE, NULL);

  clist = GTK_CLIST(lookup_widget(global.win, "lib_tree"));
  gtk_clist_freeze(clist);
  for (i1 = 0; i1 < MIME_SIZE; i1++) {
    for (dlist = global.mimetype[i1].shared; dlist; dlist = dlist->next) {
      text = (char*)(dlist->data);
#ifdef SHARE_DEBUG
      printf("adding [%s]\n", text);
#endif
      add_shared_directory(i1, text);
    }
  }
  gtk_clist_thaw(clist);
  
  if (global.auto_save) {
    text = strdup(global.auto_save);
    lib_save(text);
    free(text);
  }
  //  lib_set_modified();

  gtk_idle_add(lib_remove_idle, NULL);
  
  return;
}

file_t* file_create_from_local(char* longname) {
  file_t* new_file;
  char* pos1;
  int mime;

  mime = get_mimetype(longname);
  new_file = (file_t*)malloc(sizeof(file_t));
  
  new_file->mime_type = mime;
  new_file->longname = strdup(longname);
  pos1 = extract_short_name(new_file->longname);
  new_file->shortname = strdup(pos1);
  new_file->winname = strdup(new_file->longname);
  convert_to_win(new_file->winname);

  new_file->size = 0;
  new_file->user = strdup("");
  new_file->ip = 0;
  
  new_file->md5 = strdup("00000000000000000000000000000000");

  new_file->bitrate = 32;
  new_file->frequency = 32000;
  new_file->duration = 60;
  new_file->linespeed = 0;
  new_file->flags = 0;
  new_file->status = FILE_NONE;
  new_file->local = 1;
  new_file->vbr = 0;

  return new_file;
}

void destroy_dir_row(gpointer data) {
  char* dir;

  dir = (char*)data;
  free(dir);
}

void lib_insert_file_real(file_t *file) {
  GtkCTree* ctree;
  GtkWidget* temp;
  char* text;
  char t[500];
  char* pos;
  GtkCTreeNode *node;
  GtkCTreeNode *node2 = NULL;
  char dir[1024];
  int tt;

  //----------------------- new hotlist ctree -------------------------

  if (!file) return;
  
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));

  tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 0;

  temp = lookup_widget(global.win, "checkbutton19");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp))) {
    strcpy(t, file->shortname);
    tt = strlen(file->longname)-strlen(file->shortname);
    strncpy(dir, file->longname, tt);
    dir[tt] = 0;
    if (tt > 0 && dir[tt-1] == '/') dir[tt-1] = 0;
  } else {
    strcpy(t, file->longname);
    dir[0] = 0;
  }

  /////////////////
  pos = strtok(t, "/");
  node = NULL;
  while (pos) {
    if (strlen(pos) <= 0) {
      pos = strtok(NULL, "/");
      continue;
    }
    node2 = node;
    if (!node) node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list);
    else node = GTK_CTREE_ROW(node)->children;
    while (node) {
      text = GTK_CELL_PIXTEXT 
	(GTK_CTREE_ROW (node)->row.cell[ctree->tree_column])->text;
      if (strcmp(pos, text) == 0) break;
      node = GTK_CTREE_ROW(node)->sibling;
    }
    if (!node) break;
    if (GTK_CTREE_ROW (node)->is_leaf) break;
    strcat(dir, "/");
    strcat(dir, pos);
    pos = strtok(NULL, "/");
  }
  
  // do not insert identical files
  /*
  if (node && GTK_CTREE_ROW (node)->is_leaf) {
    file2 = (file_t*)gtk_ctree_node_get_row_data(ctree, node);    
    if (!strcmp(file2->longname, file->longname)) return 0;
  }
  */
  
  node = node2;
  do {
    strcpy(tstr[0], pos);
    strcat(dir, "/");
    strcat(dir, pos);
    pos = strtok(NULL, "/");
    if (pos) {
      node = 
	gtk_ctree_insert_node(ctree, node, NULL, list, 5,
			      global.pix.folder, 
			      global.pix.folderb,
			      global.pix.folder_open, 
			      global.pix.folder_openb,
			      FALSE, FALSE);
      gtk_ctree_node_set_row_data_full(ctree, node, (gpointer)strdup(dir),
				       (GtkDestroyNotify)destroy_dir_row);
    } else {
      sprintf(tstr[1], "%.2f MB", (double)(file->size)/1024/1024);
      if (file->vbr) sprintf(tstr[2], "av. %d", file->bitrate);
      else sprintf(tstr[2], "%d", file->bitrate);
      sprintf(tstr[3], "%d", file->frequency);
      print_time_short(tstr[4], file->duration);
      
      node = gtk_ctree_insert_node(ctree,
				   node, NULL, list, 5,
				   NULL, NULL, NULL, NULL,
				   TRUE, FALSE);
      
      gtk_ctree_node_set_row_data_full(ctree, node, (gpointer)file,
				       (GtkDestroyNotify)destroy_file_row);
    }
  } while (pos);
}

void lib_insert_file(file_t *file) {

  //----------------------- new hotlist ctree -------------------------

  if (!file) return;
  lib_insert_file_real(file);

#ifdef SHARE_DEBUG
  printf("adding [%s] %lu\n", file->longname, file->size);
#endif
  global.user.shared++;
  global.user.unshared++;
  global.user.bytes += file->size;
  if ((global.options.login_mode & L_NOT_SHARE) == 0) {
    file->flags |= FLAG_TO_SHARE;
  }

  global.user.files = g_list_append(global.user.files, file); 
}

void lib_tree_update_file(file_t *file) {
  GtkCTree* ctree;
  GtkCTreeNode* node;
  style_t* style;

  style = style_get(global.scheme, "library_unshared");

  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, file);  
  if (node == NULL) return;

  sprintf(tstr[0], "%.2f MB", (double)(file->size)/1024/1024);
  gtk_ctree_node_set_text(ctree, node, 1, tstr[0]);
  if (file->vbr) sprintf(tstr[0], "av. %d", file->bitrate);
  else sprintf(tstr[0], "%d", file->bitrate);
  gtk_ctree_node_set_text(ctree, node, 2, tstr[0]);
  sprintf(tstr[0], "%d", file->frequency);
  gtk_ctree_node_set_text(ctree, node, 3, tstr[0]);
  sprintf(tstr[0], "%d:%s%d", file->duration/60, 
	  (file->duration%60)<10?"0":"", file->duration%60);
  gtk_ctree_node_set_text(ctree, node, 4, tstr[0]);

  if (!style) return;
  
  if (file->flags & FLAG_SHARED) {
    gtk_ctree_node_set_foreground(ctree, node, NULL);
    gtk_ctree_node_set_background(ctree, node, NULL);
  } else {
    gtk_ctree_node_set_foreground(ctree, node, style->fore);
    gtk_ctree_node_set_background(ctree, node, style->back);
  }
}


void rec_set_flag (GtkCTree *ctree, GtkCTreeNode *node,
		   int* data) {
  file_t* file;
  
  if (!GTK_CTREE_ROW (node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  file->flags |= (*data);
}

void lib_set_flag(int val, GtkCTreeNode* node) {
  GtkCTree* ctree;
  
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, node, 
			   (GtkCTreeFunc)rec_set_flag, &val);
}

void rec_del_flag (GtkCTree *ctree, GtkCTreeNode *node,
		   int* data) {
  file_t* file;
  
  if (!GTK_CTREE_ROW (node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  file->flags &= (0xff^(*data));
}

void lib_del_flag(int val, GtkCTreeNode* node) {
  GtkCTree* ctree;

  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, node, 
			   (GtkCTreeFunc)rec_del_flag, &val);
}

void rec_replace_flag (GtkCTree *ctree, GtkCTreeNode *node,
		       int* data) {
  file_t* file;
  char* vals;

  vals = (char*)data;

  if (!GTK_CTREE_ROW (node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (file->flags & vals[0]) {
    file->flags &= (0xff^(vals[0]));
    file->flags |= vals[1];
    //    global.user.unshared++;
  }
}

void lib_replace_flag(int val1, int val2, GtkCTreeNode* node) {
  GtkCTree* ctree;
  char data[2];

  data[0] = val1;
  data[1] = val2;

  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, node, 
			   (GtkCTreeFunc)rec_replace_flag, data);
}

void rec_xor_flag (GtkCTree *ctree, GtkCTreeNode *node,
		   int* data) {
  file_t* file;
  char* vals;

  vals = (char*)data;

  if (!GTK_CTREE_ROW (node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (file->flags & vals[0]) {
    file->flags &= (0xff^vals[1]);
  } else {
    file->flags |= vals[1];
  }
}

void lib_xor_flag(int val1, int val2, GtkCTreeNode* node) {
  GtkCTree* ctree;
  char data[2];

  data[0] = val1;
  data[1] = val2;

  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, node, 
			   (GtkCTreeFunc)rec_xor_flag, data);
}

file_t* lib_search_file(char* longname) {
  GList* dlist;
  file_t* file;
  
  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    if (!strcmp(longname, file->longname)) return file;
  }
  return NULL;
}

void lib_search(search_pattern_t* search) {
  GList* dlist;
  file_t* file;
  int cnt = 0;

  if (global.status.building > 1) return;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    if (search_pattern_fits_file(search, file, 1)) {
      file = file_dup(file);
      file_insert_search(file, DEST_LIBRARY);
      cnt++;
    }
    if (cnt >= search->max_results) return;
  }
}

void lib_sort() {
  GtkCTree* ctree;

#ifdef SHARE_DEBUG
  printf("sorting\n");
#endif
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));

  if (global.user.shared < 5000) {
    gtk_clist_freeze(GTK_CLIST(ctree));
    gtk_ctree_sort_recursive(ctree, NULL);
    gtk_clist_thaw(GTK_CLIST(ctree));
  }
}

file_t* load_file_entry(FILE* fd) {
  char* filename;
  char* md5;
  char* size;
  char* bitrate;
  char* freq;
  char* duration;
  char* mime;

  char line[2048];
  file_t* file;
  
  if (!fgets(line, 2048-1, fd)) return NULL;

  file = (file_t*)malloc(sizeof(file_t));
  filename = arg(line, 0);
  md5 = arg(NULL, 0);
  size = arg(NULL, 0);
  bitrate = arg(NULL, 0);
  freq = arg(NULL, 0);
  duration = arg(NULL, 0);
  mime = arg(NULL, 0);

  if (!filename || ! md5 || !size || !bitrate || !freq || !duration) return NULL;

  file->longname = strdup(filename);
  file->shortname = strdup(extract_short_name(file->longname));
  file->winname = strdup(file->longname);
  convert_to_win(file->winname);
  file->md5 = strdup(md5);
  file->size = strtoul(size, NULL, 10);
  if (!strncmp(bitrate, "vbr", 3)) {
    file->bitrate = atoi(bitrate+3);
    file->vbr = 1;
  } else {
    file->bitrate = atoi(bitrate);
    file->vbr = 0;
  }
  file->frequency = atoi(freq);
  file->duration = atoi(duration);
  if (!mime) file->mime_type = MIME_MP3;
  else file->mime_type = atoi(mime);
  file->user = strdup("");
  file->ip = 0;
  file->linespeed = 0;
  file->flags = 0;
  file->status = FILE_NONE;
  file->local = 1;            //local file

  return file;
}

gint lib_load_idle(gpointer data) {
  FILE* fd = (FILE*)data;
  GtkCList* clist;
  file_t* file;
  file_t* file2;
  int i1;

#ifdef SHARE_DEBUG
  printf("load idle\n");
#endif  
  
  for (i1 = 0; i1 < 10; i1++) {
    if (!(file = load_file_entry(fd))) goto finish;
    file2 = lib_search_file(file->longname);
    if (!file2) {
      lib_insert_file(file);
    } else {
      file2->flags &= 0xff^FLAG_TO_REMOVE;
      destroy_file_row(file);
    }
  }

  return 1;
  
 finish:
  global.status.building = 0;
  fclose(fd);
  clist = GTK_CLIST(lookup_widget(global.win, "lib_tree"));
  gtk_clist_thaw(clist);
  gtk_idle_add(lib_remove_idle, NULL);

  return 0;
}

int lib_load(char* fname) {
  char filename[1024];
  FILE* fd = NULL;
  GtkCList* clist;
  GtkWidget* temp;
  char t[1024];

  if (!fname || !(*fname)) return 0;
  strcpy(t, fname);

#ifdef SHARE_DEBUG
  printf("building %d\n", global.status.building);
#endif
  if (global.status.building > 1) return 0;

#ifdef SHARE_DEBUG
  printf("lib_load\n");
#endif  
  sprintf(filename, "%s/share/%s", global.lopster_home, fname);
  fd = fopen(filename, "r");
  if (!fd) return 0;
    
  clist = GTK_CLIST(lookup_widget(global.win, "lib_tree"));
  gtk_clist_freeze(clist);
  global.status.building = 2;
  lib_set_flag(FLAG_TO_REMOVE, NULL);

  temp = lookup_widget(global.win, "entry84");
  gtk_entry_set_text(GTK_ENTRY(temp), fname);
  if (global.auto_save) free(global.auto_save);
  global.auto_save = strdup(t);

  gtk_idle_add(lib_load_idle, (gpointer)fd);
  /*
    gdk_input_add(fileno(fd), GDK_INPUT_READ,
		GTK_SIGNAL_FUNC(lib_load_func), fd);
  */
  return 1;
}

void lib_save(char* name) {
  char filename[1024];
  FILE* fd;
  GList* dlist;
  file_t* file;
  
  if (!name || !(*name)) return;

  sprintf(filename, "%s/share/%s", global.lopster_home, name);

  if ((fd = fopen(filename, "w")) == NULL) {
    g_warning(_("could not save shared files %s"), filename);
    return;
  }
  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    fprintf(fd, "\"%s\" %s %ld %s%d %d %d %d\n", 
	    file->longname, file->md5, file->size, file->vbr?"vbr":"",
	    file->bitrate, file->frequency, file->duration, file->mime_type);
  }
  fclose(fd);

  if (global.auto_save) free(global.auto_save);
  global.auto_save = strdup(name);
}

void lib_delete(char* name) {
  char filename[1024];
  
  if (!name || !(*name)) return;

  sprintf(filename, "%s/share/%s", global.lopster_home, name);
  unlink(filename);
}

void
on_delete_list_activate (GtkMenuItem     *menuitem,
			 gpointer         user_data) {
  
  GtkLabel* label;
  char* s1;

  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  s1 = label->label;
  lib_delete(s1);
}

void
on_load_list_activate (GtkMenuItem     *menuitem,
		       gpointer         user_data) {
  
  GtkLabel* label;
  char* s1;

  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  s1 = label->label;
  lib_load(s1);
}

GtkWidget* create_files_popup() {
  GtkWidget *popup;
  GtkWidget *menu1 = NULL;
  GtkWidget *menu2 = NULL;
  GtkWidget* item;
  GtkWidget* item1;
  GtkWidget* item2;
  GtkAccelGroup *popup_accels;
  char dirname[2048];
  struct dirent* entry;
  struct stat buf;
  char filename[2048];
  DIR* dir;
  int cnt = 0;

  sprintf(dirname, "%s/share", global.lopster_home);

  popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (popup));

  item1 = gtk_menu_item_new_with_label (_("Delete List"));
  gtk_widget_show (item1);
  gtk_container_add (GTK_CONTAINER (popup), item1);

  item2 = gtk_menu_item_new_with_label (_("Load List"));
  gtk_widget_show (item2);
  gtk_container_add (GTK_CONTAINER (popup), item2);

  if ((dir = opendir(dirname)) == NULL) {
    g_warning("could not open dir [%s]", dirname);
  } else {
    while ((entry = readdir(dir)) != NULL) {
      //      if (!strncmp(entry->d_name, ".", 1)) continue;
      //      if (!strncmp(entry->d_name, "..", 2)) continue;
      sprintf(filename, "%s/%s", dirname, entry->d_name);
      stat(filename, &buf);
      if (S_ISREG(buf.st_mode)) {
	if (!cnt) {  
	  menu1 = gtk_menu_new ();
	  gtk_widget_show (menu1);
	  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item1), menu1);
	  menu2 = gtk_menu_new ();
	  gtk_widget_show (menu2);
	  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item2), menu2);
	}
	item = gtk_menu_item_new_with_label (entry->d_name);
	gtk_widget_show (item);
	gtk_container_add (GTK_CONTAINER (menu1), item);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    on_delete_list_activate,
			    NULL);
	item = gtk_menu_item_new_with_label (entry->d_name);
	gtk_widget_show (item);
	gtk_container_add (GTK_CONTAINER (menu2), item);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    on_load_list_activate,
			    NULL);
	cnt++;
      }
    }
    closedir(dir);
  }
  
  if (!cnt) {
    gtk_widget_set_sensitive(item1, FALSE);
    gtk_widget_set_sensitive(item2, FALSE);
  }

  return popup;
}

void lib_update_stats() {
  GtkWidget* temp;
  char comm[1024];

  temp = lookup_widget(global.win, "label233");
  sprintf(comm, "%d/%d", global.user.shared-global.user.unshared, 
	  global.user.shared);
  gtk_label_set_text(GTK_LABEL(temp), comm);
  temp = lookup_widget(global.win, "label235");
  sprintf(comm, "%.0f", global.user.bytes / 1048576.);
  gtk_label_set_text(GTK_LABEL(temp), comm);

}

GtkWidget*
create_lib_popup2 (void)
{
  GtkWidget *lib_popup;
  GtkAccelGroup *lib_popup_accels;
  GtkWidget *item;
  GtkWidget *separator;
  GtkCTree* ctree;
  GtkCTreeNode* node;
  file_t* file;

  ctree = GTK_CTREE(global.popup_list);
  node = gtk_ctree_node_nth(ctree, global.popup_row);
  if (node) {
    file = (file_t*)gtk_ctree_node_get_row_data(ctree, node);
  } else {
    file = NULL;
  }

  lib_popup = gtk_menu_new ();
  gtk_container_set_border_width (GTK_CONTAINER (lib_popup), 2);
  lib_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (lib_popup));
  
  if (file) {
    item = gtk_menu_item_new_with_label (_("Open File"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_play_file2_activate),
			NULL);
    
    separator = gtk_menu_item_new ();
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (lib_popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);
  }
  item = gtk_menu_item_new_with_label (_("Refresh Shared List"));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (lib_popup), item);
  gtk_signal_connect (GTK_OBJECT (item), "activate",
                      GTK_SIGNAL_FUNC (on_refresh1_activate),
                      NULL);

  separator = gtk_menu_item_new ();
  gtk_widget_show (separator);
  gtk_container_add (GTK_CONTAINER (lib_popup), separator);
  gtk_widget_set_sensitive (separator, FALSE);

  if (file) {
    item = gtk_menu_item_new_with_label (_("Remove File"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_remove_activate),
			NULL);
    
    if (file->flags & FLAG_SHARED)
      item = gtk_menu_item_new_with_label (_("Unshare File"));
    else
      item = gtk_menu_item_new_with_label (_("Share File"));
    
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_unshare_activate),
			NULL);
    
    separator = gtk_menu_item_new ();
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (lib_popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);
    
    item = gtk_menu_item_new_with_label (_("Remove All"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_remove_all_activate),
			NULL);
    
    item = gtk_menu_item_new_with_label (_("Unshare All"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_unshare_all_activate),
			NULL);
    
    item = gtk_menu_item_new_with_label (_("Share All"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_share_all_activate),
			NULL);

    separator = gtk_menu_item_new ();
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (lib_popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);
    
    item = gtk_menu_item_new_with_label (_("Delete File (from Disk)"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_delete_file_activate),
			NULL);
    
    separator = gtk_menu_item_new ();
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (lib_popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);
  }
  item = gtk_menu_item_new_with_label (_("Customize List"));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (lib_popup), item);
  gtk_signal_connect (GTK_OBJECT (item), "activate",
                      GTK_SIGNAL_FUNC (on_customize_list_activate),
                      NULL);

  return lib_popup;
}

void lib_set_modified() {
  GtkWidget* temp;

  temp = lookup_widget(global.win, "entry84");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  if (global.auto_save) free(global.auto_save);
  global.auto_save = NULL;
}


void lib_compare_with_browse(user_t* userinfo) {
  GList* dlist;
  file_t* file;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    if (!browse_search_file(userinfo, file->winname) &&
	(file->flags & FLAG_SHARED)) {
      file->flags &= 0xff^FLAG_SHARED;
      global.user.unshared++;
    }
  }
  gtk_idle_add(lib_update_idle, NULL);
}
