#include <sys/types.h>
#include <sys/socket.h>
#include <dirent.h>
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "callbacks.h"
#include "connection.h"
#include "global.h"
#include "support.h"
#include "search.h"
#include "browse.h"
#include "share.h"
#include "handler.h"

/*
  load -------------.
                     >-- commit
  refresh -- remove-
*/

file_t* global_file;
GList* cur_file;

void lib_sort();
file_t* lib_search_file(char* longname);
void lib_insert_file(file_t *file);
void lib_tree_update_file(file_t *file);


unsigned int get_bits(unsigned char* buffer, int start, int no) {
  unsigned int res = 0;

  res |= buffer[0]<<24 | buffer[1]<<16 |
    buffer[2]<<8 | buffer[3];
  
  res <<= start;
  res >>= (32-no);
  return res;
}

void file_parse_mp3_header(file_t *file) {
  FILE* fd;
  unsigned char buffer[4];  // header
  int mpeg_version;
  int mpeg_layer;
  int res;
  long framesize;
  long totalframes;
  short br[2][4][16] = {
    {  // mpeg 2
      { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
      { 0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, 0 }, // 
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0 },      // 
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0 }       // 
    },
    {  // mpeg1
      { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
      { 0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 0 },     // layer III
      { 0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384, 0 },    // layer II
      { 0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, 0 }  // layer I
    }
  };

  int freq[2][4] = {
    {22050, 24000, 16000, 0},
    {44100, 48000, 32000, 0}
  };

  if ((fd = fopen(file->longname, "r")) != NULL) {
    if (fread(buffer, sizeof(char), 4, fd) != 4) {
      //      printf("could not read header [%s]\n", file->longname);
      return;
    }
    // i think this is not correct sometimes
    while (get_bits(buffer, 0, 11) != 0x7ff) {
      buffer[0] = buffer[1];
      buffer[1] = buffer[2];
      buffer[2] = buffer[3];
      if (fread(&buffer[3], sizeof(char), 1, fd) != 1) {
	//	printf("could not read header [%s]\n", file->longname);
	return;
      }
    }
  } else {
//    printf("could not open file for reading header [%s]\n", file->longname);
    return;
  }

  // sync word
  res = get_bits(buffer, 0, 12);
  
  // MPEG Version
  mpeg_version = get_bits(buffer, 12, 1);

  // MPEG Layer
  mpeg_layer = get_bits(buffer, 13, 2);

  // Protection Bit
  res = get_bits(buffer, 15, 1);

  // Bitrate
  res = get_bits(buffer, 16, 4);
  file->bitrate = br[mpeg_version][mpeg_layer][res];

  // Fequency
  res = get_bits(buffer, 20, 2);
  file->frequency = freq[mpeg_version][res];
  if ((file->bitrate == 0) || (file->frequency == 0)) {
    //    printf("corrupted header! [%s]\n", file->longname);
    return;
  }

  framesize = 144000 * file->bitrate / file->frequency;
  totalframes = (file->size / (framesize + 1)) - 1;
  file->duration = (time_t) (totalframes * 1152 / file->frequency);

  // Padding Bit
  res = get_bits(buffer, 22, 1);
  
  // Private Bit
  res = get_bits(buffer, 23, 1);
  
  // Mode
  res = get_bits(buffer, 24, 2);
  
  // Mode Extention
  res = get_bits(buffer, 26, 2);
  
  // Copyright
  res = get_bits(buffer, 28, 1);
  
  // Original Home
  res = get_bits(buffer, 29, 1);
  
  // Enphasis
  res = get_bits(buffer, 30, 2);
  
  fclose(fd);
}

void file_parse_header(file_t *file) {
  if (file->mime_type == MIME_MP3) {
    file_parse_mp3_header(file);
  } else {
    file->bitrate = 8;
    file->frequency = 0;
    file->duration = 0;
  }
}

void file_calc_md5(file_t *file) {
  //  file->md5 = MD5File(file->longname, file->md5);
  file->md5 = strdup("NOTCOMPUTED");
}

gint
lib_compare (GtkCList      *clist,
	     gconstpointer  ptr1,
	     gconstpointer  ptr2) {
  char *text1 = NULL;
  char *text2 = NULL;

  GtkCListRow *row1 = (GtkCListRow *) ptr1;
  GtkCListRow *row2 = (GtkCListRow *) ptr2;

  text1 = GTK_CELL_TEXT (row1->cell[clist->sort_column])->text;
  text2 = GTK_CELL_TEXT (row2->cell[clist->sort_column])->text;

  if (!text2)
    return (text1 != NULL);

  if (!text1)
    return -1;

  if (gtk_events_pending()) gtk_main_iteration();
  
  return strcasecmp (text1, text2);
}

suffix_t* check_suffix(char* name, int mime) {
  GList* dlist;
  char* suffix;
  suffix_t* suf;

  suffix = strrchr(name, '.');
  if (!suffix) return NULL;
  else suffix++;

  for (dlist = global.mimetype[mime].suffixes; dlist; dlist=dlist->next) {
    suf = (suffix_t*)(dlist->data);
    if (!strcasecmp(suffix, suf->suffix)) return suf;
  }
  return NULL;
}

void lib_commit_file (file_t* file) {
  char t2[2048];
  char t[1024];

  if (global.status.connection < 2) return;

  if ((file->flags & FLAG_TO_SHARE) &&
      ((file->flags & FLAG_SHARED) == 0)) {
    if (file->mime_type == MIME_MP3) {
      strcpy(t2, "\"");
      strcat(t2, file->winname);
      strcat(t2, "\" ");
      strcat(t2, file->md5);
      sprintf(t, " %ld %d %d %d",
	      file->size, file->bitrate, 
	      file->frequency, file->duration);
      strcat(t2, t);
      send_command(CMD_CLIENT_ADD_FILE, t2);
      file->flags &= 0xff^FLAG_TO_SHARE;
      file->flags |= FLAG_SHARED;
      global.user.unshared--;
      lib_tree_update_file(file);
    } else if (opennap_version(0, 37)) {
      strcpy(t2, "\"");
      strcat(t2, file->winname);
      strcat(t2, "\"");
      sprintf(t, " %ld %s %s",
	      file->size, file->md5, MimeNames_(file->mime_type));
      strcat(t2, t);
      send_command(CMD_CLIENT_SHARE_FILE, t2);
      file->flags &= 0xff^FLAG_TO_SHARE;
      file->flags |= FLAG_SHARED;
      global.user.unshared--;
      lib_tree_update_file(file);
    }
  }
}

gint lib_commit_idle(gpointer data) {
  file_t* file;
  int i1;

#ifdef SHARE_DEBUG
  printf("commit idle %d %d\n", global.status.building,
	 global.status.connection);
#endif
  if (global.status.building > 1) return 0;
  if (global.status.connection < 2) return 0;

  if (!cur_file) {
    if (global.options.login_mode & L_NOT_SHARE) {
      global.options.login_mode &= 0xff^L_NOT_SHARE;
      return 0;
    }
    cur_file = global.user.files;
  }

  for (i1 = 0; i1 < 100; i1++) {
    if (!cur_file) break;
    file = (file_t*)(cur_file->data);
    lib_commit_file(file);
    cur_file = cur_file->next;
  }
  
  if (cur_file) return 1;

  lib_update_stats();

  return 0;
}

void lib_update_file (file_t* file) {
  struct stat stats;
  int size;

  if (stat(file->longname, &stats) < 0) {
    size = file->size;
  } else {
    size = stats.st_size;
  }
  if (file->size != size) {
    global.user.bytes -= file->size;
    file->size = size;
    global.user.bytes += file->size;
    //    printf("parsing header\n");
    file_parse_header(file);
    file_calc_md5(file);
  }
  lib_tree_update_file(file);
}

int lib_update_idle(gpointer data) {
  file_t* file;
  int i1;

#ifdef SHARE_DEBUG
  printf("update idle\n");
#endif
  if (!cur_file) cur_file = global.user.files;

  for (i1 = 0; i1 < 10; i1++) {
    if (!cur_file) break;
    file = (file_t*)(cur_file->data);
    lib_update_file(file);
    cur_file = cur_file->next;
  }
  
  if (cur_file) return 1;
  
  global.status.building = 0;
  lib_update_stats();

  gtk_idle_add(lib_commit_idle, NULL);
  return 0;
}

void lib_remove_file(file_t* file) {
  char* command;
  GtkCTreeNode* node;
  GtkCTreeNode* node2;
  GtkCTree* ctree;

#ifdef SHARE_DEBUG
  printf("removing [%s]\n", file->longname);
#endif
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  if (file->flags & FLAG_SHARED) {
    // unsharing
    if (global.status.connection > 1) {
      if (opennap_version(0, 37) || !opennap_version(0,0))
	command = g_strdup_printf("\"%s\"", file->winname);
      else
	command = g_strdup_printf("%s", file->winname);
      send_command(CMD_CLIENT_REMOVE_FILE, command);
      g_free(command);
    }
    global.user.unshared++;
  }
  if (file->flags & FLAG_TO_DELETE) {
    unlink(file->longname);
  }

  if ((file->flags & FLAG_TO_DELETE) ||
      (file->flags & FLAG_TO_REMOVE)) {
    global.user.shared--;
    global.user.unshared--;
    global.user.bytes -= file->size;
    global.user.files = g_list_remove(global.user.files, file);
    node = gtk_ctree_find_by_row_data(ctree, NULL, file);
    while (node) {
      node2 = GTK_CTREE_ROW(node)->parent;
      gtk_ctree_remove_node(ctree, node);
      if (!node2 || (GTK_CTREE_ROW(node2)->children != NULL)) break;
      node = node2;
    }
  } else {
    file->flags &= 0xff^(FLAG_SHARED|FLAG_TO_UNSHARE);
    lib_tree_update_file(file);
  }
}

gint lib_remove_idle(gpointer data) {
  file_t* file;
  int i1;
  GtkCList* clist;

  global.status.building = 2;
  if (!cur_file) cur_file = global.user.files;
  
  clist = GTK_CLIST(lookup_widget(global.win, "lib_tree"));
  gtk_clist_freeze(clist);

  for (i1 = 0; i1 < 10; i1++) {
    if (!cur_file) break;
    file = (file_t*)(cur_file->data);
    
    cur_file = cur_file->next;
    if ((file->flags & FLAG_TO_REMOVE) ||
	(file->flags & FLAG_TO_DELETE) ||
	(file->flags & FLAG_TO_UNSHARE))
      lib_remove_file(file);
  }
  gtk_clist_thaw(clist);
  if (cur_file) return 1;

  // now updating stats
  lib_update_stats();

  lib_sort();

  gtk_idle_add(lib_update_idle, NULL);

  return 0;
}

int add_shared_directory(int mime, char* dirname) {
  DIR* dir;
  struct dirent* entry;
  struct stat buf;
  char name[2048];
  file_t* file;
  
  if ((dir = opendir(dirname)) == NULL) {
#ifdef SHARE_DEBUG
    printf("could not open dir [%s] (add_shared)\n", dirname);
#endif
    return 0;
  }

  while ((entry = readdir(dir)) != NULL) {
    if (!strncmp(entry->d_name, ".", 1)) continue;
    if (!strncmp(entry->d_name, "..", 2)) continue;
    
    strcpy(name, dirname);
    strcat(name, "/");
    strcat(name, entry->d_name);
    stat(name, &buf);
    if (buf.st_mode&S_IFDIR) {
      // links are followed
      //      printf("mode : %s:%x:%x\n", name, buf.st_mode, S_IFLNK);
      //      if ((buf.st_mode&S_IFLNK) == 0)
      add_shared_directory(mime, name);
    } else {
      if (check_suffix(entry->d_name, mime)) {
	file = lib_search_file(name);
	if (!file) {
	  file = file_create_from_local(name);
	  if (file) lib_insert_file(file);
	} else {
#ifdef SHARE_DEBUG
	  printf("updating [%s]\n", file->longname);
#endif
	  file->flags &= 0xff^FLAG_TO_REMOVE;
	}
      }
      if (gtk_events_pending()) gtk_main_iteration();
    }
  }
  closedir(dir);

  return 1;
}

void lib_refresh() {
  int i1 = 0;
  GList* dlist = NULL;
  GtkCList* clist;
  GtkWidget* temp;
  char* text;

#ifdef SHARE_DEBUG
  printf("refresh\n");
#endif
  // already buidling?
  if (global.status.building > 1) return;

  global.status.building = 2;

  lib_set_flag(FLAG_TO_REMOVE, NULL);

  temp = lookup_widget(global.win, "entry84");
  gtk_entry_set_text(GTK_ENTRY(temp), "");

  clist = GTK_CLIST(lookup_widget(global.win, "lib_tree"));
  gtk_clist_freeze(clist);
  for (i1 = 0; i1 < MIME_SIZE; i1++) {
    for (dlist = global.mimetype[i1].shared; dlist; dlist = dlist->next) {
      text = (char*)(dlist->data);
#ifdef SHARE_DEBUG
      printf("adding [%s]\n", text);
#endif
      add_shared_directory(i1, text);
    }
  }
  gtk_clist_thaw(clist);
  
  lib_set_modified();

  gtk_idle_add(lib_remove_idle, NULL);
  
  return;
}

file_t* file_create_from_local(char* longname) {
  file_t* new_file;
  char* pos1;
  int mime;

  mime = get_mimetype(longname);
  if (mime == 0) {
    g_warning(_("file has unknown mime type"));
    return NULL;
  }

  new_file = (file_t*)malloc(sizeof(file_t));
  
  new_file->mime_type = mime;
  new_file->longname = strdup(longname);
  pos1 = extract_short_name(new_file->longname);
  new_file->shortname = strdup(pos1);
  new_file->winname = strdup(new_file->longname);
  convert_to_win(new_file->winname);

  new_file->size = 0;
  new_file->user = strdup("");
  
  new_file->ip = 0;
  new_file->md5 = strdup("NOTCOMPUTED");;

  new_file->bitrate = 32;
  new_file->frequency = 32000;
  new_file->duration = 60;
  new_file->linespeed = 0;
  new_file->flags = 0;
  new_file->status = FILE_NONE;
  new_file->local = 1;

  return new_file;
}

void destroy_dir_row(gpointer data) {
  char* dir;

  dir = (char*)data;
  free(dir);
}

void lib_insert_file_real(file_t *file) {
  GtkCTree* ctree;
  GtkWidget* temp;
  char* text;
  char t[500];
  char* pos;
  GtkCTreeNode *node;
  GtkCTreeNode *node2 = NULL;
  char dir[1024];
  int tt;

  //----------------------- new hotlist ctree -------------------------

  if (!file) return;
  
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));

  tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 0;

  temp = lookup_widget(global.win, "checkbutton19");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp))) {
    strcpy(t, file->shortname);
    tt = strlen(file->longname)-strlen(file->shortname);
    strncpy(dir, file->longname, tt);
    dir[tt] = 0;
    if (tt > 0 && dir[tt-1] == '/') dir[tt-1] = 0;
  } else {
    strcpy(t, file->longname);
    dir[0] = 0;
  }
  
  /////////////////
  pos = strtok(t, "/");
  node = NULL;
  while (pos) {
    if (strlen(pos) <= 0) {
      pos = strtok(NULL, "/");
      continue;
    }
    node2 = node;
    if (!node) node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list);
    else node = GTK_CTREE_ROW(node)->children;
    while (node) {
      text = GTK_CELL_PIXTEXT 
	(GTK_CTREE_ROW (node)->row.cell[ctree->tree_column])->text;
      if (strcmp(pos, text) == 0) break;
      node = GTK_CTREE_ROW(node)->sibling;
    }
    if (!node) break;
    if (GTK_CTREE_ROW (node)->is_leaf) break;
    strcat(dir, "/");
    strcat(dir, pos);
    pos = strtok(NULL, "/");
  }
  
  // do not insert identical files
  /*
  if (node && GTK_CTREE_ROW (node)->is_leaf) {
    file2 = (file_t*)gtk_ctree_node_get_row_data(ctree, node);    
    if (!strcmp(file2->longname, file->longname)) return 0;
  }
  */
  
  node = node2;
  do {
    strcpy(tstr[0], pos);
    strcat(dir, "/");
    strcat(dir, pos);
    pos = strtok(NULL, "/");
    if (pos) {
      node = 
	gtk_ctree_insert_node(ctree, node, NULL, list, 5,
			      global.pix.folder, 
			      global.pix.folderb,
			      global.pix.folder_open, 
			      global.pix.folder_openb,
			      FALSE, FALSE);
      gtk_ctree_node_set_row_data_full(ctree, node, (gpointer)strdup(dir),
				       (GtkDestroyNotify)destroy_dir_row);
    } else {
      sprintf(tstr[1], "%.2f MB", (double)(file->size)/1024/1024);
      sprintf(tstr[2], "%d", file->bitrate);
      sprintf(tstr[3], "%d", file->frequency);
      sprintf(tstr[4], "%d:%s%d", file->duration/60, 
	      (file->duration%60)<10?"0":"", file->duration%60);
      
      node = gtk_ctree_insert_node(ctree,
				   node, NULL, list, 5,
				   NULL, NULL, NULL, NULL,
				   TRUE, FALSE);
      
      gtk_ctree_node_set_row_data_full(ctree, node, (gpointer)file,
				       (GtkDestroyNotify)destroy_file_row);
    }
  } while (pos);
}

void lib_insert_file(file_t *file) {

  //----------------------- new hotlist ctree -------------------------

  if (!file) return;
  lib_insert_file_real(file);

#ifdef SHARE_DEBUG
  printf("adding [%s]\n", file->longname);
#endif
  global.user.shared++;
  global.user.unshared++;
  global.user.bytes += file->size;
  file->flags |= FLAG_TO_SHARE;

  global.user.files = g_list_append(global.user.files, file); 
}

void lib_tree_update_file(file_t *file) {
  GtkCTree* ctree;
  GtkCTreeNode* node;
  static GdkColor color = { 0, 0xeeee, 0xeeee, 0xeeee };

  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  node = gtk_ctree_find_by_row_data(ctree, NULL, file);  
  if (node == NULL) return;

  sprintf(tstr[0], "%.2f MB", (double)(file->size)/1024/1024);
  gtk_ctree_node_set_text(ctree, node, 1, tstr[0]);
  sprintf(tstr[0], "%d", file->bitrate);
  gtk_ctree_node_set_text(ctree, node, 2, tstr[0]);
  sprintf(tstr[0], "%d", file->frequency);
  gtk_ctree_node_set_text(ctree, node, 3, tstr[0]);
  sprintf(tstr[0], "%d:%s%d", file->duration/60, 
	  (file->duration%60)<10?"0":"", file->duration%60);
  gtk_ctree_node_set_text(ctree, node, 4, tstr[0]);

  if (file->flags & FLAG_SHARED)
    gtk_ctree_node_set_background(ctree, node, NULL);
  else
    gtk_ctree_node_set_background(ctree, node, &color);
}


void rec_set_flag (GtkCTree *ctree, GtkCTreeNode *node,
		   int* data) {
  file_t* file;
  
  if (!GTK_CTREE_ROW (node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  file->flags |= (*data);
}

void lib_set_flag(int val, GtkCTreeNode* node) {
  GtkCTree* ctree;
  
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, node, 
			   (GtkCTreeFunc)rec_set_flag, &val);
}

void rec_del_flag (GtkCTree *ctree, GtkCTreeNode *node,
		   int* data) {
  file_t* file;
  
  if (!GTK_CTREE_ROW (node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  file->flags &= (0xff^(*data));
}

void lib_del_flag(int val, GtkCTreeNode* node) {
  GtkCTree* ctree;

  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, node, 
			   (GtkCTreeFunc)rec_del_flag, &val);
}

void rec_unshare_flag (GtkCTree *ctree, GtkCTreeNode *node,
		       int* data) {
  file_t* file;
  char* vals;

  vals = (char*)data;

  if (!GTK_CTREE_ROW (node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (file->flags & vals[0]) {
    file->flags &= (0xff^(vals[0]));
    file->flags |= vals[1];
    global.user.unshared++;
  }
}

void lib_unshare_flag(int val1, int val2, GtkCTreeNode* node) {
  GtkCTree* ctree;
  char data[2];

  data[0] = val1;
  data[1] = val2;

  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, node, 
			   (GtkCTreeFunc)rec_unshare_flag, data);
}

file_t* lib_search_file(char* longname) {
  GList* dlist;
  file_t* file;
  
  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    if (!strcmp(longname, file->longname)) return file;
  }
  return NULL;
}

void lib_search(search_pattern_t* search) {
  GList* dlist;
  file_t* file;
  int cnt = 0;

  if (global.status.building > 1) return;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    if (search_pattern_fits_file(search, file, 1)) {
      file = file_dup(file);
      file_insert_search(file, DEST_LIBRARY);
      cnt++;
    }
    if (cnt >= search->max_results) return;
  }
}

void lib_sort() {
  GtkCTree* ctree;

#ifdef SHARE_DEBUG
  printf("sorting\n");
#endif
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));

  if (global.user.shared < 5000) {
    gtk_clist_freeze(GTK_CLIST(ctree));
    gtk_ctree_sort_recursive(ctree, NULL);
    gtk_clist_thaw(GTK_CLIST(ctree));
  }
}

file_t* load_file_entry(FILE* fd) {
  char line[2048];
  file_t* file;
  char* mime;
  
  if (!fgets(line, 2048-1, fd)) return NULL;

  file = (file_t*)malloc(sizeof(file_t));
  file->longname = strdup(arg(line, 0));
  file->shortname = strdup(extract_short_name(file->longname));
  file->winname = strdup(file->longname);
  convert_to_win(file->winname);
  file->md5 = strdup(arg(NULL, 0));
  file->size = strtoul(arg(NULL, 0), NULL, 10);
  file->bitrate = atoi(arg(NULL, 0));
  file->frequency = atoi(arg(NULL, 0));
  file->duration = atoi(arg(NULL, 0));
  mime = arg(NULL, 0);
  if (!mime) file->mime_type = MIME_MP3;
  else file->mime_type = atoi(mime);
  file->user = strdup("");
  file->ip = 0;
  file->linespeed = 0;
  file->flags = 0;
  file->status = FILE_NONE;
  file->local = 1;            //local file

  return file;
}

gint lib_load_idle(gpointer data) {
  FILE* fd = (FILE*)data;
  GtkCList* clist;
  file_t* file;
  file_t* file2;
  int i1;

#ifdef SHARE_DEBUG
  printf("load idle\n");
#endif  
  
  for (i1 = 0; i1 < 10; i1++) {
    if (!(file = load_file_entry(fd))) goto finish;
    file2 = lib_search_file(file->longname);
    if (!file2) {
      lib_insert_file(file);
    } else {
      file2->flags &= 0xff^FLAG_TO_REMOVE;
      destroy_file_row(file);
    }
  }

  return 1;
  
 finish:
  global.status.building = 0;
  fclose(fd);
  clist = GTK_CLIST(lookup_widget(global.win, "lib_tree"));
  gtk_clist_thaw(clist);
  gtk_idle_add(lib_remove_idle, NULL);

  return 0;
}

int lib_load(char* fname) {
  char filename[1024];
  FILE* fd = NULL;
  GtkCList* clist;
  GtkWidget* temp;
  char t[1024];

  if (!fname || !(*fname)) return 0;
  strcpy(t, fname);

#ifdef SHARE_DEBUG
  printf("building %d\n", global.status.building);
#endif
  if (global.status.building > 1) return 0;

#ifdef SHARE_DEBUG
  printf("lib_load\n");
#endif  
  sprintf(filename, "%s/share/%s", global.lopster_home, fname);
  fd = fopen(filename, "r");
  if (!fd) return 0;
    
  clist = GTK_CLIST(lookup_widget(global.win, "lib_tree"));
  gtk_clist_freeze(clist);
  global.status.building = 2;
  lib_set_flag(FLAG_TO_REMOVE, NULL);

  temp = lookup_widget(global.win, "entry84");
  gtk_entry_set_text(GTK_ENTRY(temp), fname);
  if (global.auto_save) free(global.auto_save);
  global.auto_save = strdup(t);

  gtk_idle_add(lib_load_idle, (gpointer)fd);
  return 1;
}

void lib_save(char* name) {
  char filename[1024];
  FILE* fd;
  GList* dlist;
  file_t* file;
  
  if (!name || !(*name)) return;

  sprintf(filename, "%s/share/%s", global.lopster_home, name);

  if ((fd = fopen(filename, "w")) == NULL) {
    g_warning(_("could not save shared files %s"), filename);
    return;
  }
  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t*)(dlist->data);
    fprintf(fd, "\"%s\" %s %ld %d %d %d %d\n", 
	    file->longname, file->md5, file->size,
	    file->bitrate, file->frequency, file->duration, file->mime_type);
  }
  fclose(fd);

  if (global.auto_save) free(global.auto_save);
  global.auto_save = strdup(name);
}

void lib_delete(char* name) {
  char filename[1024];
  
  if (!name || !(*name)) return;

  sprintf(filename, "%s/share/%s", global.lopster_home, name);
  unlink(filename);
}

void
on_delete_list_activate (GtkMenuItem     *menuitem,
			 gpointer         user_data) {
  
  GtkLabel* label;
  char* s1;

  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  s1 = label->label;
  lib_delete(s1);
}

void
on_load_list_activate (GtkMenuItem     *menuitem,
		       gpointer         user_data) {
  
  GtkLabel* label;
  char* s1;

  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  s1 = label->label;
  lib_load(s1);
}

GtkWidget* create_files_popup() {
  GtkWidget *popup;
  GtkWidget *menu1 = NULL;
  GtkWidget *menu2 = NULL;
  GtkWidget* item;
  GtkWidget* item1;
  GtkWidget* item2;
  GtkAccelGroup *popup_accels;
  char dirname[2048];
  struct dirent* entry;
  struct stat buf;
  char filename[2048];
  DIR* dir;
  int cnt = 0;

  sprintf(dirname, "%s/share", global.lopster_home);

  popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (popup));

  item1 = gtk_menu_item_new_with_label (_("Delete List"));
  gtk_widget_show (item1);
  gtk_container_add (GTK_CONTAINER (popup), item1);

  item2 = gtk_menu_item_new_with_label (_("Load List"));
  gtk_widget_show (item2);
  gtk_container_add (GTK_CONTAINER (popup), item2);

  if ((dir = opendir(dirname)) == NULL) {
    g_warning("could not open dir [%s]", dirname);
  } else {
    while ((entry = readdir(dir)) != NULL) {
      //      if (!strncmp(entry->d_name, ".", 1)) continue;
      //      if (!strncmp(entry->d_name, "..", 2)) continue;
      sprintf(filename, "%s/%s", dirname, entry->d_name);
      stat(filename, &buf);
      if (S_ISREG(buf.st_mode)) {
	if (!cnt) {  
	  menu1 = gtk_menu_new ();
	  gtk_widget_show (menu1);
	  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item1), menu1);
	  menu2 = gtk_menu_new ();
	  gtk_widget_show (menu2);
	  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item2), menu2);
	}
	item = gtk_menu_item_new_with_label (entry->d_name);
	gtk_widget_show (item);
	gtk_container_add (GTK_CONTAINER (menu1), item);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    on_delete_list_activate,
			    NULL);
	item = gtk_menu_item_new_with_label (entry->d_name);
	gtk_widget_show (item);
	gtk_container_add (GTK_CONTAINER (menu2), item);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    on_load_list_activate,
			    NULL);
	cnt++;
      }
    }
    closedir(dir);
  }
  
  if (!cnt) {
    gtk_widget_set_sensitive(item1, FALSE);
    gtk_widget_set_sensitive(item2, FALSE);
  }

  return popup;
}

void lib_update_stats() {
  GtkWidget* temp;
  char comm[1024];

  temp = lookup_widget(global.win, "label233");
  sprintf(comm, "%d/%d", global.user.shared-global.user.unshared, 
	  global.user.shared);
  gtk_label_set_text(GTK_LABEL(temp), comm);
  temp = lookup_widget(global.win, "label235");
  sprintf(comm, "%.0f", global.user.bytes / 1048576.);
  gtk_label_set_text(GTK_LABEL(temp), comm);

}

GtkWidget*
create_lib_popup2 (void)
{
  GtkWidget *lib_popup;
  GtkAccelGroup *lib_popup_accels;
  GtkWidget *item;
  GtkWidget *separator;
  GtkCTree* ctree;
  GtkCTreeNode* node;
  file_t* file;

  ctree = GTK_CTREE(global.popup_list);
  node = gtk_ctree_node_nth(ctree, global.popup_row);
  if (node) {
    file = (file_t*)gtk_ctree_node_get_row_data(ctree, node);
  } else {
    file = NULL;
  }

  lib_popup = gtk_menu_new ();
  gtk_container_set_border_width (GTK_CONTAINER (lib_popup), 2);
  lib_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (lib_popup));
  
  if (file) {
    item = gtk_menu_item_new_with_label (_("Open File"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_play_file2_activate),
			NULL);
    
    separator = gtk_menu_item_new ();
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (lib_popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);
  }
  item = gtk_menu_item_new_with_label (_("Refresh Shared List"));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (lib_popup), item);
  gtk_signal_connect (GTK_OBJECT (item), "activate",
                      GTK_SIGNAL_FUNC (on_refresh1_activate),
                      NULL);

  separator = gtk_menu_item_new ();
  gtk_widget_show (separator);
  gtk_container_add (GTK_CONTAINER (lib_popup), separator);
  gtk_widget_set_sensitive (separator, FALSE);

  if (file) {
    item = gtk_menu_item_new_with_label (_("Remove File"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_remove_activate),
			NULL);
    
    if (file->flags & FLAG_SHARED)
      item = gtk_menu_item_new_with_label (_("Unshare File"));
    else
      item = gtk_menu_item_new_with_label (_("Share File"));
    
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_unshare_activate),
			NULL);
    
    separator = gtk_menu_item_new ();
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (lib_popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);
    
    item = gtk_menu_item_new_with_label (_("Delete (from Disk)"));
    gtk_widget_show (item);
    gtk_container_add (GTK_CONTAINER (lib_popup), item);
    gtk_signal_connect (GTK_OBJECT (item), "activate",
			GTK_SIGNAL_FUNC (on_delete_file_activate),
			NULL);
    
    separator = gtk_menu_item_new ();
    gtk_widget_show (separator);
    gtk_container_add (GTK_CONTAINER (lib_popup), separator);
    gtk_widget_set_sensitive (separator, FALSE);
  }
  item = gtk_menu_item_new_with_label (_("Customize List"));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (lib_popup), item);
  gtk_signal_connect (GTK_OBJECT (item), "activate",
                      GTK_SIGNAL_FUNC (on_customize_list_activate),
                      NULL);

  return lib_popup;
}

void lib_set_modified() {
  GtkWidget* temp;

  temp = lookup_widget(global.win, "entry84");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
}
