#include <sys/socket.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "interface.h"
#include "support.h"
#include "commands.h"
#include "callbacks.h"
#include "connection.h"
#include "global.h"
#include "chat.h"
#include "preferences.h"
#include "search.h"
#include "transfer.h"
#include "share.h"
#include "hotlist.h"
#include "browse.h"
#include "log.h"
#include "scheme.h"
#include "handler.h"
#include "server.h"

static HANDLER_ENTRY Protocol[] = {
  {CMD_SERVER_ERROR,            cmd_server_error},         // 0
  {CMD_SERVER_EMAIL,            cmd_login_ack},            // 3
  {CMD_SERVER_REGISTER_OK,      cmd_do_nothing},           // 8
  {CMD_SERVER_REGISTER_FAIL,    cmd_register_fail},        // 9
  {CMD_SERVER_BAD_NICK,         cmd_bad_nick},             // 10
  {CMD_CLIENT_UNSHARE_ALL,      cmd_do_nothing},           // 110
  {CMD_SERVER_SEARCH_RESULT,    cmd_search_response},      // 201
  {CMD_SERVER_SEARCH_END,       cmd_end_of_search},        // 202
  {CMD_SERVER_FILE_READY,       cmd_download_ack},         // 204
  {CMD_CLIENT_PRIVMSG,          cmd_private_message},      // 205
  {CMD_SERVER_SEND_ERROR,       cmd_get_error},            // 206
  {CMD_SERVER_USER_SIGNON,      cmd_user_sign_on},         // 209
  {CMD_SERVER_USER_SIGNOFF,     cmd_user_sign_off},        // 210
  {CMD_SERVER_BROWSE_RESPONSE,  cmd_browse_response},      // 212
  {CMD_SERVER_BROWSE_END,       cmd_browse_end},           // 213
  {CMD_SERVER_STATS,            cmd_server_stats},         // 214
  {CMD_SERVER_RESUME_MATCH,     cmd_resume_match},         // 216
  {CMD_SERVER_RESUME_MATCH_END, cmd_do_nothing},           // 217
  {CMD_SERVER_HOTLIST_ACK,      cmd_hotlist_ack},          // 301
  {CMD_HOTLIST_ERROR,           cmd_hotlist_error},        // 302
  {CMD_SERVER_DISCONNECTING,    cmd_disconnecting},        // 316
  {CMD_CLIENT_IGNORE_LIST,      cmd_do_nothing},           // 320
  {CMD_SERVER_IGNORE_ENTRY,     cmd_ignore_user},          // 321
  {CMD_CLIENT_IGNORE_USER,      cmd_ignore_user},          // 322
  {CMD_CLIENT_UNIGNORE_USER,    cmd_unignore_user},        // 323
  {CMD_SERVER_NOT_IGNORED,      cmd_do_nothing},           // 324
  {CMD_SERVER_ALREADY_IGNORED,  cmd_do_nothing},           // 325
  {CMD_CLIENT_PART,             cmd_part_channel},         // 401
  {CMD_SERVER_PUBLIC,           cmd_public_message},       // 403
  {CMD_SERVER_NOSUCH,           cmd_error_message},        // 404
  {CMD_SERVER_JOIN_ACK,         cmd_create_channel},       // 405
  {CMD_SERVER_JOIN,             cmd_join_message},         // 406
  {CMD_SERVER_PART,             cmd_part_message},         // 407
  {CMD_SERVER_CHANNEL_USER_LIST,cmd_user_online},          // 408
  {CMD_SERVER_CHANNEL_USER_LIST_END,cmd_do_nothing},       // 409
  {CMD_SERVER_TOPIC,            cmd_channel_topic},        // 410
  {CMD_CLIENT_CHANNEL_BAN_LIST, cmd_do_nothing},           // 420
  {CMD_SERVER_CHANNEL_BAN_LIST, cmd_banlist_entry},        // 421
  {CMD_CHANNEL_ALT_TOPIC,       cmd_channel_alt_topic},    // 425
  {CMD_SERVER_UPLOAD_FIREWALL,  cmd_alternate_ack},        // 501
  {CMD_SERVER_USER_SPEED,       cmd_linkspeed_response},   // 601
  {CMD_SERVER_WHOIS_RESPONSE,   cmd_eval_whois},           // 604
  {CMD_SERVER_WHOWAS,           cmd_eval_whowas},          // 605
  {CMD_SERVER_UPLOAD_REQUEST,   cmd_upload_request},       // 607
  {CMD_GET_ERROR,               cmd_get_error},            // 609
  {CMD_CLIENT_ALTER_PORT,       cmd_set_port},             // 613
  {CMD_CLIENT_BANLIST,          cmd_do_nothing},           // 615
  {CMD_SERVER_IP_BANLIST,       cmd_banlist_entry},        // 616
  {CMD_SERVER_CHANNEL_LIST_END, cmd_channel_list_end},     // 617
  {CMD_SERVER_CHANNEL_LIST,     cmd_channel_list},         // 618
  {CMD_SERVER_LIMIT,            cmd_remote_queued},        // 620
  {CMD_SERVER_MOTD,             cmd_motd},                 // 621
  {CMD_SERVER_DATA_PORT_ERROR,  cmd_data_port_error},      // 626
  {CMD_SERVER_WALLOP,           cmd_operator_message},     // 627
  {CMD_SERVER_ANNOUNCE,         cmd_global_message},       // 628
  {CMD_SERVER_NICK_BANLIST,     cmd_nick_banlist},         // 629
  {CMD_CLIENT_BROWSE_DIRECT,    cmd_browse_direct},        // 640
  {CMD_SERVER_BROWSE_DIRECT_OK, cmd_browse_direct_ok},     // 641
  {CMD_SERVER_BROWSE_DIRECT_ERR,cmd_browse_direct_err},    // 642
  {CMD_SERVER_GHOST,            cmd_ghost},                // 748
  {CMD_CLIENT_PING_SERVER,      cmd_sping},                // 750
  {CMD_SERVER_PING,             cmd_ping},                 // 751
  {CMD_SERVER_PONG,             cmd_pong},                 // 752
  {CMD_CLIENT_EMOTE,            cmd_emote},                // 824
  {CMD_SERVER_NAMES_LIST,       cmd_user_online},          // 825
  {CMD_CLIENT_FULL_CHANNEL_LIST,cmd_channel_list_end},     // 827
  {CMD_SERVER_FULL_CHANNEL_INFO,cmd_channel_list_entry},   // 828
  {CMD_SERVER_NAMES_LIST_END,   cmd_do_nothing},           // 830
  {CMD_CLIENT_GLOBAL_USER_LIST, cmd_global_user_end},      // 831
  {CMD_SERVER_GLOBAL_USER_LIST, cmd_global_user},          // 832
  {CMD_SERVER_LINKS,            cmd_server_links},         // 10112
  {CMD_CLIENT_WHO_WAS,          cmd_whowas},               // 10121
  {CMD_SERVER_USAGE_STATS,      cmd_usage_stats},          // 10115
  {CMD_CLIENT_VERSION_STATS,    cmd_version_stats},        // 10118
  {CMD_SERVER_USER_MODE,        cmd_user_mode},            // 10203
  {CMD_SERVER_BROWSE_RESULT_NEW,cmd_browse_new},           // 10302
  {CMD_CLIENT_BROWSE_NEW,       cmd_browse_end}            // 10301
};

static int Protocol_Size = sizeof (Protocol) / sizeof (HANDLER_ENTRY);

char Mode[NO_MODES][20] = {
  "ERROR",
  "BAN",
  "CHANGE",
  "KILL",
  "LEVEL",
  "SERVER",
  "MUZZLE",
  "PORT",
  "WALLOP",
  "CLOAK",
  "FLOOD",
  "PING",
  "MSG",
  "WHOIS"
};

int mode2int(char* mode) {
  int no;

  for (no = 0; no < NO_MODES; no++)
    if (!strcasecmp(mode, Mode[no])) return no;
  
  return -1;
}

void not_implemented(int type, char* data) {
#ifdef PROTOCOL_DEBUG
  log("protocol", LOG_OTHER, "** Unknown\n");
#endif
  client_message("Error", "** not implemented yet: %d: [%s]", 
		 type, data);
}

int search_handler(int code) {
  int i1;

  // for now do a linear search
  for (i1 = 0; i1 < Protocol_Size; i1++) {
    if (Protocol[i1].code == code) return i1;
  }
  return -1;
}

void handle_command(int type, char* data) {
  int i1;

  i1 = search_handler(type);
  if (i1 < 0) {
    not_implemented(type, data);
  } else {
    (Protocol[i1].handler)(data);
  }
}

HANDLER(cmd_do_nothing) {
}

HANDLER(cmd_server_error) {
  cmd_error_message(data);
}

HANDLER(cmd_login_ack) {
  int i1;
  char* text;
  chat_page_t* page;
  GtkWidget* temp;
  GList* dlist;

  if (global.clientinfo) {
    text = g_strdup_printf("Lopster %s (alias %s)", VERSION, global.clientinfo);
    gtk_window_set_title (GTK_WINDOW(global.win), text);
  } else {
    text = g_strdup_printf("Lopster %s", VERSION);
    gtk_window_set_title (GTK_WINDOW(global.win), text);
  }
  g_free(text);

  global.user.level = L_USER;
  global.user.status = STATUS_ACTIVE;
  
  while ((page = chat_page_find_unfixed()) != NULL) {
    chat_page_destroy(page);
  }

  temp = lookup_widget(global.win, "Console_text");
  gtk_text_set_point(GTK_TEXT(temp), 0);
  gtk_text_forward_delete(GTK_TEXT(temp),
			  gtk_text_get_length(GTK_TEXT(temp)));

  connect_success();
  
#ifdef PROTOCOL_DEBUG
  log("protocol", LOG_OTHER, _("Connected to %s:%d\n"), SERVER->address,
      SERVER->port);
#endif  
  if ((global.options.login_mode & L_NOT_JOIN) == 0) {
    chat_print("message", _("Joining channels....\n"));
    for (dlist = global.channels; dlist; dlist = dlist->next) {
      join_channel(dlist->data);
    }
  } else {
    global.options.login_mode &= 0xff^L_NOT_JOIN;
    chat_print("message", _("Autojoin channels suppressed!\n"));
  }
  
  chat_print("message", _("Adding Hotlist entries....\n"));
  if (global.hotlist) {
    for (i1 = 0; i1 < g_list_length(global.hotlist); i1++) {
      send_command(CMD_CLIENT_ADD_HOTLIST_SEQ, g_list_nth(global.hotlist, i1)->data);
    }
  }
  
  chat_print("message", _("Ignoring enemy users....\n"));
  if (global.enemyuser) {
    for (i1 = 0; i1 < g_list_length(global.enemyuser); i1++) {
      send_command(CMD_CLIENT_IGNORE_USER, g_list_nth(global.enemyuser, i1)->data);
    }
  }
  
  set_last_server(SERVER);

  global.status.whois_hide++;
  send_command(CMD_CLIENT_WHOIS, SERVER->nick);
}

HANDLER(cmd_register_fail) {
  socket_destroy(global.napster, (int)_("Nick already registered"));
}

HANDLER(cmd_bad_nick) {
  socket_destroy(global.napster, (int)_("Invalid Nick"));
}

HANDLER(cmd_search_response) {
  file_t* file;

  file = file_create_from_search_response(data);
  
  //  resume = resume_list_search_file(file);
  file_insert_search(file, DEST_NAPSTER);
  
  destroy_file_row(file);
}

HANDLER(cmd_end_of_search) {
  if (global.status.searching == 0) return;

  global.status.searching--;
  search_update_counter();
  
  setup_sensitive(-1);
}

HANDLER(cmd_download_ack) {
  socket_t* socket;
  transfer_t* transfer;

  char* user;
  int port;
  unsigned long ip_long;
  char* winname;
  char* md5;
  int linespeed;
  char t2[2048] = "";

  user = arg(data, 0);
  ip_long = strtoul(arg(NULL, 0), NULL, 10);
  port = atoi(arg(NULL, 0));
  winname = arg(NULL, 0);
  md5 = arg(NULL, 0);
  linespeed = atoi(arg(NULL, 0));

  socket = transfer_is_in_download(user, winname);

  if (!socket) return;
  transfer = (transfer_t*)(socket->data);
  if (!transfer_is_active(transfer)) return;

  socket->port = htons(port);
  socket->ip_long = BSWAP32(ip_long);
  transfer_set_md5(transfer, md5);
  transfer->linespeed = linespeed;
  
  if (socket->port == 0) {
    if ((global.network.port == 0) || global.network.firewall) {
      socket_end(socket, S_FIREWALL);
    } else {
      strcpy(t2, transfer->user);
      strcat(t2, " \"");
      strcat(t2, transfer->winname);
      strcat(t2, "\"");
      send_command(CMD_CLIENT_DOWNLOAD_FIREWALL, t2);
    }
  } else {
    transfer_connect_and_start(socket);
  }
}

HANDLER(cmd_private_message) {
  char t[500];
  char* user;
  char* message;
  GtkWidget* temp;
  char* dummy = "";
  time_t tim;
  chat_page_t* page;

  time(&tim);

  user = arg(data, 0);
  if (!user) return;
  message = arg(NULL, 1);
  if (!message) message = dummy;

  if (check_dcc_message(user, message)) return;

  log(user, LOG_PRIVATE, "<%s> %s\n", user, message);
  if ((page = chat_page_search(user, P_PRIVATE))) {
    sprintf(t, "%s %s %s", user, user, message);
    cmd_public_message_no_log(t, M_PUBLIC);
    
    sprintf(t, "%s_topic", page->name);
    temp = lookup_widget(global.win, t);
    sprintf(t, _("Last Message received: %s"), ctime(&tim));
    gtk_entry_set_text(GTK_ENTRY(temp), t);

  } else {
    if (global.reply) free(global.reply);
    global.reply = strdup(user);
    update_user_stats();
    
    chat_print("whisper", _("<from: "));
    chat_print("whisper", user);
    chat_print("whisper", "> ");
    chat_print_ln("whisper", message);
  }

  if (global.afk.message) send_afk(user, &global.afk);
}

HANDLER(cmd_get_error) {
  socket_t* socket;
  char* user;
  char* winname;
  
  user = arg(data, 0);
  winname = arg(data, 0);
  
  socket = transfer_is_in_download(user, winname);
  if (socket) socket_end(socket, S_UNAVAILABLE);
}

HANDLER(cmd_user_sign_on) {
  GtkCList* clist;
  int row;
  char* nick;
  int speed;
  user_t* userinfo;

  nick = arg(data, 0);
  speed = atoi(arg(NULL, 0));

  clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
  row = is_user_in_hotlist(nick);
  if (row >= 0) {
    gtk_clist_set_pixtext (clist, row, 0, nick, 5, 
			   global.pix.user1, global.pix.user1b);
    userinfo = gtk_clist_get_row_data(clist, row);
    userinfo->linespeed = speed;
    gtk_clist_set_text(clist, row, 2, LineSpeed(speed));
  } else {
    g_warning("user not in hotlist");
  }
  client_message(NULL, _("<%s> is on !"), nick);
  //  send_command(CMD_CLIENT_BROWSE, t);
}

HANDLER(cmd_user_sign_off) {
  GtkCList* clist;
  int row;
  char* nick;
  user_t* userinfo;

  nick = arg(data, 0);

  clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
  row = is_user_in_hotlist(nick);
  if (row >= 0) {
    gtk_clist_set_pixtext (clist, row, 0, nick, 5, 
			   global.pix.user2, global.pix.user2b);
    userinfo = gtk_clist_get_row_data(clist, row);
    hotlist_remove_user_files(userinfo);
    userinfo->linespeed = 0;
    gtk_clist_set_text(clist, row, 2, "");
  } else {
    g_warning("user not in hotlist");
  }
  client_message(NULL, _("<%s> is off !"), nick);
}

HANDLER(cmd_browse_response) {
  file_t* file;
  
  file = file_create_from_browse_response(data);
  file_insert_browse(file);
}

HANDLER(cmd_browse_end) {
  GtkCList* clist;
  GtkWidget* temp;
  int row;
  user_t* userinfo;
  char* pos;
  char* nick;

  nick = arg(data, 0);
  clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
  row = is_user_in_hotlist(nick);
  if (row >= 0) {
    userinfo = (user_t*)gtk_clist_get_row_data(clist, row);
    pos = g_strdup_printf("%d", userinfo->shared);
    gtk_clist_set_text(clist, row, 1, pos);
    free(pos);
  }

  temp = lookup_widget(global.win, "ctree3");
  userinfo = (user_t*)gtk_object_get_data(GTK_OBJECT(temp), 
					  "userdata");
  if (userinfo && !strcmp(userinfo->username, nick))
    gtk_ctree_sort_recursive(GTK_CTREE(temp), NULL);

  hotlist_update_user();
}

HANDLER(cmd_server_stats) {
  GtkWidget* temp;
  int songs, gics, users;
  char t[500];

  sscanf(data, "%d %d %d", &users, &songs, &gics);
  temp = lookup_widget(global.win, "label280");
  sprintf(t, _("%d Users"), users);
  gtk_label_set_text(GTK_LABEL(temp), t);
  temp = lookup_widget(global.win, "label281");
  sprintf(t, _("%d Files"), songs);
  gtk_label_set_text(GTK_LABEL(temp), t);
  temp = lookup_widget(global.win, "label282");
  sprintf(t, _("%d GB"), gics);
  gtk_label_set_text(GTK_LABEL(temp), t);
}

HANDLER(cmd_resume_match) {

  sscanf(data, "%s %*s %*s %s %*s %s %s",
	 tstr[1], tstr[0], tstr[2], tstr[3]);
  strcpy(tstr[3], LineSpeed(atoi(tstr[3])));
  
  /*
  temp = lookup_widget(global.resume_win, "clist2");
  gtk_clist_append(GTK_CLIST(temp), list);
  */
}

HANDLER(cmd_hotlist_ack) {
  GtkCList* clist;
  int row;
  char* nick;

  nick = arg(data, 0);

  clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
  row = is_user_in_hotlist(nick);
  if (row >= 0) {
    gtk_clist_set_pixtext (clist, row, 0, nick, 5,
			   global.pix.user2, global.pix.user2b);
  } else {
    g_warning("user not in hotlist");
  }
}

HANDLER(cmd_hotlist_error) {
  client_message("error", data);
}

HANDLER(cmd_disconnecting) {
  client_message(_("Message"), _("You will be disconnected"));
}

HANDLER(cmd_ignore_user) {
  char* user;
  char t[500];
  GtkCList* clist;

  // adding to intern list
  if (is_string_in_list(global.ignored_users, data)) return;    

  user = strdup(data);

  global.ignored_users =
    g_list_append(global.ignored_users, user);
  sprintf(t, "is now ignoring %s", user);

  if (global.options.send_ignore) send_emote(t);
  update_user(user);

  // maybe update the ignore win
  if (global.ignore_win == NULL) return;
  
  clist = GTK_CLIST(lookup_widget(global.ignore_win, "ignore_list"));
  strcpy(tstr[0], user);
  gtk_clist_append(clist, list);
}

HANDLER(cmd_unignore_user) {
  char* user;
  char* text;
  int i1;
  char t[500];
  GtkCList* clist;

  // removing from intern list
  user = is_string_in_list(global.ignored_users, data);
  if (user) {
    //      printf("user [%s] is ignored\n", user);
    global.ignored_users = g_list_remove(global.ignored_users, user);
    free(user);
  }
  user = data;
  sprintf(t, "is no longer ignoring %s", user);
  
  if (global.options.send_ignore) send_emote(t);
  update_user(user);

  // maybe update the ignore win
  if (global.ignore_win == NULL) return;
  
  clist = GTK_CLIST(lookup_widget(global.ignore_win, "ignore_list"));
  for (i1 = 0; i1 < clist->rows; i1++) {
    gtk_clist_get_text(clist, i1, 0, &text);
    strcpy(t, text);
    if (!strcmp(t, user)) {
      gtk_clist_remove(clist, i1);
      break;
    }
  }
}

HANDLER(cmd_part_channel) {
  chat_page_t* page;
  
  page = chat_page_search(data, P_PUBLIC);
  if (!page) return;
  chat_page_destroy(page);
}

HANDLER(cmd_public_message) {
  char page[500], user[500];
  char* pos;

  sscanf(data, "%s %s", page, user);
  pos = strchr(data, ' ');
  pos = strchr(pos+1, ' ')+1;
  log(page, LOG_CHANNEL, "<%s> %s\n", user, pos);
  
  cmd_public_message_no_log(data, M_PUBLIC);
}

HANDLER(cmd_error_message) {
  if (check_channel_wallop(data)) return;
  
  check_server_join_quit(data);
  check_server_pong(data);
  //  check_get_motd(data);
  check_op(data);
  check_pending_searches(data);
  check_share_limit(data);
  
  server_message(NULL, data);

  if (global.status.connection == 1)
    napster_disconnect(data);
  log("Messages", LOG_OTHER, "%s\n", data);
}

HANDLER(cmd_create_channel) {
  GtkWidget *channel_widget;
  GtkWidget *channel_widget2;
  GtkWidget *scrolledwindow27;
  GtkWidget *text4;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *scrolledwindow26;
  GtkWidget *clist1;
  GtkWidget *label96;
  GtkWidget *label97;
  GtkWidget *label98;
  GtkWidget *tab_label;
  GtkWidget *entry18;
  GtkWidget *button;
  GtkWidget *frame;
  GtkNotebook* notebook;
  GtkTooltips* tips;
  GtkWidget *button2;
  GtkWidget *pix1;
  GtkWidget *vbox2;
  GtkWidget *hbox2;
  GtkWidget *label;

  char str[100];
  char str1[100];
  char str2[100];
  time_t tim;
  chat_page_t* page;
  style_t* style;
  
  channel_widget = gtk_hpaned_new ();
  sprintf(str, "%s_channel", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, channel_widget);
  gtk_paned_set_gutter_size (GTK_PANED (channel_widget), 10);
  gtk_paned_set_position (GTK_PANED (channel_widget), global.paned_pos);
  gtk_widget_show(channel_widget);

  vbox = gtk_vbox_new (FALSE, 0);
  sprintf(str, "%s_vbox", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, vbox);
  gtk_widget_show (vbox);
  gtk_paned_pack1 (GTK_PANED (channel_widget), vbox, TRUE, TRUE);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_object_set_data (GTK_OBJECT (global.win), "hbox", hbox);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  text4 = gtk_text_new (NULL, NULL);
  sprintf(str, "%s_topic", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, text4);
  gtk_widget_set_style(text4, global.style[2]);  
  gtk_widget_show (text4);
  gtk_signal_connect_after (GTK_OBJECT (text4), "button_press_event",
                            GTK_SIGNAL_FUNC (on_text_button_press_event),
                            NULL);

  style = style_get(global.scheme, "text");
  if (style->font)
    gtk_widget_set_usize (text4, -1, 
			  style->font->ascent+ 
			  style->font->descent+7);
  else
    gtk_widget_set_usize (text4, -1, 22);
  gtk_text_set_word_wrap(GTK_TEXT(text4), 1);
  gtk_box_pack_start (GTK_BOX (hbox), text4, TRUE, TRUE, 0);

  //
  button2 = gtk_button_new ();
  gtk_widget_show (button2);
  gtk_box_pack_start (GTK_BOX (hbox), button2, FALSE, FALSE, 0);
  GTK_WIDGET_UNSET_FLAGS (button2, GTK_CAN_FOCUS);
  gtk_button_set_relief (GTK_BUTTON (button2), GTK_RELIEF_NONE);

  vbox2 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox2);
  gtk_container_add (GTK_CONTAINER (button2), vbox2);

  hbox2 = gtk_hbox_new (FALSE, 5);
  sprintf(str, "%s_h1", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, hbox2);
  gtk_widget_show (hbox2);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox2, FALSE, FALSE, 0);

  label = gtk_label_new (_("hide users"));
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);

  pix1 = create_pixmap (global.win, "arrowr.xpm");
  gtk_widget_show (pix1);
  gtk_box_pack_start (GTK_BOX (hbox2), pix1, FALSE, FALSE, 0);

  hbox2 = gtk_hbox_new (FALSE, 5);
  sprintf(str, "%s_h2", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, hbox2);
  //gtk_widget_show (hbox2);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox2, FALSE, FALSE, 0);

  pix1 = create_pixmap (global.win, "arrowl.xpm");
  gtk_widget_show (pix1);
  gtk_box_pack_start (GTK_BOX (hbox2), pix1, FALSE, FALSE, 0);

  gtk_signal_connect (GTK_OBJECT (button2), "clicked",
                      GTK_SIGNAL_FUNC (on_button_users_clicked),
                      NULL);
  label = gtk_label_new (_("show users"));
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);

  // new
  channel_widget2 = gtk_vpaned_new ();
  sprintf(str, "%s_channel2", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, channel_widget2);
  gtk_widget_show(channel_widget2);
  if (global.user.level > L_USER)
    gtk_paned_set_position (GTK_PANED (channel_widget2), global.paned_pos2);
  else
    gtk_paned_set_position (GTK_PANED (channel_widget2), 0);
  gtk_box_pack_start (GTK_BOX (vbox), channel_widget2, TRUE, TRUE, 0);

  //
  scrolledwindow27 = gtk_scrolled_window_new (NULL, NULL);
  sprintf(str, "%s_scrolled", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, scrolledwindow27);
  gtk_widget_show (scrolledwindow27);

  gtk_paned_pack1 (GTK_PANED (channel_widget2), scrolledwindow27, TRUE, TRUE);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow27), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_range_set_update_policy (GTK_RANGE (GTK_SCROLLED_WINDOW (scrolledwindow27)->hscrollbar), GTK_UPDATE_CONTINUOUS);
  gtk_range_set_update_policy (GTK_RANGE (GTK_SCROLLED_WINDOW (scrolledwindow27)->vscrollbar), GTK_UPDATE_CONTINUOUS);

  text4 = gtk_text_new (NULL, NULL);
  sprintf(str, "%s_text2", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, text4);
  gtk_widget_set_style(text4, global.style[2]);  
  gtk_signal_connect_after (GTK_OBJECT (text4), "button_press_event",
                            GTK_SIGNAL_FUNC (on_text_button_press_event),
                            NULL);

  gtk_widget_show (text4);
  gtk_text_set_word_wrap(GTK_TEXT(text4), 1);
  gtk_container_add (GTK_CONTAINER (scrolledwindow27), text4);

  //
  scrolledwindow27 = gtk_scrolled_window_new (NULL, NULL);
  sprintf(str, "%s_scrolled2", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, scrolledwindow27);
  gtk_widget_show (scrolledwindow27);

  gtk_paned_pack2 (GTK_PANED (channel_widget2), scrolledwindow27, TRUE, TRUE);
  //  gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow27, TRUE, TRUE, 0);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow27), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_range_set_update_policy (GTK_RANGE (GTK_SCROLLED_WINDOW (scrolledwindow27)->hscrollbar), GTK_UPDATE_CONTINUOUS);
  gtk_range_set_update_policy (GTK_RANGE (GTK_SCROLLED_WINDOW (scrolledwindow27)->vscrollbar), GTK_UPDATE_CONTINUOUS);

  text4 = gtk_text_new (NULL, NULL);
  sprintf(str, "%s_text", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, text4);
  gtk_widget_set_style(text4, global.style[2]);  
  gtk_signal_connect_after (GTK_OBJECT (text4), "button_press_event",
                            GTK_SIGNAL_FUNC (on_text_button_press_event),
                            NULL);

  gtk_widget_show (text4);
  gtk_text_set_word_wrap(GTK_TEXT(text4), 1);
  gtk_container_add (GTK_CONTAINER (scrolledwindow27), text4);

  ///

  vbox = gtk_vbox_new (FALSE, 0);
  sprintf(str, "%s_vbox", data);
  gtk_object_set_data(GTK_OBJECT (global.win), str, vbox);
  gtk_widget_show (vbox);
  gtk_paned_pack2 (GTK_PANED (channel_widget), vbox, TRUE, TRUE);

  scrolledwindow26 = gtk_scrolled_window_new (NULL, NULL);
  sprintf(str, "%s_scrolled2", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, scrolledwindow26);
  gtk_widget_show (scrolledwindow26);
  gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow26, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow26), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  clist1 = gtk_clist_new (3);
  sprintf(str, "%s_online", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, clist1);

  gtk_widget_show (clist1);
  gtk_container_add (GTK_CONTAINER (scrolledwindow26), clist1);
  gtk_clist_set_column_width (GTK_CLIST (clist1), 0, global.online_width[0]);
  gtk_clist_set_column_width (GTK_CLIST (clist1), 1, global.online_width[1]);
  gtk_clist_set_column_width (GTK_CLIST (clist1), 2, global.online_width[2]);
  gtk_clist_set_column_visibility(GTK_CLIST(clist1), 0, global.online_show[0]);
  gtk_clist_set_column_visibility(GTK_CLIST(clist1), 1, global.online_show[1]);
  gtk_clist_set_column_visibility(GTK_CLIST(clist1), 2, global.online_show[2]);
  gtk_clist_column_titles_show (GTK_CLIST (clist1));

  label96 = gtk_label_new (_("User"));
  sprintf(str, "%s_lable96", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, label96);
  gtk_widget_show (label96);
  gtk_clist_set_column_widget (GTK_CLIST (clist1), 0, label96);

  label97 = gtk_label_new (_("Share"));
  sprintf(str, "%s_label97", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, label97);
  gtk_widget_show (label97);
  gtk_clist_set_column_widget (GTK_CLIST (clist1), 1, label97);

  label98 = gtk_label_new (_("Speed"));
  sprintf(str, "%s_label98", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, label98);
  gtk_widget_show (label98);
  gtk_clist_set_column_widget (GTK_CLIST (clist1), 2, label98);


  hbox = gtk_hbox_new (FALSE, 0);
  gtk_object_set_data (GTK_OBJECT (global.win), "hbox", hbox);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);


  sprintf(str, "%s_no", data);
  entry18 = gtk_entry_new ();
  gtk_object_set_data (GTK_OBJECT (global.win), str, entry18);
  gtk_widget_show (entry18);
  gtk_box_pack_start (GTK_BOX (hbox), entry18, TRUE, TRUE, 0);

  frame = gtk_frame_new (NULL);
  gtk_object_set_data (GTK_OBJECT (global.win), "frame", frame);
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, FALSE, 0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);

  button = gtk_button_new_with_label (_("Save"));
  gtk_object_set_data (GTK_OBJECT (global.win), "button", button);
  gtk_widget_show (button);
  gtk_container_add (GTK_CONTAINER (frame), button);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (on_online_save_clicked),
                      NULL);

  tips = (GtkTooltips*)gtk_object_get_data (GTK_OBJECT (global.win),
					    "tooltips");
  gtk_tooltips_set_tip (tips, button, _("Save Chat Page Layout"), NULL);

  tab_label = gtk_label_new(data);
  sprintf(str, "%s_tab", data);
  gtk_object_set_data (GTK_OBJECT (global.win), str, tab_label);
  gtk_widget_show(tab_label);

  gtk_signal_connect (GTK_OBJECT (clist1), "button_press_event",
                      GTK_SIGNAL_FUNC (on_online_button_press_event),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (clist1), "click_column",
                      GTK_SIGNAL_FUNC (on_list_click_column),
                      NULL);
  gtk_clist_set_sort_column(GTK_CLIST(clist1), 0);
  gtk_clist_set_auto_sort(GTK_CLIST(clist1), TRUE);
  gtk_clist_set_compare_func(GTK_CLIST(clist1), user_compare);

  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook3"));
  gtk_notebook_append_page(notebook, channel_widget, tab_label);
  page = chat_page_new(data, P_PUBLIC);
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 
			gtk_notebook_page_num(GTK_NOTEBOOK(notebook), channel_widget));

  time(&tim);
  log(data, LOG_CHANNEL, "[................] %s", ctime(&tim));
  
  sprintf(str, "%s_vbox", data);
  sprintf(str1, "%s_h2", data);
  sprintf(str2, "%s_h1", data);
  set_up_button(button2, str1, str2, str);
  gtk_signal_connect (GTK_OBJECT (button2), "clicked",
                      GTK_SIGNAL_FUNC (on_hide_show_clicked),
                      NULL);
}

HANDLER(cmd_join_message) {
  char channel[1024];
  char user[1024];
  int link, share;
  char* pos;
  chat_page_t* page;

  sscanf(data, "%s %s %d %d", channel, user, &share, &link);
  
  pos = is_string_in_list(global.frienduser, user);
  page = chat_page_search(channel, P_PUBLIC);
  if (page && (global.options.show_joins || pos)) {
    chat_print_channel(page, M_PUBLIC, "user", "[");
    if (pos) chat_print_channel(page, M_PUBLIC, "friend", user);
    else chat_print_channel(page, M_PUBLIC, "join", user);
    chat_print_channel(page, M_PUBLIC, "join", " has joined");
    chat_print_channel(page, M_PUBLIC, "user", "]");
    sprintf(user, " (%s)(%d)\n",
	    LineSpeedShort[link], share);
    chat_print_channel(page, M_PUBLIC, "NULL", user);
  }
  cmd_user_online(data);
}

HANDLER(cmd_part_message) {
  char* channel;
  char* user;
  int link, share;
  GtkWidget* temp;
  GtkWidget* temp2;
  char temp_str[1024];
  char* pos;
  chat_page_t* page;

  channel = arg(data, 0);
  user = arg(NULL, 0);
  share = atoi(arg(NULL, 0));
  link = atoi(arg(NULL, 0));
  
  pos = is_string_in_list(global.frienduser, user);
  page = chat_page_search(channel, P_PUBLIC);
  if (!page) return;

  if (global.options.show_joins || pos) {
    chat_print_channel(page, M_PUBLIC, "user", "[");
    if (pos) chat_print_channel(page, M_PUBLIC, "friend", user);
    else chat_print_channel(page, M_PUBLIC, "part", user);
    chat_print_channel(page, M_PUBLIC, "part", " has left");
    chat_print_channel(page, M_PUBLIC, "user", "]");
    sprintf(temp_str, " (%s)(%d)\n",
	    LineSpeedShort[link], share);
    chat_print_channel(page, M_PUBLIC, "NULL", temp_str);
  }    

  sprintf(temp_str, "%s_online", channel);
  temp = lookup_widget(global.win, temp_str);
  share = search_user_in_list(GTK_CLIST(temp), user);
  if (share < 0) {
    g_warning("user not in list %s", user);
  } else {
    gtk_clist_remove(GTK_CLIST(temp), share);
  }
  
  sprintf(temp_str, "%s_no", channel);
  temp2 = lookup_widget(global.win, temp_str);  // online no
  sprintf(temp_str, _("%d users online !"),  GTK_CLIST(temp)->rows);
  gtk_entry_set_text(GTK_ENTRY(temp2), temp_str);
}

HANDLER(cmd_user_online) {
  char* t2;
  GtkWidget* temp;
  GtkWidget* temp2;
  int row;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  char* channel;
  char* user;
  char* share;
  char* speed;

  channel = arg(data, 0);
  user = arg(NULL, 0);
  share = arg(NULL, 0);
  speed = arg(NULL, 0);

  if (!speed) return;

  t2 = g_strdup_printf("%s_online", channel);
  temp = lookup_widget(global.win, t2);  // online list
  g_free(t2);
  
  row = search_user_in_list(GTK_CLIST(temp), user);
  strcpy(tstr[0], user);
  strcpy(tstr[1], share);
  strcpy(tstr[2], LineSpeed(atoi(speed)));
  if (row == -1) 
    row = gtk_clist_append(GTK_CLIST(temp), list);
  
  detect_speed_pixs(atoi(speed), &pixmap, &bitmap);
  gtk_clist_set_pixtext (GTK_CLIST(temp), row, 2,
			 tstr[2], 5, pixmap, bitmap);
  gtk_clist_set_pixtext (GTK_CLIST(temp), row, 0,
			 tstr[0], 5, 
			 global.pix.dummy, global.pix.dummyb);
  gtk_clist_set_text (GTK_CLIST(temp), row, 1, tstr[1]);

  t2 = g_strdup_printf("%s_no", channel);
  temp2 = lookup_widget(global.win, t2);  // online no
  g_free(t2);
  t2 = g_strdup_printf(_("%d users online !"), 
		       GTK_CLIST(temp)->rows);
  gtk_entry_set_text(GTK_ENTRY(temp2), t2);
  g_free(t2);
  
  update_user(user);
}

HANDLER(cmd_channel_topic) {
  char* room;
  char* topic;
  chat_page_t* page;
  char t[500];
  GtkWidget* temp;

  room = arg(data, 0);
  topic = arg(NULL, 1);
  
  sprintf(t, "%s_topic", room);
  temp = lookup_widget(global.win, t);
  gtk_text_set_point(GTK_TEXT(temp), 0);
  gtk_text_forward_delete(GTK_TEXT(temp),
			  gtk_text_get_length(GTK_TEXT(temp)));
  page = chat_page_search(room, P_PUBLIC);
  if (!page) return;
  chat_print_colored(page, M_TOPIC, "text", topic);
}

HANDLER(cmd_banlist_entry) {
  char* pos;
  time_t l1;
  GtkWidget* temp;

  if (!global.ban_win) return;

  temp = lookup_widget(global.ban_win, "clist3");
  sscanf(data, "%s %s", tstr[0], tstr[1]);
  data = strchr(data, '\"')+1;
  pos = strrchr(data, '\"');
  pos[0] = 0;
  strcpy(tstr[3], data);
  data = pos + 2;
  sscanf(data, "%ld\n", &l1);
  sprintf(tstr[2], "%s", ctime(&l1));
  gtk_clist_append(GTK_CLIST(temp), list);
}

HANDLER(cmd_channel_alt_topic) {
  client_message(_("Alternate Topic"), data);
}

HANDLER(cmd_alternate_ack) {
  transfer_t* transfer;
  socket_t* socket;

  char* user;
  char* ip;
  char* port;
  char* winname;
  char* md5;
  char* linespeed;
  file_t* file;
  
  user = arg(data, 0);
  ip = arg(NULL, 0);
  port = arg(NULL, 0);
  winname = arg(NULL, 0);
  md5 = arg(NULL, 0);
  linespeed = arg(NULL, 0);

  socket = transfer_is_in_upload(user, winname);
  transfer = (transfer_t*)(socket->data);

  if (!socket) return;
  if (!transfer_is_active(transfer)) return;

  transfer->linespeed = atoi(linespeed);
  transfer_set_md5(transfer, md5);
  socket->port = htons(atoi(port));
  socket->ip_long = BSWAP32(strtoul(ip, NULL, 10));

  file = lib_search_transfer(transfer);
  if (!file) {
    // this is the first lib check
    // should send someting to the user?
    g_warning("file not found in lib\n");
    return;
  }
  transfer->size = file->size;
  transfer->mime_type = file->mime_type;

  if (socket->port == 0) {
    socket_end(socket, S_FIREWALL);
  } else {
    transfer_connect_and_start(socket);
  }
}

HANDLER(cmd_linkspeed_response) {
  char* nick;
  char* linespeed;
  GtkCList* clist;
  int i1;
  transfer_t* transfer;
  socket_t* socket;

  nick = arg(data, 0);
  linespeed = arg(NULL, 0);

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    socket = (socket_t*)gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t*)(socket->data);
    if (strcmp(transfer->user, nick) == 0) {
      transfer->linespeed = atoi(linespeed);
      transfer_update(socket, 1);
    }
    i1++;
  }
}

HANDLER(cmd_eval_whois) {
  char* pos1;
  GtkEntry* entry;
  GtkWidget* temp;
  int t1;
  char str[200];
  int self;
  char nick[200];
  //  int cnt = 0;
  //  clone_t* clone;

  if (global.whois_win == NULL) {
    global.whois_win = create_whois_win();
  }
  gtk_window_set_title (GTK_WINDOW (global.whois_win), _("Whois User"));
  temp = lookup_widget(global.whois_win, "label116");
  gtk_label_set_text(GTK_LABEL(temp), _("Online"));

#ifdef GLOBAL_DEBUG
  printf("whois: %d\n", global.status.whois_hide);
#endif
  if (global.status.whois_hide == 0) {
    gtk_widget_show(global.whois_win);
  } else {
    global.status.whois_hide--;
  }

  pos1 = arg(data, 0);
  temp = lookup_widget(global.whois_win, "label527");
  gtk_label_set_text(GTK_LABEL(temp), pos1);
  if (strcmp(pos1, SERVER->nick) == 0) self = 1;
  else self = 0;
  strcpy(nick, pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry25"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, _("Level: "));
  gtk_entry_append_text(entry, pos1);
  if (self) {
    global.user.level = level2int(pos1);
#ifdef GLOBAL_DEBUG
    printf("user level: %d\n", global.user.level);
#endif
    if (global.user.level >= L_MOD) {
      temp = lookup_widget(global.win, "global_users");
      gtk_widget_set_sensitive(temp, TRUE);
      temp = lookup_widget(global.win, "banned_users");
      gtk_widget_set_sensitive(temp, TRUE);
    }
  }

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry26"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  sscanf(pos1, "%d", &t1);
  if (t1 > 60*60*24) 
    sprintf(str, _("%d days %d hours %d minutes %d seconds"), 
	    t1/(24*60*60), (t1%(24*60*60))/(60*60), 
	    (t1%(60*60))/60, t1%60);
  else if (t1 > 60*60)
    sprintf(str, _("%d hours %d minutes %d seconds"), 
	    t1/(60*60), (t1%(60*60))/60, t1%60);
  else
    sprintf(str, _("%d minutes %d seconds"), t1/60, t1%60);
  gtk_entry_set_text(entry, str);
  
  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry27"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry28"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, _("Status: "));
  gtk_entry_append_text(entry, pos1);
  if (self) {
    global.user.status = status2int(pos1);
  }

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry62"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, _("Shared: "));
  gtk_entry_append_text(entry, pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry30"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, pos1);
  gtk_entry_append_text(entry, _(" downs"));

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry59"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, pos1);
  gtk_entry_append_text(entry, _(" ups"));

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry61"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  sscanf(pos1, "%d", &t1);
  gtk_entry_set_text(entry, _("Link: "));
  gtk_entry_append_text(entry, LineSpeed(t1));

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry60"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, pos1);
  //  if (!strncmp("v2.0 BETA 6", pos1, 11)) cnt++;

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry34"));
  gtk_entry_set_text(entry, "");

  if (self) update_user_stats();
  
  // Vetinari18 "User" 3027 " #German " "Active" 1415 0 0 5 "v2.0 BETA 7"
  // 3 1 213.6.110.212 1083 0 unknown
  temp = lookup_widget(global.whois_win, "table18");
  if (global.user.level < L_MOD) {
    gtk_widget_hide(temp);
    return;
  } else {
    gtk_widget_show(temp);
  }

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry63"));
  gtk_entry_set_text(entry, pos1);
  gtk_entry_append_text(entry, _(" downs"));

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry64"));
  gtk_entry_set_text(entry, pos1);
  gtk_entry_append_text(entry, _(" ups"));

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry65"));
  gtk_entry_set_text(entry, pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry66"));
  gtk_entry_set_text(entry, _("Server: "));
  gtk_entry_append_text(entry, pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry67"));
  gtk_entry_set_text(entry, _("Data: "));
  gtk_entry_append_text(entry, pos1);
  //  if ((pos1[0] == '1') && (pos1[1] == '5')) cnt++;

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry68"));
  gtk_entry_set_text(entry, pos1);

}

HANDLER(cmd_eval_whowas) {
  char* pos1;
  GtkEntry* entry;
  GtkWidget* temp;
  time_t last_seen;
  char str[200];
  
  if (global.whois_win == NULL) {
    global.whois_win = create_whois_win();
  }
  gtk_widget_show(global.whois_win);

  gtk_window_set_title (GTK_WINDOW (global.whois_win), _("Whowas User"));
  
  temp = lookup_widget(global.whois_win, "label116");
  gtk_label_set_text(GTK_LABEL(temp), _("Last Seen"));

  pos1 = arg(data, 0);
  temp = lookup_widget(global.whois_win, "label527");
  gtk_label_set_text(GTK_LABEL(temp), pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry25"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, _("Level: "));
  gtk_entry_append_text(entry, pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry26"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  last_seen = strtoul(pos1, NULL, 10);
  sprintf(str, "%s", ctime(&last_seen));
  gtk_entry_set_text(entry, str);

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry34"));
  gtk_entry_set_text(entry, "");


  temp = lookup_widget(global.whois_win, "entry60");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);

  temp = lookup_widget(global.whois_win, "entry61");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);

  temp = lookup_widget(global.whois_win, "entry62");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);

  temp = lookup_widget(global.whois_win, "entry28");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);

  temp = lookup_widget(global.whois_win, "entry27");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);

  temp = lookup_widget(global.whois_win, "entry30");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);

  temp = lookup_widget(global.whois_win, "entry59");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);

  temp = lookup_widget(global.whois_win, "table18");
  gtk_widget_hide(temp);
}

HANDLER(cmd_upload_request) {
  transfer_t* new_trans;
  socket_t* socket;
  char* user;
  char* winname;
  char* speed;
  //  file_t* file;

  user = arg(data, 0);
  winname = arg(NULL, 0);
  speed = arg(NULL, 0);

  // do not allow uploads from enemies
  if (is_string_in_list(global.enemyuser, user)) return;
  if (global.status.exiting == E_SAFE) return;

  socket = transfer_is_in_upload(user, winname);
  
  if (!socket) {
    new_trans = transfer_new();
    new_trans->winname = strdup(winname);
    new_trans->longname = strdup(winname);
    convert_to_unix(new_trans->longname);
    new_trans->shortname =
      strdup(extract_short_name(new_trans->longname));
    new_trans->user = strdup(user);
    new_trans->type = T_UPLOAD;
    new_trans->mime_type = get_mimetype(winname);
    
    socket = socket_new(S_TRANSFER);
    socket->data = new_trans;

    // better check this when connected to user?
    /*
    file = lib_search_transfer(new_trans);
    if (file == NULL) {
    g_warning("requested file not shared");
      transfer_destroy(new_trans);
      return;
      }
    */
    if (speed)   // on napster
      new_trans->linespeed = atoi(speed);
    else         // on opennap
      send_command(CMD_CLIENT_USERSPEED, new_trans->user);
    //    new_trans->size = file->size;
    transfer_insert(socket);
    upload_start(socket);
  } else {
    socket_destroy(socket, S_REJECT);
  }

}

HANDLER(cmd_set_port) {
  int port;

  port = atoi(data);
  client_message(_("Message"), 
		 _("Server has set your data port to %d!"), port);
  create_upload_port(port, FALSE);
}

HANDLER(cmd_channel_list_end) {
  set_options_channels();
}

HANDLER(cmd_channel_list) {
  GtkWidget* temp;
  char* pos;

  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[0], data);
  data = pos+1;
  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[1], data);
  data = pos+1;
  
  strcpy(tstr[4], data);
  
  tstr[2][0] = tstr[3][0] = 0;
  temp = lookup_widget(global.win, "channel_list");
  gtk_clist_append(GTK_CLIST(temp), list);
}

HANDLER(cmd_remote_queued) {
  socket_t* socket;
  char* user;
  char* winname;
  //  char* max;
  
  user = arg(data, 0);
  winname = arg(NULL, 0);
  //  max = arg(NULL, 0);

  change_user_info(user, -2);

  socket = transfer_is_in_download(user, winname);

  if (socket) {
    socket_end(socket, S_REMOTE);
  } else g_warning("transfer not found in downloads");
}

HANDLER(cmd_motd) {
  char* mode;
  chat_page_t* page;

  if (SERVER->network == N_UNKNOWN) {
    if (get_version(data)) {
      if (global.usermode) {
	mode = make_string_from_list(global.usermode, " ");
	send_command(CMD_CLIENT_USER_MODE, "NONE");
	send_command(CMD_CLIENT_USER_MODE, mode);
      }
      lopster_links(NULL);
    }
    update_user_stats();
    setup_sensitive(0);
    gtk_idle_add(lib_commit_idle, NULL);
  }
  if (!global.links) get_server(data);
  
  page = chat_page_search("Console", P_FIX);
  if (!page) return;
  chat_print_colored(page, M_PUBLIC, "text", data);
  chat_print_colored(page, M_PUBLIC, "text", "\n");
}

HANDLER(cmd_data_port_error) {
  client_message(_("Message"), _("<%s> reports that your dataport is not reachable!"),
		 data);
}

HANDLER(cmd_operator_message) {
  char t[1024];
  char* user;
  char* message;
  chat_page_t* page;

  page = chat_page_search("Console", P_FIX);
  if (!page) return;

  chat_print_channel(page, M_PUBLIC, "message", _("(Operator) "));
  
  sprintf(t, "%s %s", "Console", data);
  cmd_public_message_no_log(t, M_PUBLIC);

  user = arg(data, 0);
  message = arg(NULL, 1);
  log("Operator", LOG_OTHER, "<%s> %s\n", user, message);
}

HANDLER(cmd_global_message) {
  char* command;
  char* user;
  char* message;

  chat_print("message", _("(Global) "));
  
  command = g_strdup_printf("%s %s", global.current_page->name, 
			    data);
  cmd_public_message_no_log(command, M_PUBLIC);
  g_free(command);

  user = arg(data, 0);
  message = arg(NULL, 1);

  log("Global", LOG_OTHER, "<%s> %s\n", user, message);
}

HANDLER(cmd_nick_banlist) {
  GtkWidget* temp;

  if (!global.ban_win) return;
  temp = lookup_widget(global.ban_win, "clist3");
  strcpy(tstr[0], data);
  tstr[1][0] = 0;
  tstr[2][0] = 0;
  tstr[3][0] = 0;
  gtk_clist_append(GTK_CLIST(temp), list);
}

HANDLER(cmd_browse_direct) {
  socket_t* socket;
  char* nick;
  char* ip;
  char* port;

  if (!data) return;
  
  nick = arg(data, 0);
  ip = arg(NULL, 0);
  port = arg(NULL, 0);

  if (nick && ip && port) {
    socket = browse_search(nick);
    if (!socket) return;
    socket->port = atoi(port);
    socket->ip_long = BSWAP32(strtoul(ip, NULL, 10));
    socket->data = strdup(nick);
    if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
      return;
    }
    socket->input = 
      gdk_input_add(socket->fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(await_conn_ack3), socket);
  } else {
    client_message(NULL, _("%s is trying to browse your files"), nick);
    send_command(CMD_SERVER_BROWSE_DIRECT_OK, nick);
  }
}

HANDLER(cmd_browse_direct_ok) {
  char* nick;
  char* ip;
  char* port;
  socket_t* socket;

  nick = arg(data, 0);
  ip = arg(NULL, 0);
  port = arg(NULL, 0);

  if (!nick || !ip || !port) return;
  
  if (atoi(port) > 0) {
    socket = browse_search(nick);
    socket->port = htons(atoi(port));
    socket->ip_long = BSWAP32(strtoul(ip, NULL, 10));
    browse_connect_and_start(socket);
  }
}

HANDLER(cmd_browse_direct_err) {
  char* nick;
  char* message;
  socket_t* socket;

  nick = arg(data, 0);
  message = arg(NULL, 0);

  socket = browse_search(nick);
  if (!socket) return;
  
  client_message("Browse", message);
  socket_destroy(socket, 0);
}

HANDLER(cmd_ghost) {
  client_message(_("Message"), _("Someone else tries to login with your Nick"));
}

HANDLER(cmd_sping) {
  time_t tim;
  char t[1024];

  time(&tim);
  strcpy(t, ctime(&tim));
  t[strlen(t)-1] = 0;
  client_message(_("Pong"), _("Pong from server [%s]"), t);
}

HANDLER(cmd_ping) {
  client_message(_("Message"), _("%s has pinged you!"), data);
  send_command(CMD_CLIENT_PONG, data);
}

HANDLER(cmd_pong) {
  time_t tim;
  char t[1024];

  time(&tim);
  strcpy(t, ctime(&tim));
  t[strlen(t)-1] = 0;
  client_message(_("Message"), _("Pong from %s [%s]"), data, t);
}

HANDLER(cmd_emote) {
  char* channel;
  char* user;
  char style[1024];
  char* message;
  chat_page_t* page;

  channel = arg(data, 0);
  user = arg(NULL, 0);
  message = arg(NULL, 0);   // quoted
  
  log(channel, LOG_CHANNEL, "<%s %s>\n", user, message);
  
  if (global.options.public_ignore) {
    if (is_string_in_list(global.ignored_users, user)) return;
  }
  
  page = chat_page_search(channel, P_PUBLIC);
  if (!page) return;

  chat_print_channel(page, M_PUBLIC, "user", "<");
  if (is_string_in_list(global.frienduser, user))
    strcpy(style, "friend");
  else if (!strcasecmp(SERVER->nick, user))
    strcpy(style, "yourself");
  else
    strcpy(style, "user");

  chat_print_channel(page, M_PUBLIC, style, user);
  chat_print_channel(page, M_PUBLIC, style, " ");

  chat_print_colored(page, M_PUBLIC, style, message);
  chat_print_channel(page, M_PUBLIC, "user", ">\n");
}

HANDLER(cmd_channel_list_entry) {
  char* pos;
  GtkWidget* temp;

  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[0], data);
  data = pos+1;
  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[1], data);
  data = pos+1;
  
  data = strchr(data, ' ')+1;
  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[3], data);
  data = pos+1;
  
  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[2], data);
  data = pos+2;
  
  pos = strrchr(data, '\"');
  pos[0] = 0;
  strcpy(tstr[4], data);
  
  temp = lookup_widget(global.win, "channel_list");
  gtk_clist_append(GTK_CLIST(temp), list);
}

HANDLER(cmd_global_user_end) {
  GtkWidget* temp;

  if (!global.user_win) return;

  temp = lookup_widget(global.user_win, "button164");
  gtk_widget_set_sensitive(temp, TRUE);
}

HANDLER(cmd_global_user) {
  char* nick;
  char* ip;
  GtkCList* clist;
  clone_t* clone;

  nick = arg(data, 0);
  ip = arg(NULL, 0);
  
  if ((clone = is_ip_in_clones(global.clones, ip)) != NULL) {
    clone->nicks = g_list_append(clone->nicks, strdup(nick));
  } else {
    clone = (clone_t*)malloc(sizeof(clone_t));
    clone->ip = strdup(ip);
    clone->nicks = g_list_append(NULL, strdup(nick));
    global.clones = g_list_append(global.clones, clone);
  }
  
  if (!global.user_win) return;
  
  clist = GTK_CLIST(lookup_widget(global.user_win, "clist12"));
  strcpy(tstr[0], nick);
  strcpy(tstr[1], ip);
  gtk_clist_append(clist, list);
}

HANDLER(cmd_server_links) {
  static GList* ldata = NULL;
  char* s1, *s2;
  link_t* new_link;
  link_t* new_link2;
  GList* dlist;
  int cnt;
  int who;

  if (strlen(data) > 0) {
    ldata = g_list_append(ldata, strdup(data));
    return;
  }
  
  if (!ldata) return;

  if (!global.links) {
    free_links(global.links);
    global.links = NULL;
  }
  global.links = (link_t*)malloc(sizeof(link_t));
  global.links->server = strdup(get_s1(ldata->data));
  global.links->links = NULL;
  global.links->ping = NULL;

  while (ldata) {
    cnt = 0;
    for (dlist = ldata; dlist; dlist = dlist->next) {
      who = 0;
      data = (char*)(dlist->data);
      s1 = strdup(get_s1(data));
      s2 = strdup(get_s2(data));
      new_link = link_search_rec(global.links, s1);
      if (new_link) {
	who = 1;
      } else {
	new_link = link_search_rec(global.links, s2);
	if (new_link) who = 2;
      }
      if (who) {
	new_link2 = (link_t*)malloc(sizeof(link_t));
	if (who == 1) new_link2->server = strdup(s2);
	else new_link2->server = strdup(s1);
	new_link2->links = NULL;
	new_link2->ping = NULL;
	new_link->links = 
	  g_list_append(new_link->links, new_link2);

	add_allowed_link(new_link->server, new_link2->server);
	add_allowed_link(new_link2->server, new_link->server);
	
	cnt++;
	ldata = g_list_remove(ldata, data);
	free(data);
	break;
      }
      free(s1);
      free(s2);
    }
    if (cnt == 0) break;
  }
  if (ldata) {
    client_message(_("Error"), _("Could not calculate Link tree"));
    for (dlist = ldata; dlist; dlist = dlist->next) {
      free(dlist->data);
    }
    g_list_free(ldata);
    ldata = NULL;
  }
  //  print_links(global.links, 0);
}

HANDLER(cmd_whowas) {
  char* pos1;
  GtkEntry* entry;
  GtkWidget* temp;
  time_t last_seen;
  char str[200];
  
  if (global.whois_win == NULL) {
    global.whois_win = create_whois_win();
  }
  gtk_widget_show(global.whois_win);

  gtk_window_set_title (GTK_WINDOW (global.whois_win), _("Whowas User"));
  
  temp = lookup_widget(global.whois_win, "label116");
  gtk_label_set_text(GTK_LABEL(temp), _("Last Seen"));

  pos1 = arg(data, 0);
  temp = lookup_widget(global.whois_win, "label527");
  gtk_label_set_text(GTK_LABEL(temp), pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry61"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, ntoa(strtoul(pos1, NULL, 10)));
  
  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry62"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, pos1);
  
  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry26"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  last_seen = strtoul(pos1, NULL, 10);
  sprintf(str, "%s", ctime(&last_seen));
  gtk_entry_set_text(entry, str);

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry34"));
  gtk_entry_set_text(entry, "");

  // --
  temp = lookup_widget(global.whois_win, "entry25");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry28");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry60");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry27");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry30");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry59");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);

  temp = lookup_widget(global.whois_win, "table18");
  gtk_widget_hide(temp);
}

HANDLER(cmd_usage_stats) {
  char* pos;
  time_t stime;
  int t1;
  char str[1024];
  unsigned long gigs;

  pos = arg(data, 0);
  server_message(_("Local Clients"), pos);

  pos = arg(NULL, 0);
  server_message(_("Local Servers"), pos);
  
  pos = arg(NULL, 0);
  server_message(_("Users"), pos);
  
  pos = arg(NULL, 0);
  server_message(_("Files"), pos);
  
  pos = arg(NULL, 0);
  gigs = strtoul(pos, NULL, 10);
  sprintf(str, "%ld", gigs/1024/1024);
  server_message(_("Gigs"), str);
  
  pos = arg(NULL, 0);
  server_message(_("Channels"), pos);
  
  pos = arg(NULL, 0);
  sscanf(pos, "%ld", &stime);
  strcpy(str, ctime(&stime));
  str[strlen(str)-1] = 0;
  server_message(_("Boot Time"), str);
  
  pos = arg(NULL, 0);
  t1 = strtoul(pos, NULL, 10);
  sprintf(str, _("%d days %d hours %d minutes %d seconds"), 
	  t1/(60*60*24), 
	  (t1%(60*60*24))/(60*60),
	  (t1%(60*60))/60, 
	  t1%60);
  server_message(_("Uptime"), str);
  
  pos = arg(NULL, 0);
  if (!pos) return;
  server_message(_("Memory"), pos);
  
  pos = arg(NULL, 0);
  if (!pos) return;
  server_message(_("Registered Users"), pos);

  pos = arg(NULL, 0);
  if (!pos) return;
  server_message(_("KBytes In/sec"), pos);

  pos = arg(NULL, 0);
  if (!pos) return;
  server_message(_("KBytes Out/sec"), pos);

  pos = arg(NULL, 0);
  if (!pos) return;
  server_message(_("Searches/sec"), pos);

  pos = arg(NULL, 0);
  if (!pos) return;
  gigs = strtoul(pos, NULL, 10);
  sprintf(str, "%ld", gigs/1024/1024);
  server_message(_("MBytes In"), str);

  pos = arg(NULL, 0);
  if (!pos) return;
  gigs = strtoul(pos, NULL, 10);
  sprintf(str, "%ld", gigs/1024/1024);
  server_message(_("MBytes Out"), str);
}

HANDLER(cmd_version_stats) {
  GtkCList* clist;
  GdkColor color = { 0, 0x9900, 0x0f00, 0x0f00 };
  GtkWidget* temp;
  GList* dlist;
  char* info;
  char* number;
  client_t* client;
  client_t* client2;
  char* short_info;
  int flag;
  int threshold;
  int row;

  if (!global.client_win) return;

  clist = GTK_CLIST(lookup_widget(global.client_win, "clist8"));
  if (strlen(data) > 0) {
    info = arg(data, 0);
    number = arg(NULL, 0);
    client = (client_t*)malloc(sizeof(client_t));
    client->cnt = 0;
    client->info = strdup(info);
    if (number) client->logins = atoi(number);
    else client->logins = 0;

    global.client_list = g_list_append(global.client_list, client);
    
    short_info = strtok(info, " -");
    if (!short_info) short_info = info;
    client2 = is_client_in_list(global.client_list, short_info);
    if (!client2) {
      client2 = (client_t*)malloc(sizeof(client_t));
      client2->cnt = 1;
      client2->info = strdup(short_info);
      client2->logins = client->logins;
      global.client_list = g_list_append(global.client_list, client2);
    } else {
      client2->cnt++;
      client2->logins += client->logins;
    }
    return;
  }
  
  temp = lookup_widget(global.client_win, "checkbutton30");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp))) {
    temp = lookup_widget(global.client_win, "spinbutton19");
    threshold = 
      gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp));
  } else {
    threshold = 0;
  }

  flag = 0;
  temp = lookup_widget(global.client_win, "checkbutton32");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp)))
    flag |= 1;  // single
  temp = lookup_widget(global.client_win, "checkbutton31");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp)))
    flag |= 2;  // sum
  
  gtk_clist_freeze(clist);
  dlist = g_list_first(global.client_list);
  while (dlist) {
    client = (client_t*)(dlist->data);
    if (client->logins >= threshold) {
      if ((client->cnt == 0) && (flag & 1)) {
	strcpy(tstr[1], client->info);
	sprintf(tstr[0], "%d", client->logins);
	row = gtk_clist_append(clist, list);
      }
      if ((client->cnt > 0) && (flag & 2)) {
	sprintf(tstr[1], "%s%s", client->info, (flag&1)?_(" (SUM)"):"");
	sprintf(tstr[0], "%d", client->logins);
	row = gtk_clist_append(clist, list);
	if (flag&1) {
	  gdk_color_alloc (gtk_widget_get_colormap (GTK_WIDGET (clist)),
			   &color);
	  gtk_clist_set_foreground(clist, row, &color);
	}
      }
    }
    dlist = dlist->next;
  }
  gtk_clist_thaw(clist);
}

HANDLER(cmd_user_mode) {
  client_message(_("Usermode"), data);
}

// deprecated
HANDLER(cmd_browse_new) {
  file_create_from_browse_new(data);
}
