#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <strings.h>
#include <sys/poll.h>

#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>

#include "callbacks.h"
#include "connection.h"
#include "support.h"
#include "global.h"
#include "transfer.h"
#include "log.h"
#include "handler.h"

const char* status_names[21] = {
  "Inactive", "Connecting...", "Getting info...", "Downloading",
  "Canceled",     "Finished!", "Timeout!",           "Rejected",
  "Incomplete",   "Uploading", "Getting info",   "Getting Info",
  "Getting Info",   "Waiting", "Firewalled", "Connection Error",
  "Queued", "Remotely Queued", "Unavailable!",        "Stopped",
  "Resume Error"
};

file_t* global_file2;
int temp_trans = 0;

GtkWidget*
create_download_popup_test (transfer_t* transfer) {
  GtkWidget *download_popup;
  GtkAccelGroup *download_popup_accels;
  GtkWidget *cancel_download;
  GtkWidget *delete_download;
  GtkWidget *separator3;
  GtkWidget *retry_download;
  GtkWidget *resume_download;
  GtkWidget *separator4;
  GtkWidget *who_is_user;
  GtkWidget *trennlinie7;
  GtkWidget *play_file;

  download_popup = gtk_menu_new ();
  gtk_object_set_data (GTK_OBJECT (download_popup), "download_popup", download_popup);
  download_popup_accels = gtk_menu_ensure_uline_accel_group (GTK_MENU (download_popup));

  cancel_download = gtk_menu_item_new_with_label ("Cancel Download");
  gtk_widget_ref (cancel_download);
  gtk_object_set_data_full (GTK_OBJECT (download_popup), "cancel_download", cancel_download,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (cancel_download);
  gtk_container_add (GTK_CONTAINER (download_popup), cancel_download);

  if (!transfer_is_active(transfer)) {
    gtk_widget_set_sensitive(cancel_download, FALSE);
  }

  delete_download = gtk_menu_item_new_with_label ("Delete Download");
  gtk_widget_ref (delete_download);
  gtk_object_set_data_full (GTK_OBJECT (download_popup), "delete_download", delete_download,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (delete_download);
  gtk_container_add (GTK_CONTAINER (download_popup), delete_download);

  separator3 = gtk_menu_item_new ();
  gtk_widget_ref (separator3);
  gtk_object_set_data_full (GTK_OBJECT (download_popup), "separator3", separator3,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (separator3);
  gtk_container_add (GTK_CONTAINER (download_popup), separator3);
  gtk_widget_set_sensitive (separator3, FALSE);

  if (!transfer_is_active(transfer)) {
    retry_download = gtk_menu_item_new_with_label ("Retry Download");
    gtk_widget_ref (retry_download);
    gtk_object_set_data_full (GTK_OBJECT (download_popup), "retry_download", retry_download,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (retry_download);
    gtk_container_add (GTK_CONTAINER (download_popup), retry_download);

    gtk_signal_connect (GTK_OBJECT (retry_download), "activate",
			GTK_SIGNAL_FUNC (on_retry_download_activate),
			NULL);
  }

#ifdef TRANSFER_DEBUG
  printf("status %d\n", transfer->status);
#endif
  if (transfer_is_active(transfer)) {
    if (transfer->status == S_STOPPED) {
      resume_download = gtk_menu_item_new_with_label ("Continue Download");
      gtk_widget_ref (resume_download);
      gtk_object_set_data_full (GTK_OBJECT (download_popup), "resume_download", resume_download,
				(GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show (resume_download);
      gtk_container_add (GTK_CONTAINER (download_popup), resume_download);
      gtk_signal_connect (GTK_OBJECT (resume_download), "activate",
			  GTK_SIGNAL_FUNC (on_resume_download_activate),
			  NULL);
      //      gtk_widget_set_sensitive(resume_download, FALSE);
    } else if (transfer->status == S_DOWNLOADING) {
      resume_download = gtk_menu_item_new_with_label ("Suspend Download");
      gtk_widget_ref (resume_download);
      gtk_object_set_data_full (GTK_OBJECT (download_popup), "stop_download", resume_download,
				(GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show (resume_download);
      gtk_container_add (GTK_CONTAINER (download_popup), resume_download);
      gtk_signal_connect (GTK_OBJECT (resume_download), "activate",
			  GTK_SIGNAL_FUNC (on_stop_download_activate),
			  NULL);
      //      gtk_widget_set_sensitive(resume_download, FALSE);
    }
  }

  separator4 = gtk_menu_item_new ();
  gtk_widget_ref (separator4);
  gtk_object_set_data_full (GTK_OBJECT (download_popup), "separator4", separator4,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (separator4);
  gtk_container_add (GTK_CONTAINER (download_popup), separator4);
  gtk_widget_set_sensitive (separator4, FALSE);

  who_is_user = gtk_menu_item_new_with_label ("Who is User");
  gtk_widget_ref (who_is_user);
  gtk_object_set_data_full (GTK_OBJECT (download_popup), "who_is_user", who_is_user,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (who_is_user);
  gtk_container_add (GTK_CONTAINER (download_popup), who_is_user);

  trennlinie7 = gtk_menu_item_new ();
  gtk_widget_ref (trennlinie7);
  gtk_object_set_data_full (GTK_OBJECT (download_popup), "trennlinie7", trennlinie7,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (trennlinie7);
  gtk_container_add (GTK_CONTAINER (download_popup), trennlinie7);
  gtk_widget_set_sensitive (trennlinie7, FALSE);

  play_file = gtk_menu_item_new_with_label ("Play File");
  gtk_widget_ref (play_file);
  gtk_object_set_data_full (GTK_OBJECT (download_popup), "play_file", play_file,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (play_file);
  gtk_container_add (GTK_CONTAINER (download_popup), play_file);

  gtk_signal_connect (GTK_OBJECT (cancel_download), "activate",
                      GTK_SIGNAL_FUNC (on_cancel_download_activate),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (delete_download), "activate",
                      GTK_SIGNAL_FUNC (on_delete_download_activate),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (who_is_user), "activate",
                      GTK_SIGNAL_FUNC (on_who_is_user_activate2),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (play_file), "activate",
                      GTK_SIGNAL_FUNC (on_play_file_activate),
                      NULL);

  return download_popup;
}


transfer_t* transfer_new() {
  transfer_t *transfer;
  int i1;

  transfer = (transfer_t*)malloc(sizeof(transfer_t));

  transfer->longname = NULL;        //
  transfer->shortname = NULL;       //
  transfer->winname = NULL;         //
  transfer->localname = NULL;
  transfer->md5 = NULL;
  transfer->user = NULL;
  transfer->linespeed = 0;
  transfer->progress = 0;
  transfer->rate = 0;
  transfer->timeleft = 0;
  transfer->start_time = 0;
  transfer->status = S_WAITING;
  transfer->size = 0;
  transfer->sock.address.ip = NULL;
  transfer->sock.address.ip_long = 0;
  transfer->sock.address.port = 0;
  transfer->sock.address.description = NULL;
  transfer->sock.address.no_connected = 0;
  transfer->sock.fd = -1;
  transfer->sock.input = -1;
  transfer->sock.timeout = -1;
  transfer->sock.cnt = 0;
  transfer->sock.max_cnt = 500;
  transfer->file = NULL;
  transfer->type = T_UNKNOWN;
  transfer->resume_check = NULL;
  transfer->check_length = 0;
  for (i1 = 0; i1 < TRANSFER_HISTORY; i1++)
    transfer->history[i1] = 0;
  transfer->hist_pos = 0;
  transfer->hist_cnt = 0;

  return transfer;
}

int transfer_is_active(transfer_t* transfer) {
  if ((transfer->status != S_WAITING) &&
      (transfer->status != S_CONNECTING) &&
      (transfer->status != S_INFO) &&
      (transfer->status != S_INFO1) &&
      (transfer->status != S_INFO2) &&
      (transfer->status != S_INFO3) &&
      (transfer->status != S_STOPPED) &&  // stopped, but active
      (transfer->status != S_QUEUED) &&   // stopped, but active
      (transfer->status != S_REMOTE) &&   // stopped, but active
      (transfer->status != S_UPLOADING) &&
      (transfer->status != S_DOWNLOADING)) {
    return 0;
  } else {
    return 1;
  }
}

void rec_search2(GtkCTree *ctree, GtkCTreeNode *node,
		 char* data) {
  file_t* file;

  if (!GTK_CTREE_ROW (node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (!strcmp(data, file->winname)) {
    global_file2 = file;
  }
}

file_t* lib_search_transfer(transfer_t* transfer) {
  GtkCTree* ctree;

  global_file2 = NULL;
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, NULL, 
			   (GtkCTreeFunc)rec_search2,
			   transfer->winname);
  return global_file2;
}

void download_start(transfer_t* transfer) {
#ifdef TRANSFER_DEBUG  
  printf("starting down [%s]\n", transfer->winname);
#endif
  if ((global.network.max_downloads > global.network.cur_downloads) &&
      (global.network.max_transfers > global.network.cur_downloads+global.network.cur_uploads)) {
    transfer->sock.timeout =
      gtk_timeout_add(UPDATE_DELAY, transfer_timeout_cb, transfer);
    transfer->status = S_WAITING;
    transfer_update(transfer);
    global.network.cur_downloads++;
#ifdef TRANSFER_DEBUG  
    printf("++ now %d\n", global.network.cur_downloads);
#endif
    send_command(CMD_CLIENT_DOWNLOAD, "%s \"%s\"", transfer->user, transfer->winname);
  } else {
    transfer_end(transfer, S_QUEUED);
  }
}

void upload_start(transfer_t* transfer) {
#ifdef TRANSFER_DEBUG  
  printf("starting up [%s]\n", transfer->winname);
#endif
  if ((global.network.max_uploads > global.network.cur_uploads) &&
      (global.network.max_transfers > global.network.cur_downloads+global.network.cur_uploads)) {
    send_command(CMD_CLIENT_UPLOAD_OK, "%s \"%s\"", transfer->user,
		 transfer->winname);
    send_command(CMD_CLIENT_USERSPEED, transfer->user);
    transfer->status = S_WAITING;
    transfer->sock.timeout =
      gtk_timeout_add(UPDATE_DELAY, transfer_timeout_cb, transfer);
    transfer_update(transfer);
    global.network.cur_uploads++;
#ifdef TRANSFER_DEBUG  
    printf("+ now %d\n", global.network.cur_uploads);
#endif
  } else {
    send_command(CMD_CLIENT_LIMIT, "%s \"%s\" %d", transfer->user, 
		 transfer->winname, global.network.cur_uploads);
    transfer_end(transfer, S_QUEUED);
  }

}

void download_file(file_t* file) {
  transfer_t* new_trans;

  new_trans = transfer_new();
  new_trans->longname = strdup(file->longname);
  new_trans->shortname = strdup(file->shortname);
  new_trans->winname = strdup(file->winname);
  new_trans->linespeed = file->linespeed;
  new_trans->user = strdup(file->user);
  new_trans->type = T_DOWNLOAD;
  new_trans->size = file->size;

  transfer_insert_download(new_trans);
  download_start(new_trans);
}

void cmd_download_ack(char* data) {
  transfer_t* transfer;
  char* pos1;
  char* pos2;
  long temp;

  char* user;
  int port;
  unsigned long ip_long;
  char* winname;
  char* md5;
  int linespeed;

  pos1 = data;
  pos2 = strchr(pos1, ' ');
  pos2[0] = 0;
  user = strdup(pos1);
  
  pos1 = pos2+1;
  pos2 = strchr(pos1, '\"');
  sscanf(pos1, "%lu %d", &temp, &port);
  ip_long = BSWAP32(temp);

  pos1 = pos2;
  pos2 = strrchr(pos1, '\"');
  pos2[0] = '\0';
  winname = strdup(pos1+1);

  pos1 = pos2+2;
  pos2 = strchr(pos1, ' ');
  pos2[0] = '\0';
  md5 = strdup(pos1);

  pos1 = pos2+1;
  linespeed = atoi(pos1);

  transfer = transfer_is_in_download(user, winname);
  free(user);
  free(winname);

  if (!transfer) {
    g_warning("transfer not found in download list, already deleted?");
    return;
  }
  if (!transfer_is_active(transfer)) {
    printf("transfer no longer active, ignoring download ack\n");
    return;
  }

  transfer->sock.address.port = port;
  transfer->sock.address.ip_long = ip_long;
  transfer->md5 = md5;
  transfer->linespeed = linespeed;
  
  if (transfer->sock.address.port == 0) {
    if ((global.network.port == 0) || global.network.firewall) {
      printf("both firewalled, cannot do download\n");
      transfer_end(transfer, S_REJECT);
    } else {
      send_command(CMD_CLIENT_DOWNLOAD_FIREWALL, "%s \"%s\"", transfer->user, transfer->winname);
    }
  } else {
    if (connect_to_user(transfer)) return;
    transfer->sock.input = 
      gdk_input_add(transfer->sock.fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(outgoing_header), transfer);
  }
}

void cmd_upload_request(char* data) {
  transfer_t* new_trans;
  char* user;
  char* winname;
  char* pos;
  file_t* file;

  pos = strchr(data, ' ');
  pos[0] = 0;
  user = strdup(data);
  data = pos+2;
  pos = strrchr(data, '\"');
  pos[0] = 0;
  winname = strdup(data);

  // do not allow uploads from enemies
  if (is_string_in_list(global.enemyuser, user)) return;
  
  new_trans = transfer_is_in_upload(user, winname);
  
  if (new_trans && (new_trans->status != S_QUEUED)) {
    printf("transfer already present\n");
    return;
  }

  if (!new_trans) {
    new_trans = transfer_new();
    new_trans->winname = strdup(winname);
    new_trans->longname = strdup(winname);
    convert_to_unix(new_trans->longname);
    new_trans->shortname =
      strdup(extract_short_name(new_trans->longname));
    new_trans->localname = NULL;
    new_trans->user = strdup(user);
    new_trans->type = T_UPLOAD;
  
    // better check this when connected to user?
    file = lib_search_transfer(new_trans);
    if (file == NULL) {
      g_warning("requested file not shared");
      free(new_trans);
      return;
    }
    new_trans->size = file->size;
    transfer_insert_upload(new_trans);
  }

  upload_start(new_trans);

  free(user);
  free(winname);
}

void cmd_alternate_ack(char* data) {
  transfer_t* transfer;
  char* pos1;
  char* pos2;
  long temp;

  char* user;
  int port;
  unsigned long ip_long;
  char* winname;
  char* md5;
  int linespeed;
  file_t* file;
  
  pos1 = data;
  pos2 = strchr(pos1, ' ');
  pos2[0] = 0;
  user = strdup(pos1);
  
  pos1 = pos2+1;
  pos2 = strchr(pos1, '\"');
  sscanf(pos1, "%lu %d", &temp, &port);
  ip_long = BSWAP32(temp);

  pos1 = pos2;
  pos2 = strrchr(pos1, '\"');
  pos2[0] = '\0';
  winname = strdup(pos1+1);

  pos1 = pos2+2;
  pos2 = strchr(pos1, ' ');
  pos2[0] = '\0';
  md5 = strdup(pos1);

  pos1 = pos2+1;
  linespeed = atoi(pos1);

  transfer = transfer_is_in_upload(user, winname);
  free(user);
  free(winname);

  if (!transfer) {
    // maybe the upload was deleted by user?
    g_warning("transfer not found in upload list, alread deleted?");
    return;
  }
  if (!transfer_is_active(transfer)) {
    printf("transfer no longer active, ignoring upload ack\n");
    return;
  }

  transfer->linespeed = linespeed;
  transfer->md5 = md5;
  transfer->sock.address.port = port;
  transfer->sock.address.ip_long = ip_long;

  file = lib_search_transfer(transfer);
  if (!file) {
    // this is the first lib check
    // should send someting to the user?
    g_warning("file not found in lib\n");
    return;
  }
  transfer->size = file->size;

  if (transfer->sock.address.port == 0) {
    g_warning("both are firewalled!");
    transfer_end(transfer, S_FIREWALL);
  } else {
    if (connect_to_user(transfer)) return;
    transfer->sock.input = 
      gdk_input_add(transfer->sock.fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(outgoing_header), transfer);
  }
}

transfer_t* transfer_is_in_download(char* user, char* winname) {
  GtkCList* clist;
  int i1;
  transfer_t* transfer;
  
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) {
      transfer = NULL;
      break;
    }
    transfer = (transfer_t*)gtk_clist_get_row_data(clist, i1);
    if ((strcmp(transfer->winname, winname) == 0) &&
	(strcmp(transfer->user, user) == 0)) {
      if ((transfer->status == S_WAITING) ||
	  (transfer->status == S_REMOTE))
	return transfer;
    }
    i1++;
  }

  return NULL;
}

transfer_t* transfer_is_in_upload(char* user, char* winname) {
  GtkCList* clist;
  int i1;
  transfer_t* transfer;
  
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) return NULL;
    transfer = (transfer_t*)gtk_clist_get_row_data(clist, i1);
    if ((strcmp(transfer->winname, winname) == 0) &&
	(strcmp(transfer->user, user) == 0)) {
      if ((transfer->status == S_WAITING) ||
	  (transfer->status == S_QUEUED))
	return transfer;
    }
    i1++;
  }

  return NULL;
}


GdkPixmap *transfer_draw_progress(GtkCList *clist, GdkPixmap *pixmap, double percent, 
				  int width, int active){
  int height;
  char text[128];
  static GdkGC *gc = NULL;
  GdkColor background = { 0, 0xbf00, 0xbf00, 0xbf00 };
  GdkColor foreground = { 0, 0xffee, 0xbbbb, 0xbbbb };
  GdkColor foreground2 = { 0, 0xbbbb, 0xbbbb, 0xffee };
  GdkWindowPrivate* temp_pix;
  
  if (!gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), &background )
      || !gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), &foreground )
      || !gdk_color_alloc( gtk_widget_get_colormap(GTK_WIDGET(clist)), &foreground2 ) ) {
    g_error("couldn't allocate colour");
  }
  
  height = GTK_CLIST(clist)->row_height;
  temp_pix = (GdkWindowPrivate*)pixmap;

  if (!pixmap || (temp_pix->width != width) ||
      (temp_pix->height != height)) {
    if (pixmap) {
      // the next line would crash the client after a few
      // column resizes
      //      free(temp_pix);
      temp_pix = NULL;
      pixmap = NULL;
    }
    pixmap = gdk_pixmap_new( global.win->window, width, 
			     GTK_CLIST(clist)->row_height, -1 );
  }
  
  if (!gc) gc = gdk_gc_new(global.win->window);

  // draw the upper/left black lines
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  gdk_draw_rectangle(pixmap, gc, TRUE, 0, 0, width, height);
  
  // draw the lower/right white lines
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->white_gc);
  gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, width-1, height-1);
  
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  // draw the background (unfilled progress)
  gdk_gc_set_foreground(gc, &background);
  gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, width - 2, height - 2);
  
  // draw the actual progress bar
  // draw the lower/right white lines
  if ((int)((width-2)*percent) > 0) {
    gdk_gc_copy(gc, GTK_WIDGET(clist)->style->white_gc);
    gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, (int)((width-2)*percent), height - 2);
  }

  // draw the upper/left black lines
  if ((int)((width-2)*percent)-1 > 0) {
    gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
    gdk_draw_rectangle(pixmap, gc, TRUE, 2, 2, (int)((width-2)*percent)-1, height - 3);
  }

  if ((int)((width-2)*percent)-2 > 0) {
    if (active) gdk_gc_set_foreground(gc, &foreground);
    else gdk_gc_set_foreground(gc, &foreground2);
    gdk_draw_rectangle(pixmap, gc, TRUE, 2, 2, (int)((width-2)*percent)-2, height - 4);
  }
  
  // draw the text on top
  g_snprintf( text, sizeof(text), "%.0f%%", percent*100);
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  gdk_draw_text( pixmap, GTK_WIDGET(clist)->style->font, gc, width/2 - 9, height - 3, text, 3 );
  
  return pixmap;
}

int transfer_update(transfer_t* transfer) {
  GtkCList* clist;
  int row;
  char str[200];
  time_t cur;
  int sec;
  double trate;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;

  time(&cur);
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  row = gtk_clist_find_row_from_data(clist, transfer);
  if (row < 0) {
    clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
    row = gtk_clist_find_row_from_data(clist, transfer);
  }
  if (row < 0) {
    printf("update: transfer not found\n");
    return 1;
  }

  // filename
  if ((transfer->type == T_DOWNLOAD) || (transfer->type == T_RESUME)) {
    if (transfer->localname) strcpy(str, transfer->localname);
    else if (transfer->shortname) strcpy(str, transfer->shortname);
    else strcpy(str, "Not known yet");
  } else if (transfer->type == T_UPLOAD) {
    if (transfer->shortname) strcpy(str, transfer->shortname);
    else strcpy(str, "Not known yet");
  } else {
    g_warning("transfer_update: this should not happen!\n");
    gtk_exit(1);
  }
  gtk_clist_set_text(clist, row, 0, str);

  // filesize
  sprintf(str, "%ld/%ld", transfer->progress, transfer->size);
  gtk_clist_set_text(clist, row, 1, str);

  // user
  if (transfer->user) strcpy(str, transfer->user);
  else strcpy(str, "Not known yet");
  gtk_clist_set_text(clist, row, 2, str);

  // status
  if ((transfer->status < 0) || (transfer->status > 20)) {
    g_warning("transfer_update: invalid status %d", 
	      transfer->status);
    gtk_exit(1);
  }
  if (transfer->status == S_WAITING)
    sprintf(str, "Waiting %d secs", 500-transfer->sock.cnt);
  else 
    strcpy(str, status_names[transfer->status]);
  gtk_clist_set_text(clist, row, 3, str);

  // linespeed
  if ((transfer->linespeed < 0) || (transfer->linespeed > 10)) {
    g_warning("transfer_update: invalid linespeed %d", 
	      transfer->linespeed);
    gtk_exit(1);
  }
  strcpy(str, LineSpeed[transfer->linespeed]);
  gtk_clist_set_text(clist, row, 4, str);

  // progress
  if ((transfer->status == S_DOWNLOADING) ||
      (transfer->status == S_FINISHED) ||
      (transfer->status == S_UPLOADING)) { 
    int last_prog;
    int first_prog;
    int intervals;

    if (transfer->size >0){
      gtk_clist_get_pixmap(clist, row, 5, &pixmap, &bitmap );
      
      pixmap = 
	transfer_draw_progress(clist, pixmap, 
			       (double)transfer->progress / 
			       (double)transfer->size, 
			       clist->column[5].width,
			       transfer->status != S_FINISHED);
      gtk_clist_set_pixmap(clist, row, 5, pixmap, NULL );
    }
    
    transfer->hist_pos++;
    if (transfer->hist_pos >= TRANSFER_HISTORY)
      transfer->hist_pos = 0;
    transfer->history[transfer->hist_pos] = transfer->progress;
    last_prog = transfer->hist_pos;
    if (transfer->hist_cnt < TRANSFER_HISTORY)
      transfer->hist_cnt++;
    if (transfer->hist_cnt <  TRANSFER_HISTORY)
      intervals = transfer->hist_cnt;
    else
      intervals = TRANSFER_HISTORY-1;
    first_prog = (last_prog+TRANSFER_HISTORY-intervals)%
      TRANSFER_HISTORY;

    if (transfer->status == S_FINISHED) {
      if (cur - transfer->start_time == 0) cur++;
      trate = transfer->size/(cur - transfer->start_time);
    } else {
      trate =
	(double)(transfer->history[last_prog]-
		 transfer->history[first_prog])/intervals;
    }
    sprintf(str, "%.2f kB/s", trate/1024);
  
    if (trate > 20*1024) {
      pixmap = global.pix.sgreen;
      bitmap = global.pix.sgreenb;
    } else if (trate > 4*1024) {
      pixmap = global.pix.syellow;
      bitmap = global.pix.syellowb;
    } else if (trate > 2*1024) {
      pixmap = global.pix.sred;
      bitmap = global.pix.sredb;
    } else if (trate > 0) {
      pixmap = global.pix.sgray;
      bitmap = global.pix.sgrayb;
    } else {
      pixmap = global.pix.dummy;
      bitmap = global.pix.dummyb;
    }
    gtk_clist_set_pixtext (clist, row, 6, str, 5, pixmap, bitmap);
    
    if (transfer->status == S_FINISHED) {
      sec = cur - transfer->start_time;
      sprintf(str, "%d:%s%d", sec/60, (sec%60<10)?"0":"", sec%60);
    } else {
      if (trate > 0) {
	sec = (int)((transfer->size-transfer->progress)/trate);
	sprintf(str, "%d:%s%d", sec/60, (sec%60<10)?"0":"", sec%60);
      } else {
	sprintf(str, "stalled");
      }
    }
    gtk_clist_set_text(clist, row, 7, str);
  }
  
  return 1;
}

void transfer_insert_download(transfer_t *transfer) {
  GtkCList* temp;
  int row;

  temp = GTK_CLIST(lookup_widget(global.win, "transfer_down"));

  tstr[0][0] = tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 
    tstr[5][0] = tstr[6][0] = 0;
  row = gtk_clist_append(temp, list);
  gtk_clist_set_row_data_full (temp, row, (gpointer)transfer, 
			       (GtkDestroyNotify)destroy_transfer_row);
  transfer_update(transfer);
}

void transfer_insert_upload(transfer_t *transfer) {
  GtkCList* temp;
  int row;

  temp = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  
  tstr[0][0] = tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 
    tstr[5][0] = tstr[6][0] = 0;
  row = gtk_clist_append(temp, &list[0]);
  gtk_clist_set_row_data_full (temp, row, (gpointer)transfer, 
			       (GtkDestroyNotify)destroy_transfer_row);
  transfer_update(transfer);
}


int connect_to_user(transfer_t* transfer) {
  int res;

  transfer->status = S_CONNECTING;
  res = connect_socket(&(transfer->sock));

  if ((res != 0) && (res != EINPROGRESS)) {
    transfer_end(transfer, S_CONERROR);
    return -1;
  }
  return 0;
}

void cmd_get_error(char* data) {
  transfer_t* transfer;
  char* user;
  char* winname;
  char* pos;
  
  pos = strchr(data, ' ');
  pos[0] = 0;
  user = strdup(data);
  data = pos+2;
  pos = strrchr(data, '\"');
  pos[0] = 0;
  winname = strdup(data);

  transfer = transfer_is_in_download(user, winname);
  free(user);
  free(winname);

  if (transfer) transfer_end(transfer, S_UNAVAILABLE);
  else {
    g_warning("transfer not found in downloads");
    gtk_exit(1);
  }
}

void cmd_remote_queued(char* data) {
  transfer_t* transfer;
  char* user;
  char* winname;
  char* pos;
  
  pos = strchr(data, ' ');
  pos[0] = 0;
  user = strdup(data);
  data = pos+2;
  pos = strrchr(data, '\"');
  pos[0] = 0;
  winname = strdup(data);

  transfer = transfer_is_in_download(user, winname);
  free(user);
  free(winname);

  if (transfer) {
    transfer_end(transfer, S_REMOTE);
    transfer->sock.timeout = 
      gtk_timeout_add(60000, transfer_remote_timeout, transfer);
  } else g_warning("transfer not found in downloads");
}

int download_inspector(gpointer data) {
  GtkCList* list;
  int i1;
  transfer_t* transfer;
  char comm[2048];
  GtkWidget* temp;

  temp = lookup_widget(global.win, "label217");
  sprintf(comm, "%d", global.network.cur_downloads);
  gtk_label_set_text(GTK_LABEL(temp), comm);
  temp = lookup_widget(global.win, "label219");
  sprintf(comm, "%d", global.network.cur_uploads);
  gtk_label_set_text(GTK_LABEL(temp), comm);

  if ((global.network.max_downloads <= global.network.cur_downloads) ||
      (global.network.max_transfers <= global.network.cur_downloads+global.network.cur_uploads))
    return 1;
  
  list = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= list->rows) break;
    transfer = (transfer_t*)gtk_clist_get_row_data(list, i1);
    if (transfer->status == S_QUEUED) {
      download_start(transfer);
      return 1;
    }
    i1++;
  }
  return 1;
}

void cmd_linkspeed_response(char* data) {
  char nick[512];
  int linespeed;
  GtkCList* clist;
  int i1;
  transfer_t* transfer;

  sscanf(data, "%s %d\n", nick, &linespeed);

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    transfer = (transfer_t*)gtk_clist_get_row_data(clist, i1);
    if (strcmp(transfer->user, nick) == 0)
      transfer->linespeed = linespeed;
    i1++;
  }
}

FILE* open_download_file(transfer_t* transfer) {
  char* pos1;
  char filename[1024];
  struct stat st;
  int cnt;

  if (transfer->type == T_RESUME) {
    if (stat(transfer->localname, &st) < 0) {
      printf("resume file does not exist! [%s]", transfer->localname);
      return NULL;
    }

    // loading resume check data
    transfer->file = fopen(transfer->localname, "r");
    if (transfer->file == NULL) {
      printf("could not open file\n");
      return NULL;
    }
    
    if (transfer->progress < st.st_size) {
      cnt = st.st_size - transfer->progress;
      fseek(transfer->file, transfer->progress, SEEK_SET);
#ifdef TRANSFER_DEBUG
      printf("set offset to %ld\n", transfer->progress);
#endif
      transfer->resume_check = (unsigned char*)malloc(cnt*sizeof(char));
      transfer->check_length = read(fileno(transfer->file),
				    transfer->resume_check, cnt);
    }
    fclose(transfer->file);
    // load done

    transfer->file = fopen(transfer->localname, "a");
    fseek(transfer->file, transfer->progress, SEEK_SET);
  } else {
    // extracting short name
    pos1 = strrchr(transfer->longname, '/');

    if (pos1 == NULL) pos1 = transfer->longname;
    else pos1++;

    sprintf(filename, "%s/%s", global.path.download, pos1);
    
    cnt = 0;
    pos1 = &filename[strlen(filename)-4];
    while (stat(filename, &st) >= 0) {
      printf("file [%s] exists..\n", filename);
      cnt++;
      sprintf(pos1, "%d.mp3", cnt);
      printf("renaming to [%s]\n", filename);
    }
    transfer->progress = 0;
    transfer->localname = strdup(filename);
    transfer->file = fopen(transfer->localname, "w");
  }
  if (transfer->file == NULL) {
    printf("could not open file\n");
  }
  return transfer->file;
}

FILE* open_upload_file(transfer_t* transfer) {
  char* filename;
  struct stat st;

  filename = transfer->longname;
  stat(filename, &st);
  transfer->file = fopen(filename, "r");

  if (transfer->file == NULL) {
    printf("**sending FILE NOT FOUND\n");
    send(transfer->sock.fd, "FILE NOT FOUND", 
	 strlen("FILE NOT FOUND"), 0);
    transfer_end(transfer, S_UNAVAILABLE);
  } else {
    transfer->size = st.st_size;
  }
  return transfer->file;
}

void download_end(transfer_t* transfer, int mode) {

  if (transfer->sock.input >= 0) {
    gdk_input_remove(transfer->sock.input);
    transfer->sock.input = -1;
  }
  if (transfer->sock.fd >= 0) {
    close(transfer->sock.fd);
    transfer->sock.fd = -1;
  }
  if (transfer->status == S_DOWNLOADING) {
    send_command(CMD_CLIENT_DOWNLOAD_END, "");
  }
  if (transfer->sock.timeout >= 0) {
    gtk_timeout_remove(transfer->sock.timeout);
    transfer->sock.timeout = -1;
    if (transfer->status != S_REMOTE) {
      global.network.cur_downloads--;
#ifdef TRANSFER_DEBUG
      printf("-- now %d\n", global.network.cur_downloads);
#endif
    }
  }

  if (transfer->resume_check) {
    free(transfer->resume_check);
    transfer->resume_check = NULL;
    transfer->check_length = 0;
  }
  
  if (transfer->file) {
    fclose(transfer->file);
    transfer->file = NULL;
    switch (mode) {
    case S_FINISHED:
    case S_STOPPED:
      // doing nothing
      break;
    default:
      // for now just delete it
      unlink(transfer->localname);
      break;
    }
  }
}

void upload_end(transfer_t* transfer, int mode) {

  if (transfer->sock.input >= 0) {
    gdk_input_remove(transfer->sock.input);
    transfer->sock.input = -1;
  }
  if (transfer->sock.fd >= 0) {
    close(transfer->sock.fd);
    transfer->sock.fd = -1;
  }
  if (transfer->status == S_UPLOADING) {
    send_command(CMD_CLIENT_UPLOAD_END, "");
  }
  if (transfer->sock.timeout >= 0) {
    gtk_timeout_remove(transfer->sock.timeout);
    transfer->sock.timeout = -1;
    global.network.cur_uploads--;
#ifdef TRANSFER_DEBUG
      printf("- now %d\n", global.network.cur_uploads);
#endif
  }  

  if (transfer->file) {
    fclose(transfer->file);
    transfer->file = NULL;
  }
}

void transfer_end(transfer_t* transfer, int mode) {
  
#ifdef TRANSFER_DEBUG
  printf("ending %d [%s]\n", mode, transfer->winname);
#endif
  if (transfer->type == T_DOWNLOAD) {
    download_end(transfer, mode);
  } else if (transfer->type == T_UPLOAD) {
    upload_end(transfer, mode);
  } else {
    if (transfer->sock.input >= 0) {
      gdk_input_remove(transfer->sock.input);
      transfer->sock.input = -1;
    }
    if (transfer->sock.fd >= 0) {
      close(transfer->sock.fd);
      transfer->sock.fd = -1;
    }
    if (transfer->sock.timeout >= 0) {
      gtk_timeout_remove(transfer->sock.timeout);
      transfer->sock.timeout = -1;
    }
    if (transfer->file) {
      fclose(transfer->file);
      transfer->file = NULL;
    }
  }
  
  transfer->status = mode;
  transfer_update(transfer);
  
  transfer->sock.cnt = 0;
}

gint outgoing_header(gpointer data, gint source, 
		     GdkInputCondition condition) {
  transfer_t* transfer;
  char c;
  int res;
  char* temp_str;
  FILE* fd;

  transfer = (transfer_t*)data;

  if (transfer->type == T_UNKNOWN) {
    g_warning("Unknown transfer type");
    gtk_exit(1);
  }

  //  printf("download read\n");
  if (transfer->status == S_CONNECTING) {
    res = recv(source, &c, 1, 0);
    if (res == -1) {
      printf("rec1 error\n");
      transfer_end(transfer, S_REJECT);
      return 1;
    }
    if (res == 0) {
      printf("received nothing\n");
      transfer_end(transfer, S_CANCELED);
      return 1;
    }

    transfer->sock.cnt = 0;
    if (c != '1') {
      transfer_end(transfer, S_REJECT);
      return 1;
    }
    if ((transfer->type == T_DOWNLOAD) || (transfer->type == T_RESUME)) {
#ifdef TRANSFER_DEBUG
      printf("sending GET %lu\n", transfer->progress);
#endif
      send(source, "GET", 3, 0);
      temp_str = g_strdup_printf("%s \"%s\" %lu", 
				 global.user.username,
				 transfer->winname, 
				 transfer->progress);
      transfer->size = 0;
    } else if (transfer->type == T_UPLOAD) {
      send(source, "SEND", 4, 0);
      temp_str = g_strdup_printf("%s \"%s\" %li", 
				 global.user.username,
				 transfer->winname, transfer->size);
      transfer->progress = 0;
    }
    send(source, temp_str, strlen(temp_str), 0);
    g_free(temp_str);
    transfer->status = S_INFO;
  } else if (transfer->status == S_INFO) {
    while ((res = recv(source, &c, 1, 0)) != -1) {
#ifdef TRANSFER_DEBUG
      printf("read [%c] [%d]\n", c, c);
#endif
      if (isdigit((unsigned char)c)) {
	if ((transfer->type == T_DOWNLOAD) || (transfer->type == T_RESUME))
	  transfer->size = transfer->size*10+ c-'0';
	else if (transfer->type == T_UPLOAD)
	  transfer->progress = transfer->progress*10+ c-'0';
      } else break;
    }
    
    if ((transfer->type == T_DOWNLOAD) || (transfer->type == T_RESUME)) {
      // returning if first non-digit not arrived
      if (res == -1) return 1;
      if (transfer->size == 0) {
	printf("Invalid file\n");
	transfer_end(transfer, S_REJECT);
	return 1;
      }
#ifdef TRANSFER_DEBUG
      printf("got size %ld\n", transfer->size);
#endif
    } else {
#ifdef TRANSFER_DEBUG
      printf("got progress %ld\n", transfer->progress);
#endif
    }

    transfer->sock.cnt = 0;
    time(&(transfer->start_time));
    gdk_input_remove(transfer->sock.input);
    transfer->sock.input = -1;
    if ((transfer->type == T_DOWNLOAD) || (transfer->type == T_RESUME)) {
      if ((fd = open_download_file(transfer)) == NULL) {
	transfer_end(transfer, S_CANCELED);
	return 1;
      }
      temp_trans = 1;
      fwrite(&c, sizeof(char), 1, fd);
      transfer->progress++;
      transfer->status = S_DOWNLOADING;
      transfer->type = T_DOWNLOAD;
      send_command(CMD_CLIENT_DOWNLOAD_START, "");
      transfer->sock.input =
	gdk_input_add(transfer->sock.fd, GDK_INPUT_READ, 
		      GTK_SIGNAL_FUNC(download_get_input), transfer);
    } else if (transfer->type == T_UPLOAD) {
      if (!open_upload_file(transfer)) return 1;
      transfer->status = S_UPLOADING;
      send_command(CMD_CLIENT_UPLOAD_START, "");
      transfer->sock.input =
	gdk_input_add(transfer->sock.fd, GDK_INPUT_WRITE, 
		      GTK_SIGNAL_FUNC(upload_push_output), transfer);
    }
  } else {
    g_warning("invalid transfer status");
    gtk_exit(1);
  }

  return 1;
}

gint incoming_header(gpointer data, gint source, 
		     GdkInputCondition condition) {
  transfer_t* transfer;
  transfer_t* transfer2;
  static char c[1024];
  static int cnt;
  char* temp_str;
  file_t* file;

  transfer = (transfer_t*)data;

  if (transfer->status == S_CONNECTING) {
    cnt = 0;
    while (recv(source, c+cnt, 1, 0) != -1) {
      cnt++;
      c[cnt] = 0;
      if (!strcmp(c, "GET")) {
	transfer->status = S_INFO1;
	transfer->type = T_UPLOAD;
	cnt = 0;
	return 1;
      }
      if (!strcmp(c, "SEND")) {
	transfer->status = S_INFO1;
	transfer->type = T_DOWNLOAD;
	cnt = 0;
	return 1;
      }
      if (cnt > 4) {
	printf("**sending INVALID REQUEST [%s]\n", c);
	send(transfer->sock.fd, "INVALID REQUEST", 
	     strlen("INVALID REQUEST"), 0);
	transfer_end(transfer, S_REJECT);
	return 1;
      }
    }
    g_warning("no valid incoming connection");
    return 1;
  } else if (transfer->status == S_INFO1) {
    while (recv(source, c+cnt, 1, 0) != -1) {
      if (c[cnt]==' ') {
	c[cnt] = 0;
	cnt = 0;
	transfer->user = strdup(c);
	transfer->status = S_INFO2;
	break;
      } else {
	cnt++;
      }
    }
  } else if (transfer->status == S_INFO2) {
    while (recv(source, c+cnt, 1, 0) != -1) {
      break;
    }
    while (recv(source, c+cnt, 1, 0) != -1) {
      if (c[cnt]=='\"') {
	c[cnt] = 0;
	cnt = 0;
	transfer->winname = strdup(c);
	transfer->longname = strdup(transfer->winname);
	convert_to_unix(transfer->longname);
	transfer->shortname = 
	  strdup(extract_short_name(transfer->longname));

	// ok, got user and filename, searching in list
	if (transfer->type == T_DOWNLOAD) {
	  transfer2 = transfer_is_in_download(transfer->user, transfer->winname);
	} else {
	  transfer2 = transfer_is_in_upload(transfer->user, transfer->winname);
	}
	if (transfer2) {
	  // copy transfer
	  transfer2->sock.fd = transfer->sock.fd;
	  transfer2->longname = strdup(transfer->longname);
	  transfer2->winname = strdup(transfer->winname);
	  transfer2->shortname = strdup(transfer->shortname);
	  transfer2->status = transfer->status;

	  // dont know how to copy inputs.
	  // for now: delete and recreate
	  gdk_input_remove(transfer->sock.input);
	  transfer2->sock.input = 
	    gdk_input_add(transfer2->sock.fd, GDK_INPUT_READ, 
			  GTK_SIGNAL_FUNC(incoming_header), transfer2);
	  transfer = transfer2;
	} else {
	  log("protocol", "hacker alert (incoming) %s %s\n", transfer->user, transfer->winname);
	  g_warning("a little hacker here?");
	  transfer_end(transfer, S_REJECT);
	  return 1;
	}

	if (!transfer_is_active(transfer2)) {
	  g_warning("transfer no longer active");
	  transfer_end(transfer, S_REJECT);
	  return 1;
	}

	if (transfer->type == T_UPLOAD) {
	  if ((file = lib_search_transfer(transfer)) == NULL) {
	    printf("**sending FILE NOT SHARED\n");
	    send(transfer->sock.fd, "FILE NOT SHARED", 
		 strlen("FILE NOT SHARED"), 0);
	    transfer_end(transfer, S_REJECT);
	    return 1;
	  } else {
	    transfer->size = file->size;
	  }
	}
	
	transfer->status = S_INFO3;
	break;
      } else {
	cnt++;
      }
    }
  } else if (transfer->status == S_INFO3) {
    while (recv(source, c+cnt, 1, 0) != -1) {
      break;
    }
    while (recv(source, c+cnt, 1, 0) != -1) {
      if (cnt > 1000) cnt = 0;
      cnt++;
    }
    c[cnt] = 0;
    if ((transfer->type == T_DOWNLOAD) || (transfer->type == T_RESUME)) {
      sscanf(c, "%ld", &transfer->size);

      if (transfer->size == 0) {
	transfer_end(transfer, S_UNAVAILABLE);
	return 1;
      }
      
#ifdef TRANSFER_DEBUG
      printf("got size %ld\n", transfer->size);
#endif
      temp_str = g_strdup_printf("%lu", transfer->progress);
#ifdef TRANSFER_DEBUG
      printf("sending offset %s\n", temp_str);
#endif
      temp_trans = 0;

      send(transfer->sock.fd, temp_str, strlen(temp_str), 0);
      g_free(temp_str);
      
      transfer->sock.cnt = 0;
      time(&(transfer->start_time));
      if (!open_download_file(transfer)) {
	transfer_end(transfer, S_CANCELED);
	return 1;
      }
      transfer->status = S_DOWNLOADING;
      transfer->type = T_DOWNLOAD;
      send_command(CMD_CLIENT_DOWNLOAD_START, "");
      gdk_input_remove(transfer->sock.input);
      transfer->sock.input =
	gdk_input_add(transfer->sock.fd, GDK_INPUT_READ, 
		      GTK_SIGNAL_FUNC(download_get_input), transfer);
    } else {
#ifdef TRANSFER_DEBUG
      printf("got [%s]\n", c);
#endif
      sscanf(c, "%ld", &transfer->progress);
      temp_str = g_strdup_printf("%lu", transfer->size);
#ifdef TRANSFER_DEBUG
      printf("got progress %ld\n", transfer->progress);
#endif      
      send(transfer->sock.fd, temp_str, strlen(temp_str), 0);
      g_free(temp_str);

      transfer->sock.cnt = 0;
      time(&(transfer->start_time));
      if (!open_upload_file(transfer)) return 1;
      transfer->status = S_UPLOADING;
      send_command(CMD_CLIENT_UPLOAD_START, "");
      gdk_input_remove(transfer->sock.input);
      transfer->sock.input =
	gdk_input_add(transfer->sock.fd, GDK_INPUT_WRITE, 
		      GTK_SIGNAL_FUNC(upload_push_output), transfer);
    }
  } else {
    printf("WARNING: invalid transfer status\n");
  }

  return 1;
}

gint upload_push_output(gpointer data, gint source, 
			GdkInputCondition condition) {
  transfer_t* transfer;
  char buf[2048 + 1];
  int n, sent;
  struct pollfd pfd;

  transfer = (transfer_t*)data;
  if (transfer->file == NULL) return 1;

  pfd.fd = source;
  pfd.events = POLLERR | POLLHUP;
  pfd.revents = 0;
  if (poll(&pfd, 1, 0) < 0) {
    printf("poll error\n");
  }

#ifdef TRANSFER_DEBUG
  if (pfd.revents > 0)
    printf("poll result: %d\n", pfd.revents);
#endif
  if (pfd.revents & (POLLERR|POLLHUP)) {
    printf("Broken Pipe?\n");
    transfer_end(transfer, S_INCOMPLETE);
    return 1;
  }


  if (transfer->progress < transfer->size) {
    lseek(fileno(transfer->file), transfer->progress, SEEK_SET);
    n = read(fileno(transfer->file), buf, 1024);
    if (n <= 0) {
      printf("read <= 0\n");
      transfer_end(transfer, S_CANCELED);
      return 1;
    }
    sent = send(source, buf, n, 0);
    if (sent <= 0) {
      printf("sent error\n");
      transfer_end(transfer, S_CANCELED);
    } else {
      transfer->sock.cnt = 0;
      transfer->progress += sent;
    }
  } else {
    transfer_end(transfer, S_FINISHED);
  }

  return 1;
}

gint download_get_input(gpointer data, gint source, GdkInputCondition condition) {
  char buf[2048];
  int n, n2;
  transfer_t* transfer;
#ifdef TRANSFER_DEBUG
  int cnt;
#endif

  transfer = (transfer_t*)data;
  if (transfer->file == NULL) return 1;
  
  bzero(buf, sizeof(buf));
  n = recv(source, buf, 2048-1, 0);
  if (n <= 0) {
    if (transfer->progress >= transfer->size) {
      transfer_end(transfer, S_FINISHED);
    } else {
      transfer_end(transfer, S_INCOMPLETE);
    }
    return 0;
  } else if (n > 0) {
    transfer->sock.cnt = 0;
    // resume check
    if (transfer->check_length > 0) {
      if (transfer->check_length > n) transfer->check_length = n;
#ifdef TRANSFER_DEBUG
      printf("checking %d bytes (%d)\n", transfer->check_length,
	     temp_trans);
#endif
      if (memcmp(buf, transfer->resume_check+temp_trans, transfer->check_length-temp_trans)) {
#ifdef TRANSFER_DEBUG
	printf("resume check failed, ending download\n");
	for (cnt = 0; cnt < transfer->check_length-temp_trans; cnt++)
	  printf("%4d %4d\n", buf[cnt], transfer->resume_check[cnt+1]);
#endif
	transfer_end(transfer, S_RESUME_ERR);
	return 0;
      }
      transfer->check_length = 0;
      free(transfer->resume_check);
      transfer->resume_check = NULL;
    }
    
    if ((n2 = fwrite(buf, n, sizeof(char), transfer->file)) < 0) {
      printf("error writing to disk %d/%d\n", n2, n);
      return 0;
    }
    transfer->progress += n;
  }
  return 0;
}

void destroy_transfer_row(gpointer data) {
  transfer_t* transfer;

  transfer = (transfer_t*)data;
  if (transfer->longname) free(transfer->longname);
  if (transfer->shortname) free(transfer->shortname);
  if (transfer->winname) free(transfer->winname);
  if (transfer->localname) free(transfer->localname);
  if (transfer->md5) free(transfer->md5);
  if (transfer->user) free(transfer->user);
 
  free(transfer);
}

int transfer_timeout_cb(gpointer data) {
  transfer_t* transfer;

  transfer = (transfer_t*)data;
  if (transfer->sock.cnt >= transfer->sock.max_cnt) {
    transfer_end(transfer, S_TIMEOUT);
    return 1;
  }
  transfer->sock.cnt++;
  transfer_update(transfer);

  return 1;
}

int transfer_remote_timeout(gpointer data) {
  transfer_t* transfer;

  transfer = (transfer_t*)data;
  gtk_timeout_remove(transfer->sock.timeout);
  transfer->sock.timeout = -1;
  download_start(transfer);
  return 1;
}

