#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <strings.h>

#include <gtk/gtk.h>

#include "connection.h"
#include "transfer.h"
#include "support.h"
#include "global.h"
#include "log.h"

// pp5066b

const char* status_names[19] = {
  "Inactive", "Connecting...", "Getting info...", "Downloading",
  "Canceled",     "Finished!", "Timeout!",           "Rejected",
  "Incomplete",   "Uploading", "Getting info",   "Getting Info",
  "Getting Info",   "Waiting", "Firewalled", "Connection Error",
  "Queued", "Remotely Queued", "Unavailable!"
};

file_t* global_file2;

transfer_t* transfer_new() {
  transfer_t *transfer;
  int i1;

  transfer = (transfer_t*)malloc(sizeof(transfer_t));

  transfer->longname = NULL;        //
  transfer->shortname = NULL;       //
  transfer->winname = NULL;         //
  transfer->localname = NULL;
  transfer->md5 = NULL;
  transfer->user = NULL;
  transfer->linespeed = 0;
  transfer->progress = 0;
  transfer->rate = 0;
  transfer->timeleft = 0;
  transfer->start_time = 0;
  transfer->status = S_INACTIVE;
  transfer->size = 0;
  transfer->sock.address.ip = NULL;
  transfer->sock.address.ip_long = 0;
  transfer->sock.address.port = 0;
  transfer->sock.address.description = NULL;
  transfer->sock.address.no_connected = 0;
  transfer->sock.fd = -1;
  transfer->sock.input = -1;
  transfer->sock.timeout = -1;
  transfer->sock.cnt = 0;
  transfer->sock.max_cnt = 500;
  transfer->file = NULL;
  transfer->type = T_UNKNOWN;
  for (i1 = 0; i1 < TRANSFER_HISTORY; i1++)
    transfer->history[i1] = 0;
  transfer->hist_pos = 0;

  return transfer;
}

int transfer_is_active(transfer_t* transfer) {
  if ((transfer->status != S_WAITING) &&
      (transfer->status != S_CONNECTING) &&
      (transfer->status != S_INFO) &&
      (transfer->status != S_INFO1) &&
      (transfer->status != S_INFO2) &&
      (transfer->status != S_INFO3) &&
      (transfer->status != S_QUEUED) &&   // stopped, but active
      (transfer->status != S_UPLOADING) &&
      (transfer->status != S_DOWNLOADING)) {
    return 0;
  } else {
    return 1;
  }
}

void rec_search2(GtkCTree *ctree, GtkCTreeNode *node,
		 char* data) {
  file_t* file;

  if (!GTK_CTREE_ROW (node)->is_leaf) return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (!strcmp(data, file->winname)) {
    global_file2 = file;
  }
}

file_t* lib_search_transfer(transfer_t* transfer) {
  GtkCTree* ctree;

  global_file2 = NULL;
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, NULL, 
			   (GtkCTreeFunc)rec_search2,
			   transfer->winname);
  return global_file2;
}

void download_file(file_t* file) {
  transfer_t* new_trans;

  new_trans = transfer_new();
  new_trans->longname = strdup(file->longname);
  new_trans->shortname = strdup(file->shortname);
  new_trans->winname = strdup(file->winname);
  new_trans->linespeed = file->linespeed;
  new_trans->user = strdup(file->user);
  new_trans->type = T_DOWNLOAD;
  new_trans->size = file->size;

  if ((global.network.max_downloads > global.network.cur_downloads) &&
      (global.network.max_transfers > global.network.cur_downloads+global.network.cur_uploads)) {
    new_trans->status = S_WAITING;
    printf("sending download request [%s]\n", file->winname);
    send_command(203, "%s \"%s\"", file->user, file->winname);
    global.network.cur_downloads++;
  } else {
    new_trans->status = S_QUEUED;
    printf("download queued [%s]\n", file->winname);
  }
  transfer_insert_download(new_trans);
}

void cmd_download_ack(char* data) {
  transfer_t* transfer;
  char* pos1;
  char* pos2;
  long temp;
  char comm[1024];

  char* user;
  int port;
  unsigned long ip_long;
  char* winname;
  char* md5;
  int linespeed;

  pos1 = data;
  pos2 = strchr(pos1, ' ');
  pos2[0] = 0;
  user = strdup(pos1);
  
  pos1 = pos2+1;
  pos2 = strchr(pos1, '\"');
  sscanf(pos1, "%lu %d", &temp, &port);
  ip_long = BSWAP32(temp);

  pos1 = pos2;
  pos2 = strrchr(pos1, '\"');
  pos2[0] = '\0';
  winname = strdup(pos1+1);

  pos1 = pos2+2;
  pos2 = strchr(pos1, ' ');
  pos2[0] = '\0';
  md5 = strdup(pos1);

  pos1 = pos2+1;
  linespeed = atoi(pos1);

  transfer = transfer_is_in_download(user, winname);
  free(user);
  free(winname);

  if (!transfer) {
    g_warning("transfer not found in download list, should not happen");
    return;
  }
  if (!transfer_is_active(transfer)) {
    printf("transfer no longer active, ignoring upload ack\n");
    return;
  }

  transfer->sock.address.port = port;
  transfer->sock.address.ip_long = ip_long;
  transfer->md5 = md5;
  transfer->linespeed = linespeed;
  
  if (transfer->sock.address.port == 0) {
    if ((global.network.port == 0) || global.network.firewall) {
      printf("both firewalled, cannot do download\n");
      transfer_end(transfer, S_REJECT);
    } else {
      printf("host firewalled\n");
      sprintf(comm, "%s \"%s\"", transfer->user, transfer->winname);
      send_command(500, comm);
    }
  } else {
    if (connect_to_user(transfer)) return;
    transfer->sock.input = 
      gdk_input_add(transfer->sock.fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(outgoing_header), transfer);
  }
}

void cmd_upload_request(char* data) {
  transfer_t* new_trans;
  char* user;
  char* winname;
  char* pos;
  file_t* file;

  printf("upload request\n");

  pos = strchr(data, ' ');
  pos[0] = 0;
  user = strdup(data);
  data = pos+2;
  pos = strrchr(data, '\"');
  pos[0] = 0;
  winname = strdup(data);

  // do not allow uploads from enemies
  if (is_string_in_list(global.enemyuser, user)) return;
  
  if ((global.network.max_uploads <= global.network.cur_uploads) ||
      (global.network.max_transfers <= global.network.cur_downloads+global.network.cur_uploads)) {
    printf("upload queued [%s]\n", winname);
    send_command(619, "%s \"%s\" %d", user, 
		 winname, global.network.cur_uploads);
    free(user);
    free(winname);
    return;
  }

  new_trans = transfer_new();
  new_trans->winname = strdup(winname);
  new_trans->longname = strdup(winname);
  convert_to_unix(new_trans->longname);
  new_trans->shortname =
    strdup(extract_short_name(new_trans->longname));
  new_trans->localname = NULL;
  new_trans->user = strdup(user);
  new_trans->type = T_UPLOAD;

  file = lib_search_transfer(new_trans);
  if (file == NULL) {
    g_warning("requested file not shared");
    free(new_trans);
    return;
  }

  new_trans->size = file->size;
  new_trans->status = S_WAITING;

  printf("sending upload ack [%s]\n", winname);
  send_command(608, "%s \"%s\"", user, winname);
  transfer_insert_upload(new_trans);
  printf("transfer inserted\n");

  // requesting users linespeed
  send_command(600, user);
  free(user);
  free(winname);
}

void cmd_alternate_ack(char* data) {
  transfer_t* transfer;
  char* pos1;
  char* pos2;
  long temp;

  char* user;
  int port;
  unsigned long ip_long;
  char* winname;
  char* md5;
  int linespeed;
  file_t* file;
  

  pos1 = data;
  pos2 = strchr(pos1, ' ');
  pos2[0] = 0;
  user = strdup(pos1);
  
  pos1 = pos2+1;
  pos2 = strchr(pos1, '\"');
  printf("[%s]\n", pos1);
  sscanf(pos1, "%lu %d", &temp, &port);
  printf("port: %d\n", port);
  ip_long = BSWAP32(temp);

  pos1 = pos2;
  pos2 = strrchr(pos1, '\"');
  pos2[0] = '\0';
  winname = strdup(pos1+1);

  pos1 = pos2+2;
  pos2 = strchr(pos1, ' ');
  pos2[0] = '\0';
  md5 = strdup(pos1);

  pos1 = pos2+1;
  linespeed = atoi(pos1);

  transfer = transfer_is_in_upload(user, winname);
  free(user);
  free(winname);

  if (!transfer) {
    g_warning("transfer not found in upload list, should not happen");
    return;
  }
  if (!transfer_is_active(transfer)) {
    printf("transfer no longer active, ignoring download ack\n");
    return;
  }

  transfer->linespeed = linespeed;
  transfer->md5 = md5;
  transfer->sock.address.port = port;
  transfer->sock.address.ip_long = ip_long;

  file = lib_search_transfer(transfer);
  if (!file) {
    g_warning("file not found in lib: should not happen\n");
    return;
  }
  transfer->size = file->size;

  if (transfer->sock.address.port == 0) {
    g_warning("both are firewalled!");
    transfer_end(transfer, S_FIREWALL);
  } else {
    if (connect_to_user(transfer)) return;
    transfer->sock.input = 
      gdk_input_add(transfer->sock.fd, GDK_INPUT_READ, 
		    GTK_SIGNAL_FUNC(outgoing_header), transfer);
  }
}

transfer_t* transfer_is_in_download(char* user, char* winname) {
  GtkCList* clist;
  int i1;
  transfer_t* transfer;
  
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) {
      transfer = NULL;
      break;
    }
    transfer = (transfer_t*)gtk_clist_get_row_data(clist, i1);
    if ((strcmp(transfer->winname, winname) == 0) &&
	(strcmp(transfer->user, user) == 0)) {
      return transfer;
    }
    i1++;
  }

  return NULL;
}

transfer_t* transfer_is_in_upload(char* user, char* winname) {
  GtkCList* clist;
  int i1;
  transfer_t* transfer;
  
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) return NULL;
    transfer = (transfer_t*)gtk_clist_get_row_data(clist, i1);
    if ((strcmp(transfer->winname, winname) == 0) &&
	(strcmp(transfer->user, user) == 0)) {
      return transfer;
    }
    i1++;
  }

  return NULL;
}

int transfer_update(transfer_t* transfer) {
  GtkCList* list;
  int row;
  char str[200];
  time_t cur;
  int sec;
  double trate;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;

  time(&cur);
  list = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  row = gtk_clist_find_row_from_data(list, transfer);
  if (row < 0) {
    list = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
    row = gtk_clist_find_row_from_data(list, transfer);
  }
  if (row < 0) {
    printf("update: transfer not found\n");
    return 1;
  }

  // filename
  if (transfer->type == T_DOWNLOAD) {
    if (transfer->localname) strcpy(str, transfer->localname);
    else if (transfer->shortname) strcpy(str, transfer->shortname);
    else strcpy(str, "Not known yet");
  } else if (transfer->type == T_UPLOAD) {
    if (transfer->shortname) strcpy(str, transfer->shortname);
    else strcpy(str, "Not known yet");
  } else {
    g_warning("transfer_update: this should not happen!\n");
    gtk_exit(1);
  }
  gtk_clist_set_text(list, row, 0, str);

  // filesize
  sprintf(str, "%ld/%ld", transfer->progress, transfer->size);
  gtk_clist_set_text(list, row, 1, str);

  // user
  if (transfer->user) strcpy(str, transfer->user);
  else strcpy(str, "Not known yet");
  gtk_clist_set_text(list, row, 2, str);

  // status
  if ((transfer->status < 0) || (transfer->status > 18)) {
    g_warning("transfer_update: invalid status %d", 
	      transfer->status);
    gtk_exit(1);
  }
  if (transfer->status == S_WAITING)
    sprintf(str, "Waiting %d secs", 500-transfer->sock.cnt);
  else 
    strcpy(str, status_names[transfer->status]);
  gtk_clist_set_text(list, row, 3, str);

  // linespeed
  if ((transfer->linespeed < 0) || (transfer->linespeed > 10)) {
    g_warning("transfer_update: invalid linespeed %d", 
	      transfer->linespeed);
    gtk_exit(1);
  }
  strcpy(str, LineSpeed[transfer->linespeed]);
  gtk_clist_set_text(list, row, 4, str);

  // progress
  if (transfer->size >0)
    sprintf(str, "%.2f %%", 100*(double)(transfer->progress)/(double)(transfer->size));
  else
    sprintf(str, "%.2f %%", .0);
  gtk_clist_set_text(list, row, 5, str);

  // transfer rate & time left
  cur = cur - transfer->start_time;
  /*
  if (cur > 0) {
    trate = (double)(transfer->progress)/(double)(cur)/1024;
    sprintf(str, "%.2f kB/s", trate);
  
    if (trate > 30) {
      pixmap = global.pix.sgreen;
      bitmap = global.pix.sgreenb;
    } else if (trate > 8) {
      pixmap = global.pix.syellow;
      bitmap = global.pix.syellowb;
    } else if (trate > 2) {
      pixmap = global.pix.sred;
      bitmap = global.pix.sredb;
    } else if (trate > 0) {
      pixmap = global.pix.sgray;
      bitmap = global.pix.sgrayb;
    } else {
      pixmap = global.pix.dummy;
      bitmap = global.pix.dummyb;
    }
    gtk_clist_set_pixtext (list, row, 6, str, 5, pixmap, bitmap);
  }
  */
  {
    int old_pos = transfer->hist_pos;

    transfer->history[transfer->hist_pos] = transfer->progress;
    transfer->hist_pos++;
    if (transfer->hist_pos >= TRANSFER_HISTORY)
      transfer->hist_pos = 0;
    trate = 
      (double)(transfer->history[old_pos]-
	       transfer->history[transfer->hist_pos])
      /TRANSFER_HISTORY/1024;
    sprintf(str, "%.2f kB/s", trate);
  
    if (trate > 30) {
      pixmap = global.pix.sgreen;
      bitmap = global.pix.sgreenb;
    } else if (trate > 8) {
      pixmap = global.pix.syellow;
      bitmap = global.pix.syellowb;
    } else if (trate > 2) {
      pixmap = global.pix.sred;
      bitmap = global.pix.sredb;
    } else if (trate > 0) {
      pixmap = global.pix.sgray;
      bitmap = global.pix.sgrayb;
    } else {
      pixmap = global.pix.dummy;
      bitmap = global.pix.dummyb;
    }
    gtk_clist_set_pixtext (list, row, 6, str, 5, pixmap, bitmap);
  }
  
  if (transfer->progress > 0) {
    sec = (int)((double)(cur)*
		((double)(transfer->size-transfer->progress)/
		 (double)(transfer->progress))
		);
  } else {
    sec = 0;
  }
  sprintf(str, "%d:%s%d", sec/60, (sec%60<10)?"0":"", sec%60);
  gtk_clist_set_text(list, row, 7, str);
  
  return 1;
}

void transfer_insert_download(transfer_t *transfer) {
  GtkCList* temp;
  int row;

  temp = GTK_CLIST(lookup_widget(global.win, "transfer_down"));

  tstr[0][0] = tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 
    tstr[5][0] = tstr[6][0] = 0;
  row = gtk_clist_append(temp, list);
  gtk_clist_set_row_data_full (temp, row, (gpointer)transfer, 
			       (GtkDestroyNotify)destroy_transfer_row);

  transfer->sock.timeout = 
    gtk_timeout_add(UPDATE_DELAY, transfer_timeout_cb, transfer);
}

void transfer_insert_upload(transfer_t *transfer) {
  GtkCList* temp;
  int row;

  temp = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  
  tstr[0][0] = tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 
    tstr[5][0] = tstr[6][0] = 0;
  row = gtk_clist_append(temp, &list[0]);
  gtk_clist_set_row_data_full (temp, row, (gpointer)transfer, 
			       (GtkDestroyNotify)destroy_transfer_row);

  global.network.cur_uploads++;
  transfer->sock.timeout = 
    gtk_timeout_add(UPDATE_DELAY, transfer_timeout_cb, transfer);
}


int connect_to_user(transfer_t* transfer) {
  int res;

  printf("connection to user\n");
  transfer->status = S_CONNECTING;
  res = connect_socket(&(transfer->sock));

  if ((res != 0) && (res != EINPROGRESS)) {
    transfer_end(transfer, S_CONERROR);
    return -1;
  }
  return 0;
}

void cmd_get_error(char* data) {
  transfer_t* transfer;
  char* user;
  char* winname;
  char* pos;
  
  pos = strchr(data, ' ');
  pos[0] = 0;
  user = strdup(data);
  data = pos+2;
  pos = strrchr(data, '\"');
  pos[0] = 0;
  winname = strdup(data);

  transfer = transfer_is_in_download(user, winname);
  free(user);
  free(winname);

  if (transfer) transfer_end(transfer, S_UNAVAILABLE);
  else {
    g_warning("transfer not found in downloads");
    gtk_exit(1);
  }
}

void cmd_remote_queued(char* data) {
  transfer_t* transfer;
  char* user;
  char* winname;
  char* pos;
  
  pos = strchr(data, ' ');
  pos[0] = 0;
  user = strdup(data);
  data = pos+2;
  pos = strrchr(data, '\"');
  pos[0] = 0;
  winname = strdup(data);

  transfer = transfer_is_in_download(user, winname);
  free(user);
  free(winname);

  if (transfer) transfer_end(transfer, S_REMOTE);
  else g_warning("transfer not found in downloads");
}

int download_inspector(gpointer data) {
  GtkCList* list;
  int i1;
  transfer_t* transfer;
  char comm[2048];
  GtkWidget* temp;

  temp = lookup_widget(global.win, "label217");
  sprintf(comm, "%d", global.network.cur_downloads);
  gtk_label_set_text(GTK_LABEL(temp), comm);
  temp = lookup_widget(global.win, "label219");
  sprintf(comm, "%d", global.network.cur_uploads);
  gtk_label_set_text(GTK_LABEL(temp), comm);

  if ((global.network.max_downloads <= global.network.cur_downloads) ||
      (global.network.max_transfers <= global.network.cur_downloads+global.network.cur_uploads))
    return 1;
  
  list = GTK_CLIST(lookup_widget(global.win, "transfer_down"));
  i1 = 0;
  while (1) {
    if (i1 >= list->rows) break;
    transfer = (transfer_t*)gtk_clist_get_row_data(list, i1);
    if (transfer->status == S_QUEUED) {
      transfer->status = S_WAITING;
      send_command(203, "%s \"%s\"", transfer->user, transfer->winname);
      global.network.cur_downloads++;
      return 1;
    }
    i1++;
  }
  return 1;
}

void cmd_linkspeed_response(char* data) {
  char nick[512];
  int linespeed;
  GtkCList* clist;
  int i1;
  transfer_t* transfer;

  sscanf(data, "%s %d\n", nick, &linespeed);

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  i1 = 0;
  while (1) {
    if (i1 >= clist->rows) break;
    transfer = (transfer_t*)gtk_clist_get_row_data(clist, i1);
    if (strcmp(transfer->user, nick) == 0)
      transfer->linespeed = linespeed;
    i1++;
  }
}

FILE* open_download_file(transfer_t* transfer) {
  char* pos1;
  char filename[1024];
  struct stat st;
  int cnt;

  // extracting short name
  pos1 = strrchr(transfer->longname, '/');

  if (pos1 == NULL) pos1 = transfer->longname;
  else pos1++;

  sprintf(filename, "%s/%s", global.path.download, pos1);

  cnt = 0;
  pos1 = &filename[strlen(filename)-4];
  while (stat(filename, &st) >= 0) {
    printf("file [%s] exists..\n", filename);
    cnt++;
    sprintf(pos1, "%d.mp3", cnt);
    printf("renaming to [%s]\n", filename);
  }

  transfer->localname = strdup(filename);

  transfer->file = fopen(filename, "w");
  if (transfer->file == NULL) {
    printf("could not open file\n");
  }
  return transfer->file;
}

FILE* open_upload_file(transfer_t* transfer) {
  char* filename;
  struct stat st;

  filename = transfer->longname;
  stat(filename, &st);
  printf("opening [%s]\n", filename);
  transfer->file = fopen(filename, "r");

  if (transfer->file == NULL) {
    printf("**sending FILE NOT FOUND\n");
    send(transfer->sock.fd, "FILE NOT FOUND", 
	 strlen("FILE NOT FOUND"), 0);
    transfer_end(transfer, S_UNAVAILABLE);
  } else {
    transfer->size = st.st_size;
  }
  return transfer->file;
}

void download_end(transfer_t* transfer, int mode) {

  printf("ending download %p mode: %d\n", transfer, mode);
  if (transfer->sock.input >= 0) {
    gdk_input_remove(transfer->sock.input);
    transfer->sock.input = -1;
  }
  if (transfer->sock.fd >= 0) {
    close(transfer->sock.fd);
    transfer->sock.fd = -1;
  }
  if (transfer->status == S_DOWNLOADING) {
    send_command(219, "");
  }  
  if (transfer->sock.timeout >= 0) {
    gtk_timeout_remove(transfer->sock.timeout);
    transfer->sock.timeout = -1;
    if (transfer->status != S_QUEUED) {
      if (global.network.cur_downloads <= 0) {
	g_warning("download_end: wrong cur_downloads");
      }
      global.network.cur_downloads--;
    }
  }

  if (transfer->file) {
    fclose(transfer->file);
    transfer->file = NULL;
    switch (mode) {
    case S_CANCELED:
      // deleting
      unlink(transfer->localname);
      break;
    case S_INCOMPLETE:
    case S_TIMEOUT:
      // moving to incomplete folder
      break;
    }
  }
}

void upload_end(transfer_t* transfer, int mode) {

  printf("ending upload %p mode: %d\n", transfer, mode);
  if (transfer->sock.input >= 0) {
    gdk_input_remove(transfer->sock.input);
    transfer->sock.input = -1;
  }
  if (transfer->sock.fd >= 0) {
    close(transfer->sock.fd);
    transfer->sock.fd = -1;
  }
  if (transfer->status == S_UPLOADING) {
    send_command(221, "");
  }
  if (transfer->sock.timeout >= 0) {
    gtk_timeout_remove(transfer->sock.timeout);
    transfer->sock.timeout = -1;
    if (global.network.cur_uploads <= 0) {
      g_warning("upload_end: wrong cur_uploads");
    }
    global.network.cur_uploads--;
  }  

  if (transfer->file) {
    fclose(transfer->file);
    transfer->file = NULL;
  }
}

void transfer_end(transfer_t* transfer, int mode) {
  
  printf("ending transfer %p mode: %d\n", transfer, mode);
  if (transfer->type == T_DOWNLOAD) {
    download_end(transfer, mode);
  } else if (transfer->type == T_UPLOAD) {
    upload_end(transfer, mode);
  } else {
    if (transfer->sock.input >= 0) {
      gdk_input_remove(transfer->sock.input);
      transfer->sock.input = -1;
    }
    if (transfer->sock.fd >= 0) {
      close(transfer->sock.fd);
      transfer->sock.fd = -1;
    }
    if (transfer->sock.timeout >= 0) {
      gtk_timeout_remove(transfer->sock.timeout);
      transfer->sock.timeout = -1;
    }  
    if (transfer->file) {
      fclose(transfer->file);
      transfer->file = NULL;
    }
    
  }
  
  transfer->status = mode;
  transfer_update(transfer);
  
  transfer->sock.cnt = 0;
  transfer->start_time = 0;
}

gint outgoing_header(gpointer data, gint source, 
		     GdkInputCondition condition) {
  transfer_t* transfer;
  char c;
  int res;
  char* temp_str;
  FILE* fd;

  transfer = (transfer_t*)data;

  if (transfer->type == T_UNKNOWN) {
    g_warning("Unknown transfer type");
    gtk_exit(1);
  }

  //  printf("download read\n");
  if (transfer->status == S_CONNECTING) {
    res = recv(source, &c, 1, 0);
    if (res == -1) {
      printf("rec1 error\n");
      transfer_end(transfer, S_REJECT);
      return 1;
    }
    if (res == 0) {
      printf("received nothing\n");
      return 1;
    }
    printf("received [%c][%d]\n", c, c);

    transfer->sock.cnt = 0;
    if (c != '1') {
      printf("connection not accepted\n");
      transfer_end(transfer, S_REJECT);
      return 1;
    }
    if (transfer->type == T_DOWNLOAD) {
      printf("**sending GET\n");
      send(source, "GET", 3, 0);
      temp_str = g_strdup_printf("%s \"%s\" %i", 
				 global.user.username,
				 transfer->winname, 0);
      transfer->size = 0;
    } else if (transfer->type == T_UPLOAD) {
      printf("**sending SEND\n");
      send(source, "SEND", 4, 0);
      temp_str = g_strdup_printf("%s \"%s\" %li", 
				 global.user.username,
				 transfer->winname, transfer->size);
      transfer->progress = 0;
    }
    printf("**sending [%s]\n", temp_str);
    send(source, temp_str, strlen(temp_str), 0);
    g_free(temp_str);
    transfer->status = S_INFO;
  } else if (transfer->status == S_INFO) {
    printf("trying to rec\n");
    while ((res = recv(source, &c, 1, 0)) != -1) {
      if (isdigit((unsigned char)c)) {
	printf("got char: [%c] = %d\n", c, c);
	if (transfer->type == T_DOWNLOAD)
	  transfer->size = transfer->size*10+ c-'0';
	else if (transfer->type == T_DOWNLOAD)
	  transfer->progress = transfer->progress*10+ c-'0';
      } else break;
    }
    
    printf("res = [%d]\n", res);

    if (transfer->type == T_DOWNLOAD) {
      // returning if first non-digit not arrived
      if (res == -1) return 1;
      printf("got size %ld\n", transfer->size);
      if (transfer->size == 0) {
	printf("Invalid file\n");
	transfer_end(transfer, S_REJECT);
	return 1;
      }
    } else {
      printf("got offset %ld\n", transfer->progress);
    }

    transfer->sock.cnt = 0;
    time(&(transfer->start_time));
    gdk_input_remove(transfer->sock.input);
    transfer->sock.input = -1;
    if (transfer->type == T_DOWNLOAD) {
      printf("open_download\n");
      if ((fd = open_download_file(transfer)) == NULL) {
	transfer_end(transfer, S_CANCELED);
	return 1;
      }
      fwrite(&c, sizeof(char), 1, fd);
      transfer->progress = 1;
      transfer->status = S_DOWNLOADING;
      send_command(218, "");
      transfer->sock.input =
	gdk_input_add(transfer->sock.fd, GDK_INPUT_READ, 
		      GTK_SIGNAL_FUNC(download_get_input), transfer);
    } else {
      printf("open_upload\n");
      if (!open_upload_file(transfer)) return 1;
      transfer->status = S_UPLOADING;
      send_command(220, "");
      transfer->sock.input =
	gdk_input_add(transfer->sock.fd, GDK_INPUT_WRITE, 
		      GTK_SIGNAL_FUNC(upload_push_output), transfer);
    }
  } else {
    g_warning("invalid transfer status");
    gtk_exit(1);
  }

  return 1;
}

gint incoming_header(gpointer data, gint source, 
		     GdkInputCondition condition) {
  transfer_t* transfer;
  transfer_t* transfer2;
  static char c[1024];
  static int cnt;
  char* temp_str;
  file_t* file;

  printf("upload read\n");
  transfer = (transfer_t*)data;

  if (transfer->status == S_CONNECTING) {
    cnt = 0;
    while (recv(source, c+cnt, 1, 0) != -1) {
      cnt++;
      c[cnt] = 0;
      if (!strcmp(c, "GET")) {
	printf("received GET\n");
	transfer->status = S_INFO1;
	transfer->type = T_UPLOAD;
	cnt = 0;
	return 1;
      }
      if (!strcmp(c, "SEND")) {
	printf("received SEND\n");
	transfer->status = S_INFO1;
	transfer->type = T_DOWNLOAD;
	cnt = 0;
	return 1;
      }
      if (cnt > 4) {
	printf("**sending INVALID REQUEST [%s]\n", c);
	send(transfer->sock.fd, "INVALID REQUEST", 
	     strlen("INVALID REQUEST"), 0);
	transfer_end(transfer, S_REJECT);
	return 1;
      }
    }
    g_warning("no valid incoming connection");
    return 1;
  } else if (transfer->status == S_INFO1) {
    printf("reading 2\n");
    while (recv(source, c+cnt, 1, 0) != -1) {
      printf("rec2[%s]\n", c);
      if (c[cnt]==' ') {
	c[cnt] = 0;
	cnt = 0;
	transfer->user = strdup(c);
	printf("advance to 2\n");
	transfer->status = S_INFO2;
	break;
      } else {
	cnt++;
      }
    }
  } else if (transfer->status == S_INFO2) {
    printf("reading 3\n");
    while (recv(source, c+cnt, 1, 0) != -1) {
      break;
    }
    while (recv(source, c+cnt, 1, 0) != -1) {
      printf("rec3[%s]\n", c);
      if (c[cnt]=='\"') {
	c[cnt] = 0;
	cnt = 0;
	transfer->winname = strdup(c);
	transfer->longname = strdup(transfer->winname);
	convert_to_unix(transfer->longname);
	transfer->shortname = 
	  strdup(extract_short_name(transfer->longname));

	// ok, got user and filename, searching in list
	if (transfer->type == T_DOWNLOAD) {
	  transfer2 = transfer_is_in_download(transfer->user, transfer->winname);
	} else {
	  transfer2 = transfer_is_in_upload(transfer->user, transfer->winname);
	}
	if (transfer2) {
	  // copy transfer
	  transfer2->sock.fd = transfer->sock.fd;
	  transfer2->longname = strdup(transfer->longname);
	  transfer2->winname = strdup(transfer->winname);
	  transfer2->shortname = strdup(transfer->shortname);
	  transfer2->status = transfer->status;

	  // dont know how to copy inputs.
	  // for now: delete and recreate
	  gdk_input_remove(transfer->sock.input);
	  transfer2->sock.input = 
	    gdk_input_add(transfer2->sock.fd, GDK_INPUT_READ, 
			  GTK_SIGNAL_FUNC(incoming_header), transfer2);
	  transfer = transfer2;
	} else {
	  log("protocol", "hacker alert (incoming)\n");
	  g_warning("a little hacker here?");
	  transfer_end(transfer, S_REJECT);
	  return 1;
	}

	if (!transfer_is_active(transfer2)) {
	  g_warning("transfer no longer active");
	  transfer_end(transfer, S_REJECT);
	  return 1;
	}

	if (transfer->type == T_UPLOAD) {
	  if ((file = lib_search_transfer(transfer)) == NULL) {
	    printf("**sending FILE NOT SHARED\n");
	    send(transfer->sock.fd, "FILE NOT SHARED", 
		 strlen("FILE NOT SHARED"), 0);
	    transfer_end(transfer, S_REJECT);
	    return 1;
	  } else {
	    transfer->size = file->size;
	  }
	}
	
	printf("advance to 3\n");
	transfer->status = S_INFO3;
	break;
      } else {
	cnt++;
      }
    }
  } else if (transfer->status == S_INFO3) {
    printf("reading 4 %d\n", cnt);
    while (recv(source, c+cnt, 1, 0) != -1) {
      printf("rec4. %d=%d\n", cnt, c[cnt]);
      break;
    }
    while (recv(source, c+cnt, 1, 0) != -1) {
      if (cnt > 1000) cnt = 0;
      printf("rec4: %d=%d %c\n", cnt, c[cnt], c[cnt]);
      cnt++;
    }
    c[cnt] = 0;
    if (transfer->type == T_DOWNLOAD) {
      sscanf(c, "%ld", &transfer->size);
      printf("got size %ld\n", transfer->size);

      if (transfer->size == 0) {
	transfer_end(transfer, S_UNAVAILABLE);
	return 1;
      }

      printf("advance to firewalled download %ld\n", transfer->size);
      temp_str = g_strdup_printf("%d", 0);

      printf("**sending %d: [%s]\n", strlen(temp_str), temp_str);
      send(transfer->sock.fd, temp_str, strlen(temp_str), 0);
      g_free(temp_str);
      
      transfer->sock.cnt = 0;
      time(&(transfer->start_time));
      if (!open_download_file(transfer)) {
	transfer_end(transfer, S_CANCELED);
	return 1;
      }
      transfer->progress = 0;
      transfer->status = S_DOWNLOADING;
      send_command(218, "");
      gdk_input_remove(transfer->sock.input);
      transfer->sock.input =
	gdk_input_add(transfer->sock.fd, GDK_INPUT_READ, 
		      GTK_SIGNAL_FUNC(download_get_input), transfer);
    } else {
      sscanf(c, "%ld", &transfer->progress);
      printf("advance to upload %ld\n", transfer->progress);
      temp_str = g_strdup_printf("%lu", transfer->size);
      
      printf("**sending %d: [%s]\n", strlen(temp_str), temp_str);
      send(transfer->sock.fd, temp_str, strlen(temp_str), 0);
      g_free(temp_str);

      transfer->sock.cnt = 0;
      time(&(transfer->start_time));
      if (!open_upload_file(transfer)) return 1;
      transfer->status = S_UPLOADING;
      send_command(220, "");
      gdk_input_remove(transfer->sock.input);
      transfer->sock.input =
	gdk_input_add(transfer->sock.fd, GDK_INPUT_WRITE, 
		      GTK_SIGNAL_FUNC(upload_push_output), transfer);
    }
  } else {
    printf("WARNING: invalid transfer status\n");
  }

  return 1;
}

gint upload_push_output(gpointer data, gint source, 
			GdkInputCondition condition) {
  transfer_t* transfer;
  char buf[2048 + 1];
  int n, sent;

  transfer = (transfer_t*)data;
  if (transfer->file == NULL) return 1;
  
  if (transfer->progress < transfer->size) {
    lseek(fileno(transfer->file), transfer->progress, SEEK_SET);
    n = read(fileno(transfer->file), buf, 1024);
    if (n <= 0) {
      printf("read <= 0\n");
      transfer_end(transfer, S_CANCELED);
      return 1;
    }
    sent = send(transfer->sock.fd, buf, n, 0);
    if (sent <= 0) {
      printf("sent error\n");
      transfer_end(transfer, S_CANCELED);
    } else {
      transfer->sock.cnt = 0;
      transfer->progress += sent;
    }
  } else {
    printf("upload finished\n");
    transfer_end(transfer, S_FINISHED);
  }

  return 1;
}

gint download_get_input(gpointer data, gint source, GdkInputCondition condition) {
   char buf[2048];
   int n, n2;
   transfer_t* transfer;

   transfer = (transfer_t*)data;

   bzero(buf, sizeof(buf));
   n = recv(source, buf, 2048-1, 0);
   if (n <= 0) {
     printf("download res: %d\n", n);
     if (transfer->progress >= transfer->size) {
       transfer_end(transfer, S_FINISHED);
     } else {
       transfer_end(transfer, S_INCOMPLETE);
     }
     return 0;
   } else if (n > 0) {
     transfer->sock.cnt = 0;
     if ((n2 = fwrite(buf, n, sizeof(char), transfer->file)) < 0) {
       printf("error writing to disk %d/%d\n", n2, n);
       return 0;
     }
     transfer->progress += n;
   }
   return 0;
}

void destroy_transfer_row(gpointer data) {
  transfer_t* transfer;

  transfer = (transfer_t*)data;
  if (transfer->longname) free(transfer->longname);
  if (transfer->shortname) free(transfer->shortname);
  if (transfer->winname) free(transfer->winname);
  if (transfer->localname) free(transfer->localname);
  if (transfer->md5) free(transfer->md5);
  if (transfer->user) free(transfer->user);
 
  free(transfer);
}

int transfer_timeout_cb(gpointer data) {
  transfer_t* transfer;

  transfer = (transfer_t*)data;
  if (transfer->sock.cnt >= transfer->sock.max_cnt) {
    transfer_end(transfer, S_TIMEOUT);
    return 1;
  }
  if (transfer->status != S_QUEUED) transfer->sock.cnt++;
  transfer_update(transfer);

  return 1;
}
