/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: meta.c,v 1.24 2002/11/28 03:09:28 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <stdlib.h> /* atoi */

#include "conf.h"
#include "util.h"
#include "security.h"
#include "meta.h"
#include "music.h"
#include "datesel.h"

struct _MetaMgr {
	GtkHBox parent;
	GtkSizeGroup *sizegroup;

	GtkWidget *date;
	GtkWidget *esubject;
	GtkWidget *cmood, *emood;
	GtkWidget *secmgr;
	GtkWidget *emusic;
	GtkWidget *pickw;
	GtkWidget *preformat;
	GtkWidget *backdate;
	GtkWidget *comments;
};

static void metamgr_init(MetaMgr *mm);
static gboolean pickeyword_to_entry  (MetaMgr *mm, Entry *entry);
static gboolean pickeyword_from_entry(MetaMgr *mm, Entry *entry);

/* gtk stuff */
GType
metamgr_get_type(void) {
	static GType mm_type = 0;
	if (!mm_type) {
		static const GTypeInfo mm_info = {
			sizeof (GtkVBoxClass),
			NULL,
			NULL,
			NULL,
			NULL,
			NULL,
			sizeof (MetaMgr),
			0,
			(GInstanceInitFunc) metamgr_init,
		};
		mm_type = g_type_register_static(GTK_TYPE_HBOX, "MetaMgr",
				&mm_info, 0);
	}
	return mm_type;
}

static GtkWidget*
labelled_box(MetaMgr *mm, const char *caption, GtkWidget *w) {
	GtkWidget *hbox, *l;

	l = gtk_label_new_with_mnemonic(caption);
	gtk_misc_set_alignment(GTK_MISC(l), 1, 0.5);
	if (mm)
		gtk_size_group_add_widget(mm->sizegroup, l);
	gtk_label_set_mnemonic_widget(GTK_LABEL(l), w);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), l, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), w, TRUE, TRUE, 0);
	return hbox;
}

static GtkWidget*
make_subject(MetaMgr *mm) {
	mm->esubject = gtk_entry_new();
	return labelled_box(mm, _("_Subject:"), mm->esubject);
}

static GtkWidget*
make_date(MetaMgr *mm) {
	mm->date = datesel_new();
	return mm->date;
}
static GtkWidget*
make_subjdate(MetaMgr *mm) {
	GtkWidget *hbox;
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), make_subject(mm), TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), make_date(mm), FALSE, FALSE, 0);
	return hbox;
}

static GtkWidget*
make_mood(MetaMgr *mm) {
	GtkWidget *box, *combo, *l;
	combo = gtk_combo_new();
	gtk_widget_set_size_request(combo, 20, -1);

	l = gtk_label_new_with_mnemonic(_("_Mood:"));
	gtk_misc_set_alignment(GTK_MISC(l), 1, 0.5);
	gtk_size_group_add_widget(mm->sizegroup, l);
	gtk_label_set_mnemonic_widget(GTK_LABEL(l), GTK_COMBO(combo)->entry);

	box = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(box), l, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), combo, TRUE, TRUE, 0);

	mm->cmood = combo;
	mm->emood = GTK_COMBO(combo)->entry;

	return box;
}
static GtkWidget*
make_pickeyword(MetaMgr *mm) {
	GtkWidget *menu, *item;
	mm->pickw = gtk_option_menu_new();
	gtk_tooltips_set_tip(app.tooltips, mm->pickw,
			_("Specify which user icon to use with this post"),
			NULL);
	menu = gtk_menu_new();
	item = gtk_menu_item_new_with_label("[default]");
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	gtk_widget_show(item);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(mm->pickw), menu);
	return labelled_box(NULL, _("_Picture:"), mm->pickw);
}

static GtkWidget*
make_security(MetaMgr *mm) {
	mm->secmgr = secmgr_new(TRUE);
	return labelled_box(NULL, _("S_ecurity:"), mm->secmgr);
}
static GtkWidget*
make_moodpic(MetaMgr *mm) {
	GtkWidget *hbox;
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(hbox), make_mood(mm), TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_vseparator_new(), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), make_pickeyword(mm), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_vseparator_new(), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), make_security(mm), FALSE, FALSE, 0);
	return hbox;
}

static void
music_grab(GtkWidget *w, MetaMgr *mm) {
	char *music;
	GError *err = NULL;
	music = music_detect(&err);
	if (music) {
		gtk_entry_set_text(GTK_ENTRY(mm->emusic), music);
		g_free(music);
	} else if (err) {
		lj_warning(GTK_WIDGET(mm), "Error detecting music: %s", err->message);
		g_error_free(err);
	}
}
static GtkWidget*
make_music(MetaMgr *mm) {
	GtkWidget *hbox, *l, *entry;

	hbox = gtk_hbox_new(FALSE, 5);

	l = gtk_label_new_with_mnemonic(_("M_usic:"));
	gtk_misc_set_alignment(GTK_MISC(l), 1, 0.5);
	gtk_size_group_add_widget(mm->sizegroup, l);
	entry = gtk_entry_new();
	gtk_label_set_mnemonic_widget(GTK_LABEL(l), entry);
	gtk_box_pack_start(GTK_BOX(hbox), l, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

	if (music_can_detect(NULL)) {
		GtkWidget *b;
		b = gtk_button_new_with_mnemonic(_("<- _Detect"));
		gtk_tooltips_set_tip(app.tooltips, b,
				_("Retrieve the current music from a currently-running XMMS"),
				NULL);
		g_signal_connect(G_OBJECT(b), "clicked",
				G_CALLBACK(music_grab), mm);
		gtk_box_pack_start(GTK_BOX(hbox), b, FALSE, FALSE, 0);
	}
	
	mm->emusic = entry;

	return hbox;
}

static GtkWidget*
make_comments(MetaMgr *mm) {
	GtkWidget *menu, *item;
	mm->comments = gtk_option_menu_new();
	gtk_tooltips_set_tip(app.tooltips, mm->comments,
			_("Specify how comments to this entry should be handled"),
			NULL);
	menu = gtk_menu_new();

	item = gtk_menu_item_new_with_label(_("Default"));
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	gtk_widget_show(item);
	item = gtk_menu_item_new_with_label(_("Don't email"));
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	gtk_widget_show(item);
	item = gtk_menu_item_new_with_label(_("Disabled"));
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	gtk_widget_show(item);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(mm->comments), menu);
	return labelled_box(NULL, _("_Comments:"), mm->comments);
}
static GtkWidget*
make_options(MetaMgr *mm) {
	GtkWidget *box, *hbox;

	hbox = gtk_hbox_new(FALSE, 10);
	mm->preformat = gtk_check_button_new_with_mnemonic(_("P_reformatted"));
	gtk_tooltips_set_tip(app.tooltips, mm->preformat,
			_("Disable LiveJournal's auto-HTML effect"),
			_("All LiveJournal posts are HTML.  By default, LiveJournal "
			"inserts a <br/> tag after every newline in your post; "
			"this makes sense most of the time and leaves you free to "
			"use HTML as you see fit.  However, sometimes (are you writing "
			"real HTML?) you want to tell LiveJournal to leave your "
			"text unmodified."));
	mm->backdate = gtk_check_button_new_with_mnemonic(_("_Backdated"));
	gtk_tooltips_set_tip(app.tooltips, mm->backdate,
			_("Allow date to be in the past and hide entry from friends views"),
			_("If you want an entry to appear in the past (ie, before "
			"your most recent entry), toggle this option.  This has "
			"the side effect of making this entry not show up in "
			"other peoples' friends lists."));
	gtk_box_pack_start(GTK_BOX(hbox), mm->preformat, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), mm->backdate, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), gtk_vseparator_new(), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), make_comments(mm), FALSE, FALSE, 0);
	box = labelled_box(mm, _("Options:"), hbox);

	return box;
}
#if 0
static GtkWidget*
make_picdisplay(MetaMgr *mm) {
	GtkWidget *frame;
	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
	gtk_widget_set_usize(frame, 100, 100);

	mm->widgets[META_PICKEYWORD] = frame;

	return frame;
}
static GtkWidget*
make_picture(MetaMgr *mm) {
	GtkWidget *vbox, *l;
	vbox = gtk_vbox_new(FALSE, 5);
	l = gtk_label_new("Picture:");
	gtk_box_pack_start(GTK_BOX(vbox), l, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), make_picdisplay(mm), FALSE, FALSE, 0);

	mm->containers[META_PICKEYWORD] = vbox;

	return vbox;
}
#endif

static GtkWidget*
make_metamain(MetaMgr *mm) {
	GtkWidget *box;
	box = gtk_vbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(box), make_subjdate(mm), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), make_moodpic(mm),  FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), make_music(mm),    FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), make_options(mm),  FALSE, FALSE, 0);
	return box;
}

static void
metamgr_init(MetaMgr *mm) {
	/* this is a complicated widget!
	 * i'll try to break the build into smaller subfunctions. */

	gtk_box_set_spacing(GTK_BOX(mm), 0);
	gtk_container_set_border_width(GTK_CONTAINER(mm), 2);

	mm->sizegroup = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

	gtk_box_pack_start(GTK_BOX(mm), make_metamain(mm), TRUE, TRUE, 0);
	/*gtk_box_pack_start(GTK_BOX(mm), make_picture(mm), FALSE, FALSE, 0);*/
}

GtkWidget*
metamgr_new(void) {
	MetaMgr *mm = METAMGR(g_object_new(metamgr_get_type(), NULL));

	return GTK_WIDGET(mm);
}

static gboolean 
meta_gtkentry_to_entry(GtkWidget *w, char **dest) {
	char *text = NULL;
	text = gtk_editable_get_chars(GTK_EDITABLE(w), 0, -1);
	string_replace(dest, text);
	return text != NULL;
}
static gboolean 
meta_gtkentry_from_entry(GtkWidget *w, char *src) {
	gtk_entry_set_text(GTK_ENTRY(w), src ? src : "");
	return src != NULL;
}

static gboolean
meta_checkbox_to_entry(GtkWidget *w, gboolean *dest) {
	*dest = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
	return *dest;
}
static gboolean
meta_checkbox_from_entry(GtkWidget *w, gboolean src) {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), src);
	return src;
}

void 
metamgr_clear(MetaMgr *mm) {
	Entry *entry;

	entry = entry_new();
	metamgr_from_entry(mm, entry);
	entry_free(entry);
	datesel_set_tm(DATESEL(mm->date), NULL);
}

void
metamgr_to_entry(MetaMgr *mm, Entry* entry) {
	meta_gtkentry_to_entry(mm->esubject,    &entry->subject);
	meta_gtkentry_to_entry(mm->emood,       &entry->mood);
	secmgr_security_get(SECMGR(mm->secmgr), &entry->security);
	meta_gtkentry_to_entry(mm->emusic,      &entry->music);
	pickeyword_to_entry(mm, entry);
	meta_checkbox_to_entry(mm->preformat,   &entry->preformatted);
	meta_checkbox_to_entry(mm->backdate,    &entry->backdated);
	entry->comments = gtk_option_menu_get_history(GTK_OPTION_MENU(mm->comments));
	datesel_get_tm(DATESEL(mm->date), &entry->time);
}

void
metamgr_from_entry(MetaMgr *mm, Entry* entry) {
	meta_gtkentry_from_entry(mm->esubject,  entry->subject);
	meta_gtkentry_from_entry(mm->emood,     entry->mood);
	secmgr_security_set(SECMGR(mm->secmgr), &entry->security);
	meta_gtkentry_from_entry(mm->emusic,    entry->music);
	pickeyword_from_entry(mm, entry);
	meta_checkbox_from_entry(mm->preformat, entry->preformatted);
	meta_checkbox_from_entry(mm->backdate,  entry->backdated);
	gtk_option_menu_set_history(GTK_OPTION_MENU(mm->comments), entry->comments);

	datesel_set_tm(DATESEL(mm->date), &entry->time);
}

void
metamgr_login_changed(MetaMgr *mm) {
	/* load moods. */
	{
		GList *l, *strings = NULL;
		Mood *m;
		char *text;

		text = gtk_editable_get_chars(GTK_EDITABLE(mm->emood), 0, -1);

		if (conf_cur_server()) {
			for (l = conf_cur_server()->moods; l; l = l->next) {
				m = l->data;
				strings = g_list_insert_sorted(strings, m->name,
						(GCompareFunc)g_ascii_strcasecmp);
			}
		}
		if (strings)
			gtk_combo_set_popdown_strings(GTK_COMBO(mm->cmood), strings);

		gtk_entry_set_text(GTK_ENTRY(mm->emood), text);
		g_free(text);
	}

	/* load userpics. */
	{
		GtkWidget *menu;
		GtkWidget *item;

		menu = gtk_menu_new();
		item = gtk_menu_item_new_with_label(_("[default]"));
		gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
		gtk_widget_show(item);

		if (conf_cur_server() && conf_cur_user()) {
			GList *l;
			for (l = conf_cur_user()->pickws; l; l = l->next) {
				item = gtk_menu_item_new_with_label(l->data);
				gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
				gtk_widget_show(item);
			}
		}

		gtk_option_menu_set_menu(GTK_OPTION_MENU(mm->pickw), menu);
	}

	/* allow custom security. */
	secmgr_custom_set_sensitive(SECMGR(mm->secmgr),
			conf_cur_server() && conf_cur_user() 
			&& conf_cur_user()->friendgroups != NULL);
}

static gboolean 
pickeyword_to_entry(MetaMgr *mm, Entry *entry) {
	int pos;
	char *kw;

	pos = gtk_option_menu_get_history(GTK_OPTION_MENU(mm->pickw));
	if (pos == 0) {
		string_replace(&entry->pickeyword, NULL);
		return FALSE;
	}

	kw = g_list_nth_data(conf_cur_user()->pickws, pos-1);
	string_replace(&entry->pickeyword, g_strdup(kw));
	return TRUE;
}

static gboolean 
pickeyword_from_entry(MetaMgr *mm, Entry *entry) {
	int pos = 0;

	if (conf_cur_user() && entry->pickeyword) {
		GList *l;
		int i = 1;

		for (l = conf_cur_user()->pickws; l; l = l->next) {
			if (g_ascii_strcasecmp(l->data, entry->pickeyword) == 0) {
				pos = i;
				break;
			}
			i++;
		}
	}
			
	gtk_option_menu_set_history(GTK_OPTION_MENU(mm->pickw), pos);
	return pos > 0;
}

