/* logjam - a GTK client for LiveJournal.
 * Copyright (C) 2000-2002 Evan Martin <evan@livejournal.com>
 *
 * vim: tabstop=4 shiftwidth=4 noexpandtab :
 * $Id: history_recent.c,v 1.4 2002/09/24 07:57:50 martine Exp $
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include "util.h"
#include "history.h"

/* how many entries to load each time "load more..." is clicked. */
#define HISTORY_LOAD_BATCH 20

typedef struct {
	GtkWidget *win;
	GtkListStore *recentstore;
	GtkWidget *recentview;
	GtkWidget *lstats;
	char *lastdate;

	GdkPixbuf *pb_friends, *pb_private;
} HistoryUI;

enum {
	COL_SECURITY,
	COL_DATE,
	COL_SUBJECT,
	COL_ITEMID,
	COL_COUNT
};

static void
recalculate_stats(HistoryUI *hui) {
	GtkTreeModel *model;
	GtkTreeIter iter;
	GdkPixbuf *pb;
	char *stats;
	//char *date;

	int ecount, pecount, fecount;

	ecount = pecount = fecount = 0;

	model = gtk_tree_view_get_model(GTK_TREE_VIEW(hui->recentview));
	gtk_tree_model_get_iter_first(model, &iter);

	do {
		gtk_tree_model_get(model, &iter,
				COL_SECURITY, &pb,
				//COL_DATE, &date,
				-1);
		ecount++;

		/* FIXME: gross hack. */
		if (pb == hui->pb_friends)
			fecount++;
		else if (pb == hui->pb_private)
			pecount++;

		//g_free(date);
	} while (gtk_tree_model_iter_next(model, &iter));
	stats = g_strdup_printf(_("%d entries displayed.  "
			"%.1f%% public / %.1f%% friends-only / %.1f%% private."),
			ecount,
			100.0f * (ecount - pecount - fecount) / (float)ecount,
			100.0f * fecount / (float)ecount,
			100.0f * pecount / (float)ecount);
	gtk_label_set_text(GTK_LABEL(hui->lstats), stats);
	g_free(stats);
}

static int
load_recent(HistoryUI *hui, GtkWidget *parent) {
	NetRequest *request;
	NetResult *result;
	int events_count;
	int i;
	GtkTreeIter iter;
	char *lastdate = NULL;
	Security sec;
	GdkPixbuf *pb;
	
	request = net_request_new("getevents");
	net_request_seti(request, "truncate", 100);
	net_request_seti(request, "prefersubject", 1);
	net_request_seti(request, "noprops", 1);
	net_request_copys(request, "selecttype", "lastn");
	net_request_seti(request, "howmany", HISTORY_LOAD_BATCH);
	if (hui->lastdate)
		net_request_copys(request, "beforedate", hui->lastdate);
	net_request_copys(request, "lineendings", "dots");
	if (conf.usejournal)
		net_request_copys(request, "usejournal", conf.usejournal);

	result = net_request_run(parent, _("Loading History..."), request);
	net_request_free(request);

	if (!net_result_succeeded(result)) {
		net_result_free(result);
		return -1;
	}

	events_count = net_result_geti(result, "events_count");
	for (i = 1; i <= events_count; i++) {
		char buf[30];
		char *ev;
		sprintf(buf, "events_%d_", i);

		gtk_list_store_append(hui->recentstore, &iter);
		lastdate = net_result_get_prefix(result, buf, "eventtime");
		security_load_from_strings(&sec, 
				net_result_get_prefix(result, buf, "security"),
				net_result_get_prefix(result, buf, "allowmask"));
		switch (sec.type) {
			case SECURITY_FRIENDS:
			case SECURITY_CUSTOM:
				pb = hui->pb_friends; break;
			case SECURITY_PRIVATE:
				pb = hui->pb_private; break;
			default:
				pb = NULL; /* public: no icon */
		}
		ev = net_result_get_prefix(result, buf, "event");
		ev = urldecode(ev);
		gtk_list_store_set(hui->recentstore, &iter,
				COL_SECURITY, pb,
				COL_DATE, lastdate,
				/* because we set prefersubjects above, the real subject
				 * is returned in event. */
				COL_SUBJECT, ev,
				COL_ITEMID, atoi(net_result_get_prefix(result, buf, "itemid")),
				-1);
		g_free(ev);
	}
	string_replace(&hui->lastdate, lastdate); 

	recalculate_stats(hui);

	return TRUE;
}

static void
load_more_cb(GtkWidget *w, HistoryUI *hui) {
	load_recent(hui, hui->win);
}

static void
row_activated_cb(GtkTreeView *treeview, GtkTreePath *arg1, GtkTreeViewColumn *arg2, HistoryUI* hui) {
	gtk_dialog_response(GTK_DIALOG(hui->win), GTK_RESPONSE_OK);
}

static GtkWidget*
make_recentview(HistoryUI *hui) {
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	
	hui->recentview = 
		gtk_tree_view_new_with_model(GTK_TREE_MODEL(hui->recentstore));
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(hui->recentview), TRUE);
	g_object_unref(G_OBJECT(hui->recentstore));
	g_signal_connect(G_OBJECT(hui->recentview), "row-activated",
			G_CALLBACK(row_activated_cb), hui);

	renderer = gtk_cell_renderer_pixbuf_new();
	column = gtk_tree_view_column_new_with_attributes(_("Security"), renderer,
			"pixbuf", COL_SECURITY,
			NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(hui->recentview), column);

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes(_("Date"), renderer,
			"text", COL_DATE,
			NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(hui->recentview), column);

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes(_("Subject"), renderer,
			"text", COL_SUBJECT,
			NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(hui->recentview), column);

	return scroll_wrap(hui->recentview);
}

static void
hui_destroy_cb(GtkWidget *w, HistoryUI *hui) {
	string_replace(&hui->lastdate, NULL); 
}

void
make_dialog(HistoryUI *hui, GtkWidget *parent) {
	GtkWidget *vbox, *bbox, *button;

	hui->win = lj_dialog_new(parent, _("History"), -1, -1);
	g_signal_connect(G_OBJECT(hui->win), "destroy",
			G_CALLBACK(hui_destroy_cb), hui);
	gtk_window_set_default_size(GTK_WINDOW(hui->win), 500, 400);
	gtk_dialog_set_default_response(GTK_DIALOG(hui->win), GTK_RESPONSE_OK);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), make_recentview(hui), TRUE, TRUE, 0);
	hui->lstats = gtk_label_new("");
	gtk_label_set_selectable(GTK_LABEL(hui->lstats), TRUE);
	gtk_box_pack_start(GTK_BOX(vbox), hui->lstats, FALSE, FALSE, 0);

	bbox = lj_dialog_buttonbox_new();
	button = lj_dialog_buttonbox_button_with_label(bbox, _("_Load More..."));
	g_signal_connect(G_OBJECT(button), "clicked",
			G_CALLBACK(load_more_cb), hui);

	lj_dialog_set_contents_buttonbox(hui->win, vbox, bbox);

	lj_dialog_add_okcancel(hui->win, NULL);
}

static Entry*
load_selected(HistoryUI *hui) {
	GtkTreeModel *model;
	GtkTreeIter iter;
	int itemid;

	gtk_tree_selection_get_selected(
			gtk_tree_view_get_selection(GTK_TREE_VIEW(hui->recentview)),
			&model, &iter);
	gtk_tree_model_get(model, &iter,
			COL_ITEMID, &itemid,
			-1);

	return history_load_itemid(hui->win, itemid);
}

Entry*
history_recent_dialog_run(GtkWidget *parent) {
	HistoryUI hui_actual = {0}, *hui = &hui_actual;
	Entry *entry = NULL;

	hui->recentstore = gtk_list_store_new(COL_COUNT,
			GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT);

	make_dialog(hui, parent);

	hui->pb_friends = gtk_widget_render_icon(hui->win, 
			"logjam-protected", GTK_ICON_SIZE_MENU, NULL);
	hui->pb_private = gtk_widget_render_icon(hui->win, 
			"logjam-private", GTK_ICON_SIZE_MENU, NULL);

	if (load_recent(hui, parent) < 0)
		return NULL;

	if (!lj_dialog_run(hui->win))
		return NULL;

	/* if we get here they picked an entry. */

	entry = load_selected(hui);

	gtk_widget_destroy(hui->win);
	return entry;
}

