/* $Id: util.c,v 1.2 2007/10/25 11:43:33 merdely Exp $ */
#include <sys/param.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <limits.h>
#define LDAP_DEPRECATED 1
#include <ldap.h>
#include <errno.h>
#include <err.h>
#include <syslog.h>
#include <login_cap.h>

#include "login_ldap.h"

int debug = 0;

void
dlog(int d, char *fmt, ...)
{
	va_list ap;

	/* 
	 * if debugging is on, print everthing to stderr
	 * otherwise, syslog it if d = 0. messing with 
	 * newlines means there wont be newlines in stuff
	 * that goes to syslog.
	 */

	va_start(ap, fmt);
	if (debug) {
		vfprintf(stderr, fmt, ap);
		fputc('\n', stderr);
	} else if (d == 0)
		vsyslog(LOG_WARNING, fmt, ap);

	va_end(ap);
}

static int
parse_server_line(char *buf, struct servinfo *s)
{
	/** 
	 * x-ldap-server=hostname,port,mode,version 
	 *
	 * must have a hostname
	 * mode can be "clear", "starttls" or "ssl"
 	 * mode defualts to clear
	 * for clear and starttls, port defaults to 389
	 * for ssl, port defualts to 636
	 * things like hostname,,starttls are accepted
	 */
	
	char *port;
	char *mode;
	char *version;
	const char *errstr;

	if (buf == NULL) {
		dlog(1, "parse_server_line got NULL buf!"); 
		return(0);
	}

	dlog(1, "parse_server_line buf = %s", buf);

	/* set the defaults */
	s->port = LDAP_PORT;
	s->mode = MODE_CLEAR;
	s->version = LDAP_VERSION3;
	
	s->host = strsep(&buf, ",");
	if (s->host == NULL || s->host[0] == '\0') {
		dlog(0, "parse_server_line no host"); 
		return(0);
	}	

	port = strsep(&buf, ",");
	if (port == NULL || port[0] == '\0')
		dlog(1, "parse_server_line port == NULL, will use default");
	else {
		dlog(1, "parse_server_line port = %s", port);
		s->port = strtonum(port, 1, 65535, &errstr);
		if (errstr != NULL) {
			dlog(1, "bad port %s: %s", port, errstr);
			return(0);
		}
	}

	mode = strsep(&buf, ",");
	if (mode == NULL || mode[0] == '\0') {
		dlog(1, "parse_server_line mode == NULL, will use default"); 
		mode = "clear";
	}

	/* set the mode */
	if (strcmp(mode, "clear") == 0)
		s->mode = MODE_CLEAR;
	else if (strcmp(mode, "starttls") == 0)
		s->mode = MODE_STARTTLS;
	else if (strcmp(mode, "ssl") == 0)
		s->mode = MODE_SSL;
	else {
		dlog(0, "parse_server_line invalid mode '%s'", mode);
		return(0);
	}

	/* see if we need to change the port */
	if ((port == NULL || port[0] == '\0') && s->mode == MODE_SSL)
		s->port = LDAPS_PORT;
	
	/* see if theres a version to use */
	version = strsep(&buf, ",");
	if (version != NULL && version[0] != '\0') 
		s->version = atoi(version);

	return(1); 
}

static void
set_tls_opts(struct auth_ctx *ctx)
{
	int rc;
	int v;

	/* 
	 * if we're going to be using ssl, point the ldap library
	 * to the appropriate certs. 
	 *
	 * if we're ever running setuid root, we should be careful
	 * we dont use the contents of /root/.ldaprc, as thats 
	 * probably not what people want. explicitly setting CERTFILE
	 * and KEYFILE here will clear any settings read from an 
	 * ldaprc. so if they're NULL (ie, not defined in login.conf)
	 * it will clear up any sticky situations
	 */ 

	dlog(1,"setting cert info");
	rc = ldap_set_option(NULL, LDAP_OPT_X_TLS_CERTFILE, ctx->usercert);
	if (rc != LDAP_OPT_SUCCESS)
		dlog(1, "couldn't set ldap certfile!");

	rc = ldap_set_option(NULL, LDAP_OPT_X_TLS_KEYFILE, ctx->userkey);
	if (rc != LDAP_OPT_SUCCESS)
		dlog(1, "coudln't set ldap keyfile!");

	if (ctx->cacert != NULL && ctx->cacert[0] != '\0') {
		rc = ldap_set_option(NULL, LDAP_OPT_X_TLS_CACERTFILE, ctx->cacert);
		if (rc != LDAP_OPT_SUCCESS)
			dlog(1, "coudln't set ldap ca certfile!");
	}

	if (ctx->cacertdir != NULL && ctx->cacertdir[0] != '\0') {
		rc = ldap_set_option(NULL, LDAP_OPT_X_TLS_CACERTDIR, ctx->cacertdir);
		if (rc != LDAP_OPT_SUCCESS) 
			dlog(1, "couldn't set ldap ca cert dir!");
	}

	if (ctx->s.mode == MODE_SSL) {
		dlog(1, "set hard tls");
		v = LDAP_OPT_X_TLS_HARD;
		rc = ldap_set_option(NULL, LDAP_OPT_X_TLS, &v);
		if (rc != LDAP_OPT_SUCCESS)
			dlog(0, "couldn't force ssl! this does not bode well!");
	} else {
		dlog(1, "clearing ssl set");
		v = LDAP_OPT_X_TLS_TRY;
		rc = ldap_set_option(NULL, LDAP_OPT_X_TLS, &v);
		if (rc != LDAP_OPT_SUCCESS)
			dlog(0, "couldn't set ssl to try!");
	}
}
	
int 
do_conn(struct auth_ctx *ctx)
{
	int rc;

	dlog(1, "host %s, port %d, version %d", ctx->s.host, ctx->s.port, ctx->s.version);

	set_tls_opts(ctx);

	/* 
	 * it would be better to use ldap_init here as the manpages state
	 * ldap_open will be depreciated. however, if the server is 
	 * unavailable we wont know till we try to bind when we use ldap_init,
	 * which makes a mess of check alternate servers from login.conf. so,
	 * we use ldap_open which will let us know straight away if a server 
	 * is unreachable servers. XXX gibberish
	 *
	 * the other option is to use ldap_initalize() and ldap_open_defconn()
	 * but since they are undocumented im avoiding them.
	 */

	ctx->ld = ldap_open(ctx->s.host, ctx->s.port);
	if (ctx->ld == NULL) {
		dlog(0, "ldap_open(%s, %d) failed", ctx->s.host, ctx->s.port);
		return(0);
	}

	dlog(1, "connect success!");

	rc = ldap_set_option(ctx->ld, LDAP_OPT_PROTOCOL_VERSION, &ctx->s.version);
	if (rc != LDAP_SUCCESS) {
		dlog(0, "error setting version %d: %s", ctx->s.version, ldap_err2string(rc));
		return(0);
	}

	dlog(1, "set version to %d", ctx->s.version);

	if (ctx->s.mode == MODE_STARTTLS) {
		dlog(1, "starttls!");
		rc = ldap_start_tls_s(ctx->ld, NULL, NULL);
		if (rc != LDAP_SUCCESS)
			dlog(0, "could not start tls!");
	}
	return(1);
}

int
conn(struct auth_ctx *ctx)
{
	/* 
	 * first we look for x-ldap-server, then x-ldap-serveralt%d
	 */
	char *servstr;
	int i = 0;
	/* x-ldap-serveralt0000*/
	char buf[21];

	servstr = login_getcapstr(ctx->lc, CAP_LDAP_SERVER, NULL, NULL);
	if (servstr == NULL) {
		dlog(0, "you need to set " CAP_LDAP_SERVER " for %s", ctx->class);
		return(0);
	}

	if (!parse_server_line(servstr, &ctx->s)) {
		dlog(0, "error parsing server line");
		return(0);
	}

	if (do_conn(ctx))
		return(1);

	dlog(1, "%s failed, trying alternates", servstr);

	for (i = 0; i < 10000; i++) {

		memset(buf, 0x00, sizeof(buf));

		if (snprintf(buf, sizeof(buf), "x-ldap-serveralt%d", i) >= sizeof(buf)) {
			/* should never happen */
			dlog(0, "serveralt string too long!");
			return(0);
		}

		servstr = login_getcapstr(ctx->lc, buf, NULL, NULL);
		if (servstr == NULL) /* end of the line */
			return(0);

		dlog(1, "trying %s %s", buf, servstr); 

		if (!parse_server_line(servstr, &ctx->s)) {
			dlog(0, "error parsing server line");
			return(0);
		}

		if (do_conn(ctx)) {
			dlog(1, " made conn i = %d", i);
			return(1);
		}

		dlog(1, "nup!");
	} 
	/* all the serveralts have failed*/
	return(0);
}

static int
getscope(char *scope)
{
	if (scope == NULL || scope[0] == '\0')
		return(LDAP_SCOPE_SUBTREE);

	if (strcmp(scope, "base") == 0)
		return(LDAP_SCOPE_BASE);
	else if (strcmp(scope, "one") == 0)
		return(LDAP_SCOPE_ONELEVEL);
	else if (strcmp(scope, "sub") == 0)
		return(LDAP_SCOPE_SUBTREE);
	else 
		return(-1);
}

/* 
 * get things from login.conf that we'll need
 * regardless of the auth type etc
 */
int
get_usearch_defaults(struct auth_ctx *ctx)
{
	char *err = "err ...?!";
	char *scope = NULL;

	ctx->ufilter = login_getcapstr(ctx->lc, CAP_LDAP_FILTER, NULL, err);
	if (ctx->ufilter == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_FILTER);
		return(0);
	} else if (ctx->ufilter == NULL) {
		dlog(0, "you need to set " CAP_LDAP_FILTER " for %s", ctx->class);
		return(0);
	}

	ctx->ufilter = parse_filter(ctx, ctx->ufilter);

	scope = login_getcapstr(ctx->lc, CAP_LDAP_USCOPE, NULL, err);
	if (scope == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_USCOPE);
		return(0);
	}

	ctx->uscope = getscope(scope);
	if (ctx->uscope == -1) {
		dlog(0, "bad user scope %s", scope);
		return(0);
	}

	dlog(1, "usearch:\n\tufilter %s\n\tscope: %s", ctx->ufilter, 
	    scope ? scope : "sub");
	return(1);
}

int
get_gsearch_defaults(struct auth_ctx *ctx)
{
	char *err = "err!@";
	char *scope = NULL;

	ctx->gfilter = login_getcapstr(ctx->lc, CAP_LDAP_GROUPFILTER, NULL, err);
	if (ctx->gfilter == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_GROUPFILTER);
		return(0);
	}
	
	if (ctx->gfilter == NULL || ctx->gfilter[0] == '\0')
		return(1); /* no groupfilter, so all done */
	
	ctx->gfilter = parse_filter(ctx, ctx->gfilter);

	ctx->groupdn = login_getcapstr(ctx->lc, CAP_LDAP_GROUPDN, NULL, err);
	if (ctx->groupdn == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_GROUPDN);
		return(0);
	}

	scope = login_getcapstr(ctx->lc, CAP_LDAP_GSCOPE, NULL, err);
	if (scope == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_GSCOPE);
		return(0);
	}

	ctx->gscope = getscope(scope);
	if (ctx->gscope == -1) {
		dlog(0, "bad group scope %s", scope);
		return(0);
	}

	return(1);
}

/*
 * get the base dn, bind dn and bind password from login.conf 
 */
int
get_bind_defaults(struct auth_ctx *ctx)
{
	char *err = "err ..."; 
	
	ctx->basedn = login_getcapstr(ctx->lc, CAP_LDAP_BASEDN, NULL, err);
	if (ctx->basedn == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_BASEDN);
		return(0);
	}

	ctx->binddn = login_getcapstr(ctx->lc, CAP_LDAP_BINDDN, NULL, err);
	if (ctx->binddn == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_BINDDN);
		return(0);
	}

	ctx->bindpw = login_getcapstr(ctx->lc, CAP_LDAP_BINDPW, NULL, err);
	if (ctx->binddn == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_BINDPW);
		return(0);
	}	

	dlog(1, "defaults:\n\tbasedn %s\n\tbinddn %s\n\tbindpw %s", 
	    ctx->basedn ? ctx->basedn : "none",
	    ctx->binddn ? ctx->binddn : "none",
	    ctx->bindpw ? ctx->bindpw : "none");

	return(1);
}

int
get_misc(struct auth_ctx *ctx)
{
	/* get some bits and bobs */
	int timeout;

	timeout = (int)login_getcapnum(ctx->lc, CAP_LDAP_TIMEOUT, (quad_t)DEFTIMEOUT, (quad_t)-1);
	if (timeout == -1) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_TIMEOUT);
		return(0);
	}
	ctx->tv.tv_sec = timeout; 
	ctx->tv.tv_usec = (timeout % 1000) * 1000;

	dlog(1, "set timeout sec %d, usec %d", ctx->tv.tv_sec, ctx->tv.tv_usec);	

	if (login_getcapbool(ctx->lc, CAP_LDAP_NOREFERRALS, 0) != 0)
		ctx->noref = 1;
	else
		ctx->noref = 0;

	dlog(1, "set noref %d", ctx->noref);

	if (login_getcapbool(ctx->lc, CAP_LDAP_REFKEEPCREDS, 0) != 0)
		ctx->keepcreds = 1;
	else
		ctx->keepcreds = 0;

	dlog(1, "set keepcreds %d", ctx->keepcreds);

	return(1);
}

int
load_ssl_certs(struct auth_ctx *ctx)
{
	/*
	 * here we look for certificates and keys
	 * we need a ca cert, a user cert and user key
	 * 
	 * the user stuff is given by x-ldap-usercert=/home/%u/.certs/
	 */

	char *err = "err ...?";

	ctx->cacert = login_getcapstr(ctx->lc, CAP_LDAP_CACERT, NULL, err);
	if (ctx->cacert == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_CACERT);
		return(0);
	}

	ctx->cacertdir = login_getcapstr(ctx->lc, CAP_LDAP_CACERTDIR, NULL, err);
	if (ctx->cacertdir == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_CACERTDIR);
		return(0);
	}

	ctx->usercert = login_getcapstr(ctx->lc, CAP_LDAP_USERCERT, NULL, err);
	if (ctx->usercert == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_CACERT);
		return(0);
	}

	if (ctx->usercert != NULL) 
		ctx->usercert = parse_filter(ctx, ctx->usercert);

	ctx->userkey = login_getcapstr(ctx->lc, CAP_LDAP_USERKEY, NULL, err);
	if (ctx->userkey == err) {
		dlog(0, "login.conf error class %s entry %s", ctx->class, CAP_LDAP_USERKEY);
		return(0);
	}
	
	if (ctx->userkey != NULL)
		ctx->userkey = parse_filter(ctx, ctx->userkey);

	dlog(1, "load_ssl_certs says:\n\tcacert %s\n\tcacertdir %s\n\tusercert %s\n\tuserkey %s",
	    ctx->cacert ? ctx->cacert : "none",
	    ctx->cacertdir ? ctx->cacertdir : "none",
	    ctx->usercert ? ctx->usercert : "none",
	    ctx->userkey ? ctx->userkey : "none");
	return(1);	
}


/*
 * Convert format specifiers from the filter in login.conf to their
 * real values. return the new filter in the filter argument.
 */
char * 
parse_filter(struct auth_ctx *ctx, char *str)
{
	char tmp[BUFSIZ];
	char hostname[MAXHOSTNAMELEN];
	char *p, *q;
	
	bzero(tmp, sizeof(tmp));

	p = str;
	q = tmp;

	/* 
	 * copy over from p to q, if we hit a %, substitute the real value,
	 * if we hit a NULL, its the end of the filter string 
	 */
	for (;;) {

		while(*p != '%' && *p != '\0' && ((q - tmp) < sizeof(tmp))) 
			*q++ = *p++;

		if ((q - tmp) == sizeof(tmp)) {
			dlog(0, "filter string too large, unable to "
			    "process: %s", p);
			return(NULL);
		}

		if (*p == '\0')
			break;
		p++;

		switch(*p) {
		
		case 'u': /* username */

			if (snprintf(tmp, sizeof(tmp), "%s%s", tmp, 
			    ctx->user) >= sizeof(tmp)) {
				dlog(0, "filter string too large, has " 
				    "been truncated: %s", tmp); 
				return(NULL);
			}

			p++;
			q += strlen(ctx->user);
			break;

		case 'h': /* hostname */

			if (gethostname(hostname, sizeof(hostname)) == -1) {
				dlog(0, "couldn't get host name for "
				    "%%h %s", strerror(errno));
				return(NULL);
			}

			if (snprintf(tmp, sizeof(tmp), "%s%s", tmp, 
			    hostname) >= sizeof(tmp)) {
				dlog(0, "filter string too large, has "
				    "been truncated: %s", tmp); 
				return(NULL);
			}

			p++;
			q += strlen(hostname);
			break;

		case 'd': /* user dn */

			if (ctx->userdn == NULL) {
				dlog(0, "no userdn has been recorded");
				return(0);
			}

			/* good to go */
			if (snprintf(tmp, sizeof(tmp), "%s%s", tmp, ctx->userdn)
			    >= sizeof(tmp)) {
				dlog(0, "filter string too large, has "
				    "been truncated: %s", tmp); 
				return(NULL);
			}

			p++;
			q += strlen(ctx->userdn);
			break;

		default:
			dlog(0, "parse_filter: invalid filter specifier "
			    "'%c'", *p);
			return(NULL);
		}
		
	}

	p = strdup(tmp);
	if (p == NULL) {
		dlog(0, "%s", strerror(errno));
		return(NULL);
	}

	return(p);
}

