/* $Id: login_ldap.c,v 1.16 2003/02/19 22:52:06 peterw Exp $
 * Copyright (c) 2002 Institute for Open Systems Technology Australia (IFOST)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * By Peter Werner <peterw@ifost.org.au>
 * Please visit http://www.ifost.org.au
 * Questions may be directed to <peterw@ifost.org.au> 
 * For information on commercial support please contact <gregb@ifost.org.au>
 */

#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <err.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <lber.h>
#include <ldap.h>
#include <login_cap.h>
#include <stdarg.h>
#include <bsd_auth.h>

/* 
 * we need to be able to distinguish between something not working (eg, can't
 * connect to server) and something being wrong (eg, something wrong in
 * /etc/login.conf). this is mainly so we dont try and check the alternate
 * server if something is wrong.
 */ 
#define LA_FAIL		0
#define LA_OK		1	
#define LA_FAIL_FATAL	2

/* login.conf strings */
#define CAP_LDAP_SERVER		"x-ldap-server"
#define CAP_LDAP_PORT		"x-ldap-port"	/* num */
#define CAP_LDAP_BASEDN		"x-ldap-basedn"
#define CAP_LDAP_FILTER		"x-ldap-filter"
#define CAP_LDAP_TLS		"x-ldap-use-tls" /* bool */
#define CAP_LDAP_BINDDN		"x-ldap-binddn"
#define CAP_LDAP_BINDPW 	"x-ldap-bindpw"
#define CAP_LDAP_GROUPFILTER 	"x-ldap-groupfilter"
#define CAP_LDAP_GROUPDN	"x-ldap-groupdn"
#define CAP_LDAP_SERVERALT	"x-ldap-server-alt" 
#define CAP_LDAP_TIMEOUT	"x-ldap-timeout"
#define CAP_LDAP_USCOPE		"x-ldap-uscope"
#define CAP_LDAP_GSCOPE		"x-ldap-gscope"
#define CAP_LDAP_NOREFERRALS	"x-ldap-noreferrals" /* bool */

int debug = 0;
int noreferrals = 0;

static int	auth_ldap(char *, char *, char *);
static int 	parse_filter(login_cap_t *, char *, char **, char *);
static int 	build_uri(login_cap_t *, char *, char **);
static int 	do_simple_bind(login_cap_t *, LDAP *, char *, char *);
static void 	free_data(login_cap_t *, LDAP *, LDAPMessage *);
static int	init_all(login_cap_t *, LDAP **, char *);
static int 	get_timeout(login_cap_t *, struct timeval *);
static int	getscope(login_cap_t *, char *);
static void	handler(int);

#define DEFTIMEO	60 /* number of seconds to wait before a timeout */
#define dprintf		if (debug) printf

/*
 * Authenticate a user using LDAP
 */

int
main(int argc, char **argv)
{
	int ret, c;
        char *class, *service, *username, *password;
	char backbuf[BUFSIZ];
	FILE *back = NULL;

        password = class = service = NULL;

	(void)signal(SIGQUIT, SIG_IGN);
	(void)signal(SIGINT, SIG_IGN);
	(void)signal(SIGALRM, handler);
	(void)setpriority(PRIO_PROCESS, 0, 0);

	openlog(NULL, LOG_ODELAY, LOG_AUTH);

        /*
         * Usage: login_xxx [-s service] [-v var] user [class]
         */
        while ((c = getopt(argc, argv, "ds:v:")) != -1)
                switch(c) {
                case 'd':
                        back = stdout;
			debug = 1;
                        break;
                case 'v':
                        break;
                case 's':       /* service */
                        service = optarg;
                        if (strcmp(service, "login") != 0 &&
				strcmp(service, "challenge") != 0 &&
				strcmp(service, "response") != 0)
                                	errx(1, "%s not supported", service);
                        break;
                default:
                        syslog(LOG_ERR, "usage error");
                        exit(1);
                }


        if (service == NULL)
                service = LOGIN_DEFSERVICE;
	
        switch(argc - optind) {
        case 2:
                class = argv[optind + 1];
		/* FALLTHROUGH */
        case 1:
                username = argv[optind];
                break;
        default:
                syslog(LOG_ERR, "usage error");
		dprintf("usage error\n");
                exit(1);
        }

	if (strlen(username) >= MAXLOGNAME) {
		syslog(LOG_ERR, "username too long");
		dprintf("username too long\n");
		exit(1);
	}

        /*
         * Filedes 3 is the back channel, where we send back results.
         */
        if (back == NULL && (back = fdopen(3, "a")) == NULL)  {
                syslog(LOG_ERR, "reopening back channel");
                dprintf("reopening back channel\n");
                exit(1);
        }

	if (strcmp(service, "login") == 0) {
		password = getpass("Password: ");
	} else if (strcmp(service, "response") == 0) {
		int n;

		/* 
		 * format of back channel message 
		 * 
		 * c h a l l e n g e \0 p a s s w o r d \0
		 *
		 * challenge can be NULL (so can password too
		 * i suppose).
		 */
 
		n = read(3, backbuf, sizeof(backbuf));
		if (n == -1) {
			syslog(LOG_ERR, "read error from back channel");
			dprintf("read error from back channel\n");
			exit(1);
		}

		/* null challenge */
		if (backbuf[0] == '\0') 
			password = backbuf + 1;
		/* skip the first string to get the password */
		else if ((password = strchr(backbuf, '\0')) != NULL) 
			password++;
		else
			syslog(LOG_ERR, "protocol error on back channel");
		
	} else if (strcmp(service, "challenge") == 0) {
		(void)fprintf(back, BI_SILENT "\n");
		exit(0);
	} else { 
		syslog(LOG_ERR, "unsupported service type %s", service);	
		errx(1, "unsupported service type %s", service);
	}

	if (password == NULL || password[0] == '\0') {
		dprintf("Null password\n");
		(void)fprintf(back, BI_REJECT "\n");
		exit(1);
	}

	ret = auth_ldap(username, class, password);

	(void) memset(password, 0, strlen(password));
					     
	if (ret == LA_OK) 
		(void)fprintf(back, BI_AUTH "\n");
	else 
		(void)fprintf(back, BI_REJECT "\n");

	closelog();

	exit(0);
	/* NOTREACHED */
}

static int
auth_ldap(char *username, char *class, char *password)
{
	int 		rc, scope;
	char 		*basedn, *userdn;
	char 		*errmsg = "error";
	char    	*filter, *groupdn;
	login_cap_t 	*lc;	
	LDAP 		*ld;
        LDAPMessage     *res, *ent;
	struct timeval	timeo;

	/* make sure we dont hang around forever */
	alarm(300); 

	lc = login_getclass(class);
	if (lc == NULL) {
		syslog(LOG_ERR, "%s: no such class", class);
		dprintf("%s: no such class\n", class);
		return(LA_FAIL);
	}

	rc = init_all(lc, &ld, CAP_LDAP_SERVER); 
	if (rc == LA_FAIL_FATAL)
		return(LA_FAIL);
	if (rc == LA_FAIL) {
		rc = init_all(lc, &ld, CAP_LDAP_SERVERALT);
		if (rc != LA_OK) 
			return(LA_FAIL);
	}	

	basedn = login_getcapstr(lc, CAP_LDAP_BASEDN, NULL, NULL);
	if (basedn == NULL) {
		syslog(LOG_ERR, "couldn't get value for " CAP_LDAP_BASEDN);	
		dprintf("couldn't get value for " CAP_LDAP_BASEDN "\n");	
		free_data(lc, ld, NULL);
		return(LA_FAIL);
	}

	rc = parse_filter(lc, username, &filter, NULL);
	if (rc == 0){
		syslog(LOG_ERR, "couldnt parse " CAP_LDAP_FILTER);
		dprintf("couldnt parse " CAP_LDAP_FILTER "\n");
		free_data(lc, ld, NULL);
		return(LA_FAIL);
	} else if (rc == 1 && filter == NULL) {
		/* this should never happen here */
		syslog(LOG_ERR, "internal error");
		dprintf("internal error\n");
		free_data(lc, ld, NULL);
		return(LA_FAIL);
	}

	if (debug) 
		(void)fprintf(stderr, "filter = %s\n", filter);

	if (get_timeout(lc, &timeo) == 0) {
		free_data(lc, ld, NULL); 
		return(LA_FAIL);	
	}

	scope = getscope(lc, CAP_LDAP_USCOPE);
	if (scope == -2)
		return(LA_FAIL);

	rc = ldap_search_st(ld, basedn, scope, filter, NULL, 0, &timeo,
	     &res);	
	dprintf("search result 0x%x\n", rc);
  	if (rc != LDAP_SUCCESS) {
		if (!(noreferrals && (rc == LDAP_PARTIAL_RESULTS))) {
			dprintf("ldap_search_st failed: %s",
				    ldap_err2string(rc));
			free_data(lc, ld, NULL);
			return(LA_FAIL);
		}
  	}

	filter = NULL;
	free(filter);

	/* 
	 * here we get the dn of the user from the first
	 * search. Then, we try and bind using the dn we
	 * just got and the password entered at the prompt.
	 * This way, we dont have to worry about passwords
	 * stored encrypted on the server (ala openldap)
	 */

	ent = ldap_first_entry(ld, res);
	if (ent == NULL) { /* no such user */
		free_data(lc, ld, res);
		return(LA_FAIL);
	}

	userdn = ldap_get_dn(ld, ent); 
	if (userdn == NULL) {
		syslog(LOG_ERR, "couldnt get dn for %s", username);
		dprintf("couldnt get dn for %s\n", username);
		free_data(lc, ld, res);
		return(LA_FAIL);
	}

	rc = do_simple_bind(lc, ld, userdn, password);
	if (rc != LA_OK) {
		free_data(lc, ld, res);
		return(LA_FAIL);
	}
		
	/* 
	 * Now we get the group details
	 */

	groupdn = login_getcapstr(lc, CAP_LDAP_GROUPDN, NULL, errmsg);
	if (groupdn == errmsg) {
		syslog(LOG_ERR,	"error occured getting " CAP_LDAP_GROUPDN);
		dprintf("error occured getting " CAP_LDAP_GROUPDN "\n");
		return(LA_FAIL);
	}

	/* 
	 * if no groupdn has been set, and we've gotten this far,
	 * the admin isnt using groups and the user is auth'ed from above
	 * so return ok
	 */
	if (groupdn == NULL) {
		free_data(lc, ld, res);
 		return(LA_OK);
	}

	rc = parse_filter(lc, username, &filter, userdn);
	if (rc == 0){
		syslog(LOG_ERR, "couldnt parse " CAP_LDAP_GROUPFILTER);
		dprintf("couldnt parse " CAP_LDAP_GROUPFILTER "\n");
		free_data(lc, ld, res);
		return(LA_FAIL);
	} else if (rc == 1 && filter == NULL) {
		/* 
		 * hmmm, the admin has set a groupdn in
		 * login.conf but hasnt set any filter.
		 * this is silly, so we say so and deny
		 * the request.
		 *
		 * NB: if you dont want to do a group 
		 * access check, dont add a groupdn.
		 */
		syslog(LOG_ERR, "groupdn but no group filter");
		dprintf("groupdn but no group filter\n");
		free_data(lc, ld, res);
		return(LA_FAIL);
	}

	if (debug)
		(void)fprintf(stderr, "filter = %s\n", filter);

	ldap_msgfree(res);
	
	if (get_timeout(lc, &timeo) == 0) {
		free_data(lc, ld, NULL);
		return(LA_FAIL);
	}

	scope = getscope(lc, CAP_LDAP_GSCOPE);
	if (scope == -2)
		return(LA_FAIL);

	rc = ldap_search_st(ld, groupdn, scope, filter, NULL, 0, &timeo,
	    &res);	
	if (rc != LDAP_SUCCESS) {
		free_data(lc, ld, NULL);
		return(LA_FAIL);
	}
	
	ent = ldap_first_entry(ld, res);
	if (ent == NULL) { 
		free_data(lc, ld, res);
		return(LA_FAIL);
	}

	free(filter);
	free_data(lc, ld, res);

	return(LA_OK);
}

static int
init_all(login_cap_t *lc, LDAP **ldp, char *capstr)
{
	char *server, *uri;
	int rc, version;
	LDAP *ld;

	server = login_getcapstr(lc, capstr, NULL, NULL);
	if (server == NULL) {
		syslog(LOG_ERR, "couldn't get %s ", capstr);
		dprintf("couldn't get %s\n", capstr);
		return(LA_FAIL_FATAL);
	}

	if (server == NULL || server[0] == '\0') {
		syslog(LOG_ERR, "no server specified for %s", capstr);
		dprintf("no server specified for %s\n", capstr);
		return(LA_FAIL_FATAL);
	}

	rc = build_uri(lc, server, &uri);
	if (rc != LA_OK) {
		syslog(LOG_ERR, "fatal error building URI");
		dprintf("fatal error bulding URI\n");
		return(rc);
	}
	
	if (debug) 
		(void)fprintf(stderr, "uri = %s\n", uri);

	rc = ldap_initialize(&ld, uri);
	if (rc != LDAP_SUCCESS) {
		syslog(LOG_ERR, "%s", ldap_err2string(rc));
		dprintf("%s\n", ldap_err2string(rc));
		return(LA_FAIL);
	}

	if (login_getcapbool(lc, CAP_LDAP_NOREFERRALS, 0) != 0) {
		rc = ldap_set_option(ld, LDAP_OPT_REFERRALS, LDAP_OPT_OFF);
		if (rc != LDAP_OPT_SUCCESS) {
			syslog(LOG_ERR, "couldnt turn LDAP_OPT_REFERRALS off");
			dprintf("couldnt turn LDAP_OPT_REFERRALS off\n");
			return(LA_FAIL);
		} 
		noreferrals = 1;
	}

	/* set the version to v3 if we're using tls */
	if (login_getcapbool(lc, CAP_LDAP_TLS, 0) != 0) {

		version = LDAP_VERSION3;

		rc = ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version); 
		if (rc != LDAP_OPT_SUCCESS) {
			syslog(LOG_ERR, "couldnt set version %d", version);
			dprintf("couldnt set version %d\n", version);
			return(LA_FAIL);
		} 
	}
		 
	rc = do_simple_bind(lc, ld, NULL, NULL);  
	if (rc != LA_OK)
		return(rc);
	
	*ldp = ld;
	return(LA_OK);
}

/*
 * Convert format specifiers from the filter in login.conf to their
 * real values. return the new filter in the filter argument.
 */
static int 
parse_filter(login_cap_t *lc, char *username, char **filter, char *dn)
{
	char tmpbuf[BUFSIZ];
	char hostname[MAXHOSTNAMELEN];
	char *tmpfilt, *errmsg = "error";
	char *def = "(uid=%u)";
	char *p, *q;
	static int pass = 0;
	
	if (pass == 0) {
		pass++;
		tmpfilt = login_getcapstr(lc, CAP_LDAP_FILTER, def, errmsg);
		if (tmpfilt == errmsg) {
			syslog(LOG_ERR, "couldn't get value for " 
			    CAP_LDAP_FILTER);
			dprintf("couldn't get value for " CAP_LDAP_FILTER "\n");
			return(0);
		} 
	} else {
		pass++;
		tmpfilt = login_getcapstr(lc, CAP_LDAP_GROUPFILTER, NULL, 
		    errmsg);

		if (tmpfilt && strcmp(tmpfilt, errmsg) == 0) {
			syslog(LOG_ERR, "couldn't get value for " 
			    CAP_LDAP_GROUPFILTER);
			dprintf("couldn't get value for " 
			    CAP_LDAP_GROUPFILTER "\n");
			return(0);
		} 
		if (tmpfilt == NULL) {
			*filter = NULL;
			return(1);
		}
	}
	
	bzero(tmpbuf, sizeof(tmpbuf));

	p = tmpfilt;
	q = tmpbuf;

	/* 
	 * copy over from p to q, if we hit a %, substitute the real value,
	 * if we hit a NULL, its the end of the filter string 
	 */
	for (;;) {

		while(*p != '%' && *p != '\0' && 
		    ((q - tmpbuf) < sizeof(tmpbuf))) 
			*q++ = *p++;

		if ((q - tmpbuf) == sizeof(tmpbuf)) {
			syslog(LOG_ERR, "filter string too large, unable to "
			    "process: %s", p);
			dprintf("filter string too large, unable to "
			    "process: %s\n", p);
			return(0);
		}

		if (*p == '\0')
			break;
		p++;

		switch(*p) {
		
		case 'u': /* username */

			if (snprintf(tmpbuf, sizeof(tmpbuf), "%s%s", tmpbuf, 
			    username) >= sizeof(tmpbuf)) {
				syslog(LOG_ERR, "filter string too large, has " 
				    "been truncated: %s", tmpbuf); 
				dprintf("filter string too large, has " 
				    "been truncated: %s\n", tmpbuf); 
				return(0);
			}

			p++;
			q += strlen(username);
			break;

		case 'h': /* hostname */

			if (gethostname(hostname, sizeof(hostname)) == -1) {
				syslog(LOG_ERR, "couldn't get host name for "
				    "%%h %m");
				dprintf("couldn't get host name for "
				    "%%h %s\n", strerror(errno));
				return(0);
			}

			if (snprintf(tmpbuf, sizeof(tmpbuf), "%s%s", tmpbuf, 
			    hostname) >= sizeof(tmpbuf)) {
				syslog(LOG_ERR, "filter string too large, has "
				    "been truncated: %s", tmpbuf); 
				dprintf("filter string too large, has "
				    "been truncated: %s\n", tmpbuf); 
				return(0);
			}

			p++;
			q += strlen(hostname);
			break;

		case 'd': /* user dn */

			if (pass == 1) {
				syslog(LOG_ERR, "'%%d' cannot be used in " 
				    CAP_LDAP_FILTER);
				dprintf("'%%d' cannot be used in " 
				    CAP_LDAP_FILTER "\n");
				return(0);
			} else if (dn == NULL) {
				syslog(LOG_ERR, "no userdn has been recorded");
				dprintf("no userdn has been recorded\n");
				return(0);
			}

			/* good to go */
			if (snprintf(tmpbuf, sizeof(tmpbuf), "%s%s", tmpbuf, dn)
			    >= sizeof(tmpbuf)) {
				syslog(LOG_ERR, "filter string too large, has "
				    "been truncated: %s", tmpbuf); 
				dprintf("filter string too large, has "
				    "been truncated: %s\n", tmpbuf); 
				return(0);
			}

			p++;
			q += strlen(dn);
			break;

		default:

			syslog(LOG_ERR, "parse_filt: invalid filter specifier "
			    "'%c'", *p);
			dprintf("parse_filt: invalid filter specifier "
			    "'%c'\n", *p);
			return(0);
		}
		
	}

	p = strdup(tmpbuf);
	if (p == NULL) {
		syslog(LOG_ERR, "%m");
		dprintf("%s\n", strerror(errno));
		return(0);
	}

	*filter = p;

	return(1);
}
	
/*
 * Convert the relevant values from login.conf to a valid URI
 */
static int
build_uri(login_cap_t *lc, char *server, char **uri)
{
	char *p, ldapuri[BUFSIZ];
	quad_t port;

	/*
 	 * It's OK if we use the defaults for these as openldap
	 * allows the library to be configured with a default 
	 *  
	 * so, something like ldaps://:123/ is still ok  
	 */

	if (login_getcapbool(lc, CAP_LDAP_TLS, 0) != 0) 
		snprintf(ldapuri, sizeof(ldapuri), "ldaps://");
	else
		snprintf(ldapuri, sizeof(ldapuri), "ldap://");
		
	if (server && strlcat(ldapuri, server, sizeof(ldapuri)) >= 
	    sizeof(ldapuri)) {
		syslog(LOG_ERR, CAP_LDAP_SERVER " (%s) made ldap uri too long:"
		    " %s", server, ldapuri);
		dprintf(CAP_LDAP_SERVER " (%s) made ldap uri too long:"
		    " %s\n", server, ldapuri);
		return(LA_FAIL_FATAL);
	}

	port = login_getcapnum(lc, CAP_LDAP_PORT, (quad_t)LDAP_PORT, 
	    (quad_t)-1);
	if (port == (quad_t)-1) {
		syslog(LOG_ERR, "couldn't get value for " CAP_LDAP_PORT);
		dprintf("couldn't get value for " CAP_LDAP_PORT "\n");
		return(LA_FAIL_FATAL);
	}

	if (snprintf(ldapuri, sizeof(ldapuri), "%s:%d/", ldapuri, (int)port) 
	    >= sizeof(ldapuri)) {
		syslog(LOG_ERR, CAP_LDAP_PORT "(%d) made ldap uri too long: %s",
		    (int)port, ldapuri);
		dprintf(CAP_LDAP_PORT "(%d) made ldap uri too long: %s\n",
		    (int)port, ldapuri);
		return(LA_FAIL_FATAL);
	}
			
	p = strdup(ldapuri);
	if (p == NULL) {
		syslog(LOG_ERR, "%m");
		dprintf("%s\n", strerror(errno));
		return(LA_FAIL_FATAL);
	}

	*uri = p;

	return(LA_OK);
}

/* 
 * attempt to bind to the LDAP server using the given username
 * and password.
 */ 
static int
do_simple_bind(login_cap_t *lc, LDAP *ld, char *username, char *password)
{
	int msgid, rc, parserc;
	struct timeval timeo;
	LDAPMessage *res;
	char *binddn, *bindpw;
	char *errmsg = "error";

	if (username == NULL) {
		binddn = login_getcapstr(lc, CAP_LDAP_BINDDN, NULL, errmsg);
		if (binddn == errmsg) {
			syslog(LOG_ERR, "couldnt get string for " 
			    CAP_LDAP_BINDDN);
			dprintf("couldnt get string for " 
			    CAP_LDAP_BINDDN "\n");
			return(LA_FAIL_FATAL);
		}
	} else
		binddn = username;

	if (password == NULL) {
		bindpw = login_getcapstr(lc, CAP_LDAP_BINDPW, NULL, errmsg);
		if (bindpw == errmsg) {
			syslog(LOG_ERR, "couldnt get string for " 
			    CAP_LDAP_BINDPW);
			dprintf("couldnt get string for " 
			    CAP_LDAP_BINDPW "\n");
			return(LA_FAIL_FATAL);
		}
	} else
		bindpw = password;
 
	if (get_timeout(lc, &timeo) == 0) 
		return(LA_FAIL_FATAL);	

	msgid = ldap_simple_bind(ld, binddn, bindpw);
	if (msgid == -1) 
		return(LA_FAIL);
	
loop:
	rc = ldap_result(ld, msgid, 0, &timeo, &res);
	switch(rc) {
		
	case -1:
		ldap_unbind(ld);
		return(LA_FAIL);

	case 0:
		goto loop;

	default:
		parserc = ldap_parse_result(ld, res, &rc, NULL, NULL, NULL, 
		    NULL, 1);	
		if (parserc != LDAP_SUCCESS) 
			return(LA_FAIL);
		
		if (rc != LDAP_SUCCESS) 
			return(LA_FAIL);
	}

	return(LA_OK);
}

static void
free_data(login_cap_t *lc, LDAP *ld, LDAPMessage *res)
{
	if (lc != NULL) 
		login_close(lc);

	if (res != NULL) 
		ldap_msgfree(res);

	if (ld != NULL) 
		ldap_unbind(ld);
}		 

static int
get_timeout(login_cap_t *lc, struct timeval *timeo)
{
	int timeout;

	timeout = (int)login_getcapnum(lc, CAP_LDAP_TIMEOUT, (quad_t)DEFTIMEO, 
	    (quad_t)-1);
	if (timeout == -1) {
		syslog(LOG_ERR, "couldnt get " CAP_LDAP_TIMEOUT);
		dprintf("couldnt get " CAP_LDAP_TIMEOUT "\n");
		return(0);
	}
	timeo->tv_sec = timeout; 
	timeo->tv_usec = (timeout % 1000) * 1000;

	return(1);
}

/*
 * assume these scopes are all #define'd >= 0
 */
static int
getscope(login_cap_t *lc, char *cap)
{
	char *scope, *error = "error";

	scope = login_getcapstr(lc, cap, NULL, error);	
	if (scope == error) {
		syslog(LOG_ERR, "couldnt get %s", cap);
		dprintf("couldnt get %s\n", cap);
		return(-2);
	}

	if (scope == NULL || scope[0] == '\0') {
		/* preserve semantics of prev version */
		if (strcmp(cap, CAP_LDAP_USCOPE) == 0)
			return(LDAP_SCOPE_ONELEVEL);
		else
			return(LDAP_SCOPE_BASE);
	}

	if (strcmp(scope, "base") == 0)
		return(LDAP_SCOPE_BASE);
	else if (strcmp(scope, "onelevel") == 0)
		return(LDAP_SCOPE_ONELEVEL);
	else if (strcmp(scope, "subtree") == 0)
		return(LDAP_SCOPE_SUBTREE);

	syslog(LOG_ERR, "dont understand scope '%s'", scope);
	dprintf("dont understand scope '%s'\n", scope);
	/* if set, must be set correctly */
	return(-2);
} 
	
static void
handler(int signo)
{
	_exit(1);
}
