#include "common.h"

__dead void
usage(void)
{
	extern char *__progname;

	fprintf(stderr, "usage: %s [-f num]\n", __progname);
	exit(1);
}

struct
fp_print_data *enroll(struct fp_dev *dev, int finger)
{
	struct fp_print_data *enrolled_print = NULL;
	int r, i = 1;

	printf("\nIn order to complete the fingerprint enrollment, you have to\n"
	    "successfully swipe your fingers %d times\n\n",
	    fp_dev_get_nr_enroll_stages(dev));
 
	do {
		printf("Swipe your %s finger.\n", fingernames[finger]);

		r = fp_enroll_finger(dev, &enrolled_print);
		if (r < 0) {
			printf("Enroll failed: %d\n", r);
			return NULL;
		}

		switch (r) {
		case FP_ENROLL_COMPLETE:
			printf("Enroll stage #%d passed!\n", i);
			break;
		case FP_ENROLL_FAIL:
			printf("Enroll failed!\n");
			return NULL;
		case FP_ENROLL_PASS:
			printf("Enroll stage #%d passed!\n", i);
			break;
		case FP_ENROLL_RETRY:
			printf("Invalid swipe, please try again.\n");
			break;
		case FP_ENROLL_RETRY_TOO_SHORT:
			printf("Short swipe, please try again.\n");
			break;
		case FP_ENROLL_RETRY_CENTER_FINGER:
			printf("Positioning error, please center your finger on the "
				"sensor and try again.\n");
			break;
		case FP_ENROLL_RETRY_REMOVE_FINGER:
			printf("Swipe failed, please remove your finger and then try "
				"again.\n");
			break;
		}
		i++;
	} while (r != FP_ENROLL_COMPLETE);

	if (!enrolled_print) {
		fprintf(stderr, "Couldn't process fingerprint, please try again!\n");
		return NULL;
	}

	printf("Enrollment for %s finger completed!\n\n", fingernames[finger]);
	return enrolled_print;
}

int
main(int argc, char **argv)
{
	int r = 1, finger = -1, ch;
	struct fp_dscv_dev *ddev;
	struct fp_dscv_dev **discovered_devs;
	struct fp_dev *dev;
	struct fp_print_data *data;
	const char *errstr;

	while ((ch = getopt(argc, argv, "f:")) != -1) {
		switch (ch) {
		case 'f':
			finger = strtonum(optarg, FINGER_MIN, FINGER_MAX, &errstr);
			if (errstr)
				errx(1, "The finger's number is %s: %s", errstr, optarg);
			break;
		default:
			usage();
			/* NOTREACHED */
		}
	}
	argc -= optind;
	argv += optind;

	if (finger < 0)
		usage();

	r = fp_init();
	if (r < 0) {
		fprintf(stderr, "Failed to initialize!\n");
		exit(1);
	}

	discovered_devs = fp_discover_devs();
	if (!discovered_devs) {
		fprintf(stderr, "Could not discover devices\n");
		goto out;
	}

	ddev = discover_device(discovered_devs);
	if (!ddev) {
		fprintf(stderr, "No devices detected.\n");
		goto out;
	}

	dev = fp_dev_open(ddev);
	fp_dscv_devs_free(discovered_devs);
	if (!dev) {
		fprintf(stderr, "Could not open device.\n");
		goto out;
	}

	data = enroll(dev, finger);
	if (!data)
		goto out_close;

	r = fp_print_data_save(data, finger);
	if (r < 0)
		fprintf(stderr, "Data save failed, code %d\n", r);

	fp_print_data_free(data);
out_close:
	fp_dev_close(dev);
out:
	fp_exit();
	return r;
}
