#include "links.h"

void get_system_name()
{
	FILE *f;
	unsigned char *p;
	memset(system_name, 0, MAX_STR_LEN);
	if (!(f = popen("uname -srm", "r"))) goto fail;
	if (fread(system_name, 1, MAX_STR_LEN - 1, f) <= 0) {
		pclose(f);
		goto fail;
	}
	pclose(f);
	for (p = system_name; *p; p++) if (*p < ' ') {
		*p = 0;
		break;
	}
	if (system_name[0]) return;
	fail:
	strcpy(system_name, SYSTEM_NAME);
}

struct option {
	int p;
	unsigned char *(*rd_cmd)(struct option *, unsigned char ***, int *);
	unsigned char *(*rd_cfg)(struct option *, unsigned char *);
	void (*wr_cfg)(struct option *, unsigned char **, int *);
	int min, max;
	void *ptr;
	unsigned char *cfg_name;
	unsigned char *cmd_name;
};

struct option links_options[];
struct option html_options[];

struct option *all_options[] = { links_options, html_options, NULL, };

unsigned char *_parse_options(int argc, unsigned char *argv[], struct option **opt)
{
	unsigned char *e, *u = NULL;
	while (argc) {
		int i;
		argv++, argc--;
		if (argv[-1][0] == '-') {
			struct option *options;
			struct option **op;
			for (op = opt; (options = *op); op++) for (i = 0; options[i].p; i++)
				if (options[i].rd_cmd && options[i].cmd_name &&
				    !strcasecmp(options[i].cmd_name, &argv[-1][1])) {
					if ((e = options[i].rd_cmd(&options[i], &argv, &argc))) {
						if (e[0]) fprintf(stderr, "Error parsing option %s: %s\n", argv[-1], e);
						return NULL;
					}
					goto found;
				}
			uu:
			fprintf(stderr, "Unknown option %s\n", argv[-1]);
			return NULL;
		} else if (!u) u = argv[-1];
		else goto uu;
		found:;
	}
	if (u) return u;
	return "";
}

unsigned char *parse_options(int argc, unsigned char *argv[])
{
	return _parse_options(argc, argv, all_options);
}

void parse_config_file(unsigned char *name, unsigned char *file, struct option *options)
{
	int err = 0;
	int line = 0;
	unsigned char *e;
	int i;
	unsigned char *n, *p;
	int nl, pl;
	while (file[0]) {
		line++;
		while (file[0] && (file[0] == ' ' || file[0] == 9)) file++;
		n = file;
		while (file[0] && file[0] > ' ') file++;
		if (file == n) {
			if (file[0]) file++;
			continue;
		}
		nl = file - n;
		while (file[0] == 9 || file[0] == ' ') file++;
		p = file;
		while (file[0] && file[0] != 10 && file[0] != 13) file++;
		pl = file - p;
		if (file[0]) {
			if ((file[1] == 10 || file[1] == 13) && file[0] != file[1]) file++;
			file++;
		}
		if (n[0] != '#') for (i = 0; options[i].p; i++) if (options[i].cfg_name && nl == strlen(options[i].cfg_name) && !casecmp(n, options[i].cfg_name, nl)) {
			unsigned char *o = memacpy(p, pl);
			if ((e = options[i].rd_cfg(&options[i], o))) {
				if (e[0]) fprintf(stderr, "Error parsing config file %s, line %d: %s\n", name, line, e), err = 1;
			}
			mem_free(o);
			goto f;
		}
		fprintf(stderr, "Unknown option in config file %s, line %d\n", name, line);
		err = 1;
		f:;
	}
	if (err) fprintf(stderr, "\007"), sleep(3);
}

unsigned char *create_config_string(struct option *options)
{
	unsigned char *s = init_str();
	int l = 0;
	int i;
	for (i = 0; options[i].p; i++) if (options[i].wr_cfg)
		options[i].wr_cfg(&options[i], &s, &l);
	add_to_str(&s, &l, NEWLINE);
	return s;
}

#define FILE_BUF	1024

unsigned char cfg_buffer[FILE_BUF];

unsigned char *read_config_file(unsigned char *name)
{
	int h, r;
	int l = 0;
	unsigned char *s;
	if ((h = open(name, O_RDONLY | O_NOCTTY)) == -1) return NULL;
	set_bin(h);
	s = init_str();
	while ((r = read(h, cfg_buffer, FILE_BUF)) > 0) {
		int i;
		for (i = 0; i < r; i++) if (!cfg_buffer[i]) cfg_buffer[i] = ' ';
		add_bytes_to_str(&s, &l, cfg_buffer, r);
	}
	if (r == -1) mem_free(s), s = NULL;
	close(h);
	return s;
}

int write_to_config_file(unsigned char *name, unsigned char *c)
{
	int rr = strlen(c);
	int r = rr;
	int h, w;
	if ((h = open(name, O_WRONLY | O_NOCTTY | O_CREAT | O_TRUNC, 0666)) == -1) return -1;
	set_bin(h);
	while (r > 0) {
		if ((w = write(h, c + rr - r, r)) <= 0) {
			close(h);
			return -1;
		}
		r -= w;
	}
	close(h);
	return 0;
}

unsigned char *get_home(int *n)
{
	struct stat st;
	unsigned char *home = stracpy(getenv("HOME"));
	unsigned char *links_home;
	unsigned char *config_dir = stracpy(getenv("CONFIG_DIR"));

	if (n) *n = 1;
	if (!home) {
		int i;
		home = stracpy(path_to_exe);
		if (!home) {
			if (config_dir) mem_free(config_dir);
			return NULL;
		}
		for (i = strlen(home) - 1; i >= 0; i--) if (dir_sep(home[i])) {
			home[i + 1] = 0;
			break;
		}
	}
	while (home[0] && dir_sep(home[strlen(home) - 1])) home[strlen(home) - 1] = 0;
	add_to_strn(&home, "/");
	links_home = stracpy(home);
	if (config_dir) 	
	{
		add_to_strn(&links_home,config_dir);
		while (links_home[0] && dir_sep(links_home[strlen(links_home) - 1])) links_home[strlen(links_home) - 1] = 0;
		if (stat(links_home,&st) != -1 && S_ISDIR(st.st_mode)) {
			add_to_strn(&links_home, "/links");
	    	} else {
			fprintf(stderr, "CONFIG_DIR set to %s. But directory %s doesn't egzist.", config_dir, links_home);
			mem_free(links_home);
			links_home = stracpy(home);
			add_to_strn(&links_home, ".links");		
		}
		mem_free(config_dir);
	} else add_to_strn(&links_home, ".links");
	if (stat(links_home, &st)) {
		if (!mkdir(links_home, 0777)) goto home_creat;
		goto first_failed;
	}
	if (S_ISDIR(st.st_mode)) goto home_ok;
	first_failed:
	mem_free(links_home);
	links_home = stracpy(home);
	add_to_strn(&links_home, "links");
	if (stat(links_home, &st)) {
		if (!mkdir(links_home, 0777)) goto home_creat;
		goto failed;
	}
	if (S_ISDIR(st.st_mode)) goto home_ok;
	failed:
	mem_free(links_home);
	mem_free(home);
	return NULL;
	home_ok:
	if (n) *n = 0;
	home_creat:
#ifdef HAVE_CHMOD
	chmod(links_home, 0700);
#endif
	add_to_strn(&links_home, "/");
	mem_free(home);
	return links_home;
}

void init_home()
{
	get_system_name();
	links_home = get_home(&first_use);
	if (!links_home) {
		fprintf(stderr, "%s\n\007", "Unable to find or create links config directory. Please check, that you have $HOME variable set correctly and that you have write permission to your home directory");
		sleep(3);
		return;
	}
}

void load_config_file(unsigned char *name, struct option *o)
{
	unsigned char *c, *config_file;
	config_file = stracpy(links_home);
	if (!config_file) return;
	add_to_strn(&config_file, name);
	if ((c = read_config_file(config_file))) goto ok;
	mem_free(config_file);
	config_file = stracpy(links_home);
	if (!config_file) return;
	add_to_strn(&config_file, ".");
	add_to_strn(&config_file, name);
	if ((c = read_config_file(config_file))) goto ok;
	mem_free(config_file);
	/*fprintf(stderr, "%s\n\007", "Unable to find or create config file in your links directory. Please check, that you have write permission to links home directory");
	sleep(3);*/
	return;
	ok:
	parse_config_file(config_file, c, o);
	mem_free(c);
	mem_free(config_file);
}

void load_config()
{
	load_config_file("links.cfg", links_options);
	load_config_file("html.cfg", html_options);
}

int write_config_file(unsigned char *name, struct option *o, struct terminal *term)
{
	unsigned char *c, *config_file;
	if (!(c = create_config_string(o))) return -1;
	config_file = stracpy(links_home);
	if (!config_file) {
		mem_free(c);
		return -1;
	}
	add_to_strn(&config_file, name);
	if (write_to_config_file(config_file, c)) {
		if (term) msg_box(term, NULL, TEXT(T_CONFIG_ERROR), AL_CENTER, TEXT(T_UNABLE_TO_WRITE_TO_CONFIG_FILE), NULL, 1, TEXT(T_CANCEL), NULL, B_ENTER | B_ESC);
		mem_free(c);
		mem_free(config_file);
		return -1;
	}
	mem_free(c);
	mem_free(config_file);
	return 0;
}

void write_config(struct terminal *term)
{
	write_config_file("links.cfg", links_options, term);
}

void write_html_config(struct terminal *term)
{
	write_config_file("html.cfg", html_options, term);
}

void add_nm(struct option *o, unsigned char **s, int *l)
{
	if (*l) add_to_str(s, l, NEWLINE);
	add_to_str(s, l, o->cfg_name);
	add_to_str(s, l, " ");
}

unsigned char *num_rd(struct option *o, unsigned char *c)
{
	unsigned char *end;
	long l = strtolx(c, &end);
	if (*end) return "Number expected";
	if (l < o->min || l > o->max) return "Out of range";
	*(int *)o->ptr = l;
	return NULL;
}

void num_wr(struct option *o, unsigned char **s, int *l)
{
	add_nm(o, s, l);
	add_knum_to_str(s, l, *(int *)o->ptr);
}

unsigned char *str_rd(struct option *o, unsigned char *c)
{
	if (strlen(c) + 1 > o->max) return "String too long";
	strcpy(o->ptr, c);
	return NULL;
}

void str_wr(struct option *o, unsigned char **s, int *l)
{
	add_nm(o, s, l);
	if (strlen(o->ptr) > o->max - 1) add_bytes_to_str(s, l, o->ptr, o->max - 1);
	else add_to_str(s, l, o->ptr);
}

unsigned char *cp_rd(struct option *o, unsigned char *c)
{
	int i;
	/*if (!strcasecmp(c, "none")) i = -1;
	else */if ((i = get_cp_index(c)) == -1) return "Unknown codepage";
	*(int *)o->ptr = i;
	return NULL;
}

void cp_wr(struct option *o, unsigned char **s, int *l)
{
	unsigned char *n = get_cp_mime_name(*(int *)o->ptr);
	add_nm(o, s, l);
	add_to_str(s, l, n);
}

unsigned char *lang_rd(struct option *o, unsigned char *c)
{
	int i;
	for (i = 0; i < n_languages(); i++)
		if (!(strcasecmp(language_name(i), c))) {
			set_language(i);
			return NULL;
		}
	return "Unknown language";
}

void lang_wr(struct option *o, unsigned char **s, int *l)
{
	add_nm(o, s, l);
	add_to_str(s, l, language_name(current_language));
}

unsigned char *get_word(unsigned char **s)
{
	unsigned char *w = *s;
	unsigned char *ww;
	if (!**s) return NULL;
	while (**s > ' ') (*s)++;
	ww = memacpy(w, *s - w);
	while (**s && **s <= ' ') (*s)++;
	return ww;
}

unsigned char *get_quoted(unsigned char **s)
{
	unsigned char *st;
	int l;
	if (!**s || **s != '"') return NULL;
	(*s)++;
	st = init_str();
	l = 0;
	while (**s && **s != '"') quak:add_chr_to_str(&st, &l, *((*s)++));
	if (!**s) {
		mem_free(st);
		return NULL;
	}
	(*s)++;
	if (**s == '"') goto quak;
	while (**s && **s <= ' ') (*s)++;
	return st;
}

void add_quoted_to_str(unsigned char **s, int *l, unsigned char *q)
{
	add_chr_to_str(s, l, '"');
	while (*q) {
		if (*q == '"') add_to_str(s, l, "\"\"");
		else add_chr_to_str(s, l, *q);
		q++;
	}
	add_chr_to_str(s, l, '"');
}

int getnum(unsigned char *s, int *n, int r1, int r2)
{
	unsigned char *e;
	long l = strtol(s, (char **)&e, 10);
	if (*e || !*s) return -1;
	if (l < r1 || l >= r2) return -1;
	*n = (int)l;
	return 0;
}

unsigned char *type_rd(struct option *o, unsigned char *c)
{
	unsigned char *err = "Error reading association specification";
	struct assoc new;
	unsigned char *w;
	int n;
	memset(&new, 0, sizeof(struct assoc));
	if (!(new.label = get_quoted(&c))) goto err;
	if (!(new.ct = get_quoted(&c))) goto err;
	if (!(new.prog = get_quoted(&c))) goto err;
	if (!(w = get_word(&c))) goto err;
	if (getnum(w, &n, 0, 32)) goto err_f;
	mem_free(w);
	new.cons = !!(n & 1);
	new.xwin = !!(n & 2);
	new.ask = !!(n & 4);
	if ((n & 8) || (n & 16)) new.block = !!(n & 16);
	else new.block = !new.xwin || new.cons;
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '9') goto err_f;
	new.system = w[0] - '0';
	mem_free(w);
	update_assoc(&new);
	err = NULL;
	err:
	if (new.label) mem_free(new.label);
	if (new.ct) mem_free(new.ct);
	if (new.prog) mem_free(new.prog);
	return err;
	err_f:
	mem_free(w);
	goto err;
}


void type_wr(struct option *o, unsigned char **s, int *l)
{
	struct assoc *a;
	foreachback(a, assoc) {
		add_nm(o, s, l);
		add_quoted_to_str(s, l, a->label);
		add_to_str(s, l, " ");
		add_quoted_to_str(s, l, a->ct);
		add_to_str(s, l, " ");
		add_quoted_to_str(s, l, a->prog);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, (!!a->cons) + (!!a->xwin) * 2 + (!!a->ask) * 4 + (!a->block) * 8 + (!!a->block) * 16);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, a->system);
	}
}

unsigned char *ext_rd(struct option *o, unsigned char *c)
{
	unsigned char *err = "Error reading extension specification";
	struct extension new;
	memset(&new, 0, sizeof(struct extension));
	if (!(new.ext = get_quoted(&c))) goto err;
	if (!(new.ct = get_quoted(&c))) goto err;
	update_ext(&new);
	err = NULL;
	err:
	if (new.ext) mem_free(new.ext);
	if (new.ct) mem_free(new.ct);
	return err;
}

void ext_wr(struct option *o, unsigned char **s, int *l)
{
	struct extension *a;
	foreachback(a, extensions) {
		add_nm(o, s, l);
		add_quoted_to_str(s, l, a->ext);
		add_to_str(s, l, " ");
		add_quoted_to_str(s, l, a->ct);
	}
}

unsigned char *prog_rd(struct option *o, unsigned char *c)
{
	unsigned char *err = "Error reading program specification";
	unsigned char *prog, *w;
	if (!(prog = get_quoted(&c))) goto err_1;
	if (!(w = get_word(&c))) goto err_2;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '9') goto err_3;
	update_prog(o->ptr, prog, w[0] - '0');
	err = NULL;
	err_3:
	mem_free(w);
	err_2:
	mem_free(prog);
	err_1:
	return err;
}

void prog_wr(struct option *o, unsigned char **s, int *l)
{
	struct protocol_program *a;
	foreachback(a, *(struct list_head *)o->ptr) {
		add_nm(o, s, l);
		add_quoted_to_str(s, l, a->prog);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, a->system);
	}
}

unsigned char *term_rd(struct option *o, unsigned char *c)
{
	struct term_spec *ts;
	unsigned char *w;
	int i;
	if (!(w = get_quoted(&c))) goto err;
	if (!(ts = new_term_spec(w))) {
		mem_free(w);
		goto end;
	}
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '2') goto err_f;
	ts->mode = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '1') goto err_f;
	ts->m11_hack = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '7') goto err_f;
	ts->col = (w[0] - '0') & 1;
	ts->restrict_852 = !!((w[0] - '0') & 2);
	ts->block_cursor = !!((w[0] - '0') & 4);
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if ((i = get_cp_index(w)) == -1) goto err_f;
	ts->charset = i;
	mem_free(w);
	end:
	return NULL;
	err_f:
	mem_free(w);
	err:
	return "Error reading terminal specification";
}

unsigned char *term2_rd(struct option *o, unsigned char *c)
{
	struct term_spec *ts;
	unsigned char *w;
	int i;
	if (!(w = get_quoted(&c))) goto err;
	if (!(ts = new_term_spec(w))) {
		mem_free(w);
		goto end;
	}
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '3') goto err_f;
	ts->mode = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '1') goto err_f;
	ts->m11_hack = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '1') goto err_f;
	ts->restrict_852 = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if (strlen(w) != 1 || w[0] < '0' || w[0] > '1') goto err_f;
	ts->col = w[0] - '0';
	mem_free(w);
	if (!(w = get_word(&c))) goto err;
	if ((i = get_cp_index(w)) == -1) goto err_f;
	ts->charset = i;
	mem_free(w);
	end:
	return NULL;
	err_f:
	mem_free(w);
	err:
	return "Error reading terminal specification";
}

void term_wr(struct option *o, unsigned char **s, int *l)
{
	struct term_spec *ts;
	foreachback(ts, term_specs) {
		add_nm(o, s, l);
		add_quoted_to_str(s, l, ts->term);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, ts->mode);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, ts->m11_hack);
		add_to_str(s, l, " ");
		add_num_to_str(s, l, !!ts->col + !!ts->restrict_852 * 2 + !!ts->block_cursor * 4);
		add_to_str(s, l, " ");
		add_to_str(s, l, get_cp_mime_name(ts->charset));
	}
}

unsigned char *gen_cmd(struct option *o, unsigned char ***argv, int *argc)
{
	unsigned char *r;
	if (!*argc) return "Parameter expected";
	(*argv)++; (*argc)--;
	if (!(r = o->rd_cfg(o, *(*argv - 1)))) return NULL;
	(*argv)--; (*argc)++;
	return r;
}

unsigned char *lookup_cmd(struct option *o, unsigned char ***argv, int *argc)
{
	ip addr;
	unsigned char *p = (unsigned char *)&addr;
	if (!*argc) return "Parameter expected";
	if (*argc >= 2) return "Too many parameters";
	(*argv)++; (*argc)--;
	if (do_real_lookup(*(*argv - 1), &addr)) {
#ifdef HAVE_HERROR
		herror("error");
#else
		fprintf(stderr, "error: host not found\n");
#endif
		return "";
	}
	printf("%d.%d.%d.%d\n", (int)p[0], (int)p[1], (int)p[2], (int)p[3]);
	fflush(stdout);
	return "";
}

unsigned char *version_cmd(struct option *o, unsigned char ***argv, int *argc)
{
	printf("Links " VERSION_STRING "\n");
	fflush(stdout);
	return "";
}

unsigned char *no_connect_cmd(struct option *o, unsigned char ***argv, int *argc)
{
	no_connect = 1;
	return NULL;
}

unsigned char *anonymous_cmd(struct option *o, unsigned char ***argv, int *argc)
{
	anonymous = 1;
	return NULL;
}
unsigned char *dump_cmd(struct option *o, unsigned char ***argv, int *argc)
{
	if (dmp != o->min && dmp) return "Can't use both -dump and -source";
	dmp = o->min;
	no_connect = 1;
	return NULL;
}

unsigned char *printhelp_cmd(struct option *o, unsigned char ***argv, int *argc)
{
/* Changed and splited - translation is much easier.
 * Print to stdout instead stderr (,,links -help | more''
 * is much better than ,,links -help 2>&1 | more'').
 */
fprintf(stdout, "%s%s%s%s%s%s\n", 

("links [options] URL\n\
Options are:\n\
\n\
 -async-dns <0>/<1>\n\
  Asynchronous DNS resolver on(1)/off(0). \n\
\n\
 -max-connections <max>\n\
  Maximum number of concurrent connections.\n\
  (default: 10)\n\
\n"),
(" -max-connections-to-host <max>\n\
  Maximum number of concurrent connection to a given host.\n\
  (default: 2)\n\
\n\
 -retries <retry>\n\
  Number of retries.\n\
  (default: 3)\n\
\n\
 -receive-timeout <sec>\n\
  Timeout on receive.\n\
  (default: 120)\n\
\n"),
(" -unrestartable-receive-timeout <sec>\n\
  Timeout on non restartable connections.\n\
  (default: 600)\n\
\n\
 -format-cache-size <num>\n\
  Number of formatted document pages cached.\n\
  (default: 5)\n\
\n\
 -memory-cache-size <Kbytes>\n\
  Cache memory in Kilobytes.\n\
  (default: 1024)\n\
\n"),
(" -http-proxy <host:port>\n\
  Host and port number of the HTTP proxy, or blank.\n\
  (default: blank)\n\
\n\
 -ftp-proxy <host:port>\n\
  Host and port number of the FTP proxy, or blank.\n\
  (default: blank)\n\
\n\
 -download-dir <path>\n\
  Default download directory.\n\
  (default: actual dir)\n\
\n"),
(" -assume-codepage <codepage>\n\
  Use the given codepage when the webpage did not specify\n\
  its codepage. (default: ISO 8859-1)\n\
\n\
 -anonymous\n\
  Restrict links so that it can run on an anonymous account.\n\
  No local file browsing. No downloads. Executing of viewers\n\
  is allowed, but user can't add or modify entries in\n\
  association table.\n\
\n\
 -no-connect\n\
  Runs links as a separate instance - instead of connecting to\n\
  existing instance.\n\
\n\
 -version\n\
  Prints the links version number and exit.\n\
\n\
 -help\n\
  Prints this help screen\n\
\n\
\n"),
("Keys:\n\
 	ESC	 display menu\n\
	^C	 quit\n\
	^P, ^N	 scroll up, down\n\
	[, ]	 scroll left, right\n\
	up, down select link\n\
	->	 follow link\n\
	<-	 go back\n\
	g	 go to url\n\
	G	 go to url based on current url\n\
	/	 search\n\
	?	 search back\n\
	n	 find next\n\
	N	 find previous\n\
	=	 document info\n\
	\\	 document source\n\
	d	 download\n\
	q	 quit\n"));

	fflush(stdout);
	return "";
}

void end_config()
{
	if (links_home) mem_free(links_home);
}

int anonymous = 0;

unsigned char system_name[MAX_STR_LEN];

unsigned char *links_home = NULL;
int first_use = 0;
int created_home = 0;

int no_connect = 0;
int base_session = 0;
int dmp = 0;

int async_lookup = 1;
int download_utime = 0;
int max_connections = 10;
int max_connections_to_host = 2;
int max_tries = 3;
int receive_timeout = 120;
int unrestartable_receive_timeout = 600;

int max_format_cache_entries = 5;
long memory_cache_size = 1048576;

int enable_html_tables = 1;
int enable_html_frames = 1;
int display_images = 1;

struct document_setup dds = { 0, 0, 1, 1, 0, 3, 0, 0 };

struct rgb default_fg = { 191, 191, 191 };
struct rgb default_bg = { 0, 0, 0 };
struct rgb default_link = { 255, 255, 255 };
struct rgb default_vlink = { 255, 255, 0 };

int default_left_margin = HTML_LEFT_MARGIN;

unsigned char http_proxy[MAX_STR_LEN] = "";
unsigned char ftp_proxy[MAX_STR_LEN] = "";

unsigned char download_dir[MAX_STR_LEN] = "";

unsigned char default_anon_pass[MAX_STR_LEN] = "somebody@host.domain";

/* These are workarounds for some CGI script bugs */
struct http_bugs http_bugs = { 0, 1, 0, 0 };
/*int bug_302_redirect = 0;*/
	/* When got 301 or 302 from POST request, change it to GET
	   - this violates RFC2068, but some buggy message board scripts rely on it */
/*int bug_post_no_keepalive = 0;*/
	/* No keepalive connection after POST request. Some buggy PHP databases report bad
	   results if GET wants to retreive data POSTed in the same connection */

struct option links_options[] = {
	1, printhelp_cmd, NULL, NULL, 0, 0, NULL, NULL, "?",
	1, printhelp_cmd, NULL, NULL, 0, 0, NULL, NULL, "h",
	1, printhelp_cmd, NULL, NULL, 0, 0, NULL, NULL, "help",
	1, printhelp_cmd, NULL, NULL, 0, 0, NULL, NULL, "-help",
	1, lookup_cmd, NULL, NULL, 0, 0, NULL, NULL, "lookup",
	1, version_cmd, NULL, NULL, 0, 0, NULL, NULL, "version",
	1, no_connect_cmd, NULL, NULL, 0, 0, NULL, NULL, "no-connect",
	1, anonymous_cmd, NULL, NULL, 0, 0, NULL, NULL, "anonymous",
	1, gen_cmd, num_rd, NULL, 0, MAXINT, &base_session, NULL, "base-session",
	1, dump_cmd, NULL, NULL, D_DUMP, 0, NULL, NULL, "dump",
	1, dump_cmd, NULL, NULL, D_SOURCE, 0, NULL, NULL, "source",
	1, gen_cmd, num_rd, num_wr, 0, 1, &async_lookup, "async_dns", "async-dns",
	1, gen_cmd, num_rd, num_wr, 0, 1, &download_utime, "download_utime", "download-utime",
	1, gen_cmd, num_rd, num_wr, 1, 16, &max_connections, "max_connections", "max-connections",
	1, gen_cmd, num_rd, num_wr, 1, 8, &max_connections_to_host, "max_connections_to_host", "max-connections-to-host",
	1, gen_cmd, num_rd, num_wr, 1, 16, &max_tries, "retries", "retries",
	1, gen_cmd, num_rd, num_wr, 1, 1800, &receive_timeout, "receive_timeout", "receive-timeout",
	1, gen_cmd, num_rd, num_wr, 1, 1800, &unrestartable_receive_timeout, "unrestartable_receive_timeout", "unrestartable-receive-timeout",
	1, gen_cmd, num_rd, num_wr, 0, 256, &max_format_cache_entries, "format_cache_size", "format-cache-size",
	1, gen_cmd, num_rd, num_wr, 0, MAXINT, &memory_cache_size, "memory_cache_size", "memory-cache-size",
	1, gen_cmd, str_rd, str_wr, 0, MAX_STR_LEN, http_proxy, "http_proxy", "http-proxy",
	1, gen_cmd, str_rd, str_wr, 0, MAX_STR_LEN, ftp_proxy, "ftp_proxy", "ftp-proxy",
	1, gen_cmd, str_rd, str_wr, 0, MAX_STR_LEN, download_dir, "download_dir", "download-dir",
	1, gen_cmd, lang_rd, lang_wr, 0, 0, &current_language, "language", "language",
	1, gen_cmd, num_rd, num_wr, 0, 1, &http_bugs.http10, "http_bugs.http10", "http-bugs.http10",
	1, gen_cmd, num_rd, num_wr, 0, 1, &http_bugs.allow_blacklist, "http_bugs.allow_blacklist", "http-bugs.allow-blacklist",
	1, gen_cmd, num_rd, num_wr, 0, 1, &http_bugs.bug_302_redirect, "http_bugs.bug_302_redirect", "http-bugs.bug-302-redirect",
	1, gen_cmd, num_rd, num_wr, 0, 1, &http_bugs.bug_post_no_keepalive, "http_bugs.bug_post_no_keepalive", "http-bugs.bug_post-no-keepalive",
	1, gen_cmd, str_rd, str_wr, 0, MAX_STR_LEN, default_anon_pass, "ftp.anonymous_password", "ftp.anonymous-password",
	1, NULL, cp_rd, NULL, 0, 0, &dds.assume_cp, "assume_codepage", "assume-codepage",
	1, NULL, term_rd, term_wr, 0, 0, NULL, "terminal", NULL,
	1, NULL, term2_rd, NULL, 0, 0, NULL, "terminal2", NULL,
	1, NULL, type_rd, type_wr, 0, 0, NULL, "association", NULL,
	1, NULL, ext_rd, ext_wr, 0, 0, NULL, "extension", NULL,
	1, NULL, prog_rd, prog_wr, 0, 0, &mailto_prog, "mailto", NULL,
	1, NULL, prog_rd, prog_wr, 0, 0, &telnet_prog, "telnet", NULL,
	1, NULL, prog_rd, prog_wr, 0, 0, &tn3270_prog, "tn3270", NULL,
	0, NULL, NULL, NULL, 0, 0, NULL, NULL, NULL,
};

struct option html_options[] = {
	1, gen_cmd, cp_rd, cp_wr, 0, 0, &dds.assume_cp, "html_assume_codepage", "html-assume-codepage",
	1, gen_cmd, num_rd, num_wr, 0, 1, &dds.tables, "html_tables", "html-tables",
	1, gen_cmd, num_rd, num_wr, 0, 1, &dds.frames, "html_frames", "html-frames",
	1, gen_cmd, num_rd, num_wr, 0, 1, &dds.images, "html_images", "html-images",
	1, gen_cmd, num_rd, num_wr, 0, 1, &dds.num_links, "html_numbered_links", "html-numbered-links",
	1, gen_cmd, num_rd, num_wr, 0, 1, &dds.table_order, "html_table_order", "html-table-order",
	1, gen_cmd, num_rd, num_wr, 0, 9, &dds.margin, "html_margin", "html-margin",
	0, NULL, NULL, NULL, 0, 0, NULL, NULL, NULL,
};

extern struct history goto_url_history ;

int load_url_history()
{
	unsigned char *history_file ;
	FILE *fp ;
	char	url[MAX_INPUT_URL_LEN];

	if (anonymous) return 0;
	/* Must have been called after init_home */
	if (!links_home)
		return 0;
	history_file = stracpy (links_home) ;
	if (!history_file)
		return 0;
	add_to_strn(&history_file, "links.his");
	if (!(fp = fopen(history_file, "r")))
	{
		mem_free (history_file) ;
		return 0;
	}
	while (fgets (url, MAX_INPUT_URL_LEN, fp))
	{
		url[strlen(url)-1] = 0 ;
		add_to_history(&goto_url_history, url) ;
	}
	fclose (fp) ;
	mem_free (history_file) ;
	return 0 ;
}

int save_url_history()
{
	struct history_item* hi ;
	unsigned char *history_file ;
	int  i = 0 ;
	FILE *fp ;
	if (anonymous) return 0;

	/* Must have been called after init_home */
	if (!links_home)
		return 0;
	history_file = stracpy (links_home) ;
	if (!history_file)
		return 0;
	add_to_strn(&history_file, "links.his");
	if (!(fp = fopen(history_file, "w"))) {
		mem_free(history_file);
		return 0;
	}
	foreachback(hi,goto_url_history.items)
	{
		if (i++ > MAX_HISTORY_ITEMS)
			break ;
		else
			fputs (hi->d, fp), fputc ('\n', fp) ;
	}
	fclose (fp) ;
	mem_free (history_file) ;
	return 0 ;
}

