// Surface.cc, Video class for SDL
// Copyright (c) 2003 Jesper Strandberg (skywarper@hotbrev.com)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "Surface.hh"

#include <string>
#include <SDL.h>
#include "Video.hh"
#include "Error.hh"

#ifdef SDL_IMAGE
#undef SDL_IMAGE
#include <SDL_image.h>
#define SDL_IMAGE
#endif // SDL_IMAGE

Surface::Surface(const std::string &filename) {
    mSurface = 0;
    load(filename);
}

Surface::Surface(int width, int height, Uint32 flags) {
    mSurface = SDL_CreateRGBSurface(flags, width, height, 32, 0, 0, 0, 0);
    if(!mSurface)
        throw Error(SDL_GetError());

    setDisplayFormat();
}

Surface::Surface(int width, int height, int bpp, Uint32 flags) {
    Uint32 rmask = 0;
    Uint32 gmask = 0;
    Uint32 bmask = 0;
    Uint32 amask = 0;

    switch(bpp) {
        case 8:
            // No rgbmask
            break;

        case 16:
#           if SDL_BYTEORDER == SDL_BIG_ENDIAN
                rmask = 0xf000;
                gmask = 0x0f00;
                bmask = 0x00f0;
                amask = 0x000f;
#           else
                rmask = 0x000f;
                gmask = 0x00f0;
                bmask = 0x0f00;
                amask = 0xf000;
#           endif
            break;

        case 24:
#           if SDL_BYTEORDER == SDL_BIGENDIAN
                rmask = 0xff0000;
                gmask = 0x00ff00;
                bmask = 0x0000ff;
                amask = 0x000000;
#           else
                rmask = 0x0000ff;
                gmask = 0x00ff00;
                bmask = 0xff0000;
                amask = 0x000000;
#           endif
            break;

        case 32:
#           if SDL_BYTEORDER == SDL_BIGENDIAN
                rmask = 0xff000000;
                gmask = 0x00ff0000;
                bmask = 0x0000ff00;
                amask = 0x000000ff;
#           else
                rmask = 0x000000ff;
                gmask = 0x0000ff00;
                bmask = 0x00ff0000;
                amask = 0xff000000;
#           endif
            break;

        default:
            throw Error("Unknown bitdepth.");
    }

    mSurface = SDL_CreateRGBSurface(flags, width, height, bpp, rmask, gmask, bmask, amask);
    if(!mSurface)
        throw Error(SDL_GetError());
}

Surface::~Surface() {
    if(mSurface)
        SDL_FreeSurface(mSurface);
}

void Surface::load(const std::string &filename) {
    if(mSurface)
        SDL_FreeSurface(mSurface);

#   ifdef SDL_IMAGE
        mSurface = IMG_Load(filename.c_str());
        if(!mSurface)
            throw Error(IMG_GetError());
#   else
        mSurface = SDL_LoadBMP(filename.c_str());
        if(!mSurface)
            throw Error(SDL_GetError());
#   endif //SDL_IMAGE
}

void Surface::blit(SDL_Rect *srcRect, SDL_Rect *dstRect, Surface *dstSurface) {
    if(SDL_BlitSurface(mSurface, srcRect, dstSurface->getSDLSurface(), dstRect) < 0)
        throw Error(SDL_GetError());
}

void Surface::blit(int x, int y, Surface *dstSurface) {
    SDL_Rect dstRect;
    dstRect.x = x;
    dstRect.y = y;

    blit(0, &dstRect, dstSurface);
}

void Surface::blit(int x, int y) {
    SDL_Rect dstRect;
    dstRect.x = x;
    dstRect.y = y;

    blit(0, &dstRect, Video::getScreen());
}

void Surface::setDisplayFormat() {
    if(!mSurface)
        throw Error("No surface present.");

    SDL_Surface *temp = SDL_DisplayFormat(mSurface);
    if(!temp)
        throw Error(SDL_GetError());

    SDL_FreeSurface(mSurface);
    mSurface = temp;
}

void Surface::setDisplayFormatAlpha() {
    if(!mSurface)
        throw Error("No surface present.");

    SDL_Surface *temp = SDL_DisplayFormatAlpha(mSurface);
    if(!temp)
        throw Error(SDL_GetError());

    SDL_FreeSurface(mSurface);
    mSurface = temp;
}

void Surface::setAlpha(Uint8 alpha, Uint32 flags) {
    if(SDL_SetAlpha(mSurface, flags, alpha) < 0)
        throw Error(SDL_GetError());
}

void Surface::setColorKey(Uint32 key, Uint32 flags) {
    if(SDL_SetColorKey(mSurface, flags, key) < 0)
        throw Error(SDL_GetError());
}

void Surface::fill(Uint32 color, SDL_Rect *rect) {
    if(SDL_FillRect(mSurface, rect, color) < 0)
        throw Error(SDL_GetError());
}

