/* Socket routine descriptions */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <netdb.h>
#include <fcntl.h>

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#include <unistd.h>
#include <iostream.h>

#include "socket.h"
#include "icq-defines.h"
#include "log.h"
#include "support.h"

#ifdef USE_SOCKS5
#define SOCKS
#include <socks.h>
#endif

extern int errno;
extern int h_errno;

//=====Constants===============================================================
const unsigned short ADDR_LOCAL = 0x01;
const unsigned short ADDR_REMOTE = 0x02;
const unsigned long MAX_RECV_SIZE = 4096;


//=====Socket==================================================================

//-----Socket::OpenSocket------------------------------------------------------
void Socket::OpenSocket(void)
{
   if (m_nDescriptor >= 0) 
   {
      // Leave the socket blocking for now until we figure out how to handle
      // non-blocking send calls
      //fcntl(descriptorVal, F_SETFL, O_NONBLOCK);
      if (sn == NULL) sn = new QSocketNotifier(m_nDescriptor, QSocketNotifier::Read);
      sn->setEnabled(true);
   }
}


//-----Socket::CloseSocket-----------------------------------------------------
void Socket::CloseSocket(void)
{
  if (sn != NULL)
  {
    delete sn;
    sn = NULL;
  }
}


//-----StdinSocket-------------------------------------------------------------
StdinSocket::StdinSocket(void)
{ 
  m_nDescriptor = STDIN_FILENO; 
  OpenSocket();
}


//=====INetSocket===============================================================

//-----INetSocket::constructor--------------------------------------------------
INetSocket::INetSocket(void)
{
  m_nDescriptor = -1;
  
  memset(&m_sRemoteAddr, 0, sizeof(struct sockaddr_in));
  memset(&m_sLocalAddr, 0, sizeof(struct sockaddr_in));
}


//-----INetSocket::dumpPacket---------------------------------------------------
void INetSocket::DumpPacket(CBuffer &b, direction d)
{
   switch(d)
   {
   case D_SENDER:
     gLog.Packet("%sPacket (%s, %d bytes) sent (%s:%d -> %s:%d):\n%s\n",
                  L_PACKETxSTR, m_szID, b.getDataSize(), LocalIpStr(), 
                  LocalPort(), RemoteIpStr(), RemotePort(), b.print());
      break;
   case D_RECEIVER:
      gLog.Packet("%sPacket (%s, %d bytes) received (%s:%d <- %s:%d):\n%s\n",
                  L_PACKETxSTR, m_szID, b.getDataSize(), LocalIpStr(), 
                  LocalPort(), RemoteIpStr(), RemotePort(), b.print());
      break;
   }
}


/*-----INetSocket::setDestination----------------------------------------------
 * Takes an ip in network order and a port in host byte order and sets the
 * remote ip and port to those values
 *---------------------------------------------------------------------------*/
bool INetSocket::SetRemoteAddr(unsigned long _nRemoteIp, unsigned short _nRemotePort)
{ 
  if (_nRemoteIp == 0) return(false);  // if the rIp is invalid, exit

  m_sRemoteAddr.sin_port = htons(_nRemotePort);
  m_sRemoteAddr.sin_addr.s_addr = _nRemoteIp;
  return(true);
}


//-----INetSocket::ResetSocket-------------------------------------------------
void INetSocket::ResetSocket(void)
{
  CloseConnection();
  
  memset(&m_sRemoteAddr, 0, sizeof(struct sockaddr_in));
  memset(&m_sLocalAddr, 0, sizeof(struct sockaddr_in));
}


/*-----INetSocket::SetAddrsFromSocket------------------------------------------
 * Sets the sockaddr_in structures using data from the connected socket
 *---------------------------------------------------------------------------*/
bool INetSocket::SetAddrsFromSocket(unsigned short _nFlags)
{
  socklen_t sizeofSockaddr;

  if (_nFlags & ADDR_LOCAL)
  {
    // Setup the local structure (getsockname() fails under SOCKS apparently)
#ifndef USE_SOCKS5
    sizeofSockaddr = sizeof(struct sockaddr_in);
    if (getsockname(m_nDescriptor, (struct sockaddr*)&m_sLocalAddr, &sizeofSockaddr) < 0)
    {
      m_nError = errno;
      return (false);
    }
    if (m_sLocalAddr.sin_addr.s_addr == INADDR_ANY) 
#endif
    {
      hostent *sLocalHost;
      char szHostName[256];
      gethostname(szHostName, 256);
      sLocalHost = gethostbyname(szHostName);
      if (sLocalHost != NULL) 
        m_sLocalAddr.sin_addr.s_addr = *((unsigned long *)sLocalHost->h_addr);
    }
  }
  
  if (_nFlags & ADDR_REMOTE)
  {
    // Set up the remote structure
    sizeofSockaddr = sizeof(struct sockaddr_in);
    if (getpeername(m_nDescriptor, (struct sockaddr *)&m_sRemoteAddr, &sizeofSockaddr) < 0)
    {
      m_nError = errno;
      return (false);
    }
  }
  
  return (true);
}


//-----INetSocket::GetIpByName-------------------------------------------------
unsigned long INetSocket::GetIpByName(char *_szHostName)
{
  struct hostent *host; 
  struct in_addr ina;

  // check if the hostname is in dot and number notation
  if (inet_aton(_szHostName, &ina)) 
     return(ina.s_addr);
  
  // try and resolve hostname
  if ((host = gethostbyname(_szHostName)) == NULL) // Couldn't resolve hostname/ip
  {
     m_nError = h_errno;
     return (0);	
  }
  
  // return the ip
  return *((unsigned long *)host->h_addr);
}


//-----INetSocket::OpenConnection-----------------------------------------------
bool INetSocket::OpenConnection(void)
{  
  // If no destination set then someone screwed up
  if(m_sRemoteAddr.sin_addr.s_addr == 0) return(false);  
   
  m_nDescriptor = socket(AF_INET, m_nSockType, 0);
  if (m_nDescriptor < 0) 
  {
    m_nError = errno;
    return(false);
  }
   
  OpenSocket();
  m_sRemoteAddr.sin_family = AF_INET;
   
  // if connect fails then call CloseConnection to clean up before returning
  socklen_t sizeofSockaddr = sizeof(struct sockaddr);
  if (connect(m_nDescriptor, (struct sockaddr *)&m_sRemoteAddr, sizeofSockaddr) < 0)
  {
    m_nError = errno;
    CloseConnection();
    return(false);
  }

  // This should always work if this above calls worked
  if (!SetAddrsFromSocket(ADDR_LOCAL /*| ADDR_REMOTE*/)) return (false);
  
  return (true);
}


//-----INetSocket::StartServer--------------------------------------------------
bool INetSocket::StartServer(unsigned int _nPort)
{
  m_nDescriptor = socket(AF_INET, m_nSockType, 0);
  if (m_nDescriptor < 0) 
  {
    m_nError = errno; 
    return (false);
  }
   
  OpenSocket();
  memset(&m_sLocalAddr.sin_zero, 0, 8);
  m_sLocalAddr.sin_family = AF_INET;
  m_sLocalAddr.sin_port = htons(_nPort);
  m_sLocalAddr.sin_addr.s_addr = htonl(INADDR_ANY);
  if (bind(m_nDescriptor, (struct sockaddr *)&m_sLocalAddr, sizeof(sockaddr_in)) < 0)
     return (false);

  if (!SetAddrsFromSocket(ADDR_LOCAL)) return (false);

  listen(m_nDescriptor, 5); // Allow 5 unprocessed connections
  return(true);
}                         


//-----INetSocket::SetRemoteAddr-----------------------------------------------
bool INetSocket::SetRemoteAddr(char *_szRemoteName, unsigned short _nRemotePort)
{ 
  return(SetRemoteAddr(GetIpByName(_szRemoteName), _nRemotePort)); 
}


//-----INetSocket::CloseConnection---------------------------------------------
void INetSocket::CloseConnection(void)
{
  CloseSocket();
  m_xRecvBuffer.Clear();
  if (m_nDescriptor != -1) 
  {
    close(m_nDescriptor); 
    m_nDescriptor = -1; 
  }
}


//-----INetSocket::SendRaw------------------------------------------------------
bool INetSocket::SendRaw(CBuffer &b)
{   
  // send the packet
  int nBytesSent; 
  unsigned long nTotalBytesSent = 0;
  while (nTotalBytesSent < b.getDataSize())
  {
    nBytesSent = send(m_nDescriptor, b.getDataStart() + nTotalBytesSent, 
                      b.getDataSize() - nTotalBytesSent, 0);
    if (nBytesSent < 0) 
    { 
      m_nError = errno;       
      CloseConnection(); 
      return(false); 
    }
    nTotalBytesSent += nBytesSent;
  }

  // Print the packet
  DumpPacket(b, D_SENDER);
  return (true);
}


/*-----INetSocket::RecvRaw---------------------------------------------------
 * Receive data on the socket.
 *---------------------------------------------------------------------------*/
bool INetSocket::RecvRaw(void)
{
  char *buffer = new char[MAX_RECV_SIZE];  
  errno = 0;
  int nBytesReceived = recv(m_nDescriptor, buffer, MAX_RECV_SIZE, 0);
  if (nBytesReceived <= 0)
  { 
    m_nError = errno;
    CloseConnection(); 
    return (false); 
  }
  m_xRecvBuffer.Create(nBytesReceived);
  m_xRecvBuffer.add(buffer, nBytesReceived);
  delete[] buffer;
  
  // Print the packet
  DumpPacket(m_xRecvBuffer, D_RECEIVER);

  return (true);
}


//=====TCPSocket===============================================================


/*-----TCPSocket::ReceiveConnection--------------------------------------------
 * Called to set up a given TCPSocket from an incoming connection on the
 * current TCPSocket
 *---------------------------------------------------------------------------*/
void TCPSocket::RecvConnection(TCPSocket &newSocket)
{
  socklen_t sizeofSockaddr = sizeof(struct sockaddr_in);
  newSocket.m_nDescriptor = accept(m_nDescriptor, (struct sockaddr *)&newSocket.m_sRemoteAddr, &sizeofSockaddr);
  newSocket.OpenSocket();
  newSocket.SetAddrsFromSocket(ADDR_LOCAL /*| ADDR_REMOTE*/);
}


/*-----TCPSocket::TransferConnectionFrom---------------------------------------
 * Transfers a connection from the given socket to the current one and closes
 * and resets the given socket
 *---------------------------------------------------------------------------*/
void TCPSocket::TransferConnectionFrom(TCPSocket &from)
{
  m_nDescriptor = from.m_nDescriptor; 
  m_sLocalAddr = from.m_sLocalAddr; 
  m_sRemoteAddr = from.m_sRemoteAddr; 
  ClearRecvBuffer();
  from.m_nDescriptor = -1;
  from.CloseSocket();
  OpenSocket();
}



/*-----TCPSocket::SendPacket---------------------------------------------------
 * Sends a packet on a socket.  The socket is blocking, so we are guaranteed
 * that the entire packet will be sent, however, it may block if the tcp
 * buffer is full.  This should not be a problem unless we are sending a huge
 * packet.
 *---------------------------------------------------------------------------*/
bool TCPSocket::SendPacket(CBuffer &b)
{
  char *pcSize = new char[2];
  pcSize[0] = (b.getDataSize()) & 0xFF;
  pcSize[1] = (b.getDataSize() >> 8) & 0xFF;
   
  // send the length of the packet, close the connection and return false if unable to send
  int nBytesSent = send(m_nDescriptor, pcSize, 2, 0);
  if (nBytesSent <= 0) 
  { 
    delete[] pcSize;
    m_nError = errno;
    CloseConnection(); 
    return (false); 
  }
  delete[] pcSize;

  // send the rest of the packet
  unsigned long nTotalBytesSent = 0;
  while (nTotalBytesSent < b.getDataSize())
  {
    nBytesSent = send(m_nDescriptor, b.getDataStart() + nTotalBytesSent, 
                      b.getDataSize() - nTotalBytesSent, 0);
    if (nBytesSent < 0) 
    { 
      m_nError = errno;       
      CloseConnection(); 
      return(false); 
    }
    nTotalBytesSent += nBytesSent;
  }

  // Print the packet
  DumpPacket(b, D_SENDER);
  
  return (true);
}


/*-----TCPSocket::ReceivePacket------------------------------------------------
 * Receive data on the socket.  Checks the buffer to see if it is empty, if
 * so, then it will create it using either the size read in from the socket 
 * (the first two bytes available) or the given size.  Then determine the
 * number of bytes needed to fill the buffer and call recv with this amount.
 * Note if the buffer is not filled, then the next call to this function will
 * append more bytes until the buffer is full.  If more bytes are available
 * then we take, then the socket will still be readable after we end.  Also
 * note that it is the responsibility of the calling procedure to reset the
 * RecvBuffer if it is full as this is not done here.
 *---------------------------------------------------------------------------*/
bool TCPSocket::RecvPacket(void)
{
  if (m_xRecvBuffer.Full()) 
  {
    gLog.Warn("%sInternal error: TCPSocket::RecvPacket(): Called with full buffer (%d bytes).\n", 
              L_WARNxSTR, m_xRecvBuffer.getDataSize());
    return (true);
  }

  int nBytesReceived = 0;
  errno = 0;
  
  // Check if the buffer is empty
  if (m_xRecvBuffer.Empty())
  {  
    char *buffer = new char[2];
    int nTwoBytes = 0;
    while (nTwoBytes != 2)
    {
      nBytesReceived = recv(m_nDescriptor, buffer + nTwoBytes, 2 - nTwoBytes, 0);
      if (nBytesReceived <= 0)  
      { 
        m_nError = errno; 
        delete[] buffer;
        CloseConnection(); 
        return (false); 
      }
      nTwoBytes += nBytesReceived;
    }
    m_xRecvBuffer.Create(((unsigned char)buffer[0]) + 
                         (((unsigned char)buffer[1]) << 8 ));
    delete[] buffer;
  }
  
  // Problem: if we got 2 bytes, then we assume there are more bytes to be read
  // This should be ok almost always, but creates the DoS attack where someone
  // sends two bytes and then nothing.  In that case we will block in recv.
  // One solution is to make the socket temporarily non-blocking for the call
  // to recv, this will however slow stuff down in this function so we avoid
  // it for now.  Another solution would be to return after the first call
  // to receive and wait to be called again, but this is really slow.
  
  //int nFlags = fcntl(m_nDescriptor, F_GETFL);
  //fcntl(m_nDescriptor, F_SETFL, O_NONBLOCK);
  
  // Determine the number of bytes left to be read into the buffer
  unsigned long nBytesLeft = m_xRecvBuffer.getDataStart() +
                             m_xRecvBuffer.getDataMaxSize() -
                             m_xRecvBuffer.getDataPosWrite();
  nBytesReceived = recv(m_nDescriptor, m_xRecvBuffer.getDataPosWrite(), nBytesLeft, 0);
  if (nBytesReceived <= 0)
  { 
    m_nError = errno;
    //if (errno == EAGAIN || errno == EWOULDBLOCK) return (true); 
    CloseConnection(); 
    return (false); 
  }
  m_xRecvBuffer.incDataPosWrite(nBytesReceived);
  
  // Print the packet if it's full
  if (m_xRecvBuffer.Full()) DumpPacket(m_xRecvBuffer, D_RECEIVER);

  //fcntl(m_nDescriptor, F_SETFL, nFlags);   
  return (true);
}

