/*
ICQ.H
header file containing all the main procedures to interface with the ICQ server at mirabilis
*/

#ifndef ICQ_H
#define ICQ_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <vector.h>
#include <list.h>
#include <qobject.h>
#include <qtimer.h>
#include <stdarg.h>

#include "icq-defines.h"
#include "socket.h"
#include "icqpacket.h"
#include "file.h"
#include "user.h"
#include "remoteserver.h"
#include "icqevent.h"



//=====ICQ=========================================================================================

class ICQ : public QObject
{
   Q_OBJECT
public:
   ICQ(void);
   ~ICQ(void);
   
   ICQEvent *sendMessage(ICQUser *u, const char *m, bool online, unsigned long id = 0);
   ICQEvent *sendReadAwayMsg(ICQUser *, bool, unsigned long id = 0);
   ICQEvent *sendUrl(ICQUser *u, const char *url, const char *description, bool online, unsigned long id = 0);
   ICQEvent *sendChat(ICQUser *u, const char *reason, bool online, unsigned long id = 0);
   ICQEvent *sendFile(ICQUser *u, const char *_szFilename, const char *_szDescription, bool online, unsigned long id = 0);
   void fileRefuse(ICQUser *u, const char *reason, unsigned long theSequence);
   void fileCancel(ICQUser *u, unsigned long seq, bool online, unsigned long id = 0);
   void fileAccept(ICQUser *u, unsigned short thePort, unsigned long theSequence);
   void chatRefuse(ICQUser *u, const char *reason, unsigned long theSequence);
   void chatAccept(ICQUser *u, unsigned short thePort, unsigned long theSequence);
   void chatCancel(ICQUser *u, unsigned long seq, bool online, unsigned long id = 0);
   
   bool icqLogon(unsigned long logonStatus);
   void icqLogoff(bool reconnect);
   bool icqSetStatus(unsigned long newStatus);
   bool icqGetUserBasicInfo(ICQUser *user);
   bool icqGetUserExtInfo(ICQUser *user);
   void icqUpdatePersonalBasicInfo(const char *, const char *, const char *, 
                                   const char *, bool);
   void icqUpdatePersonalExtInfo(const char *, unsigned short, const char *, 
                                 unsigned short, char, const char *,
                                 const char *, const char *_sAbout);
   void icqSendVisibleList(bool _bSendIfEmpty = false);
   void icqSendInvisibleList(bool _bSendIfEmpty = false);
   void icqAuthorize(unsigned long uinToAuthorize);
   void icqAlertUser(unsigned long _nUin);
   void icqRequestSystemMsg(void);
   void icqStartSearch(char *, char *, char *, char *, unsigned long);
   
   bool nextServer(void);
   void eventCancel(ICQEvent *&);
   
   void usrAdd(unsigned long id, char *filename);
   void usrRemove(ICQUser *);
   void usrSave(void);
   
   bool getIsKnownUser(unsigned long);
   void getBasicInfo(struct UserBasicInfo &us);
   void getExtInfo(struct UserExtInfo &ud);
   int getNumUsers(void);
   unsigned short getTcpServerPort(void);
   unsigned short getDefaultRemotePort(void);
   unsigned short getMaxUsersPerPacket(void)  { return m_nMaxUsersPerPacket; }
   bool getTcpPort(unsigned short);
   void setTcpPort(unsigned short, bool);
   char *getAwayMessage(void);
   ICQUser *getUser(unsigned short);
   ICQUser *getLastUser(void);
   CUsers *getUsers(void)  { return &m_cUsers; };
   void setUser(unsigned short i, ICQUser *u);
   char *getSoundPlayer(void);
   char *getSoundMsg(void);
   char *getSoundUrl(void);
   char *getSoundChat(void);
   char *getSoundFile(void);
   char *getSoundNotify(void);
   const char *getUrlViewer(void);

   void setAwayMessage(char *);
   
   void setTcpServerPort(unsigned short n)  { m_nTcpServerPort = n; }
   void setDefaultRemotePort(unsigned short n)  { m_nDefaultRemotePort = n; }   
   void setMaxUsersPerPacket(unsigned short n)  { m_nMaxUsersPerPacket = n; }   
   void setUrlViewer(const char *s);
   void setSoundEnabled(unsigned short);
   unsigned short getSoundEnabled(void);   
   void setSounds(const char *, const char *, const char *, 
                  const char *, const char *, const char *);

   ICQOwner icqOwner;
   ICQRemoteServers icqServers;
   
protected:
   // protected variables
   char remoteServer[64];
   CUsers m_cUsers;
   vector<bool> m_vbTcpPorts;
   vector <ICQEvent *> icqEvents;
   list <TCPSocket *> m_lxUnclaimedSockets;

   bool m_bAutoReconnect, 
        m_bTmpSoundEnabled;        
   unsigned long m_nDesiredStatus;
   unsigned short m_nAllowUpdateUsers,
                  m_nSoundEnabled, 
                  m_nNumUsers,
                  m_nTcpServerPort, 
                  m_nDefaultRemotePort,
                  m_nMaxUsersPerPacket;                 
   TCPSocket *tcpSocket;
   UDPSocket udpServer;
   QTimer pingTimer;
   char *m_sSoundPlayer, 
        *m_sSoundMsg, 
        *m_sSoundUrl, 
        *m_sSoundChat, 
        *m_sSoundFile, 
        *m_sSoundNotify;
   char *m_szUrlViewer;
   
   // Variables used to pass info around internally
   bool m_bEventAccepted;
   unsigned short m_nEventPort;
   char m_szEventMessage[MAX_MESSAGE_SIZE];
   
   // protected functions
   void ParseFE(char *szBuffer, char ***szSubStr, int nMaxSubStr);
   void playSound(char *);
   
   void ackUDP(unsigned short);
   void ackTCP(CPacketTcp &, ICQUser *);
   
   unsigned short processUdpPacket(CBuffer &packet);
   void processSystemMessage(CBuffer &packet, unsigned long checkUin, unsigned short newCommand, time_t timeSent);
   void processTcpPacket(CBuffer &packet, int sockfd);
   
   bool connectToServer(void);
   bool connectToUser(ICQUser *u);
   ICQUser *getUserByUIN(unsigned long u);
   void usrAdd(unsigned long id, TCPSocket &sock);
   
   void icqAddUser(ICQUser *u);
   
   ICQEvent *eventStart(INetSocket &, CPacket &, bool, 
                        unsigned long uin = 0, CUserEvent *e = NULL);

signals:
   void signal_updatedUsers();
   void signal_updatedStatus();
   void signal_doneOwnerFcn(bool isOk, unsigned short cmd);
   void signal_doneUserFcn(bool isOk, ICQEvent *e);
   void signal_doneUserBasicInfo(bool, unsigned long);
   void signal_doneUserExtInfo(bool, unsigned long);
   void signal_doneUpdatePersonalBasicInfo(bool);
   void signal_doneUpdatePersonalExtInfo(bool);
   void signal_eventResult(ICQUser *u, CUserEvent *e, bool accepted, 
                           const char *_szResponse, unsigned long port);
   void signal_doneSearch(unsigned short, char);
   void signal_userFound(struct UserBasicInfo &, unsigned long);

public slots:
   void icqUpdateContactList();
   void updateAllUsers();
   
protected slots:
   void ping();
   void eventDone(bool gotAck, int sockfd, int seq);
   void recvUDP(int);
   void recvNewTcp(int);
   void recvTcpHandshake(int _nSD);
   void recvTcp(int sockfd);
};

#endif

