#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <iostream.h>

#include "time-fix.h"

#include "icq.h"
#include "translate.h"
#include "log.h"

//-----ICQ::connectToServer-----------------------------------------------------
bool ICQ::connectToServer()
{
   // no servers!
   if (icqServers.current() == NULL) 
   {
      gLog.Error("%sInternal error: Attempt to connect to server when no servers are set.\n", 
                 L_ERRORxSTR);
      return (false);
   }
   
   // try and set the destination
   gLog.Info("%sResolving %s...\n", L_UDPxSTR, icqServers.current()->name());
   if (!udpServer.SetRemoteAddr(icqServers.current()->name(), icqServers.current()->port()))
   {
      gLog.Error("%sUnable to resolve %s:\n%s%s.\n", L_ERRORxSTR, 
               icqServers.current()->name(), L_BLANKxSTR, udpServer.ErrorStr());
      return (false);  // no route to host (not connected)
   }
   else
   {
      gLog.Info("%sICQ server found at %s:%d.\n", L_UDPxSTR, 
                udpServer.RemoteIpStr(), udpServer.RemotePort());
      return (true);
   }
}


//-----ICQ::addNewUser-----------------------------------------------------------------------------
void ICQ::icqAddUser(ICQUser *u)
{
   if (!icqOwner.getStatusOffline())
   {
      gLog.Info("%sAlerting server to new user (#%d)...\n", L_UDPxSTR, 
               icqOwner.getSequence());
      CUserGroup g("temp");
      g.addUser(u);
      CPU_ContactList p(&g, 0, 1);
      eventStart(udpServer, p, true);
      
      // update the users info from the server
      icqGetUserBasicInfo(u);
   }

   // save the new info
   usrSave();
   u->saveInfo();
   u->saveBasicInfo();
   u->saveExtInfo();
   m_cUsers.reorder(u);
}


//-----ICQ::addNewUser-----------------------------------------------------------------------------
void ICQ::icqAlertUser(unsigned long _nUin)
{
   if (icqOwner.getStatusOffline()) return;
   
   gLog.Info("%sAlerting user they were added (#%d)...\n", L_UDPxSTR, icqOwner.getSequence());
   CPU_AddUser p(_nUin);
   eventStart(udpServer, p, true);
}



//-----ICQ::nextServer-----------------------------------------------------------------------------
bool ICQ::nextServer(void)
{
   udpServer.ResetSocket();
   icqLogoff(false);
   icqServers.next();
   return(connectToServer());
}


//-----ICQ::Logon---------------------------------------------------------------
bool ICQ::icqLogon(unsigned long logonStatus)
{
   // check to see if we have a destination
   if (!udpServer.DestinationSet())
      if (!connectToServer()) return (false);   // if setting the destination fails, then return false

   gLog.Info("%sOpening socket to server.\n", L_UDPxSTR);
   if (!udpServer.OpenConnection())
   {
     gLog.Error("%sUnable to connect to %s:%d:\n%s%s.\n", L_ERRORxSTR, 
                udpServer.RemoteIpStr(), udpServer.RemotePort(), L_BLANKxSTR,
                udpServer.ErrorStr());
     return false;
   }

   connect (udpServer.sn, SIGNAL(activated(int)), this, SLOT(recvUDP(int)));
   
   gLog.Info("%sRequesting logon (#%d)...\n", L_UDPxSTR, icqOwner.getSequence());
   CPU_Logon p(udpServer.LocalIp(), logonStatus);
   eventStart(udpServer, p, true);
   m_nDesiredStatus = logonStatus;
   return(true);
}


//-----ICQ::icqLogoff-----------------------------------------------------------
void ICQ::icqLogoff(bool reconnect)
{
   // if not connected then don't both logging off
   if (udpServer.Connected())
   {
      gLog.Info("%sLogging off.\n", L_UDPxSTR);
      CPU_Logoff packet;
      udpServer.SendRaw(*packet.getBuffer());
      udpServer.CloseConnection();
      
      // close all open events
      unsigned short i;
      for (i = 0; i < icqEvents.size(); i++) 
         delete icqEvents[i];
      //icqEvents.clear();  // shitty gcc 2.7 stl doesn't have this function
      icqEvents.erase(icqEvents.begin(),icqEvents.end());
   }

   if (reconnect) 
   {
     icqLogon(icqOwner.getStatusFull());
   }
   else
   {
     // reset all the users statuses
     for (unsigned short i = 0; i < getNumUsers(); i++) 
       getUser(i)->setStatusOffline();
     emit signal_updatedUsers();
   }
   icqOwner.setStatusOffline();

   emit signal_updatedStatus();
   pingTimer.stop();
   emit signal_doneOwnerFcn(true, ICQ_CMDxSND_LOGOFF);
}             


//-----ICQ::icqUpdateContactList------------------------------------------------
void ICQ::icqUpdateContactList(void)
{   
  m_nAllowUpdateUsers = 0; 
  // send user info packet
  unsigned short numPackets = 1 + (m_cUsers.getNumUsers() / m_nMaxUsersPerPacket);
  for (unsigned short i = 0; i < numPackets; i++) 
  {
    gLog.Info("%sUpdating contact list (#%d)...\n", L_UDPxSTR, 
                          icqOwner.getSequence());
    CPU_ContactList p(m_cUsers.getGroup(0), i * m_nMaxUsersPerPacket, 
                      m_nMaxUsersPerPacket);
    eventStart(udpServer, p, true);
    m_nAllowUpdateUsers++;
  }
  
  // reset all the users statuses
  for (unsigned short i = 0; i < getNumUsers(); i++) 
    getUser(i)->setStatusOffline();
  emit signal_updatedUsers();
  
  // do this to ease down on the update user calls when the online user comes 
  // in...gets set to true when the "end of users" packet comes in
  //m_bAllowUpdateUsers = false;     
}


//-----ICQ::icqSendVisibleList--------------------------------------------------
void ICQ::icqSendVisibleList(bool _bSendIfEmpty = false)
{
   // send user info packet
   CPU_VisibleList p(m_cUsers.getGroup(0));
   if (!p.empty() || _bSendIfEmpty) 
   {
      gLog.Info("%sSending visible list (#%d)...\n", L_UDPxSTR, 
                icqOwner.getSequence() - 1);
      eventStart(udpServer, p, true);
   }
}


//-----ICQ::icqSendInvisibleList------------------------------------------------
void ICQ::icqSendInvisibleList(bool _bSendIfEmpty = false)
{
   CPU_InvisibleList p(m_cUsers.getGroup(0));
   if (!p.empty() || _bSendIfEmpty) 
   {
      gLog.Info("%sSending invisible list (#%d)...\n", L_UDPxSTR, 
                            icqOwner.getSequence() - 1);
      eventStart(udpServer, p, true);
   }
}


//-----ICQ::startSearch---------------------------------------------------------------------
void ICQ::icqStartSearch(char *nick, char *first, char *last, char *email, unsigned long s)
{
   gLog.Info("%sStarting search for user (#%d/%d)...\n", L_UDPxSTR, 
            icqOwner.getSequence(), s);
   CPU_StartSearch p(nick, first, last, email, s);
   eventStart(udpServer, p, true);
}


//-----ICQ::ping-----------------------------------------------------------------------------------
void ICQ::ping()
{
   if (icqOwner.getStatusOffline()) return;

   gLog.Info("%sPinging Mirabilis (#%d)...\n", L_UDPxSTR, icqOwner.getSequence());
   CPU_Ping p;
   eventStart(udpServer, p, true);
}


//-----ICQ::setStatus------------------------------------------------------------------------------
bool ICQ::icqSetStatus(unsigned long newStatus)
{
   if (!udpServer.Connected()) return(false);
   gLog.Info("%sChanging status (#%d)...\n", L_UDPxSTR, icqOwner.getSequence());
   CPU_SetStatus p(newStatus);
   eventStart(udpServer, p, true);
   
   m_nDesiredStatus = newStatus;
   return(true);
}


//-----ICQ::getUserBasicInfo----------------------------------------------------
bool ICQ::icqGetUserBasicInfo(ICQUser *user)
{
   gLog.Info("%sRequesting user info (#%d)...\n", L_UDPxSTR, icqOwner.getSequence());
   CPU_GetUserBasicInfo p(user->getUin());
   eventStart(udpServer, p, true);
   return(true);
}


//-----ICQ::getUserExtInfo---------------------------------------------------
bool ICQ::icqGetUserExtInfo(ICQUser *user)
{
   gLog.Info("%sRequesting extended user info (#%d)...\n", L_UDPxSTR, 
            icqOwner.getSequence());
   CPU_GetUserExtInfo p(user->getUin());
   eventStart(udpServer, p, true);
   return(true);
}


//-----ICQ::icqUpdatePersonalBasicInfo------------------------------------------
void ICQ::icqUpdatePersonalBasicInfo(const char *_sAlias, const char *_sFirstName, 
                                     const char *_sLastName, const char *_sEmail,  
                                     bool _bAuthorization)
{
   gLog.Info("%sUpdating personal information (#%d)...\n", L_UDPxSTR, 
            icqOwner.getSequence());
   CPU_UpdatePersonalBasicInfo p(_sAlias, _sFirstName, _sLastName, _sEmail, 
                                 (char)_bAuthorization);
   eventStart(udpServer, p, true);
}


//-----ICQ::icqUpdatePersonalExtInfo--------------------------------------------
void ICQ::icqUpdatePersonalExtInfo(const char *_sCity, unsigned short _nCountry,
                                   const char *_sState, unsigned short _nAge,
                                   char _cSex, const char *_sPhone,
                                   const char *_sHomepage, const char *_sAbout)
{
   gLog.Info("%sUpdating personal extended info (%d)...\n", L_UDPxSTR, 
            icqOwner.getSequence());
   CPU_UpdatePersonalExtInfo p(_sCity, _nCountry, _sState, _nAge, _cSex, 
                                 _sPhone, _sHomepage, _sAbout);
   eventStart(udpServer, p, true);
}


//-----ICQ::authorize-----------------------------------------------------------
void ICQ::icqAuthorize(unsigned long uinToAuthorize)
// authorize a user to add you to their contact list...for stupid windows people
{
   gLog.Info("%sSending authorization to %ld (#%d)...\n", L_UDPxSTR, 
             uinToAuthorize, icqOwner.getSequence());
   CPU_Authorize p(uinToAuthorize);
   eventStart(udpServer, p, true);
}


//-----ICQ::icqRequestSystemMsg-------------------------------------------------
void ICQ::icqRequestSystemMsg(void)
// request offline system messages
{
   gLog.Info("%sSending offline system message request (#%d)...\n", L_UDPxSTR, 
            icqOwner.getSequence());
   CPU_RequestSysMsg p;
   eventStart(udpServer, p, true);
}



//-----ICQ::processUdpPacket-----------------------------------------------------------------------
unsigned short ICQ::processUdpPacket(CBuffer &packet)
{
   unsigned short version, command, newCommand, theSequence, searchSequence, userPort, newStatus, messageLen, checkSequence, junkShort;
   unsigned long checkUin, userIP, junkLong;
   char junkChar;
   int i;
   ICQUser *u;
   CBuffer newPacket;
   
   // read in the standard UDP header info
   packet >> version 
          >> command 
          >> theSequence
   ;

   if (version != ICQ_VERSION) 
   { 
      gLog.Warn("%sBad UDP version number: %d.\n", L_WARNxSTR, version); 
      return(0xFFFF); 
   }
   
   switch (command) 
   {
   case ICQ_CMDxRCV_USERxONLINE:   // initial user status packet
      /* 02 00 6E 00 0B 00 8F 76 20 00 CD CD 77 90 3F 50 00 00 7F 00 00 01 04 00
         00 00 00 03 00 00 00 */
      ackUDP(theSequence);
      packet >> checkUin;
      gLog.Info("%sUser %ld is online.\n", L_UDPxSTR, checkUin);
      
      // find which user it is, verify we have them on our list
      if ((u = getUserByUIN(checkUin)) == NULL) 
      { 
         gLog.Warn("%sUnknown user (%ld) is online.\n", L_WARNxSTR, checkUin);
         break; 
      }

      // read in the relevant user information
      packet >> userIP
             >> userPort
             >> junkLong >> junkShort >> junkChar  // 7 bytes of junk
             >> newStatus  // initial status of user, anything but offline
      ;
      
      // The packet class will spit out an ip in network order on a little
      // endian machine and in little-endian on a big endian machine
      userIP = PacketIpToNetworkIp(userIP);
      u->tcpSocket.SetRemoteAddr(userIP, userPort);
      u->setStatus(newStatus);
      u->setAwayMessage(NULL);
      if (u->getOnlineNotify()) playSound(getSoundNotify());
      m_cUsers.reorder(u);  // put the user at the top of the list
      if (m_nAllowUpdateUsers == 0) emit signal_updatedUsers();
      break;
   
   case ICQ_CMDxRCV_USERxOFFLINE:  // user just went offline packet
      /* 02 00 78 00 06 00 ED 21 4E 00 */
      ackUDP(theSequence);
      packet >> checkUin;
      gLog.Info("%sUser %ld went offline.\n", L_UDPxSTR, checkUin);

      // find which user it is, verify we have them on our list
      if ((u = getUserByUIN(checkUin)) == NULL) 
      { 
         gLog.Warn("%sUnknown user (%ld) has gone offline.\n", L_WARNxSTR, checkUin);
         break; 
      }
            
      u->setStatusOffline();
      u->setAwayMessage(NULL);
      //u->tcpSocket.CloseConnection();
      
      m_cUsers.reorder(u);

      emit signal_updatedUsers();
      break;

   case ICQ_CMDxRCV_USERxINFO:   // user info packet
      /* 02 00 18 01 6C 00 10 00 50 A5 82 00 08 00 48 61 63 6B 49 43 51 00 04 00
         46 6F 6F 00 04 00 42 61 72 00 15 00 68 61 63 6B 65 72 73 40 75 77 61 74 
         65 72 6C 6F 6F 2E 63 61 00 00 00 00  */
      ackUDP(theSequence);
      packet >> checkSequence  // corresponds to the sequence number from the user information request packet...totally irrelevant.
             >> checkUin
      ;
      gLog.Info("%sReceived information for UIN %ld:\n", L_UDPxSTR, checkUin);
      // find which user it is, verify we have them on our list
      if ((u = getUserByUIN(checkUin)) == NULL) 
      { 
         gLog.Warn("%sUnknown user (%d) information.\n", L_WARNxSTR, checkUin);
         break; 
      }
      
      // read in the four data fields; alias, first name, last name, and email address
      u->setEnableSave(false);
      char temp[MAX_DATA_LEN], cAuthorization;
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> temp[i];
      u->setAlias(temp);
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> temp[i];
      u->setFirstName(temp);
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> temp[i];
      u->setLastName(temp);
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> temp[i];
      u->setEmail(temp);
      packet >> cAuthorization;
      u->setAuthorization(cAuthorization == 0 ? true : false);
      
      // translating string with Translation Table
      gTranslator.ServerToClient(u->getAlias());
      gTranslator.ServerToClient(u->getFirstName());
      gTranslator.ServerToClient(u->getLastName());

      // print out the user information
      gLog.Info("%s%s (%s %s), %s.\n", L_BLANKxSTR , u->getAlias(), 
               u->getFirstName(), u->getLastName(), u->getEmail()); 
      
      // save the user infomation
      u->setEnableSave(true);
      u->saveBasicInfo();

      emit signal_doneUserBasicInfo(true, checkUin);
      emit signal_updatedUsers();
      break;
   
   case ICQ_CMDxRCV_USERxDETAILS:   // user details packet
      /* 02 00 22 01 09 00 0A 00 96 3D 44 00 04 00 3F 3F 3F 00 2E 00 FE 01 00 00
         FF FF 01 01 00 00 1C 00 68 74 74 70 3A 2F 2F 68 65 6D 2E 70 61 73 73 61
         67 65 6E 2E 73 65 2F 67 72 72 2F 00 1E 00 49 27 6D 20 6A 75 73 74 20 61
         20 67 69 72 6C 20 69 6E 20 61 20 77 6F 72 6C 64 2E 2E 2E 00 FF FF FF FF */

      unsigned short country_code, nAge; 
      char country_status, cSex; 
 
      ackUDP(theSequence);
      packet >> checkSequence  // corresponds to the sequence number from the user information request packet...totally irrelevant.
             >> checkUin;
      gLog.Info("%sReceived extended information for UIN %ld:\n", L_UDPxSTR, checkUin);
      
      // find which user it is, verify we have them on our list
      if ((u = getUserByUIN(checkUin)) == NULL) 
      { 
         gLog.Warn("%sUnknown user (%d) extended information.\n", L_WARNxSTR, checkUin);
         break; 
      }
      
      u->setEnableSave(false);

      char sTemp[MAX_MESSAGE_SIZE]; 
     
      // City
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> sTemp[i];
      u->setCity(sTemp);
      
      // Country
      packet >> country_code;
      packet >> country_status;
      
      // State
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> sTemp[i];
      u->setState(sTemp);
      
      // Age
      packet >> nAge;
      u->setAge(nAge);
      
      // Sex
      packet >> cSex;
      u->setSex(cSex);

      // Phone Number
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> sTemp[i];
      u->setPhoneNumber(sTemp);
      
      // Homepage
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> sTemp[i];
      u->setHomepage(sTemp);
      
      // About
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> sTemp[i];
      u->setAbout(sTemp);

      // Time to transfer the int variables into their proper values 
      u->setCountry(country_code);
      
      // translating string with Translation Table
      gTranslator.ServerToClient(u->getCity());
      gTranslator.ServerToClient(u->getState());
      gTranslator.ServerToClient(u->getHomepage());
      gTranslator.ServerToClient(u->getAbout());

      // print out the user information
      gLog.Info("%s%s, %s, %s. %d years old, %s. %s, %s, %s.\n", L_BLANKxSTR,
               u->getCity(), u->getState(), u->getCountry(), u->getAge(), 
               u->getSex(), u->getPhoneNumber(), u->getHomepage(), u->getAbout()); 
      
      // save the user infomation
      u->setEnableSave(true);
      u->saveExtInfo();

      emit signal_doneUserExtInfo(true, checkUin);
      //emit signal_updatedUsers(); 
      break;
   
   case ICQ_CMDxRCV_UPDATEDxBASIC:
      ackUDP(theSequence);
      gLog.Info("%sSuccessfully updated basic info.\n", L_UDPxSTR);
      emit signal_doneUpdatePersonalBasicInfo(true);
      break;

   case ICQ_CMDxRCV_UPDATExBASICxFAIL:
      ackUDP(theSequence);
      gLog.Info("%sFailed to update basic info.\n", L_UDPxSTR);
      emit signal_doneUpdatePersonalBasicInfo(false);
      break;
                                
   case ICQ_CMDxRCV_UPDATEDxDETAIL:
      ackUDP(theSequence);
      gLog.Info("%sSuccessfully updated detail info.\n", L_UDPxSTR);
      emit signal_doneUpdatePersonalExtInfo(true);
      break;                   

   case ICQ_CMDxRCV_UPDATExDETAILxFAIL:
      ackUDP(theSequence);
      gLog.Info("%sFailed to update detail info.\n", L_UDPxSTR);
      emit signal_doneUpdatePersonalExtInfo(false);
      break;                   

   case ICQ_CMDxRCV_USERxINVALIDxUIN:  // not a good uin
      ackUDP(theSequence);
      packet >> checkUin;
      gLog.Info("%sInvalid UIN: %d.\n", L_UDPxSTR, checkUin);
      emit signal_doneUserBasicInfo(false, checkUin);
      break;

   case ICQ_CMDxRCV_USERxSTATUS:  // user changed status packet
      ackUDP(theSequence);
      packet >> checkUin;
      gLog.Info("%sUser %ld changed status.\n", L_UDPxSTR, checkUin);

      // find which user it is, verify we have them on our list
      if ((u = getUserByUIN(checkUin)) == NULL) 
      { 
         gLog.Warn("%sUnknown user (%d) changed status.\n", L_WARNxSTR, checkUin);
         break; 
      }
      
      packet >> junkLong;
      u->setStatus(junkLong);
      m_cUsers.reorder(u);
      u->setAwayMessage(NULL);
      emit signal_updatedUsers();
      break;

   case ICQ_CMDxRCV_USERxLISTxDONE:  // end of user list
      /* 02 00 1C 02 05 00 8F 76 20 00 */
      ackUDP(theSequence);
      gLog.Info("%sEnd of initial user list.\n", L_UDPxSTR);
      m_nAllowUpdateUsers--;
      emit signal_updatedUsers();
      break;

   case ICQ_CMDxRCV_SEARCHxFOUND:  // user found in search 
      /* 02 00 8C 00 03 00 05 00 8F 76 20 00 0B 00 41 70 6F 74 68 65 6F 73 69 73
         00 07 00 47 72 61 68 61 6D 00 05 00 52 6F 66 66 00 13 00 67 72 6F 66 66 40 75
         77 61 74 65 72 6C 6F 6F 2E 63 61 00 01 02 */
      ackUDP(theSequence);
      gLog.Info("%sSearch found user:\n", L_UDPxSTR);

      unsigned short i, aliasLen, firstNameLen, lastNameLen, emailLen;
      char auth;
      struct UserBasicInfo us;

      packet >> searchSequence
	          >> checkUin
      ;
	   sprintf(us.uin, "%ld", checkUin);

      packet >> aliasLen;
      for (i = 0; i < aliasLen; i++) packet >> us.alias[i];
      us.alias[i] = '\0';
      
      packet >> firstNameLen;
      for (i = 0; i < firstNameLen; i++) packet >> us.firstname[i];
      us.firstname[i] = '\0';
      
      packet >> lastNameLen;
      for (i = 0; i < lastNameLen; i++) packet >> us.lastname[i];
      us.lastname[i] = '\0';

      packet >> emailLen;
      for (i = 0; i < emailLen; i++) packet >> us.email[i];
      us.email[i] = '\0';

      // translating string with Translation Table
      gTranslator.ServerToClient(us.alias);
      gTranslator.ServerToClient(us.firstname);
      gTranslator.ServerToClient(us.lastname);

      packet >> auth;
      sprintf(us.name, "%s %s", us.firstname, us.lastname);
      gLog.Info("%s%s (%ld) <%s %s, %s>\n", L_BLANKxSTR, us.alias, checkUin, 
               us.firstname, us.lastname, us.email);
      
      emit signal_userFound(us, searchSequence);
      break;

   case ICQ_CMDxRCV_SEARCHxDONE:  // user found in search 
      /* 02 00 A0 00 04 00 05 00 00*/
      ackUDP(theSequence);
      
      char more;
      packet >> searchSequence 
             >> more;
      gLog.Info("%sSearch finished.\n", L_UDPxSTR);
      emit signal_doneSearch(searchSequence, more);
      break;
   
   case ICQ_CMDxRCV_SYSxMSGxDONE:  // end of system messages
   {   
      /* 02 00 E6 00 04 00 50 A5 82 00 */
   
      ackUDP(theSequence);
      gLog.Info("%sEnd of system messages.\n", L_UDPxSTR);
      
      // send special ack for this one as well as the usual ack
      gLog.Info("%sSending ok to erase system messages...\n", L_UDPxSTR);
      CPU_SysMsgDoneAck p(theSequence);
      eventStart(udpServer, p, true);
      break;
   }
   case ICQ_CMDxRCV_SYSxMSGxOFFLINE:  // offline system message, now have to check the sub-command
      /* 02 00 DC 00 0A 00 EC C9 45 00 CE 07 04 17 04 21 01 00 3F 00 6E 6F 2C 20 
         73 74 69 6C 6C 20 68 6F 70 69 6E 67 20 66 6F 72 20 74 68 65 20 72 65 63 
         6F 72 64 20 63 6F 6D 70 61 6E 79 2C 20 62 75 74 20 79 6F 75 20 6E 65 76 
         65 72 20 6B 6E 6F 77 2E 2E 2E 00 */
      ackUDP(theSequence);
      gLog.Info("%sOffline system message:\n", L_UDPxSTR);

      unsigned short yearSent;
      char monthSent, daySent, hourSent, minSent;
      packet >> checkUin
             >> yearSent
             >> monthSent
             >> daySent
             >> hourSent
             >> minSent
             >> newCommand
      ;
      // prepare a structure containing the relevant time information
      struct tm sentTime;
      sentTime.tm_sec  = 0;
      sentTime.tm_min  = minSent;
      sentTime.tm_hour = hourSent;
      sentTime.tm_mday = daySent;
      sentTime.tm_mon  = monthSent - 1;
      sentTime.tm_year = yearSent - 1900;
      
      // process the system message, sending the time it occured converted to a time_t structure
      m_bTmpSoundEnabled = false;  // disable sound for offline system messages to avoid segfaults on multiple messages
      processSystemMessage(packet, checkUin, newCommand, mktime(&sentTime));
      break;
   
   case ICQ_CMDxRCV_SYSxMSGxONLINE:  // online system message
      ackUDP(theSequence);
      gLog.Info("%sOnline system message:\n", L_UDPxSTR);

      packet >> checkUin
             >> newCommand;
      
      // process the system message, sending the current time as a time_t structure
      processSystemMessage(packet, checkUin, newCommand, time(NULL));
      break;
   
   case ICQ_CMDxRCV_SETxOFFLINE:  // we got put offline by mirabilis for some reason
      gLog.Info("%sKicked offline by server.\n", L_UDPxSTR);
      icqLogoff(m_bAutoReconnect);
      break;
      
   case ICQ_CMDxRCV_ACK:  // icq acknowledgement
      /* 02 00 0A 00 12 00 */
      gLog.Info("%sAck (#%d).\n", L_UDPxSTR, theSequence);
      eventDone(true, udpServer.Descriptor(), theSequence);
      break;

   case ICQ_CMDxRCV_ERROR:  // icq says go away
      gLog.Info("%sServer says bugger off (%d).\n", L_UDPxSTR, theSequence);
      eventDone(false, udpServer.Descriptor(), theSequence);  // ???
      break;
   
   case ICQ_CMDxRCV_HELLO: // hello packet from mirabilis received on gLogon
      /* 02 00 5A 00 00 00 8F 76 20 00 CD CD 76 10 02 00 01 00 05 00 00 00 00 00
         8C 00 00 00 F0 00 0A 00 0A 00 05 00 0A 00 01 */
      ackUDP(theSequence);
      gLog.Info("%sMirabilis says hello.\n", L_UDPxSTR);
      
      icqOwner.setStatus(m_nDesiredStatus);
      pingTimer.start(PING_FREQUENCY * 1000);
      emit signal_doneOwnerFcn(true, ICQ_CMDxSND_LOGON);
      emit signal_updatedStatus();

      icqUpdateContactList();
      icqSendInvisibleList();
      icqSendVisibleList();
      icqRequestSystemMsg();
      break;

   case ICQ_CMDxRCV_WRONGxPASSWD:  // incorrect password sent in gLogon
      /* 02 00 64 00 00 00 02 00 8F 76 20 00 */
      gLog.Error("%sIncorrect password: %s.\n", L_ERRORxSTR, icqOwner.getPassword());
      emit signal_doneOwnerFcn(false, ICQ_CMDxSND_LOGON);
      break;

   case ICQ_CMDxRCV_BUSY:  // server too busy to respond
      gLog.Info("%sServer busy, try again in a few minutes.\n", L_UDPxSTR);
      emit signal_doneOwnerFcn(false, ICQ_CMDxSND_LOGON);
      break;
  
   default:  // what the heck is this packet?  print it out
      ackUDP(theSequence);
      gLog.Unknown("%sUnknown UDP packet:\n%s\n", L_UNKNOWNxSTR, packet.print());
      break;
   }
   
   return(command);
}


//-----ICQ::processSystemMessage-------------------------------------------------------------------
void ICQ::processSystemMessage(CBuffer &packet, unsigned long checkUin, unsigned short newCommand, time_t timeSent)
{
   unsigned short messageLen;
   ICQUser *u;
   int i;
   
   unsigned long nMask = (newCommand | ICQ_CMDxSUB_FxMULTIREC ? E_MULTIxREC : 0);
   newCommand &= ~ICQ_CMDxSUB_FxMULTIREC;

   switch(newCommand)
   {
   case ICQ_CMDxSUB_MSG:  // system message: message through the server 
   {
      gLog.Info("%sMessage through server from %ld.\n", L_BLANKxSTR, checkUin);

      packet >> messageLen;
      
      // read in message from the packet one character at a time
      char message[messageLen + 1];
      for (i = 0; i <= messageLen; i++) packet >> message[i];
      
      // translating string with Translation Table
      gTranslator.ServerToClient (message);

      if ((u = getUserByUIN(checkUin)) == NULL) 
      { 
         gLog.Info("%sMessage from unknown user (%ld) adding them to your list.\n", 
                  L_BLANKxSTR, checkUin);
         usrAdd(checkUin, NULL);
         u = getUserByUIN(checkUin);
      }
      
      playSound(getSoundMsg());
      
      CEventMsg *e = new CEventMsg(message, ICQ_CMDxRCV_SYSxMSGxONLINE, 
                                   timeSent, nMask);
      u->AddEvent(e);
      e->AddToHistory(u, &icqOwner, D_RECEIVER);
      m_cUsers.reorder(u);
      emit signal_updatedUsers();
      break;
   }
   case ICQ_CMDxSUB_URL:  // system message: url through the server 
   {
      gLog.Info("%sUrl through server from %ld.\n", L_BLANKxSTR, checkUin);

      packet >> messageLen;

      // read in message from the packet one character at a time
      char message[messageLen];
      for (i = 0; i < messageLen; i++) packet >> message[i];

      // parse the message into url and url description
      char **szUrl = new char*[2];  // description, url
      ParseFE(message, &szUrl, 2);
      
      // translating string with Translation Table
      gTranslator.ServerToClient (szUrl[0]);

      if ((u = getUserByUIN(checkUin)) == NULL) 
      { 
         gLog.Info("%sUrl from unknown user (%ld) adding them to your list.\n", 
                  L_BLANKxSTR, checkUin);
         usrAdd(checkUin, NULL);
         u = getUserByUIN(checkUin);
      }

      playSound(getSoundUrl());

      CEventUrl *e = new CEventUrl(szUrl[1], szUrl[0], 
                                   ICQ_CMDxRCV_SYSxMSGxONLINE, timeSent, nMask);
      u->AddEvent(e);
      e->AddToHistory(u, &icqOwner, D_RECEIVER);

      m_cUsers.reorder(u);
      emit signal_updatedUsers();
      delete[] szUrl;
      break;
   }
   case ICQ_CMDxSUB_REQxAUTH:  // system message: authorisation request
   {
      /* 02 00 04 01 08 00 8F 76 20 00 06 00 41 00 41 70 6F 74 68 65 6F 73 69 73
         FE 47 72 61 68 61 6D FE 52 6F 66 66 FE 67 72 6F 66 66 40 75 77 61 74 65
         72 6C 6F 6F 2E 63 61 FE 31 FE 50 6C 65 61 73 65 20 61 75 74 68 6F 72 69 
         7A 65 20 6D 65 2E 00 */

      gLog.Info("%sAuthorization request from %ld.\n", L_BLANKxSTR, checkUin);
      unsigned short infoLen;
      packet >> infoLen;
      char message[infoLen + 1];
      for (i = 0; i < infoLen; i++) packet >> message[i];
      
      char **szFields = new char*[6];  // alias, first name, last name, email, auth, reason
      ParseFE(message, &szFields, 6);
      
      // translating string with Translation Table
      gTranslator.ServerToClient (szFields[0]);
      gTranslator.ServerToClient (szFields[1]);
      gTranslator.ServerToClient (szFields[2]);
      gTranslator.ServerToClient (szFields[5]);
      
      CEventAuth *e = new CEventAuth(checkUin, szFields[0], szFields[1], 
                                     szFields[2], szFields[3], szFields[5],
                                     ICQ_CMDxRCV_SYSxMSGxONLINE, timeSent, 0);
      icqOwner.AddEvent(e);
      e->AddToHistory(NULL, &icqOwner, D_RECEIVER);
      
      emit signal_updatedUsers();
      delete[] szFields;
      break;
   }
   case ICQ_CMDxSUB_ADDEDxTOxLIST:  // system message: added to a contact list
   {
      gLog.Info("%sUser %ld added you to their contact list.\n", L_BLANKxSTR, 
               checkUin);

      unsigned short infoLen;
      packet >> infoLen;

      // read in the user data from the packet
      char userInfo[infoLen];
      for (i = 0; i < infoLen; i++) packet >> userInfo[i];
      char **szFields = new char*[5]; // alias, first name, last name, email, auth
      ParseFE(userInfo, &szFields, 5);
            
      // translating string with Translation Table
      gTranslator.ServerToClient(szFields[0]);  // alias
      gTranslator.ServerToClient(szFields[1]);  // first name
      gTranslator.ServerToClient(szFields[2]);  // last name

      gLog.Info("%s%s (%s %s), %s\n", L_BLANKxSTR, szFields[0], szFields[1], 
               szFields[2], szFields[3]); 
                       
      CEventAdded *e = new CEventAdded(checkUin, szFields[0], szFields[1], 
                                       szFields[2], szFields[3], 
                                       ICQ_CMDxRCV_SYSxMSGxONLINE, timeSent, 0);
      icqOwner.AddEvent(e);
      e->AddToHistory(NULL, &icqOwner, D_RECEIVER);
      
      emit signal_updatedUsers();
      delete[] szFields;
      break;
   }
   case ICQ_CMDxSUB_WEBxPANEL:
   {
     /* 02 00 04 01 28 00 0A 00 00 00 0D 00 49 00 6D 79 20 6E 61 6D 65 FE FE FE 
        65 6D 61 69 6C FE 33 FE 53 65 6E 64 65 72 20 49 50 3A 20 32 30 39 2E 32 
        33 39 2E 36 2E 31 33 0D 0A 53 75 62 6A 65 63 74 3A 20 73 75 62 6A 65 63 
        74 0D 0A 74 68 65 20 6D 65 73 73 61 67 65 00 */
     gLog.Info("%sMessage through web panel.\n", L_BLANKxSTR);
     
     unsigned short nLen;
     packet >> nLen;

     // read in the user data from the packet
     char szMessage[nLen];
     for (i = 0; i < nLen; i++) packet >> szMessage[i];
     char **szFields = new char*[6]; // name, ?, ?, email, ?, message  
     ParseFE(szMessage, &szFields, 6);

     gTranslator.ServerToClient(szFields[0]);
     gTranslator.ServerToClient(szFields[5]);

     gLog.Info("%sFrom %s (%s).\n", L_BLANKxSTR, szFields[0], szFields[3]);
     CEventWebPanel *e = new CEventWebPanel(szFields[0], szFields[3], szFields[5], 
                                            ICQ_CMDxRCV_SYSxMSGxONLINE, timeSent, 0);

     icqOwner.AddEvent(e);
     e->AddToHistory(NULL, &icqOwner, D_RECEIVER);

     emit signal_updatedUsers();
     delete[] szFields;
     break;
   }
   case ICQ_CMDxSUB_EMAILxPAGER:
   {
     /* 02 00 04 01 07 00 0A 00 00 00 0E 00 5C 00 73 61 6D 40 65 75 2E 6F 72 67
        FE FE FE 73 61 6D 40 65 75 2E 6F 72 67 FE 33 FE 53 75 62 6A 65 63 74 3A
        20 5B 53 4C 41 53 48 44 4F 54 5D 0D 0A 5B 54 68 65 20 49 6E 74 65 72 6E
        65 74 5D 20 45 6E 6F 72 6D 6F 75 73 20 38 30 73 20 54 65 78 74 66 69 6C
        65 20 41 72 63 68 69 76 65 00 */
     gLog.Info("%sEmail pager message.\n", L_BLANKxSTR);
     
     unsigned short nLen;
     packet >> nLen;

     // read in the user data from the packet
     char szMessage[nLen];
     for (i = 0; i < nLen; i++) packet >> szMessage[i];
     char **szFields = new char*[6]; // name, ?, ?, email, ?, message  
     ParseFE(szMessage, &szFields, 6);

     gTranslator.ServerToClient(szFields[0]);
     gTranslator.ServerToClient(szFields[5]);

     gLog.Info("%sFrom %s (%s).\n", L_BLANKxSTR, szFields[0], szFields[3]);
     CEventEmailPager *e = new CEventEmailPager(szFields[0], szFields[3], szFields[5], 
                                                ICQ_CMDxRCV_SYSxMSGxONLINE, timeSent, 0);

     icqOwner.AddEvent(e);
     e->AddToHistory(NULL, &icqOwner, D_RECEIVER);

     emit signal_updatedUsers();
     delete[] szFields;
     break;
   }
   default:
   {
      gLog.Unknown("%sUnknown system message (%04x):\n%s\n", L_UNKNOWNxSTR, 
                  newCommand, packet.print());

      CEventUnknown *e = new CEventUnknown(&packet, newCommand, ICQ_CMDxRCV_SYSxMSGxONLINE,
                                           0, timeSent, 0);
      icqOwner.AddEvent(e);
      e->AddToHistory(NULL, &icqOwner, D_RECEIVER);      
   }
   }
}


//-----ICQ::ackUDP---------------------------------------------------------------------------------
void ICQ::ackUDP(unsigned short theSequence)
// acknowledge whatever packet we received using the relevant sequence number
{
   CPU_Ack p(theSequence);
   eventStart(udpServer, p, false);
}




//-----ICQ::recvUDP--------------------------------------------------------------------------------
void ICQ::recvUDP(int)
{
   // mirabilis contacts us using udp on this server
   udpServer.RecvRaw();
   processUdpPacket(udpServer.RecvBuffer());
   udpServer.ClearRecvBuffer();
}


