#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <iostream.h>

#include "time-fix.h"

#include "icq.h"

//-----ICQ::sendMessage----------------------------------------------------------------------------
ICQEvent *ICQ::sendMessage(ICQUser *u, const char *m, bool online, unsigned long _nSourceUin = 0)
{
   ICQEvent *result = NULL;
   char *mDos = gTranslator.NToRN(m);
   gTranslator.ClientToServer(mDos);
   CEventMsg *e = NULL; 

   if (!online) // send offline
   {
      e = new CEventMsg(m, ICQ_CMDxSND_THRUxSERVER, TIME_NOW, INT_VERSION);
      CPU_ThroughServer p(_nSourceUin, u->getUin(), ICQ_CMDxSUB_MSG, mDos);
      gLog.Info("%sSending message through server (#%d).\n", L_UDPxSTR, p.getSequence());
      result = eventStart(udpServer, p, true, u->getUin(), e);
   }
   else         // send direct
   {
      if (!connectToUser(u))
         result = NULL;
      else
      {
         e = new CEventMsg(m, ICQ_CMDxTCP_START, TIME_NOW, E_DIRECT | INT_VERSION);
         CPT_Message p(_nSourceUin, mDos, u);
         gLog.Info("%sSending message to %s (#%d).\n", L_TCPxSTR, u->getAlias(), p.getSequence());
         result = eventStart(u->tcpSocket, p, true, u->getUin(), e);
      }
   }
   
   delete mDos;
   return (result);
}


//-----ICQ::sendReadAwayMsg------------------------------------------------------------------------
ICQEvent *ICQ::sendReadAwayMsg(ICQUser *u, bool online, unsigned long _nSourceUin = 0)
{
   ICQEvent *result = NULL;

   if (!online) // send offline
   {
      CPU_ThroughServer p(_nSourceUin, u->getUin(), ICQ_CMDxTCP_READxAWAYxMSG, "");
      gLog.Info("%sFetching away message through server (#%d).\n", L_UDPxSTR, 
               p.getSequence());
      result = eventStart(udpServer, p, true, u->getUin());
   }
   else
   {
      if (!connectToUser(u)) 
         result = NULL;
      else
      {
         CPT_ReadAwayMessage p(_nSourceUin, u);
         gLog.Info("%sRequesting away message from %s (#%d).\n", L_TCPxSTR, 
                  u->getAlias(), p.getSequence());
         result = eventStart(u->tcpSocket, p, true, u->getUin());
      }      
   }
     
   return (result);   
}


//-----ICQ::sendUrl--------------------------------------------------------------------------------
ICQEvent *ICQ::sendUrl(ICQUser *u, const char *url, const char *description, bool online, unsigned long _nSourceUin = 0)
{
   // make the URL info string
  char *szDescDos = gTranslator.NToRN(description); 
  gTranslator.ClientToServer(szDescDos);
  char m[strlen(url) + strlen(szDescDos) + 2];
  CEventUrl *e = NULL;
   
  strcpy(m, szDescDos);
  int nLen = strlen(m);
  m[nLen] = (char)0xFE;
  m[nLen + 1] = '\0';
  strcat(m, url);
   
   ICQEvent *result = NULL;

   if (!online) // send offline
   {
      e = new CEventUrl(url, description, ICQ_CMDxSND_THRUxSERVER, TIME_NOW, INT_VERSION);
      CPU_ThroughServer p(_nSourceUin, u->getUin(), ICQ_CMDxSUB_URL, m);
      gLog.Info("%sSending url through server (#%d).\n", L_UDPxSTR, p.getSequence());
      result = eventStart(udpServer, p, true, u->getUin(), e);
   }
   else
   {
      if (!connectToUser(u)) 
         result = NULL;
      else
      {
         e = new CEventUrl(url, description, ICQ_CMDxTCP_START, TIME_NOW, E_DIRECT | INT_VERSION);
         CPT_Url p(_nSourceUin, m, u);
         gLog.Info("%sSending URL to %s (#%d).\n", L_TCPxSTR, u->getAlias(), p.getSequence());
         result = eventStart(u->tcpSocket, p, true, u->getUin(), e);
      }      
   }
   
   delete szDescDos;
   return (result);
}


//-----ICQ::sendFile------------------------------------------------------------
ICQEvent *ICQ::sendFile(ICQUser *u, const char *_szFilename, 
                        const char *_szDescription, bool online, 
                        unsigned long _nSourceUin = 0)
{
  ICQEvent *result = NULL;
  char *szDosDesc = gTranslator.NToRN(_szDescription);
  gTranslator.ClientToServer(szDosDesc);
  CEventFile *e = NULL;
  online = true;

  if (!connectToUser(u)) 
    result = NULL;
  else
  {
    CPT_FileTransfer p(_nSourceUin, _szFilename, szDosDesc, u);
    if (!p.IsValid()) 
      result = NULL;
    else
    {
      e = new CEventFile(_szFilename, p.GetDescription(), p.GetFileSize(), 
                         p.getSequence(), TIME_NOW, E_DIRECT | INT_VERSION);
      gLog.Info("%sSending file transfer to %s (#%d).\n", L_TCPxSTR, 
               u->getAlias(), p.getSequence());
      result = eventStart(u->tcpSocket, p, true, u->getUin(), e);
    }
  }
    
  delete szDosDesc;
  return (result);
}


//-----ICQ::fileCancel-------------------------------------------------------------------------
void ICQ::fileCancel(ICQUser *_cUser, unsigned long _nSequence, bool online, unsigned long id = 0)
{
  // add to history ??
  online = true; id = 0;
  gLog.Info("%sCancelling file transfer to %s (#%d).\n", L_TCPxSTR, _cUser->getAlias(), _nSequence);
  CPT_CancelFile p(_nSequence, _cUser);
  ackTCP(p, _cUser);
}


//-----ICQ::fileAccept-----------------------------------------------------------------------------
void ICQ::fileAccept(ICQUser *_cUser, unsigned short _nPort, unsigned long _nSequence)
{
   // basically a fancy tcp ack packet which is sent late
   // add to history ??
   gLog.Info("%sAccepting file transfer from %s (#%d).\n", L_TCPxSTR, _cUser->getAlias(), _nSequence);
   CPT_AckFileAccept p(_nPort, _nSequence, _cUser);
   ackTCP(p, _cUser);
}



//-----ICQ::chatRefuse-----------------------------------------------------------------------------
void ICQ::fileRefuse(ICQUser *_cUser, const char *_sReason, unsigned long _nSequence)
{
   // add to history ??
   gLog.Info("%sRefusing file transfer from %s (#%d).\n", L_TCPxSTR, _cUser->getAlias(), _nSequence);
   char *szReasonDos = gTranslator.NToRN(_sReason);
   gTranslator.ClientToServer(szReasonDos);
   CPT_AckFileRefuse p(szReasonDos, _nSequence, _cUser);
   ackTCP(p, _cUser);
   delete szReasonDos;
}


//-----ICQ::sendChat-------------------------------------------------------------------------------
ICQEvent *ICQ::sendChat(ICQUser *u, const char *reason, bool online, unsigned long _nSourceUin = 0)
{
  ICQEvent *result = NULL;
  char *szReasonDos = gTranslator.NToRN(reason);
  gTranslator.ClientToServer(szReasonDos);
  CEventChat *e = NULL;
  online = true;

  if (!connectToUser(u)) 
     result = NULL;
  else
  {
     CPT_ChatRequest p(_nSourceUin, szReasonDos, u);
     e = new CEventChat(reason, p.getSequence(), TIME_NOW, E_DIRECT | INT_VERSION);
     gLog.Info("%sSending chat request to %s (#%d).\n", L_TCPxSTR, u->getAlias(), p.getSequence());
     result = eventStart(u->tcpSocket, p, true, u->getUin(), e);
  }
  
  delete szReasonDos;    
  return (result);
}


//-----ICQ::chatCancel-------------------------------------------------------------------------
void ICQ::chatCancel(ICQUser *_cUser, unsigned long _nSequence, bool online, unsigned long id = 0)
{
   // add to history ??
  online = true; id = 0;
  gLog.Info("%sCancelling chat request with %s (#%d).\n", L_TCPxSTR, _cUser->getAlias(), _nSequence);
  CPT_CancelChat p(_nSequence, _cUser);
  ackTCP(p, _cUser);
}


//-----ICQ::chatRefuse-----------------------------------------------------------------------------
void ICQ::chatRefuse(ICQUser *_cUser, const char *_sReason, unsigned long _nSequence)
{
   // add to history ??
   gLog.Info("%sRefusing chat request with %s (#%d).\n", L_TCPxSTR, _cUser->getAlias(), _nSequence);
   char *szReasonDos = gTranslator.NToRN(_sReason);
   gTranslator.ClientToServer(szReasonDos);   
   CPT_AckChatRefuse p(szReasonDos, _nSequence, _cUser);
   ackTCP(p, _cUser);
   delete szReasonDos;
}


//-----ICQ::chatAccept-----------------------------------------------------------------------------
void ICQ::chatAccept(ICQUser *_cUser, unsigned short _nPort, unsigned long _nSequence)
{
   // basically a fancy tcp ack packet which is sent late
   // add to history ??
   gLog.Info("%sAccepting chat request with %s (#%d).\n", L_TCPxSTR, _cUser->getAlias(), _nSequence);
   CPT_AckChatAccept p(_nPort, _nSequence, _cUser);
   ackTCP(p, _cUser);
}


//-----ICQ::connectToUser-------------------------------------------------------
bool ICQ::connectToUser(ICQUser *u)
{
   if (u->tcpSocket.Connected()) return (true);
   
   gLog.Info("%sConnecting to %s (%d) on port %d.\n", L_TCPxSTR, u->getAlias(), u->getUin(), u->tcpSocket.RemotePort());
   
   if (!u->tcpSocket.OpenConnection())
   { 
      gLog.Warn("%sConnect failed:\n%s%s.\n", L_WARNxSTR, L_BLANKxSTR, u->tcpSocket.ErrorStr()); 
      return (false); 
   }
   
   connect(u->tcpSocket.sn, SIGNAL(activated(int)), this, SLOT(recvTcp(int)));
   gLog.Info("%sShaking hands with %s.\n", L_TCPxSTR, u->getAlias());
   
   CPacketTcp_Handshake packet(u->tcpSocket.LocalPort());
   if (!u->tcpSocket.SendPacket(*packet.getBuffer())) 
   { 
      gLog.Warn("%sHandshake failed:\n%s%s.\n", L_WARNxSTR, L_BLANKxSTR, u->tcpSocket.ErrorStr()); 
      return(false); 
   }
   return(true);
}


//-----ICQ::processTcpPacket-----------------------------------------------------------------------
void ICQ::processTcpPacket(CBuffer &packet, int sockfd)
{
   unsigned long checkUin, theSequence, senderIp, localIp, userStatus, 
                 senderPort, junkLong, nPort, nPortReversed;
   unsigned short version, command, junkShort, newCommand, messageLen, licqVersion;
   char licqChar, junkChar;
   ICQUser *u;
   static unsigned long chatUin, chatSequence, nFileUin, nFileSequence;
   
   packet >> checkUin
          >> version
          >> command      // main tcp command (start, cancel, ack)
          >> junkShort    // 00 00 to fill in the MSB of the command long int which is read in as a short
          >> checkUin
          >> newCommand   // sub command (message/chat/read away message/...)
          >> messageLen   // length of incoming message
   ;
   
   // read in the message
   char message[messageLen + 1];
   for (unsigned short i = 0; i < messageLen; i++) packet >> message[i];
   
   // read in some more stuff common to all tcp packets
   packet >> senderIp
          >> localIp
          >> senderPort
          >> junkChar      // whether use can receive tcp packets directly
          >> userStatus
   ;
   
   senderIp = PacketIpToNetworkIp(senderIp);
   localIp = PacketIpToNetworkIp(localIp);
      
   // find which user was sent
   if ((u = getUserByUIN(checkUin)) == NULL) 
   { 
      gLog.Info("%sUnknown user (UIN %d) adding them to your list.\n", L_TCPxSTR, checkUin);
      usrAdd(checkUin, NULL);
      u = getUserByUIN(checkUin);
   }

   unsigned long nMask = E_DIRECT | 
                         (newCommand | ICQ_CMDxSUB_FxMULTIREC ? E_MULTIxREC : 0);
   newCommand &= ~ICQ_CMDxSUB_FxMULTIREC;

   switch(command)
   {
   case ICQ_CMDxTCP_START:   
      switch(newCommand)  
      {
      case ICQ_CMDxSUB_MSG:  // straight message from a user
      {
         gLog.Info("%sMessage from %s (%d).\n", L_TCPxSTR, u->getAlias(), checkUin);
         
         packet >> theSequence >> licqChar >> licqVersion;
         //if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);
         
         CPT_AckMessage p(theSequence, true, u);
         ackTCP(p, u);
         playSound(getSoundMsg());
         
         // translating string with translation table    
         gTranslator.ServerToClient (message);
         
         CEventMsg *e = new CEventMsg(message, ICQ_CMDxTCP_START, TIME_NOW, 
                                      nMask | licqVersion);
         u->AddEvent(e);
         e->AddToHistory(u, &icqOwner, D_RECEIVER);
         m_cUsers.reorder(u);
         emit signal_updatedUsers();
         break;
      }
      case ICQ_CMDxTCP_READxNAxMSG:
      case ICQ_CMDxTCP_READxDNDxMSG:
      case ICQ_CMDxTCP_READxOCCUPIEDxMSG:
      case ICQ_CMDxTCP_READxAWAYxMSG:  // read away message
      {
         /* 76 1E 3F 00 03 00 EE 07 00 00 76 1E 3F 00 E8 03 01 00 00 81 61 1D 9D 81 61
            1D 9D C9 05 00 00 04 00 00 10 00 FE FF FF FF */
         gLog.Info("%s%s (%d) requested read of away message.\n", L_TCPxSTR, 
                               u->getAlias(), checkUin);
         
         packet >> theSequence >> licqChar >> licqVersion;
         //if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);
         
         CPT_AckReadAwayMessage p(newCommand, theSequence, true, u);
         ackTCP(p, u);
         break;
      }
      case ICQ_CMDxSUB_URL:  // url sent
      {
         /* BA 95 47 00 03 00 EE 07 00 00 BA 95 47 00 04 00 24 00 67 6F 6F 64 20 70 6F 
            72 6E 20 73 69 74 65 FE 68 74 74 70 3A 2F 2F 63 6F 6F 6C 70 6F 72 74 6E 2E 
            63 6F 6D 00 81 61 1D 9E 7F 00 00 01 3F 07 00 00 04 00 00 10 00 03 00 00 00 */
         gLog.Info("%sURL from %s (%d).\n", L_TCPxSTR, u->getAlias(), checkUin);
         
         packet >> theSequence >> licqChar >> licqVersion;
         //if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);
         
         CPT_AckUrl p(theSequence, true, u);
         ackTCP(p, u);
         
         // parse the message into url and url description
         char **szUrl = new char*[2]; // desc, url
         ParseFE(message, &szUrl, 2);
         
         // translating string with Translation Table
         gTranslator.ServerToClient(szUrl[0]);
         
         // format the url and url description into a message and add it to the users list
         playSound(getSoundUrl());
         CEventUrl *e = new CEventUrl(szUrl[1], szUrl[0], ICQ_CMDxTCP_START, 
                                      TIME_NOW, nMask | licqVersion);
         u->AddEvent(e);
         e->AddToHistory(u, &icqOwner, D_RECEIVER);
         m_cUsers.reorder(u);
         emit signal_updatedUsers();

         delete []szUrl;
         break;
      }
      case ICQ_CMDxSUB_CHAT:
      {
         /* 50 A5 82 00 03 00 EE 07 00 00 50 A5 82 00 02 00 0D 00 63 68 61 74 20 72
            65 71 75 65 73 74 00 CF 60 AD D3 CF 60 AD D3 28 12 00 00 04 00 00 10 00
            01 00 00 00 00 00 00 00 00 00 00 06 00 00 00 */
         gLog.Info("%sChat request from %s (%d).\n", L_TCPxSTR, u->getAlias(), checkUin);
         
         packet >> junkLong >> junkLong >> junkShort >> junkChar >> theSequence 
                >> licqChar >> licqVersion;
         //if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);

         // translating string with translation table    
         gTranslator.ServerToClient (message);
         
         playSound(getSoundChat());
         
         CEventChat *e = new CEventChat(message, theSequence, TIME_NOW, 
                                        nMask | licqVersion);
         u->AddEvent(e);
         e->AddToHistory(u, &icqOwner, D_RECEIVER);
         
         m_cUsers.reorder(u);
         emit signal_updatedUsers();
         break;
      }
      case ICQ_CMDxSUB_FILE:
      {
         /* 50 A5 82 00 03 00 EE 07 00 00 50 A5 82 00 03 00 0F 00 74 68 69 73 20 69
            73 20 61 20 66 69 6C 65 00 CF 60 AD D3 CF 60 AD D3 60 12 00 00 04 00 00
            10 00 00 00 00 00 09 00 4D 61 6B 65 66 69 6C 65 00 55 0C 00 00 00 00 00
            00 04 00 00 00 */
         gLog.Info("%sFile transfer request from %s (%d).\n", L_TCPxSTR, 
                  u->getAlias(), checkUin);
         
         unsigned short nLenFilename;
         unsigned long nFileLength;
         packet >> junkLong 
                >> nLenFilename;
         char szFilename[nLenFilename];
         for (unsigned short i = 0; i < nLenFilename; i++) 
            packet >> szFilename[i];
         packet >> nFileLength
                >> junkLong 
                >> theSequence >> licqChar >> licqVersion;
         //if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);

         // translating string with translation table    
         gTranslator.ServerToClient (message);

         playSound(getSoundFile());
         CEventFile *e = new CEventFile(szFilename, message, nFileLength,
                                        theSequence, TIME_NOW, 
                                        nMask | licqVersion);
         u->AddEvent(e);
         e->AddToHistory(u, &icqOwner, D_RECEIVER);

         m_cUsers.reorder(u);
         emit signal_updatedUsers();
         break;      
      }
      default:
         break;
      }
      break;

   case ICQ_CMDxTCP_ACK:  // message received packet      
      
      switch (newCommand) 
      {
      case ICQ_CMDxSUB_MSG:
         /* 8F 76 20 00 03 00 DA 07 00 00 8F 76 20 00 01 00 01 00 00 CF 60 AD D3 7F 
            00 00 01 5A 12 00 00 04 00 00 00 00 14 00 00 00 */
         // read in the sequence, assume it is the last long in the packet (it should be)
         packet >> theSequence;
         break;
      case ICQ_CMDxTCP_READxNAxMSG:
      case ICQ_CMDxTCP_READxDNDxMSG:
      case ICQ_CMDxTCP_READxOCCUPIEDxMSG:
      case ICQ_CMDxTCP_READxAWAYxMSG:
         packet >> theSequence;
         break;
      case ICQ_CMDxSUB_URL:
         packet >> theSequence;
         break;
      case ICQ_CMDxSUB_CHAT:
         /* 50 A5 82 00 03 00 DA 07 00 00 50 A5 82 00 02 00 03 00 6E 6F 00 CF 60 AD
            95 CF 60 AD 95 1E 3C 00 00 04 01 00 00 00 01 00 00 00 00 00 00 00 00 00
            00 01 00 00 00 */
         packet >> junkShort 
                >> junkChar
                >> nPortReversed   // port backwards
                >> nPort    // port to connect to for chat
                >> theSequence
         ;
         // Apparently this line helps, Taral sent it in
         if (nPort == 0) nPort = (nPortReversed >> 8) | ((nPortReversed & 0xFF) << 8);
         // only if this is the first chat ack packet do we do anything
         if (chatSequence == theSequence && chatUin == checkUin) return;
         chatSequence = theSequence;
         chatUin = checkUin;
         m_nEventPort = nPort;
         m_bEventAccepted = (userStatus != ICQ_TCPxACK_REFUSE);
         strncpy(m_szEventMessage, message, MAX_MESSAGE_SIZE);
         break;

      case ICQ_CMDxSUB_FILE:
         /* 50 A5 82 00 03 00 DA 07 00 00 50 A5 82 00 03 00 0A 00 6E 6F 20 74 68 61 
            6E 6B 73 00 D1 EF 04 9F 7F 00 00 01 4A 1F 00 00 04 01 00 00 00 00 00 00 
            00 01 00 00 00 00 00 00 00 00 00 00 03 00 00 00 */
         packet >> nPortReversed
                >> junkShort;
         for (int i = 0; i < junkShort; i++) packet >> junkChar;
         packet >> junkLong
                >> nPort
                >> theSequence;
         // Some clients only send the first port (reversed)
         if (nPort == 0) nPort = (nPortReversed >> 8) | ((nPortReversed & 0xFF) << 8);
         // only if this is the first file ack packet do we do anything
         if (nFileSequence == theSequence && nFileUin == checkUin) return;
         nFileSequence = theSequence;
         nFileUin = checkUin;
         m_nEventPort = nPort;
         m_bEventAccepted = (userStatus != ICQ_TCPxACK_REFUSE);
         strncpy(m_szEventMessage, message, MAX_MESSAGE_SIZE);
         break;
      default: break;
      }
      
      // translating string with translation table    
      gTranslator.ServerToClient (message);
      // output the away message if there is one (ie if user status is not online)
      if (userStatus << 24 == 0x00)
         gLog.Info("%sAck from %s (#%d).\n", L_TCPxSTR, u->getAlias(), theSequence);
      else if (userStatus << 24 == 0x01) 
         gLog.Info("%sRefusal from %s (#%d): %s\n", L_TCPxSTR, u->getAlias(), 
                  theSequence, message);
      else
      {
         // Update the away message if it's changed
         if (strcmp(u->getAwayMessage(), message) != 0)
         {
            u->setAwayMessage(message);
            u->setShowAwayMsg(true);
         }
         gLog.Info("%sAck from %s (#%d) with auto-response.\n", L_TCPxSTR, u->getAlias(), theSequence);
      }
      
      eventDone(true, sockfd, theSequence);
      break;
      
   case ICQ_CMDxTCP_CANCEL:
      switch (newCommand) 
      {
      case ICQ_CMDxSUB_CHAT:
      {
         /* 50 A5 82 00 03 00 D0 07 00 00 50 A5 82 00 02 00 01 00 00 CF 60 AD D3 CF
            60 AD D3 28 12 00 00 04 00 00 00 00 01 00 00 00 00 00 00 00 00 00 00 06
            00 00 00 */
         gLog.Info("%sChat request from %s (%ld) cancelled.\n", L_TCPxSTR, 
                  u->getAlias(), checkUin);
         CEventChatCancel *e = new CEventChatCancel(0, TIME_NOW, E_DIRECT);
         u->AddEvent(e);
         e->AddToHistory(u, &icqOwner, D_RECEIVER);

         m_cUsers.reorder(u);
         emit signal_updatedUsers();
         break;
      }
      case ICQ_CMDxSUB_FILE:
      {
         /* 50 A5 82 00 03 00 D0 07 00 00 50 A5 82 00 02 00 01 00 00 CF 60 AD D3 CF
            60 AD D3 28 12 00 00 04 00 00 00 00 01 00 00 00 00 00 00 00 00 00 00 06
            00 00 00 */
         gLog.Info("%sFile transfer request from %s (%ld) cancelled.\n", 
                  L_TCPxSTR, u->getAlias(), checkUin);
         CEventFileCancel *e = new CEventFileCancel(0, TIME_NOW, E_DIRECT);
         u->AddEvent(e);
         e->AddToHistory(u, &icqOwner, D_RECEIVER);
         
         m_cUsers.reorder(u);
         emit signal_updatedUsers();
         break;
      }
      default:
         break;
      }
      break;
   
   default:
      gLog.Unknown("%sUnknown TCP packet (command %d):\n%s\n", L_UNKNOWNxSTR, command, packet.print());
      break;
   }
} 



//-----ICQ::ackTCP---------------------------------------------------------------------------------
void ICQ::ackTCP(CPacketTcp &p, ICQUser *u)
{
   eventStart(u->tcpSocket, p, false);
}


//-----ICQ::recvNewTCP----------------------------------------------------------
void ICQ::recvNewTcp(int)
{
   // our tcp incoming server
   TCPSocket *newSocket = new TCPSocket;
   icqOwner.tcpSocket.RecvConnection(*newSocket);
   m_lxUnclaimedSockets.push_front(newSocket);
   connect(newSocket->sn, SIGNAL(activated(int)), this, SLOT(recvTcpHandshake(int)));
}


//-----recvHandshake------------------------------------------------------------
void ICQ::recvTcpHandshake(int _nSD)
{
   // Find which socket this is
   list<TCPSocket *>::iterator iter;
   for (iter = m_lxUnclaimedSockets.begin(); iter != m_lxUnclaimedSockets.end(); iter++) 
   {
      if ((*iter)->Descriptor() == _nSD) break;
   }
   // If this is not in the list, then go away
   if (iter == m_lxUnclaimedSockets.end()) return;

   // Otherwise copy the pointer, remove it from the list and disconnect the signal
   TCPSocket *newUserSocket = *iter;
   if (!newUserSocket->RecvPacket())
   {
      gLog.Warn("%sIncoming connection error on server socket:\n%s%s.\n", 
               L_WARNxSTR, L_BLANKxSTR, newUserSocket->ErrorStr());
      m_lxUnclaimedSockets.erase(iter);
      delete newUserSocket;
      return;
   }
   if (!newUserSocket->RecvBufferFull()) return;

   m_lxUnclaimedSockets.erase(iter);
   
   char cHandshake;
   unsigned long nVersion;
   newUserSocket->RecvBuffer() >> cHandshake >> nVersion;

   if ((unsigned char)cHandshake != ICQ_CMDxTCP_HANDSHAKE)
   {
      gLog.Unknown("%sUnknown TCP handshake packet (command = 0x%02X):\n%s\n", 
                   L_UNKNOWNxSTR, cHandshake, newUserSocket->RecvBuffer().print());
   }
   else
   {
      unsigned long ulJunk, newUin;//, localHost;
      newUserSocket->RecvBuffer() >> ulJunk >> newUin;
                                  //>> localHost >> localHost;
                                  //>> ulJunk >> ucJunk;
      gLog.Info("%sConnection from uin %d.\n", L_TCPxSTR, newUin);
      
      ICQUser *newUser = getUserByUIN(newUin);
      if (newUser != NULL)
      { 
         newUser->tcpSocket.TransferConnectionFrom(*newUserSocket);
         connect(newUser->tcpSocket.sn, SIGNAL(activated(int)), this, SLOT(recvTcp(int)));
      }
      else
      {
         gLog.Info("%sUnknown user (UIN %d) adding to list.\n", L_TCPxSTR, newUin);
         usrAdd(newUin, *newUserSocket);
         emit signal_updatedUsers();
      }
   }
   delete newUserSocket;
}


//-----ICQ::recvTCP-------------------------------------------------------------
void ICQ::recvTcp(int sockfd)
/* Need to add code in here to make this not O(n)...two ways: 1. make a new
 * type of QSocketNotifier that returns the uin as well, and then optimize
 * GetUserByUin to be non-O(n), or 2. make a new QSocketNotifier that returns 
 * a user pointer */
{
  for(unsigned short i = 0; i < getNumUsers(); i++)
      if (getUser(i)->tcpSocket.Descriptor() == sockfd)
      {
         ICQUser *u = getUser(i);
         if(!u->tcpSocket.RecvPacket())
         {
           if (u->tcpSocket.Error() == 0)
             gLog.Info("%sConnection to %s (%d) closed.\n", L_TCPxSTR, 
                        u->getAlias(), u->getUin());
           else
             gLog.Info("%sConnection to %s (%d) lost:\n%s%s.\n", L_TCPxSTR, 
                        u->getAlias(), u->getUin(), L_BLANKxSTR, u->tcpSocket.ErrorStr()); 
           return;
         }
         if (u->tcpSocket.RecvBufferFull()) 
         {
            processTcpPacket(u->tcpSocket.RecvBuffer(), sockfd);
            u->tcpSocket.ClearRecvBuffer();
         }         
         return;
      }
}

