/* libwmf (ipa/eps.c): library for wmf conversion
   Copyright (C) 2000 - various; see CREDITS, ChangeLog, and sources

   The libwmf Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The libwmf Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the libwmf Library; see the file COPYING.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

/* If compiling as module, rename global functions to a standard
 */
#ifdef WMF_IPA_MODULE
#define wmf_eps_device_open    wmf_device_open
#define wmf_eps_device_close   wmf_device_close
#define wmf_eps_device_begin   wmf_device_begin
#define wmf_eps_device_end     wmf_device_end
#define wmf_eps_flood_interior wmf_flood_interior
#define wmf_eps_flood_exterior wmf_flood_exterior
#define wmf_eps_draw_pixel     wmf_draw_pixel
#define wmf_eps_draw_pie       wmf_draw_pie
#define wmf_eps_draw_chord     wmf_draw_chord
#define wmf_eps_draw_arc       wmf_draw_arc
#define wmf_eps_draw_ellipse   wmf_draw_ellipse
#define wmf_eps_draw_line      wmf_draw_line
#define wmf_eps_poly_line      wmf_poly_line
#define wmf_eps_draw_polygon   wmf_draw_polygon
#define wmf_eps_draw_rectangle wmf_draw_rectangle
#define wmf_eps_rop_draw       wmf_rop_draw
#define wmf_eps_bmp_draw       wmf_bmp_draw
#define wmf_eps_bmp_read       wmf_bmp_read
#define wmf_eps_bmp_free       wmf_bmp_free
#define wmf_eps_draw_text      wmf_draw_text
#define wmf_eps_udata_init     wmf_udata_init
#define wmf_eps_udata_copy     wmf_udata_copy
#define wmf_eps_udata_set      wmf_udata_set
#define wmf_eps_udata_free     wmf_udata_free
#define wmf_eps_region_frame   wmf_region_frame
#define wmf_eps_region_paint   wmf_region_paint
#define wmf_eps_region_clip    wmf_region_clip
#endif /* WMF_IPA_MODULE */

#include <stdio.h>
#include <math.h>
#include <time.h>

#include "wmfdefs.h"

#include "libwmf/eps.h"

#include "ipa/eps.h"
#include "ipa/eps/bmp.h"
#include "ipa/eps/device.h"
#include "ipa/eps/draw.h"
#include "ipa/eps/region.h"

void wmf_eps_function (wmfAPI* API)
{	wmf_eps_t* ddata = 0;

	wmfFunctionReference* FR = (wmfFunctionReference*) API->function_reference;

/* IPA function reference links
 */
	FR->device_open    = wmf_eps_device_open;
	FR->device_close   = wmf_eps_device_close;
	FR->device_begin   = wmf_eps_device_begin;
	FR->device_end     = wmf_eps_device_end;
	FR->flood_interior = wmf_eps_flood_interior;
	FR->flood_exterior = wmf_eps_flood_exterior;
	FR->draw_pixel     = wmf_eps_draw_pixel;
	FR->draw_pie       = wmf_eps_draw_pie;
	FR->draw_chord     = wmf_eps_draw_chord;
	FR->draw_arc       = wmf_eps_draw_arc;
	FR->draw_ellipse   = wmf_eps_draw_ellipse;
	FR->draw_line      = wmf_eps_draw_line;
	FR->poly_line      = wmf_eps_poly_line;
	FR->draw_polygon   = wmf_eps_draw_polygon;
	FR->draw_rectangle = wmf_eps_draw_rectangle;
	FR->rop_draw       = wmf_eps_rop_draw;
	FR->bmp_draw       = wmf_eps_bmp_draw;
	FR->bmp_read       = wmf_eps_bmp_read;
	FR->bmp_free       = wmf_eps_bmp_free;
	FR->draw_text      = wmf_eps_draw_text;
	FR->udata_init     = wmf_eps_udata_init;
	FR->udata_copy     = wmf_eps_udata_copy;
	FR->udata_set      = wmf_eps_udata_set;
	FR->udata_free     = wmf_eps_udata_free;
	FR->region_frame   = wmf_eps_region_frame;
	FR->region_paint   = wmf_eps_region_paint;
	FR->region_clip	   = wmf_eps_region_clip;

/* Allocate device data structure
 */
	ddata = (wmf_eps_t*) wmf_malloc (API,sizeof (wmf_eps_t));

	if (ERR (API))
	{	WMF_DEBUG (API,"bailing...");
		return;
	}

	API->device_data = (void*) ddata;

/* Device data defaults
 */
	ddata->bbox.TL.x = 0;
	ddata->bbox.TL.y = 0;
	ddata->bbox.BR.x = 0;
	ddata->bbox.BR.y = 0;

	ddata->out = 0;

	ddata->offset = 0;

	ddata->Title = 0;
	ddata->Creator = 0;
	ddata->Date = 0;
	ddata->For = 0;

	ddata->eps_x = 0;
	ddata->eps_y = 0;
	ddata->eps_width = 0;
	ddata->eps_height = 0;

	ddata->page_width = 596;
	ddata->page_height = 842;

	ddata->flags = 0;
}

void wmf_eps_draw_text (wmfAPI* API,wmfDrawText_t* draw_text)
{	wmf_eps_t* ddata = WMF_EPS_GetData (API);

	FILE* out = ddata->out;

	float red;
	float green;
	float blue;
	float size;
	float ratio;
	float theta;

	unsigned int i;
	unsigned int length;

	WMF_DEBUG (API,"~~~~~~~~wmf_[eps_]draw_text");

	if (out == 0) return;

	size = (float) draw_text->font_height;

	ratio = (float) draw_text->font_ratio;

	theta = (float) (WMF_TEXT_ANGLE (draw_text->dc->font) * 180 / PI);

	fputs ("gsave % wmf_[eps_]draw_text\n",out);

	fprintf (out,"/%s findfont %f scalefont setfont\n",draw_text->dc->font->ps_name,size);

	fprintf (out,"%f %f translate 1 -1 scale %f rotate ",draw_text->pt.x,draw_text->pt.y,theta);
	fprintf (out,"%f 1 scale\n",ratio);

	fputs ("(",out);
	length = strlen (draw_text->str);
	for (i = 0; i < length; i++)
	{	if (draw_text->str[i] == ')') fputs ("\\)",out);
		else if (draw_text->str[i] == '(') fputs ("\\(",out);
		else fputc ((int) draw_text->str[i],out);
	}
	fputs (")\n",out);

	if (draw_text->dc->bgmode != TRANSPARENT)
	{	fputs ("dup stringwidth pop dup ",out);
		fprintf (out,"newpath 0 %f moveto 0 rlineto 0 %f rlineto neg 0 rlineto closepath ",
		         - 0.29 * size,1.07 * size);

		red   = (float) ((int) draw_text->dc->bgcolor.r) / 255;
		green = (float) ((int) draw_text->dc->bgcolor.g) / 255;
		blue  = (float) ((int) draw_text->dc->bgcolor.b) / 255;

		fprintf (out,"%f %f %f setrgbcolor fill ",red,green,blue);
	}

	red   = (float) ((int) draw_text->dc->textcolor.r) / 255;
	green = (float) ((int) draw_text->dc->textcolor.g) / 255;
	blue  = (float) ((int) draw_text->dc->textcolor.b) / 255;

	fprintf (out,"%f %f %f setrgbcolor ",red,green,blue);

	fputs ("0 0 moveto show\n",out);

	fputs ("grestore\n",out);
}

void wmf_eps_udata_init (wmfAPI* API,wmfUserData_t* user_data)
{	/* wmf_eps_t* ddata = WMF_EPS_GetData (API); */

	WMF_DEBUG (API,"~~~~~~~~wmf_[eps_]udata_init");

	
}

void wmf_eps_udata_copy (wmfAPI* API,wmfUserData_t* user_data)
{	/* wmf_eps_t* ddata = WMF_EPS_GetData (API); */

	WMF_DEBUG (API,"~~~~~~~~wmf_[eps_]udata_copy");

	
}

void wmf_eps_udata_set (wmfAPI* API,wmfUserData_t* user_data)
{	/* wmf_eps_t* ddata = WMF_EPS_GetData (API); */

	WMF_DEBUG (API,"~~~~~~~~wmf_[eps_]udata_set");

	
}

void wmf_eps_udata_free (wmfAPI* API,wmfUserData_t* user_data)
{	/* wmf_eps_t* ddata = WMF_EPS_GetData (API); */

	WMF_DEBUG (API,"~~~~~~~~wmf_[eps_]udata_free");

	
}

/* Write out postscript path, *minus* the ultimate `fill', then call eps_path_fill
 * Note: I KNOW there is a better way to do patterns, but my ghostscript hangs when I try!
 * 
 * Assumes that lbStyle != BS_NULL
 */
static void eps_path_fill (wmfAPI* API,wmfDC* dc,wmfD_Rect* bbox)
{	wmf_eps_t* ddata = WMF_EPS_GetData (API);

	FILE* out = ddata->out;

	float red;
	float green;
	float blue;

	float side;

	wmfBrush* brush;

	wmfRGB rgb;

	WMF_DEBUG (API,"~~~~~~~~eps_path_fill");

	if (out == 0) return;

	brush = dc->brush;

	switch (brush->lbStyle)
	{
	case BS_NULL:
		WMF_ERROR (API,"Attempt to set null fill-style!");
		API->err = wmf_E_Glitch;
	break;

	case BS_HATCHED:
		fputs ("clip ",out);

		if (dc->bgmode != TRANSPARENT)
		{	rgb = dc->bgcolor;

			red   = (float) ((int) rgb.r) / 255;
			green = (float) ((int) rgb.g) / 255;
			blue  = (float) ((int) rgb.b) / 255;

			fprintf (out,"%f %f %f setrgbcolor ",red,green,blue);

			fputs ("fill ",out);
		}

		fputs ("\n",out);
		fputs ("1 setlinewidth ",out);
		fputs ("[] 0 setdash ",out);

		rgb = brush->lbColor;

		red   = (float) ((int) rgb.r) / 255;
		green = (float) ((int) rgb.g) / 255;
		blue  = (float) ((int) rgb.b) / 255;

		fprintf (out,"%f %f %f setrgbcolor\n",red,green,blue);

		switch (brush->lbHatch)
		{
		case HS_HORIZONTAL:
			fprintf (out,"%f 5 %f { newpath dup %f exch moveto %f exch lineto stroke } for\n",
			         bbox->TL.y,bbox->BR.y,bbox->TL.x,bbox->BR.x);
		break;

		case HS_VERTICAL:
			fprintf (out,"%f 5 %f { newpath dup %f moveto %f lineto stroke } for\n",
			         bbox->TL.x,bbox->BR.x,bbox->TL.y,bbox->BR.y);
		break;

		case HS_FDIAGONAL:
			fputs ("gsave % HS_FDIAGONAL\n",out);
			fprintf (out,"%f %f translate -45 rotate ",
			         bbox->TL.x-(bbox->BR.y-bbox->TL.y)/2,(bbox->TL.y+bbox->BR.y)/2);
			side = ((bbox->BR.x-bbox->TL.x) + (bbox->BR.y-bbox->TL.y)) / 1.41421356237309504880;
			fprintf (out,"0 5 %f { newpath dup 0 moveto %f lineto stroke } for ",
			         side,side);
			fputs ("grestore\n",out);
		break;

		case HS_BDIAGONAL:
			fputs ("gsave % HS_BDIAGONAL\n",out);
			fprintf (out,"%f %f translate -45 rotate ",
			         bbox->TL.x-(bbox->BR.y-bbox->TL.y)/2,(bbox->TL.y+bbox->BR.y)/2);
			side = ((bbox->BR.x-bbox->TL.x) + (bbox->BR.y-bbox->TL.y)) / 1.41421356237309504880;
			fprintf (out,"0 5 %f { newpath dup 0 exch moveto %f exch lineto stroke } for ",
			         side,side);
			fputs ("grestore\n",out);
		break;

		case HS_CROSS:
			fprintf (out,"%f 5 %f { newpath dup %f exch moveto %f exch lineto stroke } for\n",
			         bbox->TL.y,bbox->BR.y,bbox->TL.x,bbox->BR.x);
			fprintf (out,"%f 5 %f { newpath dup %f moveto %f lineto stroke } for\n",
			         bbox->TL.x,bbox->BR.x,bbox->TL.y,bbox->BR.y);
		break;

		case HS_DIAGCROSS:
			fputs ("gsave % HS_DIAGCROSS\n",out);
			fprintf (out,"%f %f translate -45 rotate ",
			         bbox->TL.x-(bbox->BR.y-bbox->TL.y)/2,(bbox->TL.y+bbox->BR.y)/2);
			side = ((bbox->BR.x-bbox->TL.x) + (bbox->BR.y-bbox->TL.y)) / 1.41421356237309504880;
			fprintf (out,"0 5 %f { newpath dup 0 moveto %f lineto stroke } for ",
			         side,side);
			fprintf (out,"0 5 %f { newpath dup 0 exch moveto %f exch lineto stroke } for ",
			         side,side);
			fputs ("grestore\n",out);
		break;

		default:
			if (API->flags & WMF_OPT_IGNORE_NONFATAL)
			{	WMF_DEBUG (API,"Unsupported brush/hatch style!");
			}
			else
			{	WMF_ERROR (API,"Unsupported brush/hatch style!");
				API->err = wmf_E_Glitch;
			}
		break;
		}
	break;

	case BS_DIBPATTERN:
		if (dc->brush->bmp.data == 0)
		{	if (API->flags & WMF_OPT_IGNORE_NONFATAL)
			{	WMF_DEBUG (API,"Attempt to fill with non-existent pattern!");
			}
			else
			{	WMF_ERROR (API,"Attempt to fill with non-existent pattern!");
				API->err = wmf_E_Glitch;
				break;
			}
		}
		/* no break here - TODO: implement bitmap fill */
	default:
		if (API->flags & WMF_OPT_IGNORE_NONFATAL)
		{	WMF_DEBUG (API,"Unsupported brush style!");
			/* no break here */
		}
		else
		{	WMF_ERROR (API,"Unsupported brush style!");
			API->err = wmf_E_Glitch;
			break;
		}
	case BS_SOLID:
		rgb = brush->lbColor;

		red   = (float) ((int) rgb.r) / 255;
		green = (float) ((int) rgb.g) / 255;
		blue  = (float) ((int) rgb.b) / 255;

		fprintf (out,"%f %f %f setrgbcolor fill\n",red,green,blue);
	break;
	}
}

/* Write out postscript path, *minus* the ultimate `stroke', then call eps_path_stroke
 * 
 * Assumes that lopnStyle != PS_NULL
 */
static void eps_path_stroke (wmfAPI* API,wmfDC* dc,float linewidth)
{	wmf_eps_t* ddata = WMF_EPS_GetData (API);

	FILE* out = ddata->out;

	float red;
	float green;
	float blue;

	wmfPen* pen;

	wmfRGB rgb;

	WMF_DEBUG (API,"~~~~~~~~eps_path_stroke");

	if (out == 0) return;

	fputs ("\n",out);

	fprintf (out,"%f setlinewidth ",linewidth);

	pen = dc->pen;

	rgb = pen->lopnColor;

	red   = (float) ((int) rgb.r) / 255;
	green = (float) ((int) rgb.g) / 255;
	blue  = (float) ((int) rgb.b) / 255;

	fprintf (out,"%f %f %f setrgbcolor ",red,green,blue);

	switch (pen->lopnStyle & PS_ENDCAP_MASK)
	{
	case PS_ENDCAP_SQUARE:
		fputs ("2 setlinecap ",out);
	break;

	case PS_ENDCAP_ROUND:
		fputs ("1 setlinecap ",out);
	break;

	case PS_ENDCAP_FLAT:
	default:
		fputs ("0 setlinecap ",out);
	break;
	}

	switch (pen->lopnStyle & PS_JOIN_MASK)
	{
	case PS_JOIN_BEVEL:
		fputs ("2 setlinejoin ",out);
	break;

	case PS_JOIN_ROUND:
		fputs ("1 setlinejoin ",out);
	break;

	case PS_JOIN_MITER:
	default:
		fputs ("0 setlinejoin ",out);
	break;
	}

	switch (pen->lopnStyle & PS_STYLE_MASK)
	{
	case PS_DASH: /* DASH_LINE */
		fprintf (out,"[ %f %f ] 0 setdash ",
		         linewidth*10,linewidth*10);
	break;

	case PS_ALTERNATE:
	case PS_DOT: /* DOTTED_LINE */
		fprintf (out,"[ %f %f ] 0 setdash ",
		         linewidth,linewidth*2);
	break;

	case PS_DASHDOT: /* DASH_DOT_LINE */
		fprintf (out,"[ %f %f %f %f ] 0 setdash ",
		         linewidth*10,linewidth*2,linewidth,linewidth*2);
	break;

	case PS_DASHDOTDOT: /* DASH_2_DOTS_LINE */
		fprintf (out,"[ %f %f %f %f %f %f ] 0 setdash ",
		         linewidth*10,linewidth*2,linewidth,linewidth*2,linewidth,linewidth*2);
	break;

	case PS_INSIDEFRAME: /* There is nothing to do in this case... */
	case PS_SOLID:
	default:
		fputs ("[] 0 setdash ",out);
	break;
	}

	fputs ("stroke\n",out);
}
