/* libwmf (api/api.c): library for wmf conversion
   Copyright (C) 2000 - various; see CREDITS, ChangeLog, and sources

   The libwmf Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The libwmf Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the libwmf Library; see the file COPYING.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "wmfdefs.h"

#include "api/api.h"

/* API construction, initialization & destruction
 * ==============================================
 */
wmf_error_t wmf_api_create (wmfAPI** API_return,unsigned long flags,wmfAPI_Options* options)
{	wmfAPI* API = 0;

	wmfMemoryManager* MM = 0;

	wmf_error_t err = wmf_E_None;

	(*API_return) = 0;

	if (flags & WMF_OPT_ARGS) wmf_arg (&flags,options);

/* Initialize memory management array & allocate
 */
	if (flags & WMF_OPT_ALLOC)
	{	MM = (wmfMemoryManager*) options->malloc (options->context,sizeof (wmfMemoryManager));
	}
	else
	{	MM = (wmfMemoryManager*) malloc (sizeof (wmfMemoryManager));
	}
	if (MM == 0)
	{	if ((flags & WMF_OPT_NO_ERROR) == 0)
		{	fputs ("wmf_api_create: insufficient memory!\n",stderr);
		}
		return (wmf_E_InsMem);
	}
	MM->count = 0;
	MM->max = 32;
	if (flags & WMF_OPT_ALLOC)
	{	MM->list = (void**) options->malloc (options->context,MM->max * sizeof (void*));
	}
	else
	{	MM->list = (void**) malloc (MM->max * sizeof (void*));
	}
	if (MM->list == 0)
	{	if ((flags & WMF_OPT_NO_ERROR) == 0)
		{	fputs ("wmf_api_create: insufficient memory!\n",stderr);
		}
		if (flags & WMF_OPT_ALLOC)
		{	options->free (options->context,MM);
		}
		else
		{	free (MM);
		}
		return (wmf_E_InsMem);
	}
	if (flags & WMF_OPT_ALLOC)
	{	MM->context = options->context;

		MM->malloc  = options->malloc;
		MM->realloc = options->realloc;
		MM->free    = options->free;
	}
	else
	{	MM->context = 0;

		MM->malloc  = 0;
		MM->realloc = 0;
		MM->free    = 0;
	}
/* Allocate wmfAPI structure
 */
	if (flags & WMF_OPT_ALLOC)
	{	API = (wmfAPI*) options->malloc (options->context,sizeof (wmfAPI));
	}
	else
	{	API = (wmfAPI*) malloc (sizeof (wmfAPI));
	}
	if (API == 0)
	{	if ((flags & WMF_OPT_NO_ERROR) == 0)
		{	fputs ("wmf_api_create: insufficient memory!\n",stderr);
		}
		if (flags & WMF_OPT_ALLOC)
		{	options->free (options->context,MM->list);
			options->free (options->context,MM);
		}
		else
		{	free (MM->list);
			free (MM);
		}
		return (wmf_E_InsMem);
	}

	API->memory_data = (void*) MM;

/* Initialize debug, error & other streams
 */
	if (flags & WMF_OPT_NO_DEBUG) API->debug_out = 0;
	else
	{	if (flags & WMF_OPT_LOG_DEBUG) API->debug_out = options->debug_out;
		else
		{
			API->debug_out = stdout;
		}
	}

	if (flags & WMF_OPT_NO_ERROR) API->error_out = 0;
	else
	{	if (flags & WMF_OPT_LOG_ERROR) API->error_out = options->error_out;
		else
		{
			API->error_out = stderr;
		}
	}

	API->MetaHeader.pmh = &(API->PlaceableMetaHeader);
	API->MetaHeader.wmfheader = &(API->Head);
	API->File = &(API->MetaHeader);

	API->File->filein = 0; /* Input stream: file (unused) */

	API->buffer_data = 0;  /* Input stream */

	API->bbuf.read = 0;
	API->bbuf.seek = 0;
	API->bbuf.tell = 0;

/* Status function & context
 */
	API->status.context = 0;
	API->status.function = 0;

/* Zero some unset pointers 
 */
	API->function_reference = 0;

	API->font_data = 0;
	API->fonts = 0;

	API->color_data = 0;

/* Library error state:
 */
	API->err = wmf_E_None;

/* Finally: flags, etc.
 */
	API->flags = flags;

/* ---- Henceforth all allocation to be done via api ---- */

/* Create ipa function interface
 */
	API->function_reference = wmf_malloc (API,sizeof (wmfFunctionReference));

	if (ERR (API))
	{	WMF_DEBUG (API,"bailing...");
		err = wmf_api_destroy (API);
		return (err);
	}

/* Create ipa-device data
 */
	if (flags & WMF_OPT_FUNCTION)
	{	options->function (API);
	}
	else if (flags & WMF_OPT_MODULE) /* TODO... TODO... TODO... */
	{	WMF_ERROR (API,"libwmf: module interface not implemented yet...");
		WMF_ERROR (API,"        unable to initialize device layer!");
		API->err = wmf_E_Glitch;
	}
	else
	{	WMF_ERROR (API,"libwmf: unable to initialize device layer!");
		API->err = wmf_E_Glitch;
	}

	if (ERR (API))
	{	WMF_DEBUG (API,"bailing...");
		err = wmf_api_destroy (API);
		return (err);
	}

/* Create font data
 */
	wmf_ipa_font_init (API,options);
	wmf_arg_fontdirs (API,options);

	if (ERR (API))
	{	WMF_DEBUG (API,"bailing...");
		err = wmf_api_destroy (API);
		return (err);
	}

/* Create color data
 */
	wmf_ipa_color_init (API);

	if (ERR (API))
	{	WMF_DEBUG (API,"bailing...");
		err = wmf_api_destroy (API);
		return (err);
	}

/* Create player data
 */
	wmf_player_init (API);

	if (ERR (API))
	{	WMF_DEBUG (API,"bailing...");
		err = wmf_api_destroy (API);
		return (err);
	}

/* Have successfully created the API...
 */
	(*API_return) = API;

	return (wmf_E_None);
}

wmf_error_t wmf_api_destroy (wmfAPI* API) /* Basically free all alloced memory */
{	wmf_error_t err;                  /* associated with the API */

	wmfMemoryManager*     MM = (wmfMemoryManager*)     API->memory_data;
	wmfFunctionReference* FR = (wmfFunctionReference*) API->function_reference;
	wmfFontData*          FD = (wmfFontData*)          API->font_data;

	if (API->flags & API_FTLIBRARY_OPEN)
	{	FT_Done_FreeType (FD->Library);
	}

	if (FR)
	{	/* FR->device_close must be the first action of wmf_api_destroy in case
		 * FR->device_close decides, for whatever reason, to re-play the meta file
		 * or to acquire API resources...
		 */
		if ((API->flags & API_DEVICE_OPEN) && FR->device_close) FR->device_close (API);
	}

	err = API->err;

	while (MM->count)
	{	MM->count--;
		if (MM->free)
		{	MM->free (MM->context,MM->list[MM->count]);
		}
		else
		{	free (MM->list[MM->count]);
		}
	}

	if (MM->free)
	{	MM->free (MM->context,API);
		MM->free (MM->context,MM->list);
		MM->free (MM->context,MM);
	}
	else
	{	free (API);
		free (MM->list);
		free (MM);
	}

	return (err);
}

/* ERROR & DEBUG Reporting
 * =======================
 * 
 * TODO: Really need to implement proper debug/error interface using va_ stuff;
 * ====  here is the old wmfdebug code:
 * 
	 va_list argp;
	 fprintf(stream, "error: ");
	 va_start(argp, fmt);
	 vfprintf(stream, fmt, argp);
	 va_end(argp);
	 fprintf(stream, "\n");
	 fflush(stream);
 * 
 */
void wmf_assert (wmfAPI* API,char* file,int line)
{	wmf_error (API,file,line,"Assertion failed!");

	API->err = wmf_E_Assert;
}

void wmf_error (wmfAPI* API,char* file,int line,char* msg)
{	if (API->error_out == 0) return;

	fprintf (API->error_out,"ERROR: %s (%d): %s\n",file,line,msg);
	fflush (API->error_out);
}

void wmf_debug (wmfAPI* API,char* file,int line,char* msg)
{	if (API->debug_out == 0) return;

	fprintf (API->debug_out,"%s (%d): %s\n",file,line,msg);
	fflush (API->debug_out);
}

/* Memory management interface
 * ===========================
 */
void* wmf_malloc (wmfAPI* API,size_t size)
{	wmfMemoryManager* MM = (wmfMemoryManager*) API->memory_data;

	void*  mem = 0;
	void** more = 0;

	if (MM->count == MM->max)
	{	if (MM->realloc)
		{	more = (void**) MM->realloc (MM->context,MM->list,(MM->max+32) * sizeof (void*));
		}
		else
		{	more = (void**) realloc (MM->list,(MM->max+32) * sizeof (void*));
		}
		if (more == 0)
		{	WMF_ERROR (API,"wmf_[*]alloc: insufficient memory!");
			API->err = wmf_E_InsMem;
			return (0);
		}
		MM->max += 32;
		MM->list = more;
	}

	if (MM->malloc)
	{	mem = MM->malloc (MM->context,size);
	}
	else
	{	mem = malloc (size);
	}
	if (mem == 0)
	{	WMF_ERROR (API,"wmf_[*]alloc: insufficient memory!");
		API->err = wmf_E_InsMem;
		return (0);
	}

	MM->list[MM->count] = mem;
	MM->count++;

	return (mem);
}

void* wmf_calloc (wmfAPI* API,size_t number,size_t size)
{	return (wmf_malloc (API,number * size));
}

void* wmf_realloc (wmfAPI* API,void* mem,size_t size)
{	wmfMemoryManager* MM = (wmfMemoryManager*) API->memory_data;

	void* more = 0;

	int i;

	if (mem == 0) return (wmf_malloc (API,size));

	for (i = 0; i < MM->count; i++)
		if (MM->list[i] == mem)
		{	if (MM->realloc)
			{	more = MM->realloc (MM->context,mem,size);
			}
			else
			{	more = realloc (mem,size);
			}
			if (more == 0)
			{	WMF_ERROR (API,"wmf_[*]alloc: insufficient memory!");
				API->err = wmf_E_InsMem;
			}
			else
			{	MM->list[i] = more;
			}
			break;
		}

	return (more);
}

void wmf_free (wmfAPI* API,void* mem)
{	wmfMemoryManager* MM = (wmfMemoryManager*) API->memory_data;

	int i;

	for (i = 0; i < MM->count; i++)
		if (MM->list[i] == mem)
		{	if (MM->free)
			{	MM->free (MM->context,mem);
			}
			else
			{	free (mem);
			}
			MM->count--;
			MM->list[i] = MM->list[MM->count];
			break;
		}
}

char* wmf_strdup (wmfAPI* API,char* str)
{	char* cpy = 0;

	if (str == 0)
	{	if (ERR (API)) return (0);

		WMF_ERROR (API,"wmf_strdup: attempt to copy non-existent string!");
		API->err = wmf_E_Glitch;
		return (0);
	}

	cpy = (char*) wmf_malloc (API,strlen (str) + 1);

	if (ERR (API)) return (0);

	strcpy (cpy,str);

	return (cpy);
}

char* wmf_strstr (const char* haystack,const char* needle)
{
#ifdef HAVE_STRSTR
	return (strstr (haystack,needle));
#else /* HAVE_STRSTR */
	const char* ptr1;
	const char* ptr2;
	const char* ptr3;

	ptr1 = haystack;
	while (*ptr1)
	{	ptr2 = ptr1;
		ptr3 = needle;

		while ((*ptr3) == (*ptr2))
		{	if ((*ptr3) == 0) break;
			ptr2++;
			ptr3++;
		}

		if ((*ptr3) == 0) break;

		ptr1++;
	}

	return ((*ptr1) ? (char*) ptr1 : 0);
#endif /* HAVE_STRSTR */
}

/* Optional status call-back function
 * ==================================
 */
void wmf_status_function (wmfAPI* API,void* context,wmfStatus function)
{	API->status.context = context;
	API->status.function = function;
}

/* Check command line for arg.
 * ===========================
 */
static void wmf_arg (unsigned long* flags,wmfAPI_Options* options)
{	char** argv = options->argv;
	int    argc = options->argc;
	int    arg = 0;

	while ((++arg) < argc)
	{	if (strncmp (argv[arg],"--wmf-",6)) continue;

		if (strcmp (argv[arg],"--wmf-help") == 0)
		{	/* This is ignored; the controlling program can check for it... */
			continue;
		}

		if ((strcmp (argv[arg],"--wmf-error"    ) == 0)
		 || (strcmp (argv[arg],"--wmf-error=yes") == 0))
		{	(*flags) &= ~WMF_OPT_NO_ERROR;
			continue;
		}
		if (strcmp (argv[arg],"--wmf-error=no") == 0)
		{	(*flags) |= WMF_OPT_NO_ERROR;
			continue;
		}

		if ((strcmp (argv[arg],"--wmf-debug"    ) == 0)
		 || (strcmp (argv[arg],"--wmf-debug=yes") == 0))
		{	(*flags) &= ~WMF_OPT_NO_DEBUG;
			continue;
		}
		if (strcmp (argv[arg],"--wmf-debug=no") == 0)
		{	(*flags) |= WMF_OPT_NO_DEBUG;
			continue;
		}

		if (strcmp (argv[arg],"--wmf-sys-fonts") == 0)
		{	(*flags) |= WMF_OPT_SYS_FONTS;
			continue;
		}
		if (strncmp (argv[arg],"--wmf-sys-fontmap=",18) == 0)
		{	(*flags) |= WMF_OPT_SYS_FONTS;
			(*flags) |= WMF_OPT_SYS_FONTMAP;
			options->sys_fontmap_file = argv[arg] + 18;
			continue;
		}

		if (strcmp (argv[arg],"--wmf-xtra-fonts") == 0)
		{	(*flags) |= WMF_OPT_XTRA_FONTS;
			continue;
		}
		if (strncmp (argv[arg],"--wmf-xtra-fontmap=",19) == 0)
		{	(*flags) |= WMF_OPT_XTRA_FONTS;
			(*flags) |= WMF_OPT_XTRA_FONTMAP;
			options->xtra_fontmap_file = argv[arg] + 19;
			continue;
		}

		if ((strcmp (argv[arg],"--wmf-ignore-nonfatal"    ) == 0)
		 || (strcmp (argv[arg],"--wmf-ignore-nonfatal=yes") == 0))
		{	(*flags) |= WMF_OPT_IGNORE_NONFATAL;
			continue;
		}
		if (strcmp (argv[arg],"--wmf-ignore-nonfatal=no") == 0)
		{	(*flags) &= ~WMF_OPT_IGNORE_NONFATAL;
			continue;
		}

		if (strcmp (argv[arg],"--wmf-diagnostics") == 0)
		{	(*flags) |= WMF_OPT_DIAGNOSTICS;
			continue;
		}

		if (strncmp (argv[arg],"--wmf-fontdir=",14) == 0) continue; /* ignore for now */
	}

	(*flags) &= 0x000fffff;
}

static void wmf_arg_fontdirs (wmfAPI* API,wmfAPI_Options* options)
{	char** argv = options->argv;
	int    argc = options->argc;
	int    arg = 0;

	if (API->flags & WMF_OPT_ARGS)
	{	while ((++arg) < argc)
		{	if (strncmp (argv[arg],"--wmf-fontdir=",14) == 0)
			{	wmf_ipa_font_dir (API,argv[arg] + 14);
			}
		}
	}

	if (API->flags & WMF_OPT_FONTDIRS)
	{	argv = options->fontdirs;
		while (*argv)
		{	wmf_ipa_font_dir (API,(*argv));
			argv++;
		}
	}

	wmf_ipa_font_dir (API,WMF_FONTDIR);
}

char* wmf_help ()
{	static char* help = "\
Additional wmf-related options:\n\
\n\
  --wmf-error[=yes|no]            switch for error reports.\n\
  --wmf-debug[=yes|no]            switch for debug reports, if any.\n\
  --wmf-ignore-nonfatal[=yes|no]  switch to ignore (some) non-fatal errors.\n\
  --wmf-diagnostics               emit diagnostic information.\n\
  --wmf-fontdir=<path>            add <path> to list of font directories.\n\
  --wmf-sys-fonts                 use system fonts, if any found.\n\
  --wmf-sys-fontmap=<file>        use system xml-fontmap file <file>.\n\
  --wmf-xtra-fonts                use non-system fonts, if any found.\n\
  --wmf-xtra-fontmap=<file>       use non-system xml-fontmap file <file>.\n\
\n\
Report bugs to <http://www.wvware.com/>.\n";

	return (help);
}
