/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux <klaux@student.uni-kl.de>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "resources.h"


unsigned int ResourcesInit::count_ = 0;
ResourcesInit::ResourcesInit()
{
  if(count_++ == 0)
    uta::Resources::instance = new uta::Resources();
}

ResourcesInit::~ResourcesInit()
{
  count_--;
  if(count_ == 0)
    {
      delete uta::Resources::instance;
      uta::Resources::instance = NULL;
    }
}

namespace uta {

Resources* Resources::instance = NULL;

const Surface* 
Resource::surface() const 
{ 
  if(type_ == SURFACE)
    return surface_; 
  return NULL;
};
   
const Font* 
Resource::font() const 
{ 
  if(type_ == FONT)
    return font_; 
  return NULL;
};
 
const Sample* 
Resource::sample() const 
{ 
  if(type_ == SAMPLE)
    return sample_; 
  return NULL;
};
 
const Music* 
Resource::music() const 
{ 
  if(type_ == MUSIC)
    return music_; 
  return NULL;
};  
  
const Color&
Resource::color() const
{
  if(type_ == COLOR)
    return *color_;
  return uta::transparent;
}

ColorResource::ColorResource(std::string colordesc):
    Resource()
{
  color_ = new Color;
  type_ = COLOR;

  unsigned int r,g,b,a;
  int conversions = 
    sscanf(colordesc.c_str(),"%d %d %d %d",&r,&g,&b,&a);
  
  if(conversions == 4)
    color_ = new Color(r,g,b,a);
  else if(conversions == 3)
    color_ = new Color(r,g,b);
  else
    {
      color_ = new Color(blue.r, blue.b, blue.g);
      debugN(17,std::cerr << "Resource: error parsing Colorvalue: \""<<colordesc<<"\""
	     << std::endl;);
    }
  
}


FontResource::FontResource(std::string fontdescriptor_) :
  Resource()
{
  font_ = new Font;
  type_ = FONT;

  std::string tmp = fontdescriptor_;
  int pos;
  std::string filename;
  pos = tmp.find(",");
  filename = tmp.substr(0,pos);
  
  int size;
  long unsigned int fontCol;
  long unsigned int backCol;
  sscanf(tmp.substr(pos+1,tmp.size()-pos-1).c_str(),"%d,%lx,%lx",
	 &size, &fontCol, &backCol);
  debugN(17, std::cerr <<"requesting font from "<<filename<<" at "<<size<<" pt."<<std::endl;);
  font_->create(filename, size, Color(fontCol), Color(backCol));
 
}
Resources::Resources()
{ 
  debugN(17, std::cerr << "Resources: becoming alive..." << std::endl;);
  
  default_ = new Resource();
}


Resources::~Resources()
{ 
  unregisterAll();
}

void
Resources::unregisterAll()
{

  debugN(17, std::cerr << "Resources: freeing resources.....";);

  if(default_)
    {
      default_->free();
      default_ = NULL;
    }

  resources_map::iterator res = resources_.begin();
  while(res != resources_.end())
    {
      res->second->free();
      res++;
    }

  resources_.clear();
  surfaces.clear();
  fonts.clear();
  samples.clear();
  songs.clear();
  loadedFiles_.clear();

  debugN(17, std::cerr << "done."<< std::endl;);

}
 

bool
Resources::create(std::string resname, Resource* res)
{
  bool success = false;
  
  if(!res)
    return success;

  resources_map::iterator itr = resources_.find(resname);
  
  if(res->loadedProperly())
    {  
      if(itr != resources_.end())
	{
	  debug(std::cerr << "libuta::Resources: resource\""<<resname
	       << "\" already exists ! Load request will be ignored."
	       << std::endl;)
	  // remove the just loaded resource
	  res->free();

	  /* but increase reference count to the existing resource
	   * ...
	   */
	  itr->second->bind();

	  // DISABLED OVERLOADING ... as it is nonsense !
	  // 	  debugN(17,std::cerr << "Resource: \""<<resname
	  // 		 <<"\" already exists, overloading it" << std::endl;);
	  // 	  //deleting old stuff
	  // 	  itr->second->free();
	  // 	  //setting new resource
	  // 	  itr->second = res;
	}
      else
	{
	  resources_[resname]=res; 
	  success = true;
	}
    }
  else
    {
      res->free();
    }
  
  return success;
}

Resource*
Resources::check(std::string filename)
{

  if(filename.empty())
    return NULL;

  files_map::iterator itr = loadedFiles_.find(filename);
  
  if(itr != loadedFiles_.end())
    {
	  resources_map::iterator rItr = resources_.find(itr->second);
	  if(rItr == resources_.end())
		  {
		    debugN(17,std::cerr << "Resources database corrupted !!" << std::endl;);
			  return NULL;
		  }
	   else
		  {
			   return rItr->second;
		  }
    }
  else
    {
      return NULL;
    } 
}
  
void 
Resources::registerSurface(std::string resname, std::string filename)
{
  if(filename.empty() || resname.empty())
    return;
  
  SurfaceResource* res = (SurfaceResource*)check(filename);
  
  if(!res)
    {
      //if the file has not been loaded, load it now !
      res = new SurfaceResource(filename);
      loadedFiles_[filename]=resname;
    }
  else
    {
      //if the file has already been loaded increase
      //it's reference count
      res->bind();
    }
  
  if(create(resname, res))
    surfaces.push_back(resname);
  else
    {
      debugN(17,
	     std::cerr << "uta::Resources Error: unable to load surface resource \""<<resname <<"\""<<std::endl 
	     << "                      from file \"" << filename << "\"" << std::endl;);
    }
	     
  debugN(17,std::cerr << "Resources: "
	 << "Resources("<<resources_.size()<<") "
	 << "Files("<<loadedFiles_.size()<<") "
	 << "Surfaces("<<surfaces.size()<<") " << std::endl;);
				 
}

void 
Resources::registerFont(std::string resname, std::string filename)
{
  if(filename.empty() || resname.empty())
    return;

  FontResource* res = (FontResource*)check(filename);
  
  if(!res)
    {
      //if the file has not been loaded, load it now !
      res = new FontResource(filename);
      loadedFiles_[filename]=resname;
    }
  else
    {
      //if the file has already been loaded increase
      //it's reference count
      res->bind();
    }     

  if(create(resname, res))
    fonts.push_back(resname);
  else
    {
      debugN(17,
    std::cerr << "uta::Resources Error: unable to create font resource \""<<resname <<"\""<<std::endl 
	 << "                      from from specification \"" << filename << "\"" << std::endl;
	     );
    }
  debugN(17,std::cerr << "Resources: "
	 << "Resources("<<resources_.size()<<") "
	 << "Files("<<loadedFiles_.size()<<") "
	 << "Fonts("<<fonts.size()<<") " << std::endl;);

}


void 
Resources::registerSample(std::string resname, std::string filename)
{
  if(filename.empty() || resname.empty())
    return;

  SampleResource* res = (SampleResource*)check(filename);
  
  if(!res)
    {
      //if the file has not been loaded, load it now !
      res = new SampleResource(filename);
      loadedFiles_[filename]=resname;
    }
  else
    {
      //if the file has already been loaded increase
      //it's reference count
      res->bind();
    }
     
  if(create(resname, res))
    samples.push_back(resname);
  else
    {
      debugN(17,
    std::cerr << "uta::Resources Error: unable to load sample resource \""<<resname << "\""<<std::endl 
	     << "                      from file \"" << filename << "\"" << std::endl;);
    }

  debugN(17,std::cerr << "Resources: "
	 << "Resources("<<resources_.size()<<") "
	 << "Files("<<loadedFiles_.size()<<") "
	 << "Samples("<<samples.size()<<") " << std::endl;);
}

void 
Resources::registerMusic(std::string resname, std::string filename)
{
  if(filename.empty() || resname.empty())
    return;

  MusicResource* res = (MusicResource*)check(filename);
  
  if(!res)
    {
      //if the file has not been loaded, load it now !
      res = new MusicResource(filename);
      loadedFiles_[filename]=resname;
    }
  else
    {
      //if the file has already been loaded increase
      //it's reference count
      res->bind();
    }
    
  if(create(resname, res))
    songs.push_back(resname);
  else
    {
      debugN(17,
    std::cerr << "uta::Resources Error: unable to load music resource \""<<resname <<"\""<<std::endl 
	     << "                      from file \"" << filename << "\"" << std::endl;);
    }

  debugN(17,std::cerr << "Resources: "
	 << "Resources("<<resources_.size()<<") "
	 << "Files("<<loadedFiles_.size()<<") "
	 << "Songs("<<songs.size()<<") " << std::endl;);
}

void 
Resources::registerColor(std::string resname, std::string colordesc)
{

  if(colordesc.empty() || resname.empty())
    return;

  // no need to check for already loaded files ... since we do not load
  // from a file

  ColorResource* res = new ColorResource(colordesc);
  
  if(create(resname, res))
    colors.push_back(resname); 
  else
    {
      debugN(17,
    std::cerr << "uta::Resources Error: unable to create color resource \""<<resname <<"\""<<std::endl 
	     << "                      from specification \"" << colordesc << "\"" << std::endl;);
    }

  debugN(17,std::cerr << "Resources: "
	 << "Resources("<<resources_.size()<<") "
	 << "Files("<<loadedFiles_.size()<<") "
	 << "Colors("<<colors.size()<<") " << std::endl;);
}

bool
Resources::isAvailable(const std::string& resname)
{
  return (resources_.find(resname) != resources_.end());
}

const Resource*
Resources::get(std::string resname_)
{
  if(resname_.empty())
    return default_;
  
  resources_map::iterator res = resources_.find(resname_);
  
  if(res == resources_.end())
    {
      //ask application for help in finding the resource
      Resource* resource = unknownResource.emit(resname_);
      if(resource != NULL)
	{
	  create(resname_, resource);
	  if(dynamic_cast<SurfaceResource*>(resource))
	    surfaces.push_back(resname_);
	  else if(dynamic_cast<FontResource*>(resource))
	    fonts.push_back(resname_);
	  else if(dynamic_cast<MusicResource*>(resource))
	    songs.push_back(resname_);
	  else if(dynamic_cast<SampleResource*>(resource))
	    samples.push_back(resname_);
	  else if(dynamic_cast<ColorResource*>(resource))
	    colors.push_back(resname_);

	  debugN(17,std::cerr<<"Resource \""<<resname_<<"\" successfully loaded."
		 << std::endl;);
	  return resource;
	}
    }
  else
    {
      debugN(17,std::cerr<<"Resource \""<<resname_<<"\" successfully looked up."
	     << std::endl;);
      return res->second;
    }

  debugN(17, std::cerr << "Resource \""<<resname_<<"\" not found."<<std::endl;);
  return default_;
}


bool
Resources::unregister(std::string resname)
{
  bool success = false;

  if(resname.empty())
    return success;
  
  resources_map::iterator res = resources_.find(resname);
  
  if(res != resources_.end())
    {
      if(res->second->references()==1)
	{
	  //this resource is going to be deleted...
	  //we must remove the entry in the loadedFiles_ map
	  files_map::iterator itr = loadedFiles_.begin();
	  while(itr != loadedFiles_.end())
	    {
	      if(itr->second == resname)
		break;
	      itr++;
	    }
	  if(itr != loadedFiles_.end())
	    {
	      //remove the filename from the list of loaded files
	      loadedFiles_.erase(itr);
	    }
	  else
	    {
	      debugN(17,std::cerr<<"Resource: database corrupted !?"<<std::endl;);
	    }
	}
      //unbind this resources 
      res->second->free();
      //remove this ressource registration
      resources_.erase(res);
      
      success = true;
    }
  return success;
}

}

