/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux <klaux@student.uni-kl.de>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "music.h"
#include "mixer.h"
#include "debug.h"

#include <iostream>

namespace uta {

int Music::volume_ = Sound::VOLUME_MAX;

//disable the smpeg direct access ...
#undef HAVE_SMPEG

#ifdef HAVE_MIXER

Music::Music() :
  Sound(),
  channel_(0),
  data_(NULL),
  mpeg(NULL)
{
 
}

Music::Music(const std::string& filename) :
  Sound(),
  channel_(0),
  data_(NULL),
  mpeg(NULL)
{
  load(filename);
}


Music::~Music()
{
  stop();

  if(data_)
    {
      
      debugN(17,std::cerr << "Music call Mix_FreeMusic() ...";);
      if(mixer_->audioAvailable())
	Mix_FreeMusic(data_);
      else
	{
	  debug(std::cerr << "deleting Music though Mixer is not available (any more?)" 
	       << std::endl;)
	}
      debugN(17,std::cerr << " [OK]"<<std::endl;);
    }
#ifdef HAVE_SMPEG
  if(mpeg)
    SMPEG_delete(mpeg);
#endif

}


bool
Music::load(const std::string& filename)
{
  bool success = false;

  if(mixer_)
    {
      if(mixer_->audioAvailable())
	{	
#ifdef HAVE_SMPEG
 	  if(filename.find(".mp3")!=std::string::npos ||
	     filename.find(".MP3")!=std::string::npos ||
	     filename.find(".mpeg")!=std::string::npos ||
	     filename.find(".mpg")!=std::string::npos ||
	     filename.find(".MPEG")!=std::string::npos ||
	     filename.find(".MPG")!=std::string::npos )
	    {	  

	      SMPEG_Info info;
	      mpeg = SMPEG_new(filename.c_str(), &info, 0);   
	      if ( SMPEG_error(mpeg) ) 
		{
		  debugN(17, std::cerr <<"Music: " << SMPEG_error(mpeg) << std::endl;);
		  SMPEG_delete(mpeg);
		  mpeg = NULL;
		  return false;
		}      

	      SMPEG_enableaudio(mpeg, 0);
	      SMPEG_enablevideo(mpeg, 0);

	      SDL_AudioSpec audiofmt;
	      Uint16 format;
	      int freq, channels;
	      
	      /* Tell SMPEG what the audio format is */
	      Mix_QuerySpec(&freq, &format, &channels);
	      audiofmt.format = format;
	      audiofmt.freq = freq;
	      audiofmt.channels = channels;
	      SMPEG_actualSpec(mpeg, &audiofmt);

	      success = true; 
	    }
	  else
#endif
	    {
	      data_ = Mix_LoadMUS(filename.c_str());
	      
	      if(!data_)
		{
		  debugN(17,std::cerr << "Music: unable to load file." << std::endl;);
		}
	      else
		{
		  success = true;
		}

	    }	  
	}
      else
	{
	  debug(std::cerr << "Music could not load: no mixer system available." << std::endl;)
	  data_ = NULL;
	}
    }
  else
    {
      debug(std::cerr << "Music could not connect to the mixer object." << std::endl;)
      data_ = NULL;
    }
  
  return success; 
}


int
Music::setVolume(int vol)
{
  Music::volume_ = vol;
  if(!mixer_)
    return 0;
  if(!mixer_->audioAvailable())
    return 0;

  if(data_ || mpeg)
    return Mix_VolumeMusic(vol);

#ifdef HAVE_SMPEG
  if(mpeg)
    SMPEG_setvolume(mpeg, vol); 
#endif

  return 0;
}

bool
Music::play(int loop) const
{  
  bool success = false;
  int channel = 0;
  
  if(!mixer_)
    return false;
  if(!mixer_->audioAvailable())
    return false;

  if(data_)
    {
      channel = Mix_PlayMusic(data_,loop);
      if(channel >= 0)
	{
	  success = true;
	  Mix_VolumeMusic(volume_);
	}
    }
#ifdef HAVE_SMPEG
  if(mpeg)
    {
      /* Hook in the MPEG music mixer */
      Mix_HookMusic(SMPEG_playAudioSDL, mpeg);
      SMPEG_enableaudio(mpeg, 1);
      SMPEG_play(mpeg);  
    }
#endif

  return success;
}

bool
Music::playing() const
{  
  bool playing_ = false;

  if(mixer_)
    if(mixer_->audioAvailable())  
      {
	if(data_)
	  playing_ = Mix_PlayingMusic();
#ifdef HAVE_SMPEG
	else if(mpeg)
	  playing_ = (SMPEG_status(mpeg) == SMPEG_PLAYING);
#endif	
      }

  return playing_;
}


void
Music::stop() const
{
  if(mixer_)
    if(mixer_->audioAvailable())
      if(data_)
	Mix_HaltMusic();
#ifdef HAVE_SMPEG
      else if(mpeg)
	{
	  /* Stop the movie and unhook SMPEG from the mixer */
	  SMPEG_stop(mpeg);
	  Mix_HookMusic(NULL, NULL);   
	}
#endif
}

void
Music::pause() const
{  
  if(mixer_)
    if(mixer_->audioAvailable())
      Mix_PauseMusic();
}

void
Music::resume() const
{  
  if(mixer_)
    if(mixer_->audioAvailable())
      Mix_ResumeMusic();
}

void
Music::rewind() const
{  
  if(mixer_)
    if(mixer_->audioAvailable())
      Mix_RewindMusic();
}

bool
Music::paused() const
{
  if(mixer_)
    if(mixer_->audioAvailable())
      return Mix_PausedMusic();
  return false;
}

#else
//-------------------------------------------------------------
// dummy implementation when not linking against libmixer

Music::Music() :
  Sound(){
}

Music::Music(const std::string& filename) :
  Sound() 
{
  load(filename);
}

Music::~Music()
{
}

bool
Music::load(const std::string& )
{
  return false;
}

void
Music::pause() const
{  
}

void
Music::resume() const
{  
}

void
Music::rewind() const
{  
}


bool
Music::paused() const
{
  return false;
}

bool
Music::play(int ) const
{  
  return false;
}

int
Music::setVolume(int)
{
  return 0;
}

bool
Music::playing() const
{  
  return false;
}

void
Music::stop() const
{

}

#endif //HAVE_MIXER
}

