/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "multilineedit.h"
#include <ctype.h>
#include "debug.h"
#include "resources.h"

#include <stack>

namespace uta {

MultiLineEdit::MultiLineEdit(Widget* parent, int x, int y, int w, int h, 
			       const Font *font, bool readonly):
  LineEdit(parent, x, y, w, h, font),
  wordWrap_(true),
  readOnly_(readonly),
  maxLinesToStore_(100)
{
 
  enterPressed.connect(slot(this, &MultiLineEdit::newline));

  if(RES_AVAILABLE("multilineedit_color"))
    Widget::setColor(COLOR_RES("multilineedit_color"));
  if(RES_AVAILABLE("multilineedit_surface"))
    Widget::setBackground(SURFACE_RES("multilineedit_surface"),false);
    
  fonts_.resize(16);
  images_.resize(16);
  for(size_t n=0; n < 16; n++)
    {
      fonts_[n] = NULL;
      images_[n] = NULL;
    }
}

MultiLineEdit::~MultiLineEdit()
{
//   for(unsigned n=0; n < renderedLines_.size(); n++)
//    delete renderedLines_[n];

}

void
MultiLineEdit::breakText(std::string& text, std::vector<std::string>& lines, 
			 unsigned maxLines)
{
  unsigned pos = 0;
  unsigned start = 0;
  unsigned len = 0;
  unsigned linestart = 0;
  int x=0;
  std::vector<unsigned> linebreaks;

  Rect dest;
  bool breakLine = false;
  int size_x = 0;
  int size_y = textFont_->getHeight();
  
  lines.clear();
  
  pos=0;
  while(pos < text.size())
    {
      start = pos;
      breakLine = false;
      
      if(wordWrap_)
	{
	  //dont break words
	  while(pos<text.size())
	    {
	      if(isspace(text[pos]))
		break;
	      pos++;
	      size_x+= textFont_->getSize(text[pos]).x;
	    }
	  
	  if(pos<text.size())
	    {
	      if(text[pos] == '\n')
		{
		  breakLine = true;
		  len = pos-start;
		  pos++;
		}
	      else
		{
		  pos++;
		  size_x+= textFont_->getSize(text[pos]).x;
		  len = pos-start;
		}
	    }
	  else
	    {
	      len = pos-start;
	    }
	}
      else
	{
	  pos++;
	  size_x += textFont_->getSize(text[pos]).x;	    
	}
      
      if(!breakLine && (x + size_x >= width()))
	{
	  x = 0;
	  size_y += textFont_->getHeight();
	  
	  if(linestart != text.size())
	    {
	      if(start != linestart)
		{
		  //if the word is not longer than the line....
		  linebreaks.push_back(linestart);
		  lines.push_back(text.substr(linestart,start-linestart));
		  linestart = pos = start;
		}
	      else
		{
		  //the word is longer than the line, thus only the front part
		  //will be visible
		  linebreaks.push_back(linestart);
		  lines.push_back(text.substr(linestart,pos-linestart-1));
		  linestart = start = pos;
		}
	    }
	} 
      
      x += size_x;
      
      if(breakLine)
	{
	  x = 0;
	  size_y += textFont_->getHeight();
	  
	  if(linestart != text.size())
	    {
	      linebreaks.push_back(linestart);
	      lines.push_back(text.substr(linestart,pos-linestart-1));
	      linestart = start = pos;
	    }
	}
      
      size_x = 0;
	  
    }  //while pos < text.size()
  
  //add an empty line if last char is a '\n'
  if(breakLine)
    lines.push_back("");
  
  //do not forget the last line (if it is not empty)
  if(linestart != pos)
    lines.push_back(text.substr(linestart,pos-linestart));

  /* before removing lines we need to parse for escape sequences, in
   * order to not lose anything when removing lines ..
   */
  preprocessLines(lines);

  /* limit the number of lines to the requested maximum */
  if(lines.size() > maxLines)
    {
      text.erase(0,linebreaks[lines.size()-maxLines]);
    }

}


unsigned 
MultiLineEdit::visibleLines() const
{
  return height()/textFont_->getHeight();
}

void 
MultiLineEdit::preprocessLines(std::vector<std::string>& lines)
{
  std::list<std::string> prefix;
  for(unsigned n = 0; n < lines.size(); n++)
    {
      if(!lines[n].empty())
	{
	  std::string prefixStr;
	  std::list<std::string>::iterator pItr = prefix.begin();
	  while(pItr != prefix.end())
	    {
	      prefixStr+= *pItr;
	      pItr++;
	    }
	  
	  std::string::const_iterator itr = lines[n].begin();	  
	  while(itr != lines[n].end())
	    {
	      std::string tmp;
				  
	      while(itr != lines[n].end() &&
		    *itr != START_TAG && 
		    *itr != END_TAG)
		itr++;
		
	      if(itr == lines[n].end())
		break;
	      
	      if(*itr == START_TAG)
		{
		  tmp += (*itr);

		  itr++;
		  if(itr == lines[n].end())
		    break;
		  		 
		  tmp += (*itr); 

	
		  itr++;
		  if(itr == lines[n].end())
		    break;		  
		  
		  tmp += (*itr++);

		  prefix.push_back(tmp);
		  tmp = "";
		}
	      else if(*itr == END_TAG)
		{
		  if(!prefix.empty())
		    prefix.pop_back();
		  itr++;
		}
	      
	    } /// while (itr != lines[n].end())

	  lines[n] = prefixStr + lines[n];

	} // if(!lines[n].empty) {
    }
}

void
MultiLineEdit::renderTextLines(const std::vector<std::string>& lines)
{
  unsigned int max_lines = height()/textFont_->getHeight();

  int pal_ind = 0;
  int font_ind = 0;
  int img_ind = 0;
  int link_ind = 0;
  std::stack<int> context;
				  
  unsigned start = (lines.size() <= max_lines)?0:lines.size()-max_lines;

  textLines_.clear();

  for(unsigned n = start; n < lines.size(); n++)
    {
      /* to start just push an empty line, we will
       * add text chunks if needed 
       */
      textLines_.push_back(TextLine());
      if(!lines[n].empty())
	{
	  std::string::const_iterator itr = lines[n].begin();

	  while(itr != lines[n].end())
	    {
	      std::string tmp;

	      while(itr != lines[n].end() &&
	            *itr != START_TAG && 
		    *itr != END_TAG)
		{
		  tmp += (*itr);
		  itr++;
		}

	 //      if(!tmp.empty())
// 		{
		  TextChunk chunk;
		  chunk.link = link_ind;
		  chunk.image = img_ind;
		  chunk.font = font_ind;
		  chunk.palette = pal_ind;
		  chunk.text = tmp;

		  textLines_.back().push_back(chunk);
		  //	}
	      	      
	      if(itr == lines[n].end())
		break;
	      
	      if(*itr == START_TAG)
		{
		  itr++;
		  if(itr == lines[n].end())
		    break;
		  
		  if(*itr == PALETTE_TAG ||
		     *itr == FONT_TAG ||
		     *itr == IMAGE_TAG ||
		     *itr == LINK_TAG)
		    {
		      char tag = *itr;
		      itr++;
		      if(itr == lines[n].end())
			break;
		      switch(tag) 
			{
			case PALETTE_TAG:
			  // get palette number
			  pal_ind = *itr++;
			  break;
			case FONT_TAG:
			  font_ind = *itr++;
			  break;
			case IMAGE_TAG:
			  img_ind = *itr++;
			  break;
			case LINK_TAG:
			  link_ind= *itr++;
			  break;
			default:
			  break;
			}
		      context.push(tag);
		    }		      
		  else
		    {
		      itr++;
		      if(itr == lines[n].end())
			break; 
		      itr++;
		    }
		}
	      else if(*itr == END_TAG)
		{
		  if(!context.empty())
		    {
		      if(context.top() == PALETTE_TAG)
			{
			  pal_ind = 0;
			  context.pop();
			}
		      else if(context.top() == FONT_TAG)
			{
			  font_ind = 0;
			  context.pop();
			}
		      else if(context.top() == IMAGE_TAG)
			{
			  img_ind = 0;
			  context.pop();
			}
		      else if(context.top() == LINK_TAG)
			{
			  link_ind = 0;
			  context.pop();
			}
		    }
		  itr++;
		}
	      
	    } /// while (itr != lines[n].end())
	  
	} // if(lines[n].empty) {} else {
    }
}

void
MultiLineEdit::create()
{
  if(textChanged_)
    {
      unsigned int max_lines = height()/textFont_->getHeight();
      std::vector<std::string> lines;
      breakText(text_, lines, maxLinesToStore_ + max_lines);
      preprocessLines(lines);
      renderTextLines(lines);
    }

  if(!justCursor_ || textChanged_)
    {  
      Widget::create();

      int yoff = 0;
      int xoff = 0;
      int line_height = textFont_->getHeight();
      int offset = 0;
      const Font* font;

      linkAreas_.clear();
      int xsave;
      for(unsigned n=0; n < textLines_.size(); n++)
	{
	  xoff = 0;
	  xsave = 0;
	  TextLine::iterator itr = textLines_[n].begin();
	  while(itr != textLines_[n].end())
	    {
	      xsave = xoff;
	      if(itr->image > 0 && itr->image < images_.size())
		if(images_[itr->image])
		  {
		    const Surface* img = images_[itr->image];
		    offset= line_height-img->height();
		    Rect dest(xoff, yoff+offset,
			      img->width(), img->height());
		    img->blit(surface_, dest);
		    xoff += img->width();
		  }

	      if(!itr->text.empty())
		{
		  if(itr->font > 0 && itr->font < fonts_.size())
		    font = fonts_[itr->font];
		  else 
		    font = textFont_;
		  
		  /* Font::blitString() checks for a valid palette and 
		   * falls back to the default palette if none is given
		   */
		  offset = line_height - font->getHeight();
		  if(itr->palette > 0 && itr->palette < markup_.size())
		    xoff += font->blitString(itr->text, surface_, 
					     Point(xoff, yoff+offset),
					     markup_[itr->palette]);
		  else
		    xoff += font->blitString(itr->text, surface_, 
					     Point(xoff, yoff+offset),fontPal_);
		}
	      
	      if(itr->link > 0)
		{
		  LinkArea link;
		  link.rect = Rect(xsave, yoff+offset, 
				   xoff-xsave, line_height-offset);
		  link.id = itr->link;
		  linkAreas_.push_back(link);
	      
		}
	      if(xoff > surface_->width() || yoff > surface_->height())
		break;
	      itr++;
	    }
	  yoff += textFont_->getHeight();
	}
      
      //move the cursor

      if(xoff+textCursor_.width() > surface_->width())
	xoff = surface_->width() - textCursor_.width();
      
      dirtyRects_.push_back(globalCoord(textCursor_));
      textCursor_.warp(Point(xoff + 1, yoff-textCursor_.height())) ;

      //reset flags
      justCursor_ = false;
      textChanged_ = false; 
    }
  
  if(!readOnly_)
    {
      if(justCursor_)
	{
	  dirtyRects_.clear();
	  dirtyRects_.push_back(globalCoord(textCursor_));
	  justCursor_ = false;
	}

      //draw the cursor ...
      if(drawCursor_) 
	drawCursor();
      else
	deleteCursor();
    }
  
}
  
void
MultiLineEdit::setWrapping(bool flag)
{
  if(flag != wordWrap_)
    {
      textChanged_ = true;
      needsUpdate_ = true;
      wordWrap_ = flag;
    }
}
  

void MultiLineEdit::addText(const char* data)
{
  if(!data)
    return;
  text_ += std::string(data);
  //we need to be reblitted !
  textChanged_ = true;
  needsUpdate_ = true;
}

void MultiLineEdit::newline(const char*)
{
  text_ += std::string("\n");
  
  //we need to be reblitted !
  textChanged_ = true;
  needsUpdate_ = true;
}

bool
MultiLineEdit::processEvent(const Event *event)
{ 
  
  if(mouseFocus_)
    {
      if(event->type() == Event::BUTTON)
	{
	  const ButtonEvent* butEvent = (const ButtonEvent*)event;
	  
	  if(butEvent->buttons() == ButtonEvent::BUTTON_LEFT)
	    {
	      if( butEvent->state() == ButtonEvent::PRESSED )
		{
		  std::vector<LinkArea>::iterator itr = linkAreas_.begin();
		  while(itr != linkAreas_.end())
		    {
		      if(globalCoord(itr->rect).contains(butEvent->position()))
			{
			  linkActivated.emit(itr->id);
			  /* hmmm break or return true ? ...
			   * a design guideline would answer this, but
			   * no one has written it yet ... */
			  break;
			}
		      itr++;
		    }
		}
	    }
	}
    }
  
  if(readOnly_)
    return false;

  return LineEdit::processEvent(event);
  
}

void
MultiLineEdit::timer()
{

  if(readOnly_)
    return;

  LineEdit::timer();

}
void 
MultiLineEdit::setMarkupColor(int index,
			      const Color& fontCol, const Color& backCol)
{
  if(index < 1)
    return;

  int rdiff = fontCol.r - backCol.r;
  int gdiff = fontCol.g - backCol.g;
  int bdiff = fontCol.b - backCol.b;
  int adiff = fontCol.a - backCol.a;

  int r = backCol.r;
  int g = backCol.g;
  int b = backCol.b;
  int a = backCol.a;

  if((int)markup_.size() <= index)
    markup_.resize(index+1);
  
  markup_[index].clear();
  markup_[index].push_back(transparent);

  for(unsigned char num=1; num < 5; num++)
    markup_[index].push_back(Color(r + num * rdiff/4,
				   g + num * gdiff/4,
				   b + num * bdiff/4,
				   a + num * adiff/4));

  needsUpdate_ = true;
}

void 
MultiLineEdit::setMarkupFont(int index, const Font* font)
{
  if(index > 0 && index < (int)(fonts_.size()))    
    fonts_[index] = font;
}

void 
MultiLineEdit::setImage(int index, const Surface* image)
{
  if(index > 0 && index < (int)(images_.size()))    
    images_[index] = image;
}


}

