/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux <klaux@student.uni-kl.de>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "sample.h"
#include "mixer.h"
#include "debug.h"

#include <iostream>
#include <stdlib.h>    //malloc

namespace uta {

#ifdef HAVE_MIXER
Sample::Sample() :
  Sound(),
  data_(0),
  channel_(-1)
{
}

Sample::Sample(const Sample& sample) :
  Sound(sample),
  channel_(-1)
{
  if(this != &sample)
    {
      if(sample.data_ != 0)
	{
	  debugN(17, std::cerr << "Sample::Sample(const Sample&) ...";);

	  data_ = (Mix_Chunk*)malloc(sizeof(Mix_Chunk));

	  //manually copying sample data.... struct Mix_Chunk
	  data_->alen = sample.data_->alen;
	  data_->volume = sample.data_->volume;
	  
	  //allocate mem for sample data
	  data_->abuf = (Uint8*)malloc(data_->alen);
	  
	  //copy sample data
	  //memcpy(<dest>, <src>, <length>)
	  memcpy((void*)data_->abuf, 
		 (void*)sample.data_->abuf, 
		 (size_t)data_->alen);

	  debugN(17,std::cerr <<"[OK]"<<std::endl;);
	}
      else
	{
	  data_ = 0;
	}

    }
}

Sample::Sample(const std::string& filename):
  Sound(),
  channel_(-1)
{
  load(filename);
}

Sample::~Sample()
{
  if(data_)
    {
      debugN(17, std::cerr << "Sample freeing chunk ... ";);
      
      if(mixer_->audioAvailable())
	{
	  Mix_FreeChunk(data_);
	}
      else
	{
	  //do it by hand...
	  
	  if( data_->allocated ) 
	    {
	      free(data_->abuf);
	    }
	  free(data_);       
	}

      debugN(17, std::cerr << "[OK]" << std::endl;);
    }
}

int
Sample::setVolume(int vol_)
{
  int tmp = 0;
  
  if(data_ && channel_ > 0)
    {
      Mix_Volume(channel_, vol_);
      tmp  = Mix_VolumeChunk(data_, vol_);
    }

  return tmp;
}

bool
Sample::load(const std::string& filename )
{
  bool success = false;

  if((mixer_ != 0)&&(mixer_->audioAvailable()))
     {
       debugN(17,std::cerr << "loading wav ...";);
       data_ = Mix_LoadWAV(filename.c_str());
       debugN(17,std::cerr << "ok." << std::endl;);
       
       if(data_ == 0)
	 {
	   debugN(17,std::cerr << "Sample: unable to load file." << std::endl;);
	 }
       else
	 {
	   success = true;
	 }
     }
  else
    {
      data_ = 0;
    }
  
  return success;
}


bool
Sample::play(int loops) const
{
  bool success = false;

  if(mixer_ == 0)
    return false;
  if(!mixer_->audioAvailable())
    return false;

  if(data_ != 0)
    {
      //play on first available channel
      channel_ = Mix_PlayChannel(-1, data_, loops);
      if(channel_ >= 0)
	{
	  success = true;
	}
    }

  return success;
}
  
void 
Sample::stop() const
{
  if(!mixer_)
    return;
  if(!mixer_->audioAvailable())
    return;

    if(channel_ >= 0)
      Mix_HaltChannel(channel_);
}

bool
Sample::playing() const
{
  if(channel_ < 0)
    return false;

  if(!mixer_)
    return false;
  if(!mixer_->audioAvailable())
    return false;  
  
  return Mix_Playing(channel_);
}

#else
Sample::Sample() :
  Sound()
{
}

Sample::Sample(const Sample& sample) :
  Sound(sample)
{

}

Sample::Sample(const std::string& filename):
  Sound(),
  channel_(-1)
{
  load(filename);
}

Sample::~Sample()
{
 
}

int
Sample::setVolume(int)
{
  return 0;
}

bool
Sample::load(const std::string&)
{
  return false;
}


bool
Sample::play(int ) const
{
  return false;
}

void 
Sample::stop() const
{

}

bool
Sample::playing() const
{
  return false;
}

#endif
}

