/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#undef DEBUG

#include "rootwindow.h"
#include "screensurface.h"
#include "debug.h"
#include "application.h"
#include "directsurface.h"

#include <assert.h>
#undef DEBUG

namespace uta {

RootWindow::RootWindow(int resX, int resY, int bpp, bool fullscreen, bool resizeable) :
  Widget()
{
  debugN(17,cerr << "creating RootWindow...";);

  //grab the access to the rootwindow
  //the application may release it
  access_.grab();

  //set up Rect members
  upperLeft_ = Point(0,0);
  lowerRight_ = Point(resX, resY);

  rect_ = Rect(0,0,resX, resY);
  window_ = Rect(0,0,resX, resY);

  parent_ = 0;

  initMembers();

  
  surface_ = new ScreenSurface(resX, resY, bpp, fullscreen, resizeable);

  fullscreen_ = static_cast<ScreenSurface*>(surface_)->fullscreen(); 
  doublebuffered_ = static_cast<ScreenSurface*>(surface_)->doublebuffered();
  
  if(doublebuffered_)
    {
      screen_ = surface_;
      //tripple buffering ... some what slow .... need to think of a better way
      surface_ = new Surface(resX, resY);
      
      surface_->setTransparency(false);
    }
  else
    {
      screen_ = 0;
    }
  
  parentSurface_ = 0;

  debugN(17,cerr << "done." << endl;);
}


RootWindow::~RootWindow()
{
  if(screen_ != 0)
    delete screen_;
}

void
RootWindow::setTitle(const char *title)
{
  SDL_WM_SetCaption(title, 0); 
}


void
RootWindow::update()
{
  if(hidden_ || closed_)
    return;
  if(disabled_)
    return;

  //ignore disable and hide requests
  if(hideRequest_)
    {
      hideRequest_ = false;
      //      hidden_ = true;
    }

  if(disableRequest_)
    {
      disableRequest_ = false;
      //      disabled_ = true;
    }
  
  bool update_ = needsUpdate_;

  createChild();
 
  if(needsReblit_ || update_)
    {
      dirtyRects_.push_back(globalCoord(window_));
    }

//   //create the child widgets..
//   createChilds();

  std::list<Widget*> closeList;
  
  ///child list iterator.
  std::list<Widget*>::iterator childItr_;
  childItr_ = childs_.begin();
  while(childItr_ != childs_.end())
    {
      //we must not call update() of an closed widget here
      //since closed widgets selfdestruct when calling update()
      if(!(*childItr_)->isClosed())
	{	  
	 
	  //only update other widgets, if no dialog is present
	  if((*childItr_)->needsUpdate() && dialogs_.empty())
	    (*childItr_)->update();
	 	  
	  else if(update_)
	    (*childItr_)->blit();
  	  
	}      
      else
	{
	  closeList.push_back(*childItr_);
	  needsUpdate_ = true;
	}
  
       //if we update the whole screen, forget about all the other dirtyRects
   //    if(update_ || needsReblit_)
 //	(*childItr_)->dirtyRects_.clear();
   //    else
	(*childItr_)->addUpdateRects(dirtyRects_);
      
      childItr_ ++;
    }
    
  if(dialogs_.size() > 0)
    {
      //special treatment for dialogs (modal)
      //bring it ALWAYS to the front
      if(!dialogs_.back()->isClosed())
	{	 
 	  if(dialogs_.back()->needsUpdate())
	    dialogs_.back()->update();
	  else
	    if(update_)
	      dialogs_.back()->blit();	  
	}      
      else	
	{
	  closeList.push_back(dialogs_.back());
	  needsUpdate_ = true;
	}

      dialogs_.back()->addUpdateRects(dirtyRects_);
    }
  
  //kill the ones which are already dead (?)
  Widget *childPtr;
  while(!closeList.empty())
    {
      childPtr = closeList.front();
      closeList.pop_front();
      //deleting a child modifies list badly
      //this is why the local copy childPtr is needed here
      //but delete it only if the autoDelete_ flag is set
      if(childPtr->isAutoDelete())
	 delete childPtr;
      else
	childPtr->deParent();
    }
  
  needsReblit_ = false;

}

void
RootWindow::refreshRect(const Rect& rect)
{
  //dirtyRects_.push_back(rect);
  
  if(doublebuffered_)
    {
      surface_->blit(screen_, rect, localCoord(rect));	      
      static_cast<ScreenSurface*>(screen_)->flip();  
    }
  else
    {
      static_cast<ScreenSurface*>(surface_)->update(rect);	      
    }

}


void
RootWindow::updateScreen()
{
    std::list<Rect>::iterator itr = dirtyRects_.begin();
    std::list<Rect> mergedRects;
    
    while(!dirtyRects_.empty())
    {
	Rect r = dirtyRects_.front();
	bool done = false;
	dirtyRects_.pop_front();
	do 
	{
	    itr = dirtyRects_.begin();
	    while(itr != dirtyRects_.end())
	    {
		if(r.intersect(*itr).isValid())
		{
		    r = r.unite(*itr);
		    break;
		}
		itr++;
	    }
	    if(itr != dirtyRects_.end())
		dirtyRects_.erase(itr);		   
	    else
	    {
		done = true;
		mergedRects.push_front(r);
	    }
	} while(!done);
    }

    dirtyRects_ = mergedRects;

    if(doublebuffered_)
    {
      if(!dirtyRects_.empty())
	{  
	  Rect dest;
	  Rect src;
	  std::list<Rect>::iterator itr = dirtyRects_.begin();
	  
	  dest = *itr++;
	  while(itr != dirtyRects_.end())
	      dest = dest.unite(*itr++);
	  src = localCoord(dest);
	  surface_->blit(screen_, dest, src);

// 	  while(itr != dirtyRects_.end())
// 	    {
// 	      src = localCoord(*itr);
// 	      dest = *itr;
// 	      surface_->blit(screen_, dest, src);	      
// 	      itr++;
// 	    }
	}
      static_cast<ScreenSurface*>(screen_)->flip();     
    }
  else
    {
      if(!dirtyRects_.empty())
	{   
	  Rect dest;
	  std::list<Rect>::iterator itr = dirtyRects_.begin();

	  dest = *itr++;
	  while(itr != dirtyRects_.end())
	      dest = dest.unite(*itr++);
	    
	  static_cast<ScreenSurface*>(surface_)->update(dest);

// 	  while(itr != dirtyRects_.end())
// 	    {    
// 	      dest = *itr;
// 	      static_cast<ScreenSurface*>(surface_)->update(dest);	      
// 	      itr++;
// 	    }
	}
    }
    
  dirtyRects_.clear(); 
}

 
bool
RootWindow::runDialog(Widget* dialog)
{
  assert(dialog != 0);

  dialogs_.push_back(dialog);

  Rect src(dialog->upperLeft(), dialog->lowerRight());

  for(std::list<DirectSurface*>::iterator i=directSurfaces_.begin();
      i != directSurfaces_.end(); i++)
  {
      Rect cover = (*i)->rect().intersect(src);
      if(cover.isValid())
      {
	  // convert global coords to relative coords
	  cover.translate(- (*i)->rect().upperLeft().x,
			  - (*i)->rect().upperLeft().y);	  
	  (*i)->covered(cover);
      }
  }
  
  
  Surface* back = new Surface(dialog->width(), dialog->height());
  surface_->blit(back, Rect(0,0,dialog->width(), dialog->height()), src);
  backings_.push_back(back);

//  dialog->update();

  return true;
}



bool
RootWindow::closeDialog()
{
  if(dialogs_.size() == 0)
    {
      debugN(17, cerr << "uta::RootWindow::closeDialog(): there is no dialog to close !" 
		   << endl;);
      return false;
    }

 
  Surface* surf = backings_.back();
  Widget* dialog = dialogs_.back();

  Rect dest(dialog->upperLeft(), dialog->lowerRight());

  for(std::list<DirectSurface*>::iterator i=directSurfaces_.begin();
      i != directSurfaces_.end(); i++)
  {
      Rect cover = (*i)->rect().intersect(dest);
      if(cover.isValid())
      {
	  // convert global coords to relative coords
	  cover.translate(- (*i)->rect().upperLeft().x,
			  - (*i)->rect().upperLeft().y);
	  (*i)->exposed(cover);
      }
  }

  surf->blit(surface_, dest);
  refreshRect(dest);

  dialogs_.pop_back();
  backings_.pop_back();
  
//  needsUpdate_ = true;
//  sync();

  return true;
}

 
bool 
RootWindow::processEvent(const Event* event)
{
  bool processed = false;

  if(dialogs_.size() > 0)
    {
      //a modal dialog is open, so it will receive all events
      dialogs_.back()->handleEvent(event);
      
      //we swallow this event, so no one else may react
      processed = true;
    }

  return processed;
}

void
RootWindow::sync()
{ 
  update(); 
  updateScreen(); 
}

void
RootWindow::registerDirect(DirectSurface* direct)
{
    directSurfaces_.push_back(direct);
}

void
RootWindow::unregisterDirect(DirectSurface* direct)
{
    directSurfaces_.remove(direct);
}



} // namespace uta
