/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "button.h"
#include "application.h"
#include "resources.h"
#include <sigc++/bind.h>

namespace uta {

Button::Button(Widget *parent, int x, int y, int w, int h, int id) :
  Label(parent, x, y, w, h, FONT_RES("button_font")),
  isPressed_(false),
  autoRepeat_(false),
  id_(id)
{
  
  pressed.connect(slot(uapp->soundMapper(), &SoundMapper::press));
  released.connect(slot(uapp->soundMapper(), &SoundMapper::release));

  pressed.connect(bind(pressed1.slot(),id_));
  released.connect(bind(released1.slot(), id_));
  clicked.connect(bind(clicked1.slot(), id_));

  uapp->timer100.connect(slot(this, &Button::repeat));

  setClickToFocus(true);

  if(RES_AVAILABLE("button_color"))
    Widget::setColor(COLOR_RES("button_color"));
  if(RES_AVAILABLE("button_surface"))
    Widget::setBackground(SURFACE_RES("button_surface"),false);
}

Button::~Button()
{
  //EMPTY
}

bool
Button::processEvent(const Event* event)
{

  bool handled = false;

  if(autoRepeat_ && isPressed_ && event->type() == Event::TIMER)
    {
      //catch the system timer
      //this timer never gets handled
      if(((TimerEvent*)event)->id() == 0)
	{
	  pressed();
	}
    }
  
  if(mouseFocus_)
    {
      if(event->type() == Event::BUTTON)
	{
	  
	  if(((ButtonEvent*)event)->buttons() == ButtonEvent::BUTTON_LEFT)
	    {
	      if((((ButtonEvent*)event)->state() == ButtonEvent::PRESSED))
		{
		  isPressed_ = true;
		  pressed();
		}
	      
	      if(isPressed_ &&
		 (((ButtonEvent*)event)->state() == ButtonEvent::RELEASED))
		{
		  isPressed_ = false;
		  released();
		  clicked();
		}
	      handled = true;
	      needsUpdate_ = true;
	    }
	}
    };

  if(haveFocus_)
    {
      if(event->type() == Event::KEY)
	{
	  if(((KeyEvent *)event)->state() == KeyEvent::PRESSED)
	    {
	      char input = ((KeyEvent *)event)->ascii() ;
	     
	      if((input =='\n') || (input == '\r') || (input == ' '))
		{
		  isPressed_ = true;
		  pressed();
		  handled = true;
		  needsUpdate_ = true;		  
		}

	      if(((KeyEvent *)event)->key() == KEY_TAB)
		{
		  if(parent_)
		    if(((KeyEvent *)event)->modifier() && KEY_MOD_SHIFT)
		      parent_->prevFocus();
		    else
		      parent_->nextFocus();
		  handled = true;
		}
	    }

	  if(((KeyEvent *)event)->state() == KeyEvent::RELEASED && isPressed_ )
	    {
	      char input = ((KeyEvent *)event)->ascii() ;
	     
	      if((input =='\n') || (input == '\r') || (input == ' '))
		{
		  isPressed_ = false;
		  released();
		  clicked();	  
		  handled = true;
		  needsUpdate_ = true;
		}
	    }
	}     
      
    }
  else
    {
      // have lost focus, ensure we won't get triggered on a key release
      isPressed_ = false;
    }
    
  
  if(isPressed_ && !mouseFocus_)
    {
      isPressed_ = false;
      released();
      needsUpdate_ = true;
    }
  
  
  return handled;
}

void Button::press()
{
  if(isPressed_)
    {
      pressed();
    }
  else
    {
      isPressed_ = true;
      pressed();
      //request screen update
      needsUpdate_ = true;
    }
}

void Button::release()
{
  if(isPressed_)
    {
      isPressed_ = false;
      released();
      clicked();
      //request screen update
      needsUpdate_ = true;
    }
  else
    {
      released();
    }
}

  
void Button::repeat()
{
  if(haveFocus_ && isPressed_ && autoRepeat_)
    {
      isPressed_ = false;
      clicked();
      isPressed_ = true;
    }
}
}

