/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)

  Copyright (C) 1999  Karsten Laux <klaux@student.uni-kl.de>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.


  The read_png() routine has been copied from SDL_image-0.9.tar.gz
      IMGLIB:  An example image loading library for use with SDL
      Copyright (C) 1999  Sam Lantinga
      Changes: 
        - replace IMG_error by cerr
  
  The save_png() routine was originally created by  
  Philippe Lavoie <lavoie@zeus.genie.uottawa.ca> on 2 November 1998

  here is it's copyright notice:
=============================================================================
        File: SDL_png.c
     Purpose: A PNG loader and saver for the SDL library      
    Revision: 
  Created by: Philippe Lavoie          (2 November 1998)
              lavoie@zeus.genie.uottawa.ca
 Modified by: 

 Copyright notice:
          Copyright (C) 1998,1999 Philippe Lavoie
 
          This library is free software; you can redistribute it and/or
          modify it under the terms of the GNU Library General Public
          License as published by the Free Software Foundation; either
          version 2 of the License, or (at your option) any later version.
 
          This library is distributed in the hope that it will be useful,
          but WITHOUT ANY WARRANTY; without even the implied warranty of
          MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
          Library General Public License for more details.
 
          You should have received a copy of the GNU Library General Public
          License along with this library; if not, write to the Free
          Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, SA.

    Comments: The load and save routine are basically the ones you can find
             in the example.c file from the libpng distribution.

=============================================================================


  Modifications by Karsten Laux:
  * replaced SDL_SetError by cerr 
  * filehandling and writing merged to one function
  * moved vars declaration beneath setjmp to get rid of compiler warnings
*/


#include "surface.h"
#include "painter.h"
#include "color.h"
#
#include "debug.h"

#include <iostream>
#include <cstdio>
#include <cstdlib>
#include <cassert>
#include <cctype>

#include <png.h>
#include <SDL/SDL_endian.h>

namespace uta {

Pixelformat pixelformat_;

//this is used to parse gimp header imagefiles

#define HEADER_PIXEL(data,pixel) \
  pixel[0] = (((data[0] - 33) << 2) | ((data[1] - 33) >> 4)); \
  pixel[1] = ((((data[1] - 33) & 0xF) << 4) | ((data[2] - 33) >> 2)); \
  pixel[2] = ((((data[2] - 33) & 0x3) << 6) | ((data[3] - 33))); \
  data += 4;


SDL_Surface* read_png(const char* filename)
{
  FILE *fp;
  fp = fopen(filename, "rb");
  if( fp == 0)
    {
      debugN(17,cerr << "     ! Couldn't open " << filename << " for reading." 
	     << endl;);
      return 0;
    }
  else
    {
      debugN(17,cerr << "     * reading file " << filename << endl;);
    }

  png_structp png_ptr;
  png_infop info_ptr;
  png_infop end_info;

  SDL_Surface *surface;
  
  png_uint_32 width, height;
  int bit_depth, color_type, interlace_type;
  Uint32 Rmask;
  Uint32 Gmask;
  Uint32 Bmask;
  Uint32 Amask;
  SDL_Palette *palette;
  png_bytep *row_pointers;
  unsigned int row, i;
  int ckey = -1;
  png_color_16 *transv;


  png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
				   0,0,0);

  if (png_ptr == 0){
    cerr << "     ! Couldn't allocate the memory for a PNG info structs." << endl;
    return 0;
  }
  
  /* Allocate/initialize the memory for image information. */
  info_ptr = png_create_info_struct(png_ptr);
  if (info_ptr == 0)
    {
      cerr << "      ! Couldn't create the image information for a PNG file" << endl;
      png_destroy_read_struct(&png_ptr, (png_infopp)0, (png_infopp)0);
      return 0;
    }
  end_info = png_create_info_struct(png_ptr);
  if (end_info == 0)
    {
      cerr << "     ! Couldn't create the image information for a PNG file" << endl;
      png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)0);
      return 0;
    }

  /* Set error handling if you are using the setjmp/longjmp method (this is
   * the normal method of doing things with libpng).  REQUIRED unless you
   * set up your own error handlers in the png_create_read_struct() earlier.
   */
  if (setjmp(png_ptr->jmpbuf))
    {
      /* Free all of the memory associated with the png_ptr and info_ptr */
      png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)0);
      cerr << "     ! Error reading the PNG file."<< endl;
      /* If we get here, we had a problem reading the file */
      return 0;
   }
  

  /* Set up the input control if you are using standard C streams */
  png_init_io(png_ptr, fp); 

  /* Read PNG header info */
  png_read_info(png_ptr, info_ptr);
  png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
	       &color_type, &interlace_type, NULL, NULL);


  /* tell libpng to strip 16 bit/color files down to 8 bits/color */
  png_set_strip_16(png_ptr) ;
  
  /* Extract multiple pixels with bit depths of 1, 2, and 4 from a single
   * byte into separate bytes (useful for paletted and grayscale images).
   */
  png_set_packing(png_ptr);

#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) < \
    SDL_VERSIONNUM(1, 1, 5)
  //In PNG files, the alpha channel in an image is the level of opacity. 
  //so we invert it here ....
  //because SDL < 1.1.5 thinks of it as transparency
  png_set_invert_alpha(png_ptr);
#endif


  /* For images with a single "transparent colour", set colour key;
     if more than one index has transparency, use full alpha channel */
  if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS)) 
    {
      int num_trans;
      Uint8 *trans;
      png_get_tRNS(png_ptr, info_ptr, &trans, &num_trans,
		   &transv);
      if(color_type == PNG_COLOR_TYPE_PALETTE) 
	{
	  if(num_trans == 1) 
	    {
	      /* exactly one transparent value: set colour key */
	      ckey = trans[0];
	    } 
	  else
	    png_set_expand(png_ptr);
	} 
      else
	ckey = 0; /* actual value will be set later */
    }
  
  if ( (color_type == PNG_COLOR_TYPE_GRAY) ||
       (color_type == PNG_COLOR_TYPE_GRAY_ALPHA) )
    png_set_gray_to_rgb(png_ptr);
  
  png_read_update_info(png_ptr, info_ptr);
  
  png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
	       &color_type, &interlace_type, NULL, NULL);
  
  /* Allocate the SDL surface to hold the image */
  Rmask = Gmask = Bmask = Amask = 0 ; 
  if ( color_type != PNG_COLOR_TYPE_PALETTE ) 
    {
      if ( SDL_BYTEORDER == SDL_LIL_ENDIAN ) 
	{
	  Rmask = 0x000000FF;
	  Gmask = 0x0000FF00;
	  Bmask = 0x00FF0000;
	  Amask = (info_ptr->channels == 4) ? 0xFF000000 : 0;
	} 
      else 
	{
	  int s = (info_ptr->channels == 4) ? 0 : 8;
	  Rmask = 0xFF000000 >> s;
	  Gmask = 0x00FF0000 >> s;
	  Bmask = 0x0000FF00 >> s;
	  Amask = 0x000000FF >> s;
	}
    }
  surface = SDL_AllocSurface(SDL_SWSURFACE, width, height,
			     bit_depth*info_ptr->channels, 
			     Rmask,Gmask,Bmask,Amask);
  
  if ( surface == NULL ) 
    {
      cerr << "      ! out of memory"<<endl;
      goto done;
    }
  
  if(ckey != -1) 
    {
      if(color_type != PNG_COLOR_TYPE_PALETTE)
	ckey = SDL_MapRGB(surface->format, transv->red,
			  transv->green, transv->blue);
      SDL_SetColorKey(surface, SDL_SRCCOLORKEY, ckey);
    }
  
  /* Create the array of pointers to image data */
  row_pointers = (png_bytep*) malloc(sizeof(png_bytep)*height);
  if ( (row_pointers == NULL) ) 
    {
      cerr << "      ! out of memory !" << endl;
      SDL_FreeSurface(surface);
      surface = NULL;
      goto done;
    }
  for (row = 0; row < height; row++) 
    {
      row_pointers[row] = (png_bytep)
	(Uint8 *)surface->pixels + row*surface->pitch;
    }
  
  /* Read the entire image in one go */
  png_read_image(png_ptr, row_pointers);
  
  /* read rest of file, get additional chunks in info_ptr - REQUIRED */
  png_read_end(png_ptr, info_ptr);
  
  /* Load the palette, if any */
  palette = surface->format->palette;
  if ( palette && (info_ptr->num_palette > 0) ) 
    {
      palette->ncolors = info_ptr->num_palette; 
      for( i=0; i<info_ptr->num_palette; ++i ) 
	{
	  palette->colors[i].b =(Uint8)info_ptr->palette[i].blue;
	  palette->colors[i].g =(Uint8)info_ptr->palette[i].green;
	  palette->colors[i].r =(Uint8)info_ptr->palette[i].red;
	}
    }
  
 done:	/* Clean up and return */
  png_destroy_read_struct(&png_ptr, info_ptr ? &info_ptr : (png_infopp)0,
			  (png_infopp)0);
  if ( row_pointers ) 
    {
      free(row_pointers);
    }
  
//   //read file information
//   png_read_info(png_ptr, info_ptr);

//   //and expand the data into seperate variables
//   png_get_IHDR(png_ptr, info_ptr, 
// 	       &width, &height, &bit_depth, &color_type,
// 	       &interlace_type, 0, 0);

 
//   hasAlpha = color_type & PNG_COLOR_MASK_ALPHA;
//   isIndexed = color_type & PNG_COLOR_MASK_PALETTE;

//   //treat indexed images
//   if(color_type == PNG_COLOR_TYPE_PALETTE && bit_depth <= 8)
//     {
//       png_set_palette_to_rgb(png_ptr);
  
//       //expand grayscale images to full 8 bit
//       if(color_type == PNG_COLOR_TYPE_GRAY && bit_depth < 8) 
// 	png_set_gray_1_2_4_to_8(png_ptr);
 
//       //calculate alpha channel from given transparent colors
//       if(color_type == PNG_COLOR_TYPE_PALETTE)
// 	if(png_get_valid(png_ptr, info_ptr,PNG_INFO_tRNS)) 
// 	  png_set_tRNS_to_alpha(png_ptr);
//     }


    
//   if (color_type == PNG_COLOR_TYPE_GRAY ||
//       color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
//         png_set_gray_to_rgb(png_ptr);

//   png_read_update_info(png_ptr, info_ptr);
  
//   png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth, &color_type,
//  	       &interlace_type, 0, 0);

  

//   /* copy the information from the PNG format to the SDL format */
//   /* Allocate the SDL surface to hold the image */
//   Uint32 Rmask, Gmask, Bmask, Amask;

//   Rmask = Gmask = Bmask = Amask = 0 ; 
//   if ( bit_depth == 8 ) {
//     if ( SDL_BYTEORDER == SDL_LIL_ENDIAN ) {
//       Rmask = 0x000000FF;
//       Gmask = 0x0000FF00;
//       Bmask = 0x00FF0000;
//       if(hasAlpha)
// 	Amask = 0xFF000000;
//     } else {
//       Rmask = 0xFF000000;
//       Gmask = 0x00FF0000;
//       Bmask = 0x0000FF00;
//       if(hasAlpha)
// 	Amask = 0x000000FF;
//     }
//   }
  
    
//     debugN(17,cerr<<"     * allocating surface "<<width<<" x "<<height<<" pixels"<< endl;);

//   /* Create a 32bit surface, note that the colors are RGB ordered */
//   if(hasAlpha)
//     {
//       debugN(17,cerr<<"    * creating alpha-blit capable surface."<<endl;);
//       surface = SDL_AllocSurface(SDL_SWSURFACE |SDL_SRCALPHA,
// 				 width, 
// 				 height, 
// 				 bit_depth * 4, 
// 				 Rmask, Gmask, Bmask, Amask );
//     }
//   else
//     {
//       debugN(17,cerr<<"    * creating normal surface."<<endl;);
//       surface = SDL_AllocSurface(SDL_SWSURFACE,
// 				 width, 
// 				 height, 
// 				 bit_depth * 3, 
// 				 Rmask, Gmask, Bmask, Amask );
//     }
  

//   if ( surface == 0 ) 
//     {
//       cerr << "     ! failed to allocate surface memory." << endl;
//       return(0);
//     }

//   /* Allocate the memory to hold the image using the fields of info_ptr. */
//   row_pointers = (png_bytep*) malloc(sizeof(png_bytep)*height) ;
  
//   /* streaming: 
//      let libpng copy the imagedata directly to the surface
//      So create a list of pointers to the beginning of each scanline
//      inside the surface memory:
//   */
//   for (row = 0; row < (int)height; row++)
//     row_pointers[row] = (Uint8 *)surface->pixels + row*surface->pitch;

//   /* Read the entire image in one go, which will dump the pixels
//      directly into the surface memory */
//   png_read_image(png_ptr, row_pointers);

//   /* read rest of file, and get additional chunks in info_ptr - REQUIRED */
//   png_read_end(png_ptr, end_info);

  
//   /* clean up after the read, and free any memory allocated - REQUIRED */
//   png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);

//   // free the list of pointers
//   free(row_pointers) ;
  
  //close the file
  fclose(fp);

  return surface ; 
}



int write_png(const char* filename, SDL_Surface *surface)
{
  FILE *fp;

  fp = fopen(filename, "wb");
  if ( fp == 0 ) {
    cerr << "Couldn't open " << filename << "for writing" << endl;
    return -1;
  }

  png_structp png_ptr;
  png_infop info_ptr;
  
  png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,0,0,0);
  
  if (png_ptr == 0){
    cerr << "Couldn't create a write structure for the PNG file." << endl;
    fclose(fp);
    return -1;
  }

  /* Allocate/initialize the image information data.  REQUIRED */
  info_ptr = png_create_info_struct(png_ptr);
  if (info_ptr == 0){
    cerr << "Couldn't create the image information data." << endl;
    png_destroy_write_struct(&png_ptr,  (png_infopp)0);
    fclose(fp);
    return -1;
  }
  
  /* Set error handling.  REQUIRED if you aren't supplying your own
   * error hadnling functions in the png_create_write_struct() call.
   */
  if (setjmp(png_ptr->jmpbuf)){
    /* If we get here, we had a problem reading the file */
    cerr << "Error reading the file." << endl;
    png_destroy_write_struct(&png_ptr,  (png_infopp)0);
    fclose(fp);
    return -1;
  }

  int retval = 0;
  int number_passes ;
  png_colorp palette = 0;
  int i,y,pass;
  png_bytepp datap = 0;
  png_bytep *row_pointers = 0;

  /* set up the output control if you are using standard C streams */
  png_init_io(png_ptr, fp);
  
  /* Set the image information here.  Width and height are up to 2^31,
   * bit_depth is one of 1, 2, 4, 8, or 16, but valid values also depend on
   * the color_type selected. color_type is one of PNG_COLOR_TYPE_GRAY,
   * PNG_COLOR_TYPE_GRAY_ALPHA, PNG_COLOR_TYPE_PALETTE, PNG_COLOR_TYPE_RGB,
   * or PNG_COLOR_TYPE_RGB_ALPHA.  interlace is either PNG_INTERLACE_NONE or
   * PNG_INTERLACE_ADAM7, and the compression_type and filter_type MUST
   * currently be PNG_COMPRESSION_TYPE_BASE and PNG_FILTER_TYPE_BASE. 
   */
  if(surface->format->palette){
    png_set_IHDR(png_ptr,info_ptr,
		 surface->w,
		 surface->h,
		 surface->format->BitsPerPixel,
		 PNG_COLOR_TYPE_PALETTE,
		 PNG_INTERLACE_ADAM7,
		 PNG_COMPRESSION_TYPE_BASE, 
		 PNG_FILTER_TYPE_BASE);

    palette = (png_colorp)png_malloc(png_ptr, surface->format->palette->ncolors * sizeof (png_color));
    png_set_PLTE(png_ptr, info_ptr, palette, surface->format->palette->ncolors);
    for(i=0;i<info_ptr->num_palette;++i){
	    info_ptr->palette[i].blue = surface->format->palette->colors[i].b ;
	    info_ptr->palette[i].green = surface->format->palette->colors[i].g ;
	    info_ptr->palette[i].red = surface->format->palette->colors[i].r ;
    }       
  }
  else{
    if(surface->format->BitsPerPixel != 32 ){
      cerr << "The author of the PNG saver was lazy and doesn't support something other than 32bpp in non palette images." << endl;
      fclose(fp);
      return -1 ; 
    }
    if(surface->format->BitsPerPixel == 32){
      png_set_IHDR(png_ptr,info_ptr,
		   surface->w,
		   surface->h,
		   8,
		   PNG_COLOR_TYPE_RGB_ALPHA,
		   PNG_INTERLACE_NONE,
		   PNG_COMPRESSION_TYPE_BASE, 
		   PNG_FILTER_TYPE_BASE);
    }
    else{
      png_set_IHDR(png_ptr,info_ptr,
		   surface->w,
		   surface->h,
		   8,
		   PNG_COLOR_TYPE_RGB,
		   PNG_INTERLACE_NONE,
		   PNG_COMPRESSION_TYPE_BASE, 
		   PNG_FILTER_TYPE_BASE);
    }
    
  }
  
  /* Optionally write comments into the image */
  /*
    text_ptr[0].key = "Title";
    text_ptr[0].text = "unknown";
    text_ptr[0].compression = PNG_TEXT_COMPRESSION_NONE;
    text_ptr[1].key = "Author";
    text_ptr[1].text = "SDL_png";
    text_ptr[1].compression = PNG_TEXT_COMPRESSION_NONE;
    text_ptr[2].key = "Description";
    text_ptr[2].text = "This file was generated by the SDL library.";
    text_ptr[2].compression = PNG_TEXT_COMPRESSION_zTXt;
    png_set_text(png_ptr, info_ptr, text_ptr, 3);
  */
  
  /* Write the file header information.  REQUIRED */
  png_write_info(png_ptr, info_ptr);
  
  /* Once we write out the header, the compression type on the text
   * chunks gets changed to PNG_TEXT_COMPRESSION_NONE_WR or
   * PNG_TEXT_COMPRESSION_zTXt_WR, so it doesn't get written out again
   * at the end.
   */
  

  
  /* The number of passes is either 1 for non-interlaced images,
   * or 7 for interlaced images.
   */
  number_passes = png_set_interlace_handling(png_ptr);
  
  if(surface->format->BitsPerPixel==32){
    row_pointers = (png_bytep*) malloc(sizeof(png_bytep)*surface->h) ;
    for (y = 0; y < surface->h; y++){
      row_pointers[y] = (png_bytep) malloc(png_get_rowbytes(png_ptr, info_ptr));
    }
  }
  Uint32 pixel;
  uta::Color col;

  for (pass = 0; pass < number_passes; ++pass)
    {
      for (y = 0; y < surface->h; ++y)
	{
	  if(surface->format->BitsPerPixel == 32)
	    {
	      for(i=0;i< surface->w*4 ;i+=4)
		{
		  pixel = *((Uint32*)((Uint8*)surface->pixels + i + y*surface->pitch));
		  col = pixelformat_.mapToColor(pixel);
		  if ( SDL_BYTEORDER == SDL_BIG_ENDIAN ) 
		    {
		      row_pointers[y][i+3] = col.r;
		      row_pointers[y][i+2] = col.g;
		      row_pointers[y][i+1] = col.b;
		      
#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) >= \
    SDL_VERSIONNUM(1, 1, 5)
		      row_pointers[y][i] = col.a ;
#else
		      row_pointers[y][i] = 255-col.a ;
#endif
		    } 
		  else 
		    {
		      row_pointers[y][i] = col.r;
		      row_pointers[y][i+1] = col.g;
		      row_pointers[y][i+2] = col.b;
		      
#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) >= \
    SDL_VERSIONNUM(1, 1, 5)
		      row_pointers[y][i+3] = col.a ;
#else
		      row_pointers[y][i+3] = 255-col.a ;
#endif
		    }
		}

	      png_write_rows(png_ptr, &row_pointers[y], 1);
	    }
	  else
	    {
	      *datap = ((Uint8 *)surface->pixels + y*surface->pitch) ;
	      png_write_rows(png_ptr, datap, 1);
	    }
	}
    }
  
  /* It is REQUIRED to call this to finish writing the rest of the file */
  png_write_end(png_ptr, info_ptr);
  
  /* if you malloced the palette, free it here */
  if(surface->format->palette){
    free(info_ptr->palette);
  }
  
  /* if you allocated any text comments, free them here */
  
  /* clean up after the write, and free any memory allocated */
  png_destroy_write_struct(&png_ptr, (png_infopp)0);
  
  
  if(surface->format->BitsPerPixel==32){
    for (y = 0; y < surface->h; y++){
      free(row_pointers[y]) ;
    }
    free(row_pointers) ;
  }
  
  if ( fclose(fp) == EOF ) 
    {
      cerr << "Couldn't close the file %s" << endl;
      retval = -1;
    }
  
  return retval;
}

bool Surface::writeToFile(const string& filename)
{
  
  bool success = false;
  
  // set the global var
  pixelformat_ = format;

  if(filename.find(".bmp") != string::npos ||
     filename.find(".BMP") != string::npos)
    {
      success = !SDL_SaveBMP(sdlSurface_, filename.c_str());
    }
  else
    {
      success = !write_png(filename.c_str(), sdlSurface_);
    }

  if(success)
    {    
      cerr << "Surface wrote \""<<filename<<"\" successfully." << endl;
    }
  else
    {
      cerr << "Surface could not write \""<<filename<<"\" to disk."  << endl;
    }

  return success;
}


bool Surface::readFromFile(const string& filename)
{
  bool success = false;

  debugN(17,cerr<<"Surface::readFromFile("<<filename<<")"<<endl;);

  //free old surface data (if any)
  clearSurfaceData();

  //reset pixelformat
  format = Pixelformat::UNKNOWN;
  int pixformat;

  if(filename.find(".bmp") != string::npos ||
     filename.find(".BMP") != string::npos)
    {
      sdlSurface_ = SDL_LoadBMP(filename.c_str());
    }
  else
    {
      //default is to try to load PNG
      sdlSurface_ = read_png(filename.c_str());
    }

  if(sdlSurface_)
    {
      debugN(17,cerr << "     * successfully read file \"" << filename 
	    << "\""<<endl;);
      pixformat = Pixelformat::identify(sdlSurface_->format->BytesPerPixel,
					sdlSurface_->format->Rmask,
					sdlSurface_->format->Gmask,
					sdlSurface_->format->Bmask,
					sdlSurface_->format->Amask);
				
      
      if(pixformat == Pixelformat::UNKNOWN)
	{
	  debugN(17,
		 cerr.setf(ios::hex);
		 cerr << "     ! surface is in unidentified pixelformat !" 
		 << endl;
		 cerr << sdlSurface_->format->BytesPerPixel << "," 
		 << sdlSurface_->format->Rmask << "," 
		 << sdlSurface_->format->Gmask << "," 
		 << sdlSurface_->format->Bmask << "," 
		 << sdlSurface_->format->Amask << endl;
		 cerr.unsetf(ios::hex);
		 );
	}
      else
	{
	  format = pixformat;
	  debugN(17,cerr << "     * got known pixelformat:"
		 << format.asString() << endl; );
 
	  gatherPalette();

// 	  setAlpha(0);
// 	  setTransColor(Color(0,0,0));
//	  setTransparency(true);

	  success = true;
	}
    }
  
  return success;
}

bool Surface::readFromHeader(char* header_data, unsigned w, unsigned h)
{
  bool success = false;
  
  //free old surface data (if any)
  clearSurfaceData();
  
  format = Pixelformat::BGR888;
  sdlSurface_ = SDL_AllocSurface(SDL_SWSURFACE |SDL_SRCALPHA ,
				 w,
				 h,
				 format.bpp()*8,
				 format.rMask(),
				 format.gMask(),
				 format.bMask(),
				 format.aMask());
  
  gatherPalette();
 
  
  success = true;
  
  debugN(17, cerr << "parsing header_data ..."<< endl;);
  Painter* painter = new Painter(this);
  
  unsigned char pixel[3];
  //header_data is the address of the image data
  char *data = header_data;
  
  for(unsigned int y = 0; y < h; y++)
    for(unsigned int x = 0; x < w; x++)
      {
	//HEADER_PIXEL decodes the image data into RGB triplets
	//and it increments data !!
	//pixel[0..3] will contain the RGB value
	HEADER_PIXEL(data,pixel);
	//this is slow! but hey this is a constructor !
	painter->setPixel(Point(x,y), 
			  Color(pixel[0], pixel[1], pixel[2]));
      }
  
  delete painter;
  debugN(17,cerr<<"OK."<<endl;);
  
  return success;
}

char hexDecode(char c)
{
  return isdigit(c) ? c - '0' : toupper(c)-'A'+10;
}

bool 
Surface::readFromXPM(char** data)
{
  //free old surface data (if any)
  clearSurfaceData();
  
  int lineNr = 0;
  char* current = data[lineNr++];

  string line;
  line = string(current); 
  
  // parse the header line (using goold old sscanf)
  int w, h;
  int colorCount;   // number of individual colors
  int cpp;      // "chars per pixel"
  int num = sscanf(line.c_str(), "%d %d %d %d",&w, &h, &colorCount, &cpp);
  
  if(num != 4)
    {
      debugN(17,cerr<<"Surface::readFromXPM failed to parse header." << endl;);
      return false;
    }

  debugN(17,cerr<<"Surface::readFromXPM going to parse colortable with "
	 << colorCount <<" entries ..." << endl;);

  // now go for the color table
  map<string, Color> colors;
  Color color;
  string code;
  for(int i = 0; i < colorCount; i++)
    {
      current = data[lineNr++];
      
      // get the color code
      code = string(current, cpp);
      // skip color code
      current += cpp;

      // search for 'c'
      while(*current != 'c' && *current != 'g')
	current++;
      current++;
      current++;
      if(*current == '#')
	{
	  // this is a "normal" colortable entry
	  current++; // skip '#'

	  debugN(17, cerr << current << "--> ";);

	  // who needs sscanf ?
	  color.r = hexDecode(*current++)*16 + hexDecode(*current++);
	  color.g = hexDecode(*current++)*16 + hexDecode(*current++);
	  color.b = hexDecode(*current++)*16 + hexDecode(*current++);
#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) >= \
    SDL_VERSIONNUM(1, 1, 5)
	  color.a = 255;
#else
	  color.a = 0;
#endif

	  debugN(17,cerr << "("<<(int)color.r<<","<<(int)color.g<<","<<(int)color.b<<")"<<endl;);
	}
      else
	{
	  // transparent color ... some else here ??
	  color = transparent;
	}
      colors[code] = color;
    }
  
  if(*current != '\0')
    {
      debugN(17, cerr<<"[ERROR] " << endl;);
      return false;
    }
	   
  debugN(17,cerr<<" [OK]"<< endl;);

  /* Now we know enough to build the surface.
   * We will always build a 32bit surface encoding the transparency 
   * using the alpha channel. 
   */
  format = Pixelformat::ABGR8888;
  sdlSurface_ = SDL_AllocSurface(SDL_SWSURFACE | SDL_SRCALPHA ,
				 w,
				 h,
				 format.bpp()*8,
				 format.rMask(),
				 format.gMask(),
				 format.bMask(),
				 format.aMask());
  gatherPalette();
 
    
  debugN(17, cerr << "parsing XPM data  ..."<< endl;);
  Painter* painter = new Painter(this);
    
  for(int y = 0; y < h; y++)
    {
      current = data[lineNr++];
      for(int x = 0; x < w; x++)
	{
	  // extract pixel code (cpp chars)
	  code = string(current, cpp);
	  //next pixel
	  current+=cpp;
	  
	  painter->setPixel(Point(x,y), colors[code]);
	}
    }

  bool success;
  if(*current != '\0')
    {
      debugN(17, cerr<<"[ERROR] " << endl;);
      success= false;
    }
  else
    {
      debugN(17,cerr<<" [OK]"<< endl;);
      success = true;
    }

  delete painter;
  return success;

}

}

