/*
*	SDL Graphics Extension
*	Johan E. Thlin's BLib
*
*	Started 000428
*
*	License: LGPL v2+ (see the file LICENSE)
*	(c)2000 Anders Lindstrm & Johan E. Thlin
*/

/*********************************************************************
 *  This library is free software; you can redistribute it and/or    *
 *  modify it under the terms of the GNU Library General Public      *
 *  License as published by the Free Software Foundation; either     *
 *  version 2 of the License, or (at your option) any later version. *
 *********************************************************************/

/*
*  This is written by Johan E. Thlin and is slightly modified by me.
*/

#include <SDL/SDL.h>
#include "sge_draw.h"
#include "sge_blib.h"

#define SWAP(x,y,temp) temp=x;x=y;y=temp

extern Uint8 _sge_update;
extern Uint8 _sge_lock;


//==================================================================================
// Draws a horisontal line, fading the colors
//==================================================================================
void sge_FadedLine(SDL_Surface *dest,Sint16 x1,Sint16 x2,Sint16 y,Uint8 r1,Uint8 g1,Uint8 b1,Uint8 r2,Uint8 g2,Uint8 b2)
{
	Sint16 x,t;
	Sint16 tr,tg,tb;
	
	if ( x1 > x2 ) {
		SWAP(x1,x2,x);
		SWAP(r1,r2,x);
		SWAP(g1,g2,x);
		SWAP(b1,b2,x);
	}

	tr = r2 - r1;
	tg = g2 - g1;
	tb = b2 - b1;

	t = x2 - x1;
	t = t?t:1;
	
	if ( SDL_MUSTLOCK(dest) && _sge_lock )
		if ( SDL_LockSurface(dest) < 0 )
			return;
	
	
	Sint32 yp=sge_CalcYPitch(dest,y);
	
	if(yp < 0)
	  return;

	for (x = 0; x < x2 - x1; x++)
		sge_pPutPixel(dest,x+x1,yp,SDL_MapRGB( dest->format, (tr*x)/t+r1, (tg*x)/t+g1, (tb*x)/t+b1 ));


	if ( SDL_MUSTLOCK(dest) && _sge_lock )
		SDL_UnlockSurface(dest);
		
	if(_sge_update!=1){return;}
	sge_UpdateRect(dest, x1, y, x2-x1, 1);
}


//==================================================================================
// Draws a horisontal, textured line (slow - better)
//==================================================================================
void sge_sTexturedLine(SDL_Surface *dest,Sint16 x1,Sint16 x2,Sint16 y,SDL_Surface *source,Sint16 sx1,Sint16 sy1,Sint16 sx2,Sint16 sy2)
{
	Sint16 x,tx,ty;
	float t,ttx, tty;
	SDL_Color rgb;
	
	if ( x1 > x2 ) {
		SWAP(x1,x2,x);
		SWAP(sx1,sx2,x);
		SWAP(sy1,sy2,x);
	}

	tx = sx2 - sx1;
	ty = sy2 - sy1;

	t = x2 - x1;
	t = t?t:1;
	
	ttx=tx/t; tty=ty/t;
	
	if ( SDL_MUSTLOCK(dest) && _sge_lock )
		if ( SDL_LockSurface(dest) < 0 )
			return;
	
	Sint32 yp=sge_CalcYPitch(dest,y);
	
	if(yp < 0)
	  return;

	if ( dest->format->BitsPerPixel == source->format->BitsPerPixel )
		for (x = 0; x < x2 - x1; x++) {
			sge_pPutPixel(dest,x+x1,yp,sge_GetPixel(source, (Sint16)(ttx*x+sx1), (Sint16)(tty*x+sy1) ));
		}
	else	
		for (x = 0; x < x2 - x1; x++) {
			rgb=sge_GetRGB(source,sge_GetPixel(source, (Sint16)(ttx*x+sx1), (Sint16)(tty*x+sy1) ));
			sge_pPutPixel(dest,x+x1,yp,SDL_MapRGB(dest->format,rgb.r,rgb.g,rgb.b));
		}
	
			
	if ( SDL_MUSTLOCK(dest) && _sge_lock )
		SDL_UnlockSurface(dest);
		
	if(_sge_update!=1){return;}
	sge_UpdateRect(dest, x1, y, x2-x1, 1);
}


//==================================================================================
// Draws a horisontal, textured line (faster (<20%) and ugly)
//==================================================================================
void sge_TexturedLine(SDL_Surface *dest,Sint16 x1,Sint16 x2,Sint16 y,SDL_Surface *source,Sint16 sx1,Sint16 sy1,Sint16 sx2,Sint16 sy2)
{
  Sint16 x,tx,ty;
  Sint32 t;
  SDL_Color rgb;
  
  if ( x1 > x2 ) {
    SWAP(x1,x2,x);
    SWAP(sx1,sx2,x);
    SWAP(sy1,sy2,x);
  }
  
  tx = sx2 - sx1;
  ty = sy2 - sy1;
  
  t = x2 - x1;
  t = t?t:1;
  
  if ( SDL_MUSTLOCK(dest) && _sge_lock )
    if ( SDL_LockSurface(dest) < 0 )
      return;
	
  Sint32 yp=sge_CalcYPitch(dest,y);		
  
  if(yp >= 0)
    {
      if ( dest->format->BitsPerPixel == source->format->BitsPerPixel )
	for (x = 0; x < x2 - x1; x++) 
	  {
	    sge_pPutPixel(dest,x+x1,yp,sge_GetPixel(source, tx*x/t+sx1, (ty*x)/t+sy1 ));
	  }
      else	
	for (x = 0; x < x2 - x1; x++) 
	  {
	    rgb=sge_GetRGB(source,sge_GetPixel(source, tx*x/t+sx1, ty*x/t+sy1 ));
	    sge_pPutPixel(dest,x+x1,yp,SDL_MapRGB(dest->format,rgb.r,rgb.g,rgb.b));
	  }
    }
  
  if ( SDL_MUSTLOCK(dest) && _sge_lock )
    SDL_UnlockSurface(dest);
  
  if(_sge_update!=1){return;}
  sge_UpdateRect(dest, x1, y, x2-x1, 1);
}

//==================================================================================
// Draws a trigon
//==================================================================================
void sge_Trigon(SDL_Surface *dest,Sint16 x1,Sint16 y1,Sint16 x2,Sint16 y2,Sint16 x3,Sint16 y3,Uint32 color)
{
	sge_Line(dest,x1,y1,x2,y2,color);
	sge_Line(dest,x1,y1,x3,y3,color);
	sge_Line(dest,x3,y3,x2,y2,color);
}


//==================================================================================
// Draws a filled trigon
//==================================================================================
void sge_FilledTrigon(SDL_Surface *dest,Sint16 x1,Sint16 y1,Sint16 x2,Sint16 y2,Sint16 x3,Sint16 y3,Uint32 color)
{
	Sint16 xl,xs,y,t;
	
	if ( y1 > y2 ) {
		SWAP(y1,y2,y);
		SWAP(x1,x2,y);
	}
	
	if ( y2 > y3 ) {
		SWAP(y2,y3,y);
		SWAP(x2,x3,y);
	}
	
	if ( y1 > y2 ) {
		SWAP(y1,y2,y);
		SWAP(x1,x2,y);
	}
	
	for ( y = y1; y <= y3; y++) {
		if ( y > y2 ) {
			t = (y2-y3);
			t = t?t:1;
			xs = ((x2-x3)*(y-y2))/t+x2;
		}
		else {
			t = (y1-y2);
			t = t?t:1;
			xs = ((x1-x2)*(y-y1))/t+x1;
		}
		t = (y1-y3);
		xl = ((x1-x3)*(y-y1))/((t==0)?1:t)+x1;
		
		sge_HLine(dest,xl,xs,y,color);
	}
}


//==================================================================================
// Draws a gourand shaded trigon
//==================================================================================
void sge_FadedTrigon(SDL_Surface *dest,Sint16 x1,Sint16 y1,Sint16 x2,Sint16 y2,Sint16 x3,Sint16 y3,Uint32 c1,Uint32 c2,Uint32 c3)
{
	Sint16 xl,xs;
	Uint8 rl,gl,bl;
	Uint8 rs,gs,bs;
	Sint16 y;
	Uint8 c;
	Sint16 t,tl;
	SDL_Color rgb1;
	SDL_Color rgb2;
	SDL_Color rgb3;
	
	rgb1=sge_GetRGB(dest,c1);
	rgb2=sge_GetRGB(dest,c2);
	rgb3=sge_GetRGB(dest,c3);

	if ( y1 > y2 ) {
		SWAP(x1,x2,y);
		SWAP(y1,y2,y);
		SWAP(rgb1.r,rgb2.r,c);
		SWAP(rgb1.g,rgb2.g,c);
		SWAP(rgb1.b,rgb2.b,c);
	}
	
	if ( y2 > y3 ) {
		SWAP(x3,x2,y);
		SWAP(y3,y2,y);
		SWAP(rgb3.r,rgb2.r,c);
		SWAP(rgb3.g,rgb2.g,c);
		SWAP(rgb3.b,rgb2.b,c);
	}
	
	if ( y1 > y2 ) {
		SWAP(x1,x2,y);
		SWAP(y1,y2,y);
		SWAP(rgb1.r,rgb2.r,c);
		SWAP(rgb1.g,rgb2.g,c);
		SWAP(rgb1.b,rgb2.b,c);
	}

	tl = y1 - y3;
	tl = tl?tl:1;	

	for ( y = y1; y <= y3; y++) {
		if ( y > y2 ) {
			t = (y2-y3);
			t = t?t:1;
			xs = ((x2-x3)*(y-y2))/t+x2;
			rs = ((rgb2.r-rgb3.r)*(y-y2))/t+rgb2.r;
			gs = ((rgb2.g-rgb3.g)*(y-y2))/t+rgb2.g;
			bs = ((rgb2.b-rgb3.b)*(y-y2))/t+rgb2.b;
		}
		else {
			t = (y1-y2);
			t = t?t:1;
			xs = ((x1-x2)*(y-y1))/t+x1;
			rs = ((rgb1.r-rgb2.r)*(y-y1))/t+rgb1.r;
			gs = ((rgb1.g-rgb2.g)*(y-y1))/t+rgb1.g;
			bs = ((rgb1.b-rgb2.b)*(y-y1))/t+rgb1.b;
		}
		xl = ((x1-x3)*(y-y1))/tl+x1;
		rl = ((rgb1.r-rgb3.r)*(y-y1))/tl+rgb1.r;
		gl = ((rgb1.g-rgb3.g)*(y-y1))/tl+rgb1.g;
		bl = ((rgb1.b-rgb3.b)*(y-y1))/tl+rgb1.b;

		sge_FadedLine(dest,xl,xs,y,rl,gl,bl,rs,gs,bs);
	}
}


//==================================================================================
// Draws a texured trigon (fast)
//==================================================================================
void sge_TexturedTrigon(SDL_Surface *dest,Sint16 x1,Sint16 y1,Sint16 x2,Sint16 y2,Sint16 x3,Sint16 y3,SDL_Surface *source,Sint16 sx1,Sint16 sy1,Sint16 sx2,Sint16 sy2,Sint16 sx3,Sint16 sy3)
{
	Sint16 xl,xs;
	Sint16 sxs,sys,sxl,syl;
	Sint16 y;
	float t,tl;
	
	if ( y1 > y2 ) {
		SWAP(x1,x2,y);
		SWAP(y1,y2,y);
		SWAP(sx1,sx2,y);
		SWAP(sy1,sy2,y);
	}
	
	if ( y2 > y3 ) {
		SWAP(x3,x2,y);
		SWAP(y3,y2,y);
		SWAP(sx3,sx2,y);
		SWAP(sy3,sy2,y);
	}
	
	if ( y1 > y2 ) {
		SWAP(x1,x2,y);
		SWAP(y1,y2,y);
		SWAP(sx1,sx2,y);
		SWAP(sy1,sy2,y);
	}

	tl = y1 - y3;
	tl = tl?tl:1;	

	for ( y = y1; y <= y3; y++) {
		if ( y > y2 ) {
			t = (y2-y3);
			t = t?t:1;
			xs = (Sint16)(((x2-x3)*(y-y2))/t+x2);
			sxs = (Sint16)(((sx2-sx3)*(y-y2))/t+sx2);
			sys = (Sint16)(((sy2-sy3)*(y-y2))/t+sy2);
		}
		else {
			t = (y1-y2);
			t = t?t:1;
			xs = (Sint16)(((x1-x2)*(y-y1))/t+x1);
			sxs = (Sint16)(((sx1-sx2)*(y-y1))/t+sx1);
			sys = (Sint16)(((sy1-sy2)*(y-y1))/t+sy1);
		}
		xl = (Sint16)(((x1-x3)*(y-y1))/tl+x1);
		sxl = (Sint16)(((sx1-sx3)*(y-y1))/tl+sx1);
		syl = (Sint16)(((sy1-sy3)*(y-y1))/tl+sy1);

		sge_TexturedLine(dest,xl,xs,y,source,sxl,syl,sxs,sys);
	}
}


//==================================================================================
// Draws a texured trigon (slow)
//==================================================================================
void sge_sTexturedTrigon(SDL_Surface *dest,Sint16 x1,Sint16 y1,Sint16 x2,Sint16 y2,Sint16 x3,Sint16 y3,SDL_Surface *source,Sint16 sx1,Sint16 sy1,Sint16 sx2,Sint16 sy2,Sint16 sx3,Sint16 sy3)
{
	Sint16 xl,xs;
	Sint16 sxs,sys,sxl,syl;
	Sint16 y;
	float t,tl;
	
	if ( y1 > y2 ) {
		SWAP(x1,x2,y);
		SWAP(y1,y2,y);
		SWAP(sx1,sx2,y);
		SWAP(sy1,sy2,y);
	}
	
	if ( y2 > y3 ) {
		SWAP(x3,x2,y);
		SWAP(y3,y2,y);
		SWAP(sx3,sx2,y);
		SWAP(sy3,sy2,y);
	}
	
	if ( y1 > y2 ) {
		SWAP(x1,x2,y);
		SWAP(y1,y2,y);
		SWAP(sx1,sx2,y);
		SWAP(sy1,sy2,y);
	}

	tl = y1 - y3;
	tl = tl?tl:1;	

	for ( y = y1; y <= y3; y++) {
		if ( y > y2 ) {
			t = (y2-y3);
			t = t?t:1;
			xs = (Sint16)(((x2-x3)*(y-y2))/t+x2);
			sxs = (Sint16)(((sx2-sx3)*(y-y2))/t+sx2);
			sys = (Sint16)(((sy2-sy3)*(y-y2))/t+sy2);
		}
		else {
			t = (y1-y2);
			t = t?t:1;
			xs = (Sint16)(((x1-x2)*(y-y1))/t+x1);
			sxs = (Sint16)(((sx1-sx2)*(y-y1))/t+sx1);
			sys = (Sint16)(((sy1-sy2)*(y-y1))/t+sy1);
		}
		xl = (Sint16)(((x1-x3)*(y-y1))/tl+x1);
		sxl = (Sint16)(((sx1-sx3)*(y-y1))/tl+sx1);
		syl = (Sint16)(((sy1-sy3)*(y-y1))/tl+sy1);

		sge_sTexturedLine(dest,xl,xs,y,source,sxl,syl,sxs,sys);
	}
}


//==================================================================================
// Draws a texured *RECTANGLE* (fast)
//==================================================================================
void sge_TexturedRect(SDL_Surface *dest,Sint16 x1,Sint16 y1,Sint16 x2,Sint16 y2,Sint16 x3,Sint16 y3,Sint16 x4,Sint16 y4,SDL_Surface *source,Sint16 sx1,Sint16 sy1,Sint16 sx2,Sint16 sy2,Sint16 sx3,Sint16 sy3,Sint16 sx4,Sint16 sy4)
{
	Sint16 xl,xs;
	Sint16 sxs,sys,sxl,syl;
	Sint16 y;
	float t,tl;
	
	if ( y1 > y2 ) {
		SWAP(x1,x2,y);
		SWAP(y1,y2,y);
		SWAP(sx1,sx2,y);
		SWAP(sy1,sy2,y);
	}
	if ( y2 > y3 ) {
		SWAP(x3,x2,y);
		SWAP(y3,y2,y);
		SWAP(sx3,sx2,y);
		SWAP(sy3,sy2,y);
	}
	if ( y1 > y2 ) {
		SWAP(x1,x2,y);
		SWAP(y1,y2,y);
		SWAP(sx1,sx2,y);
		SWAP(sy1,sy2,y);
	}
	if ( y3 > y4 ) {
		SWAP(x3,x4,y);
		SWAP(y3,y4,y);
		SWAP(sx3,sx4,y);
		SWAP(sy3,sy4,y);
	}
	if ( y2 > y3 ) {
		SWAP(x3,x2,y);
		SWAP(y3,y2,y);
		SWAP(sx3,sx2,y);
		SWAP(sy3,sy2,y);
	}
	if ( y1 > y2 ) {
		SWAP(x1,x2,y);
		SWAP(y1,y2,y);
		SWAP(sx1,sx2,y);
		SWAP(sy1,sy2,y);
	}

	for ( y = y1; y <= y4; y++) {
		if (y <= y2 ) {
	  	t = (y1-y2);
			t = t?t:1;
			xs = (Sint16)(((x1-x2)*(y-y1))/t+x1);
			sxs = (Sint16)(((sx1-sx2)*(y-y1))/t+sx1);
			sys = (Sint16)(((sy1-sy2)*(y-y1))/t+sy1);
			
			tl = y1 - y3;
			tl = tl?tl:1;
			
			xl = (Sint16)(((x1-x3)*(y-y1))/tl+x1);
			sxl = (Sint16)(((sx1-sx3)*(y-y1))/tl+sx1);
			syl = (Sint16)(((sy1-sy3)*(y-y1))/tl+sy1);
		}
		else if ( y <= y3) {
			t = (y2-y4);
			t = t?t:1;
			xs = (Sint16)(((x2-x4)*(y-y2))/t+x2);
			sxs = (Sint16)(((sx2-sx4)*(y-y2))/t+sx2);
			sys = (Sint16)(((sy2-sy4)*(y-y2))/t+sy2);
			
			tl = y2 - y4;
			tl = tl?tl:1;
			
			xl = (Sint16)(((x1-x3)*(y-y1))/tl+x1);
			sxl = (Sint16)(((sx1-sx3)*(y-y1))/tl+sx1);
			syl = (Sint16)(((sy1-sy3)*(y-y1))/tl+sy1);
		}
		else {
			t = (y3-y4);
			t = t?t:1;
			xs = (Sint16)(((x3-x4)*(y-y3))/t+x3);
			sxs = (Sint16)(((sx3-sx4)*(y-y3))/t+sx3);
			sys = (Sint16)(((sy3-sy4)*(y-y3))/t+sy3);
			
			tl = y2 - y4;
			tl = tl?tl:1;
			
			xl = (Sint16)(((x2-x4)*(y-y2))/tl+x2);
			sxl = (Sint16)(((sx2-sx4)*(y-y2))/tl+sx2);
			syl = (Sint16)(((sy2-sy4)*(y-y2))/tl+sy2);
		}

		sge_TexturedLine(dest,xl,xs,y,source,sxl,syl,sxs,sys);
	}
}
