/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, June 1999  

#include "screensurface.h"
#include "debug.h"

namespace uta {

ScreenSurface::ScreenSurface(int w, int h, int bitPerPixel, bool fullscreen):
  Surface(),
  fullscreen_(false),
  doublebuf_(false)
{

  Uint32 flags = SDL_ANYFORMAT | SDL_HWSURFACE | SDL_HWPALETTE ;

  if(fullscreen)
    flags = flags | SDL_FULLSCREEN  | SDL_HWSURFACE | SDL_DOUBLEBUF;
  
  /* switch to video mode */
  debugN(17,cerr << "switching to video mode..."<<endl;fflush(stderr););
  sdlSurface_ = SDL_SetVideoMode(w,h,bitPerPixel, flags);
  if (sdlSurface_ == 0) {
    debugN(17,cerr << "failed: " << SDL_GetError() << endl;);
    exit(2);
  }

  /* Print out information about our surfaces */
  debugN(17,
	cerr << "\tScreen is at "<< (int)sdlSurface_->format->BitsPerPixel 
	<< " bpp" << endl;
	);
  
  if ( (sdlSurface_->flags & SDL_HWSURFACE) == SDL_HWSURFACE ) 
    {
      debugN(17,cerr << "  Screen is in video memory" << endl;);
    } 
  else 
    {
      debugN(17,cerr << "  Screen is in system memory" << endl;);
    }
  
  if ( (sdlSurface_->flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF ) 
    {
      debugN(17,cerr << "  Screen has double-buffering enabled" << endl;);
      doublebuf_ = true;
    }
  
  if ( (sdlSurface_->flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) 
    {
      debugN(17,cerr << "  Running in Fullscreen Mode" << endl;);
      fullscreen_ = true;
    }
  else
    {
      debugN(17,cerr << "  Running in Windowed Mode" << endl;);
      fullscreen_ = false;
    }
  
  //identify scteenformat
  int pixformat = Pixelformat::identify(sdlSurface_->format->BytesPerPixel,
					 sdlSurface_->format->Rmask,
					 sdlSurface_->format->Gmask,
					 sdlSurface_->format->Bmask,
					 sdlSurface_->format->Amask);
 
  //set static Pixelformat::member to spread information
  Pixelformat::displayFormat = pixformat;
  format = pixformat;

  if(format.bpp() == 1)
    {
      debugN(17, cerr << " Setting palette for Screen." << endl;);
      format.palette.resize(256);
      for(int r=0; r < 8; r++)
	for(int g=0; g < 4; g++)
	  for(int b=0; b < 8; b++)
	    {
	      format.palette[r<<5 | g<<3 | b] = Color(r<<5, g<<6, b<<5);
	    }   
      applyPalette();
      Pixelformat::displayFormat.palette = format.palette;
    }
 
  gatherPalette();
  
  debugN(17,
	if(pixformat == Pixelformat::UNKNOWN)
	{
	  cerr.setf(ios::hex);
	  cerr << "ScreenSurface: screen is in unidentified pixelformat !" 
	       << endl;
	  cerr << sdlSurface_->format->BytesPerPixel << "," 
	       << sdlSurface_->format->Rmask << "," 
	       << sdlSurface_->format->Gmask << "," 
	       << sdlSurface_->format->Bmask << "," 
	       << sdlSurface_->format->Amask << endl;
	  cerr.unsetf(ios::hex);
	}
	else
	{
	  cerr << "known screen pixelformat detected.:)" << endl;
	}
	);
}




void 
ScreenSurface::update()
{
  SDL_UpdateRects(sdlSurface_,0,0);
}
  
void
ScreenSurface::update(const Rect& dest)
{
  
  static SDL_Rect dst;
  dst.x = dest.upperLeft().x;
  dst.y = dest.upperLeft().y;
  dst.w = dest.width();
  dst.h = dest.height();
  SDL_UpdateRects(sdlSurface_,1,&dst);
  
}

void
ScreenSurface::flip()
{
  SDL_Flip(sdlSurface_);
}

#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) >= \
    SDL_VERSIONNUM(1, 1, 6)
bool
ScreenSurface::setGamma(float r, float g, float b)
{
  if(!SDL_WasInit(SDL_INIT_VIDEO))
    return false;

  int result = SDL_SetGamma(r,g,b);
  if(result != 0)
    return false;
  return true;
}
#endif

#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) >= \
    SDL_VERSIONNUM(1, 1, 7)
bool
ScreenSurface::setGammaRamp(Uint16 *red, Uint16 *green, Uint16 *blue)
{
  if(!SDL_WasInit(SDL_INIT_VIDEO))
    return false;

  int result = SDL_SetGammaRamp(red, green, blue);   
  if(result != 0)
    return false;
  return true;
}

#endif

}

