/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/


#include "widget.h"
#include "painter.h"
#include "debug.h"
#include "focus.h"
#include "directsurface.h"

#include <assert.h>

#ifdef DEBUG
//#include <typeinfo>
//#define name() typeid(this).name()
#define name() "Widget"
#endif

//too much info here ;)
#undef DEBUG

namespace uta {

  //#define DIRECT

Widget::Widget()
{
}


Widget::Widget(Widget* parent, int x, int y, int w, int h) :
  Rect(x,y,w,h) 
{
  assert(parent != 0);

  parent_ = parent;
  rect_ = Rect(x,y,w,h);
  window_ = Rect (0,0,w,h);

  initMembers();

#ifdef DIRECT
  surface_ = new DirectSurface(x, y, w, h);
#else
  //surface in memory
  surface_ = new Surface(w,h);
  surface_->setTransColor(black);
#endif

  assert(surface_ != 0);

  //register with parent and retrieve parentSurface_
  parentSurface_ = parent_->insertChild(this);

  setTransparency(true);
}

Widget::~Widget()
{
//   //do not use close() here !! because it calls the parent
//   //which at this moment is destructing too !!!!
//   if(!closed_)
//     close();

  debugN(17,cerr << name() << " has been hit for death." << endl;);

  closed_ = true;

  //clear the focus list in order to avoid hassle when killing the childs
  focusList_.clear();
  currentFocus_ = focusList_.end();

  if(parent_ )
    parent_->removeChild(this);

  Widget *childPtr;
  while(!childs_.empty())
    {
      childPtr = childs_.front();
      childs_.pop_front();
      //deleting a child modifies list childs_ (cia removeChild)
      //this is why the local copy childPtr is needed here

      if(childPtr->isAutoDelete())
	delete childPtr;
      else
	childPtr->deParent(); //just remove binding to this parent
    }

  //free own surface
  if(surface_)
    delete surface_;

  debugN(17,cerr << name() << " vanishes." << endl;);

  // if we still own focus, release it before dying !
  if(haveFocus_)
    Focus::release(this);
}

void
Widget::initMembers()
{
  // by default we do _not_ grab the focus 
  // (what does a seperator needs focus for ?)
  haveFocus_ = false;
  currentFocus_ = focusList_.begin();
  
  // by default widgets do not accept focus
  activateOnMouse_ = false;
  activateOnClick_ = false;

  /* we need to check if the mouse is already above this widget !
  */
  int mousex, mousey;
  SDL_GetMouseState(&mousex, &mousey);
  mouseFocus_ =  globalCoord(window_).contains(Point(mousex,mousey));
  
  drawBackground_ = true;
  needsUpdate_ = true;
  needsReblit_ = true;
  hidden_ = false;
  hideRequest_ = false;
  closed_ = false;
  disableRequest_ = false;
  disabled_ = false;
  tileBackground_ = true;
  autoDelete_ = true;
 
  color_ = transparent;
  colorDis_ = darkgrey;
	
  alphaDis_ = 50;
  
  backgrnd_ = 0;
}

void
Widget::setBackground(const Surface* backgrnd, bool tile)
{
  ///
  tileBackground_ = tile;
  // we don't copy the background....
  backgrnd_ = backgrnd;

  //we need update
  needsUpdate_ = true;
}

void
Widget::setColor(const Color &color)
{
  color_ = color;

  //we need update
  needsUpdate_ = true;
}

void
Widget::setDisabledColor(const Color& color)
{
    colorDis_ = color;

    needsUpdate_ = true;
}

void
Widget::setAlpha(unsigned char alpha)
{
  if(surface_ != 0)
    surface_->setAlpha(alpha);

  needsUpdate_ = true;
}

void
Widget::setDisabledAlpha(unsigned char alpha)
{
  alphaDis_ = alpha;

  needsUpdate_ = true;
}

void
Widget::blit()
{
  debugN(17,cerr << name() << " is blitted." << endl;);

  if(!hidden_)
    {
      dirtyRects_.push_back(globalCoord(window_));
      blit_complete(parentSurface_); 

      //blit the childs
      list<Widget*>::iterator childItr_;
      childItr_ = childs_.begin();
      while(childItr_ != childs_.end())
	{
	  (*childItr_)->blit();

	  //flush the dirty rects list from the child,
	  //since it's rect is covered by this widget's rect
	  (*childItr_)->dirtyRects_.clear();

	  childItr_ ++;
	}

    }
  
}

void
Widget::blit_complete(Surface* target)
{
#ifdef DIRECT
  return;
#endif

  //backBuffer_ blit if surface_ may contain transparent areas
  if((target != 0) && (surface_->usesTransparency()))
    {
      //FIXME, this is not always correct !!
      
      parent_->surface_->blit(target,
			      globalCoord(window_),
			      Rect(upperLeft(),lowerRight()));
    }
  
  //blit ourself
  surface_->blit(target, globalCoord(window_), window_);
}

  
void
Widget::show()
{
  hidden_ = false;
  hideRequest_ = false;
  needsUpdate_ = true;

  list<Widget*>::iterator childItr_;
  childItr_ = childs_.begin();
  while(childItr_ != childs_.end())
    {
      (*childItr_)->show();
      childItr_ ++;
    }  
  //force parent to update
  if(parent_ != 0)
    parent_->forceUpdate(1);

}

void
Widget::hide()
{

  hideRequest_ = true;
  needsUpdate_ = true;

  list<Widget*>::iterator childItr_;
  childItr_ = childs_.begin();
  while(childItr_ != childs_.end())
    {
      (*childItr_)->hide();
      childItr_ ++;
    }  
}

void
Widget::enable()
{
  disabled_ = false;
  //this is *needed* is a hidden widget gets disabled this flag
  //will never be reset, because update_ is not entered
  disableRequest_ = false;
  needsUpdate_ = true;  
  
  list<Widget*>::iterator childItr_;
  childItr_ = childs_.begin();
  while(childItr_ != childs_.end())
    {
      (*childItr_)->enable();
      childItr_ ++;
    }  
}

void
Widget::disable()
{ 
  disableRequest_ = true;
  needsUpdate_ = true;
  
  list<Widget*>::iterator childItr_;
  childItr_ = childs_.begin();
  while(childItr_ != childs_.end())
    {
      (*childItr_)->disable();
      childItr_ ++;
    }  
}

void
Widget::close()
{
  hide();
  //no chance...  every story has an end
  hidden_ = true;
  closed_ = true;
}

void
Widget::update()
{
  if(closed_)
    {
      //      delete this;
      return;
    }
  
  if(disabled_)
    return;
  
  if(hideRequest_)
    {
      hideRequest_ = false;
      hidden_ = true;
      
      //force parent to update
      if(parent_ != 0)
	parent_->forceUpdate(1);
    }
  
  if(disableRequest_)
    {
      disableRequest_ = false;
      disabled_ = true;
    }
	
  
  debugN(17,cerr << name() << ".update()" << endl;);

  //remember update flag
  bool update_ = needsUpdate_;
  ///child list iterator.
  list<Widget*>::iterator childItr_;
  
  //create this child
  createChild();
   
  //blit the necessary parts
  blit_dirty(parentSurface_);

  needsReblit_ = false;

  childItr_ = childs_.begin();
  list<Widget*> closeList;

  while(childItr_ != childs_.end())
    {

      //we must not call update() of an closed widget here
      //since closed widgets selfdestruct when calling update()
      if(!(*childItr_)->isClosed())
	{
#ifdef DIRECT	 
	  (*childItr_)->update();
#else 
	  if((*childItr_)->needsUpdate())
	    {
	       (*childItr_)->update();
	    }
	  else
	    if(update_)
	      (*childItr_)->blit();
#endif
	}
      else
	{
	  closeList.push_back(*childItr_);
	}

      (*childItr_)->addUpdateRects(dirtyRects_);

      childItr_ ++;
    }
 
  //kill the ones which are already dead (?)
  Widget *childPtr;
  while(!closeList.empty())
    {
      childPtr = closeList.front();
      closeList.pop_front();
      //deleting a child modifies list badly
      //this is why the local copy childPtr is needed here
      //but delete it only if the autoDelete_ flag is set
      if(childPtr->isAutoDelete())
	delete childPtr;
      else
	childPtr->deParent(); //just remove binding to this parent
    }
  

}


void
Widget::blit_dirty(Surface* target)
{
#ifdef DIRECT
  return;
#endif

  if(!dirtyRects_.empty())
    {
      Rect dest;
      Rect src;
      list<Rect>::iterator itr = dirtyRects_.begin();
      
      while(itr != dirtyRects_.end())
	{
	  src = localCoord(*itr);
	  dest = *itr;
	  debugN(17,cerr <<name()<<" blits(" << src << " >>> "<<dest << endl;);
	  surface_->blit(target, dest, src);
	  itr++;
	}
    }
}

void
Widget::createChild()
{ 
#ifdef DIRECT
  if(true)
#else
  if(needsUpdate_)
#endif
    {
      debugN(17,cerr << name() << ".createChild()" << endl;);

      if(!hidden_)
	{
  	
	  //create widget tree
	  create();
  
	  //when disabled greyout the widget
	  if(disabled_)
	    {
	      Point p;
	      Painter painter(surface_);
	      for(p.x=0; p.x < surface_->width(); p.x+=2)
		for(p.y=0; p.y < surface_->height(); p.y+=2)
		  painter.setPixel(p,colorDis_);
// 	      Surface* grey = new Surface(surface_->width(), surface_->height());

//               grey->fill(colorDis_);
//               grey->setAlpha(alphaDis_);
// 	      grey->blit(surface_);

// 	      delete grey;
	    }


	}
      else
	{
  	
	  debugN(17,cerr << name() << " now wears the Cap of Invisibility."
		<< endl;);
#ifndef DIRECT
	  //create transparent bitmap
	  surface_->clear();
#endif
  
	}
      needsUpdate_ = false;
    }
}

void
Widget::create()
{

  debugN(17,cerr << name() << ".create() " << endl;);

  dirtyRects_.push_back(globalCoord(window_));

  if(!hidden_)
    {
  
#ifndef DIRECT 
      surface_->fill(color_);
       
      //backBuffer_ blit if surface_ may contain transparent areas
      if((parentSurface_ != 0) 
	 && (surface_->usesTransparency()))
	{
	  // parent_->surface_->blit(parentSurface_, globalCoord(window_),window_); 
 	  parent_->surface_->blit(parentSurface_,
 				  globalCoord(window_),
 				  Rect(upperLeft(), lowerRight()));

	}
#endif

      if(backgrnd_ != 0 && !backgrnd_->empty())
	{
	  //scale backgrnd or tile it ...

#ifndef DIRECT
   	  if(backgrnd_->hasAlphaChannel())
	    {
	      surface_->fill(black);
	      parent_->surface_->blit(surface_,
				      Rect(0,0,width(), height()),
				      Rect(upperLeft(), lowerRight()));
	    }
#endif

// 	  if(backgrnd_->hasAlphaChannel())
// 	    {
// 	      surface_->fill(black);
// 	      parent_->surface_->blit(surface_,
// 				      globalCoord(window_),
// 				      Rect(Point(0,0),Point(width(), height())));
// 	    }

	  if(tileBackground_)
	    {
	      int w,h,x,y;
	      
	      if(surface_->width() < backgrnd_->width())
		w = surface_->width();
	      else
		w = backgrnd_->width();
	      
	      if(surface_->height() < backgrnd_->height())
		h = surface_->height();
	      else
		h = backgrnd_->height();
	      
	      int nx = 0;
	      int ny = 0;

	      //tile it
	      if(h != 0)
		ny = surface_->height() / h + ((surface_->height() % h)?1:0);
	      if(w != 0)   
		nx = surface_->width() / w + ((surface_->width() % w)?1:0);
	    
	      for(y = 0; y < ny; y++)
		for(x = 0; x < nx; x++)
		  {
		    int xsize = (w<(surface_->width()-x*w)) ? 
		      w : (surface_->width()-x*w);
		    int ysize = (h<(surface_->height()-y*h)) ? 
		      h : (surface_->height()-y*h);
		    
		    //target, destrect, srcrect
		    backgrnd_->blit(surface_, 
				    Rect(x*w,y*h,xsize,ysize), 
				    Rect(0,0,xsize,ysize));
		  }
	    }
	  else
	    {
	      if(backgrnd_->width() != surface_->width() ||
		 backgrnd_->height() != surface_->height())
		{
		  //do a scaledBlit :)
		  backgrnd_->scaledBlit(surface_);
		}
	      else
		{
		  //normal blit
		  backgrnd_->blit(surface_);
		}
	    }
	}

    }
  else
    {
      //this is for debugging
      assert(false);
    }
}



bool
Widget::needsUpdate()
{  
  
  //this widget tree is hidden or even closed
  //it surely needs no update
  if(closed_ || hidden_ || disabled_ )
    return false;

  //do we ourselves need screen update ?
  if (needsUpdate_ || needsReblit_)
    return true;

  //no ?, perhaps any child widget ?

  ///child list iterator.
  list<Widget*>::iterator childItr_;
  bool needed = false;
  childItr_ = childs_.begin();
  while(childItr_ != childs_.end())
    {
      if((*childItr_)->needsUpdate())
	{
	  needed = true;
	  //ok, we need update, no need to search any further
	  break;
	}
      childItr_++;
    }

  return needed;

}


Surface*
Widget::insertChild(Widget* child)
{

  assert(child != 0);
  assert(surface_ != 0);

  Surface* bitmap;

  //if we are the rootwidget, our own surface_ is returned
  //else parentSurface_ is returned, thus all widgets blit to root !

  if(parentSurface_ == 0)
    bitmap = surface_;
  else
    bitmap = parentSurface_;

  list<Widget*>::iterator itr = childs_.begin();
  while(itr != childs_.end())
    {
      if(*itr == child)
	break;
      itr++;
    }

  if(itr != childs_.end())
    {
      return bitmap;
    }

 
  focusList_.push_back(child);
  currentFocus_=focusList_.begin();
  
  //hmmm ... :(
  nextFocus();

  childs_.push_back(child);
  
  return bitmap;

}

void
Widget::removeChild(Widget* child)
{
  assert(child != 0);
  childs_.remove(child);
  child->parent_ = NULL;

  focusList_.remove(child);
  currentFocus_=focusList_.begin();

  //hmmm ... :(
  nextFocus();
    
  needsUpdate_ = true;
}
    
void
Widget::deParent()
{
  if(parent_)
    {
      parent_->childs_.remove(this);
      // parent_->needsUpdate_ = true;
    }
  parent_ = NULL;
}



Rect
Widget::globalCoord (Rect local)
{
  //transform local to glocal coords
  Rect  result; 
  
  if(parent_ != 0)
    result = parent_->globalCoord(local);
  else
    result = local;

  result.translate(-window_.upperLeft().x + rect_.upperLeft().x,
		   -window_.upperLeft().y + rect_.upperLeft().y);

    
  return result;
}

Rect
Widget::localCoord (Rect global)
{
  //transform global to local coords
  Rect  result; 
  
  if(parent_ != 0)
    result = parent_->localCoord(global);
  else
    result = global;

  result.translate(window_.upperLeft().x - rect_.upperLeft().x ,
		   window_.upperLeft().y - rect_.upperLeft().y );

    
  return result;
}
 
void 
Widget::addUpdateRects(list<Rect> &updateRects)
{
  //this removes all elements in dirtyRects_

  while(!dirtyRects_.empty())
    {
      updateRects.push_back(dirtyRects_.front());
      dirtyRects_.pop_front();
    }
}

bool 
Widget::handleEvent(const Event* event)
{

  if(hidden_ || closed_ || disabled_)
    return false;
 
  bool handled = false;
  
  if(event->type() == Event::MOUSE)
    {
      mouseFocus_ =  globalCoord(window_).
	contains(((MouseEvent *)event)->position());
    }
  
  //do we accept focus at all ?
  if(acceptsFocus())
    {
      //shall we try to grab the focus ?
      bool grabFocus = false;
      
      if(activateOnMouse_ && event->type() == Event::MOUSE)
	{
	  grabFocus =  mouseFocus_;
	}
      
      if(activateOnClick_ && event->type() == Event::BUTTON)
	{
	  grabFocus = globalCoord(window_).
	    contains(((ButtonEvent *)event)->position());
	  
	}
      
      //ok, try to obtain the focus...
      if(grabFocus)
	haveFocus_ = Focus::grab(this);
    }

  //can we handle this event ?
  handled = processEvent(event);

  if(!handled)
    {
      //no ?, perhaps any child ?

      ///child list iterator.
      list<Widget*>::iterator childItr_;
      childItr_ = childs_.begin();
      while(childItr_ != childs_.end())
	{
	  if((*childItr_)->handleEvent(event))
	    {
	      handled = true;
	      break;
	    }
	  childItr_++;
	}
    }
  
  return handled;
  
}


void Widget::forceUpdate(int data)
{
  switch(data)
    {
    case 0: 
      needsReblit_ = true;
      break;
    case 1:
      needsUpdate_ = true;
      break;
    default:
      needsReblit_ = true;
      break;
   }
}

bool
Widget::releaseFocus()
{
  //FIXME... sometime we may implemented locked focus ...
  haveFocus_ = false;
   
  needsUpdate_ = true;

  //ok,ok we are releasing the focus
  return true;
}

bool
Widget::grabFocus()
{
  haveFocus_ = Focus::grab(this);
  
  if(haveFocus_)
    needsUpdate_ = true;
  
  return haveFocus_;
}

void 
Widget::nextFocus()
{
  if(currentFocus_ != focusList_.end())
    {
      Widget* active = (*currentFocus_);

      do {
	currentFocus_++;
	if(currentFocus_ == focusList_.end())
	  currentFocus_ = focusList_.begin();
      } while (!(*currentFocus_)->acceptsFocus()&&(*currentFocus_ != active));

      if(active != *currentFocus_)
	(*currentFocus_)->grabFocus();
    }
}

void 
Widget::prevFocus()
{
  if(currentFocus_ != focusList_.end())
    {
      Widget* active = (*currentFocus_);
      do {
	if(currentFocus_ == focusList_.begin())
	  {
	    currentFocus_ = focusList_.end();
	    currentFocus_--;
	  }
	else
	  currentFocus_ --;
      } while (!(*currentFocus_)->acceptsFocus()&&(*currentFocus_ != active));
      
      if(active != *currentFocus_)
	(*currentFocus_)->grabFocus();
    }
}

}
