/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, June 1999  

#include "pixelformat.h"

#include "debug.h"
#include <stdio.h>

namespace uta {

Pixelformat Pixelformat::displayFormat = Pixelformat::UNKNOWN;

const char* Pixelformat::names[] = 
{ "DISPLAY", "ABGR8888", "RGBA8888","ARGB8888","BGRA8888","RGB888",
  "BGR888", "RGB0888", "BGR0888", "RGB565", "RGB555", "INDEXED", "UNKNOWN" };
  
///RGBA bitmasks
Uint32 Pixelformat::masks[][4] = 
{{0,0,0,0},
 {0x000000FF, 0x0000FF00, 0x00FF0000, 0xFF000000},
 {0xFF000000, 0x00FF0000, 0x0000FF00, 0x000000FF},
 {0x00FF0000, 0x0000FF00, 0x000000FF, 0xFF000000},
 {0x0000FF00, 0x00FF0000, 0xFF000000, 0x000000FF},
 {0x00FF0000, 0x0000FF00, 0x000000FF, 0x00000000},
 {0x000000FF, 0x0000FF00, 0x00FF0000, 0x00000000},
 {0x00FF0000, 0x0000FF00, 0x000000FF, 0x00000000},
 {0x000000FF, 0x0000FF00, 0x00FF0000, 0x00000000},
 {0x0000F800, 0x000007E0, 0x0000001F, 0x00000000},
 {0x00007C00, 0x000003E0, 0x0000001F, 0x00000000},
 {0x00, 0x00, 0x00, 0x00},
 {0x00, 0x00, 0x00, 0x00}};

int Pixelformat::bytes[] = { -1, 4, 4, 4, 4, 3, 3, 4, 4, 2, 2, 1, 0};

Pixelformat::Pixelformat(int format) :
  format_(UNKNOWN),
  bytesPerPixel_(0),
  rmask_(0),
  gmask_(0),
  bmask_(0),
  amask_(0)
{

  init(format);
}

void
Pixelformat::init(int format)
{
  if((format < UNKNOWN) && (format >= 0))
    {
      format_ = format;

      if(format_ == DISPLAY)
	{
	  format_ = Pixelformat::displayFormat;
	  palette = Pixelformat::displayFormat.palette;
	}
      
      
      rmask_ = masks[format_][0];
      rshift_ = getShift(rmask_);
      rloss_ = getLoss(rmask_);

      gmask_ = masks[format_][1];
      gshift_ = getShift(gmask_);
      gloss_ = getLoss(gmask_);

      bmask_ = masks[format_][2];
      bshift_ = getShift(bmask_);
      bloss_ = getLoss(bmask_);
      
      amask_ = masks[format_][3];
      ashift_ = getShift(amask_);
      aloss_ = getLoss(amask_);
      
      bytesPerPixel_ = bytes[format_];

      if(bytesPerPixel_ == 1 && palette.empty())
	{
	  /* Simple 6x6x6 colour cube */
	  for (int b = 0; b < 256; b += 51)
	    for (int g = 0; g < 256; g += 51)
	      for (int r = 0; r < 256; r += 51) 
		palette.push_back(uta::Color(r,g,b));
	       
	}
      valid_ = true;
      
    }
  else
    {
      format_ = UNKNOWN;
      valid_ = false;

      //      cerr << "Warning: unkown Pixelformat :" << format << endl;
    }
}

int
Pixelformat::getShift(Uint32 mask) const
{
  int i=0;
  Uint32 tmp = mask;
  
  if(tmp != 0)
    {
      for(tmp = mask; !(tmp & 0x01); tmp>>=1)
	++i;
    }
  
  return i;
}

int
Pixelformat::getLoss(Uint32 mask) const
{
  //assume we will lose all bits
  int i=8;
  Uint32 tmp = mask;
  
  if(tmp != 0)
    {
      //scroll bits down until lsb is != 0
      for(tmp = mask; !(tmp & 0x01); tmp>>=1);
      
      //how many bits are contained ?
      for(;(tmp & 0x01); tmp >>= 1)
	--i;
    }
  
  return i;
}

string 
Pixelformat::asString() const
{ 
  return string(names[format_]);
}

 
Uint32
Pixelformat::mapToPixel(const Color& col) const
{
  Uint32 pixel = 0;
  unsigned int i;
  switch(bytesPerPixel_)
    {
    case 4:
    case 3:
    case 2:
      pixel = ( ( (col.r>>rloss_) << rshift_) |
	        ( (col.g>>gloss_) << gshift_) |
		( (col.b>>bloss_) << bshift_) |
		( (col.a>>aloss_) << ashift_) ) ;
      break;
    case 1:
      {
	if(palette.size() > 0)
	  {
	    unsigned int bestindex = 0;
	    long int match;
	    long int bestmatch = 
	      (col.r - palette[bestindex].r)*(col.r - palette[bestindex].r) +
	      (col.g - palette[bestindex].g)*(col.g - palette[bestindex].g) +
	      (col.b - palette[bestindex].b)*(col.b - palette[bestindex].b);

	    //serach for best matching palette entry
	    for(i=0; i < palette.size(); i++)
	      {
		match = 
		  (col.r - palette[i].r)*(col.r - palette[i].r) +
		  (col.g - palette[i].g)*(col.g - palette[i].g) +
		  (col.b - palette[i].b)*(col.b - palette[i].b);

		if(match < bestmatch)
		  {
		    bestmatch = match;
		    bestindex = i;
		  }

		if(match == 0)
		  break;
	      };

	    pixel = bestindex;
	  }
      }
      break;
      
    default:
      //sth. unknown, ignore it
      break;
    }

  return pixel;
}


Color
Pixelformat::mapToColor(Uint32 pixel) const
{
  Color col;

  switch(bytesPerPixel_)
    {
    case 4:  
      col.r = (pixel & rmask_) >> rshift_ << rloss_;
      col.g = (pixel & gmask_) >> gshift_ << gloss_;
      col.b = (pixel & bmask_) >> bshift_ << bloss_;    
      col.a = (pixel & amask_) >> ashift_ << aloss_;
      break;
    case 3:
    case 2:
      col.r = (pixel & rmask_) >> rshift_ << rloss_;
      col.g = (pixel & gmask_) >> gshift_ << gloss_;
      col.b = (pixel & bmask_) >> bshift_ << bloss_;
      col.a = SDL_ALPHA_OPAQUE;
      break;
    case 1:
      if(pixel < palette.size())
	col = palette[pixel];
      break;
    default:
      break;
    }

  return col;
}


int
Pixelformat::identify(Uint8 bpp_, 
		       Uint32 rmask, Uint32 gmask, Uint32 bmask, Uint32 amask)
{

  debugN(17,fprintf(stderr,"%d bytes %x %x %x %x",bpp_, 
		rmask, gmask, bmask, amask););
  int result;
  if(bpp_ == 1)
    result = IND8;
  else
    {
      unsigned int i = 0;
      while(i < sizeof(Pixelformat::names))
	{
	  if((rmask == Pixelformat::masks[i][0]) &&
	     (gmask == Pixelformat::masks[i][1]) &&
	     (bmask == Pixelformat::masks[i][2]) &&
	     (amask == Pixelformat::masks[i][3]) &&
	     (bpp_ == bytes[i]))
	    break;
	  i++;
	}
      if(i != sizeof(Pixelformat::names))
	{
	  result = i;
	  debugN(17,fprintf(stderr," ... identified as %s\n.", names[result]););
	}
      else
	{
	  debugN(17,fprintf(stderr," ... not identified. :(\n"););
	  result = UNKNOWN;
	}
    }
  
  return result;
}
}

