/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "mouse.h"
#include "application.h"
#include "painter.h"

#include "debug.h"
#include <assert.h>

#include "mouse_data.h"

namespace uta {

Pointer::Pointer(const Surface* bm, const Rect &rect) :
  surface(0),
  screenArea(rect)
{
  surface = new Surface(*bm);
  assert(surface != 0);
}


Pointer::Pointer(vector<const Surface*>& bmlist, const Rect &rect ) :
  surface(0),
  screenArea(rect)
{
  assert(!bmlist.empty());
  
  vector<const Surface*>::iterator itr= bmlist.begin();
  while (itr != bmlist.end())
    {
	  bmList_.push_back(new Surface(**itr));
	  itr ++;
    }

  bmItr_ = bmList_.begin();
  surface = *bmItr_;
  
  assert(surface != 0);
}
      
 
Pointer::~Pointer()
{
  if(bmList_.empty())
    {
      delete surface;
    }
  else
    {
      bmItr_ = bmList_.begin();
      while (bmItr_ != bmList_.end())
	{
	  delete *bmItr_;
	  bmItr_ ++;
	}
    }
}


void
Pointer::tick() 
{ 
  if(!bmList_.empty())
    {
      bmItr_++; 
      if(bmItr_ == bmList_.end())
	bmItr_ = bmList_.begin();
      surface = *bmItr_;
      
    }
}

//we may only control one mouse
Mouse* Mouse::instance_ = NULL;

Mouse::Mouse() :
  defaultPointer_(0),
  pos_(Point(0,0))
{
  backBuffer_ = new Surface(MAXX, MAXY, Pixelformat::DISPLAY);
  assert(backBuffer_ != 0);

  if(uapp->rootWindow()->screen()->pixelformat().bpp() == 1)
    {
      backBuffer_->
	setPalette(uapp->rootWindow()->screen()->pixelformat().palette);
    }

  backBuffer_->setTransparency(false);

  //force sensible state
  move(Point(0,0));

  src = new_src;
  dest = new_dest;

  //load default pointer from header and set it
  //header_data, width and height are defined in umouse_data.h
  
  pointer_.readFromHeader(header_data,width,height);
  setPointer(&pointer_);

}


Mouse::~Mouse()
{
  //delete the pointers
  list<Pointer*>::iterator itr = pointers_.begin();
  while(itr != pointers_.end())
    {
      delete (*itr);
      itr++;
    } 

  if(defaultPointer_ != 0)
    delete defaultPointer_;

  delete backBuffer_;
  // delete doubleBuffer_;
  
  instance_ = NULL;
}


Mouse*
Mouse::create()
{
  if(Application::Instance == 0)
    {
      cerr << "Mouse: no Application Object found. " << endl;
      return 0;
    }

  if(!instance_)
    {
      instance_ = new Mouse();
      return instance_;
    }
  else
    {
      cerr << "Mouse: there is already an Mouse Object present." << endl;
      return 0;
    }
}


void
Mouse::setPointer(const Surface* bm)
{
  if(defaultPointer_ != 0)
    delete defaultPointer_;

  if(bm)
    defaultPointer_ = new Pointer(bm);
  else
    defaultPointer_ = new Pointer(&pointer_);

  assert(defaultPointer_ != 0);
  
  //update internal pointers and setup
  move(pos_);
}


void
Mouse::setPointer(const Surface* bm, const Rect &rect)
{  
  //check if there is already a pointer for this region
  list<Pointer*>::iterator itr = pointers_.begin();
  while(itr != pointers_.end())
    {
      if(((*itr)->screenArea == rect) || rect.contains((*itr)->screenArea)) 
	{
	  //delete this pointer, it is obsolute
	  debugN(17,
		 cerr << "Mouse: deleting old pointer for rect "<<(*itr)->screenArea << endl;);
	  delete *itr;
	  *itr = NULL;
	}
      itr++;
    }
  //remove all deleted pointers
  pointers_.remove(NULL);
      
  if(bm)
    {
      Pointer* ptr = new Pointer(bm, rect);  
      assert(ptr != 0);
      pointers_.push_front(ptr);
    }
  //update internal pointers and setup
  move(pos_);
}


void 
Mouse::setPointer(vector<const Surface*>& bmlist)
{
  
  if(defaultPointer_ != 0)
    delete defaultPointer_;

  defaultPointer_ = new Pointer(bmlist);
  assert(defaultPointer_ != 0);
  
  //update internal pointers and setup
  move(pos_);
}

void 
Mouse::setPointer(vector<const Surface*>& bmlist, const Rect& rect)
{
  //check if there is already a pointer for this region
  list<Pointer*>::iterator itr = pointers_.begin();
  while(itr != pointers_.end())
    {
      if(((*itr)->screenArea == rect) || rect.contains((*itr)->screenArea)) 
	{
	  //delete this pointer, it is obsolute
	  debugN(17,
		 cerr << "Mouse: deleting old pointer for rect "
		 <<(*itr)->screenArea << endl;);
	  delete *itr;
	  *itr = NULL;
	}
      itr++;
    }
  //remove all deleted pointers
  pointers_.remove(NULL);
      
  if(!bmlist.empty())
    {
      Pointer* ptr = new Pointer(bmlist, rect);  
      assert(ptr != 0);
      
      pointers_.push_front(ptr);
    }
  //update internal pointers and setup
  move(pos_);
}

void 
Mouse::blit()
{

  //just return if we have no MousePointer
  if(ptr == 0)
    return;
 
  //blit pointer to framebuffer
  ptr->surface->blit(Application::Instance->rootWindow()->screen(),
		    new_dest, new_src);
  
  src = new_src;

  cerr << "nosense !!!!!" ;
}

void 
Mouse::paint()
{

  //just return if we have no MousePointer
  if(ptr == 0)
    return;
 
  //store new pointer pos background to doubleBuffer_
  backBuffer_->clear();
  uapp->rootWindow()->screen()->blit(backBuffer_,new_src,new_dest);
  
  //blit pointer to framebuffer
  ptr->surface->blit(uapp->rootWindow()->screen(),new_dest, new_src);

  Application::Instance->rootWindow()->refreshRect(dest);
  Application::Instance->rootWindow()->refreshRect(new_dest);

//   if(!Application::Instance->rootWindow()->fullscreen()) 
//     {
//       //update old pos to screen
//       Application::Instance->rootWindow()->screen()->setSource(dest);
//       Application::Instance->rootWindow()->screen()->blit(NULL);
//       //update new pos to screen
//       Application::Instance->rootWindow()->screen()->setSource(new_dest);
//       Application::Instance->rootWindow()->screen()->blit(NULL);
//     }

  
  //doubleBuffer_->blit(backBuffer_);

  src = new_src;
  dest = new_dest;
}

void 
Mouse::remove()
{
  //just return if we have no MousePointer
  if(ptr == 0)
    return;

  //blit stored backgound of old mousepos to framebuffer
  //(that kinda bad, since it may have changed ..... have no idea here)
 
  backBuffer_->blit(Application::Instance->rootWindow()->screen(),dest,src);

}

void
Mouse::move(const Point& p)
{
  pos_ = p;
  
  int w = 0;
  int h = 0;

  list<Pointer*>::iterator itr = pointers_.begin();
  while(itr != pointers_.end())
    {
      if((*itr)->screenArea.contains(pos_))
	break;
      itr++;
    }
  
  if(itr != pointers_.end())
    {
      ptr = (*itr);
    }
  else
    {
      ptr = defaultPointer_;
    }

  //just return if we have no MousePointer
  if(ptr == 0)
    return;

  //is the mousepointer seen at all on the screen ?
  if(!Application::Instance->rootWindow()->contains(pos_))
    return;

  if(pos_.x + ptr->surface->width() > 
     Application::Instance->rootWindow()->width())
    w = Application::Instance->rootWindow()->width() - pos_.x;
  else
    w = ptr->surface->width();
  
  if(pos_.y + ptr->surface->height() > 
     Application::Instance->rootWindow()->height())
    h = Application::Instance->rootWindow()->height() - pos_.y;
  else
    h = ptr->surface->height();

  if(w < 0)
    w = 0;
  if(h < 0)
    h = 0;

  new_dest = Rect(pos_.x, pos_.y, w, h);

  new_src = Rect(0,0,w,h);

}

void
Mouse::refreshRect(const Rect& r)
{
  Rect _sect = r.intersect(dest);
  
  if(_sect.isValid())
    {
      //mouse pointer at least partially  covering changed screen

      int dx = _sect.upperLeft().x - dest.upperLeft().x;
      int dy = _sect.upperLeft().y - dest.upperLeft().y;
      assert(dx >= 0);
      assert(dy >= 0);

      Rect _dest = Rect(dx,dy,_sect.width(),_sect.height());

      //update backbuffer
      uapp->rootWindow()->screen()->blit(backBuffer_,_dest,_sect);      
    }
}


}

