/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "lineedit.h"
#include "application.h"
#include "resources.h"

namespace uta {

LineEdit::LineEdit(Widget* parent, int x, int y, int w, int h, 
		     const Font *font):
  Widget(parent, x,y,w,h),
  cursorPos_(0),
  cursor_(false),
  drawCursor_(false),
  textChanged_(true),
  blockInput_(false),
  cursorCount_(0),
  textFont_(font),
  textSurface_(NULL),
  justCursor_(false)
{
  if(textFont_ == 0)
    textFont_ = FONT_RES("text_font");

  textCursor_ = Rect(0,rect_.height()-5, 12, 4);

  uapp->timer100.connect(slot(this, &LineEdit::timer));

  keyInput.connect(slot(uapp->soundMapper(), &SoundMapper::click));

  setClickToFocus(true);

  if(RES_AVAILABLE("text_color"))
    if(RES_AVAILABLE("text_background"))
      setTextColor(COLOR_RES("text_color"), COLOR_RES("text_background"));
    else
      setTextColor(COLOR_RES("text_color"));
  else
    setTextColor(textFont_->fontColor(), textFont_->fadeColor());
  
  if(RES_AVAILABLE("lineedit_color"))
    Widget::setColor(COLOR_RES("lineedit_color"));
  if(RES_AVAILABLE("lineedit_surface"))
    Widget::setBackground(SURFACE_RES("lineedit_surface"),false);
}

LineEdit::~LineEdit()
{
  if(textSurface_ != NULL)
    delete textSurface_;
}


void
LineEdit::create()
{

  Rect dest;

  if(textChanged_)
    {
      justCursor_ = false;
      
      if(textSurface_ != NULL)
	{
	  delete textSurface_;
	}      
      textSurface_ = textFont_->getString(text_);
      
      if(fontPal_.size() != 0)
	textSurface_->setPalette(fontPal_);

      textChanged_ = false;
    }

  //setting background and that stuff....
  if(!justCursor_)
    Widget::create();
  
  if(!justCursor_)
    {
      dest = Rect(0,
		  rect_.height() - textSurface_->height() -1,
		  textSurface_->width(),
		  textSurface_->height());
      
      textSurface_->blit(surface_, dest);
      
      //move the cursor
      textCursor_.warp(Point(textSurface_->width() + 1, rect_.height()-5)) ;
      
      //there is no space anymore
      if(textCursor_.upperLeft().x + 2*textCursor_.width() >= rect_.width())
	blockInput_ = true;
    }

  if(justCursor_)
    {
      dirtyRects_.clear();
      justCursor_ = false;
    }

  //draw the cursor ...
  if(drawCursor_) 
    drawCursor();
  else
    deleteCursor();
  
}
  

bool
LineEdit::processEvent(const Event *event)
{
  bool handled = false;

  if(haveFocus_)
    {
      if(event->type() == Event::KEY)
	{
	  if(((KeyEvent *)event)->state() == KeyEvent::PRESSED)
	    {
	      char input = ((KeyEvent *)event)->ascii() ;
	     
	      //basic editing: there are ENTER and BS 
	      //nearly emacs ... 
	      if(!blockInput_ && isprint(input))
		{
		  text_ += string(1,input);
		  needsUpdate_ = true;
		  textChanged_ = true;
		  textChanged(text_.c_str());
		  handled = true;
		}
	      if((input =='\n') || (input == '\r'))
		{
		  enterPressed(text_.c_str());
		  handled = true;
		}
	      if(input == '\b')
		{
		  if(text_.length() > 0)
		    {
		      text_.erase(text_.length()-1,1);
		      needsUpdate_ = true;
		      textChanged_ = true;
		      blockInput_ = false;
		      textChanged(text_.c_str());
		    }
		  handled = true;
		}
	      if(handled)
		{
		  //emit signal
		  keyInput();
		}
	      if(((KeyEvent *)event)->key() == KEY_TAB)
		{
		  //TAB changes focus to next one,
		  //Shift+TAB to previous one
		  if(parent_)
		    if(((KeyEvent *)event)->modifier() && KEY_MOD_SHIFT)
		      parent_->prevFocus();
		    else
		      parent_->nextFocus();
		  handled = true;
		}
	    }
	}
    
      
    } // if(haveFocus_)
      
  
  
  if(!haveFocus_)
    {
      if(cursor_)
	{
	  drawCursor_ = false;
	  needsUpdate_ = true;
	  
	}
    }
  
  return handled;
}



void LineEdit::timer()
{
  if(haveFocus_)
    {
      drawCursor_ = true;
      
      cursorCount_++;
      if(cursorCount_ > 15)
	cursorCount_ = 0;
      
      justCursor_ = true;
      needsUpdate_ = true;
    }
}
	  

  
void 
LineEdit::drawCursor()
{  
  cursor_ = true;

  unsigned char c = (cursorCount_-8) * (cursorCount_-8) * 3 + 63;

  Color color = Color(c,c,c);

  surface_->fill(textCursor_, color);

  dirtyRects_.push_back(globalCoord(textCursor_));
}
  
void 
LineEdit::deleteCursor()
{
  cursor_ = false;

  //color_ is the widgets color
  if(backgrnd_ == NULL)
    {
      surface_->fill(textCursor_, color_);
    }
  else
    {
      backgrnd_->blit(surface_, textCursor_, textCursor_);
    }

  dirtyRects_.push_back(globalCoord(textCursor_));

}

 
void 
LineEdit::setTextColor(const Color& fontCol, const Color& backCol)
{
  int rdiff = fontCol.r - backCol.r;
  int gdiff = fontCol.g - backCol.g;
  int bdiff = fontCol.b - backCol.b;
  int adiff = fontCol.a - backCol.a;

  int r = backCol.r;
  int g = backCol.g;
  int b = backCol.b;
  int a = backCol.a;

  fontPal_.clear();
  fontPal_.push_back(transparent);

  for(unsigned char num=1; num < 5; num++)
    fontPal_.push_back(Color(r + num * rdiff/4,
			     g + num * gdiff/4,
			     b + num * bdiff/4,
			     a + num * adiff/4));

  
  needsUpdate_ = true;
}



void LineEdit::setText(const char* data)
{
  if(!data)
    return;

  text_ = data;
  blockInput_ = false;
  
  //we need to be reblitted !
  textChanged_ = true;
  needsUpdate_ = true;
}

void LineEdit::clearText()
{
  text_ = "";
  blockInput_ = false;
  
  //we need to be reblitted !
  textChanged_ = true;
  needsUpdate_ = true;
}

}

