/* tp-chan.c
 *
 * Copyright (C) 2005-2007 Nokia Corporation
 * Copyright (C) 2006-2007 Collabora Ltd. <http://www.collabora.co.uk/>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "tp-chan.h"

#include <string.h>

#include "tp-props-iface.h"
#include "tp-helpers.h"
#include "internal.h"
#include "_gen/add-signals.h"


static GObjectClass *parent_class = NULL;

/* We initialize the list of signatures here, so that we can use
 * it to add them for new interface instances later.*/

/* FIXME: This should be replaced by a more automatic way of doing
 * this. The reason for using a set of function pointers is that there is no
 * apparent cleaner way of doing this, unless DBusGProxy gains a non-varargs
 * version of dbus_g_proxy_add_signal...
 */

static void
_tp_chan_init_interface_signal_signatures (GData **signal_sigs)
{
  g_datalist_init (signal_sigs);

  g_datalist_id_set_data (signal_sigs,
      TP_IFACE_QUARK_CHANNEL_TYPE_STREAMED_MEDIA,
      tp_chan_set_streamedmedia_signatures);
  g_datalist_id_set_data (signal_sigs, TP_IFACE_QUARK_CHANNEL_TYPE_ROOM_LIST,
      tp_chan_set_roomlist_signatures);
  g_datalist_id_set_data (signal_sigs, TP_IFACE_QUARK_CHANNEL_TYPE_TEXT,
      tp_chan_set_text_signatures);
  g_datalist_id_set_data (signal_sigs,
      TP_IFACE_QUARK_CHANNEL_INTERFACE_CHAT_STATE,
      tp_chan_set_chat_state_signatures);
  g_datalist_id_set_data (signal_sigs, TP_IFACE_QUARK_CHANNEL_INTERFACE_DTMF,
      tp_chan_set_dtmf_signatures);
  g_datalist_id_set_data (signal_sigs, TP_IFACE_QUARK_CHANNEL_INTERFACE_GROUP,
      tp_chan_set_group_signatures);
  g_datalist_id_set_data (signal_sigs, TP_IFACE_QUARK_CHANNEL_INTERFACE_HOLD,
      tp_chan_set_hold_signatures);
  g_datalist_id_set_data (signal_sigs,
      TP_IFACE_QUARK_CHANNEL_INTERFACE_PASSWORD,
      tp_chan_set_password_signatures);
  g_datalist_id_set_data (signal_sigs,
      TP_IFACE_QUARK_CHANNEL_INTERFACE_MEDIA_SIGNALLING,
      tp_chan_set_media_signalling_signatures);
  g_datalist_id_set_data (signal_sigs, TP_IFACE_QUARK_CHANNEL_TYPE_TUBES,
      tp_chan_set_tubes_signatures);
}

static void
tp_chan_init (GTypeInstance *instance, gpointer g_class)
{
  TpChan *self = TELEPATHY_CHAN(instance);
  self->type = NULL;
  self->first_run = TRUE;
}


static void
tp_chan_dispose (GObject *obj)
{
  TpChan *self = TELEPATHY_CHAN (obj);

  if (self->first_run)
    {
      self->first_run = FALSE;
      g_datalist_clear (&(self->interface_list));
    }

  G_OBJECT_CLASS (parent_class)->dispose (obj);
}


static void
tp_chan_finalize (GObject *obj)
{
  TpChan *self = TELEPATHY_CHAN (obj);

  if (self->type)
    {
      g_free (self->type);
    }

  G_OBJECT_CLASS (parent_class)->finalize (obj);
}


static void
tp_chan_class_init (TpChanClass *klass)
{
  GObjectClass *obj = G_OBJECT_CLASS (klass);
  parent_class = g_type_class_peek_parent (klass);

  obj->dispose = tp_chan_dispose;
  obj->finalize = tp_chan_finalize;

  _tp_register_marshallers ();
  _tp_chan_init_interface_signal_signatures (&(klass->iface_signal_sigs));
}


GType
tp_chan_get_type (void)
{
  static GType type = 0;

  if (G_UNLIKELY (type == 0))
    {
      static const GTypeInfo info = {
          sizeof (TpChanClass),
          NULL,
          NULL,
          (GClassInitFunc) tp_chan_class_init,
          NULL,
          NULL,
          sizeof (TpChan),
          0,
          (GInstanceInitFunc) tp_chan_init
      };
      type = g_type_register_static (DBUS_TYPE_G_PROXY,
          "TpChan", &info, 0);
    }

  return type;
}



/* Public functions begin */

GQuark
tp_get_chan_interface (void)
{
  return tp_iface_quark_channel ();
}

GQuark
tp_get_chan_contactlist_interface (void)
{
  return tp_iface_quark_channel_type_contact_list ();
}


GQuark
tp_get_chan_streamed_interface (void)
{
  return tp_iface_quark_channel_type_streamed_media ();
}


GQuark
tp_get_chan_roomlist_interface (void)
{
  return tp_iface_quark_channel_type_room_list ();
}

GQuark
tp_get_chan_text_interface (void)
{
  return tp_iface_quark_channel_type_text ();
}

GQuark
tp_get_chan_tubes_interface (void)
{
  return tp_iface_quark_channel_type_tubes ();
}

GQuark
tp_get_chan_chat_state_interface (void)
{
  return tp_iface_quark_channel_interface_chat_state ();
}

GQuark
tp_get_chan_dtmf_interface (void)
{
  return tp_iface_quark_channel_interface_dtmf ();
}

GQuark
tp_get_chan_group_interface (void)
{
  return tp_iface_quark_channel_interface_group ();
}

GQuark
tp_get_chan_hold_interface (void)
{
  return tp_iface_quark_channel_interface_hold ();
}

GQuark
tp_get_chan_password_interface (void)
{
  return tp_iface_quark_channel_interface_password ();
}

GQuark
tp_get_chan_transfer_interface (void)
{
  return tp_iface_quark_channel_interface_transfer ();
}

GQuark
tp_get_chan_media_signalling_interface (void)
{
  return tp_iface_quark_channel_interface_media_signalling ();
}

TpChan *
tp_chan_new (DBusGConnection *connection,
             const gchar *bus_name,
             const gchar *object_path,
             const gchar *type,
             guint handle_type,
             guint handle)
{
  GError *error = NULL;
  gchar *unique_name;
  gchar **interfaces;
  TpChan *chan;

  g_return_val_if_fail (connection != NULL, NULL);
  g_return_val_if_fail (bus_name != NULL, NULL);
  g_return_val_if_fail (object_path != NULL, NULL);
  g_return_val_if_fail (type != NULL, NULL);

  if (!dbus_g_proxy_call (tp_get_bus_proxy (), "GetNameOwner", &error,
        G_TYPE_STRING, bus_name, G_TYPE_INVALID,
        G_TYPE_STRING, &unique_name, G_TYPE_INVALID))
    {
      _tp_warn_failure ("tp_chan_new: getting unique name", error);

      if (error != NULL)
        g_error_free (error);

      return NULL;
    }

  /* Create the channel object */
  chan = g_object_new (TELEPATHY_CHAN_TYPE,
            "name", unique_name,
            "path", object_path,
            "interface", TP_IFACE_CHANNEL,
            "connection", connection,
            NULL);

  g_free (unique_name);

  ADD_SIGNALS_FOR_CHANNEL (DBUS_G_PROXY (chan));

  g_datalist_init (&(chan->interface_list));

  /* Store interface information for the channel */
  if (tp_chan_get_interfaces (DBUS_G_PROXY (chan), &interfaces, &error))
    {
      tp_chan_local_set_interfaces (chan, interfaces);

      /* Free the strings used for interface object creation */
      g_strfreev (interfaces);
    }
  else
    {
      _tp_warn_failure ("GetInterfaces for channel", error);

      if (error != NULL)
        g_error_free (error);
    }

  /* Store necessary information for this object */
  chan->type = g_strdup (type);
  chan->handle_type = handle_type;
  chan->handle = handle;

  return chan;
}


void
tp_chan_local_set_interfaces (TpChan *self,
                              gchar **interfaces)
{
  gchar **temp_ifaces;
  gchar *chan_type = NULL;
  GError *error = NULL;
  GData **sig_list;
  void (*signature_setter_func) (DBusGProxy *proxy);
  DBusGConnection *connection;
  gchar *name, *path;

  g_return_if_fail (self != NULL);

  if (interfaces == NULL)
    return;

  sig_list = &(TELEPATHY_CHAN_GET_CLASS (self)->iface_signal_sigs);

  /* Create and store proxy objects corresponding to the
     interfaces */

  g_object_get (G_OBJECT (self),
      "connection", &connection,
      "name", &name,
      "path", &path,
      NULL);

  for (temp_ifaces = interfaces; *temp_ifaces; temp_ifaces++)
    {
      GQuark key = g_quark_from_string (*temp_ifaces);
      DBusGProxy *if_proxy;

      if (key == TP_IFACE_QUARK_PROPERTIES_INTERFACE)
        {
          if_proxy = DBUS_G_PROXY (tp_props_iface_new (connection, name,
                path));
        }
      else
        {
          if_proxy = dbus_g_proxy_new_for_name (connection, name,
              path, *temp_ifaces);

          if (if_proxy != NULL)
            {
              /* Does the interface have signals? If yes, add their signatures
                 for the interface instance by calling the
                 corresponding setter function */

              signature_setter_func = g_datalist_id_get_data (sig_list, key);

              if (signature_setter_func != NULL)
                {
                  (*signature_setter_func) (if_proxy);
                }
            }
        }
      if (if_proxy != NULL)
        {
          g_datalist_id_set_data_full (&(self->interface_list), key,
              if_proxy, g_object_unref);
        }
    }

  /* Finally, add the channel type interface */

  if (!tp_chan_get_channel_type (DBUS_G_PROXY (self), &chan_type, &error))
    {
      _tp_warn_failure ("GetChannelType", error);

      if (error != NULL)
        g_error_free (error);
    }
  else
    {
      DBusGProxy *chan_proxy = dbus_g_proxy_new_from_proxy (DBUS_G_PROXY (self),
          chan_type, NULL);

      g_datalist_id_set_data_full (&(self->interface_list),
          g_quark_from_string (chan_type), chan_proxy, g_object_unref);

      /* If the particular channel type interface has signals defined,
         call the corresponding setter function */

      signature_setter_func = g_datalist_id_get_data (sig_list,
          g_quark_from_string (chan_type));

      if (signature_setter_func != NULL)
        {
          (*signature_setter_func) (chan_proxy);
        }

      g_free (chan_type);
    }

  g_free (name);
  g_free (path);
  dbus_g_connection_unref (connection);
}

DBusGProxy *
tp_chan_get_interface (TpChan *self,
                       GQuark iface_quark)
{
  DBusGProxy *iface_proxy = NULL;

  g_return_val_if_fail (self != NULL, NULL);
  g_return_val_if_fail (iface_quark != 0, NULL);

  iface_proxy = (DBusGProxy *) g_datalist_id_get_data (
      &(self->interface_list), iface_quark);

  return iface_proxy;
}

