/*

Copyright (C) 2002 - 2004 Christian Kreibich

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "pcapnav_private.h"
#include "pcapnav_header.h"
#include "pcapnav_append.h"
#include "pcapnav_macros.h"
#include "pcapnav_trace.h"
#include "pcapnav_util.h"
#include "pcapnav_debug.h"


/* NOTE NOTE NOTE -- offsets to the user of this library are relative
 * to the *END* of the pcap file header. Hence, offset 0 means the first
 * byte after the pcap file header. This makes offset calculation
 * easier for the user, because usually the user doesn't care about
 * the trace file header when handling offsets in a trace file.
 * We thus save the user the error-prone addition/subtraction of
 * sizeof(struct pcap_file_header) all the time.
 */

static char pcap_errbuf[PCAP_ERRBUF_SIZE];


void
pcapnav_init(void)
{
  pcapnav_runtime_options.debug = 0;
  pcapnav_runtime_options.calldepth_limit = -1;
}


pcapnav_t      *
pcapnav_open_offline(const char *fname)
{
  pcapnav_t               *pn;
  u_int32_t                magic;
  struct pcap_file_header *filehdr = NULL;
  FILE                    *fp;
  struct stat              st;

  D_ENTER;

  if (!fname || fname[0] == '\0')
    {
      D(("Invalid filename: %s\n", fname));
      errno = ENOENT;
      D_RETURN_(NULL);
    }

  /* Allocate new pcapnav structure and initialize. */
  
  if (! (pn = NEW(pcapnav_t)))
    {
      D(("Out of memory.\n"));
      errno = ENOMEM;
      D_RETURN_(NULL);
    }

  if (lstat(fname, &st) < 0)
    {
      D(("lstat failed: %s\n", strerror(errno)));
      goto free_return;
    }

  pn->size = st.st_size;

  /* Allocate pcap handle */
  if (! (pn->pcap = pcap_open_offline(fname, pcap_errbuf)))
    {
      D(("%s (from pcap, re. %s)\n", pcap_errbuf, fname));
      /* Let's hope errno is meaningful now ... */
      goto free_return;
    }
  
  /* Hook pcap's file stream into our own structure: */
  pn->fp = pcap_file(pn->pcap);

  if ((fp = fopen(fname, "r")) == NULL)
    {
      D(("Could not open trace file %s for reading.\n", fname));
      /* errno set already */
      goto free_return;
    }

  if (fread((char *)&pn->trace.filehdr, sizeof(struct pcap_file_header), 1, fp) != 1)
    {
      D(("Could not read trace file header from %s\n", fname));
      /* errno set already */
      goto cleanup_return;
    }

  /* Look at magic to determine byte order. */

  magic = pn->trace.filehdr.magic;
  filehdr = &pn->trace.filehdr;      

  if (magic != TCPDUMP_MAGIC && magic != PATCHED_TCPDUMP_MAGIC)
    {
      magic = SWAPLONG(magic);

      if (magic != TCPDUMP_MAGIC && magic != PATCHED_TCPDUMP_MAGIC)
	{
	  D(("Invalid trace file %s -- didn't recognize file magic.\n", fname));
	  goto cleanup_return;
	}

      pn->trace.swapped = TRUE;

      filehdr->version_major = SWAPSHORT(filehdr->version_major);
      filehdr->version_minor = SWAPSHORT(filehdr->version_minor);
      filehdr->thiszone = SWAPLONG(filehdr->thiszone);
      filehdr->sigfigs = SWAPLONG(filehdr->sigfigs);
      filehdr->snaplen = SWAPLONG(filehdr->snaplen);
      filehdr->linktype = SWAPLONG(filehdr->linktype);      
    }
  
  if (magic == PATCHED_TCPDUMP_MAGIC)
    /*
     * XXX - the patch that's in some versions of libpcap
     * changes the packet header but not the magic number;
     * we'd have to use some hacks^H^H^H^H^Hheuristics to
     * detect that.
     */
    pn->trace.pkthdr_size = sizeof(struct pcapnav_patched_pkthdr);
  else
    pn->trace.pkthdr_size = sizeof(struct pcap_pkthdr);

  pn->chain_buf = __pcapnav_buf_new(MAX_PACKET_SIZE(pn) *
				    MAX_CHAIN_LENGTH);
  if (!pn->chain_buf)
    goto cleanup_return;

  pn->search_buf = __pcapnav_buf_new(MAX_BYTES_FOR_DEFINITE_HEADER(pn));
  if (!pn->search_buf)
    goto cleanup_return;

  /* Get length of file: */
  if (fseek(fp, 0, SEEK_END) != 0)
    {
      D(("Couldn't determine file length, fseek() failed: %s\n", strerror(errno)));
      goto cleanup_return;
    }

  if ((pn->trace.length = ftell(fp)) < 0)
    {
      D(("Couldn't determine file length, ftell() failed: %s\n", strerror(errno)));
      goto cleanup_return;
    }

  fclose(fp);  
  D_RETURN_(pn);
  
 cleanup_return:
  fclose(fp);
  
 free_return:
  FREE(pn);
  D_RETURN_(NULL);
}


void	        
pcapnav_close(pcapnav_t *pn)
{
  D_ENTER;

  if (!pn)
    D_RETURN;

  if (pn->pcap)
    pcap_close(pn->pcap);

  /* No need to flclose(pn->fp) -- we stole it from pcap */

  __pcapnav_buf_free(pn->search_buf);
  __pcapnav_buf_free(pn->chain_buf);

  FREE(pn);
  D_RETURN;
}


int                
pcapnav_get_pkthdr_size(pcapnav_t *pn)
{
  if (!pn)
    return 0;

  return pn->trace.pkthdr_size;
}


const struct pcap_file_header *
pcapnav_get_file_header(pcapnav_t *pn)
{
  if (!pn)
    return 0;

  return (const struct pcap_file_header *) &pn->trace.filehdr;
}


off_t              
pcapnav_get_offset(pcapnav_t *pn)
{
  if (!pn)
    return 0;

  return ftell(pcap_file(pn->pcap)) - sizeof(struct pcap_file_header);
}


int
pcapnav_set_offset(pcapnav_t *pn, off_t offset)
{
  int result;

  D_ENTER;
  
  if (!pn)
    D_RETURN_(-1);

  result = fseek(pcap_file(pn->pcap), offset + sizeof(struct pcap_file_header), SEEK_SET);
  D_RETURN_(result);
}


void               
pcapnav_get_timestamp(pcapnav_t *pn, struct bpf_timeval *tv)
{
  off_t position;
  struct pcap_pkthdr header;

  D_ENTER;

  if (!pn || !tv)
    D_RETURN;

  position = pcapnav_get_offset(pn);
  memset(tv, 0, sizeof(struct bpf_timeval));
  
  if (pcapnav_next(pn, &header))
    *tv = header.ts;
  
  pcapnav_set_offset(pn, position);
  D_RETURN;
}


pcap_t            *
pcapnav_pcap(pcapnav_t *pn)
{
  if (!pn)
    return NULL;

  return pn->pcap;
}



int	        
pcapnav_loop(pcapnav_t *pn, int num_packets,
	     pcap_handler callback, u_char *user_data)
{
  int n;

  D_ENTER;

  if (!pn || !callback)
    {
      D(("Invalid input.\n"));
      D_RETURN_(0);
    }
  
  n = pcap_loop(pn->pcap, num_packets, callback, user_data);
  D_RETURN_(n);
}


int	        
pcapnav_dispatch(pcapnav_t *pn, int num_packets,
		 pcap_handler callback, u_char *user_data)
{
  if (!pn || !callback)
    {
      D(("Invalid input.\n"));
      return 0;
    }

  return pcap_dispatch(pn->pcap, num_packets, callback, user_data);
}


const u_char      *
pcapnav_next(pcapnav_t *pn, struct pcap_pkthdr *header)
{
  const u_char * result;
  struct pcap_pkthdr dummy;

  D_ENTER;

  if (!pn)
    D_RETURN_(NULL);

  if (!header)
    header = &dummy;

  result = pcap_next(pn->pcap, header);
  D_RETURN_(result);
}


int                
pcapnav_has_next(pcapnav_t *pn)
{
  int result = 1;
  off_t position;

  D_ENTER;

  if (!pn)
    D_RETURN_(0);

  position = pcapnav_get_offset(pn);
  
  if (!pcapnav_next(pn, NULL))
    D_RETURN_(0);
  
  pcapnav_set_offset(pn, position);

  D_RETURN_(result);
}


pcapnav_result_t
pcapnav_goto_timestamp(pcapnav_t *pn, struct bpf_timeval *timestamp)
{
  pcapnav_result_t result;

  D_ENTER;
  
  if (!pn || !timestamp)
    D_RETURN_(PCAPNAV_ERROR);
  
  /* Make sure timespan of trace is in pn: */
  pcapnav_get_timespan(pn, NULL, NULL);
  result = __pcapnav_trace_find_packet_at_timestamp(pn, timestamp);
  
  D_RETURN_(result);
}


pcapnav_result_t             
pcapnav_goto_fraction(pcapnav_t *pn, double fraction)
{
  pcapnav_result_t result;
  off_t offset = 0;

  D_ENTER;
  
  if (!pn)
    D_RETURN_(PCAPNAV_ERROR);

  /* Make sure span of trace is in pn: */
  pcapnav_get_timespan(pn, NULL, NULL);
  
  /* Assert 0 <= fraction <= 1: */
  
  if (fraction > 1.0)
    fraction = 1.0;

  if (fraction < 0.0)
    fraction = 0.0;
  
  offset = (pn->end_offset - pn->start_offset) * fraction;
  result = __pcapnav_trace_find_packet_at_offset(pn, offset, PCAPNAV_CMP_ANY);
  
  D_RETURN_(result);
}


pcapnav_result_t             
pcapnav_goto_offset(pcapnav_t *pn, off_t offset, pcapnav_cmp_t boundary)
{
  pcapnav_result_t result;

  D_ENTER;

  if (!pn)
    D_RETURN_(PCAPNAV_ERROR);

  /* Make sure span of trace is in pn: */
  pcapnav_get_timespan(pn, NULL, NULL);
  result = __pcapnav_trace_find_packet_at_offset(pn, offset, boundary);
  
  D_RETURN_(result);
}


int             
pcapnav_get_timespan(pcapnav_t *pn, struct bpf_timeval *start, struct bpf_timeval *end)
{
  D_ENTER;

  if (!pn)
    D_RETURN_(-1);

  if ((pn->start_time.tv_sec == 0)  &&
      (pn->start_time.tv_usec == 0) &&
      (pn->end_time.tv_sec == 0)    &&
      (pn->end_time.tv_usec == 0))
    {
      /* We have not yet looked up the timespan and offsets
       * of the trace file already. */

      __pcapnav_trace_find_start(pn);
      __pcapnav_trace_find_end(pn);
    }
  
  if (start)
    *start = pn->start_time;
  
  if (end)
    *end = pn->end_time;

  D_RETURN_(0);
}


off_t
pcapnav_get_span(pcapnav_t *pn)
{
  off_t result;

  D_ENTER;

  if (!pn)
    D_RETURN_(0);

  if ((pn->start_time.tv_sec == 0)  &&
      (pn->start_time.tv_usec == 0) &&
      (pn->end_time.tv_sec == 0)    &&
      (pn->end_time.tv_usec == 0))
    {
      /* We have not yet looked up the timespan and offsets of the trace file. */

      __pcapnav_trace_find_start(pn);
      __pcapnav_trace_find_end(pn);
    }
  
  result = pn->end_offset - pn->start_offset;
  D_RETURN_(result);
}


off_t
pcapnav_get_size(pcapnav_t *pn)
{
  if (!pn)
    return 0;

  return pn->size - sizeof(struct pcap_file_header);
}


int
pcapnav_timeval_cmp(const struct bpf_timeval *tv1,
		    const struct bpf_timeval *tv2)
{
  if (!tv1 || !tv2)
    return 0;

  if (tv1->tv_sec < tv2->tv_sec)
    return -1;

  if (tv1->tv_sec > tv2->tv_sec)
    return 1;

  if (tv1->tv_usec < tv2->tv_usec)
    return -1;

  if (tv1->tv_usec > tv2->tv_usec)
    return 1;
  
  return 0;
}


void               
pcapnav_timeval_sub(const struct bpf_timeval *tv1,
		    const struct bpf_timeval *tv2,
		    struct bpf_timeval *tv_out)
{
  __pcapnav_util_timeval_sub(tv1, tv2, tv_out);
}


void               
pcapnav_timeval_add(const struct bpf_timeval *tv1,
		    const struct bpf_timeval *tv2,
		    struct bpf_timeval *tv_out)
{
  __pcapnav_util_timeval_add(tv1, tv2, tv_out);
}


double             
pcapnav_get_time_fraction(pcapnav_t *pn,
			  const struct bpf_timeval *tv)
{
  long offset;
  struct pcap_pkthdr hdr;
  double full_span, current_span, fraction, result;

  D_ENTER;

  if (!pn)
    D_RETURN_(0.0);

  /* Make sure timestamps are initialized */

  pcapnav_get_timespan(pn, NULL, NULL);
  full_span = __pcapnav_util_timeval_diff(&pn->end_time, &pn->start_time);

  if (!tv)
    {
      struct pcap_pkthdr hdr2;

      /* Obtain current packet header by reading one packet
       * and then rewinding back to original position. */
      
      if ((offset = ftell(pn->fp)) < 0)
	D_RETURN_(0.0);
      
      if (fread((void *) &hdr, sizeof(struct pcap_pkthdr), 1, pn->fp) != 1)
	D_RETURN_(0.0);
      
      if (fseek(pn->fp, offset, SEEK_SET) < 0)
	{
	  D(("fseek() failed: %s\n", strerror(errno)));
	  D_RETURN_(0.0);
	}

      __pcapnav_header_extract(pn, (u_char*) &hdr, &hdr2);
      current_span = __pcapnav_util_timeval_diff(&hdr2.ts, &pn->start_time);
    }
  else
    {
      current_span = __pcapnav_util_timeval_diff(tv, &pn->start_time);
    }

  fraction = current_span / full_span;

  if (fraction < 0.0)
    fraction = 0.0;

  if (fraction > 1.0)
    fraction = 1.0;

  result = fabs(fraction);
  D_RETURN_(result);
}


double             
pcapnav_get_space_fraction(pcapnav_t *pn, off_t offset)
{
  double fraction, result;
  
  D_ENTER;

  if (!pn || offset == 0)
    D_RETURN_(0.0);

  /* Make sure timestamps + offsets are initialized */

  pcapnav_get_timespan(pn, NULL, NULL);

  fraction =
    ((double)(offset)) /
    (pn->end_offset - pn->start_offset);

  if (fraction < 0.0)
    fraction = 0.0;

  if (fraction > 1.0)
    fraction = 1.0;

  result = fabs(fraction);
  D_RETURN_(fraction);
}


int
pcapnav_get_current_timestamp(pcapnav_t *pn, struct bpf_timeval *tv)
{
  struct pcap_pkthdr hdr;
  off_t position;

  D_ENTER;

  if (!pn || !tv)
    D_RETURN_(0);
  
  position = pcapnav_get_offset(pn);
  
  if (fread((void *) &hdr, sizeof(struct pcap_pkthdr), 1, pn->fp) != 1)
    {
      pcapnav_set_offset(pn, position);
      D_RETURN_(0);
    }

  *tv = hdr.ts;  
  pcapnav_set_offset(pn, position);

  D_RETURN_(1);
}


char *
pcapnav_geterr(pcapnav_t *pn)
{
  return pcap_geterr(pn->pcap);
}


pcap_dumper_t *
pcapnav_dump_open(pcap_t *pcap, const char *filename, pcapnav_dumpmode_t mode)
{
  if (!pcap)
    {
      D(("Input error.\n"));
      return NULL;
    }

  /* If the user requests standard output, just pass
   * through to pcap.
   */
  if (filename[0] == '-' && filename[1] == '\0')
    {
      D(("Passing through to pcap_dump_open().\n"));
      return pcap_dump_open(pcap, filename);
    }
  
  switch (mode)
    {
    case PCAPNAV_DUMP_APPEND_FAST:
      return pcapnav_append_fast(pcap, filename);
      
    case PCAPNAV_DUMP_APPEND_SAFE:
      return pcapnav_append_safe(pcap, filename);

    case PCAPNAV_DUMP_TRUNC:
    default:      
      return pcap_dump_open(pcap, filename);
    }
  
  return NULL;
}
