/*
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *  
 *  Copyright (C) 2000 - 2005 Liam Girdwood  
 *
 *  The functions in this file use the Lunar Solution ELP 2000-82B by
 *  Michelle Chapront-Touze and Jean Chapront.
 */

#include <math.h>
#include <libnova/lunar.h>
#include <libnova/vsop87.h>
#include <libnova/solar.h>
#include <libnova/earth.h>
#include <libnova/transform.h>
#include <libnova/rise_set.h>
#include <libnova/utility.h>

/* AU in KM */
#define AU			149597870

/* sequence sizes */
#define ELP1_SIZE	1023  	  	/* Main problem. Longitude periodic terms (sine) */
#define ELP2_SIZE	918    		/* Main problem. Latitude (sine) */
#define ELP3_SIZE	704    		/* Main problem. Distance (cosine) */
#define ELP4_SIZE	347    		/* Earth figure perturbations. Longitude */
#define ELP5_SIZE	316   		/* Earth figure perturbations. Latitude */
#define ELP6_SIZE	237    		/* Earth figure perturbations. Distance */
#define ELP7_SIZE	14    		/* Earth figure perturbations. Longitude/t */
#define ELP8_SIZE	11    		/* Earth figure perturbations. Latitude/t */
#define ELP9_SIZE	8    		/* Earth figure perturbations. Distance/t */
#define ELP10_SIZE	14328    	/* Planetary perturbations. Table 1 Longitude */
#define ELP11_SIZE	5233    	/* Planetary perturbations. Table 1 Latitude */
#define ELP12_SIZE	6631    	/* Planetary perturbations. Table 1 Distance */
#define ELP13_SIZE	4384    	/* Planetary perturbations. Table 1 Longitude/t */
#define ELP14_SIZE	833    		/* Planetary perturbations. Table 1 Latitude/t */
#define ELP15_SIZE	1715    	/* Planetary perturbations. Table 1 Distance/t */
#define ELP16_SIZE	170    		/* Planetary perturbations. Table 2 Longitude */
#define ELP17_SIZE	150    		/* Planetary perturbations. Table 2 Latitude */
#define ELP18_SIZE	114    		/* Planetary perturbations. Table 2 Distance */
#define ELP19_SIZE	226    		/* Planetary perturbations. Table 2 Longitude/t */
#define ELP20_SIZE	188    		/* Planetary perturbations. Table 2 Latitude/t */
#define ELP21_SIZE	169    		/* Planetary perturbations. Table 2 Distance/t */
#define ELP22_SIZE	3    		/* Tidal effects. Longitude */
#define ELP23_SIZE	2    		/* Tidal effects. Latitude */
#define ELP24_SIZE	2    		/* Tidal effects. Distance */
#define ELP25_SIZE	6    		/* Tidal effects. Longitude/t */
#define ELP26_SIZE	4    		/* Tidal effects. Latitude/t */
#define ELP27_SIZE	5    		/* Tidal effects. Distance/t */
#define ELP28_SIZE	20    		/* Moon figure perturbations. Longitude */
#define ELP29_SIZE	12    		/* Moon figure perturbations. Latitude */
#define ELP30_SIZE	14    		/* Moon figure perturbations. Distance */
#define ELP31_SIZE	11    		/* Relativistic perturbations. Longitude */
#define ELP32_SIZE	4    		/* Relativistic perturbations. Latitude */
#define ELP33_SIZE	10    		/* Relativistic perturbations. Distance */
#define ELP34_SIZE	28    		/* Planetary perturbations - solar eccentricity. Longitude/t2 */
#define ELP35_SIZE	13    		/* Planetary perturbations - solar eccentricity. Latitude/t2 */
#define ELP36_SIZE	19    		/* Planetary perturbations - solar eccentricity. Distance/t2 */


/* Chapront theory lunar constants */
#define 	RAD		(648000.0 / M_PI)
#define		DEG		(M_PI / 180.0)
#define		M_PI2 	(2.0 * M_PI)
#define		PIS2	(M_PI / 2.0)
#define 	ATH		384747.9806743165
#define 	A0		384747.9806448954
#define		AM	 	0.074801329518
#define		ALPHA 	0.002571881335
#define		DTASM	(2.0 * ALPHA / (3.0 * AM))
#define		W12		(1732559343.73604 / RAD)
#define		PRECES	(5029.0966 / RAD)
#define		C1		60.0
#define		C2		3600.0

/* Corrections of the constants for DE200/LE200 */
#define		DELNU	((0.55604 / RAD) / W12)
#define		DELE	(0.01789 / RAD)
#define		DELG	(-0.08066 / RAD)
#define		DELNP	((-0.06424 / RAD) / W12)
#define 	DELEP	(-0.12879 / RAD)

/*     Precession matrix */
#define		P1		0.10180391e-4
#define		P2		0.47020439e-6
#define		P3		-0.5417367e-9
#define		P4		-0.2507948e-11
#define		P5		0.463486e-14
#define		Q1		-0.113469002e-3
#define		Q2		0.12372674e-6
#define		Q3		0.1265417e-8
#define		Q4		-0.1371808e-11
#define		Q5		-0.320334e-14

/* used for elp1 - 3 */
struct main_problem
{
	int ilu[4];
	double A;
	double B[5];
};

/* used for elp 4 - 9 */
struct earth_pert
{
	int iz;
	int ilu[4];
	double O;
	double A;
	double P;
}; 

/* used for elp 10 - 21 */
struct planet_pert
{
	int ipla[11];
	double theta;
	double O;
	double P;
};

typedef struct earth_pert tidal_effects;
typedef struct earth_pert moon_pert;
typedef struct earth_pert rel_pert;
typedef struct earth_pert plan_sol_pert;
 
/* initialise lunar constants */
void init_lunar_constants ();

/* cache values */
static double c_JD = 0;
static double c_X = 0;
static double c_Y = 0;
static double c_Z = 0;
static double c_precision = 1.0;

/* constants with corrections for DE200 / LE200 */
static const double W1[5] = 
{
	((218.0 + (18.0 / 60.0) + (59.95571 / 3600.0))) * DEG,
	1732559343.73604 / RAD,
	-5.8883 / RAD,
	0.006604 / RAD,
	-0.00003169 / RAD
}; 

static const double W2[5] = 
{
	((83.0 + (21.0 / 60.0) + (11.67475 / 3600.0))) * DEG,
	14643420.2632 / RAD,
	-38.2776 /  RAD,
	-0.045047 / RAD,
	0.00021301 / RAD
};

static const double W3[5] = 
{
	(125.0 + (2.0 / 60.0) + (40.39816 / 3600.0)) * DEG,
	-6967919.3622 / RAD,
	6.3622 / RAD,
	0.007625 / RAD,
	-0.00003586 / RAD
};

static const double earth[5] = 
{
	(100.0 + (27.0 / 60.0) + (59.22059 / 3600.0)) * DEG,
	129597742.2758 / RAD,
	-0.0202 / RAD,
	0.000009 / RAD,
	0.00000015 / RAD	
};

static const double peri[5] =
{
	(102.0 + (56.0 / 60.0) + (14.42753 / 3600.0)) * DEG,
	1161.2283 / RAD,
	0.5327 / RAD,
	-0.000138 / RAD,
	0
};

/* Delaunay's arguments.*/
static const double del[4][5] = {
	{ 5.198466741027443, 7771.377146811758394, -0.000028449351621, 0.000000031973462, -0.000000000154365 },
	{ -0.043125180208125, 628.301955168488007, -0.000002680534843, 0.000000000712676, 0.000000000000727 },
	{ 2.355555898265799, 8328.691426955554562, 0.000157027757616, 0.000000250411114, -0.000000001186339 },
	{ 1.627905233371468, 8433.466158130539043, -0.000059392100004, -0.000000004949948, 0.000000000020217 }
};


static const double zeta[2] = 
{
	(218.0 + (18.0 / 60.0) + (59.95571 / 3600.0)) * DEG, 
	((1732559343.73604 / RAD) + PRECES)
};


/* Planetary arguments */
static const double p[8][2] = 
{
	{(252 + 15 / C1 + 3.25986 / C2 ) * DEG, 538101628.68898 / RAD },
	{(181 + 58 / C1 + 47.28305 / C2) * DEG, 210664136.43355 / RAD },
	{(100.0 + (27.0 / 60.0) + (59.22059 / 3600.0)) * DEG, 129597742.2758 / RAD},
	{(355 + 25 / C1 + 59.78866 / C2) * DEG, 68905077.59284 / RAD },
	{(34 + 21 / C1 + 5.34212 / C2) * DEG, 10925660.42861 / RAD },
	{(50 + 4 / C1 + 38.89694 / C2) * DEG, 4399609.65932 / RAD },
	{(314 + 3 / C1 + 18.01841 / C2) * DEG, 1542481.19393 / RAD },
	{(304 + 20 / C1 + 55.19575 / C2) * DEG, 786550.32074 / RAD }
};

/* precision */
static double pre[3];

/* ELP 2000-82B Arguments */
static const struct main_problem main_elp1[ELP1_SIZE] = 
{
	{{0, 0, 0, 2}, -411.602870, {168.480000, -18433.810000, -121.620000, 0.400000, -0.180000}},
	{{0, 0, 0, 4}, 0.420340, {-0.390000, 37.650000, 0.570000, 0.000000, 0.000000}},
	{{0, 0, 0, 6}, -0.000590, {0.000000, -0.080000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 1, -6}, 0.000160, {0.000000, 0.020000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 1, -4}, -0.080190, {0.770000, -7.220000, -1.460000, 0.000000, 0.000000}},
	{{0, 0, 1, -2}, 39.533930, {-395.240000, 1788.330000, 720.910000, -2.600000, 0.900000}},
	{{0, 0, 1, 0}, 22639.550000, {0.000000, 0.000000, 412529.610000, 0.000000, 0.000000}},
	{{0, 0, 1, 2}, -45.100320, {17.410000, -2019.780000, -830.200000, 0.090000, -0.020000}},
	{{0, 0, 1, 4}, 0.090920, {-0.070000, 8.140000, 1.730000, 0.000000, 0.000000}},
	{{0, 0, 1, 6}, -0.000190, {0.000000, -0.020000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 2, -4}, 0.000680, {0.070000, 0.060000, 0.030000, 0.000000, 0.000000}},
	{{0, 0, 2, -2}, -1.372590, {-7.580000, -62.490000, -50.120000, -0.070000, 0.000000}},
	{{0, 0, 2, 0}, 769.023260, {-257.510000, -47.390000, 28008.990000, -6.830000, -0.560000}},
	{{0, 0, 2, 2}, -3.997670, {2.400000, -178.940000, -146.180000, 0.050000, 0.000000}},
	{{0, 0, 2, 4}, 0.012970, {-0.010000, 1.160000, 0.480000, 0.000000, 0.000000}},
	{{0, 0, 2, 6}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 3, -4}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 3, -2}, -0.056840, {0.380000, -2.650000, -3.130000, 0.000000, 0.000000}},
	{{0, 0, 3, 0}, 36.123640, {-28.420000, -5.100000, 1972.740000, -0.930000, -0.080000}},
	{{0, 0, 3, 2}, -0.328660, {0.320000, -14.700000, -17.990000, 0.000000, 0.000000}},
	{{0, 0, 3, 4}, 0.001530, {0.000000, 0.140000, 0.080000, 0.000000, 0.000000}},
	{{0, 0, 4, -2}, -0.003540, {0.040000, -0.170000, -0.260000, 0.000000, 0.000000}},
	{{0, 0, 4, 0}, 1.933670, {-2.650000, -0.440000, 140.750000, -0.100000, 0.000000}},
	{{0, 0, 4, 2}, -0.026020, {0.040000, -1.160000, -1.900000, 0.000000, 0.000000}},
	{{0, 0, 4, 4}, 0.000160, {0.000000, 0.010000, 0.010000, 0.000000, 0.000000}},
	{{0, 0, 5, -2}, -0.000230, {0.000000, -0.010000, -0.020000, 0.000000, 0.000000}},
	{{0, 0, 5, 0}, 0.111000, {-0.230000, -0.040000, 10.100000, 0.000000, 0.000000}},
	{{0, 0, 5, 2}, -0.002010, {0.000000, -0.090000, -0.180000, 0.000000, 0.000000}},
	{{0, 0, 5, 4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 6, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 6, 0}, 0.006650, {-0.020000, 0.000000, 0.730000, 0.000000, 0.000000}},
	{{0, 0, 6, 2}, -0.000150, {0.000000, 0.000000, -0.020000, 0.000000, 0.000000}},
	{{0, 0, 7, 0}, 0.000410, {0.000000, 0.000000, 0.050000, 0.000000, 0.000000}},
	{{0, 0, 7, 2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 8, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -7, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -6, 0}, -0.000240, {0.000000, 0.000000, -0.030000, -0.010000, 0.000000}},
	{{0, 1, -5, -2}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -5, 0}, -0.003320, {-0.070000, 0.000000, -0.300000, -0.200000, 0.000000}},
	{{0, 1, -4, -2}, 0.000640, {0.010000, 0.030000, 0.050000, 0.040000, 0.000000}},
	{{0, 1, -4, 0}, -0.047020, {-1.040000, 0.030000, -3.420000, -2.820000, 0.000000}},
	{{0, 1, -4, 2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -3, -4}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -3, -2}, 0.006230, {0.140000, 0.280000, 0.340000, 0.370000, 0.000000}},
	{{0, 1, -3, 0}, -0.669400, {-15.550000, 0.380000, -36.530000, -40.100000, -0.020000}},
	{{0, 1, -3, 2}, 0.001060, {0.020000, 0.050000, 0.060000, 0.070000, 0.000000}},
	{{0, 1, -2, -4}, -0.000170, {0.000000, -0.020000, 0.000000, -0.010000, 0.000000}},
	{{0, 1, -2, -2}, 0.051650, {1.170000, 2.290000, 1.890000, 3.090000, 0.000000}},
	{{0, 1, -2, 0}, -9.679380, {-235.290000, 4.900000, -352.520000, -579.940000, -0.150000}},
	{{0, 1, -2, 2}, 0.024290, {0.740000, 1.090000, 0.890000, 1.460000, 0.000000}},
	{{0, 1, -2, 4}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -1, -4}, -0.000630, {-0.010000, -0.060000, -0.010000, -0.040000, 0.000000}},
	{{0, 1, -1, -2}, 0.301570, {6.600000, 13.340000, 5.610000, 18.070000, 0.000000}},
	{{0, 1, -1, 0}, -147.326540, {-3778.620000, 68.680000, -2688.530000, -8829.170000, -0.760000}},
	{{0, 1, -1, 2}, -0.080490, {0.250000, -3.630000, -1.390000, -4.820000, -0.010000}},
	{{0, 1, -1, 4}, 0.000130, {0.000000, 0.010000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 0, -4}, -0.000230, {0.010000, -0.020000, 0.000000, -0.010000, 0.000000}},
	{{0, 1, 0, -2}, 0.075010, {-6.720000, 3.040000, 0.850000, 4.460000, 0.000000}},
	{{0, 1, 0, 0}, -666.441860, {-5206.840000, 258.790000, -555.980000, -39887.790000, 1.730000}},
	{{0, 1, 0, 2}, 0.413420, {1.500000, 18.150000, 1.190000, 24.720000, 0.010000}},
	{{0, 1, 0, 4}, -0.000830, {0.000000, -0.070000, 0.000000, -0.050000, 0.000000}},
	{{0, 1, 1, -4}, 0.000110, {0.000000, 0.010000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 1, -2}, -0.082820, {1.730000, -3.730000, -1.500000, -4.950000, 0.000000}},
	{{0, 1, 1, 0}, -109.384190, {-2193.780000, 51.640000, -2018.130000, -6556.100000, 0.540000}},
	{{0, 1, 1, 2}, 0.263380, {4.530000, 11.650000, 5.030000, 15.780000, 0.000000}},
	{{0, 1, 1, 4}, -0.000620, {0.000000, -0.060000, -0.010000, -0.040000, 0.000000}},
	{{0, 1, 2, -4}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 2, -2}, 0.026770, {1.100000, 1.200000, 0.970000, 1.600000, 0.000000}},
	{{0, 1, 2, 0}, -7.630410, {-156.070000, 4.090000, -279.890000, -457.410000, 0.120000}},
	{{0, 1, 2, 2}, 0.043380, {0.810000, 1.920000, 1.610000, 2.600000, 0.000000}},
	{{0, 1, 2, 4}, -0.000150, {0.000000, -0.010000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 3, -2}, 0.001660, {0.060000, 0.080000, 0.090000, 0.100000, 0.000000}},
	{{0, 1, 3, 0}, -0.545940, {-11.180000, 0.340000, -29.970000, -32.730000, 0.010000}},
	{{0, 1, 3, 2}, 0.005260, {0.100000, 0.230000, 0.290000, 0.320000, 0.000000}},
	{{0, 1, 3, 4}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 4, -2}, 0.000140, {0.000000, 0.000000, 0.010000, 0.000000, 0.000000}},
	{{0, 1, 4, 0}, -0.039470, {-0.800000, 0.030000, -2.890000, -2.370000, 0.000000}},
	{{0, 1, 4, 2}, 0.000550, {0.010000, 0.020000, 0.040000, 0.030000, 0.000000}},
	{{0, 1, 5, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 5, 0}, -0.002870, {-0.060000, 0.000000, -0.260000, -0.170000, 0.000000}},
	{{0, 1, 5, 2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 6, 0}, -0.000210, {0.000000, 0.000000, -0.020000, -0.010000, 0.000000}},
	{{0, 1, 7, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -5, 0}, -0.000090, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{0, 2, -4, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -4, 0}, -0.001190, {-0.040000, 0.000000, -0.090000, -0.140000, 0.000000}},
	{{0, 2, -3, -2}, 0.000140, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{0, 2, -3, 0}, -0.015140, {-0.500000, 0.010000, -0.830000, -1.810000, 0.000000}},
	{{0, 2, -3, 2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -2, -2}, 0.001000, {0.030000, 0.040000, 0.040000, 0.120000, 0.000000}},
	{{0, 2, -2, 0}, -0.193110, {-6.230000, 0.130000, -7.040000, -23.130000, 0.000000}},
	{{0, 2, -2, 2}, 0.000600, {0.020000, 0.030000, 0.020000, 0.070000, 0.000000}},
	{{0, 2, -1, -2}, 0.004730, {0.120000, 0.210000, 0.090000, 0.570000, 0.000000}},
	{{0, 2, -1, 0}, -2.568130, {-81.060000, 1.440000, -46.940000, -307.620000, 0.000000}},
	{{0, 2, -1, 2}, 0.000130, {0.050000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{0, 2, 0, -2}, -0.003550, {-0.260000, -0.160000, 0.020000, -0.420000, 0.000000}},
	{{0, 2, 0, 0}, -7.448040, {-12.640000, 2.980000, -9.320000, -891.270000, 0.030000}},
	{{0, 2, 0, 2}, 0.005190, {-0.030000, 0.230000, 0.020000, 0.620000, 0.000000}},
	{{0, 2, 0, 4}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, 1, -2}, -0.000400, {0.080000, -0.020000, 0.000000, -0.050000, 0.000000}},
	{{0, 2, 1, 0}, -1.161770, {-19.750000, 0.430000, -21.520000, -139.130000, 0.000000}},
	{{0, 2, 1, 2}, 0.002700, {0.030000, 0.120000, 0.050000, 0.320000, 0.000000}},
	{{0, 2, 2, -2}, 0.000400, {0.020000, 0.020000, 0.020000, 0.050000, 0.000000}},
	{{0, 2, 2, 0}, -0.065130, {-0.790000, 0.020000, -2.390000, -7.800000, 0.000000}},
	{{0, 2, 2, 2}, 0.000350, {0.000000, 0.020000, 0.010000, 0.040000, 0.000000}},
	{{0, 2, 3, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, 3, 0}, -0.003410, {0.000000, 0.000000, -0.190000, -0.410000, 0.000000}},
	{{0, 2, 3, 2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, 4, 0}, -0.000150, {0.000000, 0.000000, -0.010000, -0.020000, 0.000000}},
	{{0, 3, -4, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, -3, 0}, -0.000350, {-0.010000, 0.000000, -0.020000, -0.060000, 0.000000}},
	{{0, 3, -2, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, -2, 0}, -0.004160, {-0.150000, 0.000000, -0.150000, -0.750000, 0.000000}},
	{{0, 3, -2, 2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, -1, -2}, 0.000080, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{0, 3, -1, 0}, -0.051420, {-1.820000, 0.030000, -0.940000, -9.240000, 0.000000}},
	{{0, 3, -1, 2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, 0, -2}, -0.000160, {0.000000, 0.000000, 0.000000, -0.030000, 0.000000}},
	{{0, 3, 0, 0}, -0.101360, {1.160000, 0.040000, -0.200000, -18.190000, 0.000000}},
	{{0, 3, 0, 2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{0, 3, 1, 0}, -0.017960, {-0.300000, 0.000000, -0.330000, -3.230000, 0.000000}},
	{{0, 3, 1, 2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, 2, 0}, -0.000950, {-0.010000, 0.000000, -0.030000, -0.170000, 0.000000}},
	{{0, 3, 3, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 4, -2, 0}, -0.000090, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{0, 4, -1, 0}, -0.001080, {-0.040000, 0.000000, -0.020000, -0.260000, 0.000000}},
	{{0, 4, 0, 0}, -0.001280, {0.060000, 0.000000, 0.000000, -0.300000, 0.000000}},
	{{0, 4, 1, 0}, -0.000320, {0.000000, 0.000000, 0.000000, -0.080000, 0.000000}},
	{{0, 4, 2, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 5, -1, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -4, 0, 0}, 0.000110, {0.000000, 0.000000, 0.000000, 0.030000, 0.040000}},
	{{1, -3, -2, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -3, -1, 0}, 0.000200, {0.000000, 0.000000, 0.000000, 0.040000, 0.080000}},
	{{1, -3, 0, 0}, 0.002540, {0.030000, 0.000000, 0.000000, 0.450000, 0.990000}},
	{{1, -3, 1, 0}, 0.000070, {0.000000, 0.000000, 0.000000, 0.010000, 0.030000}},
	{{1, -3, 2, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -2, -3, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{1, -2, -2, 0}, 0.000250, {0.000000, 0.000000, 0.010000, 0.030000, 0.100000}},
	{{1, -2, -1, -2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{1, -2, -1, 0}, 0.002290, {0.060000, 0.000000, 0.050000, 0.270000, 0.890000}},
	{{1, -2, 0, -2}, -0.000320, {0.000000, -0.010000, 0.000000, -0.040000, -0.120000}},
	{{1, -2, 0, 0}, 0.043040, {-0.010000, 0.000000, 0.060000, 5.140000, 16.740000}},
	{{1, -2, 0, 2}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{1, -2, 1, -2}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{1, -2, 1, 0}, -0.000330, {-0.160000, 0.000000, -0.010000, -0.040000, -0.130000}},
	{{1, -2, 2, 0}, -0.000370, {-0.030000, 0.000000, -0.020000, -0.040000, -0.140000}},
	{{1, -2, 3, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{1, -1, -5, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, -4, 0}, 0.000140, {0.000000, 0.000000, 0.010000, 0.000000, 0.050000}},
	{{1, -1, -3, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, -3, 0}, 0.000690, {0.020000, 0.000000, 0.040000, 0.040000, 0.270000}},
	{{1, -1, -3, 2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, -2, -2}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, -0.040000}},
	{{1, -1, -2, 0}, -0.006940, {-0.250000, 0.000000, -0.230000, -0.420000, -2.700000}},
	{{1, -1, -2, 2}, 0.000130, {0.000000, 0.000000, 0.000000, 0.000000, 0.050000}},
	{{1, -1, -1, -2}, -0.001000, {0.000000, -0.040000, -0.020000, -0.060000, -0.390000}},
	{{1, -1, -1, 0}, -0.136360, {-3.240000, 0.190000, -2.410000, -8.170000, -53.020000}},
	{{1, -1, -1, 2}, 0.000130, {0.000000, 0.000000, 0.000000, 0.000000, 0.050000}},
	{{1, -1, 0, -4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, 0, -2}, -0.012810, {-0.130000, -0.570000, 0.000000, -0.770000, -4.980000}},
	{{1, -1, 0, 0}, -0.556940, {-40.230000, 1.700000, -0.010000, -33.620000, -216.550000}},
	{{1, -1, 0, 2}, 0.001550, {0.090000, 0.070000, 0.000000, 0.090000, 0.600000}},
	{{1, -1, 1, -2}, -0.002120, {-0.020000, -0.090000, -0.040000, -0.130000, -0.830000}},
	{{1, -1, 1, 0}, -0.120730, {-6.320000, 0.250000, -2.380000, -7.230000, -46.940000}},
	{{1, -1, 1, 2}, 0.000600, {0.030000, 0.030000, 0.010000, 0.040000, 0.230000}},
	{{1, -1, 2, -2}, 0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.050000}},
	{{1, -1, 2, 0}, -0.014830, {-0.720000, 0.030000, -0.560000, -0.890000, -5.770000}},
	{{1, -1, 2, 2}, 0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.050000}},
	{{1, -1, 3, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, 3, 0}, -0.001490, {-0.070000, 0.000000, -0.080000, -0.090000, -0.580000}},
	{{1, -1, 3, 2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, 4, 0}, -0.000140, {0.000000, 0.000000, -0.010000, 0.000000, -0.050000}},
	{{1, -1, 5, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, -6, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{1, 0, -5, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, -5, 0}, -0.000720, {-0.020000, 0.000000, -0.060000, 0.000000, -0.280000}},
	{{1, 0, -4, -2}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{1, 0, -4, 0}, -0.009550, {-0.280000, 0.010000, -0.700000, 0.000000, -3.710000}},
	{{1, 0, -4, 2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, -3, -2}, 0.000690, {0.020000, 0.030000, 0.040000, 0.000000, 0.270000}},
	{{1, 0, -3, 0}, -0.128120, {-3.660000, 0.130000, -7.030000, -0.050000, -49.820000}},
	{{1, 0, -3, 2}, 0.000380, {0.060000, 0.020000, 0.030000, 0.000000, 0.140000}},
	{{1, 0, -2, -4}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, -2, -2}, 0.002890, {0.140000, 0.120000, 0.110000, 0.000000, 1.120000}},
	{{1, 0, -2, 0}, -1.752960, {-48.870000, 1.670000, -64.280000, -0.510000, -681.600000}},
	{{1, 0, -2, 2}, -0.001420, {0.050000, -0.070000, -0.050000, 0.000000, -0.550000}},
	{{1, 0, -1, -4}, 0.000190, {0.000000, 0.020000, 0.000000, 0.000000, 0.070000}},
	{{1, 0, -1, -2}, -0.023790, {0.120000, -1.090000, -0.420000, 0.000000, -9.250000}},
	{{1, 0, -1, 0}, -18.584670, {-437.660000, 18.910000, -346.510000, -1.000000, -7226.220000}},
	{{1, 0, -1, 2}, 0.015980, {0.320000, 0.690000, 0.300000, 0.000000, 6.210000}},
	{{1, 0, -1, 4}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, -0.040000}},
	{{1, 0, 0, -4}, 0.001230, {0.010000, 0.110000, 0.000000, 0.000000, 0.480000}},
	{{1, 0, 0, -2}, -0.583320, {-5.890000, -25.980000, -0.120000, 0.000000, -226.810000}},
	{{1, 0, 0, 0}, -124.988060, {-2831.880000, 136.180000, -86.600000, -2.680000, -48598.150000}},
	{{1, 0, 0, 2}, 0.254290, {5.730000, 11.110000, 0.260000, 0.000000, 98.870000}},
	{{1, 0, 0, 4}, -0.000540, {-0.010000, -0.050000, 0.000000, 0.000000, -0.210000}},
	{{1, 0, 1, -4}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{1, 0, 1, -2}, -0.040120, {-0.220000, -1.800000, -0.720000, 0.000000, -15.600000}},
	{{1, 0, 1, 0}, -8.453080, {-187.780000, 10.100000, -158.620000, 0.770000, -3286.750000}},
	{{1, 0, 1, 2}, 0.044500, {0.980000, 1.940000, 0.840000, 0.000000, 17.300000}},
	{{1, 0, 1, 4}, -0.000150, {0.000000, -0.010000, 0.000000, 0.000000, -0.060000}},
	{{1, 0, 2, -2}, -0.000770, {0.030000, -0.030000, -0.030000, 0.000000, -0.300000}},
	{{1, 0, 2, 0}, -0.583990, {-12.360000, 0.750000, -21.520000, 0.130000, -227.070000}},
	{{1, 0, 2, 2}, 0.005400, {0.120000, 0.240000, 0.200000, 0.000000, 2.100000}},
	{{1, 0, 2, 4}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, 0, 3, -2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{1, 0, 3, 0}, -0.040740, {-0.810000, 0.060000, -2.240000, 0.020000, -15.840000}},
	{{1, 0, 3, 2}, 0.000560, {0.010000, 0.020000, 0.030000, 0.000000, 0.220000}},
	{{1, 0, 4, 0}, -0.002850, {-0.050000, 0.000000, -0.210000, 0.000000, -1.110000}},
	{{1, 0, 4, 2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{1, 0, 5, 0}, -0.000200, {0.000000, 0.000000, -0.020000, 0.000000, -0.080000}},
	{{1, 0, 6, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, -5, 0}, 0.000110, {0.000000, 0.000000, 0.010000, 0.000000, 0.040000}},
	{{1, 1, -4, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, -4, 0}, 0.001550, {0.020000, 0.000000, 0.110000, 0.090000, 0.600000}},
	{{1, 1, -3, -2}, -0.000120, {0.000000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{1, 1, -3, 0}, 0.022960, {0.300000, 0.000000, 1.260000, 1.380000, 8.930000}},
	{{1, 1, -3, 2}, -0.000060, {0.020000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{1, 1, -2, -2}, -0.000710, {0.000000, -0.030000, -0.030000, -0.040000, -0.270000}},
	{{1, 1, -2, 0}, 0.349660, {5.360000, -0.110000, 12.830000, 20.950000, 135.950000}},
	{{1, 1, -2, 2}, 0.000630, {0.000000, 0.030000, 0.020000, 0.040000, 0.250000}},
	{{1, 1, -1, -2}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{1, 1, -1, 0}, 1.077730, {9.380000, -1.150000, 21.490000, 64.240000, 418.880000}},
	{{1, 1, -1, 2}, 0.000000, {0.030000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, 0, -2}, 0.000170, {0.030000, 0.000000, 0.070000, 0.000000, 0.070000}},
	{{1, 1, 0, 0}, 17.955120, {6.590000, -11.640000, 21.840000, 1074.240000, 6981.340000}},
	{{1, 1, 0, 2}, -0.035870, {0.000000, -1.580000, -0.050000, -2.150000, -13.950000}},
	{{1, 1, 0, 4}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{1, 1, 1, -2}, -0.001570, {0.360000, -0.040000, -0.040000, -0.070000, -0.560000}},
	{{1, 1, 1, 0}, 1.261860, {1.830000, -0.900000, 24.500000, 75.500000, 490.640000}},
	{{1, 1, 1, 2}, -0.006590, {0.000000, -0.290000, -0.130000, -0.390000, -2.560000}},
	{{1, 1, 1, 4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, 2, -2}, 0.000280, {0.030000, 0.010000, 0.000000, 0.020000, 0.110000}},
	{{1, 1, 2, 0}, 0.092050, {0.220000, -0.070000, 3.460000, 5.510000, 35.790000}},
	{{1, 1, 2, 2}, -0.000840, {0.000000, -0.040000, -0.030000, -0.050000, -0.330000}},
	{{1, 1, 3, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, 3, 0}, 0.006790, {0.020000, 0.000000, 0.380000, 0.410000, 2.640000}},
	{{1, 1, 3, 2}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, -0.040000}},
	{{1, 1, 4, 0}, 0.000500, {0.000000, 0.000000, 0.040000, 0.030000, 0.200000}},
	{{1, 1, 5, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{1, 2, -4, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{1, 2, -3, 0}, 0.000380, {0.020000, 0.000000, 0.020000, 0.050000, 0.150000}},
	{{1, 2, -2, 0}, 0.003610, {0.160000, 0.000000, 0.130000, 0.430000, 1.400000}},
	{{1, 2, -1, 0}, -0.003000, {-0.180000, 0.000000, -0.050000, -0.360000, -1.160000}},
	{{1, 2, -1, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 2, 0, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.010000, 0.010000}},
	{{1, 2, 0, 0}, -0.039680, {-0.610000, 0.040000, -0.070000, -4.750000, -15.430000}},
	{{1, 2, 0, 2}, 0.000110, {0.000000, 0.000000, 0.000000, 0.010000, 0.040000}},
	{{1, 2, 1, -2}, 0.000020, {-0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 2, 1, 0}, -0.008490, {-0.160000, 0.000000, -0.170000, -1.020000, -3.300000}},
	{{1, 2, 1, 2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{1, 2, 2, 0}, -0.001020, {-0.020000, 0.000000, -0.040000, -0.120000, -0.400000}},
	{{1, 2, 2, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 2, 3, 0}, -0.000110, {0.000000, 0.000000, 0.000000, -0.010000, -0.040000}},
	{{1, 2, 4, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 3, -2, 0}, 0.000070, {0.000000, 0.000000, 0.000000, 0.010000, 0.030000}},
	{{1, 3, -1, 0}, -0.000110, {0.000000, 0.000000, 0.000000, -0.020000, -0.040000}},
	{{1, 3, 0, -2}, 0.000000, {0.020000, -0.010000, 0.010000, 0.020000, 0.000000}},
	{{1, 3, 0, 0}, -0.000560, {-0.010000, 0.000000, 0.000000, -0.100000, -0.220000}},
	{{1, 3, 1, 0}, -0.000090, {0.000000, 0.000000, 0.000000, -0.020000, -0.030000}},
	{{1, 4, 0, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -6, 0, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -5, -2, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -5, -1, 0}, 0.000270, {0.000000, 0.000000, 0.000000, 0.080000, 0.000000}},
	{{2, -5, 0, 0}, 0.000500, {0.020000, 0.000000, 0.000000, 0.150000, 0.000000}},
	{{2, -5, 1, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{2, -4, -3, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -4, -2, 0}, 0.000320, {0.010000, 0.000000, 0.010000, 0.080000, 0.000000}},
	{{2, -4, -1, 0}, 0.008120, {0.240000, 0.000000, 0.150000, 1.940000, 0.000000}},
	{{2, -4, -1, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -4, 0, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -4, 0, 0}, 0.013310, {0.580000, 0.000000, 0.020000, 3.180000, 0.000000}},
	{{2, -4, 0, 2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -4, 1, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -4, 1, 0}, 0.001330, {0.060000, 0.000000, 0.030000, 0.320000, 0.000000}},
	{{2, -4, 2, 0}, 0.000120, {0.000000, 0.000000, 0.000000, 0.030000, 0.000000}},
	{{2, -4, 3, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, -4, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, -3, 0}, 0.000490, {0.010000, 0.000000, 0.030000, 0.090000, 0.000000}},
	{{2, -3, -2, -2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, -2, 0}, 0.009650, {0.260000, 0.000000, 0.350000, 1.730000, 0.000000}},
	{{2, -3, -2, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, -1, -2}, -0.000210, {0.000000, 0.000000, 0.000000, -0.040000, 0.000000}},
	{{2, -3, -1, 0}, 0.246970, {6.330000, -0.030000, 4.520000, 44.320000, 0.000000}},
	{{2, -3, -1, 2}, -0.000520, {-0.010000, -0.020000, 0.000000, -0.090000, 0.000000}},
	{{2, -3, 0, -2}, 0.001630, {0.000000, 0.070000, 0.000000, 0.290000, 0.000000}},
	{{2, -3, 0, 0}, 0.339830, {13.410000, -0.120000, 0.740000, 60.980000, 0.000000}},
	{{2, -3, 0, 2}, -0.000750, {-0.030000, -0.030000, 0.000000, -0.130000, 0.000000}},
	{{2, -3, 1, -2}, -0.000650, {-0.020000, -0.030000, -0.010000, -0.120000, 0.000000}},
	{{2, -3, 1, 0}, 0.033000, {1.420000, -0.020000, 0.670000, 5.920000, 0.000000}},
	{{2, -3, 1, 2}, -0.000160, {0.000000, 0.000000, 0.000000, -0.030000, 0.000000}},
	{{2, -3, 2, -2}, -0.000050, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{2, -3, 2, 0}, 0.002950, {0.140000, 0.000000, 0.110000, 0.530000, 0.000000}},
	{{2, -3, 2, 2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, 3, 0}, 0.000260, {0.010000, 0.000000, 0.010000, 0.050000, 0.000000}},
	{{2, -3, 4, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, -5, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, -4, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, -4, 0}, 0.000880, {0.010000, 0.000000, 0.060000, 0.100000, 0.000000}},
	{{2, -2, -3, -2}, -0.000130, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{2, -2, -3, 0}, 0.015440, {0.290000, 0.000000, 0.840000, 1.850000, 0.000000}},
	{{2, -2, -2, -2}, -0.001110, {-0.030000, -0.050000, -0.040000, -0.130000, 0.000000}},
	{{2, -2, -2, 0}, 0.294220, {6.880000, -0.030000, 10.730000, 35.210000, 0.000000}},
	{{2, -2, -2, 2}, -0.000720, {0.000000, -0.030000, -0.030000, -0.090000, 0.000000}},
	{{2, -2, -1, -2}, -0.004340, {-0.200000, -0.190000, -0.080000, -0.520000, 0.000000}},
	{{2, -2, -1, 0}, 7.371730, {165.380000, -1.060000, 134.790000, 881.890000, -0.040000}},
	{{2, -2, -1, 2}, -0.015410, {-0.330000, -0.690000, -0.290000, -1.840000, 0.000000}},
	{{2, -2, -1, 4}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 0, -4}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 0, -2}, 0.063140, {0.430000, 2.820000, -0.040000, 7.550000, 0.000000}},
	{{2, -2, 0, 0}, 8.050760, {289.140000, -3.140000, 21.090000, 962.910000, -0.060000}},
	{{2, -2, 0, 2}, -0.018160, {-0.620000, -0.810000, -0.120000, -2.170000, 0.000000}},
	{{2, -2, 0, 4}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 1, -4}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 1, -2}, -0.016440, {-0.380000, -0.740000, -0.300000, -1.970000, 0.000000}},
	{{2, -2, 1, 0}, 0.751730, {29.040000, -0.430000, 15.470000, 89.900000, 0.000000}},
	{{2, -2, 1, 2}, -0.003780, {-0.140000, -0.170000, -0.090000, -0.450000, 0.000000}},
	{{2, -2, 1, 4}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 2, -2}, -0.001330, {-0.030000, -0.060000, -0.050000, -0.160000, 0.000000}},
	{{2, -2, 2, 0}, 0.064390, {2.630000, -0.050000, 2.500000, 7.700000, 0.000000}},
	{{2, -2, 2, 2}, -0.000550, {-0.020000, -0.020000, -0.020000, -0.070000, 0.000000}},
	{{2, -2, 3, -2}, -0.000110, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{2, -2, 3, 0}, 0.005380, {0.230000, 0.000000, 0.310000, 0.640000, 0.000000}},
	{{2, -2, 3, 2}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 4, 0}, 0.000440, {0.020000, 0.000000, 0.030000, 0.050000, 0.000000}},
	{{2, -2, 5, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -6, 0}, 0.000100, {0.000000, 0.000000, 0.010000, 0.000000, 0.000000}},
	{{2, -1, -5, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -5, 0}, 0.001780, {0.010000, 0.000000, 0.160000, 0.110000, 0.000000}},
	{{2, -1, -4, -2}, -0.000360, {0.000000, -0.020000, -0.030000, -0.020000, 0.000000}},
	{{2, -1, -4, 0}, 0.029490, {0.410000, 0.000000, 2.150000, 1.770000, 0.000000}},
	{{2, -1, -4, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -3, -4}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -3, -2}, -0.003650, {-0.080000, -0.160000, -0.200000, -0.220000, 0.000000}},
	{{2, -1, -3, 0}, 0.478420, {8.720000, -0.050000, 26.150000, 28.630000, 0.000000}},
	{{2, -1, -3, 2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -2, -4}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -2, -2}, -0.028040, {-0.740000, -1.250000, -1.030000, -1.680000, 0.000000}},
	{{2, -1, -2, 0}, 8.605820, {187.040000, -1.140000, 313.850000, 515.030000, 0.020000}},
	{{2, -1, -2, 2}, -0.021570, {-0.210000, -0.970000, -0.790000, -1.290000, 0.000000}},
	{{2, -1, -2, 4}, 0.000120, {0.000000, 0.010000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -1, -4}, -0.000150, {0.000000, -0.010000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -1, -2}, -0.061030, {-3.570000, -2.720000, -1.220000, -3.650000, 0.000000}},
	{{2, -1, -1, 0}, 205.443150, {4157.780000, -32.990000, 3751.430000, 12284.720000, -0.130000}},
	{{2, -1, -1, 2}, -0.426240, {-8.300000, -19.010000, -7.930000, -25.480000, 0.000000}},
	{{2, -1, -1, 4}, 0.000930, {0.020000, 0.080000, 0.020000, 0.060000, 0.000000}},
	{{2, -1, 0, -4}, -0.002860, {-0.020000, -0.260000, 0.000000, -0.170000, 0.000000}},
	{{2, -1, 0, -2}, 2.146190, {18.770000, 95.810000, -0.840000, 128.220000, 0.050000}},
	{{2, -1, 0, 0}, 164.734580, {5378.280000, -77.430000, 545.710000, 9844.150000, -0.180000}},
	{{2, -1, 0, 2}, -0.382150, {-11.750000, -16.940000, -3.100000, -22.840000, 0.000000}},
	{{2, -1, 0, 4}, 0.000890, {0.030000, 0.080000, 0.010000, 0.050000, 0.000000}},
	{{2, -1, 1, -4}, 0.000180, {0.000000, 0.020000, 0.000000, 0.010000, 0.000000}},
	{{2, -1, 1, -2}, -0.374530, {-6.500000, -16.890000, -6.910000, -22.360000, 0.000000}},
	{{2, -1, 1, 0}, 14.530780, {497.460000, -9.380000, 307.250000, 867.990000, 0.000000}},
	{{2, -1, 1, 2}, -0.073730, {-2.450000, -3.260000, -1.770000, -4.400000, 0.000000}},
	{{2, -1, 1, 4}, 0.000250, {0.000000, 0.020000, 0.000000, 0.020000, 0.000000}},
	{{2, -1, 2, -4}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 2, -2}, -0.028870, {-0.540000, -1.300000, -1.060000, -1.720000, 0.000000}},
	{{2, -1, 2, 0}, 1.177040, {41.720000, -0.960000, 46.230000, 70.280000, 0.000000}},
	{{2, -1, 2, 2}, -0.010070, {-0.350000, -0.440000, -0.420000, -0.600000, 0.000000}},
	{{2, -1, 2, 4}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 3, -2}, -0.002320, {-0.050000, -0.100000, -0.130000, -0.140000, 0.000000}},
	{{2, -1, 3, 0}, 0.093200, {3.390000, -0.090000, 5.360000, 5.560000, 0.000000}},
	{{2, -1, 3, 2}, -0.001170, {-0.040000, -0.050000, -0.070000, -0.070000, 0.000000}},
	{{2, -1, 4, -2}, -0.000190, {0.000000, 0.000000, -0.010000, -0.010000, 0.000000}},
	{{2, -1, 4, 0}, 0.007300, {0.270000, 0.000000, 0.550000, 0.440000, 0.000000}},
	{{2, -1, 4, 2}, -0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 5, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 5, 0}, 0.000570, {0.020000, 0.000000, 0.050000, 0.030000, 0.000000}},
	{{2, -1, 5, 2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 6, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -8, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -7, 0}, 0.000360, {0.000000, 0.000000, 0.050000, 0.000000, 0.000000}},
	{{2, 0, -6, -2}, -0.000110, {0.000000, 0.000000, -0.010000, 0.000000, 0.000000}},
	{{2, 0, -6, 0}, 0.004970, {0.100000, 0.000000, 0.540000, 0.000000, 0.000000}},
	{{2, 0, -5, -2}, -0.001110, {-0.020000, -0.050000, -0.100000, 0.000000, 0.000000}},
	{{2, 0, -5, 0}, 0.068500, {1.400000, -0.030000, 6.240000, 0.000000, 0.000000}},
	{{2, 0, -5, 2}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -4, -4}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -4, -2}, -0.010770, {-0.240000, -0.480000, -0.790000, 0.000000, 0.000000}},
	{{2, 0, -4, 0}, 0.948270, {19.710000, -0.300000, 69.110000, 0.060000, 0.000000}},
	{{2, 0, -4, 2}, -0.000540, {0.000000, -0.030000, -0.040000, 0.000000, 0.000000}},
	{{2, 0, -3, -4}, 0.000240, {0.000000, 0.020000, 0.010000, 0.000000, 0.000000}},
	{{2, 0, -3, -2}, -0.091540, {-2.110000, -4.080000, -5.030000, 0.000000, 0.000000}},
	{{2, 0, -3, 0}, 13.194000, {279.870000, -3.190000, 721.290000, 0.650000, 0.140000}},
	{{2, 0, -3, 2}, 0.004670, {0.380000, 0.200000, 0.240000, 0.000000, 0.000000}},
	{{2, 0, -3, 4}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -2, -4}, 0.000370, {0.030000, 0.030000, 0.020000, 0.000000, 0.000000}},
	{{2, 0, -2, -2}, -0.560650, {-14.540000, -24.990000, -20.690000, -0.010000, 0.000000}},
	{{2, 0, -2, 0}, 211.654870, {4685.540000, -42.060000, 7715.640000, 7.860000, 2.120000}},
	{{2, 0, -2, 2}, -0.535720, {-4.500000, -24.120000, -19.500000, 0.040000, -0.010000}},
	{{2, 0, -2, 4}, 0.002820, {0.020000, 0.250000, 0.100000, 0.000000, 0.000000}},
	{{2, 0, -1, -6}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -1, -4}, -0.008500, {0.000000, -0.760000, -0.140000, 0.000000, 0.000000}},
	{{2, 0, -1, -2}, 0.179030, {-34.060000, 7.980000, 1.010000, 0.030000, 0.180000}},
	{{2, 0, -1, 0}, 4586.430610, {87132.460000, -842.120000, 83586.180000, -191.170000, 20.310000}},
	{{2, 0, -1, 2}, -9.366010, {-173.580000, -417.660000, -173.700000, 0.480000, -0.060000}},
	{{2, 0, -1, 4}, 0.019930, {0.360000, 1.780000, 0.390000, 0.000000, 0.000000}},
	{{2, 0, -1, 6}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 0, -6}, 0.000150, {0.000000, 0.020000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 0, -4}, -0.075180, {-0.700000, -6.720000, 0.060000, 0.000000, 0.000000}},
	{{2, 0, 0, -2}, 55.178010, {530.970000, 2463.550000, -15.350000, -6.960000, 1.200000}},
	{{2, 0, 0, 0}, 2369.912270, {69551.140000, -1472.500000, 10817.070000, -255.360000, 22.070000}},
	{{2, 0, 0, 2}, -5.741700, {-155.850000, -254.000000, -59.770000, 0.640000, -0.100000}},
	{{2, 0, 0, 4}, 0.013720, {0.350000, 1.220000, 0.210000, 0.000000, 0.000000}},
	{{2, 0, 0, 6}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 1, -6}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 1, -4}, 0.000560, {0.100000, 0.050000, 0.010000, 0.000000, 0.000000}},
	{{2, 0, 1, -2}, -6.383250, {-72.700000, -288.510000, -117.470000, 1.160000, 0.000000}},
	{{2, 0, 1, 0}, 191.955750, {5619.270000, -153.090000, 4250.550000, -30.100000, 2.780000}},
	{{2, 0, 1, 2}, -0.990230, {-27.730000, -43.700000, -25.330000, 0.160000, -0.020000}},
	{{2, 0, 1, 4}, 0.003440, {0.090000, 0.310000, 0.100000, 0.000000, 0.000000}},
	{{2, 0, 1, 6}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 2, -4}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 2, -2}, -0.453800, {-4.940000, -20.520000, -16.630000, 0.100000, 0.000000}},
	{{2, 0, 2, 0}, 14.379640, {417.800000, -13.710000, 577.470000, -3.080000, 0.280000}},
	{{2, 0, 2, 2}, -0.123860, {-3.490000, -5.450000, -5.280000, 0.030000, 0.000000}},
	{{2, 0, 2, 4}, 0.000590, {0.020000, 0.050000, 0.030000, 0.000000, 0.000000}},
	{{2, 0, 3, -4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 3, -2}, -0.033360, {-0.350000, -1.510000, -1.830000, 0.000000, 0.000000}},
	{{2, 0, 3, 0}, 1.059490, {30.460000, -1.160000, 61.740000, -0.300000, 0.030000}},
	{{2, 0, 3, 2}, -0.013320, {-0.370000, -0.590000, -0.800000, 0.000000, 0.000000}},
	{{2, 0, 3, 4}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 4, -2}, -0.002470, {-0.030000, -0.110000, -0.180000, 0.000000, 0.000000}},
	{{2, 0, 4, 0}, 0.077650, {2.200000, -0.100000, 5.930000, -0.030000, 0.000000}},
	{{2, 0, 4, 2}, -0.001310, {-0.040000, -0.060000, -0.100000, 0.000000, 0.000000}},
	{{2, 0, 4, 4}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 5, -2}, -0.000180, {0.000000, 0.000000, -0.020000, 0.000000, 0.000000}},
	{{2, 0, 5, 0}, 0.005680, {0.160000, 0.000000, 0.540000, 0.000000, 0.000000}},
	{{2, 0, 5, 2}, -0.000120, {0.000000, 0.000000, -0.010000, 0.000000, 0.000000}},
	{{2, 0, 6, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 6, 0}, 0.000410, {0.010000, 0.000000, 0.050000, 0.000000, 0.000000}},
	{{2, 0, 6, 2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 7, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -7, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -6, 0}, 0.000150, {0.010000, 0.000000, 0.020000, 0.000000, 0.000000}},
	{{2, 1, -5, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -5, 0}, 0.001730, {0.120000, 0.000000, 0.160000, 0.100000, 0.000000}},
	{{2, 1, -4, -2}, -0.000200, {-0.020000, 0.000000, -0.010000, -0.010000, 0.000000}},
	{{2, 1, -4, 0}, 0.018190, {1.370000, -0.020000, 1.320000, 1.100000, 0.000000}},
	{{2, 1, -4, 2}, -0.000030, {0.040000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{2, 1, -3, -2}, -0.001160, {-0.100000, -0.050000, -0.060000, -0.070000, 0.000000}},
	{{2, 1, -3, 0}, 0.176240, {15.800000, -0.230000, 9.590000, 10.630000, 0.050000}},
	{{2, 1, -3, 2}, -0.000240, {0.000000, -0.010000, -0.020000, -0.020000, 0.000000}},
	{{2, 1, -2, -4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -2, -2}, -0.003460, {-0.430000, -0.150000, -0.130000, -0.210000, 0.000000}},
	{{2, 1, -2, 0}, 2.488970, {257.300000, -3.720000, 89.710000, 150.190000, 0.260000}},
	{{2, 1, -2, 2}, -0.000010, {-0.340000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -1, -4}, 0.000170, {0.000000, 0.020000, 0.000000, 0.010000, 0.000000}},
	{{2, 1, -1, -2}, -0.002280, {0.760000, -0.110000, -0.050000, -0.130000, 0.000000}},
	{{2, 1, -1, 0}, -28.398100, {915.460000, -16.700000, -528.730000, -1693.300000, -0.380000}},
	{{2, 1, -1, 2}, 0.067420, {-1.630000, 3.040000, 1.290000, 4.020000, 0.000000}},
	{{2, 1, -1, 4}, -0.000160, {0.000000, -0.010000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 0, -4}, 0.001980, {0.030000, 0.180000, 0.000000, 0.120000, 0.000000}},
	{{2, 1, 0, -2}, -1.437240, {-23.320000, -64.110000, -0.160000, -86.030000, -0.040000}},
	{{2, 1, 0, 0}, -24.359100, {-661.000000, 29.130000, -128.060000, -1457.610000, -5.210000}},
	{{2, 1, 0, 2}, 0.065410, {1.460000, 2.860000, 0.750000, 3.910000, 0.030000}},
	{{2, 1, 0, 4}, -0.000170, {0.000000, -0.020000, 0.000000, -0.010000, 0.000000}},
	{{2, 1, 1, -4}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 1, -2}, 0.084720, {-0.380000, 3.850000, 1.590000, 5.050000, 0.000000}},
	{{2, 1, 1, 0}, -2.914640, {-100.960000, 4.260000, -66.130000, -174.490000, -0.750000}},
	{{2, 1, 1, 2}, 0.016050, {0.510000, 0.700000, 0.420000, 0.960000, 0.000000}},
	{{2, 1, 1, 4}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 2, -2}, 0.008890, {0.070000, 0.400000, 0.330000, 0.530000, 0.000000}},
	{{2, 1, 2, 0}, -0.290230, {-11.130000, 0.470000, -11.800000, -17.380000, -0.080000}},
	{{2, 1, 2, 2}, 0.002620, {0.100000, 0.110000, 0.110000, 0.160000, 0.000000}},
	{{2, 1, 2, 4}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 3, -2}, 0.000850, {0.010000, 0.040000, 0.050000, 0.050000, 0.000000}},
	{{2, 1, 3, 0}, -0.026760, {-1.080000, 0.050000, -1.570000, -1.600000, 0.000000}},
	{{2, 1, 3, 2}, 0.000350, {0.010000, 0.020000, 0.020000, 0.020000, 0.000000}},
	{{2, 1, 4, -2}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 4, 0}, -0.002360, {-0.100000, 0.000000, -0.180000, -0.140000, 0.000000}},
	{{2, 1, 4, 2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 5, 0}, -0.000200, {0.000000, 0.000000, -0.020000, -0.010000, 0.000000}},
	{{2, 1, 6, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -5, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -4, 0}, -0.000490, {0.000000, 0.000000, -0.040000, -0.060000, 0.000000}},
	{{2, 2, -3, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -3, 0}, -0.009440, {-0.260000, 0.000000, -0.530000, -1.140000, 0.000000}},
	{{2, 2, -3, 2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -2, -2}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -2, 0}, -0.249900, {-10.760000, 0.110000, -9.200000, -29.880000, -0.020000}},
	{{2, 2, -2, 2}, 0.000600, {0.020000, 0.030000, 0.020000, 0.070000, 0.000000}},
	{{2, 2, -1, -4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -1, -2}, -0.006540, {-0.240000, -0.290000, -0.090000, -0.780000, 0.000000}},
	{{2, 2, -1, 0}, -2.521380, {-84.660000, 0.740000, -46.070000, -302.130000, 0.070000}},
	{{2, 2, -1, 2}, 0.005150, {0.170000, 0.230000, 0.100000, 0.620000, 0.000000}},
	{{2, 2, -1, 4}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, 0, -4}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, 0, -2}, -0.023490, {-0.290000, -0.900000, 0.490000, -2.750000, 0.040000}},
	{{2, 2, 0, 0}, -0.185760, {-6.480000, 0.330000, -6.560000, -22.260000, 0.390000}},
	{{2, 2, 0, 2}, 0.000960, {0.030000, 0.040000, 0.030000, 0.110000, 0.000000}},
	{{2, 2, 1, -2}, -0.002000, {-0.120000, -0.070000, 0.000000, -0.230000, 0.000000}},
	{{2, 2, 1, 0}, -0.013430, {-0.460000, 0.030000, -0.710000, -1.610000, 0.060000}},
	{{2, 2, 1, 2}, 0.000120, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{2, 2, 2, -2}, -0.000140, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{2, 2, 2, 0}, -0.000660, {-0.010000, 0.000000, -0.060000, -0.080000, 0.000000}},
	{{2, 2, 3, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 3, -3, 0}, -0.000090, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{2, 3, -2, 0}, -0.000930, {-0.060000, 0.000000, -0.030000, -0.170000, 0.000000}},
	{{2, 3, -1, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 3, -1, 0}, -0.001880, {-0.120000, 0.000000, -0.030000, -0.340000, 0.000000}},
	{{2, 3, 0, -2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{2, 3, 0, 0}, 0.000380, {0.030000, 0.000000, 0.020000, 0.070000, 0.000000}},
	{{2, 3, 1, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 3, 1, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 4, -2, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 4, -1, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -4, -1, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -4, 0, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -3, -2, 0}, -0.000170, {0.000000, 0.000000, 0.000000, -0.030000, -0.070000}},
	{{3, -3, -1, 0}, -0.000530, {-0.020000, 0.000000, 0.000000, -0.090000, -0.210000}},
	{{3, -3, 0, -2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, -3, 0, 0}, 0.000300, {0.000000, 0.000000, 0.000000, 0.050000, 0.120000}},
	{{3, -3, 1, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{3, -2, -3, 0}, -0.000140, {0.000000, 0.000000, 0.000000, -0.020000, -0.060000}},
	{{3, -2, -2, 0}, -0.003990, {-0.100000, 0.000000, -0.150000, -0.480000, -1.550000}},
	{{3, -2, -1, -2}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{3, -2, -1, 0}, -0.011820, {-0.500000, 0.000000, -0.230000, -1.410000, -4.600000}},
	{{3, -2, -1, 2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{3, -2, 0, -2}, -0.001020, {-0.020000, -0.050000, 0.000000, -0.120000, -0.400000}},
	{{3, -2, 0, 0}, 0.005180, {0.050000, 0.000000, -0.050000, 0.620000, 2.010000}},
	{{3, -2, 1, -2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, -2, 1, 0}, 0.000370, {0.000000, 0.000000, 0.000000, 0.040000, 0.150000}},
	{{3, -1, -4, 0}, -0.000130, {0.000000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{3, -1, -3, 0}, -0.002970, {-0.060000, 0.000000, -0.160000, -0.180000, -1.160000}},
	{{3, -1, -2, 0}, -0.083110, {-1.910000, 0.020000, -3.040000, -4.980000, -32.320000}},
	{{3, -1, -2, 2}, 0.000200, {0.000000, 0.000000, 0.000000, 0.010000, 0.080000}},
	{{3, -1, -1, -2}, -0.001890, {-0.020000, -0.080000, -0.030000, -0.110000, -0.730000}},
	{{3, -1, -1, 0}, -0.231410, {-9.240000, 0.190000, -4.480000, -13.780000, -89.980000}},
	{{3, -1, -1, 2}, 0.000750, {0.030000, 0.030000, 0.020000, 0.050000, 0.290000}},
	{{3, -1, 0, -4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -1, 0, -2}, -0.019480, {-0.340000, -0.870000, 0.000000, -1.160000, -7.570000}},
	{{3, -1, 0, 0}, 0.065070, {-0.180000, 0.160000, -1.040000, 3.930000, 25.300000}},
	{{3, -1, 0, 2}, 0.000200, {0.020000, 0.000000, 0.000000, 0.010000, 0.080000}},
	{{3, -1, 1, -2}, -0.000840, {0.000000, -0.040000, -0.010000, -0.050000, -0.330000}},
	{{3, -1, 1, 0}, 0.003300, {-0.250000, 0.030000, -0.050000, 0.210000, 1.280000}},
	{{3, -1, 1, 2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{3, -1, 2, -2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, -1, 2, 0}, -0.000110, {-0.050000, 0.000000, -0.010000, 0.000000, -0.040000}},
	{{3, -1, 2, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -1, 3, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, 0, -5, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, 0, -4, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, -4, 0}, -0.001740, {0.000000, 0.000000, -0.130000, 0.000000, -0.680000}},
	{{3, 0, -3, -2}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{3, 0, -3, 0}, -0.044420, {-0.910000, 0.000000, -2.430000, -0.030000, -17.270000}},
	{{3, 0, -3, 2}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{3, 0, -2, -2}, -0.000280, {0.020000, -0.010000, 0.000000, 0.000000, -0.110000}},
	{{3, 0, -2, 0}, -1.224120, {-28.100000, 0.420000, -44.730000, -0.150000, -475.970000}},
	{{3, 0, -2, 2}, 0.002980, {0.070000, 0.130000, 0.110000, 0.000000, 1.160000}},
	{{3, 0, -2, 4}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, -1, -4}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{3, 0, -1, -2}, -0.029790, {-0.390000, -1.330000, -0.540000, 0.000000, -11.590000}},
	{{3, 0, -1, 0}, -3.209680, {-121.620000, 3.090000, -62.390000, 1.610000, -1248.010000}},
	{{3, 0, -1, 2}, 0.012150, {0.480000, 0.530000, 0.250000, 0.000000, 4.720000}},
	{{3, 0, -1, 4}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{3, 0, 0, -4}, 0.000250, {0.000000, 0.020000, 0.000000, 0.000000, 0.100000}},
	{{3, 0, 0, -2}, -0.253040, {-3.600000, -11.270000, 0.100000, 0.080000, -98.390000}},
	{{3, 0, 0, 0}, 0.404230, {-12.920000, 2.280000, -14.170000, 1.050000, 157.170000}},
	{{3, 0, 0, 2}, 0.003750, {0.240000, 0.160000, 0.110000, 0.000000, 1.460000}},
	{{3, 0, 0, 4}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, 1, -4}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{3, 0, 1, -2}, -0.010620, {-0.050000, -0.470000, -0.180000, 0.000000, -4.130000}},
	{{3, 0, 1, 0}, 0.000370, {-3.590000, 0.360000, -1.360000, 0.180000, 0.140000}},
	{{3, 0, 1, 2}, 0.000830, {0.060000, 0.030000, 0.030000, 0.000000, 0.320000}},
	{{3, 0, 2, -2}, -0.000510, {0.000000, -0.020000, -0.020000, 0.000000, -0.200000}},
	{{3, 0, 2, 0}, -0.003890, {-0.500000, 0.040000, -0.260000, 0.020000, -1.510000}},
	{{3, 0, 2, 2}, 0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.050000}},
	{{3, 0, 3, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, 3, 0}, -0.000620, {-0.050000, 0.000000, -0.040000, 0.000000, -0.240000}},
	{{3, 0, 3, 2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, 4, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{3, 1, -5, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, 1, -4, 0}, -0.000590, {-0.020000, 0.000000, -0.040000, -0.040000, -0.230000}},
	{{3, 1, -3, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 1, -3, 0}, -0.006230, {-0.300000, 0.000000, -0.340000, -0.370000, -2.420000}},
	{{3, 1, -3, 2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 1, -2, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 1, -2, 0}, -0.041890, {-2.510000, 0.020000, -1.500000, -2.520000, -16.290000}},
	{{3, 1, -2, 2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{3, 1, -1, -2}, 0.001160, {0.020000, 0.050000, 0.020000, 0.070000, 0.450000}},
	{{3, 1, -1, 0}, 0.273770, {5.060000, -0.240000, 5.220000, 16.360000, 106.450000}},
	{{3, 1, -1, 2}, -0.001380, {-0.020000, -0.060000, -0.030000, -0.080000, -0.530000}},
	{{3, 1, 0, -2}, 0.008910, {0.140000, 0.400000, -0.010000, 0.530000, 3.460000}},
	{{3, 1, 0, 0}, 0.149900, {4.450000, -0.270000, 1.610000, 8.950000, 58.290000}},
	{{3, 1, 0, 2}, -0.000960, {-0.030000, -0.040000, -0.010000, -0.060000, -0.370000}},
	{{3, 1, 1, -2}, 0.000200, {0.000000, 0.000000, 0.000000, 0.010000, 0.080000}},
	{{3, 1, 1, 0}, 0.022440, {0.700000, -0.040000, 0.590000, 1.340000, 8.730000}},
	{{3, 1, 1, 2}, -0.000220, {0.000000, 0.000000, 0.000000, -0.010000, -0.090000}},
	{{3, 1, 2, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 1, 2, 0}, 0.002560, {0.080000, 0.000000, 0.110000, 0.150000, 1.000000}},
	{{3, 1, 2, 2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{3, 1, 3, 0}, 0.000260, {0.000000, 0.000000, 0.020000, 0.020000, 0.100000}},
	{{3, 1, 4, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 2, -4, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 2, -3, 0}, -0.000200, {0.000000, 0.000000, -0.010000, -0.020000, -0.080000}},
	{{3, 2, -2, 0}, -0.000360, {-0.060000, 0.000000, -0.010000, -0.040000, -0.140000}},
	{{3, 2, -1, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 2, -1, 0}, -0.001270, {0.070000, 0.000000, -0.030000, -0.150000, -0.490000}},
	{{3, 2, 0, -2}, -0.000090, {0.000000, 0.000000, 0.000000, -0.020000, -0.040000}},
	{{3, 2, 0, 0}, -0.001910, {-0.050000, 0.000000, -0.020000, -0.230000, -0.740000}},
	{{3, 2, 0, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 2, 1, 0}, -0.000390, {-0.010000, 0.000000, 0.000000, -0.050000, -0.150000}},
	{{3, 2, 2, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, 3, -3, 0}, 0.000000, {0.000000, 0.000000, 0.000000, -0.020000, -0.030000}},
	{{3, 3, -2, -2}, 0.000000, {-0.030000, 0.010000, -0.030000, 0.000000, -0.080000}},
	{{3, 3, -2, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{3, 3, -1, 0}, -0.000140, {0.000000, 0.000000, 0.000000, -0.020000, -0.050000}},
	{{3, 3, 0, -2}, -0.000020, {-0.090000, 0.070000, -0.080000, -0.080000, 0.050000}},
	{{3, 3, 0, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -5, -2, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -5, -1, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{4, -5, 0, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -4, -3, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -4, -2, 0}, 0.000330, {0.020000, 0.000000, 0.010000, 0.080000, 0.000000}},
	{{4, -4, -1, 0}, 0.000860, {0.050000, 0.000000, 0.020000, 0.210000, 0.000000}},
	{{4, -4, 0, 0}, 0.000540, {0.040000, 0.000000, 0.000000, 0.130000, 0.000000}},
	{{4, -4, 1, 0}, 0.000090, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{4, -4, 2, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, -3, 0}, 0.000270, {0.010000, 0.000000, 0.010000, 0.050000, 0.000000}},
	{{4, -3, -2, 0}, 0.007550, {0.330000, 0.000000, 0.280000, 1.350000, 0.000000}},
	{{4, -3, -2, 2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, -1, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, -1, 0}, 0.017410, {0.970000, -0.010000, 0.360000, 3.120000, 0.000000}},
	{{4, -3, -1, 2}, -0.000090, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{4, -3, 0, -2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{4, -3, 0, 0}, 0.009710, {0.650000, 0.000000, 0.110000, 1.740000, 0.000000}},
	{{4, -3, 0, 2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, 1, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, 1, 0}, 0.001600, {0.110000, 0.000000, 0.040000, 0.290000, 0.000000}},
	{{4, -3, 1, 2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, 2, 0}, 0.000200, {0.010000, 0.000000, 0.000000, 0.040000, 0.000000}},
	{{4, -3, 3, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -5, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -4, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -3, 0}, 0.005600, {0.270000, 0.000000, 0.310000, 0.670000, 0.000000}},
	{{4, -2, -3, 2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -2, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -2, 0}, 0.157810, {6.460000, -0.060000, 5.770000, 18.870000, 0.000000}},
	{{4, -2, -2, 2}, -0.000900, {-0.040000, -0.040000, -0.030000, -0.110000, 0.000000}},
	{{4, -2, -1, -2}, 0.001080, {0.000000, 0.050000, 0.020000, 0.130000, 0.000000}},
	{{4, -2, -1, 0}, 0.308740, {16.370000, -0.230000, 6.490000, 36.900000, 0.000000}},
	{{4, -2, -1, 2}, -0.001600, {-0.080000, -0.070000, -0.040000, -0.190000, 0.000000}},
	{{4, -2, 0, -2}, 0.000910, {0.070000, 0.040000, -0.020000, 0.110000, 0.000000}},
	{{4, -2, 0, 0}, 0.152270, {9.620000, -0.170000, 1.880000, 18.190000, 0.010000}},
	{{4, -2, 0, 2}, -0.000890, {-0.050000, -0.040000, -0.020000, -0.110000, 0.000000}},
	{{4, -2, 1, -2}, -0.000520, {-0.020000, -0.020000, -0.010000, -0.060000, 0.000000}},
	{{4, -2, 1, 0}, 0.024110, {1.580000, -0.030000, 0.660000, 2.880000, 0.000000}},
	{{4, -2, 1, 2}, -0.000210, {-0.010000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{4, -2, 2, -2}, -0.000090, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{4, -2, 2, 0}, 0.002900, {0.200000, 0.000000, 0.130000, 0.350000, 0.000000}},
	{{4, -2, 2, 2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, 3, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, 3, 0}, 0.000310, {0.020000, 0.000000, 0.020000, 0.040000, 0.000000}},
	{{4, -2, 4, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -6, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -5, 0}, -0.000340, {-0.010000, 0.000000, -0.030000, -0.020000, 0.000000}},
	{{4, -1, -4, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -4, 0}, -0.000200, {0.000000, 0.000000, -0.010000, -0.010000, 0.000000}},
	{{4, -1, -3, -2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -3, 0}, 0.099820, {4.580000, -0.040000, 5.460000, 5.980000, 0.020000}},
	{{4, -1, -3, 2}, -0.000780, {-0.030000, -0.040000, -0.040000, -0.050000, 0.000000}},
	{{4, -1, -2, -4}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -2, -2}, 0.000430, {0.000000, 0.020000, 0.020000, 0.030000, 0.000000}},
	{{4, -1, -2, 0}, 2.731980, {106.730000, -1.040000, 99.830000, 163.200000, 0.110000}},
	{{4, -1, -2, 2}, -0.015330, {-0.590000, -0.680000, -0.570000, -0.920000, 0.000000}},
	{{4, -1, -2, 4}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -1, -4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -1, -2}, 0.022890, {0.270000, 1.020000, 0.410000, 1.360000, 0.020000}},
	{{4, -1, -1, 0}, 4.374160, {221.000000, -3.710000, 93.870000, 260.990000, -0.050000}},
	{{4, -1, -1, 2}, -0.022730, {-1.120000, -1.000000, -0.560000, -1.360000, 0.000000}},
	{{4, -1, -1, 4}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 0, -4}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 0, -2}, 0.006660, {0.750000, 0.260000, -0.270000, 0.390000, 0.020000}},
	{{4, -1, 0, 0}, 1.870830, {111.800000, -2.350000, 25.650000, 111.490000, 0.160000}},
	{{4, -1, 0, 2}, -0.011140, {-0.640000, -0.490000, -0.230000, -0.660000, 0.000000}},
	{{4, -1, 0, 4}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 1, -4}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 1, -2}, -0.007320, {-0.280000, -0.330000, -0.180000, -0.440000, 0.000000}},
	{{4, -1, 1, 0}, 0.282510, {17.390000, -0.420000, 8.050000, 16.820000, 0.020000}},
	{{4, -1, 1, 2}, -0.002510, {-0.150000, -0.110000, -0.080000, -0.150000, 0.000000}},
	{{4, -1, 1, 4}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 2, -2}, -0.001210, {-0.050000, -0.050000, -0.050000, -0.070000, 0.000000}},
	{{4, -1, 2, 0}, 0.032740, {2.050000, -0.060000, 1.490000, 1.950000, 0.000000}},
	{{4, -1, 2, 2}, -0.000410, {-0.030000, -0.020000, -0.020000, -0.020000, 0.000000}},
	{{4, -1, 3, -2}, -0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 3, 0}, 0.003360, {0.210000, 0.000000, 0.210000, 0.200000, 0.000000}},
	{{4, -1, 3, 2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 4, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 4, 0}, 0.000320, {0.020000, 0.000000, 0.030000, 0.020000, 0.000000}},
	{{4, -1, 5, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -7, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -6, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -6, 0}, -0.000620, {-0.030000, 0.000000, -0.070000, 0.000000, 0.000000}},
	{{4, 0, -5, -2}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -5, 0}, -0.004310, {-0.180000, 0.000000, -0.390000, 0.000000, 0.000000}},
	{{4, 0, -4, -2}, 0.000420, {0.020000, 0.020000, 0.030000, 0.000000, 0.000000}},
	{{4, 0, -4, 0}, -0.000700, {0.030000, 0.000000, -0.050000, 0.000000, 0.000000}},
	{{4, 0, -4, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -3, -4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -3, -2}, 0.000230, {0.020000, 0.010000, 0.010000, 0.000000, 0.000000}},
	{{4, 0, -3, 0}, 1.186820, {52.890000, -0.500000, 64.880000, 0.220000, 0.250000}},
	{{4, 0, -3, 2}, -0.009340, {-0.290000, -0.420000, -0.510000, 0.000000, 0.000000}},
	{{4, 0, -3, 4}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -2, -4}, 0.000130, {0.000000, 0.010000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -2, -2}, 0.006620, {0.090000, 0.290000, 0.240000, 0.000000, 0.000000}},
	{{4, 0, -2, 0}, 30.772470, {1164.740000, -12.420000, 1123.350000, -4.250000, 1.180000}},
	{{4, 0, -2, 2}, -0.169770, {-6.360000, -7.540000, -6.260000, 0.030000, 0.000000}},
	{{4, 0, -2, 4}, 0.000610, {0.020000, 0.050000, 0.020000, 0.000000, 0.000000}},
	{{4, 0, -1, -4}, 0.000170, {0.010000, 0.010000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -1, -2}, 0.300860, {5.060000, 13.370000, 5.390000, -0.190000, 0.150000}},
	{{4, 0, -1, 0}, 38.429740, {1853.280000, -37.860000, 849.770000, -9.840000, 0.080000}},
	{{4, 0, -1, 2}, -0.201340, {-9.450000, -8.850000, -5.160000, 0.050000, 0.000000}},
	{{4, 0, -1, 4}, 0.000720, {0.030000, 0.060000, 0.020000, 0.000000, 0.000000}},
	{{4, 0, 0, -4}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 0, -2}, -0.023910, {4.230000, -1.470000, -2.510000, -0.040000, 0.160000}},
	{{4, 0, 0, 0}, 13.899030, {782.370000, -19.880000, 215.740000, -5.220000, 1.370000}},
	{{4, 0, 0, 2}, -0.085000, {-4.560000, -3.710000, -1.970000, 0.030000, 0.000000}},
	{{4, 0, 0, 4}, 0.000350, {0.020000, 0.030000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 1, -4}, -0.000610, {0.000000, -0.050000, -0.010000, 0.000000, 0.000000}},
	{{4, 0, 1, -2}, -0.066010, {-2.120000, -2.980000, -1.620000, 0.030000, 0.020000}},
	{{4, 0, 1, 0}, 1.977720, {113.340000, -3.310000, 59.210000, -0.940000, 0.180000}},
	{{4, 0, 1, 2}, -0.017810, {-0.990000, -0.770000, -0.630000, 0.000000, 0.000000}},
	{{4, 0, 1, 4}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 2, -4}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 2, -2}, -0.009800, {-0.350000, -0.440000, -0.410000, 0.000000, 0.000000}},
	{{4, 0, 2, 0}, 0.218530, {12.570000, -0.410000, 10.190000, -0.130000, 0.020000}},
	{{4, 0, 2, 2}, -0.002740, {-0.150000, -0.120000, -0.140000, 0.000000, 0.000000}},
	{{4, 0, 2, 4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 3, -2}, -0.001090, {-0.040000, -0.050000, -0.070000, 0.000000, 0.000000}},
	{{4, 0, 3, 0}, 0.021490, {1.240000, -0.040000, 1.380000, -0.020000, 0.000000}},
	{{4, 0, 3, 2}, -0.000360, {-0.020000, -0.020000, -0.020000, 0.000000, 0.000000}},
	{{4, 0, 4, -2}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 4, 0}, 0.001980, {0.110000, 0.000000, 0.160000, 0.000000, 0.000000}},
	{{4, 0, 4, 2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 5, 0}, 0.000170, {0.010000, 0.000000, 0.020000, 0.000000, 0.000000}},
	{{4, 0, 6, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -6, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -5, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -4, 0}, 0.000640, {0.060000, 0.000000, 0.050000, 0.040000, 0.000000}},
	{{4, 1, -3, -2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -3, 0}, 0.026820, {3.500000, -0.050000, 1.450000, 1.600000, 0.020000}},
	{{4, 1, -3, 2}, -0.000030, {-0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -2, -2}, -0.000390, {-0.020000, -0.020000, -0.010000, -0.020000, 0.000000}},
	{{4, 1, -2, 0}, -0.357590, {7.750000, -0.180000, -13.130000, -21.490000, -0.040000}},
	{{4, 1, -2, 2}, 0.002200, {-0.030000, 0.100000, 0.080000, 0.130000, 0.000000}},
	{{4, 1, -1, -2}, -0.018750, {-0.680000, -0.830000, -0.350000, -1.120000, 0.000000}},
	{{4, 1, -1, 0}, -0.635230, {-17.620000, 0.740000, -14.320000, -38.110000, -0.210000}},
	{{4, 1, -1, 2}, 0.003530, {0.090000, 0.150000, 0.090000, 0.210000, 0.000000}},
	{{4, 1, -1, 4}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 0, -2}, 0.000510, {-0.280000, 0.040000, 0.020000, 0.030000, -0.010000}},
	{{4, 1, 0, 0}, -0.289110, {-15.100000, 0.590000, -4.730000, -17.300000, -0.060000}},
	{{4, 1, 0, 2}, 0.001860, {0.090000, 0.080000, 0.050000, 0.110000, 0.000000}},
	{{4, 1, 1, -4}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 1, -2}, 0.001830, {0.030000, 0.080000, 0.040000, 0.110000, 0.000000}},
	{{4, 1, 1, 0}, -0.050700, {-2.990000, 0.120000, -1.550000, -3.030000, -0.010000}},
	{{4, 1, 1, 2}, 0.000480, {0.030000, 0.020000, 0.020000, 0.030000, 0.000000}},
	{{4, 1, 2, -2}, 0.000320, {0.010000, 0.010000, 0.010000, 0.020000, 0.000000}},
	{{4, 1, 2, 0}, -0.006680, {-0.420000, 0.020000, -0.320000, -0.400000, 0.000000}},
	{{4, 1, 2, 2}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 3, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 3, 0}, -0.000770, {-0.050000, 0.000000, -0.050000, -0.050000, 0.000000}},
	{{4, 1, 3, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 4, 0}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -4, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -3, 0}, -0.003690, {-0.240000, 0.000000, -0.200000, -0.440000, 0.000000}},
	{{4, 2, -3, 2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -2, -2}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -2, 0}, -0.035140, {-2.040000, 0.020000, -1.290000, -4.210000, 0.000000}},
	{{4, 2, -2, 2}, 0.000190, {0.010000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{4, 2, -1, -2}, 0.000900, {0.060000, 0.040000, 0.020000, 0.110000, 0.000000}},
	{{4, 2, -1, 0}, -0.021710, {-1.510000, 0.030000, -0.570000, -2.600000, 0.010000}},
	{{4, 2, -1, 2}, 0.000120, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{4, 2, 0, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, 0, 0}, -0.002080, {-0.160000, 0.000000, -0.120000, -0.250000, 0.000000}},
	{{4, 2, 0, 2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, 1, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, 1, 0}, -0.000080, {0.000000, 0.000000, -0.010000, -0.010000, 0.000000}},
	{{4, 2, 2, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 3, -2, 0}, 0.000200, {0.000000, 0.000000, 0.000000, 0.040000, 0.000000}},
	{{4, 3, -1, 0}, 0.000230, {0.010000, 0.000000, 0.000000, 0.040000, 0.000000}},
	{{4, 3, 0, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 4, -2, -2}, 0.000000, {-0.020000, 0.000000, 0.000000, 0.040000, -0.060000}},
	{{5, -3, -2, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{5, -3, 0, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -2, -3, 0}, -0.000130, {0.000000, 0.000000, 0.000000, -0.020000, -0.050000}},
	{{5, -2, -2, 0}, -0.000520, {-0.030000, 0.000000, -0.020000, -0.060000, -0.200000}},
	{{5, -2, -1, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{5, -2, -1, 0}, -0.000110, {-0.020000, 0.000000, 0.000000, -0.010000, -0.040000}},
	{{5, -2, 0, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -2, 0, 0}, 0.000120, {0.000000, 0.000000, 0.000000, 0.010000, 0.050000}},
	{{5, -2, 1, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -1, -4, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, -1, -3, 0}, -0.001890, {-0.080000, 0.000000, -0.100000, -0.110000, -0.730000}},
	{{5, -1, -3, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -1, -2, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{5, -1, -2, 0}, -0.006930, {-0.400000, 0.000000, -0.260000, -0.410000, -2.700000}},
	{{5, -1, -2, 2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{5, -1, -1, -2}, -0.000400, {-0.020000, -0.020000, 0.000000, -0.020000, -0.160000}},
	{{5, -1, -1, 0}, -0.001540, {-0.210000, 0.010000, -0.080000, -0.090000, -0.600000}},
	{{5, -1, -1, 2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{5, -1, 0, -2}, -0.000210, {0.000000, 0.000000, 0.000000, -0.010000, -0.080000}},
	{{5, -1, 0, 0}, 0.001020, {0.010000, 0.000000, -0.020000, 0.060000, 0.400000}},
	{{5, -1, 1, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{5, -1, 1, 0}, 0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.050000}},
	{{5, 0, -4, 0}, -0.000590, {-0.030000, 0.000000, -0.040000, 0.000000, -0.230000}},
	{{5, 0, -3, 0}, -0.016860, {-0.720000, 0.000000, -0.920000, 0.000000, -6.560000}},
	{{5, 0, -3, 2}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{5, 0, -2, -2}, -0.000310, {0.000000, -0.010000, -0.010000, 0.000000, -0.120000}},
	{{5, 0, -2, 0}, -0.057250, {-3.180000, 0.070000, -2.180000, 0.050000, -22.260000}},
	{{5, 0, -2, 2}, 0.000390, {0.020000, 0.020000, 0.020000, 0.000000, 0.150000}},
	{{5, 0, -1, -2}, -0.003350, {-0.110000, -0.150000, -0.060000, 0.000000, -1.300000}},
	{{5, 0, -1, 0}, -0.013500, {-1.570000, 0.090000, -0.640000, 0.060000, -5.250000}},
	{{5, 0, -1, 2}, 0.000240, {0.020000, 0.010000, 0.000000, 0.000000, 0.090000}},
	{{5, 0, 0, -4}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 0, 0, -2}, -0.001370, {-0.060000, -0.060000, 0.000000, 0.000000, -0.530000}},
	{{5, 0, 0, 0}, 0.004390, {-0.050000, 0.030000, -0.180000, 0.030000, 1.710000}},
	{{5, 0, 0, 2}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{5, 0, 1, -2}, -0.000190, {0.000000, 0.000000, 0.000000, 0.000000, -0.070000}},
	{{5, 0, 1, 0}, 0.000500, {-0.030000, 0.000000, -0.020000, 0.000000, 0.200000}},
	{{5, 0, 1, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 0, 2, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 0, 2, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 1, -4, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, 1, -3, 0}, -0.000460, {-0.050000, 0.000000, -0.020000, -0.030000, -0.180000}},
	{{5, 1, -2, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 1, -2, 0}, 0.003920, {0.120000, 0.000000, 0.150000, 0.230000, 1.520000}},
	{{5, 1, -2, 2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{5, 1, -1, -2}, 0.000160, {0.000000, 0.000000, 0.000000, 0.000000, 0.060000}},
	{{5, 1, -1, 0}, 0.004750, {0.240000, -0.010000, 0.120000, 0.280000, 1.850000}},
	{{5, 1, -1, 2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, 1, 0, -2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{5, 1, 0, 0}, 0.001810, {0.110000, 0.000000, 0.050000, 0.110000, 0.700000}},
	{{5, 1, 0, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 1, 1, 0}, 0.000350, {0.020000, 0.000000, 0.010000, 0.020000, 0.140000}},
	{{5, 1, 2, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{5, 2, -3, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 2, -2, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 2, -1, 0}, -0.000080, {0.000000, 0.000000, 0.000000, -0.010000, -0.030000}},
	{{5, 2, 0, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, 2, 1, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 3, -2, -2}, 0.000000, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{6, -4, -3, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -4, -2, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -4, -1, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -4, 0, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -3, -3, 0}, 0.000200, {0.010000, 0.000000, 0.010000, 0.030000, 0.000000}},
	{{6, -3, -2, 0}, 0.000590, {0.040000, 0.000000, 0.020000, 0.110000, 0.000000}},
	{{6, -3, -1, 0}, 0.000590, {0.050000, 0.000000, 0.020000, 0.110000, 0.000000}},
	{{6, -3, 0, 0}, 0.000240, {0.020000, 0.000000, 0.000000, 0.040000, 0.000000}},
	{{6, -3, 1, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -4, 0}, 0.000140, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{6, -2, -3, 0}, 0.003120, {0.190000, 0.000000, 0.170000, 0.370000, 0.000000}},
	{{6, -2, -3, 2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -2, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -2, 0}, 0.008360, {0.590000, 0.000000, 0.330000, 1.000000, 0.000000}},
	{{6, -2, -2, 2}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -1, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -1, 0}, 0.007440, {0.610000, -0.010000, 0.210000, 0.890000, 0.000000}},
	{{6, -2, -1, 2}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, 0, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, 0, 0}, 0.002850, {0.260000, 0.000000, 0.070000, 0.340000, 0.000000}},
	{{6, -2, 0, 2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, 1, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, 1, 0}, 0.000560, {0.050000, 0.000000, 0.020000, 0.070000, 0.000000}},
	{{6, -2, 2, 0}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, 3, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -5, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -4, 0}, 0.001760, {0.110000, 0.000000, 0.130000, 0.110000, 0.000000}},
	{{6, -1, -4, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -3, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -3, 0}, 0.039000, {2.280000, -0.020000, 2.140000, 2.330000, 0.000000}},
	{{6, -1, -3, 2}, -0.000390, {-0.020000, -0.020000, -0.020000, -0.020000, 0.000000}},
	{{6, -1, -2, -2}, 0.000440, {0.010000, 0.020000, 0.020000, 0.030000, 0.000000}},
	{{6, -1, -2, 0}, 0.090330, {6.230000, -0.110000, 3.590000, 5.380000, 0.000000}},
	{{6, -1, -2, 2}, -0.000800, {-0.050000, -0.040000, -0.030000, -0.050000, 0.000000}},
	{{6, -1, -1, -2}, 0.000180, {0.020000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{6, -1, -1, 0}, 0.071420, {5.640000, -0.120000, 2.050000, 4.250000, 0.000000}},
	{{6, -1, -1, 2}, -0.000640, {-0.050000, -0.030000, -0.020000, -0.040000, 0.000000}},
	{{6, -1, 0, -2}, -0.000250, {0.000000, -0.010000, -0.010000, -0.010000, 0.000000}},
	{{6, -1, 0, 0}, 0.025100, {2.160000, -0.050000, 0.660000, 1.490000, 0.000000}},
	{{6, -1, 0, 2}, -0.000260, {-0.020000, -0.010000, 0.000000, -0.020000, 0.000000}},
	{{6, -1, 1, -2}, -0.000160, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, 1, 0}, 0.004770, {0.420000, -0.010000, 0.180000, 0.280000, 0.000000}},
	{{6, -1, 1, 2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, 2, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, 2, 0}, 0.000690, {0.060000, 0.000000, 0.040000, 0.040000, 0.000000}},
	{{6, -1, 2, 2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, 3, 0}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -5, 0}, 0.000200, {0.010000, 0.000000, 0.020000, 0.000000, 0.000000}},
	{{6, 0, -4, 0}, 0.013760, {0.880000, 0.000000, 1.000000, 0.000000, 0.000000}},
	{{6, 0, -4, 2}, -0.000170, {0.000000, 0.000000, -0.010000, 0.000000, 0.000000}},
	{{6, 0, -3, -2}, 0.000180, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -3, 0}, 0.292550, {16.720000, -0.190000, 16.040000, -0.080000, 0.030000}},
	{{6, 0, -3, 2}, -0.002870, {-0.160000, -0.130000, -0.160000, 0.000000, 0.000000}},
	{{6, 0, -3, 4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -2, -2}, 0.003640, {0.130000, 0.160000, 0.130000, 0.000000, 0.000000}},
	{{6, 0, -2, 0}, 0.571560, {38.340000, -0.750000, 22.970000, -0.260000, 0.020000}},
	{{6, 0, -2, 2}, -0.005090, {-0.340000, -0.220000, -0.220000, 0.000000, 0.000000}},
	{{6, 0, -2, 4}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -1, -2}, 0.000070, {0.080000, 0.000000, -0.030000, 0.000000, 0.000000}},
	{{6, 0, -1, 0}, 0.394510, {30.100000, -0.740000, 11.790000, -0.250000, 0.030000}},
	{{6, 0, -1, 2}, -0.003590, {-0.270000, -0.160000, -0.130000, 0.000000, 0.000000}},
	{{6, 0, -1, 4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 0, -4}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 0, -2}, -0.002140, {-0.070000, -0.100000, -0.070000, 0.000000, 0.000000}},
	{{6, 0, 0, 0}, 0.126160, {10.430000, -0.290000, 3.570000, -0.100000, 0.020000}},
	{{6, 0, 0, 2}, -0.001320, {-0.110000, -0.060000, -0.050000, 0.000000, 0.000000}},
	{{6, 0, 1, -4}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 1, -2}, -0.000940, {-0.050000, -0.040000, -0.030000, 0.000000, 0.000000}},
	{{6, 0, 1, 0}, 0.022850, {1.930000, -0.060000, 0.910000, -0.020000, 0.000000}},
	{{6, 0, 1, 2}, -0.000310, {-0.020000, -0.010000, -0.010000, 0.000000, 0.000000}},
	{{6, 0, 2, -2}, -0.000180, {-0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 2, 0}, 0.003180, {0.270000, 0.000000, 0.180000, 0.000000, 0.000000}},
	{{6, 0, 2, 2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 3, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 3, 0}, 0.000380, {0.030000, 0.000000, 0.030000, 0.000000, 0.000000}},
	{{6, 0, 4, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -5, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -4, 0}, 0.000280, {0.050000, 0.000000, 0.020000, 0.020000, 0.000000}},
	{{6, 1, -3, 0}, -0.004780, {0.050000, 0.000000, -0.260000, -0.290000, 0.000000}},
	{{6, 1, -3, 2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -2, -2}, -0.000260, {-0.010000, -0.010000, 0.000000, -0.020000, 0.000000}},
	{{6, 1, -2, 0}, -0.012820, {-0.470000, 0.020000, -0.520000, -0.770000, 0.000000}},
	{{6, 1, -2, 2}, 0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -1, -2}, -0.000070, {-0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -1, 0}, -0.010640, {-0.660000, 0.020000, -0.320000, -0.640000, 0.000000}},
	{{6, 1, -1, 2}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 0, -2}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 0, 0}, -0.003960, {-0.310000, 0.010000, -0.120000, -0.240000, 0.000000}},
	{{6, 1, 0, 2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 1, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 1, 0}, -0.000830, {-0.070000, 0.000000, -0.030000, -0.050000, 0.000000}},
	{{6, 1, 1, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 2, 0}, -0.000130, {-0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 3, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -4, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -3, 0}, -0.000510, {-0.040000, 0.000000, -0.030000, -0.060000, 0.000000}},
	{{6, 2, -2, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -2, 0}, -0.000610, {-0.060000, 0.000000, -0.030000, -0.070000, 0.000000}},
	{{6, 2, -1, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -1, 0}, -0.000210, {-0.020000, 0.000000, 0.000000, -0.030000, 0.000000}},
	{{6, 2, 0, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 3, -2, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, -2, -3, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, -2, -2, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, -1, -4, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{7, -1, -3, 0}, -0.000160, {-0.010000, 0.000000, 0.000000, -0.010000, -0.060000}},
	{{7, -1, -2, 0}, -0.000120, {-0.010000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{7, -1, 0, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -5, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -4, 0}, -0.000250, {-0.020000, 0.000000, -0.020000, 0.000000, -0.100000}},
	{{7, 0, -3, 0}, -0.000980, {-0.070000, 0.000000, -0.060000, 0.000000, -0.380000}},
	{{7, 0, -3, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -2, -2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{7, 0, -2, 0}, -0.000680, {-0.070000, 0.000000, -0.030000, 0.000000, -0.260000}},
	{{7, 0, -2, 2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -1, -2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{7, 0, -1, 0}, -0.000050, {-0.020000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{7, 0, 0, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, 0, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{7, 1, -3, 0}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{7, 1, -2, 0}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{7, 1, -1, 0}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{7, 1, 0, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{8, -3, -3, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -3, -2, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -3, -1, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -4, 0}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -3, 0}, 0.000190, {0.020000, 0.000000, 0.010000, 0.020000, 0.000000}},
	{{8, -2, -2, 0}, 0.000250, {0.030000, 0.000000, 0.010000, 0.030000, 0.000000}},
	{{8, -2, -1, 0}, 0.000150, {0.020000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{8, -2, 0, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, 1, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -5, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -4, 0}, 0.000580, {0.040000, 0.000000, 0.040000, 0.030000, 0.000000}},
	{{8, -1, -3, 0}, 0.001700, {0.150000, 0.000000, 0.100000, 0.100000, 0.000000}},
	{{8, -1, -3, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -2, 0}, 0.001960, {0.190000, 0.000000, 0.090000, 0.120000, 0.000000}},
	{{8, -1, -2, 2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -1, 0}, 0.001120, {0.120000, 0.000000, 0.040000, 0.070000, 0.000000}},
	{{8, -1, -1, 2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, 0, 0}, 0.000370, {0.040000, 0.000000, 0.020000, 0.020000, 0.000000}},
	{{8, -1, 1, 0}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, 2, 0}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -5, 0}, 0.000180, {0.010000, 0.000000, 0.020000, 0.000000, 0.000000}},
	{{8, 0, -4, 0}, 0.003240, {0.250000, 0.000000, 0.240000, 0.000000, 0.000000}},
	{{8, 0, -4, 2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -3, -2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -3, 0}, 0.008400, {0.720000, -0.010000, 0.490000, 0.000000, 0.000000}},
	{{8, 0, -3, 2}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -2, -2}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -2, 0}, 0.008660, {0.830000, -0.020000, 0.400000, 0.000000, 0.000000}},
	{{8, 0, -2, 2}, -0.000110, {-0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -1, -2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -1, 0}, 0.004550, {0.470000, -0.010000, 0.180000, 0.000000, 0.000000}},
	{{8, 0, -1, 2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, 0, -2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, 0, 0}, 0.001390, {0.150000, 0.000000, 0.060000, 0.000000, 0.000000}},
	{{8, 0, 0, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, 1, -2}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, 1, 0}, 0.000290, {0.030000, 0.000000, 0.020000, 0.000000, 0.000000}},
	{{8, 0, 2, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -4, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -3, 0}, -0.000240, {-0.010000, 0.000000, -0.010000, -0.010000, 0.000000}},
	{{8, 1, -2, 0}, -0.000290, {-0.020000, 0.000000, -0.010000, -0.020000, 0.000000}},
	{{8, 1, -1, 0}, -0.000170, {-0.020000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{8, 1, 0, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, 1, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 2, -3, 0}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{9, 0, -4, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{9, 0, -3, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, -1, -4, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, -1, -3, 0}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, -1, -2, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, -1, -1, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -5, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -4, 0}, 0.000120, {0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -3, 0}, 0.000170, {0.020000, 0.000000, 0.010000, 0.000000, 0.000000}},
	{{10, 0, -2, 0}, 0.000130, {0.020000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -1, 0}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, 0, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}}
};

static const struct main_problem main_elp2[ELP2_SIZE] =
{
	{{0, 0, 0, 1}, 18461.400000, {0.000000, 412529.610000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 0, 3}, -6.296640, {7.680000, -422.650000, -13.210000, 0.020000, -0.020000}},
	{{0, 0, 0, 5}, 0.005920, {0.000000, 0.660000, 0.020000, 0.000000, 0.000000}},
	{{0, 0, 1, -5}, -0.002360, {0.020000, -0.260000, -0.050000, 0.000000, 0.000000}},
	{{0, 0, 1, -3}, 2.798710, {-17.130000, 188.680000, 50.780000, -0.110000, 0.040000}},
	{{0, 0, 1, -1}, 999.700790, {-563.820000, 22508.540000, 18298.820000, -0.920000, -0.210000}},
	{{0, 0, 1, 1}, 1010.174300, {-93.160000, 22571.830000, 18386.360000, -0.760000, -0.170000}},
	{{0, 0, 1, 3}, -1.019410, {0.950000, -68.420000, -19.880000, 0.000000, 0.000000}},
	{{0, 0, 1, 5}, 0.001560, {0.000000, 0.170000, 0.030000, 0.000000, 0.000000}},
	{{0, 0, 2, -5}, -0.000440, {0.000000, -0.050000, -0.020000, 0.000000, 0.000000}},
	{{0, 0, 2, -3}, -0.130350, {-0.400000, -8.780000, -4.770000, 0.000000, 0.000000}},
	{{0, 0, 2, -1}, 31.759850, {-167.640000, 723.950000, 1159.970000, -1.090000, 0.220000}},
	{{0, 0, 2, 1}, 61.912290, {-23.730000, 1382.400000, 2253.180000, -0.620000, -0.060000}},
	{{0, 0, 2, 3}, -0.117870, {0.110000, -7.910000, -4.410000, 0.000000, 0.000000}},
	{{0, 0, 2, 5}, 0.000260, {0.000000, 0.030000, 0.010000, 0.000000, 0.000000}},
	{{0, 0, 3, -5}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 3, -3}, -0.001460, {0.040000, -0.100000, -0.080000, 0.000000, 0.000000}},
	{{0, 0, 3, -1}, 1.581310, {-10.250000, 36.060000, 86.530000, -0.080000, 0.000000}},
	{{0, 0, 3, 1}, 3.984070, {-3.300000, 88.860000, 217.430000, -0.110000, -0.010000}},
	{{0, 0, 3, 3}, -0.011810, {0.010000, -0.790000, -0.650000, 0.000000, 0.000000}},
	{{0, 0, 3, 5}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 4, -3}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 4, -1}, 0.091570, {-0.700000, 2.090000, 6.680000, 0.000000, 0.000000}},
	{{0, 0, 4, 1}, 0.263250, {-0.370000, 5.860000, 19.150000, -0.010000, 0.000000}},
	{{0, 0, 4, 3}, -0.001090, {0.000000, -0.070000, -0.080000, 0.000000, 0.000000}},
	{{0, 0, 5, -1}, 0.005660, {-0.050000, 0.130000, 0.510000, 0.000000, 0.000000}},
	{{0, 0, 5, 1}, 0.017680, {-0.040000, 0.390000, 1.610000, 0.000000, 0.000000}},
	{{0, 0, 5, 3}, -0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 6, -1}, 0.000360, {0.000000, 0.000000, 0.040000, 0.000000, 0.000000}},
	{{0, 0, 6, 1}, 0.001200, {0.000000, 0.030000, 0.130000, 0.000000, 0.000000}},
	{{0, 0, 7, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 7, 1}, 0.000080, {0.000000, 0.000000, 0.010000, 0.000000, 0.000000}},
	{{0, 1, -6, -1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -6, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -5, -1}, -0.000530, {-0.010000, -0.010000, -0.050000, -0.030000, 0.000000}},
	{{0, 1, -5, 1}, -0.000140, {0.000000, 0.000000, -0.010000, 0.000000, 0.000000}},
	{{0, 1, -4, -3}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -4, -1}, -0.006460, {-0.140000, -0.140000, -0.470000, -0.390000, 0.000000}},
	{{0, 1, -4, 1}, -0.001860, {-0.020000, -0.040000, -0.140000, -0.110000, 0.000000}},
	{{0, 1, -3, -3}, 0.000230, {0.000000, 0.010000, 0.010000, 0.010000, 0.000000}},
	{{0, 1, -3, -1}, -0.074790, {-1.690000, -1.630000, -4.080000, -4.480000, 0.000000}},
	{{0, 1, -3, 1}, -0.023650, {-0.350000, -0.530000, -1.290000, -1.420000, 0.000000}},
	{{0, 1, -3, 3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -2, -3}, 0.001540, {0.030000, 0.100000, 0.060000, 0.090000, 0.000000}},
	{{0, 1, -2, -1}, -0.793220, {-18.590000, -17.320000, -28.860000, -47.520000, -0.010000}},
	{{0, 1, -2, 1}, -0.301290, {-4.790000, -6.710000, -11.010000, -18.040000, -0.010000}},
	{{0, 1, -2, 3}, 0.001910, {0.060000, 0.130000, 0.070000, 0.120000, 0.000000}},
	{{0, 1, -1, -5}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -1, -3}, 0.006790, {0.140000, 0.450000, 0.140000, 0.410000, 0.000000}},
	{{0, 1, -1, -1}, -6.731730, {-159.360000, -146.940000, -122.890000, -403.360000, -0.030000}},
	{{0, 1, -1, 1}, -5.632600, {-148.810000, -123.960000, -102.800000, -337.620000, -0.040000}},
	{{0, 1, -1, 3}, -0.009110, {-0.200000, -0.610000, -0.150000, -0.550000, 0.000000}},
	{{0, 1, 0, -3}, 0.000100, {-0.170000, 0.000000, 0.060000, 0.000000, 0.000000}},
	{{0, 1, 0, -1}, -4.839830, {70.860000, -100.970000, -8.740000, -289.050000, 0.020000}},
	{{0, 1, 0, 1}, -6.460360, {28.140000, -136.250000, -20.210000, -386.090000, 0.290000}},
	{{0, 1, 0, 3}, 0.011570, {0.060000, 0.770000, 0.040000, 0.690000, 0.000000}},
	{{0, 1, 0, 5}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 1, -3}, -0.007030, {-0.040000, -0.470000, -0.130000, -0.420000, 0.000000}},
	{{0, 1, 1, -1}, -5.076140, {-113.160000, -112.230000, -93.120000, -304.280000, 0.000000}},
	{{0, 1, 1, 1}, -5.311510, {-99.030000, -115.960000, -99.190000, -318.300000, 0.060000}},
	{{0, 1, 1, 3}, 0.006680, {0.100000, 0.450000, 0.130000, 0.400000, 0.000000}},
	{{0, 1, 1, 5}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 2, -3}, 0.001410, {0.040000, 0.090000, 0.050000, 0.080000, 0.000000}},
	{{0, 1, 2, -1}, -0.312920, {-4.920000, -6.960000, -11.470000, -18.750000, 0.000000}},
	{{0, 1, 2, 1}, -0.638840, {-12.560000, -13.950000, -23.510000, -38.290000, 0.010000}},
	{{0, 1, 2, 3}, 0.001350, {0.020000, 0.090000, 0.050000, 0.080000, 0.000000}},
	{{0, 1, 3, -3}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 3, -1}, -0.024190, {-0.360000, -0.540000, -1.330000, -1.450000, 0.000000}},
	{{0, 1, 3, 1}, -0.061760, {-1.230000, -1.350000, -3.400000, -3.700000, 0.000000}},
	{{0, 1, 3, 3}, 0.000200, {0.000000, 0.010000, 0.010000, 0.010000, 0.000000}},
	{{0, 1, 4, -1}, -0.001910, {-0.030000, -0.040000, -0.140000, -0.110000, 0.000000}},
	{{0, 1, 4, 1}, -0.005480, {-0.110000, -0.120000, -0.400000, -0.330000, 0.000000}},
	{{0, 1, 4, 3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 5, -1}, -0.000150, {0.000000, 0.000000, -0.010000, 0.000000, 0.000000}},
	{{0, 1, 5, 1}, -0.000460, {0.000000, -0.010000, -0.040000, -0.030000, 0.000000}},
	{{0, 1, 6, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 6, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -5, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -4, -1}, -0.000160, {0.000000, 0.000000, -0.010000, -0.020000, 0.000000}},
	{{0, 2, -4, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -3, -1}, -0.001690, {-0.050000, -0.040000, -0.090000, -0.200000, 0.000000}},
	{{0, 2, -3, 1}, -0.000490, {-0.010000, -0.010000, -0.030000, -0.060000, 0.000000}},
	{{0, 2, -2, -3}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -2, -1}, -0.015710, {-0.490000, -0.340000, -0.570000, -1.880000, 0.000000}},
	{{0, 2, -2, 1}, -0.005360, {-0.120000, -0.120000, -0.200000, -0.640000, 0.000000}},
	{{0, 2, -2, 3}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -1, -3}, 0.000100, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{0, 2, -1, -1}, -0.113350, {-3.250000, -2.460000, -2.070000, -13.580000, 0.000000}},
	{{0, 2, -1, 1}, -0.095110, {-3.170000, -2.080000, -1.730000, -11.390000, 0.000000}},
	{{0, 2, -1, 3}, -0.000090, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{0, 2, 0, -3}, -0.000140, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{0, 2, 0, -1}, -0.018010, {3.130000, -0.330000, -0.140000, -2.140000, 0.000000}},
	{{0, 2, 0, 1}, -0.057290, {2.090000, -1.190000, -0.390000, -6.840000, 0.000000}},
	{{0, 2, 0, 3}, 0.000160, {0.000000, 0.010000, 0.000000, 0.020000, 0.000000}},
	{{0, 2, 1, -3}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, 1, -1}, -0.061870, {-1.290000, -1.370000, -1.130000, -7.410000, 0.000000}},
	{{0, 2, 1, 1}, -0.055040, {-0.810000, -1.210000, -1.040000, -6.590000, 0.000000}},
	{{0, 2, 1, 3}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, 2, -3}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, 2, -1}, -0.002840, {-0.020000, -0.060000, -0.100000, -0.340000, 0.000000}},
	{{0, 2, 2, 1}, -0.005280, {-0.060000, -0.120000, -0.190000, -0.630000, 0.000000}},
	{{0, 2, 2, 3}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, 3, -1}, -0.000160, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{0, 2, 3, 1}, -0.000370, {0.000000, 0.000000, -0.020000, -0.040000, 0.000000}},
	{{0, 2, 4, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, -3, -1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, -3, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, -2, -1}, -0.000330, {-0.010000, 0.000000, -0.010000, -0.060000, 0.000000}},
	{{0, 3, -2, 1}, -0.000110, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{0, 3, -1, -1}, -0.002160, {-0.070000, -0.050000, -0.040000, -0.390000, 0.000000}},
	{{0, 3, -1, 1}, -0.001900, {-0.070000, -0.040000, -0.030000, -0.340000, 0.000000}},
	{{0, 3, 0, -1}, 0.000660, {0.110000, 0.020000, 0.000000, 0.120000, 0.000000}},
	{{0, 3, 0, 1}, -0.000310, {0.080000, 0.000000, 0.000000, -0.060000, 0.000000}},
	{{0, 3, 1, -1}, -0.001090, {-0.020000, -0.020000, -0.020000, -0.190000, 0.000000}},
	{{0, 3, 1, 1}, -0.000830, {-0.010000, -0.020000, -0.020000, -0.150000, 0.000000}},
	{{0, 3, 2, -1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, 2, 1}, -0.000080, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{0, 4, -1, -1}, -0.000040, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{0, 4, -1, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 4, 0, -1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 4, 0, 1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 4, 1, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 4, 1, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -3, -1, -1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{1, -3, 0, -1}, 0.000110, {0.000000, 0.000000, 0.000000, 0.020000, 0.040000}},
	{{1, -3, 0, 1}, 0.000120, {0.000000, 0.000000, 0.000000, 0.020000, 0.050000}},
	{{1, -3, 1, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -2, -2, -1}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{1, -2, -2, 1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, -2, -1, -1}, 0.000640, {0.000000, 0.010000, 0.010000, 0.080000, 0.250000}},
	{{1, -2, -1, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -2, 0, -1}, 0.002040, {0.020000, 0.050000, 0.000000, 0.240000, 0.790000}},
	{{1, -2, 0, 1}, 0.002140, {0.000000, 0.050000, 0.000000, 0.260000, 0.830000}},
	{{1, -2, 1, -1}, -0.000600, {-0.010000, -0.010000, -0.010000, -0.070000, -0.230000}},
	{{1, -2, 1, 1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{1, -2, 2, -1}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{1, -2, 2, 1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, -1, -4, -1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{1, -1, -3, -1}, 0.000180, {0.000000, 0.000000, 0.010000, 0.010000, 0.070000}},
	{{1, -1, -3, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, -1, -2, -1}, 0.000960, {0.000000, 0.020000, 0.040000, 0.060000, 0.370000}},
	{{1, -1, -2, 1}, -0.001370, {-0.030000, -0.030000, -0.050000, -0.080000, -0.530000}},
	{{1, -1, -1, -3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, -1, -1, -1}, 0.010310, {0.070000, 0.230000, 0.200000, 0.620000, 4.010000}},
	{{1, -1, -1, 1}, 0.000800, {0.040000, 0.030000, 0.000000, 0.050000, 0.310000}},
	{{1, -1, 0, -3}, -0.000520, {0.000000, -0.030000, 0.000000, -0.030000, -0.200000}},
	{{1, -1, 0, -1}, -0.013460, {-1.070000, -0.270000, 0.000000, -0.820000, -5.230000}},
	{{1, -1, 0, 1}, -0.018290, {-1.520000, -0.330000, 0.000000, -1.110000, -7.110000}},
	{{1, -1, 0, 3}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{1, -1, 1, -3}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{1, -1, 1, -1}, -0.020120, {-0.420000, -0.450000, -0.380000, -1.200000, -7.830000}},
	{{1, -1, 1, 1}, -0.008450, {-0.450000, -0.170000, -0.170000, -0.510000, -3.290000}},
	{{1, -1, 1, 3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, 2, -1}, -0.001750, {-0.040000, -0.040000, -0.060000, -0.100000, -0.680000}},
	{{1, -1, 2, 1}, -0.001460, {-0.070000, -0.030000, -0.060000, -0.090000, -0.570000}},
	{{1, -1, 3, -1}, -0.000150, {0.000000, 0.000000, 0.000000, 0.000000, -0.060000}},
	{{1, -1, 3, 1}, -0.000180, {0.000000, 0.000000, -0.010000, -0.010000, -0.070000}},
	{{1, -1, 4, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, 4, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, -5, -1}, -0.000110, {0.000000, 0.000000, -0.010000, 0.000000, -0.040000}},
	{{1, 0, -5, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, 0, -4, -1}, -0.001220, {-0.040000, -0.030000, -0.090000, 0.000000, -0.470000}},
	{{1, 0, -4, 1}, -0.000430, {0.000000, 0.000000, -0.030000, 0.000000, -0.170000}},
	{{1, 0, -3, -1}, -0.012550, {-0.370000, -0.270000, -0.690000, 0.000000, -4.880000}},
	{{1, 0, -3, 1}, -0.005370, {-0.120000, -0.120000, -0.290000, 0.000000, -2.090000}},
	{{1, 0, -3, 3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, -2, -3}, -0.000250, {0.000000, -0.020000, 0.000000, 0.000000, -0.100000}},
	{{1, 0, -2, -1}, -0.109640, {-3.190000, -2.330000, -4.030000, -0.020000, -42.630000}},
	{{1, 0, -2, 1}, -0.078460, {-2.000000, -1.720000, -2.870000, -0.030000, -30.510000}},
	{{1, 0, -2, 3}, -0.000200, {0.000000, -0.010000, 0.000000, 0.000000, -0.080000}},
	{{1, 0, -1, -3}, -0.003690, {-0.040000, -0.250000, -0.070000, 0.000000, -1.440000}},
	{{1, 0, -1, -1}, -0.429890, {-13.130000, -8.830000, -8.150000, -0.010000, -167.150000}},
	{{1, 0, -1, 1}, 0.139280, {1.860000, 3.370000, 2.460000, 0.040000, 54.160000}},
	{{1, 0, -1, 3}, 0.002340, {0.050000, 0.160000, 0.040000, 0.000000, 0.910000}},
	{{1, 0, 0, -5}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{1, 0, 0, -3}, -0.032260, {-0.410000, -2.150000, 0.000000, 0.000000, -12.540000}},
	{{1, 0, 0, -1}, -4.805780, {-102.500000, -103.390000, -3.120000, -0.090000, -1868.590000}},
	{{1, 0, 0, 1}, -5.368440, {-119.820000, -113.860000, -3.160000, -0.080000, -2087.370000}},
	{{1, 0, 0, 3}, 0.006450, {0.150000, 0.430000, 0.020000, 0.000000, 2.510000}},
	{{1, 0, 0, 5}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, 1, -5}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, 1, -3}, -0.000190, {0.010000, -0.010000, 0.000000, 0.000000, -0.070000}},
	{{1, 0, 1, -1}, -0.588930, {-7.870000, -13.100000, -10.900000, 0.030000, -228.990000}},
	{{1, 0, 1, 1}, -0.667410, {-14.680000, -14.150000, -12.470000, 0.070000, -259.500000}},
	{{1, 0, 1, 3}, 0.001420, {0.030000, 0.090000, 0.030000, 0.000000, 0.550000}},
	{{1, 0, 2, -1}, -0.036360, {-0.400000, -0.810000, -1.340000, 0.000000, -14.140000}},
	{{1, 0, 2, 1}, -0.063830, {-1.350000, -1.350000, -2.350000, 0.020000, -24.820000}},
	{{1, 0, 2, 3}, 0.000210, {0.000000, 0.010000, 0.000000, 0.000000, 0.080000}},
	{{1, 0, 3, -1}, -0.002470, {-0.020000, -0.050000, -0.140000, 0.000000, -0.960000}},
	{{1, 0, 3, 1}, -0.005530, {-0.110000, -0.120000, -0.300000, 0.000000, -2.150000}},
	{{1, 0, 3, 3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, 4, -1}, -0.000170, {0.000000, 0.000000, -0.010000, 0.000000, -0.070000}},
	{{1, 0, 4, 1}, -0.000450, {0.000000, 0.000000, -0.030000, 0.000000, -0.180000}},
	{{1, 0, 5, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, 5, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, 1, -5, -1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, -4, -1}, 0.000170, {0.000000, 0.000000, 0.010000, 0.010000, 0.070000}},
	{{1, 1, -4, 1}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{1, 1, -3, -1}, 0.001860, {0.020000, 0.040000, 0.100000, 0.110000, 0.720000}},
	{{1, 1, -3, 1}, 0.000930, {0.000000, 0.020000, 0.050000, 0.060000, 0.360000}},
	{{1, 1, -2, -3}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, -2, -1}, 0.015970, {0.200000, 0.350000, 0.590000, 0.960000, 6.210000}},
	{{1, 1, -2, 1}, 0.016800, {0.230000, 0.380000, 0.610000, 1.010000, 6.530000}},
	{{1, 1, -2, 3}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{1, 1, -1, -3}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{1, 1, -1, -1}, 0.006350, {-0.380000, 0.070000, 0.200000, 0.360000, 2.460000}},
	{{1, 1, -1, 1}, -0.055900, {-0.180000, -1.290000, -0.970000, -3.360000, -21.740000}},
	{{1, 1, -1, 3}, -0.000210, {0.000000, -0.010000, 0.000000, -0.010000, -0.080000}},
	{{1, 1, 0, -3}, 0.000800, {0.000000, 0.050000, 0.000000, 0.050000, 0.310000}},
	{{1, 1, 0, -1}, 0.804260, {0.220000, 17.530000, 1.110000, 48.120000, 312.710000}},
	{{1, 1, 0, 1}, 0.802630, {0.340000, 17.410000, 0.850000, 48.020000, 312.080000}},
	{{1, 1, 0, 3}, -0.000830, {0.000000, -0.060000, 0.000000, -0.050000, -0.320000}},
	{{1, 1, 1, -3}, -0.000210, {0.010000, -0.010000, 0.000000, -0.010000, -0.080000}},
	{{1, 1, 1, -1}, 0.034650, {0.090000, 0.750000, 0.710000, 2.070000, 13.480000}},
	{{1, 1, 1, 1}, 0.101760, {0.150000, 2.200000, 1.970000, 6.090000, 39.570000}},
	{{1, 1, 1, 3}, -0.000200, {0.000000, -0.010000, 0.000000, -0.010000, -0.080000}},
	{{1, 1, 2, -3}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, 2, -1}, 0.003170, {0.000000, 0.070000, 0.120000, 0.190000, 1.230000}},
	{{1, 1, 2, 1}, 0.010160, {0.030000, 0.220000, 0.380000, 0.610000, 3.950000}},
	{{1, 1, 2, 3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, 1, 3, -1}, 0.000270, {0.000000, 0.000000, 0.020000, 0.020000, 0.110000}},
	{{1, 1, 3, 1}, 0.000930, {0.000000, 0.020000, 0.050000, 0.060000, 0.360000}},
	{{1, 1, 4, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, 4, 1}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{1, 2, -3, -1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{1, 2, -3, 1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 2, -2, -1}, 0.000170, {0.000000, 0.000000, 0.000000, 0.020000, 0.060000}},
	{{1, 2, -2, 1}, 0.000160, {0.000000, 0.000000, 0.000000, 0.020000, 0.060000}},
	{{1, 2, -1, -1}, -0.000080, {0.000000, 0.000000, 0.000000, -0.010000, -0.030000}},
	{{1, 2, -1, 1}, -0.000390, {-0.020000, 0.000000, 0.000000, -0.050000, -0.150000}},
	{{1, 2, 0, -1}, -0.001700, {-0.030000, -0.040000, 0.000000, -0.200000, -0.660000}},
	{{1, 2, 0, 1}, -0.002040, {-0.030000, -0.040000, 0.000000, -0.240000, -0.790000}},
	{{1, 2, 1, -1}, -0.000220, {0.000000, 0.000000, 0.000000, -0.030000, -0.090000}},
	{{1, 2, 1, 1}, -0.000720, {-0.010000, -0.020000, -0.010000, -0.090000, -0.280000}},
	{{1, 2, 2, -1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, 2, 2, 1}, -0.000120, {0.000000, 0.000000, 0.000000, -0.010000, -0.050000}},
	{{1, 2, 3, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 3, 0, -1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{1, 3, 0, 1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{2, -5, -1, -1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -5, -1, 1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -5, 0, -1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{2, -5, 0, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -4, -2, -1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -4, -1, -1}, 0.000310, {0.000000, 0.000000, 0.000000, 0.070000, 0.000000}},
	{{2, -4, -1, 1}, 0.000340, {0.000000, 0.000000, 0.000000, 0.080000, 0.000000}},
	{{2, -4, 0, -1}, 0.001180, {0.030000, 0.030000, 0.000000, 0.280000, 0.000000}},
	{{2, -4, 0, 1}, 0.000620, {0.030000, 0.010000, 0.000000, 0.150000, 0.000000}},
	{{2, -4, 1, -1}, 0.000080, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{2, -4, 1, 1}, 0.000100, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{2, -4, 2, 1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, -3, -1}, 0.000060, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{2, -3, -2, -1}, 0.000770, {0.020000, 0.020000, 0.030000, 0.140000, 0.000000}},
	{{2, -3, -2, 1}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, -1, -1}, 0.009140, {0.250000, 0.200000, 0.170000, 1.640000, 0.000000}},
	{{2, -3, -1, 1}, 0.010420, {0.250000, 0.230000, 0.190000, 1.870000, 0.000000}},
	{{2, -3, -1, 3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, 0, -3}, 0.000080, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{2, -3, 0, -1}, 0.036470, {0.910000, 0.810000, 0.020000, 6.540000, 0.000000}},
	{{2, -3, 0, 1}, 0.016030, {0.620000, 0.350000, 0.060000, 2.880000, 0.000000}},
	{{2, -3, 0, 3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, 1, -3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, 1, -1}, 0.002400, {0.070000, 0.050000, 0.050000, 0.430000, 0.000000}},
	{{2, -3, 1, 1}, 0.002560, {0.110000, 0.060000, 0.050000, 0.460000, 0.000000}},
	{{2, -3, 2, -1}, 0.000210, {0.000000, 0.000000, 0.000000, 0.040000, 0.000000}},
	{{2, -3, 2, 1}, 0.000310, {0.010000, 0.000000, 0.010000, 0.060000, 0.000000}},
	{{2, -3, 3, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, 3, 1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, -4, -1}, 0.000120, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{2, -2, -4, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, -3, -1}, 0.001780, {0.030000, 0.040000, 0.100000, 0.210000, 0.000000}},
	{{2, -2, -3, 1}, 0.000300, {0.000000, 0.000000, 0.020000, 0.040000, 0.000000}},
	{{2, -2, -2, -1}, 0.022850, {0.530000, 0.510000, 0.830000, 2.730000, 0.000000}},
	{{2, -2, -2, 1}, -0.001920, {0.000000, -0.040000, -0.070000, -0.230000, 0.000000}},
	{{2, -2, -2, 3}, -0.000090, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{2, -2, -1, -3}, 0.000240, {0.000000, 0.020000, 0.000000, 0.030000, 0.000000}},
	{{2, -2, -1, -1}, 0.268650, {6.320000, 5.970000, 4.910000, 32.140000, 0.000000}},
	{{2, -2, -1, 1}, 0.314740, {6.780000, 6.970000, 5.750000, 37.650000, 0.000000}},
	{{2, -2, -1, 3}, -0.000450, {0.000000, -0.030000, 0.000000, -0.050000, 0.000000}},
	{{2, -2, 0, -3}, 0.002730, {0.030000, 0.180000, 0.000000, 0.330000, 0.000000}},
	{{2, -2, 0, -1}, 1.085870, {23.510000, 24.250000, 0.470000, 129.870000, 0.000000}},
	{{2, -2, 0, 1}, 0.383530, {13.480000, 8.450000, 1.720000, 45.870000, 0.000000}},
	{{2, -2, 0, 3}, -0.000440, {-0.010000, -0.030000, 0.000000, -0.050000, 0.000000}},
	{{2, -2, 1, -3}, -0.000720, {-0.020000, -0.050000, -0.010000, -0.090000, 0.000000}},
	{{2, -2, 1, -1}, 0.069150, {1.690000, 1.510000, 1.310000, 8.270000, 0.000000}},
	{{2, -2, 1, 1}, 0.058480, {2.240000, 1.280000, 1.260000, 6.990000, 0.000000}},
	{{2, -2, 1, 3}, -0.000110, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{2, -2, 2, -3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 2, -1}, 0.005550, {0.160000, 0.120000, 0.210000, 0.660000, 0.000000}},
	{{2, -2, 2, 1}, 0.006770, {0.270000, 0.150000, 0.270000, 0.810000, 0.000000}},
	{{2, -2, 2, 3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 3, -1}, 0.000460, {0.010000, 0.000000, 0.030000, 0.060000, 0.000000}},
	{{2, -2, 3, 1}, 0.000700, {0.030000, 0.020000, 0.040000, 0.080000, 0.000000}},
	{{2, -2, 4, -1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 4, 1}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -6, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -5, -1}, 0.000290, {0.000000, 0.000000, 0.030000, 0.020000, 0.000000}},
	{{2, -1, -5, 1}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -4, -1}, 0.004160, {0.060000, 0.090000, 0.300000, 0.250000, 0.000000}},
	{{2, -1, -4, 1}, 0.000800, {0.000000, 0.020000, 0.060000, 0.050000, 0.000000}},
	{{2, -1, -3, -3}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -3, -1}, 0.055020, {0.990000, 1.230000, 3.010000, 3.290000, 0.000000}},
	{{2, -1, -3, 1}, 0.009350, {0.120000, 0.210000, 0.510000, 0.560000, 0.000000}},
	{{2, -1, -3, 3}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -2, -3}, 0.000330, {-0.010000, 0.020000, 0.010000, 0.020000, 0.000000}},
	{{2, -1, -2, -1}, 0.650250, {13.750000, 14.460000, 23.700000, 38.910000, 0.000000}},
	{{2, -1, -2, 1}, -0.062080, {-0.480000, -1.400000, -2.270000, -3.710000, 0.000000}},
	{{2, -1, -2, 3}, -0.002670, {-0.030000, -0.180000, -0.100000, -0.160000, 0.000000}},
	{{2, -1, -1, -5}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -1, -3}, 0.010340, {0.060000, 0.690000, 0.170000, 0.620000, 0.000000}},
	{{2, -1, -1, -1}, 7.434880, {154.990000, 165.140000, 135.900000, 444.610000, -0.020000}},
	{{2, -1, -1, 1}, 8.868530, {174.250000, 196.330000, 161.680000, 530.290000, -0.010000}},
	{{2, -1, -1, 3}, -0.011770, {-0.230000, -0.790000, -0.240000, -0.700000, 0.000000}},
	{{2, -1, -1, 5}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 0, -5}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 0, -3}, 0.088150, {0.980000, 5.900000, -0.130000, 5.270000, 0.000000}},
	{{2, -1, 0, -1}, 29.577940, {553.390000, 661.640000, 12.430000, 1767.420000, 0.160000}},
	{{2, -1, 0, 1}, 7.958910, {253.110000, 175.210000, 44.350000, 475.600000, 0.000000}},
	{{2, -1, 0, 3}, -0.009440, {-0.280000, -0.630000, -0.110000, -0.560000, 0.000000}},
	{{2, -1, 0, 5}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 1, -5}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 1, -3}, -0.016690, {-0.300000, -1.120000, -0.310000, -1.000000, 0.000000}},
	{{2, -1, 1, -1}, 1.766060, {35.980000, 38.730000, 33.430000, 105.500000, 0.020000}},
	{{2, -1, 1, 1}, 1.134660, {38.360000, 24.880000, 25.430000, 67.780000, 0.000000}},
	{{2, -1, 1, 3}, -0.002240, {-0.070000, -0.150000, -0.060000, -0.130000, 0.000000}},
	{{2, -1, 2, -3}, -0.000610, {-0.010000, -0.040000, -0.020000, -0.040000, 0.000000}},
	{{2, -1, 2, -1}, 0.128970, {2.890000, 2.780000, 4.820000, 7.700000, 0.000000}},
	{{2, -1, 2, 1}, 0.123870, {4.350000, 2.710000, 4.970000, 7.400000, 0.000000}},
	{{2, -1, 2, 3}, -0.000360, {-0.010000, -0.020000, -0.020000, -0.020000, 0.000000}},
	{{2, -1, 3, -3}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 3, -1}, 0.009850, {0.240000, 0.210000, 0.550000, 0.590000, 0.000000}},
	{{2, -1, 3, 1}, 0.012110, {0.440000, 0.260000, 0.700000, 0.720000, 0.000000}},
	{{2, -1, 3, 3}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 4, -1}, 0.000760, {0.020000, 0.020000, 0.060000, 0.050000, 0.000000}},
	{{2, -1, 4, 1}, 0.001110, {0.040000, 0.020000, 0.080000, 0.070000, 0.000000}},
	{{2, -1, 5, -1}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 5, 1}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -7, -1}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -7, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -6, -1}, 0.000920, {0.020000, 0.020000, 0.100000, 0.000000, 0.000000}},
	{{2, 0, -6, 1}, 0.000200, {0.000000, 0.000000, 0.020000, 0.000000, 0.000000}},
	{{2, 0, -5, -3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -5, -1}, 0.011270, {0.230000, 0.250000, 1.030000, 0.000000, 0.000000}},
	{{2, 0, -5, 1}, 0.002300, {0.030000, 0.050000, 0.210000, 0.000000, 0.000000}},
	{{2, 0, -4, -3}, -0.000140, {0.000000, 0.000000, -0.010000, 0.000000, 0.000000}},
	{{2, 0, -4, -1}, 0.133810, {2.770000, 2.960000, 9.750000, 0.000000, 0.000000}},
	{{2, 0, -4, 1}, 0.024960, {0.390000, 0.560000, 1.820000, 0.000000, 0.000000}},
	{{2, 0, -4, 3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -3, -3}, 0.000230, {-0.020000, 0.010000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -3, -1}, 1.515640, {31.760000, 33.590000, 82.840000, 0.070000, 0.000000}},
	{{2, 0, -3, 1}, 0.254080, {3.880000, 5.750000, 13.920000, 0.000000, 0.010000}},
	{{2, 0, -3, 3}, 0.002320, {0.050000, 0.160000, 0.130000, 0.000000, 0.000000}},
	{{2, 0, -3, 5}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -2, -5}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -2, -3}, 0.020450, {0.100000, 1.370000, 0.700000, 0.000000, 0.000000}},
	{{2, 0, -2, -1}, 15.566350, {330.350000, 344.990000, 567.470000, 0.230000, 0.030000}},
	{{2, 0, -2, 1}, -1.624430, {-20.880000, -36.380000, -59.200000, 0.180000, -0.020000}},
	{{2, 0, -2, 3}, -0.065610, {-0.770000, -4.410000, -2.380000, 0.000000, 0.000000}},
	{{2, 0, -2, 5}, 0.000120, {0.000000, 0.010000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -1, -5}, -0.000460, {0.000000, -0.050000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -1, -3}, 0.329070, {3.620000, 22.030000, 5.640000, -0.020000, 0.010000}},
	{{2, 0, -1, -1}, 166.575280, {3219.290000, 3695.900000, 3042.800000, -5.990000, 0.390000}},
	{{2, 0, -1, 1}, 199.485150, {3726.280000, 4413.250000, 3627.390000, -8.680000, 0.730000}},
	{{2, 0, -1, 3}, -0.244840, {-4.550000, -16.390000, -4.940000, 0.020000, 0.000000}},
	{{2, 0, -1, 5}, 0.000380, {0.000000, 0.040000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 0, -5}, -0.002210, {-0.020000, -0.250000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 0, -3}, 2.186370, {23.850000, 146.450000, -2.380000, -0.270000, 0.030000}},
	{{2, 0, 0, -1}, 623.657830, {9963.620000, 13978.190000, 245.160000, -70.260000, 5.000000}},
	{{2, 0, 0, 1}, 117.261610, {3313.250000, 2575.660000, 874.520000, -12.650000, 1.790000}},
	{{2, 0, 0, 3}, -0.144530, {-3.710000, -9.640000, -2.050000, 0.020000, 0.000000}},
	{{2, 0, 0, 5}, 0.000260, {0.000000, 0.030000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 1, -5}, -0.000750, {0.000000, -0.080000, -0.010000, 0.000000, 0.000000}},
	{{2, 0, 1, -3}, -0.291160, {-3.490000, -19.630000, -5.360000, 0.050000, 0.000000}},
	{{2, 0, 1, -1}, 33.357430, {512.770000, 735.760000, 630.310000, -5.210000, 0.430000}},
	{{2, 0, 1, 1}, 15.121650, {433.440000, 331.250000, 359.200000, -2.400000, 0.310000}},
	{{2, 0, 1, 3}, -0.030380, {-0.820000, -2.020000, -0.850000, 0.000000, 0.000000}},
	{{2, 0, 1, 5}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 2, -5}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 2, -3}, -0.009950, {-0.120000, -0.680000, -0.360000, 0.000000, 0.000000}},
	{{2, 0, 2, -1}, 2.146180, {33.220000, 46.620000, 80.120000, -0.450000, 0.040000}},
	{{2, 0, 2, 1}, 1.519760, {43.480000, 33.210000, 62.710000, -0.330000, 0.040000}},
	{{2, 0, 2, 3}, -0.004510, {-0.120000, -0.300000, -0.200000, 0.000000, 0.000000}},
	{{2, 0, 2, 5}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 3, -3}, -0.000930, {0.000000, -0.060000, -0.050000, 0.000000, 0.000000}},
	{{2, 0, 3, -1}, 0.146420, {2.280000, 3.130000, 8.160000, -0.040000, 0.000000}},
	{{2, 0, 3, 1}, 0.137950, {3.920000, 3.010000, 8.150000, -0.040000, 0.000000}},
	{{2, 0, 3, 3}, -0.000560, {-0.020000, -0.040000, -0.030000, 0.000000, 0.000000}},
	{{2, 0, 4, -3}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 4, -1}, 0.010270, {0.160000, 0.220000, 0.760000, 0.000000, 0.000000}},
	{{2, 0, 4, 1}, 0.011860, {0.330000, 0.260000, 0.910000, 0.000000, 0.000000}},
	{{2, 0, 4, 3}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 5, -1}, 0.000730, {0.010000, 0.020000, 0.070000, 0.000000, 0.000000}},
	{{2, 0, 5, 1}, 0.000990, {0.030000, 0.020000, 0.090000, 0.000000, 0.000000}},
	{{2, 0, 6, -1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 6, 1}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -6, -1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -5, -1}, 0.000280, {0.020000, 0.000000, 0.030000, 0.020000, 0.000000}},
	{{2, 1, -5, 1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -4, -1}, 0.002470, {0.190000, 0.050000, 0.180000, 0.150000, 0.000000}},
	{{2, 1, -4, 1}, 0.000340, {0.030000, 0.000000, 0.030000, 0.020000, 0.000000}},
	{{2, 1, -3, -3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -3, -1}, 0.018180, {1.670000, 0.380000, 0.990000, 1.100000, 0.000000}},
	{{2, 1, -3, 1}, 0.001620, {0.170000, 0.030000, 0.090000, 0.100000, 0.000000}},
	{{2, 1, -2, -3}, -0.000150, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -2, -1}, 0.079130, {12.080000, 1.570000, 2.840000, 4.800000, 0.000000}},
	{{2, 1, -2, 1}, 0.054290, {3.180000, 1.160000, 1.970000, 3.260000, 0.000000}},
	{{2, 1, -2, 3}, 0.000170, {-0.030000, 0.010000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -1, -3}, -0.003420, {0.050000, -0.230000, -0.060000, -0.200000, 0.000000}},
	{{2, 1, -1, -1}, -0.791050, {41.530000, -18.400000, -14.910000, -47.090000, -0.010000}},
	{{2, 1, -1, 1}, -1.317880, {39.710000, -29.980000, -24.580000, -78.590000, 0.000000}},
	{{2, 1, -1, 3}, 0.001920, {-0.030000, 0.130000, 0.040000, 0.110000, 0.000000}},
	{{2, 1, 0, -5}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 0, -3}, -0.054570, {-0.950000, -3.650000, 0.010000, -3.270000, 0.000000}},
	{{2, 1, 0, -1}, -12.094700, {-205.760000, -271.060000, -3.140000, -723.840000, -0.310000}},
	{{2, 1, 0, 1}, -1.264330, {-30.650000, -27.380000, -10.590000, -75.650000, -0.410000}},
	{{2, 1, 0, 3}, 0.001770, {0.040000, 0.120000, 0.030000, 0.110000, 0.000000}},
	{{2, 1, 1, -5}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 1, -3}, 0.004650, {0.020000, 0.310000, 0.090000, 0.280000, 0.000000}},
	{{2, 1, 1, -1}, -0.822750, {-17.020000, -18.200000, -15.410000, -49.250000, -0.040000}},
	{{2, 1, 1, 1}, -0.237020, {-7.850000, -5.100000, -5.790000, -14.190000, -0.080000}},
	{{2, 1, 1, 3}, 0.000520, {0.020000, 0.030000, 0.020000, 0.030000, 0.000000}},
	{{2, 1, 2, -3}, 0.000210, {0.000000, 0.010000, 0.000000, 0.010000, 0.000000}},
	{{2, 1, 2, -1}, -0.062830, {-1.470000, -1.370000, -2.340000, -3.760000, 0.000000}},
	{{2, 1, 2, 1}, -0.031420, {-1.170000, -0.670000, -1.310000, -1.880000, -0.010000}},
	{{2, 1, 2, 3}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 3, -3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 3, -1}, -0.004960, {-0.130000, -0.110000, -0.280000, -0.300000, 0.000000}},
	{{2, 1, 3, 1}, -0.003550, {-0.140000, -0.080000, -0.210000, -0.210000, 0.000000}},
	{{2, 1, 3, 3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 4, -1}, -0.000400, {-0.010000, 0.000000, -0.030000, -0.020000, 0.000000}},
	{{2, 1, 4, 1}, -0.000370, {-0.020000, 0.000000, -0.030000, -0.020000, 0.000000}},
	{{2, 1, 5, -1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 5, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -4, -1}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -4, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -3, -1}, -0.001020, {-0.030000, -0.020000, -0.060000, -0.120000, 0.000000}},
	{{2, 2, -3, 1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -2, -3}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -2, -1}, -0.012620, {-0.470000, -0.280000, -0.460000, -1.510000, 0.000000}},
	{{2, 2, -2, 1}, -0.005040, {-0.270000, -0.110000, -0.190000, -0.600000, 0.000000}},
	{{2, 2, -2, 3}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -1, -3}, -0.000460, {-0.020000, -0.030000, 0.000000, -0.060000, 0.000000}},
	{{2, 2, -1, -1}, -0.105350, {-3.660000, -2.340000, -1.910000, -12.620000, 0.000000}},
	{{2, 2, -1, 1}, -0.113300, {-3.820000, -2.500000, -2.080000, -13.580000, 0.000000}},
	{{2, 2, -1, 3}, 0.000120, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{2, 2, 0, -3}, -0.001030, {-0.020000, -0.060000, 0.020000, -0.120000, 0.000000}},
	{{2, 2, 0, -1}, -0.134150, {-2.410000, -3.040000, -0.260000, -16.070000, 0.000000}},
	{{2, 2, 0, 1}, -0.014820, {-0.520000, -0.320000, -0.530000, -1.780000, 0.030000}},
	{{2, 2, 0, 3}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, 1, -3}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, 1, -1}, -0.006170, {-0.080000, -0.140000, -0.130000, -0.740000, 0.000000}},
	{{2, 2, 1, 1}, -0.001450, {-0.050000, -0.030000, -0.080000, -0.170000, 0.000000}},
	{{2, 2, 2, -1}, -0.000250, {0.000000, 0.000000, -0.010000, -0.030000, 0.000000}},
	{{2, 2, 2, 1}, -0.000090, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{2, 3, -2, -1}, -0.000060, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{2, 3, -2, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 3, -1, -1}, -0.000090, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{2, 3, -1, 1}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 3, 0, -1}, 0.000370, {0.010000, 0.000000, 0.000000, 0.070000, 0.000000}},
	{{2, 3, 0, 1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 3, 1, -1}, 0.000060, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{3, -3, -1, -1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, -3, -1, 1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{3, -3, 0, -1}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, -2, -3, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -2, -2, -1}, -0.000140, {0.000000, 0.000000, 0.000000, -0.020000, -0.060000}},
	{{3, -2, -2, 1}, -0.000160, {0.000000, 0.000000, 0.000000, -0.020000, -0.060000}},
	{{3, -2, -1, -1}, -0.001010, {-0.030000, -0.020000, -0.020000, -0.120000, -0.390000}},
	{{3, -2, -1, 1}, -0.000630, {-0.030000, -0.010000, -0.010000, -0.080000, -0.250000}},
	{{3, -2, 0, -3}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, -2, 0, -1}, -0.001310, {-0.050000, -0.030000, 0.000000, -0.160000, -0.510000}},
	{{3, -2, 0, 1}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{3, -2, 1, -1}, -0.000140, {0.000000, 0.000000, 0.000000, -0.020000, -0.060000}},
	{{3, -2, 2, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -1, -4, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -1, -3, -1}, -0.000220, {0.000000, 0.000000, -0.010000, -0.010000, -0.080000}},
	{{3, -1, -2, -1}, -0.002820, {-0.070000, -0.060000, -0.100000, -0.170000, -1.100000}},
	{{3, -1, -2, 1}, -0.003290, {-0.070000, -0.070000, -0.120000, -0.200000, -1.280000}},
	{{3, -1, -1, -3}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, -0.040000}},
	{{3, -1, -1, -1}, -0.021040, {-0.530000, -0.470000, -0.390000, -1.260000, -8.180000}},
	{{3, -1, -1, 1}, -0.013560, {-0.550000, -0.290000, -0.270000, -0.810000, -5.270000}},
	{{3, -1, -1, 3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -1, 0, -3}, -0.000760, {-0.010000, -0.050000, 0.000000, -0.050000, -0.300000}},
	{{3, -1, 0, -1}, -0.025720, {-0.940000, -0.600000, -0.070000, -1.530000, -10.000000}},
	{{3, -1, 0, 1}, -0.000420, {-0.190000, 0.000000, -0.100000, -0.020000, -0.160000}},
	{{3, -1, 0, 3}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -1, 1, -1}, -0.002750, {-0.110000, -0.060000, -0.060000, -0.160000, -1.070000}},
	{{3, -1, 1, 1}, -0.000300, {-0.050000, 0.000000, -0.020000, -0.020000, -0.120000}},
	{{3, -1, 2, -1}, -0.000270, {-0.010000, 0.000000, -0.010000, -0.020000, -0.100000}},
	{{3, -1, 2, 1}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{3, -1, 3, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -1, 3, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, -4, -1}, -0.000160, {0.000000, 0.000000, -0.010000, 0.000000, -0.060000}},
	{{3, 0, -4, 1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{3, 0, -3, -1}, -0.002870, {-0.040000, -0.060000, -0.160000, 0.000000, -1.120000}},
	{{3, 0, -3, 1}, 0.000150, {0.000000, 0.000000, 0.000000, 0.000000, 0.060000}},
	{{3, 0, -3, 3}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, -2, -3}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{3, 0, -2, -1}, -0.039410, {-0.880000, -0.870000, -1.440000, 0.000000, -15.320000}},
	{{3, 0, -2, 1}, -0.048520, {-1.040000, -1.060000, -1.770000, 0.000000, -18.870000}},
	{{3, 0, -2, 3}, 0.000160, {0.000000, 0.010000, 0.000000, 0.000000, 0.060000}},
	{{3, 0, -1, -3}, -0.001280, {-0.010000, -0.090000, -0.020000, 0.000000, -0.500000}},
	{{3, 0, -1, -1}, -0.305170, {-7.430000, -6.740000, -5.660000, 0.080000, -118.660000}},
	{{3, 0, -1, 1}, -0.205930, {-7.880000, -4.390000, -4.050000, 0.120000, -80.070000}},
	{{3, 0, -1, 3}, 0.000380, {0.020000, 0.020000, 0.000000, 0.000000, 0.150000}},
	{{3, 0, 0, -3}, -0.010090, {-0.130000, -0.680000, 0.000000, 0.000000, -3.920000}},
	{{3, 0, 0, -1}, -0.351830, {-11.870000, -8.150000, -0.990000, 0.220000, -136.800000}},
	{{3, 0, 0, 1}, -0.028400, {-2.770000, -0.490000, -1.370000, 0.100000, -11.040000}},
	{{3, 0, 0, 3}, 0.000160, {0.000000, 0.010000, 0.000000, 0.000000, 0.060000}},
	{{3, 0, 1, -3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, 1, -1}, -0.036110, {-1.260000, -0.830000, -0.750000, 0.030000, -14.040000}},
	{{3, 0, 1, 1}, -0.006580, {-0.560000, -0.120000, -0.290000, 0.020000, -2.560000}},
	{{3, 0, 1, 3}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{3, 0, 2, -1}, -0.003220, {-0.120000, -0.070000, -0.130000, 0.000000, -1.250000}},
	{{3, 0, 2, 1}, -0.001080, {-0.080000, -0.020000, -0.060000, 0.000000, -0.420000}},
	{{3, 0, 3, -1}, -0.000270, {-0.010000, 0.000000, -0.020000, 0.000000, -0.110000}},
	{{3, 0, 3, 1}, -0.000140, {0.000000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{3, 0, 4, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, 4, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 1, -4, -1}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{3, 1, -4, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 1, -3, -1}, -0.000440, {-0.020000, 0.000000, -0.020000, -0.030000, -0.170000}},
	{{3, 1, -3, 1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{3, 1, -2, -1}, -0.001520, {-0.090000, -0.030000, -0.060000, -0.090000, -0.590000}},
	{{3, 1, -2, 1}, -0.002610, {-0.120000, -0.060000, -0.090000, -0.160000, -1.020000}},
	{{3, 1, -1, -3}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.010000}},
	{{3, 1, -1, -1}, 0.013210, {0.230000, 0.290000, 0.250000, 0.790000, 5.140000}},
	{{3, 1, -1, 1}, 0.020830, {0.370000, 0.450000, 0.390000, 1.240000, 8.100000}},
	{{3, 1, -1, 3}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, 1, 0, -3}, 0.000310, {0.000000, 0.020000, 0.000000, 0.020000, 0.120000}},
	{{3, 1, 0, -1}, 0.034360, {0.680000, 0.770000, 0.090000, 2.050000, 13.360000}},
	{{3, 1, 0, 1}, 0.013510, {0.390000, 0.290000, 0.170000, 0.810000, 5.250000}},
	{{3, 1, 0, 3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{3, 1, 1, -1}, 0.004260, {0.090000, 0.090000, 0.090000, 0.250000, 1.660000}},
	{{3, 1, 1, 1}, 0.002580, {0.080000, 0.050000, 0.070000, 0.150000, 1.000000}},
	{{3, 1, 2, -1}, 0.000440, {0.000000, 0.000000, 0.020000, 0.030000, 0.170000}},
	{{3, 1, 2, 1}, 0.000350, {0.010000, 0.000000, 0.020000, 0.020000, 0.140000}},
	{{3, 1, 3, -1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{3, 1, 3, 1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{3, 2, -2, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 2, -2, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 2, -1, -1}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, 2, -1, 1}, -0.000100, {0.000000, 0.000000, 0.000000, -0.010000, -0.040000}},
	{{3, 2, 0, -1}, -0.000560, {0.000000, -0.010000, 0.000000, -0.070000, -0.220000}},
	{{3, 2, 0, 1}, -0.000160, {0.000000, 0.000000, 0.000000, -0.020000, -0.060000}},
	{{3, 2, 1, -1}, -0.000090, {0.000000, 0.000000, 0.000000, -0.010000, -0.030000}},
	{{3, 2, 1, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, 2, 2, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 3, -1, -3}, 0.000010, {0.000000, 0.000000, -0.020000, -0.010000, 0.030000}},
	{{3, 3, -1, -1}, 0.000000, {-0.010000, 0.000000, -0.010000, -0.020000, 0.010000}},
	{{3, 3, -1, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -4, -2, 1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -4, -1, -1}, 0.000070, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{4, -4, -1, 1}, 0.000070, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{4, -4, 0, -1}, 0.000080, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{4, -4, 0, 1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{4, -4, 1, -1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, -2, -1}, 0.000130, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{4, -3, -2, 1}, 0.000590, {0.030000, 0.010000, 0.020000, 0.110000, 0.000000}},
	{{4, -3, -1, -1}, 0.001650, {0.070000, 0.040000, 0.030000, 0.300000, 0.000000}},
	{{4, -3, -1, 1}, 0.001340, {0.070000, 0.030000, 0.030000, 0.240000, 0.000000}},
	{{4, -3, 0, -3}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, 0, -1}, 0.001610, {0.080000, 0.040000, 0.000000, 0.290000, 0.000000}},
	{{4, -3, 0, 1}, 0.000800, {0.050000, 0.020000, 0.010000, 0.140000, 0.000000}},
	{{4, -3, 1, -1}, 0.000240, {0.010000, 0.000000, 0.000000, 0.040000, 0.000000}},
	{{4, -3, 1, 1}, 0.000170, {0.010000, 0.000000, 0.000000, 0.030000, 0.000000}},
	{{4, -3, 2, -1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, 2, 1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -4, -1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -3, -1}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -3, 1}, 0.000110, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{4, -2, -2, -3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -2, -1}, 0.002830, {0.110000, 0.060000, 0.100000, 0.340000, 0.000000}},
	{{4, -2, -2, 1}, 0.012300, {0.500000, 0.270000, 0.450000, 1.470000, 0.000000}},
	{{4, -2, -2, 3}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -1, -3}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -1, -1}, 0.034620, {1.290000, 0.770000, 0.650000, 4.140000, 0.000000}},
	{{4, -2, -1, 1}, 0.023800, {1.250000, 0.520000, 0.530000, 2.850000, 0.000000}},
	{{4, -2, -1, 3}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, 0, -3}, 0.000290, {0.000000, 0.020000, 0.000000, 0.040000, 0.000000}},
	{{4, -2, 0, -1}, 0.028990, {1.410000, 0.650000, 0.180000, 3.460000, 0.000000}},
	{{4, -2, 0, 1}, 0.012700, {0.790000, 0.270000, 0.200000, 1.520000, 0.000000}},
	{{4, -2, 0, 3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, 1, -3}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, 1, -1}, 0.004170, {0.210000, 0.090000, 0.100000, 0.500000, 0.000000}},
	{{4, -2, 1, 1}, 0.002550, {0.170000, 0.060000, 0.080000, 0.300000, 0.000000}},
	{{4, -2, 2, -1}, 0.000470, {0.030000, 0.010000, 0.020000, 0.060000, 0.000000}},
	{{4, -2, 2, 1}, 0.000370, {0.030000, 0.000000, 0.020000, 0.040000, 0.000000}},
	{{4, -2, 3, -1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, 3, 1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -6, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -5, -1}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -4, -1}, -0.000580, {-0.020000, -0.010000, -0.040000, -0.030000, 0.000000}},
	{{4, -1, -4, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -3, -3}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -3, -1}, -0.000900, {-0.030000, -0.020000, -0.050000, -0.050000, 0.000000}},
	{{4, -1, -3, 1}, 0.001910, {0.120000, 0.040000, 0.100000, 0.110000, 0.000000}},
	{{4, -1, -3, 3}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -2, -3}, -0.000330, {-0.010000, -0.020000, -0.010000, -0.020000, 0.000000}},
	{{4, -1, -2, -1}, 0.052510, {1.990000, 1.160000, 1.920000, 3.140000, 0.000000}},
	{{4, -1, -2, 1}, 0.213760, {8.240000, 4.680000, 7.800000, 12.770000, 0.000000}},
	{{4, -1, -2, 3}, -0.000600, {-0.020000, -0.040000, -0.020000, -0.040000, 0.000000}},
	{{4, -1, -1, -3}, -0.000650, {-0.040000, -0.040000, -0.010000, -0.040000, 0.000000}},
	{{4, -1, -1, -1}, 0.595800, {21.190000, 13.270000, 11.160000, 35.530000, 0.020000}},
	{{4, -1, -1, 1}, 0.338820, {16.920000, 7.360000, 7.740000, 20.210000, 0.000000}},
	{{4, -1, -1, 3}, -0.000750, {-0.040000, -0.050000, -0.020000, -0.040000, 0.000000}},
	{{4, -1, 0, -5}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 0, -3}, 0.005340, {0.130000, 0.360000, 0.000000, 0.320000, 0.000000}},
	{{4, -1, 0, -1}, 0.414960, {19.000000, 9.230000, 2.870000, 24.710000, 0.000000}},
	{{4, -1, 0, 1}, 0.157910, {9.240000, 3.410000, 2.730000, 9.410000, 0.000000}},
	{{4, -1, 0, 3}, -0.000370, {-0.020000, -0.020000, 0.000000, -0.020000, 0.000000}},
	{{4, -1, 1, -3}, -0.000620, {-0.020000, -0.040000, -0.010000, -0.040000, 0.000000}},
	{{4, -1, 1, -1}, 0.056860, {2.690000, 1.240000, 1.350000, 3.380000, 0.000000}},
	{{4, -1, 1, 1}, 0.030090, {1.830000, 0.650000, 0.930000, 1.790000, 0.000000}},
	{{4, -1, 1, 3}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 2, -3}, -0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 2, -1}, 0.006140, {0.300000, 0.130000, 0.250000, 0.370000, 0.000000}},
	{{4, -1, 2, 1}, 0.004210, {0.260000, 0.090000, 0.200000, 0.250000, 0.000000}},
	{{4, -1, 2, 3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 3, -3}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 3, -1}, 0.000600, {0.030000, 0.010000, 0.040000, 0.040000, 0.000000}},
	{{4, -1, 3, 1}, 0.000500, {0.030000, 0.010000, 0.030000, 0.030000, 0.000000}},
	{{4, -1, 4, -1}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 4, 1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -7, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -6, -1}, -0.000150, {0.000000, 0.000000, -0.020000, 0.000000, 0.000000}},
	{{4, 0, -6, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -5, -1}, -0.001160, {-0.050000, -0.030000, -0.110000, 0.000000, 0.000000}},
	{{4, 0, -4, -3}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -4, -1}, -0.006800, {-0.280000, -0.150000, -0.500000, 0.000000, 0.000000}},
	{{4, 0, -4, 1}, 0.000290, {0.000000, 0.000000, 0.020000, 0.000000, 0.000000}},
	{{4, 0, -4, 3}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -3, -3}, -0.000660, {-0.020000, -0.040000, -0.040000, 0.000000, 0.000000}},
	{{4, 0, -3, -1}, -0.007710, {-0.290000, -0.170000, -0.420000, 0.000000, 0.000000}},
	{{4, 0, -3, 1}, 0.021740, {1.300000, 0.470000, 1.190000, 0.020000, 0.000000}},
	{{4, 0, -3, 3}, -0.001270, {-0.040000, -0.090000, -0.070000, 0.000000, 0.000000}},
	{{4, 0, -2, -3}, -0.003900, {-0.140000, -0.260000, -0.140000, 0.000000, 0.000000}},
	{{4, 0, -2, -1}, 0.633710, {23.670000, 13.990000, 23.160000, -0.030000, 0.080000}},
	{{4, 0, -2, 1}, 2.413890, {90.530000, 52.840000, 87.950000, -0.350000, 0.070000}},
	{{4, 0, -2, 3}, -0.006320, {-0.240000, -0.420000, -0.240000, 0.000000, 0.000000}},
	{{4, 0, -2, 5}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -1, -5}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -1, -3}, -0.005540, {-0.340000, -0.370000, -0.100000, 0.000000, 0.000000}},
	{{4, 0, -1, -1}, 6.579620, {223.570000, 146.750000, 123.070000, -1.940000, 0.260000}},
	{{4, 0, -1, 1}, 2.998500, {142.570000, 65.020000, 71.120000, -0.780000, 0.060000}},
	{{4, 0, -1, 3}, -0.006640, {-0.310000, -0.440000, -0.190000, 0.000000, 0.000000}},
	{{4, 0, -1, 5}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 0, -5}, -0.000130, {0.000000, -0.010000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 0, -3}, 0.062570, {1.510000, 4.190000, -0.040000, -0.030000, 0.020000}},
	{{4, 0, 0, -1}, 3.674490, {157.550000, 81.630000, 29.570000, -1.610000, 0.060000}},
	{{4, 0, 0, 1}, 1.191880, {65.390000, 25.660000, 23.110000, -0.450000, 0.080000}},
	{{4, 0, 0, 3}, -0.002880, {-0.150000, -0.190000, -0.080000, 0.000000, 0.000000}},
	{{4, 0, 1, -3}, -0.005520, {-0.150000, -0.380000, -0.110000, 0.000000, 0.000000}},
	{{4, 0, 1, -1}, 0.473380, {20.490000, 10.320000, 11.590000, -0.250000, 0.020000}},
	{{4, 0, 1, 1}, 0.212590, {11.970000, 4.560000, 6.920000, -0.100000, 0.010000}},
	{{4, 0, 1, 3}, -0.000680, {-0.040000, -0.040000, -0.030000, 0.000000, 0.000000}},
	{{4, 0, 2, -3}, -0.000900, {-0.020000, -0.060000, -0.030000, 0.000000, 0.000000}},
	{{4, 0, 2, -1}, 0.048340, {2.100000, 1.030000, 2.040000, -0.030000, 0.000000}},
	{{4, 0, 2, 1}, 0.028280, {1.600000, 0.600000, 1.370000, -0.020000, 0.000000}},
	{{4, 0, 2, 3}, -0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 3, -3}, -0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 3, -1}, 0.004490, {0.200000, 0.090000, 0.270000, 0.000000, 0.000000}},
	{{4, 0, 3, 1}, 0.003230, {0.180000, 0.070000, 0.210000, 0.000000, 0.000000}},
	{{4, 0, 3, 3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 4, -1}, 0.000400, {0.020000, 0.000000, 0.030000, 0.000000, 0.000000}},
	{{4, 0, 4, 1}, 0.000340, {0.020000, 0.000000, 0.030000, 0.000000, 0.000000}},
	{{4, 0, 5, -1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 5, 1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -5, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -4, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -4, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -3, -1}, 0.000630, {0.050000, 0.010000, 0.030000, 0.040000, 0.000000}},
	{{4, 1, -3, 1}, 0.001920, {0.190000, 0.040000, 0.100000, 0.110000, 0.000000}},
	{{4, 1, -2, -3}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -2, -1}, -0.002030, {0.570000, -0.050000, -0.080000, -0.120000, 0.000000}},
	{{4, 1, -2, 1}, -0.029570, {0.540000, -0.670000, -1.080000, -1.780000, 0.000000}},
	{{4, 1, -2, 3}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -1, -3}, -0.000150, {-0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -1, -1}, -0.171910, {-3.930000, -3.890000, -3.200000, -10.300000, -0.020000}},
	{{4, 1, -1, 1}, -0.050970, {-1.360000, -1.110000, -1.240000, -3.060000, -0.020000}},
	{{4, 1, -1, 3}, 0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 0, -3}, -0.003040, {-0.090000, -0.200000, 0.000000, -0.180000, 0.000000}},
	{{4, 1, 0, -1}, -0.113080, {-4.830000, -2.510000, -0.930000, -6.760000, -0.020000}},
	{{4, 1, 0, 1}, -0.025490, {-1.260000, -0.540000, -0.520000, -1.530000, 0.000000}},
	{{4, 1, 0, 3}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 1, -3}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 1, -1}, -0.016920, {-0.790000, -0.370000, -0.420000, -1.010000, 0.000000}},
	{{4, 1, 1, 1}, -0.005580, {-0.320000, -0.120000, -0.190000, -0.330000, 0.000000}},
	{{4, 1, 1, 3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 2, -3}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 2, -1}, -0.001960, {-0.100000, -0.040000, -0.080000, -0.120000, 0.000000}},
	{{4, 1, 2, 1}, -0.000880, {-0.050000, -0.020000, -0.040000, -0.050000, 0.000000}},
	{{4, 1, 3, -1}, -0.000200, {-0.010000, 0.000000, -0.010000, -0.010000, 0.000000}},
	{{4, 1, 3, 1}, -0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 4, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 4, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -3, -1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -3, 1}, -0.000200, {-0.010000, 0.000000, -0.010000, -0.020000, 0.000000}},
	{{4, 2, -2, -3}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -2, -1}, -0.001140, {-0.070000, -0.020000, -0.040000, -0.140000, 0.000000}},
	{{4, 2, -2, 1}, -0.002770, {-0.160000, -0.060000, -0.100000, -0.330000, 0.000000}},
	{{4, 2, -1, -3}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -1, -1}, -0.004380, {-0.250000, -0.090000, -0.080000, -0.520000, 0.000000}},
	{{4, 2, -1, 1}, -0.001790, {-0.120000, -0.040000, -0.050000, -0.210000, 0.000000}},
	{{4, 2, 0, -1}, -0.000560, {-0.030000, -0.010000, -0.020000, -0.070000, 0.000000}},
	{{4, 2, 0, 1}, -0.000250, {-0.020000, 0.000000, -0.010000, -0.030000, 0.000000}},
	{{4, 2, 1, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, 1, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 3, -2, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 3, -1, -1}, 0.000080, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{4, 3, -1, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 3, 0, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -2, -2, -1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, -2, -2, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, -2, -1, -1}, -0.000100, {0.000000, 0.000000, 0.000000, -0.010000, -0.040000}},
	{{5, -2, -1, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -2, 0, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -1, -3, -1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{5, -1, -3, 1}, -0.000140, {0.000000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{5, -1, -2, -1}, -0.000570, {-0.020000, -0.010000, -0.020000, -0.030000, -0.220000}},
	{{5, -1, -2, 1}, -0.000610, {-0.040000, -0.010000, -0.020000, -0.040000, -0.240000}},
	{{5, -1, -1, -3}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -1, -1, -1}, -0.001270, {-0.070000, -0.030000, -0.030000, -0.080000, -0.500000}},
	{{5, -1, -1, 1}, -0.000290, {-0.030000, 0.000000, -0.010000, -0.020000, -0.110000}},
	{{5, -1, 0, -3}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, -1, 0, -1}, -0.000320, {-0.030000, 0.000000, 0.000000, -0.020000, -0.120000}},
	{{5, -1, 1, -1}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, 0, -4, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 0, -3, -1}, -0.000330, {-0.010000, 0.000000, -0.020000, 0.000000, -0.130000}},
	{{5, 0, -3, 1}, -0.001240, {-0.050000, -0.030000, -0.070000, 0.000000, -0.480000}},
	{{5, 0, -2, -1}, -0.005050, {-0.210000, -0.110000, -0.190000, 0.000000, -1.960000}},
	{{5, 0, -2, 1}, -0.005190, {-0.290000, -0.110000, -0.200000, 0.000000, -2.020000}},
	{{5, 0, -2, 3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 0, -1, -3}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, -0.040000}},
	{{5, 0, -1, -1}, -0.010490, {-0.540000, -0.240000, -0.220000, 0.010000, -4.080000}},
	{{5, 0, -1, 1}, -0.002450, {-0.210000, -0.050000, -0.090000, 0.000000, -0.950000}},
	{{5, 0, -1, 3}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 0, 0, -3}, -0.000410, {-0.010000, -0.030000, 0.000000, 0.000000, -0.160000}},
	{{5, 0, 0, -1}, -0.002540, {-0.190000, -0.060000, -0.070000, 0.000000, -0.990000}},
	{{5, 0, 0, 1}, -0.000270, {-0.050000, 0.000000, -0.030000, 0.000000, -0.100000}},
	{{5, 0, 1, -3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{5, 0, 1, -1}, -0.000450, {-0.030000, -0.010000, -0.020000, 0.000000, -0.170000}},
	{{5, 0, 1, 1}, -0.000060, {-0.010000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, 0, 2, -1}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, 0, 2, 1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 1, -3, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 1, -3, 1}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, 1, -2, -1}, 0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.060000}},
	{{5, 1, -2, 1}, 0.000410, {0.010000, 0.000000, 0.020000, 0.020000, 0.160000}},
	{{5, 1, -1, -1}, 0.000970, {0.040000, 0.020000, 0.020000, 0.060000, 0.380000}},
	{{5, 1, -1, 1}, 0.000530, {0.030000, 0.010000, 0.010000, 0.030000, 0.210000}},
	{{5, 1, 0, -3}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 1, 0, -1}, 0.000510, {0.020000, 0.010000, 0.000000, 0.030000, 0.200000}},
	{{5, 1, 0, 1}, 0.000230, {0.010000, 0.000000, 0.000000, 0.010000, 0.090000}},
	{{5, 1, 1, -1}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{5, 1, 1, 1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{5, 1, 2, -1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 2, -1, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 2, 0, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -3, -3, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -3, -2, -1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -3, -2, 1}, 0.000060, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{6, -3, -1, -1}, 0.000100, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{6, -3, -1, 1}, 0.000060, {0.000000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{6, -3, 0, -1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -3, 0, 1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -3, 1, -1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -3, -1}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -3, 1}, 0.000330, {0.020000, 0.000000, 0.020000, 0.040000, 0.000000}},
	{{6, -2, -2, -1}, 0.000880, {0.050000, 0.020000, 0.030000, 0.100000, 0.000000}},
	{{6, -2, -2, 1}, 0.000870, {0.060000, 0.020000, 0.040000, 0.100000, 0.000000}},
	{{6, -2, -1, -3}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -1, -1}, 0.001370, {0.090000, 0.030000, 0.030000, 0.160000, 0.000000}},
	{{6, -2, -1, 1}, 0.000780, {0.060000, 0.020000, 0.020000, 0.090000, 0.000000}},
	{{6, -2, 0, -1}, 0.000670, {0.050000, 0.010000, 0.010000, 0.080000, 0.000000}},
	{{6, -2, 0, 1}, 0.000320, {0.030000, 0.000000, 0.000000, 0.040000, 0.000000}},
	{{6, -2, 1, -1}, 0.000130, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{6, -2, 1, 1}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, 2, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, 2, 1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -4, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -4, 1}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -3, -1}, 0.000890, {0.050000, 0.020000, 0.050000, 0.050000, 0.000000}},
	{{6, -1, -3, 1}, 0.004150, {0.240000, 0.090000, 0.230000, 0.250000, 0.000000}},
	{{6, -1, -3, 3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -2, -1}, 0.010910, {0.590000, 0.240000, 0.400000, 0.650000, 0.000000}},
	{{6, -1, -2, 1}, 0.009390, {0.640000, 0.200000, 0.380000, 0.560000, 0.000000}},
	{{6, -1, -2, 3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -1, -3}, 0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -1, -1}, 0.014860, {0.960000, 0.330000, 0.350000, 0.880000, 0.000000}},
	{{6, -1, -1, 1}, 0.007570, {0.590000, 0.160000, 0.240000, 0.450000, 0.000000}},
	{{6, -1, -1, 3}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, 0, -3}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, 0, -1}, 0.006520, {0.470000, 0.140000, 0.120000, 0.390000, 0.000000}},
	{{6, -1, 0, 1}, 0.002880, {0.240000, 0.060000, 0.090000, 0.170000, 0.000000}},
	{{6, -1, 0, 3}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, 1, -1}, 0.001220, {0.090000, 0.030000, 0.040000, 0.070000, 0.000000}},
	{{6, -1, 1, 1}, 0.000630, {0.060000, 0.010000, 0.030000, 0.040000, 0.000000}},
	{{6, -1, 2, -1}, 0.000170, {0.010000, 0.000000, 0.000000, 0.010000, 0.000000}},
	{{6, -1, 2, 1}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, 3, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, 3, 1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -4, -1}, 0.000140, {0.000000, 0.000000, 0.010000, 0.000000, 0.000000}},
	{{6, 0, -4, 1}, 0.000720, {0.050000, 0.020000, 0.050000, 0.000000, 0.000000}},
	{{6, 0, -4, 3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -3, -1}, 0.007020, {0.400000, 0.150000, 0.380000, 0.000000, 0.000000}},
	{{6, 0, -3, 1}, 0.031180, {1.770000, 0.680000, 1.710000, 0.000000, 0.000000}},
	{{6, 0, -3, 3}, -0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -2, -3}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -2, -1}, 0.080960, {4.280000, 1.790000, 3.000000, -0.040000, 0.000000}},
	{{6, 0, -2, 1}, 0.059630, {3.970000, 1.280000, 2.460000, -0.030000, 0.000000}},
	{{6, 0, -2, 3}, -0.000210, {-0.010000, -0.010000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -1, -3}, 0.001070, {0.040000, 0.070000, 0.020000, 0.000000, 0.000000}},
	{{6, 0, -1, -1}, 0.094030, {5.840000, 2.080000, 2.280000, -0.070000, 0.000000}},
	{{6, 0, -1, 1}, 0.042170, {3.170000, 0.890000, 1.370000, -0.030000, 0.000000}},
	{{6, 0, -1, 3}, -0.000140, {-0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 0, -3}, 0.000600, {0.040000, 0.040000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 0, -1}, 0.036740, {2.540000, 0.800000, 0.730000, -0.030000, 0.000000}},
	{{6, 0, 0, 1}, 0.014650, {1.190000, 0.310000, 0.470000, -0.010000, 0.000000}},
	{{6, 0, 0, 3}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 1, -3}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 1, -1}, 0.006540, {0.460000, 0.140000, 0.220000, 0.000000, 0.000000}},
	{{6, 0, 1, 1}, 0.003050, {0.250000, 0.060000, 0.130000, 0.000000, 0.000000}},
	{{6, 0, 1, 3}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 2, -3}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 2, -1}, 0.000880, {0.060000, 0.020000, 0.040000, 0.000000, 0.000000}},
	{{6, 0, 2, 1}, 0.000480, {0.040000, 0.010000, 0.030000, 0.000000, 0.000000}},
	{{6, 0, 3, -1}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 3, 1}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 4, -1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -4, 1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -3, -1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -3, 1}, -0.000540, {0.000000, -0.010000, -0.030000, -0.030000, 0.000000}},
	{{6, 1, -2, -1}, -0.002560, {-0.080000, -0.060000, -0.090000, -0.150000, 0.000000}},
	{{6, 1, -2, 1}, -0.001370, {-0.050000, -0.030000, -0.060000, -0.080000, 0.000000}},
	{{6, 1, -1, -3}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -1, -1}, -0.003500, {-0.190000, -0.080000, -0.080000, -0.210000, 0.000000}},
	{{6, 1, -1, 1}, -0.001160, {-0.070000, -0.020000, -0.040000, -0.070000, 0.000000}},
	{{6, 1, 0, -3}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 0, -1}, -0.001530, {-0.100000, -0.030000, -0.030000, -0.090000, 0.000000}},
	{{6, 1, 0, 1}, -0.000470, {-0.040000, 0.000000, -0.020000, -0.030000, 0.000000}},
	{{6, 1, 1, -1}, -0.000310, {-0.020000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{6, 1, 1, 1}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 2, -1}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 2, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -3, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -3, 1}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -2, -1}, -0.000090, {0.000000, 0.000000, 0.000000, -0.010000, 0.000000}},
	{{6, 2, -2, 1}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -1, -1}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -1, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, -1, -3, -1}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, -1, -3, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, -1, -2, -1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{7, -1, -2, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, -1, -1, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -4, 1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -3, -1}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{7, 0, -3, 1}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, -0.040000}},
	{{7, 0, -2, -1}, -0.000240, {-0.020000, 0.000000, 0.000000, 0.000000, -0.090000}},
	{{7, 0, -2, 1}, -0.000100, {0.000000, 0.000000, 0.000000, 0.000000, -0.040000}},
	{{7, 0, -1, -3}, -0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -1, -1}, -0.000130, {-0.010000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{7, 0, -1, 1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{7, 0, 0, -1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, -0.010000}},
	{{7, 1, -2, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 1, -2, 1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 1, -1, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 1, -1, 1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -3, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -3, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -2, -1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -2, 1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -1, -1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -1, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, 0, -1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -4, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -4, 1}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -3, -1}, 0.000190, {0.010000, 0.000000, 0.010000, 0.010000, 0.000000}},
	{{8, -1, -3, 1}, 0.000220, {0.020000, 0.000000, 0.010000, 0.010000, 0.000000}},
	{{8, -1, -2, -1}, 0.000380, {0.030000, 0.000000, 0.020000, 0.020000, 0.000000}},
	{{8, -1, -2, 1}, 0.000250, {0.020000, 0.000000, 0.010000, 0.010000, 0.000000}},
	{{8, -1, -1, -1}, 0.000290, {0.030000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{8, -1, -1, 1}, 0.000150, {0.020000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, 0, -1}, 0.000110, {0.010000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, 0, 1}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, 1, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, 1, 1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -5, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -4, -1}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -4, 1}, 0.000420, {0.030000, 0.000000, 0.030000, 0.000000, 0.000000}},
	{{8, 0, -3, -1}, 0.001070, {0.080000, 0.020000, 0.060000, 0.000000, 0.000000}},
	{{8, 0, -3, 1}, 0.001080, {0.090000, 0.020000, 0.060000, 0.000000, 0.000000}},
	{{8, 0, -2, -3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -2, -1}, 0.001890, {0.150000, 0.040000, 0.080000, 0.000000, 0.000000}},
	{{8, 0, -2, 1}, 0.001110, {0.110000, 0.020000, 0.050000, 0.000000, 0.000000}},
	{{8, 0, -1, -3}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -1, -1}, 0.001300, {0.120000, 0.030000, 0.040000, 0.000000, 0.000000}},
	{{8, 0, -1, 1}, 0.000600, {0.060000, 0.010000, 0.030000, 0.000000, 0.000000}},
	{{8, 0, 0, -1}, 0.000450, {0.040000, 0.000000, 0.010000, 0.000000, 0.000000}},
	{{8, 0, 0, 1}, 0.000200, {0.020000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, 1, -1}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, 1, 1}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, 2, -1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -3, -1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -3, 1}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -2, -1}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -2, 1}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -1, -1}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -1, 1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, 0, -1}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -4, -1}, 0.000010, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -4, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -3, -1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -3, 1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -2, -1}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -2, 1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -1, -1}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}}
};


static const struct main_problem main_elp3 [ELP3_SIZE] = 
{
	{{0, 0, 0, 0}, 385000.527190, {-7992.630000, -11.060000, 21578.080000, -4.530000, 11.390000}},
	{{0, 0, 0, 2}, -3.148370, {-204.480000, -138.940000, 159.640000, -0.390000, 0.120000}},
	{{0, 0, 0, 4}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 1, -4}, 0.000380, {0.030000, 0.030000, -0.030000, 0.000000, 0.000000}},
	{{0, 0, 1, -2}, 79.661830, {-359.450000, 3583.790000, 1454.020000, -2.370000, 0.850000}},
	{{0, 0, 1, 0}, -20905.322060, {6888.230000, -35.830000, -380331.740000, 22.310000, 1.770000}},
	{{0, 0, 1, 2}, -0.103260, {-9.140000, -4.530000, 8.080000, 0.000000, 0.020000}},
	{{0, 0, 2, -4}, -0.008370, {0.070000, -0.750000, -0.310000, 0.000000, 0.000000}},
	{{0, 0, 2, -2}, -4.421240, {18.290000, -198.910000, -161.900000, 0.140000, -0.040000}},
	{{0, 0, 2, 0}, -569.923320, {374.440000, -1.990000, -20737.330000, 5.790000, 0.440000}},
	{{0, 0, 2, 2}, -0.001020, {-0.370000, -0.040000, 0.620000, 0.000000, 0.000000}},
	{{0, 0, 3, -4}, 0.000330, {0.000000, 0.030000, 0.020000, 0.000000, 0.000000}},
	{{0, 0, 3, -2}, -0.001110, {0.000000, 0.000000, -0.050000, 0.000000, 0.000000}},
	{{0, 0, 3, 0}, -23.210320, {26.490000, -0.140000, -1266.640000, 0.630000, 0.050000}},
	{{0, 0, 3, 2}, 0.000260, {0.000000, 0.000000, 0.060000, 0.000000, 0.000000}},
	{{0, 0, 4, -4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 4, -2}, 0.000270, {0.000000, 0.000000, 0.020000, 0.000000, 0.000000}},
	{{0, 0, 4, 0}, -1.116930, {1.980000, 0.000000, -81.260000, 0.060000, 0.000000}},
	{{0, 0, 4, 2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 5, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 0, 5, 0}, -0.058880, {0.150000, 0.000000, -5.350000, 0.000000, 0.000000}},
	{{0, 0, 6, 0}, -0.003290, {0.000000, 0.000000, -0.360000, 0.000000, 0.000000}},
	{{0, 0, 7, 0}, -0.000190, {0.000000, 0.000000, -0.020000, 0.000000, 0.000000}},
	{{0, 1, -6, 0}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -5, 0}, -0.001740, {-0.030000, 0.000000, -0.160000, -0.100000, 0.000000}},
	{{0, 1, -4, 0}, -0.026740, {-0.560000, 0.000000, -1.940000, -1.600000, 0.000000}},
	{{0, 1, -3, 0}, -0.422420, {-9.350000, 0.200000, -23.040000, -25.300000, 0.000000}},
	{{0, 1, -3, 2}, 0.000150, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -2, -2}, -0.000190, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -2, 0}, -7.002930, {-161.630000, 3.290000, -254.780000, -419.550000, -0.120000}},
	{{0, 1, -2, 2}, -0.050200, {-1.170000, -2.240000, -1.830000, -3.010000, 0.000000}},
	{{0, 1, -2, 4}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, -1, -2}, -0.007690, {-0.150000, -0.340000, -0.040000, -0.460000, 0.000000}},
	{{0, 1, -1, 0}, -129.624760, {-3122.770000, 61.800000, -2362.460000, -7767.840000, -0.740000}},
	{{0, 1, -1, 2}, 0.334650, {7.260000, 14.910000, 5.990000, 20.060000, 0.000000}},
	{{0, 1, -1, 4}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 0, -2}, -0.185680, {-4.480000, -8.290000, 0.540000, -11.120000, 0.000000}},
	{{0, 1, 0, 0}, 48.890100, {1082.400000, -25.990000, 156.780000, 2926.500000, -3.760000}},
	{{0, 1, 0, 2}, -0.158030, {-3.350000, -7.060000, -0.080000, -9.460000, -0.030000}},
	{{0, 1, 1, -4}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 1, -2}, -0.248110, {-2.720000, -11.080000, -4.500000, -14.870000, 0.000000}},
	{{0, 1, 1, 0}, 104.758960, {2165.070000, -47.620000, 1929.100000, 6279.070000, -0.600000}},
	{{0, 1, 1, 2}, -0.006820, {-0.050000, -0.310000, -0.170000, -0.410000, 0.000000}},
	{{0, 1, 2, -4}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 2, -2}, 0.032650, {0.360000, 1.450000, 1.200000, 1.960000, 0.000000}},
	{{0, 1, 2, 0}, 5.751050, {119.470000, -2.620000, 210.760000, 344.760000, -0.090000}},
	{{0, 1, 2, 2}, -0.000300, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 3, -2}, -0.000800, {-0.040000, -0.040000, -0.040000, -0.050000, 0.000000}},
	{{0, 1, 3, 0}, 0.355090, {7.330000, -0.160000, 19.480000, 21.290000, 0.000000}},
	{{0, 1, 4, -2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 1, 4, 0}, 0.023040, {0.470000, 0.000000, 1.680000, 1.380000, 0.000000}},
	{{0, 1, 5, 0}, 0.001530, {0.030000, 0.000000, 0.140000, 0.090000, 0.000000}},
	{{0, 1, 6, 0}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -5, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, -4, 0}, -0.000660, {-0.020000, 0.000000, -0.050000, -0.080000, 0.000000}},
	{{0, 2, -3, 0}, -0.009380, {-0.290000, 0.000000, -0.510000, -1.120000, 0.000000}},
	{{0, 2, -2, 0}, -0.136180, {-4.100000, 0.090000, -4.960000, -16.310000, 0.000000}},
	{{0, 2, -2, 2}, -0.000990, {-0.030000, -0.040000, -0.040000, -0.120000, 0.000000}},
	{{0, 2, -1, -2}, -0.000310, {0.000000, 0.000000, 0.000000, -0.040000, 0.000000}},
	{{0, 2, -1, 0}, -2.117280, {-59.780000, 1.220000, -38.700000, -253.610000, 0.000000}},
	{{0, 2, -1, 2}, 0.004500, {0.120000, 0.200000, 0.080000, 0.540000, 0.000000}},
	{{0, 2, 0, -2}, -0.005580, {-0.140000, -0.250000, 0.000000, -0.670000, 0.000000}},
	{{0, 2, 0, 0}, 1.065750, {16.470000, -0.600000, 3.280000, 127.540000, 0.000000}},
	{{0, 2, 0, 2}, -0.003420, {-0.070000, -0.150000, 0.000000, -0.410000, 0.000000}},
	{{0, 2, 1, -2}, -0.002370, {0.000000, -0.110000, -0.040000, -0.280000, 0.000000}},
	{{0, 2, 1, 0}, 1.165620, {22.200000, -0.390000, 21.490000, 139.600000, 0.000000}},
	{{0, 2, 1, 2}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 2, 2, -2}, 0.000180, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{0, 2, 2, 0}, 0.050720, {0.700000, 0.000000, 1.860000, 6.070000, 0.000000}},
	{{0, 2, 3, 0}, 0.002280, {0.000000, 0.000000, 0.130000, 0.270000, 0.000000}},
	{{0, 2, 4, 0}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, -3, 0}, -0.000210, {0.000000, 0.000000, 0.000000, -0.040000, 0.000000}},
	{{0, 3, -2, 0}, -0.002870, {-0.100000, 0.000000, -0.100000, -0.510000, 0.000000}},
	{{0, 3, -2, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, -1, 0}, -0.038940, {-1.140000, 0.030000, -0.710000, -6.990000, 0.000000}},
	{{0, 3, -1, 2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, 0, -2}, -0.000170, {0.000000, 0.000000, 0.000000, -0.030000, 0.000000}},
	{{0, 3, 0, 0}, 0.021270, {0.030000, 0.000000, 0.080000, 3.820000, 0.000000}},
	{{0, 3, 0, 2}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, 1, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 3, 1, 0}, 0.019120, {0.390000, 0.000000, 0.350000, 3.430000, 0.000000}},
	{{0, 3, 2, 0}, 0.000780, {0.000000, 0.000000, 0.030000, 0.140000, 0.000000}},
	{{0, 3, 3, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 4, -2, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{0, 4, -1, 0}, -0.000730, {-0.020000, 0.000000, 0.000000, -0.170000, 0.000000}},
	{{0, 4, 0, 0}, 0.000340, {0.000000, 0.000000, 0.000000, 0.080000, 0.000000}},
	{{0, 4, 1, 0}, 0.000370, {0.000000, 0.000000, 0.000000, 0.090000, 0.000000}},
	{{1, -4, 0, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{1, -3, -1, 0}, 0.000350, {0.000000, 0.000000, 0.000000, 0.060000, 0.130000}},
	{{1, -3, 0, 0}, -0.001610, {0.000000, 0.000000, 0.000000, -0.290000, -0.630000}},
	{{1, -2, -3, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -2, -2, 0}, 0.000290, {0.000000, 0.000000, 0.000000, 0.040000, 0.110000}},
	{{1, -2, -1, 0}, 0.008440, {0.100000, 0.000000, 0.170000, 1.010000, 3.280000}},
	{{1, -2, 0, -2}, -0.000240, {0.000000, 0.000000, 0.000000, -0.030000, -0.090000}},
	{{1, -2, 0, 0}, -0.029480, {0.180000, 0.000000, -0.030000, -3.520000, -11.460000}},
	{{1, -2, 0, 2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -2, 1, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -2, 1, 0}, 0.001590, {0.160000, 0.000000, 0.030000, 0.190000, 0.620000}},
	{{1, -2, 2, 0}, 0.000360, {0.030000, 0.000000, 0.000000, 0.040000, 0.140000}},
	{{1, -2, 3, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, -4, 0}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{1, -1, -3, 0}, 0.000650, {0.000000, 0.000000, 0.040000, 0.040000, 0.250000}},
	{{1, -1, -2, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, -2, 0}, -0.004220, {-0.240000, 0.000000, -0.130000, -0.260000, -1.640000}},
	{{1, -1, -2, 2}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{1, -1, -1, -2}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, -0.040000}},
	{{1, -1, -1, 0}, 0.114110, {0.000000, -0.020000, 2.310000, 6.800000, 44.370000}},
	{{1, -1, -1, 2}, 0.000210, {0.000000, 0.000000, 0.000000, 0.000000, 0.080000}},
	{{1, -1, 0, -2}, -0.013660, {-0.210000, -0.610000, 0.000000, -0.820000, -5.310000}},
	{{1, -1, 0, 0}, 0.497570, {32.610000, -1.420000, 0.140000, 30.000000, 193.460000}},
	{{1, -1, 0, 2}, -0.000740, {-0.030000, -0.030000, 0.000000, -0.040000, -0.290000}},
	{{1, -1, 1, -2}, 0.000440, {0.000000, 0.020000, 0.000000, 0.030000, 0.170000}},
	{{1, -1, 1, 0}, 0.109980, {5.320000, -0.170000, 2.140000, 6.590000, 42.760000}},
	{{1, -1, 1, 2}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{1, -1, 2, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, -1, 2, 0}, 0.011210, {0.520000, 0.000000, 0.420000, 0.670000, 4.360000}},
	{{1, -1, 3, 0}, 0.000970, {0.040000, 0.000000, 0.050000, 0.060000, 0.380000}},
	{{1, -1, 4, 0}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{1, 0, -6, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 0, -5, 0}, -0.000430, {0.000000, 0.000000, -0.040000, 0.000000, -0.170000}},
	{{1, 0, -4, 0}, -0.006360, {-0.180000, 0.000000, -0.460000, 0.000000, -2.470000}},
	{{1, 0, -3, -2}, -0.000190, {0.000000, 0.000000, 0.000000, 0.000000, -0.070000}},
	{{1, 0, -3, 0}, -0.099380, {-2.830000, 0.090000, -5.450000, -0.040000, -38.640000}},
	{{1, 0, -3, 2}, -0.000480, {0.000000, -0.020000, -0.030000, 0.000000, -0.190000}},
	{{1, 0, -2, -2}, -0.002780, {-0.040000, -0.120000, -0.100000, 0.000000, -1.080000}},
	{{1, 0, -2, 0}, -1.738520, {-48.130000, 1.590000, -63.710000, -0.500000, -675.990000}},
	{{1, 0, -2, 2}, 0.004570, {0.030000, 0.200000, 0.160000, 0.000000, 1.780000}},
	{{1, 0, -1, -2}, -0.042320, {-0.600000, -1.870000, -0.770000, 0.000000, -16.450000}},
	{{1, 0, -1, 0}, -8.379090, {-219.270000, 9.530000, -156.070000, -0.840000, -3257.990000}},
	{{1, 0, -1, 2}, -0.024710, {-0.500000, -1.080000, -0.460000, 0.000000, -9.610000}},
	{{1, 0, 0, -2}, -0.795640, {-11.230000, -35.240000, -0.220000, 0.000000, -309.370000}},
	{{1, 0, 0, 0}, 108.742650, {2351.060000, -120.480000, 69.820000, 2.010000, 42281.560000}},
	{{1, 0, 0, 2}, -0.018740, {-0.510000, -0.830000, -0.130000, 0.000000, -7.290000}},
	{{1, 0, 1, -4}, 0.000170, {0.000000, 0.000000, 0.000000, 0.000000, 0.060000}},
	{{1, 0, 1, -2}, 0.060550, {0.870000, 2.670000, 1.140000, 0.000000, 23.540000}},
	{{1, 0, 1, 0}, 6.321990, {138.540000, -6.740000, 118.100000, -0.560000, 2458.130000}},
	{{1, 0, 1, 2}, -0.001640, {-0.050000, -0.070000, -0.040000, 0.000000, -0.640000}},
	{{1, 0, 2, -2}, 0.002030, {0.000000, 0.090000, 0.070000, 0.000000, 0.790000}},
	{{1, 0, 2, 0}, 0.378520, {7.870000, -0.400000, 13.920000, -0.090000, 147.180000}},
	{{1, 0, 2, 2}, -0.000130, {0.000000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{1, 0, 3, -2}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{1, 0, 3, 0}, 0.023680, {0.460000, -0.020000, 1.300000, 0.000000, 9.210000}},
	{{1, 0, 4, 0}, 0.001520, {0.030000, 0.000000, 0.110000, 0.000000, 0.590000}},
	{{1, 0, 5, 0}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{1, 1, -5, 0}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{1, 1, -4, 0}, 0.001010, {0.000000, 0.000000, 0.070000, 0.060000, 0.390000}},
	{{1, 1, -3, 0}, 0.017290, {0.210000, 0.000000, 0.950000, 1.040000, 6.720000}},
	{{1, 1, -3, 2}, 0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.050000}},
	{{1, 1, -2, -2}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{1, 1, -2, 0}, 0.332260, {4.670000, -0.100000, 12.190000, 19.900000, 129.180000}},
	{{1, 1, -2, 2}, -0.001480, {0.000000, -0.070000, -0.050000, -0.090000, -0.580000}},
	{{1, 1, -1, -2}, 0.001560, {0.000000, 0.070000, 0.030000, 0.090000, 0.610000}},
	{{1, 1, -1, 0}, 0.851270, {-2.920000, -0.500000, 15.940000, 50.910000, 330.990000}},
	{{1, 1, -1, 2}, 0.003310, {-0.020000, 0.150000, 0.060000, 0.200000, 1.290000}},
	{{1, 1, 0, -2}, 0.033790, {0.040000, 1.480000, 0.110000, 2.020000, 13.140000}},
	{{1, 1, 0, 0}, -16.675330, {-4.780000, 10.810000, -19.360000, -997.670000, -6483.730000}},
	{{1, 1, 0, 2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{1, 1, 1, -2}, -0.005620, {0.000000, -0.250000, -0.110000, -0.340000, -2.180000}},
	{{1, 1, 1, 0}, -0.933350, {-1.070000, 0.640000, -18.080000, -55.840000, -362.910000}},
	{{1, 1, 2, -2}, -0.000370, {-0.020000, 0.000000, 0.000000, -0.020000, -0.140000}},
	{{1, 1, 2, 0}, -0.059040, {-0.120000, 0.040000, -2.220000, -3.530000, -22.960000}},
	{{1, 1, 3, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 1, 3, 0}, -0.003920, {0.000000, 0.000000, -0.220000, -0.230000, -1.520000}},
	{{1, 1, 4, 0}, -0.000270, {0.000000, 0.000000, -0.020000, 0.000000, -0.100000}},
	{{1, 2, -4, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 2, -3, 0}, 0.000280, {0.000000, 0.000000, 0.000000, 0.030000, 0.110000}},
	{{1, 2, -2, 0}, 0.003240, {0.140000, 0.000000, 0.120000, 0.390000, 1.260000}},
	{{1, 2, -1, 0}, 0.004380, {0.140000, 0.000000, 0.080000, 0.520000, 1.700000}},
	{{1, 2, 0, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 2, 0, 0}, 0.037290, {0.600000, -0.030000, 0.070000, 4.470000, 14.500000}},
	{{1, 2, 1, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{1, 2, 1, 0}, 0.006460, {0.120000, 0.000000, 0.130000, 0.770000, 2.510000}},
	{{1, 2, 2, 0}, 0.000670, {0.000000, 0.000000, 0.030000, 0.080000, 0.260000}},
	{{1, 2, 3, 0}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{1, 3, -2, 0}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{1, 3, -1, 0}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{1, 3, 0, 0}, 0.000540, {0.000000, 0.000000, 0.000000, 0.100000, 0.210000}},
	{{1, 3, 1, 0}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{2, -6, 0, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -5, -1, 0}, -0.000120, {0.000000, 0.000000, 0.000000, -0.040000, 0.000000}},
	{{2, -5, 0, 0}, -0.000580, {-0.030000, 0.000000, 0.000000, -0.170000, 0.000000}},
	{{2, -5, 1, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -4, -2, 0}, 0.000360, {0.000000, 0.000000, 0.000000, 0.090000, 0.000000}},
	{{2, -4, -1, 0}, -0.004260, {-0.080000, 0.000000, -0.080000, -1.020000, 0.000000}},
	{{2, -4, 0, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -4, 0, 0}, -0.015780, {-0.660000, 0.000000, 0.000000, -3.780000, 0.000000}},
	{{2, -4, 1, 0}, -0.001130, {-0.050000, 0.000000, -0.020000, -0.270000, 0.000000}},
	{{2, -4, 2, 0}, -0.000090, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{2, -3, -4, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, -3, 0}, 0.000460, {0.000000, 0.000000, 0.020000, 0.080000, 0.000000}},
	{{2, -3, -2, 0}, 0.011160, {0.260000, 0.000000, 0.410000, 2.000000, 0.000000}},
	{{2, -3, -2, 2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, -1, -2}, 0.000260, {0.000000, 0.000000, 0.000000, 0.050000, 0.000000}},
	{{2, -3, -1, 0}, -0.148080, {-2.640000, 0.030000, -2.710000, -26.580000, 0.000000}},
	{{2, -3, -1, 2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, 0, -2}, 0.001040, {0.030000, 0.050000, 0.000000, 0.190000, 0.000000}},
	{{2, -3, 0, 0}, -0.410760, {-15.670000, 0.090000, -0.510000, -73.710000, 0.000000}},
	{{2, -3, 1, -2}, 0.000270, {0.000000, 0.000000, 0.000000, 0.050000, 0.000000}},
	{{2, -3, 1, 0}, -0.028550, {-1.200000, 0.000000, -0.560000, -5.120000, 0.000000}},
	{{2, -3, 2, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -3, 2, 0}, -0.002090, {-0.100000, 0.000000, -0.080000, -0.380000, 0.000000}},
	{{2, -3, 3, 0}, -0.000160, {0.000000, 0.000000, 0.000000, -0.030000, 0.000000}},
	{{2, -2, -5, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, -4, 0}, 0.000670, {0.000000, 0.000000, 0.050000, 0.080000, 0.000000}},
	{{2, -2, -3, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, -3, 0}, 0.015170, {0.230000, 0.000000, 0.830000, 1.810000, 0.000000}},
	{{2, -2, -2, -2}, 0.000410, {0.000000, 0.000000, 0.000000, 0.050000, 0.000000}},
	{{2, -2, -2, 0}, 0.343040, {6.870000, -0.030000, 12.500000, 41.040000, 0.000000}},
	{{2, -2, -2, 2}, 0.000990, {0.000000, 0.040000, 0.040000, 0.120000, 0.000000}},
	{{2, -2, -1, -2}, 0.009570, {0.130000, 0.430000, 0.170000, 1.150000, 0.000000}},
	{{2, -2, -1, 0}, -4.950490, {-84.900000, 0.910000, -90.350000, -592.240000, 0.030000}},
	{{2, -2, -1, 2}, 0.001680, {0.040000, 0.080000, 0.040000, 0.200000, 0.000000}},
	{{2, -2, 0, -2}, 0.028390, {0.780000, 1.270000, -0.110000, 3.400000, 0.000000}},
	{{2, -2, 0, 0}, -9.885190, {-346.350000, 1.990000, -14.480000, -1182.310000, 0.070000}},
	{{2, -2, 1, -4}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 1, -2}, 0.007920, {0.090000, 0.360000, 0.140000, 0.950000, 0.000000}},
	{{2, -2, 1, 0}, -0.657580, {-25.020000, 0.180000, -13.070000, -78.640000, 0.000000}},
	{{2, -2, 2, -2}, 0.000840, {0.020000, 0.040000, 0.030000, 0.100000, 0.000000}},
	{{2, -2, 2, 0}, -0.046080, {-1.860000, 0.000000, -1.760000, -5.510000, 0.000000}},
	{{2, -2, 3, -2}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -2, 3, 0}, -0.003350, {-0.140000, 0.000000, -0.190000, -0.400000, 0.000000}},
	{{2, -2, 4, 0}, -0.000250, {0.000000, 0.000000, -0.020000, -0.030000, 0.000000}},
	{{2, -2, 5, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -6, 0}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -5, 0}, 0.001190, {0.000000, 0.000000, 0.110000, 0.070000, 0.000000}},
	{{2, -1, -4, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, -4, 0}, 0.023340, {0.280000, 0.000000, 1.700000, 1.400000, 0.000000}},
	{{2, -1, -3, -2}, 0.000790, {0.000000, 0.040000, 0.040000, 0.050000, 0.000000}},
	{{2, -1, -3, 0}, 0.495060, {7.680000, 0.000000, 27.040000, 29.620000, 0.000000}},
	{{2, -1, -3, 2}, -0.000460, {0.000000, -0.020000, -0.020000, -0.030000, 0.000000}},
	{{2, -1, -2, -2}, 0.015180, {0.170000, 0.680000, 0.540000, 0.910000, 0.000000}},
	{{2, -1, -2, 0}, 10.056540, {188.870000, -1.290000, 366.360000, 601.570000, -0.030000}},
	{{2, -1, -2, 2}, 0.030240, {0.360000, 1.360000, 1.100000, 1.810000, 0.000000}},
	{{2, -1, -1, -2}, 0.323360, {4.560000, 14.420000, 5.730000, 19.340000, 0.000000}},
	{{2, -1, -1, 0}, -152.143140, {-2564.200000, 30.400000, -2771.740000, -9097.840000, 0.060000}},
	{{2, -1, -1, 2}, 0.037340, {0.840000, 1.680000, 0.850000, 2.230000, 0.000000}},
	{{2, -1, 0, -4}, -0.000350, {0.000000, -0.030000, 0.000000, -0.020000, 0.000000}},
	{{2, -1, 0, -2}, 0.657100, {17.830000, 29.300000, -2.920000, 39.260000, 0.000000}},
	{{2, -1, 0, 0}, -204.593570, {-6583.800000, 38.790000, -377.950000, -12225.820000, 0.270000}},
	{{2, -1, 0, 2}, 0.001080, {-0.090000, 0.050000, 0.210000, 0.060000, 0.000000}},
	{{2, -1, 1, -4}, -0.000850, {0.000000, -0.080000, 0.000000, -0.050000, 0.000000}},
	{{2, -1, 1, -2}, 0.209420, {1.810000, 9.460000, 3.820000, 12.500000, 0.000000}},
	{{2, -1, 1, 0}, -12.831850, {-434.140000, 2.980000, -260.020000, -766.510000, 0.020000}},
	{{2, -1, 1, 2}, 0.000000, {0.000000, 0.000000, 0.020000, 0.000000, 0.000000}},
	{{2, -1, 2, -4}, 0.000130, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 2, -2}, 0.018040, {0.340000, 0.810000, 0.660000, 1.080000, 0.000000}},
	{{2, -1, 2, 0}, -0.848830, {-29.750000, 0.220000, -32.800000, -50.680000, 0.000000}},
	{{2, -1, 3, -2}, 0.001270, {0.030000, 0.060000, 0.070000, 0.080000, 0.000000}},
	{{2, -1, 3, 0}, -0.058500, {-2.100000, 0.000000, -3.330000, -3.490000, 0.000000}},
	{{2, -1, 4, -2}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, -1, 4, 0}, -0.004130, {-0.150000, 0.000000, -0.310000, -0.250000, 0.000000}},
	{{2, -1, 5, 0}, -0.000300, {0.000000, 0.000000, -0.030000, 0.000000, 0.000000}},
	{{2, -1, 6, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -7, 0}, 0.000200, {0.000000, 0.000000, 0.030000, 0.000000, 0.000000}},
	{{2, 0, -6, 0}, 0.003040, {0.060000, 0.000000, 0.330000, 0.000000, 0.000000}},
	{{2, 0, -5, -2}, 0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -5, 0}, 0.047400, {0.930000, 0.000000, 4.320000, 0.000000, 0.000000}},
	{{2, 0, -5, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -4, -2}, 0.001880, {0.030000, 0.080000, 0.140000, 0.000000, 0.000000}},
	{{2, 0, -4, 0}, 0.778540, {15.430000, -0.130000, 56.710000, 0.050000, 0.000000}},
	{{2, 0, -4, 2}, 0.000480, {0.020000, 0.020000, 0.040000, 0.000000, 0.000000}},
	{{2, 0, -3, -2}, 0.029100, {0.440000, 1.300000, 1.570000, 0.000000, 0.000000}},
	{{2, 0, -3, 0}, 14.402620, {282.850000, -2.350000, 786.990000, 0.580000, 0.140000}},
	{{2, 0, -3, 2}, -0.020080, {-0.430000, -0.890000, -1.090000, 0.000000, 0.000000}},
	{{2, 0, -3, 4}, -0.000180, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -2, -4}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -2, -2}, 0.472630, {7.200000, 21.060000, 16.920000, 0.000000, 0.000000}},
	{{2, 0, -2, 0}, 246.157680, {4806.360000, -51.780000, 8962.910000, -0.430000, 1.110000}},
	{{2, 0, -2, 2}, 0.774050, {9.680000, 34.650000, 28.170000, -0.040000, 0.000000}},
	{{2, 0, -2, 4}, -0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -1, -4}, -0.000390, {0.000000, -0.040000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, -1, -2}, 8.751700, {133.450000, 390.010000, 155.980000, -0.280000, 0.200000}},
	{{2, 0, -1, 0}, -3699.104680, {-63127.050000, 818.000000, -67236.740000, 147.860000, -15.950000}},
	{{2, 0, -1, 2}, 0.596330, {12.130000, 26.760000, 14.460000, -0.080000, 0.030000}},
	{{2, 0, 0, -4}, -0.007450, {-0.110000, -0.670000, -0.100000, 0.000000, 0.000000}},
	{{2, 0, 0, -2}, 10.321290, {282.080000, 460.160000, -60.760000, -1.290000, 0.060000}},
	{{2, 0, 0, 0}, -2955.966510, {-86674.510000, 507.990000, -7597.690000, 323.540000, -19.090000}},
	{{2, 0, 0, 2}, 0.031430, {-0.690000, 1.430000, 2.990000, 0.000000, 0.000000}},
	{{2, 0, 1, -4}, -0.018660, {-0.110000, -1.670000, -0.330000, 0.000000, 0.000000}},
	{{2, 0, 1, -2}, 4.131180, {20.860000, 186.870000, 75.090000, -0.760000, 0.000000}},
	{{2, 0, 1, 0}, -170.732740, {-4963.300000, 29.720000, -3579.160000, 26.820000, -1.820000}},
	{{2, 0, 1, 2}, 0.001810, {-0.120000, 0.080000, 0.320000, 0.000000, 0.000000}},
	{{2, 0, 2, -4}, 0.002190, {0.000000, 0.200000, 0.080000, 0.000000, 0.000000}},
	{{2, 0, 2, -2}, 0.283990, {2.900000, 12.830000, 10.350000, -0.060000, 0.000000}},
	{{2, 0, 2, 0}, -10.444720, {-300.480000, 1.810000, -410.750000, 2.240000, -0.170000}},
	{{2, 0, 2, 2}, 0.000140, {0.000000, 0.000000, 0.030000, 0.000000, 0.000000}},
	{{2, 0, 3, -4}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 3, -2}, 0.018110, {0.190000, 0.820000, 0.990000, 0.000000, 0.000000}},
	{{2, 0, 3, 0}, -0.669680, {-19.020000, 0.120000, -38.590000, 0.190000, 0.000000}},
	{{2, 0, 4, -2}, 0.001190, {0.000000, 0.050000, 0.090000, 0.000000, 0.000000}},
	{{2, 0, 4, 0}, -0.044220, {-1.240000, 0.000000, -3.360000, 0.000000, 0.000000}},
	{{2, 0, 5, -2}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 0, 5, 0}, -0.002980, {-0.080000, 0.000000, -0.280000, 0.000000, 0.000000}},
	{{2, 0, 6, 0}, -0.000200, {0.000000, 0.000000, -0.020000, 0.000000, 0.000000}},
	{{2, 1, -6, 0}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -5, 0}, 0.001160, {0.080000, 0.000000, 0.110000, 0.070000, 0.000000}},
	{{2, 1, -4, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -4, 0}, 0.014170, {1.080000, 0.000000, 1.030000, 0.850000, 0.000000}},
	{{2, 1, -3, -2}, 0.000210, {0.020000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, -3, 0}, 0.168580, {15.790000, -0.240000, 9.160000, 10.170000, 0.050000}},
	{{2, 1, -3, 2}, 0.000300, {0.000000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{2, 1, -2, -2}, 0.000520, {0.270000, 0.020000, 0.000000, 0.030000, 0.000000}},
	{{2, 1, -2, 0}, 0.143680, {122.130000, -1.910000, 4.580000, 9.230000, 0.130000}},
	{{2, 1, -2, 2}, -0.002030, {0.310000, -0.100000, -0.080000, -0.120000, 0.000000}},
	{{2, 1, -1, -2}, -0.047360, {2.580000, -2.160000, -0.890000, -2.820000, 0.000000}},
	{{2, 1, -1, 0}, 24.209350, {-837.260000, 13.300000, 449.340000, 1443.450000, 0.260000}},
	{{2, 1, -1, 2}, -0.005880, {-0.100000, -0.260000, -0.140000, -0.350000, 0.000000}},
	{{2, 1, 0, -4}, 0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 0, -2}, -0.135720, {-4.940000, -6.040000, 0.540000, -8.130000, 0.030000}},
	{{2, 1, 0, 0}, 30.824980, {873.360000, -8.400000, 94.400000, 1844.570000, 3.860000}},
	{{2, 1, 0, 2}, -0.001430, {-0.040000, -0.060000, -0.040000, -0.090000, 0.000000}},
	{{2, 1, 1, -4}, 0.000400, {0.000000, 0.040000, 0.000000, 0.020000, 0.000000}},
	{{2, 1, 1, -2}, -0.055270, {1.020000, -2.520000, -1.020000, -3.290000, 0.000000}},
	{{2, 1, 1, 0}, 2.616500, {90.880000, -1.010000, 56.020000, 156.640000, 0.490000}},
	{{2, 1, 1, 2}, -0.000150, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 2, -4}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 2, -2}, -0.005370, {-0.030000, -0.240000, -0.200000, -0.320000, 0.000000}},
	{{2, 1, 2, 0}, 0.212520, {8.110000, -0.090000, 8.450000, 12.720000, 0.050000}},
	{{2, 1, 3, -2}, -0.000440, {0.000000, -0.020000, -0.020000, -0.030000, 0.000000}},
	{{2, 1, 3, 0}, 0.017040, {0.680000, 0.000000, 0.990000, 1.020000, 0.000000}},
	{{2, 1, 4, -2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 1, 4, 0}, 0.001350, {0.060000, 0.000000, 0.100000, 0.080000, 0.000000}},
	{{2, 1, 5, 0}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -4, 0}, -0.000410, {0.000000, 0.000000, -0.030000, -0.050000, 0.000000}},
	{{2, 2, -3, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, -3, 0}, -0.010360, {-0.290000, 0.000000, -0.580000, -1.250000, 0.000000}},
	{{2, 2, -2, -2}, -0.000550, {0.000000, -0.020000, -0.020000, -0.070000, 0.000000}},
	{{2, 2, -2, 0}, -0.108880, {-2.720000, 0.000000, -3.960000, -13.050000, 0.000000}},
	{{2, 2, -2, 2}, -0.000460, {0.000000, -0.020000, 0.000000, -0.060000, 0.000000}},
	{{2, 2, -1, -2}, -0.010720, {-0.370000, -0.470000, -0.170000, -1.280000, 0.000000}},
	{{2, 2, -1, 0}, 2.353800, {79.520000, -0.720000, 42.860000, 282.050000, -0.060000}},
	{{2, 2, -1, 2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, 0, -2}, 0.000800, {0.020000, 0.040000, 0.040000, 0.100000, 0.000000}},
	{{2, 2, 0, 0}, 0.147640, {4.880000, -0.050000, 4.900000, 17.690000, -0.290000}},
	{{2, 2, 1, -2}, 0.001870, {0.110000, 0.070000, 0.000000, 0.220000, 0.000000}},
	{{2, 2, 1, 0}, 0.009120, {0.290000, 0.000000, 0.460000, 1.090000, -0.040000}},
	{{2, 2, 2, -2}, 0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 2, 2, 0}, 0.000360, {0.000000, 0.000000, 0.030000, 0.040000, 0.000000}},
	{{2, 3, -3, 0}, -0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 3, -2, 0}, -0.000520, {-0.020000, 0.000000, -0.020000, -0.090000, 0.000000}},
	{{2, 3, -1, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{2, 3, -1, 0}, 0.002080, {0.140000, 0.000000, 0.030000, 0.370000, 0.000000}},
	{{2, 3, 0, 0}, -0.000240, {-0.020000, 0.000000, 0.000000, -0.040000, 0.000000}},
	{{2, 4, -1, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -4, -1, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -4, 0, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -3, -2, 0}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{3, -3, -1, 0}, 0.000650, {0.030000, 0.000000, 0.000000, 0.120000, 0.250000}},
	{{3, -3, 0, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -3, 0, 0}, -0.000550, {0.000000, 0.000000, 0.000000, -0.100000, -0.210000}},
	{{3, -3, 1, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -2, -3, 0}, -0.000160, {0.000000, 0.000000, 0.000000, -0.020000, -0.060000}},
	{{3, -2, -2, 0}, 0.002240, {0.030000, 0.000000, 0.080000, 0.270000, 0.870000}},
	{{3, -2, -1, -2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, -2, -1, 0}, 0.013890, {0.550000, 0.000000, 0.260000, 1.660000, 5.400000}},
	{{3, -2, 0, -2}, 0.000700, {0.000000, 0.030000, 0.000000, 0.080000, 0.270000}},
	{{3, -2, 0, 0}, -0.010480, {-0.230000, 0.000000, 0.040000, -1.250000, -4.070000}},
	{{3, -2, 1, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -2, 1, 0}, -0.000670, {0.000000, 0.000000, 0.000000, -0.080000, -0.260000}},
	{{3, -2, 2, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -1, -4, 0}, -0.000120, {0.000000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{3, -1, -3, 0}, -0.003340, {-0.050000, 0.000000, -0.180000, -0.200000, -1.300000}},
	{{3, -1, -2, -2}, -0.000140, {0.000000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{3, -1, -2, 0}, 0.052930, {0.920000, 0.000000, 1.930000, 3.170000, 20.580000}},
	{{3, -1, -2, 2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, -1, -1, -2}, -0.000890, {0.000000, -0.040000, 0.000000, -0.050000, -0.340000}},
	{{3, -1, -1, 0}, 0.255610, {9.610000, -0.150000, 4.830000, 15.240000, 99.380000}},
	{{3, -1, 0, -2}, 0.014410, {0.210000, 0.640000, 0.000000, 0.860000, 5.610000}},
	{{3, -1, 0, 0}, -0.157080, {-2.880000, -0.040000, 0.830000, -9.420000, -61.080000}},
	{{3, -1, 1, -2}, 0.000490, {0.000000, 0.020000, 0.000000, 0.030000, 0.190000}},
	{{3, -1, 1, 0}, -0.008870, {-0.030000, 0.000000, -0.090000, -0.540000, -3.450000}},
	{{3, -1, 2, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, -1, 2, 0}, -0.000300, {0.000000, 0.000000, 0.000000, 0.000000, -0.120000}},
	{{3, 0, -5, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, -4, 0}, -0.001700, {0.000000, 0.000000, -0.120000, 0.000000, -0.660000}},
	{{3, 0, -3, -2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, -3, 0}, -0.048430, {-0.720000, 0.000000, -2.650000, -0.030000, -18.830000}},
	{{3, 0, -3, 2}, -0.000130, {0.000000, 0.000000, 0.000000, 0.000000, -0.050000}},
	{{3, 0, -2, -2}, -0.002060, {0.000000, -0.090000, -0.070000, 0.000000, -0.800000}},
	{{3, 0, -2, 0}, 0.862430, {16.640000, -0.340000, 31.480000, 0.130000, 335.340000}},
	{{3, 0, -2, 2}, -0.001010, {-0.030000, -0.040000, -0.040000, 0.000000, -0.390000}},
	{{3, 0, -1, -2}, -0.008680, {0.000000, -0.390000, -0.140000, 0.000000, -3.380000}},
	{{3, 0, -1, 0}, 3.258230, {117.770000, -2.270000, 61.910000, -1.340000, 1266.880000}},
	{{3, 0, -1, 2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{3, 0, 0, -4}, -0.000180, {0.000000, 0.000000, 0.000000, 0.000000, -0.070000}},
	{{3, 0, 0, -2}, 0.200990, {2.460000, 8.950000, -0.040000, -0.070000, 78.150000}},
	{{3, 0, 0, 0}, -1.418930, {-19.430000, -0.690000, 10.760000, -0.630000, -551.710000}},
	{{3, 0, 0, 2}, -0.000150, {0.000000, 0.000000, 0.000000, 0.000000, -0.060000}},
	{{3, 0, 1, -4}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{3, 0, 1, -2}, 0.006710, {0.020000, 0.300000, 0.120000, 0.000000, 2.610000}},
	{{3, 0, 1, 0}, -0.065450, {0.720000, -0.100000, -0.320000, -0.110000, -25.450000}},
	{{3, 0, 1, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 0, 2, -2}, 0.000350, {0.000000, 0.000000, 0.000000, 0.000000, 0.140000}},
	{{3, 0, 2, 0}, -0.001110, {0.210000, 0.000000, 0.030000, 0.000000, -0.430000}},
	{{3, 0, 3, 0}, 0.000160, {0.030000, 0.000000, 0.000000, 0.000000, 0.060000}},
	{{3, 0, 4, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 1, -5, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 1, -4, 0}, -0.000650, {-0.030000, 0.000000, -0.050000, -0.040000, -0.250000}},
	{{3, 1, -3, 0}, -0.006800, {-0.290000, 0.000000, -0.370000, -0.410000, -2.640000}},
	{{3, 1, -3, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 1, -2, -2}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, -0.040000}},
	{{3, 1, -2, 0}, 0.038100, {2.030000, 0.000000, 1.370000, 2.290000, 14.820000}},
	{{3, 1, -2, 2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{3, 1, -1, -2}, -0.000230, {0.000000, 0.000000, 0.000000, 0.000000, -0.090000}},
	{{3, 1, -1, 0}, -0.212590, {-4.000000, 0.170000, -4.030000, -12.700000, -82.660000}},
	{{3, 1, -1, 2}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{3, 1, 0, -2}, -0.008080, {-0.120000, -0.360000, 0.000000, -0.480000, -3.140000}},
	{{3, 1, 0, 0}, -0.107660, {-3.220000, 0.120000, -1.090000, -6.420000, -41.860000}},
	{{3, 1, 1, -2}, -0.000180, {0.000000, 0.000000, 0.000000, 0.000000, -0.070000}},
	{{3, 1, 1, 0}, -0.014480, {-0.440000, 0.000000, -0.370000, -0.860000, -5.630000}},
	{{3, 1, 2, 0}, -0.001500, {-0.050000, 0.000000, -0.070000, -0.090000, -0.580000}},
	{{3, 1, 3, 0}, -0.000140, {0.000000, 0.000000, 0.000000, 0.000000, -0.060000}},
	{{3, 2, -3, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{3, 2, -2, 0}, 0.000320, {0.050000, 0.000000, 0.000000, 0.040000, 0.120000}},
	{{3, 2, -1, 0}, 0.000990, {-0.050000, 0.000000, 0.020000, 0.120000, 0.390000}},
	{{3, 2, 0, -2}, 0.000090, {0.000000, 0.000000, 0.000000, 0.020000, 0.040000}},
	{{3, 2, 0, 0}, 0.001620, {0.040000, 0.000000, 0.000000, 0.190000, 0.630000}},
	{{3, 2, 1, 0}, 0.000280, {0.000000, 0.000000, 0.000000, 0.030000, 0.110000}},
	{{3, 2, 2, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 3, -2, -2}, 0.000000, {-0.030000, 0.000000, -0.030000, 0.000000, -0.070000}},
	{{3, 3, -2, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{3, 3, -1, 0}, 0.000100, {0.000000, 0.000000, 0.000000, 0.020000, 0.040000}},
	{{3, 3, 0, -2}, 0.000020, {0.080000, -0.060000, 0.070000, 0.080000, -0.050000}},
	{{4, -5, -1, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -5, 0, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -4, -2, 0}, -0.000220, {0.000000, 0.000000, 0.000000, -0.050000, 0.000000}},
	{{4, -4, -1, 0}, -0.000780, {-0.050000, 0.000000, 0.000000, -0.190000, 0.000000}},
	{{4, -4, 0, 0}, -0.000450, {-0.030000, 0.000000, 0.000000, -0.110000, 0.000000}},
	{{4, -4, 1, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, -3, 0}, -0.000110, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{4, -3, -2, 0}, -0.005130, {-0.220000, 0.000000, -0.190000, -0.920000, 0.000000}},
	{{4, -3, -1, -2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -3, -1, 0}, -0.015710, {-0.870000, 0.000000, -0.310000, -2.820000, 0.000000}},
	{{4, -3, 0, -2}, -0.000220, {0.000000, 0.000000, 0.000000, -0.040000, 0.000000}},
	{{4, -3, 0, 0}, -0.008170, {-0.540000, 0.000000, -0.080000, -1.470000, 0.000000}},
	{{4, -3, 1, 0}, -0.001140, {-0.080000, 0.000000, -0.030000, -0.210000, 0.000000}},
	{{4, -3, 2, 0}, -0.000130, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{4, -2, -5, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -4, 0}, -0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -3, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -3, 0}, -0.002390, {-0.120000, 0.000000, -0.130000, -0.290000, 0.000000}},
	{{4, -2, -3, 2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -2, -2}, -0.000360, {0.000000, 0.000000, 0.000000, -0.040000, 0.000000}},
	{{4, -2, -2, 0}, -0.108340, {-4.380000, 0.040000, -3.950000, -12.960000, 0.000000}},
	{{4, -2, -2, 2}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, -1, -2}, -0.001450, {-0.040000, -0.060000, -0.030000, -0.170000, 0.000000}},
	{{4, -2, -1, 0}, -0.279060, {-14.760000, 0.110000, -5.610000, -33.360000, 0.000000}},
	{{4, -2, -1, 2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, 0, -2}, -0.004030, {-0.180000, -0.180000, 0.000000, -0.480000, 0.000000}},
	{{4, -2, 0, 0}, -0.128060, {-8.090000, 0.050000, -1.350000, -15.300000, -0.020000}},
	{{4, -2, 1, -2}, 0.000230, {0.000000, 0.000000, 0.000000, 0.030000, 0.000000}},
	{{4, -2, 1, 0}, -0.017280, {-1.130000, 0.000000, -0.450000, -2.060000, 0.000000}},
	{{4, -2, 2, -2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -2, 2, 0}, -0.001840, {-0.120000, 0.000000, -0.080000, -0.220000, 0.000000}},
	{{4, -2, 3, 0}, -0.000180, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{4, -1, -6, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -5, 0}, -0.000410, {0.000000, 0.000000, -0.040000, -0.020000, 0.000000}},
	{{4, -1, -4, -2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -4, 0}, -0.002300, {-0.060000, 0.000000, -0.170000, -0.140000, 0.000000}},
	{{4, -1, -3, -2}, -0.000590, {-0.020000, -0.030000, -0.030000, -0.040000, 0.000000}},
	{{4, -1, -3, 0}, -0.043160, {-2.190000, 0.020000, -2.360000, -2.590000, 0.000000}},
	{{4, -1, -3, 2}, 0.000770, {0.020000, 0.030000, 0.040000, 0.050000, 0.000000}},
	{{4, -1, -2, -2}, -0.006610, {-0.250000, -0.290000, -0.240000, -0.400000, 0.000000}},
	{{4, -1, -2, 0}, -1.897100, {-73.130000, 0.790000, -69.170000, -113.340000, -0.120000}},
	{{4, -1, -2, 2}, 0.001600, {0.070000, 0.070000, 0.060000, 0.100000, 0.000000}},
	{{4, -1, -1, -4}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, -1, -2}, -0.026980, {-0.670000, -1.200000, -0.500000, -1.610000, 0.000000}},
	{{4, -1, -1, 0}, -3.958120, {-199.780000, 1.540000, -80.750000, -236.160000, 0.150000}},
	{{4, -1, -1, 2}, 0.000690, {0.030000, 0.030000, 0.030000, 0.040000, 0.000000}},
	{{4, -1, 0, -4}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 0, -2}, -0.057890, {-2.540000, -2.570000, 0.140000, -3.440000, -0.020000}},
	{{4, -1, 0, 0}, -1.571450, {-94.020000, 0.590000, -18.480000, -93.650000, -0.260000}},
	{{4, -1, 0, 2}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 1, -4}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 1, -2}, 0.002720, {0.120000, 0.120000, 0.070000, 0.160000, 0.000000}},
	{{4, -1, 1, 0}, -0.202860, {-12.450000, 0.080000, -5.510000, -12.080000, -0.030000}},
	{{4, -1, 2, -2}, 0.000550, {0.020000, 0.020000, 0.020000, 0.030000, 0.000000}},
	{{4, -1, 2, 0}, -0.020790, {-1.290000, 0.000000, -0.920000, -1.240000, 0.000000}},
	{{4, -1, 3, -2}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, -1, 3, 0}, -0.001940, {-0.120000, 0.000000, -0.120000, -0.120000, 0.000000}},
	{{4, -1, 4, 0}, -0.000170, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -7, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -6, 0}, -0.000550, {-0.020000, 0.000000, -0.060000, 0.000000, 0.000000}},
	{{4, 0, -5, -2}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -5, 0}, -0.005420, {-0.210000, 0.000000, -0.490000, 0.000000, 0.000000}},
	{{4, 0, -4, -2}, -0.000720, {-0.030000, -0.030000, -0.050000, 0.000000, 0.000000}},
	{{4, 0, -4, 0}, -0.029390, {-0.870000, 0.000000, -2.140000, 0.000000, 0.000000}},
	{{4, 0, -4, 2}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -3, -2}, -0.007440, {-0.270000, -0.330000, -0.410000, 0.000000, 0.000000}},
	{{4, 0, -3, 0}, -0.514230, {-25.810000, 0.280000, -28.100000, -0.190000, -0.150000}},
	{{4, 0, -3, 2}, 0.009250, {0.290000, 0.410000, 0.500000, 0.000000, 0.000000}},
	{{4, 0, -2, -4}, -0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -2, -2}, -0.079820, {-2.910000, -3.540000, -2.910000, 0.000000, 0.000000}},
	{{4, 0, -2, 0}, -21.636270, {-808.810000, 9.470000, -788.010000, 2.770000, -1.210000}},
	{{4, 0, -2, 2}, 0.014830, {0.600000, 0.660000, 0.610000, 0.000000, 0.000000}},
	{{4, 0, -1, -4}, -0.000800, {-0.020000, -0.070000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, -1, -2}, -0.321760, {-7.980000, -14.290000, -5.950000, 0.150000, -0.110000}},
	{{4, 0, -1, 0}, -34.782450, {-1678.730000, 13.500000, -725.230000, 8.970000, 0.880000}},
	{{4, 0, -1, 2}, 0.005500, {0.250000, 0.250000, 0.210000, 0.000000, 0.000000}},
	{{4, 0, 0, -4}, 0.000320, {0.000000, 0.030000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 0, -2}, -0.507930, {-21.440000, -22.550000, 1.370000, 0.280000, -0.170000}},
	{{4, 0, 0, 0}, -11.649930, {-657.420000, 4.130000, -155.600000, 4.420000, -2.170000}},
	{{4, 0, 0, 2}, 0.000670, {0.020000, 0.030000, 0.050000, 0.000000, 0.000000}},
	{{4, 0, 1, -4}, 0.000500, {0.000000, 0.040000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 1, -2}, 0.019380, {0.760000, 0.890000, 0.570000, 0.000000, 0.000000}},
	{{4, 0, 1, 0}, -1.422550, {-81.300000, 0.500000, -40.470000, 0.680000, -0.210000}},
	{{4, 0, 1, 2}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 2, -2}, 0.004000, {0.160000, 0.180000, 0.170000, 0.000000, 0.000000}},
	{{4, 0, 2, 0}, -0.139220, {-7.970000, 0.050000, -6.330000, 0.080000, 0.000000}},
	{{4, 0, 3, -2}, 0.000440, {0.000000, 0.020000, 0.030000, 0.000000, 0.000000}},
	{{4, 0, 3, 0}, -0.012430, {-0.710000, 0.000000, -0.780000, 0.000000, 0.000000}},
	{{4, 0, 4, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 0, 4, 0}, -0.001060, {-0.060000, 0.000000, -0.090000, 0.000000, 0.000000}},
	{{4, 0, 5, 0}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -5, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -4, 0}, 0.000930, {0.050000, 0.000000, 0.070000, 0.060000, 0.000000}},
	{{4, 1, -3, 0}, -0.023990, {-2.480000, 0.030000, -1.300000, -1.430000, 0.000000}},
	{{4, 1, -3, 2}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -2, -2}, 0.000360, {-0.050000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{4, 1, -2, 0}, 0.236960, {-7.250000, 0.120000, 8.690000, 14.250000, 0.020000}},
	{{4, 1, -2, 2}, -0.000220, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, -1, -2}, 0.015510, {0.500000, 0.690000, 0.290000, 0.920000, 0.000000}},
	{{4, 1, -1, 0}, 0.578810, {16.490000, -0.100000, 12.270000, 34.720000, 0.130000}},
	{{4, 1, -1, 2}, -0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 0, -2}, 0.019690, {0.950000, 0.870000, 0.000000, 1.180000, 0.000000}},
	{{4, 1, 0, 0}, 0.244530, {12.940000, -0.100000, 3.440000, 14.630000, 0.040000}},
	{{4, 1, 0, 2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 1, -4}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 1, -2}, 0.000130, {0.030000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 1, 0}, 0.036820, {2.170000, -0.020000, 1.070000, 2.200000, 0.000000}},
	{{4, 1, 2, -2}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 1, 2, 0}, 0.004300, {0.270000, 0.000000, 0.200000, 0.260000, 0.000000}},
	{{4, 1, 3, 0}, 0.000450, {0.030000, 0.000000, 0.030000, 0.030000, 0.000000}},
	{{4, 1, 4, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -4, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -3, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, -3, 0}, 0.002520, {0.170000, 0.000000, 0.140000, 0.300000, 0.000000}},
	{{4, 2, -2, -2}, 0.000210, {0.000000, 0.000000, 0.000000, 0.030000, 0.000000}},
	{{4, 2, -2, 0}, 0.026580, {1.570000, 0.000000, 0.970000, 3.180000, 0.000000}},
	{{4, 2, -1, -2}, -0.000640, {-0.040000, -0.030000, 0.000000, -0.080000, 0.000000}},
	{{4, 2, -1, 0}, 0.018680, {1.310000, 0.000000, 0.450000, 2.240000, 0.000000}},
	{{4, 2, 0, -2}, 0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 2, 0, 0}, 0.001360, {0.110000, 0.000000, 0.090000, 0.160000, 0.000000}},
	{{4, 2, 1, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{4, 3, -2, 0}, -0.000150, {0.000000, 0.000000, 0.000000, -0.030000, 0.000000}},
	{{4, 3, -1, 0}, -0.000200, {0.000000, 0.000000, 0.000000, -0.040000, 0.000000}},
	{{4, 3, 0, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -3, -2, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -2, -3, 0}, 0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{5, -2, -2, 0}, 0.000440, {0.030000, 0.000000, 0.000000, 0.050000, 0.170000}},
	{{5, -2, -1, -2}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -2, -1, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, -2, 0, 0}, -0.000160, {0.000000, 0.000000, 0.000000, -0.020000, -0.060000}},
	{{5, -2, 1, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -1, -4, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -1, -3, 0}, 0.001280, {0.060000, 0.000000, 0.070000, 0.080000, 0.500000}},
	{{5, -1, -2, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -1, -2, 0}, 0.005730, {0.320000, 0.000000, 0.220000, 0.340000, 2.230000}},
	{{5, -1, -1, -2}, 0.000340, {0.000000, 0.000000, 0.000000, 0.020000, 0.130000}},
	{{5, -1, -1, 0}, -0.000560, {0.070000, 0.000000, 0.020000, -0.040000, -0.220000}},
	{{5, -1, 0, -2}, 0.000210, {0.000000, 0.000000, 0.000000, 0.000000, 0.080000}},
	{{5, -1, 0, 0}, -0.001530, {-0.060000, 0.000000, 0.000000, -0.090000, -0.600000}},
	{{5, -1, 1, -2}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, -1, 1, 0}, -0.000190, {0.000000, 0.000000, 0.000000, 0.000000, -0.070000}},
	{{5, 0, -4, 0}, 0.000260, {0.000000, 0.000000, 0.020000, 0.000000, 0.100000}},
	{{5, 0, -3, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 0, -3, 0}, 0.011490, {0.490000, 0.000000, 0.630000, 0.000000, 4.470000}},
	{{5, 0, -3, 2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 0, -2, -2}, 0.000300, {0.000000, 0.000000, 0.000000, 0.000000, 0.120000}},
	{{5, 0, -2, 0}, 0.045910, {2.500000, -0.040000, 1.730000, -0.030000, 17.850000}},
	{{5, 0, -1, -2}, 0.002810, {0.100000, 0.120000, 0.050000, 0.000000, 1.090000}},
	{{5, 0, -1, 0}, -0.002480, {0.570000, -0.020000, 0.220000, -0.040000, -0.960000}},
	{{5, 0, 0, -2}, 0.001420, {0.070000, 0.060000, 0.000000, 0.000000, 0.550000}},
	{{5, 0, 0, 0}, -0.008140, {-0.250000, 0.000000, 0.070000, 0.000000, -3.170000}},
	{{5, 0, 1, -2}, 0.000200, {0.000000, 0.000000, 0.000000, 0.000000, 0.080000}},
	{{5, 0, 1, 0}, -0.000890, {0.000000, 0.000000, 0.000000, 0.000000, -0.350000}},
	{{5, 0, 2, 0}, -0.000060, {0.000000, 0.000000, 0.000000, 0.000000, -0.020000}},
	{{5, 1, -4, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 1, -3, 0}, 0.000380, {0.040000, 0.000000, 0.020000, 0.020000, 0.150000}},
	{{5, 1, -2, 0}, -0.002560, {-0.070000, 0.000000, -0.100000, -0.150000, -1.000000}},
	{{5, 1, -1, -2}, -0.000150, {0.000000, 0.000000, 0.000000, 0.000000, -0.060000}},
	{{5, 1, -1, 0}, -0.003110, {-0.160000, 0.000000, -0.080000, -0.190000, -1.210000}},
	{{5, 1, 0, -2}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{5, 1, 0, 0}, -0.001110, {-0.070000, 0.000000, -0.030000, -0.070000, -0.430000}},
	{{5, 1, 1, 0}, -0.000200, {0.000000, 0.000000, 0.000000, 0.000000, -0.080000}},
	{{5, 1, 2, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{5, 2, -1, 0}, 0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.030000}},
	{{5, 2, 0, 0}, 0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -4, -2, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -4, -1, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -3, -3, 0}, -0.000120, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{6, -3, -2, 0}, -0.000440, {-0.030000, 0.000000, 0.000000, -0.080000, 0.000000}},
	{{6, -3, -1, 0}, -0.000430, {-0.040000, 0.000000, 0.000000, -0.080000, 0.000000}},
	{{6, -3, 0, 0}, -0.000170, {0.000000, 0.000000, 0.000000, -0.030000, 0.000000}},
	{{6, -3, 1, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -4, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -3, 0}, -0.001950, {-0.120000, 0.000000, -0.110000, -0.230000, 0.000000}},
	{{6, -2, -2, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -2, 0}, -0.006180, {-0.440000, 0.000000, -0.240000, -0.740000, 0.000000}},
	{{6, -2, -1, -2}, -0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, -1, 0}, -0.005410, {-0.440000, 0.000000, -0.140000, -0.650000, 0.000000}},
	{{6, -2, 0, -2}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -2, 0, 0}, -0.001940, {-0.170000, 0.000000, -0.040000, -0.230000, 0.000000}},
	{{6, -2, 1, 0}, -0.000350, {-0.030000, 0.000000, 0.000000, -0.040000, 0.000000}},
	{{6, -2, 2, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -4, 0}, -0.000850, {-0.060000, 0.000000, -0.060000, -0.050000, 0.000000}},
	{{6, -1, -3, -2}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -3, 0}, -0.024360, {-1.430000, 0.000000, -1.330000, -1.450000, 0.000000}},
	{{6, -1, -3, 2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -2, -2}, -0.000410, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{6, -1, -2, 0}, -0.066780, {-4.610000, 0.040000, -2.600000, -3.980000, 0.000000}},
	{{6, -1, -2, 2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, -1, -1, -2}, -0.001210, {-0.070000, -0.050000, -0.020000, -0.070000, 0.000000}},
	{{6, -1, -1, 0}, -0.051910, {-4.100000, 0.030000, -1.420000, -3.090000, 0.000000}},
	{{6, -1, 0, -2}, -0.000490, {-0.040000, -0.020000, 0.000000, -0.030000, 0.000000}},
	{{6, -1, 0, 0}, -0.017090, {-1.470000, 0.000000, -0.420000, -1.010000, 0.000000}},
	{{6, -1, 1, 0}, -0.002960, {-0.260000, 0.000000, -0.110000, -0.180000, 0.000000}},
	{{6, -1, 2, 0}, -0.000390, {-0.040000, 0.000000, -0.020000, -0.020000, 0.000000}},
	{{6, -1, 3, 0}, -0.000050, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -5, 0}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -4, 0}, -0.006590, {-0.450000, 0.000000, -0.480000, 0.000000, 0.000000}},
	{{6, 0, -4, 2}, 0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -3, -2}, -0.000270, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -3, 0}, -0.183160, {-10.480000, 0.120000, -10.020000, 0.050000, -0.020000}},
	{{6, 0, -3, 2}, 0.000280, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -2, -2}, -0.003260, {-0.130000, -0.140000, -0.120000, 0.000000, 0.000000}},
	{{6, 0, -2, 0}, -0.422500, {-28.360000, 0.260000, -16.580000, 0.190000, 0.000000}},
	{{6, 0, -2, 2}, 0.000200, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, -1, -2}, -0.007850, {-0.440000, -0.350000, -0.130000, 0.000000, 0.000000}},
	{{6, 0, -1, 0}, -0.286630, {-21.860000, 0.160000, -8.120000, 0.180000, -0.030000}},
	{{6, 0, -1, 2}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 0, -4}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 0, -2}, -0.002700, {-0.190000, -0.120000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 0, 0}, -0.085870, {-7.100000, 0.050000, -2.250000, 0.070000, -0.020000}},
	{{6, 0, 1, 0}, -0.014200, {-1.200000, 0.000000, -0.550000, 0.000000, 0.000000}},
	{{6, 0, 2, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 2, 0}, -0.001820, {-0.160000, 0.000000, -0.100000, 0.000000, 0.000000}},
	{{6, 0, 3, 0}, -0.000200, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 0, 4, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -4, 0}, -0.000240, {-0.040000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -3, 0}, 0.002780, {-0.060000, 0.000000, 0.150000, 0.170000, 0.000000}},
	{{6, 1, -2, -2}, 0.000190, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, -2, 0}, 0.009480, {0.350000, 0.000000, 0.370000, 0.570000, 0.000000}},
	{{6, 1, -1, -2}, 0.000420, {0.030000, 0.000000, 0.000000, 0.020000, 0.000000}},
	{{6, 1, -1, 0}, 0.007800, {0.490000, 0.000000, 0.220000, 0.470000, 0.000000}},
	{{6, 1, 0, -2}, 0.000160, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 1, 0, 0}, 0.002720, {0.210000, 0.000000, 0.070000, 0.160000, 0.000000}},
	{{6, 1, 1, 0}, 0.000520, {0.040000, 0.000000, 0.020000, 0.030000, 0.000000}},
	{{6, 1, 2, 0}, 0.000080, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -4, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{6, 2, -3, 0}, 0.000340, {0.030000, 0.000000, 0.020000, 0.040000, 0.000000}},
	{{6, 2, -2, 0}, 0.000440, {0.040000, 0.000000, 0.000000, 0.050000, 0.000000}},
	{{6, 2, -1, 0}, 0.000140, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, -1, -4, 0}, 0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, -1, -3, 0}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.040000}},
	{{7, -1, -2, 0}, 0.000060, {0.000000, 0.000000, 0.000000, 0.000000, 0.020000}},
	{{7, -1, -1, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -4, 0}, 0.000150, {0.000000, 0.000000, 0.000000, 0.000000, 0.060000}},
	{{7, 0, -3, 0}, 0.000660, {0.050000, 0.000000, 0.040000, 0.000000, 0.260000}},
	{{7, 0, -2, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -2, 0}, 0.000320, {0.040000, 0.000000, 0.000000, 0.000000, 0.130000}},
	{{7, 0, -1, -2}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 0, -1, 0}, -0.000080, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{7, 0, 0, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{7, 1, -3, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{7, 1, -2, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, -0.030000}},
	{{7, 1, -1, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -4, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -3, 0}, -0.000130, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, -2, 0}, -0.000160, {0.000000, 0.000000, 0.000000, -0.020000, 0.000000}},
	{{8, -2, -1, 0}, -0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -2, 0, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -4, 0}, -0.000330, {-0.030000, 0.000000, -0.020000, -0.020000, 0.000000}},
	{{8, -1, -3, 0}, -0.001090, {-0.100000, 0.000000, -0.060000, -0.070000, 0.000000}},
	{{8, -1, -2, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -2, 0}, -0.001260, {-0.120000, 0.000000, -0.060000, -0.080000, 0.000000}},
	{{8, -1, -1, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, -1, 0}, -0.000700, {-0.080000, 0.000000, -0.030000, -0.040000, 0.000000}},
	{{8, -1, 0, 0}, -0.000220, {-0.020000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, -1, 1, 0}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -5, 0}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -4, 0}, -0.001840, {-0.140000, 0.000000, -0.130000, 0.000000, 0.000000}},
	{{8, 0, -3, -2}, -0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -3, 0}, -0.005390, {-0.460000, 0.000000, -0.310000, 0.000000, 0.000000}},
	{{8, 0, -2, -2}, -0.000120, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -2, 0}, -0.005590, {-0.530000, 0.000000, -0.250000, 0.000000, 0.000000}},
	{{8, 0, -1, -2}, -0.000090, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, -1, 0}, -0.002850, {-0.290000, 0.000000, -0.110000, 0.000000, 0.000000}},
	{{8, 0, 0, -2}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, 0, 0}, -0.000830, {-0.090000, 0.000000, -0.030000, 0.000000, 0.000000}},
	{{8, 0, 1, 0}, -0.000160, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 0, 2, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -4, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -3, 0}, 0.000150, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -2, 0}, 0.000190, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, -1, 0}, 0.000110, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{8, 1, 0, 0}, 0.000040, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, -1, -3, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, -1, -2, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -5, 0}, -0.000020, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -4, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -3, 0}, -0.000100, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -2, 0}, -0.000070, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}},
	{{10, 0, -1, 0}, -0.000030, {0.000000, 0.000000, 0.000000, 0.000000, 0.000000}}
};


static const struct earth_pert earth_pert_elp4 [ELP4_SIZE] = 
{
	{0, {0, 0, 0, 1}, 270.000000, 0.000030, 0.075000},
	{0, {0, 0, 0, 2}, 0.000000, 0.000370, 0.037000},
	{0, {0, 0, 1, -2}, 180.000000, 0.004800, 0.074000},
	{0, {0, 0, 1, -1}, 90.000000, 0.000010, 5.997000},
	{0, {0, 0, 2, -2}, 0.000000, 0.000850, 2.998000},
	{0, {0, 0, 2, 0}, 180.000000, 0.000250, 0.038000},
	{0, {0, 0, 3, -2}, 0.000000, 0.000030, 0.077000},
	{0, {0, 0, 3, 0}, 180.000000, 0.000020, 0.025000},
	{0, {0, 0, 4, 0}, 180.000000, 0.000010, 0.019000},
	{0, {0, 1, -2, 0}, 180.000000, 0.000030, 0.039000},
	{0, {0, 1, -1, 0}, 180.000000, 0.000480, 0.082000},
	{0, {0, 1, -1, 2}, 180.000000, 0.000010, 0.069000},
	{0, {0, 1, 0, 0}, 180.000000, 0.000470, 1.000000},
	{0, {0, 1, 1, -2}, 180.000000, 0.000050, 0.079000},
	{0, {0, 1, 1, 0}, 180.000000, 0.000190, 0.070000},
	{0, {0, 1, 2, 0}, 180.000000, 0.000010, 0.036000},
	{0, {0, 2, -1, 0}, 180.000000, 0.000030, 0.089000},
	{0, {1, -1, -1, 0}, 180.000000, 0.000010, 0.530000},
	{0, {1, -1, 0, 0}, 180.000000, 0.000150, 0.088000},
	{0, {1, -1, 1, 0}, 180.000000, 0.000010, 0.041000},
	{0, {1, 0, -2, 0}, 180.000000, 0.000010, 0.071000},
	{0, {1, 0, -1, 0}, 180.000000, 0.000040, 1.127000},
	{0, {1, 0, 0, -2}, 0.000000, 0.000010, 0.069000},
	{0, {1, 0, 0, 0}, 180.000000, 0.001000, 0.081000},
	{0, {1, 0, 1, 0}, 180.000000, 0.000060, 0.039000},
	{0, {1, 1, -1, 0}, 0.000000, 0.000190, 8.850000},
	{0, {1, 1, 0, 0}, 180.000000, 0.000700, 0.075000},
	{0, {1, 1, 1, 0}, 180.000000, 0.000050, 0.038000},
	{0, {2, -1, -2, 0}, 180.000000, 0.000020, 0.360000},
	{0, {2, -1, -1, 0}, 0.000000, 0.000610, 0.095000},
	{0, {2, -1, 0, -2}, 0.000000, 0.000020, 0.322000},
	{0, {2, -1, 0, 0}, 0.000000, 0.000010, 0.042000},
	{0, {2, -1, 1, -2}, 0.000000, 0.000040, 0.099000},
	{0, {2, 0, -3, 2}, 180.000000, 0.000010, 0.085000},
	{0, {2, 0, -2, 0}, 180.000000, 0.000770, 0.564000},
	{0, {2, 0, -2, 2}, 0.000000, 0.000060, 0.040000},
	{0, {2, 0, -1, -2}, 180.000000, 0.000050, 0.065000},
	{0, {2, 0, -1, 0}, 0.000000, 0.022060, 0.087000},
	{0, {2, 0, 0, -2}, 0.000000, 0.000640, 0.474000},
	{0, {2, 0, 0, 0}, 0.000000, 0.001020, 0.040000},
	{0, {2, 0, 1, -2}, 0.000000, 0.000890, 0.090000},
	{0, {2, 0, 1, 0}, 0.000000, 0.000040, 0.026000},
	{0, {2, 0, 2, -2}, 0.000000, 0.000070, 0.041000},
	{0, {2, 1, -2, 0}, 0.000000, 0.000060, 1.292000},
	{0, {2, 1, -1, 0}, 180.000000, 0.000060, 0.080000},
	{0, {2, 1, 0, -2}, 180.000000, 0.000020, 0.903000},
	{0, {2, 1, 0, 0}, 180.000000, 0.000010, 0.039000},
	{0, {2, 2, -1, 0}, 0.000000, 0.000090, 0.074000},
	{0, {3, 0, -1, 0}, 180.000000, 0.000010, 0.042000},
	{0, {4, -1, -2, 0}, 0.000000, 0.000020, 0.046000},
	{0, {4, -1, -1, 0}, 0.000000, 0.000020, 0.028000},
	{0, {4, 0, -2, 0}, 0.000000, 0.000260, 0.044000},
	{0, {4, 0, -1, -2}, 0.000000, 0.000020, 0.107000},
	{0, {4, 0, -1, 0}, 0.000000, 0.000190, 0.028000},
	{0, {4, 0, 0, -2}, 0.000000, 0.000020, 0.044000},
	{0, {4, 0, 0, 0}, 0.000000, 0.000020, 0.020000},
	{0, {4, 0, 1, -2}, 0.000000, 0.000010, 0.028000},
	{1, {-6, 0, 0, -1}, 0.001000, 0.000010, 0.013000},
	{1, {-6, 0, 1, -1}, 0.001000, 0.000030, 0.016000},
	{1, {-6, 0, 2, -1}, 0.000680, 0.000030, 0.021000},
	{1, {-6, 0, 3, -1}, 0.000360, 0.000010, 0.029000},
	{1, {-4, -1, 0, -1}, 180.001030, 0.000010, 0.020000},
	{1, {-4, -1, 1, -1}, 180.000810, 0.000020, 0.027000},
	{1, {-4, -1, 1, 1}, 180.000720, 0.000020, 0.096000},
	{1, {-4, 0, -2, -1}, 0.000990, 0.000020, 0.013000},
	{1, {-4, 0, -2, 1}, 180.000500, 0.000010, 0.020000},
	{1, {-4, 0, -1, -1}, 0.001090, 0.000140, 0.016000},
	{1, {-4, 0, -1, 1}, 180.000500, 0.000090, 0.028000},
	{1, {-4, 0, 0, -1}, 0.001000, 0.000750, 0.020000},
	{1, {-4, 0, 0, 1}, 180.000170, 0.000110, 0.044000},
	{1, {-4, 0, 1, -3}, 180.000930, 0.000020, 0.016000},
	{1, {-4, 0, 1, -1}, 0.000800, 0.001410, 0.028000},
	{1, {-4, 0, 1, 1}, 0.000910, 0.000220, 0.106000},
	{1, {-4, 0, 2, -3}, 180.000910, 0.000010, 0.020000},
	{1, {-4, 0, 2, -1}, 0.000290, 0.000650, 0.043000},
	{1, {-4, 1, -1, -1}, 0.000980, 0.000020, 0.016000},
	{1, {-4, 1, 0, -1}, 0.000990, 0.000100, 0.021000},
	{1, {-4, 1, 1, -1}, 0.000800, 0.000170, 0.028000},
	{1, {-4, 1, 1, 1}, 0.000820, 0.000020, 0.119000},
	{1, {-4, 1, 2, -1}, 0.000300, 0.000060, 0.045000},
	{1, {-4, 2, 0, -1}, 0.000940, 0.000010, 0.021000},
	{1, {-4, 2, 1, -1}, 0.000980, 0.000010, 0.029000},
	{1, {-3, 0, 0, -1}, 0.000820, 0.000040, 0.027000},
	{1, {-3, 0, 0, 1}, 180.000730, 0.000230, 0.097000},
	{1, {-3, 0, 1, -1}, 180.000680, 0.000060, 0.042000},
	{1, {-3, 0, 1, 1}, 180.000720, 0.000030, 0.340000},
	{1, {-3, 1, 0, 1}, 180.000720, 0.000020, 0.107000},
	{1, {-2, -2, 1, -1}, 180.000310, 0.000020, 0.074000},
	{1, {-2, -2, 1, 1}, 180.000290, 0.000010, 0.075000},
	{1, {-2, -1, -2, -1}, 180.001230, 0.000020, 0.019000},
	{1, {-2, -1, -2, 1}, 0.000560, 0.000010, 0.039000},
	{1, {-2, -1, -1, -1}, 180.001190, 0.000120, 0.026000},
	{1, {-2, -1, -1, 1}, 0.000650, 0.000090, 0.082000},
	{1, {-2, -1, 0, -1}, 180.001090, 0.000530, 0.039000},
	{1, {-2, -1, 0, 1}, 180.000760, 0.001230, 0.949000},
	{1, {-2, -1, 1, -1}, 180.000470, 0.000530, 0.080000},
	{1, {-2, -1, 2, -1}, 180.001480, 0.000060, 1.388000},
	{1, {-2, 0, -3, -1}, 0.001220, 0.000090, 0.015000},
	{1, {-2, 0, -3, 1}, 180.000440, 0.000050, 0.027000},
	{1, {-2, 0, -2, -3}, 180.000930, 0.000010, 0.013000},
	{1, {-2, 0, -2, -1}, 0.001190, 0.000970, 0.019000},
	{1, {-2, 0, -2, 1}, 180.000500, 0.000650, 0.041000},
	{1, {-2, 0, -1, -3}, 180.001010, 0.000080, 0.015000},
	{1, {-2, 0, -1, -1}, 0.001130, 0.009130, 0.026000},
	{1, {-2, 0, -1, 1}, 180.000490, 0.008870, 0.089000},
	{1, {-2, 0, 0, -3}, 180.000920, 0.000370, 0.019000},
	{1, {-2, 0, 0, -1}, 0.001000, 0.065690, 0.040000},
	{1, {-2, 0, 0, 0}, 90.000000, 0.000060, 0.088000},
	{1, {-2, 0, 0, 1}, 0.000760, 0.047460, 0.487000},
	{1, {-2, 0, 0, 3}, 180.000770, 0.000130, 0.035000},
	{1, {-2, 0, 1, -3}, 180.000730, 0.000440, 0.026000},
	{1, {-2, 0, 1, -1}, 0.000290, 0.049620, 0.087000},
	{1, {-2, 0, 1, 1}, 0.000450, 0.000260, 0.065000},
	{1, {-2, 0, 1, 3}, 180.000760, 0.000010, 0.024000},
	{1, {-2, 0, 2, -3}, 180.000430, 0.000010, 0.040000},
	{1, {-2, 0, 2, -1}, 180.002080, 0.002490, 0.581000},
	{1, {-2, 0, 2, 1}, 180.000700, 0.000480, 0.035000},
	{1, {-2, 0, 3, -1}, 180.001550, 0.000450, 0.067000},
	{1, {-2, 0, 3, 1}, 180.000730, 0.000080, 0.024000},
	{1, {-2, 0, 4, -1}, 180.001450, 0.000050, 0.035000},
	{1, {-2, 1, -2, -1}, 0.001180, 0.000080, 0.020000},
	{1, {-2, 1, -2, 1}, 180.000490, 0.000040, 0.043000},
	{1, {-2, 1, -1, -1}, 0.001110, 0.000720, 0.027000},
	{1, {-2, 1, -1, 1}, 180.000520, 0.000490, 0.098000},
	{1, {-2, 1, 0, -3}, 180.000910, 0.000030, 0.020000},
	{1, {-2, 1, 0, -1}, 0.000980, 0.004820, 0.042000},
	{1, {-2, 1, 0, 1}, 0.000750, 0.001870, 0.327000},
	{1, {-2, 1, 1, -3}, 180.000890, 0.000020, 0.027000},
	{1, {-2, 1, 1, -1}, 0.000310, 0.002250, 0.095000},
	{1, {-2, 1, 1, 1}, 180.000780, 0.000050, 0.061000},
	{1, {-2, 1, 2, -1}, 180.002250, 0.000090, 0.368000},
	{1, {-2, 1, 2, 1}, 180.000710, 0.000030, 0.034000},
	{1, {-2, 1, 3, -1}, 180.001640, 0.000010, 0.063000},
	{1, {-2, 2, -1, -1}, 0.001110, 0.000040, 0.028000},
	{1, {-2, 2, -1, 1}, 180.000540, 0.000020, 0.109000},
	{1, {-2, 2, 0, -1}, 0.000970, 0.000240, 0.044000},
	{1, {-2, 2, 0, 1}, 0.000720, 0.000060, 0.247000},
	{1, {-2, 2, 1, -1}, 0.000310, 0.000080, 0.105000},
	{1, {-2, 3, 0, -1}, 0.000940, 0.000010, 0.046000},
	{1, {-1, -1, -1, -1}, 0.001220, 0.000040, 0.037000},
	{1, {-1, -1, 0, -1}, 0.001070, 0.000190, 0.075000},
	{1, {-1, -1, 0, 1}, 0.000000, 0.000010, 0.075000},
	{1, {-1, -1, 1, -1}, 180.001490, 0.000010, 5.998000},
	{1, {-1, 0, -2, -1}, 180.001310, 0.000030, 0.026000},
	{1, {-1, 0, -1, -1}, 180.001310, 0.000230, 0.039000},
	{1, {-1, 0, -1, 1}, 180.001220, 0.000020, 0.857000},
	{1, {-1, 0, 0, -3}, 0.000930, 0.000010, 0.025000},
	{1, {-1, 0, 0, -1}, 180.001340, 0.000750, 0.081000},
	{1, {-1, 0, 0, 1}, 180.000670, 0.000560, 0.069000},
	{1, {-1, 0, 1, -1}, 0.001270, 0.000280, 1.200000},
	{1, {-1, 0, 1, 1}, 180.000720, 0.000020, 0.036000},
	{1, {-1, 0, 2, -1}, 0.001290, 0.000060, 0.071000},
	{1, {-1, 1, 0, -1}, 179.999900, 0.000040, 0.088000},
	{1, {-1, 1, 0, 1}, 180.000720, 0.000010, 0.065000},
	{1, {0, -2, -1, -1}, 180.001380, 0.000020, 0.065000},
	{1, {0, -2, 0, -1}, 0.000720, 0.000030, 0.487000},
	{1, {0, -2, 1, -1}, 0.001160, 0.000070, 0.089000},
	{1, {0, -1, -3, -1}, 180.001330, 0.000030, 0.024000},
	{1, {0, -1, -2, -1}, 180.001330, 0.000290, 0.036000},
	{1, {0, -1, -2, 1}, 0.000250, 0.000030, 1.389000},
	{1, {0, -1, -1, -3}, 0.000900, 0.000010, 0.024000},
	{1, {0, -1, -1, -1}, 180.001410, 0.001870, 0.070000},
	{1, {0, -1, -1, 1}, 359.999280, 0.000080, 0.080000},
	{1, {0, -1, 0, -3}, 0.000920, 0.000010, 0.036000},
	{1, {0, -1, 0, -1}, 0.000670, 0.002820, 0.949000},
	{1, {0, -1, 0, 1}, 0.000450, 0.000100, 0.039000},
	{1, {0, -1, 1, -1}, 0.001180, 0.003870, 0.082000},
	{1, {0, -1, 1, 1}, 0.000700, 0.000260, 0.026000},
	{1, {0, -1, 2, -1}, 0.001220, 0.000470, 0.039000},
	{1, {0, -1, 2, 1}, 0.000700, 0.000040, 0.019000},
	{1, {0, -1, 3, -1}, 0.001220, 0.000050, 0.026000},
	{1, {0, 0, -5, -1}, 0.001270, 0.000010, 0.015000},
	{1, {0, 0, -4, -1}, 0.001290, 0.000160, 0.019000},
	{1, {0, 0, -3, -3}, 180.001130, 0.000030, 0.015000},
	{1, {0, 0, -3, -1}, 0.001290, 0.002280, 0.025000},
	{1, {0, 0, -3, 1}, 180.000290, 0.000130, 0.077000},
	{1, {0, 0, -2, -3}, 180.001090, 0.000320, 0.019000},
	{1, {0, 0, -2, -1}, 0.001280, 0.032790, 0.038000},
	{1, {0, 0, -2, 1}, 180.000240, 0.003870, 3.574000},
	{1, {0, 0, -1, -3}, 180.001020, 0.002610, 0.025000},
	{1, {0, 0, -1, -1}, 0.001270, 0.491410, 0.075000},
	{1, {0, 0, -1, 0}, 90.000000, 0.000100, 8.847000},
	{1, {0, 0, -1, 1}, 0.000510, 0.050360, 0.074000},
	{1, {0, 0, -1, 3}, 180.000600, 0.000170, 0.025000},
	{1, {0, 0, 0, -5}, 0.000860, 0.000030, 0.019000},
	{1, {0, 0, 0, -3}, 180.000860, 0.014920, 0.037000},
	{1, {0, 0, 0, -1}, 0.000940, 7.063040, 18.613000},
	{1, {0, 0, 0, 0}, 270.000000, 0.000270, 0.075000},
	{1, {0, 0, 0, 1}, 0.000710, 0.360610, 0.037000},
	{1, {0, 0, 0, 3}, 180.000710, 0.000740, 0.019000},
	{1, {0, 0, 1, -3}, 180.000540, 0.000500, 0.073000},
	{1, {0, 0, 1, -1}, 0.001270, 0.493310, 0.076000},
	{1, {0, 0, 1, 0}, 270.000000, 0.000020, 0.038000},
	{1, {0, 0, 1, 1}, 0.000700, 0.038380, 0.025000},
	{1, {0, 0, 1, 3}, 180.000740, 0.000150, 0.015000},
	{1, {0, 0, 2, -3}, 180.000610, 0.000040, 2.582000},
	{1, {0, 0, 2, -1}, 0.001260, 0.034020, 0.038000},
	{1, {0, 0, 2, 1}, 0.000690, 0.003310, 0.019000},
	{1, {0, 0, 2, 3}, 180.000730, 0.000020, 0.012000},
	{1, {0, 0, 3, -1}, 0.001250, 0.002420, 0.025000},
	{1, {0, 0, 3, 1}, 0.000670, 0.000270, 0.015000},
	{1, {0, 0, 4, -1}, 0.001250, 0.000170, 0.019000},
	{1, {0, 0, 4, 1}, 0.000670, 0.000020, 0.013000},
	{1, {0, 0, 5, -1}, 0.001270, 0.000010, 0.015000},
	{1, {0, 1, -3, -1}, 0.001330, 0.000040, 0.026000},
	{1, {0, 1, -2, -1}, 0.001350, 0.000370, 0.039000},
	{1, {0, 1, -2, 1}, 180.000500, 0.000030, 0.781000},
	{1, {0, 1, -1, -3}, 180.000900, 0.000010, 0.026000},
	{1, {0, 1, -1, -1}, 0.001430, 0.002490, 0.081000},
	{1, {0, 1, -1, 1}, 0.002720, 0.000020, 0.069000},
	{1, {0, 1, 0, -1}, 180.000790, 0.002400, 1.057000},
	{1, {0, 1, 0, 1}, 180.000760, 0.000340, 0.036000},
	{1, {0, 1, 1, -1}, 180.001160, 0.002920, 0.070000},
	{1, {0, 1, 1, 1}, 180.000730, 0.000220, 0.024000},
	{1, {0, 1, 2, -1}, 180.001210, 0.000370, 0.036000},
	{1, {0, 1, 2, 1}, 180.000710, 0.000040, 0.018000},
	{1, {0, 1, 3, -1}, 180.001220, 0.000040, 0.025000},
	{1, {0, 2, -1, -1}, 0.001460, 0.000040, 0.088000},
	{1, {0, 2, 0, -1}, 180.000720, 0.000030, 0.514000},
	{1, {0, 2, 1, -1}, 180.001190, 0.000030, 0.066000},
	{1, {1, -1, 0, -1}, 180.000800, 0.000030, 0.088000},
	{1, {1, 0, -2, -1}, 0.001820, 0.000030, 0.070000},
	{1, {1, 0, -1, -1}, 179.999290, 0.000110, 1.063000},
	{1, {1, 0, -1, 1}, 180.000520, 0.000020, 0.039000},
	{1, {1, 0, 0, -1}, 180.000880, 0.003420, 0.081000},
	{1, {1, 0, 0, 1}, 180.000720, 0.000210, 0.026000},
	{1, {1, 0, 1, -1}, 180.001040, 0.000430, 0.039000},
	{1, {1, 0, 1, 1}, 180.000710, 0.000040, 0.019000},
	{1, {1, 0, 2, -1}, 180.001100, 0.000040, 0.026000},
	{1, {1, 1, -1, -1}, 180.000070, 0.000060, 16.874000},
	{1, {1, 1, 0, -1}, 0.000590, 0.000640, 0.075000},
	{1, {1, 1, 0, 1}, 0.000720, 0.000030, 0.025000},
	{1, {1, 1, 1, -1}, 0.000850, 0.000070, 0.038000},
	{1, {2, -3, 0, -1}, 0.000940, 0.000010, 0.046000},
	{1, {2, -2, -1, -1}, 0.000400, 0.000100, 0.106000},
	{1, {2, -2, -1, 1}, 0.000720, 0.000010, 0.028000},
	{1, {2, -2, 0, -1}, 0.000900, 0.000310, 0.044000},
	{1, {2, -2, 0, 1}, 0.000720, 0.000020, 0.020000},
	{1, {2, -2, 1, -1}, 0.001040, 0.000050, 0.028000},
	{1, {2, -1, -3, -1}, 180.001620, 0.000010, 0.062000},
	{1, {2, -1, -2, -1}, 180.002960, 0.000060, 0.354000},
	{1, {2, -1, -2, 1}, 0.000510, 0.000030, 0.042000},
	{1, {2, -1, -1, -1}, 0.000410, 0.002880, 0.096000},
	{1, {2, -1, -1, 1}, 0.000690, 0.000370, 0.027000},
	{1, {2, -1, 0, -1}, 0.000910, 0.006460, 0.042000},
	{1, {2, -1, 0, 1}, 0.000720, 0.000310, 0.020000},
	{1, {2, -1, 1, -1}, 0.001030, 0.000910, 0.027000},
	{1, {2, -1, 1, 1}, 0.000690, 0.000060, 0.016000},
	{1, {2, -1, 2, -1}, 0.001060, 0.000100, 0.020000},
	{1, {2, 0, -4, -1}, 180.001530, 0.000050, 0.035000},
	{1, {2, 0, -3, -1}, 180.001730, 0.000370, 0.066000},
	{1, {2, 0, -3, 1}, 180.000030, 0.000050, 0.085000},
	{1, {2, 0, -2, -3}, 0.001210, 0.000020, 0.035000},
	{1, {2, 0, -2, -1}, 180.003220, 0.001420, 0.547000},
	{1, {2, 0, -2, 1}, 0.000520, 0.000670, 0.040000},
	{1, {2, 0, -1, -1}, 0.000420, 0.064560, 0.088000},
	{1, {2, 0, -1, 1}, 0.000710, 0.008150, 0.026000},
	{1, {2, 0, -1, 3}, 180.000360, 0.000040, 0.015000},
	{1, {2, 0, 0, -3}, 179.998270, 0.000060, 0.463000},
	{1, {2, 0, 0, -1}, 0.000900, 0.096420, 0.041000},
	{1, {2, 0, 0, 1}, 0.000700, 0.004800, 0.019000},
	{1, {2, 0, 0, 3}, 180.000380, 0.000020, 0.013000},
	{1, {2, 0, 1, -3}, 180.001140, 0.000140, 0.090000},
	{1, {2, 0, 1, -1}, 0.001020, 0.012340, 0.026000},
	{1, {2, 0, 1, 1}, 0.000700, 0.000800, 0.015000},
	{1, {2, 0, 2, -3}, 180.001180, 0.000020, 0.041000},
	{1, {2, 0, 2, -1}, 0.001070, 0.001260, 0.020000},
	{1, {2, 0, 2, 1}, 0.000730, 0.000090, 0.013000},
	{1, {2, 0, 3, -1}, 0.001110, 0.000120, 0.016000},
	{1, {2, 0, 3, 1}, 0.000620, 0.000010, 0.011000},
	{1, {2, 1, -1, -1}, 180.000040, 0.000170, 0.080000},
	{1, {2, 1, -1, 1}, 180.000730, 0.000060, 0.025000},
	{1, {2, 1, 0, -1}, 180.000880, 0.001130, 0.039000},
	{1, {2, 1, 0, 1}, 180.000730, 0.000050, 0.019000},
	{1, {2, 1, 1, -1}, 180.000980, 0.000210, 0.026000},
	{1, {2, 1, 1, 1}, 180.000990, 0.000010, 0.015000},
	{1, {2, 1, 2, -1}, 180.001050, 0.000030, 0.019000},
	{1, {2, 2, -1, -1}, 180.000330, 0.000050, 0.074000},
	{1, {3, 0, -2, -1}, 179.999610, 0.000010, 0.095000},
	{1, {3, 0, -1, -1}, 180.000700, 0.000130, 0.042000},
	{1, {3, 0, -1, 1}, 180.000720, 0.000010, 0.020000},
	{1, {3, 1, -1, -1}, 0.000730, 0.000010, 0.040000},
	{1, {3, 1, 0, -1}, 0.000880, 0.000010, 0.026000},
	{1, {4, -2, -1, -1}, 0.000910, 0.000020, 0.029000},
	{1, {4, -2, 0, -1}, 0.000940, 0.000010, 0.021000},
	{1, {4, -1, -2, -1}, 0.000410, 0.000080, 0.046000},
	{1, {4, -1, -2, 1}, 0.000920, 0.000010, 0.021000},
	{1, {4, -1, -1, -1}, 0.000770, 0.000240, 0.028000},
	{1, {4, -1, -1, 1}, 0.000710, 0.000020, 0.016000},
	{1, {4, -1, 0, -1}, 0.000900, 0.000150, 0.021000},
	{1, {4, -1, 1, -1}, 0.000910, 0.000030, 0.016000},
	{1, {4, 0, -3, -1}, 359.998180, 0.000010, 0.104000},
	{1, {4, 0, -3, 1}, 0.000560, 0.000010, 0.027000},
	{1, {4, 0, -2, -1}, 0.000420, 0.000880, 0.044000},
	{1, {4, 0, -2, 1}, 0.000700, 0.000140, 0.020000},
	{1, {4, 0, -1, -1}, 0.000780, 0.002170, 0.028000},
	{1, {4, 0, -1, 1}, 0.000710, 0.000160, 0.016000},
	{1, {4, 0, 0, -1}, 0.000900, 0.001150, 0.020000},
	{1, {4, 0, 0, 1}, 0.000720, 0.000060, 0.013000},
	{1, {4, 0, 1, -1}, 0.000980, 0.000210, 0.016000},
	{1, {4, 0, 1, 1}, 0.000380, 0.000010, 0.011000},
	{1, {4, 0, 2, -1}, 0.000930, 0.000030, 0.013000},
	{1, {4, 1, -1, -1}, 180.000770, 0.000040, 0.027000},
	{1, {4, 1, 0, -1}, 180.000890, 0.000030, 0.020000},
	{1, {6, 0, -3, -1}, 0.000370, 0.000010, 0.029000},
	{1, {6, 0, -2, -1}, 0.000670, 0.000040, 0.021000},
	{1, {6, 0, -1, -1}, 0.000900, 0.000040, 0.016000},
	{1, {6, 0, 0, -1}, 0.000900, 0.000020, 0.013000},
	{2, {-2, -1, 0, 0}, 0.000000, 0.000010, 1.000000},
	{2, {-2, 0, -1, -2}, 180.000000, 0.000040, 0.026000},
	{2, {-2, 0, -1, 0}, 0.000000, 0.000130, 0.089000},
	{2, {-2, 0, 0, -2}, 180.000000, 0.000270, 0.040000},
	{2, {-2, 0, 0, 0}, 180.000000, 0.000430, 0.500000},
	{2, {-2, 0, 1, -2}, 180.000000, 0.000250, 0.086000},
	{2, {-2, 0, 1, 0}, 180.000000, 0.000020, 0.066000},
	{2, {-2, 0, 2, -2}, 0.000000, 0.000010, 0.600000},
	{2, {-2, 1, 0, -2}, 180.000000, 0.000020, 0.042000},
	{2, {-2, 1, 0, 0}, 180.000000, 0.000030, 0.333000},
	{2, {-2, 1, 1, -2}, 180.000000, 0.000010, 0.094000},
	{2, {0, -1, 0, -2}, 180.000000, 0.000030, 0.903000},
	{2, {0, -1, 1, -2}, 180.000000, 0.000020, 0.082000},
	{2, {0, 0, -2, -2}, 180.000000, 0.000160, 0.038000},
	{2, {0, 0, -2, 0}, 0.000000, 0.000090, 4.424000},
	{2, {0, 0, -1, -4}, 0.000000, 0.000010, 0.025000},
	{2, {0, 0, -1, -2}, 180.000000, 0.002390, 0.075000},
	{2, {0, 0, -1, -1}, 270.000000, 0.000010, 16.863000},
	{2, {0, 0, -1, 0}, 180.000000, 0.000800, 0.074000},
	{2, {0, 0, 0, -4}, 0.000000, 0.000070, 0.037000},
	{2, {0, 0, 0, -2}, 180.000000, 0.036380, 9.307000},
	{2, {0, 0, 0, -1}, 90.000000, 0.000030, 0.075000},
	{2, {0, 0, 0, 0}, 180.000000, 0.002390, 0.037000},
	{2, {0, 0, 1, -2}, 180.000000, 0.002410, 0.076000},
	{2, {0, 0, 1, 0}, 180.000000, 0.000250, 0.025000},
	{2, {0, 0, 2, -2}, 180.000000, 0.000170, 0.038000},
	{2, {0, 0, 2, 0}, 180.000000, 0.000020, 0.019000},
	{2, {0, 1, 0, -2}, 0.000000, 0.000020, 1.120000},
	{2, {0, 1, 1, -2}, 0.000000, 0.000010, 0.071000},
	{2, {1, 0, 0, -2}, 0.000000, 0.000020, 0.082000},
	{2, {1, 1, 0, -2}, 180.000000, 0.000040, 0.075000},
	{2, {2, -1, -1, -2}, 180.000000, 0.000020, 0.096000},
	{2, {2, -1, 0, -2}, 180.000000, 0.000040, 0.042000},
	{2, {2, 0, -1, -2}, 180.000000, 0.000390, 0.088000},
	{2, {2, 0, -1, 0}, 180.000000, 0.000060, 0.026000},
	{2, {2, 0, 0, -2}, 180.000000, 0.000570, 0.041000},
	{2, {2, 0, 0, 0}, 180.000000, 0.000030, 0.019000},
	{2, {2, 0, 1, -2}, 180.000000, 0.000070, 0.026000},
	{2, {4, 0, -1, -2}, 180.000000, 0.000010, 0.028000}
};

static const struct earth_pert earth_pert_elp5 [ELP5_SIZE] = 
{
	{0, {0, 0, 0, 3}, 0.000000, 0.000030, 0.025000},
	{0, {0, 0, 1, -3}, 180.000000, 0.000210, 0.037000},
	{0, {0, 0, 1, -1}, 0.000000, 0.000560, 5.997000},
	{0, {0, 0, 1, 1}, 0.000000, 0.000470, 0.037000},
	{0, {0, 0, 2, -3}, 0.000000, 0.000040, 0.073000},
	{0, {0, 0, 2, -1}, 180.000000, 0.001440, 0.076000},
	{0, {0, 0, 2, 1}, 0.000000, 0.000030, 0.025000},
	{0, {0, 0, 3, -1}, 180.000000, 0.000070, 0.038000},
	{0, {0, 1, -1, -1}, 180.000000, 0.000020, 0.039000},
	{0, {0, 1, -1, 1}, 180.000000, 0.000020, 0.857000},
	{0, {0, 1, 0, -1}, 180.000000, 0.000040, 0.081000},
	{0, {0, 1, 0, 1}, 180.000000, 0.000040, 0.069000},
	{0, {0, 1, 1, -1}, 180.000000, 0.000010, 1.200000},
	{0, {0, 1, 1, 1}, 180.000000, 0.000020, 0.036000},
	{0, {1, 0, 0, -1}, 180.000000, 0.000040, 0.949000},
	{0, {1, 0, 0, 1}, 180.000000, 0.000050, 0.039000},
	{0, {1, 1, -1, -1}, 0.000000, 0.000010, 0.075000},
	{0, {1, 1, -1, 1}, 0.000000, 0.000010, 0.074000},
	{0, {1, 1, 0, -1}, 180.000000, 0.000030, 18.600000},
	{0, {1, 1, 0, 1}, 180.000000, 0.000030, 0.037000},
	{0, {2, -1, -1, -1}, 0.000000, 0.000030, 0.340000},
	{0, {2, -1, -1, 1}, 0.000000, 0.000040, 0.042000},
	{0, {2, -1, 0, -1}, 180.000000, 0.000060, 0.097000},
	{0, {2, 0, -2, -1}, 180.000000, 0.000050, 0.066000},
	{0, {2, 0, -2, 1}, 180.000000, 0.000210, 0.086000},
	{0, {2, 0, -1, -1}, 0.000000, 0.001260, 0.515000},
	{0, {2, 0, -1, 1}, 0.000000, 0.001080, 0.040000},
	{0, {2, 0, 0, -3}, 0.000000, 0.000040, 0.064000},
	{0, {2, 0, 0, -1}, 180.000000, 0.002170, 0.088000},
	{0, {2, 0, 0, 1}, 0.000000, 0.000130, 0.026000},
	{0, {2, 0, 1, -3}, 0.000000, 0.000030, 0.440000},
	{0, {2, 0, 1, -1}, 180.000000, 0.000080, 0.041000},
	{0, {2, 0, 2, -1}, 180.000000, 0.000020, 0.026000},
	{0, {2, 1, -1, -1}, 180.000000, 0.000010, 1.063000},
	{0, {2, 1, 0, -1}, 0.000000, 0.000090, 0.081000},
	{0, {2, 2, 0, -1}, 0.000000, 0.000010, 0.075000},
	{0, {4, 0, -2, 1}, 0.000000, 0.000030, 0.027000},
	{0, {4, 0, -1, -1}, 0.000000, 0.000010, 0.044000},
	{0, {4, 0, -1, 1}, 0.000000, 0.000020, 0.020000},
	{0, {4, 0, 0, -1}, 180.000000, 0.000010, 0.028000},
	{1, {-6, 0, 0, 0}, 0.000760, 0.000020, 0.016000},
	{1, {-6, 0, 1, 0}, 0.000750, 0.000050, 0.021000},
	{1, {-6, 0, 2, 0}, 0.000750, 0.000040, 0.029000},
	{1, {-4, -1, 0, 0}, 180.000750, 0.000060, 0.027000},
	{1, {-4, -1, 1, 0}, 180.000740, 0.000080, 0.042000},
	{1, {-4, 0, -2, 0}, 0.000760, 0.000030, 0.016000},
	{1, {-4, 0, -1, -2}, 0.000950, 0.000020, 0.013000},
	{1, {-4, 0, -1, 0}, 0.000750, 0.000250, 0.020000},
	{1, {-4, 0, -1, 2}, 180.000460, 0.000010, 0.044000},
	{1, {-4, 0, 0, -2}, 0.000940, 0.000090, 0.016000},
	{1, {-4, 0, 0, 0}, 0.000730, 0.001880, 0.028000},
	{1, {-4, 0, 0, 2}, 0.000720, 0.000090, 0.108000},
	{1, {-4, 0, 1, -2}, 0.000830, 0.000170, 0.020000},
	{1, {-4, 0, 1, 0}, 0.000710, 0.003220, 0.044000},
	{1, {-4, 0, 2, -2}, 0.000550, 0.000090, 0.027000},
	{1, {-4, 0, 2, 0}, 0.000700, 0.000290, 0.104000},
	{1, {-4, 1, -1, 0}, 0.000750, 0.000030, 0.021000},
	{1, {-4, 1, 0, -2}, 0.000940, 0.000010, 0.016000},
	{1, {-4, 1, 0, 0}, 0.000750, 0.000210, 0.028000},
	{1, {-4, 1, 1, -2}, 0.000940, 0.000020, 0.021000},
	{1, {-4, 1, 1, 0}, 0.000720, 0.000290, 0.046000},
	{1, {-4, 1, 2, -2}, 0.000930, 0.000010, 0.028000},
	{1, {-4, 1, 2, 0}, 0.000720, 0.000020, 0.116000},
	{1, {-4, 2, 0, 0}, 0.000720, 0.000010, 0.029000},
	{1, {-4, 2, 1, 0}, 0.000720, 0.000020, 0.048000},
	{1, {-3, -1, 0, 0}, 0.000720, 0.000020, 0.040000},
	{1, {-3, 0, -1, 0}, 180.000720, 0.000020, 0.027000},
	{1, {-3, 0, 0, 0}, 180.000740, 0.000170, 0.042000},
	{1, {-3, 0, 0, 2}, 180.000720, 0.000010, 0.322000},
	{1, {-3, 0, 1, 0}, 180.000730, 0.000140, 0.095000},
	{1, {-3, 0, 2, 0}, 180.000720, 0.000020, 0.360000},
	{1, {-3, 1, 0, 0}, 180.000720, 0.000010, 0.044000},
	{1, {-2, -2, 0, 0}, 180.000370, 0.000130, 0.075000},
	{1, {-2, -2, 1, 0}, 180.000720, 0.000050, 8.853000},
	{1, {-2, -1, -2, 0}, 180.000750, 0.000030, 0.026000},
	{1, {-2, -1, -1, -2}, 180.000950, 0.000010, 0.019000},
	{1, {-2, -1, -1, 0}, 180.000730, 0.000410, 0.039000},
	{1, {-2, -1, 0, -2}, 180.000940, 0.000060, 0.026000},
	{1, {-2, -1, 0, 0}, 180.000710, 0.005890, 0.081000},
	{1, {-2, -1, 0, 2}, 180.000790, 0.000070, 0.069000},
	{1, {-2, -1, 1, -2}, 180.000660, 0.000050, 0.039000},
	{1, {-2, -1, 1, 0}, 180.000740, 0.000340, 1.127000},
	{1, {-2, -1, 2, 0}, 0.000720, 0.000030, 0.071000},
	{1, {-2, 0, -3, -2}, 0.000940, 0.000010, 0.013000},
	{1, {-2, 0, -3, 0}, 0.000770, 0.000070, 0.020000},
	{1, {-2, 0, -2, -2}, 0.001110, 0.000140, 0.015000},
	{1, {-2, 0, -2, 0}, 0.000770, 0.001100, 0.026000},
	{1, {-2, 0, -2, 2}, 180.000470, 0.000020, 0.091000},
	{1, {-2, 0, -1, -2}, 0.001040, 0.001040, 0.019000},
	{1, {-2, 0, -1, 0}, 0.000750, 0.016470, 0.041000},
	{1, {-2, 0, -1, 2}, 180.000590, 0.000480, 0.450000},
	{1, {-2, 0, 0, -4}, 180.000900, 0.000010, 0.015000},
	{1, {-2, 0, 0, -2}, 0.000930, 0.005620, 0.026000},
	{1, {-2, 0, 0, 0}, 0.000720, 0.298540, 0.088000},
	{1, {-2, 0, 0, 2}, 0.000770, 0.002730, 0.065000},
	{1, {-2, 0, 1, -4}, 180.000880, 0.000020, 0.019000},
	{1, {-2, 0, 1, -2}, 0.000650, 0.005620, 0.040000},
	{1, {-2, 0, 1, 0}, 0.000710, 0.073320, 0.530000},
	{1, {-2, 0, 1, 2}, 0.000720, 0.000430, 0.035000},
	{1, {-2, 0, 2, -2}, 179.999510, 0.000020, 0.085000},
	{1, {-2, 0, 2, 0}, 0.000660, 0.006720, 0.066000},
	{1, {-2, 0, 2, 2}, 0.000720, 0.000030, 0.024000},
	{1, {-2, 0, 3, 0}, 0.000630, 0.000630, 0.035000},
	{1, {-2, 0, 4, 0}, 0.000610, 0.000050, 0.024000},
	{1, {-2, 1, -2, -2}, 0.000940, 0.000010, 0.016000},
	{1, {-2, 1, -2, 0}, 0.000750, 0.000060, 0.027000},
	{1, {-2, 1, -1, -2}, 0.001040, 0.000080, 0.020000},
	{1, {-2, 1, -1, 0}, 0.000750, 0.000860, 0.042000},
	{1, {-2, 1, -1, 2}, 180.000600, 0.000030, 0.311000},
	{1, {-2, 1, 0, -2}, 0.000940, 0.000380, 0.027000},
	{1, {-2, 1, 0, 0}, 0.000720, 0.014040, 0.096000},
	{1, {-2, 1, 0, 2}, 0.000760, 0.000110, 0.061000},
	{1, {-2, 1, 1, -2}, 0.000650, 0.000250, 0.042000},
	{1, {-2, 1, 1, 0}, 0.000710, 0.003280, 0.346000},
	{1, {-2, 1, 1, 2}, 0.000730, 0.000010, 0.034000},
	{1, {-2, 1, 2, 0}, 0.000680, 0.000280, 0.062000},
	{1, {-2, 1, 3, 0}, 0.000720, 0.000020, 0.034000},
	{1, {-2, 2, -1, 0}, 0.000740, 0.000030, 0.044000},
	{1, {-2, 2, 0, -2}, 0.000930, 0.000020, 0.028000},
	{1, {-2, 2, 0, 0}, 0.000730, 0.000510, 0.107000},
	{1, {-2, 2, 1, -2}, 0.000910, 0.000010, 0.044000},
	{1, {-2, 2, 1, 0}, 0.000720, 0.000120, 0.257000},
	{1, {-2, 2, 2, 0}, 0.000720, 0.000010, 0.058000},
	{1, {-2, 3, 0, 0}, 0.000720, 0.000020, 0.119000},
	{1, {-1, -1, -1, 0}, 0.000720, 0.000020, 0.075000},
	{1, {-1, -1, 0, -2}, 0.000940, 0.000020, 0.037000},
	{1, {-1, -1, 0, 0}, 0.000720, 0.000360, 25770.018000},
	{1, {-1, 0, -2, 0}, 180.000720, 0.000020, 0.039000},
	{1, {-1, 0, -1, -2}, 180.001120, 0.000030, 0.026000},
	{1, {-1, 0, -1, 0}, 180.000720, 0.000290, 0.082000},
	{1, {-1, 0, 0, -2}, 180.000990, 0.000130, 0.039000},
	{1, {-1, 0, 0, 0}, 180.000730, 0.002120, 1.000000},
	{1, {-1, 0, 0, 2}, 180.000690, 0.000050, 0.036000},
	{1, {-1, 0, 1, 0}, 180.000740, 0.000170, 0.070000},
	{1, {-1, 0, 2, 0}, 180.000710, 0.000040, 0.036000},
	{1, {0, -2, -1, 0}, 180.000720, 0.000030, 0.530000},
	{1, {0, -2, 1, 0}, 180.000710, 0.000050, 0.041000},
	{1, {0, -1, -3, 0}, 180.000720, 0.000010, 0.036000},
	{1, {0, -1, -2, -2}, 180.001190, 0.000040, 0.024000},
	{1, {0, -1, -2, 0}, 180.000800, 0.000140, 0.071000},
	{1, {0, -1, -1, -2}, 180.001140, 0.000180, 0.036000},
	{1, {0, -1, -1, 0}, 180.000760, 0.002270, 1.127000},
	{1, {0, -1, 0, -2}, 180.001920, 0.000020, 0.069000},
	{1, {0, -1, 0, 0}, 180.000580, 0.002520, 0.081000},
	{1, {0, -1, 1, -2}, 0.002020, 0.000040, 0.819000},
	{1, {0, -1, 1, 0}, 180.000680, 0.002760, 0.039000},
	{1, {0, -1, 2, 0}, 180.000650, 0.000310, 0.026000},
	{1, {0, -1, 3, 0}, 180.000690, 0.000030, 0.019000},
	{1, {0, 0, -4, -2}, 0.001210, 0.000030, 0.015000},
	{1, {0, 0, -4, 0}, 0.000640, 0.000050, 0.025000},
	{1, {0, 0, -3, -2}, 0.001190, 0.000330, 0.019000},
	{1, {0, 0, -3, 0}, 0.000690, 0.000910, 0.038000},
	{1, {0, 0, -2, -4}, 180.000890, 0.000010, 0.015000},
	{1, {0, 0, -2, -2}, 0.001160, 0.003770, 0.025000},
	{1, {0, 0, -2, 0}, 0.000670, 0.017450, 0.076000},
	{1, {0, 0, -2, 2}, 180.000380, 0.000330, 0.073000},
	{1, {0, 0, -1, -4}, 180.000980, 0.000080, 0.019000},
	{1, {0, 0, -1, -2}, 0.001090, 0.039770, 0.037000},
	{1, {0, 0, -1, 0}, 0.000750, 0.455860, 8.847000},
	{1, {0, 0, -1, 2}, 0.000620, 0.004370, 0.037000},
	{1, {0, 0, 0, -4}, 180.000860, 0.000340, 0.025000},
	{1, {0, 0, 0, -2}, 0.000860, 0.326220, 0.074000},
	{1, {0, 0, 0, -1}, 270.000000, 0.000010, 18.613000},
	{1, {0, 0, 0, 0}, 180.000710, 8.045080, 0.075000},
	{1, {0, 0, 0, 1}, 270.000000, 0.000010, 0.037000},
	{1, {0, 0, 0, 2}, 0.000710, 0.008340, 0.025000},
	{1, {0, 0, 0, 4}, 180.000320, 0.000020, 0.015000},
	{1, {0, 0, 1, -4}, 180.000670, 0.000090, 0.037000},
	{1, {0, 0, 1, -2}, 0.004640, 0.002390, 4.536000},
	{1, {0, 0, 1, 0}, 180.000690, 0.415710, 0.038000},
	{1, {0, 0, 1, 2}, 0.000700, 0.001330, 0.019000},
	{1, {0, 0, 2, -2}, 0.001470, 0.000790, 0.077000},
	{1, {0, 0, 2, 0}, 180.000660, 0.024030, 0.025000},
	{1, {0, 0, 2, 2}, 0.000710, 0.000150, 0.015000},
	{1, {0, 0, 3, -2}, 0.001370, 0.000070, 0.038000},
	{1, {0, 0, 3, 0}, 180.000620, 0.001460, 0.019000},
	{1, {0, 0, 3, 2}, 0.000750, 0.000020, 0.013000},
	{1, {0, 0, 4, 0}, 180.000580, 0.000090, 0.015000},
	{1, {0, 1, -3, 0}, 0.000720, 0.000010, 0.039000},
	{1, {0, 1, -2, -2}, 0.001190, 0.000040, 0.026000},
	{1, {0, 1, -2, 0}, 0.000770, 0.000140, 0.082000},
	{1, {0, 1, -1, -2}, 0.001140, 0.000230, 0.039000},
	{1, {0, 1, -1, 0}, 0.000740, 0.002570, 0.898000},
	{1, {0, 1, -1, 2}, 0.000870, 0.000010, 0.036000},
	{1, {0, 1, 0, -2}, 0.000710, 0.000030, 0.080000},
	{1, {0, 1, 0, 0}, 0.000950, 0.002060, 0.070000},
	{1, {0, 1, 0, 2}, 180.000720, 0.000020, 0.024000},
	{1, {0, 1, 1, -2}, 180.002170, 0.000030, 1.283000},
	{1, {0, 1, 1, 0}, 0.000700, 0.002120, 0.036000},
	{1, {0, 1, 2, 0}, 0.000650, 0.000250, 0.024000},
	{1, {0, 1, 3, 0}, 0.000690, 0.000020, 0.018000},
	{1, {0, 2, -1, 0}, 0.000720, 0.000040, 0.473000},
	{1, {0, 2, 0, 0}, 0.000880, 0.000020, 0.065000},
	{1, {0, 2, 1, 0}, 0.000720, 0.000020, 0.035000},
	{1, {1, 0, -2, 0}, 0.000720, 0.000040, 1.292000},
	{1, {1, 0, -1, 0}, 180.001370, 0.000030, 0.080000},
	{1, {1, 0, 0, -2}, 180.001020, 0.000030, 0.903000},
	{1, {1, 0, 0, 0}, 0.000710, 0.002180, 0.039000},
	{1, {1, 0, 1, -2}, 180.001310, 0.000010, 0.082000},
	{1, {1, 0, 1, 0}, 0.000670, 0.000260, 0.026000},
	{1, {1, 0, 2, 0}, 0.000680, 0.000020, 0.019000},
	{1, {1, 1, -1, 0}, 0.000220, 0.000080, 0.074000},
	{1, {1, 1, 0, -2}, 0.000280, 0.000010, 9.303000},
	{1, {1, 1, 0, 0}, 180.000730, 0.000320, 0.037000},
	{1, {1, 1, 1, 0}, 180.000700, 0.000040, 0.025000},
	{1, {2, -2, -1, 0}, 180.000710, 0.000130, 0.044000},
	{1, {2, -2, 0, -2}, 0.001020, 0.000020, 0.108000},
	{1, {2, -2, 0, 0}, 180.000700, 0.000150, 0.028000},
	{1, {2, -2, 1, 0}, 180.000690, 0.000020, 0.020000},
	{1, {2, -1, -2, -2}, 180.001460, 0.000020, 0.062000},
	{1, {2, -1, -2, 0}, 0.000910, 0.000020, 0.094000},
	{1, {2, -1, -1, -2}, 180.002530, 0.000030, 0.334000},
	{1, {2, -1, -1, 0}, 180.000730, 0.003710, 0.042000},
	{1, {2, -1, -1, 2}, 0.000720, 0.000020, 0.020000},
	{1, {2, -1, 0, -2}, 0.001010, 0.000490, 0.097000},
	{1, {2, -1, 0, 0}, 180.000700, 0.003150, 0.027000},
	{1, {2, -1, 0, 2}, 0.000730, 0.000010, 0.016000},
	{1, {2, -1, 1, -2}, 0.001120, 0.000080, 0.043000},
	{1, {2, -1, 1, 0}, 180.000660, 0.000430, 0.020000},
	{1, {2, -1, 2, 0}, 180.000670, 0.000040, 0.016000},
	{1, {2, 0, -4, 0}, 180.000720, 0.000010, 0.067000},
	{1, {2, 0, -3, -2}, 180.001380, 0.000070, 0.035000},
	{1, {2, 0, -3, 0}, 180.000740, 0.000130, 0.602000},
	{1, {2, 0, -2, -2}, 180.001500, 0.000380, 0.066000},
	{1, {2, 0, -2, 0}, 0.001640, 0.000330, 0.086000},
	{1, {2, 0, -2, 2}, 0.000620, 0.000100, 0.026000},
	{1, {2, 0, -1, -4}, 180.000830, 0.000010, 0.035000},
	{1, {2, 0, -1, -2}, 180.002400, 0.000660, 0.501000},
	{1, {2, 0, -1, 0}, 180.000730, 0.083500, 0.040000},
	{1, {2, 0, -1, 2}, 0.000690, 0.000330, 0.019000},
	{1, {2, 0, 0, -4}, 180.000720, 0.000040, 0.064000},
	{1, {2, 0, 0, -2}, 0.001020, 0.010050, 0.089000},
	{1, {2, 0, 0, 0}, 180.000700, 0.046380, 0.026000},
	{1, {2, 0, 0, 2}, 0.000730, 0.000180, 0.015000},
	{1, {2, 0, 1, -2}, 0.001120, 0.001390, 0.041000},
	{1, {2, 0, 1, 0}, 180.000670, 0.005630, 0.019000},
	{1, {2, 0, 1, 2}, 0.000720, 0.000040, 0.013000},
	{1, {2, 0, 2, -2}, 0.001150, 0.000140, 0.026000},
	{1, {2, 0, 2, 0}, 180.000620, 0.000540, 0.015000},
	{1, {2, 0, 3, -2}, 0.000930, 0.000010, 0.020000},
	{1, {2, 0, 3, 0}, 180.000660, 0.000050, 0.013000},
	{1, {2, 1, -2, 0}, 180.000720, 0.000020, 0.079000},
	{1, {2, 1, -1, -2}, 0.001210, 0.000010, 1.006000},
	{1, {2, 1, -1, 0}, 0.000740, 0.000550, 0.039000},
	{1, {2, 1, 0, -2}, 180.001030, 0.000190, 0.082000},
	{1, {2, 1, 0, 0}, 0.000710, 0.000490, 0.026000},
	{1, {2, 1, 1, -2}, 180.001140, 0.000030, 0.039000},
	{1, {2, 1, 1, 0}, 0.000690, 0.000090, 0.019000},
	{1, {2, 1, 2, 0}, 0.000670, 0.000010, 0.015000},
	{1, {2, 2, -1, 0}, 0.000710, 0.000050, 0.037000},
	{1, {3, 0, -2, 0}, 0.000720, 0.000020, 0.042000},
	{1, {3, 0, -1, 0}, 0.000700, 0.000080, 0.027000},
	{1, {3, 0, 0, 0}, 0.000720, 0.000010, 0.020000},
	{1, {4, -2, -1, 0}, 180.000720, 0.000010, 0.021000},
	{1, {4, -1, -2, 0}, 180.000700, 0.000080, 0.028000},
	{1, {4, -1, -1, -2}, 0.000740, 0.000020, 0.046000},
	{1, {4, -1, -1, 0}, 180.000690, 0.000130, 0.021000},
	{1, {4, -1, 0, -2}, 0.000960, 0.000020, 0.029000},
	{1, {4, -1, 0, 0}, 180.000670, 0.000060, 0.016000},
	{1, {4, -1, 1, 0}, 180.000660, 0.000010, 0.013000},
	{1, {4, 0, -3, 0}, 180.000340, 0.000010, 0.043000},
	{1, {4, 0, -2, 0}, 180.000740, 0.000970, 0.028000},
	{1, {4, 0, -1, -2}, 0.000740, 0.000190, 0.044000},
	{1, {4, 0, -1, 0}, 180.000710, 0.001130, 0.020000},
	{1, {4, 0, -1, 2}, 0.000560, 0.000010, 0.013000},
	{1, {4, 0, 0, -2}, 0.000960, 0.000190, 0.028000},
	{1, {4, 0, 0, 0}, 180.000670, 0.000430, 0.016000},
	{1, {4, 0, 1, -2}, 0.001030, 0.000040, 0.020000},
	{1, {4, 0, 1, 0}, 180.000660, 0.000070, 0.013000},
	{1, {4, 1, -2, 0}, 0.000720, 0.000010, 0.027000},
	{1, {4, 1, -1, 0}, 0.000690, 0.000020, 0.020000},
	{1, {6, 0, -3, 0}, 180.000690, 0.000010, 0.021000},
	{1, {6, 0, -2, 0}, 180.000670, 0.000020, 0.016000},
	{1, {6, 0, -1, 0}, 180.000660, 0.000010, 0.013000},
	{2, {-4, 0, 0, -1}, 180.000000, 0.000020, 0.028000},
	{2, {-4, 0, 1, -1}, 180.000000, 0.000030, 0.044000},
	{2, {-2, -1, 0, -1}, 0.000000, 0.000060, 0.081000},
	{2, {-2, 0, -2, -1}, 180.000000, 0.000010, 0.026000},
	{2, {-2, 0, -1, -1}, 180.000000, 0.000160, 0.041000},
	{2, {-2, 0, 0, -3}, 180.000000, 0.000030, 0.026000},
	{2, {-2, 0, 0, -1}, 180.000000, 0.002930, 0.088000},
	{2, {-2, 0, 0, 1}, 180.000000, 0.000140, 0.065000},
	{2, {-2, 0, 1, -3}, 180.000000, 0.000030, 0.040000},
	{2, {-2, 0, 1, -1}, 180.000000, 0.000720, 0.545000},
	{2, {-2, 0, 1, 1}, 180.000000, 0.000010, 0.035000},
	{2, {-2, 0, 2, -1}, 180.000000, 0.000070, 0.066000},
	{2, {-2, 1, 0, -1}, 180.000000, 0.000140, 0.096000},
	{2, {-2, 1, 1, -1}, 180.000000, 0.000030, 0.353000},
	{2, {-1, 0, 0, -1}, 0.000000, 0.000020, 1.057000},
	{2, {0, -1, -1, -1}, 0.000000, 0.000020, 1.063000},
	{2, {0, -1, 0, -1}, 0.000000, 0.000030, 0.081000},
	{2, {0, -1, 1, -1}, 0.000000, 0.000030, 0.039000},
	{2, {0, 0, -3, -1}, 180.000000, 0.000010, 0.038000},
	{2, {0, 0, -2, -3}, 180.000000, 0.000020, 0.025000},
	{2, {0, 0, -2, -1}, 180.000000, 0.000200, 0.076000},
	{2, {0, 0, -2, 1}, 180.000000, 0.000050, 0.073000},
	{2, {0, 0, -1, -3}, 180.000000, 0.000190, 0.037000},
	{2, {0, 0, -1, -1}, 180.000000, 0.004410, 16.863000},
	{2, {0, 0, -1, 1}, 180.000000, 0.000060, 0.037000},
	{2, {0, 0, 0, -3}, 180.000000, 0.001560, 0.074000},
	{2, {0, 0, 0, -1}, 0.000000, 0.080420, 0.075000},
	{2, {0, 0, 0, 1}, 180.000000, 0.000030, 0.025000},
	{2, {0, 0, 1, -1}, 0.000000, 0.004250, 0.038000},
	{2, {0, 0, 2, -1}, 0.000000, 0.000260, 0.025000},
	{2, {0, 0, 3, -1}, 0.000000, 0.000020, 0.019000},
	{2, {0, 1, -1, -1}, 180.000000, 0.000020, 0.944000},
	{2, {0, 1, 1, -1}, 180.000000, 0.000020, 0.036000},
	{2, {1, 0, 0, -1}, 180.000000, 0.000020, 0.039000},
	{2, {1, 1, -1, -1}, 180.000000, 0.000020, 0.074000},
	{2, {2, -1, -1, -1}, 0.000000, 0.000040, 0.042000},
	{2, {2, -1, 0, -1}, 0.000000, 0.000030, 0.027000},
	{2, {2, 0, -1, -1}, 0.000000, 0.000850, 0.040000},
	{2, {2, 0, 0, -3}, 180.000000, 0.000050, 0.089000},
	{2, {2, 0, 0, -1}, 0.000000, 0.000490, 0.026000},
	{2, {2, 0, 1, -1}, 0.000000, 0.000060, 0.019000},
	{2, {4, 0, -1, -1}, 0.000000, 0.000010, 0.020000}
};

static const struct earth_pert earth_pert_elp6 [ELP6_SIZE] = 
{
	{0, {0, 0, 0, 0}, 90.000000, 0.043010, 99999.999000},
	{0, {0, 0, 0, 1}, 180.000000, 0.000030, 0.075000},
	{0, {0, 0, 0, 2}, 270.000000, 0.000040, 0.037000},
	{0, {0, 0, 1, -2}, 270.000000, 0.004440, 0.074000},
	{0, {0, 0, 1, 0}, 90.000000, 0.005310, 0.075000},
	{0, {0, 0, 2, -2}, 90.000000, 0.000260, 2.998000},
	{0, {0, 0, 2, 0}, 90.000000, 0.000420, 0.038000},
	{0, {0, 0, 2, 2}, 270.000000, 0.000040, 0.019000},
	{0, {0, 0, 3, -2}, 270.000000, 0.000020, 0.077000},
	{0, {0, 0, 3, 0}, 90.000000, 0.000030, 0.025000},
	{0, {0, 1, -2, 0}, 270.000000, 0.000040, 0.039000},
	{0, {0, 1, -1, 0}, 270.000000, 0.000370, 0.082000},
	{0, {0, 1, 0, 0}, 90.000000, 0.000090, 1.000000},
	{0, {0, 1, 1, -2}, 270.000000, 0.000050, 0.079000},
	{0, {0, 1, 1, 0}, 90.000000, 0.000170, 0.070000},
	{0, {0, 1, 2, 0}, 90.000000, 0.000020, 0.036000},
	{0, {0, 2, 0, 0}, 270.000000, 0.000020, 0.500000},
	{0, {1, -1, 0, 0}, 90.000000, 0.000120, 0.088000},
	{0, {1, 0, -1, 0}, 270.000000, 0.000050, 1.127000},
	{0, {1, 0, 0, 0}, 90.000000, 0.000880, 0.081000},
	{0, {1, 0, 1, 0}, 90.000000, 0.000060, 0.039000},
	{0, {1, 1, -1, 0}, 270.000000, 0.000040, 8.850000},
	{0, {1, 1, 0, 0}, 90.000000, 0.000640, 0.075000},
	{0, {1, 1, 1, 0}, 90.000000, 0.000040, 0.038000},
	{0, {2, -1, -1, 0}, 270.000000, 0.000530, 0.095000},
	{0, {2, -1, 1, -2}, 270.000000, 0.000030, 0.099000},
	{0, {2, 0, -3, 0}, 90.000000, 0.000030, 0.067000},
	{0, {2, 0, -2, 0}, 90.000000, 0.000820, 0.564000},
	{0, {2, 0, -2, 2}, 270.000000, 0.000030, 0.040000},
	{0, {2, 0, -1, 0}, 270.000000, 0.019610, 0.087000},
	{0, {2, 0, -1, 2}, 270.000000, 0.000030, 0.026000},
	{0, {2, 0, 0, -2}, 90.000000, 0.000210, 0.474000},
	{0, {2, 0, 0, 0}, 270.000000, 0.000470, 0.040000},
	{0, {2, 0, 1, -2}, 270.000000, 0.000710, 0.090000},
	{0, {2, 0, 2, -2}, 270.000000, 0.000050, 0.041000},
	{0, {2, 1, -1, 0}, 90.000000, 0.000080, 0.080000},
	{0, {2, 1, 0, 0}, 90.000000, 0.000030, 0.039000},
	{0, {2, 2, -1, 0}, 270.000000, 0.000090, 0.074000},
	{0, {4, -1, -1, 0}, 270.000000, 0.000020, 0.028000},
	{0, {4, 0, -2, 0}, 270.000000, 0.000170, 0.044000},
	{0, {4, 0, -1, 0}, 270.000000, 0.000170, 0.028000},
	{0, {4, 0, 0, 0}, 270.000000, 0.000020, 0.020000},
	{1, {-6, 0, 1, -1}, 90.000940, 0.000020, 0.016000},
	{1, {-6, 0, 2, -1}, 90.000700, 0.000030, 0.021000},
	{1, {-4, -1, 1, -1}, 270.000940, 0.000030, 0.027000},
	{1, {-4, 0, -1, -1}, 90.001030, 0.000120, 0.016000},
	{1, {-4, 0, -1, 1}, 270.000270, 0.000050, 0.028000},
	{1, {-4, 0, 0, -1}, 90.000960, 0.000750, 0.020000},
	{1, {-4, 0, 0, 1}, 90.000860, 0.000400, 0.044000},
	{1, {-4, 0, 1, -1}, 90.000790, 0.001490, 0.028000},
	{1, {-4, 0, 1, 1}, 90.000790, 0.000270, 0.106000},
	{1, {-4, 0, 2, -1}, 90.000280, 0.000460, 0.043000},
	{1, {-4, 0, 2, 1}, 90.000670, 0.000080, 0.261000},
	{1, {-4, 1, 0, -1}, 90.000950, 0.000100, 0.021000},
	{1, {-4, 1, 0, 1}, 90.000720, 0.000050, 0.046000},
	{1, {-4, 1, 1, -1}, 90.000790, 0.000170, 0.028000},
	{1, {-4, 1, 1, 1}, 90.000720, 0.000020, 0.119000},
	{1, {-4, 1, 2, -1}, 90.000290, 0.000040, 0.045000},
	{1, {-3, 0, 0, -1}, 90.000920, 0.000080, 0.027000},
	{1, {-3, 0, 0, 1}, 270.000720, 0.000180, 0.097000},
	{1, {-3, 0, 1, -1}, 270.000690, 0.000070, 0.042000},
	{1, {-2, -2, 1, -1}, 270.000240, 0.000020, 0.074000},
	{1, {-2, -1, -1, -1}, 270.001090, 0.000140, 0.026000},
	{1, {-2, -1, -1, 1}, 90.000590, 0.000060, 0.082000},
	{1, {-2, -1, 0, -1}, 270.000970, 0.000960, 0.039000},
	{1, {-2, -1, 0, 1}, 90.000720, 0.000120, 0.949000},
	{1, {-2, -1, 1, -1}, 270.000470, 0.000440, 0.080000},
	{1, {-2, -1, 1, 1}, 90.000650, 0.000050, 0.070000},
	{1, {-2, -1, 2, -1}, 90.000720, 0.000020, 1.388000},
	{1, {-2, 0, -3, -1}, 90.001170, 0.000070, 0.015000},
	{1, {-2, 0, -3, 1}, 270.000470, 0.000030, 0.027000},
	{1, {-2, 0, -2, -1}, 90.001130, 0.000800, 0.019000},
	{1, {-2, 0, -2, 1}, 270.000440, 0.000440, 0.041000},
	{1, {-2, 0, -1, -1}, 90.001080, 0.009250, 0.026000},
	{1, {-2, 0, -1, 1}, 270.000460, 0.006170, 0.089000},
	{1, {-2, 0, -1, 3}, 90.000740, 0.000030, 0.064000},
	{1, {-2, 0, 0, -1}, 90.000960, 0.095540, 0.040000},
	{1, {-2, 0, 0, 0}, 180.000000, 0.000050, 0.088000},
	{1, {-2, 0, 0, 1}, 270.000790, 0.008540, 0.487000},
	{1, {-2, 0, 1, -3}, 270.000880, 0.000030, 0.026000},
	{1, {-2, 0, 1, -1}, 90.000280, 0.038550, 0.087000},
	{1, {-2, 0, 1, 1}, 270.000700, 0.007850, 0.065000},
	{1, {-2, 0, 2, -3}, 270.000300, 0.000020, 0.040000},
	{1, {-2, 0, 2, -1}, 90.002160, 0.002800, 0.581000},
	{1, {-2, 0, 2, 1}, 270.000710, 0.000400, 0.035000},
	{1, {-2, 0, 3, -1}, 90.001590, 0.000470, 0.067000},
	{1, {-2, 0, 3, 1}, 270.000910, 0.000020, 0.024000},
	{1, {-2, 0, 4, -1}, 90.001490, 0.000040, 0.035000},
	{1, {-2, 1, -2, -1}, 90.001120, 0.000070, 0.020000},
	{1, {-2, 1, -2, 1}, 270.000500, 0.000020, 0.043000},
	{1, {-2, 1, -1, -1}, 90.001090, 0.000690, 0.027000},
	{1, {-2, 1, -1, 1}, 270.000480, 0.000280, 0.098000},
	{1, {-2, 1, 0, -1}, 90.000960, 0.006570, 0.042000},
	{1, {-2, 1, 0, 1}, 270.000770, 0.000560, 0.327000},
	{1, {-2, 1, 1, -1}, 90.000290, 0.001600, 0.095000},
	{1, {-2, 1, 1, 1}, 270.000710, 0.000290, 0.061000},
	{1, {-2, 1, 2, -1}, 90.002270, 0.000110, 0.368000},
	{1, {-2, 2, -1, -1}, 90.000940, 0.000030, 0.028000},
	{1, {-2, 2, 0, -1}, 90.000950, 0.000320, 0.044000},
	{1, {-2, 2, 0, 1}, 270.000720, 0.000030, 0.247000},
	{1, {-2, 2, 1, -1}, 90.000300, 0.000050, 0.105000},
	{1, {-1, -1, -1, -1}, 90.001210, 0.000030, 0.037000},
	{1, {-1, -1, 0, -1}, 90.001070, 0.000180, 0.075000},
	{1, {-1, -1, 0, 1}, 270.000520, 0.000040, 0.075000},
	{1, {-1, 0, -1, -1}, 270.001280, 0.000180, 0.039000},
	{1, {-1, 0, -1, 1}, 270.000720, 0.000050, 0.857000},
	{1, {-1, 0, 0, -1}, 270.001370, 0.000630, 0.081000},
	{1, {-1, 0, 0, 1}, 90.000680, 0.000750, 0.069000},
	{1, {-1, 0, 1, -1}, 270.001230, 0.000140, 1.200000},
	{1, {-1, 0, 1, 1}, 90.000720, 0.000040, 0.036000},
	{1, {-1, 0, 2, -1}, 270.001290, 0.000060, 0.071000},
	{1, {-1, 1, 0, -1}, 269.999920, 0.000030, 0.088000},
	{1, {0, -2, -1, -1}, 270.001370, 0.000020, 0.065000},
	{1, {0, -2, 1, -1}, 270.001150, 0.000060, 0.089000},
	{1, {0, -1, -3, -1}, 270.001270, 0.000020, 0.024000},
	{1, {0, -1, -2, -1}, 270.001340, 0.000220, 0.036000},
	{1, {0, -1, -2, 1}, 270.000730, 0.000030, 1.389000},
	{1, {0, -1, -1, -1}, 270.001400, 0.001810, 0.070000},
	{1, {0, -1, -1, 1}, 90.001620, 0.000100, 0.080000},
	{1, {0, -1, 0, -1}, 90.000670, 0.000280, 0.949000},
	{1, {0, -1, 0, 1}, 90.000870, 0.000130, 0.039000},
	{1, {0, -1, 1, -1}, 270.001180, 0.003410, 0.082000},
	{1, {0, -1, 1, 1}, 270.000000, 0.000030, 0.026000},
	{1, {0, -1, 2, -1}, 270.001210, 0.000340, 0.039000},
	{1, {0, -1, 3, -1}, 270.001270, 0.000030, 0.026000},
	{1, {0, 0, -4, -1}, 90.001280, 0.000100, 0.019000},
	{1, {0, 0, -3, -1}, 90.001280, 0.001500, 0.025000},
	{1, {0, 0, -3, 1}, 270.000000, 0.000060, 0.077000},
	{1, {0, 0, -2, -1}, 90.001280, 0.024690, 0.038000},
	{1, {0, 0, -2, 1}, 90.000620, 0.004850, 3.574000},
	{1, {0, 0, -1, -1}, 90.001280, 0.452760, 0.075000},
	{1, {0, 0, -1, 1}, 270.000610, 0.085080, 0.074000},
	{1, {0, 0, 0, -3}, 90.000670, 0.000120, 0.037000},
	{1, {0, 0, 0, -1}, 90.000010, 0.028090, 18.613000},
	{1, {0, 0, 0, 0}, 180.000000, 0.000250, 0.075000},
	{1, {0, 0, 0, 1}, 90.000500, 0.003900, 0.037000},
	{1, {0, 0, 1, -3}, 270.000370, 0.001080, 0.073000},
	{1, {0, 0, 1, -1}, 270.001260, 0.456480, 0.076000},
	{1, {0, 0, 1, 1}, 90.000240, 0.000250, 0.025000},
	{1, {0, 0, 2, -3}, 270.002800, 0.000040, 2.582000},
	{1, {0, 0, 2, -1}, 270.001260, 0.024880, 0.038000},
	{1, {0, 0, 2, 1}, 90.000000, 0.000020, 0.019000},
	{1, {0, 0, 3, -1}, 270.001260, 0.001530, 0.025000},
	{1, {0, 0, 3, 1}, 270.000000, 0.000020, 0.015000},
	{1, {0, 0, 4, -1}, 270.001260, 0.000100, 0.019000},
	{1, {0, 1, -3, -1}, 90.001270, 0.000030, 0.026000},
	{1, {0, 1, -2, -1}, 90.001340, 0.000270, 0.039000},
	{1, {0, 1, -2, 1}, 90.000720, 0.000050, 0.781000},
	{1, {0, 1, -1, -1}, 90.001430, 0.002180, 0.081000},
	{1, {0, 1, -1, 1}, 270.000840, 0.000250, 0.069000},
	{1, {0, 1, 0, -1}, 90.000800, 0.000240, 1.057000},
	{1, {0, 1, 0, 1}, 90.000640, 0.000150, 0.036000},
	{1, {0, 1, 1, -1}, 90.001160, 0.002790, 0.070000},
	{1, {0, 1, 2, -1}, 90.001210, 0.000280, 0.036000},
	{1, {0, 1, 3, -1}, 90.001270, 0.000020, 0.025000},
	{1, {0, 2, -1, -1}, 90.001470, 0.000030, 0.088000},
	{1, {0, 2, 1, -1}, 90.001200, 0.000030, 0.066000},
	{1, {1, -1, 0, -1}, 90.000800, 0.000020, 0.088000},
	{1, {1, 0, -2, -1}, 90.001800, 0.000030, 0.070000},
	{1, {1, 0, -1, -1}, 269.999530, 0.000060, 1.063000},
	{1, {1, 0, -1, 1}, 90.000720, 0.000020, 0.039000},
	{1, {1, 0, 0, -1}, 90.000880, 0.002990, 0.081000},
	{1, {1, 0, 0, 1}, 90.000510, 0.000020, 0.026000},
	{1, {1, 0, 1, -1}, 90.001040, 0.000310, 0.039000},
	{1, {1, 0, 2, -1}, 90.000900, 0.000020, 0.026000},
	{1, {1, 1, 0, -1}, 270.000580, 0.000590, 0.075000},
	{1, {1, 1, 1, -1}, 270.000850, 0.000050, 0.038000},
	{1, {2, -2, -1, -1}, 270.000410, 0.000070, 0.106000},
	{1, {2, -2, 0, -1}, 270.000930, 0.000360, 0.044000},
	{1, {2, -2, 1, -1}, 270.000940, 0.000030, 0.028000},
	{1, {2, -1, -2, -1}, 270.002810, 0.000080, 0.354000},
	{1, {2, -1, -2, 1}, 270.000590, 0.000030, 0.042000},
	{1, {2, -1, -1, -1}, 270.000420, 0.002180, 0.096000},
	{1, {2, -1, -1, 1}, 270.000390, 0.000060, 0.027000},
	{1, {2, -1, 0, -1}, 270.000920, 0.007450, 0.042000},
	{1, {2, -1, 0, 1}, 270.000000, 0.000020, 0.020000},
	{1, {2, -1, 1, -1}, 270.001050, 0.000750, 0.027000},
	{1, {2, -1, 2, -1}, 270.001120, 0.000070, 0.020000},
	{1, {2, 0, -4, -1}, 270.001480, 0.000040, 0.035000},
	{1, {2, 0, -3, -1}, 270.001680, 0.000420, 0.066000},
	{1, {2, 0, -3, 1}, 90.000240, 0.000060, 0.085000},
	{1, {2, 0, -2, -3}, 270.001220, 0.000020, 0.035000},
	{1, {2, 0, -2, -1}, 270.003000, 0.001770, 0.547000},
	{1, {2, 0, -2, 1}, 270.000640, 0.000780, 0.040000},
	{1, {2, 0, -1, -3}, 270.001200, 0.000180, 0.065000},
	{1, {2, 0, -1, -1}, 270.000430, 0.053430, 0.088000},
	{1, {2, 0, -1, 1}, 270.000620, 0.000590, 0.026000},
	{1, {2, 0, 0, -1}, 270.000920, 0.106890, 0.041000},
	{1, {2, 0, 0, 1}, 270.000950, 0.000020, 0.019000},
	{1, {2, 0, 1, -3}, 90.001070, 0.000100, 0.090000},
	{1, {2, 0, 1, -1}, 270.001050, 0.009920, 0.026000},
	{1, {2, 0, 2, -1}, 270.001110, 0.000840, 0.020000},
	{1, {2, 0, 3, -1}, 270.001150, 0.000070, 0.016000},
	{1, {2, 1, -2, -1}, 90.000720, 0.000020, 1.208000},
	{1, {2, 1, -1, -1}, 90.000080, 0.000160, 0.080000},
	{1, {2, 1, 0, -1}, 90.000920, 0.001150, 0.039000},
	{1, {2, 1, 1, -1}, 90.001040, 0.000160, 0.026000},
	{1, {2, 2, -1, -1}, 90.000330, 0.000050, 0.074000},
	{1, {3, 0, -1, -1}, 90.000700, 0.000120, 0.042000},
	{1, {3, 0, 0, -1}, 270.000960, 0.000070, 0.027000},
	{1, {4, -1, -2, -1}, 270.000420, 0.000060, 0.046000},
	{1, {4, -1, -1, -1}, 270.000780, 0.000200, 0.028000},
	{1, {4, -1, 0, -1}, 270.000930, 0.000110, 0.021000},
	{1, {4, 0, -2, -1}, 270.000420, 0.000630, 0.044000},
	{1, {4, 0, -1, -1}, 270.000780, 0.001760, 0.028000},
	{1, {4, 0, 0, -1}, 270.000930, 0.000840, 0.020000},
	{1, {4, 0, 1, -1}, 270.001050, 0.000130, 0.016000},
	{1, {4, 1, -1, -1}, 90.000940, 0.000030, 0.027000},
	{1, {6, 0, -2, -1}, 270.000700, 0.000030, 0.021000},
	{1, {6, 0, -1, -1}, 270.000940, 0.000020, 0.016000},
	{2, {-2, 0, -1, -2}, 270.000000, 0.000050, 0.026000},
	{2, {-2, 0, -1, 0}, 90.000000, 0.000090, 0.089000},
	{2, {-2, 0, 0, -2}, 270.000000, 0.000470, 0.040000},
	{2, {-2, 0, 0, 0}, 90.000000, 0.000040, 0.500000},
	{2, {-2, 0, 1, -2}, 270.000000, 0.000190, 0.086000},
	{2, {-2, 1, 0, -2}, 270.000000, 0.000040, 0.042000},
	{2, {0, -1, 0, 0}, 90.000000, 0.000020, 0.039000},
	{2, {0, 0, -2, -2}, 270.000000, 0.000120, 0.038000},
	{2, {0, 0, -2, 0}, 270.000000, 0.000050, 4.424000},
	{2, {0, 0, -1, -2}, 270.000000, 0.002190, 0.075000},
	{2, {0, 0, -1, 0}, 90.000000, 0.000770, 0.074000},
	{2, {0, 0, 0, -2}, 270.000000, 0.000280, 9.307000},
	{2, {0, 0, 0, -1}, 0.000000, 0.000030, 0.075000},
	{2, {0, 0, 0, 0}, 90.000000, 0.004480, 0.037000},
	{2, {0, 0, 1, -2}, 90.000000, 0.002230, 0.076000},
	{2, {0, 0, 1, 0}, 90.000000, 0.000500, 0.025000},
	{2, {0, 0, 2, -2}, 90.000000, 0.000120, 0.038000},
	{2, {0, 0, 2, 0}, 90.000000, 0.000060, 0.019000},
	{2, {1, 0, 0, -2}, 270.000000, 0.000020, 0.082000},
	{2, {1, 0, 0, 0}, 90.000000, 0.000030, 0.026000},
	{2, {1, 1, 0, -2}, 90.000000, 0.000040, 0.075000},
	{2, {2, -1, 0, -2}, 90.000000, 0.000040, 0.042000},
	{2, {2, 0, -1, -2}, 90.000000, 0.000320, 0.088000},
	{2, {2, 0, -1, 0}, 90.000000, 0.000090, 0.026000},
	{2, {2, 0, 0, -2}, 90.000000, 0.000560, 0.041000},
	{2, {2, 0, 0, 0}, 90.000000, 0.000050, 0.019000},
	{2, {2, 0, 1, -2}, 90.000000, 0.000050, 0.026000}
};

static const struct earth_pert earth_pert_elp7 [ELP7_SIZE] = 
{
	{1, {-2, 0, 0, -1}, 180.000000, 0.000030, 0.040000},
	{1, {-2, 0, 0, 1}, 180.000000, 0.000020, 0.487000},
	{1, {-2, 0, 1, -1}, 180.000000, 0.000020, 0.087000},
	{1, {0, 0, -2, -1}, 180.000000, 0.000010, 0.038000},
	{1, {0, 0, -1, -1}, 180.000000, 0.000210, 0.075000},
	{1, {0, 0, -1, 1}, 180.000000, 0.000020, 0.074000},
	{1, {0, 0, 0, -1}, 180.000000, 0.003000, 18.613000},
	{1, {0, 0, 0, 1}, 180.000000, 0.000150, 0.037000},
	{1, {0, 0, 1, -1}, 180.000000, 0.000210, 0.076000},
	{1, {0, 0, 1, 1}, 180.000000, 0.000020, 0.025000},
	{1, {0, 0, 2, -1}, 180.000000, 0.000020, 0.038000},
	{1, {2, 0, -1, -1}, 180.000000, 0.000030, 0.088000},
	{1, {2, 0, 0, -1}, 180.000000, 0.000040, 0.041000},
	{2, {0, 0, 0, -2}, 0.000000, 0.000040, 9.307000}
};

static const struct earth_pert earth_pert_elp8 [ELP8_SIZE] =
{
	{1, {-2, 0, 0, 0}, 180.000000, 0.000120, 0.088000},
	{1, {-2, 0, 1, 0}, 180.000000, 0.000030, 0.530000},
	{1, {0, 0, -1, -2}, 180.000000, 0.000010, 0.037000},
	{1, {0, 0, -1, 0}, 180.000000, 0.000190, 8.847000},
	{1, {0, 0, 0, -2}, 180.000000, 0.000140, 0.074000},
	{1, {0, 0, 0, 0}, 0.000000, 0.003420, 0.075000},
	{1, {0, 0, 1, 0}, 0.000000, 0.000180, 0.038000},
	{1, {0, 0, 2, 0}, 0.000000, 0.000010, 0.025000},
	{1, {2, 0, -1, 0}, 0.000000, 0.000040, 0.040000},
	{1, {2, 0, 0, 0}, 0.000000, 0.000020, 0.026000},
	{2, {0, 0, 0, -1}, 180.000000, 0.000090, 0.075000}
};

static const struct earth_pert earth_pert_elp9 [ELP9_SIZE] = 
{
	{0, {0, 0, 0, 0}, 270.000000, 0.000040, 99999.999000},
	{1, {-2, 0, 0, -1}, 270.000000, 0.000040, 0.040000},
	{1, {-2, 0, 1, -1}, 270.000000, 0.000020, 0.087000},
	{1, {0, 0, -1, -1}, 270.000000, 0.000190, 0.075000},
	{1, {0, 0, -1, 1}, 90.000000, 0.000030, 0.074000},
	{1, {0, 0, 1, -1}, 90.000000, 0.000190, 0.076000},
	{1, {2, 0, -1, -1}, 90.000000, 0.000020, 0.088000},
	{1, {2, 0, 0, -1}, 90.000000, 0.000040, 0.041000}
};

static const struct planet_pert plan_pert_elp10 [ELP10_SIZE] = 
{
	{{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2}, 359.998310, 0.000200, 0.037000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 0, 1, -2}, 359.982540, 0.000070, 0.074000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0}, 359.936740, 0.009620, 0.075000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2}, 180.035080, 0.000020, 0.025000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0}, 359.958640, 0.000480, 0.038000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0}, 359.943750, 0.000050, 0.025000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 1, -2, 0}, 180.001750, 0.000040, 0.071000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 1, -1, 0}, 180.018100, 0.000710, 1.127000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 1, 0, -1}, 97.018100, 0.000110, 0.949000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0}, 180.015290, 0.004140, 0.081000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1}, 248.645910, 0.000020, 0.039000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 1, 1, -1}, 259.468000, 0.000010, 0.082000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0}, 180.017440, 0.000290, 0.039000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, -4, 0}, 180.945220, 0.000010, 0.035000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, -3, 0}, 185.233510, 0.000030, 0.067000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, -2, -2}, 0.042080, 0.000010, 0.035000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, -2, 0}, 0.078450, 0.002120, 0.564000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, -1, 0}, 0.122040, 0.037260, 0.087000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, -1, 2}, 180.102460, 0.000080, 0.026000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, 0, -2}, 0.236690, 0.000240, 0.474000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0}, 0.099940, 0.015850, 0.040000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 2}, 180.111930, 0.000050, 0.019000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, 1, -2}, 180.015930, 0.000070, 0.090000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0}, 0.116880, 0.001350, 0.026000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 2, 2, 0}, 359.933350, 0.000090, 0.020000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 3, -1, 0}, 180.124560, 0.000060, 0.042000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 3, 0, -1}, 214.900870, 0.000010, 0.042000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 4, -3, 0}, 0.564020, 0.000060, 0.103000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 4, -2, 0}, 0.122670, 0.000490, 0.044000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 4, -1, 0}, 0.114760, 0.000570, 0.028000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 4, 0, -2}, 0.020340, 0.000020, 0.044000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0}, 0.170480, 0.000200, 0.020000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 4, 1, 0}, 0.110190, 0.000050, 0.016000},
	{{0, 0, 0, 0, 0, 0, 0, 0, 6, -1, 0}, 0.123720, 0.000010, 0.016000},
	{{0, 0, 0, 0, 0, 0, 0, 1, -2, 0, 0}, 319.837980, 0.000020, 0.040000},
	{{0, 0, 0, 0, 0, 0, 0, 1, -2, 1, 0}, 49.171040, 0.000050, 0.087000},
	{{0, 0, 0, 0, 0, 0, 0, 1, 0, -2, 0}, 261.188640, 0.000020, 0.038000},
	{{0, 0, 0, 0, 0, 0, 0, 1, 0, -1, 0}, 261.051780, 0.000310, 0.075000},
	{{0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0}, 260.136670, 0.000740, 164.772000},
	{{0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0}, 261.051780, 0.000310, 0.075000},
	{{0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0}, 261.188640, 0.000020, 0.038000},
	{{0, 0, 0, 0, 0, 0, 0, 1, 2, -1, 0}, 47.398080, 0.000040, 0.087000},
	{{0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0}, 314.650280, 0.000020, 0.040000},
	{{0, 0, 0, 0, 0, 0, 0, 2, -2, 1, 0}, 153.556420, 0.000030, 0.087000},
	{{0, 0, 0, 0, 0, 0, 0, 2, 0, -1, 0}, 333.584190, 0.000150, 0.076000},
	{{0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0}, 333.767450, 0.000350, 82.386000},
	{{0, 0, 0, 0, 0, 0, 0, 2, 0, 1, 0}, 333.584190, 0.000150, 0.075000},
	{{0, 0, 0, 0, 0, 0, 0, 2, 2, -1, 0}, 153.556420, 0.000030, 0.087000},
	{{0, 0, 0, 0, 0, 0, 1, -2, -2, 0, 0}, 146.218010, 0.000070, 0.040000},
	{{0, 0, 0, 0, 0, 0, 1, -2, -2, 1, 0}, 255.188350, 0.000180, 0.087000},
	{{0, 0, 0, 0, 0, 0, 1, -2, -2, 2, 0}, 282.494070, 0.000020, 0.564000},
	{{0, 0, 0, 0, 0, 0, 1, -2, 0, -2, 0}, 101.059520, 0.000080, 0.038000},
	{{0, 0, 0, 0, 0, 0, 1, -2, 0, -1, 0}, 101.147090, 0.001250, 0.075000},
	{{0, 0, 0, 0, 0, 0, 1, -2, 0, 0, 0}, 95.097310, 0.002970, 4234.399000},
	{{0, 0, 0, 0, 0, 0, 1, -2, 0, 1, 0}, 101.147090, 0.001250, 0.075000},
	{{0, 0, 0, 0, 0, 0, 1, -2, 0, 2, 0}, 94.537330, 0.000080, 0.038000},
	{{0, 0, 0, 0, 0, 0, 1, -2, 2, -2, 0}, 282.494060, 0.000020, 0.564000},
	{{0, 0, 0, 0, 0, 0, 1, -2, 2, -1, 0}, 255.290880, 0.000180, 0.087000},
	{{0, 0, 0, 0, 0, 0, 1, -2, 2, 0, 0}, 146.451320, 0.000070, 0.040000},
	{{0, 0, 0, 0, 0, 0, 1, 0, -2, -1, 0}, 184.372050, 0.000030, 0.026000},
	{{0, 0, 0, 0, 0, 0, 1, 0, -2, 0, 0}, 187.931480, 0.000170, 0.040000},
	{{0, 0, 0, 0, 0, 0, 1, 0, -2, 1, 0}, 173.348890, 0.000300, 0.087000},
	{{0, 0, 0, 0, 0, 0, 1, 0, 0, -2, 0}, 237.229130, 0.000030, 0.038000},
	{{0, 0, 0, 0, 0, 0, 1, 0, 0, -1, 0}, 240.145080, 0.000470, 0.076000},
	{{0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0}, 242.328730, 0.001100, 84.021000},
	{{0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0}, 240.301670, 0.000470, 0.075000},
	{{0, 0, 0, 0, 0, 0, 1, 0, 0, 2, 0}, 237.200810, 0.000030, 0.038000},
	{{0, 0, 0, 0, 0, 0, 1, 0, 2, -1, 0}, 175.303970, 0.000300, 0.087000},
	{{0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0}, 189.591040, 0.000170, 0.040000},
	{{0, 0, 0, 0, 0, 0, 1, 0, 2, 1, 0}, 186.066110, 0.000030, 0.026000},
	{{0, 0, 0, 0, 0, 0, 2, -4, 0, 0, 0}, 348.773020, 0.000010, 2117.199000},
	{{0, 0, 0, 0, 0, 0, 2, -2, 0, 0, 0}, 359.694280, 0.000050, 85.722000},
	{{0, 0, 0, 0, 0, 0, 2, 0, -2, 1, 0}, 161.931690, 0.000020, 0.087000},
	{{0, 0, 0, 0, 0, 0, 2, 0, 0, -1, 0}, 341.875690, 0.000130, 0.076000},
	{{0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0}, 341.495090, 0.000300, 42.010000},
	{{0, 0, 0, 0, 0, 0, 2, 0, 0, 1, 0}, 341.875690, 0.000130, 0.075000},
	{{0, 0, 0, 0, 0, 0, 2, 0, 2, -1, 0}, 161.931690, 0.000020, 0.087000},
	{{0, 0, 0, 0, 0, 0, 3, 0, 0, -1, 0}, 152.666110, 0.000010, 0.076000},
	{{0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0}, 169.572460, 0.000020, 28.007000},
	{{0, 0, 0, 0, 0, 0, 3, 0, 0, 1, 0}, 152.666110, 0.000010, 0.075000},
	{{0, 0, 0, 0, 0, 1, -3, 0, -2, 0, 0}, 290.104910, 0.000010, 0.040000},
	{{0, 0, 0, 0, 0, 1, -3, 0, -2, 1, 0}, 125.752920, 0.000010, 0.087000},
	{{0, 0, 0, 0, 0, 1, -3, 0, 0, -1, 0}, 306.446770, 0.000080, 0.075000},
	{{0, 0, 0, 0, 0, 1, -3, 0, 0, 0, 0}, 312.562400, 0.000150, 568.851000},
	{{0, 0, 0, 0, 0, 1, -3, 0, 0, 1, 0}, 306.446770, 0.000080, 0.075000},
	{{0, 0, 0, 0, 0, 1, -3, 0, 2, -1, 0}, 125.752920, 0.000010, 0.087000},
	{{0, 0, 0, 0, 0, 1, -3, 0, 2, 0, 0}, 289.809720, 0.000010, 0.040000},
	{{0, 0, 0, 0, 0, 1, -2, 0, 0, 0, 0}, 271.996750, 0.000010, 98.581000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -4, 0, 0}, 267.905870, 0.000050, 0.020000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -4, 1, 0}, 268.663580, 0.000110, 0.028000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -4, 2, 0}, 269.919760, 0.000060, 0.044000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -3, 0, 0}, 268.039390, 0.000010, 0.027000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -2, -2, 0}, 266.758800, 0.000060, 0.020000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -2, -1, 0}, 266.271430, 0.000390, 0.026000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -2, -1, 2}, 87.848400, 0.000010, 0.090000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -2, 0, -2}, 267.942600, 0.000010, 0.019000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -2, 0, 0}, 267.682270, 0.003900, 0.040000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -2, 0, 2}, 267.860530, 0.000080, 0.467000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -2, 1, 0}, 269.197970, 0.005840, 0.087000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -2, 1, 2}, 87.848400, 0.000010, 0.065000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -2, 2, 0}, 272.363420, 0.000200, 0.553000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -1, 0, 0}, 87.452040, 0.000120, 0.081000},
	{{0, 0, 0, 0, 0, 1, 0, 0, -1, 1, 0}, 87.848400, 0.000020, 1.086000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, -3, 0}, 260.456770, 0.000030, 0.025000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, -2, 0}, 260.879680, 0.000580, 0.038000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, -1, -2}, 81.509860, 0.000020, 0.025000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, -1, 0}, 260.787140, 0.008540, 0.076000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, -1, 2}, 261.509860, 0.000010, 0.073000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0}, 260.214090, 0.019010, 29.457000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, 1, -2}, 261.509860, 0.000010, 0.074000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0}, 260.756370, 0.008500, 0.075000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 2}, 81.509860, 0.000020, 0.025000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, 2, 0}, 260.827400, 0.000580, 0.038000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 0, 3, 0}, 260.374480, 0.000030, 0.025000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 1, -1, 0}, 87.249610, 0.000020, 1.172000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 1, 0, -1}, 66.178080, 0.000020, 0.981000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0}, 87.691390, 0.000100, 0.081000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 2, -2, 0}, 272.859860, 0.000190, 0.575000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 2, -1, -2}, 87.249690, 0.000010, 0.065000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 2, -1, 0}, 268.861710, 0.005830, 0.087000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 2, 0, -2}, 267.238460, 0.000080, 0.482000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 2, 0, 0}, 267.106880, 0.003920, 0.040000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 2, 0, 2}, 267.159100, 0.000010, 0.019000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 2, 1, -2}, 87.249690, 0.000010, 0.089000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 2, 1, 0}, 266.098690, 0.000380, 0.026000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 2, 2, 0}, 266.301730, 0.000050, 0.019000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 3, -2, 0}, 267.090520, 0.000010, 0.094000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 3, 0, 0}, 267.084410, 0.000010, 0.027000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 4, -2, 0}, 269.498020, 0.000060, 0.043000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 4, -1, 0}, 268.419740, 0.000110, 0.028000},
	{{0, 0, 0, 0, 0, 1, 0, 0, 4, 0, 0}, 266.954440, 0.000050, 0.020000},
	{{0, 0, 0, 0, 0, 2, -3, 0, 0, 0, 0}, 211.809410, 0.000030, 31.066000},
	{{0, 0, 0, 0, 0, 2, 0, 0, -2, -1, 0}, 166.003970, 0.000040, 0.026000},
	{{0, 0, 0, 0, 0, 2, 0, 0, -2, 0, 0}, 168.358590, 0.000220, 0.041000},
	{{0, 0, 0, 0, 0, 2, 0, 0, -2, 1, 0}, 162.809390, 0.000490, 0.088000},
	{{0, 0, 0, 0, 0, 2, 0, 0, -2, 2, 0}, 164.041600, 0.000010, 0.543000},
	{{0, 0, 0, 0, 0, 2, 0, 0, -1, 0, 1}, 324.971620, 0.000010, 0.892000},
	{{0, 0, 0, 0, 0, 2, 0, 0, 0, -2, 0}, 325.004450, 0.000030, 0.038000},
	{{0, 0, 0, 0, 0, 2, 0, 0, 0, -1, 0}, 326.155300, 0.000500, 0.076000},
	{{0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0}, 326.833400, 0.001110, 14.729000},
	{{0, 0, 0, 0, 0, 2, 0, 0, 0, 1, 0}, 326.421670, 0.000510, 0.075000},
	{{0, 0, 0, 0, 0, 2, 0, 0, 0, 2, 0}, 324.958520, 0.000030, 0.038000},
	{{0, 0, 0, 0, 0, 2, 0, 0, 2, -2, 0}, 165.606270, 0.000010, 0.586000},
	{{0, 0, 0, 0, 0, 2, 0, 0, 2, -1, 0}, 163.996850, 0.000430, 0.087000},
	{{0, 0, 0, 0, 0, 2, 0, 0, 2, 0, 0}, 177.843610, 0.000140, 0.040000},
	{{0, 0, 0, 0, 0, 2, 0, 0, 2, 1, 0}, 169.745600, 0.000030, 0.026000},
	{{0, 0, 0, 0, 0, 3, 0, 0, -2, 0, 0}, 48.747290, 0.000030, 0.041000},
	{{0, 0, 0, 0, 0, 3, 0, 0, -2, 1, 0}, 64.882000, 0.000030, 0.088000},
	{{0, 0, 0, 0, 0, 3, 0, 0, 0, -1, 0}, 259.336320, 0.000030, 0.076000},
	{{0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0}, 272.757250, 0.000100, 9.819000},
	{{0, 0, 0, 0, 0, 3, 0, 0, 0, 1, 0}, 259.336320, 0.000030, 0.075000},
	{{0, 0, 0, 0, 0, 3, 0, 0, 2, 0, 0}, 48.780470, 0.000010, 0.040000},
	{{0, 0, 0, 0, 0, 5, 0, 0, 0, -1, 0}, 309.553480, 0.000020, 0.076000},
	{{0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0}, 307.749060, 0.000040, 5.891000},
	{{0, 0, 0, 0, 0, 5, 0, 0, 0, 1, 0}, 309.553480, 0.000020, 0.074000},
	{{0, 0, 0, 0, 1, -5, 0, 0, -2, 0, 0}, 200.353690, 0.000080, 0.040000},
	{{0, 0, 0, 0, 1, -5, 0, 0, -2, 1, 0}, 193.557430, 0.000160, 0.086000},
	{{0, 0, 0, 0, 1, -5, 0, 0, 0, -1, 0}, 258.010730, 0.000120, 0.075000},
	{{0, 0, 0, 0, 1, -5, 0, 0, 0, 0, 0}, 269.352850, 0.000220, 11.705000},
	{{0, 0, 0, 0, 1, -5, 0, 0, 0, 1, 0}, 252.228920, 0.000120, 0.076000},
	{{0, 0, 0, 0, 1, -5, 0, 0, 2, -1, 0}, 187.232290, 0.000170, 0.088000},
	{{0, 0, 0, 0, 1, -5, 0, 0, 2, 0, 0}, 193.554500, 0.000090, 0.041000},
	{{0, 0, 0, 0, 1, -4, 0, 0, 0, 0, 0}, 190.619950, 0.000020, 19.422000},
	{{0, 0, 0, 0, 1, -3, 0, 0, -2, 0, 0}, 134.888460, 0.000050, 0.040000},
	{{0, 0, 0, 0, 1, -3, 0, 0, -2, 1, 0}, 123.440260, 0.000090, 0.087000},
	{{0, 0, 0, 0, 1, -3, 0, 0, 0, -1, 0}, 235.512580, 0.000100, 0.075000},
	{{0, 0, 0, 0, 1, -3, 0, 0, 0, 0, 0}, 274.554630, 0.000400, 57.013000},
	{{0, 0, 0, 0, 1, -3, 0, 0, 0, 1, 0}, 235.512580, 0.000100, 0.076000},
	{{0, 0, 0, 0, 1, -3, 0, 0, 2, -1, 0}, 120.517450, 0.000090, 0.087000},
	{{0, 0, 0, 0, 1, -3, 0, 0, 2, 0, 0}, 131.884200, 0.000050, 0.040000},
	{{0, 0, 0, 0, 1, -2, 0, 0, -2, -1, 0}, 355.569270, 0.000040, 0.026000},
	{{0, 0, 0, 0, 1, -2, 0, 0, -2, 0, 0}, 352.347870, 0.000250, 0.040000},
	{{0, 0, 0, 0, 1, -2, 0, 0, -2, 1, 0}, 6.183260, 0.000460, 0.087000},
	{{0, 0, 0, 0, 1, -2, 0, 0, -2, 2, 0}, 34.778020, 0.000010, 0.559000},
	{{0, 0, 0, 0, 1, -2, 0, 0, 0, -2, 0}, 298.258580, 0.000040, 0.038000},
	{{0, 0, 0, 0, 1, -2, 0, 0, 0, -1, 0}, 292.795120, 0.000620, 0.076000},
	{{0, 0, 0, 0, 1, -2, 0, 0, 0, 0, 0}, 273.876300, 0.001290, 60.947000},
	{{0, 0, 0, 0, 1, -2, 0, 0, 0, 1, 0}, 292.623920, 0.000630, 0.075000},
	{{0, 0, 0, 0, 1, -2, 0, 0, 0, 2, 0}, 297.734140, 0.000040, 0.038000},
	{{0, 0, 0, 0, 1, -2, 0, 0, 2, -2, 0}, 31.067820, 0.000010, 0.569000},
	{{0, 0, 0, 0, 1, -2, 0, 0, 2, -1, 0}, 3.957950, 0.000460, 0.087000},
	{{0, 0, 0, 0, 1, -2, 0, 0, 2, 0, 0}, 350.335850, 0.000250, 0.040000},
	{{0, 0, 0, 0, 1, -2, 0, 0, 2, 1, 0}, 353.653950, 0.000040, 0.026000},
	{{0, 0, 0, 0, 1, -1, 0, 0, -2, -1, 0}, 177.145810, 0.000030, 0.026000},
	{{0, 0, 0, 0, 1, -1, 0, 0, -2, 0, 0}, 177.145810, 0.000170, 0.041000},
	{{0, 0, 0, 0, 1, -1, 0, 0, -2, 1, 0}, 177.263910, 0.000320, 0.087000},
	{{0, 0, 0, 0, 1, -1, 0, 0, -2, 2, 0}, 177.145810, 0.000010, 0.548000},
	{{0, 0, 0, 0, 1, -1, 0, 0, 0, -2, 0}, 175.974130, 0.000010, 0.038000},
	{{0, 0, 0, 0, 1, -1, 0, 0, 0, -1, 0}, 176.593500, 0.000120, 0.076000},
	{{0, 0, 0, 0, 1, -1, 0, 0, 0, 0, 0}, 348.754120, 0.000220, 19.859000},
	{{0, 0, 0, 0, 1, -1, 0, 0, 0, 1, 0}, 176.411290, 0.000120, 0.075000},
	{{0, 0, 0, 0, 1, -1, 0, 0, 0, 2, 0}, 175.991310, 0.000010, 0.038000},
	{{0, 0, 0, 0, 1, -1, 0, 0, 2, -2, 0}, 176.873320, 0.000010, 0.580000},
	{{0, 0, 0, 0, 1, -1, 0, 0, 2, -1, 0}, 176.955200, 0.000320, 0.087000},
	{{0, 0, 0, 0, 1, -1, 0, 0, 2, 0, 0}, 176.873330, 0.000160, 0.040000},
	{{0, 0, 0, 0, 1, -1, 0, 0, 2, 1, 0}, 176.873330, 0.000030, 0.026000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -6, 0, 0}, 177.977310, 0.000010, 0.013000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -6, 1, 0}, 354.883200, 0.000020, 0.016000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -6, 2, 0}, 354.681770, 0.000020, 0.021000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -4, -2, 0}, 176.559380, 0.000010, 0.013000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -4, -1, 0}, 348.734380, 0.000060, 0.016000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -4, 0, 0}, 0.269030, 0.000400, 0.020000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -4, 1, 0}, 0.196990, 0.001020, 0.028000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -4, 1, 2}, 355.976110, 0.000020, 0.108000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -4, 2, 0}, 1.476260, 0.000780, 0.044000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -4, 3, 0}, 351.560750, 0.000030, 0.104000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -3, 0, 0}, 279.656300, 0.000050, 0.027000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -3, 1, 0}, 204.051580, 0.000050, 0.042000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -3, 2, 0}, 71.271240, 0.000030, 0.095000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, -3, 0}, 355.569110, 0.000020, 0.016000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, -2, 0}, 344.565210, 0.000210, 0.020000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, -1, -2}, 175.119880, 0.000020, 0.015000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, -1, 0}, 350.458160, 0.002860, 0.026000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, -1, 2}, 169.142220, 0.000080, 0.090000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, 0, -2}, 174.543800, 0.000050, 0.019000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, 0, 0}, 355.134540, 0.033090, 0.041000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, 0, 2}, 354.999650, 0.000720, 0.456000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, 1, -2}, 180.410950, 0.000130, 0.026000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, 1, 0}, 359.809460, 0.057560, 0.088000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, 1, 2}, 353.421670, 0.000030, 0.065000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, 2, 0}, 5.195360, 0.002530, 0.538000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -2, 3, 0}, 21.271650, 0.000120, 0.066000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -1, -1, 0}, 196.411790, 0.000100, 0.039000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -1, -1, 1}, 78.503080, 0.000070, 0.083000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -1, 0, -1}, 258.878050, 0.000080, 0.039000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -1, 0, 0}, 188.328080, 0.001000, 0.081000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -1, 0, 1}, 78.904330, 0.000340, 0.879000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -1, 1, -1}, 258.535950, 0.000080, 0.080000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -1, 1, 0}, 192.246550, 0.000160, 1.030000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -1, 1, 1}, 78.483390, 0.000050, 0.069000},
	{{0, 0, 0, 0, 1, 0, 0, 0, -1, 2, 0}, 155.945960, 0.000020, 0.070000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, -4, 0}, 289.448770, 0.000010, 0.019000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, -3, 0}, 292.783160, 0.000190, 0.025000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, -2, -2}, 122.504590, 0.000010, 0.019000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, -2, 0}, 295.891190, 0.002730, 0.038000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, -1, -2}, 113.490870, 0.000080, 0.025000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, -1, 0}, 294.984140, 0.040460, 0.076000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, -1, 2}, 292.251620, 0.000060, 0.073000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 0, -2}, 257.494800, 0.000030, 0.037000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 0, -1}, 256.855030, 0.000020, 0.075000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0}, 286.696280, 0.086340, 11.862000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 2}, 211.840370, 0.000030, 0.037000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 1, -2}, 293.460320, 0.000100, 0.074000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0}, 293.671100, 0.040400, 0.075000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 2}, 112.536790, 0.000080, 0.025000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 2, 0}, 293.089410, 0.002790, 0.038000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 2, 2}, 122.475740, 0.000010, 0.019000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 3, 0}, 291.651120, 0.000190, 0.025000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 0, 4, 0}, 288.786280, 0.000010, 0.019000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, -2, 0}, 207.989790, 0.000010, 0.071000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, -1, -1}, 66.058850, 0.000060, 0.070000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, -1, 0}, 186.723340, 0.000130, 1.246000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, -1, 1}, 258.552830, 0.000100, 0.079000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, 0, -1}, 77.028950, 0.000410, 1.032000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0}, 177.953290, 0.000740, 0.080000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1}, 258.554740, 0.000090, 0.039000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, 1, -1}, 105.428940, 0.000060, 0.081000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0}, 174.338690, 0.000070, 0.039000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 1, 2, -1}, 77.200360, 0.000010, 0.039000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, -3, 0}, 26.873540, 0.000120, 0.067000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, -2, 0}, 7.909490, 0.002370, 0.592000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, -1, -2}, 352.299720, 0.000030, 0.065000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, -1, 0}, 356.137180, 0.052740, 0.086000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, -1, 2}, 177.368290, 0.000120, 0.026000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, 0, -2}, 345.349300, 0.000690, 0.494000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, 0, 0}, 346.268720, 0.028480, 0.040000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, 0, 2}, 173.972480, 0.000040, 0.019000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, 1, -2}, 163.490620, 0.000070, 0.089000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, 1, 0}, 337.949660, 0.002550, 0.026000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, 1, 2}, 172.039270, 0.000010, 0.015000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, 2, 0}, 331.014620, 0.000200, 0.019000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 2, 3, 0}, 356.169220, 0.000020, 0.015000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 3, -1, -1}, 358.709830, 0.000010, 0.095000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 3, -1, 0}, 155.030440, 0.000040, 0.042000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 3, 0, -1}, 178.411280, 0.000020, 0.042000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 3, 0, 0}, 22.351320, 0.000020, 0.027000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 4, -3, 0}, 350.397180, 0.000030, 0.102000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 4, -2, 0}, 355.551170, 0.000700, 0.043000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 4, -1, -2}, 348.736530, 0.000020, 0.106000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 4, -1, 0}, 347.838210, 0.000890, 0.028000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 4, 0, -2}, 347.030930, 0.000010, 0.044000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 4, 0, 0}, 342.847000, 0.000340, 0.020000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 4, 1, 0}, 356.225900, 0.000060, 0.016000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 4, 2, 0}, 166.652130, 0.000010, 0.013000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 6, -2, 0}, 351.098400, 0.000020, 0.021000},
	{{0, 0, 0, 0, 1, 0, 0, 0, 6, -1, 0}, 343.772500, 0.000010, 0.016000},
	{{0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0}, 272.589050, 0.000020, 8.457000},
	{{0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0}, 94.564490, 0.000040, 6.570000},
	{{0, 0, 0, 0, 1, 3, 0, 0, 0, 0, 0}, 37.108410, 0.000020, 5.372000},
	{{0, 0, 0, 0, 2, -7, 0, 0, 0, -1, 0}, 1.583250, 0.000010, 0.075000},
	{{0, 0, 0, 0, 2, -7, 0, 0, 0, 0, 0}, 355.255110, 0.000080, 14.487000},
	{{0, 0, 0, 0, 2, -7, 0, 0, 0, 1, 0}, 1.583250, 0.000010, 0.076000},
	{{0, 0, 0, 0, 2, -6, 0, 0, -2, 0, 0}, 101.182940, 0.000030, 0.040000},
	{{0, 0, 0, 0, 2, -6, 0, 0, -2, 1, 0}, 101.182920, 0.000040, 0.087000},
	{{0, 0, 0, 0, 2, -6, 0, 0, 0, -1, 0}, 108.513560, 0.000040, 0.075000},
	{{0, 0, 0, 0, 2, -6, 0, 0, 0, 0, 0}, 110.073080, 0.000040, 28.507000},
	{{0, 0, 0, 0, 2, -6, 0, 0, 0, 1, 0}, 108.513560, 0.000040, 0.076000},
	{{0, 0, 0, 0, 2, -6, 0, 0, 2, -1, 0}, 100.402350, 0.000040, 0.087000},
	{{0, 0, 0, 0, 2, -6, 0, 0, 2, 0, 0}, 100.402350, 0.000030, 0.040000},
	{{0, 0, 0, 0, 2, -6, 3, 0, -2, 0, 0}, 15.092230, 0.000030, 0.040000},
	{{0, 0, 0, 0, 2, -6, 3, 0, -2, 1, 0}, 171.284420, 0.000080, 0.087000},
	{{0, 0, 0, 0, 2, -6, 3, 0, -2, 2, 0}, 180.217900, 0.000010, 0.564000},
	{{0, 0, 0, 0, 2, -6, 3, 0, 0, -2, 0}, 0.043910, 0.000040, 0.038000},
	{{0, 0, 0, 0, 2, -6, 3, 0, 0, -1, 0}, 359.847760, 0.000640, 0.075000},
	{{0, 0, 0, 0, 2, -6, 3, 0, 0, 0, 0}, 358.470670, 0.001460, 1597.835000},
	{{0, 0, 0, 0, 2, -6, 3, 0, 0, 1, 0}, 359.847760, 0.000640, 0.075000},
	{{0, 0, 0, 0, 2, -6, 3, 0, 0, 2, 0}, 0.043910, 0.000040, 0.038000},
	{{0, 0, 0, 0, 2, -6, 3, 0, 2, -2, 0}, 180.217940, 0.000010, 0.564000},
	{{0, 0, 0, 0, 2, -6, 3, 0, 2, -1, 0}, 171.193450, 0.000080, 0.087000},
	{{0, 0, 0, 0, 2, -6, 3, 0, 2, 0, 0}, 14.875320, 0.000040, 0.040000},
	{{0, 0, 0, 0, 2, -6, 4, -2, 0, 0, 0}, 0.410990, 0.000010, 2566.169000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -4, 0, 0}, 256.074550, 0.000020, 0.020000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -4, 1, 0}, 190.135540, 0.000120, 0.028000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -4, 2, 0}, 177.025680, 0.000180, 0.044000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, -2, 0}, 331.248390, 0.000080, 0.020000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, -1, 0}, 334.646370, 0.000600, 0.026000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, -1, 2}, 165.960950, 0.000020, 0.090000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, 0, 0}, 259.598900, 0.001550, 0.040000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, 0, 2}, 280.679190, 0.000030, 0.475000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, 1, -2}, 344.542000, 0.000010, 0.026000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, 1, 0}, 178.712680, 0.014000, 0.087000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, 2, 0}, 173.174560, 0.001310, 0.564000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, 3, 0}, 170.011260, 0.000110, 0.067000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -2, 4, 0}, 166.784710, 0.000010, 0.035000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -1, -1, 0}, 165.960950, 0.000020, 0.039000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -1, 0, 0}, 62.251280, 0.000050, 0.081000},
	{{0, 0, 0, 0, 2, -5, 0, 0, -1, 1, 0}, 353.433850, 0.000060, 1.129000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, -4, 0}, 345.960950, 0.000020, 0.019000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, -3, 0}, 346.651570, 0.000330, 0.025000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, -2, -2}, 165.960950, 0.000020, 0.019000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, -2, 0}, 346.459960, 0.004630, 0.038000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, -1, -2}, 166.306890, 0.000130, 0.025000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, -1, 0}, 346.487040, 0.068260, 0.075000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, -1, 2}, 346.068760, 0.000120, 0.074000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 0, -2}, 342.387920, 0.000040, 0.037000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 0, 0}, 347.766530, 0.163760, 883.328000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 0, 2}, 342.387920, 0.000040, 0.037000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 1, -2}, 346.068760, 0.000120, 0.074000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 1, 0}, 346.489720, 0.068260, 0.075000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 1, 2}, 166.306890, 0.000130, 0.025000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 2, 0}, 346.461920, 0.004630, 0.038000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 2, 2}, 165.960950, 0.000020, 0.019000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 3, 0}, 346.652670, 0.000330, 0.025000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 0, 4, 0}, 345.960950, 0.000020, 0.019000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 1, -1, 0}, 353.370220, 0.000060, 1.126000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 1, 0, 0}, 72.263970, 0.000050, 0.081000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 1, 1, 0}, 165.960950, 0.000020, 0.039000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, -4, 0}, 166.814010, 0.000010, 0.035000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, -3, 0}, 169.975810, 0.000120, 0.067000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, -2, 0}, 173.135360, 0.001320, 0.563000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, -1, 0}, 178.608540, 0.014080, 0.087000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, -1, 2}, 344.542000, 0.000010, 0.026000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, 0, -2}, 278.189360, 0.000030, 0.474000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, 0, 0}, 255.533600, 0.001550, 0.040000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, 1, -2}, 165.960950, 0.000020, 0.090000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, 1, 0}, 334.579650, 0.000590, 0.026000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 2, 2, 0}, 330.626670, 0.000080, 0.020000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 4, -2, 0}, 177.219430, 0.000180, 0.044000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 4, -1, 0}, 189.857570, 0.000130, 0.028000},
	{{0, 0, 0, 0, 2, -5, 0, 0, 4, 0, 0}, 253.415930, 0.000020, 0.020000},
	{{0, 0, 0, 0, 2, -4, 0, 0, -2, -1, 0}, 250.785510, 0.000010, 0.026000},
	{{0, 0, 0, 0, 2, -4, 0, 0, -2, 0, 0}, 250.785500, 0.000110, 0.040000},
	{{0, 0, 0, 0, 2, -4, 0, 0, -2, 1, 0}, 250.141600, 0.000200, 0.087000},
	{{0, 0, 0, 0, 2, -4, 0, 0, 0, -1, 0}, 262.699570, 0.000050, 0.076000},
	{{0, 0, 0, 0, 2, -4, 0, 0, 0, 0, 0}, 315.358140, 0.000060, 30.473000},
	{{0, 0, 0, 0, 2, -4, 0, 0, 0, 1, 0}, 263.039640, 0.000050, 0.075000},
	{{0, 0, 0, 0, 2, -4, 0, 0, 2, -1, 0}, 250.405810, 0.000200, 0.087000},
	{{0, 0, 0, 0, 2, -4, 0, 0, 2, 0, 0}, 251.032150, 0.000100, 0.040000},
	{{0, 0, 0, 0, 2, -4, 0, 0, 2, 1, 0}, 251.032150, 0.000010, 0.026000},
	{{0, 0, 0, 0, 2, -3, 0, 0, -2, -1, 0}, 300.404270, 0.000020, 0.026000},
	{{0, 0, 0, 0, 2, -3, 0, 0, -2, 0, 0}, 300.404270, 0.000130, 0.041000},
	{{0, 0, 0, 0, 2, -3, 0, 0, -2, 1, 0}, 300.345160, 0.000220, 0.088000},
	{{0, 0, 0, 0, 2, -3, 0, 0, 0, -2, 0}, 300.719450, 0.000010, 0.038000},
	{{0, 0, 0, 0, 2, -3, 0, 0, 0, -1, 0}, 300.845190, 0.000210, 0.076000},
	{{0, 0, 0, 0, 2, -3, 0, 0, 0, 0, 0}, 301.994650, 0.000280, 14.978000},
	{{0, 0, 0, 0, 2, -3, 0, 0, 0, 1, 0}, 300.842490, 0.000210, 0.075000},
	{{0, 0, 0, 0, 2, -3, 0, 0, 0, 2, 0}, 300.719370, 0.000010, 0.038000},
	{{0, 0, 0, 0, 2, -3, 0, 0, 2, -1, 0}, 300.295740, 0.000200, 0.087000},
	{{0, 0, 0, 0, 2, -3, 0, 0, 2, 0, 0}, 300.370050, 0.000120, 0.040000},
	{{0, 0, 0, 0, 2, -3, 0, 0, 2, 1, 0}, 300.369910, 0.000010, 0.026000},
	{{0, 0, 0, 0, 2, -2, 0, 0, -4, 1, 0}, 359.181660, 0.000010, 0.028000},
	{{0, 0, 0, 0, 2, -2, 0, 0, -2, -1, 0}, 359.176990, 0.000050, 0.026000},
	{{0, 0, 0, 0, 2, -2, 0, 0, -2, 0, 0}, 359.175180, 0.000290, 0.041000},
	{{0, 0, 0, 0, 2, -2, 0, 0, -2, 1, 0}, 359.391490, 0.000540, 0.088000},
	{{0, 0, 0, 0, 2, -2, 0, 0, -2, 2, 0}, 359.181660, 0.000020, 0.533000},
	{{0, 0, 0, 0, 2, -2, 0, 0, 0, -2, 0}, 356.787320, 0.000020, 0.038000},
	{{0, 0, 0, 0, 2, -2, 0, 0, 0, -1, 0}, 356.716540, 0.000220, 0.076000},
	{{0, 0, 0, 0, 2, -2, 0, 0, 0, 0, 0}, 356.943340, 0.000340, 9.929000},
	{{0, 0, 0, 0, 2, -2, 0, 0, 0, 1, 0}, 356.113360, 0.000200, 0.075000},
	{{0, 0, 0, 0, 2, -2, 0, 0, 0, 2, 0}, 356.835120, 0.000020, 0.038000},
	{{0, 0, 0, 0, 2, -2, 0, 0, 2, -2, 0}, 358.653890, 0.000020, 0.598000},
	{{0, 0, 0, 0, 2, -2, 0, 0, 2, -1, 0}, 358.805790, 0.000490, 0.086000},
	{{0, 0, 0, 0, 2, -2, 0, 0, 2, 0, 0}, 358.646900, 0.000260, 0.040000},
	{{0, 0, 0, 0, 2, -2, 0, 0, 2, 1, 0}, 358.653850, 0.000040, 0.026000},
	{{0, 0, 0, 0, 2, -1, 0, 0, -2, 0, 0}, 165.338700, 0.000020, 0.041000},
	{{0, 0, 0, 0, 2, -1, 0, 0, 0, 0, 0}, 355.118260, 0.000070, 7.426000},
	{{0, 0, 0, 0, 2, -1, 0, 0, 2, 0, 0}, 160.289130, 0.000020, 0.040000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -4, 1, 0}, 134.386320, 0.000040, 0.028000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -4, 2, 0}, 119.989220, 0.000020, 0.044000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -2, -1, 0}, 11.330480, 0.000080, 0.026000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -2, 0, 0}, 18.034210, 0.000650, 0.041000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -2, 0, 2}, 22.015610, 0.000020, 0.439000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -2, 1, -1}, 132.889910, 0.000030, 0.040000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -2, 1, 0}, 34.782870, 0.000990, 0.088000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -2, 2, -1}, 131.862280, 0.000020, 0.087000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -2, 2, 0}, 57.793560, 0.000050, 0.515000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -1, 0, 0}, 140.963500, 0.000080, 0.082000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -1, 0, 1}, 46.107530, 0.000050, 0.818000},
	{{0, 0, 0, 0, 2, 0, 0, 0, -1, 1, -1}, 336.538960, 0.000030, 0.081000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, -3, 0}, 319.081350, 0.000020, 0.025000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, -2, 0}, 317.518690, 0.000280, 0.038000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, -1, -1}, 133.314700, 0.000020, 0.038000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, -1, 0}, 318.598290, 0.004260, 0.076000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, 0, -1}, 133.171160, 0.000350, 0.075000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0}, 316.177030, 0.008170, 5.931000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, 1, -1}, 132.889910, 0.002280, 540.002000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, 1, 0}, 317.842380, 0.004250, 0.074000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, 2, -1}, 133.672900, 0.000140, 0.075000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, 2, 0}, 316.092720, 0.000300, 0.037000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, 3, -1}, 133.314700, 0.000020, 0.038000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 0, 3, 0}, 320.208990, 0.000020, 0.025000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 1, 0, -1}, 109.054240, 0.000090, 1.130000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 1, 0, 0}, 154.023870, 0.000030, 0.080000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 2, -3, 0}, 247.153100, 0.000010, 0.067000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 2, -2, 0}, 80.734620, 0.000050, 0.623000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 2, -1, 0}, 353.565290, 0.001240, 0.086000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 2, 0, -2}, 334.628290, 0.000040, 0.516000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 2, 0, -1}, 313.846070, 0.000020, 0.087000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0}, 333.893300, 0.001270, 0.040000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 2, 1, -1}, 132.889910, 0.000030, 0.040000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 2, 1, 0}, 330.392360, 0.000140, 0.026000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 4, -2, 0}, 8.914550, 0.000010, 0.043000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 4, -1, 0}, 321.136910, 0.000020, 0.027000},
	{{0, 0, 0, 0, 2, 0, 0, 0, 4, 0, 0}, 350.593900, 0.000010, 0.020000},
	{{0, 0, 0, 0, 2, 5, 0, 0, 2, -3, 0}, 119.788600, 0.000020, 0.068000},
	{{0, 0, 0, 0, 2, 5, 0, 0, 2, -2, 0}, 119.788600, 0.000020, 0.697000},
	{{0, 0, 0, 0, 2, 5, 0, 0, 2, -1, 0}, 119.788600, 0.000020, 0.085000},
	{{0, 0, 0, 0, 2, 5, 0, 0, 2, 0, 0}, 119.788600, 0.000020, 0.040000},
	{{0, 0, 0, 0, 3, -7, 0, 0, 0, 0, 0}, 56.265880, 0.000050, 65.464000},
	{{0, 0, 0, 0, 3, -6, 0, 0, 0, 0, 0}, 24.631630, 0.000020, 20.316000},
	{{0, 0, 0, 0, 3, -5, 0, 0, -2, -1, 0}, 84.563250, 0.000040, 0.026000},
	{{0, 0, 0, 0, 3, -5, 0, 0, -2, 0, 0}, 83.710820, 0.000210, 0.041000},
	{{0, 0, 0, 0, 3, -5, 0, 0, -2, 1, 0}, 84.492310, 0.000380, 0.088000},
	{{0, 0, 0, 0, 3, -5, 0, 0, 0, -2, 0}, 88.644950, 0.000020, 0.038000},
	{{0, 0, 0, 0, 3, -5, 0, 0, 0, -1, 0}, 88.126670, 0.000280, 0.076000},
	{{0, 0, 0, 0, 3, -5, 0, 0, 0, 0, 0}, 86.229030, 0.000510, 12.023000},
	{{0, 0, 0, 0, 3, -5, 0, 0, 0, 1, 0}, 88.110110, 0.000280, 0.075000},
	{{0, 0, 0, 0, 3, -5, 0, 0, 0, 2, 0}, 87.566730, 0.000020, 0.038000},
	{{0, 0, 0, 0, 3, -5, 0, 0, 2, -2, 0}, 84.802060, 0.000020, 0.591000},
	{{0, 0, 0, 0, 3, -5, 0, 0, 2, -1, 0}, 84.314610, 0.000380, 0.086000},
	{{0, 0, 0, 0, 3, -5, 0, 0, 2, 0, 0}, 84.801570, 0.000210, 0.040000},
	{{0, 0, 0, 0, 3, -5, 0, 0, 2, 1, 0}, 84.802100, 0.000030, 0.026000},
	{{0, 0, 0, 0, 3, -4, 0, 0, -2, 0, 0}, 120.936000, 0.000030, 0.041000},
	{{0, 0, 0, 0, 3, -4, 0, 0, -2, 1, 0}, 120.934480, 0.000050, 0.088000},
	{{0, 0, 0, 0, 3, -4, 0, 0, 0, -1, 0}, 122.742310, 0.000020, 0.076000},
	{{0, 0, 0, 0, 3, -4, 0, 0, 0, 0, 0}, 161.954320, 0.000020, 8.538000},
	{{0, 0, 0, 0, 3, -4, 0, 0, 0, 1, 0}, 122.742310, 0.000020, 0.075000},
	{{0, 0, 0, 0, 3, -4, 0, 0, 2, -1, 0}, 120.503650, 0.000040, 0.086000},
	{{0, 0, 0, 0, 3, -4, 0, 0, 2, 0, 0}, 120.503650, 0.000030, 0.040000},
	{{0, 0, 0, 0, 3, -3, 0, 0, -2, 0, 0}, 342.073910, 0.000040, 0.041000},
	{{0, 0, 0, 0, 3, -3, 0, 0, -2, 1, 0}, 342.073910, 0.000060, 0.088000},
	{{0, 0, 0, 0, 3, -3, 0, 0, 0, -1, 0}, 350.340540, 0.000020, 0.076000},
	{{0, 0, 0, 0, 3, -3, 0, 0, 0, 0, 0}, 69.789280, 0.000040, 6.620000},
	{{0, 0, 0, 0, 3, -3, 0, 0, 0, 1, 0}, 350.340540, 0.000020, 0.075000},
	{{0, 0, 0, 0, 3, -3, 0, 0, 2, -1, 0}, 337.345170, 0.000040, 0.086000},
	{{0, 0, 0, 0, 3, -3, 0, 0, 2, 0, 0}, 337.345190, 0.000030, 0.040000},
	{{0, 0, 0, 0, 3, -2, 0, 0, -2, 0, 0}, 356.810540, 0.000030, 0.041000},
	{{0, 0, 0, 0, 3, -2, 0, 0, -2, 1, 0}, 356.810540, 0.000040, 0.089000},
	{{0, 0, 0, 0, 3, -2, 0, 0, 0, -1, 0}, 335.801590, 0.000040, 0.077000},
	{{0, 0, 0, 0, 3, -2, 0, 0, 0, 0, 0}, 332.189000, 0.000090, 5.405000},
	{{0, 0, 0, 0, 3, -2, 0, 0, 0, 1, 0}, 335.801590, 0.000040, 0.074000},
	{{0, 0, 0, 0, 3, -2, 0, 0, 2, -1, 0}, 346.517310, 0.000040, 0.086000},
	{{0, 0, 0, 0, 3, -2, 0, 0, 2, 0, 0}, 347.946880, 0.000020, 0.040000},
	{{0, 0, 0, 0, 3, 0, 0, 0, -4, 2, 0}, 324.195900, 0.000020, 0.044000},
	{{0, 0, 0, 0, 3, 0, 0, 0, -2, 0, 0}, 110.593530, 0.000100, 0.041000},
	{{0, 0, 0, 0, 3, 0, 0, 0, -2, 1, 0}, 87.868330, 0.000080, 0.089000},
	{{0, 0, 0, 0, 3, 0, 0, 0, -1, 1, -1}, 326.393340, 0.000020, 0.081000},
	{{0, 0, 0, 0, 3, 0, 0, 0, 0, -2, 0}, 314.350810, 0.000020, 0.038000},
	{{0, 0, 0, 0, 3, 0, 0, 0, 0, -1, 0}, 313.535020, 0.000320, 0.077000},
	{{0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0}, 310.219190, 0.000520, 3.954000},
	{{0, 0, 0, 0, 3, 0, 0, 0, 0, 1, 0}, 315.894880, 0.000360, 0.074000},
	{{0, 0, 0, 0, 3, 0, 0, 0, 0, 2, 0}, 314.123740, 0.000020, 0.037000},
	{{0, 0, 0, 0, 3, 0, 0, 0, 1, 0, -1}, 115.200650, 0.000010, 1.249000},
	{{0, 0, 0, 0, 3, 0, 0, 0, 2, -1, 0}, 1.892530, 0.000040, 0.085000},
	{{0, 0, 0, 0, 3, 0, 0, 0, 2, 0, 0}, 325.417690, 0.000050, 0.040000},
	{{0, 0, 0, 0, 4, -11, 3, 0, 0, 0, 0}, 47.870580, 0.000050, 1975.368000},
	{{0, 0, 0, 0, 4, -10, 0, 0, -2, 0, 0}, 39.193620, 0.000020, 0.040000},
	{{0, 0, 0, 0, 4, -10, 0, 0, -2, 1, 0}, 13.367380, 0.000070, 0.087000},
	{{0, 0, 0, 0, 4, -10, 0, 0, 0, -2, 0}, 177.394040, 0.000020, 0.038000},
	{{0, 0, 0, 0, 4, -10, 0, 0, 0, -1, 0}, 178.522400, 0.000270, 0.075000},
	{{0, 0, 0, 0, 4, -10, 0, 0, 0, 0, 0}, 184.689140, 0.000750, 441.664000},
	{{0, 0, 0, 0, 4, -10, 0, 0, 0, 1, 0}, 178.522400, 0.000270, 0.075000},
	{{0, 0, 0, 0, 4, -10, 0, 0, 0, 2, 0}, 177.394040, 0.000020, 0.038000},
	{{0, 0, 0, 0, 4, -10, 0, 0, 2, -1, 0}, 13.118720, 0.000070, 0.087000},
	{{0, 0, 0, 0, 4, -10, 0, 0, 2, 0, 0}, 37.901350, 0.000020, 0.040000},
	{{0, 0, 0, 0, 4, -9, 0, 0, 0, 0, 0}, 263.490040, 0.000020, 31.562000},
	{{0, 0, 0, 0, 4, -8, 0, 0, 0, 0, 0}, 311.035890, 0.000030, 15.237000},
	{{0, 0, 0, 0, 4, -7, 0, 0, 0, 0, 0}, 188.325940, 0.000030, 10.042000},
	{{0, 0, 0, 0, 4, -6, 0, 0, 0, 0, 0}, 54.719500, 0.000020, 7.489000},
	{{0, 0, 0, 0, 4, -5, 0, 0, -2, 0, 0}, 74.669640, 0.000020, 0.041000},
	{{0, 0, 0, 0, 4, -5, 0, 0, -2, 1, 0}, 74.669640, 0.000030, 0.088000},
	{{0, 0, 0, 0, 4, -5, 0, 0, 0, -1, 0}, 95.686040, 0.000020, 0.076000},
	{{0, 0, 0, 0, 4, -5, 0, 0, 0, 0, 0}, 248.087340, 0.000050, 5.971000},
	{{0, 0, 0, 0, 4, -5, 0, 0, 0, 1, -1}, 345.247850, 0.000070, 1389.349000},
	{{0, 0, 0, 0, 4, -5, 0, 0, 0, 1, 0}, 95.686040, 0.000020, 0.074000},
	{{0, 0, 0, 0, 4, -5, 0, 0, 2, -1, 0}, 81.340840, 0.000030, 0.086000},
	{{0, 0, 0, 0, 4, -5, 0, 0, 2, 0, 0}, 81.350180, 0.000020, 0.040000},
	{{0, 0, 0, 0, 4, -4, 0, 0, 0, 0, 0}, 2.658620, 0.000030, 4.965000},
	{{0, 0, 0, 0, 4, -3, 0, 0, 0, 0, 0}, 14.009810, 0.000010, 4.249000},
	{{0, 0, 0, 0, 4, 0, 0, 0, -2, 0, 0}, 114.003200, 0.000010, 0.041000},
	{{0, 0, 0, 0, 4, 0, 0, 0, 0, -1, 0}, 306.230840, 0.000010, 0.077000},
	{{0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0}, 300.535860, 0.000020, 2.965000},
	{{0, 0, 0, 0, 4, 0, 0, 0, 0, 1, 0}, 306.230840, 0.000010, 0.074000},
	{{0, 0, 0, 0, 4, 0, 0, 0, 0, 2, -2}, 117.634940, 0.000020, 270.001000},
	{{0, 0, 0, 0, 6, -5, 0, 0, 2, -3, 0}, 320.021870, 0.000020, 0.068000},
	{{0, 0, 0, 0, 6, -5, 0, 0, 2, -2, 0}, 320.021870, 0.000020, 0.695000},
	{{0, 0, 0, 0, 6, -5, 0, 0, 2, -1, 0}, 320.021870, 0.000020, 0.085000},
	{{0, 0, 0, 0, 6, -5, 0, 0, 2, 0, 0}, 320.021870, 0.000020, 0.040000},
	{{0, 0, 0, 1, -2, 0, 0, 0, -2, 0, 0}, 150.154020, 0.000020, 0.041000},
	{{0, 0, 0, 1, -2, 0, 0, 0, -2, 1, 0}, 330.222320, 0.000030, 0.090000},
	{{0, 0, 0, 1, -1, 0, 0, 0, -2, 0, 0}, 357.163340, 0.000010, 0.041000},
	{{0, 0, 0, 1, -1, 0, 0, 0, 0, -1, 0}, 175.453280, 0.000010, 0.078000},
	{{0, 0, 0, 1, -1, 0, 0, 0, 0, 1, 0}, 175.453280, 0.000010, 0.073000},
	{{0, 0, 0, 1, 0, -1, 0, 0, 0, 0, 0}, 171.793410, 0.000010, 2.009000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -4, 0, 0}, 39.411790, 0.000010, 0.020000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -4, 2, 0}, 38.910860, 0.000010, 0.045000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -2, -1, 0}, 39.081110, 0.000080, 0.027000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -2, 0, 0}, 40.902040, 0.000880, 0.041000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -2, 0, 2}, 38.574250, 0.000020, 0.379000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -2, 1, 0}, 39.205570, 0.001300, 0.091000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -2, 2, 0}, 38.867320, 0.000080, 0.434000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -1, -1, 1}, 128.708470, 0.000020, 0.086000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -1, 0, -1}, 308.374410, 0.000010, 0.040000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -1, 0, 1}, 128.332970, 0.000080, 0.631000},
	{{0, 0, 0, 1, 0, 0, 0, 0, -1, 2, -1}, 308.374410, 0.000010, 0.799000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 0, -1, 0}, 20.121280, 0.000090, 0.079000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0}, 206.829430, 0.001650, 1.881000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0}, 346.932500, 0.000050, 0.073000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 0, 2, 0}, 9.973690, 0.000010, 0.037000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 1, -2, 1}, 308.758760, 0.000010, 5.303000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 1, -1, -1}, 128.685760, 0.000030, 0.073000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 1, -1, 1}, 308.708470, 0.000120, 0.077000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 1, 0, -1}, 128.805320, 0.000230, 1.915000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0}, 29.723970, 0.000040, 0.078000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 1}, 308.772910, 0.000050, 0.038000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 1, 1, -1}, 128.685760, 0.000030, 0.079000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 2, -2, 0}, 34.254000, 0.000060, 0.805000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 2, -1, 0}, 30.333590, 0.000870, 0.083000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 2, 0, -2}, 26.418650, 0.000010, 0.635000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 2, 0, 0}, 26.550920, 0.000330, 0.040000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 2, 1, 0}, 30.522860, 0.000040, 0.026000},
	{{0, 0, 0, 1, 0, 0, 0, 0, 4, -1, 0}, 28.775010, 0.000010, 0.027000},
	{{0, 0, 0, 2, -3, 0, 0, 0, 0, 0, 0}, 13.206790, 0.000010, 1.234000},
	{{0, 0, 0, 2, -2, 0, 0, 0, -2, 0, 0}, 4.744620, 0.000010, 0.042000},
	{{0, 0, 0, 2, -2, 0, 0, 0, 0, 0, 0}, 0.166380, 0.000050, 1.118000},
	{{0, 0, 0, 2, -1, 0, 0, 0, -2, 0, 0}, 26.036380, 0.000010, 0.042000},
	{{0, 0, 0, 2, 0, -2, 0, 0, -2, 0, 0}, 27.147220, 0.000030, 0.042000},
	{{0, 0, 0, 2, 0, -2, 0, 0, -2, 1, 0}, 27.147220, 0.000040, 0.095000},
	{{0, 0, 0, 2, 0, -2, 0, 0, 0, -1, 0}, 23.589790, 0.000010, 0.082000},
	{{0, 0, 0, 2, 0, -2, 0, 0, 0, 0, 0}, 15.823210, 0.000070, 1.005000},
	{{0, 0, 0, 2, 0, -2, 0, 0, 0, 1, 0}, 23.589790, 0.000010, 0.070000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -4, -1, 0}, 309.502660, 0.000010, 0.016000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -4, 0, 0}, 124.907250, 0.000050, 0.021000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -4, 1, 0}, 125.500390, 0.000090, 0.028000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -4, 2, 0}, 129.583220, 0.000040, 0.046000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -3, 2, -1}, 343.795320, 0.000010, 0.044000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -2, -2, 0}, 135.391050, 0.000020, 0.020000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -2, -1, 0}, 129.441000, 0.000240, 0.027000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -2, 0, 0}, 132.936600, 0.002240, 0.042000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -2, 0, 2}, 133.769760, 0.000030, 0.315000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -2, 1, 0}, 134.977890, 0.002180, 0.096000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -2, 2, 0}, 136.907600, 0.000070, 0.352000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -1, -2, -1}, 163.741950, 0.000020, 0.020000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -1, -1, -1}, 163.738060, 0.000150, 0.026000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -1, -1, 1}, 163.751760, 0.000030, 0.090000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -1, 0, -1}, 163.711850, 0.000810, 0.040000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -1, 0, 1}, 344.275310, 0.000270, 0.472000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -1, 1, -1}, 343.797790, 0.000760, 0.087000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -1, 2, -1}, 343.760570, 0.000140, 0.561000},
	{{0, 0, 0, 2, 0, 0, 0, 0, -1, 3, -1}, 343.760770, 0.000010, 0.066000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 0, -2, 0}, 147.907600, 0.000100, 0.039000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 0, -1, 0}, 157.820750, 0.001030, 0.082000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 0, 0, -1}, 343.721990, 0.000020, 0.081000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0}, 183.352870, 0.003210, 0.940000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 0, 1, 0}, 210.568120, 0.000430, 0.070000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 0, 2, 0}, 262.257620, 0.000040, 0.036000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, -3, -1}, 163.748350, 0.000050, 0.025000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, -2, -1}, 163.748350, 0.000740, 0.038000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, -1, -3}, 343.771010, 0.000030, 0.025000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, -1, -1}, 163.746860, 0.010930, 0.075000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, -1, 1}, 343.541630, 0.000200, 0.074000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, 0, -3}, 343.825680, 0.000080, 0.037000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, 0, -1}, 163.721990, 0.063690, 104.137000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, 0, 0}, 46.573690, 0.000020, 0.074000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, 0, 1}, 343.756500, 0.001720, 0.037000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, 1, -3}, 163.705490, 0.000020, 0.074000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, 1, -1}, 163.749840, 0.010900, 0.075000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, 1, 1}, 343.749480, 0.000210, 0.025000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, 2, -1}, 163.748350, 0.000740, 0.038000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 1, 3, -1}, 163.748350, 0.000050, 0.025000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 2, -2, 0}, 18.893320, 0.000020, 1.407000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 2, -1, 0}, 320.290870, 0.000090, 0.080000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 2, 0, -2}, 281.411340, 0.000010, 0.958000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 2, 0, -1}, 343.721990, 0.000020, 0.081000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 2, 0, 0}, 279.954940, 0.000190, 0.039000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 2, 1, 0}, 308.309280, 0.000030, 0.026000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 3, -3, -1}, 343.760770, 0.000010, 0.067000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 3, -2, -1}, 343.760770, 0.000140, 0.567000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 3, -1, -1}, 343.793080, 0.000830, 0.087000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 3, -1, 1}, 343.751800, 0.000040, 0.026000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 3, 0, -1}, 163.719630, 0.000670, 0.040000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 3, 0, 1}, 343.751830, 0.000020, 0.019000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 3, 1, -1}, 163.738060, 0.000150, 0.026000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 3, 2, -1}, 163.741950, 0.000020, 0.020000},
	{{0, 0, 0, 2, 0, 0, 0, 0, 5, -2, -1}, 343.795320, 0.000010, 0.044000},
	{{0, 0, 0, 3, -2, 0, 0, 0, -2, 0, 0}, 207.958680, 0.000010, 0.043000},
	{{0, 0, 0, 3, -2, 0, 0, 0, 2, -1, 0}, 31.109310, 0.000030, 0.077000},
	{{0, 0, 0, 3, 0, 0, 0, 0, -2, 0, 0}, 326.009280, 0.000030, 0.043000},
	{{0, 0, 0, 3, 0, 0, 0, 0, -2, 1, 0}, 325.940900, 0.000030, 0.101000},
	{{0, 0, 0, 3, 0, 0, 0, 0, 0, -1, 0}, 279.577170, 0.000030, 0.086000},
	{{0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0}, 268.261710, 0.000160, 0.627000},
	{{0, 0, 0, 3, 0, 0, 0, 0, 0, 1, 0}, 293.391210, 0.000010, 0.067000},
	{{0, 0, 0, 3, 0, 0, 0, 0, 1, 0, -1}, 2.518960, 0.000020, 1.847000},
	{{0, 0, 0, 3, 0, 0, 0, 0, 2, -3, 0}, 146.587330, 0.000010, 0.074000},
	{{0, 0, 0, 3, 0, 0, 0, 0, 2, -2, 0}, 75.419930, 0.000010, 5.588000},
	{{0, 0, 0, 3, 0, 0, 0, 0, 2, -1, 0}, 90.422490, 0.000110, 0.076000},
	{{0, 0, 0, 3, 1, 0, 0, 0, 2, -3, 0}, 105.224100, 0.000010, 0.075000},
	{{0, 0, 0, 3, 1, 0, 0, 0, 2, -1, 0}, 285.224100, 0.000010, 0.076000},
	{{0, 0, 0, 4, -2, 0, 0, 0, 2, -1, 0}, 236.752600, 0.000020, 0.074000},
	{{0, 0, 0, 4, -1, 0, 0, 0, 0, 0, 0}, 256.669940, 0.000010, 0.490000},
	{{0, 0, 0, 4, 0, 0, 0, 0, -2, 0, 0}, 37.016470, 0.000050, 0.044000},
	{{0, 0, 0, 4, 0, 0, 0, 0, -2, 1, 0}, 37.430350, 0.000050, 0.107000},
	{{0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0}, 349.801730, 0.000020, 0.470000},
	{{0, 0, 0, 4, 0, 0, 0, 0, 2, 0, -2}, 142.187340, 0.000010, 52.068000},
	{{0, 0, 1, -8, 3, 0, 0, 0, -3, 1, 2}, 235.589430, 0.000010, 166.090000},
	{{0, 0, 1, -8, 3, 0, 0, 0, 2, -1, 0}, 235.637320, 0.000010, 0.118000},
	{{0, 0, 1, -8, 3, 0, 0, 0, 2, 0, 0}, 235.639700, 0.000030, 0.046000},
	{{0, 0, 1, -7, 2, 0, 0, 0, -2, 0, 0}, 51.767520, 0.000010, 0.037000},
	{{0, 0, 1, -7, 2, 0, 0, 0, -2, 2, 0}, 51.767520, 0.000010, 1.283000},
	{{0, 0, 1, -6, 0, 0, 0, 0, 2, 0, 0}, 119.905420, 0.000020, 0.044000},
	{{0, 0, 1, -6, 1, 0, 0, 0, -2, 0, 2}, 77.246130, 0.000020, 578.012000},
	{{0, 0, 1, -5, -2, 0, 0, 0, -2, 1, 0}, 356.214980, 0.000010, 0.075000},
	{{0, 0, 1, -5, -2, 0, 0, 0, -2, 3, 0}, 176.214980, 0.000010, 0.076000},
	{{0, 0, 1, -5, -1, 0, 0, 0, -2, 1, 0}, 208.940370, 0.000020, 0.076000},
	{{0, 0, 1, -5, 0, 0, 0, 0, -2, 1, 0}, 81.698360, 0.000040, 0.076000},
	{{0, 0, 1, -5, 0, 0, 0, 0, 0, 0, 0}, 247.045130, 0.000040, 0.603000},
	{{0, 0, 1, -5, 2, 0, 0, 0, -2, 1, 0}, 112.915390, 0.000020, 0.077000},
	{{0, 0, 1, -4, -1, 0, 0, 0, -2, 1, 0}, 52.506180, 0.000010, 0.079000},
	{{0, 0, 1, -4, -1, 0, 0, 0, -2, 3, 0}, 52.506180, 0.000010, 0.072000},
	{{0, 0, 1, -4, 0, 0, 0, 0, -2, 0, 0}, 225.009620, 0.000030, 0.039000},
	{{0, 0, 1, -4, 0, 0, 0, 0, -2, 1, 0}, 174.066980, 0.000030, 0.079000},
	{{0, 0, 1, -4, 0, 0, 0, 0, -2, 2, 0}, 47.868230, 0.000010, 1.545000},
	{{0, 0, 1, -4, 0, 0, 0, 0, -1, -1, 1}, 345.164560, 0.000060, 0.075000},
	{{0, 0, 1, -4, 0, 0, 0, 0, -1, 0, -1}, 165.714810, 0.000030, 0.037000},
	{{0, 0, 1, -4, 0, 0, 0, 0, -1, 0, 1}, 345.867860, 0.000380, 13.704000},
	{{0, 0, 1, -4, 0, 0, 0, 0, -1, 1, 1}, 345.164560, 0.000060, 0.076000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 0, -1, 0}, 9.972170, 0.000090, 0.070000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 0, 0, 0}, 15.112140, 0.000830, 0.888000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 0, 1, 0}, 31.496300, 0.000350, 0.082000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 0, 2, 0}, 18.534820, 0.000020, 0.039000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 1, 0, -1}, 167.844740, 0.000010, 0.459000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 2, -2, 0}, 41.778530, 0.000040, 0.345000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 2, -1, 0}, 41.785180, 0.000850, 0.097000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 2, 0, -2}, 41.500840, 0.000010, 0.309000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 2, 0, 0}, 39.884480, 0.000780, 0.042000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 2, 1, 0}, 41.149430, 0.000110, 0.027000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 4, -1, 0}, 43.659210, 0.000020, 0.028000},
	{{0, 0, 1, -4, 0, 0, 0, 0, 4, 0, 0}, 43.148060, 0.000020, 0.021000},
	{{0, 0, 1, -4, 0, 2, 0, 0, -1, 0, 1}, 241.061750, 0.000010, 197.046000},
	{{0, 0, 1, -4, 1, 0, 0, 0, 2, -2, 0}, 277.352360, 0.000010, 0.355000},
	{{0, 0, 1, -4, 1, 0, 0, 0, 2, 0, 0}, 277.352360, 0.000010, 0.042000},
	{{0, 0, 1, -4, 2, 0, 0, 0, 0, 0, 0}, 122.725220, 0.000040, 1.044000},
	{{0, 0, 1, -4, 2, 0, 0, 0, 2, -1, 0}, 115.726130, 0.000020, 0.095000},
	{{0, 0, 1, -4, 2, 0, 0, 0, 2, 0, 0}, 102.260170, 0.000020, 0.042000},
	{{0, 0, 1, -4, 3, 0, 0, 0, 0, 0, 0}, 137.233450, 0.000030, 1.144000},
	{{0, 0, 1, -4, 3, 0, 0, 0, 2, 0, 0}, 137.466240, 0.000010, 0.042000},
	{{0, 0, 1, -3, 0, 0, 0, 0, -2, -2, 0}, 228.863240, 0.000010, 0.019000},
	{{0, 0, 1, -3, 0, 0, 0, 0, -2, 0, 0}, 236.420110, 0.000050, 0.039000},
	{{0, 0, 1, -3, 0, 0, 0, 0, -2, 1, 0}, 126.309910, 0.000150, 0.083000},
	{{0, 0, 1, -3, 0, 0, 0, 0, -2, 2, 0}, 8.587050, 0.000010, 0.848000},
	{{0, 0, 1, -3, 0, 0, 0, 0, -2, 3, 0}, 302.285160, 0.000020, 0.069000},
	{{0, 0, 1, -3, 0, 0, 0, 0, -1, 0, -1}, 201.081610, 0.000020, 0.038000},
	{{0, 0, 1, -3, 0, 0, 0, 0, -1, 0, 0}, 122.429170, 0.000020, 0.077000},
	{{0, 0, 1, -3, 0, 0, 0, 0, -1, 0, 1}, 20.964410, 0.000140, 2.180000},
	{{0, 0, 1, -3, 0, 0, 0, 0, -1, 1, -1}, 201.284650, 0.000090, 0.076000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 0, -1, 0}, 280.018860, 0.000280, 0.072000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 0, 0, 0}, 301.978680, 0.001590, 1.681000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 0, 1, 0}, 293.641130, 0.000220, 0.079000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 1, 0, -1}, 24.275500, 0.000060, 0.607000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 1, 1, -1}, 21.284650, 0.000020, 0.086000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 2, -2, 0}, 356.474540, 0.000010, 0.422000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 2, -1, 0}, 81.339960, 0.000450, 0.092000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 2, 0, 0}, 74.854250, 0.000320, 0.041000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 2, 1, 0}, 80.572140, 0.000040, 0.027000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 4, -1, 0}, 81.047780, 0.000010, 0.028000},
	{{0, 0, 1, -3, 0, 0, 0, 0, 4, 0, 0}, 262.653490, 0.000020, 0.020000},
	{{0, 0, 1, -3, 1, 0, 0, 0, 0, 0, 0}, 161.603150, 0.000020, 1.958000},
	{{0, 0, 1, -2, -3, 0, 0, 0, 0, 0, 0}, 103.732700, 0.000020, 3.162000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -6, 2, 0}, 318.975170, 0.000010, 0.021000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -4, -1, 0}, 318.639560, 0.000050, 0.016000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -4, 0, 0}, 319.655120, 0.000260, 0.020000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -4, 0, 2}, 318.481120, 0.000030, 0.044000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -4, 1, 0}, 318.283950, 0.000660, 0.028000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -4, 1, 2}, 318.430150, 0.000010, 0.106000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -4, 2, 0}, 321.100290, 0.000430, 0.043000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -3, 0, 0}, 317.241400, 0.000020, 0.027000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -3, 0, 1}, 273.341080, 0.000050, 0.042000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -3, 1, 0}, 137.988440, 0.000020, 0.042000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -3, 1, 1}, 93.267750, 0.000040, 0.095000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, -3, 0}, 318.830360, 0.000020, 0.015000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, -2, 0}, 317.096010, 0.000210, 0.019000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, -1, 0}, 316.905220, 0.002410, 0.026000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, -1, 2}, 137.025180, 0.000070, 0.089000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, 0, -2}, 139.622460, 0.000040, 0.019000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, 0, 0}, 318.229430, 0.023010, 0.040000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, 0, 2}, 318.000100, 0.000540, 0.489000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, 1, -2}, 141.081900, 0.000070, 0.026000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, 1, 0}, 320.919880, 0.031940, 0.087000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, 1, 2}, 319.461430, 0.000020, 0.065000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, 2, 0}, 327.676880, 0.000880, 0.585000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -2, 3, 0}, 343.193580, 0.000030, 0.067000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -1, -1, 0}, 137.202000, 0.000070, 0.039000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -1, -1, 1}, 273.314720, 0.000070, 0.082000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -1, 0, -1}, 232.986990, 0.000030, 0.039000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -1, 0, 0}, 139.080290, 0.000570, 0.080000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -1, 0, 1}, 56.720620, 0.000120, 1.010000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -1, 1, -1}, 232.978240, 0.000010, 0.079000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -1, 1, 0}, 144.710370, 0.000050, 1.214000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -1, 1, 1}, 93.314720, 0.000050, 0.070000},
	{{0, 0, 1, -2, 0, 0, 0, 0, -1, 3, -1}, 232.978240, 0.000010, 0.072000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, -4, 0}, 310.957510, 0.000010, 0.019000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, -3, 0}, 313.252460, 0.000300, 0.025000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, -2, -2}, 134.710380, 0.000030, 0.019000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, -2, 0}, 313.271560, 0.004310, 0.038000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, -1, -2}, 134.828290, 0.000150, 0.025000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, -1, 0}, 313.303470, 0.063230, 0.075000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, -1, 2}, 313.240760, 0.000120, 0.074000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 0, -2}, 143.302880, 0.000150, 0.037000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 0, 0}, 318.137760, 0.324550, 15.781000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 0, 2}, 145.384880, 0.000130, 0.037000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 1, -2}, 312.476180, 0.000090, 0.073000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 1, 0}, 313.479200, 0.064060, 0.076000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 1, 2}, 134.828290, 0.000150, 0.025000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 2, 0}, 313.526660, 0.004330, 0.038000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 2, 2}, 134.711400, 0.000030, 0.019000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 3, 0}, 313.187880, 0.000300, 0.025000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 0, 4, 0}, 311.140080, 0.000010, 0.019000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 1, -1, 0}, 143.540520, 0.000070, 1.052000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 1, -1, 1}, 244.814030, 0.000030, 0.080000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 1, 0, -1}, 52.653080, 0.000120, 0.895000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 1, 0, 0}, 141.663730, 0.000670, 0.081000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 1, 1, 0}, 137.131960, 0.000090, 0.039000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, -3, 0}, 341.198290, 0.000030, 0.066000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, -2, 0}, 325.202890, 0.001110, 0.544000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, -1, -2}, 319.840050, 0.000020, 0.065000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, -1, 0}, 321.961250, 0.035490, 0.088000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, -1, 2}, 141.948310, 0.000080, 0.026000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, 0, -2}, 320.045790, 0.000540, 0.461000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, 0, 0}, 320.197580, 0.025130, 0.041000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, 0, 2}, 139.806380, 0.000040, 0.019000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, 1, -2}, 138.189750, 0.000070, 0.090000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, 1, 0}, 319.042230, 0.002550, 0.026000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, 1, 2}, 140.124750, 0.000010, 0.015000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 2, 2, 0}, 317.567240, 0.000220, 0.020000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 3, -1, 0}, 140.028160, 0.000030, 0.042000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 3, 0, 0}, 321.385220, 0.000020, 0.027000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 4, -3, 0}, 323.772310, 0.000010, 0.104000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 4, -2, 0}, 322.004140, 0.000490, 0.044000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 4, -1, -2}, 320.262100, 0.000010, 0.107000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 4, -1, 0}, 321.235250, 0.000700, 0.028000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 4, 0, 0}, 319.084760, 0.000300, 0.020000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 4, 1, 0}, 319.304050, 0.000060, 0.016000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 6, -2, 0}, 320.023250, 0.000020, 0.021000},
	{{0, 0, 1, -2, 0, 0, 0, 0, 6, -1, 0}, 319.612940, 0.000010, 0.016000},
	{{0, 0, 1, -2, 0, 2, 0, 0, -2, -1, 0}, 33.824930, 0.000030, 0.026000},
	{{0, 0, 1, -2, 0, 2, 0, 0, -2, 0, 0}, 33.609300, 0.000210, 0.040000},
	{{0, 0, 1, -2, 0, 2, 0, 0, -2, 1, 0}, 34.903050, 0.000350, 0.087000},
	{{0, 0, 1, -2, 0, 2, 0, 0, -2, 2, 0}, 33.824930, 0.000010, 0.562000},
	{{0, 0, 1, -2, 0, 2, 0, 0, 0, -2, 0}, 26.344710, 0.000020, 0.038000},
	{{0, 0, 1, -2, 0, 2, 0, 0, 0, -1, 0}, 27.524860, 0.000360, 0.075000},
	{{0, 0, 1, -2, 0, 2, 0, 0, 0, 0, 0}, 31.833980, 0.001970, 220.858000},
	{{0, 0, 1, -2, 0, 2, 0, 0, 0, 1, 0}, 27.502670, 0.000360, 0.075000},
	{{0, 0, 1, -2, 0, 2, 0, 0, 0, 2, 0}, 26.348720, 0.000020, 0.038000},
	{{0, 0, 1, -2, 0, 2, 0, 0, 2, -2, 0}, 33.774990, 0.000010, 0.565000},
	{{0, 0, 1, -2, 0, 2, 0, 0, 2, -1, 0}, 34.853370, 0.000350, 0.087000},
	{{0, 0, 1, -2, 0, 2, 0, 0, 2, 0, 0}, 33.562570, 0.000210, 0.040000},
	{{0, 0, 1, -2, 0, 2, 0, 0, 2, 1, 0}, 33.774990, 0.000030, 0.026000},
	{{0, 0, 1, -2, 1, 0, 0, 0, -2, 0, 0}, 195.158930, 0.000040, 0.040000},
	{{0, 0, 1, -2, 1, 0, 0, 0, -2, 1, 0}, 195.757640, 0.000050, 0.087000},
	{{0, 0, 1, -2, 1, 0, 0, 0, 0, -1, 0}, 193.105970, 0.000060, 0.076000},
	{{0, 0, 1, -2, 1, 0, 0, 0, 0, 0, 0}, 196.888330, 0.000260, 47.766000},
	{{0, 0, 1, -2, 1, 0, 0, 0, 0, 1, 0}, 193.105970, 0.000060, 0.075000},
	{{0, 0, 1, -2, 1, 0, 0, 0, 2, -1, 0}, 194.233830, 0.000050, 0.087000},
	{{0, 0, 1, -2, 1, 0, 0, 0, 2, 0, 0}, 196.008160, 0.000030, 0.040000},
	{{0, 0, 1, -2, 2, 0, 0, 0, 0, 0, 0}, 7.227640, 0.000010, 9.502000},
	{{0, 0, 1, -2, 3, 0, 0, 0, 0, 0, 0}, 199.122210, 0.000010, 5.276000},
	{{0, 0, 1, -1, -2, 0, 0, 0, -2, 0, 0}, 121.100220, 0.000020, 0.041000},
	{{0, 0, 1, -1, -2, 0, 0, 0, 0, -1, 0}, 301.791810, 0.000010, 0.077000},
	{{0, 0, 1, -1, -2, 0, 0, 0, 0, 1, 0}, 301.791810, 0.000010, 0.074000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -6, 2, 0}, 0.649000, 0.000010, 0.021000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -4, 0, 0}, 0.735340, 0.000050, 0.020000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -4, 1, 0}, 0.656060, 0.000150, 0.028000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -4, 2, 0}, 0.655370, 0.000090, 0.044000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -3, 2, 0}, 0.604890, 0.000010, 0.099000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, -2, 0}, 0.646350, 0.000030, 0.020000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, -1, 0}, 0.648040, 0.000390, 0.027000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, -1, 2}, 180.627410, 0.000010, 0.094000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, 0, -2}, 180.648060, 0.000010, 0.020000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, 0, 0}, 0.655410, 0.004660, 0.041000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, 0, 2}, 0.637490, 0.000090, 0.388000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, 1, 0}, 0.680350, 0.006920, 0.091000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, 1, 2}, 180.627410, 0.000010, 0.063000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, 2, 0}, 0.684200, 0.000320, 0.446000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -2, 3, 0}, 0.634930, 0.000020, 0.065000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -1, 0, 0}, 180.470950, 0.000170, 0.084000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -1, 0, 1}, 92.769820, 0.000040, 0.657000},
	{{0, 0, 1, -1, 0, 0, 0, 0, -1, 1, 0}, 180.627410, 0.000010, 0.738000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 0, -2, 0}, 0.528290, 0.000220, 0.038000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 0, -1, 0}, 0.539300, 0.003420, 0.078000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 0, 0, 0}, 0.883940, 0.010510, 2.135000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 0, 1, -2}, 181.111350, 0.000020, 0.076000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 0, 1, 0}, 0.432880, 0.002960, 0.073000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 0, 2, 0}, 0.394120, 0.000190, 0.037000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 0, 3, -2}, 181.111350, 0.000020, 0.075000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 0, 3, 0}, 0.286990, 0.000010, 0.025000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 1, -1, 1}, 274.240360, 0.000040, 0.077000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 1, 0, -1}, 94.553170, 0.000130, 1.708000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 1, 0, 0}, 0.283300, 0.000130, 0.078000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 1, 0, 1}, 274.465150, 0.000020, 0.038000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 2, -2, 0}, 359.855770, 0.000150, 0.766000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 2, -1, 0}, 0.349360, 0.004500, 0.084000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 2, 0, -2}, 0.250860, 0.000070, 0.610000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 2, 0, 0}, 0.325280, 0.002180, 0.040000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 2, 1, 0}, 0.337940, 0.000220, 0.026000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 2, 2, 0}, 0.288760, 0.000020, 0.019000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 3, -2, 0}, 180.197430, 0.000010, 0.090000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 4, -2, 0}, 0.419490, 0.000070, 0.043000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 4, -1, 0}, 0.352460, 0.000070, 0.027000},
	{{0, 0, 1, -1, 0, 0, 0, 0, 4, 0, 0}, 0.306290, 0.000050, 0.020000},
	{{0, 0, 1, -1, 1, 0, 0, 0, 2, 0, 0}, 212.066860, 0.000010, 0.040000},
	{{0, 0, 1, -1, 3, 0, 0, 0, 1, -2, 1}, 127.674560, 0.000030, 1023.362000},
	{{0, 0, 1, 0, -6, 0, 0, 0, 0, 0, 0}, 334.155380, 0.000030, 2.024000},
	{{0, 0, 1, 0, -6, 5, 0, 0, 0, 0, 0}, 198.894870, 0.000040, 1.506000},
	{{0, 0, 1, 0, -5, 0, 0, 0, -2, 0, 0}, 327.392280, 0.000050, 0.041000},
	{{0, 0, 1, 0, -5, 0, 0, 0, -2, 1, 0}, 328.069380, 0.000060, 0.092000},
	{{0, 0, 1, 0, -5, 0, 0, 0, 0, -1, 0}, 323.413150, 0.000070, 0.079000},
	{{0, 0, 1, 0, -5, 0, 0, 0, 0, 0, 0}, 322.872090, 0.000300, 1.729000},
	{{0, 0, 1, 0, -5, 0, 0, 0, 0, 1, 0}, 323.413150, 0.000070, 0.072000},
	{{0, 0, 1, 0, -5, 0, 0, 0, 2, -1, 0}, 326.881740, 0.000020, 0.083000},
	{{0, 0, 1, 0, -5, 0, 0, 0, 2, 0, 0}, 325.924620, 0.000020, 0.040000},
	{{0, 0, 1, 0, -5, 2, 0, 0, -2, 0, 0}, 312.396440, 0.000010, 0.042000},
	{{0, 0, 1, 0, -5, 2, 0, 0, 0, 0, 0}, 302.199210, 0.000070, 1.547000},
	{{0, 0, 1, 0, -5, 3, 0, 0, -2, 0, 0}, 310.903870, 0.000020, 0.042000},
	{{0, 0, 1, 0, -5, 3, 0, 0, 0, 0, 0}, 309.514390, 0.000050, 1.470000},
	{{0, 0, 1, 0, -5, 4, 0, 0, 0, 0, 0}, 186.073100, 0.000020, 1.400000},
	{{0, 0, 1, 0, -5, 5, 0, 0, -2, 0, 0}, 183.850270, 0.000040, 0.042000},
	{{0, 0, 1, 0, -5, 5, 0, 0, -2, 1, 0}, 184.610910, 0.000040, 0.093000},
	{{0, 0, 1, 0, -5, 5, 0, 0, 0, -1, 0}, 180.243080, 0.000040, 0.080000},
	{{0, 0, 1, 0, -5, 5, 0, 0, 0, 0, 0}, 172.896020, 0.000180, 1.337000},
	{{0, 0, 1, 0, -5, 5, 0, 0, 0, 1, 0}, 180.243080, 0.000040, 0.071000},
	{{0, 0, 1, 0, -5, 5, 0, 0, 2, -1, 0}, 337.153960, 0.000030, 0.082000},
	{{0, 0, 1, 0, -5, 7, 0, 0, 0, 0, 0}, 167.613980, 0.000020, 1.225000},
	{{0, 0, 1, 0, -5, 10, 0, 0, 0, 0, 0}, 343.193160, 0.000020, 1.089000},
	{{0, 0, 1, 0, -4, 0, 0, 0, -4, 1, 0}, 317.608410, 0.000020, 0.028000},
	{{0, 0, 1, 0, -4, 0, 0, 0, -2, -1, 0}, 317.226040, 0.000050, 0.027000},
	{{0, 0, 1, 0, -4, 0, 0, 0, -2, 0, 0}, 317.726090, 0.000450, 0.042000},
	{{0, 0, 1, 0, -4, 0, 0, 0, -2, 1, 0}, 317.451000, 0.000640, 0.092000},
	{{0, 0, 1, 0, -4, 0, 0, 0, -2, 2, 0}, 317.771530, 0.000020, 0.410000},
	{{0, 0, 1, 0, -4, 0, 0, 0, -1, -1, 1}, 121.204910, 0.000020, 0.087000},
	{{0, 0, 1, 0, -4, 0, 0, 0, -1, 0, 1}, 122.262080, 0.000020, 0.583000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 0, -2, 0}, 313.486450, 0.000040, 0.039000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 0, -1, 0}, 313.567500, 0.000590, 0.079000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 0, 0, 0}, 313.059060, 0.002670, 1.509000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 0, 1, 0}, 312.993760, 0.000500, 0.072000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 0, 2, 0}, 313.515920, 0.000040, 0.037000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 1, -2, 1}, 301.445250, 0.000040, 17.401000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 1, -1, 1}, 301.204880, 0.000120, 0.076000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 1, 0, -1}, 120.559300, 0.000100, 2.558000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 1, 0, 0}, 313.392910, 0.000020, 0.077000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 1, 0, 1}, 300.642030, 0.000020, 0.038000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 2, -2, 0}, 128.401080, 0.000020, 0.900000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 2, -1, 0}, 332.011770, 0.000120, 0.082000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 2, 0, 0}, 316.318440, 0.000140, 0.039000},
	{{0, 0, 1, 0, -4, 0, 0, 0, 2, 1, 0}, 316.995350, 0.000010, 0.026000},
	{{0, 0, 1, 0, -4, 2, 0, 0, -2, 0, 0}, 313.238330, 0.000060, 0.042000},
	{{0, 0, 1, 0, -4, 2, 0, 0, -2, 1, 0}, 313.705810, 0.000070, 0.093000},
	{{0, 0, 1, 0, -4, 2, 0, 0, 0, -1, 0}, 308.340300, 0.000060, 0.080000},
	{{0, 0, 1, 0, -4, 2, 0, 0, 0, 0, 0}, 304.377160, 0.000270, 1.369000},
	{{0, 0, 1, 0, -4, 2, 0, 0, 0, 1, 0}, 308.340300, 0.000060, 0.071000},
	{{0, 0, 1, 0, -4, 2, 0, 0, 1, -3, 1}, 281.301920, 0.000010, 0.075000},
	{{0, 0, 1, 0, -4, 2, 0, 0, 1, -2, 1}, 281.091910, 0.000120, 95.906000},
	{{0, 0, 1, 0, -4, 2, 0, 0, 1, -1, 1}, 101.054640, 0.000060, 0.075000},
	{{0, 0, 1, 0, -4, 2, 0, 0, 1, 0, -1}, 100.992070, 0.000010, 3.095000},
	{{0, 0, 1, 0, -4, 2, 0, 0, 2, -1, 0}, 15.127370, 0.000010, 0.082000},
	{{0, 0, 1, 0, -4, 2, 0, 0, 2, 0, 0}, 314.730200, 0.000010, 0.039000},
	{{0, 0, 1, 0, -4, 3, 0, 0, -2, 0, 0}, 6.008750, 0.000040, 0.042000},
	{{0, 0, 1, 0, -4, 3, 0, 0, -2, 1, 0}, 6.008740, 0.000030, 0.093000},
	{{0, 0, 1, 0, -4, 3, 0, 0, 0, -1, 0}, 5.632450, 0.000020, 0.080000},
	{{0, 0, 1, 0, -4, 3, 0, 0, 0, 0, 0}, 356.244750, 0.000130, 1.308000},
	{{0, 0, 1, 0, -4, 3, 0, 0, 0, 1, 0}, 5.632450, 0.000020, 0.071000},
	{{0, 0, 1, 0, -4, 3, 0, 0, 2, -1, 0}, 160.084160, 0.000030, 0.082000},
	{{0, 0, 1, 0, -4, 4, 0, 0, -2, 0, 0}, 239.615910, 0.000020, 0.042000},
	{{0, 0, 1, 0, -4, 4, 0, 0, 0, 0, 0}, 238.966610, 0.000050, 1.252000},
	{{0, 0, 1, 0, -4, 5, 0, 0, -2, 0, 0}, 117.638950, 0.000080, 0.042000},
	{{0, 0, 1, 0, -4, 5, 0, 0, -2, 1, 0}, 118.321610, 0.000110, 0.094000},
	{{0, 0, 1, 0, -4, 5, 0, 0, 0, -1, 0}, 111.880520, 0.000110, 0.080000},
	{{0, 0, 1, 0, -4, 5, 0, 0, 0, 0, 0}, 107.791830, 0.000550, 1.201000},
	{{0, 0, 1, 0, -4, 5, 0, 0, 0, 1, 0}, 110.217530, 0.000090, 0.071000},
	{{0, 0, 1, 0, -4, 5, 0, 0, 1, 0, -1}, 271.915290, 0.000070, 4.520000},
	{{0, 0, 1, 0, -4, 5, 0, 0, 2, -3, 0}, 300.304060, 0.000020, 0.070000},
	{{0, 0, 1, 0, -4, 5, 0, 0, 2, -2, 0}, 274.527340, 0.000010, 1.062000},
	{{0, 0, 1, 0, -4, 5, 0, 0, 2, -1, 0}, 98.109670, 0.000040, 0.081000},
	{{0, 0, 1, 0, -4, 5, 0, 0, 2, 0, 0}, 127.648240, 0.000030, 0.039000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -4, 0, 0}, 308.927910, 0.000050, 0.021000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -4, 1, 0}, 309.203520, 0.000130, 0.028000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -4, 2, 0}, 310.426360, 0.000060, 0.045000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -3, 0, 0}, 313.106030, 0.000010, 0.028000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -2, -2, 0}, 308.980800, 0.000040, 0.020000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -2, -1, 0}, 309.252620, 0.000390, 0.027000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -2, 0, -2}, 128.814140, 0.000010, 0.020000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -2, 0, 0}, 309.452720, 0.004360, 0.042000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -2, 0, 2}, 309.616750, 0.000070, 0.350000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -2, 1, 0}, 309.402230, 0.005870, 0.093000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -2, 2, 0}, 307.957890, 0.000270, 0.397000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -2, 3, 0}, 307.851030, 0.000020, 0.063000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -1, -2, 1}, 290.661520, 0.000050, 0.040000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -1, -1, 1}, 290.634350, 0.000510, 0.087000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -1, 0, -1}, 291.845220, 0.000030, 0.040000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -1, 0, 0}, 128.694610, 0.000200, 0.086000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -1, 0, 1}, 112.175510, 0.000090, 0.555000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -1, 1, -1}, 291.450990, 0.000050, 0.085000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -1, 1, 0}, 131.160830, 0.000020, 0.612000},
	{{0, 0, 1, 0, -3, 0, 0, 0, -1, 1, 1}, 111.361120, 0.000030, 0.066000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 0, -2, 0}, 306.403030, 0.000310, 0.039000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 0, -1, -2}, 125.230970, 0.000020, 0.025000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 0, -1, 0}, 306.558230, 0.004580, 0.080000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 0, 0, 0}, 304.606990, 0.023140, 1.339000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 0, 1, 0}, 304.889430, 0.004150, 0.071000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 0, 1, 2}, 125.230970, 0.000020, 0.024000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 0, 2, 0}, 304.625670, 0.000270, 0.037000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, -3, 1}, 290.621810, 0.000070, 0.076000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, -2, -1}, 290.922320, 0.000020, 0.037000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, -2, 1}, 290.644890, 0.001580, 37.265000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, -1, -1}, 111.025270, 0.000160, 0.074000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, -1, 0}, 304.959300, 0.000120, 7.147000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, -1, 1}, 110.668540, 0.002470, 0.075000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, 0, -1}, 110.936720, 0.001190, 3.261000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, 0, 0}, 308.030010, 0.000330, 0.076000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, 0, 1}, 291.624250, 0.000060, 0.038000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, 1, -1}, 111.025270, 0.000160, 0.077000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, 1, 0}, 307.993150, 0.000020, 0.038000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 1, 1, 1}, 290.924050, 0.000020, 0.025000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 2, -3, 0}, 294.031540, 0.000030, 0.070000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 2, -2, 0}, 111.891380, 0.000310, 0.974000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 2, -1, 0}, 88.258410, 0.001470, 0.082000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 2, 0, -2}, 308.873160, 0.000050, 0.735000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 2, 0, 0}, 317.178540, 0.000830, 0.039000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 2, 1, 0}, 314.527590, 0.000100, 0.026000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 3, -3, 1}, 290.661560, 0.000020, 0.087000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 3, -2, 1}, 110.697730, 0.000020, 0.040000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 3, -1, 1}, 110.669680, 0.000020, 0.026000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 4, -3, 0}, 112.951760, 0.000200, 0.096000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 4, -2, 0}, 112.277200, 0.000500, 0.042000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 4, -1, 0}, 111.447310, 0.000300, 0.027000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 4, 0, 0}, 105.446950, 0.000030, 0.020000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 6, -3, 0}, 112.984140, 0.000020, 0.028000},
	{{0, 0, 1, 0, -3, 0, 0, 0, 6, -2, 0}, 113.019720, 0.000010, 0.021000},
	{{0, 0, 1, 0, -3, 1, 0, 0, -2, 0, 0}, 130.317200, 0.000020, 0.042000},
	{{0, 0, 1, 0, -3, 1, 0, 0, 0, 0, 0}, 115.944100, 0.000080, 1.280000},
	{{0, 0, 1, 0, -3, 1, 0, 0, 2, -1, 0}, 104.375100, 0.000040, 0.082000},
	{{0, 0, 1, 0, -3, 2, 0, 0, -2, -1, 0}, 356.285510, 0.000010, 0.027000},
	{{0, 0, 1, 0, -3, 2, 0, 0, -2, 0, 0}, 356.285520, 0.000140, 0.042000},
	{{0, 0, 1, 0, -3, 2, 0, 0, -2, 1, 0}, 356.335060, 0.000180, 0.094000},
	{{0, 0, 1, 0, -3, 2, 0, 0, 0, -1, 0}, 355.207300, 0.000150, 0.080000},
	{{0, 0, 1, 0, -3, 2, 0, 0, 0, 0, 0}, 353.301120, 0.000590, 1.227000},
	{{0, 0, 1, 0, -3, 2, 0, 0, 0, 1, 0}, 355.023340, 0.000110, 0.071000},
	{{0, 0, 1, 0, -3, 2, 0, 0, 2, -3, 0}, 0.641870, 0.000020, 0.070000},
	{{0, 0, 1, 0, -3, 2, 0, 0, 2, -2, 0}, 1.165610, 0.000020, 1.043000},
	{{0, 0, 1, 0, -3, 2, 0, 0, 2, -1, 0}, 357.658830, 0.000060, 0.081000},
	{{0, 0, 1, 0, -3, 2, 0, 0, 2, 0, 0}, 357.611550, 0.000050, 0.039000},
	{{0, 0, 1, 0, -3, 2, 0, 0, 4, -3, 0}, 177.905690, 0.000010, 0.095000},
	{{0, 0, 1, 0, -3, 2, 0, 0, 4, -2, 0}, 177.492660, 0.000010, 0.042000},
	{{0, 0, 1, 0, -3, 3, 0, 0, -2, 0, 0}, 60.487790, 0.000070, 0.042000},
	{{0, 0, 1, 0, -3, 3, 0, 0, -2, 1, 0}, 60.502120, 0.000070, 0.094000},
	{{0, 0, 1, 0, -3, 3, 0, 0, 0, -1, 0}, 60.356730, 0.000050, 0.081000},
	{{0, 0, 1, 0, -3, 3, 0, 0, 0, 0, 0}, 60.307290, 0.000210, 1.178000},
	{{0, 0, 1, 0, -3, 3, 0, 0, 0, 1, 0}, 60.356730, 0.000050, 0.071000},
	{{0, 0, 1, 0, -3, 3, 0, 0, 2, -1, 0}, 57.640290, 0.000020, 0.081000},
	{{0, 0, 1, 0, -3, 3, 0, 0, 2, 0, 0}, 58.491420, 0.000010, 0.039000},
	{{0, 0, 1, 0, -3, 4, 0, 0, -2, 0, 0}, 296.759590, 0.000020, 0.042000},
	{{0, 0, 1, 0, -3, 4, 0, 0, 0, 0, 0}, 291.955680, 0.000030, 1.133000},
	{{0, 0, 1, 0, -3, 5, 0, 0, -4, 1, 0}, 172.516500, 0.000010, 0.028000},
	{{0, 0, 1, 0, -3, 5, 0, 0, -2, -1, 0}, 172.555210, 0.000040, 0.027000},
	{{0, 0, 1, 0, -3, 5, 0, 0, -2, 0, 0}, 172.569760, 0.000410, 0.042000},
	{{0, 0, 1, 0, -3, 5, 0, 0, -2, 1, 0}, 172.464180, 0.000550, 0.095000},
	{{0, 0, 1, 0, -3, 5, 0, 0, -2, 2, 0}, 172.516320, 0.000020, 0.372000},
	{{0, 0, 1, 0, -3, 5, 0, 0, -1, 0, 0}, 172.581860, 0.000030, 0.087000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 0, -2, 0}, 173.039740, 0.000020, 0.039000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 0, -1, 0}, 172.862100, 0.000380, 0.081000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 0, 0, 0}, 172.326350, 0.001710, 1.091000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 0, 1, 0}, 172.944150, 0.000250, 0.071000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 0, 2, 0}, 173.008150, 0.000020, 0.036000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 1, -1, 0}, 172.587010, 0.000060, 33.536000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 1, 0, -1}, 12.258800, 0.000010, 7.303000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 1, 0, 0}, 352.581860, 0.000170, 0.075000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 2, -3, 0}, 1.541210, 0.000010, 0.071000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 2, -1, 0}, 171.320920, 0.000120, 0.081000},
	{{0, 0, 1, 0, -3, 5, 0, 0, 2, 0, 0}, 172.450150, 0.000080, 0.039000},
	{{0, 0, 1, 0, -3, 6, -3, 0, 0, 0, 0}, 349.498380, 0.000020, 1.093000},
	{{0, 0, 1, 0, -2, -5, 0, 0, 0, 0, 0}, 321.165170, 0.000030, 1.511000},
	{{0, 0, 1, 0, -2, -2, 0, 0, 0, 0, 0}, 97.393120, 0.000030, 1.310000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -6, 1, 0}, 306.290790, 0.000020, 0.017000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -6, 2, 0}, 305.914260, 0.000030, 0.021000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -4, -1, 0}, 305.665310, 0.000070, 0.016000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -4, 0, 0}, 306.661660, 0.000380, 0.021000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -4, 1, 0}, 306.606290, 0.000900, 0.028000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -4, 1, 2}, 306.354470, 0.000010, 0.117000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -4, 2, 0}, 305.829230, 0.000600, 0.045000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -4, 3, 0}, 301.605410, 0.000030, 0.113000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -3, 0, 0}, 308.207510, 0.000010, 0.028000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -3, 1, 0}, 126.165010, 0.000060, 0.043000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -3, 1, 1}, 280.640010, 0.000010, 0.104000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, -3, 0}, 305.766300, 0.000030, 0.016000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, -2, 0}, 304.559170, 0.000260, 0.020000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, -1, -2}, 125.589290, 0.000020, 0.016000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, -1, 0}, 305.553720, 0.003130, 0.027000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, -1, 2}, 125.692530, 0.000070, 0.097000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, 0, -2}, 124.640400, 0.000070, 0.020000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, 0, 0}, 306.028040, 0.034740, 0.042000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, 0, 2}, 306.271970, 0.000500, 0.340000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, 1, -2}, 125.606050, 0.000130, 0.027000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, 1, 0}, 305.994160, 0.045700, 0.094000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, 1, 2}, 305.373310, 0.000020, 0.062000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, 2, 0}, 305.970850, 0.001960, 0.384000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -2, 3, 0}, 308.155630, 0.000100, 0.063000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, -2, 1}, 280.516230, 0.000050, 0.041000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, -1, -1}, 280.506170, 0.000030, 0.026000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, -1, 0}, 125.783750, 0.000150, 0.040000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, -1, 1}, 280.518640, 0.000690, 0.088000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, 0, -1}, 280.508280, 0.000160, 0.040000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, 0, 0}, 125.857040, 0.002330, 0.087000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, 0, 1}, 100.502690, 0.001040, 0.530000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, 1, -1}, 280.538880, 0.000400, 0.085000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, 1, 0}, 125.981960, 0.000250, 0.582000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, 1, 1}, 100.550300, 0.000180, 0.066000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, 2, -1}, 280.516410, 0.000030, 0.644000},
	{{0, 0, 1, 0, -2, 0, 0, 0, -1, 2, 0}, 125.809930, 0.000020, 0.067000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, -3, 0}, 300.042240, 0.000200, 0.026000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, -2, 0}, 300.115070, 0.002610, 0.039000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, -1, -2}, 122.267210, 0.000090, 0.025000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, -1, 0}, 301.001670, 0.037800, 0.080000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, -1, 2}, 291.773860, 0.000020, 0.069000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, 0, -2}, 118.849710, 0.000020, 0.038000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, 0, -1}, 100.514480, 0.000020, 0.079000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, 0, 0}, 301.005990, 0.164250, 1.203000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, 0, 2}, 121.444260, 0.000110, 0.036000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, 1, 0}, 301.497680, 0.028890, 0.071000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, 1, 2}, 122.034290, 0.000080, 0.024000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, 2, 0}, 301.266710, 0.002020, 0.037000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 0, 3, 0}, 301.522720, 0.000130, 0.025000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, -3, 0}, 308.254100, 0.000010, 0.038000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, -3, 1}, 280.540490, 0.000020, 0.076000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, -2, -1}, 100.522540, 0.000080, 0.037000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, -2, 0}, 310.578720, 0.000180, 0.075000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, -2, 1}, 280.537250, 0.000550, 8.998000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, -1, -1}, 100.523300, 0.001550, 0.074000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, -1, 0}, 305.890400, 0.003270, 17.980000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, -1, 1}, 100.524470, 0.004450, 0.075000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 0, -3}, 283.508100, 0.000010, 0.037000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 0, -1}, 100.530100, 0.011830, 4.497000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 0, 0}, 306.193170, 0.006970, 0.076000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 0, 1}, 280.506100, 0.002030, 0.038000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 1, -1}, 100.524170, 0.001650, 0.077000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 1, 0}, 306.138740, 0.000470, 0.038000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 1, 1}, 280.506990, 0.000240, 0.025000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 2, -1}, 100.523210, 0.000110, 0.038000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 2, 0}, 306.090150, 0.000030, 0.025000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 1, 2, 1}, 280.507380, 0.000020, 0.019000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, -4, 0}, 292.644920, 0.000010, 0.036000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, -3, 0}, 275.013050, 0.000040, 0.070000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, -2, 0}, 288.469890, 0.000490, 1.061000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, -1, 0}, 301.711050, 0.018190, 0.081000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, -1, 2}, 130.795140, 0.000030, 0.026000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, 0, -2}, 304.986120, 0.000400, 0.784000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, 0, 0}, 303.968000, 0.008720, 0.039000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, 0, 2}, 125.808930, 0.000020, 0.019000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, 1, -2}, 125.009960, 0.000020, 0.083000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, 1, 0}, 303.242760, 0.000990, 0.026000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 2, 2, 0}, 305.074960, 0.000100, 0.019000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 3, -2, -1}, 280.518400, 0.000020, 0.501000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 3, -2, 0}, 305.544060, 0.000030, 0.088000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 3, -2, 1}, 100.524100, 0.000060, 0.040000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 3, -1, -1}, 280.472280, 0.000070, 0.089000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 3, -1, 0}, 305.923180, 0.000120, 0.041000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 3, 0, -1}, 100.660500, 0.000030, 0.041000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 3, 0, 0}, 305.898910, 0.000080, 0.026000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 3, 0, 1}, 280.507220, 0.000030, 0.019000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 4, -3, 0}, 103.085550, 0.000180, 0.095000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 4, -2, 0}, 289.480980, 0.000690, 0.042000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 4, -1, 0}, 289.229820, 0.000980, 0.027000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 4, 0, 0}, 297.360320, 0.000190, 0.020000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 4, 1, 0}, 304.959580, 0.000010, 0.016000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 6, -3, 0}, 286.192750, 0.000020, 0.028000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 6, -2, 0}, 283.290880, 0.000040, 0.021000},
	{{0, 0, 1, 0, -2, 0, 0, 0, 6, -1, 0}, 283.058910, 0.000010, 0.016000},
	{{0, 0, 1, 0, -2, 1, 0, 0, -2, 0, 0}, 182.976990, 0.000060, 0.042000},
	{{0, 0, 1, 0, -2, 1, 0, 0, -2, 1, 0}, 182.976990, 0.000070, 0.094000},
	{{0, 0, 1, 0, -2, 1, 0, 0, 0, -1, 0}, 182.533130, 0.000050, 0.081000},
	{{0, 0, 1, 0, -2, 1, 0, 0, 0, 0, 0}, 181.621760, 0.000190, 1.156000},
	{{0, 0, 1, 0, -2, 1, 0, 0, 0, 1, 0}, 182.533130, 0.000050, 0.071000},
	{{0, 0, 1, 0, -2, 1, 0, 0, 2, -1, 0}, 181.109350, 0.000030, 0.081000},
	{{0, 0, 1, 0, -2, 1, 0, 0, 2, 0, 0}, 181.109370, 0.000020, 0.039000},
	{{0, 0, 1, 0, -2, 2, 0, 0, -2, -1, 0}, 28.726000, 0.000010, 0.027000},
	{{0, 0, 1, 0, -2, 2, 0, 0, -2, 0, 0}, 28.726000, 0.000100, 0.042000},
	{{0, 0, 1, 0, -2, 2, 0, 0, -2, 1, 0}, 29.404560, 0.000130, 0.094000},
	{{0, 0, 1, 0, -2, 2, 0, 0, 0, -1, 0}, 23.139300, 0.000100, 0.081000},
	{{0, 0, 1, 0, -2, 2, 0, 0, 0, 0, 0}, 22.877820, 0.000390, 1.112000},
	{{0, 0, 1, 0, -2, 2, 0, 0, 0, 1, 0}, 20.609650, 0.000070, 0.071000},
	{{0, 0, 1, 0, -2, 2, 0, 0, 1, -1, 0}, 12.805190, 0.000080, 81.442000},
	{{0, 0, 1, 0, -2, 2, 0, 0, 1, 0, -1}, 114.869110, 0.000020, 6.474000},
	{{0, 0, 1, 0, -2, 2, 0, 0, 1, 0, 0}, 191.063710, 0.000070, 0.075000},
	{{0, 0, 1, 0, -2, 2, 0, 0, 2, -1, 0}, 3.514770, 0.000030, 0.081000},
	{{0, 0, 1, 0, -2, 2, 0, 0, 2, 0, 0}, 9.774070, 0.000020, 0.039000},
	{{0, 0, 1, 0, -2, 3, 0, 0, -2, 0, 0}, 177.030370, 0.000020, 0.042000},
	{{0, 0, 1, 0, -2, 3, 0, 0, 0, 0, 0}, 237.177380, 0.000060, 1.072000},
	{{0, 0, 1, 0, -2, 3, 0, 0, 1, 0, -1}, 108.707340, 0.000040, 8.297000},
	{{0, 0, 1, 0, -2, 4, 0, 0, -2, 0, 0}, 348.120640, 0.000060, 0.042000},
	{{0, 0, 1, 0, -2, 4, 0, 0, -2, 1, 0}, 348.120650, 0.000070, 0.095000},
	{{0, 0, 1, 0, -2, 4, 0, 0, 0, -1, 0}, 349.133730, 0.000050, 0.081000},
	{{0, 0, 1, 0, -2, 4, 0, 0, 0, 0, 0}, 350.556370, 0.000260, 1.034000},
	{{0, 0, 1, 0, -2, 4, 0, 0, 0, 1, 0}, 349.133730, 0.000050, 0.070000},
	{{0, 0, 1, 0, -2, 4, 0, 0, 1, 0, -1}, 186.903520, 0.000010, 11.550000},
	{{0, 0, 1, 0, -2, 4, 0, 0, 2, -1, 0}, 351.988180, 0.000030, 0.080000},
	{{0, 0, 1, 0, -2, 4, 0, 0, 2, 0, 0}, 346.480270, 0.000010, 0.039000},
	{{0, 0, 1, 0, -2, 5, 0, 0, -4, 1, 0}, 244.139120, 0.000020, 0.028000},
	{{0, 0, 1, 0, -2, 5, 0, 0, -4, 2, 0}, 269.654400, 0.000010, 0.046000},
	{{0, 0, 1, 0, -2, 5, 0, 0, -2, -1, 0}, 128.307630, 0.000050, 0.027000},
	{{0, 0, 1, 0, -2, 5, 0, 0, -2, 0, 0}, 193.449750, 0.000280, 0.042000},
	{{0, 0, 1, 0, -2, 5, 0, 0, -2, 1, 0}, 244.372350, 0.000780, 0.095000},
	{{0, 0, 1, 0, -2, 5, 0, 0, -2, 2, 0}, 253.996870, 0.000060, 0.360000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 0, -2, 0}, 97.684470, 0.000060, 0.039000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 0, -1, 0}, 112.556070, 0.000420, 0.082000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 0, 0, 0}, 200.609980, 0.000680, 0.999000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 0, 1, 0}, 256.234950, 0.000370, 0.070000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 0, 2, 0}, 259.546500, 0.000030, 0.036000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 1, 0, -1}, 266.230980, 0.000040, 19.000000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 1, 0, 1}, 86.299260, 0.000050, 0.037000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 2, -3, 0}, 343.748280, 0.000010, 0.071000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 2, -2, 0}, 215.126220, 0.000010, 1.294000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 2, -1, 0}, 86.056160, 0.000070, 0.080000},
	{{0, 0, 1, 0, -2, 5, 0, 0, 2, 0, 0}, 181.739040, 0.000030, 0.039000},
	{{0, 0, 1, 0, -2, 6, 0, 0, 1, 0, -1}, 90.458640, 0.000030, 53.519000},
	{{0, 0, 1, 0, -2, 7, 0, 0, 1, 0, -1}, 48.458300, 0.000010, 65.520000},
	{{0, 0, 1, 0, -1, -5, 0, 0, -2, 0, 0}, 318.207290, 0.000040, 0.042000},
	{{0, 0, 1, 0, -1, -5, 0, 0, -2, 1, 0}, 318.207630, 0.000050, 0.093000},
	{{0, 0, 1, 0, -1, -5, 0, 0, 0, -1, 0}, 314.506090, 0.000040, 0.080000},
	{{0, 0, 1, 0, -1, -5, 0, 0, 0, 0, 0}, 310.643600, 0.000190, 1.341000},
	{{0, 0, 1, 0, -1, -5, 0, 0, 0, 1, 0}, 314.506090, 0.000040, 0.071000},
	{{0, 0, 1, 0, -1, -5, 0, 0, 2, 0, 0}, 319.241700, 0.000010, 0.039000},
	{{0, 0, 1, 0, -1, -3, 0, 0, 0, 0, 0}, 338.799030, 0.000010, 1.229000},
	{{0, 0, 1, 0, -1, -2, 0, 0, -2, 0, 0}, 107.019420, 0.000030, 0.042000},
	{{0, 0, 1, 0, -1, -2, 0, 0, -2, 1, 0}, 107.019440, 0.000020, 0.094000},
	{{0, 0, 1, 0, -1, -2, 0, 0, 0, -1, 0}, 107.019610, 0.000010, 0.081000},
	{{0, 0, 1, 0, -1, -2, 0, 0, 0, 0, 0}, 106.344980, 0.000120, 1.180000},
	{{0, 0, 1, 0, -1, -2, 0, 0, 0, 1, 0}, 107.019610, 0.000010, 0.071000},
	{{0, 0, 1, 0, -1, -1, 0, 0, -2, 0, 0}, 310.206860, 0.000010, 0.042000},
	{{0, 0, 1, 0, -1, -1, 0, 0, 0, 0, 0}, 307.277790, 0.000050, 1.134000},
	{{0, 0, 1, 0, -1, -1, 0, 0, 1, 0, -1}, 293.883070, 0.000010, 5.813000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -6, -1, 0}, 1.895890, 0.000010, 0.012000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -6, 0, 0}, 0.491900, 0.000030, 0.014000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -6, 1, 0}, 1.354640, 0.000080, 0.017000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -6, 2, 0}, 6.795500, 0.000090, 0.021000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -6, 3, 0}, 1.729020, 0.000040, 0.030000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -5, 1, 0}, 2.102600, 0.000010, 0.021000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, -2, 0}, 1.067740, 0.000050, 0.013000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, -1, 0}, 359.988010, 0.000280, 0.016000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, -1, 2}, 1.768350, 0.000010, 0.029000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, 0, 0}, 2.846470, 0.001880, 0.021000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, 0, 2}, 1.782340, 0.000050, 0.046000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, 1, -2}, 181.698440, 0.000020, 0.016000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, 1, 0}, 2.423160, 0.004430, 0.028000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, 1, 2}, 2.139080, 0.000020, 0.118000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, 2, -2}, 181.669130, 0.000010, 0.020000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, 2, 0}, 2.189600, 0.002820, 0.045000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -4, 3, 0}, 1.642970, 0.000070, 0.114000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -3, -2, 0}, 1.839380, 0.000010, 0.016000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -3, -1, 0}, 1.675040, 0.000040, 0.020000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -3, 0, 0}, 1.480380, 0.000150, 0.028000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -3, 1, 0}, 183.900710, 0.000090, 0.044000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -3, 2, 0}, 182.249840, 0.000080, 0.103000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, -3, 0}, 1.632240, 0.000110, 0.016000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, -2, 0}, 1.599540, 0.001190, 0.020000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, -2, 2}, 181.818980, 0.000020, 0.043000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, -1, -2}, 181.651240, 0.000070, 0.016000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, -1, 0}, 1.965100, 0.014550, 0.027000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, -1, 2}, 181.609440, 0.000350, 0.098000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 0, -2}, 181.666370, 0.000400, 0.020000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 0, 0}, 1.802800, 0.164970, 0.042000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 0, 2}, 1.802140, 0.002280, 0.331000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 1, -2}, 181.715580, 0.000450, 0.027000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 1, 0}, 1.743330, 0.211180, 0.095000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 1, 2}, 181.639800, 0.000060, 0.061000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 2, -2}, 181.540350, 0.000030, 0.041000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 2, 0}, 1.709050, 0.008890, 0.372000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 2, 2}, 181.734320, 0.000050, 0.034000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 3, 0}, 2.185500, 0.000530, 0.063000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -2, 4, 0}, 2.210860, 0.000040, 0.034000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, -2, 0}, 0.652030, 0.000080, 0.026000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, -1, 0}, 0.560290, 0.000800, 0.040000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, -1, 1}, 277.892670, 0.000040, 0.089000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, 0, -2}, 180.710180, 0.000030, 0.026000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, 0, -1}, 274.616020, 0.000010, 0.040000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, 0, 0}, 0.621110, 0.009780, 0.087000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, 0, 1}, 96.664150, 0.000080, 0.508000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, 1, -1}, 275.641690, 0.000030, 0.086000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, 1, 0}, 0.518380, 0.000740, 0.555000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, 1, 1}, 93.048320, 0.000010, 0.066000},
	{{0, 0, 1, 0, -1, 0, 0, 0, -1, 2, 0}, 0.733640, 0.000080, 0.066000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, -4, 0}, 0.968310, 0.000040, 0.019000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, -3, 0}, 0.892730, 0.000670, 0.026000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, -2, -2}, 181.164830, 0.000060, 0.019000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, -2, 0}, 1.256950, 0.009430, 0.039000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, -2, 2}, 181.751470, 0.000030, 0.801000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, -1, -2}, 181.354100, 0.000310, 0.026000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, -1, 0}, 1.235820, 0.143470, 0.081000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, -1, 2}, 358.136640, 0.000060, 0.069000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 0, -2}, 183.266440, 0.000080, 0.039000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 0, 0}, 1.228900, 0.638800, 1.092000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 0, 2}, 181.204100, 0.000320, 0.036000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 1, -2}, 348.262080, 0.000040, 0.079000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 1, 0}, 1.111680, 0.109930, 0.071000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 1, 2}, 181.161540, 0.000270, 0.024000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 2, -2}, 183.120050, 0.000020, 1.718000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 2, 0}, 1.036370, 0.007660, 0.036000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 2, 2}, 180.938950, 0.000040, 0.018000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 3, 0}, 1.112850, 0.000550, 0.025000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 0, 4, 0}, 1.064010, 0.000040, 0.019000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, -3, 0}, 181.137640, 0.000050, 0.038000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, -2, -1}, 97.007400, 0.000010, 0.038000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, -2, 0}, 180.862750, 0.000490, 0.076000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, -2, 1}, 277.389040, 0.000010, 5.117000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, -1, -2}, 181.066670, 0.000030, 0.037000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, -1, -1}, 96.642870, 0.000210, 0.075000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, -1, 0}, 0.697360, 0.023380, 34.860000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, -1, 1}, 96.675090, 0.000240, 0.074000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, -1, 2}, 181.066670, 0.000030, 0.037000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 0, -2}, 180.734330, 0.000110, 0.074000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 0, -1}, 96.790660, 0.001600, 7.243000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 0, 0}, 180.709130, 0.067110, 0.075000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 0, 1}, 276.849400, 0.000350, 0.037000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 0, 2}, 0.710190, 0.000140, 0.025000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 1, -1}, 96.642870, 0.000210, 0.076000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 1, 0}, 180.711410, 0.004600, 0.038000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 1, 1}, 276.810410, 0.000040, 0.025000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 1, 2}, 0.710190, 0.000020, 0.019000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 2, -2}, 180.461070, 0.000010, 0.077000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 2, 0}, 180.708310, 0.000330, 0.025000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 1, 3, 0}, 180.710190, 0.000020, 0.019000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, -4, 0}, 182.379150, 0.000010, 0.037000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, -3, 0}, 182.167740, 0.000080, 0.071000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, -2, 0}, 179.691450, 0.000830, 1.165000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, -1, -2}, 1.629340, 0.000030, 0.069000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, -1, 0}, 0.579200, 0.051080, 0.081000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, -1, 2}, 180.816800, 0.000110, 0.025000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, 0, -2}, 0.311180, 0.001620, 0.839000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, 0, 0}, 0.433960, 0.031110, 0.039000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, 0, 2}, 181.516280, 0.000050, 0.019000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, 1, -2}, 179.202900, 0.000080, 0.083000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, 1, 0}, 0.427240, 0.003460, 0.026000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, 2, 0}, 1.428080, 0.000310, 0.019000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 2, 3, 0}, 0.655520, 0.000030, 0.015000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 3, -3, 0}, 0.737340, 0.000030, 0.573000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 3, -2, 0}, 0.744240, 0.000610, 0.087000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 3, -1, 0}, 180.691060, 0.000660, 0.040000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 3, 0, 0}, 180.745400, 0.000600, 0.026000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 3, 1, 0}, 180.710150, 0.000090, 0.020000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 4, -3, 0}, 181.429440, 0.000020, 0.094000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 4, -2, 0}, 1.315600, 0.000690, 0.042000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 4, -1, -2}, 0.221720, 0.000030, 0.097000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 4, -1, 0}, 359.771090, 0.000930, 0.027000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 4, 0, -2}, 0.283650, 0.000020, 0.042000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 4, 0, 0}, 1.644090, 0.000370, 0.020000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 4, 1, 0}, 0.632620, 0.000080, 0.016000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 5, -1, 0}, 180.709290, 0.000020, 0.020000},
	{{0, 0, 1, 0, -1, 0, 0, 0, 6, -2, 0}, 0.812010, 0.000020, 0.021000},
	{{0, 0, 1, 0, -1, 1, 0, 0, -2, 0, 0}, 225.458220, 0.000040, 0.042000},
	{{0, 0, 1, 0, -1, 1, 0, 0, -2, 1, 0}, 225.458220, 0.000040, 0.095000},
	{{0, 0, 1, 0, -1, 1, 0, 0, 0, -1, 0}, 219.577810, 0.000010, 0.081000},
	{{0, 0, 1, 0, -1, 1, 0, 0, 0, 0, 0}, 206.653930, 0.000090, 1.053000},
	{{0, 0, 1, 0, -1, 1, 0, 0, 0, 1, 0}, 219.577810, 0.000010, 0.070000},
	{{0, 0, 1, 0, -1, 1, 0, 0, 1, 0, -1}, 278.337010, 0.000010, 9.604000},
	{{0, 0, 1, 0, -1, 2, 0, 0, -2, 0, 0}, 49.887170, 0.000030, 0.042000},
	{{0, 0, 1, 0, -1, 2, 0, 0, -2, 1, 0}, 48.164150, 0.000040, 0.095000},
	{{0, 0, 1, 0, -1, 2, 0, 0, 0, -1, 0}, 36.616540, 0.000010, 0.081000},
	{{0, 0, 1, 0, -1, 2, 0, 0, 0, 0, 0}, 14.774250, 0.000070, 1.017000},
	{{0, 0, 1, 0, -1, 2, 0, 0, 0, 1, 0}, 36.616540, 0.000010, 0.070000},
	{{0, 0, 1, 0, -1, 3, 0, 0, 0, 0, 0}, 189.440460, 0.000020, 0.983000},
	{{0, 0, 1, 0, -1, 4, 0, 0, 1, 0, -1}, 43.794170, 0.000030, 439.132000},
	{{0, 0, 1, 0, -1, 5, 0, 0, -2, 0, 0}, 234.934380, 0.000010, 0.042000},
	{{0, 0, 1, 0, -1, 5, 0, 0, 0, 0, 0}, 222.793490, 0.000020, 0.921000},
	{{0, 0, 1, 0, -1, 5, 0, 0, 1, 0, -1}, 105.768990, 0.000040, 31.575000},
	{{0, 0, 1, 0, 0, -5, 0, 0, -2, -1, 0}, 314.848970, 0.000030, 0.027000},
	{{0, 0, 1, 0, 0, -5, 0, 0, -2, 0, 0}, 314.552120, 0.000180, 0.042000},
	{{0, 0, 1, 0, 0, -5, 0, 0, -2, 1, 0}, 314.865320, 0.000250, 0.094000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 0, -1, 0}, 311.708870, 0.000230, 0.080000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 0, 0, 0}, 307.675620, 0.000930, 1.204000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 0, 1, 0}, 310.486770, 0.000170, 0.071000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 0, 2, 0}, 312.430340, 0.000010, 0.037000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 1, -1, 1}, 109.268870, 0.000020, 0.075000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 1, 0, -1}, 109.148540, 0.000070, 4.474000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 1, 0, 0}, 309.967760, 0.000030, 0.076000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 2, -3, 0}, 198.397260, 0.000010, 0.070000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 2, -1, 0}, 298.497000, 0.000110, 0.081000},
	{{0, 0, 1, 0, 0, -5, 0, 0, 2, 0, 0}, 313.309020, 0.000040, 0.039000},
	{{0, 0, 1, 0, 0, -4, 0, 0, -2, 0, 0}, 108.780120, 0.000040, 0.042000},
	{{0, 0, 1, 0, 0, -4, 0, 0, -2, 1, 0}, 109.044540, 0.000050, 0.094000},
	{{0, 0, 1, 0, 0, -4, 0, 0, 0, -1, 0}, 110.457120, 0.000040, 0.081000},
	{{0, 0, 1, 0, 0, -4, 0, 0, 0, 0, 0}, 113.275430, 0.000190, 1.157000},
	{{0, 0, 1, 0, 0, -4, 0, 0, 0, 1, 0}, 110.457120, 0.000040, 0.071000},
	{{0, 0, 1, 0, 0, -4, 0, 0, 1, -1, 0}, 110.385510, 0.000010, 43.858000},
	{{0, 0, 1, 0, 0, -4, 0, 0, 1, 0, -1}, 299.237790, 0.000030, 5.276000},
	{{0, 0, 1, 0, 0, -4, 0, 0, 2, -1, 0}, 108.766290, 0.000020, 0.081000},
	{{0, 0, 1, 0, 0, -3, 0, 0, -4, 1, 0}, 16.832840, 0.000010, 0.028000},
	{{0, 0, 1, 0, 0, -3, 0, 0, -2, -1, 0}, 16.685560, 0.000040, 0.027000},
	{{0, 0, 1, 0, 0, -3, 0, 0, -2, 0, 0}, 16.509390, 0.000380, 0.042000},
	{{0, 0, 1, 0, 0, -3, 0, 0, -2, 1, 0}, 17.035480, 0.000460, 0.094000},
	{{0, 0, 1, 0, 0, -3, 0, 0, -2, 2, 0}, 17.045710, 0.000010, 0.374000},
	{{0, 0, 1, 0, 0, -3, 0, 0, -1, 0, 0}, 17.014660, 0.000080, 0.087000},
	{{0, 0, 1, 0, 0, -3, 0, 0, -1, 0, 1}, 206.547760, 0.000040, 0.512000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 0, -2, 0}, 16.928960, 0.000020, 0.039000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 0, -1, 0}, 16.840040, 0.000370, 0.081000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 0, 0, 0}, 16.547440, 0.001720, 1.113000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 0, 1, 0}, 16.767870, 0.000270, 0.071000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 0, 2, 0}, 16.931900, 0.000020, 0.036000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 1, -1, -1}, 206.530660, 0.000050, 0.075000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 1, -1, 0}, 16.974030, 0.000450, 89.713000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 1, -1, 1}, 206.530710, 0.000070, 0.074000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 1, 0, -1}, 206.521040, 0.000370, 6.426000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 1, 0, 0}, 197.017380, 0.000370, 0.075000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 1, 0, 1}, 26.522780, 0.000080, 0.037000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 1, 1, -1}, 206.530660, 0.000050, 0.076000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 1, 1, 0}, 197.014670, 0.000030, 0.038000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 2, -1, 0}, 16.794340, 0.000130, 0.081000},
	{{0, 0, 1, 0, 0, -3, 0, 0, 2, 0, 0}, 16.944540, 0.000070, 0.039000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -4, 0, 0}, 285.950260, 0.000030, 0.021000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -4, 1, 0}, 285.822110, 0.000070, 0.028000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -4, 2, 0}, 285.946790, 0.000050, 0.045000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -2, -2, 0}, 286.005820, 0.000030, 0.020000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -2, -1, 0}, 285.681580, 0.000210, 0.027000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -2, 0, 0}, 285.716650, 0.002450, 0.042000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -2, 0, 2}, 286.277240, 0.000040, 0.329000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -2, 1, 0}, 285.986620, 0.003120, 0.095000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -2, 2, 0}, 286.408250, 0.000140, 0.370000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -1, -1, 1}, 293.922210, 0.000050, 0.089000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -1, 0, -1}, 293.921920, 0.000020, 0.040000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -1, 0, 0}, 286.300100, 0.000080, 0.087000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -1, 0, 1}, 113.922090, 0.000180, 0.504000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -1, 1, -1}, 293.922090, 0.000060, 0.086000},
	{{0, 0, 1, 0, 0, -2, 0, 0, -1, 1, 1}, 113.922180, 0.000040, 0.066000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 0, -2, 0}, 285.614710, 0.000170, 0.039000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 0, -1, 0}, 285.656970, 0.002450, 0.081000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 0, 0, 0}, 285.556000, 0.010960, 1.073000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 0, 1, 0}, 285.692710, 0.001860, 0.070000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 0, 2, 0}, 285.633240, 0.000120, 0.036000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 0, 3, 0}, 105.976840, 0.000010, 0.025000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, -2, -1}, 113.922170, 0.000030, 0.038000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, -2, 0}, 110.516660, 0.000010, 0.076000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, -2, 1}, 293.922310, 0.000020, 4.720000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, -1, -1}, 113.922170, 0.000390, 0.075000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, -1, 0}, 285.929790, 0.000100, 22.176000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, -1, 1}, 113.922190, 0.000380, 0.074000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, 0, -1}, 113.922220, 0.003070, 8.220000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, 0, 0}, 106.272910, 0.000660, 0.075000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, 0, 1}, 293.922220, 0.000730, 0.037000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, 1, -1}, 113.922170, 0.000390, 0.076000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, 1, 0}, 106.272930, 0.000040, 0.038000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, 1, 1}, 293.922210, 0.000080, 0.025000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 1, 2, -1}, 113.922170, 0.000030, 0.038000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 2, -2, 0}, 107.465500, 0.000020, 1.188000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 2, -1, 0}, 286.691280, 0.000620, 0.081000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 2, 0, -2}, 286.406760, 0.000020, 0.851000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 2, 0, 0}, 286.250590, 0.000420, 0.039000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 2, 1, 0}, 286.073650, 0.000060, 0.026000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 3, -1, -1}, 293.922070, 0.000030, 0.088000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 3, -1, 1}, 293.922210, 0.000020, 0.026000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 4, -2, 0}, 283.116320, 0.000010, 0.042000},
	{{0, 0, 1, 0, 0, -2, 0, 0, 4, -1, 0}, 286.499690, 0.000010, 0.027000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -4, -2, 0}, 0.722160, 0.000010, 0.013000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -4, 0, 0}, 0.220810, 0.000120, 0.021000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -4, 1, 0}, 0.669070, 0.000270, 0.028000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -4, 2, 0}, 0.700820, 0.000130, 0.045000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -3, 0, 0}, 0.844420, 0.000020, 0.028000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, -2, 0}, 0.684890, 0.000090, 0.020000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, -1, 0}, 0.643940, 0.000850, 0.027000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, -1, 2}, 180.697330, 0.000020, 0.098000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, 0, -2}, 180.661540, 0.000020, 0.020000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, 0, 0}, 0.725060, 0.009590, 0.042000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, 0, 2}, 0.712520, 0.000140, 0.325000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, 1, 0}, 0.684920, 0.012110, 0.095000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, 1, 2}, 180.697330, 0.000020, 0.061000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, 2, 0}, 0.755170, 0.000530, 0.365000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -2, 3, 0}, 0.811290, 0.000030, 0.063000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -1, -1, 0}, 0.296480, 0.000020, 0.041000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -1, 0, 0}, 358.506480, 0.000040, 0.088000},
	{{0, 0, 1, 0, 0, -1, 0, 0, -1, 0, 1}, 169.361600, 0.000030, 0.495000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, -3, 0}, 0.447990, 0.000030, 0.026000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, -2, 0}, 0.484170, 0.000580, 0.039000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, -1, -2}, 180.456160, 0.000030, 0.026000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, -1, 0}, 0.457990, 0.008630, 0.081000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, -1, 2}, 0.464610, 0.000010, 0.069000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, 0, -2}, 180.526630, 0.000030, 0.039000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, 0, 0}, 0.432960, 0.038970, 1.035000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, 0, 2}, 180.599610, 0.000020, 0.036000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, 1, -2}, 0.464610, 0.000010, 0.079000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, 1, 0}, 0.404770, 0.006540, 0.070000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, 1, 2}, 180.456160, 0.000030, 0.024000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, 2, 0}, 0.410530, 0.000440, 0.036000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 0, 3, 0}, 0.286880, 0.000030, 0.025000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, -2, 0}, 181.886570, 0.000020, 0.076000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, -1, -1}, 170.106570, 0.000080, 0.075000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, -1, 0}, 181.444880, 0.000010, 12.652000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, -1, 1}, 170.161040, 0.000060, 0.074000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, 0, -1}, 170.126320, 0.000660, 11.401000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, 0, 0}, 180.312630, 0.001540, 0.075000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, 0, 1}, 350.123120, 0.000200, 0.037000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, 1, -1}, 170.106570, 0.000080, 0.076000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, 1, 0}, 180.318360, 0.000110, 0.038000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 1, 1, 1}, 350.123120, 0.000020, 0.025000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 2, -2, 0}, 179.996010, 0.000110, 1.238000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 2, -1, 0}, 0.182150, 0.002250, 0.080000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 2, 0, -2}, 0.227880, 0.000090, 0.876000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 2, 0, 0}, 0.144370, 0.001640, 0.039000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 2, 1, 0}, 0.410380, 0.000200, 0.026000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 3, -1, 0}, 180.296480, 0.000020, 0.040000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 4, -2, 0}, 0.271180, 0.000020, 0.042000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 4, -1, 0}, 0.443690, 0.000030, 0.027000},
	{{0, 0, 1, 0, 0, -1, 0, 0, 4, 0, 0}, 0.147000, 0.000020, 0.020000},
	{{0, 0, 1, 0, 0, 0, -3, 0, -2, 0, 0}, 92.895540, 0.000020, 0.042000},
	{{0, 0, 1, 0, 0, 0, -3, 0, 0, 0, 0}, 92.951490, 0.000070, 1.037000},
	{{0, 0, 1, 0, 0, 0, -3, 2, 0, 0, 0}, 100.705230, 0.000010, 1.024000},
	{{0, 0, 1, 0, 0, 0, -2, 0, -2, -1, 0}, 276.998750, 0.000020, 0.027000},
	{{0, 0, 1, 0, 0, 0, -2, 0, -2, 0, 0}, 276.998520, 0.000130, 0.042000},
	{{0, 0, 1, 0, 0, 0, -2, 0, -2, 1, 0}, 276.996290, 0.000180, 0.095000},
	{{0, 0, 1, 0, 0, 0, -2, 0, 0, -1, 0}, 277.013340, 0.000150, 0.081000},
	{{0, 0, 1, 0, 0, 0, -2, 0, 0, 0, 0}, 276.999930, 0.000680, 1.024000},
	{{0, 0, 1, 0, 0, 0, -2, 0, 0, 1, 0}, 277.019240, 0.000110, 0.070000},
	{{0, 0, 1, 0, 0, 0, -2, 0, 1, 0, -1}, 74.074920, 0.000040, 12.892000},
	{{0, 0, 1, 0, 0, 0, -2, 0, 1, 0, 1}, 254.075090, 0.000020, 0.037000},
	{{0, 0, 1, 0, 0, 0, -2, 0, 2, -1, 0}, 276.183120, 0.000020, 0.080000},
	{{0, 0, 1, 0, 0, 0, -2, 0, 2, 0, 0}, 276.826180, 0.000020, 0.039000},
	{{0, 0, 1, 0, 0, 0, -1, -2, 0, 0, 0}, 273.086700, 0.000010, 1.025000},
	{{0, 0, 1, 0, 0, 0, -1, 0, -2, -1, 0}, 353.644410, 0.000040, 0.027000},
	{{0, 0, 1, 0, 0, 0, -1, 0, -2, 0, 0}, 353.727370, 0.000230, 0.042000},
	{{0, 0, 1, 0, 0, 0, -1, 0, -2, 1, 0}, 353.572970, 0.000310, 0.095000},
	{{0, 0, 1, 0, 0, 0, -1, 0, 0, -1, 0}, 354.944610, 0.000250, 0.082000},
	{{0, 0, 1, 0, 0, 0, -1, 0, 0, 0, 0}, 356.071180, 0.001040, 1.012000},
	{{0, 0, 1, 0, 0, 0, -1, 0, 0, 1, 0}, 355.466050, 0.000170, 0.070000},
	{{0, 0, 1, 0, 0, 0, -1, 0, 0, 2, 0}, 354.708830, 0.000010, 0.036000},
	{{0, 0, 1, 0, 0, 0, -1, 0, 1, 0, 0}, 176.643630, 0.000020, 0.075000},
	{{0, 0, 1, 0, 0, 0, -1, 0, 2, -1, 0}, 7.271700, 0.000040, 0.080000},
	{{0, 0, 1, 0, 0, 0, -1, 0, 2, 0, 0}, 358.448110, 0.000040, 0.039000},
	{{0, 0, 1, 0, 0, 0, 0, -2, -2, 0, 0}, 283.594470, 0.000090, 0.042000},
	{{0, 0, 1, 0, 0, 0, 0, -2, -2, 1, 0}, 283.594960, 0.000120, 0.095000},
	{{0, 0, 1, 0, 0, 0, 0, -2, 0, -1, 0}, 283.586670, 0.000100, 0.082000},
	{{0, 0, 1, 0, 0, 0, 0, -2, 0, 0, 0}, 283.573150, 0.000420, 1.012000},
	{{0, 0, 1, 0, 0, 0, 0, -2, 0, 1, 0}, 283.585740, 0.000090, 0.070000},
	{{0, 0, 1, 0, 0, 0, 0, -2, 1, 0, -1}, 131.785140, 0.000060, 15.174000},
	{{0, 0, 1, 0, 0, 0, 0, -2, 1, 0, 1}, 311.785130, 0.000030, 0.037000},
	{{0, 0, 1, 0, 0, 0, 0, -2, 2, -1, 0}, 283.531880, 0.000010, 0.080000},
	{{0, 0, 1, 0, 0, 0, 0, -2, 2, 0, 0}, 283.568640, 0.000010, 0.039000},
	{{0, 0, 1, 0, 0, 0, 0, -1, -2, 0, 0}, 1.699660, 0.000090, 0.042000},
	{{0, 0, 1, 0, 0, 0, 0, -1, -2, 1, 0}, 1.716130, 0.000110, 0.095000},
	{{0, 0, 1, 0, 0, 0, 0, -1, 0, -1, 0}, 1.378500, 0.000090, 0.082000},
	{{0, 0, 1, 0, 0, 0, 0, -1, 0, 0, 0}, 0.879070, 0.000410, 1.006000},
	{{0, 0, 1, 0, 0, 0, 0, -1, 0, 1, 0}, 1.378500, 0.000090, 0.070000},
	{{0, 0, 1, 0, 0, 0, 0, -1, 2, -1, 0}, 358.202650, 0.000010, 0.080000},
	{{0, 0, 1, 0, 0, 0, 0, -1, 2, 0, 0}, 0.348160, 0.000020, 0.039000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -4, 0, 0}, 72.280170, 0.000120, 0.021000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -4, 1, 0}, 66.933950, 0.000120, 0.028000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -4, 2, 0}, 68.063770, 0.000070, 0.046000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -2, -2, 0}, 60.297670, 0.000030, 0.020000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -2, -1, 0}, 67.945370, 0.000300, 0.027000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -2, 0, 0}, 64.630660, 0.003090, 0.042000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -2, 0, 1}, 103.346680, 0.000020, 0.097000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -2, 0, 2}, 61.970760, 0.000040, 0.322000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -2, 1, 0}, 63.155790, 0.003900, 0.095000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -2, 2, 0}, 60.565400, 0.000190, 0.360000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -2, 3, 0}, 89.515590, 0.000040, 0.062000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, -2, 1}, 112.421270, 0.000010, 0.041000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, -1, -1}, 95.133710, 0.000020, 0.026000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, -1, 0}, 84.882270, 0.000020, 0.041000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, -1, 1}, 278.392870, 0.000100, 0.089000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, 0, -1}, 91.718700, 0.000220, 0.040000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, 0, 0}, 108.921320, 0.000040, 0.088000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, 0, 1}, 279.632890, 0.001120, 0.487000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, 1, -1}, 91.642970, 0.000440, 0.087000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, 1, 1}, 287.430490, 0.000210, 0.065000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, 2, -1}, 88.904180, 0.000030, 0.581000},
	{{0, 0, 1, 0, 0, 0, 0, 0, -1, 2, 1}, 312.277070, 0.000030, 0.035000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 0, -3, 0}, 91.642310, 0.000010, 0.026000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 0, -2, 0}, 90.348740, 0.000250, 0.039000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 0, -1, 0}, 90.297810, 0.003550, 0.082000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 0, 0, -1}, 281.288590, 0.000100, 0.081000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}, 95.383940, 0.016990, 1.000000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1}, 103.351810, 0.000080, 0.069000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 0, 1, -1}, 283.351350, 0.000010, 1.200000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0}, 88.409240, 0.002670, 0.070000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 0, 2, 0}, 89.853060, 0.000190, 0.036000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, -2, -1}, 276.127010, 0.000070, 0.038000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, -1, -1}, 274.427300, 0.000910, 0.075000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, -1, 0}, 262.194150, 0.000080, 8.850000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, -1, 1}, 288.201320, 0.000850, 0.074000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, 0, -1}, 298.979350, 0.004560, 18.600000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0}, 264.995360, 0.001160, 0.075000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1}, 112.295960, 0.006250, 0.037000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, 1, -1}, 274.148070, 0.001000, 0.076000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 0}, 264.803050, 0.000080, 0.038000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, 1, 1}, 111.925070, 0.000700, 0.025000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, 2, -1}, 271.437340, 0.000060, 0.038000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 1, 2, 1}, 109.287900, 0.000070, 0.019000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 2, -3, 0}, 293.732610, 0.000020, 0.071000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 2, -2, 0}, 303.054330, 0.000200, 1.292000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 2, -1, 0}, 16.061550, 0.002420, 0.080000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 2, 0, -2}, 52.616150, 0.000040, 0.903000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 2, 0, -1}, 99.837470, 0.000060, 0.081000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 2, 0, 0}, 45.434830, 0.000930, 0.039000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 2, 1, 0}, 57.734790, 0.000120, 0.026000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 3, -3, -1}, 77.063070, 0.000020, 0.066000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 3, -2, -1}, 85.729300, 0.000010, 0.547000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 3, -2, 1}, 289.297150, 0.000020, 0.040000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 3, -1, -1}, 96.602470, 0.000220, 0.088000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 3, -1, 0}, 264.882270, 0.000020, 0.040000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 3, -1, 1}, 111.961230, 0.000140, 0.026000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 3, 0, -1}, 111.437770, 0.000210, 0.041000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 3, 0, 1}, 111.961220, 0.000090, 0.019000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 3, 1, -1}, 111.963950, 0.000030, 0.026000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 4, -2, 0}, 35.668160, 0.000040, 0.042000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 4, -1, 0}, 5.708280, 0.000030, 0.027000},
	{{0, 0, 1, 0, 0, 0, 0, 0, 4, 0, 0}, 354.151500, 0.000020, 0.020000},
	{{0, 0, 1, 0, 0, 0, 1, -2, -2, 0, 0}, 163.791360, 0.000020, 0.042000},
	{{0, 0, 1, 0, 0, 0, 1, -2, 0, 0, 0}, 176.834450, 0.000060, 1.000000},
	{{0, 0, 1, 0, 0, 0, 1, 0, -2, 0, 0}, 83.748990, 0.000030, 0.042000},
	{{0, 0, 1, 0, 0, 0, 1, 0, 0, -1, 0}, 84.425060, 0.000010, 0.082000},
	{{0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0}, 86.976440, 0.000040, 0.988000},
	{{0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0}, 84.425060, 0.000010, 0.070000},
	{{0, 0, 1, 0, 0, 1, -3, 0, -2, 0, 0}, 24.731290, 0.000020, 0.042000},
	{{0, 0, 1, 0, 0, 1, -3, 0, 0, 0, 0}, 23.313550, 0.000030, 1.002000},
	{{0, 0, 1, 0, 0, 1, 0, 0, -4, 1, 0}, 165.687700, 0.000010, 0.028000},
	{{0, 0, 1, 0, 0, 1, 0, 0, -2, -1, 0}, 165.720630, 0.000040, 0.027000},
	{{0, 0, 1, 0, 0, 1, 0, 0, -2, 0, 0}, 165.361760, 0.000440, 0.042000},
	{{0, 0, 1, 0, 0, 1, 0, 0, -2, 1, 0}, 166.624760, 0.000470, 0.096000},
	{{0, 0, 1, 0, 0, 1, 0, 0, -2, 2, 0}, 165.653090, 0.000030, 0.356000},
	{{0, 0, 1, 0, 0, 1, 0, 0, -1, 0, -1}, 334.357540, 0.000040, 0.040000},
	{{0, 0, 1, 0, 0, 1, 0, 0, -1, 0, 1}, 334.905040, 0.000040, 0.479000},
	{{0, 0, 1, 0, 0, 1, 0, 0, -1, 1, -1}, 154.072840, 0.000100, 0.087000},
	{{0, 0, 1, 0, 0, 1, 0, 0, -1, 2, -1}, 154.144830, 0.000010, 0.570000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 0, -1, 0}, 164.605350, 0.000220, 0.082000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0}, 167.081400, 0.000660, 0.967000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0}, 171.442690, 0.000060, 0.070000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 1, -2, -1}, 334.181740, 0.000070, 0.038000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 1, -1, -1}, 334.181740, 0.000990, 0.075000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 1, -1, 1}, 334.298320, 0.000020, 0.074000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 1, 0, -1}, 334.276320, 0.005070, 50.462000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 1}, 334.188060, 0.000570, 0.037000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 1, 1, -1}, 334.181740, 0.000990, 0.076000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 1, 1, 1}, 334.189590, 0.000060, 0.025000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 1, 2, -1}, 334.181740, 0.000070, 0.038000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 2, -1, 0}, 348.999770, 0.000020, 0.080000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 2, 0, 0}, 348.371320, 0.000020, 0.039000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 3, -2, -1}, 154.144830, 0.000010, 0.557000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 3, -1, -1}, 154.044240, 0.000070, 0.087000},
	{{0, 0, 1, 0, 0, 1, 0, 0, 3, 0, -1}, 334.241350, 0.000080, 0.040000},
	{{0, 0, 1, 0, 0, 2, 0, 0, -2, 0, 0}, 63.329970, 0.000030, 0.042000},
	{{0, 0, 1, 0, 0, 2, 0, 0, -2, 1, 0}, 63.412850, 0.000020, 0.096000},
	{{0, 0, 1, 0, 0, 2, 0, 0, -1, 0, -1}, 236.290920, 0.000010, 0.040000},
	{{0, 0, 1, 0, 0, 2, 0, 0, -1, 1, -1}, 57.452820, 0.000010, 0.087000},
	{{0, 0, 1, 0, 0, 2, 0, 0, 0, 0, 0}, 66.095020, 0.000040, 0.936000},
	{{0, 0, 1, 0, 0, 2, 0, 0, 1, -1, -1}, 236.729330, 0.000170, 0.076000},
	{{0, 0, 1, 0, 0, 2, 0, 0, 1, 0, -1}, 236.290920, 0.000950, 70.769000},
	{{0, 0, 1, 0, 0, 2, 0, 0, 1, 0, 1}, 55.590230, 0.000030, 0.037000},
	{{0, 0, 1, 0, 0, 2, 0, 0, 1, 1, -1}, 236.729330, 0.000170, 0.075000},
	{{0, 0, 1, 0, 0, 2, 0, 0, 3, -1, -1}, 57.452820, 0.000010, 0.087000},
	{{0, 0, 1, 0, 0, 2, 0, 0, 3, 0, -1}, 236.290920, 0.000010, 0.040000},
	{{0, 0, 1, 0, 0, 3, 0, 0, 1, 0, -1}, 139.652310, 0.000010, 20.799000},
	{{0, 0, 1, 0, 1, -6, 3, 0, 0, 0, 0}, 192.392730, 0.000020, 1.091000},
	{{0, 0, 1, 0, 1, -5, 0, 0, -4, 1, 0}, 11.668010, 0.000020, 0.028000},
	{{0, 0, 1, 0, 1, -5, 0, 0, -2, -1, 0}, 11.693930, 0.000050, 0.027000},
	{{0, 0, 1, 0, 1, -5, 0, 0, -2, 0, 0}, 11.705270, 0.000450, 0.042000},
	{{0, 0, 1, 0, 1, -5, 0, 0, -2, 1, 0}, 11.621280, 0.000610, 0.095000},
	{{0, 0, 1, 0, 1, -5, 0, 0, -2, 2, 0}, 11.678160, 0.000020, 0.372000},
	{{0, 0, 1, 0, 1, -5, 0, 0, -1, 0, 0}, 11.207610, 0.000040, 0.087000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 0, -2, 0}, 11.931440, 0.000020, 0.039000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 0, -1, 0}, 11.830280, 0.000410, 0.081000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 0, 0, 0}, 11.178670, 0.001820, 1.093000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 0, 1, 0}, 11.745290, 0.000290, 0.071000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 0, 2, 0}, 11.960650, 0.000020, 0.036000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 1, -1, 0}, 10.915980, 0.000070, 36.292000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 1, 0, 0}, 191.207630, 0.000190, 0.075000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 2, -3, 0}, 189.789910, 0.000010, 0.071000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 2, -1, 0}, 8.787020, 0.000120, 0.081000},
	{{0, 0, 1, 0, 1, -5, 0, 0, 2, 0, 0}, 10.383190, 0.000070, 0.039000},
	{{0, 0, 1, 0, 1, -4, 0, 0, -2, 0, 0}, 239.609960, 0.000020, 0.042000},
	{{0, 0, 1, 0, 1, -4, 0, 0, 0, 0, 0}, 239.605910, 0.000030, 1.054000},
	{{0, 0, 1, 0, 1, -3, 0, 0, -2, -1, 0}, 308.900790, 0.000020, 0.027000},
	{{0, 0, 1, 0, 1, -3, 0, 0, -2, 0, 0}, 308.900790, 0.000090, 0.042000},
	{{0, 0, 1, 0, 1, -3, 0, 0, -2, 1, 0}, 308.907050, 0.000120, 0.095000},
	{{0, 0, 1, 0, 1, -3, 0, 0, 0, -1, 0}, 308.372610, 0.000090, 0.081000},
	{{0, 0, 1, 0, 1, -3, 0, 0, 0, 0, 0}, 306.818220, 0.000400, 1.018000},
	{{0, 0, 1, 0, 1, -3, 0, 0, 0, 1, 0}, 308.152540, 0.000070, 0.070000},
	{{0, 0, 1, 0, 1, -3, 0, 0, 1, 0, -1}, 136.656530, 0.000010, 14.024000},
	{{0, 0, 1, 0, 1, -3, 0, 0, 2, -1, 0}, 298.880180, 0.000010, 0.080000},
	{{0, 0, 1, 0, 1, -3, 0, 0, 2, 0, 0}, 305.228580, 0.000010, 0.039000},
	{{0, 0, 1, 0, 1, -2, 0, 0, -4, 1, 0}, 183.232210, 0.000010, 0.028000},
	{{0, 0, 1, 0, 1, -2, 0, 0, -2, -1, 0}, 183.212510, 0.000050, 0.027000},
	{{0, 0, 1, 0, 1, -2, 0, 0, -2, 0, 0}, 183.157740, 0.000450, 0.042000},
	{{0, 0, 1, 0, 1, -2, 0, 0, -2, 1, 0}, 183.228920, 0.000570, 0.096000},
	{{0, 0, 1, 0, 1, -2, 0, 0, -2, 2, 0}, 183.266140, 0.000030, 0.358000},
	{{0, 0, 1, 0, 1, -2, 0, 0, 0, -2, 0}, 182.258350, 0.000020, 0.039000},
	{{0, 0, 1, 0, 1, -2, 0, 0, 0, -1, 0}, 182.071200, 0.000430, 0.082000},
	{{0, 0, 1, 0, 1, -2, 0, 0, 0, 0, 0}, 181.054940, 0.001860, 0.984000},
	{{0, 0, 1, 0, 1, -2, 0, 0, 0, 1, 0}, 181.596780, 0.000270, 0.070000},
	{{0, 0, 1, 0, 1, -2, 0, 0, 0, 2, 0}, 182.367470, 0.000020, 0.036000},
	{{0, 0, 1, 0, 1, -2, 0, 0, 1, 0, -1}, 79.182080, 0.000020, 26.769000},
	{{0, 0, 1, 0, 1, -2, 0, 0, 1, 0, 0}, 0.571330, 0.000030, 0.075000},
	{{0, 0, 1, 0, 1, -2, 0, 0, 2, -1, 0}, 175.518580, 0.000070, 0.080000},
	{{0, 0, 1, 0, 1, -2, 0, 0, 2, 0, 0}, 179.153150, 0.000060, 0.039000},
	{{0, 0, 1, 0, 1, -1, 0, 0, -2, 0, 0}, 66.363150, 0.000020, 0.042000},
	{{0, 0, 1, 0, 1, -1, 0, 0, -2, 1, 0}, 66.363150, 0.000020, 0.096000},
	{{0, 0, 1, 0, 1, -1, 0, 0, -1, 0, -1}, 273.735230, 0.000060, 0.040000},
	{{0, 0, 1, 0, 1, -1, 0, 0, -1, 1, -1}, 94.199330, 0.000070, 0.087000},
	{{0, 0, 1, 0, 1, -1, 0, 0, -1, 2, -1}, 93.990440, 0.000010, 0.565000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 0, 0, 0}, 34.372380, 0.000040, 0.952000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 1, -2, -1}, 273.911230, 0.000060, 0.038000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 1, -1, -1}, 273.911230, 0.000870, 0.075000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 1, 0, -1}, 273.735230, 0.004910, 293.315000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 1, 0, 1}, 274.417680, 0.000050, 0.037000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 1, 1, -1}, 273.911230, 0.000870, 0.075000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 1, 2, -1}, 273.911230, 0.000060, 0.038000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 3, -2, -1}, 93.990440, 0.000010, 0.563000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 3, -1, -1}, 94.199330, 0.000070, 0.087000},
	{{0, 0, 1, 0, 1, -1, 0, 0, 3, 0, -1}, 273.735230, 0.000060, 0.040000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -4, -1, 0}, 72.577940, 0.000020, 0.016000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -4, 0, 0}, 246.273040, 0.000090, 0.021000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -4, 1, 0}, 249.376830, 0.000170, 0.028000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -4, 2, 0}, 251.658100, 0.000090, 0.046000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -3, 0, 0}, 247.551970, 0.000010, 0.028000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -2, -2, 0}, 244.678310, 0.000030, 0.020000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -2, -1, 0}, 241.020630, 0.000350, 0.027000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -2, 0, 0}, 244.627270, 0.004080, 0.042000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -2, 0, 2}, 245.484420, 0.000060, 0.313000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -2, 1, 0}, 247.510340, 0.004740, 0.096000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -2, 2, 0}, 251.712720, 0.000220, 0.350000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -1, -1, -1}, 75.536530, 0.000060, 0.026000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -1, -1, 1}, 76.898210, 0.000030, 0.090000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -1, 0, -1}, 75.547880, 0.000410, 0.040000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -1, 0, 0}, 59.881380, 0.000030, 0.089000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -1, 0, 1}, 256.093040, 0.000360, 0.468000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -1, 1, -1}, 255.894110, 0.000220, 0.087000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -1, 1, 1}, 254.504880, 0.000020, 0.065000},
	{{0, 0, 1, 0, 1, 0, 0, 0, -1, 2, -1}, 255.376590, 0.000060, 0.554000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 0, -2, 0}, 226.444370, 0.000140, 0.039000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 0, -1, 0}, 227.426820, 0.001880, 0.082000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0}, 235.356960, 0.007200, 0.922000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0}, 241.158900, 0.001090, 0.070000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 0, 2, 0}, 245.658000, 0.000050, 0.036000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, -3, -1}, 75.627420, 0.000010, 0.025000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, -2, -1}, 75.355870, 0.000290, 0.038000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, -1, -1}, 75.575870, 0.004260, 0.076000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, -1, 0}, 57.703210, 0.000020, 5.069000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, -1, 1}, 255.798870, 0.000290, 0.073000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, 0, -3}, 257.537170, 0.000030, 0.037000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, 0, -1}, 75.471820, 0.025400, 32.746000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0}, 38.926340, 0.000070, 0.074000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1}, 255.515320, 0.001530, 0.037000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, 1, -1}, 75.578670, 0.004240, 0.075000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, 1, 1}, 255.508520, 0.000160, 0.025000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, 2, -1}, 75.577270, 0.000290, 0.038000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 1, 3, -1}, 75.627420, 0.000010, 0.025000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 2, -2, 0}, 19.851000, 0.000040, 1.450000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 2, -1, 0}, 60.538650, 0.000170, 0.080000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 2, 0, 0}, 96.922480, 0.000100, 0.039000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 3, -2, -1}, 255.627420, 0.000050, 0.574000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 3, -1, -1}, 255.740480, 0.000330, 0.087000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 3, -1, 1}, 255.508610, 0.000030, 0.026000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 3, 0, -1}, 75.477180, 0.000230, 0.040000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 3, 0, 1}, 255.508800, 0.000020, 0.019000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 3, 1, -1}, 75.536530, 0.000060, 0.026000},
	{{0, 0, 1, 0, 1, 0, 0, 0, 4, -1, 0}, 67.037280, 0.000010, 0.027000},
	{{0, 0, 1, 0, 1, 5, 0, 0, 0, -1, 0}, 299.544010, 0.000020, 0.083000},
	{{0, 0, 1, 0, 1, 5, 0, 0, 0, 1, 0}, 299.544010, 0.000020, 0.069000},
	{{0, 0, 1, 0, 1, 5, 0, 0, 2, -2, 0}, 299.686450, 0.000010, 1.923000},
	{{0, 0, 1, 0, 1, 5, 0, 0, 2, 0, 0}, 299.686450, 0.000010, 0.038000},
	{{0, 0, 1, 0, 2, -8, 0, 0, 0, 0, 0}, 205.290390, 0.000030, 1.115000},
	{{0, 0, 1, 0, 2, -7, 0, 0, -2, 0, 0}, 113.766450, 0.000040, 0.042000},
	{{0, 0, 1, 0, 2, -7, 0, 0, -2, 1, 0}, 113.766450, 0.000050, 0.095000},
	{{0, 0, 1, 0, 2, -7, 0, 0, 0, -1, 0}, 113.669360, 0.000030, 0.081000},
	{{0, 0, 1, 0, 2, -7, 0, 0, 0, 0, 0}, 113.558640, 0.000160, 1.074000},
	{{0, 0, 1, 0, 2, -7, 0, 0, 0, 1, 0}, 113.669360, 0.000030, 0.070000},
	{{0, 0, 1, 0, 2, -7, 0, 0, 1, 0, -1}, 302.634600, 0.000040, 8.144000},
	{{0, 0, 1, 0, 2, -6, 0, 0, -2, 0, 0}, 191.225420, 0.000080, 0.042000},
	{{0, 0, 1, 0, 2, -6, 0, 0, -2, 1, 0}, 191.230190, 0.000080, 0.095000},
	{{0, 0, 1, 0, 2, -6, 0, 0, 0, -1, 0}, 191.098350, 0.000060, 0.081000},
	{{0, 0, 1, 0, 2, -6, 0, 0, 0, 0, 0}, 190.860720, 0.000280, 1.036000},
	{{0, 0, 1, 0, 2, -6, 0, 0, 0, 1, 0}, 191.098350, 0.000060, 0.070000},
	{{0, 0, 1, 0, 2, -6, 0, 0, 2, -1, 0}, 189.893820, 0.000020, 0.080000},
	{{0, 0, 1, 0, 2, -6, 0, 0, 2, 0, 0}, 190.329940, 0.000020, 0.039000},
	{{0, 0, 1, 0, 2, -6, 3, 0, -2, 0, 0}, 257.327530, 0.000030, 0.042000},
	{{0, 0, 1, 0, 2, -6, 3, 0, -2, 1, 0}, 257.327530, 0.000020, 0.095000},
	{{0, 0, 1, 0, 2, -6, 3, 0, 0, -1, 0}, 257.327530, 0.000010, 0.082000},
	{{0, 0, 1, 0, 2, -6, 3, 0, 0, 0, 0}, 255.739600, 0.000090, 0.999000},
	{{0, 0, 1, 0, 2, -6, 3, 0, 0, 1, 0}, 257.327530, 0.000010, 0.070000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -4, -1, 0}, 67.015600, 0.000010, 0.016000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -4, 0, 0}, 65.533010, 0.000050, 0.021000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -4, 1, 0}, 66.527060, 0.000140, 0.028000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -4, 2, 0}, 66.957990, 0.000080, 0.046000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -2, -2, 0}, 66.930710, 0.000040, 0.020000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -2, -1, 0}, 67.265640, 0.000320, 0.027000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -2, 0, -2}, 246.820260, 0.000010, 0.020000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -2, 0, 0}, 67.171100, 0.004230, 0.042000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -2, 0, 2}, 67.085340, 0.000060, 0.322000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -2, 1, 0}, 66.729350, 0.005860, 0.095000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -2, 2, 0}, 66.577090, 0.000290, 0.361000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -2, 3, 0}, 67.222250, 0.000020, 0.062000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -1, -1, 1}, 58.999110, 0.000020, 0.089000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -1, 0, 0}, 255.302910, 0.000010, 0.088000},
	{{0, 0, 1, 0, 2, -5, 0, 0, -1, 0, 1}, 239.044330, 0.000030, 0.487000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 0, -3, 0}, 65.685120, 0.000010, 0.026000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 0, -2, 0}, 66.660600, 0.000190, 0.039000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 0, -1, 0}, 65.897320, 0.003310, 0.082000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 0, 0, 0}, 65.649390, 0.016960, 1.001000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 0, 1, 0}, 65.390640, 0.003130, 0.070000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 0, 2, 0}, 64.963120, 0.000220, 0.036000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 1, -1, -1}, 239.055260, 0.000010, 0.075000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 1, -1, 0}, 246.605830, 0.000040, 8.940000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 1, -1, 1}, 239.053420, 0.000080, 0.074000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 1, 0, -1}, 239.056180, 0.000700, 18.216000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 1, 0, 0}, 245.308440, 0.000460, 0.075000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 1, 0, 1}, 59.059280, 0.000640, 0.037000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 1, 1, -1}, 239.055260, 0.000010, 0.076000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 1, 1, 0}, 245.351940, 0.000030, 0.038000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 1, 1, 1}, 59.059360, 0.000070, 0.025000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 2, -2, 0}, 259.236960, 0.000050, 1.290000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 2, -1, 0}, 65.187150, 0.000630, 0.080000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 2, 0, -2}, 64.637060, 0.000040, 0.902000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 2, 0, 0}, 62.225150, 0.000650, 0.039000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 2, 1, 0}, 64.596150, 0.000100, 0.026000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 3, 0, -1}, 59.060880, 0.000020, 0.041000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 4, -2, 0}, 78.000110, 0.000010, 0.042000},
	{{0, 0, 1, 0, 2, -5, 0, 0, 4, -1, 0}, 64.471370, 0.000020, 0.027000},
	{{0, 0, 1, 0, 2, -4, 0, 0, -2, 0, 0}, 283.386470, 0.000020, 0.042000},
	{{0, 0, 1, 0, 2, -4, 0, 0, 0, 0, 0}, 295.939490, 0.000100, 0.968000},
	{{0, 0, 1, 0, 2, -4, 0, 0, 1, -1, -1}, 322.384080, 0.000020, 0.075000},
	{{0, 0, 1, 0, 2, -4, 0, 0, 1, 0, -1}, 323.172350, 0.000130, 47.735000},
	{{0, 0, 1, 0, 2, -4, 0, 0, 1, 0, 1}, 323.192130, 0.000020, 0.037000},
	{{0, 0, 1, 0, 2, -4, 0, 0, 1, 1, -1}, 322.384080, 0.000020, 0.076000},
	{{0, 0, 1, 0, 2, -3, 0, 0, -2, 0, 0}, 183.775140, 0.000040, 0.042000},
	{{0, 0, 1, 0, 2, -3, 0, 0, -2, 1, 0}, 183.775140, 0.000040, 0.096000},
	{{0, 0, 1, 0, 2, -3, 0, 0, 0, -1, 0}, 183.775140, 0.000010, 0.082000},
	{{0, 0, 1, 0, 2, -3, 0, 0, 0, 0, 0}, 175.387630, 0.000090, 0.937000},
	{{0, 0, 1, 0, 2, -3, 0, 0, 0, 1, 0}, 183.775140, 0.000010, 0.070000},
	{{0, 0, 1, 0, 2, -3, 0, 0, 1, -1, -1}, 30.020940, 0.000030, 0.076000},
	{{0, 0, 1, 0, 2, -3, 0, 0, 1, 0, -1}, 30.290320, 0.000180, 76.932000},
	{{0, 0, 1, 0, 2, -3, 0, 0, 1, 1, -1}, 30.020940, 0.000030, 0.075000},
	{{0, 0, 1, 0, 2, -2, 0, 0, -2, 0, 0}, 240.323910, 0.000050, 0.042000},
	{{0, 0, 1, 0, 2, -2, 0, 0, -2, 1, 0}, 240.323910, 0.000060, 0.096000},
	{{0, 0, 1, 0, 2, -2, 0, 0, 0, -1, 0}, 235.668310, 0.000020, 0.082000},
	{{0, 0, 1, 0, 2, -2, 0, 0, 0, 0, 0}, 227.605870, 0.000090, 0.909000},
	{{0, 0, 1, 0, 2, -2, 0, 0, 0, 1, 0}, 235.668310, 0.000020, 0.070000},
	{{0, 0, 1, 0, 2, -2, 0, 0, 1, -1, -1}, 93.618380, 0.000020, 0.076000},
	{{0, 0, 1, 0, 2, -2, 0, 0, 1, 0, -1}, 95.389290, 0.000160, 21.301000},
	{{0, 0, 1, 0, 2, -2, 0, 0, 1, 1, -1}, 93.618380, 0.000020, 0.075000},
	{{0, 0, 1, 0, 2, 0, 0, 0, -2, -1, 0}, 248.300680, 0.000020, 0.027000},
	{{0, 0, 1, 0, 2, 0, 0, 0, -2, 0, 0}, 247.040350, 0.000100, 0.042000},
	{{0, 0, 1, 0, 2, 0, 0, 0, -2, 1, 0}, 257.947370, 0.000100, 0.097000},
	{{0, 0, 1, 0, 2, 0, 0, 0, 0, -1, 0}, 229.803460, 0.000100, 0.083000},
	{{0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0}, 229.898700, 0.000220, 0.856000},
	{{0, 0, 1, 0, 2, 0, 0, 0, 0, 1, 0}, 232.387440, 0.000030, 0.069000},
	{{0, 0, 1, 0, 2, 0, 0, 0, 1, -1, -1}, 75.795160, 0.000010, 0.076000},
	{{0, 0, 1, 0, 2, 0, 0, 0, 1, 0, -1}, 81.426250, 0.000090, 8.708000},
	{{0, 0, 1, 0, 2, 0, 0, 0, 1, 1, -1}, 75.795160, 0.000010, 0.075000},
	{{0, 0, 1, 0, 2, 0, 0, 0, 2, -3, 0}, 213.300600, 0.000010, 0.072000},
	{{0, 0, 1, 0, 2, 0, 0, 0, 2, -1, 0}, 42.224620, 0.000020, 0.079000},
	{{0, 0, 1, 0, 2, 5, 0, 0, 2, -2, 0}, 110.160340, 0.000010, 2.295000},
	{{0, 0, 1, 0, 2, 5, 0, 0, 2, 0, 0}, 110.160340, 0.000010, 0.038000},
	{{0, 0, 1, 0, 3, -10, 0, 0, 0, 0, 0}, 200.930490, 0.000020, 1.095000},
	{{0, 0, 1, 0, 3, -7, 0, 0, -2, 0, 0}, 12.467880, 0.000020, 0.042000},
	{{0, 0, 1, 0, 3, -7, 0, 0, 0, 0, 0}, 10.653710, 0.000040, 0.985000},
	{{0, 0, 1, 0, 3, -5, 0, 0, -2, 0, 0}, 328.930240, 0.000030, 0.042000},
	{{0, 0, 1, 0, 3, -5, 0, 0, -2, 1, 0}, 329.769280, 0.000030, 0.096000},
	{{0, 0, 1, 0, 3, -5, 0, 0, 0, -1, 0}, 324.335590, 0.000010, 0.082000},
	{{0, 0, 1, 0, 3, -5, 0, 0, 0, 0, 0}, 315.408980, 0.000050, 0.923000},
	{{0, 0, 1, 0, 3, -5, 0, 0, 0, 1, 0}, 324.335590, 0.000010, 0.070000},
	{{0, 0, 1, 0, 3, -5, 0, 0, 1, -1, -1}, 170.737230, 0.000030, 0.076000},
	{{0, 0, 1, 0, 3, -5, 0, 0, 1, 0, -1}, 172.014800, 0.000190, 34.006000},
	{{0, 0, 1, 0, 3, -5, 0, 0, 1, 1, -1}, 170.737230, 0.000030, 0.075000},
	{{0, 0, 1, 0, 3, -4, 0, 0, 0, 0, 0}, 355.877050, 0.000020, 0.895000},
	{{0, 0, 1, 0, 4, -10, 0, 0, -2, 0, 0}, 258.140340, 0.000040, 0.042000},
	{{0, 0, 1, 0, 4, -10, 0, 0, -2, 1, 0}, 258.140340, 0.000040, 0.095000},
	{{0, 0, 1, 0, 4, -10, 0, 0, 0, -1, 0}, 258.140340, 0.000020, 0.082000},
	{{0, 0, 1, 0, 4, -10, 0, 0, 0, 0, 0}, 256.500760, 0.000130, 1.002000},
	{{0, 0, 1, 0, 4, -10, 0, 0, 0, 1, 0}, 258.140340, 0.000020, 0.070000},
	{{0, 0, 1, 0, 4, -1, 0, 0, 2, -3, 0}, 306.949000, 0.000010, 0.073000},
	{{0, 0, 1, 0, 4, -1, 0, 0, 2, -1, 0}, 306.949000, 0.000010, 0.078000},
	{{0, 0, 1, 0, 5, -5, 0, 0, 0, -1, 0}, 139.777280, 0.000020, 0.083000},
	{{0, 0, 1, 0, 5, -5, 0, 0, 0, 1, 0}, 139.777280, 0.000020, 0.069000},
	{{0, 0, 1, 0, 5, -5, 0, 0, 2, -2, 0}, 138.957930, 0.000010, 1.915000},
	{{0, 0, 1, 0, 5, -5, 0, 0, 2, 0, 0}, 138.957930, 0.000010, 0.038000},
	{{0, 0, 1, 0, 5, -3, 0, 0, 2, -3, 0}, 68.550960, 0.000010, 0.073000},
	{{0, 0, 1, 0, 5, -3, 0, 0, 2, -1, 0}, 68.550960, 0.000010, 0.078000},
	{{0, 0, 1, 0, 5, -2, 0, 0, 2, -2, 0}, 127.141170, 0.000020, 2.379000},
	{{0, 0, 1, 0, 5, -2, 0, 0, 2, 0, 0}, 127.141170, 0.000020, 0.038000},
	{{0, 0, 1, 0, 6, -5, 0, 0, 2, -2, 0}, 310.393610, 0.000010, 2.284000},
	{{0, 0, 1, 0, 6, -5, 0, 0, 2, 0, 0}, 310.393610, 0.000010, 0.038000},
	{{0, 0, 1, 0, 6, -2, 0, 0, 2, -3, 0}, 117.757500, 0.000010, 0.074000},
	{{0, 0, 1, 0, 6, -2, 0, 0, 2, -1, 0}, 117.757500, 0.000010, 0.077000},
	{{0, 0, 1, 0, 10, -2, 0, 0, 2, -2, 0}, 259.853630, 0.000020, 884.771000},
	{{0, 0, 1, 0, 14, -12, 0, 0, 2, -2, 0}, 262.861940, 0.000020, 881.891000},
	{{0, 0, 1, 1, -2, 0, 0, 0, 2, -3, 0}, 358.155270, 0.000010, 0.073000},
	{{0, 0, 1, 1, -2, 0, 0, 0, 2, -1, 0}, 358.155270, 0.000010, 0.078000},
	{{0, 0, 1, 1, -1, 0, 0, 0, 2, -1, 0}, 26.169570, 0.000020, 0.077000},
	{{0, 0, 1, 1, 0, 0, 0, 0, -2, -1, 0}, 274.128700, 0.000020, 0.027000},
	{{0, 0, 1, 1, 0, 0, 0, 0, -2, 0, 0}, 271.856600, 0.000140, 0.043000},
	{{0, 0, 1, 1, 0, 0, 0, 0, -2, 1, 0}, 278.929240, 0.000120, 0.101000},
	{{0, 0, 1, 1, 0, 0, 0, 0, -1, 0, 1}, 156.992560, 0.000010, 0.387000},
	{{0, 0, 1, 1, 0, 0, 0, 0, 0, -1, 0}, 252.582940, 0.000150, 0.085000},
	{{0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0}, 246.600470, 0.000530, 0.653000},
	{{0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0}, 256.441330, 0.000090, 0.068000},
	{{0, 0, 1, 1, 0, 0, 0, 0, 1, 0, -1}, 335.074160, 0.000050, 2.092000},
	{{0, 0, 1, 1, 0, 0, 0, 0, 2, -2, 0}, 58.066530, 0.000020, 4.127000},
	{{0, 0, 1, 1, 0, 0, 0, 0, 2, -1, 0}, 61.768010, 0.000280, 0.077000},
	{{0, 0, 1, 1, 0, 0, 0, 0, 2, 0, 0}, 59.599740, 0.000040, 0.038000},
	{{0, 0, 1, 1, 1, 0, 0, 0, 2, -1, 0}, 274.216650, 0.000020, 0.076000},
	{{0, 0, 1, 1, 2, 0, 0, 0, 2, -2, 0}, 302.508540, 0.000010, 13.564000},
	{{0, 0, 1, 1, 3, 0, 0, 0, 2, -2, 0}, 284.649770, 0.000010, 94.519000},
	{{0, 0, 1, 2, -3, 0, 0, 0, 2, -2, 0}, 59.881520, 0.000020, 27.444000},
	{{0, 0, 1, 2, -3, 0, 0, 0, 2, -1, 0}, 244.521680, 0.000040, 0.075000},
	{{0, 0, 1, 2, -2, 0, 0, 0, 0, 0, 0}, 209.286800, 0.000010, 0.528000},
	{{0, 0, 1, 2, -1, 0, 0, 0, 2, -1, 0}, 240.794570, 0.000020, 0.074000},
	{{0, 0, 1, 2, 0, 0, 0, 0, -2, -1, 0}, 41.419040, 0.000010, 0.028000},
	{{0, 0, 1, 2, 0, 0, 0, 0, -2, 0, 0}, 38.006990, 0.000150, 0.044000},
	{{0, 0, 1, 2, 0, 0, 0, 0, -2, 1, 0}, 42.876700, 0.000100, 0.106000},
	{{0, 0, 1, 2, 0, 0, 0, 0, -1, 0, -1}, 60.785060, 0.000050, 0.042000},
	{{0, 0, 1, 2, 0, 0, 0, 0, -1, 1, -1}, 240.858400, 0.000040, 0.095000},
	{{0, 0, 1, 2, 0, 0, 0, 0, 0, -1, 0}, 51.498930, 0.000030, 0.089000},
	{{0, 0, 1, 2, 0, 0, 0, 0, 0, 1, 0}, 68.967790, 0.000010, 0.065000},
	{{0, 0, 1, 2, 0, 0, 0, 0, 1, -1, -1}, 60.811420, 0.000070, 0.082000},
	{{0, 0, 1, 2, 0, 0, 0, 0, 1, 0, -1}, 34.027510, 0.000020, 0.991000},
	{{0, 0, 1, 2, 0, 0, 0, 0, 1, 1, -1}, 240.811420, 0.000050, 0.070000},
	{{0, 0, 1, 2, 1, 0, 0, 0, 2, 0, -2}, 281.767750, 0.000010, 24.912000},
	{{0, 0, 1, 3, -2, 0, 0, 0, 2, -1, 0}, 249.686740, 0.000020, 0.072000},
	{{0, 0, 1, 3, 0, 0, 0, 0, 2, -3, 0}, 310.748340, 0.000010, 0.080000},
	{{0, 0, 1, 3, 0, 0, 0, 0, 2, -1, 0}, 310.748340, 0.000010, 0.071000},
	{{0, 0, 1, 4, -2, 0, 0, 0, 2, -1, 0}, 278.818670, 0.000020, 0.069000},
	{{0, 0, 1, 4, -2, 0, 0, 0, 2, 0, 0}, 99.018940, 0.000020, 0.036000},
	{{0, 0, 1, 4, -1, 0, 0, 0, 2, 0, 0}, 125.417080, 0.000010, 0.036000},
	{{0, 0, 1, 5, -2, 0, 0, 0, 2, -1, 0}, 307.830830, 0.000020, 0.067000},
	{{0, 0, 1, 5, -2, 0, 0, 0, 2, 0, 0}, 308.078220, 0.000010, 0.035000},
	{{0, 0, 2, -8, 0, 0, 0, 0, -2, 1, 0}, 173.572390, 0.000010, 0.073000},
	{{0, 0, 2, -8, 0, 0, 0, 0, -2, 3, 0}, 173.572390, 0.000010, 0.078000},
	{{0, 0, 2, -8, 2, 0, 0, 0, -2, 0, 0}, 51.633150, 0.000020, 0.037000},
	{{0, 0, 2, -8, 3, 0, 0, 0, -2, 1, 0}, 150.663310, 0.000020, 0.074000},
	{{0, 0, 2, -8, 3, 0, 0, 0, -2, 3, 0}, 149.443490, 0.000020, 0.077000},
	{{0, 0, 2, -8, 3, 0, 0, 0, 0, 0, 0}, 132.702900, 0.000240, 0.500000},
	{{0, 0, 2, -8, 3, 0, 0, 0, 0, 1, 0}, 132.500100, 0.000090, 0.089000},
	{{0, 0, 2, -8, 3, 0, 0, 0, 2, -2, 0}, 132.696290, 0.000020, 0.265000},
	{{0, 0, 2, -8, 3, 0, 0, 0, 2, -1, 0}, 132.853830, 0.000470, 0.105000},
	{{0, 0, 2, -8, 3, 0, 0, 0, 2, 0, 0}, 132.636050, 0.000560, 0.044000},
	{{0, 0, 2, -8, 3, 0, 0, 0, 2, 1, 0}, 132.691050, 0.000060, 0.028000},
	{{0, 0, 2, -8, 3, 0, 0, 0, 4, -1, 0}, 132.688180, 0.000040, 0.029000},
	{{0, 0, 2, -8, 5, 0, 0, 0, -2, 1, 0}, 288.001260, 0.000030, 0.075000},
	{{0, 0, 2, -7, 0, 0, 0, 0, -2, 1, 0}, 107.529210, 0.000010, 0.076000},
	{{0, 0, 2, -7, 0, 0, 0, 0, -2, 2, 0}, 202.419120, 0.000030, 19.147000},
	{{0, 0, 2, -7, 0, 0, 0, 0, -2, 3, 0}, 107.529210, 0.000010, 0.075000},
	{{0, 0, 2, -7, 0, 0, 0, 0, 0, 0, 0}, 222.076170, 0.000020, 0.581000},
	{{0, 0, 2, -7, 0, 0, 0, 0, 1, 1, -1}, 246.780220, 0.000020, 0.095000},
	{{0, 0, 2, -7, 0, 0, 0, 0, 1, 2, -1}, 247.376100, 0.000010, 0.042000},
	{{0, 0, 2, -6, 0, 0, 0, 0, -2, 1, 0}, 55.732760, 0.000030, 0.079000},
	{{0, 0, 2, -6, 0, 0, 0, 0, -2, 3, 0}, 63.481690, 0.000010, 0.072000},
	{{0, 0, 2, -6, 0, 0, 0, 0, -1, 0, 1}, 310.269490, 0.000020, 7.335000},
	{{0, 0, 2, -6, 0, 0, 0, 0, 0, -1, 0}, 334.031540, 0.000010, 0.069000},
	{{0, 0, 2, -6, 0, 0, 0, 0, 0, 0, 0}, 333.332550, 0.000160, 0.840000},
	{{0, 0, 2, -6, 0, 0, 0, 0, 0, 1, 0}, 7.541640, 0.000080, 0.083000},
	{{0, 0, 2, -6, 0, 0, 0, 0, 2, -1, 0}, 16.719170, 0.000150, 0.097000},
	{{0, 0, 2, -6, 0, 0, 0, 0, 2, 0, 0}, 16.211130, 0.000170, 0.042000},
	{{0, 0, 2, -6, 0, 0, 0, 0, 2, 1, 0}, 16.184430, 0.000010, 0.027000},
	{{0, 0, 2, -6, 0, 4, 0, 0, -1, 0, 1}, 310.171190, 0.000040, 1827.649000},
	{{0, 0, 2, -6, 2, 0, 0, 0, 0, -1, 0}, 67.951390, 0.000010, 0.070000},
	{{0, 0, 2, -6, 2, 0, 0, 0, 0, 0, 0}, 73.475960, 0.000070, 0.979000},
	{{0, 0, 2, -6, 2, 0, 0, 0, 0, 1, 0}, 67.951390, 0.000010, 0.082000},
	{{0, 0, 2, -6, 2, 0, 0, 0, 2, -1, 0}, 74.063370, 0.000050, 0.096000},
	{{0, 0, 2, -6, 2, 0, 0, 0, 2, 0, 0}, 73.924570, 0.000050, 0.042000},
	{{0, 0, 2, -6, 3, 0, 0, 0, 0, 0, 0}, 107.448730, 0.000060, 1.067000},
	{{0, 0, 2, -6, 3, 0, 0, 0, 2, 0, 0}, 105.853780, 0.000020, 0.042000},
	{{0, 0, 2, -5, 0, 0, 0, 0, -2, 0, 0}, 245.905240, 0.000030, 0.039000},
	{{0, 0, 2, -5, 0, 0, 0, 0, -2, 1, 0}, 116.312940, 0.000020, 0.082000},
	{{0, 0, 2, -5, 0, 0, 0, 0, -1, 0, 1}, 351.497360, 0.000070, 2.529000},
	{{0, 0, 2, -5, 0, 0, 0, 0, -1, 1, -1}, 171.961090, 0.000080, 0.076000},
	{{0, 0, 2, -5, 0, 0, 0, 0, -1, 2, -1}, 171.768670, 0.000020, 16.174000},
	{{0, 0, 2, -5, 0, 0, 0, 0, 0, -1, 0}, 269.809670, 0.000180, 0.072000},
	{{0, 0, 2, -5, 0, 0, 0, 0, 0, 0, 0}, 279.162980, 0.001090, 1.519000},
	{{0, 0, 2, -5, 0, 0, 0, 0, 0, 1, 0}, 280.485890, 0.000190, 0.079000},
	{{0, 0, 2, -5, 0, 0, 0, 0, 1, 0, -1}, 353.274290, 0.000030, 0.584000},
	{{0, 0, 2, -5, 0, 0, 0, 0, 2, -1, 0}, 8.860170, 0.000170, 0.092000},
	{{0, 0, 2, -5, 0, 0, 0, 0, 2, 0, 0}, 5.159940, 0.000150, 0.042000},
	{{0, 0, 2, -5, 0, 0, 0, 0, 2, 1, 0}, 2.082730, 0.000010, 0.027000},
	{{0, 0, 2, -5, 3, 0, 0, 0, 0, 0, 0}, 137.742160, 0.000020, 2.466000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -4, 0, 0}, 299.594850, 0.000080, 0.020000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -4, 1, 0}, 302.022840, 0.000200, 0.028000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -4, 2, 0}, 302.991300, 0.000120, 0.043000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -3, 0, 0}, 300.699630, 0.000020, 0.027000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, -2, 0}, 297.811880, 0.000090, 0.019000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, -1, 0}, 299.892520, 0.000710, 0.026000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, -1, 2}, 121.217140, 0.000010, 0.089000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, 0, -2}, 122.025160, 0.000010, 0.019000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, 0, 0}, 300.820140, 0.006910, 0.040000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, 0, 2}, 300.952590, 0.000170, 0.505000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, 1, -2}, 121.865090, 0.000020, 0.026000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, 1, 0}, 302.557530, 0.010240, 0.086000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, 2, 0}, 306.850220, 0.000320, 0.607000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -2, 3, 0}, 309.215380, 0.000020, 0.067000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -1, 0, 0}, 120.133500, 0.000160, 0.080000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -1, 0, 1}, 23.368340, 0.000060, 1.079000},
	{{0, 0, 2, -4, 0, 0, 0, 0, -1, 1, 0}, 123.119330, 0.000020, 1.315000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, -3, 0}, 297.641780, 0.000080, 0.025000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, -2, 0}, 297.938590, 0.001160, 0.038000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, -1, -2}, 118.833480, 0.000040, 0.025000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, -1, 0}, 297.825340, 0.017120, 0.075000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, -1, 2}, 297.747340, 0.000030, 0.074000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, 0, -2}, 123.139150, 0.000040, 0.037000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, 0, 0}, 300.139130, 0.091780, 7.890000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, 0, 1}, 23.325940, 0.000020, 0.075000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, 0, 2}, 123.139150, 0.000040, 0.037000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, 1, -2}, 297.747340, 0.000030, 0.073000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, 1, 0}, 298.040830, 0.017700, 0.076000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, 1, 2}, 118.833480, 0.000040, 0.025000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, 2, 0}, 297.878650, 0.001180, 0.038000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 0, 3, 0}, 297.597360, 0.000080, 0.025000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 1, -1, 0}, 123.749700, 0.000030, 0.986000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 1, 0, -1}, 24.126890, 0.000070, 0.847000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 1, 0, 0}, 112.475820, 0.000310, 0.082000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 1, 1, 0}, 121.807110, 0.000020, 0.039000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, -3, 0}, 308.375800, 0.000020, 0.066000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, -2, 0}, 305.388740, 0.000450, 0.526000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, -1, 0}, 303.411170, 0.012840, 0.088000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, -1, 2}, 122.569620, 0.000030, 0.026000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, 0, -2}, 302.512950, 0.000190, 0.448000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, 0, 0}, 302.715440, 0.008740, 0.041000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, 0, 2}, 122.657930, 0.000020, 0.019000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, 1, -2}, 122.771070, 0.000010, 0.091000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, 1, 0}, 301.634270, 0.000860, 0.026000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 2, 2, 0}, 301.404100, 0.000090, 0.020000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 3, -1, 0}, 95.308280, 0.000020, 0.042000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 3, 0, 0}, 303.304410, 0.000020, 0.027000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 4, -2, 0}, 303.453460, 0.000150, 0.044000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 4, -1, 0}, 302.806300, 0.000260, 0.028000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 4, 0, 0}, 302.346440, 0.000110, 0.020000},
	{{0, 0, 2, -4, 0, 0, 0, 0, 4, 1, 0}, 302.685590, 0.000020, 0.016000},
	{{0, 0, 2, -4, 0, 1, 0, 0, 0, 0, 0}, 199.206560, 0.000010, 10.777000},
	{{0, 0, 2, -4, 0, 2, 0, 0, 0, 0, 0}, 0.628210, 0.000060, 16.995000},
	{{0, 0, 2, -4, 1, 0, 0, 0, -2, 0, 0}, 155.281160, 0.000070, 0.040000},
	{{0, 0, 2, -4, 1, 0, 0, 0, -2, 1, 0}, 155.690210, 0.000110, 0.087000},
	{{0, 0, 2, -4, 1, 0, 0, 0, 0, -1, 0}, 153.341010, 0.000120, 0.075000},
	{{0, 0, 2, -4, 1, 0, 0, 0, 0, 0, 0}, 154.612920, 0.000770, 23.567000},
	{{0, 0, 2, -4, 1, 0, 0, 0, 0, 1, 0}, 153.386900, 0.000120, 0.076000},
	{{0, 0, 2, -4, 1, 0, 0, 0, 2, -1, 0}, 155.998540, 0.000120, 0.087000},
	{{0, 0, 2, -4, 1, 0, 0, 0, 2, 0, 0}, 155.470360, 0.000070, 0.040000},
	{{0, 0, 2, -4, 2, 0, 0, 0, -2, -1, 0}, 0.614610, 0.000030, 0.026000},
	{{0, 0, 2, -4, 2, 0, 0, 0, -2, 0, 0}, 0.552560, 0.000190, 0.040000},
	{{0, 0, 2, -4, 2, 0, 0, 0, -2, 1, 0}, 0.851980, 0.000310, 0.087000},
	{{0, 0, 2, -4, 2, 0, 0, 0, -2, 2, 0}, 0.614620, 0.000010, 0.551000},
	{{0, 0, 2, -4, 2, 0, 0, 0, 0, -2, 0}, 359.012360, 0.000020, 0.038000},
	{{0, 0, 2, -4, 2, 0, 0, 0, 0, -1, 0}, 359.242570, 0.000340, 0.076000},
	{{0, 0, 2, -4, 2, 0, 0, 0, 0, 0, 0}, 0.035160, 0.001760, 23.883000},
	{{0, 0, 2, -4, 2, 0, 0, 0, 0, 1, 0}, 359.180710, 0.000330, 0.075000},
	{{0, 0, 2, -4, 2, 0, 0, 0, 0, 2, 0}, 359.019370, 0.000020, 0.038000},
	{{0, 0, 2, -4, 2, 0, 0, 0, 2, -2, 0}, 0.508930, 0.000010, 0.577000},
	{{0, 0, 2, -4, 2, 0, 0, 0, 2, -1, 0}, 0.725620, 0.000270, 0.087000},
	{{0, 0, 2, -4, 2, 0, 0, 0, 2, 0, 0}, 0.508300, 0.000190, 0.040000},
	{{0, 0, 2, -4, 2, 0, 0, 0, 2, 1, 0}, 0.538830, 0.000020, 0.026000},
	{{0, 0, 2, -4, 3, 0, 0, 0, 0, 0, 0}, 176.234170, 0.000050, 7.926000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -4, -1, 0}, 330.911740, 0.000010, 0.016000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -4, 0, 0}, 330.898790, 0.000080, 0.020000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -4, 1, 0}, 330.819220, 0.000220, 0.028000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -4, 2, 0}, 330.923240, 0.000130, 0.044000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -3, 0, 0}, 331.118630, 0.000020, 0.027000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -2, -2, 0}, 331.702560, 0.000060, 0.020000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -2, -1, 0}, 330.681150, 0.000680, 0.027000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -2, 0, -2}, 150.766390, 0.000020, 0.020000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -2, 0, 0}, 330.977510, 0.007390, 0.041000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -2, 0, 2}, 331.001420, 0.000140, 0.398000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -2, 1, -2}, 150.799280, 0.000020, 0.026000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -2, 1, 0}, 330.923950, 0.010850, 0.090000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -2, 2, 0}, 331.076670, 0.000470, 0.459000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -2, 3, 0}, 331.373410, 0.000030, 0.065000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -1, 0, 0}, 150.675360, 0.000250, 0.084000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -1, 0, 1}, 52.850660, 0.000060, 0.686000},
	{{0, 0, 2, -3, 0, 0, 0, 0, -1, 1, 0}, 150.762500, 0.000040, 0.774000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, -3, 0}, 329.735440, 0.000040, 0.025000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, -2, 0}, 329.811470, 0.000570, 0.038000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, -1, -2}, 150.043320, 0.000030, 0.025000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, -1, 0}, 329.851530, 0.008490, 0.078000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, -1, 2}, 329.799710, 0.000010, 0.071000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, 0, -2}, 150.746410, 0.000020, 0.038000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, 0, 0}, 330.286950, 0.040140, 2.470000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, 0, 2}, 150.746410, 0.000020, 0.037000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, 1, -2}, 154.798340, 0.000030, 0.076000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, 1, 0}, 329.781440, 0.007530, 0.073000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, 1, 2}, 150.043320, 0.000030, 0.025000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, 2, -2}, 153.556050, 0.000010, 13.999000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, 2, 0}, 329.742120, 0.000520, 0.037000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 0, 3, 0}, 329.656050, 0.000030, 0.025000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 1, -1, -1}, 53.532360, 0.000020, 0.072000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 1, -1, 1}, 233.385220, 0.000070, 0.077000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 1, 0, -1}, 53.416270, 0.000220, 1.541000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 1, 0, 0}, 331.109280, 0.000060, 0.078000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 1, 0, 1}, 233.395200, 0.000030, 0.038000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 1, 1, -1}, 53.532360, 0.000020, 0.079000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 2, -2, 0}, 331.043940, 0.000220, 0.730000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 2, -1, -2}, 150.522470, 0.000010, 0.067000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 2, -1, 0}, 331.231760, 0.006530, 0.084000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 2, 0, -2}, 330.500980, 0.000100, 0.587000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 2, 0, 0}, 330.416420, 0.003680, 0.040000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 2, 1, -2}, 150.522470, 0.000010, 0.087000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 2, 1, 0}, 330.458660, 0.000380, 0.026000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 2, 2, 0}, 330.438970, 0.000040, 0.019000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 4, -2, 0}, 331.083230, 0.000070, 0.043000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 4, -1, 0}, 330.732170, 0.000120, 0.027000},
	{{0, 0, 2, -3, 0, 0, 0, 0, 4, 0, 0}, 330.838010, 0.000050, 0.020000},
	{{0, 0, 2, -3, 1, 0, 0, 0, 0, 0, 0}, 353.366560, 0.000020, 2.044000},
	{{0, 0, 2, -3, 1, 0, 0, 0, 2, 0, 0}, 183.009560, 0.000010, 0.040000},
	{{0, 0, 2, -3, 2, 0, 0, 0, 2, -3, 0}, 209.840560, 0.000010, 0.069000},
	{{0, 0, 2, -3, 2, 0, 0, 0, 2, -1, 0}, 209.840560, 0.000010, 0.083000},
	{{0, 0, 2, -3, 2, 0, 0, 0, 2, 0, 0}, 29.777520, 0.000010, 0.040000},
	{{0, 0, 2, -3, 3, 0, 0, 0, 0, 0, 0}, 351.969470, 0.000010, 1.520000},
	{{0, 0, 2, -2, -3, 0, 0, 0, 0, 0, 0}, 320.724180, 0.000010, 1.463000},
	{{0, 0, 2, -2, -2, 0, 0, 0, -2, 0, 0}, 120.937600, 0.000010, 0.042000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -6, 0, 0}, 177.244200, 0.000010, 0.014000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -6, 1, 0}, 359.041980, 0.000020, 0.017000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -6, 2, 0}, 358.931110, 0.000040, 0.021000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -4, -1, 0}, 0.975460, 0.000090, 0.016000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -4, 0, 0}, 357.170700, 0.000510, 0.021000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -4, 0, 2}, 358.671430, 0.000010, 0.046000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -4, 1, 0}, 357.618980, 0.001190, 0.028000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -4, 1, 2}, 358.694520, 0.000020, 0.119000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -4, 2, 0}, 358.451380, 0.000770, 0.045000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -4, 3, 0}, 359.180010, 0.000030, 0.114000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -3, -1, 0}, 358.261920, 0.000010, 0.020000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -3, 0, 0}, 358.929790, 0.000040, 0.028000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -3, 1, 0}, 176.346550, 0.000020, 0.044000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, -3, 0}, 358.812150, 0.000040, 0.016000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, -2, 0}, 359.089350, 0.000330, 0.020000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, -1, -2}, 178.913500, 0.000020, 0.016000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, -1, 0}, 358.131180, 0.004040, 0.027000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, -1, 2}, 178.729980, 0.000070, 0.098000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, 0, -2}, 179.099590, 0.000090, 0.020000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, 0, 0}, 358.579750, 0.046120, 0.042000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, 0, 2}, 358.767100, 0.000640, 0.329000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, 1, -2}, 178.899310, 0.000170, 0.027000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, 1, 0}, 358.868730, 0.059040, 0.095000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, 1, 2}, 359.286380, 0.000020, 0.061000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, 2, 0}, 359.116210, 0.002540, 0.369000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -2, 3, 0}, 358.226120, 0.000130, 0.063000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -1, -1, 0}, 2.195200, 0.000040, 0.041000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -1, -1, 1}, 279.512160, 0.000030, 0.089000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -1, 0, -1}, 100.087150, 0.000010, 0.040000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -1, 0, 0}, 0.469970, 0.001190, 0.087000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -1, 0, 1}, 98.757840, 0.000060, 0.502000},
	{{0, 0, 2, -2, 0, 0, 0, 0, -1, 1, 0}, 1.623330, 0.000070, 0.548000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, -3, 0}, 359.848720, 0.000210, 0.026000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, -2, 0}, 359.284740, 0.002740, 0.039000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, -1, -2}, 179.556950, 0.000110, 0.026000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, -1, 0}, 359.516370, 0.042130, 0.081000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, 0, -2}, 178.520600, 0.000020, 0.039000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, 0, 0}, 359.764870, 0.194430, 1.068000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, 0, 2}, 179.692110, 0.000140, 0.036000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, 1, 0}, 0.071170, 0.032590, 0.070000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, 1, 2}, 179.664570, 0.000090, 0.024000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, 2, 0}, 0.402260, 0.002290, 0.036000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, 2, 2}, 179.667600, 0.000010, 0.018000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 0, 3, 0}, 359.974110, 0.000150, 0.025000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, -3, 0}, 180.436210, 0.000020, 0.038000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, -2, 0}, 180.502460, 0.000170, 0.076000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, -1, -1}, 99.340310, 0.000140, 0.075000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, -1, 0}, 0.275480, 0.001390, 20.152000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, -1, 1}, 99.532510, 0.000130, 0.074000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, 0, -2}, 180.234880, 0.000020, 0.074000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, 0, -1}, 99.526230, 0.001130, 8.537000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, 0, 0}, 180.276030, 0.012010, 0.075000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, 0, 1}, 279.444070, 0.000280, 0.037000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, 0, 2}, 0.234860, 0.000020, 0.025000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, 1, -1}, 99.340310, 0.000140, 0.076000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, 1, 0}, 180.230890, 0.000830, 0.038000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, 1, 1}, 279.409210, 0.000030, 0.025000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 1, 2, 0}, 180.234890, 0.000060, 0.025000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, -3, 0}, 183.613550, 0.000010, 0.071000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, -2, 0}, 180.687750, 0.000350, 1.194000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, -1, 0}, 0.451560, 0.013150, 0.081000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, -1, 2}, 179.963110, 0.000030, 0.025000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, 0, -2}, 0.844250, 0.000490, 0.854000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, 0, 0}, 0.890770, 0.008750, 0.039000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, 0, 2}, 179.528100, 0.000020, 0.019000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, 1, -2}, 180.793720, 0.000020, 0.083000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, 1, 0}, 1.041190, 0.001010, 0.026000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 2, 2, 0}, 359.617220, 0.000120, 0.019000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 3, -2, 0}, 0.400360, 0.000060, 0.087000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 3, -1, 0}, 180.224540, 0.000140, 0.040000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 3, 0, 0}, 180.018150, 0.000090, 0.026000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 4, -2, 0}, 359.988570, 0.000160, 0.042000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 4, -1, 0}, 1.391660, 0.000270, 0.027000},
	{{0, 0, 2, -2, 0, 0, 0, 0, 4, 0, 0}, 0.786000, 0.000090, 0.020000},
	{{0, 0, 2, -2, 0, 2, 0, 0, -2, 0, 0}, 290.922360, 0.000020, 0.042000},
	{{0, 0, 2, -2, 0, 2, 0, 0, -2, 1, 0}, 290.922360, 0.000020, 0.095000},
	{{0, 0, 2, -2, 0, 2, 0, 0, 0, 0, 0}, 290.945540, 0.000030, 0.996000},
	{{0, 0, 2, -2, 1, 0, 0, 0, 0, 0, 0}, 97.772950, 0.000010, 0.980000},
	{{0, 0, 2, -2, 1, 0, 0, 0, 1, 0, -1}, 120.933360, 0.000010, 30.461000},
	{{0, 0, 2, -2, 1, 0, 0, 0, 2, -3, 0}, 204.719580, 0.000010, 0.071000},
	{{0, 0, 2, -2, 1, 0, 0, 0, 2, -1, 0}, 204.719580, 0.000010, 0.080000},
	{{0, 0, 2, -2, 2, 0, 0, 0, 1, 0, -1}, 165.354860, 0.000010, 19.427000},
	{{0, 0, 2, -1, 0, 0, 0, 0, -4, 1, 0}, 255.724630, 0.000020, 0.029000},
	{{0, 0, 2, -1, 0, 0, 0, 0, -2, -1, 0}, 255.535130, 0.000050, 0.027000},
	{{0, 0, 2, -1, 0, 0, 0, 0, -2, 0, 0}, 255.006210, 0.000500, 0.043000},
	{{0, 0, 2, -1, 0, 0, 0, 0, -2, 1, 0}, 257.411310, 0.000530, 0.100000},
	{{0, 0, 2, -1, 0, 0, 0, 0, -2, 2, 0}, 260.262740, 0.000030, 0.308000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 0, -1, 0}, 234.623980, 0.000390, 0.085000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 0, 0, 0}, 228.749620, 0.001400, 0.681000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 0, 1, 0}, 232.839710, 0.000170, 0.068000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 1, -1, -1}, 308.942390, 0.000020, 0.078000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 1, -1, 1}, 129.692010, 0.000020, 0.071000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 1, 0, -1}, 309.056540, 0.000170, 2.412000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 1, 0, 0}, 210.131220, 0.000020, 0.072000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 1, 0, 1}, 128.959290, 0.000020, 0.037000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 1, 1, -1}, 308.942390, 0.000020, 0.073000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 2, -2, 0}, 44.315940, 0.000070, 3.271000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 2, -1, 0}, 39.715290, 0.000550, 0.077000},
	{{0, 0, 2, -1, 0, 0, 0, 0, 2, 0, 0}, 43.776410, 0.000090, 0.038000},
	{{0, 0, 2, -1, 2, 0, 0, 0, 2, -1, 0}, 281.918980, 0.000020, 0.076000},
	{{0, 0, 2, 0, -7, 0, 0, 0, 0, 0, 0}, 233.411290, 0.000010, 0.709000},
	{{0, 0, 2, 0, -7, 3, 0, 0, 0, 0, 0}, 197.993170, 0.000010, 0.662000},
	{{0, 0, 2, 0, -7, 5, 0, 0, -2, 0, 0}, 102.115250, 0.000020, 0.043000},
	{{0, 0, 2, 0, -7, 5, 0, 0, 0, 0, 0}, 102.483710, 0.000030, 0.633000},
	{{0, 0, 2, 0, -7, 10, 0, 0, 2, -3, 0}, 247.917360, 0.000020, 0.075000},
	{{0, 0, 2, 0, -7, 10, 0, 0, 2, -2, 0}, 77.168680, 0.000030, 40.542000},
	{{0, 0, 2, 0, -7, 10, 0, 0, 2, -1, 0}, 67.917360, 0.000020, 0.076000},
	{{0, 0, 2, 0, -6, 0, 0, 0, -2, 0, 0}, 218.520110, 0.000040, 0.043000},
	{{0, 0, 2, 0, -6, 0, 0, 0, -2, 1, 0}, 218.755200, 0.000040, 0.100000},
	{{0, 0, 2, 0, -6, 0, 0, 0, 0, -1, 0}, 219.604120, 0.000010, 0.085000},
	{{0, 0, 2, 0, -6, 0, 0, 0, 0, 0, 0}, 220.527070, 0.000100, 0.669000},
	{{0, 0, 2, 0, -6, 0, 0, 0, 0, 1, 0}, 219.604120, 0.000010, 0.068000},
	{{0, 0, 2, 0, -6, 0, 0, 0, 2, -1, 0}, 41.063970, 0.000050, 0.077000},
	{{0, 0, 2, 0, -6, 2, 0, 0, -2, 0, 0}, 199.259220, 0.000020, 0.043000},
	{{0, 0, 2, 0, -6, 2, 0, 0, 0, 0, 0}, 200.882520, 0.000030, 0.640000},
	{{0, 0, 2, 0, -6, 3, 0, 0, -2, 0, 0}, 217.473300, 0.000010, 0.043000},
	{{0, 0, 2, 0, -6, 3, 0, 0, 0, 0, 0}, 219.218270, 0.000030, 0.627000},
	{{0, 0, 2, 0, -6, 3, 0, 0, 2, -1, 0}, 210.600760, 0.000020, 0.076000},
	{{0, 0, 2, 0, -6, 4, 0, 0, -2, 0, 0}, 129.709010, 0.000020, 0.043000},
	{{0, 0, 2, 0, -6, 4, 0, 0, 0, 0, 0}, 129.858820, 0.000020, 0.614000},
	{{0, 0, 2, 0, -6, 4, 0, 0, 2, -1, 0}, 131.000690, 0.000020, 0.076000},
	{{0, 0, 2, 0, -6, 5, 0, 0, -2, 0, 0}, 80.019380, 0.000050, 0.043000},
	{{0, 0, 2, 0, -6, 5, 0, 0, -2, 1, 0}, 80.019420, 0.000040, 0.102000},
	{{0, 0, 2, 0, -6, 5, 0, 0, 0, -1, 0}, 79.275180, 0.000050, 0.086000},
	{{0, 0, 2, 0, -6, 5, 0, 0, 0, 0, 0}, 80.882660, 0.000100, 0.601000},
	{{0, 0, 2, 0, -6, 5, 0, 0, 0, 1, 0}, 80.019980, 0.000010, 0.067000},
	{{0, 0, 2, 0, -6, 5, 0, 0, 2, -3, 0}, 139.151690, 0.000020, 0.075000},
	{{0, 0, 2, 0, -6, 5, 0, 0, 2, -2, 0}, 255.401860, 0.000030, 9.083000},
	{{0, 0, 2, 0, -6, 5, 0, 0, 2, -1, 0}, 259.075760, 0.000190, 0.076000},
	{{0, 0, 2, 0, -6, 7, 0, 0, 0, 0, 0}, 353.833780, 0.000010, 0.577000},
	{{0, 0, 2, 0, -6, 8, 0, 0, 2, -2, 0}, 231.833930, 0.000030, 121.093000},
	{{0, 0, 2, 0, -6, 10, 0, 0, 0, 0, 0}, 161.671570, 0.000010, 0.545000},
	{{0, 0, 2, 0, -6, 10, 0, 0, 2, -3, 0}, 335.233450, 0.000020, 0.076000},
	{{0, 0, 2, 0, -6, 10, 0, 0, 2, -1, 0}, 155.233450, 0.000020, 0.075000},
	{{0, 0, 2, 0, -5, 0, 0, 0, -2, -1, 0}, 209.097530, 0.000030, 0.027000},
	{{0, 0, 2, 0, -5, 0, 0, 0, -2, 0, 0}, 209.172030, 0.000280, 0.043000},
	{{0, 0, 2, 0, -5, 0, 0, 0, -2, 1, 0}, 209.104700, 0.000320, 0.101000},
	{{0, 0, 2, 0, -5, 0, 0, 0, 0, -2, 0}, 209.968750, 0.000020, 0.040000},
	{{0, 0, 2, 0, -5, 0, 0, 0, 0, -1, 0}, 209.892660, 0.000390, 0.086000},
	{{0, 0, 2, 0, -5, 0, 0, 0, 0, 0, 0}, 210.153740, 0.000840, 0.634000},
	{{0, 0, 2, 0, -5, 0, 0, 0, 0, 1, 0}, 210.754930, 0.000140, 0.067000},
	{{0, 0, 2, 0, -5, 0, 0, 0, 1, 0, -1}, 298.551090, 0.000010, 1.906000},
	{{0, 0, 2, 0, -5, 0, 0, 0, 2, -2, 0}, 29.341020, 0.000120, 5.114000},
	{{0, 0, 2, 0, -5, 0, 0, 0, 2, -1, 0}, 29.508030, 0.000780, 0.077000},
	{{0, 0, 2, 0, -5, 0, 0, 0, 2, 0, 0}, 29.835930, 0.000080, 0.038000},
	{{0, 0, 2, 0, -5, 2, 0, 0, -2, 0, 0}, 190.259180, 0.000070, 0.043000},
	{{0, 0, 2, 0, -5, 2, 0, 0, -2, 1, 0}, 190.259180, 0.000060, 0.102000},
	{{0, 0, 2, 0, -5, 2, 0, 0, 0, -1, 0}, 191.181470, 0.000080, 0.086000},
	{{0, 0, 2, 0, -5, 2, 0, 0, 0, 0, 0}, 190.847890, 0.000160, 0.607000},
	{{0, 0, 2, 0, -5, 2, 0, 0, 0, 1, 0}, 190.659570, 0.000030, 0.067000},
	{{0, 0, 2, 0, -5, 2, 0, 0, 2, -3, 0}, 201.884550, 0.000010, 0.075000},
	{{0, 0, 2, 0, -5, 2, 0, 0, 2, -2, 0}, 10.340540, 0.000030, 7.834000},
	{{0, 0, 2, 0, -5, 2, 0, 0, 2, -1, 0}, 11.485890, 0.000240, 0.076000},
	{{0, 0, 2, 0, -5, 3, 0, 0, -2, 0, 0}, 214.006980, 0.000040, 0.043000},
	{{0, 0, 2, 0, -5, 3, 0, 0, -2, 1, 0}, 214.006980, 0.000040, 0.102000},
	{{0, 0, 2, 0, -5, 3, 0, 0, 0, -2, 0}, 33.612960, 0.000010, 0.040000},
	{{0, 0, 2, 0, -5, 3, 0, 0, 0, -1, 0}, 212.487170, 0.000040, 0.086000},
	{{0, 0, 2, 0, -5, 3, 0, 0, 0, 0, 0}, 212.697080, 0.000080, 0.595000},
	{{0, 0, 2, 0, -5, 3, 0, 0, 2, -3, 0}, 102.513000, 0.000020, 0.075000},
	{{0, 0, 2, 0, -5, 3, 0, 0, 2, -2, 0}, 34.694050, 0.000030, 10.673000},
	{{0, 0, 2, 0, -5, 3, 0, 0, 2, -1, 0}, 32.102570, 0.000160, 0.076000},
	{{0, 0, 2, 0, -5, 4, 0, 0, -2, 0, 0}, 64.014510, 0.000020, 0.043000},
	{{0, 0, 2, 0, -5, 4, 0, 0, 0, -1, 0}, 58.287730, 0.000030, 0.087000},
	{{0, 0, 2, 0, -5, 4, 0, 0, 0, 0, 0}, 62.337300, 0.000050, 0.583000},
	{{0, 0, 2, 0, -5, 4, 0, 0, 2, -3, 0}, 73.737070, 0.000030, 0.075000},
	{{0, 0, 2, 0, -5, 4, 0, 0, 2, -2, 0}, 239.284700, 0.000050, 16.738000},
	{{0, 0, 2, 0, -5, 4, 0, 0, 2, -1, 0}, 238.287470, 0.000160, 0.076000},
	{{0, 0, 2, 0, -5, 5, 0, 0, -2, -1, 0}, 56.732360, 0.000020, 0.028000},
	{{0, 0, 2, 0, -5, 5, 0, 0, -2, 0, 0}, 56.739990, 0.000140, 0.043000},
	{{0, 0, 2, 0, -5, 5, 0, 0, -2, 1, 0}, 56.746900, 0.000150, 0.103000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 0, -2, 0}, 53.800320, 0.000020, 0.040000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 0, -1, 0}, 54.492990, 0.000870, 0.087000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 0, 0, 0}, 55.929180, 0.000480, 0.572000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 0, 1, 0}, 56.482390, 0.000060, 0.067000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 2, -3, 0}, 247.873350, 0.000040, 0.075000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 2, -2, 0}, 233.933040, 0.002710, 38.763000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 2, -1, 0}, 234.224550, 0.004100, 0.076000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 2, 0, 0}, 231.368390, 0.000280, 0.038000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 2, 1, 0}, 234.217210, 0.000020, 0.025000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 4, -3, 0}, 232.030120, 0.000050, 0.087000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 4, -2, 0}, 234.133890, 0.000080, 0.040000},
	{{0, 0, 2, 0, -5, 5, 0, 0, 4, -1, 0}, 234.203160, 0.000040, 0.026000},
	{{0, 0, 2, 0, -5, 6, 0, 0, 2, -3, 0}, 168.606190, 0.000020, 0.075000},
	{{0, 0, 2, 0, -5, 6, 0, 0, 2, -2, 0}, 178.180810, 0.000070, 122.705000},
	{{0, 0, 2, 0, -5, 6, 0, 0, 2, -1, 0}, 348.606190, 0.000020, 0.075000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -4, 0, 0}, 199.892150, 0.000030, 0.021000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -4, 1, 0}, 199.900650, 0.000060, 0.029000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -4, 2, 0}, 199.902460, 0.000040, 0.047000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -2, -2, 0}, 199.122130, 0.000010, 0.020000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -2, -1, 0}, 199.859740, 0.000230, 0.028000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -2, 0, 0}, 199.377040, 0.002480, 0.043000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -2, 0, 2}, 199.648550, 0.000030, 0.265000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -2, 1, 0}, 199.812120, 0.002580, 0.102000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -2, 2, 0}, 199.551810, 0.000110, 0.291000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -1, 0, 0}, 20.205450, 0.000030, 0.093000},
	{{0, 0, 2, 0, -4, 0, 0, 0, -1, 0, 1}, 295.247830, 0.000010, 0.368000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 0, -3, 0}, 199.869280, 0.000010, 0.026000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 0, -2, 0}, 200.261180, 0.000250, 0.040000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 0, -1, 0}, 200.250560, 0.004250, 0.086000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 0, 0, 0}, 200.474140, 0.006720, 0.601000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 0, 1, 0}, 201.013440, 0.001120, 0.067000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 0, 2, 0}, 201.043920, 0.000070, 0.035000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 1, 0, -1}, 289.340980, 0.000080, 1.642000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 1, 0, 0}, 21.432240, 0.000040, 0.071000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 2, -3, 0}, 19.923610, 0.000100, 0.075000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 2, -2, 0}, 19.834950, 0.002730, 8.990000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 2, -1, -2}, 19.940000, 0.000020, 0.073000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 2, -1, 0}, 19.760440, 0.013120, 0.076000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 2, -1, 2}, 199.940000, 0.000030, 0.025000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 2, 0, -2}, 20.013700, 0.000010, 2.248000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 2, 0, 0}, 21.579310, 0.001050, 0.038000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 2, 1, 0}, 20.049400, 0.000080, 0.025000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 4, -3, 0}, 19.639190, 0.000040, 0.088000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 4, -2, 0}, 19.847860, 0.000200, 0.041000},
	{{0, 0, 2, 0, -4, 0, 0, 0, 4, -1, 0}, 20.010360, 0.000130, 0.026000},
	{{0, 0, 2, 0, -4, 1, 0, 0, -2, 0, 0}, 15.304100, 0.000020, 0.043000},
	{{0, 0, 2, 0, -4, 1, 0, 0, 0, 0, 0}, 15.655540, 0.000040, 0.589000},
	{{0, 0, 2, 0, -4, 1, 0, 0, 2, -3, 0}, 27.326750, 0.000010, 0.075000},
	{{0, 0, 2, 0, -4, 1, 0, 0, 2, -2, 0}, 192.354670, 0.000020, 12.939000},
	{{0, 0, 2, 0, -4, 1, 0, 0, 2, -1, 0}, 189.723030, 0.000060, 0.076000},
	{{0, 0, 2, 0, -4, 2, 0, 0, -2, -1, 0}, 182.704110, 0.000020, 0.028000},
	{{0, 0, 2, 0, -4, 2, 0, 0, -2, 0, 0}, 182.713540, 0.000180, 0.043000},
	{{0, 0, 2, 0, -4, 2, 0, 0, -2, 1, 0}, 182.722160, 0.000190, 0.103000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 0, -2, 0}, 182.193000, 0.000020, 0.040000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 0, -1, 0}, 182.416050, 0.000720, 0.087000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 0, 0, 0}, 182.286980, 0.000570, 0.578000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 0, 1, 0}, 182.461770, 0.000080, 0.067000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 2, -2, 0}, 2.277870, 0.001290, 23.075000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 2, -1, 0}, 2.382720, 0.003040, 0.076000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 2, 0, 0}, 2.257180, 0.000220, 0.038000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 4, -3, 0}, 359.107630, 0.000030, 0.087000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 4, -2, 0}, 2.244290, 0.000050, 0.040000},
	{{0, 0, 2, 0, -4, 2, 0, 0, 4, -1, 0}, 2.368930, 0.000030, 0.026000},
	{{0, 0, 2, 0, -4, 3, 0, 0, -2, 0, 0}, 239.162340, 0.000070, 0.044000},
	{{0, 0, 2, 0, -4, 3, 0, 0, -2, 1, 0}, 239.162340, 0.000070, 0.103000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 0, -2, 0}, 57.234150, 0.000030, 0.040000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 0, -1, 0}, 237.189250, 0.000850, 0.087000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 0, 0, 0}, 237.953050, 0.000260, 0.567000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 0, 1, 0}, 238.751550, 0.000030, 0.067000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 2, -4, 0}, 52.640180, 0.000020, 0.038000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 2, -3, 0}, 52.610780, 0.000230, 0.075000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 2, -2, 0}, 57.323170, 0.008190, 106.494000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 2, -1, 0}, 57.163510, 0.004950, 0.075000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 2, 0, 0}, 57.218680, 0.000340, 0.038000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 2, 1, 0}, 57.146920, 0.000020, 0.025000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 4, -3, 0}, 58.914990, 0.000140, 0.087000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 4, -2, 0}, 57.349520, 0.000160, 0.040000},
	{{0, 0, 2, 0, -4, 3, 0, 0, 4, -1, 0}, 57.204290, 0.000050, 0.026000},
	{{0, 0, 2, 0, -4, 4, -2, 0, 2, -2, 0}, 174.859960, 0.000400, 1326.783000},
	{{0, 0, 2, 0, -4, 4, 0, 0, -2, 0, 0}, 117.582380, 0.000020, 0.044000},
	{{0, 0, 2, 0, -4, 4, 0, 0, 0, -1, 0}, 296.938260, 0.000040, 0.087000},
	{{0, 0, 2, 0, -4, 4, 0, 0, 0, 0, 0}, 117.288630, 0.000030, 0.556000},
	{{0, 0, 2, 0, -4, 4, 0, 0, 2, -2, 0}, 295.363600, 0.000120, 40.721000},
	{{0, 0, 2, 0, -4, 4, 0, 0, 2, -1, 0}, 116.938350, 0.000200, 0.075000},
	{{0, 0, 2, 0, -4, 5, 0, 0, -4, 1, 0}, 351.228380, 0.000020, 0.029000},
	{{0, 0, 2, 0, -4, 5, 0, 0, -2, -1, 0}, 351.227390, 0.000050, 0.028000},
	{{0, 0, 2, 0, -4, 5, 0, 0, -2, 0, 0}, 351.236540, 0.000470, 0.044000},
	{{0, 0, 2, 0, -4, 5, 0, 0, -2, 1, 0}, 351.235500, 0.000500, 0.104000},
	{{0, 0, 2, 0, -4, 5, 0, 0, -2, 2, 0}, 351.230180, 0.000010, 0.277000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 0, -2, 0}, 170.984220, 0.000090, 0.041000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 0, -1, 0}, 171.197810, 0.001030, 0.088000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 0, 0, 0}, 351.467130, 0.001050, 0.546000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 0, 1, 0}, 351.292840, 0.000170, 0.066000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 0, 2, 0}, 349.149850, 0.000030, 0.035000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 2, -3, 0}, 148.633060, 0.000050, 0.076000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 2, -2, 0}, 170.863240, 0.001330, 17.093000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 2, -1, 0}, 351.238590, 0.006320, 0.075000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 2, 0, -2}, 167.596220, 0.000010, 3.636000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 2, 0, 0}, 351.490830, 0.000350, 0.038000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 2, 1, 0}, 351.214740, 0.000030, 0.025000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 2, 2, 0}, 168.243170, 0.000010, 0.019000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 4, -3, 0}, 183.235860, 0.000020, 0.087000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 4, -2, 0}, 350.488140, 0.000070, 0.040000},
	{{0, 0, 2, 0, -4, 5, 0, 0, 4, -1, 0}, 351.196890, 0.000060, 0.026000},
	{{0, 0, 2, 0, -4, 6, -3, 0, 0, 0, 0}, 168.762550, 0.000010, 0.546000},
	{{0, 0, 2, 0, -4, 6, -3, 0, 2, -1, 0}, 173.239910, 0.000030, 0.075000},
	{{0, 0, 2, 0, -3, -5, 0, 0, 0, 0, 0}, 219.397820, 0.000010, 0.634000},
	{{0, 0, 2, 0, -3, -1, 0, 0, 0, 0, 0}, 154.758350, 0.000010, 0.584000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -6, 1, 0}, 190.066980, 0.000010, 0.017000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -4, -1, 0}, 189.998460, 0.000040, 0.016000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -4, 0, 0}, 189.125780, 0.000190, 0.021000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -4, 0, 2}, 190.235470, 0.000020, 0.048000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -4, 1, 0}, 190.626470, 0.000430, 0.029000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -4, 2, 0}, 190.340990, 0.000230, 0.047000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -3, 0, 0}, 190.192860, 0.000020, 0.028000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -3, 1, 0}, 10.209720, 0.000020, 0.045000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, -3, 0}, 190.116820, 0.000020, 0.016000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, -2, 0}, 190.279430, 0.000200, 0.020000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, -1, 0}, 190.292270, 0.002200, 0.028000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, -1, 2}, 10.250220, 0.000020, 0.106000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, 0, -2}, 10.405970, 0.000040, 0.020000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, 0, 0}, 190.207510, 0.018410, 0.043000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, 0, 2}, 190.220670, 0.000160, 0.259000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, 1, -2}, 10.380900, 0.000040, 0.027000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, 1, 0}, 190.297860, 0.018140, 0.103000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, 2, 0}, 190.405480, 0.000750, 0.284000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -2, 3, 0}, 190.329090, 0.000050, 0.060000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -1, -1, 0}, 10.085770, 0.000060, 0.042000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -1, -1, 1}, 187.732800, 0.000020, 0.096000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -1, 0, 0}, 11.038220, 0.000320, 0.094000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -1, 0, 1}, 292.549510, 0.000040, 0.357000},
	{{0, 0, 2, 0, -3, 0, 0, 0, -1, 1, 0}, 10.428940, 0.000040, 0.380000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, -3, 0}, 190.433920, 0.000110, 0.026000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, -2, -2}, 10.113120, 0.000020, 0.019000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, -2, 0}, 190.268690, 0.003200, 0.040000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, -2, 2}, 10.053970, 0.000040, 0.481000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, -1, -2}, 10.173320, 0.000190, 0.026000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, -1, 0}, 190.179700, 0.095180, 0.087000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, -1, 2}, 190.694190, 0.000010, 0.065000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, 0, -2}, 8.630240, 0.000020, 0.040000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, 0, 0}, 190.436000, 0.051670, 0.572000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, 0, 2}, 10.393900, 0.000060, 0.035000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, 1, 0}, 190.515720, 0.008130, 0.067000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, 1, 2}, 10.615300, 0.000030, 0.024000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, 2, 0}, 190.677580, 0.000580, 0.035000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 0, 3, 0}, 190.765760, 0.000040, 0.024000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 1, -2, 0}, 190.648340, 0.000040, 0.081000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 1, -1, -1}, 276.013230, 0.000030, 0.080000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 1, -1, 0}, 10.207560, 0.000440, 1.163000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 1, -1, 1}, 99.378190, 0.000030, 0.070000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 1, 0, -1}, 278.286090, 0.000310, 1.442000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 1, 0, 0}, 8.011310, 0.000390, 0.071000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 1, 0, 1}, 100.504000, 0.000020, 0.036000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 1, 1, -1}, 276.013230, 0.000030, 0.072000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 1, 1, 0}, 8.097040, 0.000030, 0.037000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -5, 0}, 10.153140, 0.000050, 0.025000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -4, 0}, 10.153900, 0.000640, 0.038000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -3, -2}, 190.077770, 0.000040, 0.025000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -3, 0}, 10.151700, 0.009360, 0.075000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -2, -2}, 190.010790, 0.000240, 0.037000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -2, 0}, 10.038350, 0.282810, 37.133000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -2, 2}, 190.044350, 0.000290, 0.037000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -1, -2}, 10.077010, 0.000700, 0.073000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -1, 0}, 10.070010, 0.445320, 0.076000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, -1, 2}, 190.082550, 0.000920, 0.025000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, 0, -2}, 11.696790, 0.000140, 2.774000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, 0, 0}, 10.085800, 0.031740, 0.038000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, 0, 2}, 190.082860, 0.000160, 0.019000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, 1, -2}, 190.206190, 0.000050, 0.078000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, 1, 0}, 9.993650, 0.002230, 0.025000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, 1, 2}, 190.085870, 0.000020, 0.015000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 2, 2, 0}, 10.101320, 0.000170, 0.019000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 3, -2, 0}, 190.013050, 0.000130, 0.081000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 3, -1, 0}, 190.082850, 0.000170, 0.039000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 3, 0, 0}, 190.085850, 0.000020, 0.026000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, -4, 0}, 9.876140, 0.000120, 0.555000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, -3, 0}, 9.991810, 0.004400, 0.087000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, -3, 2}, 190.025390, 0.000020, 0.026000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, -2, -2}, 10.052060, 0.000040, 0.469000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, -2, 0}, 9.967310, 0.008970, 0.040000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, -2, 2}, 190.068950, 0.000050, 0.019000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, -1, -2}, 190.049880, 0.000110, 0.090000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, -1, 0}, 10.081500, 0.004470, 0.026000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, -1, 2}, 190.085350, 0.000020, 0.015000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, 0, 0}, 10.159160, 0.000660, 0.020000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 4, 1, 0}, 10.372020, 0.000060, 0.016000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 6, -4, 0}, 9.993880, 0.000060, 0.044000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 6, -3, 0}, 10.049520, 0.000170, 0.028000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 6, -2, 0}, 10.057480, 0.000150, 0.020000},
	{{0, 0, 2, 0, -3, 0, 0, 0, 6, -1, 0}, 10.085110, 0.000050, 0.016000},
	{{0, 0, 2, 0, -3, 0, 1, 0, 2, -3, 0}, 0.730830, 0.000010, 0.075000},
	{{0, 0, 2, 0, -3, 0, 1, 0, 2, -2, 0}, 180.737490, 0.000020, 66.542000},
	{{0, 0, 2, 0, -3, 0, 1, 0, 2, -1, 0}, 180.730830, 0.000010, 0.076000},
	{{0, 0, 2, 0, -3, 0, 2, 0, 2, -2, 0}, 349.757140, 0.000300, 319.860000},
	{{0, 0, 2, 0, -3, 0, 2, 0, 2, -1, 0}, 349.777960, 0.000070, 0.075000},
	{{0, 0, 2, 0, -3, 1, 0, 0, -2, 0, 0}, 3.602120, 0.000070, 0.044000},
	{{0, 0, 2, 0, -3, 1, 0, 0, -2, 1, 0}, 3.602120, 0.000070, 0.103000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 0, -3, 0}, 181.058210, 0.000020, 0.026000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 0, -2, 0}, 181.314150, 0.000200, 0.040000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 0, -1, 0}, 181.397710, 0.001210, 0.087000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 0, 0, 0}, 5.950270, 0.000050, 0.561000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 0, 1, 0}, 3.602120, 0.000010, 0.067000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 2, -4, 0}, 179.815680, 0.000020, 0.038000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 2, -3, 0}, 179.822090, 0.000370, 0.075000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 2, -2, 0}, 181.297640, 0.011260, 142.500000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 2, -1, 0}, 1.456380, 0.004790, 0.075000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 2, 0, 0}, 1.462260, 0.000310, 0.038000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 2, 1, 0}, 1.438290, 0.000030, 0.025000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 4, -3, 0}, 181.918740, 0.000180, 0.087000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 4, -2, 0}, 181.250990, 0.000060, 0.040000},
	{{0, 0, 2, 0, -3, 1, 0, 0, 4, -1, 0}, 1.488460, 0.000030, 0.026000},
	{{0, 0, 2, 0, -3, 2, 0, 0, -2, 0, 0}, 202.892510, 0.000080, 0.044000},
	{{0, 0, 2, 0, -3, 2, 0, 0, -2, 1, 0}, 202.892510, 0.000080, 0.103000},
	{{0, 0, 2, 0, -3, 2, 0, 0, 0, -2, 0}, 11.291520, 0.000010, 0.040000},
	{{0, 0, 2, 0, -3, 2, 0, 0, 0, -1, 0}, 10.647590, 0.000180, 0.087000},
	{{0, 0, 2, 0, -3, 2, 0, 0, 0, 0, 0}, 197.634210, 0.000150, 0.551000},
	{{0, 0, 2, 0, -3, 2, 0, 0, 0, 1, 0}, 195.913620, 0.000030, 0.066000},
	{{0, 0, 2, 0, -3, 2, 0, 0, 2, -3, 0}, 1.447420, 0.000040, 0.076000},
	{{0, 0, 2, 0, -3, 2, 0, 0, 2, -2, 0}, 10.513720, 0.000330, 24.411000},
	{{0, 0, 2, 0, -3, 2, 0, 0, 2, -1, 0}, 191.465560, 0.001060, 0.075000},
	{{0, 0, 2, 0, -3, 2, 0, 0, 2, 0, 0}, 193.457600, 0.000060, 0.038000},
	{{0, 0, 2, 0, -3, 2, 0, 0, 4, -2, 0}, 191.142040, 0.000010, 0.040000},
	{{0, 0, 2, 0, -3, 3, 0, 0, 0, 0, 0}, 10.260920, 0.000010, 0.541000},
	{{0, 0, 2, 0, -3, 3, 0, 0, 2, -3, 0}, 254.912140, 0.000020, 0.076000},
	{{0, 0, 2, 0, -3, 3, 0, 0, 2, -1, 0}, 334.075210, 0.000020, 0.075000},
	{{0, 0, 2, 0, -3, 5, 0, 0, -2, 0, 0}, 76.384360, 0.000050, 0.044000},
	{{0, 0, 2, 0, -3, 5, 0, 0, -2, 1, 0}, 76.020450, 0.000040, 0.105000},
	{{0, 0, 2, 0, -3, 5, 0, 0, 0, -1, 0}, 77.505620, 0.000010, 0.088000},
	{{0, 0, 2, 0, -3, 5, 0, 0, 0, 0, 0}, 83.823840, 0.000040, 0.522000},
	{{0, 0, 2, 0, -3, 5, 0, 0, 0, 1, 0}, 77.505620, 0.000010, 0.066000},
	{{0, 0, 2, 0, -3, 5, 0, 0, 2, -2, 0}, 159.397840, 0.000020, 7.002000},
	{{0, 0, 2, 0, -3, 5, 0, 0, 2, -1, 0}, 130.218540, 0.000050, 0.075000},
	{{0, 0, 2, 0, -3, 5, 0, 0, 2, 0, 0}, 159.397840, 0.000020, 0.038000},
	{{0, 0, 2, 0, -2, -5, 0, 0, -2, 0, 0}, 208.355040, 0.000030, 0.043000},
	{{0, 0, 2, 0, -2, -5, 0, 0, -2, 1, 0}, 208.351470, 0.000030, 0.102000},
	{{0, 0, 2, 0, -2, -5, 0, 0, 0, -1, 0}, 209.411990, 0.000020, 0.086000},
	{{0, 0, 2, 0, -2, -5, 0, 0, 0, 0, 0}, 209.459790, 0.000070, 0.602000},
	{{0, 0, 2, 0, -2, -5, 0, 0, 2, -3, 0}, 209.884740, 0.000020, 0.075000},
	{{0, 0, 2, 0, -2, -5, 0, 0, 2, -2, 0}, 28.397520, 0.000010, 8.900000},
	{{0, 0, 2, 0, -2, -5, 0, 0, 2, -1, 0}, 29.411980, 0.000110, 0.076000},
	{{0, 0, 2, 0, -2, -3, 0, 0, 0, 0, 0}, 181.094960, 0.000010, 0.578000},
	{{0, 0, 2, 0, -2, -3, 0, 0, 2, -3, 0}, 178.909670, 0.000020, 0.075000},
	{{0, 0, 2, 0, -2, -3, 0, 0, 2, -1, 0}, 358.909670, 0.000020, 0.076000},
	{{0, 0, 2, 0, -2, -2, 0, 0, -2, 0, 0}, 336.864750, 0.000010, 0.044000},
	{{0, 0, 2, 0, -2, -2, 0, 0, 0, -1, 0}, 345.693350, 0.000150, 0.087000},
	{{0, 0, 2, 0, -2, -2, 0, 0, 0, 0, 0}, 339.009710, 0.000030, 0.567000},
	{{0, 0, 2, 0, -2, -2, 0, 0, 2, -3, 0}, 182.364080, 0.000010, 0.075000},
	{{0, 0, 2, 0, -2, -2, 0, 0, 2, -2, 0}, 165.357110, 0.001410, 95.036000},
	{{0, 0, 2, 0, -2, -2, 0, 0, 2, -1, 0}, 165.636090, 0.000910, 0.076000},
	{{0, 0, 2, 0, -2, -2, 0, 0, 2, 0, 0}, 165.662160, 0.000060, 0.038000},
	{{0, 0, 2, 0, -2, -2, 0, 0, 4, -3, 0}, 163.791480, 0.000030, 0.087000},
	{{0, 0, 2, 0, -2, -2, 0, 0, 4, -2, 0}, 165.371580, 0.000030, 0.040000},
	{{0, 0, 2, 0, -2, -1, 0, 0, -2, 0, 0}, 146.975170, 0.000020, 0.044000},
	{{0, 0, 2, 0, -2, -1, 0, 0, 0, -1, 0}, 348.082620, 0.000040, 0.087000},
	{{0, 0, 2, 0, -2, -1, 0, 0, 0, 0, 0}, 150.521190, 0.000020, 0.556000},
	{{0, 0, 2, 0, -2, -1, 0, 0, 2, -3, 0}, 160.401090, 0.000040, 0.076000},
	{{0, 0, 2, 0, -2, -1, 0, 0, 2, -2, 0}, 349.511660, 0.000100, 42.689000},
	{{0, 0, 2, 0, -2, -1, 0, 0, 2, -1, 0}, 168.082490, 0.000190, 0.075000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -6, 1, 0}, 180.981620, 0.000040, 0.017000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -6, 2, 0}, 181.352400, 0.000030, 0.022000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -6, 3, 0}, 180.864550, 0.000010, 0.031000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -4, -1, 0}, 178.534330, 0.000120, 0.016000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -4, 0, 0}, 181.897610, 0.000930, 0.021000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -4, 1, -2}, 0.862790, 0.000020, 0.016000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -4, 1, 0}, 181.419090, 0.002110, 0.029000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -4, 1, 2}, 180.918460, 0.000020, 0.133000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -4, 2, 0}, 181.280670, 0.001210, 0.047000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -4, 3, 0}, 180.895330, 0.000040, 0.127000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -3, -1, 0}, 181.113480, 0.000040, 0.021000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -3, 0, 0}, 181.396270, 0.000050, 0.028000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -3, 1, 0}, 0.953580, 0.000120, 0.045000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -3, 2, 0}, 1.140250, 0.000020, 0.114000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, -3, 0}, 181.690190, 0.000030, 0.016000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, -2, 0}, 181.587090, 0.000220, 0.020000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, -1, -2}, 0.814450, 0.000040, 0.016000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, -1, 0}, 181.400700, 0.005270, 0.028000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, -1, 2}, 0.866420, 0.000160, 0.107000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, 0, -2}, 0.800530, 0.000190, 0.020000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, 0, 0}, 180.959180, 0.084080, 0.044000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, 0, 2}, 180.900690, 0.000790, 0.254000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, 1, -2}, 0.823260, 0.000240, 0.027000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, 1, 0}, 180.861100, 0.089110, 0.104000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, 1, 2}, 0.962780, 0.000040, 0.058000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, 2, 0}, 180.860090, 0.003800, 0.277000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, 3, 0}, 180.986020, 0.000250, 0.059000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -2, 4, 0}, 180.876770, 0.000020, 0.033000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -1, -2, 0}, 180.329000, 0.000030, 0.027000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -1, -1, 0}, 179.512670, 0.000060, 0.042000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -1, -1, 1}, 177.587180, 0.000040, 0.096000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -1, 0, 0}, 1.337640, 0.001290, 0.095000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -1, 0, 1}, 353.598030, 0.000060, 0.347000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -1, 1, 0}, 0.462880, 0.000150, 0.368000},
	{{0, 0, 2, 0, -2, 0, 0, 0, -1, 2, 0}, 0.186060, 0.000020, 0.063000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, -4, 0}, 0.070570, 0.000090, 0.020000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, -3, 0}, 0.014260, 0.001070, 0.026000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, -2, -2}, 180.061210, 0.000090, 0.019000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, -2, 0}, 359.985420, 0.014340, 0.041000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, -1, -2}, 179.959240, 0.000370, 0.026000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, -1, 0}, 359.984500, 0.182560, 0.088000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, -1, 2}, 2.460090, 0.000020, 0.065000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, 0, -2}, 180.098680, 0.000070, 0.040000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, 0, 0}, 180.436080, 0.184570, 0.546000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, 0, 2}, 0.427630, 0.000110, 0.035000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, 1, -2}, 0.367030, 0.000140, 0.085000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, 1, 0}, 180.409270, 0.033370, 0.066000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, 1, 2}, 0.311380, 0.000070, 0.024000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, 2, 0}, 180.337740, 0.002300, 0.035000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, 2, 2}, 0.512570, 0.000010, 0.018000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 0, 3, 0}, 180.409990, 0.000150, 0.024000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 1, -2, 0}, 180.097180, 0.000160, 0.081000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 1, -1, -1}, 172.498020, 0.000010, 0.080000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 1, -1, 0}, 180.060300, 0.000680, 1.059000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 1, -1, 1}, 177.587180, 0.000020, 0.070000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 1, 0, -1}, 203.821420, 0.000070, 1.286000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 1, 0, 0}, 0.601560, 0.001280, 0.070000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 1, 1, -1}, 172.498020, 0.000010, 0.071000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 1, 1, 0}, 0.184460, 0.000110, 0.036000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -5, 0}, 0.054920, 0.000020, 0.025000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -4, 0}, 0.056110, 0.000350, 0.038000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -3, -2}, 180.051030, 0.000030, 0.025000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -3, 0}, 0.066380, 0.005530, 0.076000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -2, -2}, 180.077330, 0.000290, 0.037000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -2, 0}, 0.086420, 0.245150, 17.430000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -2, 2}, 180.054520, 0.000110, 0.037000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -1, -2}, 180.119560, 0.002510, 0.074000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -1, 0}, 180.119770, 1.143070, 0.075000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, -1, 2}, 0.122420, 0.002270, 0.025000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, 0, -2}, 0.102860, 0.001580, 3.621000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, 0, 0}, 180.131790, 0.068630, 0.038000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, 0, 2}, 0.143400, 0.000380, 0.019000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, 1, -2}, 180.103270, 0.000450, 0.077000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, 1, 0}, 180.178240, 0.005040, 0.025000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, 1, 2}, 0.121830, 0.000050, 0.015000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, 2, -2}, 180.113720, 0.000040, 0.038000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, 2, 0}, 179.964600, 0.000340, 0.019000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 2, 3, 0}, 179.947930, 0.000030, 0.015000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 3, -2, 0}, 0.182290, 0.000040, 0.080000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 3, -1, 0}, 0.123830, 0.000470, 0.039000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 3, 0, 0}, 0.142860, 0.000050, 0.026000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -5, 0}, 0.100010, 0.000010, 0.067000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -4, 0}, 0.099680, 0.000150, 0.583000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -3, 0}, 0.090140, 0.004290, 0.087000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -3, 2}, 180.077730, 0.000020, 0.026000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -2, -2}, 0.114890, 0.000080, 0.488000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -2, 0}, 180.128770, 0.012290, 0.040000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -2, 2}, 0.129180, 0.000070, 0.019000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -1, -2}, 0.123530, 0.000430, 0.089000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -1, 0}, 180.126290, 0.010380, 0.026000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, -1, 2}, 0.121850, 0.000060, 0.015000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, 0, -2}, 0.268600, 0.000020, 0.041000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, 0, 0}, 180.114990, 0.001410, 0.019000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 4, 1, 0}, 180.129850, 0.000120, 0.015000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 5, -2, 0}, 0.121980, 0.000020, 0.027000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 6, -4, 0}, 0.088890, 0.000060, 0.043000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 6, -3, 0}, 180.141070, 0.000130, 0.028000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 6, -2, 0}, 180.128250, 0.000260, 0.020000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 6, -1, 0}, 180.129650, 0.000120, 0.016000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 6, 0, 0}, 180.121850, 0.000030, 0.013000},
	{{0, 0, 2, 0, -2, 0, 0, 0, 8, -3, 0}, 180.121980, 0.000020, 0.016000},
	{{0, 0, 2, 0, -2, 1, 0, 0, -2, 0, 0}, 41.732640, 0.000020, 0.044000},
	{{0, 0, 2, 0, -2, 1, 0, 0, 0, 0, 0}, 33.742770, 0.000030, 0.536000},
	{{0, 0, 2, 0, -2, 1, 0, 0, 2, -3, 0}, 331.132160, 0.000020, 0.076000},
	{{0, 0, 2, 0, -2, 1, 0, 0, 2, -1, 0}, 6.002220, 0.000050, 0.075000},
	{{0, 0, 2, 0, -2, 2, 0, 0, -2, 0, 0}, 283.488980, 0.000050, 0.044000},
	{{0, 0, 2, 0, -2, 2, 0, 0, -2, 1, 0}, 283.491580, 0.000050, 0.104000},
	{{0, 0, 2, 0, -2, 2, 0, 0, 0, -1, 0}, 100.101210, 0.000040, 0.088000},
	{{0, 0, 2, 0, -2, 2, 0, 0, 0, 0, 0}, 282.631020, 0.000090, 0.527000},
	{{0, 0, 2, 0, -2, 2, 0, 0, 0, 1, 0}, 283.141000, 0.000010, 0.066000},
	{{0, 0, 2, 0, -2, 2, 0, 0, 2, -2, 0}, 104.884390, 0.000020, 7.983000},
	{{0, 0, 2, 0, -2, 2, 0, 0, 2, -1, 0}, 280.954320, 0.000240, 0.075000},
	{{0, 0, 2, 0, -2, 3, 0, 0, -2, 0, 0}, 169.176080, 0.000060, 0.044000},
	{{0, 0, 2, 0, -2, 3, 0, 0, -2, 1, 0}, 169.169690, 0.000050, 0.105000},
	{{0, 0, 2, 0, -2, 3, 0, 0, 0, -1, 0}, 351.461290, 0.000040, 0.088000},
	{{0, 0, 2, 0, -2, 3, 0, 0, 0, 0, 0}, 170.302390, 0.000110, 0.517000},
	{{0, 0, 2, 0, -2, 3, 0, 0, 0, 1, 0}, 169.548360, 0.000020, 0.066000},
	{{0, 0, 2, 0, -2, 3, 0, 0, 2, -2, 0}, 348.064030, 0.000020, 6.281000},
	{{0, 0, 2, 0, -2, 3, 0, 0, 2, -1, 0}, 170.933980, 0.000270, 0.075000},
	{{0, 0, 2, 0, -2, 3, 0, 0, 2, 0, 0}, 170.933980, 0.000020, 0.037000},
	{{0, 0, 2, 0, -2, 5, 0, 0, -2, 0, 0}, 106.953160, 0.000030, 0.044000},
	{{0, 0, 2, 0, -2, 5, 0, 0, -2, 1, 0}, 168.165190, 0.000020, 0.105000},
	{{0, 0, 2, 0, -2, 5, 0, 0, 0, 0, 0}, 95.449810, 0.000020, 0.500000},
	{{0, 0, 2, 0, -2, 5, 0, 0, 2, -3, 0}, 140.766100, 0.000030, 0.077000},
	{{0, 0, 2, 0, -2, 5, 0, 0, 2, -1, 0}, 134.076060, 0.000030, 0.074000},
	{{0, 0, 2, 0, -1, -5, 0, 0, -2, -1, 0}, 198.781290, 0.000010, 0.028000},
	{{0, 0, 2, 0, -1, -5, 0, 0, -2, 0, 0}, 198.764250, 0.000130, 0.043000},
	{{0, 0, 2, 0, -1, -5, 0, 0, -2, 1, 0}, 198.763740, 0.000140, 0.103000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 0, -2, 0}, 198.855170, 0.000020, 0.040000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 0, -1, 0}, 198.985870, 0.000780, 0.087000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 0, 0, 0}, 199.277730, 0.000460, 0.573000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 0, 1, 0}, 199.102710, 0.000060, 0.067000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 2, -3, 0}, 54.525650, 0.000020, 0.075000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 2, -2, 0}, 18.661980, 0.002240, 35.635000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 2, -1, 0}, 18.962500, 0.003620, 0.076000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 2, 0, 0}, 18.953650, 0.000260, 0.038000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 4, -3, 0}, 15.269330, 0.000050, 0.087000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 4, -2, 0}, 18.717830, 0.000070, 0.040000},
	{{0, 0, 2, 0, -1, -5, 0, 0, 4, -1, 0}, 18.950850, 0.000030, 0.026000},
	{{0, 0, 2, 0, -1, -4, 0, 0, 2, -3, 0}, 196.551830, 0.000020, 0.075000},
	{{0, 0, 2, 0, -1, -4, 0, 0, 2, -2, 0}, 299.693320, 0.000110, 169.910000},
	{{0, 0, 2, 0, -1, -4, 0, 0, 2, -1, 0}, 156.419130, 0.000050, 0.075000},
	{{0, 0, 2, 0, -1, -3, 0, 0, 0, 0, 0}, 137.581760, 0.000010, 0.551000},
	{{0, 0, 2, 0, -1, -3, 0, 0, 2, -3, 0}, 309.045880, 0.000030, 0.076000},
	{{0, 0, 2, 0, -1, -3, 0, 0, 2, -2, 0}, 297.152710, 0.000010, 25.105000},
	{{0, 0, 2, 0, -1, -3, 0, 0, 2, -1, 0}, 252.202370, 0.000010, 0.075000},
	{{0, 0, 2, 0, -1, -2, 0, 0, -2, 0, 0}, 351.918410, 0.000060, 0.044000},
	{{0, 0, 2, 0, -1, -2, 0, 0, -2, 1, 0}, 351.918410, 0.000050, 0.104000},
	{{0, 0, 2, 0, -1, -2, 0, 0, 0, -1, 0}, 169.626630, 0.000080, 0.088000},
	{{0, 0, 2, 0, -1, -2, 0, 0, 0, 0, 0}, 351.039390, 0.000110, 0.541000},
	{{0, 0, 2, 0, -1, -2, 0, 0, 0, 1, 0}, 351.918420, 0.000010, 0.066000},
	{{0, 0, 2, 0, -1, -2, 0, 0, 2, -3, 0}, 146.216990, 0.000010, 0.076000},
	{{0, 0, 2, 0, -1, -2, 0, 0, 2, -2, 0}, 171.273660, 0.000070, 13.554000},
	{{0, 0, 2, 0, -1, -2, 0, 0, 2, -1, 0}, 349.908330, 0.000460, 0.075000},
	{{0, 0, 2, 0, -1, -2, 0, 0, 2, 0, 0}, 349.908490, 0.000030, 0.038000},
	{{0, 0, 2, 0, -1, -1, 0, 0, -2, 0, 0}, 173.334670, 0.000020, 0.044000},
	{{0, 0, 2, 0, -1, -1, 0, 0, 0, 0, 0}, 173.451350, 0.000030, 0.531000},
	{{0, 0, 2, 0, -1, -1, 0, 0, 2, -3, 0}, 358.177820, 0.000020, 0.076000},
	{{0, 0, 2, 0, -1, -1, 0, 0, 2, -2, 0}, 3.177090, 0.000010, 9.283000},
	{{0, 0, 2, 0, -1, -1, 0, 0, 2, -1, 0}, 169.601130, 0.000040, 0.075000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -6, 1, 0}, 261.788350, 0.000020, 0.017000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -6, 2, 0}, 261.985260, 0.000010, 0.022000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -6, 3, 0}, 260.347820, 0.000010, 0.031000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -4, -1, 0}, 261.374420, 0.000090, 0.016000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -4, 0, 0}, 261.486370, 0.000280, 0.021000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -4, 1, 0}, 262.433060, 0.000630, 0.029000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -4, 2, 0}, 262.777250, 0.000300, 0.048000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -3, 0, 0}, 262.900820, 0.000020, 0.028000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -3, 1, 0}, 86.467740, 0.000020, 0.046000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, -2, 0}, 263.933250, 0.000140, 0.020000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, -1, 0}, 263.286690, 0.001540, 0.028000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, -1, 2}, 84.738500, 0.000030, 0.108000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, 0, -2}, 85.326130, 0.000040, 0.020000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, 0, 0}, 264.378850, 0.016440, 0.044000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, 0, 2}, 264.378940, 0.000140, 0.249000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, 1, -2}, 84.993930, 0.000020, 0.027000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, 1, 0}, 264.832120, 0.015370, 0.105000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, 1, 2}, 84.854610, 0.000030, 0.058000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, 2, 0}, 264.947440, 0.000640, 0.271000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -2, 3, 0}, 264.283990, 0.000060, 0.059000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -1, -2, 0}, 262.502770, 0.000010, 0.027000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -1, -1, 0}, 257.767510, 0.000060, 0.042000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -1, 0, 0}, 254.072920, 0.000530, 0.096000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -1, 0, 1}, 273.552720, 0.000030, 0.337000},
	{{0, 0, 2, 0, -1, 0, 0, 0, -1, 1, 0}, 244.219010, 0.000030, 0.357000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 0, -3, 0}, 227.575870, 0.000010, 0.026000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 0, -2, 0}, 247.831490, 0.000380, 0.041000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 0, -1, 0}, 245.457490, 0.004560, 0.088000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 0, 0, -2}, 90.374150, 0.000010, 0.040000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 0, 0, 0}, 270.160440, 0.015890, 0.522000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 0, 1, 0}, 271.983720, 0.002510, 0.066000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 0, 2, 0}, 276.512760, 0.000150, 0.035000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 0, 3, 0}, 236.077860, 0.000020, 0.024000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 1, -2, 0}, 84.369330, 0.000020, 0.082000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 1, -1, -1}, 257.668530, 0.000030, 0.081000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 1, -1, 0}, 248.945790, 0.000040, 0.972000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 1, -1, 1}, 79.552850, 0.000030, 0.069000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 1, 0, -1}, 257.932500, 0.000280, 1.160000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 1, 0, 0}, 253.046050, 0.000320, 0.070000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 1, 0, 1}, 78.794610, 0.000020, 0.036000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 1, 1, -1}, 257.668530, 0.000030, 0.071000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 1, 1, 0}, 257.773260, 0.000050, 0.036000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, -3, 0}, 260.510650, 0.000020, 0.076000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, -2, 0}, 201.315080, 0.000620, 7.058000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, -1, -2}, 320.375530, 0.000040, 0.074000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, -1, -1}, 79.893040, 0.000030, 39.875000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, -1, 0}, 307.248960, 0.012990, 0.075000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, -1, 2}, 125.669250, 0.000030, 0.025000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, 0, -2}, 194.820260, 0.000040, 5.213000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, 0, -1}, 80.555490, 0.000040, 0.076000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, 0, 0}, 299.596800, 0.000720, 0.038000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, 1, -2}, 339.130260, 0.000020, 0.077000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 2, 1, 0}, 282.547310, 0.000050, 0.025000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 3, -1, 0}, 257.772360, 0.000010, 0.039000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 4, -2, 0}, 304.781090, 0.000170, 0.040000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 4, -1, 0}, 305.634160, 0.000110, 0.026000},
	{{0, 0, 2, 0, -1, 0, 0, 0, 4, 0, 0}, 329.375320, 0.000020, 0.019000},
	{{0, 0, 2, 0, -1, 5, 0, 0, 2, -2, 0}, 128.621170, 0.000020, 3.211000},
	{{0, 0, 2, 0, -1, 5, 0, 0, 2, 0, 0}, 128.621170, 0.000020, 0.037000},
	{{0, 0, 2, 0, 0, -7, 0, 0, 2, -2, 0}, 15.198870, 0.000040, 85.805000},
	{{0, 0, 2, 0, 0, -7, 0, 0, 2, -1, 0}, 8.462910, 0.000020, 0.076000},
	{{0, 0, 2, 0, 0, -6, 0, 0, 2, -2, 0}, 8.234460, 0.000020, 44.857000},
	{{0, 0, 2, 0, 0, -6, 0, 0, 2, -1, 0}, 191.285810, 0.000030, 0.075000},
	{{0, 0, 2, 0, 0, -6, 3, 0, 0, 0, 0}, 12.025260, 0.000010, 0.546000},
	{{0, 0, 2, 0, 0, -6, 3, 0, 2, -1, 0}, 7.969070, 0.000030, 0.075000},
	{{0, 0, 2, 0, 0, -5, 0, 0, -4, 1, 0}, 187.841830, 0.000020, 0.029000},
	{{0, 0, 2, 0, 0, -5, 0, 0, -2, -1, 0}, 187.157080, 0.000050, 0.028000},
	{{0, 0, 2, 0, 0, -5, 0, 0, -2, 0, 0}, 185.899220, 0.000470, 0.044000},
	{{0, 0, 2, 0, 0, -5, 0, 0, -2, 1, 0}, 189.059480, 0.000520, 0.104000},
	{{0, 0, 2, 0, 0, -5, 0, 0, -2, 2, 0}, 189.104140, 0.000010, 0.277000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 0, -2, 0}, 7.617590, 0.000100, 0.041000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 0, -1, 0}, 7.373740, 0.001110, 0.088000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 0, 0, 0}, 187.724800, 0.001080, 0.546000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 0, 1, 0}, 188.494230, 0.000170, 0.066000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 0, 2, 0}, 188.794040, 0.000030, 0.035000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 2, -3, 0}, 15.964550, 0.000040, 0.076000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 2, -2, 0}, 7.954740, 0.001490, 17.781000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 2, -1, 0}, 187.648950, 0.006750, 0.075000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 2, 0, -2}, 8.638990, 0.000010, 3.607000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 2, 0, 0}, 187.228670, 0.000380, 0.038000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 2, 1, 0}, 187.665300, 0.000030, 0.025000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 2, 2, 0}, 8.696290, 0.000010, 0.019000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 4, -3, 0}, 5.145380, 0.000020, 0.087000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 4, -2, 0}, 187.721340, 0.000070, 0.040000},
	{{0, 0, 2, 0, 0, -5, 0, 0, 4, -1, 0}, 187.669970, 0.000060, 0.026000},
	{{0, 0, 2, 0, 0, -4, 0, 0, -2, 0, 0}, 5.373120, 0.000090, 0.044000},
	{{0, 0, 2, 0, 0, -4, 0, 0, -2, 1, 0}, 5.373140, 0.000100, 0.104000},
	{{0, 0, 2, 0, 0, -4, 0, 0, 0, -2, 0}, 186.885810, 0.000010, 0.041000},
	{{0, 0, 2, 0, 0, -4, 0, 0, 0, -1, 0}, 186.231490, 0.000130, 0.088000},
	{{0, 0, 2, 0, 0, -4, 0, 0, 0, 0, 0}, 7.603410, 0.000250, 0.536000},
	{{0, 0, 2, 0, 0, -4, 0, 0, 0, 1, 0}, 6.056300, 0.000050, 0.066000},
	{{0, 0, 2, 0, 0, -4, 0, 0, 2, -3, 0}, 268.231340, 0.000010, 0.076000},
	{{0, 0, 2, 0, 0, -4, 0, 0, 2, -2, 0}, 185.537920, 0.000100, 11.088000},
	{{0, 0, 2, 0, 0, -4, 0, 0, 2, -1, 0}, 6.184640, 0.000850, 0.075000},
	{{0, 0, 2, 0, 0, -4, 0, 0, 2, 0, 0}, 6.534570, 0.000040, 0.038000},
	{{0, 0, 2, 0, 0, -4, 0, 0, 4, -2, 0}, 6.919480, 0.000010, 0.040000},
	{{0, 0, 2, 0, 0, -3, 0, 0, -4, 1, 0}, 272.937980, 0.000030, 0.029000},
	{{0, 0, 2, 0, 0, -3, 0, 0, -2, -1, 0}, 273.046980, 0.000080, 0.028000},
	{{0, 0, 2, 0, 0, -3, 0, 0, -2, 0, 0}, 273.180740, 0.000710, 0.044000},
	{{0, 0, 2, 0, 0, -3, 0, 0, -2, 1, 0}, 272.736080, 0.000710, 0.104000},
	{{0, 0, 2, 0, 0, -3, 0, 0, -2, 2, 0}, 272.720410, 0.000040, 0.272000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 0, -2, 0}, 93.126690, 0.000040, 0.041000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 0, -1, 0}, 92.941900, 0.000500, 0.088000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 0, 0, 0}, 273.018020, 0.001500, 0.527000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 0, 1, 0}, 272.833460, 0.000260, 0.066000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 0, 2, 0}, 272.747900, 0.000020, 0.035000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 1, 0, -1}, 294.420200, 0.000020, 1.184000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 2, -3, 0}, 117.050410, 0.000030, 0.076000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 2, -2, 0}, 92.754040, 0.000300, 8.056000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 2, -1, 0}, 272.875590, 0.004570, 0.075000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 2, 0, -2}, 92.718310, 0.000020, 4.776000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 2, 0, 0}, 272.951770, 0.000220, 0.038000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 4, -2, 0}, 273.020680, 0.000060, 0.040000},
	{{0, 0, 2, 0, 0, -3, 0, 0, 4, -1, 0}, 272.885700, 0.000040, 0.026000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -4, -1, 0}, 0.216800, 0.000010, 0.016000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -4, 0, 0}, 180.202690, 0.000050, 0.021000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -4, 1, 0}, 180.205070, 0.000080, 0.029000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -4, 2, 0}, 180.212680, 0.000040, 0.048000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -2, -2, 0}, 180.224540, 0.000030, 0.020000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -2, -1, 0}, 180.230630, 0.000240, 0.028000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -2, 0, 0}, 180.234050, 0.003580, 0.044000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -2, 0, 2}, 180.232470, 0.000010, 0.248000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -2, 1, 0}, 180.211680, 0.003580, 0.105000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -2, 2, 0}, 180.238340, 0.000160, 0.270000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -1, 0, 0}, 359.920710, 0.000040, 0.096000},
	{{0, 0, 2, 0, 0, -2, 0, 0, -1, 0, 1}, 10.526510, 0.000010, 0.335000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 0, -3, 0}, 0.454130, 0.000020, 0.026000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 0, -2, 0}, 359.851860, 0.000080, 0.041000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 0, -1, 0}, 359.883180, 0.001680, 0.088000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 0, 0, 0}, 180.105310, 0.007310, 0.518000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 0, 1, 0}, 180.095980, 0.001310, 0.066000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 0, 2, 0}, 180.018310, 0.000090, 0.035000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 0, 3, 0}, 0.247970, 0.000010, 0.024000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 1, 0, -1}, 189.781150, 0.000010, 1.139000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 1, 0, 0}, 359.929080, 0.000050, 0.070000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, -3, 0}, 173.222450, 0.000030, 0.076000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, -2, 0}, 359.986170, 0.000780, 6.326000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, -1, -2}, 180.017870, 0.000080, 0.074000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, -1, -1}, 189.711880, 0.000010, 115.334000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, -1, 0}, 180.033530, 0.018730, 0.075000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, -1, 2}, 0.034300, 0.000040, 0.025000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, 0, -2}, 359.891290, 0.000070, 5.700000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, 0, 0}, 180.064860, 0.000880, 0.037000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, 1, -2}, 180.005550, 0.000040, 0.076000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 2, 1, 0}, 180.005550, 0.000040, 0.025000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 4, -3, 0}, 357.229570, 0.000010, 0.086000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 4, -2, 0}, 180.043750, 0.000250, 0.040000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 4, -1, 0}, 180.039920, 0.000150, 0.026000},
	{{0, 0, 2, 0, 0, -2, 0, 0, 4, 0, 0}, 180.048660, 0.000020, 0.019000},
	{{0, 0, 2, 0, 0, -1, 0, 0, -4, 0, 0}, 258.563950, 0.000010, 0.021000},
	{{0, 0, 2, 0, 0, -1, 0, 0, -4, 1, 0}, 258.096160, 0.000020, 0.029000},
	{{0, 0, 2, 0, 0, -1, 0, 0, -2, -1, 0}, 258.378270, 0.000120, 0.028000},
	{{0, 0, 2, 0, 0, -1, 0, 0, -2, 0, 0}, 258.423060, 0.001010, 0.044000},
	{{0, 0, 2, 0, 0, -1, 0, 0, -2, 0, 2}, 258.447740, 0.000010, 0.245000},
	{{0, 0, 2, 0, 0, -1, 0, 0, -2, 1, 0}, 258.458870, 0.000980, 0.105000},
	{{0, 0, 2, 0, 0, -1, 0, 0, -2, 2, 0}, 258.481530, 0.000050, 0.267000},
	{{0, 0, 2, 0, 0, -1, 0, 0, 0, -1, 0}, 256.501200, 0.000270, 0.089000},
	{{0, 0, 2, 0, 0, -1, 0, 0, 0, 0, 0}, 259.081320, 0.001070, 0.509000},
	{{0, 0, 2, 0, 0, -1, 0, 0, 0, 1, 0}, 259.627420, 0.000150, 0.066000},
	{{0, 0, 2, 0, 0, -1, 0, 0, 1, 0, -1}, 246.322250, 0.000020, 1.096000},
	{{0, 0, 2, 0, 0, -1, 0, 0, 2, -2, 0}, 254.482820, 0.000020, 5.207000},
	{{0, 0, 2, 0, 0, -1, 0, 0, 2, -1, 0}, 262.643380, 0.000770, 0.074000},
	{{0, 0, 2, 0, 0, -1, 0, 0, 2, 0, 0}, 259.251160, 0.000040, 0.037000},
	{{0, 0, 2, 0, 0, -1, 0, 0, 4, -2, 0}, 262.419340, 0.000010, 0.040000},
	{{0, 0, 2, 0, 0, 0, -3, 0, -2, 0, 0}, 353.606840, 0.000030, 0.044000},
	{{0, 0, 2, 0, 0, 0, -3, 0, 0, 0, 0}, 353.699780, 0.000030, 0.509000},
	{{0, 0, 2, 0, 0, 0, -3, 0, 2, -1, 0}, 354.081930, 0.000030, 0.074000},
	{{0, 0, 2, 0, 0, 0, -2, 0, -2, 0, 0}, 178.798450, 0.000090, 0.044000},
	{{0, 0, 2, 0, 0, 0, -2, 0, -2, 1, 0}, 178.798450, 0.000080, 0.105000},
	{{0, 0, 2, 0, 0, 0, -2, 0, 0, -1, 0}, 1.371150, 0.000040, 0.089000},
	{{0, 0, 2, 0, 0, 0, -2, 0, 0, 0, 0}, 179.396040, 0.000150, 0.506000},
	{{0, 0, 2, 0, 0, 0, -2, 0, 0, 1, 0}, 176.847940, 0.000020, 0.066000},
	{{0, 0, 2, 0, 0, 0, -2, 0, 2, -1, 0}, 179.760670, 0.000280, 0.074000},
	{{0, 0, 2, 0, 0, 0, -2, 0, 2, 0, 0}, 359.986920, 0.000020, 0.037000},
	{{0, 0, 2, 0, 0, 0, -1, 0, -2, 0, 0}, 249.047210, 0.000030, 0.044000},
	{{0, 0, 2, 0, 0, 0, -1, 0, 0, 0, 0}, 247.253640, 0.000020, 0.503000},
	{{0, 0, 2, 0, 0, 0, 0, -2, -2, 0, 0}, 180.219050, 0.000030, 0.044000},
	{{0, 0, 2, 0, 0, 0, 0, -2, 0, 0, 0}, 180.125200, 0.000050, 0.503000},
	{{0, 0, 2, 0, 0, 0, 0, -2, 2, -1, 0}, 180.018970, 0.000070, 0.074000},
	{{0, 0, 2, 0, 0, 0, 0, -1, -2, 0, 0}, 258.845400, 0.000020, 0.044000},
	{{0, 0, 2, 0, 0, 0, 0, -1, 0, 0, 0}, 258.679980, 0.000010, 0.502000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -4, 0, 0}, 357.227810, 0.000020, 0.021000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -4, 1, 0}, 0.897750, 0.000020, 0.029000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -2, -1, 0}, 349.558780, 0.000050, 0.028000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -2, 0, 0}, 1.689490, 0.000620, 0.044000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -2, 1, 0}, 0.945700, 0.000550, 0.105000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -2, 2, 0}, 11.447470, 0.000060, 0.265000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -1, 0, -1}, 11.153410, 0.000050, 0.042000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -1, 0, 0}, 203.562740, 0.000040, 0.096000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -1, 0, 1}, 194.377840, 0.000110, 0.327000},
	{{0, 0, 2, 0, 0, 0, 0, 0, -1, 1, -1}, 10.397260, 0.000010, 0.095000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 0, -1, 0}, 356.893610, 0.000140, 0.089000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0}, 20.938600, 0.003160, 0.500000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 0, 1, 0}, 20.405050, 0.000350, 0.066000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 0, 2, 0}, 46.074930, 0.000020, 0.035000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 1, -1, 0}, 204.536690, 0.000010, 0.898000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 1, 0, -1}, 8.091330, 0.000240, 1.057000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 1, 0, 0}, 204.624370, 0.000070, 0.070000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 1, 0, 1}, 186.155030, 0.000050, 0.036000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 2, -3, 0}, 210.826710, 0.000030, 0.077000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 2, -1, 0}, 23.149970, 0.002030, 0.074000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 2, 0, -2}, 24.357680, 0.000020, 9.300000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 2, 0, -1}, 192.712130, 0.000080, 0.075000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 2, 0, 0}, 9.805210, 0.000090, 0.037000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 2, 1, 0}, 202.239610, 0.000010, 0.025000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 4, -2, 0}, 20.873490, 0.000030, 0.040000},
	{{0, 0, 2, 0, 0, 0, 0, 0, 4, -1, 0}, 23.052080, 0.000020, 0.026000},
	{{0, 0, 2, 0, 0, 1, 0, 0, -2, 0, 0}, 63.320400, 0.000030, 0.044000},
	{{0, 0, 2, 0, 0, 1, 0, 0, -2, 1, 0}, 63.290340, 0.000010, 0.106000},
	{{0, 0, 2, 0, 0, 1, 0, 0, 0, 0, 0}, 66.742700, 0.000010, 0.492000},
	{{0, 0, 2, 0, 0, 2, 0, 0, 2, -1, 0}, 307.952130, 0.000020, 0.074000},
	{{0, 0, 2, 0, 1, -9, 0, 0, 2, -2, 0}, 86.508070, 0.000010, 210.377000},
	{{0, 0, 2, 0, 1, -5, 0, 0, -2, 0, 0}, 287.028560, 0.000040, 0.044000},
	{{0, 0, 2, 0, 1, -5, 0, 0, -2, 1, 0}, 289.358550, 0.000040, 0.105000},
	{{0, 0, 2, 0, 1, -5, 0, 0, 0, -1, 0}, 250.304440, 0.000010, 0.088000},
	{{0, 0, 2, 0, 1, -5, 0, 0, 0, 0, 0}, 315.213860, 0.000040, 0.522000},
	{{0, 0, 2, 0, 1, -5, 0, 0, 2, -3, 0}, 281.609410, 0.000010, 0.076000},
	{{0, 0, 2, 0, 1, -5, 0, 0, 2, -2, 0}, 7.340850, 0.000020, 7.115000},
	{{0, 0, 2, 0, 1, -5, 0, 0, 2, -1, 0}, 18.630960, 0.000080, 0.075000},
	{{0, 0, 2, 0, 1, -2, 0, 0, -2, 0, 0}, 79.222840, 0.000050, 0.044000},
	{{0, 0, 2, 0, 1, -2, 0, 0, -2, 1, 0}, 79.224220, 0.000040, 0.106000},
	{{0, 0, 2, 0, 1, -2, 0, 0, 0, -1, 0}, 78.832980, 0.000010, 0.089000},
	{{0, 0, 2, 0, 1, -2, 0, 0, 0, 0, 0}, 77.843410, 0.000040, 0.496000},
	{{0, 0, 2, 0, 1, -2, 0, 0, 0, 1, 0}, 78.832980, 0.000010, 0.065000},
	{{0, 0, 2, 0, 1, -2, 0, 0, 2, -3, 0}, 258.430920, 0.000010, 0.077000},
	{{0, 0, 2, 0, 1, -2, 0, 0, 2, -1, 0}, 78.430920, 0.000010, 0.074000},
	{{0, 0, 2, 0, 1, 0, 0, 0, -4, 0, 0}, 323.199600, 0.000010, 0.021000},
	{{0, 0, 2, 0, 1, 0, 0, 0, -4, 2, 0}, 143.032380, 0.000010, 0.048000},
	{{0, 0, 2, 0, 1, 0, 0, 0, -2, -1, 0}, 139.484990, 0.000040, 0.028000},
	{{0, 0, 2, 0, 1, 0, 0, 0, -2, 0, 0}, 139.783650, 0.000250, 0.044000},
	{{0, 0, 2, 0, 1, 0, 0, 0, -2, 1, 0}, 141.848270, 0.000240, 0.106000},
	{{0, 0, 2, 0, 1, 0, 0, 0, -1, 0, -1}, 332.537420, 0.000020, 0.042000},
	{{0, 0, 2, 0, 1, 0, 0, 0, -1, 1, -1}, 152.835980, 0.000010, 0.096000},
	{{0, 0, 2, 0, 1, 0, 0, 0, 0, -1, 0}, 136.659960, 0.000090, 0.090000},
	{{0, 0, 2, 0, 1, 0, 0, 0, 0, 0, 0}, 133.567490, 0.000150, 0.480000},
	{{0, 0, 2, 0, 1, 0, 0, 0, 1, -1, -1}, 332.640340, 0.000030, 0.082000},
	{{0, 0, 2, 0, 1, 0, 0, 0, 1, 0, -1}, 332.973250, 0.000020, 0.970000},
	{{0, 0, 2, 0, 1, 0, 0, 0, 1, 1, -1}, 152.690490, 0.000010, 0.070000},
	{{0, 0, 2, 0, 1, 0, 0, 0, 2, -1, 0}, 67.111610, 0.000070, 0.074000},
	{{0, 0, 2, 0, 1, 0, 0, 0, 2, 0, -2}, 166.387040, 0.000020, 43.055000},
	{{0, 0, 2, 0, 1, 2, 0, 0, 2, -3, 0}, 135.697030, 0.000020, 0.078000},
	{{0, 0, 2, 0, 1, 2, 0, 0, 2, -1, 0}, 135.697030, 0.000020, 0.073000},
	{{0, 0, 2, 0, 1, 5, 0, 0, 2, -3, 0}, 110.058140, 0.000020, 0.078000},
	{{0, 0, 2, 0, 1, 5, 0, 0, 2, -2, 0}, 110.448110, 0.000020, 2.083000},
	{{0, 0, 2, 0, 1, 5, 0, 0, 2, -1, 0}, 110.058140, 0.000020, 0.073000},
	{{0, 0, 2, 0, 1, 5, 0, 0, 2, 0, 0}, 110.448110, 0.000020, 0.037000},
	{{0, 0, 2, 0, 2, -10, 0, 0, 2, -3, 0}, 197.615990, 0.000020, 0.076000},
	{{0, 0, 2, 0, 2, -10, 0, 0, 2, -2, 0}, 197.679670, 0.000010, 18.146000},
	{{0, 0, 2, 0, 2, -10, 0, 0, 2, -1, 0}, 17.615990, 0.000020, 0.075000},
	{{0, 0, 2, 0, 2, -8, 0, 0, -2, 0, 0}, 101.769910, 0.000020, 0.044000},
	{{0, 0, 2, 0, 2, -8, 0, 0, 0, 0, 0}, 101.736400, 0.000040, 0.527000},
	{{0, 0, 2, 0, 2, -8, 0, 0, 2, -1, 0}, 101.409280, 0.000070, 0.075000},
	{{0, 0, 2, 0, 2, -7, 0, 0, -2, 0, 0}, 8.273210, 0.000060, 0.044000},
	{{0, 0, 2, 0, 2, -7, 0, 0, -2, 1, 0}, 8.273210, 0.000060, 0.105000},
	{{0, 0, 2, 0, 2, -7, 0, 0, 0, -1, 0}, 188.855310, 0.000040, 0.088000},
	{{0, 0, 2, 0, 2, -7, 0, 0, 0, 0, 0}, 8.291090, 0.000120, 0.518000},
	{{0, 0, 2, 0, 2, -7, 0, 0, 0, 1, 0}, 8.273210, 0.000010, 0.066000},
	{{0, 0, 2, 0, 2, -7, 0, 0, 2, -2, 0}, 188.748770, 0.000020, 6.371000},
	{{0, 0, 2, 0, 2, -7, 0, 0, 2, -1, 0}, 8.735060, 0.000280, 0.075000},
	{{0, 0, 2, 0, 2, -7, 0, 0, 2, 0, 0}, 8.735030, 0.000020, 0.037000},
	{{0, 0, 2, 0, 2, -6, 0, 0, 2, -3, 0}, 268.743310, 0.000010, 0.077000},
	{{0, 0, 2, 0, 2, -6, 0, 0, 2, -1, 0}, 268.743310, 0.000010, 0.074000},
	{{0, 0, 2, 0, 2, -5, 0, 0, -4, 1, 0}, 323.367760, 0.000010, 0.029000},
	{{0, 0, 2, 0, 2, -5, 0, 0, -2, -1, 0}, 323.317430, 0.000040, 0.028000},
	{{0, 0, 2, 0, 2, -5, 0, 0, -2, 0, 0}, 323.369740, 0.000390, 0.044000},
	{{0, 0, 2, 0, 2, -5, 0, 0, -2, 1, 0}, 323.224040, 0.000400, 0.105000},
	{{0, 0, 2, 0, 2, -5, 0, 0, -2, 2, 0}, 323.362420, 0.000020, 0.265000},
	{{0, 0, 2, 0, 2, -5, 0, 0, 0, -1, 0}, 321.826010, 0.000130, 0.089000},
	{{0, 0, 2, 0, 2, -5, 0, 0, 0, 0, 0}, 322.394710, 0.000390, 0.500000},
	{{0, 0, 2, 0, 2, -5, 0, 0, 0, 1, 0}, 322.715390, 0.000050, 0.066000},
	{{0, 0, 2, 0, 2, -5, 0, 0, 2, -3, 0}, 348.163030, 0.000030, 0.077000},
	{{0, 0, 2, 0, 2, -5, 0, 0, 2, -2, 0}, 323.698840, 0.000010, 4.447000},
	{{0, 0, 2, 0, 2, -5, 0, 0, 2, -1, 0}, 328.818560, 0.000150, 0.074000},
	{{0, 0, 2, 0, 2, 0, 0, 0, -2, 0, 0}, 138.263630, 0.000020, 0.044000},
	{{0, 0, 2, 0, 2, 0, 0, 0, 2, -3, 0}, 128.323620, 0.000030, 0.078000},
	{{0, 0, 2, 0, 2, 0, 0, 0, 2, -1, 0}, 308.323620, 0.000030, 0.073000},
	{{0, 0, 2, 0, 3, -5, 0, 0, 2, -2, 0}, 328.854440, 0.000020, 3.235000},
	{{0, 0, 2, 0, 3, -5, 0, 0, 2, 0, 0}, 328.523290, 0.000030, 0.037000},
	{{0, 0, 2, 0, 3, -2, 0, 0, 2, -3, 0}, 316.218330, 0.000010, 0.078000},
	{{0, 0, 2, 0, 3, -2, 0, 0, 2, -1, 0}, 316.218330, 0.000010, 0.073000},
	{{0, 0, 2, 0, 3, -1, 0, 0, 2, -3, 0}, 126.939630, 0.000020, 0.078000},
	{{0, 0, 2, 0, 3, -1, 0, 0, 2, -1, 0}, 126.939630, 0.000020, 0.073000},
	{{0, 0, 2, 0, 4, -3, 0, 0, 2, -3, 0}, 248.541600, 0.000020, 0.078000},
	{{0, 0, 2, 0, 4, -3, 0, 0, 2, -1, 0}, 248.541600, 0.000020, 0.073000},
	{{0, 0, 2, 0, 4, -3, 0, 0, 2, 0, 0}, 248.541600, 0.000010, 0.037000},
	{{0, 0, 2, 0, 4, -2, 0, 0, 2, -3, 0}, 127.337480, 0.000020, 0.078000},
	{{0, 0, 2, 0, 4, -2, 0, 0, 2, -2, 0}, 127.131800, 0.000030, 2.019000},
	{{0, 0, 2, 0, 4, -2, 0, 0, 2, -1, 0}, 127.337480, 0.000020, 0.073000},
	{{0, 0, 2, 0, 4, -2, 0, 0, 2, 0, -2}, 307.076760, 0.000010, 6.181000},
	{{0, 0, 2, 0, 4, -2, 0, 0, 2, 0, 0}, 127.131800, 0.000030, 0.037000},
	{{0, 0, 2, 0, 5, -5, 0, 0, 2, -3, 0}, 264.916990, 0.000020, 0.078000},
	{{0, 0, 2, 0, 5, -5, 0, 0, 2, -2, 0}, 310.681380, 0.000010, 2.093000},
	{{0, 0, 2, 0, 5, -5, 0, 0, 2, -1, 0}, 264.916990, 0.000020, 0.073000},
	{{0, 0, 2, 0, 5, -5, 0, 0, 2, 0, 0}, 310.681380, 0.000010, 0.037000},
	{{0, 0, 2, 1, -1, 0, 0, 0, 0, 0, 0}, 228.600120, 0.000010, 0.409000},
	{{0, 0, 2, 1, 0, 0, 0, 0, 0, 0, 0}, 253.110000, 0.000020, 0.395000},
	{{0, 0, 2, 1, 0, 0, 0, 0, 2, -3, 0}, 283.579090, 0.000010, 0.080000},
	{{0, 0, 2, 1, 0, 0, 0, 0, 2, -1, 0}, 283.579090, 0.000010, 0.071000},
	{{0, 0, 2, 2, -1, 0, 0, 0, 2, -3, 0}, 102.541360, 0.000010, 0.083000},
	{{0, 0, 2, 2, -1, 0, 0, 0, 2, -1, 0}, 102.541360, 0.000010, 0.069000},
	{{0, 0, 2, 2, -1, 0, 0, 0, 2, 0, 0}, 282.541360, 0.000020, 0.036000},
	{{0, 0, 3, -10, 2, 0, 0, 0, -2, 0, 0}, 22.511410, 0.000010, 0.037000},
	{{0, 0, 3, -10, 2, 0, 0, 0, -2, 2, 0}, 202.511410, 0.000010, 2.672000},
	{{0, 0, 3, -9, 0, 0, 0, 0, -2, 1, 0}, 207.616920, 0.000090, 0.075000},
	{{0, 0, 3, -9, 0, 0, 0, 0, -2, 2, 0}, 26.837880, 0.000100, 89.761000},
	{{0, 0, 3, -9, 0, 0, 0, 0, 0, 1, 0}, 27.616580, 0.000020, 0.087000},
	{{0, 0, 3, -9, 0, 0, 0, 2, -2, 2, 0}, 44.970770, 0.000010, 1002.658000},
	{{0, 0, 3, -9, 0, 0, 1, 0, -2, 2, 0}, 0.074070, 0.000040, 1313.736000},
	{{0, 0, 3, -9, 2, 0, 0, 0, -2, 0, 0}, 51.519600, 0.000030, 0.038000},
	{{0, 0, 3, -8, 0, 0, 0, 0, 0, -1, 0}, 348.732150, 0.000010, 0.069000},
	{{0, 0, 3, -8, 0, 0, 0, 0, 0, 0, 0}, 304.592870, 0.000040, 0.798000},
	{{0, 0, 3, -8, 0, 0, 0, 0, 0, 1, 0}, 348.732150, 0.000010, 0.083000},
	{{0, 0, 3, -8, 0, 0, 0, 0, 2, -1, 0}, 349.663840, 0.000040, 0.098000},
	{{0, 0, 3, -8, 0, 0, 0, 0, 2, 0, 0}, 349.656870, 0.000040, 0.043000},
	{{0, 0, 3, -8, 1, 5, 0, 0, 0, -1, 0}, 21.863940, 0.000010, 0.070000},
	{{0, 0, 3, -8, 1, 5, 0, 0, 0, 0, 0}, 21.825770, 0.000090, 1.001000},
	{{0, 0, 3, -8, 1, 5, 0, 0, 0, 1, 0}, 21.863940, 0.000010, 0.082000},
	{{0, 0, 3, -8, 1, 5, 0, 0, 2, -1, 0}, 22.315940, 0.000070, 0.095000},
	{{0, 0, 3, -8, 1, 5, 0, 0, 2, 0, 0}, 21.999160, 0.000070, 0.042000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -4, 0, 0}, 209.726390, 0.000020, 0.020000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -4, 1, 0}, 209.716540, 0.000040, 0.027000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -4, 2, 0}, 209.716880, 0.000020, 0.042000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -2, -2, 0}, 209.715560, 0.000020, 0.019000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -2, -1, 0}, 208.764620, 0.000070, 0.026000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -2, 0, 0}, 209.493560, 0.000530, 0.039000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -2, 0, 2}, 209.717350, 0.000030, 0.903000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -2, 1, 0}, 210.390210, 0.000370, 0.080000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -2, 2, 0}, 32.099180, 0.000030, 1.293000},
	{{0, 0, 3, -8, 3, 0, 0, 0, -1, 0, 0}, 28.939120, 0.000040, 0.075000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 0, -2, 0}, 38.221200, 0.000030, 0.036000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 0, -1, 0}, 31.310240, 0.001010, 0.070000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 0, 0, 0}, 29.797470, 0.011210, 0.999000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 0, 1, 0}, 29.217850, 0.003600, 0.082000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 0, 2, 0}, 28.913810, 0.000310, 0.039000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 0, 3, 0}, 27.915750, 0.000010, 0.026000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 1, -2, 0}, 29.906110, 0.000010, 0.066000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, -3, 0}, 30.368120, 0.000030, 0.062000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, -2, 0}, 30.302480, 0.000380, 0.360000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, -1, -2}, 209.746920, 0.000020, 0.061000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, -1, 0}, 29.961250, 0.009710, 0.095000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, 0, -2}, 29.744500, 0.000130, 0.322000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, 0, 0}, 29.681060, 0.009050, 0.042000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, 0, 2}, 209.749010, 0.000020, 0.020000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, 1, -2}, 209.746440, 0.000010, 0.099000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, 1, 0}, 29.464930, 0.000890, 0.027000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 2, 2, 0}, 29.363210, 0.000100, 0.020000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 3, -1, 0}, 209.740230, 0.000010, 0.044000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 3, 0, 0}, 29.740230, 0.000020, 0.028000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 4, -2, 0}, 30.028310, 0.000190, 0.046000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 4, -1, 0}, 29.809190, 0.000390, 0.028000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 4, 0, 0}, 29.654920, 0.000160, 0.021000},
	{{0, 0, 3, -8, 3, 0, 0, 0, 4, 1, 0}, 29.732290, 0.000040, 0.016000},
	{{0, 0, 3, -8, 5, -5, 0, 0, 2, 0, 0}, 165.952720, 0.000020, 0.042000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -3, 2, -1}, 144.548930, 0.000190, 0.040000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -3, 3, -1}, 144.409860, 0.000230, 0.087000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -2, 0, 0}, 203.691630, 0.000030, 0.039000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -1, 0, -1}, 322.161690, 0.000050, 0.038000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -1, 0, 1}, 322.274950, 0.000040, 3.012000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -1, 1, -1}, 322.162540, 0.000700, 0.075000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -1, 2, -3}, 328.433700, 0.000010, 0.037000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -1, 2, -1}, 144.439170, 0.018060, 650.233000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -1, 2, 1}, 328.433700, 0.000010, 0.037000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -1, 3, -1}, 144.477580, 0.000860, 0.075000},
	{{0, 0, 3, -7, 0, 0, 0, 0, -1, 4, -1}, 144.478140, 0.000060, 0.038000},
	{{0, 0, 3, -7, 0, 0, 0, 0, 0, -1, 0}, 250.468170, 0.000080, 0.072000},
	{{0, 0, 3, -7, 0, 0, 0, 0, 0, 0, 0}, 255.387250, 0.000400, 1.385000},
	{{0, 0, 3, -7, 0, 0, 0, 0, 0, 1, 0}, 251.469780, 0.000080, 0.080000},
	{{0, 0, 3, -7, 0, 0, 0, 0, 1, 0, -1}, 143.176750, 0.000020, 0.563000},
	{{0, 0, 3, -7, 0, 0, 0, 0, 1, 1, -1}, 143.843520, 0.000540, 0.087000},
	{{0, 0, 3, -7, 0, 0, 0, 0, 1, 2, -1}, 144.349120, 0.000230, 0.040000},
	{{0, 0, 3, -7, 0, 0, 0, 0, 1, 3, -1}, 144.394160, 0.000030, 0.026000},
	{{0, 0, 3, -7, 0, 0, 0, 0, 2, -1, 0}, 351.934880, 0.000090, 0.093000},
	{{0, 0, 3, -7, 0, 0, 0, 0, 2, 0, 0}, 341.783710, 0.000090, 0.042000},
	{{0, 0, 3, -7, 1, 0, 0, 0, 0, 0, 0}, 92.847460, 0.000010, 1.569000},
	{{0, 0, 3, -7, 2, 0, 0, 0, 0, 0, 0}, 300.596360, 0.000010, 1.808000},
	{{0, 0, 3, -6, 0, 0, 0, 0, -4, 0, 0}, 271.976400, 0.000020, 0.020000},
	{{0, 0, 3, -6, 0, 0, 0, 0, -4, 2, 0}, 273.161720, 0.000020, 0.043000},
	{{0, 0, 3, -6, 0, 0, 0, 0, -2, -1, 0}, 272.008250, 0.000130, 0.026000},
	{{0, 0, 3, -6, 0, 0, 0, 0, -2, 0, 0}, 271.354340, 0.001130, 0.040000},
	{{0, 0, 3, -6, 0, 0, 0, 0, -2, 0, 2}, 271.746550, 0.000030, 0.522000},
	{{0, 0, 3, -6, 0, 0, 0, 0, -2, 1, 0}, 273.645100, 0.001650, 0.086000},
	{{0, 0, 3, -6, 0, 0, 0, 0, -2, 2, 0}, 280.939730, 0.000040, 0.631000},
	{{0, 0, 3, -6, 0, 0, 0, 0, -1, 0, 0}, 93.238110, 0.000020, 0.080000},
	{{0, 0, 3, -6, 0, 0, 0, 0, -1, 0, 1}, 352.311670, 0.000030, 1.158000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 0, -2, 0}, 268.665880, 0.000200, 0.037000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 0, -1, 0}, 268.916530, 0.003100, 0.074000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 0, -1, 1}, 170.658600, 0.000020, 42.830000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 0, 0, 0}, 271.342540, 0.016730, 5.260000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 0, 1, 0}, 269.157510, 0.003250, 0.077000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 0, 2, 0}, 269.245340, 0.000220, 0.038000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 0, 3, 0}, 268.534870, 0.000010, 0.025000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 1, -2, 0}, 94.110470, 0.000010, 0.070000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 1, 0, -1}, 353.646890, 0.000030, 0.804000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 1, 0, 0}, 93.028650, 0.000030, 0.082000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 2, -2, 0}, 278.569850, 0.000080, 0.509000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 2, -1, 0}, 275.236880, 0.002340, 0.089000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 2, 0, -2}, 274.745560, 0.000030, 0.435000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 2, 0, 0}, 274.964650, 0.001620, 0.041000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 2, 1, 0}, 273.485710, 0.000170, 0.026000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 4, -2, 0}, 274.533630, 0.000030, 0.044000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 4, -1, 0}, 274.524850, 0.000030, 0.028000},
	{{0, 0, 3, -6, 0, 0, 0, 0, 4, 0, 0}, 274.685770, 0.000020, 0.020000},
	{{0, 0, 3, -6, 0, 2, 0, 0, 0, 0, 0}, 331.076610, 0.000020, 8.183000},
	{{0, 0, 3, -6, 1, 0, 0, 0, -2, 0, 0}, 124.888750, 0.000010, 0.040000},
	{{0, 0, 3, -6, 1, 0, 0, 0, 0, 0, 0}, 126.450610, 0.000090, 9.452000},
	{{0, 0, 3, -6, 1, 0, 0, 0, 2, 0, 0}, 125.450240, 0.000020, 0.041000},
	{{0, 0, 3, -6, 2, 0, 0, 0, -4, 1, 0}, 331.159560, 0.000010, 0.028000},
	{{0, 0, 3, -6, 2, 0, 0, 0, -2, -1, 0}, 331.159550, 0.000050, 0.026000},
	{{0, 0, 3, -6, 2, 0, 0, 0, -2, 0, 0}, 331.075400, 0.000460, 0.040000},
	{{0, 0, 3, -6, 2, 0, 0, 0, -2, 0, 2}, 331.159560, 0.000010, 0.479000},
	{{0, 0, 3, -6, 2, 0, 0, 0, -2, 1, 0}, 331.363130, 0.000720, 0.087000},
	{{0, 0, 3, -6, 2, 0, 0, 0, -2, 2, 0}, 331.154290, 0.000030, 0.571000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 0, -2, 0}, 329.911490, 0.000060, 0.038000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 0, -1, 0}, 329.959500, 0.000920, 0.075000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 0, 0, 0}, 330.474140, 0.004910, 46.520000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 0, 1, 0}, 329.968190, 0.000920, 0.076000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 0, 2, 0}, 329.923020, 0.000060, 0.038000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 2, -2, 0}, 331.726370, 0.000020, 0.557000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 2, -1, 0}, 331.440390, 0.000730, 0.087000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 2, 0, -2}, 331.231320, 0.000010, 0.470000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 2, 0, 0}, 331.122520, 0.000470, 0.040000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 2, 1, 0}, 331.218660, 0.000050, 0.026000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 4, -2, 0}, 331.404050, 0.000010, 0.044000},
	{{0, 0, 3, -6, 2, 0, 0, 0, 4, -1, 0}, 331.231320, 0.000020, 0.028000},
	{{0, 0, 3, -6, 3, 0, 0, 0, 0, 0, 0}, 235.663750, 0.000050, 15.922000},
	{{0, 0, 3, -5, -4, 0, 0, 0, -2, 0, 0}, 127.663300, 0.000010, 0.040000},
	{{0, 0, 3, -5, -4, 0, 0, 0, -2, 1, 0}, 127.663300, 0.000020, 0.087000},
	{{0, 0, 3, -5, -4, 0, 0, 0, 0, 0, 0}, 179.452460, 0.000010, 229.298000},
	{{0, 0, 3, -5, -4, 0, 0, 0, 2, -1, 0}, 127.770340, 0.000020, 0.087000},
	{{0, 0, 3, -5, -4, 0, 0, 0, 2, 0, 0}, 127.770340, 0.000010, 0.040000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -4, 0, 0}, 301.538020, 0.000040, 0.020000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -4, 1, 0}, 301.592680, 0.000110, 0.028000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -4, 2, 0}, 301.797530, 0.000060, 0.044000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -2, -2, 0}, 301.729290, 0.000030, 0.020000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -2, -1, 0}, 301.396940, 0.000300, 0.027000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -2, 0, 0}, 301.847890, 0.003410, 0.041000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -2, 0, 2}, 301.888790, 0.000060, 0.408000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -2, 1, 0}, 301.792700, 0.005100, 0.090000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -2, 2, 0}, 302.048410, 0.000220, 0.473000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -2, 3, 0}, 301.732120, 0.000010, 0.065000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -1, 0, 0}, 121.509260, 0.000110, 0.083000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -1, 0, 1}, 22.822020, 0.000030, 0.717000},
	{{0, 0, 3, -5, 0, 0, 0, 0, -1, 1, 0}, 121.836080, 0.000010, 0.814000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, -3, 0}, 299.695050, 0.000010, 0.025000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, -2, 0}, 300.210530, 0.000290, 0.038000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, -1, 0}, 300.170100, 0.004100, 0.077000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, 0, 0}, 300.803870, 0.020090, 2.928000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, 1, -2}, 118.466300, 0.000180, 0.075000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, 1, 0}, 300.068520, 0.003740, 0.074000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, 2, -2}, 118.126510, 0.000370, 123.995000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, 2, 0}, 299.954490, 0.000240, 0.037000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, 3, -2}, 115.269600, 0.000010, 0.075000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 0, 3, 0}, 299.551870, 0.000010, 0.025000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 1, -1, 0}, 121.182570, 0.000010, 1.833000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 1, -1, 1}, 203.176170, 0.000030, 0.078000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 1, 0, -1}, 23.012260, 0.000130, 1.404000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 1, 0, 0}, 303.321130, 0.000010, 0.079000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 1, 0, 1}, 203.021770, 0.000020, 0.038000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 2, -2, 0}, 301.957980, 0.000110, 0.698000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 2, -1, 0}, 302.071600, 0.003410, 0.085000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 2, 0, -2}, 301.242540, 0.000050, 0.566000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 2, 0, 0}, 301.136690, 0.001960, 0.040000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 2, 1, -2}, 298.510550, 0.000030, 0.087000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 2, 1, 0}, 301.183860, 0.000200, 0.026000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 2, 2, 0}, 301.382560, 0.000020, 0.019000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 4, -2, 0}, 301.696470, 0.000040, 0.043000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 4, -1, 0}, 301.427430, 0.000070, 0.027000},
	{{0, 0, 3, -5, 0, 0, 0, 0, 4, 0, 0}, 301.625770, 0.000040, 0.020000},
	{{0, 0, 3, -5, 2, 0, 0, 0, 0, 0, 0}, 0.406370, 0.000020, 1.960000},
	{{0, 0, 3, -5, 2, 0, 0, 0, 2, 0, 0}, 0.985150, 0.000010, 0.040000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -4, -1, 0}, 330.161110, 0.000030, 0.016000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -4, 0, 0}, 329.009700, 0.000130, 0.021000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -4, 1, 0}, 327.463650, 0.000290, 0.028000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -4, 2, 0}, 328.469940, 0.000180, 0.045000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -3, 1, 0}, 150.321390, 0.000040, 0.044000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, -2, 0}, 328.802240, 0.000090, 0.020000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, -1, 0}, 327.554570, 0.000970, 0.027000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, -1, 2}, 148.614910, 0.000010, 0.097000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, 0, -2}, 148.746510, 0.000030, 0.020000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, 0, 0}, 328.291860, 0.010990, 0.042000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, 0, 2}, 328.453120, 0.000150, 0.335000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, 1, -2}, 148.739090, 0.000030, 0.027000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, 1, 0}, 328.675740, 0.014310, 0.094000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, 2, 0}, 329.422850, 0.000630, 0.378000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -2, 3, 0}, 328.170780, 0.000040, 0.063000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -1, -1, 1}, 234.815450, 0.000050, 0.088000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -1, 0, -1}, 54.526280, 0.000010, 0.040000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -1, 0, 0}, 151.032200, 0.002270, 0.087000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -1, 0, 1}, 54.517510, 0.000060, 0.519000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -1, 1, 0}, 150.817310, 0.000240, 0.568000},
	{{0, 0, 3, -4, 0, 0, 0, 0, -1, 2, 0}, 151.113380, 0.000020, 0.067000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, -3, 0}, 330.060330, 0.000040, 0.026000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, -2, 0}, 329.662500, 0.000660, 0.039000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, -1, -2}, 149.910720, 0.000020, 0.025000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, -1, 0}, 329.613830, 0.009940, 0.081000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, -1, 2}, 329.935970, 0.000010, 0.069000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, 0, -2}, 149.849070, 0.000030, 0.039000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, 0, 0}, 330.087140, 0.046670, 1.145000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, 0, 2}, 149.734350, 0.000030, 0.036000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, 1, 0}, 330.630190, 0.007890, 0.071000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, 1, 2}, 149.910720, 0.000020, 0.024000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, 2, 0}, 330.980900, 0.000550, 0.037000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 0, 3, 0}, 331.042820, 0.000040, 0.025000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, -3, 0}, 331.554750, 0.000040, 0.038000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, -2, 0}, 331.556490, 0.000610, 0.075000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, -1, -2}, 151.650980, 0.000010, 0.037000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, -1, -1}, 54.795100, 0.000120, 0.074000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, -1, 0}, 331.253590, 0.014250, 72.752000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, -1, 1}, 54.813580, 0.000220, 0.075000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, -1, 2}, 151.650980, 0.000010, 0.037000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, 0, -2}, 331.235610, 0.000020, 0.074000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, 0, -1}, 54.848770, 0.000950, 5.540000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, 0, 0}, 331.232930, 0.011260, 0.076000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, 0, 1}, 234.839830, 0.000180, 0.038000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, 0, 2}, 151.235530, 0.000020, 0.025000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, 1, -1}, 54.795100, 0.000120, 0.076000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, 1, 0}, 331.236870, 0.000760, 0.038000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, 1, 1}, 234.838110, 0.000020, 0.025000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 1, 2, 0}, 331.235620, 0.000050, 0.025000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 2, -2, 0}, 315.206990, 0.000040, 1.110000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 2, -1, -2}, 151.340180, 0.000010, 0.069000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 2, -1, 0}, 330.113360, 0.004750, 0.081000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 2, 0, -2}, 331.233770, 0.000130, 0.810000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 2, 0, 0}, 331.656670, 0.002600, 0.039000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 2, 1, -2}, 151.340180, 0.000010, 0.083000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 2, 1, 0}, 332.142630, 0.000290, 0.026000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 2, 2, 0}, 330.328840, 0.000030, 0.019000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 3, -2, 0}, 331.059310, 0.000190, 0.087000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 3, -1, 0}, 331.237450, 0.000300, 0.040000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 3, 0, 0}, 331.237230, 0.000120, 0.026000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 4, -2, 0}, 322.417230, 0.000070, 0.042000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 4, -1, 0}, 330.589130, 0.000080, 0.027000},
	{{0, 0, 3, -4, 0, 0, 0, 0, 4, 0, 0}, 335.378030, 0.000030, 0.020000},
	{{0, 0, 3, -4, 1, 0, 0, 0, -2, 0, 0}, 158.531030, 0.000010, 0.042000},
	{{0, 0, 3, -4, 1, 0, 0, 0, 0, 0, 0}, 176.707240, 0.000060, 1.044000},
	{{0, 0, 3, -4, 2, 0, 0, 0, -2, 0, 0}, 236.725690, 0.000030, 0.042000},
	{{0, 0, 3, -4, 2, 0, 0, 0, -2, 1, 0}, 236.725690, 0.000040, 0.096000},
	{{0, 0, 3, -4, 2, 0, 0, 0, 0, -1, 0}, 232.893250, 0.000010, 0.082000},
	{{0, 0, 3, -4, 2, 0, 0, 0, 0, 0, 0}, 222.572210, 0.000080, 0.960000},
	{{0, 0, 3, -4, 2, 0, 0, 0, 0, 1, 0}, 232.893250, 0.000010, 0.070000},
	{{0, 0, 3, -4, 2, 0, 0, 0, 2, 0, 0}, 29.501650, 0.000010, 0.039000},
	{{0, 0, 3, -3, -3, 0, 0, 0, 0, 0, 0}, 307.068080, 0.000010, 0.868000},
	{{0, 0, 3, -3, -2, 0, 0, 0, -2, 0, 0}, 120.842550, 0.000010, 0.043000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -4, 0, 0}, 191.310890, 0.000050, 0.021000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -4, 1, 0}, 190.417200, 0.000100, 0.029000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -4, 2, 0}, 186.919650, 0.000050, 0.046000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -2, -2, 0}, 187.444620, 0.000040, 0.020000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -2, -1, 0}, 188.227240, 0.000320, 0.027000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -2, 0, 0}, 187.759650, 0.004050, 0.043000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -2, 0, 2}, 188.061880, 0.000030, 0.285000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -2, 1, 0}, 187.344440, 0.004450, 0.099000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -2, 2, 0}, 187.871780, 0.000200, 0.315000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -1, 0, 0}, 5.857550, 0.000070, 0.091000},
	{{0, 0, 3, -3, 0, 0, 0, 0, -1, 2, 0}, 1.053700, 0.000010, 0.064000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 0, -3, 0}, 182.260090, 0.000010, 0.026000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 0, -2, 0}, 183.051210, 0.000320, 0.040000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 0, -1, 0}, 183.149130, 0.004550, 0.084000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 0, 0, 0}, 182.646150, 0.014540, 0.712000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 0, 1, 0}, 182.417040, 0.002280, 0.068000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 0, 2, 0}, 182.490600, 0.000150, 0.036000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 1, -2, 0}, 2.657120, 0.000020, 0.079000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 1, -1, -1}, 279.364980, 0.000010, 0.077000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 1, -1, 0}, 3.872040, 0.000010, 1.931000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 1, 0, -1}, 277.642180, 0.000140, 2.848000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 1, 0, 0}, 8.374930, 0.000040, 0.073000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 1, 1, -1}, 279.364980, 0.000010, 0.073000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 2, -3, 0}, 355.445610, 0.000030, 0.073000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 2, -2, 0}, 1.110290, 0.000720, 2.710000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 2, -1, 0}, 0.867500, 0.006390, 0.078000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 2, 0, -2}, 359.110520, 0.000020, 1.423000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 2, 0, 0}, 1.048730, 0.000620, 0.038000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 2, 1, 0}, 356.667350, 0.000040, 0.025000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 4, -2, 0}, 0.560600, 0.000080, 0.041000},
	{{0, 0, 3, -3, 0, 0, 0, 0, 4, -1, 0}, 0.572100, 0.000080, 0.027000},
	{{0, 0, 3, -3, 2, 0, 0, 0, 2, -3, 0}, 60.843290, 0.000010, 0.074000},
	{{0, 0, 3, -3, 2, 0, 0, 0, 2, -1, 0}, 60.843290, 0.000010, 0.077000},
	{{0, 0, 3, -2, -2, 0, 0, 0, 2, -2, 0}, 257.525700, 0.000030, 166.820000},
	{{0, 0, 3, -2, -2, 0, 0, 0, 2, -1, 0}, 200.057960, 0.000020, 0.075000},
	{{0, 0, 3, -2, -1, 0, 0, 0, 2, -2, 0}, 11.991090, 0.000020, 12.770000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -4, -1, 0}, 75.937810, 0.000030, 0.016000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -4, 0, 0}, 257.108520, 0.000110, 0.021000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -4, 1, 0}, 256.420890, 0.000180, 0.029000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -4, 2, 0}, 256.528950, 0.000090, 0.048000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -3, 0, 0}, 256.483340, 0.000010, 0.028000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -2, -2, 0}, 256.864230, 0.000030, 0.020000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -2, -1, 0}, 256.611740, 0.000390, 0.028000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -2, 0, 0}, 256.491560, 0.004590, 0.044000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -2, 0, 2}, 256.844390, 0.000040, 0.247000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -2, 1, 0}, 256.926200, 0.004230, 0.105000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -2, 2, 0}, 256.907420, 0.000190, 0.269000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -1, 0, 0}, 257.315210, 0.000080, 0.096000},
	{{0, 0, 3, -2, 0, 0, 0, 0, -1, 2, 0}, 257.265730, 0.000010, 0.062000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 0, -3, 0}, 77.255930, 0.000010, 0.026000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 0, -2, 0}, 256.078670, 0.000120, 0.041000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 0, -1, 0}, 255.821510, 0.001380, 0.088000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 0, 0, -1}, 310.844690, 0.000030, 0.087000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 0, 0, 0}, 254.395930, 0.004590, 0.516000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 0, 1, 0}, 256.023760, 0.000740, 0.066000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 0, 2, 0}, 256.326330, 0.000040, 0.035000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 1, 0, -1}, 307.662750, 0.000090, 1.133000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 1, 0, 0}, 257.232860, 0.000070, 0.070000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 2, -3, 0}, 237.151080, 0.000010, 0.076000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 2, -2, 0}, 255.581560, 0.000140, 6.150000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 2, -1, -1}, 130.336010, 0.000640, 241.391000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 2, -1, 0}, 259.025320, 0.001960, 0.075000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 2, 0, -1}, 130.758420, 0.000210, 0.075000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 2, 0, 0}, 259.569420, 0.000170, 0.037000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 4, -2, 0}, 259.112300, 0.000030, 0.040000},
	{{0, 0, 3, -2, 0, 0, 0, 0, 4, -1, 0}, 258.991520, 0.000020, 0.026000},
	{{0, 0, 3, -2, 2, 0, 0, 0, 2, 0, -2}, 340.271420, 0.000010, 436.527000},
	{{0, 0, 3, -1, -2, 0, 0, 0, 0, 0, 0}, 180.160390, 0.000010, 0.435000},
	{{0, 0, 3, -1, -2, 0, 0, 0, 2, -3, 0}, 1.514090, 0.000010, 0.079000},
	{{0, 0, 3, -1, -2, 0, 0, 0, 2, -1, 0}, 181.514090, 0.000010, 0.073000},
	{{0, 0, 3, -1, 0, 0, 0, 0, -2, 0, 0}, 159.023830, 0.000040, 0.045000},
	{{0, 0, 3, -1, 0, 0, 0, 0, -2, 1, 0}, 158.965420, 0.000020, 0.111000},
	{{0, 0, 3, -1, 0, 0, 0, 0, 0, 0, 0}, 221.082420, 0.000070, 0.405000},
	{{0, 0, 3, -1, 0, 0, 0, 0, 1, 0, -1}, 339.045930, 0.000020, 0.707000},
	{{0, 0, 3, -1, 0, 0, 0, 0, 2, -1, 0}, 233.530700, 0.000030, 0.072000},
	{{0, 0, 3, 0, -6, 0, 0, 0, -2, 0, 0}, 204.451370, 0.000050, 0.045000},
	{{0, 0, 3, 0, -6, 0, 0, 0, -2, 1, 0}, 204.751110, 0.000020, 0.111000},
	{{0, 0, 3, 0, -6, 0, 0, 0, 0, -1, 0}, 32.370480, 0.000020, 0.093000},
	{{0, 0, 3, 0, -6, 0, 0, 0, 0, 0, 0}, 209.152550, 0.000070, 0.401000},
	{{0, 0, 3, 0, -6, 0, 0, 0, 0, 1, 0}, 208.684070, 0.000010, 0.063000},
	{{0, 0, 3, 0, -6, 0, 0, 0, 2, -1, 0}, 210.639430, 0.000130, 0.072000},
	{{0, 0, 3, 0, -6, 0, 0, 0, 2, 0, 0}, 32.048390, 0.000030, 0.037000},
	{{0, 0, 3, 0, -6, 2, 0, 0, 0, 0, 0}, 185.337870, 0.000010, 0.390000},
	{{0, 0, 3, 0, -6, 2, 0, 0, 2, -3, 0}, 10.494860, 0.000010, 0.080000},
	{{0, 0, 3, 0, -6, 2, 0, 0, 2, -1, 0}, 190.494860, 0.000010, 0.071000},
	{{0, 0, 3, 0, -6, 5, 0, 0, -2, 0, 0}, 32.006370, 0.000020, 0.045000},
	{{0, 0, 3, 0, -6, 5, 0, 0, 0, 0, 0}, 37.131000, 0.000030, 0.375000},
	{{0, 0, 3, 0, -6, 5, 0, 0, 2, -3, 0}, 244.175850, 0.000010, 0.081000},
	{{0, 0, 3, 0, -6, 5, 0, 0, 2, -2, 0}, 69.032900, 0.000020, 1.124000},
	{{0, 0, 3, 0, -6, 5, 0, 0, 2, -1, 0}, 46.736410, 0.000050, 0.071000},
	{{0, 0, 3, 0, -5, 0, 0, 0, -2, -1, 0}, 194.015810, 0.000040, 0.028000},
	{{0, 0, 3, 0, -5, 0, 0, 0, -2, 0, 0}, 193.928810, 0.000260, 0.045000},
	{{0, 0, 3, 0, -5, 0, 0, 0, -2, 1, 0}, 193.838590, 0.000230, 0.112000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 0, -1, 0}, 40.996940, 0.000020, 0.094000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 0, 0, 0}, 198.096690, 0.000440, 0.388000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 0, 1, 0}, 198.638180, 0.000090, 0.063000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 0, 2, 0}, 198.929860, 0.000010, 0.034000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 1, 0, -1}, 290.761280, 0.000020, 0.656000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 2, -2, 0}, 200.594540, 0.000030, 1.243000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 2, -1, 0}, 199.525180, 0.001030, 0.071000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 2, 0, -2}, 200.602440, 0.000040, 2.123000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 2, 0, 0}, 21.566550, 0.000120, 0.037000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 2, 1, 0}, 20.642370, 0.000030, 0.025000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 4, -2, 0}, 199.397220, 0.000010, 0.039000},
	{{0, 0, 3, 0, -5, 0, 0, 0, 4, 0, 0}, 200.601300, 0.000020, 0.019000},
	{{0, 0, 3, 0, -5, 2, 0, 0, -2, 0, 0}, 166.257680, 0.000020, 0.045000},
	{{0, 0, 3, 0, -5, 2, 0, 0, 0, 0, 0}, 174.148880, 0.000030, 0.378000},
	{{0, 0, 3, 0, -5, 2, 0, 0, 2, -3, 0}, 1.526600, 0.000010, 0.081000},
	{{0, 0, 3, 0, -5, 2, 0, 0, 2, -2, 0}, 181.251750, 0.000010, 1.146000},
	{{0, 0, 3, 0, -5, 2, 0, 0, 2, -1, 0}, 181.196080, 0.000050, 0.071000},
	{{0, 0, 3, 0, -5, 3, 0, 0, 2, -3, 0}, 59.699350, 0.000010, 0.081000},
	{{0, 0, 3, 0, -5, 3, 0, 0, 2, -2, 0}, 239.699350, 0.000010, 1.103000},
	{{0, 0, 3, 0, -5, 3, 0, 0, 2, -1, 0}, 239.699350, 0.000010, 0.071000},
	{{0, 0, 3, 0, -5, 3, 0, 0, 2, 0, 0}, 59.699350, 0.000010, 0.036000},
	{{0, 0, 3, 0, -5, 5, 0, 0, -2, 0, 0}, 343.079280, 0.000080, 0.045000},
	{{0, 0, 3, 0, -5, 5, 0, 0, -2, 1, 0}, 343.207850, 0.000060, 0.115000},
	{{0, 0, 3, 0, -5, 5, 0, 0, 0, -1, 0}, 261.118230, 0.000020, 0.095000},
	{{0, 0, 3, 0, -5, 5, 0, 0, 0, 0, 0}, 347.118760, 0.000110, 0.364000},
	{{0, 0, 3, 0, -5, 5, 0, 0, 0, 1, 0}, 342.152830, 0.000020, 0.062000},
	{{0, 0, 3, 0, -5, 5, 0, 0, 0, 2, 0}, 348.351450, 0.000010, 0.034000},
	{{0, 0, 3, 0, -5, 5, 0, 0, 2, -3, 0}, 338.746790, 0.000010, 0.081000},
	{{0, 0, 3, 0, -5, 5, 0, 0, 2, -1, 0}, 345.494120, 0.000200, 0.070000},
	{{0, 0, 3, 0, -5, 5, 0, 0, 2, 0, 0}, 168.291840, 0.000060, 0.036000},
	{{0, 0, 3, 0, -5, 5, 0, 0, 2, 2, 0}, 168.474860, 0.000010, 0.019000},
	{{0, 0, 3, 0, -4, 0, 0, 0, -4, 0, 0}, 182.491770, 0.000030, 0.021000},
	{{0, 0, 3, 0, -4, 0, 0, 0, -4, 1, 0}, 182.546910, 0.000030, 0.030000},
	{{0, 0, 3, 0, -4, 0, 0, 0, -4, 2, 0}, 182.488030, 0.000010, 0.049000},
	{{0, 0, 3, 0, -4, 0, 0, 0, -2, -1, 0}, 182.365020, 0.000140, 0.028000},
	{{0, 0, 3, 0, -4, 0, 0, 0, -2, 0, 0}, 181.651960, 0.001670, 0.045000},
	{{0, 0, 3, 0, -4, 0, 0, 0, -2, 0, 2}, 182.027300, 0.000010, 0.210000},
	{{0, 0, 3, 0, -4, 0, 0, 0, -2, 1, 0}, 182.403010, 0.001450, 0.113000},
	{{0, 0, 3, 0, -4, 0, 0, 0, -2, 2, 0}, 180.231320, 0.000060, 0.225000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 0, -2, 0}, 39.144500, 0.000020, 0.042000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 0, -1, 0}, 48.270730, 0.000300, 0.094000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 0, 0, 0}, 186.610240, 0.002410, 0.376000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 0, 1, 0}, 187.125480, 0.000670, 0.063000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 0, 2, 0}, 192.849730, 0.000030, 0.034000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 1, -3, 0}, 14.469890, 0.000050, 0.040000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 1, -2, 0}, 14.136250, 0.000180, 0.087000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 1, 0, -1}, 280.190940, 0.000060, 0.622000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 2, -3, 0}, 186.923750, 0.000030, 0.081000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 2, -2, 0}, 184.621050, 0.000170, 1.125000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 2, -1, 0}, 189.354570, 0.005930, 0.071000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 2, 0, -2}, 190.429920, 0.000230, 1.801000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 2, 0, 0}, 12.528380, 0.000880, 0.036000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 2, 1, 0}, 10.341070, 0.000090, 0.025000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 2, 2, 0}, 10.226360, 0.000020, 0.019000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 3, -4, 0}, 14.638230, 0.000140, 0.075000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 3, -3, 0}, 14.469890, 0.004520, 569.346000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 3, -2, 0}, 193.700450, 0.000380, 0.075000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 3, -1, 0}, 193.700350, 0.000030, 0.038000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 4, -2, 0}, 188.430300, 0.000090, 0.039000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 4, -1, 0}, 189.805230, 0.000050, 0.026000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 4, 0, 0}, 7.154680, 0.000030, 0.019000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 5, -4, 0}, 14.403800, 0.000070, 0.087000},
	{{0, 0, 3, 0, -4, 0, 0, 0, 5, -3, 0}, 14.469890, 0.000050, 0.040000},
	{{0, 0, 3, 0, -4, 1, 0, 0, 2, -3, 0}, 181.302960, 0.000010, 0.081000},
	{{0, 0, 3, 0, -4, 1, 0, 0, 2, -1, 0}, 1.302960, 0.000010, 0.071000},
	{{0, 0, 3, 0, -4, 2, 0, 0, -2, 0, 0}, 98.240510, 0.000020, 0.045000},
	{{0, 0, 3, 0, -4, 2, 0, 0, 0, -1, 0}, 79.446950, 0.000030, 0.095000},
	{{0, 0, 3, 0, -4, 2, 0, 0, 0, 0, 0}, 113.475900, 0.000020, 0.366000},
	{{0, 0, 3, 0, -4, 2, 0, 0, 2, -3, 0}, 17.869300, 0.000010, 0.081000},
	{{0, 0, 3, 0, -4, 2, 0, 0, 2, -1, 0}, 195.952180, 0.000010, 0.070000},
	{{0, 0, 3, 0, -4, 3, 0, 0, 0, -1, 0}, 134.210380, 0.000040, 0.095000},
	{{0, 0, 3, 0, -4, 3, 0, 0, 2, -1, 0}, 134.211650, 0.000020, 0.070000},
	{{0, 0, 3, 0, -4, 5, 0, 0, -2, 0, 0}, 246.120930, 0.000060, 0.046000},
	{{0, 0, 3, 0, -4, 5, 0, 0, -2, 1, 0}, 246.154730, 0.000040, 0.116000},
	{{0, 0, 3, 0, -4, 5, 0, 0, 0, -1, 0}, 68.277530, 0.000060, 0.096000},
	{{0, 0, 3, 0, -4, 5, 0, 0, 0, 0, 0}, 244.332390, 0.000030, 0.353000},
	{{0, 0, 3, 0, -4, 5, 0, 0, 2, -1, 0}, 68.276580, 0.000030, 0.070000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -4, 0, 0}, 150.032120, 0.000090, 0.021000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -4, 1, 0}, 162.447000, 0.000170, 0.030000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -4, 2, 0}, 165.848140, 0.000060, 0.049000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -3, 0, 0}, 160.870050, 0.000010, 0.029000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -2, -2, 0}, 153.408020, 0.000060, 0.021000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -2, -1, 0}, 153.287590, 0.000530, 0.028000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -2, 0, -2}, 346.342870, 0.000020, 0.020000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -2, 0, 0}, 163.624000, 0.006550, 0.045000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -2, 0, 2}, 166.310230, 0.000050, 0.206000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -2, 1, 0}, 165.262670, 0.005560, 0.114000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -2, 2, 0}, 165.278310, 0.000250, 0.221000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -2, 3, 0}, 165.249480, 0.000020, 0.056000},
	{{0, 0, 3, 0, -3, 0, 0, 0, -1, 0, 0}, 347.998960, 0.000090, 0.104000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 0, -2, 0}, 72.305940, 0.000100, 0.042000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 0, -1, 0}, 75.770000, 0.004420, 0.095000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 0, 0, -2}, 351.765350, 0.000010, 0.041000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 0, 0, 0}, 169.783990, 0.008510, 0.364000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 0, 1, -2}, 180.091460, 0.000010, 0.092000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 0, 1, 0}, 175.090600, 0.002460, 0.062000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 0, 2, 0}, 174.693190, 0.000170, 0.034000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 0, 3, 0}, 177.415190, 0.000010, 0.024000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 1, 0, 0}, 355.032840, 0.000080, 0.066000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, -3, 0}, 224.055740, 0.000080, 0.081000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, -2, 0}, 151.436560, 0.000690, 1.028000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, -1, -2}, 179.683940, 0.000030, 0.079000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, -1, 0}, 173.718020, 0.021660, 0.070000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, -1, 2}, 359.405610, 0.000050, 0.024000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, 0, -2}, 180.508490, 0.000830, 1.564000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, 0, 0}, 5.440540, 0.003680, 0.036000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, 0, 2}, 179.632260, 0.000020, 0.018000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, 1, -2}, 358.944070, 0.000020, 0.072000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, 1, 0}, 6.324090, 0.000320, 0.025000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 2, 2, 0}, 359.918980, 0.000070, 0.019000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 3, -4, 0}, 180.094710, 0.000010, 0.076000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 3, -2, 0}, 211.419650, 0.000050, 0.075000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 3, -1, 0}, 267.148840, 0.000020, 0.038000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 4, -3, 0}, 78.868590, 0.000030, 0.080000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 4, -2, 0}, 160.785390, 0.000320, 0.039000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 4, -1, -2}, 180.091460, 0.000010, 0.083000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 4, -1, 0}, 146.227670, 0.000110, 0.026000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 4, 0, 0}, 359.726840, 0.000040, 0.019000},
	{{0, 0, 3, 0, -3, 0, 0, 0, 4, 2, 0}, 0.035160, 0.000010, 0.013000},
	{{0, 0, 3, 0, -3, 1, 0, 0, 0, -1, 0}, 78.501210, 0.000050, 0.095000},
	{{0, 0, 3, 0, -3, 1, 0, 0, 2, -1, 0}, 78.500730, 0.000030, 0.070000},
	{{0, 0, 3, 0, -2, -5, 0, 0, -2, 0, 0}, 190.404470, 0.000020, 0.045000},
	{{0, 0, 3, 0, -2, -5, 0, 0, 0, 0, 0}, 194.936150, 0.000030, 0.376000},
	{{0, 0, 3, 0, -2, -5, 0, 0, 2, -3, 0}, 21.127470, 0.000020, 0.081000},
	{{0, 0, 3, 0, -2, -5, 0, 0, 2, -2, 0}, 199.605060, 0.000010, 1.127000},
	{{0, 0, 3, 0, -2, -5, 0, 0, 2, -1, 0}, 199.103790, 0.000050, 0.071000},
	{{0, 0, 3, 0, -2, -5, 0, 0, 3, -3, 0}, 18.091920, 0.000180, 1601.744000},
	{{0, 0, 3, 0, -2, -2, 0, 0, 2, -3, 0}, 172.545150, 0.000010, 0.082000},
	{{0, 0, 3, 0, -2, -2, 0, 0, 2, -1, 0}, 352.545150, 0.000010, 0.070000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -4, 0, 0}, 75.883880, 0.000160, 0.021000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -4, 1, 0}, 76.308850, 0.000330, 0.030000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -4, 2, 0}, 75.607610, 0.000100, 0.050000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -3, 0, 0}, 76.287820, 0.000020, 0.029000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -3, 3, 0}, 256.263870, 0.000010, 0.182000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -2, -3, 0}, 257.332590, 0.000010, 0.016000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -2, -2, 0}, 75.016970, 0.000050, 0.021000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -2, -1, 0}, 75.256300, 0.000530, 0.028000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -2, 0, -2}, 255.468460, 0.000020, 0.021000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -2, 0, 0}, 75.716080, 0.008600, 0.046000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -2, 0, 2}, 75.712000, 0.000060, 0.202000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -2, 1, 0}, 75.667540, 0.007020, 0.116000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -2, 2, 0}, 76.743440, 0.000340, 0.217000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -2, 3, 0}, 77.144810, 0.000020, 0.056000},
	{{0, 0, 3, 0, -2, 0, 0, 0, -1, 0, 0}, 254.406820, 0.000060, 0.105000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 0, -3, 0}, 256.447240, 0.000080, 0.027000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 0, -2, 0}, 256.887940, 0.000760, 0.042000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 0, -1, -2}, 77.184900, 0.000020, 0.027000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 0, -1, 0}, 256.833210, 0.008130, 0.096000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 0, 0, 0}, 73.982850, 0.004870, 0.353000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 0, 1, 0}, 73.080450, 0.000950, 0.062000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 0, 2, 0}, 70.454780, 0.000060, 0.034000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 1, -2, -1}, 75.997540, 0.000020, 0.040000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 1, -1, -1}, 77.096000, 0.000120, 0.087000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 1, 0, -1}, 259.428670, 0.000060, 0.563000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 2, -3, 0}, 262.376710, 0.000030, 0.082000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 2, -2, 0}, 259.656680, 0.000600, 0.946000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 2, -1, 0}, 268.938470, 0.004520, 0.070000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 2, 0, -2}, 316.397430, 0.000040, 1.382000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 2, 0, 0}, 241.951990, 0.000630, 0.036000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 2, 1, 0}, 228.283280, 0.000060, 0.024000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 3, -3, -1}, 59.857980, 0.000010, 0.075000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 3, -2, -1}, 75.997540, 0.001610, 277.143000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 3, -2, 0}, 77.204800, 0.000020, 0.074000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 3, -1, -1}, 257.555590, 0.000420, 0.075000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 3, -1, 0}, 77.184900, 0.000070, 0.037000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 3, 0, -1}, 257.286110, 0.000030, 0.038000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 4, -3, 0}, 77.973720, 0.000010, 0.080000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 4, -2, 0}, 263.060440, 0.000130, 0.039000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 4, -1, 0}, 256.842350, 0.000160, 0.026000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 4, 0, 0}, 263.463360, 0.000030, 0.019000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 5, -3, -1}, 77.271660, 0.000030, 0.087000},
	{{0, 0, 3, 0, -2, 0, 0, 0, 5, -2, -1}, 75.997540, 0.000020, 0.040000},
	{{0, 0, 3, 0, -2, 5, 0, 0, 2, -3, 0}, 308.908940, 0.000010, 0.083000},
	{{0, 0, 3, 0, -2, 5, 0, 0, 2, -1, 0}, 308.908940, 0.000010, 0.069000},
	{{0, 0, 3, 0, -1, -5, 0, 0, -2, 0, 0}, 171.214170, 0.000070, 0.045000},
	{{0, 0, 3, 0, -1, -5, 0, 0, -2, 1, 0}, 173.435750, 0.000050, 0.114000},
	{{0, 0, 3, 0, -1, -5, 0, 0, 0, -1, 0}, 45.114520, 0.000050, 0.095000},
	{{0, 0, 3, 0, -1, -5, 0, 0, 0, 0, 0}, 183.032290, 0.000110, 0.364000},
	{{0, 0, 3, 0, -1, -5, 0, 0, 0, 2, 0}, 188.729790, 0.000010, 0.034000},
	{{0, 0, 3, 0, -1, -5, 0, 0, 2, -3, 0}, 188.230310, 0.000020, 0.081000},
	{{0, 0, 3, 0, -1, -5, 0, 0, 2, -1, 0}, 183.065850, 0.000180, 0.070000},
	{{0, 0, 3, 0, -1, -5, 0, 0, 2, 0, 0}, 8.390220, 0.000060, 0.036000},
	{{0, 0, 3, 0, -1, -5, 0, 0, 2, 2, 0}, 8.708260, 0.000010, 0.019000},
	{{0, 0, 3, 0, -1, 0, 0, 0, -4, 0, 0}, 166.865670, 0.000030, 0.021000},
	{{0, 0, 3, 0, -1, 0, 0, 0, -4, 1, 0}, 163.240950, 0.000040, 0.030000},
	{{0, 0, 3, 0, -1, 0, 0, 0, -2, -1, 0}, 164.967720, 0.000120, 0.029000},
	{{0, 0, 3, 0, -1, 0, 0, 0, -2, 0, 0}, 165.059690, 0.001050, 0.046000},
	{{0, 0, 3, 0, -1, 0, 0, 0, -2, 1, 0}, 165.500090, 0.000780, 0.117000},
	{{0, 0, 3, 0, -1, 0, 0, 0, -2, 2, 0}, 166.018480, 0.000040, 0.213000},
	{{0, 0, 3, 0, -1, 0, 0, 0, -1, 0, 0}, 154.836330, 0.000020, 0.106000},
	{{0, 0, 3, 0, -1, 0, 0, 0, 0, -2, 0}, 14.754380, 0.000010, 0.042000},
	{{0, 0, 3, 0, -1, 0, 0, 0, 0, -1, 0}, 143.968420, 0.000150, 0.097000},
	{{0, 0, 3, 0, -1, 0, 0, 0, 0, 0, 0}, 174.406710, 0.000350, 0.343000},
	{{0, 0, 3, 0, -1, 0, 0, 0, 0, 1, 0}, 182.255160, 0.000050, 0.062000},
	{{0, 0, 3, 0, -1, 0, 0, 0, 2, -1, 0}, 53.196640, 0.000030, 0.069000},
	{{0, 0, 3, 0, 0, -5, 0, 0, -2, 0, 0}, 74.805760, 0.000060, 0.046000},
	{{0, 0, 3, 0, 0, -5, 0, 0, -2, 1, 0}, 83.045260, 0.000050, 0.116000},
	{{0, 0, 3, 0, 0, -5, 0, 0, 0, -1, 0}, 264.728440, 0.000060, 0.096000},
	{{0, 0, 3, 0, 0, -5, 0, 0, 0, 0, 0}, 63.978610, 0.000040, 0.353000},
	{{0, 0, 3, 0, 0, -5, 0, 0, 2, -1, 0}, 304.684350, 0.000040, 0.070000},
	{{0, 0, 3, 0, 0, -5, 0, 0, 2, 0, 0}, 185.466610, 0.000020, 0.036000},
	{{0, 0, 3, 0, 0, -5, 0, 0, 3, -2, -1}, 171.663730, 0.000040, 403.851000},
	{{0, 0, 3, 0, 0, -4, 0, 0, -2, 0, 0}, 270.376570, 0.000060, 0.046000},
	{{0, 0, 3, 0, 0, -4, 0, 0, -2, 1, 0}, 270.778350, 0.000020, 0.116000},
	{{0, 0, 3, 0, 0, -4, 0, 0, 0, 0, 0}, 271.394880, 0.000050, 0.349000},
	{{0, 0, 3, 0, 0, -4, 0, 0, 0, 1, 0}, 269.971230, 0.000020, 0.062000},
	{{0, 0, 3, 0, 0, -4, 0, 0, 2, -1, 0}, 272.727680, 0.000120, 0.070000},
	{{0, 0, 3, 0, 0, -4, 0, 0, 2, 0, 0}, 92.727680, 0.000050, 0.036000},
	{{0, 0, 3, 0, 0, -3, 0, 0, -2, -1, 0}, 176.800070, 0.000020, 0.028000},
	{{0, 0, 3, 0, 0, -3, 0, 0, -2, 0, 0}, 176.770920, 0.000200, 0.046000},
	{{0, 0, 3, 0, 0, -3, 0, 0, -2, 1, 0}, 176.627090, 0.000160, 0.117000},
	{{0, 0, 3, 0, 0, -3, 0, 0, 0, -1, 0}, 352.958090, 0.000030, 0.097000},
	{{0, 0, 3, 0, 0, -3, 0, 0, 0, 0, 0}, 177.609600, 0.000210, 0.345000},
	{{0, 0, 3, 0, 0, -3, 0, 0, 0, 1, 0}, 178.072790, 0.000070, 0.062000},
	{{0, 0, 3, 0, 0, -3, 0, 0, 2, -2, 0}, 180.008730, 0.000020, 0.890000},
	{{0, 0, 3, 0, 0, -3, 0, 0, 2, -1, 0}, 180.444390, 0.000440, 0.070000},
	{{0, 0, 3, 0, 0, -3, 0, 0, 2, 0, -2}, 180.008730, 0.000010, 1.265000},
	{{0, 0, 3, 0, 0, -3, 0, 0, 2, 0, 0}, 0.005600, 0.000090, 0.036000},
	{{0, 0, 3, 0, 0, -3, 0, 0, 2, 1, 0}, 359.967410, 0.000030, 0.024000},
	{{0, 0, 3, 0, 0, -2, 0, 0, -4, 1, 0}, 76.952170, 0.000010, 0.030000},
	{{0, 0, 3, 0, 0, -2, 0, 0, -2, -1, 0}, 76.940750, 0.000050, 0.029000},
	{{0, 0, 3, 0, 0, -2, 0, 0, -2, 0, 0}, 76.947050, 0.000330, 0.046000},
	{{0, 0, 3, 0, 0, -2, 0, 0, -2, 1, 0}, 76.947570, 0.000260, 0.117000},
	{{0, 0, 3, 0, 0, -2, 0, 0, 0, -2, 0}, 257.105510, 0.000010, 0.042000},
	{{0, 0, 3, 0, 0, -2, 0, 0, 0, -1, 0}, 257.548780, 0.000050, 0.097000},
	{{0, 0, 3, 0, 0, -2, 0, 0, 0, 0, 0}, 76.652100, 0.000220, 0.341000},
	{{0, 0, 3, 0, 0, -2, 0, 0, 0, 1, 0}, 76.653160, 0.000030, 0.062000},
	{{0, 0, 3, 0, 0, -2, 0, 0, 2, -2, 0}, 257.119600, 0.000010, 0.864000},
	{{0, 0, 3, 0, 0, -2, 0, 0, 2, -1, 0}, 257.083690, 0.000070, 0.069000},
	{{0, 0, 3, 0, 0, -2, 0, 0, 2, 0, 0}, 81.300980, 0.000010, 0.036000},
	{{0, 0, 3, 0, 0, -1, 0, 0, -2, 0, 0}, 155.890280, 0.000100, 0.046000},
	{{0, 0, 3, 0, 0, -1, 0, 0, -2, 1, 0}, 155.897120, 0.000060, 0.117000},
	{{0, 0, 3, 0, 0, -1, 0, 0, 0, 0, 0}, 157.567470, 0.000020, 0.337000},
	{{0, 0, 3, 0, 0, 0, 0, 0, -2, 0, 0}, 268.745080, 0.000080, 0.046000},
	{{0, 0, 3, 0, 0, 0, 0, 0, -2, 1, 0}, 290.948610, 0.000040, 0.118000},
	{{0, 0, 3, 0, 0, 0, 0, 0, -1, 0, 1}, 120.820670, 0.000020, 0.247000},
	{{0, 0, 3, 0, 0, 0, 0, 0, 0, -1, 0}, 308.363350, 0.000020, 0.098000},
	{{0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0}, 292.649000, 0.000210, 0.333000},
	{{0, 0, 3, 0, 0, 0, 0, 0, 0, 1, 0}, 294.452730, 0.000030, 0.062000},
	{{0, 0, 3, 0, 0, 0, 0, 0, 1, 0, -1}, 304.133720, 0.000030, 0.514000},
	{{0, 0, 3, 0, 0, 0, 0, 0, 2, -3, 0}, 216.155440, 0.000030, 0.083000},
	{{0, 0, 3, 0, 0, 0, 0, 0, 2, -2, 0}, 39.098360, 0.000030, 0.816000},
	{{0, 0, 3, 0, 0, 0, 0, 0, 2, -1, 0}, 303.942700, 0.000040, 0.069000},
	{{0, 0, 3, 0, 0, 0, 0, 0, 2, 0, 0}, 78.126290, 0.000030, 0.036000},
	{{0, 0, 3, 0, 0, 0, 0, 0, 4, 0, 0}, 51.160610, 0.000020, 0.019000},
	{{0, 0, 3, 0, 1, 0, 0, 0, -2, 0, 0}, 38.973240, 0.000020, 0.046000},
	{{0, 0, 3, 0, 1, 0, 0, 0, 2, -3, 0}, 306.867120, 0.000010, 0.084000},
	{{0, 0, 3, 0, 1, 0, 0, 0, 2, -2, 0}, 126.867190, 0.000010, 0.763000},
	{{0, 0, 3, 0, 1, 0, 0, 0, 2, -1, 0}, 126.867120, 0.000010, 0.069000},
	{{0, 0, 3, 0, 1, 0, 0, 0, 2, 0, 0}, 305.385780, 0.000020, 0.036000},
	{{0, 0, 3, 0, 2, -5, 0, 0, -2, 0, 0}, 220.308970, 0.000030, 0.046000},
	{{0, 0, 3, 0, 2, -5, 0, 0, 0, 0, 0}, 219.032880, 0.000010, 0.333000},
	{{0, 0, 3, 0, 2, -5, 0, 0, 2, -3, 0}, 149.142210, 0.000010, 0.083000},
	{{0, 0, 3, 0, 2, -5, 0, 0, 2, -1, 0}, 149.142210, 0.000010, 0.069000},
	{{0, 0, 3, 0, 3, -2, 0, 0, 2, -3, 0}, 307.122430, 0.000010, 0.084000},
	{{0, 0, 3, 0, 3, -2, 0, 0, 2, -2, 0}, 127.122430, 0.000010, 0.709000},
	{{0, 0, 3, 0, 3, -2, 0, 0, 2, -1, 0}, 307.122430, 0.000010, 0.068000},
	{{0, 0, 3, 0, 3, -2, 0, 0, 2, 0, 0}, 127.122430, 0.000010, 0.036000},
	{{0, 0, 3, 0, 4, -5, 0, 0, 2, 0, 0}, 192.610990, 0.000010, 0.036000},
	{{0, 0, 3, 2, 0, 0, 0, 0, 2, -2, 0}, 168.971080, 0.000020, 0.437000},
	{{0, 0, 3, 2, 0, 0, 0, 0, 2, 0, 0}, 168.924700, 0.000030, 0.035000},
	{{0, 0, 3, 3, 0, 0, 0, 0, 2, -2, 0}, 184.688200, 0.000010, 0.354000},
	{{0, 0, 3, 3, 0, 0, 0, 0, 2, 0, 0}, 184.688200, 0.000010, 0.034000},
	{{0, 0, 4, -11, 0, 0, 0, 0, -2, 1, 0}, 332.227780, 0.000050, 0.075000},
	{{0, 0, 4, -11, 0, 0, 0, 0, -2, 3, 0}, 323.170010, 0.000010, 0.076000},
	{{0, 0, 4, -11, 0, 2, 0, 0, -2, 2, 0}, 356.986630, 0.000010, 151.221000},
	{{0, 0, 4, -11, 1, 0, 0, 0, -2, 1, 0}, 350.562730, 0.000020, 0.075000},
	{{0, 0, 4, -11, 1, 0, 0, 0, -2, 2, 0}, 218.874480, 0.000020, 102.094000},
	{{0, 0, 4, -10, 0, 0, 0, 0, 0, 0, 0}, 259.102510, 0.000010, 0.759000},
	{{0, 0, 4, -10, 0, 0, 0, 0, 2, 0, 0}, 323.073570, 0.000020, 0.043000},
	{{0, 0, 4, -10, 2, 0, 0, 0, -2, 0, 0}, 51.321510, 0.000020, 0.039000},
	{{0, 0, 4, -10, 2, 0, 0, 0, -2, 1, 0}, 51.508670, 0.000010, 0.079000},
	{{0, 0, 4, -10, 2, 0, 0, 0, -2, 2, 0}, 231.241070, 0.000010, 1.598000},
	{{0, 0, 4, -10, 2, 0, 0, 0, -2, 3, 0}, 51.508670, 0.000010, 0.072000},
	{{0, 0, 4, -9, 0, 0, 0, 0, -1, 0, 1}, 293.382400, 0.000020, 3.723000},
	{{0, 0, 4, -9, 0, 0, 0, 0, 0, -1, 0}, 255.111630, 0.000020, 0.071000},
	{{0, 0, 4, -9, 0, 0, 0, 0, 0, 0, 0}, 231.551520, 0.000160, 1.274000},
	{{0, 0, 4, -9, 0, 0, 0, 0, 0, 1, 0}, 271.086250, 0.000030, 0.080000},
	{{0, 0, 4, -9, 0, 0, 0, 0, 2, -1, 0}, 325.707120, 0.000060, 0.093000},
	{{0, 0, 4, -9, 0, 0, 0, 0, 2, 0, 0}, 313.113090, 0.000050, 0.042000},
	{{0, 0, 4, -8, 0, 0, 0, 0, -2, -1, 0}, 242.610290, 0.000030, 0.026000},
	{{0, 0, 4, -8, 0, 0, 0, 0, -2, 0, 0}, 243.311590, 0.000230, 0.040000},
	{{0, 0, 4, -8, 0, 0, 0, 0, -2, 1, 0}, 245.016500, 0.000320, 0.085000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 0, -2, 0}, 240.725820, 0.000050, 0.037000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 0, -1, 0}, 240.023700, 0.000690, 0.074000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 0, 0, 0}, 242.876290, 0.003730, 3.945000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 0, 1, 0}, 240.769220, 0.000740, 0.077000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 0, 2, 0}, 241.240330, 0.000060, 0.038000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 1, 0, -1}, 324.556700, 0.000010, 0.765000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 2, -2, 0}, 247.122240, 0.000020, 0.493000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 2, -1, 0}, 248.260640, 0.000530, 0.089000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 2, 0, 0}, 246.518990, 0.000380, 0.041000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 2, 1, 0}, 246.924350, 0.000040, 0.026000},
	{{0, 0, 4, -8, 0, 0, 0, 0, 4, -1, 0}, 246.946650, 0.000010, 0.028000},
	{{0, 0, 4, -8, 1, 0, 0, 0, 0, 0, 0}, 94.085140, 0.000010, 5.911000},
	{{0, 0, 4, -8, 1, 5, 0, 0, -4, 0, 0}, 99.173350, 0.000010, 0.020000},
	{{0, 0, 4, -8, 1, 5, 0, 0, -4, 1, 0}, 279.173350, 0.000030, 0.028000},
	{{0, 0, 4, -8, 1, 5, 0, 0, -4, 2, 0}, 279.173480, 0.000010, 0.044000},
	{{0, 0, 4, -8, 1, 5, 0, 0, -2, -1, 0}, 279.036660, 0.000070, 0.026000},
	{{0, 0, 4, -8, 1, 5, 0, 0, -2, 0, 0}, 279.046530, 0.000640, 0.040000},
	{{0, 0, 4, -8, 1, 5, 0, 0, -2, 0, 2}, 279.173350, 0.000020, 0.474000},
	{{0, 0, 4, -8, 1, 5, 0, 0, -2, 1, 0}, 279.609170, 0.000990, 0.087000},
	{{0, 0, 4, -8, 1, 5, 0, 0, -2, 2, 0}, 279.173560, 0.000040, 0.564000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 0, -2, 0}, 276.718150, 0.000080, 0.038000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 0, -1, 0}, 276.834490, 0.001250, 0.075000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 0, 0, 0}, 278.083710, 0.006500, 1750.265000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 0, 1, 0}, 276.833480, 0.001250, 0.075000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 0, 2, 0}, 276.717630, 0.000080, 0.038000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 2, -2, 0}, 279.170010, 0.000040, 0.564000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 2, -1, 0}, 279.605720, 0.000990, 0.087000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 2, 0, -2}, 279.170210, 0.000020, 0.475000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 2, 0, 0}, 279.043590, 0.000640, 0.040000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 2, 1, 0}, 279.033780, 0.000070, 0.026000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 4, -2, 0}, 279.170090, 0.000010, 0.044000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 4, -1, 0}, 279.170210, 0.000030, 0.028000},
	{{0, 0, 4, -8, 1, 5, 0, 0, 4, 0, 0}, 99.170210, 0.000010, 0.020000},
	{{0, 0, 4, -8, 2, 0, 0, 0, -2, 0, 0}, 301.454020, 0.000020, 0.040000},
	{{0, 0, 4, -8, 2, 0, 0, 0, -2, 1, 0}, 301.190910, 0.000010, 0.086000},
	{{0, 0, 4, -8, 2, 0, 0, 0, 0, -1, 0}, 301.514870, 0.000050, 0.075000},
	{{0, 0, 4, -8, 2, 0, 0, 0, 0, 0, 0}, 300.218140, 0.000180, 11.784000},
	{{0, 0, 4, -8, 2, 0, 0, 0, 0, 1, 0}, 301.514870, 0.000050, 0.076000},
	{{0, 0, 4, -8, 2, 0, 0, 0, 2, -1, 0}, 301.532490, 0.000010, 0.088000},
	{{0, 0, 4, -8, 2, 0, 0, 0, 2, 0, 0}, 301.644150, 0.000020, 0.041000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -6, 0, 0}, 286.779960, 0.000020, 0.013000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -6, 1, 0}, 286.780890, 0.000040, 0.016000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -6, 2, 0}, 286.782030, 0.000050, 0.021000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -6, 3, 0}, 286.782140, 0.000020, 0.029000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -4, -1, 0}, 286.545320, 0.000170, 0.016000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -4, 0, 0}, 286.688320, 0.001060, 0.020000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -4, 0, 2}, 286.780200, 0.000020, 0.044000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -4, 1, -2}, 106.781400, 0.000020, 0.016000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -4, 1, 0}, 286.877730, 0.002640, 0.028000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -4, 1, 2}, 286.779010, 0.000020, 0.107000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -4, 2, 0}, 287.110110, 0.001890, 0.044000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -4, 3, 0}, 287.695140, 0.000050, 0.103000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -3, -1, 0}, 286.780510, 0.000030, 0.020000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -3, 0, 0}, 286.776180, 0.000030, 0.027000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -3, 1, 0}, 106.942860, 0.000170, 0.042000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -3, 2, 0}, 107.298320, 0.000060, 0.094000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, -3, 0}, 286.210740, 0.000080, 0.016000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, -2, 0}, 286.200690, 0.000760, 0.020000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, -2, 2}, 106.780880, 0.000010, 0.041000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, -1, -2}, 106.633830, 0.000060, 0.015000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, -1, 0}, 286.409350, 0.008760, 0.026000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, -1, 2}, 106.376780, 0.000270, 0.090000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 0, -2}, 106.709150, 0.000240, 0.019000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 0, 0}, 286.689570, 0.090020, 0.040000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 0, 2}, 286.680010, 0.002100, 0.475000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 1, -2}, 107.078420, 0.000330, 0.026000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 1, 0}, 287.112160, 0.139600, 0.087000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 2, -2}, 106.783310, 0.000020, 0.040000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 2, 0}, 287.815150, 0.004840, 0.564000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 2, 2}, 106.781530, 0.000030, 0.035000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 3, 0}, 288.989690, 0.000220, 0.067000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -2, 4, 0}, 289.774640, 0.000010, 0.035000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -1, -1, 0}, 106.193020, 0.000220, 0.039000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -1, 0, 0}, 106.690760, 0.002370, 0.081000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -1, 1, 0}, 107.802520, 0.000210, 1.128000},
	{{0, 0, 4, -8, 3, 0, 0, 0, -1, 2, 0}, 109.473260, 0.000020, 0.071000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, -4, 0}, 284.824400, 0.000050, 0.019000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, -3, 0}, 284.937580, 0.000800, 0.025000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, -2, -2}, 105.027940, 0.000060, 0.019000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, -2, 0}, 284.989300, 0.011620, 0.038000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, -2, 2}, 104.677630, 0.000020, 3.003000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, -1, -2}, 105.240100, 0.000390, 0.025000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, -1, 0}, 284.987760, 0.170830, 0.075000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, -1, 2}, 284.451770, 0.000240, 0.074000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 0, -2}, 107.223980, 0.000340, 0.037000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 0, 0}, 285.987070, 0.901140, 1783.359000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 0, 2}, 107.226590, 0.000340, 0.037000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 1, -2}, 284.427560, 0.000230, 0.074000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 1, 0}, 284.987670, 0.170820, 0.075000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 1, 2}, 105.240740, 0.000390, 0.025000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 2, -2}, 104.678250, 0.000020, 2.993000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 2, 0}, 284.989810, 0.011620, 0.038000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 2, 2}, 105.028290, 0.000060, 0.019000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 3, 0}, 284.938030, 0.000800, 0.025000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 0, 4, 0}, 284.824790, 0.000050, 0.019000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 1, -2, 0}, 109.474860, 0.000020, 0.071000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 1, -1, 0}, 107.801200, 0.000220, 1.127000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 1, 0, 0}, 106.690930, 0.002370, 0.081000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 1, 1, 0}, 106.194180, 0.000220, 0.039000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, -4, 0}, 289.773370, 0.000010, 0.035000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, -3, 0}, 288.989830, 0.000220, 0.067000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, -2, -2}, 106.784160, 0.000030, 0.035000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, -2, 0}, 287.815380, 0.004850, 0.564000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, -2, 2}, 106.783540, 0.000020, 0.040000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, -1, 0}, 287.113880, 0.139710, 0.087000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, -1, 2}, 107.080170, 0.000330, 0.026000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, 0, -2}, 286.682610, 0.002100, 0.474000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, 0, 0}, 286.692690, 0.090090, 0.040000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, 0, 2}, 106.710650, 0.000240, 0.019000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, 1, -2}, 106.379290, 0.000270, 0.090000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, 1, 0}, 286.410410, 0.008750, 0.026000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, 1, 2}, 106.635540, 0.000060, 0.015000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, 2, -2}, 106.784060, 0.000010, 0.041000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, 2, 0}, 286.202100, 0.000760, 0.020000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 2, 3, 0}, 286.212960, 0.000080, 0.016000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 3, -2, 0}, 107.302150, 0.000060, 0.094000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 3, -1, 0}, 106.945180, 0.000170, 0.042000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 3, 0, 0}, 286.787770, 0.000030, 0.027000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 3, 1, 0}, 286.784460, 0.000030, 0.020000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 4, -3, 0}, 287.693010, 0.000050, 0.103000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 4, -2, 0}, 287.111860, 0.001890, 0.044000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 4, -1, -2}, 286.784820, 0.000020, 0.107000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 4, -1, 0}, 286.879760, 0.002640, 0.028000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 4, -1, 2}, 106.783910, 0.000020, 0.016000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 4, 0, -2}, 286.783910, 0.000020, 0.044000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 4, 0, 0}, 286.693010, 0.001070, 0.020000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 4, 1, 0}, 286.548180, 0.000170, 0.016000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 6, -3, 0}, 286.783620, 0.000020, 0.029000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 6, -2, 0}, 286.783740, 0.000050, 0.021000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 6, -1, 0}, 286.783820, 0.000040, 0.016000},
	{{0, 0, 4, -8, 3, 0, 0, 0, 6, 0, 0}, 286.783870, 0.000020, 0.013000},
	{{0, 0, 4, -8, 5, -5, 0, 0, -2, 0, 0}, 61.301250, 0.000070, 0.040000},
	{{0, 0, 4, -8, 5, -5, 0, 0, -2, 1, 0}, 61.301250, 0.000150, 0.087000},
	{{0, 0, 4, -8, 5, -5, 0, 0, 0, -1, 0}, 241.300370, 0.000020, 0.075000},
	{{0, 0, 4, -8, 5, -5, 0, 0, 0, 0, 0}, 241.300810, 0.000010, 590.730000},
	{{0, 0, 4, -8, 5, -5, 0, 0, 0, 1, 0}, 241.301250, 0.000020, 0.075000},
	{{0, 0, 4, -8, 5, -5, 0, 0, 2, -1, 0}, 61.300370, 0.000150, 0.087000},
	{{0, 0, 4, -8, 5, -5, 0, 0, 2, 0, 0}, 61.300370, 0.000070, 0.040000},
	{{0, 0, 4, -7, -2, 0, 0, 0, 0, 0, 0}, 33.696040, 0.000010, 9.124000},
	{{0, 0, 4, -7, -1, 0, 0, 0, 0, 0, 0}, 220.184630, 0.000010, 5.157000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -4, 0, 0}, 272.711250, 0.000030, 0.020000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -4, 1, 0}, 272.522200, 0.000020, 0.028000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -4, 2, 0}, 272.576060, 0.000030, 0.044000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -3, 2, -1}, 41.502240, 0.000010, 0.042000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -3, 3, -1}, 41.474250, 0.000010, 0.095000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -2, -2, 0}, 272.776720, 0.000020, 0.020000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -2, -1, 0}, 272.420720, 0.000150, 0.027000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -2, 0, 0}, 273.020890, 0.001700, 0.041000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -2, 0, 2}, 272.828990, 0.000040, 0.419000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -2, 1, 0}, 272.814220, 0.002610, 0.089000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -2, 2, 0}, 272.880770, 0.000100, 0.487000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -1, 0, 0}, 92.309860, 0.000030, 0.083000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -1, 0, 1}, 353.516570, 0.000010, 0.751000},
	{{0, 0, 4, -7, 0, 0, 0, 0, -1, 2, 0}, 92.626870, 0.000010, 0.069000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 0, -2, 0}, 270.563560, 0.000140, 0.038000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 0, -1, 0}, 270.601190, 0.002190, 0.077000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 0, 0, 0}, 271.446430, 0.010990, 3.594000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 0, 1, 0}, 270.408450, 0.002010, 0.074000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 0, 2, 0}, 270.320880, 0.000130, 0.037000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 0, 3, 0}, 92.532180, 0.000010, 0.025000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 1, -2, 0}, 272.169680, 0.000010, 0.072000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 1, 0, -1}, 353.414150, 0.000070, 1.289000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 1, 0, 0}, 92.169680, 0.000010, 0.079000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 2, -2, 0}, 272.046310, 0.000070, 0.669000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 2, -1, 0}, 272.772060, 0.001890, 0.085000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 2, 0, -2}, 272.169680, 0.000030, 0.547000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 2, 0, 0}, 272.189080, 0.001090, 0.040000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 2, 1, 0}, 271.989860, 0.000130, 0.026000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 4, -2, 0}, 272.340000, 0.000020, 0.043000},
	{{0, 0, 4, -7, 0, 0, 0, 0, 4, 0, 0}, 272.382720, 0.000020, 0.020000},
	{{0, 0, 4, -7, 1, 0, 0, 0, 0, 0, 0}, 318.225230, 0.000010, 2.759000},
	{{0, 0, 4, -7, 2, 0, 0, 0, 0, 0, 0}, 331.253580, 0.000020, 2.238000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -4, 0, 0}, 300.539170, 0.000040, 0.021000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -4, 1, 0}, 300.530060, 0.000080, 0.028000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -4, 2, 0}, 300.412700, 0.000050, 0.045000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -2, -2, 0}, 300.517290, 0.000030, 0.020000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -2, -1, 0}, 300.599920, 0.000280, 0.027000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -2, 0, 0}, 300.389340, 0.003320, 0.042000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -2, 0, 2}, 300.458230, 0.000040, 0.343000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -2, 1, 0}, 300.616240, 0.004390, 0.094000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -2, 2, 0}, 300.428640, 0.000200, 0.387000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -1, -1, 1}, 204.001940, 0.000040, 0.088000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -1, 0, 0}, 121.586080, 0.000210, 0.087000},
	{{0, 0, 4, -6, 0, 0, 0, 0, -1, 0, 1}, 23.520220, 0.000040, 0.537000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 0, -2, 0}, 300.900660, 0.000210, 0.039000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 0, -1, 0}, 300.197980, 0.002960, 0.080000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 0, 0, 0}, 301.217300, 0.014100, 1.235000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 0, 1, 0}, 301.393580, 0.002370, 0.071000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 0, 2, 0}, 301.280650, 0.000150, 0.037000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, -2, 0}, 299.731380, 0.000020, 0.075000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, -2, 1}, 204.011700, 0.000030, 11.156000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, -1, -1}, 23.990720, 0.000050, 0.074000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, -1, 0}, 301.284500, 0.000230, 12.968000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, -1, 1}, 24.001940, 0.000220, 0.075000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, 0, -1}, 24.021140, 0.000420, 4.101000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, 0, 0}, 302.220850, 0.000610, 0.076000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, 0, 1}, 204.014810, 0.000080, 0.038000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, 1, -1}, 23.990720, 0.000050, 0.077000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 1, 1, 0}, 302.241050, 0.000040, 0.038000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 2, -2, 0}, 282.460530, 0.000060, 1.037000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 2, -1, 0}, 296.095650, 0.002090, 0.081000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 2, 0, -2}, 302.109620, 0.000040, 0.771000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 2, 0, 0}, 301.061350, 0.000990, 0.039000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 2, 1, 0}, 301.785170, 0.000130, 0.026000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 4, -3, 0}, 74.818410, 0.000010, 0.095000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 4, -2, 0}, 284.262260, 0.000040, 0.042000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 4, -1, 0}, 276.318600, 0.000050, 0.027000},
	{{0, 0, 4, -6, 0, 0, 0, 0, 4, 0, 0}, 302.234720, 0.000020, 0.020000},
	{{0, 0, 4, -6, 1, 0, 0, 0, 0, 0, 0}, 150.720780, 0.000020, 1.118000},
	{{0, 0, 4, -6, 1, 0, 0, 0, 2, 0, 0}, 153.571360, 0.000010, 0.039000},
	{{0, 0, 4, -6, 2, 0, 0, 0, -2, 0, 0}, 281.535880, 0.000040, 0.042000},
	{{0, 0, 4, -6, 2, 0, 0, 0, -2, 1, 0}, 289.643520, 0.000040, 0.095000},
	{{0, 0, 4, -6, 2, 0, 0, 0, 0, -1, 0}, 328.674110, 0.000010, 0.081000},
	{{0, 0, 4, -6, 2, 0, 0, 0, 0, 0, 0}, 330.486980, 0.000100, 1.022000},
	{{0, 0, 4, -6, 2, 0, 0, 0, 0, 1, 0}, 328.674110, 0.000010, 0.070000},
	{{0, 0, 4, -6, 2, 0, 0, 0, 2, 0, 0}, 0.579420, 0.000020, 0.039000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -4, 0, 0}, 150.542800, 0.000030, 0.021000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -4, 1, 0}, 155.878210, 0.000070, 0.029000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -4, 2, 0}, 156.158470, 0.000040, 0.046000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -2, -2, 0}, 156.632470, 0.000020, 0.020000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -2, -1, 0}, 155.633100, 0.000210, 0.027000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -2, 0, 0}, 156.495040, 0.002530, 0.043000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -2, 0, 2}, 156.600790, 0.000030, 0.290000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -2, 1, 0}, 156.198950, 0.002820, 0.099000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -2, 2, 0}, 156.949850, 0.000120, 0.321000},
	{{0, 0, 4, -5, 0, 0, 0, 0, -1, 0, 0}, 336.161380, 0.000040, 0.091000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 0, -2, 0}, 153.103850, 0.000190, 0.040000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 0, -1, 0}, 153.166020, 0.002920, 0.084000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 0, 0, 0}, 152.760920, 0.009480, 0.745000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 0, 1, 0}, 152.582010, 0.001520, 0.069000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 0, 2, 0}, 152.644660, 0.000100, 0.036000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 1, -2, 0}, 336.944310, 0.000020, 0.078000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 1, -1, -1}, 236.077380, 0.000030, 0.077000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 1, -1, 1}, 56.451250, 0.000020, 0.072000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 1, 0, -1}, 236.087840, 0.000260, 3.475000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 1, 0, 0}, 336.944310, 0.000020, 0.073000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 1, 0, 1}, 56.047270, 0.000030, 0.037000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 1, 1, -1}, 236.077380, 0.000030, 0.074000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 2, -2, 0}, 331.818930, 0.000400, 2.313000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 2, -1, 0}, 331.335000, 0.003820, 0.078000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 2, 0, -2}, 332.477090, 0.000020, 1.306000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 2, 0, 0}, 333.254080, 0.000410, 0.038000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 2, 1, 0}, 330.012580, 0.000030, 0.025000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 4, -2, 0}, 331.204600, 0.000050, 0.041000},
	{{0, 0, 4, -5, 0, 0, 0, 0, 4, -1, 0}, 331.418340, 0.000050, 0.027000},
	{{0, 0, 4, -5, 2, 0, 0, 0, 2, 0, 0}, 29.280890, 0.000010, 0.038000},
	{{0, 0, 4, -4, -2, 0, 0, 0, 2, -2, 0}, 304.811260, 0.000010, 14.417000},
	{{0, 0, 4, -4, 0, -3, 0, 0, 2, -1, 0}, 43.451490, 0.000020, 0.075000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -4, 0, 0}, 213.107920, 0.000040, 0.021000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -4, 1, 0}, 211.514170, 0.000040, 0.029000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -4, 2, 0}, 205.210000, 0.000030, 0.047000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -2, -2, 0}, 205.250960, 0.000010, 0.020000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -2, -1, 0}, 216.574600, 0.000120, 0.028000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -2, 0, 0}, 206.725570, 0.001750, 0.044000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -2, 0, 2}, 206.271670, 0.000020, 0.251000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -2, 1, 0}, 205.671510, 0.001730, 0.104000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -2, 2, 0}, 207.278780, 0.000070, 0.274000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -1, -1, 0}, 228.316660, 0.000010, 0.042000},
	{{0, 0, 4, -4, 0, 0, 0, 0, -1, 0, 0}, 48.297210, 0.000090, 0.095000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 0, -3, 0}, 2.378780, 0.000020, 0.026000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 0, -2, 0}, 357.277060, 0.000220, 0.041000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 0, -1, 0}, 356.657260, 0.003030, 0.088000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 0, 0, 0}, 191.186990, 0.004040, 0.534000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 0, 1, 0}, 189.865000, 0.000680, 0.066000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 0, 2, 0}, 188.014140, 0.000050, 0.035000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 1, 0, -1}, 278.919830, 0.000070, 1.220000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 1, 0, 0}, 48.298580, 0.000050, 0.070000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 2, -3, 0}, 350.406440, 0.000040, 0.076000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 2, -2, 0}, 0.336680, 0.002140, 10.076000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 2, -1, -2}, 181.447510, 0.000060, 0.074000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 2, -1, 0}, 181.912760, 0.020080, 0.075000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 2, -1, 2}, 1.974490, 0.000040, 0.025000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 2, 0, -2}, 357.846900, 0.000070, 4.269000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 2, 0, 0}, 182.436990, 0.001040, 0.038000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 2, 1, -2}, 180.642810, 0.000020, 0.077000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 2, 1, 0}, 180.637160, 0.000030, 0.025000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 4, -3, 0}, 2.417530, 0.000040, 0.086000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 4, -2, 0}, 182.148080, 0.000240, 0.040000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 4, -1, 0}, 182.225440, 0.000160, 0.026000},
	{{0, 0, 4, -4, 0, 0, 0, 0, 4, 0, 0}, 181.820930, 0.000030, 0.019000},
	{{0, 0, 4, -4, 2, 0, 0, 0, 2, -1, 0}, 241.929320, 0.000020, 0.074000},
	{{0, 0, 4, -3, -2, 0, 0, 0, 0, 0, 0}, 151.592580, 0.000010, 0.447000},
	{{0, 0, 4, -3, 0, 0, 0, 0, -4, 0, 0}, 258.788290, 0.000020, 0.021000},
	{{0, 0, 4, -3, 0, 0, 0, 0, -4, 1, 0}, 76.833050, 0.000010, 0.030000},
	{{0, 0, 4, -3, 0, 0, 0, 0, -2, -1, 0}, 76.467450, 0.000040, 0.028000},
	{{0, 0, 4, -3, 0, 0, 0, 0, -2, 0, 0}, 76.970830, 0.000370, 0.045000},
	{{0, 0, 4, -3, 0, 0, 0, 0, -2, 1, 0}, 77.022550, 0.000340, 0.110000},
	{{0, 0, 4, -3, 0, 0, 0, 0, 0, -1, 0}, 86.806080, 0.000180, 0.092000},
	{{0, 0, 4, -3, 0, 0, 0, 0, 0, 0, 0}, 92.530300, 0.000250, 0.416000},
	{{0, 0, 4, -3, 0, 0, 0, 0, 0, 1, 0}, 70.817560, 0.000040, 0.064000},
	{{0, 0, 4, -3, 0, 0, 0, 0, 1, 0, -1}, 310.019020, 0.000060, 0.740000},
	{{0, 0, 4, -3, 0, 0, 0, 0, 2, -2, 0}, 73.162550, 0.000010, 1.585000},
	{{0, 0, 4, -3, 0, 0, 0, 0, 2, -1, 0}, 41.966870, 0.000260, 0.072000},
	{{0, 0, 4, -3, 0, 0, 0, 0, 2, 0, 0}, 195.290040, 0.000030, 0.037000},
	{{0, 0, 4, -2, 0, 0, 0, 0, -2, -1, 0}, 154.688540, 0.000030, 0.029000},
	{{0, 0, 4, -2, 0, 0, 0, 0, -2, 0, 0}, 154.628300, 0.000260, 0.046000},
	{{0, 0, 4, -2, 0, 0, 0, 0, -2, 1, 0}, 155.375390, 0.000220, 0.117000},
	{{0, 0, 4, -2, 0, 0, 0, 0, 0, -1, 0}, 154.776040, 0.000050, 0.097000},
	{{0, 0, 4, -2, 0, 0, 0, 0, 0, 0, 0}, 165.139100, 0.000100, 0.341000},
	{{0, 0, 4, -2, 0, 0, 0, 0, 0, 1, 0}, 159.508980, 0.000010, 0.062000},
	{{0, 0, 4, -2, 0, 0, 0, 0, 2, -3, 0}, 186.598200, 0.000010, 0.083000},
	{{0, 0, 4, -2, 0, 0, 0, 0, 2, -1, 0}, 186.598200, 0.000010, 0.069000},
	{{0, 0, 4, 0, -7, 0, 0, 0, 0, 0, 0}, 209.689240, 0.000010, 0.293000},
	{{0, 0, 4, 0, -7, 0, 0, 0, 2, -1, 0}, 211.318620, 0.000040, 0.067000},
	{{0, 0, 4, 0, -7, 0, 0, 0, 2, 0, 0}, 30.780330, 0.000020, 0.036000},
	{{0, 0, 4, 0, -7, 5, 0, 0, 2, -2, 0}, 17.576510, 0.000010, 0.554000},
	{{0, 0, 4, 0, -7, 5, 0, 0, 2, 0, 0}, 197.576510, 0.000010, 0.035000},
	{{0, 0, 4, 0, -6, 0, 0, 0, -2, 0, 0}, 195.359080, 0.000060, 0.047000},
	{{0, 0, 4, 0, -6, 0, 0, 0, -2, 1, 0}, 195.418810, 0.000030, 0.125000},
	{{0, 0, 4, 0, -6, 0, 0, 0, 0, -1, 0}, 22.901720, 0.000020, 0.102000},
	{{0, 0, 4, 0, -6, 0, 0, 0, 0, 0, 0}, 198.496980, 0.000050, 0.286000},
	{{0, 0, 4, 0, -6, 0, 0, 0, 0, 1, 0}, 198.458220, 0.000020, 0.060000},
	{{0, 0, 4, 0, -6, 0, 0, 0, 2, -2, 0}, 200.624880, 0.000010, 0.581000},
	{{0, 0, 4, 0, -6, 0, 0, 0, 2, -1, 0}, 200.626790, 0.000210, 0.067000},
	{{0, 0, 4, 0, -6, 0, 0, 0, 2, 0, 0}, 20.512360, 0.000090, 0.035000},
	{{0, 0, 4, 0, -6, 0, 0, 0, 2, 1, 0}, 20.595410, 0.000020, 0.024000},
	{{0, 0, 4, 0, -6, 2, 0, 0, 2, -2, 0}, 180.866600, 0.000010, 0.559000},
	{{0, 0, 4, 0, -6, 2, 0, 0, 2, 0, 0}, 0.866600, 0.000010, 0.035000},
	{{0, 0, 4, 0, -6, 5, 0, 0, -2, 0, 0}, 341.141430, 0.000020, 0.047000},
	{{0, 0, 4, 0, -6, 5, 0, 0, 0, 0, 0}, 345.152460, 0.000010, 0.273000},
	{{0, 0, 4, 0, -6, 5, 0, 0, 2, -2, 0}, 348.386640, 0.000020, 0.529000},
	{{0, 0, 4, 0, -6, 5, 0, 0, 2, -1, 0}, 348.189250, 0.000040, 0.066000},
	{{0, 0, 4, 0, -6, 5, 0, 0, 2, 0, 0}, 168.532150, 0.000010, 0.035000},
	{{0, 0, 4, 0, -5, 0, 0, 0, -2, -1, 0}, 183.443290, 0.000030, 0.029000},
	{{0, 0, 4, 0, -5, 0, 0, 0, -2, 0, 0}, 183.374330, 0.000240, 0.047000},
	{{0, 0, 4, 0, -5, 0, 0, 0, -2, 1, 0}, 182.932560, 0.000190, 0.127000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 0, -1, 0}, 15.462800, 0.000060, 0.103000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 0, 0, 0}, 186.639340, 0.000190, 0.279000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 0, 1, 0}, 190.633900, 0.000100, 0.059000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 0, 2, 0}, 189.968260, 0.000010, 0.033000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 1, 0, -1}, 279.548520, 0.000010, 0.396000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 2, -2, 0}, 190.308820, 0.000030, 0.554000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 2, -1, 0}, 190.280790, 0.001030, 0.066000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 2, 0, -2}, 190.333790, 0.000030, 0.680000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 2, 0, 0}, 10.322750, 0.000380, 0.035000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 2, 1, 0}, 10.302750, 0.000070, 0.024000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 4, -2, 0}, 190.194870, 0.000020, 0.038000},
	{{0, 0, 4, 0, -5, 0, 0, 0, 4, -1, 0}, 10.302390, 0.000010, 0.025000},
	{{0, 0, 4, 0, -4, 0, 0, 0, -4, 1, 0}, 166.597430, 0.000030, 0.031000},
	{{0, 0, 4, 0, -4, 0, 0, 0, -2, -1, 0}, 166.826140, 0.000080, 0.029000},
	{{0, 0, 4, 0, -4, 0, 0, 0, -2, 0, 0}, 164.959940, 0.000760, 0.047000},
	{{0, 0, 4, 0, -4, 0, 0, 0, -2, 1, 0}, 166.037430, 0.000540, 0.128000},
	{{0, 0, 4, 0, -4, 0, 0, 0, -2, 2, 0}, 164.980700, 0.000020, 0.184000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 0, -2, 0}, 6.579900, 0.000010, 0.044000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 0, -1, 0}, 355.896550, 0.000200, 0.104000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 0, 0, 0}, 170.234240, 0.000440, 0.273000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 0, 1, 0}, 175.746200, 0.000340, 0.059000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 0, 2, 0}, 178.883340, 0.000020, 0.033000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 2, -2, 0}, 180.123760, 0.000100, 0.529000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 2, -1, 0}, 179.869500, 0.003080, 0.066000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 2, 0, -2}, 179.795190, 0.000100, 0.643000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 2, 0, 0}, 0.662100, 0.001250, 0.035000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 2, 1, 0}, 359.635860, 0.000130, 0.024000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 2, 2, 0}, 0.698120, 0.000010, 0.018000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 4, -2, 0}, 179.206800, 0.000050, 0.038000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 4, 0, 0}, 359.382660, 0.000030, 0.019000},
	{{0, 0, 4, 0, -4, 0, 0, 0, 4, 1, 0}, 179.795190, 0.000010, 0.015000},
	{{0, 0, 4, 0, -3, -5, 0, 0, 2, -2, 0}, 198.202500, 0.000010, 0.555000},
	{{0, 0, 4, 0, -3, -5, 0, 0, 2, 0, 0}, 18.202500, 0.000010, 0.035000},
	{{0, 0, 4, 0, -3, 0, 0, 0, -4, 0, 0}, 242.830850, 0.000020, 0.022000},
	{{0, 0, 4, 0, -3, 0, 0, 0, -4, 1, 0}, 63.536770, 0.000030, 0.031000},
	{{0, 0, 4, 0, -3, 0, 0, 0, -2, -1, 0}, 64.179130, 0.000070, 0.029000},
	{{0, 0, 4, 0, -3, 0, 0, 0, -2, 0, 0}, 63.970470, 0.000690, 0.048000},
	{{0, 0, 4, 0, -3, 0, 0, 0, -2, 1, 0}, 64.008150, 0.000500, 0.129000},
	{{0, 0, 4, 0, -3, 0, 0, 0, -2, 2, 0}, 64.097480, 0.000030, 0.181000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 0, -3, 0}, 220.114680, 0.000010, 0.028000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 0, -2, 0}, 265.045030, 0.000020, 0.044000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 0, -1, 0}, 339.565300, 0.000150, 0.105000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 0, 0, 0}, 64.175140, 0.000250, 0.267000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 0, 1, 0}, 63.124640, 0.000060, 0.059000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 1, 0, -1}, 259.109160, 0.000010, 0.371000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 2, -2, 0}, 5.107370, 0.000010, 0.507000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 2, -1, 0}, 2.196200, 0.000250, 0.066000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 2, 0, 0}, 175.370620, 0.000090, 0.035000},
	{{0, 0, 4, 0, -3, 0, 0, 0, 2, 1, 0}, 204.591450, 0.000020, 0.024000},
	{{0, 0, 4, 0, -2, -5, 0, 0, 2, -3, 0}, 8.586590, 0.000010, 0.088000},
	{{0, 0, 4, 0, -2, -5, 0, 0, 2, -2, 0}, 188.585930, 0.000010, 0.530000},
	{{0, 0, 4, 0, -2, -5, 0, 0, 2, -1, 0}, 188.498140, 0.000040, 0.066000},
	{{0, 0, 4, 0, -2, -5, 0, 0, 2, 0, 0}, 8.514340, 0.000010, 0.035000},
	{{0, 0, 4, 0, -2, 0, 0, 0, -4, 1, 0}, 331.911630, 0.000030, 0.031000},
	{{0, 0, 4, 0, -2, 0, 0, 0, -4, 3, 0}, 153.916450, 0.000020, 0.170000},
	{{0, 0, 4, 0, -2, 0, 0, 0, -2, -1, 0}, 331.073510, 0.000060, 0.029000},
	{{0, 0, 4, 0, -2, 0, 0, 0, -2, 0, 0}, 331.491170, 0.000560, 0.048000},
	{{0, 0, 4, 0, -2, 0, 0, 0, -2, 1, 0}, 331.062870, 0.000380, 0.131000},
	{{0, 0, 4, 0, -2, 0, 0, 0, -2, 2, 0}, 331.610480, 0.000020, 0.178000},
	{{0, 0, 4, 0, -2, 0, 0, 0, 0, -2, 0}, 154.138430, 0.000040, 0.044000},
	{{0, 0, 4, 0, -2, 0, 0, 0, 0, -1, 0}, 153.649610, 0.000290, 0.106000},
	{{0, 0, 4, 0, -2, 0, 0, 0, 0, 0, 0}, 329.333880, 0.000100, 0.261000},
	{{0, 0, 4, 0, -2, 0, 0, 0, 2, -3, 0}, 153.640880, 0.000010, 0.089000},
	{{0, 0, 4, 0, -2, 0, 0, 0, 2, -2, 0}, 154.300140, 0.000010, 0.486000},
	{{0, 0, 4, 0, -2, 0, 0, 0, 2, -1, 0}, 156.540690, 0.000040, 0.065000},
	{{0, 0, 4, 0, -2, 5, 0, 0, 2, -3, 0}, 119.423020, 0.000020, 0.091000},
	{{0, 0, 4, 0, -2, 5, 0, 0, 2, -2, 0}, 119.280680, 0.000020, 0.449000},
	{{0, 0, 4, 0, -2, 5, 0, 0, 2, -1, 0}, 119.423020, 0.000020, 0.065000},
	{{0, 0, 4, 0, -1, 0, 0, 0, -2, 0, 0}, 66.118830, 0.000060, 0.048000},
	{{0, 0, 4, 0, -1, 0, 0, 0, -2, 1, 0}, 64.731200, 0.000030, 0.132000},
	{{0, 0, 4, 0, -1, 0, 0, 0, 0, 0, 0}, 73.785190, 0.000010, 0.255000},
	{{0, 0, 4, 0, -1, 0, 0, 0, 2, -1, 0}, 317.912380, 0.000030, 0.065000},
	{{0, 0, 4, 0, -1, 0, 0, 0, 2, 0, 0}, 168.049550, 0.000070, 0.035000},
	{{0, 0, 4, 0, 0, -5, 0, 0, 2, 0, 0}, 92.673420, 0.000010, 0.035000},
	{{0, 0, 4, 0, 0, -4, 0, 0, -2, 0, 0}, 176.171320, 0.000030, 0.048000},
	{{0, 0, 4, 0, 0, -4, 0, 0, 0, 0, 0}, 177.496050, 0.000010, 0.259000},
	{{0, 0, 4, 0, 0, -4, 0, 0, 2, -1, 0}, 179.887210, 0.000050, 0.065000},
	{{0, 0, 4, 0, 0, -4, 0, 0, 2, 0, 0}, 359.887220, 0.000030, 0.035000},
	{{0, 0, 4, 0, 0, -3, 0, 0, -2, 0, 0}, 74.944160, 0.000020, 0.048000},
	{{0, 0, 4, 0, 0, -2, 0, 0, -2, 0, 0}, 333.834220, 0.000030, 0.048000},
	{{0, 0, 4, 0, 0, -2, 0, 0, 0, 0, 0}, 334.426740, 0.000010, 0.254000},
	{{0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0}, 200.824580, 0.000030, 0.250000},
	{{0, 0, 4, 0, 0, 0, 0, 0, 2, -3, 0}, 306.541870, 0.000030, 0.091000},
	{{0, 0, 4, 0, 0, 0, 0, 0, 2, -2, 0}, 128.064720, 0.000040, 0.449000},
	{{0, 0, 4, 0, 0, 0, 0, 0, 2, 0, -2}, 308.223680, 0.000030, 0.528000},
	{{0, 0, 4, 0, 0, 0, 0, 0, 2, 0, 0}, 128.143080, 0.000080, 0.035000},
	{{0, 0, 4, 0, 0, 0, 0, 0, 2, 2, 0}, 128.215870, 0.000020, 0.018000},
	{{0, 0, 4, 0, 0, 0, 0, 0, 4, -2, 0}, 308.119290, 0.000010, 0.037000},
	{{0, 0, 4, 0, 1, 0, 0, 0, 2, -1, 0}, 119.690630, 0.000040, 0.064000},
	{{0, 0, 4, 0, 1, 0, 0, 0, 2, 0, 0}, 297.343970, 0.000060, 0.035000},
	{{0, 0, 4, 0, 2, -5, 0, 0, 2, -3, 0}, 318.693330, 0.000020, 0.091000},
	{{0, 0, 4, 0, 2, -5, 0, 0, 2, -2, 0}, 319.514610, 0.000020, 0.449000},
	{{0, 0, 4, 0, 2, -5, 0, 0, 2, -1, 0}, 318.693330, 0.000020, 0.065000},
	{{0, 0, 4, 1, -2, 0, 0, 0, 2, 0, 0}, 98.622100, 0.000020, 0.034000},
	{{0, 0, 4, 1, -1, 0, 0, 0, 2, 0, 0}, 305.301410, 0.000020, 0.034000},
	{{0, 0, 4, 2, -2, 0, 0, 0, 2, 0, 0}, 127.977360, 0.000020, 0.034000},
	{{0, 0, 5, -14, 4, 0, 0, 0, -2, 0, 2}, 258.798600, 0.000020, 855.192000},
	{{0, 0, 5, -12, 0, 0, 0, 0, -2, 1, 0}, 324.394070, 0.000020, 0.078000},
	{{0, 0, 5, -11, 0, 0, 0, 0, 0, 0, 0}, 230.886360, 0.000050, 1.179000},
	{{0, 0, 5, -11, 0, 0, 0, 0, 2, -1, 0}, 305.786460, 0.000050, 0.094000},
	{{0, 0, 5, -11, 0, 0, 0, 0, 2, 0, 0}, 306.926150, 0.000050, 0.042000},
	{{0, 0, 5, -11, 1, 0, 0, 0, 0, 0, 0}, 216.972940, 0.000010, 1.309000},
	{{0, 0, 5, -10, 0, 0, 0, 0, -2, 0, 0}, 212.750520, 0.000060, 0.040000},
	{{0, 0, 5, -10, 0, 0, 0, 0, -2, 1, 0}, 212.493940, 0.000080, 0.085000},
	{{0, 0, 5, -10, 0, 0, 0, 0, 0, -1, 0}, 212.135220, 0.000170, 0.074000},
	{{0, 0, 5, -10, 0, 0, 0, 0, 0, 0, 0}, 214.341090, 0.000930, 3.156000},
	{{0, 0, 5, -10, 0, 0, 0, 0, 0, 1, 0}, 213.378750, 0.000190, 0.077000},
	{{0, 0, 5, -10, 0, 0, 0, 0, 2, -1, 0}, 221.377050, 0.000120, 0.090000},
	{{0, 0, 5, -10, 0, 0, 0, 0, 2, 0, 0}, 219.005560, 0.000100, 0.041000},
	{{0, 0, 5, -10, 0, 0, 0, 0, 2, 1, 0}, 219.673650, 0.000020, 0.027000},
	{{0, 0, 5, -10, 2, 0, 0, 0, 0, 0, 0}, 273.493030, 0.000020, 6.746000},
	{{0, 0, 5, -10, 3, 0, 0, 0, 0, 0, 0}, 39.889250, 0.000030, 15.643000},
	{{0, 0, 5, -9, -2, 0, 0, 0, 0, -1, 0}, 178.693260, 0.000020, 0.076000},
	{{0, 0, 5, -9, -2, 0, 0, 0, 0, 0, 0}, 358.153070, 0.000010, 21.630000},
	{{0, 0, 5, -9, -2, 0, 0, 0, 0, 1, 0}, 178.693260, 0.000020, 0.075000},
	{{0, 0, 5, -9, -1, 0, 0, 0, 0, -1, 0}, 24.300800, 0.000010, 0.076000},
	{{0, 0, 5, -9, -1, 0, 0, 0, 0, 1, 0}, 24.300800, 0.000010, 0.075000},
	{{0, 0, 5, -9, 0, 0, 0, 0, -4, 0, 0}, 243.825920, 0.000020, 0.020000},
	{{0, 0, 5, -9, 0, 0, 0, 0, -4, 2, 0}, 243.703260, 0.000020, 0.044000},
	{{0, 0, 5, -9, 0, 0, 0, 0, -2, -1, 0}, 243.412530, 0.000100, 0.026000},
	{{0, 0, 5, -9, 0, 0, 0, 0, -2, 0, 0}, 243.749400, 0.000950, 0.041000},
	{{0, 0, 5, -9, 0, 0, 0, 0, -2, 0, 2}, 243.841460, 0.000020, 0.431000},
	{{0, 0, 5, -9, 0, 0, 0, 0, -2, 1, 0}, 244.045760, 0.001460, 0.089000},
	{{0, 0, 5, -9, 0, 0, 0, 0, -2, 2, 0}, 243.870320, 0.000060, 0.503000},
	{{0, 0, 5, -9, 0, 0, 0, 0, -1, 0, 0}, 63.286560, 0.000010, 0.082000},
	{{0, 0, 5, -9, 0, 0, 0, 0, -1, 2, 0}, 63.286560, 0.000010, 0.070000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 0, -2, 0}, 240.437230, 0.000070, 0.038000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 0, -1, 0}, 241.182610, 0.001300, 0.077000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 0, 0, 0}, 242.296690, 0.006760, 4.655000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 0, 1, 0}, 240.955140, 0.001250, 0.074000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 0, 2, 0}, 240.772410, 0.000080, 0.037000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 1, 0, -1}, 324.746710, 0.000020, 1.192000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 2, -2, 0}, 243.012470, 0.000050, 0.641000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 2, -1, 0}, 243.525820, 0.001150, 0.085000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 2, 0, -2}, 243.158860, 0.000020, 0.528000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 2, 0, 0}, 243.136920, 0.000680, 0.040000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 2, 1, 0}, 242.900200, 0.000080, 0.026000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 4, -2, 0}, 243.170540, 0.000010, 0.043000},
	{{0, 0, 5, -9, 0, 0, 0, 0, 4, -1, 0}, 243.213480, 0.000020, 0.027000},
	{{0, 0, 5, -9, 1, 0, 0, 0, 0, 0, 0}, 274.967250, 0.000010, 3.343000},
	{{0, 0, 5, -9, 2, 0, 0, 0, 0, 0, 0}, 302.113840, 0.000010, 2.608000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -4, 0, 0}, 271.978570, 0.000020, 0.021000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -4, 1, 0}, 271.991120, 0.000010, 0.028000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -4, 2, 0}, 272.086640, 0.000020, 0.045000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -2, -1, 0}, 271.978050, 0.000100, 0.027000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -2, 0, 0}, 271.983800, 0.001100, 0.042000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -2, 0, 2}, 271.974910, 0.000020, 0.350000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -2, 1, 0}, 272.001410, 0.001510, 0.093000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -2, 2, 0}, 271.974500, 0.000050, 0.397000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -1, -1, 1}, 174.296360, 0.000070, 0.087000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -1, 0, 0}, 92.818670, 0.000020, 0.086000},
	{{0, 0, 5, -8, 0, 0, 0, 0, -1, 0, 1}, 354.316480, 0.000010, 0.555000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 0, -2, 0}, 271.882810, 0.000060, 0.039000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 0, -1, 0}, 273.524740, 0.000920, 0.080000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 0, 0, 0}, 272.055020, 0.004740, 1.340000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 0, 1, 0}, 271.885870, 0.000800, 0.071000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 0, 2, 0}, 271.854550, 0.000050, 0.037000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 1, -3, 1}, 174.472220, 0.000010, 0.076000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 1, -2, 1}, 174.301570, 0.000220, 38.061000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 1, -1, 0}, 273.151480, 0.000030, 7.118000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 1, -1, 1}, 354.296170, 0.000340, 0.075000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 1, 0, -1}, 354.309350, 0.000170, 3.255000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 1, 0, 0}, 272.818660, 0.000120, 0.076000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 2, -2, 0}, 2.250830, 0.000020, 0.973000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 2, -1, 0}, 287.376530, 0.000660, 0.082000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 2, 0, 0}, 276.302140, 0.000350, 0.039000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 2, 1, 0}, 273.051200, 0.000040, 0.026000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 4, -3, 0}, 44.709410, 0.000010, 0.096000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 4, -2, 0}, 44.971800, 0.000040, 0.042000},
	{{0, 0, 5, -8, 0, 0, 0, 0, 4, -1, 0}, 17.579010, 0.000020, 0.027000},
	{{0, 0, 5, -8, 1, 0, 0, 0, 0, 0, 0}, 308.630330, 0.000010, 1.204000},
	{{0, 0, 5, -8, 1, 5, 0, 0, -2, 0, 0}, 174.667660, 0.000070, 0.042000},
	{{0, 0, 5, -8, 1, 5, 0, 0, -2, 1, 0}, 174.785190, 0.000070, 0.095000},
	{{0, 0, 5, -8, 1, 5, 0, 0, 0, -1, 0}, 172.889200, 0.000010, 0.082000},
	{{0, 0, 5, -8, 1, 5, 0, 0, 0, 0, 0}, 172.238960, 0.000100, 0.999000},
	{{0, 0, 5, -8, 1, 5, 0, 0, 0, 1, 0}, 172.889200, 0.000010, 0.070000},
	{{0, 0, 5, -8, 2, 0, 0, 0, 0, 0, 0}, 327.561320, 0.000020, 1.093000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -6, 1, 0}, 184.531210, 0.000010, 0.017000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -4, -2, 0}, 184.428870, 0.000020, 0.013000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -4, -1, 0}, 184.151840, 0.000030, 0.016000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -4, 0, 0}, 184.360990, 0.000170, 0.021000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -4, 1, 0}, 184.594330, 0.000400, 0.028000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -4, 2, 0}, 184.950200, 0.000170, 0.046000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -3, 0, 0}, 184.686180, 0.000020, 0.028000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -3, 1, 0}, 4.686180, 0.000010, 0.044000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -3, 3, 0}, 4.686180, 0.000010, 0.273000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, -2, 0}, 184.286170, 0.000110, 0.020000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, -1, 0}, 184.495640, 0.000880, 0.027000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, -1, 2}, 4.969570, 0.000010, 0.099000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, 0, -2}, 5.098620, 0.000020, 0.020000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, 0, 0}, 184.817550, 0.008970, 0.042000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, 0, 2}, 184.773320, 0.000140, 0.322000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, 1, 0}, 185.262570, 0.009620, 0.095000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, 1, 2}, 4.969570, 0.000010, 0.061000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, 2, 0}, 185.807850, 0.000370, 0.361000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -2, 3, 0}, 185.999340, 0.000030, 0.062000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -1, 0, 0}, 9.173820, 0.000040, 0.088000},
	{{0, 0, 5, -8, 3, 0, 0, 0, -1, 1, 0}, 4.568700, 0.000010, 0.530000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 0, -3, 0}, 183.910120, 0.000010, 0.026000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 0, -2, 0}, 184.894080, 0.000310, 0.039000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 0, -1, 0}, 185.635850, 0.003530, 0.082000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 0, 0, 0}, 187.283720, 0.010910, 1.001000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 0, 1, 0}, 191.809400, 0.000970, 0.070000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 0, 2, 0}, 267.965090, 0.000010, 0.036000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 1, 0, 0}, 158.377290, 0.000040, 0.075000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 2, -2, 0}, 180.860250, 0.000030, 1.291000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 2, -1, 0}, 0.539340, 0.000400, 0.080000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 2, 0, -2}, 1.735390, 0.000040, 0.902000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 2, 0, 0}, 1.025590, 0.000520, 0.039000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 2, 1, 0}, 1.672630, 0.000040, 0.026000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 2, 2, 0}, 3.726680, 0.000020, 0.019000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 4, -2, 0}, 3.755940, 0.000020, 0.042000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 4, -1, 0}, 3.315460, 0.000040, 0.027000},
	{{0, 0, 5, -8, 3, 0, 0, 0, 4, 0, 0}, 6.122570, 0.000010, 0.020000},
	{{0, 0, 5, -8, 5, -5, 0, 0, -2, 0, 0}, 319.672780, 0.000020, 0.042000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -4, 0, 0}, 126.482740, 0.000020, 0.021000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -4, 1, 0}, 126.378730, 0.000020, 0.029000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -4, 2, 0}, 126.744980, 0.000020, 0.046000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -2, -2, 0}, 126.608460, 0.000020, 0.020000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -2, -1, 0}, 126.358220, 0.000110, 0.027000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -2, 0, 0}, 127.368270, 0.001410, 0.043000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -2, 0, 2}, 126.692050, 0.000020, 0.295000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -2, 1, 0}, 126.213930, 0.001580, 0.098000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -2, 2, 0}, 127.211150, 0.000070, 0.328000},
	{{0, 0, 5, -7, 0, 0, 0, 0, -1, 0, 1}, 206.018990, 0.000010, 0.429000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 0, -2, 0}, 124.554190, 0.000110, 0.040000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 0, -1, 0}, 124.125750, 0.001590, 0.083000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 0, 0, 0}, 123.470480, 0.005340, 0.782000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 0, 1, 0}, 123.144340, 0.000860, 0.069000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 0, 2, 0}, 123.268540, 0.000060, 0.036000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 1, -1, -1}, 204.443140, 0.000030, 0.077000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 1, -1, 1}, 24.824480, 0.000020, 0.072000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 1, 0, -1}, 204.459950, 0.000260, 4.456000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 1, 0, 1}, 24.429660, 0.000030, 0.037000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 1, 1, -1}, 204.443140, 0.000030, 0.074000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 2, -3, 0}, 298.584750, 0.000010, 0.073000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 2, -2, 0}, 304.136910, 0.000180, 2.017000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 2, -1, 0}, 301.816630, 0.001780, 0.078000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 2, 0, -2}, 302.450370, 0.000020, 1.206000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 2, 0, 0}, 302.946650, 0.000190, 0.038000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 2, 1, 0}, 302.792730, 0.000020, 0.025000},
	{{0, 0, 5, -7, 0, 0, 0, 0, 4, -2, 0}, 301.356230, 0.000020, 0.041000},
	{{0, 0, 5, -7, 2, 0, 0, 0, 2, 0, 0}, 0.523970, 0.000010, 0.038000},
	{{0, 0, 5, -7, 3, 0, 0, 0, 2, -1, 0}, 137.114770, 0.000020, 0.077000},
	{{0, 0, 5, -6, -2, 0, 0, 0, 2, -1, 0}, 274.039270, 0.000030, 0.076000},
	{{0, 0, 5, -6, -1, 0, 0, 0, 2, -3, 0}, 268.868650, 0.000010, 0.075000},
	{{0, 0, 5, -6, -1, 0, 0, 0, 2, -2, 0}, 301.350820, 0.000010, 20.650000},
	{{0, 0, 5, -6, -1, 0, 0, 0, 2, -1, 0}, 295.768910, 0.000030, 0.076000},
	{{0, 0, 5, -6, 0, -1, 0, 0, 2, -2, 0}, 269.309430, 0.000060, 518.400000},
	{{0, 0, 5, -6, 0, 0, 0, 0, -4, 1, 0}, 165.921110, 0.000020, 0.029000},
	{{0, 0, 5, -6, 0, 0, 0, 0, -2, -2, 0}, 330.101950, 0.000010, 0.020000},
	{{0, 0, 5, -6, 0, 0, 0, 0, -2, -1, 0}, 235.330220, 0.000020, 0.028000},
	{{0, 0, 5, -6, 0, 0, 0, 0, -2, 0, 0}, 169.257840, 0.000730, 0.044000},
	{{0, 0, 5, -6, 0, 0, 0, 0, -2, 1, 0}, 166.768500, 0.000850, 0.103000},
	{{0, 0, 5, -6, 0, 0, 0, 0, -2, 2, 0}, 169.605300, 0.000030, 0.279000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 0, -3, 0}, 331.850170, 0.000060, 0.026000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 0, -2, 0}, 331.453950, 0.000720, 0.040000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 0, -1, -2}, 151.741840, 0.000020, 0.026000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 0, -1, 0}, 331.173730, 0.008380, 0.087000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 0, 0, 0}, 157.540970, 0.001950, 0.553000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 0, 1, 0}, 156.029600, 0.000380, 0.066000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 0, 2, 0}, 154.399150, 0.000030, 0.035000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 1, -1, 0}, 151.741870, 0.000040, 1.084000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 1, 0, -1}, 237.308390, 0.000100, 1.323000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 1, 0, 0}, 149.019460, 0.000020, 0.071000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, -4, 0}, 332.793340, 0.000030, 0.038000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, -3, 0}, 332.763170, 0.000440, 0.076000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, -2, -2}, 152.054470, 0.000010, 0.037000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, -2, 0}, 331.682910, 0.016820, 27.873000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, -2, 2}, 152.054470, 0.000010, 0.037000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, -1, -2}, 151.705970, 0.000090, 0.074000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, -1, 0}, 151.732400, 0.042820, 0.075000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, -1, 2}, 331.741800, 0.000090, 0.025000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, 0, -2}, 331.559230, 0.000060, 3.360000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, 0, 0}, 151.786040, 0.002630, 0.038000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, 0, 2}, 331.741760, 0.000020, 0.019000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, 1, -2}, 151.506080, 0.000010, 0.077000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 2, 1, 0}, 151.705390, 0.000160, 0.025000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 3, -1, 0}, 331.741910, 0.000020, 0.039000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 4, -3, 0}, 331.335610, 0.000280, 0.087000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 4, -2, 0}, 151.791520, 0.000380, 0.040000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 4, -1, 0}, 151.753560, 0.000370, 0.026000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 4, 0, 0}, 151.731240, 0.000060, 0.019000},
	{{0, 0, 5, -6, 0, 0, 0, 0, 6, -2, 0}, 151.753460, 0.000010, 0.020000},
	{{0, 0, 5, -6, 1, 0, 0, 0, 0, 0, 0}, 5.387130, 0.000010, 0.528000},
	{{0, 0, 5, -6, 2, 0, 0, 0, 2, -2, 0}, 209.029690, 0.000020, 4.890000},
	{{0, 0, 5, -5, -1, 0, 0, 0, -2, 0, 0}, 327.438670, 0.000010, 0.044000},
	{{0, 0, 5, -5, 0, 0, 0, 0, -2, -1, 0}, 137.149400, 0.000040, 0.028000},
	{{0, 0, 5, -5, 0, 0, 0, 0, -2, 0, 0}, 133.486800, 0.000240, 0.045000},
	{{0, 0, 5, -5, 0, 0, 0, 0, -2, 1, 0}, 136.285950, 0.000210, 0.109000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 0, -2, 0}, 4.137940, 0.000020, 0.041000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 0, -1, 0}, 19.191380, 0.000280, 0.092000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 0, 0, 0}, 173.743430, 0.001280, 0.427000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 0, 1, 0}, 175.787560, 0.000220, 0.064000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 0, 2, 0}, 171.963340, 0.000020, 0.035000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 1, 0, -1}, 282.354700, 0.000040, 0.777000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 2, -2, 0}, 179.241880, 0.000050, 1.762000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 2, -1, 0}, 179.307380, 0.002610, 0.072000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 2, 0, -2}, 180.703950, 0.000180, 4.273000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 2, 0, 0}, 2.360530, 0.000200, 0.037000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 2, 1, 0}, 0.381970, 0.000050, 0.025000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 4, -2, 0}, 179.526800, 0.000040, 0.040000},
	{{0, 0, 5, -5, 0, 0, 0, 0, 4, -1, 0}, 178.074520, 0.000010, 0.026000},
	{{0, 0, 5, -5, 2, 0, 0, 0, 0, 0, 0}, 57.369180, 0.000010, 0.398000},
	{{0, 0, 5, -4, -2, 0, 0, 0, -2, 0, 0}, 149.142990, 0.000010, 0.045000},
	{{0, 0, 5, -4, 0, 0, 0, 0, -2, -1, 0}, 86.586570, 0.000020, 0.028000},
	{{0, 0, 5, -4, 0, 0, 0, 0, -2, 0, 0}, 88.535380, 0.000170, 0.046000},
	{{0, 0, 5, -4, 0, 0, 0, 0, -2, 1, 0}, 87.049420, 0.000130, 0.116000},
	{{0, 0, 5, -4, 0, 0, 0, 0, 0, -2, 0}, 255.580310, 0.000010, 0.042000},
	{{0, 0, 5, -4, 0, 0, 0, 0, 0, -1, 0}, 249.708830, 0.000160, 0.096000},
	{{0, 0, 5, -4, 0, 0, 0, 0, 0, 0, 0}, 86.605890, 0.000080, 0.348000},
	{{0, 0, 5, -4, 0, 0, 0, 0, 0, 1, 0}, 67.961590, 0.000040, 0.062000},
	{{0, 0, 5, -4, 0, 0, 0, 0, 1, 0, -1}, 309.950160, 0.000040, 0.550000},
	{{0, 0, 5, -4, 0, 0, 0, 0, 2, -1, 0}, 358.543160, 0.000130, 0.070000},
	{{0, 0, 5, -4, 0, 0, 0, 0, 2, 0, 0}, 206.476430, 0.000050, 0.036000},
	{{0, 0, 5, -4, 0, 0, 0, 0, 3, -1, -1}, 309.940120, 0.000030, 0.075000},
	{{0, 0, 5, -3, -1, 0, 0, 0, 0, 0, 0}, 197.928090, 0.000010, 0.301000},
	{{0, 0, 5, -3, 0, 0, 0, 0, -2, 0, 0}, 327.618370, 0.000030, 0.047000},
	{{0, 0, 5, -3, 0, 0, 0, 0, 0, 0, 0}, 315.909090, 0.000010, 0.294000},
	{{0, 0, 5, -2, 0, 0, 0, 0, -2, 0, 0}, 50.801350, 0.000020, 0.048000},
	{{0, 0, 5, -2, 0, 0, 0, 0, 2, -2, 0}, 266.385150, 0.000020, 0.462000},
	{{0, 0, 5, -2, 0, 0, 0, 0, 2, 0, 0}, 266.385150, 0.000020, 0.035000},
	{{0, 0, 5, 0, -8, 0, 0, 0, 2, 0, 0}, 30.778390, 0.000010, 0.034000},
	{{0, 0, 5, 0, -7, 0, 0, 0, -2, 0, 0}, 195.146960, 0.000020, 0.049000},
	{{0, 0, 5, 0, -7, 0, 0, 0, 2, -1, 0}, 200.507100, 0.000050, 0.063000},
	{{0, 0, 5, 0, -7, 0, 0, 0, 2, 0, 0}, 20.507100, 0.000050, 0.034000},
	{{0, 0, 5, 0, -7, 5, 0, 0, 2, -2, 0}, 347.689890, 0.000010, 0.356000},
	{{0, 0, 5, 0, -7, 5, 0, 0, 2, 0, 0}, 167.689890, 0.000010, 0.034000},
	{{0, 0, 5, 0, -6, 0, 0, 0, -2, 0, 0}, 182.564870, 0.000050, 0.049000},
	{{0, 0, 5, 0, -6, 0, 0, 0, -2, 1, 0}, 182.661730, 0.000020, 0.143000},
	{{0, 0, 5, 0, -6, 0, 0, 0, 0, -1, 0}, 10.051680, 0.000020, 0.114000},
	{{0, 0, 5, 0, -6, 0, 0, 0, 0, 0, 0}, 180.197340, 0.000010, 0.223000},
	{{0, 0, 5, 0, -6, 0, 0, 0, 0, 1, 0}, 190.051690, 0.000010, 0.056000},
	{{0, 0, 5, 0, -6, 0, 0, 0, 2, -2, 0}, 190.051680, 0.000010, 0.368000},
	{{0, 0, 5, 0, -6, 0, 0, 0, 2, -1, 0}, 190.051680, 0.000210, 0.063000},
	{{0, 0, 5, 0, -6, 0, 0, 0, 2, 0, 0}, 9.873380, 0.000120, 0.034000},
	{{0, 0, 5, 0, -6, 0, 0, 0, 2, 1, 0}, 10.051680, 0.000030, 0.024000},
	{{0, 0, 5, 0, -5, 0, 0, 0, -2, 0, 0}, 165.090180, 0.000140, 0.050000},
	{{0, 0, 5, 0, -5, 0, 0, 0, -2, 1, 0}, 164.999570, 0.000060, 0.145000},
	{{0, 0, 5, 0, -5, 0, 0, 0, 0, -1, 0}, 20.460820, 0.000020, 0.115000},
	{{0, 0, 5, 0, -5, 0, 0, 0, 0, 0, 0}, 139.955990, 0.000020, 0.218000},
	{{0, 0, 5, 0, -5, 0, 0, 0, 0, 1, 0}, 174.425760, 0.000050, 0.056000},
	{{0, 0, 5, 0, -5, 0, 0, 0, 2, -2, 0}, 181.761360, 0.000010, 0.357000},
	{{0, 0, 5, 0, -5, 0, 0, 0, 2, -1, 0}, 178.997640, 0.000500, 0.062000},
	{{0, 0, 5, 0, -5, 0, 0, 0, 2, 0, -2}, 179.142820, 0.000020, 0.405000},
	{{0, 0, 5, 0, -5, 0, 0, 0, 2, 0, 0}, 359.315340, 0.000320, 0.034000},
	{{0, 0, 5, 0, -5, 0, 0, 0, 2, 1, 0}, 359.105170, 0.000060, 0.023000},
	{{0, 0, 5, 0, -5, 0, 0, 0, 4, -1, 0}, 359.142810, 0.000030, 0.025000},
	{{0, 0, 5, 0, -4, 0, 0, 0, -2, 0, 0}, 62.822660, 0.000110, 0.050000},
	{{0, 0, 5, 0, -4, 0, 0, 0, -2, 1, 0}, 62.950720, 0.000070, 0.147000},
	{{0, 0, 5, 0, -4, 0, 0, 0, 0, -1, 0}, 246.258000, 0.000020, 0.116000},
	{{0, 0, 5, 0, -4, 0, 0, 0, 0, 0, 0}, 65.413920, 0.000020, 0.214000},
	{{0, 0, 5, 0, -4, 0, 0, 0, 0, 1, 0}, 58.176900, 0.000010, 0.056000},
	{{0, 0, 5, 0, -4, 0, 0, 0, 2, -1, 0}, 18.497930, 0.000060, 0.062000},
	{{0, 0, 5, 0, -4, 0, 0, 0, 2, 0, 0}, 195.994020, 0.000060, 0.034000},
	{{0, 0, 5, 0, -3, -5, 0, 0, 2, -2, 0}, 187.923290, 0.000010, 0.357000},
	{{0, 0, 5, 0, -3, -5, 0, 0, 2, 0, 0}, 7.923290, 0.000010, 0.034000},
	{{0, 0, 5, 0, -3, 0, 0, 0, -2, 0, 0}, 319.220200, 0.000060, 0.050000},
	{{0, 0, 5, 0, -3, 0, 0, 0, -2, 1, 0}, 315.209420, 0.000030, 0.148000},
	{{0, 0, 5, 0, -3, 0, 0, 0, 0, 0, 0}, 316.156380, 0.000010, 0.211000},
	{{0, 0, 5, 0, -3, 5, 0, 0, 2, 0, 0}, 298.875750, 0.000010, 0.034000},
	{{0, 0, 5, 0, -2, 0, 0, 0, -2, 0, 0}, 228.398410, 0.000040, 0.050000},
	{{0, 0, 5, 0, -2, 0, 0, 0, -2, 1, 0}, 227.717960, 0.000010, 0.150000},
	{{0, 0, 5, 0, -2, 0, 0, 0, 2, -1, 0}, 248.043160, 0.000020, 0.061000},
	{{0, 0, 5, 0, -2, 0, 0, 0, 2, 0, 0}, 250.556120, 0.000050, 0.034000},
	{{0, 0, 5, 0, -1, 0, 0, 0, 0, -1, 0}, 309.530500, 0.000010, 0.120000},
	{{0, 0, 5, 0, -1, 0, 0, 0, 0, 1, 0}, 309.530500, 0.000010, 0.055000},
	{{0, 0, 5, 0, -1, 0, 0, 0, 2, -3, 0}, 130.411710, 0.000010, 0.099000},
	{{0, 0, 5, 0, -1, 0, 0, 0, 2, -2, 0}, 309.975450, 0.000040, 0.318000},
	{{0, 0, 5, 0, -1, 0, 0, 0, 2, 0, 0}, 311.029690, 0.000040, 0.034000},
	{{0, 0, 5, 0, 0, 0, 0, 0, -2, 0, 0}, 302.153280, 0.000020, 0.051000},
	{{0, 0, 5, 0, 0, 0, 0, 0, -2, 2, 0}, 302.153280, 0.000020, 0.148000},
	{{0, 0, 5, 0, 0, 0, 0, 0, 0, -2, 0}, 158.018720, 0.000010, 0.046000},
	{{0, 0, 5, 0, 0, 0, 0, 0, 0, 0, 0}, 14.867130, 0.000040, 0.200000},
	{{0, 0, 5, 0, 0, 0, 0, 0, 0, 2, 0}, 37.365810, 0.000010, 0.032000},
	{{0, 0, 5, 0, 0, 0, 0, 0, 2, -2, 0}, 197.950440, 0.000030, 0.310000},
	{{0, 0, 5, 0, 0, 0, 0, 0, 4, 0, 0}, 215.587290, 0.000030, 0.018000},
	{{0, 0, 5, 0, 1, -5, 0, 0, 2, 0, 0}, 138.279890, 0.000010, 0.034000},
	{{0, 0, 6, -16, 4, 5, 0, 0, 0, 0, 0}, 98.554290, 0.000030, 0.500000},
	{{0, 0, 6, -16, 4, 5, 0, 0, 2, -1, 0}, 98.554290, 0.000060, 0.105000},
	{{0, 0, 6, -16, 4, 5, 0, 0, 2, 0, 0}, 98.554290, 0.000070, 0.044000},
	{{0, 0, 6, -15, 0, 0, 0, 0, 2, 0, 0}, 0.785290, 0.000020, 0.044000},
	{{0, 0, 6, -13, 0, 0, 0, 0, 0, -1, 0}, 274.742570, 0.000010, 0.071000},
	{{0, 0, 6, -13, 0, 0, 0, 0, 0, 0, 0}, 270.377970, 0.000060, 1.097000},
	{{0, 0, 6, -13, 0, 0, 0, 0, 0, 1, 0}, 274.742570, 0.000010, 0.081000},
	{{0, 0, 6, -13, 0, 0, 0, 0, 2, -1, 0}, 283.392190, 0.000060, 0.095000},
	{{0, 0, 6, -13, 0, 0, 0, 0, 2, 0, 0}, 283.392190, 0.000050, 0.042000},
	{{0, 0, 6, -12, 0, 0, 0, 0, -2, 0, 0}, 182.304840, 0.000010, 0.040000},
	{{0, 0, 6, -12, 0, 0, 0, 0, -2, 1, 0}, 182.307470, 0.000020, 0.084000},
	{{0, 0, 6, -12, 0, 0, 0, 0, 0, -1, 0}, 184.201910, 0.000050, 0.073000},
	{{0, 0, 6, -12, 0, 0, 0, 0, 0, 0, 0}, 186.110800, 0.000280, 2.630000},
	{{0, 0, 6, -12, 0, 0, 0, 0, 0, 1, 0}, 184.201910, 0.000050, 0.078000},
	{{0, 0, 6, -12, 0, 0, 0, 0, 2, -1, 0}, 193.940160, 0.000030, 0.090000},
	{{0, 0, 6, -12, 0, 0, 0, 0, 2, 0, 0}, 193.940160, 0.000030, 0.041000},
	{{0, 0, 6, -12, 2, 0, 0, 0, 0, -1, 0}, 229.812700, 0.000010, 0.074000},
	{{0, 0, 6, -12, 2, 0, 0, 0, 0, 1, 0}, 229.812700, 0.000010, 0.077000},
	{{0, 0, 6, -12, 3, 0, 0, 0, 0, 0, 0}, 46.098290, 0.000050, 7.856000},
	{{0, 0, 6, -11, -2, 0, 0, 0, 0, -1, 0}, 332.617700, 0.000010, 0.075000},
	{{0, 0, 6, -11, -2, 0, 0, 0, 0, 0, 0}, 150.582950, 0.000010, 58.360000},
	{{0, 0, 6, -11, -2, 0, 0, 0, 0, 1, 0}, 332.617700, 0.000010, 0.076000},
	{{0, 0, 6, -11, 0, 0, 0, 0, -4, 1, 0}, 214.834150, 0.000010, 0.028000},
	{{0, 0, 6, -11, 0, 0, 0, 0, -4, 2, 0}, 214.852460, 0.000010, 0.044000},
	{{0, 0, 6, -11, 0, 0, 0, 0, -2, -1, 0}, 214.837660, 0.000070, 0.026000},
	{{0, 0, 6, -11, 0, 0, 0, 0, -2, 0, 0}, 214.720720, 0.000640, 0.041000},
	{{0, 0, 6, -11, 0, 0, 0, 0, -2, 0, 2}, 214.852330, 0.000010, 0.443000},
	{{0, 0, 6, -11, 0, 0, 0, 0, -2, 1, 0}, 215.260000, 0.000990, 0.088000},
	{{0, 0, 6, -11, 0, 0, 0, 0, -2, 2, 0}, 214.882080, 0.000040, 0.519000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 0, -2, 0}, 212.793780, 0.000060, 0.038000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 0, -1, 0}, 211.861890, 0.000960, 0.076000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 0, 0, 0}, 213.130320, 0.005060, 6.602000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 0, 1, 0}, 211.636000, 0.000920, 0.075000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 0, 2, 0}, 211.023570, 0.000050, 0.038000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 1, 0, -1}, 295.649390, 0.000010, 1.108000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 2, -2, 0}, 214.112020, 0.000030, 0.616000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 2, -1, 0}, 214.682750, 0.000820, 0.086000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 2, 0, -2}, 214.228940, 0.000010, 0.511000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 2, 0, 0}, 214.167910, 0.000500, 0.040000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 2, 1, 0}, 214.293150, 0.000050, 0.026000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 4, -2, 0}, 214.270820, 0.000010, 0.043000},
	{{0, 0, 6, -11, 0, 0, 0, 0, 4, -1, 0}, 214.269880, 0.000020, 0.027000},
	{{0, 0, 6, -11, 3, 0, 0, 0, 0, 0, 0}, 78.422780, 0.000020, 2.473000},
	{{0, 0, 6, -10, 0, 0, 0, 0, -4, 1, 0}, 243.254770, 0.000010, 0.028000},
	{{0, 0, 6, -10, 0, 0, 0, 0, -2, -1, 0}, 243.253970, 0.000040, 0.027000},
	{{0, 0, 6, -10, 0, 0, 0, 0, -2, 0, 0}, 243.248410, 0.000390, 0.042000},
	{{0, 0, 6, -10, 0, 0, 0, 0, -2, 1, 0}, 243.329550, 0.000540, 0.093000},
	{{0, 0, 6, -10, 0, 0, 0, 0, -2, 2, 0}, 243.255750, 0.000030, 0.407000},
	{{0, 0, 6, -10, 0, 0, 0, 0, -1, -1, 1}, 324.932160, 0.000020, 0.087000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 0, -2, 0}, 240.919730, 0.000010, 0.039000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 0, -1, 0}, 242.380460, 0.000330, 0.080000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 0, 0, 0}, 242.972770, 0.001750, 1.464000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 0, 1, 0}, 242.327380, 0.000270, 0.072000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 0, 2, 0}, 242.372400, 0.000020, 0.037000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 1, -2, 1}, 144.936790, 0.000060, 26.959000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 1, -1, 1}, 144.932160, 0.000110, 0.076000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 1, 0, -1}, 324.929410, 0.000070, 2.698000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 1, 0, 0}, 241.120260, 0.000020, 0.077000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 2, -1, 0}, 248.551390, 0.000310, 0.082000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 2, 0, 0}, 243.967200, 0.000170, 0.039000},
	{{0, 0, 6, -10, 0, 0, 0, 0, 2, 1, 0}, 244.038230, 0.000010, 0.026000},
	{{0, 0, 6, -10, 2, 0, 0, 0, 0, 0, 0}, 122.961450, 0.000010, 1.174000},
	{{0, 0, 6, -10, 3, 0, 0, 0, -2, 0, 0}, 107.856470, 0.000010, 0.042000},
	{{0, 0, 6, -10, 3, 0, 0, 0, 0, 0, 0}, 108.284420, 0.000060, 1.068000},
	{{0, 0, 6, -9, 0, 0, 0, 0, -4, 1, 0}, 97.487600, 0.000030, 0.029000},
	{{0, 0, 6, -9, 0, 0, 0, 0, -2, -1, 0}, 97.420200, 0.000080, 0.027000},
	{{0, 0, 6, -9, 0, 0, 0, 0, -2, 0, 0}, 98.092050, 0.000780, 0.043000},
	{{0, 0, 6, -9, 0, 0, 0, 0, -2, 0, 2}, 97.555500, 0.000010, 0.301000},
	{{0, 0, 6, -9, 0, 0, 0, 0, -2, 1, 0}, 97.217960, 0.000910, 0.097000},
	{{0, 0, 6, -9, 0, 0, 0, 0, -2, 2, 0}, 96.954900, 0.000050, 0.335000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 0, -2, 0}, 94.971170, 0.000040, 0.040000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 0, -1, 0}, 95.275860, 0.000870, 0.083000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 0, 0, 0}, 94.528280, 0.002970, 0.823000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 0, 1, 0}, 94.544970, 0.000490, 0.069000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 0, 2, 0}, 94.359620, 0.000030, 0.036000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 1, -2, 0}, 271.779860, 0.000010, 0.077000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 1, -1, -1}, 174.342050, 0.000030, 0.076000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 1, 0, -1}, 174.365430, 0.000210, 6.208000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 1, 0, 0}, 91.779860, 0.000010, 0.074000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 1, 0, 1}, 354.335380, 0.000020, 0.037000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 1, 1, -2}, 94.315680, 0.000040, 176.039000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 1, 1, -1}, 174.342050, 0.000030, 0.075000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 2, -2, 0}, 274.276760, 0.000060, 1.788000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 2, -1, 0}, 272.606790, 0.000760, 0.079000},
	{{0, 0, 6, -9, 0, 0, 0, 0, 2, 0, 0}, 275.107540, 0.000060, 0.039000},
	{{0, 0, 6, -8, -2, 0, 0, 0, 0, 0, 0}, 244.028910, 0.000010, 0.634000},
	{{0, 0, 6, -8, 0, 0, 0, 0, -4, 1, 0}, 132.651660, 0.000010, 0.029000},
	{{0, 0, 6, -8, 0, 0, 0, 0, -2, -1, 0}, 126.490480, 0.000100, 0.028000},
	{{0, 0, 6, -8, 0, 0, 0, 0, -2, 0, 0}, 131.573550, 0.000500, 0.043000},
	{{0, 0, 6, -8, 0, 0, 0, 0, -2, 1, 0}, 133.357120, 0.000390, 0.103000},
	{{0, 0, 6, -8, 0, 0, 0, 0, -2, 2, 0}, 137.244510, 0.000020, 0.284000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 0, -2, 0}, 122.709870, 0.000210, 0.040000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 0, -1, 0}, 122.629300, 0.006640, 0.087000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 0, 0, 0}, 124.517600, 0.002060, 0.573000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 0, 1, 0}, 125.543600, 0.000310, 0.067000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 0, 2, 0}, 127.041140, 0.000020, 0.035000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 1, -1, 0}, 302.336450, 0.000030, 1.163000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 1, 0, -1}, 205.294510, 0.000090, 1.444000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 1, 0, 0}, 122.240970, 0.000020, 0.071000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, -4, 0}, 301.503960, 0.000050, 0.038000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, -3, 0}, 301.491630, 0.000670, 0.075000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, -2, -2}, 122.337810, 0.000020, 0.037000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, -2, 0}, 302.356460, 0.020230, 36.376000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, -2, 2}, 122.337810, 0.000020, 0.037000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, -1, -2}, 302.336250, 0.000060, 0.073000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, -1, 0}, 302.343460, 0.033170, 0.076000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, -1, 2}, 122.336330, 0.000060, 0.025000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, 0, -2}, 302.504800, 0.000030, 2.770000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, 0, 0}, 302.301810, 0.002430, 0.038000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 2, 1, 0}, 302.368760, 0.000180, 0.025000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 4, -3, 0}, 302.729960, 0.000310, 0.087000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 4, -2, 0}, 302.355240, 0.000660, 0.040000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 4, -1, 0}, 302.347020, 0.000340, 0.026000},
	{{0, 0, 6, -8, 0, 0, 0, 0, 4, 0, 0}, 302.256410, 0.000030, 0.020000},
	{{0, 0, 6, -8, 0, 1, 0, 0, 2, -2, 0}, 164.892080, 0.000020, 154.875000},
	{{0, 0, 6, -8, 1, 0, 0, 0, 0, 0, 0}, 356.855370, 0.000010, 0.546000},
	{{0, 0, 6, -8, 1, 0, 0, 0, 2, -1, 0}, 288.176580, 0.000030, 0.075000},
	{{0, 0, 6, -8, 2, 0, 0, 0, 2, 0, 0}, 180.286710, 0.000010, 0.038000},
	{{0, 0, 6, -8, 3, 0, 0, 0, -4, 1, 0}, 82.368490, 0.000030, 0.029000},
	{{0, 0, 6, -8, 3, 0, 0, 0, -2, -1, 0}, 82.673420, 0.000060, 0.028000},
	{{0, 0, 6, -8, 3, 0, 0, 0, -2, 0, 0}, 82.380890, 0.000540, 0.044000},
	{{0, 0, 6, -8, 3, 0, 0, 0, -2, 1, 0}, 82.823720, 0.000460, 0.105000},
	{{0, 0, 6, -8, 3, 0, 0, 0, -2, 2, 0}, 82.647690, 0.000020, 0.265000},
	{{0, 0, 6, -8, 3, 0, 0, 0, 0, -1, 0}, 84.182990, 0.000110, 0.089000},
	{{0, 0, 6, -8, 3, 0, 0, 0, 0, 0, 0}, 84.496910, 0.000240, 0.500000},
	{{0, 0, 6, -8, 3, 0, 0, 0, 2, -3, 0}, 112.220200, 0.000010, 0.077000},
	{{0, 0, 6, -7, -6, 0, 0, 0, 2, -2, 0}, 194.085650, 0.000020, 612.237000},
	{{0, 0, 6, -7, -2, 0, 0, 0, 2, -1, -2}, 272.419060, 0.000020, 0.075000},
	{{0, 0, 6, -7, -2, 0, 0, 0, 2, 0, -2}, 274.942530, 0.000100, 483.014000},
	{{0, 0, 6, -7, -2, 0, 0, 0, 2, 1, -2}, 272.419060, 0.000020, 0.075000},
	{{0, 0, 6, -7, 0, 0, 0, 0, -2, -1, 0}, 121.145590, 0.000030, 0.028000},
	{{0, 0, 6, -7, 0, 0, 0, 0, -2, 0, 0}, 115.838560, 0.000180, 0.045000},
	{{0, 0, 6, -7, 0, 0, 0, 0, -2, 1, 0}, 120.293110, 0.000170, 0.109000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 0, -2, 0}, 333.394890, 0.000020, 0.041000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 0, -1, 0}, 340.225830, 0.000300, 0.091000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 0, 0, 0}, 146.138230, 0.001030, 0.439000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 0, 1, 0}, 146.484060, 0.000210, 0.064000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 0, 2, 0}, 145.725930, 0.000020, 0.035000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 1, 0, -1}, 239.707070, 0.000060, 0.817000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 2, -3, 0}, 154.814160, 0.000020, 0.078000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 2, -2, 0}, 151.127200, 0.000030, 1.983000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 2, -1, -2}, 151.534370, 0.000010, 0.076000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 2, -1, 0}, 150.756920, 0.002640, 0.073000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 2, 0, -2}, 151.534370, 0.000250, 5.859000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 2, 0, 0}, 332.791250, 0.000170, 0.037000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 2, 1, -2}, 151.534370, 0.000010, 0.074000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 2, 1, 0}, 331.351030, 0.000050, 0.025000},
	{{0, 0, 6, -7, 0, 0, 0, 0, 4, -2, 0}, 150.731030, 0.000040, 0.040000},
	{{0, 0, 6, -6, 0, 0, 0, 0, -2, -1, 0}, 149.656970, 0.000020, 0.028000},
	{{0, 0, 6, -6, 0, 0, 0, 0, -2, 0, 0}, 150.743810, 0.000120, 0.046000},
	{{0, 0, 6, -6, 0, 0, 0, 0, -2, 1, 0}, 136.401120, 0.000070, 0.115000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 0, -2, 0}, 249.708530, 0.000030, 0.042000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 0, -1, 0}, 231.979040, 0.000360, 0.096000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 0, 0, 0}, 179.268550, 0.000630, 0.356000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 0, 1, 0}, 173.098750, 0.000110, 0.062000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 1, 0, -1}, 283.256030, 0.000030, 0.569000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 2, -2, 0}, 202.112200, 0.000060, 0.965000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 2, -1, 0}, 187.682710, 0.001190, 0.070000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 2, 0, -2}, 180.609290, 0.000040, 1.424000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 2, 0, 0}, 353.500090, 0.000160, 0.036000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 2, 1, 0}, 0.545970, 0.000040, 0.025000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 3, -2, -1}, 103.246380, 0.000040, 55.903000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 3, -1, -1}, 103.220610, 0.000050, 0.076000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 4, -2, 0}, 179.881160, 0.000020, 0.039000},
	{{0, 0, 6, -6, 0, 0, 0, 0, 4, -1, 0}, 0.507110, 0.000010, 0.026000},
	{{0, 0, 6, -5, 0, 0, 0, 0, -2, 0, 0}, 41.159040, 0.000050, 0.047000},
	{{0, 0, 6, -5, 0, 0, 0, 0, -2, 1, 0}, 41.008930, 0.000020, 0.123000},
	{{0, 0, 6, -5, 0, 0, 0, 0, 0, -1, 0}, 236.592170, 0.000040, 0.101000},
	{{0, 0, 6, -5, 0, 0, 0, 0, 0, 0, 0}, 53.390520, 0.000020, 0.299000},
	{{0, 0, 6, -5, 0, 0, 0, 0, 0, 1, 0}, 34.114170, 0.000020, 0.060000},
	{{0, 0, 6, -5, 0, 0, 0, 0, 1, 0, -1}, 309.731760, 0.000020, 0.437000},
	{{0, 0, 6, -5, 0, 0, 0, 0, 2, -1, 0}, 30.133990, 0.000140, 0.067000},
	{{0, 0, 6, -5, 0, 0, 0, 0, 2, 0, 0}, 208.652930, 0.000060, 0.036000},
	{{0, 0, 6, -4, 0, 0, 0, 0, -2, 0, 0}, 327.931940, 0.000020, 0.048000},
	{{0, 0, 6, 0, -8, 0, 0, 0, 2, -1, 0}, 200.248110, 0.000020, 0.059000},
	{{0, 0, 6, 0, -8, 0, 0, 0, 2, 0, 0}, 20.248110, 0.000020, 0.033000},
	{{0, 0, 6, 0, -7, 0, 0, 0, 0, 0, 0}, 10.051800, 0.000010, 0.185000},
	{{0, 0, 6, 0, -7, 0, 0, 0, 2, -1, 0}, 189.584550, 0.000040, 0.059000},
	{{0, 0, 6, 0, -7, 0, 0, 0, 2, 0, 0}, 9.584550, 0.000040, 0.033000},
	{{0, 0, 6, 0, -6, 0, 0, 0, -2, 0, 0}, 162.225890, 0.000020, 0.052000},
	{{0, 0, 6, 0, -6, 0, 0, 0, 0, 0, 0}, 5.275560, 0.000020, 0.182000},
	{{0, 0, 6, 0, -6, 0, 0, 0, 2, -1, 0}, 178.117150, 0.000100, 0.059000},
	{{0, 0, 6, 0, -6, 0, 0, 0, 2, 0, 0}, 358.117140, 0.000090, 0.033000},
	{{0, 0, 6, 0, -6, 0, 0, 0, 2, 1, 0}, 358.117140, 0.000010, 0.023000},
	{{0, 0, 6, 0, -5, 0, 0, 0, -2, 0, 0}, 59.893670, 0.000030, 0.052000},
	{{0, 0, 6, 0, -5, 0, 0, 0, 2, -1, 0}, 23.284720, 0.000020, 0.059000},
	{{0, 0, 6, 0, -5, 0, 0, 0, 2, 0, 0}, 203.285340, 0.000020, 0.033000},
	{{0, 0, 6, 0, -2, 0, 0, 0, 0, 0, 0}, 308.357190, 0.000010, 0.171000},
	{{0, 0, 6, 0, -2, 0, 0, 0, 0, 2, 0}, 308.357190, 0.000010, 0.031000},
	{{0, 0, 6, 0, 0, 0, 0, 0, 2, -2, 0}, 322.325270, 0.000010, 0.237000},
	{{0, 0, 6, 0, 0, 0, 0, 0, 2, -1, 0}, 326.762030, 0.000020, 0.057000},
	{{0, 0, 6, 0, 0, 0, 0, 0, 2, 0, 0}, 322.325270, 0.000010, 0.033000},
	{{0, 0, 7, -17, 0, -2, 0, 0, -2, 0, 2}, 356.309150, 0.000040, 997.332000},
	{{0, 0, 7, -16, 4, 5, 0, 0, -2, -1, 0}, 175.617360, 0.000020, 0.026000},
	{{0, 0, 7, -16, 4, 5, 0, 0, -2, 0, 0}, 175.617360, 0.000020, 0.039000},
	{{0, 0, 7, -16, 4, 5, 0, 0, -2, 1, 0}, 175.617360, 0.000070, 0.080000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 0, -1, 0}, 355.617360, 0.000200, 0.070000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 0, 0, 0}, 355.617360, 0.001200, 1.000000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 0, 1, 0}, 355.617360, 0.000270, 0.082000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 2, -2, 0}, 355.617360, 0.000060, 0.360000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 2, -1, 0}, 355.617360, 0.001150, 0.095000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 2, 0, -2}, 355.617360, 0.000010, 0.322000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 2, 0, 0}, 355.617360, 0.000870, 0.042000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 2, 1, 0}, 355.617360, 0.000110, 0.027000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 4, -1, 0}, 355.617360, 0.000040, 0.028000},
	{{0, 0, 7, -16, 4, 5, 0, 0, 4, 0, 0}, 355.617360, 0.000020, 0.021000},
	{{0, 0, 7, -15, 0, 0, 0, 0, -2, 1, 0}, 78.451200, 0.000030, 0.080000},
	{{0, 0, 7, -15, 0, 0, 0, 0, 0, -1, 0}, 251.870760, 0.000010, 0.070000},
	{{0, 0, 7, -15, 0, 0, 0, 0, 0, 0, 0}, 256.029940, 0.000300, 1.025000},
	{{0, 0, 7, -15, 0, 0, 0, 0, 0, 1, 0}, 254.796160, 0.000110, 0.081000},
	{{0, 0, 7, -15, 0, 0, 0, 0, 2, -1, 0}, 258.304740, 0.000280, 0.095000},
	{{0, 0, 7, -15, 0, 0, 0, 0, 2, 0, 0}, 257.804200, 0.000240, 0.042000},
	{{0, 0, 7, -15, 0, 0, 0, 0, 2, 1, 0}, 258.030640, 0.000030, 0.027000},
	{{0, 0, 7, -14, 0, 0, 0, 0, 0, 0, 0}, 158.050110, 0.000060, 2.254000},
	{{0, 0, 7, -14, 1, 0, 0, 0, 0, 0, 0}, 186.561470, 0.000020, 2.783000},
	{{0, 0, 7, -14, 3, 0, 0, 0, 0, 0, 0}, 17.050750, 0.000010, 5.245000},
	{{0, 0, 7, -13, -1, 0, 0, 0, -2, 0, 0}, 116.214640, 0.000060, 0.040000},
	{{0, 0, 7, -13, -1, 0, 0, 0, -2, 1, 0}, 116.678610, 0.000070, 0.087000},
	{{0, 0, 7, -13, -1, 0, 0, 0, 0, -1, 0}, 115.106570, 0.000130, 0.075000},
	{{0, 0, 7, -13, -1, 0, 0, 0, 0, 0, 0}, 117.137090, 0.000550, 263.130000},
	{{0, 0, 7, -13, -1, 0, 0, 0, 0, 1, 0}, 115.106570, 0.000130, 0.075000},
	{{0, 0, 7, -13, -1, 0, 0, 0, 2, -1, 0}, 116.672200, 0.000070, 0.087000},
	{{0, 0, 7, -13, -1, 0, 0, 0, 2, 0, 0}, 116.207350, 0.000050, 0.040000},
	{{0, 0, 7, -13, 0, 0, 0, 0, -4, 1, 0}, 185.846090, 0.000030, 0.028000},
	{{0, 0, 7, -13, 0, 0, 0, 0, -4, 2, 0}, 185.861430, 0.000010, 0.044000},
	{{0, 0, 7, -13, 0, 0, 0, 0, -2, -1, 0}, 185.846060, 0.000060, 0.026000},
	{{0, 0, 7, -13, 0, 0, 0, 0, -2, 0, 0}, 185.692150, 0.000630, 0.041000},
	{{0, 0, 7, -13, 0, 0, 0, 0, -2, 0, 2}, 185.846110, 0.000010, 0.455000},
	{{0, 0, 7, -13, 0, 0, 0, 0, -2, 1, 0}, 186.335820, 0.000980, 0.088000},
	{{0, 0, 7, -13, 0, 0, 0, 0, -2, 2, 0}, 185.875330, 0.000040, 0.537000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 0, -2, 0}, 182.600340, 0.000070, 0.038000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 0, -1, 0}, 182.805500, 0.001070, 0.076000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 0, 0, -2}, 4.193600, 0.000010, 0.037000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 0, 0, 0}, 184.173280, 0.005580, 11.350000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 0, 0, 2}, 4.193600, 0.000010, 0.037000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 0, 1, 0}, 182.662240, 0.001050, 0.075000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 0, 2, 0}, 182.511790, 0.000070, 0.038000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 2, -2, 0}, 185.345710, 0.000040, 0.593000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 2, -1, 0}, 185.802090, 0.000870, 0.086000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 2, 0, -2}, 185.385940, 0.000010, 0.495000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 2, 0, 0}, 185.245650, 0.000550, 0.040000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 2, 1, 0}, 185.400370, 0.000050, 0.026000},
	{{0, 0, 7, -13, 0, 0, 0, 0, 4, -1, 0}, 185.385940, 0.000020, 0.028000},
	{{0, 0, 7, -13, 1, 0, 0, 0, 0, 0, 0}, 220.097640, 0.000010, 5.800000},
	{{0, 0, 7, -13, 3, 0, 0, 0, 0, 0, 0}, 49.525840, 0.000020, 2.932000},
	{{0, 0, 7, -12, 0, 0, 0, 0, -2, -1, 0}, 214.455680, 0.000020, 0.027000},
	{{0, 0, 7, -12, 0, 0, 0, 0, -2, 0, 0}, 214.418700, 0.000120, 0.041000},
	{{0, 0, 7, -12, 0, 0, 0, 0, -2, 1, 0}, 214.487870, 0.000200, 0.092000},
	{{0, 0, 7, -12, 0, 0, 0, 0, 0, -1, 0}, 213.567220, 0.000120, 0.079000},
	{{0, 0, 7, -12, 0, 0, 0, 0, 0, 0, 0}, 213.581120, 0.000660, 1.613000},
	{{0, 0, 7, -12, 0, 0, 0, 0, 0, 1, 0}, 213.547720, 0.000100, 0.072000},
	{{0, 0, 7, -12, 0, 0, 0, 0, 1, 0, -1}, 295.835790, 0.000020, 2.304000},
	{{0, 0, 7, -12, 0, 0, 0, 0, 2, -1, 0}, 215.067120, 0.000140, 0.083000},
	{{0, 0, 7, -12, 0, 0, 0, 0, 2, 0, 0}, 214.812660, 0.000080, 0.039000},
	{{0, 0, 7, -12, 3, 0, 0, 0, 0, 0, 0}, 78.434430, 0.000030, 1.146000},
	{{0, 0, 7, -11, -2, 0, 0, 0, 0, 0, 0}, 184.739430, 0.000010, 1.017000},
	{{0, 0, 7, -11, -1, 0, 0, 0, 1, 0, -1}, 266.121830, 0.000010, 74.609000},
	{{0, 0, 7, -11, 0, 0, 0, 0, -4, 1, 0}, 68.976530, 0.000020, 0.029000},
	{{0, 0, 7, -11, 0, 0, 0, 0, -2, -1, 0}, 68.948930, 0.000050, 0.027000},
	{{0, 0, 7, -11, 0, 0, 0, 0, -2, 0, 0}, 68.892400, 0.000470, 0.042000},
	{{0, 0, 7, -11, 0, 0, 0, 0, -2, 1, 0}, 68.617060, 0.000560, 0.097000},
	{{0, 0, 7, -11, 0, 0, 0, 0, -2, 2, 0}, 68.933200, 0.000030, 0.342000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 0, -2, 0}, 66.107390, 0.000020, 0.039000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 0, -1, 0}, 66.439550, 0.000490, 0.083000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 0, 0, 0}, 66.058730, 0.001790, 0.868000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 0, 1, 0}, 65.844820, 0.000300, 0.069000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 0, 2, 0}, 66.742950, 0.000020, 0.036000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 1, -1, -1}, 144.337650, 0.000020, 0.076000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 1, 0, -1}, 144.609550, 0.000170, 10.235000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 1, 0, 1}, 324.577250, 0.000020, 0.037000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 1, 1, -1}, 144.337650, 0.000020, 0.075000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 2, -2, 0}, 247.547970, 0.000030, 1.606000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 2, -1, 0}, 245.096400, 0.000290, 0.079000},
	{{0, 0, 7, -11, 0, 0, 0, 0, 2, 0, 0}, 248.949810, 0.000020, 0.039000},
	{{0, 0, 7, -10, 0, 0, 0, 0, -2, -1, 0}, 101.110830, 0.000020, 0.028000},
	{{0, 0, 7, -10, 0, 0, 0, 0, -2, 0, 0}, 101.357310, 0.000160, 0.043000},
	{{0, 0, 7, -10, 0, 0, 0, 0, -2, 1, 0}, 102.118600, 0.000160, 0.102000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 0, -2, 0}, 93.000240, 0.000050, 0.040000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 0, -1, 0}, 93.774350, 0.001180, 0.086000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 0, 0, 0}, 95.107400, 0.000800, 0.594000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 0, 1, 0}, 96.159720, 0.000130, 0.067000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 1, 0, -1}, 175.378100, 0.000070, 1.589000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 2, -3, 0}, 273.494400, 0.000040, 0.075000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 2, -2, 0}, 273.329500, 0.001110, 11.006000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 2, -1, 0}, 273.152240, 0.005070, 0.076000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 2, 0, -2}, 273.470810, 0.000020, 2.356000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 2, 0, 0}, 273.145830, 0.000430, 0.038000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 2, 1, 0}, 273.246100, 0.000040, 0.025000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 4, -3, 0}, 273.242070, 0.000020, 0.088000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 4, -2, 0}, 273.159970, 0.000080, 0.041000},
	{{0, 0, 7, -10, 0, 0, 0, 0, 4, -1, 0}, 273.126640, 0.000050, 0.026000},
	{{0, 0, 7, -10, 1, 0, 0, 0, 2, -2, 0}, 124.265950, 0.000230, 152.550000},
	{{0, 0, 7, -10, 1, 0, 0, 0, 2, -1, 0}, 142.654340, 0.000080, 0.075000},
	{{0, 0, 7, -10, 2, 0, 0, 0, 2, -2, 0}, 331.202980, 0.000010, 12.862000},
	{{0, 0, 7, -10, 2, 0, 0, 0, 2, 0, 0}, 151.202980, 0.000010, 0.038000},
	{{0, 0, 7, -9, -2, 0, 0, 0, 0, 0, 0}, 221.758630, 0.000010, 0.489000},
	{{0, 0, 7, -9, 0, 0, 0, 0, -2, -2, 0}, 88.837850, 0.000010, 0.020000},
	{{0, 0, 7, -9, 0, 0, 0, 0, -2, 0, 0}, 92.450580, 0.000110, 0.044000},
	{{0, 0, 7, -9, 0, 0, 0, 0, -2, 1, 0}, 85.131680, 0.000090, 0.108000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 0, -2, 0}, 303.815960, 0.000020, 0.041000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 0, -1, 0}, 306.957280, 0.000220, 0.091000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 0, 0, 0}, 116.850780, 0.000590, 0.452000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 0, 1, 0}, 114.650080, 0.000120, 0.065000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 0, 2, 0}, 119.413810, 0.000020, 0.035000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 1, 0, -1}, 206.606470, 0.000050, 0.861000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 2, -2, 0}, 122.383970, 0.000020, 2.269000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 2, -1, -2}, 122.490520, 0.000020, 0.076000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 2, -1, 0}, 121.763700, 0.001860, 0.073000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 2, 0, -2}, 122.490520, 0.000300, 9.319000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 2, 0, 0}, 303.461460, 0.000110, 0.037000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 2, 1, -2}, 122.490520, 0.000020, 0.075000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 2, 1, 0}, 302.350660, 0.000030, 0.025000},
	{{0, 0, 7, -9, 0, 0, 0, 0, 4, -2, 0}, 121.641160, 0.000030, 0.040000},
	{{0, 0, 7, -9, 2, 0, 0, 0, 2, -2, 0}, 180.075260, 0.000010, 1.641000},
	{{0, 0, 7, -9, 2, 0, 0, 0, 2, 0, 0}, 180.075260, 0.000010, 0.037000},
	{{0, 0, 7, -8, -2, 0, 0, 0, 0, 0, 0}, 273.107080, 0.000010, 0.388000},
	{{0, 0, 7, -8, 0, 0, 0, 0, -2, -1, 0}, 135.903540, 0.000020, 0.028000},
	{{0, 0, 7, -8, 0, 0, 0, 0, -2, 0, 0}, 136.551360, 0.000100, 0.045000},
	{{0, 0, 7, -8, 0, 0, 0, 0, -2, 1, 0}, 129.532960, 0.000070, 0.114000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 0, -2, 0}, 350.890780, 0.000010, 0.042000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 0, -1, 0}, 10.717960, 0.000340, 0.095000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 0, 0, 0}, 147.416460, 0.000550, 0.364000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 0, 1, 0}, 148.589670, 0.000130, 0.062000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 0, 2, 0}, 149.455250, 0.000010, 0.034000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 1, 0, -1}, 241.335820, 0.000030, 0.591000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 2, -2, 0}, 106.345330, 0.000020, 1.028000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 2, -1, 0}, 144.770640, 0.001030, 0.070000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 2, 0, -2}, 151.328080, 0.000040, 1.565000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 2, 0, 0}, 336.191070, 0.000200, 0.036000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 2, 1, 0}, 331.410520, 0.000040, 0.025000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 4, -2, 0}, 151.212840, 0.000020, 0.039000},
	{{0, 0, 7, -8, 0, 0, 0, 0, 4, -1, 0}, 331.434840, 0.000010, 0.026000},
	{{0, 0, 7, -8, 3, 0, 0, 0, -2, 0, 0}, 339.862510, 0.000030, 0.046000},
	{{0, 0, 7, -8, 3, 0, 0, 0, -2, 1, 0}, 339.847810, 0.000010, 0.118000},
	{{0, 0, 7, -8, 3, 0, 0, 0, 3, -1, -2}, 338.095330, 0.000020, 204.109000},
	{{0, 0, 7, -7, 0, 0, 0, 0, -2, 0, 0}, 167.718510, 0.000060, 0.047000},
	{{0, 0, 7, -7, 0, 0, 0, 0, 0, -1, 0}, 292.335790, 0.000020, 0.100000},
	{{0, 0, 7, -7, 0, 0, 0, 0, 0, 0, 0}, 178.639390, 0.000320, 0.305000},
	{{0, 0, 7, -7, 0, 0, 0, 0, 0, 1, 0}, 176.360740, 0.000060, 0.060000},
	{{0, 0, 7, -7, 0, 0, 0, 0, 1, 0, -1}, 283.884450, 0.000020, 0.450000},
	{{0, 0, 7, -7, 0, 0, 0, 0, 2, -2, 0}, 180.407390, 0.000020, 0.665000},
	{{0, 0, 7, -7, 0, 0, 0, 0, 2, -1, 0}, 181.252120, 0.000500, 0.068000},
	{{0, 0, 7, -7, 0, 0, 0, 0, 2, 0, -2}, 180.417640, 0.000020, 0.854000},
	{{0, 0, 7, -7, 0, 0, 0, 0, 2, 0, 0}, 0.854090, 0.000130, 0.036000},
	{{0, 0, 7, -7, 0, 0, 0, 0, 2, 1, 0}, 0.410490, 0.000030, 0.024000},
	{{0, 0, 7, -6, -2, 0, 0, 0, 0, 0, 0}, 330.614250, 0.000020, 0.275000},
	{{0, 0, 7, -6, 0, 0, 0, 0, -2, 0, 0}, 43.289490, 0.000030, 0.048000},
	{{0, 0, 7, -6, 0, 0, 0, 0, -2, 1, 0}, 35.619370, 0.000010, 0.130000},
	{{0, 0, 7, -6, 0, 0, 0, 0, 0, -1, 0}, 181.782580, 0.000020, 0.106000},
	{{0, 0, 7, -6, 0, 0, 0, 0, 0, 0, 0}, 40.162000, 0.000020, 0.262000},
	{{0, 0, 7, -6, 0, 0, 0, 0, 0, 1, 0}, 29.834860, 0.000010, 0.059000},
	{{0, 0, 7, -6, 0, 0, 0, 0, 1, 0, -1}, 309.352160, 0.000010, 0.363000},
	{{0, 0, 7, -6, 0, 0, 0, 0, 2, -1, 0}, 29.845160, 0.000100, 0.065000},
	{{0, 0, 7, -6, 0, 0, 0, 0, 2, 0, 0}, 209.850720, 0.000060, 0.035000},
	{{0, 0, 7, 0, -8, 0, 0, 0, 2, 0, 0}, 8.931100, 0.000020, 0.032000},
	{{0, 0, 7, 0, -7, 0, 0, 0, 2, -1, 0}, 176.698420, 0.000030, 0.056000},
	{{0, 0, 7, 0, -7, 0, 0, 0, 2, 0, 0}, 356.698420, 0.000030, 0.032000},
	{{0, 0, 8, -19, 0, 0, 0, 0, -2, 0, 2}, 202.696180, 0.000030, 180.816000},
	{{0, 0, 8, -19, 0, 3, 0, 0, 2, 0, 0}, 277.161150, 0.000030, 0.044000},
	{{0, 0, 8, -17, 0, 0, 0, 0, -1, 0, 1}, 175.194400, 0.000050, 66.080000},
	{{0, 0, 8, -17, 0, 0, 0, 0, 0, -1, 0}, 234.308320, 0.000010, 0.070000},
	{{0, 0, 8, -17, 0, 0, 0, 0, 0, 0, 0}, 234.342770, 0.000060, 0.963000},
	{{0, 0, 8, -17, 0, 0, 0, 0, 0, 1, 0}, 234.308320, 0.000010, 0.082000},
	{{0, 0, 8, -17, 0, 0, 0, 0, 2, -1, 0}, 232.434730, 0.000050, 0.096000},
	{{0, 0, 8, -17, 0, 0, 0, 0, 2, 0, 0}, 232.434730, 0.000040, 0.042000},
	{{0, 0, 8, -16, 0, 0, 0, 0, 0, 0, 0}, 121.394900, 0.000020, 1.973000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -4, -1, 0}, 252.680440, 0.000020, 0.016000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -4, 0, 0}, 252.680440, 0.000110, 0.020000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -4, 1, 0}, 252.680440, 0.000290, 0.028000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -4, 2, 0}, 252.680440, 0.000200, 0.044000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -3, 0, 0}, 252.680440, 0.000020, 0.027000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -3, 3, 0}, 72.680440, 0.000010, 0.376000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, -2, 0}, 252.680440, 0.000070, 0.020000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, -1, 0}, 252.680440, 0.000680, 0.026000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, -1, 2}, 72.680440, 0.000010, 0.090000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, 0, -2}, 72.680440, 0.000020, 0.019000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, 0, 0}, 252.680440, 0.008650, 0.040000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, 0, 2}, 252.680440, 0.000210, 0.474000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, 1, -2}, 72.680440, 0.000030, 0.026000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, 1, 0}, 252.680440, 0.017090, 0.087000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, 2, 0}, 252.680440, 0.000830, 0.564000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -2, 3, 0}, 252.680440, 0.000060, 0.067000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -1, -1, 0}, 72.680440, 0.000020, 0.039000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -1, 0, 0}, 72.680440, 0.000230, 0.081000},
	{{0, 0, 8, -16, 4, 5, 0, 0, -1, 1, 0}, 72.680440, 0.000050, 1.127000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 0, -3, 0}, 72.680440, 0.000020, 0.025000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 0, -2, 0}, 72.680440, 0.000150, 0.038000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 0, -1, 0}, 72.680440, 0.001990, 0.075000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 0, 0, 0}, 72.680440, 0.001880, 94316.280000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 0, 1, 0}, 72.680440, 0.001990, 0.075000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 0, 2, 0}, 72.680440, 0.000150, 0.038000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 0, 3, 0}, 72.680440, 0.000020, 0.025000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 1, -1, 0}, 72.680440, 0.000050, 1.127000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 1, 0, 0}, 72.680440, 0.000230, 0.081000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 1, 1, 0}, 72.680440, 0.000020, 0.039000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, -3, 0}, 252.680440, 0.000060, 0.067000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, -2, 0}, 252.680440, 0.000830, 0.564000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, -1, 0}, 252.680440, 0.017090, 0.087000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, -1, 2}, 72.680440, 0.000030, 0.026000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, 0, -2}, 252.680440, 0.000210, 0.474000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, 0, 0}, 252.680440, 0.008650, 0.040000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, 0, 2}, 72.680440, 0.000020, 0.019000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, 1, -2}, 72.680440, 0.000010, 0.090000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, 1, 0}, 252.680440, 0.000680, 0.026000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 2, 2, 0}, 252.680440, 0.000070, 0.020000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 3, -3, 0}, 72.680440, 0.000010, 0.376000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 3, 0, 0}, 252.680440, 0.000020, 0.027000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 4, -2, 0}, 252.680440, 0.000200, 0.044000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 4, -1, 0}, 252.680440, 0.000290, 0.028000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 4, 0, 0}, 252.680440, 0.000110, 0.020000},
	{{0, 0, 8, -16, 4, 5, 0, 0, 4, 1, 0}, 252.680440, 0.000020, 0.016000},
	{{0, 0, 8, -16, 6, 0, 0, 0, -2, 0, 0}, 261.024760, 0.000070, 0.040000},
	{{0, 0, 8, -16, 6, 0, 0, 0, -2, 1, 0}, 261.024760, 0.000120, 0.087000},
	{{0, 0, 8, -16, 6, 0, 0, 0, 0, 1, 0}, 81.024760, 0.000020, 0.075000},
	{{0, 0, 8, -16, 6, 0, 0, 0, 2, -1, 0}, 261.024760, 0.000120, 0.087000},
	{{0, 0, 8, -16, 6, 0, 0, 0, 2, 0, 0}, 261.024760, 0.000070, 0.040000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -4, 0, 0}, 156.840070, 0.000030, 0.020000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -4, 1, 0}, 156.841570, 0.000080, 0.028000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -4, 2, 0}, 157.633710, 0.000040, 0.044000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -2, -2, 0}, 156.834210, 0.000030, 0.020000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -2, -1, 0}, 156.211530, 0.000260, 0.026000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -2, 0, 0}, 156.708470, 0.002680, 0.040000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -2, 0, 2}, 156.807770, 0.000050, 0.469000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -2, 1, 0}, 157.360800, 0.004140, 0.087000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -2, 2, 0}, 158.413430, 0.000160, 0.556000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -2, 3, 0}, 156.845410, 0.000010, 0.066000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -1, 0, 0}, 336.592250, 0.000100, 0.081000},
	{{0, 0, 8, -15, 0, 0, 0, 0, -1, 1, 0}, 336.807770, 0.000010, 1.097000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, -3, 0}, 152.805150, 0.000020, 0.025000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, -2, 0}, 153.807530, 0.000350, 0.038000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, -1, -2}, 334.483960, 0.000020, 0.025000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, -1, 0}, 153.770740, 0.005030, 0.076000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, 0, -2}, 336.818950, 0.000010, 0.037000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, 0, 0}, 155.309140, 0.026110, 40.427000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, 0, 2}, 336.818950, 0.000010, 0.037000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, 1, 0}, 153.729870, 0.005010, 0.075000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, 1, 2}, 334.483960, 0.000020, 0.025000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, 2, 0}, 153.752760, 0.000340, 0.038000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 0, 3, 0}, 152.764630, 0.000020, 0.025000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 1, -1, 0}, 336.645840, 0.000010, 1.160000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 1, 0, 0}, 336.620540, 0.000070, 0.081000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 2, -3, 0}, 156.637250, 0.000010, 0.067000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 2, -2, 0}, 158.197090, 0.000150, 0.572000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 2, -1, 0}, 157.222110, 0.003980, 0.087000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 2, 0, -2}, 156.645840, 0.000060, 0.480000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 2, 0, 0}, 156.539150, 0.002570, 0.040000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 2, 1, 0}, 156.060700, 0.000250, 0.026000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 2, 2, 0}, 156.672280, 0.000030, 0.020000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 4, -2, 0}, 157.368880, 0.000050, 0.044000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 4, -1, 0}, 156.692690, 0.000070, 0.028000},
	{{0, 0, 8, -15, 0, 0, 0, 0, 4, 0, 0}, 156.676820, 0.000030, 0.020000},
	{{0, 0, 8, -15, 2, 0, 0, 0, 0, -1, 0}, 18.953480, 0.000010, 0.077000},
	{{0, 0, 8, -15, 2, 0, 0, 0, 0, 1, 0}, 18.953480, 0.000010, 0.074000},
	{{0, 0, 8, -14, 0, 0, 0, 0, -2, 0, 0}, 185.614220, 0.000060, 0.041000},
	{{0, 0, 8, -14, 0, 0, 0, 0, -2, 1, 0}, 185.614290, 0.000090, 0.092000},
	{{0, 0, 8, -14, 0, 0, 0, 0, 0, -1, 0}, 182.672800, 0.000050, 0.079000},
	{{0, 0, 8, -14, 0, 0, 0, 0, 0, 0, 0}, 184.334930, 0.000290, 1.797000},
	{{0, 0, 8, -14, 0, 0, 0, 0, 0, 1, 0}, 182.672800, 0.000050, 0.072000},
	{{0, 0, 8, -14, 0, 0, 0, 0, 2, -1, 0}, 184.995050, 0.000070, 0.083000},
	{{0, 0, 8, -14, 0, 0, 0, 0, 2, 0, 0}, 186.159950, 0.000040, 0.040000},
	{{0, 0, 8, -14, 3, 0, 0, 0, 0, 0, 0}, 49.413480, 0.000010, 1.236000},
	{{0, 0, 8, -13, 0, -1, 0, 0, 1, -1, -1}, 140.831400, 0.000020, 0.075000},
	{{0, 0, 8, -13, 0, -1, 0, 0, 1, 0, -1}, 173.131690, 0.000310, 2556.886000},
	{{0, 0, 8, -13, 0, -1, 0, 0, 1, 1, -1}, 140.831400, 0.000020, 0.075000},
	{{0, 0, 8, -13, 0, 0, 0, 0, -4, 1, 0}, 41.718310, 0.000010, 0.028000},
	{{0, 0, 8, -13, 0, 0, 0, 0, -2, -1, 0}, 41.681190, 0.000040, 0.027000},
	{{0, 0, 8, -13, 0, 0, 0, 0, -2, 0, 0}, 41.593140, 0.000350, 0.042000},
	{{0, 0, 8, -13, 0, 0, 0, 0, -2, 1, 0}, 41.867030, 0.000430, 0.096000},
	{{0, 0, 8, -13, 0, 0, 0, 0, -2, 2, 0}, 41.782550, 0.000020, 0.349000},
	{{0, 0, 8, -13, 0, 0, 0, 0, 0, -2, 0}, 38.300270, 0.000010, 0.039000},
	{{0, 0, 8, -13, 0, 0, 0, 0, 0, -1, 0}, 38.596600, 0.000320, 0.082000},
	{{0, 0, 8, -13, 0, 0, 0, 0, 0, 0, 0}, 37.649220, 0.001310, 0.919000},
	{{0, 0, 8, -13, 0, 0, 0, 0, 0, 1, 0}, 37.190690, 0.000200, 0.070000},
	{{0, 0, 8, -13, 0, 0, 0, 0, 1, -1, -1}, 112.411750, 0.000020, 0.076000},
	{{0, 0, 8, -13, 0, 0, 0, 0, 1, 0, -1}, 113.542440, 0.000180, 29.122000},
	{{0, 0, 8, -13, 0, 0, 0, 0, 1, 1, -1}, 112.411750, 0.000020, 0.075000},
	{{0, 0, 8, -13, 0, 0, 0, 0, 2, -1, 0}, 216.296570, 0.000090, 0.080000},
	{{0, 0, 8, -13, 0, 0, 0, 0, 2, 0, 0}, 37.635770, 0.000020, 0.039000},
	{{0, 0, 8, -12, 0, 0, 0, 0, -2, 0, 0}, 70.665490, 0.000080, 0.043000},
	{{0, 0, 8, -12, 0, 0, 0, 0, -2, 1, 0}, 71.228570, 0.000080, 0.101000},
	{{0, 0, 8, -12, 0, 0, 0, 0, 0, -2, 0}, 64.232460, 0.000020, 0.040000},
	{{0, 0, 8, -12, 0, 0, 0, 0, 0, -1, 0}, 64.802090, 0.000360, 0.086000},
	{{0, 0, 8, -12, 0, 0, 0, 0, 0, 0, 0}, 65.759560, 0.000360, 0.617000},
	{{0, 0, 8, -12, 0, 0, 0, 0, 0, 1, 0}, 66.935730, 0.000080, 0.067000},
	{{0, 0, 8, -12, 0, 0, 0, 0, 1, 0, -1}, 145.719110, 0.000040, 1.767000},
	{{0, 0, 8, -12, 0, 0, 0, 0, 2, -2, 0}, 244.823650, 0.000210, 6.484000},
	{{0, 0, 8, -12, 0, 0, 0, 0, 2, -1, 0}, 244.220440, 0.001420, 0.076000},
	{{0, 0, 8, -12, 0, 0, 0, 0, 2, 0, 0}, 244.303650, 0.000150, 0.038000},
	{{0, 0, 8, -12, 0, 0, 0, 0, 4, -2, 0}, 244.133690, 0.000020, 0.041000},
	{{0, 0, 8, -12, 2, 0, 0, 0, 2, -2, 0}, 314.299130, 0.000040, 69.540000},
	{{0, 0, 8, -12, 2, 0, 0, 0, 2, -1, 0}, 129.289690, 0.000070, 0.075000},
	{{0, 0, 8, -11, -2, 0, 0, 0, 0, 0, 0}, 231.572640, 0.000010, 0.504000},
	{{0, 0, 8, -11, 0, 0, 0, 0, -2, 0, 0}, 52.262740, 0.000060, 0.044000},
	{{0, 0, 8, -11, 0, 0, 0, 0, -2, 1, 0}, 41.954140, 0.000040, 0.107000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 0, -2, 0}, 272.438080, 0.000020, 0.041000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 0, -1, 0}, 275.933300, 0.000170, 0.090000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 0, 0, 0}, 86.948180, 0.000290, 0.465000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 0, 1, 0}, 84.662700, 0.000060, 0.065000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 1, 0, -1}, 175.955740, 0.000040, 0.911000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 2, -1, -2}, 93.559820, 0.000070, 0.076000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 2, -1, 0}, 92.898230, 0.001160, 0.073000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 2, 0, -2}, 93.468440, 0.000650, 22.758000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 2, 0, 0}, 274.115890, 0.000100, 0.037000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 2, 1, -2}, 93.559820, 0.000070, 0.075000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 2, 1, 0}, 273.453930, 0.000010, 0.025000},
	{{0, 0, 8, -11, 0, 0, 0, 0, 4, -2, 0}, 92.737550, 0.000020, 0.040000},
	{{0, 0, 8, -10, 0, 0, 0, 0, -2, 0, 0}, 110.651260, 0.000070, 0.045000},
	{{0, 0, 8, -10, 0, 0, 0, 0, -2, 1, 0}, 103.425460, 0.000040, 0.114000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 0, -2, 0}, 302.195890, 0.000010, 0.042000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 0, -1, 0}, 323.650730, 0.000100, 0.095000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 0, 0, 0}, 120.120380, 0.000340, 0.373000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 0, 1, 0}, 119.814660, 0.000070, 0.063000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 1, 0, -1}, 207.572640, 0.000030, 0.614000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 2, -2, 0}, 118.519360, 0.000020, 1.100000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 2, -1, 0}, 121.122530, 0.000780, 0.071000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 2, 0, -2}, 122.442120, 0.000030, 1.737000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 2, 0, 0}, 302.480860, 0.000110, 0.036000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 2, 1, 0}, 302.420190, 0.000030, 0.025000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 3, -3, 0}, 302.780380, 0.000020, 45.185000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 3, -2, 0}, 122.780390, 0.000030, 0.075000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 4, -2, 0}, 122.338160, 0.000010, 0.039000},
	{{0, 0, 8, -10, 0, 0, 0, 0, 4, 0, 0}, 122.441430, 0.000020, 0.019000},
	{{0, 0, 8, -9, 0, 0, 0, 0, -2, 0, 0}, 145.268430, 0.000060, 0.046000},
	{{0, 0, 8, -9, 0, 0, 0, 0, -2, 1, 0}, 144.667240, 0.000020, 0.121000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 0, -1, 0}, 293.124050, 0.000020, 0.100000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 0, 0, 0}, 149.559150, 0.000320, 0.311000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 0, 1, 0}, 148.510520, 0.000060, 0.061000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 1, 0, -1}, 242.902200, 0.000020, 0.463000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 2, -2, 0}, 151.382450, 0.000020, 0.694000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 2, -1, 0}, 151.403690, 0.000590, 0.068000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 2, 0, -2}, 151.383420, 0.000020, 0.903000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 2, 0, 0}, 330.971770, 0.000160, 0.036000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 2, 1, 0}, 331.209110, 0.000030, 0.024000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 4, -2, 0}, 149.934220, 0.000010, 0.038000},
	{{0, 0, 8, -9, 0, 0, 0, 0, 4, -1, 0}, 331.372640, 0.000010, 0.025000},
	{{0, 0, 8, -8, 0, 0, 0, 0, -2, 0, 0}, 159.669330, 0.000020, 0.048000},
	{{0, 0, 8, -8, 0, 0, 0, 0, 0, -1, 0}, 359.630610, 0.000030, 0.105000},
	{{0, 0, 8, -8, 0, 0, 0, 0, 0, 0, 0}, 177.893540, 0.000170, 0.267000},
	{{0, 0, 8, -8, 0, 0, 0, 0, 0, 1, 0}, 180.054460, 0.000010, 0.059000},
	{{0, 0, 8, -8, 0, 0, 0, 0, 1, 0, -1}, 284.293740, 0.000010, 0.371000},
	{{0, 0, 8, -8, 0, 0, 0, 0, 2, -2, 0}, 180.061320, 0.000010, 0.507000},
	{{0, 0, 8, -8, 0, 0, 0, 0, 2, -1, 0}, 179.800850, 0.000250, 0.066000},
	{{0, 0, 8, -8, 0, 0, 0, 0, 2, 0, 0}, 0.100570, 0.000110, 0.035000},
	{{0, 0, 8, -8, 0, 0, 0, 0, 2, 1, 0}, 0.063000, 0.000030, 0.024000},
	{{0, 0, 8, -7, -2, 0, 0, 0, 0, 0, 0}, 330.151480, 0.000010, 0.243000},
	{{0, 0, 8, -7, 0, 0, 0, 0, -2, 0, 0}, 45.814310, 0.000010, 0.049000},
	{{0, 0, 8, -7, 0, 0, 0, 0, 0, 0, 0}, 48.985820, 0.000020, 0.234000},
	{{0, 0, 8, -7, 0, 0, 0, 0, 0, 1, 0}, 30.569510, 0.000010, 0.057000},
	{{0, 0, 8, -7, 0, 0, 0, 0, 2, -1, 0}, 30.577760, 0.000070, 0.063000},
	{{0, 0, 8, -7, 0, 0, 0, 0, 2, 0, 0}, 210.582590, 0.000050, 0.034000},
	{{0, 0, 9, -19, 0, 3, 0, 0, 0, -2, 0}, 174.224320, 0.000010, 0.036000},
	{{0, 0, 9, -19, 0, 3, 0, 0, 0, -1, 0}, 174.224090, 0.000030, 0.070000},
	{{0, 0, 9, -19, 0, 3, 0, 0, 0, 0, 0}, 174.224170, 0.000230, 1.000000},
	{{0, 0, 9, -19, 0, 3, 0, 0, 0, 1, 0}, 174.224120, 0.000050, 0.082000},
	{{0, 0, 9, -19, 0, 3, 0, 0, 2, -1, 0}, 174.224220, 0.000240, 0.095000},
	{{0, 0, 9, -19, 0, 3, 0, 0, 2, 0, 0}, 174.224210, 0.000190, 0.042000},
	{{0, 0, 9, -19, 0, 3, 0, 0, 2, 1, 0}, 174.224220, 0.000020, 0.027000},
	{{0, 0, 9, -17, 0, 0, 0, 0, -4, 1, 0}, 127.720040, 0.000010, 0.028000},
	{{0, 0, 9, -17, 0, 0, 0, 0, -2, -1, 0}, 127.720030, 0.000040, 0.026000},
	{{0, 0, 9, -17, 0, 0, 0, 0, -2, 0, 0}, 127.558730, 0.000380, 0.040000},
	{{0, 0, 9, -17, 0, 0, 0, 0, -2, 1, 0}, 128.207850, 0.000580, 0.087000},
	{{0, 0, 9, -17, 0, 0, 0, 0, -2, 2, 0}, 127.697300, 0.000020, 0.576000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 0, -2, 0}, 125.285460, 0.000060, 0.038000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 0, -1, 0}, 125.086550, 0.000770, 0.075000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 0, 0, 0}, 126.491160, 0.004070, 25.886000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 0, 1, 0}, 125.174050, 0.000780, 0.076000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 0, 2, 0}, 125.288420, 0.000060, 0.038000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 2, -2, 0}, 128.054470, 0.000030, 0.552000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 2, -1, 0}, 128.562030, 0.000610, 0.087000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 2, 0, 0}, 127.864860, 0.000400, 0.040000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 2, 1, 0}, 128.034770, 0.000040, 0.026000},
	{{0, 0, 9, -17, 0, 0, 0, 0, 4, -1, 0}, 128.034770, 0.000010, 0.028000},
	{{0, 0, 9, -17, 1, 0, 0, 0, 0, 0, 0}, 173.431370, 0.000010, 21.895000},
	{{0, 0, 9, -16, 0, 0, 0, 0, -2, 0, 0}, 156.749710, 0.000030, 0.041000},
	{{0, 0, 9, -16, 0, 0, 0, 0, -2, 1, 0}, 156.749820, 0.000020, 0.091000},
	{{0, 0, 9, -16, 0, 0, 0, 0, 0, -1, 0}, 156.615560, 0.000020, 0.078000},
	{{0, 0, 9, -16, 0, 0, 0, 0, 0, 0, 0}, 154.688330, 0.000120, 2.028000},
	{{0, 0, 9, -16, 0, 0, 0, 0, 0, 1, 0}, 156.615560, 0.000020, 0.073000},
	{{0, 0, 9, -16, 0, 0, 0, 0, 2, -1, 0}, 155.263050, 0.000020, 0.084000},
	{{0, 0, 9, -16, 0, 0, 0, 0, 2, 0, 0}, 156.375650, 0.000020, 0.040000},
	{{0, 0, 9, -16, 4, 5, 0, 0, -4, 0, 0}, 149.743510, 0.000030, 0.021000},
	{{0, 0, 9, -16, 4, 5, 0, 0, -4, 1, 0}, 149.743510, 0.000030, 0.028000},
	{{0, 0, 9, -16, 4, 5, 0, 0, -4, 2, 0}, 149.743510, 0.000020, 0.046000},
	{{0, 0, 9, -16, 4, 5, 0, 0, -2, -1, 0}, 149.743510, 0.000100, 0.027000},
	{{0, 0, 9, -16, 4, 5, 0, 0, -2, 0, 0}, 149.743510, 0.000870, 0.042000},
	{{0, 0, 9, -16, 4, 5, 0, 0, -2, 0, 2}, 149.743510, 0.000010, 0.322000},
	{{0, 0, 9, -16, 4, 5, 0, 0, -2, 1, 0}, 149.743510, 0.001140, 0.095000},
	{{0, 0, 9, -16, 4, 5, 0, 0, -2, 2, 0}, 149.743510, 0.000060, 0.360000},
	{{0, 0, 9, -16, 4, 5, 0, 0, 0, -1, 0}, 149.743510, 0.000250, 0.082000},
	{{0, 0, 9, -16, 4, 5, 0, 0, 0, 0, 0}, 149.743510, 0.001210, 1.000000},
	{{0, 0, 9, -16, 4, 5, 0, 0, 0, 1, 0}, 149.743510, 0.000200, 0.070000},
	{{0, 0, 9, -16, 4, 5, 0, 0, 2, -2, 0}, 149.743510, 0.000010, 1.292000},
	{{0, 0, 9, -16, 4, 5, 0, 0, 2, -1, 0}, 329.743510, 0.000060, 0.080000},
	{{0, 0, 9, -16, 4, 5, 0, 0, 2, 0, 0}, 329.743510, 0.000040, 0.039000},
	{{0, 0, 9, -16, 4, 5, 0, 0, 2, 1, 0}, 329.743510, 0.000020, 0.026000},
	{{0, 0, 9, -15, 0, 0, 0, 0, -4, 1, 0}, 21.756480, 0.000020, 0.028000},
	{{0, 0, 9, -15, 0, 0, 0, 0, -2, -1, 0}, 21.459060, 0.000070, 0.027000},
	{{0, 0, 9, -15, 0, 0, 0, 0, -2, 0, 0}, 23.476730, 0.000670, 0.042000},
	{{0, 0, 9, -15, 0, 0, 0, 0, -2, 1, 0}, 21.628200, 0.000810, 0.096000},
	{{0, 0, 9, -15, 0, 0, 0, 0, -2, 2, 0}, 22.441090, 0.000040, 0.357000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 0, -2, 0}, 13.576750, 0.000020, 0.039000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 0, -1, 0}, 15.102610, 0.000520, 0.082000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 0, 0, 0}, 12.348440, 0.002090, 0.976000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 0, 1, 0}, 9.714220, 0.000310, 0.070000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 0, 2, 0}, 9.094270, 0.000010, 0.036000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 1, -1, -1}, 268.080740, 0.000020, 0.075000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 1, 0, -1}, 267.801350, 0.000170, 34.449000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 1, 0, 0}, 191.628140, 0.000040, 0.075000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 1, 0, 1}, 267.678280, 0.000040, 0.037000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 1, 1, -1}, 268.080740, 0.000020, 0.076000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 2, -1, 0}, 334.767440, 0.000020, 0.080000},
	{{0, 0, 9, -15, 0, 0, 0, 0, 2, 0, 0}, 355.087010, 0.000040, 0.039000},
	{{0, 0, 9, -14, 0, 0, 0, 0, -2, 0, 0}, 40.526230, 0.000040, 0.043000},
	{{0, 0, 9, -14, 0, 0, 0, 0, -2, 1, 0}, 40.526170, 0.000040, 0.101000},
	{{0, 0, 9, -14, 0, 0, 0, 0, 0, -1, 0}, 36.377370, 0.000120, 0.085000},
	{{0, 0, 9, -14, 0, 0, 0, 0, 0, 0, 0}, 37.324080, 0.000140, 0.643000},
	{{0, 0, 9, -14, 0, 0, 0, 0, 0, 1, 0}, 37.695410, 0.000030, 0.068000},
	{{0, 0, 9, -14, 0, 0, 0, 0, 1, 0, -1}, 116.339530, 0.000020, 1.989000},
	{{0, 0, 9, -14, 0, 0, 0, 0, 2, -2, 0}, 215.103410, 0.000050, 4.596000},
	{{0, 0, 9, -14, 0, 0, 0, 0, 2, -1, 0}, 215.879510, 0.000440, 0.077000},
	{{0, 0, 9, -14, 0, 0, 0, 0, 2, 0, 0}, 215.713850, 0.000060, 0.038000},
	{{0, 0, 9, -13, 0, 0, 0, 0, -2, 0, 0}, 359.049740, 0.000030, 0.044000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 0, -1, 0}, 245.572160, 0.000120, 0.090000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 0, 0, -2}, 64.772510, 0.000010, 0.040000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 0, 0, 0}, 53.917190, 0.000120, 0.479000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 0, 1, -2}, 244.545920, 0.000020, 0.087000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 0, 1, 0}, 57.269650, 0.000030, 0.065000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 1, 0, -1}, 146.118040, 0.000020, 0.967000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 2, -2, -2}, 64.618930, 0.000010, 0.038000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 2, -1, -2}, 64.656350, 0.000210, 0.075000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 2, -1, 0}, 63.899220, 0.000680, 0.074000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 2, 0, -2}, 64.772510, 0.000950, 51.478000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 2, 0, 0}, 64.205040, 0.000070, 0.037000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 2, 1, -2}, 64.656350, 0.000210, 0.076000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 2, 2, -2}, 64.618930, 0.000010, 0.038000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 4, -1, -2}, 244.545920, 0.000020, 0.087000},
	{{0, 0, 9, -13, 0, 0, 0, 0, 4, 0, -2}, 64.772510, 0.000010, 0.040000},
	{{0, 0, 9, -12, 0, 0, 0, 0, -2, 0, 0}, 81.844700, 0.000040, 0.045000},
	{{0, 0, 9, -12, 0, 0, 0, 0, -2, 1, 0}, 81.741870, 0.000010, 0.113000},
	{{0, 0, 9, -12, 0, 0, 0, 0, 0, -1, 0}, 273.471420, 0.000070, 0.094000},
	{{0, 0, 9, -12, 0, 0, 0, 0, 0, 0, 0}, 91.277580, 0.000190, 0.382000},
	{{0, 0, 9, -12, 0, 0, 0, 0, 0, 1, 0}, 93.325870, 0.000030, 0.063000},
	{{0, 0, 9, -12, 0, 0, 0, 0, 1, 0, -1}, 176.572510, 0.000020, 0.639000},
	{{0, 0, 9, -12, 0, 0, 0, 0, 2, -2, 0}, 93.417660, 0.000010, 1.182000},
	{{0, 0, 9, -12, 0, 0, 0, 0, 2, -1, 0}, 93.426750, 0.000480, 0.071000},
	{{0, 0, 9, -12, 0, 0, 0, 0, 2, 0, -2}, 93.432570, 0.000020, 1.952000},
	{{0, 0, 9, -12, 0, 0, 0, 0, 2, 0, 0}, 273.417990, 0.000070, 0.037000},
	{{0, 0, 9, -12, 0, 0, 0, 0, 2, 1, 0}, 273.425670, 0.000010, 0.025000},
	{{0, 0, 9, -12, 2, 0, 0, 0, 2, -3, 0}, 150.929730, 0.000010, 0.082000},
	{{0, 0, 9, -12, 2, 0, 0, 0, 2, -1, 0}, 330.929730, 0.000010, 0.070000},
	{{0, 0, 9, -11, -2, 0, 0, 0, 0, 0, 0}, 231.910890, 0.000010, 0.335000},
	{{0, 0, 9, -11, 0, 0, 0, 0, -2, 0, 0}, 118.450550, 0.000050, 0.046000},
	{{0, 0, 9, -11, 0, 0, 0, 0, 0, -1, 0}, 303.694180, 0.000040, 0.099000},
	{{0, 0, 9, -11, 0, 0, 0, 0, 0, 0, 0}, 120.643490, 0.000220, 0.317000},
	{{0, 0, 9, -11, 0, 0, 0, 0, 0, 1, 0}, 121.214910, 0.000040, 0.061000},
	{{0, 0, 9, -11, 0, 0, 0, 0, 1, 0, -1}, 208.566070, 0.000020, 0.477000},
	{{0, 0, 9, -11, 0, 0, 0, 0, 2, -2, 0}, 122.379130, 0.000020, 0.726000},
	{{0, 0, 9, -11, 0, 0, 0, 0, 2, -1, 0}, 122.376790, 0.000440, 0.068000},
	{{0, 0, 9, -11, 0, 0, 0, 0, 2, 0, -2}, 122.379700, 0.000010, 0.958000},
	{{0, 0, 9, -11, 0, 0, 0, 0, 2, 0, 0}, 301.613360, 0.000110, 0.036000},
	{{0, 0, 9, -11, 0, 0, 0, 0, 2, 1, 0}, 302.134870, 0.000020, 0.024000},
	{{0, 0, 9, -10, -2, 0, 0, 0, 0, 0, 0}, 272.360200, 0.000010, 0.285000},
	{{0, 0, 9, -10, 0, 0, 0, 0, -2, 0, 0}, 144.465670, 0.000050, 0.047000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 0, -1, 0}, 334.889690, 0.000010, 0.104000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 0, 0, 0}, 149.732930, 0.000180, 0.272000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 0, 1, 0}, 147.491300, 0.000030, 0.059000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 1, 0, -1}, 244.373260, 0.000010, 0.380000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 2, -2, 0}, 151.223520, 0.000020, 0.524000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 2, -1, 0}, 151.204940, 0.000340, 0.066000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 2, 0, -2}, 151.226850, 0.000010, 0.635000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 2, 0, 0}, 332.743770, 0.000130, 0.035000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 2, 1, 0}, 331.684310, 0.000030, 0.024000},
	{{0, 0, 9, -10, 0, 0, 0, 0, 4, 0, 0}, 151.226770, 0.000020, 0.019000},
	{{0, 0, 9, -9, -2, 0, 0, 0, 0, 0, 0}, 301.218410, 0.000010, 0.247000},
	{{0, 0, 9, -9, -1, 0, 0, 0, 0, 0, 0}, 146.965480, 0.000010, 0.242000},
	{{0, 0, 9, -9, 0, 0, 0, 0, -2, 0, 0}, 154.121750, 0.000020, 0.049000},
	{{0, 0, 9, -9, 0, 0, 0, 0, 0, -1, 0}, 359.465520, 0.000010, 0.111000},
	{{0, 0, 9, -9, 0, 0, 0, 0, 0, 0, 0}, 175.656250, 0.000080, 0.237000},
	{{0, 0, 9, -9, 0, 0, 0, 0, 2, -1, 0}, 179.469940, 0.000120, 0.064000},
	{{0, 0, 9, -9, 0, 0, 0, 0, 2, 0, 0}, 359.468270, 0.000080, 0.035000},
	{{0, 0, 9, -9, 0, 0, 0, 0, 2, 1, -4}, 186.670870, 0.000020, 0.087000},
	{{0, 0, 9, -9, 0, 0, 0, 0, 4, -1, -4}, 4.410560, 0.000080, 0.075000},
	{{0, 0, 9, -9, 0, 0, 0, 0, 4, 0, -4}, 192.804770, 0.000530, 4535.679000},
	{{0, 0, 9, -9, 0, 0, 0, 0, 4, 1, -4}, 4.410560, 0.000080, 0.075000},
	{{0, 0, 9, -9, 0, 0, 0, 0, 6, -1, -4}, 186.670870, 0.000020, 0.087000},
	{{0, 0, 9, -8, 0, 0, 0, 0, 2, -1, 0}, 31.026460, 0.000050, 0.062000},
	{{0, 0, 9, -8, 0, 0, 0, 0, 2, 0, 0}, 211.030810, 0.000040, 0.034000},
	{{0, 0, 9, -7, 0, 0, 0, 0, 2, 0, 0}, 59.719030, 0.000010, 0.033000},
	{{0, 0, 10, -21, 2, 0, 0, 0, 0, 0, 0}, 230.142170, 0.000010, 1.003000},
	{{0, 0, 10, -21, 2, 0, 0, 0, 2, 0, 0}, 230.964600, 0.000010, 0.042000},
	{{0, 0, 10, -19, 0, 0, 0, 0, -2, 0, 0}, 98.562020, 0.000020, 0.040000},
	{{0, 0, 10, -19, 0, 0, 0, 0, -2, 1, 0}, 98.562090, 0.000030, 0.086000},
	{{0, 0, 10, -19, 0, 0, 0, 0, 0, -1, 0}, 98.307970, 0.000040, 0.075000},
	{{0, 0, 10, -19, 0, 0, 0, 0, 0, 0, 0}, 98.020060, 0.000230, 9.804000},
	{{0, 0, 10, -19, 0, 0, 0, 0, 0, 1, 0}, 98.307970, 0.000040, 0.076000},
	{{0, 0, 10, -19, 0, 0, 0, 0, 2, -1, 0}, 99.591750, 0.000030, 0.088000},
	{{0, 0, 10, -19, 0, 0, 0, 0, 2, 0, 0}, 99.591750, 0.000020, 0.041000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -4, 0, 0}, 71.287350, 0.000030, 0.020000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -4, 1, 0}, 71.287350, 0.000060, 0.028000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -4, 2, 0}, 71.287350, 0.000050, 0.044000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -2, -2, 0}, 71.287350, 0.000020, 0.020000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -2, -1, 0}, 71.287350, 0.000130, 0.026000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -2, 0, 0}, 71.287350, 0.001730, 0.040000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -2, 0, 2}, 71.287350, 0.000050, 0.475000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -2, 1, 0}, 71.287350, 0.003400, 0.087000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -2, 2, 0}, 71.287360, 0.000160, 0.564000},
	{{0, 0, 10, -19, 0, 3, 0, 0, -1, 0, 0}, 251.287390, 0.000060, 0.081000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 0, -2, 0}, 251.287250, 0.000030, 0.038000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 0, -1, 0}, 251.287240, 0.000390, 0.075000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 0, 0, 0}, 251.287310, 0.000380, 6408.190000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 0, 1, 0}, 251.287380, 0.000390, 0.075000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 0, 2, 0}, 251.287370, 0.000030, 0.038000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 1, 0, 0}, 251.287230, 0.000060, 0.081000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 2, -2, 0}, 71.287270, 0.000160, 0.564000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 2, -1, 0}, 71.287270, 0.003400, 0.087000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 2, 0, -2}, 71.287270, 0.000050, 0.474000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 2, 0, 0}, 71.287270, 0.001730, 0.040000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 2, 1, 0}, 71.287270, 0.000130, 0.026000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 2, 2, 0}, 71.287270, 0.000020, 0.020000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 4, -2, 0}, 71.287270, 0.000050, 0.044000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 4, -1, 0}, 71.287270, 0.000060, 0.028000},
	{{0, 0, 10, -19, 0, 3, 0, 0, 4, 0, 0}, 71.287270, 0.000030, 0.020000},
	{{0, 0, 10, -19, 1, 0, 0, 0, 0, 0, 0}, 288.816620, 0.000010, 56.510000},
	{{0, 0, 10, -18, 0, 0, 0, 0, -2, 0, 0}, 127.806310, 0.000010, 0.041000},
	{{0, 0, 10, -18, 0, 0, 0, 0, 0, 0, 0}, 125.557480, 0.000060, 2.327000},
	{{0, 0, 10, -17, 0, 0, 0, 0, -2, -1, 0}, 142.200720, 0.000020, 0.027000},
	{{0, 0, 10, -17, 0, 0, 0, 0, -2, 0, 0}, 142.200680, 0.000100, 0.042000},
	{{0, 0, 10, -17, 0, 0, 0, 0, -2, 1, 0}, 141.242430, 0.000120, 0.095000},
	{{0, 0, 10, -17, 0, 0, 0, 0, 0, -1, 0}, 150.066010, 0.000110, 0.081000},
	{{0, 0, 10, -17, 0, 0, 0, 0, 0, 0, 0}, 152.285270, 0.000480, 1.040000},
	{{0, 0, 10, -17, 0, 0, 0, 0, 0, 1, 0}, 152.905870, 0.000080, 0.070000},
	{{0, 0, 10, -17, 0, 0, 0, 0, 1, 0, -1}, 238.099910, 0.000030, 10.823000},
	{{0, 0, 10, -17, 0, 0, 0, 0, 2, -3, 0}, 151.682120, 0.000010, 0.071000},
	{{0, 0, 10, -17, 0, 0, 0, 0, 2, -1, 0}, 163.229640, 0.000030, 0.080000},
	{{0, 0, 10, -17, 0, 0, 0, 0, 2, 0, 0}, 160.544410, 0.000020, 0.039000},
	{{0, 0, 10, -16, 0, 0, 0, 0, -2, 0, 0}, 10.770680, 0.000020, 0.043000},
	{{0, 0, 10, -16, 0, 0, 0, 0, 0, -1, 0}, 6.627900, 0.000030, 0.085000},
	{{0, 0, 10, -16, 0, 0, 0, 0, 0, 0, 0}, 8.225040, 0.000070, 0.670000},
	{{0, 0, 10, -16, 0, 0, 0, 0, 1, 0, -1}, 78.894070, 0.000010, 2.276000},
	{{0, 0, 10, -16, 0, 0, 0, 0, 2, -2, 0}, 183.711440, 0.000010, 3.559000},
	{{0, 0, 10, -16, 0, 0, 0, 0, 2, -1, 0}, 186.627900, 0.000140, 0.077000},
	{{0, 0, 10, -16, 0, 0, 0, 0, 2, 0, 0}, 186.624230, 0.000010, 0.038000},
	{{0, 0, 10, -16, 4, 5, 0, 0, -2, 0, 0}, 46.806580, 0.000080, 0.044000},
	{{0, 0, 10, -16, 4, 5, 0, 0, -2, 1, 0}, 46.806580, 0.000060, 0.105000},
	{{0, 0, 10, -16, 4, 5, 0, 0, 0, 0, 0}, 46.806580, 0.000030, 0.500000},
	{{0, 0, 10, -15, -3, 0, 0, 0, 2, -2, 0}, 166.867520, 0.000020, 455.769000},
	{{0, 0, 10, -15, -3, 0, 0, 0, 2, -1, 0}, 193.065350, 0.000030, 0.075000},
	{{0, 0, 10, -15, 0, 0, 0, 0, -2, 0, 0}, 289.187320, 0.000060, 0.044000},
	{{0, 0, 10, -15, 0, 0, 0, 0, -2, 1, 0}, 289.143050, 0.000050, 0.106000},
	{{0, 0, 10, -15, 0, 0, 0, 0, 0, -1, 0}, 221.177290, 0.000070, 0.089000},
	{{0, 0, 10, -15, 0, 0, 0, 0, 0, 0, 0}, 348.422670, 0.000050, 0.494000},
	{{0, 0, 10, -15, 0, 0, 0, 0, 0, 1, 0}, 343.885180, 0.000010, 0.065000},
	{{0, 0, 10, -15, 0, 0, 0, 0, 1, 0, -1}, 116.631750, 0.000010, 1.030000},
	{{0, 0, 10, -15, 0, 0, 0, 0, 2, -1, 0}, 32.268850, 0.000380, 0.074000},
	{{0, 0, 10, -15, 1, 0, 0, 0, 2, -1, -2}, 248.472260, 0.000040, 0.075000},
	{{0, 0, 10, -15, 1, 0, 0, 0, 2, 0, -2}, 245.970580, 0.000210, 662.430000},
	{{0, 0, 10, -15, 1, 0, 0, 0, 2, 1, -2}, 248.472260, 0.000040, 0.075000},
	{{0, 0, 10, -14, 0, 0, 0, 0, -2, 0, 0}, 53.317920, 0.000020, 0.045000},
	{{0, 0, 10, -14, 0, 0, 0, 0, 0, -1, 0}, 244.427440, 0.000040, 0.093000},
	{{0, 0, 10, -14, 0, 0, 0, 0, 0, 0, 0}, 62.755760, 0.000080, 0.391000},
	{{0, 0, 10, -14, 0, 0, 0, 0, 0, 1, 0}, 64.471720, 0.000020, 0.063000},
	{{0, 0, 10, -14, 0, 0, 0, 0, 1, 0, -1}, 146.546040, 0.000010, 0.666000},
	{{0, 0, 10, -14, 0, 0, 0, 0, 2, -2, 0}, 64.458280, 0.000010, 1.278000},
	{{0, 0, 10, -14, 0, 0, 0, 0, 2, -1, 0}, 64.443710, 0.000260, 0.071000},
	{{0, 0, 10, -14, 0, 0, 0, 0, 2, 0, -2}, 64.439250, 0.000010, 2.228000},
	{{0, 0, 10, -14, 0, 0, 0, 0, 2, 0, 0}, 244.438220, 0.000040, 0.037000},
	{{0, 0, 10, -13, 0, 0, 0, 0, -2, 0, 0}, 91.534310, 0.000020, 0.046000},
	{{0, 0, 10, -13, 0, 0, 0, 0, 0, -1, 0}, 273.450320, 0.000030, 0.098000},
	{{0, 0, 10, -13, 0, 0, 0, 0, 0, 0, 0}, 92.530910, 0.000110, 0.324000},
	{{0, 0, 10, -13, 0, 0, 0, 0, 0, 1, 0}, 93.274980, 0.000020, 0.061000},
	{{0, 0, 10, -13, 0, 0, 0, 0, 1, 0, -1}, 177.215920, 0.000010, 0.492000},
	{{0, 0, 10, -13, 0, 0, 0, 0, 2, -2, 0}, 93.383240, 0.000010, 0.761000},
	{{0, 0, 10, -13, 0, 0, 0, 0, 2, -1, 0}, 93.375400, 0.000290, 0.069000},
	{{0, 0, 10, -13, 0, 0, 0, 0, 2, 0, 0}, 273.368400, 0.000080, 0.036000},
	{{0, 0, 10, -13, 0, 0, 0, 0, 2, 1, 0}, 273.375800, 0.000010, 0.024000},
	{{0, 0, 10, -12, 0, 0, 0, 0, -2, 0, 0}, 119.034090, 0.000030, 0.047000},
	{{0, 0, 10, -12, 0, 0, 0, 0, 0, -1, 0}, 303.356120, 0.000030, 0.104000},
	{{0, 0, 10, -12, 0, 0, 0, 0, 0, 0, 0}, 120.374400, 0.000140, 0.276000},
	{{0, 0, 10, -12, 0, 0, 0, 0, 0, 1, 0}, 121.964090, 0.000020, 0.059000},
	{{0, 0, 10, -12, 0, 0, 0, 0, 1, 0, -1}, 209.571100, 0.000010, 0.390000},
	{{0, 0, 10, -12, 0, 0, 0, 0, 2, -2, 0}, 122.278970, 0.000020, 0.542000},
	{{0, 0, 10, -12, 0, 0, 0, 0, 2, -1, 0}, 122.729650, 0.000270, 0.066000},
	{{0, 0, 10, -12, 0, 0, 0, 0, 2, 0, 0}, 302.231050, 0.000100, 0.035000},
	{{0, 0, 10, -12, 0, 0, 0, 0, 2, 1, 0}, 302.279290, 0.000030, 0.024000},
	{{0, 0, 10, -11, -2, 0, 0, 0, 0, 0, 0}, 272.092630, 0.000020, 0.251000},
	{{0, 0, 10, -11, -1, 0, 0, 0, 0, 0, 0}, 118.344550, 0.000010, 0.246000},
	{{0, 0, 10, -11, 0, 0, 0, 0, -2, 0, 0}, 139.446200, 0.000020, 0.049000},
	{{0, 0, 10, -11, 0, 0, 0, 0, 0, -1, 0}, 333.318140, 0.000020, 0.110000},
	{{0, 0, 10, -11, 0, 0, 0, 0, 0, 0, 0}, 148.867520, 0.000110, 0.241000},
	{{0, 0, 10, -11, 0, 0, 0, 0, 0, 1, 0}, 150.454390, 0.000020, 0.057000},
	{{0, 0, 10, -11, 0, 0, 0, 0, 2, -2, 0}, 150.995620, 0.000010, 0.421000},
	{{0, 0, 10, -11, 0, 0, 0, 0, 2, -1, 0}, 151.832320, 0.000210, 0.064000},
	{{0, 0, 10, -11, 0, 0, 0, 0, 2, 0, 0}, 330.927890, 0.000090, 0.035000},
	{{0, 0, 10, -11, 0, 0, 0, 0, 2, 1, 0}, 330.996280, 0.000030, 0.024000},
	{{0, 0, 10, -10, 0, 0, 0, 0, 0, -1, 0}, 178.489450, 0.000010, 0.117000},
	{{0, 0, 10, -10, 0, 0, 0, 0, 0, 0, 0}, 175.367270, 0.000030, 0.214000},
	{{0, 0, 10, -10, 0, 0, 0, 0, 0, 1, 0}, 178.489450, 0.000010, 0.056000},
	{{0, 0, 10, -10, 0, 0, 0, 0, 2, -1, 0}, 178.485730, 0.000060, 0.062000},
	{{0, 0, 10, -10, 0, 0, 0, 0, 2, 0, 0}, 358.483400, 0.000050, 0.034000},
	{{0, 0, 10, -9, 0, 0, 0, 0, 2, -1, 0}, 31.271440, 0.000040, 0.060000},
	{{0, 0, 10, -9, 0, 0, 0, 0, 2, 0, 0}, 211.275460, 0.000030, 0.033000},
	{{0, 0, 11, -21, 0, 0, 0, 0, 0, -1, 1}, 8.026770, 0.000010, 720.555000},
	{{0, 0, 11, -21, 0, 0, 0, 0, 0, 0, 0}, 66.343130, 0.000030, 6.047000},
	{{0, 0, 11, -21, 2, 0, 0, 0, -2, 0, 0}, 128.136410, 0.000110, 0.040000},
	{{0, 0, 11, -21, 2, 0, 0, 0, -2, 1, 0}, 129.487100, 0.000160, 0.087000},
	{{0, 0, 11, -21, 2, 0, 0, 0, 0, -2, 0}, 122.890950, 0.000010, 0.038000},
	{{0, 0, 11, -21, 2, 0, 0, 0, 0, -1, 0}, 123.520870, 0.000240, 0.075000},
	{{0, 0, 11, -21, 2, 0, 0, 0, 0, 0, 0}, 126.340250, 0.001180, 308.674000},
	{{0, 0, 11, -21, 2, 0, 0, 0, 0, 1, 0}, 123.511840, 0.000240, 0.075000},
	{{0, 0, 11, -21, 2, 0, 0, 0, 0, 2, 0}, 122.892600, 0.000010, 0.038000},
	{{0, 0, 11, -21, 2, 0, 0, 0, 2, -1, 0}, 129.455630, 0.000160, 0.087000},
	{{0, 0, 11, -21, 2, 0, 0, 0, 2, 0, 0}, 128.107540, 0.000110, 0.040000},
	{{0, 0, 11, -20, 0, 0, 0, 0, 0, 0, 0}, 97.367710, 0.000030, 2.730000},
	{{0, 0, 11, -19, 0, 0, 0, 0, -2, 0, 0}, 123.836800, 0.000020, 0.042000},
	{{0, 0, 11, -19, 0, 0, 0, 0, -2, 1, 0}, 123.836840, 0.000020, 0.094000},
	{{0, 0, 11, -19, 0, 0, 0, 0, 0, 0, 0}, 126.925810, 0.000080, 1.114000},
	{{0, 0, 11, -19, 0, 0, 0, 0, 1, -1, 0}, 129.033480, 0.000040, 90.987000},
	{{0, 0, 11, -19, 0, 3, 0, 0, -2, -1, 0}, 328.350410, 0.000030, 0.027000},
	{{0, 0, 11, -19, 0, 3, 0, 0, -2, 0, 0}, 328.350410, 0.000170, 0.042000},
	{{0, 0, 11, -19, 0, 3, 0, 0, -2, 1, 0}, 328.350410, 0.000230, 0.095000},
	{{0, 0, 11, -19, 0, 3, 0, 0, 0, -1, 0}, 328.350490, 0.000060, 0.082000},
	{{0, 0, 11, -19, 0, 3, 0, 0, 0, 0, 0}, 328.350450, 0.000240, 1.000000},
	{{0, 0, 11, -19, 0, 3, 0, 0, 0, 1, 0}, 328.350520, 0.000030, 0.070000},
	{{0, 0, 11, -18, 0, 0, 0, 0, -2, 0, 0}, 341.283030, 0.000020, 0.043000},
	{{0, 0, 11, -18, 0, 0, 0, 0, 0, 0, 0}, 339.517540, 0.000030, 0.699000},
	{{0, 0, 11, -18, 0, 0, 0, 0, 2, -1, 0}, 157.703990, 0.000050, 0.077000},
	{{0, 0, 11, -17, 0, 0, 0, 0, -2, 0, 0}, 30.040530, 0.000020, 0.044000},
	{{0, 0, 11, -17, 0, 0, 0, 0, 0, -1, 0}, 186.349270, 0.000050, 0.089000},
	{{0, 0, 11, -17, 0, 0, 0, 0, 0, 0, 0}, 19.239340, 0.000040, 0.510000},
	{{0, 0, 11, -17, 0, 0, 0, 0, 2, -1, 0}, 6.349250, 0.000230, 0.074000},
	{{0, 0, 11, -17, 0, 0, 0, 0, 2, 0, 0}, 186.561330, 0.000020, 0.037000},
	{{0, 0, 11, -16, 0, 0, 0, 0, 0, -1, 0}, 215.407560, 0.000020, 0.093000},
	{{0, 0, 11, -16, 0, 0, 0, 0, 0, 0, 0}, 33.998890, 0.000040, 0.401000},
	{{0, 0, 11, -16, 0, 0, 0, 0, 0, 1, 0}, 35.599920, 0.000010, 0.063000},
	{{0, 0, 11, -16, 0, 0, 0, 0, 2, -1, 0}, 35.474110, 0.000130, 0.072000},
	{{0, 0, 11, -16, 0, 0, 0, 0, 2, 0, 0}, 215.468330, 0.000040, 0.037000},
	{{0, 0, 11, -15, 0, 0, 0, 0, -2, 0, 0}, 77.659650, 0.000020, 0.046000},
	{{0, 0, 11, -15, 0, 0, 0, 0, 0, -1, 0}, 244.392630, 0.000020, 0.098000},
	{{0, 0, 11, -15, 0, 0, 0, 0, 0, 0, 0}, 66.627620, 0.000050, 0.331000},
	{{0, 0, 11, -15, 0, 0, 0, 0, 0, 1, 0}, 64.399070, 0.000020, 0.061000},
	{{0, 0, 11, -15, 0, 0, 0, 0, 2, -1, 0}, 63.835040, 0.000150, 0.069000},
	{{0, 0, 11, -15, 0, 0, 0, 0, 2, 0, 0}, 244.391980, 0.000050, 0.036000},
	{{0, 0, 11, -14, 0, 0, 0, 0, -2, 0, 0}, 84.187390, 0.000030, 0.047000},
	{{0, 0, 11, -14, 0, 0, 0, 0, 0, -1, 0}, 273.416440, 0.000020, 0.103000},
	{{0, 0, 11, -14, 0, 0, 0, 0, 0, 0, 0}, 91.677830, 0.000070, 0.281000},
	{{0, 0, 11, -14, 0, 0, 0, 0, 0, 1, 0}, 86.373420, 0.000010, 0.059000},
	{{0, 0, 11, -14, 0, 0, 0, 0, 1, 0, -1}, 177.878980, 0.000010, 0.400000},
	{{0, 0, 11, -14, 0, 0, 0, 0, 2, -1, 0}, 93.298280, 0.000180, 0.066000},
	{{0, 0, 11, -14, 0, 0, 0, 0, 2, 0, 0}, 273.293810, 0.000090, 0.035000},
	{{0, 0, 11, -14, 0, 0, 0, 0, 4, -4, 0}, 273.309150, 0.000010, 119.249000},
	{{0, 0, 11, -13, 0, 0, 0, 0, -2, 0, 0}, 114.900800, 0.000030, 0.048000},
	{{0, 0, 11, -13, 0, 0, 0, 0, 0, -1, 0}, 302.135830, 0.000020, 0.109000},
	{{0, 0, 11, -13, 0, 0, 0, 0, 0, 0, 0}, 120.060660, 0.000090, 0.245000},
	{{0, 0, 11, -13, 0, 0, 0, 0, 0, 1, 0}, 122.135370, 0.000010, 0.058000},
	{{0, 0, 11, -13, 0, 0, 0, 0, 2, -2, 0}, 122.136160, 0.000010, 0.432000},
	{{0, 0, 11, -13, 0, 0, 0, 0, 2, -1, 0}, 122.136530, 0.000170, 0.064000},
	{{0, 0, 11, -13, 0, 0, 0, 0, 2, 0, 0}, 302.173380, 0.000080, 0.035000},
	{{0, 0, 11, -13, 0, 0, 0, 0, 2, 1, 0}, 302.136510, 0.000020, 0.024000},
	{{0, 0, 11, -12, 0, 0, 0, 0, -2, 0, 0}, 139.863820, 0.000020, 0.050000},
	{{0, 0, 11, -12, 0, 0, 0, 0, 0, -1, 0}, 316.565130, 0.000010, 0.116000},
	{{0, 0, 11, -12, 0, 0, 0, 0, 0, 0, 0}, 147.874370, 0.000060, 0.216000},
	{{0, 0, 11, -12, 0, 0, 0, 0, 2, -1, 0}, 150.678740, 0.000120, 0.062000},
	{{0, 0, 11, -12, 0, 0, 0, 0, 2, 0, 0}, 330.678100, 0.000090, 0.034000},
	{{0, 0, 11, -12, 0, 0, 0, 0, 2, 1, 0}, 330.678680, 0.000020, 0.023000},
	{{0, 0, 11, -11, 0, 0, 0, 0, 0, 0, 0}, 168.799780, 0.000010, 0.194000},
	{{0, 0, 11, -11, 0, 0, 0, 0, 2, -1, 0}, 176.748770, 0.000030, 0.060000},
	{{0, 0, 11, -11, 0, 0, 0, 0, 2, 0, 0}, 356.746890, 0.000040, 0.033000},
	{{0, 0, 11, -10, 0, 0, 0, 0, 2, -1, 0}, 29.144030, 0.000030, 0.058000},
	{{0, 0, 11, -10, 0, 0, 0, 0, 2, 0, 0}, 211.352200, 0.000040, 0.033000},
	{{0, 0, 12, -22, 0, 0, 0, 0, 0, 0, 0}, 68.378070, 0.000020, 3.301000},
	{{0, 0, 12, -21, 0, 0, 0, 0, 0, 0, 0}, 98.112810, 0.000020, 1.198000},
	{{0, 0, 12, -21, 2, 0, 0, 0, -2, 0, 0}, 12.624880, 0.000020, 0.042000},
	{{0, 0, 12, -21, 2, 0, 0, 0, 0, 0, 0}, 357.919700, 0.000020, 0.997000},
	{{0, 0, 12, -20, 0, 0, 0, 0, 0, 0, 0}, 310.073910, 0.000010, 0.732000},
	{{0, 0, 12, -20, 0, 0, 0, 0, 2, -1, 0}, 124.833640, 0.000020, 0.078000},
	{{0, 0, 12, -19, 0, 0, 0, 0, 0, -1, 0}, 157.624060, 0.000030, 0.088000},
	{{0, 0, 12, -19, 0, 0, 0, 0, 2, -2, 0}, 157.584960, 0.000010, 8.066000},
	{{0, 0, 12, -19, 0, 0, 0, 0, 2, -1, 0}, 337.624060, 0.000150, 0.075000},
	{{0, 0, 12, -19, 0, 3, 0, 0, -2, 0, 0}, 225.413470, 0.000010, 0.044000},
	{{0, 0, 12, -18, 0, 0, 0, 0, 0, 0, 0}, 7.768900, 0.000010, 0.412000},
	{{0, 0, 12, -18, 0, 0, 0, 0, 2, -1, 0}, 5.687770, 0.000080, 0.072000},
	{{0, 0, 12, -18, 0, 0, 0, 0, 2, 0, 0}, 186.497310, 0.000020, 0.037000},
	{{0, 0, 12, -17, 0, 0, 0, 0, 0, 0, 0}, 39.460900, 0.000020, 0.338000},
	{{0, 0, 12, -17, 0, 0, 0, 0, 0, 1, 0}, 31.916380, 0.000010, 0.062000},
	{{0, 0, 12, -17, 0, 0, 0, 0, 2, -1, 0}, 35.424000, 0.000080, 0.069000},
	{{0, 0, 12, -17, 0, 0, 0, 0, 2, 0, 0}, 215.419170, 0.000040, 0.036000},
	{{0, 0, 12, -16, 0, 0, 0, 0, 0, -1, 0}, 244.351720, 0.000010, 0.102000},
	{{0, 0, 12, -16, 0, 0, 0, 0, 0, 0, 0}, 63.687560, 0.000030, 0.286000},
	{{0, 0, 12, -16, 0, 0, 0, 0, 0, 1, 0}, 64.314690, 0.000010, 0.060000},
	{{0, 0, 12, -16, 0, 0, 0, 0, 2, -1, 0}, 64.330770, 0.000100, 0.067000},
	{{0, 0, 12, -16, 0, 0, 0, 0, 2, 0, 0}, 244.326810, 0.000050, 0.035000},
	{{0, 0, 12, -15, 0, 0, 0, 0, 0, -1, 0}, 273.359120, 0.000010, 0.108000},
	{{0, 0, 12, -15, 0, 0, 0, 0, 0, 0, 0}, 91.609280, 0.000030, 0.248000},
	{{0, 0, 12, -15, 0, 0, 0, 0, 0, 1, 0}, 84.075590, 0.000010, 0.058000},
	{{0, 0, 12, -15, 0, 0, 0, 0, 2, -1, 0}, 93.192580, 0.000120, 0.064000},
	{{0, 0, 12, -15, 0, 0, 0, 0, 2, 0, 0}, 274.505990, 0.000070, 0.035000},
	{{0, 0, 12, -14, 0, 0, 0, 0, 0, -1, 0}, 301.945640, 0.000010, 0.115000},
	{{0, 0, 12, -14, 0, 0, 0, 0, 0, 0, 0}, 119.892930, 0.000050, 0.219000},
	{{0, 0, 12, -14, 0, 0, 0, 0, 2, -1, 0}, 121.946570, 0.000110, 0.062000},
	{{0, 0, 12, -14, 0, 0, 0, 0, 2, 0, 0}, 301.946240, 0.000090, 0.034000},
	{{0, 0, 12, -14, 0, 0, 0, 0, 2, 1, 0}, 301.946540, 0.000020, 0.024000},
	{{0, 0, 12, -13, 0, 0, 0, 0, 0, 0, 0}, 145.862040, 0.000020, 0.197000},
	{{0, 0, 12, -13, 0, 0, 0, 0, 2, -1, 0}, 150.253830, 0.000080, 0.060000},
	{{0, 0, 12, -13, 0, 0, 0, 0, 2, 0, 0}, 330.253030, 0.000070, 0.034000},
	{{0, 0, 12, -12, 0, 0, 0, 0, 2, 0, 0}, 353.138410, 0.000020, 0.033000},
	{{0, 0, 12, -11, 0, 0, 0, 0, 2, -1, 0}, 31.327310, 0.000020, 0.057000},
	{{0, 0, 12, -11, 0, 0, 0, 0, 2, 0, 0}, 211.279230, 0.000030, 0.032000},
	{{0, 0, 13, -21, 0, 0, 0, 0, 0, -1, 0}, 128.263630, 0.000030, 0.088000},
	{{0, 0, 13, -21, 0, 0, 0, 0, 2, -2, 0}, 128.673240, 0.000030, 16.498000},
	{{0, 0, 13, -21, 0, 0, 0, 0, 2, -1, 0}, 308.263670, 0.000130, 0.075000},
	{{0, 0, 13, -20, 0, 0, 0, 0, 2, -1, 0}, 336.061230, 0.000040, 0.072000},
	{{0, 0, 13, -20, 0, 0, 0, 0, 2, 0, 0}, 157.524950, 0.000020, 0.037000},
	{{0, 0, 13, -19, 0, 0, 0, 0, 2, -1, 0}, 6.457320, 0.000050, 0.070000},
	{{0, 0, 13, -19, 0, 0, 0, 0, 2, 0, 0}, 186.437850, 0.000020, 0.036000},
	{{0, 0, 13, -18, 0, 0, 0, 0, 0, 0, 0}, 35.524890, 0.000010, 0.292000},
	{{0, 0, 13, -18, 0, 0, 0, 0, 2, -1, 0}, 35.363420, 0.000060, 0.067000},
	{{0, 0, 13, -18, 0, 0, 0, 0, 2, 0, 0}, 215.359170, 0.000030, 0.036000},
	{{0, 0, 13, -17, 0, 0, 0, 0, 0, 0, 0}, 63.314070, 0.000020, 0.252000},
	{{0, 0, 13, -17, 0, 0, 0, 0, 0, 1, 0}, 64.213900, 0.000010, 0.058000},
	{{0, 0, 13, -17, 0, 0, 0, 0, 2, -1, 0}, 64.245120, 0.000070, 0.065000},
	{{0, 0, 13, -17, 0, 0, 0, 0, 2, 0, 0}, 244.241350, 0.000040, 0.035000},
	{{0, 0, 13, -16, 0, 0, 0, 0, 0, 0, 0}, 91.079860, 0.000020, 0.223000},
	{{0, 0, 13, -16, 0, 0, 0, 0, 0, 1, 0}, 106.642650, 0.000020, 0.056000},
	{{0, 0, 13, -16, 0, 0, 0, 0, 2, -1, 0}, 93.068960, 0.000080, 0.063000},
	{{0, 0, 13, -16, 0, 0, 0, 0, 2, 0, 0}, 273.068750, 0.000060, 0.034000},
	{{0, 0, 13, -15, 0, 0, 0, 0, 0, 0, 0}, 118.464350, 0.000020, 0.199000},
	{{0, 0, 13, -15, 0, 0, 0, 0, 2, -1, 0}, 121.703740, 0.000080, 0.061000},
	{{0, 0, 13, -15, 0, 0, 0, 0, 2, 0, 0}, 301.703350, 0.000060, 0.034000},
	{{0, 0, 13, -14, 0, 0, 0, 0, 0, 0, 0}, 144.202600, 0.000010, 0.180000},
	{{0, 0, 13, -14, 0, 0, 0, 0, 2, -1, 0}, 149.691650, 0.000040, 0.059000},
	{{0, 0, 13, -14, 0, 0, 0, 0, 2, 0, 0}, 329.690710, 0.000040, 0.033000},
	{{0, 0, 13, -12, 0, 0, 0, 0, 2, -1, 0}, 31.052550, 0.000020, 0.055000},
	{{0, 0, 13, -12, 0, 0, 0, 0, 2, 0, 0}, 211.053760, 0.000020, 0.032000},
	{{0, 0, 13, -11, 0, 0, 0, 0, 2, 0, 0}, 63.905120, 0.000020, 0.031000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 0, -3, 0}, 98.664130, 0.000010, 0.026000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 0, -2, 0}, 98.627040, 0.000070, 0.040000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 0, -1, 0}, 278.672990, 0.000150, 0.087000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 0, 0, 0}, 278.669720, 0.000020, 0.565000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 2, -4, 0}, 97.859320, 0.000020, 0.038000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 2, -3, 0}, 97.857760, 0.000250, 0.075000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 2, -2, -2}, 278.664130, 0.000020, 0.037000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 2, -2, 0}, 98.664130, 0.008000, 362.998000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 2, -2, 2}, 278.664130, 0.000020, 0.037000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 2, -1, 0}, 98.668190, 0.001620, 0.075000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 2, 0, 0}, 98.668150, 0.000110, 0.038000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 4, -3, 0}, 98.986980, 0.000130, 0.087000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 4, -2, 0}, 98.689060, 0.000110, 0.040000},
	{{0, 0, 14, -23, 0, 0, 0, 0, 4, -1, 0}, 98.664130, 0.000010, 0.026000},
	{{0, 0, 14, -23, 4, 0, 0, 0, 2, 0, -2}, 70.284120, 0.000070, 1053.903000},
	{{0, 0, 14, -22, 0, 0, 0, 0, 2, -1, 0}, 306.348880, 0.000020, 0.073000},
	{{0, 0, 14, -21, 0, 0, 0, 0, 2, -1, 0}, 337.499060, 0.000020, 0.070000},
	{{0, 0, 14, -21, 0, 0, 0, 0, 2, 0, 0}, 157.470370, 0.000020, 0.036000},
	{{0, 0, 14, -21, 0, 0, 0, 0, 3, -2, -1}, 238.328400, 0.000020, 146.030000},
	{{0, 0, 14, -20, 0, 0, 0, 0, 2, -1, 0}, 6.396970, 0.000030, 0.067000},
	{{0, 0, 14, -20, 0, 0, 0, 0, 2, 0, 0}, 186.381920, 0.000020, 0.036000},
	{{0, 0, 14, -19, 0, 0, 0, 0, 0, 0, 0}, 35.227750, 0.000020, 0.257000},
	{{0, 0, 14, -19, 0, 0, 0, 0, 2, -1, 0}, 35.288470, 0.000040, 0.065000},
	{{0, 0, 14, -19, 0, 0, 0, 0, 2, 0, 0}, 215.284860, 0.000030, 0.035000},
	{{0, 0, 14, -18, 0, 0, 0, 0, 0, 0, 0}, 62.863140, 0.000010, 0.226000},
	{{0, 0, 14, -18, 0, 0, 0, 0, 2, -1, 0}, 66.102690, 0.000050, 0.063000},
	{{0, 0, 14, -18, 0, 0, 0, 0, 2, 0, 0}, 245.761300, 0.000050, 0.034000},
	{{0, 0, 14, -17, 0, 0, 0, 0, 0, -1, 0}, 92.899130, 0.000010, 0.121000},
	{{0, 0, 14, -17, 0, 0, 0, 0, 0, 0, 0}, 90.424940, 0.000010, 0.202000},
	{{0, 0, 14, -17, 0, 0, 0, 0, 0, 1, 0}, 92.899130, 0.000010, 0.055000},
	{{0, 0, 14, -17, 0, 0, 0, 0, 2, -1, 0}, 92.898780, 0.000050, 0.061000},
	{{0, 0, 14, -17, 0, 0, 0, 0, 2, 0, 0}, 272.898550, 0.000050, 0.034000},
	{{0, 0, 14, -16, 0, 0, 0, 0, 0, 0, 0}, 117.408030, 0.000010, 0.182000},
	{{0, 0, 14, -16, 0, 0, 0, 0, 2, -1, 0}, 121.400800, 0.000040, 0.059000},
	{{0, 0, 14, -16, 0, 0, 0, 0, 2, 0, 0}, 301.400360, 0.000050, 0.033000},
	{{0, 0, 14, -15, 0, 0, 0, 0, 2, -1, 0}, 149.661420, 0.000020, 0.057000},
	{{0, 0, 14, -15, 0, 0, 0, 0, 2, 0, 0}, 328.944800, 0.000040, 0.033000},
	{{0, 0, 14, -13, 0, 0, 0, 0, 2, 0, 0}, 210.660320, 0.000010, 0.031000},
	{{0, 0, 15, -25, 0, 0, 0, 0, 2, -1, 0}, 67.303160, 0.000020, 0.076000},
	{{0, 0, 15, -22, 0, 0, 0, 0, 2, 0, 0}, 157.412960, 0.000020, 0.036000},
	{{0, 0, 15, -21, 0, 0, 0, 0, 2, -1, 0}, 6.315680, 0.000020, 0.065000},
	{{0, 0, 15, -21, 0, 0, 0, 0, 2, 0, 0}, 186.315650, 0.000020, 0.035000},
	{{0, 0, 15, -20, 0, 0, 0, 0, 2, -1, 0}, 35.193630, 0.000030, 0.063000},
	{{0, 0, 15, -20, 0, 0, 0, 0, 2, 0, 0}, 215.193860, 0.000030, 0.034000},
	{{0, 0, 15, -19, 0, 0, 0, 0, 2, -1, 0}, 64.010400, 0.000030, 0.061000},
	{{0, 0, 15, -19, 0, 0, 0, 0, 2, 0, 0}, 244.010320, 0.000040, 0.034000},
	{{0, 0, 15, -18, 0, 0, 0, 0, 2, -1, 0}, 92.690900, 0.000040, 0.059000},
	{{0, 0, 15, -18, 0, 0, 0, 0, 2, 0, 0}, 272.690760, 0.000050, 0.033000},
	{{0, 0, 15, -17, 0, 0, 0, 0, 2, -1, 0}, 121.028300, 0.000030, 0.057000},
	{{0, 0, 15, -17, 0, 0, 0, 0, 2, 0, 0}, 301.027790, 0.000030, 0.033000},
	{{0, 0, 15, -16, 0, 0, 0, 0, 2, -1, 0}, 147.937130, 0.000020, 0.056000},
	{{0, 0, 15, -16, 0, 0, 0, 0, 2, 0, 0}, 327.936620, 0.000030, 0.032000},
	{{0, 0, 15, -14, 0, 0, 0, 0, 2, 0, 0}, 206.560970, 0.000010, 0.031000},
	{{0, 0, 16, -23, 0, 0, 0, 0, 2, 0, 0}, 157.348970, 0.000020, 0.035000},
	{{0, 0, 16, -22, 0, 0, 0, 0, 2, -1, 0}, 6.236630, 0.000020, 0.063000},
	{{0, 0, 16, -22, 0, 0, 0, 0, 2, 0, 0}, 186.236600, 0.000010, 0.034000},
	{{0, 0, 16, -21, 0, 0, 0, 0, 2, -1, 0}, 35.085580, 0.000030, 0.061000},
	{{0, 0, 16, -21, 0, 0, 0, 0, 2, 0, 0}, 215.085530, 0.000020, 0.034000},
	{{0, 0, 16, -20, 0, 0, 0, 0, 2, -1, 0}, 63.851120, 0.000030, 0.059000},
	{{0, 0, 16, -20, 0, 0, 0, 0, 2, 0, 0}, 243.851700, 0.000040, 0.033000},
	{{0, 0, 16, -19, 0, 0, 0, 0, 0, 0, 0}, 92.441580, 0.000020, 0.170000},
	{{0, 0, 16, -19, 0, 0, 0, 0, 2, -1, 0}, 92.440130, 0.000030, 0.058000},
	{{0, 0, 16, -19, 0, 0, 0, 0, 2, 0, 0}, 272.441410, 0.000040, 0.033000},
	{{0, 0, 16, -18, 0, 0, 0, 0, 2, -1, 0}, 120.569530, 0.000020, 0.056000},
	{{0, 0, 16, -18, 0, 0, 0, 0, 2, 0, 0}, 300.573190, 0.000040, 0.032000},
	{{0, 0, 16, -17, 0, 0, 0, 0, 2, 0, 0}, 326.527400, 0.000020, 0.032000},
	{{0, 0, 16, -15, 0, 0, 0, 0, 2, 0, 0}, 209.206690, 0.000020, 0.031000},
	{{0, 0, 16, -5, -1, 0, 0, 0, 0, -1, 0}, 272.097580, 0.000020, 658.208000},
	{{0, 0, 17, -32, 0, 0, 0, 0, 0, 0, 0}, 282.422850, 0.000040, 71.966000},
	{{0, 0, 17, -30, 0, 0, 0, 0, 1, 0, -1}, 243.793780, 0.000030, 232.951000},
	{{0, 0, 17, -28, 0, 0, 0, 0, 2, 0, -2}, 221.726260, 0.000050, 188.324000},
	{{0, 0, 17, -24, 0, 0, 0, 0, 2, 0, 0}, 157.275780, 0.000020, 0.035000},
	{{0, 0, 17, -23, 0, 0, 0, 0, 2, 0, 0}, 186.142220, 0.000020, 0.034000},
	{{0, 0, 17, -22, 0, 0, 0, 0, 2, -1, 0}, 34.956590, 0.000020, 0.060000},
	{{0, 0, 17, -22, 0, 0, 0, 0, 2, 0, 0}, 214.956540, 0.000020, 0.033000},
	{{0, 0, 17, -21, 0, 0, 0, 0, 2, -1, 0}, 63.663190, 0.000020, 0.058000},
	{{0, 0, 17, -21, 0, 0, 0, 0, 2, 0, 0}, 243.663920, 0.000030, 0.033000},
	{{0, 0, 17, -20, 0, 0, 0, 0, 2, -1, 0}, 92.146200, 0.000030, 0.056000},
	{{0, 0, 17, -20, 0, 0, 0, 0, 2, 0, 0}, 272.146080, 0.000030, 0.032000},
	{{0, 0, 17, -19, 0, 0, 0, 0, 2, -1, 0}, 120.018850, 0.000020, 0.054000},
	{{0, 0, 17, -19, 0, 0, 0, 0, 2, 0, 0}, 300.018610, 0.000020, 0.032000},
	{{0, 0, 17, -18, 0, 0, 0, 0, 2, 0, 0}, 327.217100, 0.000010, 0.031000},
	{{0, 0, 18, -24, 0, 0, 0, 0, 2, 0, 0}, 186.032300, 0.000010, 0.033000},
	{{0, 0, 18, -23, 0, 0, 0, 0, 2, -1, 0}, 34.805440, 0.000020, 0.058000},
	{{0, 0, 18, -23, 0, 0, 0, 0, 2, 0, 0}, 214.805390, 0.000020, 0.033000},
	{{0, 0, 18, -22, 0, 0, 0, 0, 2, -1, 0}, 63.444740, 0.000020, 0.056000},
	{{0, 0, 18, -22, 0, 0, 0, 0, 2, 0, 0}, 243.444650, 0.000020, 0.032000},
	{{0, 0, 18, -21, 0, 0, 0, 0, 2, 0, 0}, 271.799150, 0.000020, 0.032000},
	{{0, 0, 18, -20, 0, 0, 0, 0, 2, 0, 0}, 299.339570, 0.000020, 0.031000},
	{{0, 0, 19, -25, 0, 0, 0, 0, 2, 0, 0}, 185.904880, 0.000010, 0.033000},
	{{0, 0, 19, -24, 0, 0, 0, 0, 2, 0, 0}, 214.630550, 0.000010, 0.032000},
	{{0, 0, 19, -23, 0, 0, 0, 0, 2, 0, 0}, 243.191220, 0.000020, 0.032000},
	{{0, 0, 19, -22, 0, 0, 0, 0, 2, 0, 0}, 271.393540, 0.000020, 0.031000},
	{{0, 0, 19, -21, 0, 0, 0, 0, 2, 0, 0}, 298.499920, 0.000010, 0.031000},
	{{0, 0, 19, -17, -3, 0, 0, 0, -4, 3, 0}, 255.593860, 0.000030, 1361.910000},
	{{0, 0, 20, -26, 0, 0, 0, 0, 2, 0, 0}, 185.759720, 0.000020, 0.032000},
	{{0, 0, 20, -25, 0, 0, 0, 0, 2, 0, 0}, 214.430430, 0.000010, 0.032000},
	{{0, 0, 20, -24, 0, 0, 0, 0, 2, 0, 0}, 242.900490, 0.000010, 0.031000},
	{{0, 0, 20, -23, 0, 0, 0, 0, 2, 0, 0}, 270.914980, 0.000020, 0.031000},
	{{0, 0, 20, -18, 0, 0, 0, 0, -3, 1, 1}, 140.454600, 0.000160, 584.305000},
	{{0, 0, 20, -16, 0, 0, -1, 0, -2, 1, 0}, 105.223220, 0.000020, 1701.651000},
	{{0, 0, 20, -16, 0, 0, 0, 0, -4, 1, 0}, 327.067010, 0.000020, 0.040000},
	{{0, 0, 20, -16, 0, 0, 0, 0, -4, 2, 0}, 327.067010, 0.000020, 0.087000},
	{{0, 0, 20, -16, 0, 0, 0, 0, -2, 0, 0}, 327.021070, 0.000070, 0.076000},
	{{0, 0, 20, -16, 0, 0, 0, 0, -2, 1, 0}, 327.067010, 0.001480, 88.385000},
	{{0, 0, 20, -16, 0, 0, 0, 0, -2, 2, 0}, 327.104610, 0.000090, 0.075000},
	{{0, 0, 20, -16, 0, 0, 0, 0, 0, 0, 0}, 327.067010, 0.000020, 0.087000},
	{{0, 0, 20, -16, 0, 0, 0, 0, 0, 1, 0}, 327.067010, 0.000020, 0.040000},
	{{0, 0, 20, -13, 2, 0, 0, 0, 0, -1, 0}, 59.402550, 0.000020, 1043.255000},
	{{0, 0, 21, -26, 0, 0, 0, 0, 2, 0, 0}, 214.201770, 0.000010, 0.031000},
	{{0, 0, 21, -25, 0, 0, 0, 0, 2, 0, 0}, 242.566050, 0.000010, 0.031000},
	{{0, 0, 21, -24, 0, 0, 0, 0, 2, 0, 0}, 270.360930, 0.000010, 0.030000},
	{{0, 0, 21, -18, 0, 0, 0, 0, -2, 1, 0}, 15.764310, 0.000140, 19.211000},
	{{0, 0, 22, -19, -5, 0, 0, 0, -2, 1, 0}, 87.782730, 0.000010, 190.106000},
	{{0, 0, 23, -21, -4, 0, 0, 0, -2, 1, 0}, 326.309020, 0.000010, 63.795000},
	{{0, 0, 23, -20, 0, 0, 0, 0, -1, 0, 0}, 100.984870, 0.000090, 411.317000},
	{{0, 0, 23, -20, 2, 0, 0, 0, -1, 1, -1}, 258.278900, 0.000080, 1726.003000},
	{{0, 0, 26, -27, -2, 0, 0, 0, -2, 0, 0}, 268.214230, 0.000010, 0.075000},
	{{0, 0, 26, -27, -2, 0, 0, 0, -2, 1, 0}, 356.237470, 0.000010, 171.793000},
	{{0, 0, 26, -27, -2, 0, 0, 0, -2, 2, 0}, 268.214230, 0.000010, 0.075000},
	{{0, 0, 26, -27, -1, 0, 0, 0, -2, 1, 0}, 103.459620, 0.000010, 12.742000},
	{{0, 0, 26, -27, 0, 0, 0, 0, -2, 1, 0}, 336.957000, 0.000010, 6.143000},
	{{0, 0, 26, -24, 0, 0, 0, 0, 0, -1, 0}, 278.293820, 0.000040, 61.818000},
	{{0, 0, 26, -22, 0, 0, 0, 0, 1, -1, -1}, 229.823820, 0.000070, 152.117000},
	{{0, 0, 27, -29, 0, 0, 0, 0, -2, 1, 0}, 53.215640, 0.000020, 10.059000},
	{{0, 0, 28, -31, 0, 0, -3, 0, -2, 1, 0}, 34.454280, 0.000130, 2901.861000},
	{{0, 0, 28, -31, 0, 0, 0, 0, -2, 0, 0}, 29.610850, 0.000020, 0.076000},
	{{0, 0, 28, -31, 0, 0, 0, 0, -2, 1, 0}, 28.796190, 0.000460, 27.739000},
	{{0, 0, 28, -31, 0, 0, 0, 0, -2, 2, 0}, 28.363350, 0.000030, 0.075000},
	{{0, 0, 29, -33, 0, 0, 0, 0, -2, 0, 0}, 305.758940, 0.000030, 0.075000},
	{{0, 0, 29, -33, 0, 0, 0, 0, -2, 1, 0}, 305.758940, 0.000500, 36.607000},
	{{0, 0, 29, -33, 0, 0, 0, 0, -2, 2, 0}, 305.758940, 0.000030, 0.076000},
	{{0, 0, 29, -32, -6, 0, 0, 0, -2, 1, 0}, 125.457040, 0.000030, 685.011000},
	{{0, 0, 29, -31, 0, 0, 0, 0, -1, 1, -1}, 211.610910, 0.000020, 56.451000},
	{{0, 0, 29, -29, -4, 0, 0, 0, 0, -1, 0}, 52.728770, 0.000010, 84.635000},
	{{0, 0, 29, -29, 0, 0, 0, 0, 0, 1, -2}, 336.968190, 0.000010, 123.278000},
	{{0, 0, 30, -35, 0, 0, 0, 0, -2, 1, 0}, 250.325360, 0.000040, 11.027000},
	{{0, 0, 30, -35, 2, 0, 0, 0, -2, 1, 0}, 79.509160, 0.000020, 12.833000},
	{{0, 0, 30, -33, -3, 0, 0, 0, -1, -1, 1}, 125.604110, 0.000140, 2071.859000},
	{{0, 0, 30, -33, -1, 0, 0, 0, -1, 0, 0}, 242.865470, 0.000030, 730.361000},
	{{0, 0, 31, -37, 0, 0, 0, 0, -2, 1, 0}, 179.398090, 0.000020, 6.491000},
	{{0, 0, 31, -37, 2, 0, 0, 0, -2, 1, 0}, 92.983040, 0.000010, 68.710000},
	{{0, 0, 31, -34, 0, 0, 0, 0, 0, -3, 2}, 244.981070, 0.000040, 2022.190000},
	{{0, 0, 32, -39, 2, 0, 0, 0, -2, 1, 0}, 22.697200, 0.000010, 20.486000},
	{{0, 0, 32, -39, 3, 0, 0, 0, -2, 1, 0}, 352.818730, 0.000010, 28.177000},
	{{0, 0, 32, -35, 0, -4, 0, 0, 0, -1, 0}, 284.843820, 0.000050, 2015.126000},
	{{0, 0, 33, -41, 3, 0, 0, 0, -2, 1, 0}, 339.586410, 0.000010, 35.871000},
	{{0, 0, 33, -40, -3, 0, 0, 0, -2, 1, 0}, 88.940430, 0.000040, 494.910000},
	{{0, 0, 33, -37, -1, 0, 0, 0, 0, -1, 0}, 6.721210, 0.000010, 80.800000},
	{{0, 0, 34, -43, 0, 0, 0, 0, -2, -1, 2}, 271.072560, 0.000010, 93.931000},
	{{0, 0, 34, -43, 4, 0, 0, 0, -2, 1, 0}, 261.073700, 0.000020, 144.041000},
	{{0, 0, 34, -42, -2, 0, 0, 0, -2, 1, 0}, 69.864080, 0.000010, 52.868000},
	{{0, 0, 34, -41, 0, 0, 0, 0, -1, -1, 1}, 33.747560, 0.000170, 958.407000},
	{{0, 0, 34, -41, 2, 0, 0, 0, -1, -1, 0}, 210.399780, 0.000010, 0.075000},
	{{0, 0, 34, -41, 2, 0, 0, 0, -1, 0, 0}, 210.399780, 0.000200, 1236.942000},
	{{0, 0, 34, -41, 2, 0, 0, 0, -1, 1, 0}, 210.399780, 0.000010, 0.075000},
	{{0, 0, 34, -39, 0, 0, 0, 0, 0, -1, 0}, 30.962160, 0.000180, 116.831000},
	{{0, 0, 34, -37, 0, 0, 0, 0, 1, -1, -1}, 182.371160, 0.000020, 55.060000},
	{{0, 0, 35, -44, -2, 0, 0, 0, -2, 1, 0}, 25.913010, 0.000020, 22.496000},
	{{0, 0, 35, -44, -1, 0, 0, 0, -2, 1, 0}, 263.244900, 0.000020, 25.094000},
	{{0, 0, 35, -41, 0, 0, 0, 0, 0, -1, 0}, 352.961370, 0.000010, 18.245000},
	{{0, 0, 36, -46, -1, 0, 0, 0, -2, 1, 0}, 221.494450, 0.000010, 42.523000},
	{{0, 0, 36, -44, 0, -2, 0, 0, -1, 1, -1}, 149.082470, 0.000010, 401.021000},
	{{0, 0, 37, -48, 0, 0, 0, 0, -2, 0, 0}, 222.481130, 0.000030, 0.075000},
	{{0, 0, 37, -48, 0, 0, 0, 0, -2, 1, 0}, 251.778490, 0.000160, 387.400000},
	{{0, 0, 37, -48, 0, 0, 0, 0, -2, 2, 0}, 233.589430, 0.000010, 0.075000},
	{{0, 0, 37, -46, 0, 0, 0, 0, -1, 1, -1}, 2.116910, 0.000080, 142.421000},
	{{0, 0, 37, -45, 2, 0, 0, 0, 0, -1, 0}, 356.177140, 0.000030, 77.298000},
	{{0, 0, 38, -47, 3, 0, 0, 0, 0, -1, 0}, 297.284480, 0.000010, 125.022000},
	{{0, 0, 40, -53, -4, 0, 0, 0, -2, 1, 0}, 270.348710, 0.000010, 561.851000},
	{{0, 0, 40, -50, -2, 0, 0, 0, 0, -1, 0}, 352.606790, 0.000010, 116.607000},
	{{0, 0, 40, -48, -2, 0, 0, 0, 1, -1, -1}, 189.706380, 0.000030, 973.771000},
	{{0, 0, 40, -48, 0, 0, 0, 0, 1, 0, -2}, 301.001920, 0.000020, 347.370000},
	{{0, 0, 41, -56, 3, 0, 0, 0, -2, 1, 0}, 126.318940, 0.000010, 318.264000},
	{{0, 0, 42, -54, 0, -1, 0, 0, 0, -2, 0}, 311.880460, 0.000010, 0.075000},
	{{0, 0, 42, -54, 0, -1, 0, 0, 0, -1, 0}, 311.880460, 0.000210, 1533.043000},
	{{0, 0, 42, -54, 0, -1, 0, 0, 0, 0, 0}, 311.880460, 0.000010, 0.075000},
	{{0, 0, 43, -55, -2, 0, 0, 0, 0, 1, -2}, 312.303430, 0.000010, 1957.040000},
	{{0, 0, 43, -52, 0, 0, 0, 0, 2, -1, -2}, 273.606640, 0.000010, 92.025000},
	{{0, 0, 44, -61, -1, 0, 0, 0, -2, 1, 0}, 31.136630, 0.000070, 820.282000},
	{{0, 0, 44, -56, 1, 0, 0, 0, 1, -1, -1}, 297.365360, 0.000130, 2145.021000},
	{{0, 0, 45, -63, 0, 0, 0, 0, -2, 1, 0}, 178.897290, 0.000010, 45.137000},
	{{0, 0, 48, -69, 2, 0, 0, 0, -2, 1, 0}, 11.692270, 0.000050, 1518.942000},
	{{0, 0, 48, -67, 0, 0, 0, 0, -1, 0, 0}, 335.322190, 0.000080, 118.917000},
	{{0, 1, -42, 51, 0, 0, 0, 0, 0, 1, 0}, 279.858780, 0.000010, 353.939000},
	{{0, 1, -37, 45, 0, 0, 0, 0, 2, -1, 0}, 92.724440, 0.000020, 30.256000},
	{{0, 1, -36, 43, 0, 0, 0, 0, 2, -1, 0}, 63.397270, 0.000010, 32.986000},
	{{0, 1, -11, 0, 0, 0, 0, 0, 4, -1, -2}, 154.757860, 0.000050, 3701.205000},
	{{0, 1, -10, 0, 0, 0, 0, 0, -2, 0, 0}, 289.629150, 0.000010, 0.030000},
	{{0, 1, -10, 0, 0, 0, 0, 0, -2, 2, 0}, 289.629150, 0.000010, 0.152000},
	{{0, 1, -8, 0, 0, 0, 0, 0, -2, 1, 0}, 120.654750, 0.000020, 0.056000},
	{{0, 1, -8, 0, 2, 0, 0, 0, -2, 0, 0}, 322.501920, 0.000010, 0.032000},
	{{0, 1, -8, 0, 2, 0, 0, 0, -2, 2, 0}, 142.501920, 0.000010, 0.226000},
	{{0, 1, -8, 10, 0, 0, 0, 0, -1, 0, 1}, 214.606960, 0.000020, 258.482000},
	{{0, 1, -8, 12, 0, 0, 0, 0, -2, 0, 0}, 145.524670, 0.000060, 0.040000},
	{{0, 1, -8, 12, 0, 0, 0, 0, -2, 1, 0}, 145.642400, 0.000080, 0.087000},
	{{0, 1, -8, 12, 0, 0, 0, 0, 0, -1, 0}, 144.847680, 0.000110, 0.075000},
	{{0, 1, -8, 12, 0, 0, 0, 0, 0, 0, 0}, 144.552960, 0.000510, 174.398000},
	{{0, 1, -8, 12, 0, 0, 0, 0, 0, 1, 0}, 144.847680, 0.000110, 0.075000},
	{{0, 1, -8, 12, 0, 0, 0, 0, 2, -1, 0}, 145.643470, 0.000080, 0.087000},
	{{0, 1, -8, 12, 0, 0, 0, 0, 2, 0, 0}, 145.524770, 0.000060, 0.040000},
	{{0, 1, -8, 13, 0, 0, 0, 0, 0, 0, 0}, 354.258300, 0.000010, 1.861000},
	{{0, 1, -7, 0, 0, 0, 0, 0, -2, -1, 0}, 211.177390, 0.000010, 0.023000},
	{{0, 1, -7, 0, 0, 0, 0, 0, -2, 0, 0}, 217.879080, 0.000030, 0.033000},
	{{0, 1, -7, 0, 0, 0, 0, 0, -2, 1, 0}, 214.005510, 0.000020, 0.059000},
	{{0, 1, -7, 0, 0, 0, 0, 0, -2, 2, 0}, 218.683070, 0.000010, 0.278000},
	{{0, 1, -7, 0, 0, 0, 0, 0, -2, 3, 0}, 36.871420, 0.000010, 0.104000},
	{{0, 1, -7, 0, 2, 0, 0, 0, -2, 0, 0}, 231.686650, 0.000050, 0.033000},
	{{0, 1, -7, 0, 2, 0, 0, 0, -2, 1, 0}, 231.641160, 0.000020, 0.060000},
	{{0, 1, -7, 0, 2, 0, 0, 0, 0, -2, 0}, 231.642620, 0.000010, 0.032000},
	{{0, 1, -7, 0, 2, 0, 0, 0, 0, 0, 0}, 231.642620, 0.000010, 0.192000},
	{{0, 1, -7, 2, 0, 0, 0, 0, -2, 0, 0}, 231.908460, 0.000020, 0.034000},
	{{0, 1, -7, 2, 0, 0, 0, 0, -2, 1, 0}, 51.694520, 0.000020, 0.063000},
	{{0, 1, -7, 2, 0, 0, 0, 0, -2, 2, 0}, 231.908460, 0.000020, 0.394000},
	{{0, 1, -7, 2, 0, 0, 0, 0, -2, 3, 0}, 51.694520, 0.000020, 0.093000},
	{{0, 1, -7, 11, 0, 0, 0, 0, 0, 0, 0}, 325.289390, 0.000010, 2.110000},
	{{0, 1, -6, 0, 0, 0, 0, 0, -2, -1, 0}, 324.100090, 0.000020, 0.024000},
	{{0, 1, -6, 0, 0, 0, 0, 0, -2, 2, 0}, 346.164700, 0.000050, 0.385000},
	{{0, 1, -6, 0, 0, 0, 0, 0, 0, -2, 0}, 142.693450, 0.000010, 0.032000},
	{{0, 1, -6, 0, 0, 0, 0, 0, 0, -1, 0}, 144.100090, 0.000030, 0.057000},
	{{0, 1, -6, 0, 0, 0, 0, 0, 0, 0, 0}, 152.432530, 0.000050, 0.229000},
	{{0, 1, -6, 0, 0, 0, 0, 0, 2, -1, 0}, 331.401070, 0.000030, 0.141000},
	{{0, 1, -6, 0, 1, 0, 0, 0, -2, 0, 0}, 52.691110, 0.000010, 0.034000},
	{{0, 1, -6, 0, 1, 0, 0, 0, -2, 1, 0}, 50.469310, 0.000010, 0.063000},
	{{0, 1, -6, 0, 1, 0, 0, 0, 0, -1, 0}, 230.469310, 0.000010, 0.057000},
	{{0, 1, -6, 0, 1, 0, 0, 0, 0, 1, 0}, 230.469310, 0.000010, 0.112000},
	{{0, 1, -6, 0, 2, 0, 0, 0, -2, 0, 0}, 291.824360, 0.000050, 0.035000},
	{{0, 1, -6, 0, 2, 0, 0, 0, -2, 1, 0}, 291.956650, 0.000010, 0.064000},
	{{0, 1, -6, 0, 2, 0, 0, 0, -2, 3, 0}, 291.956650, 0.000010, 0.092000},
	{{0, 1, -6, 2, 0, 0, 0, 0, -2, 0, 0}, 272.305790, 0.000030, 0.036000},
	{{0, 1, -6, 2, 0, 0, 0, 0, -2, 2, 0}, 273.613580, 0.000020, 0.651000},
	{{0, 1, -6, 3, 0, 0, 0, 0, -2, 0, 0}, 288.628100, 0.000010, 0.036000},
	{{0, 1, -6, 3, 0, 0, 0, 0, -2, 1, 0}, 108.628100, 0.000020, 0.070000},
	{{0, 1, -6, 3, 0, 0, 0, 0, -2, 2, 0}, 288.628100, 0.000010, 0.995000},
	{{0, 1, -6, 3, 0, 0, 0, 0, -2, 3, 0}, 108.628100, 0.000020, 0.082000},
	{{0, 1, -6, 6, 0, 0, 0, 0, 0, 0, 0}, 209.498560, 0.000010, 0.844000},
	{{0, 1, -6, 8, 0, 0, 0, 0, 0, 0, 0}, 271.980850, 0.000010, 8.264000},
	{{0, 1, -5, 0, -1, 0, 0, 0, -2, 0, 0}, 58.491550, 0.000020, 0.035000},
	{{0, 1, -5, 0, -1, 0, 0, 0, -2, 2, 0}, 65.673590, 0.000010, 0.594000},
	{{0, 1, -5, 0, 0, 0, 0, 0, -4, 0, 0}, 231.780360, 0.000020, 0.019000},
	{{0, 1, -5, 0, 0, 0, 0, 0, -4, 3, 0}, 51.841840, 0.000030, 0.076000},
	{{0, 1, -5, 0, 0, 0, 0, 0, -2, -2, 0}, 51.779160, 0.000010, 0.018000},
	{{0, 1, -5, 0, 0, 0, 0, 0, -2, 0, 0}, 53.783870, 0.000050, 0.036000},
	{{0, 1, -5, 0, 0, 0, 0, 0, -2, 1, 0}, 231.149370, 0.000020, 0.067000},
	{{0, 1, -5, 0, 0, 0, 0, 0, -2, 3, 0}, 232.024080, 0.000040, 0.086000},
	{{0, 1, -5, 0, 0, 0, 0, 0, 0, -2, 0}, 51.676570, 0.000010, 0.033000},
	{{0, 1, -5, 0, 0, 0, 0, 0, 0, 0, 0}, 323.652360, 0.000040, 0.296000},
	{{0, 1, -5, 0, 1, 0, 0, 0, -2, 0, 0}, 308.771670, 0.000030, 0.036000},
	{{0, 1, -5, 0, 1, 0, 0, 0, -2, 1, 0}, 161.348070, 0.000030, 0.068000},
	{{0, 1, -5, 0, 1, 0, 0, 0, -2, 2, 0}, 221.257000, 0.000010, 0.660000},
	{{0, 1, -5, 2, 0, 0, 0, 0, 0, 0, 0}, 172.334420, 0.000010, 0.433000},
	{{0, 1, -5, 3, 0, 0, 0, 0, -4, 3, 0}, 181.638610, 0.000010, 0.087000},
	{{0, 1, -5, 3, 0, 0, 0, 0, -2, 1, 0}, 354.319410, 0.000090, 0.075000},
	{{0, 1, -5, 3, 0, 0, 0, 0, -2, 2, 0}, 133.113570, 0.000070, 184.957000},
	{{0, 1, -5, 3, 0, 0, 0, 0, -2, 3, 0}, 13.995360, 0.000050, 0.075000},
	{{0, 1, -5, 3, 0, 0, 0, 0, 0, 1, 0}, 181.638610, 0.000010, 0.087000},
	{{0, 1, -5, 5, 0, 0, 0, 0, -1, 2, -1}, 330.723020, 0.000020, 238.317000},
	{{0, 1, -5, 5, 0, 0, 0, 0, 0, 0, 0}, 73.005070, 0.000010, 1.397000},
	{{0, 1, -5, 6, 0, 0, 0, 0, 0, 0, 0}, 260.624700, 0.000010, 5.424000},
	{{0, 1, -5, 8, -3, 0, 0, 0, 0, 0, 0}, 71.948840, 0.000010, 1.597000},
	{{0, 1, -4, -2, 0, 0, 0, 0, -2, 0, 0}, 11.070710, 0.000030, 0.035000},
	{{0, 1, -4, -2, 0, 0, 0, 0, -2, 2, 0}, 11.027630, 0.000020, 0.601000},
	{{0, 1, -4, 0, -1, 0, 0, 0, -2, 0, 0}, 54.541870, 0.000060, 0.037000},
	{{0, 1, -4, 0, -1, 0, 0, 0, -2, 1, 0}, 62.346100, 0.000100, 0.072000},
	{{0, 1, -4, 0, -1, 0, 0, 0, -2, 3, 0}, 329.757320, 0.000010, 0.080000},
	{{0, 1, -4, 0, -1, 0, 0, 0, 0, 1, 0}, 242.346630, 0.000020, 0.093000},
	{{0, 1, -4, 0, 0, 0, 0, 0, -4, 0, 0}, 128.843430, 0.000020, 0.019000},
	{{0, 1, -4, 0, 0, 0, 0, 0, -4, 2, 0}, 128.843430, 0.000020, 0.039000},
	{{0, 1, -4, 0, 0, 0, 0, 0, -2, 0, 0}, 107.915540, 0.000060, 0.037000},
	{{0, 1, -4, 0, 0, 0, 0, 0, -2, 1, 0}, 241.968970, 0.000070, 0.072000},
	{{0, 1, -4, 0, 0, 0, 0, 0, -2, 2, 0}, 149.718420, 0.000030, 1.665000},
	{{0, 1, -4, 0, 0, 0, 0, 0, -2, 3, 0}, 358.564160, 0.000030, 0.079000},
	{{0, 1, -4, 0, 0, 0, 0, 0, -1, 0, 0}, 118.180290, 0.000020, 0.068000},
	{{0, 1, -4, 0, 0, 0, 0, 0, -1, 0, 1}, 237.233920, 0.000040, 0.757000},
	{{0, 1, -4, 0, 0, 0, 0, 0, 0, -1, 0}, 241.770890, 0.000030, 0.064000},
	{{0, 1, -4, 0, 0, 0, 0, 0, 0, 0, 0}, 245.668950, 0.000280, 0.421000},
	{{0, 1, -4, 0, 0, 0, 0, 0, 0, 1, 0}, 251.054140, 0.000040, 0.092000},
	{{0, 1, -4, 0, 0, 0, 0, 0, 1, 0, -1}, 60.723920, 0.000030, 0.292000},
	{{0, 1, -4, 0, 0, 0, 0, 0, 2, -1, 0}, 243.907610, 0.000050, 0.110000},
	{{0, 1, -4, 0, 0, 0, 0, 0, 2, 0, 0}, 242.991290, 0.000070, 0.045000},
	{{0, 1, -4, 0, 1, 0, 0, 0, -2, 1, 0}, 308.669530, 0.000030, 0.073000},
	{{0, 1, -4, 0, 2, 0, 0, 0, 2, -2, 0}, 90.680120, 0.000020, 0.251000},
	{{0, 1, -4, 0, 2, 0, 0, 0, 2, 0, 0}, 270.680120, 0.000020, 0.044000},
	{{0, 1, -4, 0, 3, 0, 0, 0, -2, 0, 2}, 357.639600, 0.000030, 71.265000},
	{{0, 1, -4, 0, 3, 0, 0, 0, 0, 0, 0}, 346.287920, 0.000010, 0.471000},
	{{0, 1, -4, 1, 0, 0, 0, 0, -2, 1, 0}, 297.130220, 0.000030, 0.075000},
	{{0, 1, -4, 2, 0, 0, 0, 0, 0, 0, 0}, 272.790170, 0.000020, 0.763000},
	{{0, 1, -4, 3, 0, 0, 0, 0, 0, 0, 0}, 353.423900, 0.000020, 1.283000},
	{{0, 1, -4, 4, 0, 0, 0, 0, 0, -1, 0}, 29.736640, 0.000030, 0.074000},
	{{0, 1, -4, 4, 0, 0, 0, 0, 0, 0, 0}, 216.784070, 0.000020, 4.037000},
	{{0, 1, -4, 4, 0, 0, 0, 0, 0, 1, 0}, 29.736640, 0.000030, 0.077000},
	{{0, 1, -4, 4, 0, 0, 0, 0, 2, -2, 0}, 29.598910, 0.000010, 0.495000},
	{{0, 1, -4, 4, 0, 0, 0, 0, 2, 0, 0}, 29.598910, 0.000010, 0.041000},
	{{0, 1, -3, -1, 0, 0, 0, 0, -2, 1, 0}, 270.419080, 0.000020, 0.075000},
	{{0, 1, -3, 0, -2, 0, 0, 0, -2, 0, 0}, 51.582620, 0.000030, 0.038000},
	{{0, 1, -3, 0, -2, 0, 0, 0, -2, 1, 0}, 230.083320, 0.000060, 0.077000},
	{{0, 1, -3, 0, -2, 0, 0, 0, -2, 3, 0}, 54.616140, 0.000010, 0.074000},
	{{0, 1, -3, 0, -1, 0, 0, 0, -2, 1, 0}, 304.097190, 0.000020, 0.077000},
	{{0, 1, -3, 0, -1, 0, 0, 0, -2, 3, 0}, 23.009020, 0.000010, 0.074000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -4, 2, 0}, 342.312710, 0.000020, 0.041000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -3, 0, 1}, 165.306240, 0.000010, 0.040000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -3, 1, 1}, 348.375690, 0.000010, 0.085000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -2, -1, 0}, 334.362940, 0.000020, 0.025000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -2, 0, 0}, 322.877610, 0.000080, 0.038000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -2, 1, 0}, 338.122800, 0.001090, 0.078000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -2, 2, 0}, 336.137760, 0.000100, 2.503000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -2, 3, 0}, 355.675550, 0.000050, 0.073000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -1, -2, 3}, 348.330660, 0.000020, 78.110000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -1, -1, 1}, 166.465330, 0.000180, 0.074000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -1, 0, -1}, 340.670740, 0.000130, 0.037000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -1, 0, 0}, 336.251510, 0.000160, 0.073000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -1, 0, 1}, 165.270380, 0.001000, 3.118000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -1, 1, -1}, 346.532730, 0.000060, 0.072000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -1, 1, 0}, 336.225750, 0.000020, 2.051000},
	{{0, 1, -3, 0, 0, 0, 0, 0, -1, 1, 1}, 166.465330, 0.000180, 0.077000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 0, -2, 0}, 137.500800, 0.000030, 0.036000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 0, -1, 0}, 157.335610, 0.000430, 0.068000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 0, 0, 0}, 157.204140, 0.003790, 0.728000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 0, 1, 0}, 163.802840, 0.000820, 0.084000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 0, 2, 0}, 161.219700, 0.000050, 0.040000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 1, -1, -1}, 346.687120, 0.000010, 0.064000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 1, 0, -1}, 346.592980, 0.000230, 0.412000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 1, 0, 0}, 337.687030, 0.000030, 0.091000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 1, 0, 1}, 167.894770, 0.000090, 0.041000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 1, 1, -1}, 346.687120, 0.000010, 0.092000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 2, -2, 0}, 167.870360, 0.000040, 0.318000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 2, -1, 0}, 154.131720, 0.000210, 0.099000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 2, 0, 0}, 160.390550, 0.000230, 0.043000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 2, 1, 0}, 157.819320, 0.000020, 0.027000},
	{{0, 1, -3, 0, 0, 0, 0, 0, 4, 0, 0}, 167.382080, 0.000020, 0.021000},
	{{0, 1, -3, 0, 1, 0, 0, 0, -2, 1, 0}, 266.873160, 0.000010, 0.078000},
	{{0, 1, -3, 0, 1, 0, 0, 0, -2, 3, 0}, 86.873160, 0.000010, 0.073000},
	{{0, 1, -3, 0, 1, 0, 0, 0, 0, 0, 0}, 271.771670, 0.000010, 0.775000},
	{{0, 1, -3, 0, 1, 0, 0, 0, 2, -2, 0}, 89.356280, 0.000030, 0.326000},
	{{0, 1, -3, 0, 1, 0, 0, 0, 2, 0, 0}, 89.208310, 0.000020, 0.043000},
	{{0, 1, -3, 0, 2, 0, 0, 0, -2, 0, 0}, 179.907000, 0.000030, 0.039000},
	{{0, 1, -3, 0, 2, 0, 0, 0, -2, 1, 0}, 178.665270, 0.000050, 0.079000},
	{{0, 1, -3, 0, 2, 0, 0, 0, 0, -2, 0}, 180.435000, 0.000010, 0.036000},
	{{0, 1, -3, 0, 2, 0, 0, 0, 0, 0, 0}, 180.874440, 0.000060, 0.829000},
	{{0, 1, -3, 0, 2, 0, 0, 0, 2, -1, 0}, 179.866580, 0.000030, 0.097000},
	{{0, 1, -3, 0, 2, 0, 0, 0, 2, 0, 0}, 180.255440, 0.000010, 0.042000},
	{{0, 1, -3, 0, 2, 0, 0, 0, 2, 2, 0}, 179.894910, 0.000010, 0.020000},
	{{0, 1, -3, 0, 3, 0, 0, 0, 2, 0, 0}, 170.906800, 0.000040, 0.042000},
	{{0, 1, -3, 1, 0, 0, 0, 0, 0, 0, 0}, 163.751810, 0.000020, 1.187000},
	{{0, 1, -3, 2, 0, 0, 0, 0, -2, 2, 0}, 179.213420, 0.000010, 0.684000},
	{{0, 1, -3, 2, 0, 0, 0, 0, 0, -1, 0}, 357.730090, 0.000020, 0.074000},
	{{0, 1, -3, 2, 0, 0, 0, 0, 0, 1, 0}, 357.730090, 0.000020, 0.077000},
	{{0, 1, -3, 2, 0, 0, 0, 0, 2, -3, 0}, 0.206490, 0.000030, 0.065000},
	{{0, 1, -3, 2, 0, 0, 0, 0, 2, -2, 0}, 179.458710, 0.000020, 0.480000},
	{{0, 1, -3, 2, 0, 0, 0, 0, 2, -1, 0}, 0.206490, 0.000030, 0.090000},
	{{0, 1, -3, 2, 0, 0, 0, 0, 2, 0, 0}, 179.913930, 0.000040, 0.041000},
	{{0, 1, -3, 3, 0, 0, 0, 0, 0, 0, 0}, 36.959770, 0.000050, 4.534000},
	{{0, 1, -3, 6, 0, 0, 0, 0, 2, -3, 0}, 103.216120, 0.000020, 0.076000},
	{{0, 1, -3, 6, 0, 0, 0, 0, 2, -1, 0}, 103.216120, 0.000020, 0.075000},
	{{0, 1, -2, -3, 0, 0, 0, 0, -2, 1, 0}, 11.747990, 0.000010, 0.074000},
	{{0, 1, -2, -3, 0, 0, 0, 0, -2, 3, 0}, 110.501700, 0.000010, 0.077000},
	{{0, 1, -2, 0, -1, 0, 0, 0, -2, 1, 0}, 204.026560, 0.000020, 0.084000},
	{{0, 1, -2, 0, -1, 0, 0, 0, -2, 3, 0}, 24.026560, 0.000020, 0.069000},
	{{0, 1, -2, 0, -1, 0, 0, 0, 2, -2, 0}, 279.594300, 0.000010, 0.448000},
	{{0, 1, -2, 0, -1, 0, 0, 0, 2, 0, 0}, 99.594300, 0.000010, 0.041000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -4, 0, 0}, 147.421740, 0.000020, 0.020000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -4, 1, 0}, 121.746220, 0.000050, 0.027000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -4, 2, 0}, 209.203390, 0.000010, 0.043000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -3, 0, 1}, 256.663700, 0.000010, 0.042000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -3, 1, 1}, 32.932280, 0.000020, 0.093000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -3, 2, -1}, 76.622580, 0.000010, 0.041000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -2, -2, 0}, 101.828110, 0.000020, 0.019000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -2, -1, 0}, 114.285120, 0.000260, 0.026000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -2, -1, 2}, 117.637520, 0.000020, 0.087000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -2, 0, 0}, 121.838230, 0.001760, 0.040000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -2, 0, 1}, 76.620690, 0.000100, 0.086000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -2, 0, 2}, 117.863220, 0.000060, 0.577000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -2, 1, 0}, 210.077550, 0.001230, 0.084000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -2, 2, 0}, 269.499470, 0.000210, 0.715000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -2, 3, 0}, 276.270820, 0.000060, 0.068000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, -2, 1}, 256.658630, 0.000020, 0.039000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, -1, -1}, 84.290290, 0.000040, 0.025000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, -1, 0}, 88.451180, 0.000020, 0.038000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, -1, 1}, 256.647520, 0.000440, 0.080000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, 0, -1}, 76.605260, 0.000380, 0.038000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, 0, 0}, 88.332290, 0.000320, 0.078000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, 0, 1}, 256.710040, 0.001650, 1.472000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, 1, -1}, 76.981800, 0.000870, 0.077000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, 1, 0}, 91.659320, 0.000070, 1.951000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, 1, 1}, 256.638830, 0.000310, 0.072000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, 2, -1}, 76.632070, 0.000080, 2.892000},
	{{0, 1, -2, 0, 0, 0, 0, 0, -1, 2, 1}, 256.660810, 0.000020, 0.037000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, -3, 0}, 107.205070, 0.000040, 0.025000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, -2, 0}, 105.567380, 0.000670, 0.037000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, -2, 2}, 295.422690, 0.000040, 24.413000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, -1, -2}, 286.229440, 0.000030, 0.025000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, -1, 0}, 106.025720, 0.009130, 0.073000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, -1, 1}, 256.587150, 0.000080, 4.814000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, -1, 2}, 299.022920, 0.000070, 0.076000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 0, -2}, 282.167080, 0.000010, 0.037000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 0, -1}, 76.332230, 0.000150, 0.072000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 0, 0}, 103.715670, 0.044780, 2.670000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 0, 1}, 256.614460, 0.000340, 0.077000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 0, 2}, 282.167080, 0.000010, 0.038000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 1, -2}, 108.919750, 0.000010, 0.072000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 1, -1}, 76.628310, 0.000020, 1.848000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 1, 0}, 106.632340, 0.008520, 0.078000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 1, 1}, 256.614460, 0.000020, 0.038000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 1, 2}, 286.229440, 0.000030, 0.025000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 2, 0}, 106.005550, 0.000540, 0.038000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 0, 3, 0}, 106.830910, 0.000040, 0.025000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, -2, -1}, 256.649540, 0.000010, 0.036000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, -2, 1}, 76.672330, 0.000040, 0.913000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, -1, -1}, 256.681720, 0.000240, 0.068000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, -1, 0}, 272.599610, 0.000030, 0.793000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, -1, 1}, 76.816260, 0.000570, 0.082000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, 0, -1}, 256.717430, 0.002180, 0.700000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, 0, 0}, 280.642840, 0.000420, 0.083000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, 0, 1}, 78.670420, 0.000500, 0.039000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, 1, -1}, 256.534620, 0.000460, 0.085000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, 1, 1}, 78.199810, 0.000040, 0.026000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 1, 2, -1}, 256.636350, 0.000020, 0.040000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 2, -3, 0}, 285.997860, 0.000010, 0.065000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 2, -2, 0}, 225.600850, 0.000110, 0.465000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 2, -1, 0}, 228.433420, 0.001860, 0.090000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 2, 0, -2}, 211.101220, 0.000010, 0.403000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 2, 0, 0}, 236.846420, 0.001370, 0.041000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 2, 1, 0}, 217.538580, 0.000080, 0.027000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 2, 2, 0}, 259.635580, 0.000030, 0.020000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 4, -2, 0}, 248.744260, 0.000020, 0.044000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 4, -1, 0}, 263.069730, 0.000040, 0.028000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 4, 0, 0}, 219.571300, 0.000010, 0.020000},
	{{0, 1, -2, 0, 0, 0, 0, 0, 4, 1, 0}, 87.841080, 0.000020, 0.016000},
	{{0, 1, -2, 0, 0, 1, 0, 0, 0, -2, 2}, 359.794770, 0.000010, 142.558000},
	{{0, 1, -2, 0, 0, 1, 0, 0, 0, 0, 0}, 0.038810, 0.000010, 2.937000},
	{{0, 1, -2, 0, 1, 0, 0, 0, -2, 0, 0}, 173.288440, 0.000060, 0.040000},
	{{0, 1, -2, 0, 1, 0, 0, 0, -2, 1, 0}, 177.970830, 0.000180, 0.085000},
	{{0, 1, -2, 0, 1, 0, 0, 0, 0, -1, 0}, 356.284300, 0.000030, 0.074000},
	{{0, 1, -2, 0, 1, 0, 0, 0, 0, 0, 0}, 1.635360, 0.000080, 3.446000},
	{{0, 1, -2, 0, 1, 0, 0, 0, 0, 1, 0}, 357.500310, 0.000080, 0.077000},
	{{0, 1, -2, 0, 1, 0, 0, 0, 2, -2, 0}, 1.625930, 0.000010, 0.484000},
	{{0, 1, -2, 0, 1, 0, 0, 0, 2, -1, 0}, 358.690080, 0.000020, 0.089000},
	{{0, 1, -2, 0, 1, 0, 0, 0, 2, 0, 0}, 177.529070, 0.000030, 0.041000},
	{{0, 1, -2, 0, 1, 0, 0, 0, 2, 2, 0}, 178.721600, 0.000010, 0.020000},
	{{0, 1, -2, 0, 2, 0, 0, 0, -2, 0, 0}, 235.698430, 0.000010, 0.040000},
	{{0, 1, -2, 0, 2, 0, 0, 0, -2, 1, 0}, 240.845910, 0.000020, 0.086000},
	{{0, 1, -2, 0, 2, 0, 0, 0, 0, -1, 0}, 60.845910, 0.000080, 0.074000},
	{{0, 1, -2, 0, 2, 0, 0, 0, 0, 0, 0}, 212.098900, 0.000020, 4.858000},
	{{0, 1, -2, 0, 2, 0, 0, 0, 0, 1, 0}, 60.845910, 0.000080, 0.077000},
	{{0, 1, -2, 0, 2, 0, 0, 0, 2, -2, 0}, 83.016090, 0.000020, 0.505000},
	{{0, 1, -2, 0, 2, 0, 0, 0, 2, -1, 0}, 240.845910, 0.000020, 0.089000},
	{{0, 1, -2, 0, 2, 0, 0, 0, 2, 0, 0}, 83.016090, 0.000020, 0.041000},
	{{0, 1, -2, 0, 3, 0, 0, 0, 0, 0, 0}, 221.825580, 0.000010, 8.226000},
	{{0, 1, -2, 0, 4, 0, 0, 0, 0, 0, 0}, 241.063070, 0.000020, 26.839000},
	{{0, 1, -2, 1, 0, 0, 0, 0, 0, -1, 0}, 168.887430, 0.000020, 0.076000},
	{{0, 1, -2, 1, 0, 0, 0, 0, 0, 0, 0}, 316.817060, 0.000010, 6.361000},
	{{0, 1, -2, 1, 0, 0, 0, 0, 0, 1, 0}, 168.887430, 0.000020, 0.075000},
	{{0, 1, -2, 2, 0, 0, 0, 0, -2, 0, 0}, 220.796430, 0.000020, 0.042000},
	{{0, 1, -2, 2, 0, 0, 0, 0, -2, 1, 0}, 220.718720, 0.000010, 0.093000},
	{{0, 1, -2, 2, 0, 0, 0, 0, -2, 2, 0}, 220.941540, 0.000030, 0.406000},
	{{0, 1, -2, 2, 0, 0, 0, 0, -2, 3, 0}, 220.718720, 0.000010, 0.064000},
	{{0, 1, -2, 2, 0, 0, 0, 0, 0, -1, 0}, 222.501390, 0.000020, 0.080000},
	{{0, 1, -2, 2, 0, 0, 0, 0, 0, 0, 0}, 280.741610, 0.000010, 1.452000},
	{{0, 1, -2, 2, 0, 0, 0, 0, 0, 1, 0}, 222.501390, 0.000020, 0.072000},
	{{0, 1, -2, 2, 0, 0, 0, 0, 2, -3, 0}, 44.755520, 0.000040, 0.070000},
	{{0, 1, -2, 2, 0, 0, 0, 0, 2, -2, 0}, 44.755530, 0.000030, 0.922000},
	{{0, 1, -2, 2, 0, 0, 0, 0, 2, -1, 0}, 44.755520, 0.000040, 0.082000},
	{{0, 1, -2, 2, 0, 0, 0, 0, 2, 0, 0}, 45.437620, 0.000020, 0.039000},
	{{0, 1, -2, 3, 0, 0, 0, 0, -2, 0, 0}, 236.435730, 0.000020, 0.043000},
	{{0, 1, -2, 3, 0, 0, 0, 0, -2, 2, 0}, 236.435730, 0.000020, 0.334000},
	{{0, 1, -2, 3, 0, 0, 0, 0, 2, -3, 0}, 236.451620, 0.000030, 0.072000},
	{{0, 1, -2, 3, 0, 0, 0, 0, 2, -2, 0}, 236.880380, 0.000020, 1.807000},
	{{0, 1, -2, 3, 0, 0, 0, 0, 2, -1, 0}, 236.451620, 0.000030, 0.079000},
	{{0, 1, -2, 3, 0, 0, 0, 0, 2, 0, 0}, 236.880380, 0.000020, 0.039000},
	{{0, 1, -2, 4, 0, 0, 0, 0, 2, -3, 0}, 268.583280, 0.000030, 0.075000},
	{{0, 1, -2, 4, 0, 0, 0, 0, 2, -2, 0}, 97.555560, 0.000010, 45.963000},
	{{0, 1, -2, 4, 0, 0, 0, 0, 2, -1, 0}, 217.111180, 0.000030, 0.076000},
	{{0, 1, -1, -5, 0, 0, 0, 0, -2, 1, 0}, 24.066860, 0.000020, 0.074000},
	{{0, 1, -1, -2, 0, 0, 0, 0, 0, 0, 0}, 120.667430, 0.000010, 2.284000},
	{{0, 1, -1, 0, -2, 0, 0, 0, -2, 0, 0}, 24.956620, 0.000020, 0.041000},
	{{0, 1, -1, 0, -2, 0, 0, 0, -2, 2, 0}, 204.956620, 0.000020, 0.448000},
	{{0, 1, -1, 0, -2, 0, 0, 0, 0, 0, 0}, 6.133700, 0.000010, 2.189000},
	{{0, 1, -1, 0, -1, 0, 0, 0, -2, 0, 0}, 211.704000, 0.000060, 0.041000},
	{{0, 1, -1, 0, -1, 0, 0, 0, -2, 2, 0}, 91.855860, 0.000020, 0.432000},
	{{0, 1, -1, 0, -1, 0, 0, 0, 0, 0, 0}, 72.786570, 0.000020, 1.848000},
	{{0, 1, -1, 0, -1, 0, 0, 0, 2, -3, 0}, 193.925940, 0.000010, 0.069000},
	{{0, 1, -1, 0, -1, 0, 0, 0, 2, -2, 0}, 193.925940, 0.000010, 0.811000},
	{{0, 1, -1, 0, -1, 0, 0, 0, 2, -1, 0}, 193.925900, 0.000040, 0.083000},
	{{0, 1, -1, 0, -1, 0, 0, 0, 2, 0, 0}, 191.710940, 0.000070, 0.040000},
	{{0, 1, -1, 0, 0, -1, 0, 0, 0, 0, 0}, 88.271430, 0.000010, 1.690000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -6, 0, 0}, 179.987110, 0.000020, 0.014000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -6, 1, 0}, 179.983480, 0.000050, 0.017000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -6, 2, 0}, 179.983270, 0.000060, 0.021000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -6, 3, 0}, 179.771060, 0.000020, 0.030000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -4, -2, 0}, 180.000840, 0.000010, 0.013000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -4, -1, 0}, 179.990480, 0.000200, 0.016000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -4, 0, 0}, 179.992910, 0.001130, 0.020000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -4, 0, 2}, 179.968450, 0.000010, 0.045000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -4, 1, -2}, 359.982720, 0.000020, 0.016000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -4, 1, 0}, 179.992510, 0.002660, 0.028000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -4, 2, 0}, 179.967310, 0.001750, 0.045000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -4, 3, 0}, 180.357600, 0.000060, 0.110000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -3, -1, 0}, 179.971910, 0.000030, 0.020000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -3, 0, 0}, 179.813950, 0.000020, 0.027000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -3, 1, 0}, 359.991470, 0.000230, 0.043000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -3, 2, 0}, 0.060950, 0.000040, 0.100000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -3, 3, 0}, 359.988090, 0.000010, 0.304000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, -3, 0}, 180.003860, 0.000090, 0.016000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, -2, 0}, 180.012870, 0.000840, 0.020000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, -1, -2}, 359.984290, 0.000040, 0.016000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, -1, 0}, 179.989740, 0.009390, 0.027000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, -1, 2}, 359.990450, 0.000230, 0.095000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, 0, -2}, 359.986620, 0.000250, 0.020000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, 0, 0}, 179.981520, 0.099220, 0.041000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, 0, 2}, 179.984190, 0.001640, 0.366000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, 1, -2}, 359.982690, 0.000310, 0.027000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, 1, 0}, 179.984360, 0.131220, 0.092000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, 2, -2}, 359.977770, 0.000030, 0.041000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, 2, 0}, 179.988180, 0.004970, 0.417000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, 3, 0}, 179.948370, 0.000170, 0.064000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -2, 4, 0}, 179.919900, 0.000010, 0.035000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -1, -1, 0}, 0.005880, 0.000470, 0.040000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -1, -1, 1}, 50.920300, 0.000030, 0.086000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -1, 0, -1}, 220.632960, 0.000010, 0.040000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -1, 0, 0}, 0.007560, 0.006660, 0.085000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -1, 0, 1}, 149.121310, 0.000020, 0.596000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -1, 1, 0}, 0.001660, 0.000770, 0.661000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -1, 2, -1}, 220.632960, 0.000010, 0.743000},
	{{0, 1, -1, 0, 0, 0, 0, 0, -1, 2, 0}, 359.105050, 0.000020, 0.068000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, -4, 0}, 180.012820, 0.000050, 0.019000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, -3, 0}, 180.010410, 0.000720, 0.026000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, -2, -2}, 0.010370, 0.000050, 0.019000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, -2, 0}, 180.009070, 0.010190, 0.039000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, -2, 2}, 0.013310, 0.000010, 1.043000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, -1, -2}, 0.007960, 0.000370, 0.025000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, -1, 0}, 180.008880, 0.153500, 0.079000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, -1, 2}, 180.046500, 0.000190, 0.070000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 0, -2}, 359.995720, 0.000230, 0.038000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 0, 0}, 180.009880, 0.821550, 1.599000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 0, 2}, 0.010390, 0.000500, 0.036000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 1, -2}, 180.205080, 0.000120, 0.077000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 1, 0}, 180.015110, 0.127220, 0.072000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 1, 2}, 0.013330, 0.000310, 0.025000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 2, -2}, 359.880230, 0.000030, 3.425000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 2, 0}, 180.016020, 0.008850, 0.037000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 2, 2}, 0.011820, 0.000060, 0.019000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 3, 0}, 180.017770, 0.000620, 0.025000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 0, 4, 0}, 180.019340, 0.000040, 0.019000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, -2, 0}, 180.102480, 0.000250, 0.074000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, -2, 1}, 247.054340, 0.000030, 10.557000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, -1, -1}, 31.604140, 0.000030, 0.073000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, -1, 0}, 180.012890, 0.002920, 3.824000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, -1, 1}, 230.920290, 0.000130, 0.076000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, 0, -2}, 180.020540, 0.000030, 0.072000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, 0, -1}, 24.501420, 0.000250, 2.335000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, 0, 0}, 180.020100, 0.012180, 0.077000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, 0, 1}, 189.749040, 0.000040, 0.038000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, 0, 2}, 0.021500, 0.000020, 0.025000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, 1, -1}, 31.604140, 0.000030, 0.078000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, 1, 0}, 180.021590, 0.000750, 0.038000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 1, 2, 0}, 179.965110, 0.000050, 0.025000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, -4, 0}, 359.715140, 0.000010, 0.036000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, -2, 0}, 359.995550, 0.001760, 0.871000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, -1, -2}, 0.165060, 0.000020, 0.068000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, -1, 0}, 179.879730, 0.012920, 0.083000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, -1, 2}, 359.953740, 0.000030, 0.026000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, 0, -2}, 180.020670, 0.000890, 0.675000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, 0, 0}, 180.014640, 0.022010, 0.039000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, 0, 2}, 359.887840, 0.000020, 0.019000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, 1, -2}, 0.082470, 0.000080, 0.085000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, 1, 0}, 179.998900, 0.002860, 0.026000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, 2, 0}, 180.086000, 0.000280, 0.019000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 2, 3, 0}, 180.065020, 0.000030, 0.015000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 3, -2, 0}, 179.854850, 0.000030, 0.089000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 3, -1, 0}, 180.018300, 0.000170, 0.041000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 3, 0, 0}, 180.018630, 0.000120, 0.027000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 3, 2, 0}, 180.039140, 0.000020, 0.016000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 4, -3, 0}, 0.337750, 0.000060, 0.097000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 4, -2, 0}, 179.806870, 0.000190, 0.042000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 4, -1, -2}, 180.029830, 0.000020, 0.100000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 4, -1, 0}, 179.978890, 0.000510, 0.027000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 4, 0, 0}, 179.995460, 0.000220, 0.020000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 4, 1, 0}, 179.997140, 0.000040, 0.016000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 5, -2, 0}, 180.019610, 0.000010, 0.028000},
	{{0, 1, -1, 0, 0, 0, 0, 0, 6, -1, 0}, 0.026870, 0.000030, 0.016000},
	{{0, 1, -1, 0, 0, 1, 0, 0, 0, 0, 0}, 85.913320, 0.000020, 1.516000},
	{{0, 1, -1, 0, 1, 0, 0, 0, -2, 0, 0}, 344.898430, 0.000040, 0.042000},
	{{0, 1, -1, 0, 1, 0, 0, 0, -2, 1, 0}, 169.063610, 0.000010, 0.093000},
	{{0, 1, -1, 0, 1, 0, 0, 0, 0, -1, 0}, 349.063610, 0.000070, 0.080000},
	{{0, 1, -1, 0, 1, 0, 0, 0, 0, 0, 0}, 295.597370, 0.000010, 1.409000},
	{{0, 1, -1, 0, 1, 0, 0, 0, 0, 1, 0}, 349.063610, 0.000070, 0.072000},
	{{0, 1, -1, 0, 1, 0, 0, 0, 1, -2, 1}, 105.146230, 0.000020, 95.933000},
	{{0, 1, -1, 0, 1, 0, 0, 0, 1, -1, 1}, 91.854730, 0.000090, 0.075000},
	{{0, 1, -1, 0, 1, 0, 0, 0, 2, -3, 0}, 314.474970, 0.000020, 0.070000},
	{{0, 1, -1, 0, 1, 0, 0, 0, 2, -2, 0}, 314.420200, 0.000020, 0.940000},
	{{0, 1, -1, 0, 1, 0, 0, 0, 2, -1, 0}, 315.095990, 0.000040, 0.082000},
	{{0, 1, -1, 0, 1, 0, 0, 0, 2, 0, 0}, 330.477100, 0.000070, 0.039000},
	{{0, 1, -1, 0, 2, 0, 0, 0, 0, 0, 0}, 74.451500, 0.000010, 1.259000},
	{{0, 1, -1, 0, 2, 0, 0, 0, 1, 0, -1}, 354.963230, 0.000010, 3.852000},
	{{0, 1, -1, 2, 0, 0, 0, 0, 2, -3, 0}, 211.598560, 0.000020, 0.075000},
	{{0, 1, -1, 2, 0, 0, 0, 0, 2, -1, 0}, 231.876200, 0.000060, 0.076000},
	{{0, 1, 0, -7, 0, 0, 0, 0, -2, 0, 2}, 176.646400, 0.000010, 88.775000},
	{{0, 1, 0, -6, 0, 0, 0, 0, -2, 1, 0}, 206.527040, 0.000030, 0.077000},
	{{0, 1, 0, -3, 0, 0, 0, 0, 0, -1, 0}, 0.655870, 0.000020, 0.076000},
	{{0, 1, 0, -3, 0, 0, 0, 0, 0, 0, 0}, 324.217070, 0.000030, 32.819000},
	{{0, 1, 0, -3, 0, 0, 0, 0, 0, 1, 0}, 0.655870, 0.000020, 0.075000},
	{{0, 1, 0, -2, 0, 0, 0, 0, -2, 2, 0}, 321.255310, 0.000020, 0.428000},
	{{0, 1, 0, -2, 0, 0, 0, 0, 0, -1, 0}, 321.562240, 0.000030, 0.079000},
	{{0, 1, 0, -2, 0, 0, 0, 0, 0, 1, 0}, 321.562240, 0.000030, 0.072000},
	{{0, 1, 0, -2, 0, 0, 0, 0, 2, -3, 0}, 139.617500, 0.000040, 0.069000},
	{{0, 1, 0, -2, 0, 0, 0, 0, 2, -2, 0}, 140.140830, 0.000020, 0.825000},
	{{0, 1, 0, -2, 0, 0, 0, 0, 2, -1, 0}, 139.617500, 0.000040, 0.083000},
	{{0, 1, 0, -2, 0, 0, 0, 0, 2, 0, 0}, 140.140830, 0.000020, 0.040000},
	{{0, 1, 0, 0, -3, 0, 0, 0, 0, 0, 0}, 316.078240, 0.000010, 0.729000},
	{{0, 1, 0, 0, -2, 0, 0, 0, -2, 0, 0}, 303.461540, 0.000060, 0.043000},
	{{0, 1, 0, 0, -2, 0, 0, 0, 2, -3, 0}, 140.803770, 0.000020, 0.074000},
	{{0, 1, 0, 0, -2, 0, 0, 0, 2, -2, 0}, 308.549500, 0.000010, 3.154000},
	{{0, 1, 0, 0, -2, 0, 0, 0, 2, -1, 0}, 320.803770, 0.000020, 0.077000},
	{{0, 1, 0, 0, -2, 0, 0, 0, 2, 0, 0}, 128.549500, 0.000010, 0.038000},
	{{0, 1, 0, 0, -1, 0, 0, 0, -2, 0, 0}, 180.153330, 0.000040, 0.043000},
	{{0, 1, 0, 0, -1, 0, 0, 0, -2, 1, 0}, 0.990830, 0.000030, 0.101000},
	{{0, 1, 0, 0, -1, 0, 0, 0, -2, 2, 0}, 178.692080, 0.000010, 0.302000},
	{{0, 1, 0, 0, -1, 0, 0, 0, 0, -2, 0}, 180.990830, 0.000030, 0.040000},
	{{0, 1, 0, 0, -1, 0, 0, 0, 0, -1, 0}, 0.454200, 0.000040, 0.085000},
	{{0, 1, 0, 0, -1, 0, 0, 0, 0, 0, 0}, 4.897740, 0.000040, 0.649000},
	{{0, 1, 0, 0, -1, 0, 0, 0, 0, 1, 0}, 0.454200, 0.000040, 0.068000},
	{{0, 1, 0, 0, -1, 0, 0, 0, 2, -3, 0}, 3.603980, 0.000030, 0.074000},
	{{0, 1, 0, 0, -1, 0, 0, 0, 2, -2, 0}, 4.073730, 0.000020, 4.296000},
	{{0, 1, 0, 0, -1, 0, 0, 0, 2, -1, 0}, 183.603980, 0.000030, 0.077000},
	{{0, 1, 0, 0, -1, 0, 0, 0, 2, 0, 0}, 3.494270, 0.000070, 0.038000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -4, -2, 0}, 76.973300, 0.000010, 0.013000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -4, -1, 0}, 77.861140, 0.000030, 0.016000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -4, 0, 0}, 78.477250, 0.000240, 0.021000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -4, 1, 0}, 77.300980, 0.000400, 0.029000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -4, 2, 0}, 77.863180, 0.000160, 0.047000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -3, 0, 0}, 76.363280, 0.000020, 0.028000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -3, 0, 1}, 103.360930, 0.000020, 0.045000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -3, 1, -1}, 103.337320, 0.000010, 0.028000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -3, 1, 1}, 103.338330, 0.000010, 0.114000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -3, 3, 0}, 256.363280, 0.000010, 0.233000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, -2, 0}, 76.878970, 0.000110, 0.020000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, -1, 0}, 77.144900, 0.000930, 0.027000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, -1, 2}, 256.126260, 0.000010, 0.105000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, 0, -2}, 257.036540, 0.000020, 0.020000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, 0, 0}, 77.055400, 0.009420, 0.043000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, 0, 2}, 76.648420, 0.000100, 0.268000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, 1, 0}, 76.964720, 0.008730, 0.101000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, 1, 2}, 256.126260, 0.000010, 0.059000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, 2, 0}, 74.649920, 0.000330, 0.294000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, 3, 0}, 85.014460, 0.000070, 0.060000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -2, 4, 0}, 80.289380, 0.000010, 0.033000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -1, -1, -1}, 103.340530, 0.000070, 0.027000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -1, -1, 1}, 283.795160, 0.000090, 0.095000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -1, 0, -1}, 103.307560, 0.000770, 0.041000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -1, 0, 0}, 258.053410, 0.000220, 0.093000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -1, 0, 1}, 283.339260, 0.001980, 0.373000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -1, 1, -1}, 102.525400, 0.000450, 0.092000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -1, 1, 1}, 283.343990, 0.000220, 0.063000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -1, 2, -1}, 99.182760, 0.000010, 0.426000},
	{{0, 1, 0, 0, 0, 0, 0, 0, -1, 2, 1}, 283.338600, 0.000010, 0.034000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 0, -3, 0}, 84.441250, 0.000030, 0.026000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 0, -2, 0}, 80.775780, 0.000490, 0.040000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 0, -1, 0}, 81.703430, 0.006540, 0.086000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 0, 0, -1}, 282.866590, 0.000120, 0.085000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0}, 82.232990, 0.014570, 0.615000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1}, 103.337450, 0.000050, 0.066000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0}, 81.229070, 0.001910, 0.067000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 0, 2, 0}, 83.187980, 0.000110, 0.036000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, -2, -1}, 103.335150, 0.000090, 0.039000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, -2, 0}, 263.742640, 0.000010, 0.080000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, -2, 1}, 283.476060, 0.000030, 1.105000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, -1, -1}, 103.333580, 0.001410, 0.079000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, -1, 0}, 270.377040, 0.000030, 1.354000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, -1, 1}, 283.337330, 0.000880, 0.071000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, 0, -3}, 283.331000, 0.000010, 0.038000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, 0, -1}, 103.333710, 0.008940, 1.749000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0}, 271.092500, 0.000170, 0.071000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1}, 283.332670, 0.001140, 0.036000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, 1, -1}, 103.383660, 0.001260, 0.072000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1}, 283.333650, 0.000120, 0.025000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 1, 2, -1}, 103.335030, 0.000090, 0.037000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 2, -3, 0}, 269.578780, 0.000040, 0.075000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 2, -3, 1}, 282.434380, 0.000030, 54.757000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 2, -2, 0}, 265.802800, 0.001950, 6.734000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 2, -2, 1}, 103.341660, 0.000030, 0.075000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 2, -1, -1}, 103.331160, 0.000040, 3.172000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 2, -1, 0}, 270.206610, 0.008210, 0.076000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 2, 0, -1}, 103.337480, 0.000200, 0.077000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 2, 0, 0}, 275.311560, 0.000620, 0.038000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 2, 1, 0}, 271.842060, 0.000050, 0.025000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 3, -2, -1}, 283.335860, 0.000020, 0.832000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 3, -1, -1}, 283.327130, 0.000180, 0.083000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 3, -1, 1}, 283.331110, 0.000020, 0.026000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 3, 0, -1}, 103.336750, 0.000080, 0.040000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 3, 1, -1}, 103.333810, 0.000010, 0.026000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 4, -3, 0}, 265.517950, 0.000030, 0.088000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 4, -2, 0}, 273.653290, 0.000090, 0.041000},
	{{0, 1, 0, 0, 0, 0, 0, 0, 4, -1, 0}, 270.792780, 0.000070, 0.026000},
	{{0, 1, 0, 0, 1, 0, 0, 0, -2, 0, 0}, 80.433150, 0.000010, 0.043000},
	{{0, 1, 0, 0, 1, 0, 0, 0, -2, 2, 0}, 260.433150, 0.000010, 0.287000},
	{{0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0}, 94.520450, 0.000010, 0.585000},
	{{0, 1, 0, 0, 1, 0, 0, 0, 2, -3, 0}, 285.809370, 0.000020, 0.075000},
	{{0, 1, 0, 0, 1, 0, 0, 0, 2, -2, 0}, 264.994700, 0.000020, 15.579000},
	{{0, 1, 0, 0, 1, 0, 0, 0, 2, -1, 0}, 302.409220, 0.000060, 0.076000},
	{{0, 1, 0, 0, 1, 0, 0, 0, 2, 0, 0}, 151.683450, 0.000010, 0.038000},
	{{0, 1, 0, 0, 2, 0, 0, 0, 2, -3, 0}, 149.992340, 0.000040, 0.076000},
	{{0, 1, 0, 0, 2, 0, 0, 0, 2, -2, 0}, 342.798250, 0.000020, 49.716000},
	{{0, 1, 0, 0, 2, 0, 0, 0, 2, -1, 0}, 111.602870, 0.000030, 0.075000},
	{{0, 1, 1, -5, 0, 0, 0, 0, 0, 0, 0}, 100.470260, 0.000020, 30.398000},
	{{0, 1, 1, -3, 0, 0, 0, 0, 1, 0, -1}, 242.602960, 0.000010, 42.928000},
	{{0, 1, 1, -2, 0, 0, 0, 0, -2, 0, 0}, 180.131170, 0.000030, 0.043000},
	{{0, 1, 1, -2, 0, 0, 0, 0, -2, 1, 0}, 0.051720, 0.000030, 0.101000},
	{{0, 1, 1, -2, 0, 0, 0, 0, -2, 3, 0}, 0.051720, 0.000010, 0.060000},
	{{0, 1, 1, -2, 0, 0, 0, 0, 0, -1, 0}, 0.051720, 0.000010, 0.086000},
	{{0, 1, 1, -2, 0, 0, 0, 0, 0, 0, 0}, 232.002120, 0.000010, 0.640000},
	{{0, 1, 1, -2, 0, 0, 0, 0, 0, 1, 0}, 0.051720, 0.000010, 0.067000},
	{{0, 1, 1, -2, 0, 0, 0, 0, 2, -3, 0}, 183.774560, 0.000010, 0.074000},
	{{0, 1, 1, -2, 0, 0, 0, 0, 2, -1, 0}, 183.774560, 0.000010, 0.077000},
	{{0, 1, 1, -1, 0, 0, 0, 0, 2, 0, -2}, 37.481810, 0.000020, 73.036000},
	{{0, 1, 1, 0, -4, 0, 0, 0, 0, 0, 0}, 199.688870, 0.000010, 0.437000},
	{{0, 1, 1, 0, -3, 0, 0, 0, -2, 0, 0}, 191.537010, 0.000040, 0.045000},
	{{0, 1, 1, 0, -3, 0, 0, 0, -2, 2, 0}, 9.149970, 0.000010, 0.241000},
	{{0, 1, 1, 0, -2, 0, 0, 0, -2, 1, 0}, 180.043130, 0.000090, 0.111000},
	{{0, 1, 1, 0, -2, 0, 0, 0, -2, 3, 0}, 0.147690, 0.000010, 0.057000},
	{{0, 1, 1, 0, -2, 0, 0, 0, 0, -1, 0}, 180.062920, 0.000020, 0.093000},
	{{0, 1, 1, 0, -2, 0, 0, 0, 0, 0, 0}, 180.762110, 0.000050, 0.407000},
	{{0, 1, 1, 0, -2, 0, 0, 0, 0, 1, 0}, 180.062920, 0.000020, 0.064000},
	{{0, 1, 1, 0, -2, 0, 0, 0, 2, -3, 0}, 0.341140, 0.000010, 0.080000},
	{{0, 1, 1, 0, -2, 0, 0, 0, 2, -1, 0}, 180.142290, 0.000050, 0.072000},
	{{0, 1, 1, 0, -2, 0, 0, 0, 2, 0, 0}, 0.093840, 0.000020, 0.037000},
	{{0, 1, 1, 0, -1, 0, 0, 0, -2, 0, 0}, 90.652940, 0.000030, 0.045000},
	{{0, 1, 1, 0, -1, 0, 0, 0, -2, 2, 0}, 90.652940, 0.000030, 0.232000},
	{{0, 1, 1, 0, -1, 0, 0, 0, 0, -2, 0}, 345.847560, 0.000030, 0.042000},
	{{0, 1, 1, 0, -1, 0, 0, 0, 0, 0, 0}, 189.161970, 0.000030, 0.394000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -4, 0, 0}, 96.294410, 0.000010, 0.021000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -4, 1, 0}, 348.797850, 0.000040, 0.030000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -2, -1, 0}, 351.830350, 0.000090, 0.028000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -2, 0, 0}, 355.715680, 0.000870, 0.045000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -2, 1, 0}, 357.265650, 0.000670, 0.113000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -2, 2, 0}, 2.000130, 0.000060, 0.227000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -1, 0, -1}, 9.967780, 0.000040, 0.043000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -1, 0, 0}, 204.517260, 0.000050, 0.103000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -1, 0, 1}, 190.206050, 0.000110, 0.272000},
	{{0, 1, 1, 0, 0, 0, 0, 0, -1, 1, -1}, 9.967770, 0.000010, 0.101000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 0, -2, 0}, 17.241820, 0.000020, 0.042000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 0, -1, 0}, 7.131550, 0.000410, 0.094000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0}, 22.283190, 0.002730, 0.381000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0}, 24.154930, 0.000270, 0.063000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0}, 18.782920, 0.000030, 0.034000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 1, 0, -1}, 12.684820, 0.000160, 0.636000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0}, 205.152820, 0.000050, 0.067000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1}, 191.705080, 0.000030, 0.035000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 2, -3, 0}, 280.061380, 0.000020, 0.081000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 2, -2, 0}, 13.604080, 0.000020, 1.174000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 2, -1, 0}, 21.706670, 0.000660, 0.071000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 2, 0, -2}, 26.744140, 0.000010, 1.931000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 0}, 222.691930, 0.000040, 0.037000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 3, -2, 0}, 23.894200, 0.000020, 0.076000},
	{{0, 1, 1, 0, 0, 0, 0, 0, 4, -2, 0}, 30.281790, 0.000010, 0.039000},
	{{0, 1, 1, 0, 2, 0, 0, 0, 2, -2, 0}, 308.159280, 0.000010, 0.980000},
	{{0, 1, 1, 0, 2, 0, 0, 0, 2, -1, 0}, 308.159280, 0.000030, 0.070000},
	{{0, 1, 1, 0, 2, 0, 0, 0, 2, 0, 0}, 128.159280, 0.000010, 0.036000},
	{{0, 1, 2, -6, 0, 0, 0, 0, 0, 0, 0}, 273.782650, 0.000010, 2.297000},
	{{0, 1, 2, -4, 0, 0, 0, 0, -2, 0, 0}, 330.647020, 0.000010, 0.043000},
	{{0, 1, 2, -4, 0, 0, 0, 0, -2, 2, 0}, 330.647020, 0.000010, 0.306000},
	{{0, 1, 2, -3, 0, 0, 0, 0, 2, -1, 0}, 266.522140, 0.000020, 0.074000},
	{{0, 1, 2, 0, -3, 0, 0, 0, 0, 0, 0}, 241.272780, 0.000010, 0.297000},
	{{0, 1, 2, 0, -2, 0, 0, 0, -2, 0, 0}, 269.307710, 0.000020, 0.047000},
	{{0, 1, 2, 0, -2, 0, 0, 0, -2, 2, 0}, 89.307710, 0.000020, 0.191000},
	{{0, 1, 2, 0, -2, 0, 0, 0, 0, 0, 0}, 90.174110, 0.000010, 0.289000},
	{{0, 1, 2, 0, 0, 0, 0, 0, -2, 0, 0}, 275.678630, 0.000090, 0.047000},
	{{0, 1, 2, 0, 0, 0, 0, 0, -2, 1, 0}, 282.047490, 0.000040, 0.127000},
	{{0, 1, 2, 0, 0, 0, 0, 0, -1, 0, 1}, 122.286300, 0.000020, 0.214000},
	{{0, 1, 2, 0, 0, 0, 0, 0, 0, -1, 0}, 305.597080, 0.000030, 0.104000},
	{{0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0}, 293.108730, 0.000200, 0.276000},
	{{0, 1, 2, 0, 0, 0, 0, 0, 0, 1, 0}, 291.985870, 0.000020, 0.059000},
	{{0, 1, 2, 0, 0, 0, 0, 0, 1, 0, -1}, 305.432530, 0.000020, 0.389000},
	{{0, 1, 2, 0, 0, 0, 0, 0, 2, -3, 0}, 219.459180, 0.000030, 0.088000},
	{{0, 1, 2, 0, 0, 0, 0, 0, 2, -2, 0}, 45.060260, 0.000040, 0.540000},
	{{0, 1, 2, 0, 0, 0, 0, 0, 2, -1, 0}, 271.033080, 0.000030, 0.066000},
	{{0, 1, 2, 0, 0, 0, 0, 0, 2, 0, 0}, 72.878440, 0.000040, 0.035000},
	{{0, 1, 2, 0, 0, 0, 0, 0, 2, 1, 0}, 55.710890, 0.000010, 0.024000},
	{{0, 1, 2, 0, 1, 0, 0, 0, 0, 0, 0}, 128.859340, 0.000020, 0.270000},
	{{0, 1, 2, 0, 1, 0, 0, 0, 2, -2, 0}, 126.895960, 0.000010, 0.517000},
	{{0, 1, 2, 0, 1, 0, 0, 0, 2, -1, 0}, 135.602730, 0.000010, 0.066000},
	{{0, 1, 2, 0, 1, 0, 0, 0, 2, 0, 0}, 308.444160, 0.000120, 0.035000},
	{{0, 1, 2, 3, 0, 0, 0, 0, 2, -2, 0}, 184.688010, 0.000010, 0.290000},
	{{0, 1, 2, 3, 0, 0, 0, 0, 2, 0, 0}, 184.688010, 0.000010, 0.033000},
	{{0, 1, 3, -8, 0, 0, 0, 0, 0, 0, 0}, 245.047220, 0.000010, 2.688000},
	{{0, 1, 3, -8, 3, 0, 0, 0, 0, 0, 0}, 108.118160, 0.000010, 1.600000},
	{{0, 1, 3, -5, 0, 0, 0, 0, 0, 0, 0}, 150.614760, 0.000010, 0.508000},
	{{0, 1, 3, 0, -1, 0, 0, 0, 2, -3, 0}, 320.599810, 0.000030, 0.095000},
	{{0, 1, 3, 0, -1, 0, 0, 0, 2, -2, 0}, 309.072810, 0.000050, 0.361000},
	{{0, 1, 3, 0, -1, 0, 0, 0, 2, -1, 0}, 320.599810, 0.000030, 0.062000},
	{{0, 1, 3, 0, -1, 0, 0, 0, 2, 0, 0}, 300.079990, 0.000050, 0.034000},
	{{0, 1, 3, 0, 0, 0, 0, 0, -2, 0, 0}, 89.463910, 0.000020, 0.050000},
	{{0, 1, 3, 0, 0, 0, 0, 0, -2, 2, 0}, 131.906710, 0.000010, 0.156000},
	{{0, 1, 3, 0, 0, 0, 0, 0, 0, -1, 0}, 128.213120, 0.000020, 0.116000},
	{{0, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0}, 208.258080, 0.000030, 0.216000},
	{{0, 1, 3, 0, 0, 0, 0, 0, 0, 1, 0}, 128.213120, 0.000020, 0.056000},
	{{0, 1, 3, 0, 0, 0, 0, 0, 1, 0, 0}, 128.215030, 0.000020, 0.059000},
	{{0, 1, 3, 0, 0, 0, 0, 0, 2, -2, 0}, 128.141340, 0.000030, 0.351000},
	{{0, 1, 3, 0, 0, 0, 0, 0, 2, -1, 0}, 306.652290, 0.000020, 0.062000},
	{{0, 1, 3, 0, 0, 0, 0, 0, 2, 0, -2}, 308.215030, 0.000030, 0.397000},
	{{0, 1, 3, 0, 0, 0, 0, 0, 2, 0, 0}, 129.298930, 0.000040, 0.034000},
	{{0, 1, 3, 0, 0, 0, 0, 0, 2, 2, 0}, 128.213110, 0.000020, 0.018000},
	{{0, 1, 3, 0, 1, 0, 0, 0, 2, -2, 0}, 114.559330, 0.000010, 0.341000},
	{{0, 1, 3, 0, 1, 0, 0, 0, 2, 0, 0}, 294.559330, 0.000010, 0.034000},
	{{0, 1, 4, -7, 0, 0, 0, 0, 2, -1, 0}, 303.842430, 0.000020, 0.075000},
	{{0, 1, 4, -2, 0, 0, 0, 0, 2, -2, 0}, 266.385690, 0.000020, 0.359000},
	{{0, 1, 4, -2, 0, 0, 0, 0, 2, 0, 0}, 266.385690, 0.000020, 0.034000},
	{{0, 1, 4, 0, -1, 0, 0, 0, 0, 0, 0}, 309.244820, 0.000030, 0.180000},
	{{0, 1, 4, 0, -1, 0, 0, 0, 2, -3, 0}, 309.247320, 0.000030, 0.105000},
	{{0, 1, 4, 0, -1, 0, 0, 0, 2, -2, 0}, 128.783300, 0.000020, 0.265000},
	{{0, 1, 4, 0, -1, 0, 0, 0, 2, -1, 0}, 309.247320, 0.000030, 0.059000},
	{{0, 1, 4, 0, -1, 0, 0, 0, 2, 0, 0}, 128.908370, 0.000030, 0.033000},
	{{0, 1, 4, 0, 0, 0, 0, 0, -2, 0, 0}, 302.148150, 0.000020, 0.052000},
	{{0, 1, 4, 0, 0, 0, 0, 0, -2, 2, 0}, 302.148150, 0.000020, 0.135000},
	{{0, 1, 4, 0, 0, 0, 0, 0, 0, -2, 0}, 122.083190, 0.000010, 0.048000},
	{{0, 1, 4, 0, 0, 0, 0, 0, 0, 0, 0}, 302.083190, 0.000010, 0.178000},
	{{0, 1, 4, 0, 0, 0, 0, 0, 2, -3, 0}, 37.362690, 0.000030, 0.106000},
	{{0, 1, 4, 0, 0, 0, 0, 0, 2, -2, 0}, 207.603970, 0.000050, 0.260000},
	{{0, 1, 4, 0, 0, 0, 0, 0, 2, -1, 0}, 59.942990, 0.000010, 0.058000},
	{{0, 1, 4, 0, 0, 0, 0, 0, 2, 0, 0}, 343.286910, 0.000020, 0.033000},
	{{0, 1, 4, 0, 0, 0, 0, 0, 2, 1, 0}, 34.495600, 0.000010, 0.023000},
	{{0, 1, 5, -9, 0, 0, 0, 0, 2, -1, 0}, 268.472900, 0.000030, 0.075000},
	{{0, 1, 5, -2, 0, 0, 0, 0, 2, -3, 0}, 128.304010, 0.000010, 0.106000},
	{{0, 1, 5, -2, 0, 0, 0, 0, 2, -2, 0}, 128.184570, 0.000020, 0.264000},
	{{0, 1, 5, -2, 0, 0, 0, 0, 2, -1, 0}, 128.304010, 0.000010, 0.059000},
	{{0, 1, 5, -2, 0, 0, 0, 0, 2, 0, 0}, 130.343360, 0.000040, 0.033000},
	{{0, 1, 5, 0, -3, 0, 0, 0, 2, 0, 0}, 138.992520, 0.000020, 0.032000},
	{{0, 1, 5, 0, -2, 0, 0, 0, 0, 0, 0}, 128.315210, 0.000010, 0.155000},
	{{0, 1, 5, 0, -2, 0, 0, 0, 0, 2, 0}, 308.315210, 0.000010, 0.030000},
	{{0, 1, 5, 0, -2, 0, 0, 0, 2, -3, 0}, 308.357600, 0.000020, 0.117000},
	{{0, 1, 5, 0, -2, 0, 0, 0, 2, -2, 0}, 308.314570, 0.000020, 0.214000},
	{{0, 1, 5, 0, -2, 0, 0, 0, 2, -1, 0}, 128.357600, 0.000020, 0.056000},
	{{0, 1, 5, 0, -1, 0, 0, 0, 2, -2, 0}, 38.698840, 0.000010, 0.210000},
	{{0, 1, 5, 0, -1, 0, 0, 0, 2, 0, 0}, 38.698840, 0.000010, 0.032000},
	{{0, 1, 5, 0, 0, 0, 0, 0, 0, -1, 0}, 152.012520, 0.000010, 0.151000},
	{{0, 1, 5, 0, 0, 0, 0, 0, 0, 1, 0}, 152.012520, 0.000010, 0.050000},
	{{0, 1, 5, 0, 0, 0, 0, 0, 2, 2, 0}, 152.012520, 0.000020, 0.017000},
	{{0, 1, 6, -11, 0, 0, 0, 0, 2, -3, 0}, 142.279770, 0.000020, 0.075000},
	{{0, 1, 6, -11, 0, 0, 0, 0, 2, -2, 0}, 67.731130, 0.000100, 335.665000},
	{{0, 1, 6, -11, 0, 0, 0, 0, 2, -1, 0}, 142.279770, 0.000020, 0.075000},
	{{0, 1, 6, 0, 0, 0, 0, 0, 2, 0, 0}, 239.442580, 0.000030, 0.031000},
	{{0, 1, 9, -20, 0, 0, 0, 0, 0, 0, 0}, 248.078080, 0.000010, 122.527000},
	{{0, 1, 29, -36, 0, 0, 0, 0, -2, 1, 0}, 10.307510, 0.000010, 317.190000},
	{{0, 1, 37, -51, 0, 0, 0, 0, -2, 1, 0}, 155.607010, 0.000020, 35.857000},
	{{0, 1, 43, -59, 0, 0, 0, 0, 0, -2, 0}, 97.328560, 0.000010, 0.075000},
	{{0, 1, 43, -59, 0, 0, 0, 0, 0, -1, 0}, 97.328560, 0.000240, 2513.483000},
	{{0, 1, 43, -59, 0, 0, 0, 0, 0, 0, 0}, 97.328560, 0.000010, 0.075000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -4, 1, 0}, 295.930280, 0.000010, 0.026000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -4, 2, 0}, 295.930280, 0.000090, 0.040000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -4, 3, 0}, 290.962450, 0.000150, 0.087000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -2, 0, 0}, 301.493730, 0.000010, 0.038000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -2, 1, 0}, 290.921120, 0.000280, 0.075000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -2, 2, -2}, 115.930280, 0.000020, 0.037000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -2, 2, 0}, 295.930280, 0.008080, 3054.914000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -2, 2, 2}, 115.930280, 0.000020, 0.037000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -2, 3, 0}, 321.439840, 0.000150, 0.075000},
	{{0, 2, -13, 15, 0, 0, 0, 0, -2, 4, 0}, 301.493730, 0.000010, 0.038000},
	{{0, 2, -13, 15, 0, 0, 0, 0, 0, 1, 0}, 298.251980, 0.000120, 0.087000},
	{{0, 2, -13, 15, 0, 0, 0, 0, 0, 2, 0}, 295.930280, 0.000090, 0.040000},
	{{0, 2, -13, 15, 0, 0, 0, 0, 0, 3, 0}, 295.930280, 0.000010, 0.026000},
	{{0, 2, -11, 0, 0, 0, 0, 0, -2, 0, 0}, 289.624680, 0.000010, 0.031000},
	{{0, 2, -11, 0, 0, 0, 0, 0, -2, 2, 0}, 289.624680, 0.000010, 0.167000},
	{{0, 2, -10, 11, 0, 0, 0, 0, 0, 0, 0}, 87.154110, 0.000010, 1.111000},
	{{0, 2, -9, 0, 0, 0, 0, 0, -2, 1, 0}, 121.158150, 0.000020, 0.058000},
	{{0, 2, -9, 0, 2, 0, 0, 0, -2, 0, 0}, 322.551820, 0.000020, 0.033000},
	{{0, 2, -9, 8, 0, 0, 0, 0, 0, 0, 0}, 46.496700, 0.000010, 0.669000},
	{{0, 2, -9, 11, 0, 0, 0, 0, 0, 0, 0}, 341.363430, 0.000010, 10.043000},
	{{0, 2, -8, 0, 0, 0, 0, 0, -2, -1, 0}, 213.239190, 0.000010, 0.023000},
	{{0, 2, -8, 0, 0, 0, 0, 0, -2, 1, 0}, 213.239190, 0.000010, 0.062000},
	{{0, 2, -8, 0, 0, 0, 0, 0, 0, 0, 0}, 200.155880, 0.000010, 0.211000},
	{{0, 2, -8, 0, 2, 0, 0, 0, -2, 0, 0}, 231.681770, 0.000040, 0.034000},
	{{0, 2, -8, 0, 2, 0, 0, 0, -2, 1, 0}, 231.637150, 0.000020, 0.062000},
	{{0, 2, -8, 0, 2, 0, 0, 0, 0, 0, 0}, 231.641330, 0.000020, 0.218000},
	{{0, 2, -8, 0, 3, 0, 0, 0, -2, 0, 0}, 221.890060, 0.000010, 0.034000},
	{{0, 2, -8, 0, 3, 0, 0, 0, -2, 2, 0}, 221.890060, 0.000010, 0.367000},
	{{0, 2, -8, 2, 0, 0, 0, 0, -2, 0, 0}, 231.908270, 0.000020, 0.035000},
	{{0, 2, -8, 2, 0, 0, 0, 0, -2, 1, 0}, 51.778350, 0.000030, 0.066000},
	{{0, 2, -8, 2, 0, 0, 0, 0, -2, 2, 0}, 231.908270, 0.000020, 0.523000},
	{{0, 2, -8, 2, 0, 0, 0, 0, -2, 3, 0}, 51.778350, 0.000030, 0.088000},
	{{0, 2, -8, 5, 0, 0, 0, 0, -2, 0, 2}, 322.264410, 0.000010, 58.829000},
	{{0, 2, -8, 6, 0, 0, 0, 0, 0, 0, 0}, 0.269230, 0.000010, 0.642000},
	{{0, 2, -8, 7, 0, 0, 0, 0, 0, 0, 0}, 17.336840, 0.000020, 0.974000},
	{{0, 2, -8, 7, 0, 0, 0, 0, 2, 0, 0}, 2.164350, 0.000020, 0.042000},
	{{0, 2, -8, 9, 0, 0, 0, 0, -2, 0, 0}, 263.805250, 0.000020, 0.040000},
	{{0, 2, -8, 9, 0, 0, 0, 0, -2, 1, 0}, 263.805250, 0.000020, 0.087000},
	{{0, 2, -8, 9, 0, 0, 0, 0, 0, -1, 0}, 261.442650, 0.000040, 0.076000},
	{{0, 2, -8, 9, 0, 0, 0, 0, 0, 0, 0}, 263.805250, 0.000170, 27.621000},
	{{0, 2, -8, 9, 0, 0, 0, 0, 0, 1, 0}, 261.442650, 0.000040, 0.075000},
	{{0, 2, -8, 9, 0, 0, 0, 0, 2, -1, 0}, 263.805250, 0.000020, 0.087000},
	{{0, 2, -8, 9, 0, 0, 0, 0, 2, 0, 0}, 263.805250, 0.000020, 0.040000},
	{{0, 2, -7, 0, 0, 0, 0, 0, -2, -1, 0}, 324.014960, 0.000020, 0.024000},
	{{0, 2, -7, 0, 0, 0, 0, 0, -2, 0, 0}, 138.460320, 0.000040, 0.035000},
	{{0, 2, -7, 0, 0, 0, 0, 0, -2, 1, 0}, 132.056830, 0.000040, 0.066000},
	{{0, 2, -7, 0, 0, 0, 0, 0, -2, 2, 0}, 344.659270, 0.000020, 0.506000},
	{{0, 2, -7, 0, 0, 0, 0, 0, -2, 3, 0}, 152.594400, 0.000050, 0.089000},
	{{0, 2, -7, 0, 0, 0, 0, 0, 0, -2, 0}, 142.755710, 0.000010, 0.033000},
	{{0, 2, -7, 0, 0, 0, 0, 0, 0, 0, 0}, 157.112470, 0.000050, 0.267000},
	{{0, 2, -7, 0, 0, 0, 0, 0, 0, 2, 0}, 114.128740, 0.000020, 0.044000},
	{{0, 2, -7, 0, 0, 0, 0, 0, 2, -1, 0}, 329.124960, 0.000050, 0.129000},
	{{0, 2, -7, 0, 1, 0, 0, 0, -2, 2, 0}, 51.208170, 0.000020, 0.529000},
	{{0, 2, -7, 0, 1, 0, 0, 0, -2, 3, 0}, 231.208180, 0.000010, 0.088000},
	{{0, 2, -7, 0, 1, 0, 0, 0, 0, -1, 0}, 230.468020, 0.000020, 0.059000},
	{{0, 2, -7, 0, 1, 0, 0, 0, 0, 1, 0}, 230.468020, 0.000020, 0.104000},
	{{0, 2, -7, 0, 2, 0, 0, 0, -2, 0, 0}, 291.822750, 0.000050, 0.035000},
	{{0, 2, -7, 2, 0, 0, 0, 0, -2, 0, 0}, 261.981030, 0.000010, 0.036000},
	{{0, 2, -7, 3, 0, 0, 0, 0, -2, 0, 0}, 288.626810, 0.000010, 0.037000},
	{{0, 2, -7, 3, 0, 0, 0, 0, -2, 0, 2}, 288.972600, 0.000040, 21.566000},
	{{0, 2, -7, 3, 0, 0, 0, 0, -2, 2, 0}, 288.626810, 0.000010, 2.632000},
	{{0, 2, -7, 6, 0, 0, 0, 0, 0, 0, 0}, 8.619850, 0.000010, 1.789000},
	{{0, 2, -7, 7, 0, 0, 0, 0, -2, 0, 0}, 234.685960, 0.000060, 0.040000},
	{{0, 2, -7, 7, 0, 0, 0, 0, -2, 1, 0}, 234.097210, 0.000080, 0.087000},
	{{0, 2, -7, 7, 0, 0, 0, 0, 0, -1, 0}, 237.694090, 0.000090, 0.075000},
	{{0, 2, -7, 7, 0, 0, 0, 0, 0, 0, 0}, 235.317610, 0.000530, 36.814000},
	{{0, 2, -7, 7, 0, 0, 0, 0, 0, 1, 0}, 237.666830, 0.000090, 0.076000},
	{{0, 2, -7, 7, 0, 0, 0, 0, 2, -1, 0}, 234.011320, 0.000090, 0.087000},
	{{0, 2, -7, 7, 0, 0, 0, 0, 2, 0, 0}, 234.595030, 0.000060, 0.040000},
	{{0, 2, -6, 0, -1, 0, 0, 0, -2, 0, 0}, 58.435990, 0.000020, 0.036000},
	{{0, 2, -6, 0, -1, 0, 0, 0, -2, 1, 0}, 65.743870, 0.000010, 0.070000},
	{{0, 2, -6, 0, -1, 0, 0, 0, -2, 2, 0}, 65.743870, 0.000010, 0.944000},
	{{0, 2, -6, 0, -1, 0, 0, 0, -2, 3, 0}, 245.743870, 0.000010, 0.082000},
	{{0, 2, -6, 0, 0, 0, 0, 0, -4, 0, 0}, 231.785210, 0.000020, 0.019000},
	{{0, 2, -6, 0, 0, 0, 0, 0, -4, 3, 0}, 64.640680, 0.000010, 0.080000},
	{{0, 2, -6, 0, 0, 0, 0, 0, -3, 2, 0}, 231.785210, 0.000010, 0.075000},
	{{0, 2, -6, 0, 0, 0, 0, 0, -3, 4, 0}, 231.785210, 0.000010, 0.076000},
	{{0, 2, -6, 0, 0, 0, 0, 0, -2, 0, 0}, 52.502170, 0.000100, 0.036000},
	{{0, 2, -6, 0, 0, 0, 0, 0, -2, 1, 0}, 276.422560, 0.000030, 0.070000},
	{{0, 2, -6, 0, 0, 0, 0, 0, -2, 2, 0}, 51.905650, 0.000070, 1.026000},
	{{0, 2, -6, 0, 0, 0, 0, 0, -2, 3, 0}, 237.353910, 0.000080, 0.081000},
	{{0, 2, -6, 0, 0, 0, 0, 0, 0, -1, 0}, 51.908290, 0.000010, 0.062000},
	{{0, 2, -6, 0, 0, 0, 0, 0, 0, 0, 0}, 324.576320, 0.000050, 0.364000},
	{{0, 2, -6, 0, 0, 0, 0, 0, 0, 1, 0}, 31.255670, 0.000040, 0.095000},
	{{0, 2, -6, 0, 0, 0, 0, 0, 2, 0, 0}, 214.434700, 0.000020, 0.045000},
	{{0, 2, -6, 0, 1, 0, 0, 0, -2, 0, 0}, 324.960300, 0.000030, 0.036000},
	{{0, 2, -6, 0, 1, 0, 0, 0, -2, 2, 0}, 220.950940, 0.000010, 1.123000},
	{{0, 2, -6, 2, 0, 0, 0, 0, -2, 1, 0}, 213.892080, 0.000020, 0.076000},
	{{0, 2, -6, 2, 0, 0, 0, 0, -2, 2, 0}, 4.225950, 0.000010, 11.308000},
	{{0, 2, -6, 2, 0, 0, 0, 0, -2, 3, 0}, 213.892080, 0.000020, 0.075000},
	{{0, 2, -6, 3, 0, 0, 0, 0, 0, 0, 0}, 318.905820, 0.000020, 0.867000},
	{{0, 2, -6, 5, 0, 0, 0, 0, -2, 0, 0}, 208.894210, 0.000020, 0.040000},
	{{0, 2, -6, 5, 0, 0, 0, 0, 0, 0, 0}, 222.985490, 0.000080, 11.046000},
	{{0, 2, -6, 5, 0, 0, 0, 0, 2, 0, 0}, 208.894210, 0.000010, 0.041000},
	{{0, 2, -6, 6, 0, 0, 0, 0, 0, 0, 0}, 56.338190, 0.000020, 2.267000},
	{{0, 2, -6, 8, -3, 0, 0, 0, -2, 0, 0}, 252.168810, 0.000020, 0.043000},
	{{0, 2, -6, 8, -3, 0, 0, 0, 0, 0, 0}, 251.997270, 0.000030, 0.799000},
	{{0, 2, -5, -3, 0, 0, 0, 0, -2, 0, 0}, 355.312640, 0.000010, 0.036000},
	{{0, 2, -5, -3, 0, 0, 0, 0, -2, 1, 0}, 175.312640, 0.000010, 0.067000},
	{{0, 2, -5, -3, 0, 0, 0, 0, -2, 2, 0}, 355.312640, 0.000010, 0.637000},
	{{0, 2, -5, -3, 0, 0, 0, 0, -2, 3, 0}, 175.312640, 0.000010, 0.086000},
	{{0, 2, -5, -2, 0, 0, 0, 0, -2, 0, 0}, 11.108360, 0.000030, 0.036000},
	{{0, 2, -5, -2, 0, 0, 0, 0, -2, 2, 0}, 11.024870, 0.000020, 0.963000},
	{{0, 2, -5, 0, -2, 0, 0, 0, -2, 1, 0}, 341.494810, 0.000070, 0.075000},
	{{0, 2, -5, 0, -1, 0, 0, 0, -2, 0, 0}, 235.789120, 0.000010, 0.038000},
	{{0, 2, -5, 0, -1, 0, 0, 0, -2, 1, 0}, 151.729350, 0.000040, 0.075000},
	{{0, 2, -5, 0, 0, -1, 0, 0, -2, 1, 0}, 233.891100, 0.000030, 0.075000},
	{{0, 2, -5, 0, 0, -1, 0, 0, -2, 2, 0}, 63.365830, 0.000020, 112.558000},
	{{0, 2, -5, 0, 0, 0, -2, 0, -2, 1, 0}, 338.080500, 0.000040, 0.075000},
	{{0, 2, -5, 0, 0, 0, -2, 0, -2, 2, 0}, 175.299490, 0.000110, 793.895000},
	{{0, 2, -5, 0, 0, 0, 0, 0, -4, 0, 0}, 116.997220, 0.000020, 0.020000},
	{{0, 2, -5, 0, 0, 0, 0, 0, -4, 1, 0}, 67.668580, 0.000020, 0.026000},
	{{0, 2, -5, 0, 0, 0, 0, 0, -4, 2, 0}, 68.108320, 0.000040, 0.040000},
	{{0, 2, -5, 0, 0, 0, 0, 0, -2, -2, 0}, 133.865770, 0.000010, 0.019000},
	{{0, 2, -5, 0, 0, 0, 0, 0, -2, 0, 0}, 79.937040, 0.000170, 0.038000},
	{{0, 2, -5, 0, 0, 0, 0, 0, -2, 1, 0}, 67.610400, 0.002000, 0.076000},
	{{0, 2, -5, 0, 0, 0, 0, 0, -2, 2, 0}, 69.088970, 0.001330, 39.899000},
	{{0, 2, -5, 0, 0, 0, 0, 0, -2, 3, 0}, 54.541150, 0.000120, 0.075000},
	{{0, 2, -5, 0, 0, 0, 0, 0, -1, 0, 1}, 238.622270, 0.000070, 1.438000},
	{{0, 2, -5, 0, 0, 0, 0, 0, 0, -1, 0}, 241.177390, 0.000040, 0.067000},
	{{0, 2, -5, 0, 0, 0, 0, 0, 0, 0, 0}, 238.873860, 0.000370, 0.572000},
	{{0, 2, -5, 0, 0, 0, 0, 0, 0, 1, 0}, 244.149030, 0.000420, 0.087000},
	{{0, 2, -5, 0, 0, 0, 0, 0, 1, 0, -1}, 62.241390, 0.000040, 0.357000},
	{{0, 2, -5, 0, 0, 0, 0, 0, 2, -1, 0}, 131.451830, 0.000130, 0.103000},
	{{0, 2, -5, 0, 0, 0, 0, 0, 2, 0, 0}, 146.961170, 0.000140, 0.044000},
	{{0, 2, -5, 0, 0, 0, 0, 0, 2, 1, 0}, 129.727600, 0.000020, 0.028000},
	{{0, 2, -5, 0, 0, 1, 0, 0, -2, 1, 0}, 48.700030, 0.000030, 0.076000},
	{{0, 2, -5, 0, 1, 0, 0, 0, -2, 1, 0}, 60.006450, 0.000050, 0.076000},
	{{0, 2, -5, 0, 1, 0, 0, 0, -2, 2, 0}, 230.590940, 0.000010, 9.144000},
	{{0, 2, -5, 0, 1, 0, 0, 0, -2, 3, 0}, 121.487780, 0.000010, 0.075000},
	{{0, 2, -5, 0, 2, 0, 0, 0, -2, 1, 0}, 270.714560, 0.000020, 0.077000},
	{{0, 2, -5, 2, 0, 0, 0, 0, 0, 0, 0}, 286.812650, 0.000010, 1.459000},
	{{0, 2, -5, 3, 0, 0, 0, 0, 0, 0, 0}, 187.799060, 0.000050, 6.498000},
	{{0, 2, -5, 4, 0, 0, 0, 0, 0, -1, 0}, 209.597610, 0.000020, 0.078000},
	{{0, 2, -5, 4, 0, 0, 0, 0, 0, 1, 0}, 209.597610, 0.000020, 0.073000},
	{{0, 2, -5, 4, 0, 0, 0, 0, 2, -2, 0}, 29.597610, 0.000010, 0.716000},
	{{0, 2, -5, 4, 0, 0, 0, 0, 2, 0, 0}, 29.597610, 0.000010, 0.040000},
	{{0, 2, -5, 5, 0, 0, 0, 0, 0, 0, 0}, 57.524260, 0.000020, 1.100000},
	{{0, 2, -5, 6, 0, 0, 0, 0, 0, 0, 0}, 86.339470, 0.000010, 0.694000},
	{{0, 2, -4, -2, 0, 0, 0, 0, -2, 1, 0}, 307.206580, 0.000020, 0.075000},
	{{0, 2, -4, -2, 0, 0, 0, 0, -2, 2, 0}, 109.868890, 0.000020, 26.107000},
	{{0, 2, -4, -2, 0, 0, 0, 0, -2, 3, 0}, 307.206580, 0.000020, 0.076000},
	{{0, 2, -4, 0, -2, 0, 0, 0, -2, 0, 0}, 51.841370, 0.000020, 0.039000},
	{{0, 2, -4, 0, -2, 0, 0, 0, -2, 1, 0}, 51.841370, 0.000030, 0.081000},
	{{0, 2, -4, 0, -1, 0, 0, 0, 0, 0, 0}, 237.298180, 0.000010, 1.200000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -4, 0, 0}, 355.591920, 0.000010, 0.020000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -4, 1, 0}, 194.257850, 0.000020, 0.027000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -2, -1, 0}, 186.326040, 0.000030, 0.026000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -2, 0, 0}, 195.598470, 0.000270, 0.039000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -2, 0, 2}, 193.070080, 0.000020, 0.736000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -2, 1, 0}, 325.831010, 0.000330, 0.082000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -2, 2, 0}, 349.183190, 0.000040, 0.976000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -2, 3, 0}, 187.504470, 0.000020, 0.070000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, -1, 0}, 158.339370, 0.000040, 0.038000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, -1, 1}, 350.498900, 0.000040, 0.077000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, 0, -1}, 293.488410, 0.000020, 0.038000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, 0, 0}, 157.370630, 0.000530, 0.076000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, 0, 1}, 359.767090, 0.000070, 3.281000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, 1, -1}, 352.646150, 0.000560, 0.075000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, 1, 0}, 155.689340, 0.000150, 7.244000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, 1, 1}, 350.498900, 0.000040, 0.074000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, 2, -1}, 172.553350, 0.000310, 34.842000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, 2, 0}, 182.475600, 0.000020, 0.075000},
	{{0, 2, -4, 0, 0, 0, 0, 0, -1, 3, -1}, 172.502490, 0.000010, 0.076000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 0, -2, 0}, 166.441880, 0.000080, 0.037000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 0, -1, 0}, 152.778660, 0.000780, 0.071000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 0, 0, 0}, 129.371640, 0.004130, 1.335000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 0, 1, 0}, 101.161670, 0.000600, 0.080000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 0, 2, 0}, 88.760600, 0.000040, 0.039000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 1, -1, 0}, 337.822410, 0.000010, 0.611000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 1, -1, 1}, 169.607750, 0.000050, 0.085000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 1, 0, -1}, 349.879070, 0.000180, 0.555000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 1, 0, 0}, 334.765510, 0.000170, 0.086000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 1, 0, 1}, 169.762570, 0.000050, 0.040000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 1, 1, -1}, 173.009660, 0.000120, 0.087000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 2, -2, 0}, 28.319430, 0.000090, 0.396000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 2, -1, 0}, 19.213720, 0.002590, 0.093000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 2, 0, -2}, 18.116400, 0.000040, 0.350000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 2, 0, 0}, 19.008720, 0.002260, 0.042000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 2, 1, 0}, 16.500780, 0.000240, 0.027000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 2, 2, 0}, 19.142950, 0.000020, 0.020000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 4, -2, 0}, 16.619750, 0.000050, 0.045000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 4, -1, 0}, 21.331650, 0.000070, 0.028000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 4, 0, 0}, 65.872770, 0.000020, 0.021000},
	{{0, 2, -4, 0, 0, 0, 0, 0, 4, 1, 0}, 191.389510, 0.000010, 0.016000},
	{{0, 2, -4, 0, 1, 0, 0, 0, 0, 0, 0}, 251.130380, 0.000030, 1.505000},
	{{0, 2, -4, 0, 1, 0, 0, 0, 2, 0, 0}, 89.354580, 0.000050, 0.042000},
	{{0, 2, -4, 0, 2, 0, 0, 0, -2, 0, 0}, 179.906810, 0.000030, 0.039000},
	{{0, 2, -4, 0, 2, 0, 0, 0, -2, 1, 0}, 179.906810, 0.000020, 0.083000},
	{{0, 2, -4, 0, 2, 0, 0, 0, 0, -2, 0}, 180.433710, 0.000010, 0.037000},
	{{0, 2, -4, 0, 2, 0, 0, 0, 0, -1, 0}, 179.826510, 0.000020, 0.072000},
	{{0, 2, -4, 0, 2, 0, 0, 0, 0, 0, 0}, 181.319430, 0.000130, 1.723000},
	{{0, 2, -4, 0, 2, 0, 0, 0, 0, 1, 0}, 179.826510, 0.000020, 0.079000},
	{{0, 2, -4, 0, 2, 0, 0, 0, 0, 2, 0}, 180.272600, 0.000010, 0.039000},
	{{0, 2, -4, 0, 2, 0, 0, 0, 2, -1, 0}, 179.849360, 0.000030, 0.092000},
	{{0, 2, -4, 0, 2, 0, 0, 0, 2, 0, 0}, 179.849360, 0.000030, 0.041000},
	{{0, 2, -4, 0, 3, 0, 0, 0, 0, 0, 0}, 174.936350, 0.000020, 2.016000},
	{{0, 2, -4, 1, 0, 0, 0, 0, 0, 0, 0}, 276.844930, 0.000030, 4.603000},
	{{0, 2, -4, 2, 0, 0, 0, 0, -2, 0, 0}, 358.684060, 0.000030, 0.041000},
	{{0, 2, -4, 2, 0, 0, 0, 0, -2, 2, 0}, 356.716620, 0.000010, 0.479000},
	{{0, 2, -4, 2, 0, 0, 0, 0, 0, -1, 0}, 359.103990, 0.000040, 0.077000},
	{{0, 2, -4, 2, 0, 0, 0, 0, 0, 0, 0}, 1.203600, 0.000090, 3.181000},
	{{0, 2, -4, 2, 0, 0, 0, 0, 0, 1, 0}, 359.103990, 0.000040, 0.074000},
	{{0, 2, -4, 2, 0, 0, 0, 0, 2, -2, 0}, 179.456590, 0.000010, 0.685000},
	{{0, 2, -4, 2, 0, 0, 0, 0, 2, 0, 0}, 179.917060, 0.000040, 0.040000},
	{{0, 2, -4, 4, 0, 0, 0, 0, -2, 0, 0}, 57.975690, 0.000010, 0.043000},
	{{0, 2, -4, 4, 0, 0, 0, 0, -2, 2, 0}, 57.975690, 0.000010, 0.317000},
	{{0, 2, -4, 4, 0, 0, 0, 0, 2, -3, 0}, 61.473550, 0.000030, 0.073000},
	{{0, 2, -4, 4, 0, 0, 0, 0, 2, -1, 0}, 61.473550, 0.000030, 0.078000},
	{{0, 2, -3, -4, 0, 0, 0, 0, -2, 1, 0}, 253.088800, 0.000020, 0.075000},
	{{0, 2, -3, -4, 0, 0, 0, 0, -2, 3, 0}, 253.088800, 0.000020, 0.076000},
	{{0, 2, -3, -1, 0, 0, 0, 0, 0, 0, 0}, 286.822920, 0.000010, 3.563000},
	{{0, 2, -3, 0, -3, 0, 0, 0, -2, 0, 0}, 300.944780, 0.000020, 0.040000},
	{{0, 2, -3, 0, -3, 0, 0, 0, -2, 1, 0}, 318.416510, 0.000030, 0.087000},
	{{0, 2, -3, 0, -3, 0, 0, 0, 0, -1, 0}, 231.886550, 0.000040, 0.075000},
	{{0, 2, -3, 0, -3, 0, 0, 0, 0, 0, 0}, 4.963850, 0.000030, 535.690000},
	{{0, 2, -3, 0, -3, 0, 0, 0, 0, 1, 0}, 231.886550, 0.000040, 0.075000},
	{{0, 2, -3, 0, -3, 0, 0, 0, 2, -1, 0}, 318.423740, 0.000030, 0.087000},
	{{0, 2, -3, 0, -3, 0, 0, 0, 2, 0, 0}, 300.919860, 0.000020, 0.040000},
	{{0, 2, -3, 0, -2, 0, 0, 0, 0, 0, 0}, 101.064350, 0.000020, 12.131000},
	{{0, 2, -3, 0, -1, 0, 0, 0, 0, -1, 0}, 301.029220, 0.000030, 0.076000},
	{{0, 2, -3, 0, -1, 0, 0, 0, 0, 0, 0}, 120.079860, 0.000030, 5.997000},
	{{0, 2, -3, 0, -1, 0, 0, 0, 0, 1, 0}, 301.029220, 0.000030, 0.075000},
	{{0, 2, -3, 0, 0, -1, 0, 0, 0, 0, 0}, 356.198400, 0.000010, 4.606000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -6, 1, 0}, 272.100540, 0.000020, 0.016000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -6, 2, 0}, 271.165870, 0.000030, 0.021000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -4, -1, 0}, 270.613270, 0.000070, 0.016000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -4, 0, 0}, 271.973850, 0.000530, 0.020000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -4, 1, 0}, 271.617430, 0.001210, 0.028000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -4, 1, 2}, 271.553650, 0.000020, 0.110000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -4, 2, 0}, 271.603140, 0.000820, 0.044000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -4, 3, 0}, 271.135610, 0.000020, 0.106000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -3, -1, 0}, 271.765330, 0.000010, 0.020000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -3, 0, 1}, 256.630850, 0.000010, 0.043000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -3, 1, 0}, 90.862460, 0.000050, 0.042000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -3, 1, 1}, 20.608200, 0.000020, 0.098000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, -3, 0}, 271.208960, 0.000020, 0.016000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, -2, -2}, 91.816240, 0.000010, 0.013000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, -2, 0}, 271.140150, 0.000330, 0.020000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, -1, -2}, 91.042910, 0.000020, 0.015000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, -1, 0}, 271.161020, 0.003950, 0.026000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, -1, 2}, 90.987290, 0.000090, 0.092000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 0, -2}, 90.574760, 0.000080, 0.019000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 0, 0}, 271.380940, 0.042140, 0.041000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 0, 1}, 256.577670, 0.000030, 0.090000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 0, 2}, 271.584330, 0.000820, 0.424000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 1, -2}, 91.415250, 0.000140, 0.026000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 1, -1}, 257.324780, 0.000010, 0.041000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 1, 0}, 271.566250, 0.061880, 0.089000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 1, 2}, 270.931530, 0.000030, 0.064000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 2, 0}, 271.644860, 0.002320, 0.494000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -2, 3, 0}, 271.829290, 0.000150, 0.065000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, -2, 1}, 256.610860, 0.000020, 0.040000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, -1, -1}, 112.473710, 0.000010, 0.026000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, -1, 0}, 90.646960, 0.000150, 0.039000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, -1, 1}, 256.624610, 0.000380, 0.084000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, 0, -1}, 76.780700, 0.000120, 0.039000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, 0, 0}, 90.628550, 0.001480, 0.083000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, 0, 1}, 256.869390, 0.000670, 0.766000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, 1, -1}, 77.865330, 0.000290, 0.081000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, 1, 0}, 91.721010, 0.000170, 0.879000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, 1, 1}, 256.877020, 0.000130, 0.069000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, 2, -1}, 76.640160, 0.000030, 1.030000},
	{{0, 2, -3, 0, 0, 0, 0, 0, -1, 2, 0}, 91.242770, 0.000030, 0.069000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, -4, 0}, 267.795380, 0.000010, 0.019000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, -3, 0}, 268.023000, 0.000270, 0.025000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, -2, -2}, 89.057040, 0.000030, 0.019000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, -2, 0}, 268.688920, 0.004080, 0.038000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, -1, -2}, 89.161740, 0.000160, 0.025000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, -1, -1}, 257.339270, 0.000060, 0.038000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, -1, 0}, 268.696480, 0.060300, 0.077000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, -1, 1}, 76.364310, 0.000020, 2.394000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, -1, 2}, 269.828810, 0.000070, 0.072000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 0, -2}, 91.748620, 0.000150, 0.038000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 0, -1}, 257.323450, 0.000890, 0.076000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 0, 0}, 269.953930, 0.343550, 3.983000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 0, 1}, 76.577670, 0.000160, 0.073000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 0, 2}, 90.693800, 0.000210, 0.037000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 1, -2}, 262.046510, 0.000120, 0.075000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 1, -1}, 256.565240, 0.000330, 11.864000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 1, 0}, 268.568750, 0.055400, 0.074000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 1, 2}, 88.793910, 0.000120, 0.025000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 2, -2}, 79.077920, 0.000010, 12.126000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 2, -1}, 257.259060, 0.000020, 0.075000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 2, 0}, 268.571880, 0.003780, 0.037000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 2, 2}, 89.077110, 0.000030, 0.019000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 3, 0}, 268.714980, 0.000250, 0.025000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 0, 4, 0}, 267.530980, 0.000010, 0.019000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, -2, -1}, 256.579240, 0.000030, 0.037000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, -2, 1}, 76.595940, 0.000080, 2.131000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, -1, -1}, 256.576480, 0.000450, 0.071000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, -1, 0}, 266.530070, 0.000060, 1.572000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, -1, 1}, 76.653230, 0.001180, 0.078000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, 0, -1}, 256.564540, 0.003970, 1.246000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, 0, 0}, 119.922420, 0.000080, 0.079000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, 0, 1}, 77.129050, 0.000830, 0.038000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, 1, -1}, 256.481230, 0.000560, 0.080000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, 1, 0}, 87.545290, 0.000020, 0.039000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, 1, 1}, 76.992480, 0.000070, 0.025000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, 2, -1}, 256.578750, 0.000030, 0.039000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 1, 2, 0}, 270.517200, 0.000010, 0.026000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, -3, 0}, 271.300190, 0.000040, 0.068000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, -2, 0}, 272.137490, 0.001060, 0.657000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, -2, 2}, 90.667800, 0.000010, 0.039000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, -1, 0}, 271.018200, 0.034510, 0.085000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, -1, 2}, 91.215510, 0.000070, 0.026000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, 0, -2}, 269.815140, 0.000580, 0.539000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, 0, -1}, 77.464820, 0.000220, 0.086000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, 0, 0}, 269.986470, 0.021720, 0.040000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, 0, 2}, 90.326060, 0.000050, 0.019000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, 1, -2}, 90.238460, 0.000070, 0.088000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, 1, -1}, 77.324780, 0.000010, 0.040000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, 1, 0}, 269.383620, 0.002130, 0.026000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 2, 2, 0}, 269.711750, 0.000210, 0.019000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 3, -3, -1}, 76.636050, 0.000020, 0.063000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 3, -2, 1}, 69.652210, 0.000010, 0.041000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 3, -1, 0}, 90.146570, 0.000020, 0.041000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 3, 0, -1}, 256.643680, 0.000010, 0.042000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 3, 0, 0}, 270.041780, 0.000030, 0.027000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 4, -3, 0}, 273.674810, 0.000010, 0.100000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 4, -2, 0}, 268.558720, 0.000370, 0.043000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 4, -1, -2}, 270.409410, 0.000010, 0.104000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 4, -1, 0}, 268.275150, 0.000570, 0.027000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 4, 0, -2}, 270.517200, 0.000030, 0.044000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 4, 0, 0}, 267.538740, 0.000220, 0.020000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 4, 1, 0}, 271.365020, 0.000050, 0.016000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 5, -2, 0}, 270.146580, 0.000010, 0.028000},
	{{0, 2, -3, 0, 0, 0, 0, 0, 6, -2, 0}, 275.208950, 0.000010, 0.021000},
	{{0, 2, -3, 0, 0, 1, 0, 0, 0, -1, 0}, 179.783680, 0.000010, 0.077000},
	{{0, 2, -3, 0, 0, 1, 0, 0, 0, 0, 0}, 359.739860, 0.000010, 3.509000},
	{{0, 2, -3, 0, 0, 1, 0, 0, 0, 1, 0}, 179.783680, 0.000010, 0.074000},
	{{0, 2, -3, 0, 0, 2, 0, 0, 0, 0, 0}, 93.561810, 0.000010, 3.135000},
	{{0, 2, -3, 0, 1, 0, 0, 0, -2, 0, 0}, 175.482050, 0.000090, 0.041000},
	{{0, 2, -3, 0, 1, 0, 0, 0, -2, 1, 0}, 3.029820, 0.000050, 0.090000},
	{{0, 2, -3, 0, 1, 0, 0, 0, -2, 2, 0}, 358.646630, 0.000010, 0.474000},
	{{0, 2, -3, 0, 1, 0, 0, 0, 0, 0, 0}, 359.255510, 0.000280, 2.982000},
	{{0, 2, -3, 0, 1, 0, 0, 0, 0, 1, -2}, 181.356040, 0.000030, 0.075000},
	{{0, 2, -3, 0, 1, 0, 0, 0, 0, 2, -2}, 134.651510, 0.000090, 544.385000},
	{{0, 2, -3, 0, 1, 0, 0, 0, 0, 2, 0}, 178.717380, 0.000020, 0.037000},
	{{0, 2, -3, 0, 1, 0, 0, 0, 2, -1, 0}, 358.560510, 0.000030, 0.085000},
	{{0, 2, -3, 0, 1, 0, 0, 0, 2, 0, 0}, 177.271930, 0.000010, 0.040000},
	{{0, 2, -3, 0, 2, 0, 0, 0, -2, 0, 0}, 51.451150, 0.000030, 0.041000},
	{{0, 2, -3, 0, 2, 0, 0, 0, 0, 0, 0}, 55.782230, 0.000050, 2.383000},
	{{0, 2, -3, 0, 2, 0, 0, 0, 2, -2, 0}, 60.194910, 0.000030, 0.738000},
	{{0, 2, -3, 0, 2, 0, 0, 0, 2, -1, 0}, 60.309830, 0.000060, 0.084000},
	{{0, 2, -3, 0, 2, 0, 0, 0, 2, 0, 0}, 60.194910, 0.000030, 0.040000},
	{{0, 2, -3, 0, 3, 0, 0, 0, 0, 0, 0}, 65.492880, 0.000020, 1.984000},
	{{0, 2, -3, 1, 0, 0, 0, 0, 0, 0, 0}, 359.269140, 0.000020, 1.278000},
	{{0, 2, -3, 2, 0, 0, 0, 0, -2, 0, 0}, 40.174710, 0.000010, 0.043000},
	{{0, 2, -3, 2, 0, 0, 0, 0, -2, 1, 0}, 40.510860, 0.000010, 0.098000},
	{{0, 2, -3, 2, 0, 0, 0, 0, -2, 2, 0}, 40.236630, 0.000030, 0.324000},
	{{0, 2, -3, 2, 0, 0, 0, 0, -2, 3, 0}, 40.510860, 0.000010, 0.061000},
	{{0, 2, -3, 2, 0, 0, 0, 0, 0, 0, 0}, 256.271720, 0.000020, 0.761000},
	{{0, 2, -3, 2, 0, 0, 0, 0, 2, -2, 0}, 44.719050, 0.000030, 2.176000},
	{{0, 2, -3, 2, 0, 0, 0, 0, 2, 0, 0}, 45.367310, 0.000020, 0.038000},
	{{0, 2, -3, 3, 0, 0, 0, 0, 2, -3, 0}, 237.604050, 0.000030, 0.076000},
	{{0, 2, -3, 3, 0, 0, 0, 0, 2, -2, 0}, 236.879090, 0.000010, 13.873000},
	{{0, 2, -3, 3, 0, 0, 0, 0, 2, -1, 0}, 231.221380, 0.000050, 0.075000},
	{{0, 2, -3, 3, 0, 0, 0, 0, 2, 0, 0}, 236.879090, 0.000010, 0.038000},
	{{0, 2, -2, -2, 0, 0, 0, 0, 0, -1, 0}, 47.368060, 0.000010, 0.077000},
	{{0, 2, -2, -2, 0, 0, 0, 0, 0, 0, 0}, 151.689150, 0.000010, 5.328000},
	{{0, 2, -2, -2, 0, 0, 0, 0, 0, 1, 0}, 47.368060, 0.000010, 0.074000},
	{{0, 2, -2, -1, 0, 0, 0, 0, 1, -3, 1}, 166.282350, 0.000010, 0.075000},
	{{0, 2, -2, -1, 0, 0, 0, 0, 1, -2, 1}, 166.282350, 0.000270, 1124.126000},
	{{0, 2, -2, -1, 0, 0, 0, 0, 1, -1, 1}, 166.282350, 0.000010, 0.075000},
	{{0, 2, -2, 0, -2, 0, 0, 0, -2, 0, 0}, 205.439510, 0.000010, 0.042000},
	{{0, 2, -2, 0, -2, 0, 0, 0, 0, 0, 0}, 341.340300, 0.000010, 0.924000},
	{{0, 2, -2, 0, -2, 0, 0, 0, 1, 0, -1}, 175.320190, 0.000010, 34.878000},
	{{0, 2, -2, 0, -1, 0, 0, 0, -2, 0, 0}, 224.931970, 0.000090, 0.042000},
	{{0, 2, -2, 0, -1, 0, 0, 0, -2, 1, 0}, 12.460780, 0.000030, 0.097000},
	{{0, 2, -2, 0, -1, 0, 0, 0, -2, 2, 0}, 9.796770, 0.000040, 0.340000},
	{{0, 2, -2, 0, -1, 0, 0, 0, -2, 3, 0}, 192.042370, 0.000020, 0.062000},
	{{0, 2, -2, 0, -1, 0, 0, 0, 0, -2, 0}, 193.056320, 0.000010, 0.039000},
	{{0, 2, -2, 0, -1, 0, 0, 0, 0, -1, 0}, 11.569310, 0.000010, 0.083000},
	{{0, 2, -2, 0, -1, 0, 0, 0, 0, 0, 0}, 86.450170, 0.000020, 0.857000},
	{{0, 2, -2, 0, -1, 0, 0, 0, 0, 1, 0}, 11.569310, 0.000010, 0.069000},
	{{0, 2, -2, 0, -1, 0, 0, 0, 1, 0, -1}, 76.961450, 0.000020, 8.852000},
	{{0, 2, -2, 0, -1, 0, 0, 0, 2, -2, 0}, 193.996210, 0.000010, 1.647000},
	{{0, 2, -2, 0, -1, 0, 0, 0, 2, -1, 0}, 192.205450, 0.000050, 0.079000},
	{{0, 2, -2, 0, -1, 0, 0, 0, 2, 0, 0}, 191.765440, 0.000060, 0.039000},
	{{0, 2, -2, 0, 0, -1, 0, 0, 0, 0, 0}, 270.412060, 0.000010, 0.822000},
	{{0, 2, -2, 0, 0, -1, 0, 0, 2, -1, 0}, 92.482940, 0.000030, 0.079000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -6, 0, 0}, 359.927160, 0.000030, 0.014000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -6, 1, 0}, 0.439300, 0.000030, 0.017000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -6, 2, 0}, 0.540230, 0.000070, 0.022000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -6, 3, 0}, 0.367630, 0.000020, 0.030000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -5, 2, 0}, 180.403730, 0.000010, 0.029000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, -2, 0}, 359.900940, 0.000030, 0.013000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, -1, 0}, 359.430530, 0.000200, 0.016000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, -1, 2}, 0.308380, 0.000010, 0.029000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, 0, -2}, 180.323430, 0.000010, 0.013000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, 0, 0}, 0.707390, 0.001480, 0.021000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, 0, 2}, 0.281340, 0.000050, 0.047000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, 1, -2}, 180.304970, 0.000010, 0.016000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, 1, 0}, 0.582060, 0.003460, 0.029000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, 1, 2}, 0.368110, 0.000020, 0.123000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, 2, 0}, 0.490960, 0.002130, 0.046000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -4, 3, 0}, 0.480420, 0.000050, 0.118000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -3, -1, 0}, 0.434440, 0.000040, 0.020000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -3, 0, 0}, 1.209650, 0.000040, 0.028000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -3, 1, 0}, 180.305910, 0.000220, 0.044000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -3, 2, 0}, 180.433210, 0.000080, 0.107000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, -3, 0}, 0.290290, 0.000080, 0.016000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, -2, 0}, 0.284590, 0.000840, 0.020000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, -2, 2}, 180.324760, 0.000010, 0.043000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, -1, -2}, 180.292380, 0.000070, 0.016000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, -1, 0}, 0.446800, 0.011200, 0.027000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, -1, 2}, 180.300060, 0.000240, 0.101000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 0, -2}, 180.302780, 0.000320, 0.020000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 0, 0}, 0.366590, 0.134320, 0.043000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 0, 2}, 0.350100, 0.001540, 0.298000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 1, -2}, 180.304590, 0.000370, 0.027000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 1, 0}, 0.311770, 0.160900, 0.098000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 1, 2}, 180.291560, 0.000080, 0.060000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 2, -2}, 180.299000, 0.000030, 0.042000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 2, 0}, 0.227640, 0.007040, 0.331000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 2, 2}, 180.283260, 0.000030, 0.033000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 3, 0}, 0.232370, 0.000540, 0.061000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -2, 4, 0}, 0.221170, 0.000050, 0.034000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -1, -2, 0}, 180.607660, 0.000020, 0.027000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -1, -1, 0}, 180.119850, 0.000220, 0.041000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -1, -1, 1}, 185.454640, 0.000030, 0.091000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -1, 0, -1}, 188.365370, 0.000110, 0.041000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -1, 0, 0}, 180.064280, 0.002760, 0.090000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -1, 0, 1}, 3.668320, 0.000200, 0.434000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -1, 1, -1}, 11.977360, 0.000030, 0.089000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -1, 1, 0}, 180.103360, 0.000290, 0.468000},
	{{0, 2, -2, 0, 0, 0, 0, 0, -1, 2, 0}, 179.933860, 0.000030, 0.065000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, -3, 0}, 0.296780, 0.000250, 0.026000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, -2, -2}, 180.234180, 0.000030, 0.019000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, -2, 0}, 0.486000, 0.003760, 0.040000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, -1, -2}, 180.354760, 0.000130, 0.026000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, -1, 0}, 0.398520, 0.056350, 0.083000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, -1, 2}, 180.339850, 0.000090, 0.067000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 0, -2}, 359.053210, 0.000030, 0.039000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 0, 0}, 0.204480, 0.301550, 0.799000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 0, 2}, 180.186610, 0.000140, 0.036000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 1, -2}, 358.826260, 0.000040, 0.081000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 1, 0}, 0.156810, 0.050710, 0.069000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 1, 2}, 180.214080, 0.000140, 0.024000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 2, 0}, 0.077100, 0.003530, 0.036000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 2, 2}, 180.192590, 0.000020, 0.018000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 3, -2}, 179.741520, 0.000020, 0.071000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 0, 3, 0}, 0.227090, 0.000230, 0.024000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, -2, -1}, 187.482860, 0.000040, 0.038000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, -1, -2}, 359.989470, 0.000020, 0.038000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, -1, -1}, 187.482860, 0.000520, 0.077000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, -1, 0}, 0.067300, 0.000280, 2.747000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, -1, 1}, 5.543640, 0.000190, 0.073000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 0, -2}, 359.988720, 0.000360, 0.076000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 0, -1}, 185.612330, 0.003340, 5.069000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 0, 0}, 0.059690, 0.002020, 0.073000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 0, 1}, 5.871580, 0.000360, 0.037000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 1, -2}, 359.971780, 0.000270, 32.762000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 1, -1}, 187.482860, 0.000520, 0.074000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 1, 0}, 359.850650, 0.000090, 0.037000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 1, 1}, 5.894570, 0.000040, 0.025000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 2, -2}, 0.011470, 0.000040, 0.075000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 1, 2, -1}, 187.482860, 0.000040, 0.037000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, -4, 0}, 0.735610, 0.000030, 0.037000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, -3, 0}, 0.880370, 0.000360, 0.073000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, -2, -2}, 179.873970, 0.000020, 0.037000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, -2, 0}, 359.828620, 0.005590, 1.912000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, -2, 2}, 180.112980, 0.000020, 0.038000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, -1, -2}, 0.075430, 0.000120, 0.071000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, -1, 0}, 359.997620, 0.135930, 0.079000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, -1, 2}, 180.014410, 0.000290, 0.025000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, 0, -2}, 359.992900, 0.002230, 1.168000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, 0, 0}, 359.986670, 0.035530, 0.038000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, 0, 2}, 180.084150, 0.000130, 0.019000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, 1, -2}, 179.868820, 0.000110, 0.081000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, 1, 0}, 0.008620, 0.003250, 0.025000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, 1, 2}, 180.082300, 0.000010, 0.015000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, 2, -2}, 179.976570, 0.000020, 0.039000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, 2, 0}, 0.128320, 0.000310, 0.019000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 2, 3, 0}, 0.061980, 0.000020, 0.015000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 3, -2, 0}, 180.031520, 0.000040, 0.084000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 3, -1, -1}, 11.977360, 0.000030, 0.086000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 3, -1, 0}, 180.032730, 0.000060, 0.040000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 3, 0, -2}, 179.989470, 0.000070, 0.087000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 3, 0, -1}, 185.391250, 0.000020, 0.040000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 3, 0, 0}, 0.006270, 0.000040, 0.026000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 4, -3, 0}, 358.608560, 0.000050, 0.091000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 4, -2, 0}, 0.058940, 0.001810, 0.041000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 4, -1, -2}, 359.893290, 0.000020, 0.094000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 4, -1, 0}, 0.016080, 0.001720, 0.027000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 4, 0, 0}, 359.999540, 0.000490, 0.020000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 4, 1, 0}, 0.055590, 0.000070, 0.016000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 5, -3, 0}, 359.994370, 0.000010, 0.043000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 6, -3, 0}, 0.046850, 0.000020, 0.028000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 6, -2, 0}, 0.041200, 0.000040, 0.020000},
	{{0, 2, -2, 0, 0, 0, 0, 0, 6, -1, 0}, 0.010760, 0.000050, 0.016000},
	{{0, 2, -2, 0, 0, 1, 0, 0, 2, -1, 0}, 87.677540, 0.000030, 0.078000},
	{{0, 2, -2, 0, 1, 0, 0, 0, -2, 0, 0}, 346.480100, 0.000060, 0.043000},
	{{0, 2, -2, 0, 1, 0, 0, 0, -2, 1, 0}, 171.904570, 0.000040, 0.099000},
	{{0, 2, -2, 0, 1, 0, 0, 0, -2, 3, 0}, 352.220660, 0.000030, 0.061000},
	{{0, 2, -2, 0, 1, 0, 0, 0, 0, 0, 0}, 212.183120, 0.000020, 0.749000},
	{{0, 2, -2, 0, 1, 0, 0, 0, 2, -2, 0}, 349.581330, 0.000010, 2.280000},
	{{0, 2, -2, 0, 1, 0, 0, 0, 2, -1, 0}, 336.378060, 0.000070, 0.078000},
	{{0, 2, -2, 0, 1, 0, 0, 0, 2, 0, 0}, 316.887510, 0.000050, 0.038000},
	{{0, 2, -2, 0, 2, 0, 0, 0, 0, 0, 0}, 7.310720, 0.000010, 0.704000},
	{{0, 2, -2, 1, 0, 0, 0, 0, 2, -3, 0}, 226.019310, 0.000020, 0.075000},
	{{0, 2, -2, 1, 0, 0, 0, 0, 2, -2, 0}, 358.897530, 0.000030, 114.769000},
	{{0, 2, -2, 1, 0, 0, 0, 0, 2, -1, 0}, 226.019310, 0.000020, 0.075000},
	{{0, 2, -1, -4, 0, 0, 0, 0, 0, 0, 0}, 161.416330, 0.000010, 8.045000},
	{{0, 2, -1, -3, 0, 0, 0, 0, -2, 0, 0}, 303.120270, 0.000020, 0.042000},
	{{0, 2, -1, -3, 0, 0, 0, 0, -2, 2, 0}, 303.120270, 0.000020, 0.412000},
	{{0, 2, -1, -3, 0, 0, 0, 0, 0, -1, 0}, 303.120270, 0.000010, 0.079000},
	{{0, 2, -1, -3, 0, 0, 0, 0, 0, 1, 0}, 303.120270, 0.000010, 0.072000},
	{{0, 2, -1, -3, 0, 0, 0, 0, 2, -3, 0}, 303.564920, 0.000020, 0.070000},
	{{0, 2, -1, -3, 0, 0, 0, 0, 2, -2, 0}, 303.564920, 0.000020, 0.894000},
	{{0, 2, -1, -3, 0, 0, 0, 0, 2, -1, 0}, 303.564920, 0.000020, 0.082000},
	{{0, 2, -1, -3, 0, 0, 0, 0, 2, 0, 0}, 303.564920, 0.000020, 0.039000},
	{{0, 2, -1, -2, 0, 0, 0, 0, -2, 0, 0}, 127.733290, 0.000030, 0.042000},
	{{0, 2, -1, -2, 0, 0, 0, 0, -2, 2, 0}, 135.098890, 0.000040, 0.338000},
	{{0, 2, -1, -2, 0, 0, 0, 0, 0, 0, 0}, 355.008340, 0.000010, 0.842000},
	{{0, 2, -1, -2, 0, 0, 0, 0, 2, -2, 0}, 139.612800, 0.000030, 1.706000},
	{{0, 2, -1, -2, 0, 0, 0, 0, 2, 0, 0}, 139.690310, 0.000020, 0.039000},
	{{0, 2, -1, -1, 0, 0, 0, 0, 2, -3, 0}, 93.391660, 0.000010, 0.075000},
	{{0, 2, -1, -1, 0, 0, 0, 0, 2, -1, 0}, 273.391660, 0.000010, 0.076000},
	{{0, 2, -1, 0, -3, 0, 0, 0, 0, 0, 0}, 129.634780, 0.000010, 0.500000},
	{{0, 2, -1, 0, -2, 0, 0, 0, -2, 0, 0}, 116.198930, 0.000060, 0.044000},
	{{0, 2, -1, 0, -2, 0, 0, 0, 0, 0, 0}, 292.329480, 0.000040, 0.480000},
	{{0, 2, -1, 0, -2, 0, 0, 0, 2, -3, 0}, 321.186150, 0.000020, 0.077000},
	{{0, 2, -1, 0, -2, 0, 0, 0, 2, -1, 0}, 141.439730, 0.000050, 0.074000},
	{{0, 2, -1, 0, -2, 0, 0, 0, 2, 0, -2}, 170.199840, 0.000040, 39.852000},
	{{0, 2, -1, 0, -1, 0, 0, 0, -2, 0, 0}, 2.536580, 0.000030, 0.044000},
	{{0, 2, -1, 0, -1, 0, 0, 0, -2, 1, 0}, 181.350980, 0.000100, 0.107000},
	{{0, 2, -1, 0, -1, 0, 0, 0, -2, 2, 0}, 178.732040, 0.000010, 0.254000},
	{{0, 2, -1, 0, -1, 0, 0, 0, 0, -2, 0}, 185.135210, 0.000010, 0.041000},
	{{0, 2, -1, 0, -1, 0, 0, 0, 0, 0, 0}, 176.705310, 0.000060, 0.462000},
	{{0, 2, -1, 0, -1, 0, 0, 0, 0, 1, 0}, 181.468640, 0.000040, 0.065000},
	{{0, 2, -1, 0, -1, 0, 0, 0, 2, -2, 0}, 182.478680, 0.000020, 2.546000},
	{{0, 2, -1, 0, -1, 0, 0, 0, 2, -1, 0}, 177.548320, 0.000160, 0.073000},
	{{0, 2, -1, 0, -1, 0, 0, 0, 2, 0, -2}, 183.040030, 0.000050, 16.889000},
	{{0, 2, -1, 0, -1, 0, 0, 0, 2, 0, 0}, 4.042200, 0.000090, 0.037000},
	{{0, 2, -1, 0, 0, -1, 0, 0, -2, 0, 0}, 180.215670, 0.000010, 0.044000},
	{{0, 2, -1, 0, 0, -1, 0, 0, -2, 2, 0}, 0.215670, 0.000010, 0.251000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -6, 1, 0}, 257.359260, 0.000010, 0.017000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -4, -1, 0}, 257.079480, 0.000060, 0.017000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -4, 0, 0}, 256.119090, 0.000190, 0.021000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -4, 1, 0}, 257.067240, 0.000500, 0.029000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -4, 2, 0}, 256.519300, 0.000230, 0.048000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -4, 3, 0}, 249.927830, 0.000010, 0.134000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -3, 0, 0}, 257.533160, 0.000020, 0.029000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -3, 0, 1}, 103.126760, 0.000020, 0.047000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -3, 1, -1}, 103.228160, 0.000010, 0.029000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -3, 1, 0}, 76.664890, 0.000020, 0.046000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -3, 1, 1}, 103.298200, 0.000010, 0.122000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, -2, 0}, 257.148300, 0.000110, 0.020000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, -1, 0}, 257.054770, 0.001200, 0.028000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, -1, 2}, 77.386360, 0.000020, 0.112000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, 0, -2}, 77.017520, 0.000030, 0.020000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, 0, 0}, 257.092350, 0.013450, 0.044000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, 0, 2}, 257.570550, 0.000100, 0.229000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, 1, -2}, 77.117500, 0.000010, 0.028000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, 1, 0}, 257.140130, 0.012030, 0.108000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, 1, 2}, 77.257410, 0.000030, 0.057000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, 2, 0}, 258.447660, 0.000540, 0.248000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -2, 3, 0}, 254.192940, 0.000030, 0.058000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -1, -1, -1}, 103.206450, 0.000040, 0.027000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -1, -1, 1}, 283.566450, 0.000100, 0.100000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -1, 0, -1}, 103.223770, 0.000550, 0.042000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -1, 0, 1}, 283.220030, 0.001320, 0.303000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -1, 1, -1}, 101.848420, 0.000330, 0.097000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -1, 1, 0}, 77.330030, 0.000010, 0.319000},
	{{0, 2, -1, 0, 0, 0, 0, 0, -1, 1, 1}, 283.212350, 0.000130, 0.060000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 0, -2, 0}, 256.629160, 0.000190, 0.041000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 0, -1, 0}, 257.055100, 0.002620, 0.091000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 0, 0, -2}, 77.268350, 0.000010, 0.041000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 0, 0, -1}, 283.330200, 0.000010, 0.090000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 0, 0, 0}, 252.157700, 0.008730, 0.444000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 0, 0, 1}, 103.329450, 0.000030, 0.064000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 0, 1, 0}, 253.845660, 0.001470, 0.064000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 0, 2, 0}, 253.196340, 0.000100, 0.035000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, -2, -1}, 103.288960, 0.000040, 0.040000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, -2, 1}, 283.316930, 0.000020, 0.653000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, -1, -1}, 103.292620, 0.000730, 0.083000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, -1, 0}, 77.212890, 0.000010, 0.733000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, -1, 1}, 283.294450, 0.000490, 0.068000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, 0, -1}, 103.304000, 0.004420, 0.835000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, 0, 0}, 276.744600, 0.000050, 0.068000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, 0, 1}, 283.296100, 0.000490, 0.036000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, 1, -1}, 103.457920, 0.000560, 0.069000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, 1, 1}, 283.304650, 0.000050, 0.024000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 1, 2, -1}, 103.358380, 0.000030, 0.036000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 2, -3, 0}, 261.491900, 0.000070, 0.078000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 2, -2, 0}, 256.298790, 0.000150, 2.096000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 2, -1, -1}, 286.586740, 0.000030, 3.223000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 2, -1, 0}, 243.585890, 0.002000, 0.073000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 2, 0, -2}, 246.096590, 0.000200, 6.968000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 2, 0, -1}, 283.337960, 0.000240, 0.074000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 2, 0, 0}, 71.021650, 0.000130, 0.037000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 3, -2, -1}, 283.311340, 0.000030, 1.734000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 3, -1, -1}, 283.326790, 0.000230, 0.079000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 3, 0, -1}, 103.303630, 0.000050, 0.039000},
	{{0, 2, -1, 0, 0, 0, 0, 0, 4, -2, 0}, 205.667460, 0.000010, 0.040000},
	{{0, 2, -1, 0, 1, 0, 0, 0, 2, -3, 0}, 75.750720, 0.000010, 0.079000},
	{{0, 2, -1, 0, 1, 0, 0, 0, 2, -1, 0}, 255.750720, 0.000010, 0.072000},
	{{0, 2, 0, -6, 0, 0, 0, 0, 0, 0, 0}, 68.540100, 0.000020, 16.410000},
	{{0, 2, 0, -5, 0, 0, 0, 0, 0, 0, 0}, 110.674640, 0.000010, 1.687000},
	{{0, 2, 0, -3, 0, 0, 0, 0, 2, -3, 0}, 231.641300, 0.000020, 0.075000},
	{{0, 2, 0, -3, 0, 0, 0, 0, 2, -2, 0}, 151.779090, 0.000020, 8.473000},
	{{0, 2, 0, -3, 0, 0, 0, 0, 2, -1, 0}, 254.837560, 0.000040, 0.076000},
	{{0, 2, 0, -2, 0, 0, 0, 0, -2, 0, 0}, 359.780270, 0.000030, 0.044000},
	{{0, 2, 0, -2, 0, 0, 0, 0, -2, 1, 0}, 179.894190, 0.000030, 0.108000},
	{{0, 2, 0, -2, 0, 0, 0, 0, -2, 3, 0}, 179.894190, 0.000020, 0.058000},
	{{0, 2, 0, -2, 0, 0, 0, 0, 0, -2, 0}, 179.894190, 0.000010, 0.041000},
	{{0, 2, 0, -2, 0, 0, 0, 0, 0, -1, 0}, 179.894190, 0.000010, 0.090000},
	{{0, 2, 0, -2, 0, 0, 0, 0, 0, 0, 0}, 179.894190, 0.000010, 0.457000},
	{{0, 2, 0, -2, 0, 0, 0, 0, 0, 1, 0}, 179.894190, 0.000010, 0.065000},
	{{0, 2, 0, -2, 0, 0, 0, 0, 2, -3, 0}, 167.160480, 0.000010, 0.078000},
	{{0, 2, 0, -2, 0, 0, 0, 0, 2, -1, 0}, 167.160480, 0.000010, 0.073000},
	{{0, 2, 0, 0, -3, 0, 0, 0, -2, 0, 0}, 8.937600, 0.000040, 0.046000},
	{{0, 2, 0, 0, -3, 0, 0, 0, -2, 2, 0}, 9.133430, 0.000010, 0.210000},
	{{0, 2, 0, 0, -3, 0, 0, 0, 0, 0, 0}, 190.886330, 0.000010, 0.334000},
	{{0, 2, 0, 0, -2, 0, 0, 0, -2, 0, 0}, 179.954000, 0.000020, 0.046000},
	{{0, 2, 0, 0, -2, 0, 0, 0, -2, 1, 0}, 0.116390, 0.000110, 0.119000},
	{{0, 2, 0, 0, -2, 0, 0, 0, -2, 2, 0}, 0.307150, 0.000010, 0.206000},
	{{0, 2, 0, 0, -2, 0, 0, 0, -2, 3, 0}, 0.063450, 0.000020, 0.055000},
	{{0, 2, 0, 0, -2, 0, 0, 0, 0, -2, 0}, 0.061780, 0.000020, 0.043000},
	{{0, 2, 0, 0, -2, 0, 0, 0, 0, -1, 0}, 0.105740, 0.000010, 0.098000},
	{{0, 2, 0, 0, -2, 0, 0, 0, 0, 0, 0}, 0.009060, 0.000090, 0.324000},
	{{0, 2, 0, 0, -2, 0, 0, 0, 0, 1, 0}, 0.105740, 0.000010, 0.061000},
	{{0, 2, 0, 0, -2, 0, 0, 0, 2, -3, 0}, 0.093650, 0.000010, 0.084000},
	{{0, 2, 0, 0, -2, 0, 0, 0, 2, -2, 0}, 180.056240, 0.000020, 0.764000},
	{{0, 2, 0, 0, -2, 0, 0, 0, 2, -1, 0}, 180.093650, 0.000010, 0.069000},
	{{0, 2, 0, 0, -2, 0, 0, 0, 2, 0, 0}, 0.093650, 0.000010, 0.036000},
	{{0, 2, 0, 0, -1, 0, 0, 0, -2, 0, 0}, 90.689730, 0.000020, 0.046000},
	{{0, 2, 0, 0, -1, 0, 0, 0, -2, 2, 0}, 90.569560, 0.000030, 0.202000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -4, 0, 0}, 3.406890, 0.000010, 0.022000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -4, 1, 0}, 124.842870, 0.000020, 0.030000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -2, -1, 0}, 140.707890, 0.000090, 0.029000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -2, 0, 0}, 126.544300, 0.000670, 0.047000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -2, 1, 0}, 126.237580, 0.000490, 0.121000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -2, 2, 0}, 67.417300, 0.000020, 0.199000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -1, 0, -1}, 9.197110, 0.000030, 0.044000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -1, 0, 0}, 204.976580, 0.000040, 0.110000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -1, 0, 1}, 190.306290, 0.000060, 0.232000},
	{{0, 2, 0, 0, 0, 0, 0, 0, -1, 1, -1}, 9.197110, 0.000010, 0.108000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 0, -2, 0}, 23.304620, 0.000020, 0.043000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 0, -1, 0}, 43.603430, 0.000200, 0.100000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0}, 31.139410, 0.002040, 0.308000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 0, 1, 0}, 30.175020, 0.000210, 0.061000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 0, 2, 0}, 110.153530, 0.000010, 0.034000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 1, 0, -1}, 13.529170, 0.000090, 0.455000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 1, 0, 0}, 205.528840, 0.000030, 0.064000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 2, -3, 0}, 118.527800, 0.000020, 0.085000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 2, -2, 0}, 3.411740, 0.000010, 0.677000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 2, -1, 0}, 29.102680, 0.000390, 0.068000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 2, 0, 0}, 205.173070, 0.000080, 0.036000},
	{{0, 2, 0, 0, 0, 0, 0, 0, 2, 1, 0}, 208.639980, 0.000010, 0.024000},
	{{0, 2, 0, 0, 2, 0, 0, 0, 2, -2, 0}, 308.159090, 0.000010, 0.608000},
	{{0, 2, 0, 0, 2, 0, 0, 0, 2, 0, 0}, 128.556610, 0.000020, 0.036000},
	{{0, 2, 1, -8, 0, 0, 0, 0, -2, 0, 0}, 19.932670, 0.000050, 0.040000},
	{{0, 2, 1, -8, 0, 0, 0, 0, -2, 1, 0}, 19.932670, 0.000090, 0.087000},
	{{0, 2, 1, -8, 0, 0, 0, 0, 0, -1, 0}, 24.299530, 0.000020, 0.075000},
	{{0, 2, 1, -8, 0, 0, 0, 0, 0, 0, 0}, 0.533090, 0.000120, 411.948000},
	{{0, 2, 1, -8, 0, 0, 0, 0, 0, 1, 0}, 24.299530, 0.000020, 0.075000},
	{{0, 2, 1, -8, 0, 0, 0, 0, 2, -1, 0}, 20.002150, 0.000090, 0.087000},
	{{0, 2, 1, -8, 0, 0, 0, 0, 2, 0, 0}, 20.002150, 0.000050, 0.040000},
	{{0, 2, 1, -5, 0, 0, 0, 0, 0, 0, 0}, 120.844940, 0.000020, 0.628000},
	{{0, 2, 1, -5, 0, 0, 0, 0, 2, -3, 0}, 121.538240, 0.000010, 0.074000},
	{{0, 2, 1, -5, 0, 0, 0, 0, 2, -1, 0}, 121.538240, 0.000010, 0.076000},
	{{0, 2, 1, -4, 0, 0, 0, 0, -2, 0, 0}, 150.582570, 0.000010, 0.044000},
	{{0, 2, 1, -4, 0, 0, 0, 0, -2, 2, 0}, 150.401740, 0.000020, 0.257000},
	{{0, 2, 1, 0, -2, 0, 0, 0, -2, 0, 0}, 269.208180, 0.000020, 0.048000},
	{{0, 2, 1, 0, -2, 0, 0, 0, -2, 2, 0}, 89.208180, 0.000020, 0.171000},
	{{0, 2, 1, 0, -1, 0, 0, 0, 0, 0, 0}, 18.297310, 0.000010, 0.240000},
	{{0, 2, 1, 0, 0, 0, 0, 0, -2, 0, 0}, 334.013960, 0.000060, 0.049000},
	{{0, 2, 1, 0, 0, 0, 0, 0, -2, 1, 0}, 341.303690, 0.000030, 0.138000},
	{{0, 2, 1, 0, 0, 0, 0, 0, -1, 0, 1}, 123.547570, 0.000020, 0.189000},
	{{0, 2, 1, 0, 0, 0, 0, 0, 0, -1, 0}, 288.812410, 0.000040, 0.111000},
	{{0, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0}, 295.541790, 0.000130, 0.235000},
	{{0, 2, 1, 0, 0, 0, 0, 0, 0, 1, 0}, 288.181730, 0.000020, 0.057000},
	{{0, 2, 1, 0, 0, 0, 0, 0, 1, 0, -1}, 306.204540, 0.000020, 0.313000},
	{{0, 2, 1, 0, 0, 0, 0, 0, 2, -1, 0}, 274.631270, 0.000030, 0.064000},
	{{0, 2, 1, 0, 0, 0, 0, 0, 2, 0, 0}, 113.346980, 0.000020, 0.034000},
	{{0, 2, 1, 0, 1, 0, 0, 0, 2, -3, 0}, 128.364040, 0.000040, 0.094000},
	{{0, 2, 1, 0, 1, 0, 0, 0, 2, -2, 0}, 126.860500, 0.000010, 0.390000},
	{{0, 2, 1, 0, 1, 0, 0, 0, 2, -1, 0}, 304.484710, 0.000020, 0.063000},
	{{0, 2, 1, 0, 1, 0, 0, 0, 2, 0, 0}, 128.236770, 0.000070, 0.034000},
	{{0, 2, 1, 2, 0, 0, 0, 0, 2, 0, 0}, 168.895380, 0.000010, 0.033000},
	{{0, 2, 2, -10, 0, 0, 0, 0, 0, 0, 0}, 65.153930, 0.000010, 15.199000},
	{{0, 2, 2, -8, 0, 0, 0, 0, 0, 0, 0}, 60.622840, 0.000020, 1.002000},
	{{0, 2, 2, -8, 3, 0, 0, 0, -2, 0, 0}, 288.338520, 0.000020, 0.043000},
	{{0, 2, 2, -8, 3, 0, 0, 0, 0, 0, 0}, 288.166520, 0.000030, 0.800000},
	{{0, 2, 2, -7, 0, 0, 0, 0, 0, 0, 0}, 92.673900, 0.000010, 0.654000},
	{{0, 2, 2, 0, -4, 0, 0, 0, 0, 0, 0}, 174.029420, 0.000010, 0.204000},
	{{0, 2, 2, 0, -2, 0, 0, 0, 2, -2, 0}, 335.389470, 0.000010, 0.302000},
	{{0, 2, 2, 0, -2, 0, 0, 0, 2, 0, 0}, 335.389470, 0.000010, 0.034000},
	{{0, 2, 2, 0, -1, 0, 0, 0, 2, -3, 0}, 141.308600, 0.000020, 0.101000},
	{{0, 2, 2, 0, -1, 0, 0, 0, 2, -2, 0}, 220.486470, 0.000010, 0.295000},
	{{0, 2, 2, 0, -1, 0, 0, 0, 2, -1, 0}, 141.308600, 0.000020, 0.060000},
	{{0, 2, 2, 0, -1, 0, 0, 0, 2, 0, 0}, 155.317460, 0.000060, 0.033000},
	{{0, 2, 2, 0, 0, 0, 0, 0, -2, 0, 0}, 131.037270, 0.000020, 0.051000},
	{{0, 2, 2, 0, 0, 0, 0, 0, -2, 1, 0}, 308.243910, 0.000010, 0.160000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 0, -1, 0}, 128.243910, 0.000060, 0.125000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0}, 221.306900, 0.000030, 0.190000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 0, 1, 0}, 128.243910, 0.000040, 0.054000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 0, 2, 0}, 307.238950, 0.000020, 0.031000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 2, -2, 0}, 126.070980, 0.000060, 0.288000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 2, -1, 0}, 127.608220, 0.000040, 0.060000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 2, 0, 0}, 127.479030, 0.000110, 0.033000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 2, 1, 0}, 128.299550, 0.000020, 0.023000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 2, 2, 0}, 128.243910, 0.000060, 0.018000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 4, -2, 0}, 306.070980, 0.000010, 0.035000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 4, -1, 0}, 128.200130, 0.000010, 0.024000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 4, 0, 0}, 308.243910, 0.000010, 0.018000},
	{{0, 2, 2, 0, 0, 0, 0, 0, 4, 1, 0}, 308.200130, 0.000010, 0.015000},
	{{0, 2, 2, 0, 1, 0, 0, 0, 2, 0, 0}, 300.376620, 0.000040, 0.033000},
	{{0, 2, 3, -3, 0, 0, 0, 0, 2, -2, 0}, 251.372550, 0.000010, 0.347000},
	{{0, 2, 3, -3, 0, 0, 0, 0, 2, 0, 0}, 251.372550, 0.000010, 0.034000},
	{{0, 2, 3, -2, 0, 0, 0, 0, 2, -2, 0}, 84.581080, 0.000020, 0.293000},
	{{0, 2, 3, -2, 0, 0, 0, 0, 2, 0, 0}, 84.581080, 0.000020, 0.033000},
	{{0, 2, 3, 0, -2, 0, 0, 0, 2, 0, 0}, 40.271820, 0.000010, 0.032000},
	{{0, 2, 3, 0, -1, 0, 0, 0, 0, -1, 0}, 309.480820, 0.000020, 0.141000},
	{{0, 2, 3, 0, -1, 0, 0, 0, 0, 0, 0}, 129.531340, 0.000030, 0.162000},
	{{0, 2, 3, 0, -1, 0, 0, 0, 2, -2, 0}, 128.783110, 0.000020, 0.228000},
	{{0, 2, 3, 0, -1, 0, 0, 0, 2, -1, 0}, 129.486890, 0.000090, 0.057000},
	{{0, 2, 3, 0, -1, 0, 0, 0, 2, 0, 0}, 129.480820, 0.000050, 0.032000},
	{{0, 2, 3, 0, 0, 0, 0, 0, -2, 0, 0}, 302.097990, 0.000030, 0.054000},
	{{0, 2, 3, 0, 0, 0, 0, 0, -2, 2, 0}, 302.096650, 0.000040, 0.125000},
	{{0, 2, 3, 0, 0, 0, 0, 0, 0, -2, 0}, 122.071280, 0.000010, 0.049000},
	{{0, 2, 3, 0, 0, 0, 0, 0, 0, -1, 0}, 209.692250, 0.000030, 0.143000},
	{{0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0}, 17.798550, 0.000020, 0.160000},
	{{0, 2, 3, 0, 0, 0, 0, 0, 0, 1, 0}, 209.692250, 0.000030, 0.051000},
	{{0, 2, 3, 0, 0, 0, 0, 0, 0, 2, 0}, 25.103370, 0.000020, 0.031000},
	{{0, 2, 3, 0, 0, 0, 0, 0, 2, 0, 0}, 193.383440, 0.000010, 0.032000},
	{{0, 2, 4, -2, 0, 0, 0, 0, 2, -3, 0}, 308.146470, 0.000010, 0.113000},
	{{0, 2, 4, -2, 0, 0, 0, 0, 2, -2, 0}, 308.229660, 0.000050, 0.227000},
	{{0, 2, 4, -2, 0, 0, 0, 0, 2, -1, 0}, 308.178090, 0.000020, 0.057000},
	{{0, 2, 4, -2, 0, 0, 0, 0, 2, 0, 0}, 308.205450, 0.000060, 0.032000},
	{{0, 2, 4, 0, -3, 0, 0, 0, 2, 0, 0}, 318.109300, 0.000030, 0.032000},
	{{0, 2, 4, 0, -2, 0, 0, 0, 0, 0, 0}, 308.379640, 0.000020, 0.141000},
	{{0, 2, 4, 0, -2, 0, 0, 0, 2, -3, 0}, 128.358030, 0.000020, 0.126000},
	{{0, 2, 4, 0, -2, 0, 0, 0, 2, -2, 0}, 308.314380, 0.000020, 0.188000},
	{{0, 2, 4, 0, -2, 0, 0, 0, 2, -1, 0}, 128.358030, 0.000020, 0.054000},
	{{0, 2, 4, 0, -2, 0, 0, 0, 2, 0, 0}, 308.331810, 0.000060, 0.031000},
	{{0, 2, 4, 0, -1, 0, 0, 0, 2, 0, 0}, 38.644900, 0.000020, 0.031000},
	{{0, 2, 4, 0, 0, 0, 0, 0, 0, -1, 0}, 123.910390, 0.000020, 0.167000},
	{{0, 2, 4, 0, 0, 0, 0, 0, 2, -2, 0}, 324.393070, 0.000010, 0.183000},
	{{0, 2, 4, 0, 0, 0, 0, 0, 2, 0, 0}, 160.470210, 0.000010, 0.031000},
	{{0, 2, 5, 0, 0, 0, 0, 0, 2, 0, 0}, 239.015200, 0.000020, 0.030000},
	{{0, 2, 18, -19, 0, 0, 0, 0, -2, -1, 2}, 261.062860, 0.000010, 1216.760000},
	{{0, 2, 21, -26, 0, 0, 0, 0, -3, 1, 1}, 187.912310, 0.000060, 1396.538000},
	{{0, 2, 21, -24, 0, 0, 0, 0, -2, 1, 0}, 205.945380, 0.000010, 112.528000},
	{{0, 2, 27, -28, 0, 0, 0, 0, 2, -1, -2}, 220.771430, 0.000070, 1662.840000},
	{{0, 2, 38, -56, 0, 0, 0, 0, -2, 1, 0}, 77.242430, 0.000020, 199.649000},
	{{0, 3, -15, 23, 0, 0, 0, 0, 2, 0, -2}, 135.343940, 0.000010, 449.647000},
	{{0, 3, -12, 0, 0, 0, 0, 0, -2, 0, 0}, 289.614000, 0.000010, 0.031000},
	{{0, 3, -10, 0, 0, 0, 0, 0, -2, 1, 0}, 121.851740, 0.000020, 0.060000},
	{{0, 3, -10, 0, 0, 0, 0, 0, -2, 3, 0}, 121.851740, 0.000020, 0.101000},
	{{0, 3, -10, 0, 2, 0, 0, 0, -2, 0, 0}, 322.611070, 0.000020, 0.034000},
	{{0, 3, -9, 0, 0, 0, 0, 0, -2, -1, 0}, 215.252550, 0.000010, 0.024000},
	{{0, 3, -9, 0, 0, 0, 0, 0, -2, 0, 0}, 38.891780, 0.000020, 0.035000},
	{{0, 3, -9, 0, 0, 0, 0, 0, -2, 3, 0}, 230.838740, 0.000020, 0.092000},
	{{0, 3, -9, 0, 0, 0, 0, 0, 0, 0, 0}, 246.963050, 0.000020, 0.243000},
	{{0, 3, -9, 0, 1, 0, 0, 0, -2, 1, 0}, 321.437760, 0.000010, 0.064000},
	{{0, 3, -9, 0, 1, 0, 0, 0, -2, 3, 0}, 141.437760, 0.000010, 0.091000},
	{{0, 3, -9, 0, 2, 0, 0, 0, -2, 0, 0}, 231.670570, 0.000030, 0.035000},
	{{0, 3, -9, 0, 2, 0, 0, 0, -2, 1, 0}, 231.599650, 0.000020, 0.065000},
	{{0, 3, -9, 0, 2, 0, 0, 0, -2, 3, 0}, 231.599650, 0.000020, 0.090000},
	{{0, 3, -9, 0, 2, 0, 0, 0, 0, 0, 0}, 231.638570, 0.000020, 0.253000},
	{{0, 3, -9, 0, 3, 0, 0, 0, -2, 0, 0}, 221.887300, 0.000010, 0.035000},
	{{0, 3, -9, 0, 3, 0, 0, 0, -2, 2, 0}, 221.887300, 0.000010, 0.477000},
	{{0, 3, -9, 2, 0, 0, 0, 0, -2, 0, 0}, 231.906970, 0.000020, 0.036000},
	{{0, 3, -9, 2, 0, 0, 0, 0, -2, 1, 0}, 51.906970, 0.000020, 0.069000},
	{{0, 3, -9, 2, 0, 0, 0, 0, -2, 2, 0}, 231.906970, 0.000020, 0.778000},
	{{0, 3, -9, 2, 0, 0, 0, 0, -2, 3, 0}, 51.906970, 0.000020, 0.084000},
	{{0, 3, -9, 4, 0, 0, 0, 0, -2, 1, 0}, 115.331700, 0.000010, 0.074000},
	{{0, 3, -9, 4, 0, 0, 0, 0, -2, 3, 0}, 115.331700, 0.000010, 0.077000},
	{{0, 3, -9, 4, 0, 0, 0, 0, 0, 0, 0}, 351.358350, 0.000010, 0.501000},
	{{0, 3, -9, 4, 0, 0, 0, 0, 2, 0, 0}, 352.373950, 0.000020, 0.044000},
	{{0, 3, -8, 0, 0, 0, 0, 0, -2, -1, 0}, 323.997490, 0.000040, 0.024000},
	{{0, 3, -8, 0, 0, 0, 0, 0, -2, 0, 0}, 164.959160, 0.000020, 0.036000},
	{{0, 3, -8, 0, 0, 0, 0, 0, -2, 1, 0}, 69.792960, 0.000010, 0.068000},
	{{0, 3, -8, 0, 0, 0, 0, 0, -2, 2, 0}, 315.505870, 0.000030, 0.741000},
	{{0, 3, -8, 0, 0, 0, 0, 0, -2, 3, 0}, 135.667700, 0.000030, 0.084000},
	{{0, 3, -8, 0, 0, 0, 0, 0, 0, -2, 0}, 142.817210, 0.000010, 0.034000},
	{{0, 3, -8, 0, 0, 0, 0, 0, 0, -1, 0}, 322.817210, 0.000010, 0.061000},
	{{0, 3, -8, 0, 0, 0, 0, 0, 0, 0, 0}, 149.076850, 0.000030, 0.320000},
	{{0, 3, -8, 0, 0, 0, 0, 0, 0, 1, 0}, 322.817210, 0.000010, 0.099000},
	{{0, 3, -8, 0, 0, 0, 0, 0, 2, -1, 0}, 332.321870, 0.000030, 0.120000},
	{{0, 3, -8, 0, 1, 0, 0, 0, -2, -2, 0}, 50.465260, 0.000010, 0.018000},
	{{0, 3, -8, 0, 1, 0, 0, 0, -2, 2, 0}, 51.201830, 0.000020, 0.790000},
	{{0, 3, -8, 0, 1, 0, 0, 0, -2, 3, 0}, 231.201840, 0.000020, 0.083000},
	{{0, 3, -8, 0, 2, 0, 0, 0, -2, 0, 0}, 291.819400, 0.000040, 0.036000},
	{{0, 3, -8, 0, 2, 0, 0, 0, -2, 1, 0}, 111.892110, 0.000050, 0.069000},
	{{0, 3, -8, 0, 2, 0, 0, 0, -2, 3, 0}, 292.298370, 0.000010, 0.083000},
	{{0, 3, -8, 2, 0, 0, 0, 0, -2, 1, 0}, 244.907510, 0.000010, 0.074000},
	{{0, 3, -8, 2, 0, 0, 0, 0, -2, 3, 0}, 244.907510, 0.000010, 0.077000},
	{{0, 3, -8, 3, 0, 0, 0, 0, -2, 0, 0}, 288.624050, 0.000010, 0.038000},
	{{0, 3, -8, 3, 0, 0, 0, 0, -2, 2, 0}, 288.624050, 0.000010, 4.071000},
	{{0, 3, -8, 4, 0, 0, 0, 0, -2, 1, 0}, 69.349420, 0.000030, 0.080000},
	{{0, 3, -8, 4, 0, 0, 0, 0, -2, 2, 0}, 250.765750, 0.000010, 1.286000},
	{{0, 3, -8, 4, 0, 0, 0, 0, 0, -1, 0}, 241.084230, 0.000020, 0.070000},
	{{0, 3, -8, 4, 0, 0, 0, 0, 0, 0, 0}, 248.242620, 0.000400, 1.003000},
	{{0, 3, -8, 4, 0, 0, 0, 0, 0, 1, 0}, 249.198980, 0.000150, 0.082000},
	{{0, 3, -8, 4, 0, 0, 0, 0, 2, -2, 0}, 249.843050, 0.000010, 0.361000},
	{{0, 3, -8, 4, 0, 0, 0, 0, 2, -1, 0}, 249.648000, 0.000360, 0.095000},
	{{0, 3, -8, 4, 0, 0, 0, 0, 2, 0, 0}, 249.987700, 0.000300, 0.042000},
	{{0, 3, -8, 4, 0, 0, 0, 0, 2, 1, 0}, 249.776440, 0.000030, 0.027000},
	{{0, 3, -8, 4, 0, 0, 0, 0, 4, 0, 0}, 70.009900, 0.000020, 0.021000},
	{{0, 3, -8, 5, 0, 0, 0, 0, 0, 0, 0}, 312.587990, 0.000010, 2.151000},
	{{0, 3, -8, 6, 0, 0, 0, 0, 0, 0, 0}, 210.606930, 0.000050, 14.998000},
	{{0, 3, -8, 8, -3, 0, 0, 0, 0, 0, 0}, 163.062660, 0.000010, 1.140000},
	{{0, 3, -7, 0, -1, 0, 0, 0, -2, 0, 0}, 58.378200, 0.000020, 0.037000},
	{{0, 3, -7, 0, -1, 0, 0, 0, -2, 0, 2}, 61.590350, 0.000030, 9.980000},
	{{0, 3, -7, 0, -1, 0, 0, 0, -2, 1, 0}, 267.246770, 0.000080, 0.073000},
	{{0, 3, -7, 0, -1, 0, 0, 0, -2, 2, 0}, 65.793190, 0.000010, 2.306000},
	{{0, 3, -7, 0, 0, -1, 0, 0, -2, 0, 2}, 324.227910, 0.000010, 20.062000},
	{{0, 3, -7, 0, 0, 0, 0, 0, -4, 0, 0}, 231.788510, 0.000020, 0.019000},
	{{0, 3, -7, 0, 0, 0, 0, 0, -2, -1, 2}, 50.727770, 0.000030, 0.075000},
	{{0, 3, -7, 0, 0, 0, 0, 0, -2, 0, 0}, 52.815970, 0.000070, 0.037000},
	{{0, 3, -7, 0, 0, 0, 0, 0, -2, 0, 2}, 295.919980, 0.000030, 62.897000},
	{{0, 3, -7, 0, 0, 0, 0, 0, -2, 1, 0}, 297.142080, 0.000030, 0.074000},
	{{0, 3, -7, 0, 0, 0, 0, 0, -2, 1, 2}, 50.727770, 0.000030, 0.076000},
	{{0, 3, -7, 0, 0, 0, 0, 0, -2, 2, 0}, 51.912230, 0.000030, 2.862000},
	{{0, 3, -7, 0, 0, 0, 0, 0, -2, 3, 0}, 267.977560, 0.000020, 0.077000},
	{{0, 3, -7, 0, 0, 0, 0, 0, 0, -1, 0}, 208.100990, 0.000020, 0.065000},
	{{0, 3, -7, 0, 0, 0, 0, 0, 0, -1, 2}, 51.788510, 0.000020, 0.087000},
	{{0, 3, -7, 0, 0, 0, 0, 0, 0, 0, 0}, 256.420790, 0.000060, 0.471000},
	{{0, 3, -7, 0, 0, 0, 0, 0, 0, 1, 0}, 208.100990, 0.000020, 0.090000},
	{{0, 3, -7, 0, 0, 0, 0, 0, 2, -1, 0}, 190.934300, 0.000090, 0.107000},
	{{0, 3, -7, 0, 0, 0, 0, 0, 2, 0, 0}, 191.809620, 0.000100, 0.044000},
	{{0, 3, -7, 0, 0, 0, 1, 0, -2, 0, 2}, 238.391430, 0.000010, 250.179000},
	{{0, 3, -7, 0, 0, 1, 0, 0, -2, 0, 2}, 319.485020, 0.000010, 55.406000},
	{{0, 3, -7, 0, 0, 1, 0, 0, -2, 1, 0}, 139.426660, 0.000030, 0.074000},
	{{0, 3, -7, 0, 0, 1, 0, 0, -2, 3, 0}, 319.426660, 0.000030, 0.077000},
	{{0, 3, -7, 0, 1, 0, 0, 0, -2, 0, 0}, 0.688130, 0.000020, 0.037000},
	{{0, 3, -7, 0, 1, 0, 0, 0, -2, 0, 2}, 180.688130, 0.000020, 14.619000},
	{{0, 3, -7, 0, 1, 0, 0, 0, -2, 1, 0}, 2.619490, 0.000030, 0.074000},
	{{0, 3, -7, 0, 1, 0, 0, 0, -2, 2, 0}, 70.924360, 0.000020, 3.772000},
	{{0, 3, -7, 0, 1, 0, 0, 0, -2, 3, 0}, 2.619490, 0.000030, 0.077000},
	{{0, 3, -7, 0, 4, 0, 0, 0, -2, 1, 0}, 4.214650, 0.000030, 0.075000},
	{{0, 3, -7, 0, 4, 0, 0, 0, -2, 2, 0}, 163.492620, 0.000010, 81.999000},
	{{0, 3, -7, 2, 0, 0, 0, 0, -2, 1, 0}, 24.871980, 0.000010, 0.080000},
	{{0, 3, -7, 2, 0, 0, 0, 0, -2, 3, 0}, 24.871980, 0.000010, 0.072000},
	{{0, 3, -7, 2, 0, 0, 0, 0, -1, 0, 1}, 357.911810, 0.000060, 158.825000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -4, 0, 0}, 147.640870, 0.000040, 0.020000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -4, 1, 0}, 147.533480, 0.000100, 0.028000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -4, 2, 0}, 147.197880, 0.000060, 0.044000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -2, -2, 0}, 147.640860, 0.000030, 0.020000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -2, -1, 0}, 147.975310, 0.000320, 0.026000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -2, 0, 0}, 147.705860, 0.003380, 0.040000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -2, 0, 2}, 147.640850, 0.000070, 0.474000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -2, 1, 0}, 147.321070, 0.005340, 0.087000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -2, 2, 0}, 146.761820, 0.000200, 0.563000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -1, 0, 0}, 327.705700, 0.000100, 0.081000},
	{{0, 3, -7, 4, 0, 0, 0, 0, -1, 1, 0}, 327.640850, 0.000010, 1.123000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, -3, 0}, 150.312540, 0.000020, 0.025000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, -2, 0}, 149.412330, 0.000400, 0.038000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, -1, -2}, 328.880000, 0.000020, 0.025000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, -1, 0}, 149.428430, 0.005880, 0.075000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, 0, -2}, 327.002330, 0.000010, 0.037000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, 0, 0}, 148.231030, 0.032450, 302.434000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, 0, 2}, 327.002330, 0.000010, 0.037000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, 1, 0}, 149.431920, 0.005880, 0.075000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, 1, 2}, 328.880000, 0.000020, 0.025000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, 2, 0}, 149.416840, 0.000400, 0.038000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 0, 3, 0}, 149.554410, 0.000030, 0.025000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 1, -1, 0}, 327.650370, 0.000010, 1.132000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 1, 0, 0}, 327.702000, 0.000100, 0.081000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 2, -2, 0}, 146.778480, 0.000200, 0.565000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 2, -1, 0}, 147.330550, 0.005310, 0.087000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 2, 0, -2}, 147.650360, 0.000070, 0.475000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 2, 0, 0}, 147.714850, 0.003360, 0.040000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 2, 1, 0}, 147.984100, 0.000320, 0.026000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 2, 2, 0}, 147.650350, 0.000030, 0.020000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 4, -2, 0}, 147.205130, 0.000060, 0.044000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 4, -1, 0}, 147.533780, 0.000090, 0.028000},
	{{0, 3, -7, 4, 0, 0, 0, 0, 4, 0, 0}, 147.650340, 0.000040, 0.020000},
	{{0, 3, -7, 5, 0, 0, 0, 0, 0, 0, 0}, 256.552570, 0.000010, 1.869000},
	{{0, 3, -7, 6, 0, 0, 0, 0, 1, 0, -1}, 333.730170, 0.000020, 77.464000},
	{{0, 3, -7, 7, 0, 0, 0, 0, 0, 0, 0}, 267.322700, 0.000010, 0.626000},
	{{0, 3, -7, 7, 0, 0, 0, 0, 2, -3, 0}, 86.413570, 0.000010, 0.074000},
	{{0, 3, -7, 7, 0, 0, 0, 0, 2, -1, 0}, 86.413570, 0.000010, 0.076000},
	{{0, 3, -6, -3, 0, 0, 0, 0, -2, 0, 0}, 355.312450, 0.000010, 0.036000},
	{{0, 3, -6, -3, 0, 0, 0, 0, -2, 1, 0}, 175.312450, 0.000020, 0.070000},
	{{0, 3, -6, -3, 0, 0, 0, 0, -2, 2, 0}, 355.312450, 0.000010, 1.059000},
	{{0, 3, -6, -3, 0, 0, 0, 0, -2, 3, 0}, 175.312450, 0.000020, 0.081000},
	{{0, 3, -6, -2, 0, 0, 0, 0, -2, 0, 0}, 11.105590, 0.000030, 0.037000},
	{{0, 3, -6, -2, 0, 0, 0, 0, -2, 1, 0}, 10.668730, 0.000010, 0.073000},
	{{0, 3, -6, -2, 0, 0, 0, 0, -2, 2, 0}, 11.020410, 0.000020, 2.423000},
	{{0, 3, -6, -2, 0, 0, 0, 0, -2, 3, 0}, 10.668730, 0.000010, 0.078000},
	{{0, 3, -6, -1, 0, 0, 0, 0, -2, 1, 0}, 51.972110, 0.000020, 0.076000},
	{{0, 3, -6, -1, 0, 0, 0, 0, -2, 3, 0}, 51.972110, 0.000020, 0.075000},
	{{0, 3, -6, 0, -2, 0, 0, 0, -2, 1, 0}, 6.888820, 0.000040, 0.078000},
	{{0, 3, -6, 0, -2, 0, 0, 0, -2, 3, 0}, 180.286690, 0.000020, 0.073000},
	{{0, 3, -6, 0, -1, 0, 0, 0, -4, 3, 0}, 55.933510, 0.000010, 0.092000},
	{{0, 3, -6, 0, -1, 0, 0, 0, -2, 0, 0}, 235.751350, 0.000010, 0.039000},
	{{0, 3, -6, 0, -1, 0, 0, 0, -2, 1, 0}, 236.770360, 0.000090, 0.079000},
	{{0, 3, -6, 0, -1, 0, 0, 0, -2, 3, 0}, 232.450910, 0.000020, 0.072000},
	{{0, 3, -6, 0, -1, 0, 0, 0, 0, -1, 0}, 235.751350, 0.000010, 0.069000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -4, 0, 0}, 120.176130, 0.000020, 0.020000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -3, 0, 1}, 238.814340, 0.000020, 0.040000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -3, 1, 1}, 59.666100, 0.000030, 0.087000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -2, -1, 0}, 267.618110, 0.000020, 0.026000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -2, 0, 0}, 253.046350, 0.000050, 0.039000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -2, 1, 0}, 241.556420, 0.000020, 0.079000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -2, 3, 0}, 163.815610, 0.000040, 0.072000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, -2, 1}, 239.280170, 0.000020, 0.038000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, -1, -1}, 59.012150, 0.000020, 0.025000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, -1, 1}, 239.134920, 0.000350, 0.075000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, 0, -1}, 59.012150, 0.000180, 0.037000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, 0, 0}, 23.166380, 0.000030, 0.074000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, 0, 1}, 238.814340, 0.001990, 14.355000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, 1, -1}, 58.692810, 0.000040, 0.073000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, 1, 1}, 239.134920, 0.000350, 0.076000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, 2, 0}, 84.686310, 0.000010, 0.077000},
	{{0, 3, -6, 0, 0, 0, 0, 0, -1, 2, 1}, 239.280170, 0.000020, 0.038000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 0, -2, 0}, 286.949140, 0.000010, 0.036000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 0, -1, 0}, 100.865360, 0.000150, 0.070000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 0, 0, 0}, 97.281470, 0.001290, 0.890000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 0, 1, 0}, 90.847100, 0.000430, 0.082000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 0, 2, 0}, 87.296350, 0.000010, 0.039000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 1, -1, 1}, 59.666100, 0.000030, 0.088000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 1, 0, -1}, 60.153020, 0.000090, 0.459000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 1, 0, 1}, 254.276810, 0.000020, 0.041000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 2, -2, 0}, 90.599360, 0.000080, 0.345000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 2, -1, 0}, 90.556010, 0.001530, 0.097000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 2, 0, -2}, 88.766010, 0.000020, 0.310000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 2, 0, 0}, 90.345320, 0.001350, 0.042000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 2, 1, 0}, 89.441390, 0.000160, 0.027000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 4, -2, 0}, 89.753890, 0.000030, 0.046000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 4, -1, 0}, 90.212400, 0.000030, 0.028000},
	{{0, 3, -6, 0, 0, 0, 0, 0, 4, 0, 0}, 89.566660, 0.000040, 0.021000},
	{{0, 3, -6, 0, 0, 2, 0, 0, -1, -1, 1}, 74.168120, 0.000010, 0.075000},
	{{0, 3, -6, 0, 0, 2, 0, 0, -1, 0, 1}, 95.036960, 0.000020, 565.470000},
	{{0, 3, -6, 0, 0, 2, 0, 0, -1, 1, 1}, 74.168120, 0.000010, 0.075000},
	{{0, 3, -6, 0, 1, 0, 0, 0, -1, 0, 1}, 154.832840, 0.000020, 68.309000},
	{{0, 3, -6, 0, 1, 0, 0, 0, 0, 0, 0}, 185.366920, 0.000020, 0.962000},
	{{0, 3, -6, 0, 1, 0, 0, 0, 2, 0, 0}, 185.700810, 0.000030, 0.042000},
	{{0, 3, -6, 0, 2, 0, 0, 0, -1, 0, 1}, 83.521060, 0.000010, 10.107000},
	{{0, 3, -6, 0, 2, 0, 0, 0, 0, 0, 0}, 69.299600, 0.000020, 1.047000},
	{{0, 3, -6, 0, 2, 0, 0, 0, 2, -1, 0}, 288.724230, 0.000040, 0.095000},
	{{0, 3, -6, 0, 2, 0, 0, 0, 2, 0, 0}, 288.724230, 0.000030, 0.042000},
	{{0, 3, -6, 0, 3, 0, 0, 0, 0, 0, 0}, 59.300600, 0.000020, 1.149000},
	{{0, 3, -6, 2, 0, 0, 0, 0, -2, 0, 0}, 94.491960, 0.000020, 0.040000},
	{{0, 3, -6, 2, 0, 0, 0, 0, -2, 1, 0}, 94.491960, 0.000030, 0.087000},
	{{0, 3, -6, 2, 0, 0, 0, 0, 0, -1, 0}, 90.971820, 0.000010, 0.075000},
	{{0, 3, -6, 2, 0, 0, 0, 0, 0, 0, 0}, 93.475690, 0.000170, 16.650000},
	{{0, 3, -6, 2, 0, 0, 0, 0, 0, 1, 0}, 90.971820, 0.000010, 0.076000},
	{{0, 3, -6, 2, 0, 0, 0, 0, 2, -1, 0}, 94.686740, 0.000030, 0.088000},
	{{0, 3, -6, 2, 0, 0, 0, 0, 2, 0, 0}, 94.686740, 0.000020, 0.041000},
	{{0, 3, -6, 3, 0, 0, 0, 0, 0, 0, 0}, 153.865590, 0.000020, 2.120000},
	{{0, 3, -6, 4, 0, 0, 0, 0, -2, -1, 0}, 50.457050, 0.000030, 0.027000},
	{{0, 3, -6, 4, 0, 0, 0, 0, -2, 0, 0}, 49.068050, 0.000220, 0.042000},
	{{0, 3, -6, 4, 0, 0, 0, 0, -2, 1, 0}, 50.361290, 0.000260, 0.095000},
	{{0, 3, -6, 4, 0, 0, 0, 0, 0, -1, 0}, 62.227520, 0.000050, 0.082000},
	{{0, 3, -6, 4, 0, 0, 0, 0, 0, 0, 0}, 117.975700, 0.000090, 0.997000},
	{{0, 3, -6, 4, 0, 0, 0, 0, 0, 1, 0}, 208.240520, 0.000030, 0.070000},
	{{0, 3, -6, 4, 0, 0, 0, 0, 2, -1, 0}, 224.713840, 0.000020, 0.080000},
	{{0, 3, -6, 4, 0, 0, 0, 0, 2, 0, 0}, 219.828330, 0.000030, 0.039000},
	{{0, 3, -5, -3, 0, 0, 0, 0, -2, 1, 0}, 98.356780, 0.000040, 0.076000},
	{{0, 3, -5, -3, 0, 0, 0, 0, -2, 2, 0}, 46.820850, 0.000050, 18.007000},
	{{0, 3, -5, -3, 0, 0, 0, 0, -2, 3, 0}, 98.356780, 0.000040, 0.075000},
	{{0, 3, -5, 0, -2, 0, 0, 0, -2, 0, 0}, 51.840070, 0.000020, 0.040000},
	{{0, 3, -5, 0, -2, 0, 0, 0, 0, -1, 0}, 231.840070, 0.000020, 0.074000},
	{{0, 3, -5, 0, -2, 0, 0, 0, 0, 1, 0}, 231.840070, 0.000020, 0.077000},
	{{0, 3, -5, 0, -1, 0, 0, 0, 0, -2, 0}, 216.781440, 0.000010, 0.037000},
	{{0, 3, -5, 0, -1, 0, 0, 0, 0, 0, 0}, 101.045650, 0.000030, 4.814000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -4, -1, 0}, 344.133310, 0.000020, 0.016000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -4, 0, 0}, 345.566580, 0.000150, 0.020000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -4, 1, 0}, 344.079990, 0.000340, 0.028000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -4, 2, 0}, 346.990990, 0.000250, 0.043000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -3, 0, 0}, 342.276670, 0.000020, 0.027000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -3, 1, 0}, 166.642200, 0.000010, 0.042000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, -2, 0}, 344.359600, 0.000110, 0.019000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, -1, 0}, 342.997220, 0.000980, 0.026000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, -1, 2}, 178.384650, 0.000020, 0.089000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, 0, -2}, 165.302740, 0.000020, 0.019000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, 0, 0}, 344.004270, 0.010700, 0.040000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, 0, 1}, 348.409330, 0.000040, 0.087000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, 0, 2}, 344.161170, 0.000270, 0.504000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, 1, -2}, 165.191970, 0.000040, 0.026000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, 1, 0}, 345.882830, 0.019170, 0.086000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, 2, 0}, 347.120770, 0.000790, 0.606000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -2, 3, 0}, 345.686710, 0.000060, 0.067000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -1, -1, 0}, 165.933510, 0.000020, 0.039000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -1, 0, -1}, 169.847830, 0.000030, 0.039000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -1, 0, 0}, 155.008230, 0.000280, 0.080000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -1, 0, 1}, 350.650740, 0.000130, 1.075000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -1, 1, -1}, 166.275930, 0.000030, 0.079000},
	{{0, 3, -5, 0, 0, 0, 0, 0, -1, 1, 0}, 162.656380, 0.000050, 1.310000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, -3, 0}, 337.631230, 0.000060, 0.025000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, -2, 0}, 338.457330, 0.000980, 0.038000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, -1, -2}, 160.825850, 0.000040, 0.025000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, -1, 0}, 338.067200, 0.014230, 0.075000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, -1, 1}, 349.987960, 0.000070, 23.081000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, -1, 2}, 339.466650, 0.000030, 0.074000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 0, -2}, 169.205300, 0.000070, 0.037000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 0, -1}, 168.576980, 0.000040, 0.074000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 0, 0}, 344.529950, 0.082770, 8.102000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 0, 1}, 168.409320, 0.000190, 0.075000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 0, 2}, 170.501490, 0.000040, 0.037000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 1, -2}, 339.466650, 0.000030, 0.073000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 1, 0}, 338.663730, 0.014980, 0.076000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 1, 2}, 160.825850, 0.000040, 0.025000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 2, 0}, 338.057980, 0.001000, 0.038000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 0, 3, 0}, 335.773100, 0.000060, 0.025000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 1, -2, 0}, 188.076190, 0.000020, 0.070000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 1, -1, -1}, 344.999500, 0.000030, 0.069000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 1, -1, 0}, 101.289640, 0.000020, 0.990000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 1, -1, 1}, 167.364930, 0.000080, 0.081000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 1, 0, -1}, 347.196770, 0.000330, 0.849000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 1, 0, 0}, 46.475180, 0.000220, 0.082000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 1, 0, 1}, 167.035100, 0.000080, 0.039000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 1, 1, -1}, 344.999500, 0.000030, 0.083000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 1, 1, 0}, 26.347520, 0.000040, 0.039000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, -3, 0}, 348.751470, 0.000060, 0.066000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, -2, 0}, 346.855230, 0.000950, 0.527000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, -1, 0}, 346.950200, 0.022990, 0.088000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, -1, 2}, 166.135310, 0.000050, 0.026000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, 0, -2}, 346.485120, 0.000300, 0.448000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, 0, 0}, 346.498740, 0.014240, 0.041000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, 0, 2}, 166.270700, 0.000030, 0.019000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, 1, -2}, 166.126860, 0.000030, 0.091000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, 1, 0}, 346.151730, 0.001300, 0.026000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 2, 2, 0}, 345.370250, 0.000110, 0.020000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 3, -2, 0}, 13.575670, 0.000070, 0.096000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 3, -1, 0}, 17.447510, 0.000110, 0.042000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 3, 0, 0}, 6.306780, 0.000070, 0.027000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 4, -2, 0}, 349.962900, 0.000330, 0.044000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 4, -1, 0}, 348.482910, 0.000430, 0.028000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 4, 0, -2}, 346.315530, 0.000020, 0.044000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 4, 0, 0}, 343.792600, 0.000130, 0.020000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 4, 1, 0}, 345.734060, 0.000030, 0.016000},
	{{0, 3, -5, 0, 0, 0, 0, 0, 6, -1, 0}, 345.809080, 0.000010, 0.016000},
	{{0, 3, -5, 0, 0, 2, 0, 0, -2, 0, 0}, 178.794260, 0.000020, 0.040000},
	{{0, 3, -5, 0, 0, 2, 0, 0, 0, -1, 0}, 179.673780, 0.000040, 0.075000},
	{{0, 3, -5, 0, 0, 2, 0, 0, 0, 0, 0}, 178.710010, 0.000130, 18.007000},
	{{0, 3, -5, 0, 0, 2, 0, 0, 0, 1, 0}, 179.673780, 0.000040, 0.076000},
	{{0, 3, -5, 0, 0, 2, 0, 0, 2, 0, 0}, 178.813450, 0.000020, 0.041000},
	{{0, 3, -5, 0, 0, 3, 0, 0, -2, 0, 0}, 87.116050, 0.000020, 0.040000},
	{{0, 3, -5, 0, 0, 3, 0, 0, -2, 1, 0}, 87.116050, 0.000030, 0.087000},
	{{0, 3, -5, 0, 0, 3, 0, 0, 0, -1, 0}, 87.325830, 0.000040, 0.075000},
	{{0, 3, -5, 0, 0, 3, 0, 0, 0, 0, 0}, 87.279080, 0.000170, 46.327000},
	{{0, 3, -5, 0, 0, 3, 0, 0, 0, 1, 0}, 87.325830, 0.000040, 0.076000},
	{{0, 3, -5, 0, 0, 3, 0, 0, 2, -1, 0}, 87.124200, 0.000030, 0.087000},
	{{0, 3, -5, 0, 0, 3, 0, 0, 2, 0, 0}, 87.124200, 0.000020, 0.040000},
	{{0, 3, -5, 0, 1, 0, 0, 0, -2, 0, 0}, 82.289550, 0.000100, 0.040000},
	{{0, 3, -5, 0, 1, 0, 0, 0, -2, 1, 0}, 82.276180, 0.000170, 0.087000},
	{{0, 3, -5, 0, 1, 0, 0, 0, 0, -2, 0}, 83.051960, 0.000010, 0.038000},
	{{0, 3, -5, 0, 1, 0, 0, 0, 0, -1, 0}, 82.294250, 0.000210, 0.075000},
	{{0, 3, -5, 0, 1, 0, 0, 0, 0, 0, 0}, 80.274110, 0.000980, 25.559000},
	{{0, 3, -5, 0, 1, 0, 0, 0, 0, 1, 0}, 82.461970, 0.000230, 0.076000},
	{{0, 3, -5, 0, 1, 0, 0, 0, 0, 2, 0}, 83.045010, 0.000010, 0.038000},
	{{0, 3, -5, 0, 1, 0, 0, 0, 2, -3, 0}, 89.658430, 0.000010, 0.066000},
	{{0, 3, -5, 0, 1, 0, 0, 0, 2, -1, 0}, 83.141990, 0.000200, 0.087000},
	{{0, 3, -5, 0, 1, 0, 0, 0, 2, 0, 0}, 84.601150, 0.000160, 0.040000},
	{{0, 3, -5, 0, 1, 0, 0, 0, 2, 1, 0}, 83.993790, 0.000020, 0.026000},
	{{0, 3, -5, 0, 2, 0, 0, 0, -4, 1, 0}, 178.988190, 0.000020, 0.028000},
	{{0, 3, -5, 0, 2, 0, 0, 0, -2, -1, 0}, 178.998400, 0.000060, 0.026000},
	{{0, 3, -5, 0, 2, 0, 0, 0, -2, 0, 0}, 179.002600, 0.000600, 0.040000},
	{{0, 3, -5, 0, 2, 0, 0, 0, -2, 0, 2}, 178.971040, 0.000010, 0.465000},
	{{0, 3, -5, 0, 2, 0, 0, 0, -2, 1, 0}, 178.641990, 0.000890, 0.087000},
	{{0, 3, -5, 0, 2, 0, 0, 0, -2, 2, 0}, 178.971330, 0.000040, 0.550000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 0, -2, 0}, 180.229230, 0.000080, 0.038000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 0, -1, 0}, 180.300210, 0.001060, 0.076000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 0, 0, -2}, 358.910000, 0.000010, 0.037000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 0, 0, 0}, 178.909090, 0.006120, 22.135000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 0, 0, 2}, 358.910000, 0.000010, 0.037000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 0, 1, 0}, 180.317110, 0.001050, 0.075000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 0, 2, 0}, 180.193490, 0.000080, 0.038000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 2, -2, 0}, 175.360590, 0.000010, 0.578000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 2, -1, 0}, 178.610600, 0.000830, 0.087000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 2, 0, -2}, 178.966470, 0.000010, 0.485000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 2, 0, 0}, 178.924700, 0.000510, 0.040000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 2, 1, 0}, 178.942440, 0.000050, 0.026000},
	{{0, 3, -5, 0, 2, 0, 0, 0, 4, -1, 0}, 178.966470, 0.000020, 0.028000},
	{{0, 3, -5, 0, 3, 0, 0, 0, -2, 0, 0}, 170.283520, 0.000020, 0.041000},
	{{0, 3, -5, 0, 3, 0, 0, 0, -2, 1, 0}, 170.283520, 0.000020, 0.088000},
	{{0, 3, -5, 0, 3, 0, 0, 0, 0, -1, 0}, 169.268440, 0.000020, 0.076000},
	{{0, 3, -5, 0, 3, 0, 0, 0, 0, 0, 0}, 167.787790, 0.000170, 7.723000},
	{{0, 3, -5, 0, 3, 0, 0, 0, 0, 1, 0}, 169.268440, 0.000020, 0.075000},
	{{0, 3, -5, 0, 3, 0, 0, 0, 2, -1, 0}, 170.199410, 0.000040, 0.086000},
	{{0, 3, -5, 0, 3, 0, 0, 0, 2, 0, 0}, 170.272900, 0.000020, 0.040000},
	{{0, 3, -5, 0, 4, 0, 0, 0, 0, 0, 0}, 158.733950, 0.000030, 4.678000},
	{{0, 3, -5, 2, 0, 0, 0, 0, -2, 0, 0}, 359.160630, 0.000030, 0.042000},
	{{0, 3, -5, 2, 0, 0, 0, 0, 0, -1, 0}, 180.822890, 0.000020, 0.081000},
	{{0, 3, -5, 2, 0, 0, 0, 0, 0, 0, 0}, 359.798920, 0.000100, 1.064000},
	{{0, 3, -5, 2, 0, 0, 0, 0, 0, 1, 0}, 180.822890, 0.000020, 0.070000},
	{{0, 3, -5, 2, 0, 0, 0, 0, 2, -3, 0}, 179.455740, 0.000020, 0.071000},
	{{0, 3, -5, 2, 0, 0, 0, 0, 2, -1, 0}, 179.455740, 0.000020, 0.081000},
	{{0, 3, -5, 2, 0, 0, 0, 0, 2, 0, 0}, 181.826200, 0.000020, 0.039000},
	{{0, 3, -5, 3, 0, 0, 0, 0, 2, -3, 0}, 210.033510, 0.000010, 0.074000},
	{{0, 3, -5, 3, 0, 0, 0, 0, 2, -1, 0}, 210.033510, 0.000010, 0.077000},
	{{0, 3, -5, 4, 0, 0, 0, 0, -2, 0, 0}, 312.527750, 0.000020, 0.044000},
	{{0, 3, -4, -5, 0, 0, 0, 0, -2, 2, 0}, 110.879660, 0.000020, 127.646000},
	{{0, 3, -4, -3, 0, 0, 0, 0, -1, 2, -1}, 120.960530, 0.000010, 565.421000},
	{{0, 3, -4, -2, 0, 0, 0, 0, 0, 0, 0}, 302.190300, 0.000010, 5.353000},
	{{0, 3, -4, -1, 0, 0, 0, 0, 0, 0, 0}, 171.624400, 0.000020, 2.900000},
	{{0, 3, -4, 0, -1, 0, 0, 0, 2, -3, 0}, 279.703460, 0.000010, 0.070000},
	{{0, 3, -4, 0, -1, 0, 0, 0, 2, -1, 0}, 99.703460, 0.000010, 0.081000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -6, 1, 0}, 271.188260, 0.000020, 0.017000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -6, 2, 0}, 270.658980, 0.000010, 0.021000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -4, -1, 0}, 271.266390, 0.000080, 0.016000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -4, 0, 0}, 269.884910, 0.000440, 0.021000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -4, 0, 2}, 270.626350, 0.000020, 0.046000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -4, 1, 0}, 270.038380, 0.001000, 0.028000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -4, 1, 2}, 270.584730, 0.000010, 0.118000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -4, 2, 0}, 270.033760, 0.000620, 0.045000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -4, 3, 0}, 270.354520, 0.000020, 0.113000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -3, 0, 0}, 90.319310, 0.000070, 0.028000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -3, 0, 1}, 76.621370, 0.000020, 0.044000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -3, 1, 0}, 90.342840, 0.000190, 0.044000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -3, 1, 1}, 76.584280, 0.000010, 0.105000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, -3, 0}, 270.590410, 0.000020, 0.016000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, -2, 0}, 270.632880, 0.000260, 0.020000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, -1, -2}, 90.607920, 0.000020, 0.016000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, -1, 0}, 270.403070, 0.003340, 0.027000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, -1, 2}, 90.595770, 0.000060, 0.097000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, 0, -2}, 90.609120, 0.000060, 0.020000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, 0, 0}, 270.557410, 0.036680, 0.042000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, 0, 2}, 270.594670, 0.000480, 0.335000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, 1, -2}, 90.626780, 0.000120, 0.027000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, 1, 0}, 270.606420, 0.046620, 0.094000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, 1, 2}, 270.715640, 0.000020, 0.062000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, 2, 0}, 270.793740, 0.001980, 0.377000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -2, 3, 0}, 270.727820, 0.000110, 0.063000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, -2, 0}, 270.192540, 0.000080, 0.026000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, -1, 0}, 270.265610, 0.000310, 0.040000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, -1, 1}, 76.455650, 0.000150, 0.088000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, -1, 2}, 270.341110, 0.000020, 0.477000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, 0, -2}, 270.282160, 0.000020, 0.026000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, 0, -1}, 256.588200, 0.000300, 0.040000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, 0, 0}, 90.349690, 0.011970, 0.087000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, 0, 1}, 76.599680, 0.001020, 0.518000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, 1, -1}, 256.622260, 0.000280, 0.086000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, 1, 0}, 90.335330, 0.001250, 0.567000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, 1, 1}, 76.628220, 0.000100, 0.066000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, 2, -1}, 256.584180, 0.000020, 0.626000},
	{{0, 3, -4, 0, 0, 0, 0, 0, -1, 2, 0}, 90.282870, 0.000110, 0.067000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, -3, 0}, 271.427490, 0.000190, 0.026000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, -2, 0}, 271.347480, 0.002650, 0.039000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, -1, -2}, 90.866380, 0.000090, 0.025000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, -1, 0}, 271.173910, 0.039450, 0.081000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, -1, 2}, 272.815900, 0.000020, 0.069000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 0, -2}, 91.091010, 0.000040, 0.039000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 0, -1}, 256.483460, 0.000070, 0.080000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 0, 0}, 270.993160, 0.175110, 1.141000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 0, 1}, 76.311810, 0.000040, 0.070000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 0, 2}, 90.870850, 0.000130, 0.036000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 1, -1}, 260.385880, 0.000020, 1.409000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 1, 0}, 271.086660, 0.029110, 0.071000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 1, 2}, 90.928580, 0.000080, 0.024000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 2, 0}, 271.119700, 0.002020, 0.037000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 0, 3, 0}, 271.056750, 0.000130, 0.025000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -5, 0}, 90.310130, 0.000010, 0.019000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -4, 0}, 269.952810, 0.000020, 0.025000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -3, -2}, 270.310130, 0.000010, 0.019000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -3, 0}, 269.902850, 0.000280, 0.038000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -2, -2}, 90.161060, 0.000020, 0.025000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -2, 0}, 269.895230, 0.004050, 0.075000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -2, 1}, 92.563330, 0.000030, 6.397000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -1, -2}, 90.354850, 0.000080, 0.037000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -1, -1}, 77.795710, 0.000050, 0.074000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -1, 0}, 270.342920, 0.103860, 95.796000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -1, 1}, 255.847850, 0.000140, 0.075000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, -1, 2}, 90.354850, 0.000080, 0.037000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 0, -2}, 270.287940, 0.000120, 0.074000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 0, -1}, 256.477500, 0.002970, 5.644000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 0, 0}, 270.286040, 0.065140, 0.075000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 0, 1}, 77.233880, 0.000470, 0.037000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 0, 2}, 90.292180, 0.000140, 0.025000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 1, -1}, 77.288800, 0.000080, 0.076000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 1, 0}, 270.289590, 0.004430, 0.038000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 1, 1}, 76.222510, 0.000040, 0.025000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 1, 2}, 90.310130, 0.000010, 0.019000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 2, 0}, 270.287580, 0.000310, 0.025000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 1, 3, 0}, 270.310130, 0.000010, 0.019000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, -3, 0}, 166.941470, 0.000040, 0.071000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, -2, 0}, 80.456050, 0.000420, 1.114000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, -1, -2}, 91.782700, 0.000020, 0.069000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, -1, -1}, 81.612660, 0.000010, 0.940000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, -1, 0}, 282.081850, 0.004310, 0.081000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, 0, -2}, 271.827250, 0.000220, 0.812000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, 0, -1}, 76.528310, 0.000060, 0.082000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, 0, 0}, 272.396620, 0.004620, 0.039000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, 1, -2}, 91.782700, 0.000020, 0.083000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, 1, 0}, 271.751140, 0.000560, 0.026000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 2, 2, 0}, 270.767290, 0.000080, 0.019000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 3, -3, 0}, 271.332320, 0.000030, 0.560000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 3, -2, 0}, 270.568250, 0.001490, 0.087000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 3, -1, -2}, 270.341110, 0.000020, 0.472000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 3, -1, -1}, 256.694330, 0.000070, 0.088000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 3, -1, 0}, 270.332060, 0.001990, 0.040000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 3, 0, -1}, 256.645160, 0.000020, 0.041000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 3, 0, 0}, 270.297500, 0.000700, 0.026000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 3, 1, 0}, 270.293100, 0.000090, 0.020000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 4, -3, 0}, 93.384560, 0.000060, 0.094000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 4, -2, 0}, 282.789720, 0.000370, 0.042000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 4, -1, 0}, 281.151150, 0.000520, 0.027000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 4, 0, 0}, 274.755620, 0.000080, 0.020000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 4, 1, 0}, 91.695370, 0.000030, 0.016000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 5, -3, 0}, 270.565240, 0.000020, 0.044000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 5, -2, 0}, 270.401290, 0.000040, 0.028000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 5, -1, 0}, 270.327250, 0.000030, 0.020000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 6, -3, 0}, 282.988630, 0.000010, 0.028000},
	{{0, 3, -4, 0, 0, 0, 0, 0, 6, -2, 0}, 283.100520, 0.000020, 0.021000},
	{{0, 3, -4, 0, 1, 0, 0, 0, -2, 0, 0}, 203.349150, 0.000020, 0.042000},
	{{0, 3, -4, 0, 1, 0, 0, 0, -2, 1, 0}, 354.524430, 0.000110, 0.095000},
	{{0, 3, -4, 0, 1, 0, 0, 0, -2, 2, 0}, 83.095880, 0.000010, 0.366000},
	{{0, 3, -4, 0, 1, 0, 0, 0, -2, 3, 0}, 355.984420, 0.000010, 0.063000},
	{{0, 3, -4, 0, 1, 0, 0, 0, 0, -1, 0}, 342.334300, 0.000010, 0.081000},
	{{0, 3, -4, 0, 1, 0, 0, 0, 0, 0, 0}, 356.762130, 0.000440, 1.041000},
	{{0, 3, -4, 0, 1, 0, 0, 0, 0, 1, 0}, 342.334300, 0.000010, 0.070000},
	{{0, 3, -4, 0, 1, 0, 0, 0, 0, 2, 0}, 178.688720, 0.000030, 0.036000},
	{{0, 3, -4, 0, 1, 0, 0, 0, 2, -2, 0}, 178.165900, 0.000010, 1.230000},
	{{0, 3, -4, 0, 1, 0, 0, 0, 2, -1, 0}, 327.785410, 0.000040, 0.080000},
	{{0, 3, -4, 0, 1, 0, 0, 0, 2, 0, 0}, 178.165850, 0.000010, 0.039000},
	{{0, 3, -4, 0, 2, 0, 0, 0, -2, 0, 0}, 83.490650, 0.000040, 0.042000},
	{{0, 3, -4, 0, 2, 0, 0, 0, -2, 1, 0}, 83.490650, 0.000050, 0.096000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 0, -1, 0}, 240.204880, 0.000010, 0.082000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 0, 0, 0}, 107.387480, 0.000040, 0.957000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 0, 1, 0}, 240.204880, 0.000010, 0.070000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 1, -1, -1}, 230.961690, 0.000020, 0.075000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 1, 0, -1}, 112.122420, 0.000060, 116.455000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 1, 1, -1}, 230.961690, 0.000020, 0.075000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 2, -3, 0}, 60.691540, 0.000010, 0.072000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 2, -2, 0}, 60.552030, 0.000010, 1.372000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 2, -1, 0}, 88.413150, 0.000030, 0.080000},
	{{0, 3, -4, 0, 2, 0, 0, 0, 2, 0, 0}, 60.272590, 0.000040, 0.039000},
	{{0, 3, -4, 1, 0, 0, 0, 0, 2, -3, 0}, 359.527870, 0.000010, 0.073000},
	{{0, 3, -4, 1, 0, 0, 0, 0, 2, -1, 0}, 359.527870, 0.000010, 0.078000},
	{{0, 3, -4, 2, 0, 0, 0, 0, 2, -3, 0}, 111.430750, 0.000020, 0.076000},
	{{0, 3, -4, 2, 0, 0, 0, 0, 2, -1, 0}, 111.430750, 0.000020, 0.075000},
	{{0, 3, -4, 3, 0, 0, 0, 0, 2, -3, 0}, 51.716420, 0.000010, 0.080000},
	{{0, 3, -4, 3, 0, 0, 0, 0, 2, -2, 0}, 236.876330, 0.000010, 1.434000},
	{{0, 3, -4, 3, 0, 0, 0, 0, 2, -1, 0}, 51.716420, 0.000010, 0.072000},
	{{0, 3, -4, 3, 0, 0, 0, 0, 2, 0, 0}, 236.876330, 0.000010, 0.037000},
	{{0, 3, -3, -2, 0, 0, 0, 0, 0, 0, 0}, 33.603230, 0.000010, 1.230000},
	{{0, 3, -3, 0, -2, 0, 0, 0, -2, 0, 0}, 205.439320, 0.000010, 0.043000},
	{{0, 3, -3, 0, -2, 0, 0, 0, 0, 0, 0}, 340.447980, 0.000010, 0.586000},
	{{0, 3, -3, 0, -2, 0, 0, 0, 2, -2, 0}, 14.763280, 0.000010, 15.139000},
	{{0, 3, -3, 0, -2, 0, 0, 0, 2, -1, 0}, 30.309520, 0.000030, 0.076000},
	{{0, 3, -3, 0, -1, 0, 0, 0, -4, 2, 0}, 50.938530, 0.000010, 0.047000},
	{{0, 3, -3, 0, -1, 0, 0, 0, -2, 0, 0}, 206.569850, 0.000080, 0.044000},
	{{0, 3, -3, 0, -1, 0, 0, 0, -2, 1, 0}, 13.056130, 0.000030, 0.103000},
	{{0, 3, -3, 0, -1, 0, 0, 0, -2, 2, 0}, 272.295120, 0.000030, 0.280000},
	{{0, 3, -3, 0, -1, 0, 0, 0, -2, 3, 0}, 13.056130, 0.000020, 0.059000},
	{{0, 3, -3, 0, -1, 0, 0, 0, 0, -2, 0}, 193.056130, 0.000010, 0.040000},
	{{0, 3, -3, 0, -1, 0, 0, 0, 0, -1, 0}, 86.963480, 0.000010, 0.087000},
	{{0, 3, -3, 0, -1, 0, 0, 0, 0, 0, 0}, 46.310700, 0.000030, 0.558000},
	{{0, 3, -3, 0, -1, 0, 0, 0, 2, -3, 0}, 283.220410, 0.000020, 0.076000},
	{{0, 3, -3, 0, -1, 0, 0, 0, 2, -2, 0}, 214.120940, 0.000060, 54.802000},
	{{0, 3, -3, 0, -1, 0, 0, 0, 2, -1, 0}, 262.565530, 0.000080, 0.075000},
	{{0, 3, -3, 0, -1, 0, 0, 0, 2, 0, 0}, 191.810540, 0.000050, 0.038000},
	{{0, 3, -3, 0, -1, 0, 0, 0, 4, -3, 0}, 86.963480, 0.000010, 0.087000},
	{{0, 3, -3, 0, 0, -3, 0, 0, 2, -3, 0}, 116.871520, 0.000050, 0.075000},
	{{0, 3, -3, 0, 0, -3, 0, 0, 2, -2, 0}, 122.063660, 0.000700, 1412.983000},
	{{0, 3, -3, 0, 0, -3, 0, 0, 2, -1, 0}, 116.871520, 0.000050, 0.075000},
	{{0, 3, -3, 0, 0, -1, 0, 0, 2, -3, 0}, 93.217480, 0.000010, 0.076000},
	{{0, 3, -3, 0, 0, -1, 0, 0, 2, -2, 0}, 272.899560, 0.000010, 14.577000},
	{{0, 3, -3, 0, 0, -1, 0, 0, 2, -1, 0}, 92.790850, 0.000070, 0.075000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -6, 1, 0}, 4.004480, 0.000010, 0.017000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -4, -1, 0}, 356.957140, 0.000030, 0.016000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -4, 0, 0}, 7.106320, 0.000140, 0.021000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -4, 0, 2}, 2.087240, 0.000020, 0.048000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -4, 1, 0}, 4.923480, 0.000340, 0.029000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -4, 2, 0}, 5.257030, 0.000170, 0.047000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -4, 3, 0}, 179.595560, 0.000010, 0.128000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -3, 1, 0}, 182.969740, 0.000020, 0.046000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, -2, 0}, 0.523950, 0.000240, 0.020000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, -1, 0}, 1.881090, 0.002150, 0.028000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, -1, 2}, 182.039870, 0.000030, 0.108000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, 0, -2}, 182.650120, 0.000030, 0.020000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, 0, 0}, 2.821740, 0.015020, 0.044000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, 0, 2}, 3.198270, 0.000110, 0.251000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, 1, -2}, 182.692240, 0.000010, 0.027000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, 1, 0}, 2.665940, 0.014870, 0.104000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, 1, 2}, 182.364400, 0.000030, 0.058000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, 2, 0}, 1.644130, 0.000820, 0.274000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -2, 3, 0}, 1.012690, 0.000130, 0.059000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -1, -2, 0}, 177.138300, 0.000030, 0.027000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -1, -1, 0}, 176.842720, 0.000170, 0.042000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -1, 0, -1}, 178.578430, 0.000030, 0.042000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -1, 0, 0}, 340.393920, 0.000330, 0.095000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -1, 0, 1}, 354.245770, 0.000130, 0.341000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -1, 1, -1}, 0.334640, 0.000010, 0.094000},
	{{0, 3, -3, 0, 0, 0, 0, 0, -1, 1, 0}, 359.436670, 0.000060, 0.362000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, -4, 0}, 0.097270, 0.000040, 0.020000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, -3, 0}, 0.143690, 0.000630, 0.026000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, -2, -2}, 179.986280, 0.000050, 0.019000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, -2, 0}, 0.139990, 0.008840, 0.041000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, -1, -2}, 180.110740, 0.000250, 0.026000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, -1, 0}, 0.120460, 0.125370, 0.088000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, -1, 2}, 179.637380, 0.000040, 0.065000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 0, -2}, 359.763600, 0.000070, 0.040000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 0, -1}, 11.319030, 0.000030, 0.087000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 0, 0}, 179.099750, 0.043400, 0.533000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 0, 2}, 358.641400, 0.000030, 0.035000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 1, -2}, 0.122110, 0.000200, 0.085000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 1, 0}, 179.181270, 0.007370, 0.066000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 1, 2}, 358.880140, 0.000030, 0.024000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 2, -2}, 359.526470, 0.000020, 0.648000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 2, 0}, 179.874510, 0.000540, 0.035000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 0, 3, 0}, 179.611270, 0.000050, 0.024000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, -2, 0}, 174.998520, 0.000060, 0.082000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, -1, -1}, 182.090300, 0.000090, 0.080000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, -1, 0}, 183.043550, 0.000190, 1.011000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, -1, 1}, 6.592030, 0.000060, 0.069000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, 0, -2}, 0.011780, 0.000030, 0.079000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, 0, -1}, 184.509640, 0.000710, 1.215000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, 0, 0}, 358.282580, 0.002260, 0.070000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, 0, 1}, 4.698410, 0.000060, 0.036000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, 1, -1}, 182.090300, 0.000090, 0.071000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 1, 1, 0}, 356.871020, 0.000160, 0.036000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, -4, 0}, 3.938140, 0.000070, 0.038000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, -3, -2}, 180.584200, 0.000010, 0.025000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, -3, 0}, 3.139590, 0.001220, 0.076000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, -2, -2}, 180.368270, 0.000130, 0.037000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, -2, 0}, 0.041670, 0.071990, 9.751000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, -1, -2}, 179.993990, 0.001870, 0.074000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, -1, -1}, 192.877030, 0.000040, 15.575000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, -1, 0}, 179.981440, 0.643710, 0.075000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, -1, 2}, 359.979180, 0.001280, 0.025000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, 0, -2}, 0.033940, 0.003050, 4.330000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, 0, -1}, 191.318990, 0.000150, 0.076000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, 0, 0}, 179.942770, 0.030300, 0.038000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, 0, 2}, 359.963690, 0.000180, 0.019000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, 1, -2}, 180.015030, 0.000720, 0.077000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, 1, 0}, 179.843750, 0.002070, 0.025000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, 1, 2}, 359.980760, 0.000030, 0.015000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, 2, -2}, 180.014600, 0.000050, 0.038000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 2, 2, 0}, 179.820110, 0.000080, 0.019000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 3, -2, 0}, 181.488580, 0.000020, 0.080000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 3, -1, 0}, 359.021320, 0.000280, 0.039000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 3, 0, 0}, 0.025060, 0.000080, 0.026000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 3, 1, 0}, 0.060440, 0.000010, 0.019000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, -4, 0}, 358.636270, 0.000050, 0.598000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, -3, 0}, 359.536430, 0.001330, 0.086000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, -2, -2}, 0.008470, 0.000040, 0.499000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, -2, 0}, 179.978590, 0.007650, 0.040000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, -2, 2}, 359.984580, 0.000050, 0.019000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, -1, -2}, 359.998180, 0.000350, 0.089000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, -1, 0}, 179.968910, 0.005770, 0.026000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, -1, 2}, 359.981680, 0.000030, 0.015000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, 0, -2}, 359.860650, 0.000010, 0.041000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, 0, 0}, 179.984280, 0.000710, 0.019000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 4, 1, 0}, 179.913680, 0.000040, 0.015000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 5, -2, 0}, 359.988180, 0.000010, 0.027000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 6, -4, 0}, 359.433510, 0.000020, 0.043000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 6, -3, 0}, 180.223830, 0.000090, 0.028000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 6, -2, 0}, 179.982320, 0.000160, 0.020000},
	{{0, 3, -3, 0, 0, 0, 0, 0, 6, -1, 0}, 179.964430, 0.000060, 0.016000},
	{{0, 3, -3, 0, 0, 1, 0, 0, 2, -3, 0}, 87.684060, 0.000010, 0.076000},
	{{0, 3, -3, 0, 0, 1, 0, 0, 2, -2, 0}, 87.674780, 0.000010, 7.326000},
	{{0, 3, -3, 0, 0, 1, 0, 0, 2, -1, 0}, 87.689010, 0.000040, 0.075000},
	{{0, 3, -3, 0, 1, 0, 0, 0, -2, 0, 0}, 328.016110, 0.000070, 0.044000},
	{{0, 3, -3, 0, 1, 0, 0, 0, 0, -1, 0}, 171.265450, 0.000010, 0.089000},
	{{0, 3, -3, 0, 1, 0, 0, 0, 0, 0, 0}, 187.958840, 0.000050, 0.510000},
	{{0, 3, -3, 0, 1, 0, 0, 0, 0, 1, 0}, 171.265450, 0.000010, 0.066000},
	{{0, 3, -3, 0, 1, 0, 0, 0, 2, -2, 0}, 37.405540, 0.000010, 5.352000},
	{{0, 3, -3, 0, 1, 0, 0, 0, 2, -1, 0}, 202.713940, 0.000040, 0.074000},
	{{0, 3, -3, 0, 1, 0, 0, 0, 2, 0, 0}, 332.059200, 0.000040, 0.037000},
	{{0, 3, -3, 0, 2, 0, 0, 0, 2, -3, 0}, 327.041260, 0.000010, 0.077000},
	{{0, 3, -3, 0, 2, 0, 0, 0, 2, -1, 0}, 183.886430, 0.000030, 0.074000},
	{{0, 3, -2, -3, 0, 0, 0, 0, 2, -3, 0}, 303.513440, 0.000030, 0.073000},
	{{0, 3, -2, -3, 0, 0, 0, 0, 2, -2, 0}, 303.564730, 0.000010, 2.030000},
	{{0, 3, -2, -3, 0, 0, 0, 0, 2, -1, 0}, 303.513440, 0.000030, 0.078000},
	{{0, 3, -2, -3, 0, 0, 0, 0, 2, 0, 0}, 303.564730, 0.000010, 0.038000},
	{{0, 3, -2, -2, 0, 0, 0, 0, 0, 0, 0}, 102.186070, 0.000010, 0.552000},
	{{0, 3, -2, -2, 0, 0, 0, 0, 2, -2, 0}, 125.990300, 0.000010, 25.521000},
	{{0, 3, -2, -2, 0, 0, 0, 0, 2, -1, 0}, 156.776260, 0.000020, 0.075000},
	{{0, 3, -2, -2, 0, 0, 0, 0, 2, 0, 0}, 139.681990, 0.000020, 0.038000},
	{{0, 3, -2, 0, -2, 0, 0, 0, -2, 0, 0}, 89.596780, 0.000020, 0.045000},
	{{0, 3, -2, 0, -2, 0, 0, 0, -2, 1, 0}, 119.791530, 0.000010, 0.114000},
	{{0, 3, -2, 0, -2, 0, 0, 0, -2, 2, 0}, 269.596780, 0.000020, 0.223000},
	{{0, 3, -2, 0, -2, 0, 0, 0, 2, -3, 0}, 308.545450, 0.000010, 0.081000},
	{{0, 3, -2, 0, -2, 0, 0, 0, 2, -1, 0}, 308.545450, 0.000010, 0.070000},
	{{0, 3, -2, 0, -1, 0, 0, 0, -2, 0, 0}, 2.310980, 0.000010, 0.046000},
	{{0, 3, -2, 0, -1, 0, 0, 0, -2, 2, 0}, 1.488920, 0.000030, 0.219000},
	{{0, 3, -2, 0, -1, 0, 0, 0, 0, -2, 0}, 184.996260, 0.000010, 0.042000},
	{{0, 3, -2, 0, -1, 0, 0, 0, 0, -1, 0}, 4.785240, 0.000020, 0.096000},
	{{0, 3, -2, 0, -1, 0, 0, 0, 0, 0, 0}, 179.612560, 0.000010, 0.358000},
	{{0, 3, -2, 0, -1, 0, 0, 0, 2, -1, 0}, 184.785240, 0.000100, 0.070000},
	{{0, 3, -2, 0, -1, 0, 0, 0, 2, 0, 0}, 4.004420, 0.000080, 0.036000},
	{{0, 3, -2, 0, 0, -1, 0, 0, -2, 0, 0}, 180.215480, 0.000010, 0.046000},
	{{0, 3, -2, 0, 0, -1, 0, 0, -2, 2, 0}, 0.215480, 0.000010, 0.217000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -4, 0, 0}, 256.564430, 0.000030, 0.021000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -4, 1, 0}, 254.918020, 0.000020, 0.030000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -3, 0, 1}, 103.182240, 0.000020, 0.048000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -3, 1, 1}, 103.315860, 0.000020, 0.132000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -2, -2, 0}, 257.126180, 0.000020, 0.021000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -2, -1, 0}, 256.959670, 0.000290, 0.028000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -2, 0, 0}, 256.612570, 0.001610, 0.046000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -2, 0, 2}, 258.141150, 0.000010, 0.201000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -2, 1, 0}, 257.166010, 0.001350, 0.116000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -2, 2, 0}, 255.301530, 0.000060, 0.215000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -1, -1, -1}, 103.141260, 0.000020, 0.028000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -1, -1, 1}, 283.563470, 0.000090, 0.107000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -1, 0, -1}, 103.164440, 0.000420, 0.044000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -1, 0, 0}, 255.812150, 0.000030, 0.105000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -1, 0, 1}, 283.156560, 0.000870, 0.254000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -1, 1, -1}, 103.315880, 0.000220, 0.104000},
	{{0, 3, -2, 0, 0, 0, 0, 0, -1, 1, 1}, 283.230550, 0.000080, 0.058000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 0, -3, 0}, 256.884510, 0.000030, 0.027000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 0, -2, 0}, 256.858650, 0.000470, 0.042000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 0, -1, 0}, 256.689040, 0.005940, 0.096000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 0, 0, -1}, 282.774280, 0.000040, 0.095000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 0, 0, 0}, 191.121530, 0.000410, 0.348000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 0, 0, 1}, 103.322820, 0.000030, 0.061000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 0, 1, 0}, 131.123660, 0.000050, 0.062000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 0, 2, 0}, 78.941950, 0.000010, 0.034000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 1, -2, 1}, 283.323710, 0.000010, 0.464000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 1, -1, -1}, 283.115920, 0.000050, 0.087000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 1, -1, 1}, 283.306350, 0.000260, 0.065000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 1, 0, -1}, 103.310100, 0.002740, 0.549000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 1, 0, 0}, 273.386520, 0.000030, 0.066000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 1, 0, 1}, 283.298390, 0.000250, 0.035000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 1, 1, -1}, 103.309500, 0.000330, 0.066000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 1, 1, 1}, 283.299500, 0.000030, 0.024000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 1, 2, -1}, 103.307790, 0.000030, 0.035000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 2, -3, 0}, 86.727800, 0.000040, 0.082000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 2, -2, 0}, 257.245980, 0.000460, 0.907000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 2, -1, 0}, 256.217780, 0.003650, 0.070000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 2, 0, -2}, 256.541110, 0.000060, 1.300000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 2, 0, -1}, 283.341090, 0.000080, 0.070000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 2, 0, 0}, 257.548610, 0.000140, 0.036000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 3, -2, -1}, 283.273930, 0.000430, 20.498000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 3, -2, 0}, 77.049350, 0.000020, 0.074000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 3, -1, -1}, 103.287460, 0.002390, 0.075000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 3, -1, 0}, 77.044010, 0.000040, 0.037000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 3, 0, -1}, 103.293230, 0.000200, 0.038000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 4, -3, 0}, 262.281630, 0.000020, 0.079000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 4, -2, 0}, 256.375800, 0.000100, 0.039000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 4, -1, 0}, 257.352890, 0.000080, 0.026000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 5, -2, -1}, 103.287310, 0.000030, 0.040000},
	{{0, 3, -2, 0, 0, 0, 0, 0, 5, -1, -1}, 103.287420, 0.000020, 0.026000},
	{{0, 3, -1, -6, 0, 0, 0, 0, 0, 0, 0}, 80.321980, 0.000010, 1.457000},
	{{0, 3, -1, -4, 0, 0, 0, 0, 2, -3, 0}, 146.946790, 0.000020, 0.075000},
	{{0, 3, -1, -4, 0, 0, 0, 0, 2, -2, 0}, 301.816640, 0.000030, 41.349000},
	{{0, 3, -1, -3, 0, 0, 0, 0, 0, 0, 0}, 310.377250, 0.000010, 0.438000},
	{{0, 3, -1, -2, 0, 0, 0, 0, -2, 0, 0}, 0.495020, 0.000030, 0.046000},
	{{0, 3, -1, -2, 0, 0, 0, 0, -2, 1, 0}, 179.894000, 0.000010, 0.115000},
	{{0, 3, -1, -2, 0, 0, 0, 0, -2, 2, 0}, 0.495020, 0.000030, 0.218000},
	{{0, 3, -1, -2, 0, 0, 0, 0, -2, 3, 0}, 179.894000, 0.000010, 0.056000},
	{{0, 3, -1, -2, 0, 0, 0, 0, 0, -2, 0}, 179.894000, 0.000010, 0.042000},
	{{0, 3, -1, -2, 0, 0, 0, 0, 0, 0, 0}, 179.894000, 0.000010, 0.355000},
	{{0, 3, -1, 0, -2, 0, 0, 0, -2, -2, 0}, 180.105550, 0.000010, 0.021000},
	{{0, 3, -1, 0, -2, 0, 0, 0, -2, 1, 0}, 180.024490, 0.000010, 0.129000},
	{{0, 3, -1, 0, -2, 0, 0, 0, -2, 2, 0}, 0.310880, 0.000010, 0.182000},
	{{0, 3, -1, 0, -2, 0, 0, 0, 0, -2, 0}, 359.989550, 0.000030, 0.044000},
	{{0, 3, -1, 0, -2, 0, 0, 0, 0, -1, 0}, 0.105550, 0.000010, 0.105000},
	{{0, 3, -1, 0, -2, 0, 0, 0, 0, 0, 0}, 359.989550, 0.000030, 0.270000},
	{{0, 3, -1, 0, -2, 0, 0, 0, 0, 1, 0}, 0.105550, 0.000010, 0.059000},
	{{0, 3, -1, 0, -1, 0, 0, 0, -2, 0, 0}, 90.497840, 0.000020, 0.048000},
	{{0, 3, -1, 0, -1, 0, 0, 0, -2, 2, 0}, 90.407490, 0.000030, 0.180000},
	{{0, 3, -1, 0, 0, 0, 0, 0, -2, -1, 0}, 39.255490, 0.000030, 0.029000},
	{{0, 3, -1, 0, 0, 0, 0, 0, -2, 0, 0}, 43.920930, 0.000330, 0.048000},
	{{0, 3, -1, 0, 0, 0, 0, 0, -2, 1, 0}, 42.141670, 0.000220, 0.131000},
	{{0, 3, -1, 0, 0, 0, 0, 0, -2, 2, 0}, 13.049370, 0.000010, 0.177000},
	{{0, 3, -1, 0, 0, 0, 0, 0, -1, 0, -1}, 8.197710, 0.000020, 0.046000},
	{{0, 3, -1, 0, 0, 0, 0, 0, -1, 0, 0}, 205.309300, 0.000040, 0.118000},
	{{0, 3, -1, 0, 0, 0, 0, 0, -1, 0, 1}, 189.955910, 0.000040, 0.203000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 0, -1, 0}, 87.782670, 0.000200, 0.107000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 0, 0, 0}, 26.195700, 0.001800, 0.258000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 0, 1, 0}, 23.698840, 0.000170, 0.058000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 1, -1, -1}, 180.350510, 0.000020, 0.096000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 1, 0, -1}, 13.278120, 0.000050, 0.354000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 1, 0, 0}, 208.717220, 0.000020, 0.062000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 1, 0, 1}, 182.615790, 0.000010, 0.034000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 2, -3, 0}, 165.635740, 0.000030, 0.090000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 2, -2, 0}, 26.295830, 0.000020, 0.476000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 2, -1, 0}, 35.472280, 0.000280, 0.065000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 2, 0, 0}, 200.506650, 0.000090, 0.035000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 2, 1, 0}, 203.807260, 0.000010, 0.024000},
	{{0, 3, -1, 0, 0, 0, 0, 0, 4, -2, -2}, 246.513210, 0.000070, 200.852000},
	{{0, 3, 0, -8, 3, 0, 0, 0, 0, 0, 0}, 199.229090, 0.000010, 1.142000},
	{{0, 3, 0, -5, 0, 0, 0, 0, 0, 0, 0}, 120.467180, 0.000010, 0.451000},
	{{0, 3, 0, 0, -2, 0, 0, 0, -2, 0, 0}, 269.026510, 0.000020, 0.050000},
	{{0, 3, 0, 0, -2, 0, 0, 0, -2, 2, 0}, 89.026510, 0.000020, 0.154000},
	{{0, 3, 0, 0, -1, 0, 0, 0, 0, 0, 0}, 19.876910, 0.000010, 0.209000},
	{{0, 3, 0, 0, 0, 0, 0, 0, -2, 0, 0}, 293.597090, 0.000060, 0.050000},
	{{0, 3, 0, 0, 0, 0, 0, 0, -2, 1, 0}, 289.943030, 0.000030, 0.151000},
	{{0, 3, 0, 0, 0, 0, 0, 0, -1, 0, 1}, 124.547830, 0.000010, 0.169000},
	{{0, 3, 0, 0, 0, 0, 0, 0, 0, -1, 0}, 287.505480, 0.000020, 0.119000},
	{{0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0}, 291.787580, 0.000100, 0.205000},
	{{0, 3, 0, 0, 0, 0, 0, 0, 0, 1, 0}, 292.042930, 0.000010, 0.055000},
	{{0, 3, 0, 0, 0, 0, 0, 0, 1, 0, -1}, 307.113120, 0.000020, 0.262000},
	{{0, 3, 0, 0, 0, 0, 0, 0, 2, -3, 0}, 218.638570, 0.000020, 0.098000},
	{{0, 3, 0, 0, 0, 0, 0, 0, 2, -1, 0}, 218.638570, 0.000020, 0.061000},
	{{0, 3, 0, 0, 0, 0, 0, 0, 2, 0, 0}, 192.368510, 0.000020, 0.034000},
	{{0, 3, 0, 0, 1, 0, 0, 0, 2, 0, 0}, 131.165290, 0.000060, 0.034000},
	{{0, 3, 1, 0, -1, 0, 0, 0, 2, -2, 0}, 147.832890, 0.000010, 0.249000},
	{{0, 3, 1, 0, -1, 0, 0, 0, 2, -1, 0}, 141.308410, 0.000030, 0.058000},
	{{0, 3, 1, 0, -1, 0, 0, 0, 2, 0, 0}, 154.352320, 0.000030, 0.033000},
	{{0, 3, 1, 0, 0, 0, 0, 0, -2, 1, 0}, 308.175220, 0.000020, 0.178000},
	{{0, 3, 1, 0, 0, 0, 0, 0, 0, -1, 0}, 308.326870, 0.000020, 0.136000},
	{{0, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0}, 206.450840, 0.000020, 0.170000},
	{{0, 3, 1, 0, 0, 0, 0, 0, 0, 1, 0}, 308.326870, 0.000020, 0.052000},
	{{0, 3, 1, 0, 0, 0, 0, 0, 2, -2, 0}, 126.206600, 0.000050, 0.244000},
	{{0, 3, 1, 0, 0, 0, 0, 0, 2, 0, 0}, 127.737790, 0.000060, 0.033000},
	{{0, 3, 1, 0, 0, 0, 0, 0, 2, 1, 0}, 308.232310, 0.000020, 0.023000},
	{{0, 3, 1, 0, 0, 0, 0, 0, 2, 2, 0}, 128.211620, 0.000020, 0.018000},
	{{0, 3, 1, 0, 0, 0, 0, 0, 4, -2, 0}, 306.206600, 0.000010, 0.035000},
	{{0, 3, 1, 0, 1, 0, 0, 0, 2, -2, 0}, 114.915050, 0.000010, 0.239000},
	{{0, 3, 1, 0, 1, 0, 0, 0, 2, -1, 0}, 299.517730, 0.000030, 0.057000},
	{{0, 3, 1, 0, 1, 0, 0, 0, 2, 0, 0}, 294.915050, 0.000010, 0.033000},
	{{0, 3, 2, -9, 0, 0, 0, 0, 2, 0, -2}, 243.478240, 0.000010, 62.037000},
	{{0, 3, 2, -3, 0, 0, 0, 0, 2, -2, 0}, 251.372360, 0.000010, 0.285000},
	{{0, 3, 2, -3, 0, 0, 0, 0, 2, 0, 0}, 251.372360, 0.000010, 0.033000},
	{{0, 3, 2, 0, -1, 0, 0, 0, 2, 0, 0}, 309.711090, 0.000020, 0.032000},
	{{0, 3, 2, 0, 0, 0, 0, 0, -2, 0, 0}, 302.082390, 0.000040, 0.056000},
	{{0, 3, 2, 0, 0, 0, 0, 0, -2, 2, 0}, 302.080980, 0.000050, 0.116000},
	{{0, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0}, 41.794370, 0.000030, 0.145000},
	{{0, 3, 2, 0, 0, 0, 0, 0, 2, 0, 0}, 211.833560, 0.000020, 0.032000},
	{{0, 3, 3, -2, 0, 0, 0, 0, 2, -3, 0}, 308.146280, 0.000010, 0.122000},
	{{0, 3, 3, -2, 0, 0, 0, 0, 2, -2, 0}, 308.090900, 0.000010, 0.198000},
	{{0, 3, 3, -2, 0, 0, 0, 0, 2, -1, 0}, 308.146280, 0.000010, 0.055000},
	{{0, 3, 3, -2, 0, 0, 0, 0, 2, 0, 0}, 308.090900, 0.000010, 0.032000},
	{{0, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0}, 333.060630, 0.000030, 0.127000},
	{{0, 3, 3, 0, 0, 0, 0, 0, 2, -2, 0}, 325.404690, 0.000010, 0.164000},
	{{0, 3, 5, -15, 0, 0, 0, 0, 2, -3, 0}, 156.703710, 0.000030, 0.076000},
	{{0, 3, 5, -15, 0, 0, 0, 0, 2, -1, 0}, 156.703710, 0.000030, 0.075000},
	{{0, 3, 5, 0, -2, 0, 0, 0, -4, 3, 0}, 184.662040, 0.000130, 4160.319000},
	{{0, 3, 7, -19, 0, 0, 0, 0, 2, -3, 0}, 304.874340, 0.000020, 0.075000},
	{{0, 3, 7, -19, 0, 0, 0, 0, 2, -2, 0}, 311.204880, 0.000250, 1812.670000},
	{{0, 3, 7, -19, 0, 0, 0, 0, 2, -1, 0}, 304.874340, 0.000020, 0.075000},
	{{0, 3, 27, -37, 0, 0, 0, 0, -1, -1, 1}, 43.227520, 0.000010, 441.870000},
	{{0, 3, 30, -44, 0, 0, 0, 0, -2, 1, 0}, 38.914670, 0.000100, 1378.938000},
	{{0, 4, -49, 55, 0, 0, 0, 0, 0, 1, 0}, 334.521570, 0.000070, 2078.312000},
	{{0, 4, -16, 15, 0, 0, 0, 0, -2, 1, 0}, 254.750460, 0.000010, 0.077000},
	{{0, 4, -16, 15, 0, 0, 0, 0, -2, 3, 0}, 254.750460, 0.000010, 0.074000},
	{{0, 4, -15, 14, 0, 0, 0, 0, -1, -1, 1}, 286.279690, 0.000020, 0.075000},
	{{0, 4, -15, 14, 0, 0, 0, 0, -1, 0, 1}, 286.279690, 0.000300, 1778.605000},
	{{0, 4, -15, 14, 0, 0, 0, 0, -1, 1, 1}, 286.279690, 0.000020, 0.075000},
	{{0, 4, -12, 7, 0, 0, 0, 0, -2, 2, 0}, 279.685050, 0.000060, 476.157000},
	{{0, 4, -11, 0, 2, 0, 0, 0, -2, 0, 0}, 322.673340, 0.000020, 0.034000},
	{{0, 4, -11, 5, 0, 0, 0, 0, -2, 2, 0}, 289.227810, 0.000020, 15.275000},
	{{0, 4, -10, 0, 0, 0, 0, 0, -2, -1, 0}, 217.103170, 0.000010, 0.024000},
	{{0, 4, -10, 0, 0, 0, 0, 0, -2, 0, 0}, 217.478250, 0.000010, 0.035000},
	{{0, 4, -10, 0, 0, 0, 0, 0, -2, 1, 0}, 98.189240, 0.000010, 0.067000},
	{{0, 4, -10, 0, 0, 0, 0, 0, -2, 2, 0}, 217.478250, 0.000010, 0.580000},
	{{0, 4, -10, 0, 0, 0, 0, 0, 0, -1, 0}, 66.960370, 0.000010, 0.060000},
	{{0, 4, -10, 0, 0, 0, 0, 0, 0, 0, 0}, 246.962830, 0.000020, 0.286000},
	{{0, 4, -10, 0, 0, 0, 0, 0, 0, 1, 0}, 66.960370, 0.000010, 0.102000},
	{{0, 4, -10, 0, 1, 0, 0, 0, -2, 1, 0}, 321.500030, 0.000010, 0.067000},
	{{0, 4, -10, 0, 1, 0, 0, 0, -2, 3, 0}, 321.500030, 0.000010, 0.086000},
	{{0, 4, -10, 0, 2, 0, 0, 0, -2, 0, 0}, 231.662260, 0.000010, 0.036000},
	{{0, 4, -10, 0, 2, 0, 0, 0, -2, 1, 0}, 51.662260, 0.000010, 0.068000},
	{{0, 4, -10, 0, 2, 0, 0, 0, -2, 2, 0}, 231.662260, 0.000010, 0.643000},
	{{0, 4, -10, 0, 2, 0, 0, 0, -2, 3, 0}, 51.662260, 0.000010, 0.085000},
	{{0, 4, -10, 0, 2, 0, 0, 0, 0, -2, 0}, 51.634110, 0.000010, 0.034000},
	{{0, 4, -10, 0, 2, 0, 0, 0, 0, 0, 0}, 231.634110, 0.000010, 0.300000},
	{{0, 4, -10, 0, 3, 0, 0, 0, -2, 0, 0}, 221.882830, 0.000030, 0.036000},
	{{0, 4, -10, 0, 3, 0, 0, 0, -2, 1, 0}, 41.882830, 0.000020, 0.068000},
	{{0, 4, -10, 2, 0, 0, 0, 0, -2, 0, 0}, 231.904220, 0.000020, 0.037000},
	{{0, 4, -10, 2, 0, 0, 0, 0, -2, 2, 0}, 231.904220, 0.000020, 1.514000},
	{{0, 4, -10, 3, 0, 0, 0, 0, -2, 1, 0}, 332.551430, 0.000020, 0.075000},
	{{0, 4, -10, 3, 0, 0, 0, 0, -2, 3, 0}, 332.551430, 0.000020, 0.076000},
	{{0, 4, -9, 0, -1, 0, 0, 0, -2, 1, 0}, 331.513450, 0.000030, 0.071000},
	{{0, 4, -9, 0, 0, 0, 0, 0, -2, -3, 0}, 324.001860, 0.000020, 0.015000},
	{{0, 4, -9, 0, 0, 0, 0, 0, -2, -1, 0}, 324.001860, 0.000020, 0.025000},
	{{0, 4, -9, 0, 0, 0, 0, 0, -2, 1, 0}, 20.067820, 0.000030, 0.072000},
	{{0, 4, -9, 0, 0, 0, 0, 0, -2, 2, 0}, 312.086520, 0.000050, 1.381000},
	{{0, 4, -9, 0, 0, 0, 0, 0, -2, 3, 0}, 37.038150, 0.000030, 0.080000},
	{{0, 4, -9, 0, 0, 0, 0, 0, 0, -2, 0}, 142.875790, 0.000010, 0.034000},
	{{0, 4, -9, 0, 0, 0, 0, 0, 0, -1, 0}, 322.875790, 0.000010, 0.063000},
	{{0, 4, -9, 0, 0, 0, 0, 0, 0, 0, 0}, 133.107690, 0.000040, 0.400000},
	{{0, 4, -9, 0, 0, 0, 0, 0, 0, 1, 0}, 322.875790, 0.000010, 0.093000},
	{{0, 4, -9, 0, 0, 0, 0, 0, 2, -1, 0}, 313.558810, 0.000030, 0.111000},
	{{0, 4, -9, 0, 1, 0, 0, 0, -2, -2, 0}, 50.460800, 0.000010, 0.019000},
	{{0, 4, -9, 0, 1, 0, 0, 0, -2, 1, 0}, 115.749160, 0.000030, 0.072000},
	{{0, 4, -9, 0, 1, 0, 0, 0, -2, 2, 0}, 51.193520, 0.000020, 1.564000},
	{{0, 4, -9, 0, 1, 0, 0, 0, 0, -2, 0}, 50.460800, 0.000020, 0.035000},
	{{0, 4, -9, 0, 1, 0, 0, 0, 0, 0, 0}, 50.460800, 0.000020, 0.414000},
	{{0, 4, -9, 0, 2, 0, 0, 0, -2, 0, 0}, 291.841230, 0.000030, 0.037000},
	{{0, 4, -9, 0, 2, 0, 0, 0, -2, 1, 0}, 32.187730, 0.000010, 0.072000},
	{{0, 4, -9, 0, 2, 0, 0, 0, -2, 2, 0}, 111.768480, 0.000020, 1.801000},
	{{0, 4, -9, 0, 2, 0, 0, 0, -2, 3, 0}, 112.352560, 0.000010, 0.079000},
	{{0, 4, -9, 1, 0, 0, 0, 0, -2, 1, 0}, 51.338990, 0.000020, 0.074000},
	{{0, 4, -9, 1, 0, 0, 0, 0, -2, 3, 0}, 51.338990, 0.000020, 0.077000},
	{{0, 4, -9, 2, 0, 0, 0, 0, -2, 1, 0}, 320.074510, 0.000010, 0.077000},
	{{0, 4, -9, 2, 0, 0, 0, 0, -2, 3, 0}, 320.074510, 0.000010, 0.074000},
	{{0, 4, -9, 3, 0, 0, 0, 0, -2, 0, 0}, 288.619580, 0.000010, 0.039000},
	{{0, 4, -9, 3, 0, 0, 0, 0, -2, 1, 0}, 288.619580, 0.000020, 0.081000},
	{{0, 4, -9, 3, 0, 0, 0, 0, -2, 2, 0}, 288.619580, 0.000010, 1.148000},
	{{0, 4, -9, 3, 0, 0, 0, 0, -2, 3, 0}, 288.619580, 0.000020, 0.071000},
	{{0, 4, -9, 5, 0, 0, 0, 0, 0, 0, 0}, 39.331940, 0.000010, 6.230000},
	{{0, 4, -9, 8, 0, 0, 0, 0, 2, -3, 0}, 266.924230, 0.000020, 0.075000},
	{{0, 4, -9, 8, 0, 0, 0, 0, 2, -1, 0}, 266.924230, 0.000020, 0.076000},
	{{0, 4, -8, 0, -1, 0, 0, 0, -2, 0, 0}, 245.827030, 0.000010, 0.038000},
	{{0, 4, -8, 0, -1, 0, 0, 0, -2, 1, 0}, 324.284420, 0.000010, 0.077000},
	{{0, 4, -8, 0, -1, 0, 0, 0, -2, 3, 0}, 144.284420, 0.000010, 0.074000},
	{{0, 4, -8, 0, -1, 0, 0, 0, 0, -1, 0}, 240.474210, 0.000010, 0.067000},
	{{0, 4, -8, 0, -1, 0, 0, 0, 0, 1, 0}, 240.474210, 0.000010, 0.086000},
	{{0, 4, -8, 0, 0, -1, 0, 0, -2, 1, 0}, 324.315450, 0.000060, 0.077000},
	{{0, 4, -8, 0, 0, 0, 0, 0, -4, 0, 0}, 231.789560, 0.000020, 0.020000},
	{{0, 4, -8, 0, 0, 0, 0, 0, -4, 2, 0}, 51.789560, 0.000020, 0.041000},
	{{0, 4, -8, 0, 0, 0, 0, 0, -2, 0, 0}, 48.022290, 0.000030, 0.038000},
	{{0, 4, -8, 0, 0, 0, 0, 0, -2, 1, 0}, 107.517630, 0.000040, 0.077000},
	{{0, 4, -8, 0, 0, 0, 0, 0, -2, 2, 0}, 60.479620, 0.000010, 3.622000},
	{{0, 4, -8, 0, 0, 0, 0, 0, -2, 3, 0}, 133.872100, 0.000010, 0.074000},
	{{0, 4, -8, 0, 0, 0, 0, 0, -1, 0, 1}, 326.221910, 0.000010, 2.252000},
	{{0, 4, -8, 0, 0, 0, 0, 0, 0, -1, 2}, 51.789560, 0.000030, 0.083000},
	{{0, 4, -8, 0, 0, 0, 0, 0, 0, 0, 0}, 294.860960, 0.000070, 0.668000},
	{{0, 4, -8, 0, 0, 0, 0, 0, 2, 0, 0}, 177.766370, 0.000020, 0.043000},
	{{0, 4, -8, 0, 1, 0, 0, 0, -2, 0, 0}, 41.602620, 0.000030, 0.038000},
	{{0, 4, -8, 0, 1, 0, 0, 0, -2, 1, 0}, 180.300310, 0.000100, 0.078000},
	{{0, 4, -8, 0, 1, 0, 0, 0, 0, -1, 0}, 38.683530, 0.000010, 0.068000},
	{{0, 4, -8, 0, 1, 0, 0, 0, 0, 1, 0}, 38.683530, 0.000010, 0.084000},
	{{0, 4, -8, 2, 0, 0, 0, 0, 0, 0, 0}, 341.892980, 0.000010, 2.301000},
	{{0, 4, -8, 3, 0, 0, 0, 0, 0, 0, 0}, 226.963150, 0.000020, 10.294000},
	{{0, 4, -7, -3, 0, 0, 0, 0, -2, 0, 0}, 355.311150, 0.000010, 0.037000},
	{{0, 4, -7, -3, 0, 0, 0, 0, -2, 0, 2}, 103.202370, 0.000010, 68.630000},
	{{0, 4, -7, -3, 0, 0, 0, 0, -2, 2, 0}, 355.311150, 0.000010, 3.135000},
	{{0, 4, -7, -2, 0, 0, 0, 0, -2, 0, 0}, 11.228990, 0.000010, 0.038000},
	{{0, 4, -7, 0, -1, 0, 0, 0, -2, 0, 0}, 235.711930, 0.000010, 0.039000},
	{{0, 4, -7, 0, 0, 0, 0, 0, -2, -2, 0}, 132.720330, 0.000020, 0.019000},
	{{0, 4, -7, 0, 0, 0, 0, 0, -2, 0, 0}, 66.788810, 0.000040, 0.040000},
	{{0, 4, -7, 0, 0, 0, 0, 0, -2, 1, 0}, 57.678080, 0.000100, 0.083000},
	{{0, 4, -7, 0, 0, 0, 0, 0, -2, 2, 0}, 319.845380, 0.000010, 0.784000},
	{{0, 4, -7, 0, 0, 0, 0, 0, -1, 0, 0}, 244.811520, 0.000030, 0.078000},
	{{0, 4, -7, 0, 0, 0, 0, 0, -1, 0, 1}, 63.136380, 0.000030, 1.799000},
	{{0, 4, -7, 0, 0, 0, 0, 0, -1, 1, -1}, 241.730360, 0.000010, 0.077000},
	{{0, 4, -7, 0, 0, 0, 0, 0, -1, 3, -1}, 61.730360, 0.000010, 0.074000},
	{{0, 4, -7, 0, 0, 0, 0, 0, 0, -1, 0}, 257.524880, 0.000140, 0.073000},
	{{0, 4, -7, 0, 0, 0, 0, 0, 0, 0, 0}, 241.527320, 0.000550, 2.008000},
	{{0, 4, -7, 0, 0, 0, 0, 0, 0, 1, 0}, 247.712830, 0.000100, 0.078000},
	{{0, 4, -7, 0, 0, 0, 0, 0, 1, 0, -1}, 64.433770, 0.000060, 0.644000},
	{{0, 4, -7, 0, 0, 0, 0, 0, 2, -2, 0}, 80.063420, 0.000010, 0.440000},
	{{0, 4, -7, 0, 0, 0, 0, 0, 2, -1, 0}, 89.194500, 0.000210, 0.091000},
	{{0, 4, -7, 0, 0, 0, 0, 0, 2, 0, 0}, 88.957650, 0.000110, 0.041000},
	{{0, 4, -7, 0, 0, 0, 0, 0, 2, 1, 0}, 82.943580, 0.000020, 0.027000},
	{{0, 4, -7, 0, 1, 0, 0, 0, 0, 0, 0}, 12.054160, 0.000020, 2.418000},
	{{0, 4, -7, 0, 2, 0, 0, 0, 0, 0, 0}, 72.363450, 0.000010, 3.037000},
	{{0, 4, -7, 0, 3, 0, 0, 0, 0, 0, 0}, 103.381100, 0.000020, 4.082000},
	{{0, 4, -7, 0, 6, 0, 0, 0, -2, 0, 0}, 157.579590, 0.000010, 0.040000},
	{{0, 4, -7, 0, 6, 0, 0, 0, 2, 0, 0}, 157.579590, 0.000010, 0.040000},
	{{0, 4, -7, 5, 0, 0, 0, 0, 0, 0, 0}, 238.191250, 0.000010, 0.463000},
	{{0, 4, -6, -1, 0, 0, 0, 0, 0, 0, 0}, 238.429680, 0.000010, 33.794000},
	{{0, 4, -6, 0, -2, 0, 0, 0, -2, 0, 0}, 51.837320, 0.000020, 0.041000},
	{{0, 4, -6, 0, -2, 0, 0, 0, 0, -1, 0}, 51.837320, 0.000010, 0.077000},
	{{0, 4, -6, 0, -2, 0, 0, 0, 0, 1, 0}, 51.837320, 0.000010, 0.074000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -4, 0, 0}, 344.924540, 0.000040, 0.020000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -4, 1, 0}, 344.232000, 0.000080, 0.028000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -4, 2, 0}, 349.777220, 0.000040, 0.045000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -2, -2, 0}, 345.276570, 0.000030, 0.020000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -2, -1, 0}, 346.349620, 0.000270, 0.027000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -2, 0, 0}, 345.361110, 0.003100, 0.041000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -2, 0, 2}, 346.108990, 0.000050, 0.383000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -2, 1, 0}, 345.807330, 0.004110, 0.091000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -2, 2, 0}, 342.128190, 0.000170, 0.439000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -1, -1, 1}, 354.998830, 0.000070, 0.085000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -1, 0, 0}, 162.351780, 0.000180, 0.084000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -1, 0, 1}, 354.176800, 0.000030, 0.643000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -1, 1, -1}, 168.538280, 0.000020, 0.083000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -1, 1, 1}, 24.045290, 0.000010, 0.068000},
	{{0, 4, -6, 0, 0, 0, 0, 0, -1, 3, -1}, 348.138880, 0.000010, 0.069000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, -3, 0}, 345.654900, 0.000020, 0.025000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, -2, 0}, 345.240270, 0.000390, 0.038000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, -1, 0}, 342.657130, 0.005110, 0.078000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, 0, -2}, 165.165190, 0.000010, 0.038000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, 0, -1}, 348.547160, 0.000030, 0.077000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, 0, 0}, 343.213240, 0.025870, 1.992000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, 0, 2}, 165.165190, 0.000010, 0.037000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, 1, 0}, 342.735420, 0.004390, 0.073000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, 2, 0}, 340.751180, 0.000280, 0.037000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 0, 3, 0}, 339.482210, 0.000010, 0.025000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 1, -2, 1}, 169.025930, 0.000050, 4.584000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 1, -1, -1}, 349.213360, 0.000090, 0.072000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 1, -1, 0}, 339.997330, 0.000050, 2.598000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 1, -1, 1}, 169.232070, 0.000310, 0.077000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 1, 0, -1}, 349.418390, 0.000790, 1.813000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 1, 0, 0}, 340.606950, 0.000300, 0.078000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 1, 0, 1}, 169.342300, 0.000170, 0.038000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 1, 1, -1}, 349.213360, 0.000090, 0.079000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 1, 1, 0}, 340.446120, 0.000020, 0.038000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 2, -2, 0}, 171.847250, 0.000080, 0.786000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 2, -1, 0}, 321.232190, 0.000130, 0.083000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 2, 0, -2}, 344.724660, 0.000020, 0.623000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 2, 0, 0}, 341.925920, 0.000630, 0.040000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 2, 1, 0}, 344.233450, 0.000100, 0.026000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 4, -2, 0}, 175.483300, 0.000020, 0.043000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 4, -1, 0}, 191.323370, 0.000020, 0.027000},
	{{0, 4, -6, 0, 0, 0, 0, 0, 4, 0, 0}, 165.326440, 0.000040, 0.020000},
	{{0, 4, -6, 0, 1, 0, 0, 0, 0, 0, 0}, 50.148590, 0.000020, 1.705000},
	{{0, 4, -6, 0, 1, 0, 0, 0, 2, -2, 0}, 269.543650, 0.000030, 0.842000},
	{{0, 4, -6, 0, 1, 0, 0, 0, 2, 0, 0}, 89.412780, 0.000020, 0.039000},
	{{0, 4, -6, 0, 2, 0, 0, 0, -2, 0, 0}, 179.871380, 0.000050, 0.042000},
	{{0, 4, -6, 0, 2, 0, 0, 0, 0, -1, 0}, 0.312910, 0.000020, 0.079000},
	{{0, 4, -6, 0, 2, 0, 0, 0, 0, 0, 0}, 177.042010, 0.000110, 1.491000},
	{{0, 4, -6, 0, 2, 0, 0, 0, 0, 1, 0}, 0.312910, 0.000020, 0.072000},
	{{0, 4, -6, 0, 2, 0, 0, 0, 2, -3, 0}, 359.967570, 0.000010, 0.070000},
	{{0, 4, -6, 0, 2, 0, 0, 0, 2, -2, 0}, 359.967560, 0.000010, 0.906000},
	{{0, 4, -6, 0, 2, 0, 0, 0, 2, 0, 0}, 359.967560, 0.000010, 0.039000},
	{{0, 4, -6, 0, 3, 0, 0, 0, 0, 0, 0}, 169.456380, 0.000020, 1.325000},
	{{0, 4, -6, 1, 0, 0, 0, 0, 1, 0, -1}, 75.185240, 0.000020, 50.030000},
	{{0, 4, -6, 2, 0, 0, 0, 0, 0, -1, 0}, 180.097950, 0.000010, 0.086000},
	{{0, 4, -6, 2, 0, 0, 0, 0, 0, 1, 0}, 180.097950, 0.000010, 0.067000},
	{{0, 4, -6, 2, 0, 0, 0, 0, 2, -3, 0}, 128.672020, 0.000030, 0.074000},
	{{0, 4, -6, 2, 0, 0, 0, 0, 2, -1, 0}, 128.672020, 0.000030, 0.077000},
	{{0, 4, -6, 3, 0, 0, 0, 0, 2, 0, -2}, 302.564920, 0.000010, 96.289000},
	{{0, 4, -5, -3, 0, 0, 0, 0, 0, -1, 0}, 327.978230, 0.000010, 0.075000},
	{{0, 4, -5, -3, 0, 0, 0, 0, 0, 0, 0}, 162.574090, 0.000010, 10.758000},
	{{0, 4, -5, -3, 0, 0, 0, 0, 0, 1, 0}, 327.978230, 0.000010, 0.076000},
	{{0, 4, -5, 0, -1, 0, 0, 0, -2, 0, 0}, 103.965320, 0.000010, 0.043000},
	{{0, 4, -5, 0, -1, 0, 0, 0, -2, 2, 0}, 283.965320, 0.000010, 0.313000},
	{{0, 4, -5, 0, -1, 0, 0, 0, 2, -1, 0}, 98.189880, 0.000040, 0.078000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -4, 0, 0}, 89.813490, 0.000030, 0.021000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -4, 1, 0}, 93.553260, 0.000090, 0.029000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -4, 2, 0}, 92.965380, 0.000060, 0.047000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -3, 0, 1}, 76.543210, 0.000020, 0.045000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -3, 1, 1}, 76.542860, 0.000020, 0.112000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -2, -2, 0}, 94.109350, 0.000030, 0.020000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -2, -1, 0}, 94.183360, 0.000200, 0.027000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -2, 0, 0}, 94.018690, 0.003390, 0.043000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -2, 0, 2}, 93.292490, 0.000020, 0.277000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -2, 1, 0}, 93.332890, 0.004100, 0.100000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -2, 2, 0}, 94.907520, 0.000120, 0.305000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -2, 3, 0}, 270.748460, 0.000030, 0.060000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, -2, 1}, 76.521440, 0.000010, 0.042000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, -1, -1}, 86.338340, 0.000030, 0.027000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, -1, 1}, 76.466890, 0.000040, 0.093000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, 0, -1}, 76.527620, 0.000360, 0.041000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, 0, 0}, 269.871250, 0.000060, 0.092000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, 0, 1}, 256.711540, 0.000830, 0.391000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, 1, -1}, 78.639360, 0.000160, 0.091000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, 1, 1}, 256.537930, 0.000090, 0.063000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, 2, -1}, 256.361330, 0.000020, 0.450000},
	{{0, 4, -5, 0, 0, 0, 0, 0, -1, 2, 0}, 268.088700, 0.000010, 0.064000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 0, -3, 0}, 272.525790, 0.000030, 0.026000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 0, -2, 0}, 271.910370, 0.000340, 0.040000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 0, -1, 0}, 270.661530, 0.005410, 0.085000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 0, 0, -1}, 256.450970, 0.000080, 0.084000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 0, 0, 0}, 130.839920, 0.000440, 0.666000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 0, 0, 1}, 76.368930, 0.000030, 0.067000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 0, 1, 0}, 262.237940, 0.000210, 0.068000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 0, 2, 0}, 279.761950, 0.000010, 0.036000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, -2, -1}, 76.513430, 0.000090, 0.038000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, -2, 0}, 278.418390, 0.000010, 0.079000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, -2, 1}, 251.712250, 0.000010, 1.279000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, -1, -1}, 76.513460, 0.001350, 0.078000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, -1, 0}, 89.427270, 0.000080, 1.626000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, -1, 1}, 256.445580, 0.000790, 0.071000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, 0, -3}, 257.156320, 0.000010, 0.038000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, 0, -1}, 76.510910, 0.009870, 2.231000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, 0, 0}, 88.050330, 0.000290, 0.072000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, 0, 1}, 255.846930, 0.001180, 0.037000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, 1, -1}, 76.512820, 0.001260, 0.073000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, 1, 0}, 88.596210, 0.000020, 0.037000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, 1, 1}, 256.096240, 0.000130, 0.025000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 1, 2, -1}, 76.513500, 0.000090, 0.037000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, -4, 0}, 275.492100, 0.000010, 0.037000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, -3, 0}, 299.972970, 0.000030, 0.074000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, -2, 0}, 90.857910, 0.002690, 3.678000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, -1, -2}, 91.516500, 0.000040, 0.072000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, -1, -1}, 76.465680, 0.000030, 2.280000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, -1, 0}, 91.399170, 0.029520, 0.077000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, -1, 2}, 271.592190, 0.000060, 0.025000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, 0, -2}, 92.393260, 0.000250, 1.652000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, 0, -1}, 76.463130, 0.000170, 0.078000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, 0, 0}, 92.152560, 0.004110, 0.038000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, 1, 0}, 92.833620, 0.000400, 0.025000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 2, 2, 0}, 92.228520, 0.000040, 0.019000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 3, -2, -1}, 256.515270, 0.000010, 0.754000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 3, -1, -1}, 256.516020, 0.000130, 0.084000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 3, -1, 1}, 256.513170, 0.000030, 0.026000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 3, 0, -1}, 76.514260, 0.000080, 0.040000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 3, 0, 0}, 92.480610, 0.000020, 0.026000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 3, 0, 1}, 256.513190, 0.000020, 0.019000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 3, 1, -1}, 76.510840, 0.000010, 0.026000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 4, -3, 0}, 93.053240, 0.000060, 0.089000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 4, -2, 0}, 89.662410, 0.000380, 0.041000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 4, -1, 0}, 89.114850, 0.000250, 0.027000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 4, 0, 0}, 96.132630, 0.000110, 0.020000},
	{{0, 4, -5, 0, 0, 0, 0, 0, 4, 1, 0}, 92.646790, 0.000010, 0.016000},
	{{0, 4, -5, 0, 0, 1, 0, 0, 2, -1, 0}, 359.716910, 0.000050, 0.077000},
	{{0, 4, -5, 0, 0, 2, 0, 0, 2, -3, 0}, 261.381640, 0.000010, 0.074000},
	{{0, 4, -5, 0, 0, 2, 0, 0, 2, -1, 0}, 81.381640, 0.000010, 0.077000},
	{{0, 4, -5, 0, 1, 0, 0, 0, -2, 0, 0}, 189.195210, 0.000100, 0.043000},
	{{0, 4, -5, 0, 1, 0, 0, 0, -2, 1, 0}, 188.103640, 0.000010, 0.101000},
	{{0, 4, -5, 0, 1, 0, 0, 0, 0, -1, 0}, 178.138180, 0.000040, 0.086000},
	{{0, 4, -5, 0, 1, 0, 0, 0, 0, 0, 0}, 171.360380, 0.000030, 0.630000},
	{{0, 4, -5, 0, 1, 0, 0, 0, 0, 1, 0}, 178.009470, 0.000030, 0.067000},
	{{0, 4, -5, 0, 1, 0, 0, 0, 2, -3, 0}, 359.769120, 0.000030, 0.074000},
	{{0, 4, -5, 0, 1, 0, 0, 0, 2, -1, 0}, 89.765510, 0.000050, 0.077000},
	{{0, 4, -5, 0, 1, 0, 0, 0, 2, 0, 0}, 178.713080, 0.000010, 0.038000},
	{{0, 4, -5, 0, 1, 0, 0, 0, 2, 2, 0}, 358.713080, 0.000010, 0.019000},
	{{0, 4, -5, 0, 2, 0, 0, 0, 0, 0, 0}, 62.741920, 0.000020, 0.599000},
	{{0, 4, -5, 0, 2, 0, 0, 0, 2, -3, 0}, 229.894150, 0.000020, 0.075000},
	{{0, 4, -5, 0, 2, 0, 0, 0, 2, -2, 0}, 240.316860, 0.000010, 9.679000},
	{{0, 4, -5, 0, 2, 0, 0, 0, 2, -1, 0}, 49.894150, 0.000020, 0.076000},
	{{0, 4, -5, 0, 2, 0, 0, 0, 2, 0, 0}, 60.323310, 0.000040, 0.038000},
	{{0, 4, -5, 0, 3, 0, 0, 0, 2, -3, 0}, 91.255350, 0.000010, 0.075000},
	{{0, 4, -5, 0, 3, 0, 0, 0, 2, -2, 0}, 38.073200, 0.000020, 52.601000},
	{{0, 4, -5, 0, 3, 0, 0, 0, 2, -1, 0}, 271.255350, 0.000010, 0.076000},
	{{0, 4, -5, 2, 0, 0, 0, 0, 2, -3, 0}, 224.657630, 0.000040, 0.080000},
	{{0, 4, -5, 2, 0, 0, 0, 0, 2, -1, 0}, 224.657630, 0.000040, 0.071000},
	{{0, 4, -5, 3, 0, 0, 0, 0, -2, 0, 0}, 236.427210, 0.000020, 0.046000},
	{{0, 4, -5, 3, 0, 0, 0, 0, -2, 2, 0}, 236.427210, 0.000020, 0.205000},
	{{0, 4, -4, -8, 0, 0, 0, 0, -2, 2, 0}, 295.867600, 0.000020, 44.178000},
	{{0, 4, -4, 0, -2, 0, 0, 0, -2, 0, 0}, 205.438020, 0.000010, 0.045000},
	{{0, 4, -4, 0, -1, 0, 0, 0, -2, 0, 0}, 190.216320, 0.000050, 0.045000},
	{{0, 4, -4, 0, -1, 0, 0, 0, -2, 1, 0}, 13.054830, 0.000030, 0.110000},
	{{0, 4, -4, 0, -1, 0, 0, 0, -2, 2, 0}, 185.514980, 0.000010, 0.239000},
	{{0, 4, -4, 0, -1, 0, 0, 0, 0, -2, 0}, 193.054830, 0.000010, 0.042000},
	{{0, 4, -4, 0, -1, 0, 0, 0, 0, 0, 0}, 51.536230, 0.000010, 0.414000},
	{{0, 4, -4, 0, -1, 0, 0, 0, 2, -2, 0}, 14.079310, 0.000010, 1.553000},
	{{0, 4, -4, 0, -1, 0, 0, 0, 2, -1, 0}, 12.648580, 0.000060, 0.072000},
	{{0, 4, -4, 0, -1, 0, 0, 0, 2, 0, 0}, 194.079310, 0.000010, 0.037000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -4, 0, 0}, 0.084150, 0.000050, 0.021000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -4, 1, 0}, 359.746810, 0.000090, 0.030000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -4, 2, 0}, 359.853600, 0.000060, 0.049000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -3, 0, 0}, 357.748990, 0.000010, 0.029000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -2, -2, 0}, 359.522720, 0.000080, 0.021000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -2, -1, 0}, 358.924680, 0.000370, 0.028000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -2, 0, 0}, 359.152100, 0.004690, 0.045000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -2, 0, 2}, 0.739470, 0.000030, 0.217000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -2, 1, 0}, 359.208430, 0.004390, 0.111000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -2, 2, 0}, 359.089540, 0.000330, 0.234000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -2, 3, 0}, 358.924070, 0.000050, 0.057000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -1, -1, 0}, 180.039120, 0.000020, 0.043000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -1, 0, -1}, 255.419440, 0.000010, 0.043000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -1, 0, 0}, 1.404560, 0.000140, 0.101000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -1, 0, 1}, 15.510970, 0.000050, 0.281000},
	{{0, 4, -4, 0, 0, 0, 0, 0, -1, 1, 0}, 359.939320, 0.000040, 0.295000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, -3, 0}, 0.791260, 0.000030, 0.027000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, -2, 0}, 0.574600, 0.000570, 0.042000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, -1, 0}, 0.255430, 0.008820, 0.093000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, -1, 2}, 180.050050, 0.000020, 0.062000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, 0, -2}, 178.792300, 0.000100, 0.041000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, 0, 0}, 180.070810, 0.046260, 0.400000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, 0, 2}, 358.726170, 0.000070, 0.034000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, 1, -2}, 180.090620, 0.000170, 0.090000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, 1, 0}, 180.076540, 0.007460, 0.063000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, 1, 2}, 0.093670, 0.000030, 0.023000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, 2, 0}, 180.009930, 0.000560, 0.034000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 0, 3, 0}, 179.809780, 0.000040, 0.024000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 1, -2, 0}, 0.947930, 0.000100, 0.086000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 1, -1, -1}, 189.951830, 0.000040, 0.085000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 1, -1, 0}, 0.677010, 0.000120, 0.619000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 1, -1, 1}, 2.668340, 0.000020, 0.067000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 1, 0, -1}, 190.909120, 0.000370, 0.690000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 1, 0, 0}, 0.102280, 0.001110, 0.067000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 1, 0, 1}, 10.943720, 0.000020, 0.035000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 1, 1, -1}, 189.951830, 0.000040, 0.068000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 1, 1, 0}, 0.139120, 0.000060, 0.036000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, -3, 0}, 180.607570, 0.000140, 0.080000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, -2, -2}, 179.703680, 0.000010, 0.038000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, -2, 0}, 179.883310, 0.001900, 1.374000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, -2, 2}, 0.027460, 0.000010, 0.036000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, -1, -2}, 179.692250, 0.000170, 0.078000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, -1, 0}, 180.009790, 0.082350, 0.072000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, -1, 2}, 0.030580, 0.000160, 0.024000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, 0, -2}, 180.016820, 0.003920, 2.535000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, 0, 0}, 0.095830, 0.009090, 0.037000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, 0, 2}, 180.044910, 0.000010, 0.018000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, 1, 0}, 0.081950, 0.000800, 0.025000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 2, 2, 0}, 359.892010, 0.000090, 0.019000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 3, -3, -1}, 176.996850, 0.000010, 0.074000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 3, -3, 0}, 180.022810, 0.000050, 6.292000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 3, -2, 0}, 180.107750, 0.000390, 0.076000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 3, -1, -1}, 356.996850, 0.000010, 0.077000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 3, 0, -2}, 180.064520, 0.000040, 0.078000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 3, 0, 0}, 0.068820, 0.000040, 0.025000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 3, 1, 0}, 0.064550, 0.000010, 0.019000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 4, -3, 0}, 178.039620, 0.000010, 0.082000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 4, -2, 0}, 180.044430, 0.001110, 0.039000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 4, -1, -2}, 180.291560, 0.000040, 0.084000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 4, -1, 0}, 180.093930, 0.000530, 0.026000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 4, 0, -2}, 0.081770, 0.000020, 0.040000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 4, 0, 0}, 359.940880, 0.000040, 0.019000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 4, 1, 0}, 0.229600, 0.000050, 0.015000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 6, -3, 0}, 180.017410, 0.000020, 0.027000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 6, -2, 0}, 180.023850, 0.000020, 0.020000},
	{{0, 4, -4, 0, 0, 0, 0, 0, 6, -1, 0}, 0.038610, 0.000020, 0.016000},
	{{0, 4, -4, 0, 1, 0, 0, 0, -2, 0, 0}, 345.497240, 0.000050, 0.045000},
	{{0, 4, -4, 0, 1, 0, 0, 0, -2, 1, 0}, 171.264160, 0.000030, 0.112000},
	{{0, 4, -4, 0, 1, 0, 0, 0, 0, 0, 0}, 171.264160, 0.000020, 0.387000},
	{{0, 4, -4, 0, 1, 0, 0, 0, 2, -3, 0}, 168.802030, 0.000020, 0.080000},
	{{0, 4, -4, 0, 1, 0, 0, 0, 2, -1, 0}, 140.447580, 0.000060, 0.071000},
	{{0, 4, -3, -4, 0, 0, 0, 0, 0, 0, 0}, 30.446650, 0.000010, 0.727000},
	{{0, 4, -3, -3, 0, 0, 0, 0, 2, -3, 0}, 304.593590, 0.000010, 0.076000},
	{{0, 4, -3, -3, 0, 0, 0, 0, 2, -2, 0}, 303.526010, 0.000040, 7.518000},
	{{0, 4, -3, -3, 0, 0, 0, 0, 2, -1, 0}, 304.593590, 0.000010, 0.075000},
	{{0, 4, -3, -3, 0, 0, 0, 0, 2, 0, 0}, 303.563430, 0.000010, 0.038000},
	{{0, 4, -3, -1, 0, 0, 0, 0, 0, 0, 0}, 180.434560, 0.000020, 0.337000},
	{{0, 4, -3, 0, -2, 0, 0, 0, -2, 0, 0}, 78.344220, 0.000010, 0.047000},
	{{0, 4, -3, 0, -2, 0, 0, 0, -2, 1, 0}, 119.790240, 0.000010, 0.123000},
	{{0, 4, -3, 0, -2, 0, 0, 0, -2, 2, 0}, 88.127060, 0.000030, 0.196000},
	{{0, 4, -3, 0, -1, 0, 0, 0, -2, 0, 0}, 178.722420, 0.000010, 0.047000},
	{{0, 4, -3, 0, -1, 0, 0, 0, -2, 2, 0}, 178.722420, 0.000010, 0.193000},
	{{0, 4, -3, 0, -1, 0, 0, 0, 0, -2, 0}, 181.706490, 0.000010, 0.043000},
	{{0, 4, -3, 0, -1, 0, 0, 0, 0, 0, 0}, 181.094470, 0.000050, 0.293000},
	{{0, 4, -3, 0, -1, 0, 0, 0, 0, 2, 0}, 181.138910, 0.000020, 0.033000},
	{{0, 4, -3, 0, -1, 0, 0, 0, 2, -3, 0}, 3.964210, 0.000020, 0.086000},
	{{0, 4, -3, 0, -1, 0, 0, 0, 2, -1, 0}, 183.964210, 0.000020, 0.067000},
	{{0, 4, -3, 0, -1, 0, 0, 0, 2, 0, 0}, 3.964980, 0.000070, 0.036000},
	{{0, 4, -3, 0, 0, -1, 0, 0, -2, 0, 0}, 180.214190, 0.000010, 0.047000},
	{{0, 4, -3, 0, 0, -1, 0, 0, -2, 2, 0}, 0.214190, 0.000010, 0.191000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -4, 0, 0}, 77.049440, 0.000020, 0.022000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -4, 1, 0}, 252.255740, 0.000010, 0.031000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -3, 0, 1}, 103.298940, 0.000020, 0.050000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -3, 1, 1}, 103.313720, 0.000010, 0.144000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -2, -1, 0}, 253.418380, 0.000060, 0.029000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -2, 0, 0}, 251.393080, 0.000420, 0.047000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -2, 1, 0}, 253.263290, 0.000440, 0.125000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -2, 2, 0}, 273.914900, 0.000040, 0.190000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -2, 3, 0}, 86.933750, 0.000010, 0.054000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -1, -1, -1}, 102.955310, 0.000030, 0.028000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -1, -1, 1}, 283.646780, 0.000070, 0.115000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -1, 0, -1}, 102.907430, 0.000330, 0.045000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -1, 0, 1}, 283.311010, 0.000590, 0.220000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -1, 1, -1}, 103.299640, 0.000170, 0.111000},
	{{0, 4, -3, 0, 0, 0, 0, 0, -1, 1, 1}, 283.310550, 0.000050, 0.056000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 0, -2, 0}, 257.190220, 0.000060, 0.043000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 0, -1, 0}, 254.501730, 0.000600, 0.103000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 0, 0, -1}, 283.305110, 0.000030, 0.101000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 0, 0, 0}, 101.980350, 0.000480, 0.286000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 0, 0, 1}, 103.316370, 0.000020, 0.059000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 0, 1, 0}, 81.088600, 0.000100, 0.060000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 1, -1, -1}, 103.320500, 0.000290, 0.093000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 1, -1, 1}, 283.327760, 0.000170, 0.062000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 1, 0, -1}, 103.342360, 0.001960, 0.408000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 1, 0, 1}, 283.327450, 0.000130, 0.034000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 1, 1, -1}, 103.322320, 0.000240, 0.064000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 1, 1, 1}, 283.330090, 0.000020, 0.024000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 1, 2, -1}, 103.319210, 0.000010, 0.035000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 2, -3, 0}, 80.979470, 0.000030, 0.087000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 2, -2, 0}, 258.846140, 0.000060, 0.579000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 2, -1, 0}, 259.394180, 0.000810, 0.067000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 2, 0, -2}, 259.840550, 0.000030, 0.717000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 2, 0, -1}, 283.346340, 0.000050, 0.067000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 2, 0, 0}, 80.722650, 0.000250, 0.035000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 2, 1, 0}, 76.587250, 0.000060, 0.024000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 3, -2, -1}, 103.318170, 0.000010, 1.483000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 3, -1, -1}, 103.324930, 0.000130, 0.072000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 3, 0, -1}, 103.328240, 0.000030, 0.037000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 4, -1, 0}, 85.378530, 0.000010, 0.025000},
	{{0, 4, -3, 0, 0, 0, 0, 0, 4, 0, 0}, 95.414540, 0.000010, 0.019000},
	{{0, 4, -3, 0, 1, 0, 0, 0, 0, 0, 0}, 16.005140, 0.000010, 0.279000},
	{{0, 4, -2, -4, 0, 0, 0, 0, 0, 0, 0}, 269.383220, 0.000010, 0.421000},
	{{0, 4, -2, -4, 0, 0, 0, 0, 2, -3, 0}, 302.016440, 0.000020, 0.079000},
	{{0, 4, -2, -4, 0, 0, 0, 0, 2, -1, 0}, 302.016440, 0.000020, 0.072000},
	{{0, 4, -2, -2, 0, 0, 0, 0, -2, 0, 0}, 0.654840, 0.000010, 0.047000},
	{{0, 4, -2, -2, 0, 0, 0, 0, -2, 2, 0}, 0.654840, 0.000010, 0.192000},
	{{0, 4, -2, -2, 0, 0, 0, 0, 0, -2, 0}, 179.892700, 0.000010, 0.043000},
	{{0, 4, -2, -2, 0, 0, 0, 0, 0, 0, 0}, 179.892700, 0.000010, 0.291000},
	{{0, 4, -2, 0, -3, 0, 0, 0, 0, 0, 0}, 189.692870, 0.000010, 0.235000},
	{{0, 4, -2, 0, -2, 0, 0, 0, -2, -2, 0}, 0.104260, 0.000010, 0.021000},
	{{0, 4, -2, 0, -2, 0, 0, 0, -2, 2, 0}, 0.310920, 0.000010, 0.164000},
	{{0, 4, -2, 0, -2, 0, 0, 0, 0, 0, 0}, 180.065910, 0.000020, 0.231000},
	{{0, 4, -2, 0, -1, 0, 0, 0, -2, 0, 0}, 90.165280, 0.000030, 0.049000},
	{{0, 4, -2, 0, -1, 0, 0, 0, -2, 2, 0}, 90.165280, 0.000030, 0.162000},
	{{0, 4, -2, 0, -1, 0, 0, 0, 0, 0, 0}, 347.768660, 0.000010, 0.226000},
	{{0, 4, -2, 0, -1, 0, 0, 0, 0, 2, 0}, 347.768660, 0.000010, 0.032000},
	{{0, 4, -2, 0, 0, 0, 0, 0, -2, -1, 0}, 30.764000, 0.000030, 0.030000},
	{{0, 4, -2, 0, 0, 0, 0, 0, -2, 0, 0}, 31.881950, 0.000290, 0.049000},
	{{0, 4, -2, 0, 0, 0, 0, 0, -2, 1, 0}, 29.685540, 0.000190, 0.143000},
	{{0, 4, -2, 0, 0, 0, 0, 0, -2, 2, 0}, 57.827610, 0.000010, 0.159000},
	{{0, 4, -2, 0, 0, 0, 0, 0, -1, 0, 0}, 207.579420, 0.000030, 0.127000},
	{{0, 4, -2, 0, 0, 0, 0, 0, -1, 0, 1}, 189.512360, 0.000020, 0.180000},
	{{0, 4, -2, 0, 0, 0, 0, 0, 0, -2, 0}, 340.906760, 0.000020, 0.045000},
	{{0, 4, -2, 0, 0, 0, 0, 0, 0, -1, 0}, 33.168000, 0.000190, 0.114000},
	{{0, 4, -2, 0, 0, 0, 0, 0, 0, 0, 0}, 25.922180, 0.001470, 0.222000},
	{{0, 4, -2, 0, 0, 0, 0, 0, 0, 1, 0}, 26.019360, 0.000150, 0.056000},
	{{0, 4, -2, 0, 0, 0, 0, 0, 1, 0, -1}, 11.209630, 0.000030, 0.290000},
	{{0, 4, -2, 0, 0, 0, 0, 0, 1, 0, 0}, 206.087290, 0.000020, 0.059000},
	{{0, 4, -2, 0, 0, 0, 0, 0, 2, -2, 0}, 26.224890, 0.000020, 0.367000},
	{{0, 4, -2, 0, 0, 0, 0, 0, 2, -1, 0}, 26.126900, 0.000240, 0.063000},
	{{0, 4, -2, 0, 0, 0, 0, 0, 2, 0, 0}, 206.455400, 0.000100, 0.034000},
	{{0, 4, -1, -7, 0, 0, 0, 0, 2, -3, 0}, 2.922890, 0.000010, 0.075000},
	{{0, 4, -1, -7, 0, 0, 0, 0, 2, -2, 0}, 354.175400, 0.000030, 159.092000},
	{{0, 4, -1, -7, 0, 0, 0, 0, 2, -1, 0}, 2.922890, 0.000010, 0.075000},
	{{0, 4, -1, -5, 0, 0, 0, 0, 0, 0, 0}, 121.799390, 0.000010, 0.352000},
	{{0, 4, -1, 0, -3, 0, 0, 0, -2, 0, 0}, 356.691050, 0.000010, 0.051000},
	{{0, 4, -1, 0, -3, 0, 0, 0, -2, 2, 0}, 176.691050, 0.000010, 0.142000},
	{{0, 4, -1, 0, 0, 0, 0, 0, -2, 0, 0}, 288.334380, 0.000050, 0.052000},
	{{0, 4, -1, 0, 0, 0, 0, 0, -2, 1, 0}, 289.272010, 0.000020, 0.167000},
	{{0, 4, -1, 0, 0, 0, 0, 0, 0, 0, 0}, 288.808190, 0.000080, 0.182000},
	{{0, 4, -1, 0, 0, 0, 0, 0, 1, 0, -1}, 307.798360, 0.000020, 0.225000},
	{{0, 4, -1, 0, 0, 0, 0, 0, 2, -2, 0}, 37.359960, 0.000030, 0.268000},
	{{0, 4, -1, 0, 0, 0, 0, 0, 2, -1, 0}, 293.327340, 0.000020, 0.059000},
	{{0, 4, -1, 0, 0, 0, 0, 0, 2, 0, 0}, 72.708580, 0.000020, 0.033000},
	{{0, 4, -1, 0, 1, 0, 0, 0, 2, 0, 0}, 132.618900, 0.000050, 0.033000},
	{{0, 4, 0, 0, -2, 0, 0, 0, 2, -2, 0}, 153.690310, 0.000010, 0.219000},
	{{0, 4, 0, 0, -2, 0, 0, 0, 2, 0, 0}, 153.690310, 0.000010, 0.032000},
	{{0, 4, 0, 0, -1, 0, 0, 0, 2, -2, 0}, 142.380520, 0.000020, 0.215000},
	{{0, 4, 0, 0, -1, 0, 0, 0, 2, -1, 0}, 141.307110, 0.000020, 0.056000},
	{{0, 4, 0, 0, -1, 0, 0, 0, 2, 0, 0}, 149.626970, 0.000040, 0.032000},
	{{0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0}, 269.033620, 0.000040, 0.154000},
	{{0, 4, 0, 0, 0, 0, 0, 0, 0, 1, 0}, 128.137580, 0.000010, 0.051000},
	{{0, 4, 0, 0, 0, 0, 0, 0, 0, 2, 0}, 308.152820, 0.000040, 0.030000},
	{{0, 4, 0, 0, 0, 0, 0, 0, 0, 3, 0}, 128.137580, 0.000010, 0.022000},
	{{0, 4, 0, 0, 0, 0, 0, 0, 2, -3, 0}, 308.140050, 0.000010, 0.117000},
	{{0, 4, 0, 0, 0, 0, 0, 0, 2, -2, 0}, 127.988520, 0.000050, 0.212000},
	{{0, 4, 0, 0, 0, 0, 0, 0, 2, 0, 0}, 129.663690, 0.000030, 0.032000},
	{{0, 4, 0, 0, 0, 0, 0, 0, 2, 1, 0}, 308.137580, 0.000020, 0.022000},
	{{0, 4, 0, 0, 1, 0, 0, 0, 2, -2, 0}, 299.161760, 0.000010, 0.208000},
	{{0, 4, 0, 0, 1, 0, 0, 0, 2, -1, 0}, 299.516440, 0.000030, 0.055000},
	{{0, 4, 0, 0, 1, 0, 0, 0, 2, 0, 0}, 296.644260, 0.000030, 0.032000},
	{{0, 4, 1, 0, -2, 0, 0, 0, 2, -2, 0}, 225.021010, 0.000010, 0.180000},
	{{0, 4, 1, 0, -2, 0, 0, 0, 2, 0, 0}, 45.021010, 0.000010, 0.031000},
	{{0, 4, 1, 0, -1, 0, 0, 0, 2, -3, 0}, 129.245440, 0.000020, 0.131000},
	{{0, 4, 1, 0, -1, 0, 0, 0, 2, -1, 0}, 129.245440, 0.000020, 0.053000},
	{{0, 4, 1, 0, -1, 0, 0, 0, 2, 0, 0}, 128.621940, 0.000040, 0.031000},
	{{0, 4, 1, 0, -1, 0, 0, 0, 2, 2, 0}, 129.529850, 0.000010, 0.017000},
	{{0, 4, 1, 0, 0, 0, 0, 0, -2, 0, 0}, 302.062740, 0.000040, 0.058000},
	{{0, 4, 1, 0, 0, 0, 0, 0, -2, 2, 0}, 302.066680, 0.000050, 0.108000},
	{{0, 4, 1, 0, 0, 0, 0, 0, 0, 0, 0}, 59.129410, 0.000020, 0.133000},
	{{0, 4, 1, 0, 0, 0, 0, 0, 2, -3, 0}, 217.389220, 0.000010, 0.133000},
	{{0, 4, 1, 0, 0, 0, 0, 0, 2, -2, 0}, 218.707380, 0.000020, 0.175000},
	{{0, 4, 1, 0, 0, 0, 0, 0, 2, -1, 0}, 217.389220, 0.000010, 0.053000},
	{{0, 4, 1, 0, 0, 0, 0, 0, 2, 0, 0}, 218.707380, 0.000020, 0.031000},
	{{0, 4, 2, -2, 0, 0, 0, 0, 2, -2, 0}, 308.088150, 0.000010, 0.177000},
	{{0, 4, 2, -2, 0, 0, 0, 0, 2, 0, 0}, 308.088150, 0.000010, 0.031000},
	{{0, 4, 2, 0, -3, 0, 0, 0, 2, -2, 0}, 318.107820, 0.000010, 0.154000},
	{{0, 4, 2, 0, -3, 0, 0, 0, 2, 0, 0}, 318.107820, 0.000010, 0.030000},
	{{0, 4, 2, 0, -2, 0, 0, 0, 0, 0, 0}, 308.356540, 0.000010, 0.120000},
	{{0, 4, 2, 0, -2, 0, 0, 0, 0, 2, 0}, 128.356540, 0.000010, 0.029000},
	{{0, 4, 2, 0, 0, 0, 0, 0, 0, 0, 0}, 333.371040, 0.000020, 0.118000},
	{{0, 4, 2, 0, 0, 0, 0, 0, 2, -1, 0}, 153.371040, 0.000030, 0.050000},
	{{0, 4, 5, 0, 0, 0, 0, 0, -2, 1, 0}, 169.960040, 0.000020, 49.129000},
	{{0, 4, 7, 0, -3, 0, 0, 0, 0, -1, 0}, 163.134580, 0.000020, 152.083000},
	{{0, 4, 22, -32, 0, 0, 0, 0, -2, 1, 0}, 215.645940, 0.000010, 154.818000},
	{{0, 5, -14, 0, 0, 0, 0, 0, -2, 0, 0}, 289.621970, 0.000010, 0.033000},
	{{0, 5, -14, 0, 0, 0, 0, 0, -2, 2, 0}, 289.621970, 0.000010, 0.244000},
	{{0, 5, -14, 15, 0, 0, 0, 0, 2, 0, -2}, 337.847420, 0.000020, 214.986000},
	{{0, 5, -13, 6, 0, 0, 0, 0, -2, 1, 0}, 290.075130, 0.000040, 0.076000},
	{{0, 5, -13, 6, 0, 0, 0, 0, -2, 2, 0}, 110.090790, 0.000010, 10.901000},
	{{0, 5, -13, 6, 0, 0, 0, 0, -2, 3, 0}, 290.075130, 0.000040, 0.075000},
	{{0, 5, -12, 0, 2, 0, 0, 0, -2, 0, 0}, 322.734840, 0.000020, 0.035000},
	{{0, 5, -12, 4, 0, 0, 0, 0, -2, 1, 0}, 343.456500, 0.000010, 0.076000},
	{{0, 5, -12, 4, 0, 0, 0, 0, -2, 3, 0}, 343.456500, 0.000010, 0.075000},
	{{0, 5, -11, 0, 0, 0, 0, 0, -2, -1, 0}, 218.754320, 0.000020, 0.024000},
	{{0, 5, -11, 0, 0, 0, 0, 0, -2, 0, 0}, 217.499700, 0.000010, 0.036000},
	{{0, 5, -11, 0, 0, 0, 0, 0, -2, 1, 0}, 223.945770, 0.000020, 0.070000},
	{{0, 5, -11, 0, 0, 0, 0, 0, -2, 2, 0}, 217.499700, 0.000010, 0.910000},
	{{0, 5, -11, 0, 0, 0, 0, 0, 0, -1, 0}, 66.968990, 0.000020, 0.062000},
	{{0, 5, -11, 0, 0, 0, 0, 0, 0, 0, 0}, 266.750160, 0.000030, 0.348000},
	{{0, 5, -11, 0, 0, 0, 0, 0, 0, 1, 0}, 66.968990, 0.000020, 0.096000},
	{{0, 5, -11, 0, 1, 0, 0, 0, -2, 1, 0}, 141.905830, 0.000030, 0.070000},
	{{0, 5, -11, 0, 2, 0, 0, 0, -2, 0, 0}, 231.651910, 0.000010, 0.036000},
	{{0, 5, -11, 0, 2, 0, 0, 0, -2, 1, 0}, 51.651920, 0.000040, 0.070000},
	{{0, 5, -11, 0, 2, 0, 0, 0, -2, 2, 0}, 231.651910, 0.000010, 1.076000},
	{{0, 5, -11, 0, 3, 0, 0, 0, -2, 0, 0}, 221.876490, 0.000020, 0.037000},
	{{0, 5, -11, 2, 0, 0, 0, 0, -2, 0, 0}, 231.769320, 0.000040, 0.038000},
	{{0, 5, -11, 2, 0, 0, 0, 0, -2, 1, 0}, 184.696530, 0.000010, 0.075000},
	{{0, 5, -11, 2, 0, 0, 0, 0, -2, 2, 0}, 252.888150, 0.000030, 28.573000},
	{{0, 5, -11, 2, 0, 0, 0, 0, -2, 3, 0}, 184.696530, 0.000010, 0.076000},
	{{0, 5, -10, 0, -1, 0, 0, 0, -2, 1, 0}, 156.699930, 0.000010, 0.074000},
	{{0, 5, -10, 0, -1, 0, 0, 0, -2, 3, 0}, 336.699930, 0.000010, 0.076000},
	{{0, 5, -10, 0, 0, 0, 0, 0, -2, 0, 0}, 308.000220, 0.000030, 0.038000},
	{{0, 5, -10, 0, 0, 0, 0, 0, -2, 2, 0}, 301.758050, 0.000050, 10.166000},
	{{0, 5, -10, 0, 0, 0, 0, 0, -1, 0, 0}, 144.029500, 0.000020, 0.070000},
	{{0, 5, -10, 0, 0, 0, 0, 0, 0, 0, 0}, 121.178660, 0.000030, 0.534000},
	{{0, 5, -10, 0, 0, 0, 0, 0, 2, -2, 0}, 237.791400, 0.000010, 0.274000},
	{{0, 5, -10, 0, 0, 0, 0, 0, 2, -1, 0}, 294.835600, 0.000030, 0.104000},
	{{0, 5, -10, 0, 0, 0, 0, 0, 2, 0, 0}, 274.348260, 0.000030, 0.044000},
	{{0, 5, -10, 0, 0, 1, 0, 0, -2, 1, 0}, 232.355720, 0.000020, 0.075000},
	{{0, 5, -10, 0, 0, 3, 0, 0, -2, 1, 0}, 43.357670, 0.000030, 0.075000},
	{{0, 5, -10, 0, 0, 3, 0, 0, -2, 3, 0}, 43.357670, 0.000030, 0.075000},
	{{0, 5, -10, 0, 1, 0, 0, 0, -2, 0, 0}, 51.183170, 0.000010, 0.038000},
	{{0, 5, -10, 0, 1, 0, 0, 0, -2, 1, 0}, 322.758290, 0.000050, 0.075000},
	{{0, 5, -10, 0, 1, 0, 0, 0, -2, 2, 0}, 88.455840, 0.000020, 71.113000},
	{{0, 5, -10, 0, 1, 0, 0, 0, -2, 3, 0}, 265.989290, 0.000010, 0.076000},
	{{0, 5, -10, 0, 1, 0, 0, 0, 0, -1, 0}, 50.731870, 0.000020, 0.066000},
	{{0, 5, -10, 0, 1, 0, 0, 0, 0, 0, 0}, 50.454460, 0.000030, 0.559000},
	{{0, 5, -10, 0, 1, 0, 0, 0, 0, 1, 0}, 50.731870, 0.000020, 0.087000},
	{{0, 5, -10, 0, 2, 0, 0, 0, -2, 1, 0}, 199.280880, 0.000020, 0.076000},
	{{0, 5, -10, 0, 2, 0, 0, 0, -2, 2, 0}, 319.761860, 0.000020, 14.237000},
	{{0, 5, -10, 0, 3, 0, 0, 0, -2, 1, 0}, 287.224580, 0.000010, 0.076000},
	{{0, 5, -10, 0, 3, 0, 0, 0, -2, 3, 0}, 107.224580, 0.000010, 0.075000},
	{{0, 5, -10, 3, 0, 0, 0, 0, -2, 0, 0}, 288.613250, 0.000010, 0.040000},
	{{0, 5, -10, 3, 0, 0, 0, 0, -2, 2, 0}, 288.613250, 0.000010, 0.668000},
	{{0, 5, -10, 3, 0, 0, 0, 0, 0, -1, 0}, 108.613250, 0.000020, 0.074000},
	{{0, 5, -10, 3, 0, 0, 0, 0, 0, 1, 0}, 108.613250, 0.000020, 0.077000},
	{{0, 5, -10, 7, 0, 0, 0, 0, 2, -3, 0}, 57.364390, 0.000010, 0.076000},
	{{0, 5, -10, 7, 0, 0, 0, 0, 2, -1, 0}, 57.364390, 0.000010, 0.075000},
	{{0, 5, -9, -2, 0, 0, 0, 0, -2, 1, 0}, 282.067780, 0.000010, 0.075000},
	{{0, 5, -9, -2, 0, 0, 0, 0, -2, 3, 0}, 282.067780, 0.000010, 0.076000},
	{{0, 5, -9, 0, -1, 0, 0, 0, -2, 0, 0}, 242.121430, 0.000030, 0.039000},
	{{0, 5, -9, 0, -1, 0, 0, 0, -2, 1, 0}, 245.849290, 0.000010, 0.080000},
	{{0, 5, -9, 0, -1, 0, 0, 0, -2, 2, 0}, 234.935020, 0.000010, 1.224000},
	{{0, 5, -9, 0, -1, 0, 0, 0, -2, 3, 0}, 65.849290, 0.000010, 0.071000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -4, 0, 0}, 231.788110, 0.000020, 0.020000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -4, 2, 0}, 54.261620, 0.000010, 0.042000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -2, -1, 0}, 51.691480, 0.000010, 0.026000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -2, 0, 0}, 37.485690, 0.000050, 0.039000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -2, 0, 2}, 231.788110, 0.000010, 0.810000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -2, 1, 0}, 56.690650, 0.000060, 0.081000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -2, 2, 0}, 44.850950, 0.000020, 1.109000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -2, 2, 2}, 231.788110, 0.000010, 0.036000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -2, 3, 0}, 83.214330, 0.000030, 0.071000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -1, 0, 0}, 309.413720, 0.000130, 0.076000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -1, 1, 0}, 321.173710, 0.000150, 68.379000},
	{{0, 5, -9, 0, 0, 0, 0, 0, -1, 2, 0}, 232.233050, 0.000040, 0.075000},
	{{0, 5, -9, 0, 0, 0, 0, 0, 0, 0, 0}, 197.524110, 0.000170, 1.146000},
	{{0, 5, -9, 0, 0, 0, 0, 0, 0, 1, -2}, 51.788110, 0.000020, 0.069000},
	{{0, 5, -9, 0, 0, 0, 0, 0, 0, 1, 0}, 172.349700, 0.000070, 0.081000},
	{{0, 5, -9, 0, 0, 0, 0, 0, 1, 0, 0}, 128.610030, 0.000020, 0.087000},
	{{0, 5, -9, 0, 0, 0, 0, 0, 2, -1, 0}, 173.772290, 0.000220, 0.094000},
	{{0, 5, -9, 0, 0, 0, 0, 0, 2, 0, 0}, 173.572010, 0.000200, 0.042000},
	{{0, 5, -9, 0, 0, 0, 0, 0, 2, 1, 0}, 173.896120, 0.000020, 0.027000},
	{{0, 5, -9, 0, 1, 0, 0, 0, -2, 0, 0}, 41.516550, 0.000030, 0.039000},
	{{0, 5, -9, 0, 1, 0, 0, 0, -2, 1, 0}, 41.895340, 0.000020, 0.082000},
	{{0, 5, -9, 0, 1, 0, 0, 0, -2, 2, 0}, 43.669930, 0.000010, 1.014000},
	{{0, 5, -9, 0, 1, 0, 0, 0, 0, 0, 0}, 233.640200, 0.000010, 1.269000},
	{{0, 5, -9, 2, 0, 0, 0, 0, 0, 0, 0}, 254.957210, 0.000010, 5.236000},
	{{0, 5, -9, 3, 0, 0, 0, 0, 1, -2, 1}, 188.345210, 0.000070, 413.749000},
	{{0, 5, -9, 5, 0, 0, 0, 0, 2, -3, 0}, 199.057920, 0.000020, 0.076000},
	{{0, 5, -9, 5, 0, 0, 0, 0, 2, -1, 0}, 199.057920, 0.000020, 0.075000},
	{{0, 5, -8, -3, 0, 0, 0, 0, -2, 0, 0}, 355.308390, 0.000010, 0.038000},
	{{0, 5, -8, -3, 0, 0, 0, 0, -2, 2, 0}, 355.308390, 0.000010, 3.262000},
	{{0, 5, -8, -2, 0, 0, 0, 0, -2, 1, 0}, 11.005760, 0.000020, 0.081000},
	{{0, 5, -8, -2, 0, 0, 0, 0, -2, 3, 0}, 11.005760, 0.000020, 0.071000},
	{{0, 5, -8, 0, -2, 0, 0, 0, 0, -1, 0}, 18.041240, 0.000030, 0.075000},
	{{0, 5, -8, 0, -2, 0, 0, 0, 0, 1, 0}, 18.041240, 0.000030, 0.076000},
	{{0, 5, -8, 0, -1, 0, 0, 0, -2, 0, 0}, 235.672620, 0.000010, 0.040000},
	{{0, 5, -8, 0, -1, 0, 0, 0, -2, 1, 0}, 55.672620, 0.000030, 0.087000},
	{{0, 5, -8, 0, -1, 0, 0, 0, 0, -1, 0}, 147.012580, 0.000020, 0.076000},
	{{0, 5, -8, 0, -1, 0, 0, 0, 0, 1, 0}, 147.012580, 0.000020, 0.075000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -4, 0, 0}, 80.546040, 0.000030, 0.020000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -4, 1, 0}, 72.165310, 0.000060, 0.028000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -4, 2, 0}, 86.049060, 0.000050, 0.044000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -2, -2, 0}, 97.679620, 0.000030, 0.020000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -2, -1, 0}, 71.275380, 0.000220, 0.026000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -2, 0, 0}, 72.103780, 0.002310, 0.041000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -2, 0, 2}, 72.241170, 0.000050, 0.447000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -2, 1, 0}, 72.396510, 0.003370, 0.088000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -2, 2, 0}, 59.621850, 0.000110, 0.526000},
	{{0, 5, -8, 0, 0, 0, 0, 0, -1, 0, 0}, 251.681620, 0.000070, 0.082000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, -3, 0}, 66.908910, 0.000010, 0.025000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, -2, 0}, 67.593420, 0.000280, 0.038000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, -1, 0}, 69.478880, 0.004180, 0.076000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, 0, -2}, 252.376030, 0.000010, 0.037000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, 0, -1}, 243.764310, 0.000060, 0.075000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, 0, 0}, 70.235540, 0.022960, 7.836000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, 0, 2}, 252.376030, 0.000010, 0.037000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, 1, -1}, 63.752710, 0.000020, 25.549000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, 1, 0}, 69.452280, 0.004030, 0.075000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, 2, 0}, 69.298900, 0.000270, 0.038000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 0, 3, 0}, 68.823490, 0.000010, 0.025000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 1, 0, -1}, 67.688150, 0.000120, 1.080000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 1, 0, 0}, 255.044230, 0.000030, 0.080000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 1, 1, -1}, 272.367180, 0.000010, 0.081000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 2, -2, 0}, 73.525020, 0.000090, 0.607000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 2, -1, 0}, 72.298660, 0.002530, 0.086000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 2, 0, -2}, 71.636090, 0.000040, 0.505000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 2, 0, 0}, 71.421360, 0.001660, 0.040000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 2, 1, 0}, 71.031930, 0.000170, 0.026000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 2, 2, 0}, 71.630920, 0.000020, 0.019000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 3, 0, -1}, 272.204360, 0.000010, 0.042000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 4, -2, 0}, 72.856720, 0.000020, 0.043000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 4, -1, 0}, 71.870790, 0.000030, 0.028000},
	{{0, 5, -8, 0, 0, 0, 0, 0, 4, 0, 0}, 71.635220, 0.000020, 0.020000},
	{{0, 5, -8, 0, 1, 0, 0, 0, 0, 0, 0}, 199.487770, 0.000010, 4.719000},
	{{0, 5, -8, 0, 2, 0, 0, 0, 0, -1, 0}, 91.998140, 0.000030, 0.077000},
	{{0, 5, -8, 0, 2, 0, 0, 0, 0, 0, 0}, 267.140240, 0.000050, 3.376000},
	{{0, 5, -8, 0, 2, 0, 0, 0, 0, 1, 0}, 91.998140, 0.000030, 0.074000},
	{{0, 5, -8, 0, 2, 0, 0, 0, 2, -1, 0}, 270.987120, 0.000020, 0.085000},
	{{0, 5, -8, 0, 3, 0, 0, 0, 0, 0, 0}, 193.738340, 0.000040, 2.628000},
	{{0, 5, -8, 2, 0, 0, 0, 0, 0, 0, 0}, 93.113650, 0.000010, 0.840000},
	{{0, 5, -8, 3, 0, 0, 0, 0, 2, -3, 0}, 356.822160, 0.000030, 0.075000},
	{{0, 5, -8, 3, 0, 0, 0, 0, 2, -2, 0}, 175.161670, 0.000020, 19.475000},
	{{0, 5, -8, 3, 0, 0, 0, 0, 2, -1, 0}, 356.822160, 0.000030, 0.076000},
	{{0, 5, -8, 4, 0, 0, 0, 0, 0, 0, 0}, 30.592270, 0.000010, 0.444000},
	{{0, 5, -7, -4, 0, 0, 0, 0, 0, 0, 0}, 112.336110, 0.000050, 1.001000},
	{{0, 5, -7, -4, 0, 0, 0, 0, 2, -1, 0}, 112.335210, 0.000040, 0.095000},
	{{0, 5, -7, -4, 0, 0, 0, 0, 2, 0, 0}, 112.335210, 0.000040, 0.042000},
	{{0, 5, -7, -3, 0, 0, 0, 0, 0, 0, 0}, 27.053460, 0.000010, 2.139000},
	{{0, 5, -7, -2, 0, 0, 0, 0, 0, -1, 0}, 166.343590, 0.000010, 0.076000},
	{{0, 5, -7, -2, 0, 0, 0, 0, 0, 1, 0}, 166.343590, 0.000010, 0.075000},
	{{0, 5, -7, 0, -2, 0, 0, 0, -2, 0, 0}, 51.832850, 0.000010, 0.042000},
	{{0, 5, -7, 0, -2, 0, 0, 0, -2, 2, 0}, 231.832850, 0.000010, 0.366000},
	{{0, 5, -7, 0, -1, 0, 0, 0, 1, 0, -1}, 207.413410, 0.000020, 95.659000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -4, 0, 0}, 167.947740, 0.000040, 0.021000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -4, 1, 0}, 165.473180, 0.000070, 0.028000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -4, 2, 0}, 165.113670, 0.000050, 0.046000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -2, -2, 0}, 167.045780, 0.000030, 0.020000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -2, -1, 0}, 166.645750, 0.000230, 0.027000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -2, 0, 0}, 169.340990, 0.002900, 0.042000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -2, 0, 2}, 168.297700, 0.000020, 0.309000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -2, 1, 0}, 168.349970, 0.003650, 0.097000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -2, 2, 0}, 173.540940, 0.000150, 0.345000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -1, -1, -1}, 169.327030, 0.000040, 0.026000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -1, -1, 1}, 166.065230, 0.000060, 0.090000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -1, 0, -1}, 169.222250, 0.000340, 0.041000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -1, 0, 0}, 344.091040, 0.000080, 0.089000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -1, 0, 1}, 350.572980, 0.000450, 0.458000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -1, 1, -1}, 349.852230, 0.000090, 0.088000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -1, 1, 1}, 0.615550, 0.000030, 0.065000},
	{{0, 5, -7, 0, 0, 0, 0, 0, -1, 2, -1}, 349.428960, 0.000030, 0.541000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 0, -2, 0}, 158.720950, 0.000070, 0.039000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 0, -1, 0}, 170.321840, 0.001420, 0.082000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 0, 0, 0}, 170.492270, 0.008180, 0.887000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 0, 1, 0}, 169.107440, 0.001290, 0.070000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 0, 2, 0}, 175.283730, 0.000090, 0.036000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, -2, -1}, 169.357700, 0.000170, 0.038000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, -2, 0}, 350.194020, 0.000020, 0.077000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, -1, -1}, 169.370650, 0.002710, 0.076000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, -1, 0}, 155.875570, 0.000020, 4.156000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, -1, 1}, 350.544130, 0.000460, 0.073000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, 0, -3}, 350.294410, 0.000020, 0.037000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, 0, -1}, 169.267620, 0.018100, 13.541000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, 0, 0}, 154.981520, 0.000200, 0.074000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, 0, 1}, 349.307210, 0.001630, 0.037000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, 1, -1}, 169.370650, 0.002710, 0.075000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, 1, 1}, 349.282330, 0.000180, 0.025000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 1, 2, -1}, 169.370650, 0.000180, 0.038000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 2, -3, 0}, 137.604690, 0.000020, 0.072000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 2, -2, 0}, 160.982710, 0.000110, 1.547000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 2, -1, -2}, 195.249220, 0.000040, 0.070000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 2, -1, 0}, 164.299730, 0.002810, 0.079000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 2, 0, -2}, 175.203340, 0.000070, 1.021000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 2, 0, 0}, 166.825960, 0.000930, 0.039000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 2, 1, -2}, 15.249220, 0.000050, 0.081000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 2, 1, 0}, 167.059490, 0.000100, 0.026000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 3, -2, -1}, 349.430250, 0.000030, 0.588000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 3, -1, -1}, 349.574830, 0.000180, 0.087000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 3, -1, 1}, 349.282360, 0.000040, 0.026000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 3, 0, -1}, 169.266840, 0.000140, 0.040000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 3, 0, 1}, 349.282410, 0.000020, 0.019000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 3, 1, -1}, 169.327030, 0.000040, 0.026000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 4, -2, 0}, 148.051100, 0.000020, 0.042000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 4, -1, -2}, 195.309120, 0.000010, 0.095000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 4, -1, 0}, 161.094270, 0.000050, 0.027000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 4, 0, -2}, 15.232700, 0.000040, 0.042000},
	{{0, 5, -7, 0, 0, 0, 0, 0, 4, 0, 0}, 166.816010, 0.000050, 0.020000},
	{{0, 5, -7, 0, 0, 1, 0, 0, 0, 0, 0}, 268.922240, 0.000010, 0.861000},
	{{0, 5, -7, 0, 2, 0, 0, 0, -2, 0, 0}, 179.898290, 0.000020, 0.043000},
	{{0, 5, -7, 0, 2, 0, 0, 0, 0, -1, 0}, 359.859210, 0.000030, 0.084000},
	{{0, 5, -7, 0, 2, 0, 0, 0, 0, 0, 0}, 175.803980, 0.000030, 0.771000},
	{{0, 5, -7, 0, 2, 0, 0, 0, 0, 1, 0}, 359.859210, 0.000030, 0.069000},
	{{0, 5, -7, 0, 2, 0, 0, 0, 0, 2, 0}, 359.880050, 0.000010, 0.036000},
	{{0, 5, -7, 0, 2, 0, 0, 0, 2, -1, 0}, 164.686710, 0.000040, 0.078000},
	{{0, 5, -7, 0, 2, 0, 0, 0, 2, 0, 0}, 359.953620, 0.000010, 0.038000},
	{{0, 5, -7, 0, 3, 0, 0, 0, 0, 0, 0}, 169.630020, 0.000010, 0.724000},
	{{0, 5, -7, 0, 3, 0, 0, 0, 2, 0, 0}, 170.128780, 0.000030, 0.038000},
	{{0, 5, -7, 2, 0, 0, 0, 0, 2, -3, 0}, 337.239450, 0.000020, 0.078000},
	{{0, 5, -7, 2, 0, 0, 0, 0, 2, -2, 0}, 0.181320, 0.000030, 2.398000},
	{{0, 5, -7, 2, 0, 0, 0, 0, 2, -1, 0}, 337.239450, 0.000020, 0.073000},
	{{0, 5, -7, 2, 0, 0, 0, 0, 2, 0, -2}, 185.594800, 0.000020, 11.983000},
	{{0, 5, -7, 2, 0, 0, 0, 0, 2, 0, 0}, 0.181320, 0.000030, 0.037000},
	{{0, 5, -6, -4, 0, 0, 0, 0, -4, 1, 0}, 9.412150, 0.000010, 0.028000},
	{{0, 5, -6, -4, 0, 0, 0, 0, -2, -1, 0}, 9.412150, 0.000040, 0.026000},
	{{0, 5, -6, -4, 0, 0, 0, 0, -2, 0, 0}, 9.365730, 0.000320, 0.040000},
	{{0, 5, -6, -4, 0, 0, 0, 0, -2, 1, 0}, 10.184610, 0.000540, 0.087000},
	{{0, 5, -6, -4, 0, 0, 0, 0, -2, 2, 0}, 9.412150, 0.000020, 0.563000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 0, -2, 0}, 5.599850, 0.000040, 0.038000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 0, -1, 0}, 5.261160, 0.000530, 0.075000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 0, 0, 0}, 9.004970, 0.003230, 1137.638000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 0, 1, 0}, 5.255910, 0.000530, 0.075000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 0, 2, 0}, 5.600340, 0.000040, 0.038000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 2, -2, 0}, 9.400210, 0.000020, 0.564000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 2, -1, 0}, 10.172220, 0.000540, 0.087000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 2, 0, 0}, 9.354980, 0.000320, 0.040000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 2, 1, 0}, 9.400210, 0.000040, 0.026000},
	{{0, 5, -6, -4, 0, 0, 0, 0, 4, -1, 0}, 9.400210, 0.000010, 0.028000},
	{{0, 5, -6, -2, 0, 0, 0, 0, 1, 0, -1}, 79.700220, 0.000030, 95.404000},
	{{0, 5, -6, 0, -1, 0, 0, 0, -2, 0, 0}, 104.024580, 0.000010, 0.044000},
	{{0, 5, -6, 0, -1, 0, 0, 0, -2, 2, 0}, 284.024580, 0.000010, 0.262000},
	{{0, 5, -6, 0, -1, 0, 0, 0, 0, -2, 0}, 27.696450, 0.000010, 0.041000},
	{{0, 5, -6, 0, -1, 0, 0, 0, 0, 0, 0}, 207.696450, 0.000010, 0.489000},
	{{0, 5, -6, 0, -1, 0, 0, 0, 2, -3, 0}, 284.867000, 0.000020, 0.077000},
	{{0, 5, -6, 0, -1, 0, 0, 0, 2, -1, 0}, 104.867000, 0.000020, 0.074000},
	{{0, 5, -6, 0, 0, -1, 0, 0, 2, 0, -2}, 3.421590, 0.000020, 72.134000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -2, -2, 0}, 273.771490, 0.000020, 0.020000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -2, -1, 0}, 87.466160, 0.000140, 0.028000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -2, 0, 0}, 85.425600, 0.001130, 0.044000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -2, 0, 2}, 86.204630, 0.000010, 0.236000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -2, 1, -2}, 272.295770, 0.000010, 0.028000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -2, 1, 0}, 85.333080, 0.001160, 0.107000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -2, 2, -2}, 92.322940, 0.000010, 0.044000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -2, 2, 0}, 82.613030, 0.000030, 0.256000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -2, 3, 0}, 282.775480, 0.000020, 0.058000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -1, -1, 1}, 256.396420, 0.000010, 0.099000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -1, 0, -1}, 75.479950, 0.000180, 0.042000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -1, 0, 0}, 93.818240, 0.000020, 0.098000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -1, 0, 1}, 256.596860, 0.000390, 0.314000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -1, 1, -1}, 77.186860, 0.000080, 0.096000},
	{{0, 5, -6, 0, 0, 0, 0, 0, -1, 1, 1}, 256.332220, 0.000040, 0.061000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, -3, 0}, 92.138030, 0.000020, 0.027000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, -2, 0}, 91.094740, 0.000230, 0.041000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, -1, -2}, 272.304560, 0.000120, 0.026000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, -1, 0}, 91.398580, 0.003860, 0.090000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, 0, -2}, 272.296490, 0.000720, 0.040000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, 0, -1}, 76.384770, 0.000050, 0.089000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, 0, 0}, 272.668960, 0.003670, 0.470000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, 0, 1}, 76.331560, 0.000020, 0.064000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, 1, -2}, 92.390800, 0.000260, 0.087000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, 1, 0}, 272.322040, 0.000840, 0.065000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, 2, -2}, 92.322940, 0.000080, 0.557000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 0, 2, 0}, 272.471570, 0.000050, 0.035000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, -2, -1}, 76.375940, 0.000030, 0.039000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, -1, -1}, 76.453090, 0.000460, 0.082000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, -1, 0}, 92.553310, 0.000030, 0.806000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, -1, 1}, 256.385190, 0.000300, 0.068000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, 0, -2}, 92.297560, 0.000030, 0.081000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, 0, -1}, 76.355490, 0.003100, 0.931000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, 0, 0}, 90.734110, 0.000150, 0.069000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, 0, 1}, 253.959180, 0.000310, 0.036000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, 1, -1}, 76.597210, 0.000370, 0.070000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, 1, 1}, 254.703030, 0.000030, 0.024000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 1, 2, -1}, 76.376280, 0.000030, 0.036000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, -3, -2}, 272.347040, 0.000040, 0.025000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, -3, 0}, 133.384440, 0.000020, 0.078000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, -2, -2}, 272.312290, 0.000500, 0.038000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, -2, 0}, 102.696270, 0.000110, 2.828000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, -1, -4}, 92.436170, 0.000010, 0.025000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, -1, -2}, 272.309520, 0.007420, 0.076000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, -1, -1}, 253.680240, 0.000030, 5.352000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, -1, 0}, 271.881330, 0.022670, 0.073000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, -1, 2}, 91.818940, 0.000050, 0.025000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 0, -4}, 92.541500, 0.000070, 0.037000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 0, -2}, 272.295770, 0.052720, 49.790000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 0, -1}, 256.369440, 0.000370, 0.074000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 0, 0}, 92.870570, 0.003420, 0.037000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 1, -2}, 272.312290, 0.007380, 0.075000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 1, -1}, 256.369440, 0.000020, 0.037000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 1, 0}, 93.238790, 0.000450, 0.025000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 2, -2}, 272.312290, 0.000500, 0.038000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 2, 0}, 92.366180, 0.000060, 0.019000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 2, 3, -2}, 272.312290, 0.000040, 0.025000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 3, -3, 0}, 91.847480, 0.000010, 1.875000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 3, -1, -1}, 262.234170, 0.000120, 0.080000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 3, 0, -1}, 76.401670, 0.000030, 0.039000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 3, 0, 0}, 92.454980, 0.000020, 0.025000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 4, -2, -2}, 92.322940, 0.000080, 0.570000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 4, -2, 0}, 273.211210, 0.000320, 0.040000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 4, -1, -2}, 92.359930, 0.000390, 0.087000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 4, -1, 0}, 273.380240, 0.000150, 0.026000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 4, 0, -2}, 272.294710, 0.000490, 0.040000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 4, 0, 0}, 98.616610, 0.000080, 0.019000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 4, 1, -2}, 272.305610, 0.000100, 0.026000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 4, 1, 0}, 92.571590, 0.000010, 0.015000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 6, -2, -2}, 92.322940, 0.000010, 0.044000},
	{{0, 5, -6, 0, 0, 0, 0, 0, 6, -1, -2}, 272.295770, 0.000010, 0.028000},
	{{0, 5, -6, 0, 0, 1, 0, 0, 2, -3, 0}, 179.770120, 0.000020, 0.078000},
	{{0, 5, -6, 0, 0, 1, 0, 0, 2, -1, 0}, 179.770120, 0.000020, 0.073000},
	{{0, 5, -6, 0, 1, 0, 0, 0, -2, 0, 0}, 191.566460, 0.000090, 0.044000},
	{{0, 5, -6, 0, 1, 0, 0, 0, -2, 1, 0}, 189.254240, 0.000010, 0.108000},
	{{0, 5, -6, 0, 1, 0, 0, 0, 0, 0, 0}, 324.106360, 0.000020, 0.452000},
	{{0, 5, -6, 0, 1, 0, 0, 0, 2, -3, 0}, 201.099420, 0.000020, 0.078000},
	{{0, 5, -6, 0, 2, 0, 0, 0, 0, 0, 0}, 72.942780, 0.000020, 0.436000},
	{{0, 5, -6, 0, 2, 0, 0, 0, 2, -3, 0}, 257.297300, 0.000020, 0.079000},
	{{0, 5, -6, 0, 2, 0, 0, 0, 2, -1, 0}, 257.297300, 0.000020, 0.073000},
	{{0, 5, -6, 1, 0, 0, 0, 0, 0, 0, 0}, 179.549750, 0.000010, 0.376000},
	{{0, 5, -6, 2, 0, 0, 0, 0, 2, -3, 0}, 223.656990, 0.000020, 0.084000},
	{{0, 5, -6, 2, 0, 0, 0, 0, 2, -2, 0}, 43.656990, 0.000020, 0.706000},
	{{0, 5, -6, 2, 0, 0, 0, 0, 2, -1, 0}, 223.656990, 0.000020, 0.068000},
	{{0, 5, -6, 2, 0, 0, 0, 0, 2, 0, 0}, 43.656990, 0.000020, 0.036000},
	{{0, 5, -6, 3, 0, 0, 0, 0, -2, 0, 0}, 236.420880, 0.000020, 0.048000},
	{{0, 5, -6, 3, 0, 0, 0, 0, -2, 2, 0}, 236.420880, 0.000020, 0.182000},
	{{0, 5, -5, -4, 0, 0, 0, 0, -2, 0, 0}, 261.370650, 0.000030, 0.042000},
	{{0, 5, -5, -4, 0, 0, 0, 0, -2, 1, 0}, 261.370650, 0.000020, 0.095000},
	{{0, 5, -5, -4, 0, 0, 0, 0, 0, 0, 0}, 244.151230, 0.000020, 0.999000},
	{{0, 5, -5, -3, 0, 0, 0, 0, 2, -3, 0}, 34.533180, 0.000010, 0.074000},
	{{0, 5, -5, -3, 0, 0, 0, 0, 2, -1, 0}, 34.533180, 0.000010, 0.077000},
	{{0, 5, -5, 0, -2, 0, 0, 0, -2, 0, 0}, 205.435270, 0.000010, 0.046000},
	{{0, 5, -5, 0, -1, 0, 0, 0, -2, 0, 0}, 191.397950, 0.000060, 0.046000},
	{{0, 5, -5, 0, -1, 0, 0, 0, -2, 1, 0}, 13.052070, 0.000010, 0.119000},
	{{0, 5, -5, 0, -1, 0, 0, 0, -2, 2, 0}, 185.531890, 0.000010, 0.208000},
	{{0, 5, -5, 0, -1, 0, 0, 0, -2, 3, 0}, 193.052070, 0.000010, 0.055000},
	{{0, 5, -5, 0, -1, 0, 0, 0, 0, -2, 0}, 193.052070, 0.000010, 0.043000},
	{{0, 5, -5, 0, -1, 0, 0, 0, 0, 0, 0}, 54.387630, 0.000020, 0.329000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -4, -1, 0}, 179.517090, 0.000010, 0.017000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -4, 0, 0}, 359.115430, 0.000030, 0.022000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -4, 1, 0}, 358.794440, 0.000030, 0.030000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -4, 2, 0}, 1.010110, 0.000020, 0.050000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -2, -1, 0}, 356.504640, 0.000040, 0.029000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -2, 0, 0}, 358.249870, 0.001200, 0.046000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -2, 1, 0}, 358.303510, 0.001040, 0.120000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -2, 2, 0}, 358.896310, 0.000180, 0.204000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -2, 3, 0}, 356.391490, 0.000030, 0.055000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -1, -1, 0}, 180.286010, 0.000010, 0.044000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -1, 0, 0}, 0.066010, 0.000190, 0.108000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -1, 0, 1}, 14.863390, 0.000040, 0.239000},
	{{0, 5, -5, 0, 0, 0, 0, 0, -1, 1, 0}, 0.000570, 0.000050, 0.249000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, -3, 0}, 356.536370, 0.000020, 0.027000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, -2, 0}, 359.837510, 0.000140, 0.043000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, -1, 0}, 358.679020, 0.001970, 0.099000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, 0, -2}, 175.220010, 0.000110, 0.042000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, 0, 0}, 180.028510, 0.033050, 0.320000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, 0, 2}, 0.117890, 0.000030, 0.033000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, 1, -2}, 184.320860, 0.000040, 0.096000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, 1, 0}, 180.051320, 0.005320, 0.061000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, 2, 0}, 180.027460, 0.000430, 0.034000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, 3, 0}, 179.994270, 0.000040, 0.023000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 0, 4, 0}, 180.028550, 0.000010, 0.018000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 1, -2, 0}, 183.178460, 0.000020, 0.091000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 1, -1, -1}, 190.024210, 0.000020, 0.089000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 1, -1, 0}, 0.137080, 0.000090, 0.446000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 1, 0, -2}, 180.063530, 0.000040, 0.088000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 1, 0, -1}, 190.024210, 0.000270, 0.482000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 1, 0, 0}, 0.293820, 0.000640, 0.065000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 1, 1, -1}, 190.024210, 0.000020, 0.065000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 1, 1, 0}, 0.442440, 0.000030, 0.035000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, -2, 0}, 180.030080, 0.001240, 0.739000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, -1, -2}, 174.648150, 0.000090, 0.082000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, -1, 0}, 180.072990, 0.037590, 0.068000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, -1, 2}, 0.039140, 0.000080, 0.024000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, 0, -2}, 179.897730, 0.001260, 0.980000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, 0, 0}, 0.047880, 0.009520, 0.036000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, 0, 2}, 180.022090, 0.000040, 0.018000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, 1, -2}, 355.443530, 0.000080, 0.070000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, 1, 0}, 0.016660, 0.000800, 0.024000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 2, 2, 0}, 359.874780, 0.000120, 0.018000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 3, -2, 0}, 0.158370, 0.000120, 0.073000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 3, -1, -2}, 0.062580, 0.000010, 7.515000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 3, -1, -1}, 179.361560, 0.000020, 0.074000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 3, 0, -2}, 0.063530, 0.000180, 0.075000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 3, 0, 0}, 0.080320, 0.000030, 0.025000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 3, 1, 0}, 0.066340, 0.000010, 0.019000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 4, -3, 0}, 182.091880, 0.000030, 0.078000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 4, -2, 0}, 180.066020, 0.000520, 0.038000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 4, -1, -2}, 180.049450, 0.000040, 0.080000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 4, -1, 0}, 180.069250, 0.000160, 0.025000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 4, 0, -2}, 0.017790, 0.000020, 0.039000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 4, 0, 0}, 359.976340, 0.000100, 0.019000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 4, 1, 0}, 0.013500, 0.000040, 0.015000},
	{{0, 5, -5, 0, 0, 0, 0, 0, 6, -1, 0}, 0.035010, 0.000010, 0.016000},
	{{0, 5, -5, 0, 1, 0, 0, 0, -2, 0, 0}, 345.679100, 0.000040, 0.046000},
	{{0, 5, -5, 0, 1, 0, 0, 0, -2, 1, 0}, 171.261400, 0.000010, 0.121000},
	{{0, 5, -5, 0, 1, 0, 0, 0, -2, 3, 0}, 351.261400, 0.000010, 0.055000},
	{{0, 5, -5, 0, 1, 0, 0, 0, 0, 0, 0}, 171.261400, 0.000020, 0.311000},
	{{0, 5, -5, 0, 1, 0, 0, 0, 2, -2, 0}, 166.289600, 0.000020, 0.695000},
	{{0, 5, -5, 0, 1, 0, 0, 0, 2, -1, 0}, 346.289640, 0.000040, 0.068000},
	{{0, 5, -5, 0, 1, 0, 0, 0, 2, 0, 0}, 158.617540, 0.000030, 0.036000},
	{{0, 5, -4, -3, 0, 0, 0, 0, 2, -3, 0}, 123.560680, 0.000020, 0.080000},
	{{0, 5, -4, -3, 0, 0, 0, 0, 2, -2, 0}, 303.560680, 0.000010, 1.318000},
	{{0, 5, -4, -3, 0, 0, 0, 0, 2, -1, 0}, 123.560680, 0.000020, 0.071000},
	{{0, 5, -4, -3, 0, 0, 0, 0, 2, 0, 0}, 303.560680, 0.000010, 0.037000},
	{{0, 5, -4, -1, 0, 0, 0, 0, 0, 0, 0}, 180.431800, 0.000010, 0.278000},
	{{0, 5, -4, 0, -3, 0, 0, 0, 0, 0, 0}, 113.521500, 0.000020, 0.258000},
	{{0, 5, -4, 0, -2, 0, 0, 0, 0, 0, 0}, 276.752510, 0.000010, 0.253000},
	{{0, 5, -4, 0, -1, 0, 0, 0, -2, 0, 0}, 178.698270, 0.000010, 0.048000},
	{{0, 5, -4, 0, -1, 0, 0, 0, -2, 1, 0}, 181.486610, 0.000030, 0.134000},
	{{0, 5, -4, 0, -1, 0, 0, 0, -2, 2, 0}, 178.698270, 0.000010, 0.172000},
	{{0, 5, -4, 0, -1, 0, 0, 0, -2, 3, 0}, 1.486610, 0.000030, 0.052000},
	{{0, 5, -4, 0, -1, 0, 0, 0, 0, -2, 0}, 181.700830, 0.000010, 0.045000},
	{{0, 5, -4, 0, -1, 0, 0, 0, 0, -1, 0}, 181.274810, 0.000010, 0.109000},
	{{0, 5, -4, 0, -1, 0, 0, 0, 0, 0, 0}, 181.470930, 0.000080, 0.247000},
	{{0, 5, -4, 0, -1, 0, 0, 0, 0, 1, 0}, 181.274810, 0.000010, 0.058000},
	{{0, 5, -4, 0, -1, 0, 0, 0, 2, -2, 0}, 183.924900, 0.000020, 0.441000},
	{{0, 5, -4, 0, -1, 0, 0, 0, 2, -1, 0}, 183.924900, 0.000020, 0.064000},
	{{0, 5, -4, 0, -1, 0, 0, 0, 2, 0, 0}, 3.924900, 0.000020, 0.035000},
	{{0, 5, -4, 0, 0, -1, 0, 0, -2, 0, 0}, 180.211430, 0.000010, 0.048000},
	{{0, 5, -4, 0, 0, -1, 0, 0, -2, 2, 0}, 0.211430, 0.000010, 0.170000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -3, 0, 1}, 103.290320, 0.000020, 0.051000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -2, -1, 0}, 249.560000, 0.000020, 0.030000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -2, 0, 0}, 234.178590, 0.000080, 0.049000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -2, 1, 0}, 250.505910, 0.000180, 0.136000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -2, 2, 0}, 283.555450, 0.000020, 0.169000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -2, 3, 0}, 250.732140, 0.000010, 0.052000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -1, -1, -1}, 103.297200, 0.000020, 0.029000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -1, -1, 1}, 283.724750, 0.000050, 0.124000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -1, 0, -1}, 103.294700, 0.000290, 0.046000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -1, 0, 1}, 283.460540, 0.000420, 0.193000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -1, 1, -1}, 103.293230, 0.000140, 0.119000},
	{{0, 5, -4, 0, 0, 0, 0, 0, -1, 1, 1}, 283.415040, 0.000030, 0.054000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 0, -2, 0}, 241.553160, 0.000010, 0.045000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 0, -1, 0}, 253.940780, 0.000190, 0.110000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 0, 0, -1}, 283.309560, 0.000010, 0.108000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 0, 0, 0}, 95.599540, 0.000290, 0.242000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 0, 1, 0}, 75.105900, 0.000050, 0.058000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 1, -1, -1}, 103.320650, 0.000220, 0.098000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 1, -1, 1}, 283.326690, 0.000120, 0.060000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 1, 0, -1}, 103.327210, 0.001450, 0.325000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 1, 0, 1}, 283.323880, 0.000080, 0.033000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 1, 1, -1}, 103.323710, 0.000160, 0.061000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 2, -2, 0}, 266.996820, 0.000040, 0.425000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 2, -1, 0}, 263.825340, 0.000520, 0.064000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 2, 0, -2}, 262.831820, 0.000020, 0.495000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 2, 0, -1}, 283.348810, 0.000030, 0.065000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 2, 0, 0}, 81.926190, 0.000310, 0.035000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 2, 1, 0}, 78.764710, 0.000070, 0.024000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 3, -1, -1}, 103.323450, 0.000060, 0.069000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 3, 0, -1}, 103.326790, 0.000030, 0.036000},
	{{0, 5, -4, 0, 0, 0, 0, 0, 4, -1, 0}, 83.784640, 0.000020, 0.025000},
	{{0, 5, -3, -13, 0, 0, 0, 0, -2, 1, 0}, 328.443500, 0.000010, 0.075000},
	{{0, 5, -3, -13, 0, 0, 0, 0, -2, 3, 0}, 328.443500, 0.000010, 0.075000},
	{{0, 5, -3, -4, 0, 0, 0, 0, 0, 0, 0}, 303.536130, 0.000010, 0.333000},
	{{0, 5, -3, -3, 0, 0, 0, 0, -2, 0, 0}, 336.219450, 0.000010, 0.047000},
	{{0, 5, -3, -3, 0, 0, 0, 0, -2, 2, 0}, 336.219450, 0.000010, 0.188000},
	{{0, 5, -3, -2, 0, 0, 0, 0, -2, 0, 0}, 0.621590, 0.000010, 0.048000},
	{{0, 5, -3, -2, 0, 0, 0, 0, -2, 1, 0}, 179.689770, 0.000020, 0.135000},
	{{0, 5, -3, -2, 0, 0, 0, 0, -2, 2, 0}, 0.621590, 0.000010, 0.171000},
	{{0, 5, -3, -2, 0, 0, 0, 0, 0, -2, 0}, 179.889940, 0.000010, 0.045000},
	{{0, 5, -3, -2, 0, 0, 0, 0, 0, 0, 0}, 179.889940, 0.000010, 0.246000},
	{{0, 5, -3, 0, -2, 0, 0, 0, -2, -2, 0}, 0.101500, 0.000010, 0.022000},
	{{0, 5, -3, 0, -2, 0, 0, 0, -2, 2, 0}, 0.308580, 0.000020, 0.149000},
	{{0, 5, -3, 0, -2, 0, 0, 0, 0, -2, 0}, 359.762980, 0.000020, 0.046000},
	{{0, 5, -3, 0, -1, 0, 0, 0, -2, 0, 0}, 89.843340, 0.000020, 0.051000},
	{{0, 5, -3, 0, -1, 0, 0, 0, -2, 2, 0}, 269.843340, 0.000020, 0.147000},
	{{0, 5, -3, 0, 0, 0, 0, 0, -2, -1, 0}, 28.893560, 0.000030, 0.030000},
	{{0, 5, -3, 0, 0, 0, 0, 0, -2, 0, 0}, 29.999730, 0.000280, 0.051000},
	{{0, 5, -3, 0, 0, 0, 0, 0, -2, 1, 0}, 28.132680, 0.000170, 0.157000},
	{{0, 5, -3, 0, 0, 0, 0, 0, -2, 2, 0}, 25.996160, 0.000020, 0.145000},
	{{0, 5, -3, 0, 0, 0, 0, 0, -1, 0, -1}, 185.742200, 0.000010, 0.048000},
	{{0, 5, -3, 0, 0, 0, 0, 0, -1, 0, 0}, 205.642460, 0.000020, 0.138000},
	{{0, 5, -3, 0, 0, 0, 0, 0, -1, 0, 1}, 188.993500, 0.000010, 0.162000},
	{{0, 5, -3, 0, 0, 0, 0, 0, 0, -1, 0}, 25.763460, 0.000170, 0.123000},
	{{0, 5, -3, 0, 0, 0, 0, 0, 0, 0, 0}, 26.391200, 0.001190, 0.195000},
	{{0, 5, -3, 0, 0, 0, 0, 0, 0, 1, 0}, 25.990230, 0.000120, 0.054000},
	{{0, 5, -3, 0, 0, 0, 0, 0, 0, 2, 0}, 149.877870, 0.000010, 0.032000},
	{{0, 5, -3, 0, 0, 0, 0, 0, 1, 0, -1}, 8.063020, 0.000020, 0.245000},
	{{0, 5, -3, 0, 0, 0, 0, 0, 1, 0, 0}, 206.273310, 0.000020, 0.057000},
	{{0, 5, -3, 0, 0, 0, 0, 0, 2, -2, 0}, 336.820290, 0.000020, 0.298000},
	{{0, 5, -3, 0, 0, 0, 0, 0, 2, -1, 0}, 29.034890, 0.000190, 0.060000},
	{{0, 5, -3, 0, 0, 0, 0, 0, 2, 0, 0}, 216.673130, 0.000100, 0.033000},
	{{0, 5, -2, 0, -3, 0, 0, 0, -2, 0, 0}, 356.688290, 0.000010, 0.053000},
	{{0, 5, -2, 0, -3, 0, 0, 0, -2, 2, 0}, 176.688290, 0.000010, 0.131000},
	{{0, 5, -2, 0, 0, 0, 0, 0, -2, 0, 0}, 290.034970, 0.000020, 0.054000},
	{{0, 5, -2, 0, 0, 0, 0, 0, -2, 1, 0}, 288.463150, 0.000020, 0.187000},
	{{0, 5, -2, 0, 0, 0, 0, 0, 0, 0, 0}, 288.020330, 0.000080, 0.163000},
	{{0, 5, -2, 0, 0, 0, 0, 0, 1, 0, -1}, 308.314130, 0.000010, 0.197000},
	{{0, 5, -2, 0, 0, 0, 0, 0, 2, 0, 0}, 249.941980, 0.000020, 0.032000},
	{{0, 5, -2, 0, 1, 0, 0, 0, 2, 0, 0}, 133.506890, 0.000040, 0.032000},
	{{0, 5, -1, 0, -2, 0, 0, 0, 2, -2, 0}, 333.687550, 0.000010, 0.193000},
	{{0, 5, -1, 0, -2, 0, 0, 0, 2, 0, 0}, 153.687550, 0.000010, 0.032000},
	{{0, 5, -1, 0, -1, 0, 0, 0, 2, -2, 0}, 141.867890, 0.000010, 0.190000},
	{{0, 5, -1, 0, -1, 0, 0, 0, 2, 0, 0}, 144.851980, 0.000080, 0.031000},
	{{0, 5, -1, 0, 0, 0, 0, 0, -2, 0, 0}, 181.550700, 0.000010, 0.057000},
	{{0, 5, -1, 0, 0, 0, 0, 0, -2, 2, 0}, 181.550700, 0.000010, 0.112000},
	{{0, 5, -1, 0, 0, 0, 0, 0, 0, -2, 0}, 146.797690, 0.000010, 0.052000},
	{{0, 5, -1, 0, 0, 0, 0, 0, 0, -1, 0}, 128.212470, 0.000020, 0.163000},
	{{0, 5, -1, 0, 0, 0, 0, 0, 0, 0, 0}, 187.974540, 0.000020, 0.140000},
	{{0, 5, -1, 0, 0, 0, 0, 0, 0, 1, 0}, 128.141990, 0.000030, 0.049000},
	{{0, 5, -1, 0, 0, 0, 0, 0, 2, -3, 0}, 128.059080, 0.000010, 0.127000},
	{{0, 5, -1, 0, 0, 0, 0, 0, 2, -2, 0}, 128.033850, 0.000020, 0.187000},
	{{0, 5, -1, 0, 0, 0, 0, 0, 2, -1, 0}, 308.059080, 0.000010, 0.054000},
	{{0, 5, -1, 0, 0, 0, 0, 0, 2, 0, 0}, 131.410070, 0.000020, 0.031000},
	{{0, 5, -1, 0, 0, 0, 0, 0, 2, 2, 0}, 128.212470, 0.000020, 0.017000},
	{{0, 5, -1, 0, 1, 0, 0, 0, 2, -2, 0}, 299.517200, 0.000030, 0.184000},
	{{0, 5, -1, 0, 1, 0, 0, 0, 2, 0, 0}, 298.037760, 0.000050, 0.031000},
	{{0, 5, 0, 0, -1, 0, 0, 0, 0, 0, 0}, 129.527090, 0.000010, 0.124000},
	{{0, 5, 0, 0, -1, 0, 0, 0, 0, 2, 0}, 129.527090, 0.000010, 0.029000},
	{{0, 5, 0, 0, -1, 0, 0, 0, 2, 0, 0}, 128.783760, 0.000040, 0.031000},
	{{0, 5, 0, 0, 0, 0, 0, 0, -2, 0, 0}, 302.050320, 0.000050, 0.060000},
	{{0, 5, 0, 0, 0, 0, 0, 0, -2, 2, 0}, 302.050320, 0.000050, 0.101000},
	{{0, 5, 0, 0, 0, 0, 0, 0, 2, -2, 0}, 212.281910, 0.000030, 0.157000},
	{{0, 5, 0, 0, 0, 0, 0, 0, 2, -1, 0}, 210.019500, 0.000020, 0.051000},
	{{0, 5, 1, 0, 0, 0, 0, 0, 2, -1, 0}, 153.604800, 0.000020, 0.049000},
	{{0, 5, 31, -52, 0, 0, 0, 0, -2, 1, 0}, 323.507770, 0.000020, 587.442000},
	{{0, 6, -14, 4, 0, 0, 0, 0, -2, 0, 2}, 79.805100, 0.000020, 79.413000},
	{{0, 6, -14, 6, 0, 0, 0, 0, -1, 0, 1}, 162.817380, 0.000010, 334.478000},
	{{0, 6, -14, 8, 0, 0, 0, 0, 0, -1, 0}, 174.654970, 0.000010, 0.075000},
	{{0, 6, -14, 8, 0, 0, 0, 0, 0, 0, 0}, 53.992660, 0.000010, 151.217000},
	{{0, 6, -14, 8, 0, 0, 0, 0, 0, 1, 0}, 174.654970, 0.000010, 0.075000},
	{{0, 6, -13, 0, 2, 0, 0, 0, -2, 0, 0}, 322.793420, 0.000020, 0.036000},
	{{0, 6, -13, 0, 2, 0, 0, 0, -2, 1, 0}, 142.793420, 0.000020, 0.069000},
	{{0, 6, -13, 4, 0, 0, 0, 0, -2, 1, 0}, 81.323460, 0.000010, 0.079000},
	{{0, 6, -13, 4, 0, 0, 0, 0, -2, 3, 0}, 81.323460, 0.000010, 0.072000},
	{{0, 6, -12, 0, 0, 0, 0, 0, -4, 0, 0}, 40.207540, 0.000020, 0.019000},
	{{0, 6, -12, 0, 0, 0, 0, 0, -2, -1, 0}, 220.207540, 0.000020, 0.025000},
	{{0, 6, -12, 0, 0, 0, 0, 0, -2, -1, 2}, 220.207540, 0.000010, 0.075000},
	{{0, 6, -12, 0, 0, 0, 0, 0, -2, 0, 0}, 219.605420, 0.000030, 0.037000},
	{{0, 6, -12, 0, 0, 0, 0, 0, -2, 0, 2}, 218.342380, 0.000010, 7.177000},
	{{0, 6, -12, 0, 0, 0, 0, 0, -2, 1, 0}, 220.207540, 0.000020, 0.073000},
	{{0, 6, -12, 0, 0, 0, 0, 0, -2, 1, 2}, 220.207540, 0.000010, 0.076000},
	{{0, 6, -12, 0, 0, 0, 0, 0, -2, 2, 0}, 219.605420, 0.000030, 2.115000},
	{{0, 6, -12, 0, 0, 0, 0, 0, 0, 0, 0}, 276.701500, 0.000020, 0.445000},
	{{0, 6, -12, 0, 0, 4, 0, 0, -2, 0, 2}, 46.326530, 0.000030, 282.735000},
	{{0, 6, -12, 0, 2, 0, 0, 0, -2, 0, 0}, 231.630250, 0.000040, 0.037000},
	{{0, 6, -12, 0, 2, 0, 0, 0, -2, 0, 2}, 202.988490, 0.000020, 34.155000},
	{{0, 6, -12, 0, 2, 0, 0, 0, 0, -1, 0}, 51.619460, 0.000010, 0.065000},
	{{0, 6, -12, 0, 2, 0, 0, 0, 0, 1, 0}, 51.619460, 0.000010, 0.089000},
	{{0, 6, -12, 0, 3, 0, 0, 0, -2, 0, 0}, 221.868180, 0.000010, 0.037000},
	{{0, 6, -12, 0, 3, 0, 0, 0, -2, 1, 0}, 39.653570, 0.000010, 0.074000},
	{{0, 6, -12, 0, 3, 0, 0, 0, -2, 2, 0}, 41.868180, 0.000010, 4.547000},
	{{0, 6, -12, 0, 3, 0, 0, 0, -2, 3, 0}, 39.653570, 0.000010, 0.077000},
	{{0, 6, -12, 0, 4, 0, 0, 0, -2, 1, 0}, 32.274260, 0.000030, 0.075000},
	{{0, 6, -12, 1, 0, 0, 0, 0, -2, 1, 0}, 142.430430, 0.000010, 0.076000},
	{{0, 6, -12, 1, 0, 0, 0, 0, -2, 3, 0}, 142.430430, 0.000010, 0.075000},
	{{0, 6, -12, 2, 0, 0, 0, 0, -2, 0, 0}, 233.649010, 0.000050, 0.039000},
	{{0, 6, -12, 2, 0, 0, 0, 0, -2, 2, 0}, 231.762220, 0.000040, 1.693000},
	{{0, 6, -12, 2, 0, 0, 0, 0, 0, -1, 0}, 231.831010, 0.000010, 0.069000},
	{{0, 6, -12, 2, 0, 0, 0, 0, 0, 1, 0}, 231.831010, 0.000010, 0.083000},
	{{0, 6, -11, -1, 0, 0, 0, 0, -2, 1, 0}, 42.480040, 0.000020, 0.075000},
	{{0, 6, -11, -1, 0, 0, 0, 0, -2, 2, 0}, 106.863280, 0.000020, 220.866000},
	{{0, 6, -11, -1, 0, 0, 0, 0, -2, 3, 0}, 42.480040, 0.000020, 0.075000},
	{{0, 6, -11, 0, -1, 0, 0, 0, -2, 1, 0}, 332.713390, 0.000030, 0.078000},
	{{0, 6, -11, 0, -1, 0, 0, 0, -2, 3, 0}, 152.713390, 0.000030, 0.073000},
	{{0, 6, -11, 0, 0, 0, 0, 0, -4, 3, 0}, 352.400180, 0.000010, 0.091000},
	{{0, 6, -11, 0, 0, 0, 0, 0, -2, -1, 2}, 324.025440, 0.000010, 0.081000},
	{{0, 6, -11, 0, 0, 0, 0, 0, -2, 0, 0}, 358.530530, 0.000010, 0.038000},
	{{0, 6, -11, 0, 0, 0, 0, 0, -2, 1, 0}, 208.826500, 0.000040, 0.079000},
	{{0, 6, -11, 0, 0, 0, 0, 0, -2, 1, 2}, 324.025440, 0.000010, 0.071000},
	{{0, 6, -11, 0, 0, 0, 0, 0, -2, 2, 0}, 333.082130, 0.000060, 1.897000},
	{{0, 6, -11, 0, 0, 0, 0, 0, -2, 3, 0}, 197.217630, 0.000040, 0.073000},
	{{0, 6, -11, 0, 0, 0, 0, 0, 0, 0, 0}, 183.067570, 0.000020, 0.802000},
	{{0, 6, -11, 0, 0, 0, 0, 0, 0, 2, 0}, 169.743280, 0.000020, 0.040000},
	{{0, 6, -11, 0, 0, 0, 0, 0, 2, -3, 0}, 334.740310, 0.000020, 0.061000},
	{{0, 6, -11, 0, 0, 0, 0, 0, 2, -2, 0}, 57.852840, 0.000010, 0.331000},
	{{0, 6, -11, 0, 0, 0, 0, 0, 2, -1, 0}, 334.740310, 0.000020, 0.098000},
	{{0, 6, -11, 0, 0, 0, 0, 0, 2, 0, 0}, 57.852840, 0.000010, 0.043000},
	{{0, 6, -11, 0, 1, 0, 0, 0, -2, 0, 0}, 51.170750, 0.000010, 0.039000},
	{{0, 6, -11, 0, 1, 0, 0, 0, -2, 1, 0}, 344.447690, 0.000030, 0.079000},
	{{0, 6, -11, 0, 1, 0, 0, 0, -2, 2, 0}, 51.170750, 0.000010, 1.635000},
	{{0, 6, -11, 0, 1, 0, 0, 0, 0, -1, 0}, 50.436740, 0.000040, 0.069000},
	{{0, 6, -11, 0, 1, 0, 0, 0, 0, 0, 0}, 50.446150, 0.000030, 0.860000},
	{{0, 6, -11, 0, 1, 0, 0, 0, 0, 1, 0}, 50.436740, 0.000040, 0.083000},
	{{0, 6, -11, 0, 2, 0, 0, 0, -2, 1, 0}, 291.957400, 0.000030, 0.080000},
	{{0, 6, -11, 0, 2, 0, 0, 0, -2, 3, 0}, 291.957400, 0.000030, 0.072000},
	{{0, 6, -11, 0, 3, 0, 0, 0, 0, 0, 0}, 35.647740, 0.000040, 1.006000},
	{{0, 6, -11, 0, 3, 0, 0, 0, 2, -1, 0}, 35.901020, 0.000050, 0.095000},
	{{0, 6, -11, 0, 3, 0, 0, 0, 2, 0, 0}, 35.901020, 0.000040, 0.042000},
	{{0, 6, -11, 2, 0, 0, 0, 0, -2, 1, 0}, 95.396400, 0.000010, 0.086000},
	{{0, 6, -11, 2, 0, 0, 0, 0, -2, 3, 0}, 95.396400, 0.000010, 0.067000},
	{{0, 6, -11, 2, 0, 0, 0, 0, 0, -1, 0}, 98.600470, 0.000020, 0.074000},
	{{0, 6, -11, 2, 0, 0, 0, 0, 0, 1, 0}, 98.600470, 0.000020, 0.076000},
	{{0, 6, -11, 3, 0, 0, 0, 0, 0, -1, 0}, 288.604930, 0.000010, 0.077000},
	{{0, 6, -11, 3, 0, 0, 0, 0, 0, 1, 0}, 288.604930, 0.000010, 0.074000},
	{{0, 6, -10, -3, 0, 0, 0, 0, -2, 1, 0}, 86.499690, 0.000020, 0.075000},
	{{0, 6, -10, -3, 0, 0, 0, 0, -2, 3, 0}, 86.499690, 0.000020, 0.076000},
	{{0, 6, -10, 0, -1, 0, 0, 0, -2, 0, 0}, 254.105500, 0.000040, 0.040000},
	{{0, 6, -10, 0, -1, 0, 0, 0, -2, 1, 0}, 241.626130, 0.000070, 0.085000},
	{{0, 6, -10, 0, -1, 0, 0, 0, 0, -2, 2}, 323.899880, 0.000030, 425.789000},
	{{0, 6, -10, 0, 0, 0, 0, 0, -4, 2, 0}, 17.615760, 0.000010, 0.043000},
	{{0, 6, -10, 0, 0, 0, 0, 0, -2, -2, 0}, 51.704940, 0.000020, 0.019000},
	{{0, 6, -10, 0, 0, 0, 0, 0, -2, 0, 0}, 94.972980, 0.000060, 0.040000},
	{{0, 6, -10, 0, 0, 0, 0, 0, -2, 0, 2}, 231.784070, 0.000030, 0.537000},
	{{0, 6, -10, 0, 0, 0, 0, 0, -2, 1, 0}, 120.963430, 0.000070, 0.085000},
	{{0, 6, -10, 0, 0, 0, 0, 0, -2, 2, 0}, 60.202670, 0.000020, 0.655000},
	{{0, 6, -10, 0, 0, 0, 0, 0, -2, 3, 0}, 55.068040, 0.000030, 0.068000},
	{{0, 6, -10, 0, 0, 0, 0, 0, 0, -1, 0}, 212.950890, 0.000040, 0.074000},
	{{0, 6, -10, 0, 0, 0, 0, 0, 0, 0, 0}, 196.296760, 0.000080, 4.051000},
	{{0, 6, -10, 0, 0, 0, 0, 0, 0, 1, 0}, 198.196440, 0.000050, 0.077000},
	{{0, 6, -10, 0, 0, 0, 0, 0, 1, 0, -1}, 150.918110, 0.000010, 0.769000},
	{{0, 6, -10, 0, 0, 0, 0, 0, 2, -1, 0}, 152.015180, 0.000100, 0.089000},
	{{0, 6, -10, 0, 0, 0, 0, 0, 2, 0, 0}, 156.575380, 0.000070, 0.041000},
	{{0, 6, -10, 0, 1, 0, 0, 0, -2, 0, 0}, 41.436530, 0.000030, 0.040000},
	{{0, 6, -10, 0, 1, 0, 0, 0, -2, 1, 0}, 39.000000, 0.000030, 0.086000},
	{{0, 6, -10, 0, 1, 0, 0, 0, -2, 3, 0}, 219.000000, 0.000030, 0.067000},
	{{0, 6, -10, 0, 1, 0, 0, 0, 0, -1, 0}, 66.603080, 0.000010, 0.075000},
	{{0, 6, -10, 0, 1, 0, 0, 0, 0, 1, 0}, 66.603080, 0.000010, 0.076000},
	{{0, 6, -10, 0, 2, 0, 0, 0, 0, -1, 0}, 312.044600, 0.000010, 0.075000},
	{{0, 6, -10, 0, 2, 0, 0, 0, 0, 0, 0}, 346.210800, 0.000020, 12.779000},
	{{0, 6, -10, 0, 2, 0, 0, 0, 0, 1, 0}, 312.044600, 0.000010, 0.076000},
	{{0, 6, -10, 0, 3, 0, 0, 0, -4, 1, 0}, 293.079240, 0.000010, 0.028000},
	{{0, 6, -10, 0, 3, 0, 0, 0, -2, -1, 0}, 293.079240, 0.000040, 0.026000},
	{{0, 6, -10, 0, 3, 0, 0, 0, -2, 0, 0}, 293.027680, 0.000390, 0.040000},
	{{0, 6, -10, 0, 3, 0, 0, 0, -2, 1, 0}, 293.591060, 0.000610, 0.087000},
	{{0, 6, -10, 0, 3, 0, 0, 0, -2, 2, 0}, 293.079820, 0.000020, 0.562000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 0, -2, 0}, 290.705470, 0.000050, 0.038000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 0, -1, 0}, 290.547060, 0.000680, 0.075000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 0, 0, 0}, 292.614280, 0.003730, 165.227000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 0, 1, 0}, 290.540030, 0.000680, 0.075000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 0, 2, 0}, 290.705240, 0.000050, 0.038000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 2, -2, 0}, 293.070230, 0.000020, 0.566000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 2, -1, 0}, 293.586450, 0.000600, 0.087000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 2, 0, 0}, 293.019480, 0.000380, 0.040000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 2, 1, 0}, 293.070830, 0.000040, 0.026000},
	{{0, 6, -10, 0, 3, 0, 0, 0, 4, -1, 0}, 293.070830, 0.000010, 0.028000},
	{{0, 6, -10, 0, 4, 0, 0, 0, 0, 0, 0}, 203.251750, 0.000030, 11.067000},
	{{0, 6, -9, -3, 0, 0, 0, 0, -2, 0, 0}, 355.303930, 0.000010, 0.039000},
	{{0, 6, -9, -3, 0, 0, 0, 0, -2, 1, 0}, 355.303930, 0.000010, 0.081000},
	{{0, 6, -9, -3, 0, 0, 0, 0, -2, 2, 0}, 355.303930, 0.000010, 1.073000},
	{{0, 6, -9, -3, 0, 0, 0, 0, -2, 3, 0}, 355.303930, 0.000010, 0.070000},
	{{0, 6, -9, -2, 0, 0, 0, 0, -2, 1, 0}, 191.214340, 0.000010, 0.085000},
	{{0, 6, -9, -2, 0, 0, 0, 0, -2, 3, 0}, 191.214340, 0.000010, 0.068000},
	{{0, 6, -9, -2, 0, 0, 0, 0, 0, -1, 0}, 190.995430, 0.000020, 0.074000},
	{{0, 6, -9, -2, 0, 0, 0, 0, 0, 1, 0}, 190.995430, 0.000020, 0.077000},
	{{0, 6, -9, 0, -1, 0, 0, 0, -2, 0, 0}, 235.634110, 0.000010, 0.042000},
	{{0, 6, -9, 0, -1, 0, 0, 0, 0, -1, 0}, 55.634110, 0.000010, 0.079000},
	{{0, 6, -9, 0, -1, 0, 0, 0, 0, 1, 0}, 55.634110, 0.000010, 0.072000},
	{{0, 6, -9, 0, 0, 0, 0, 0, -4, 1, 0}, 71.477350, 0.000010, 0.028000},
	{{0, 6, -9, 0, 0, 0, 0, 0, -4, 2, 0}, 122.825540, 0.000020, 0.045000},
	{{0, 6, -9, 0, 0, 0, 0, 0, -2, -1, 0}, 68.840680, 0.000040, 0.027000},
	{{0, 6, -9, 0, 0, 0, 0, 0, -2, 0, 0}, 71.388500, 0.000380, 0.042000},
	{{0, 6, -9, 0, 0, 0, 0, 0, -2, 1, 0}, 71.741300, 0.000470, 0.093000},
	{{0, 6, -9, 0, 0, 0, 0, 0, -2, 2, 0}, 57.383480, 0.000020, 0.396000},
	{{0, 6, -9, 0, 0, 0, 0, 0, -1, -1, 1}, 246.551870, 0.000060, 0.087000},
	{{0, 6, -9, 0, 0, 0, 0, 0, -1, 0, 0}, 250.051080, 0.000030, 0.086000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 0, -2, 0}, 64.409220, 0.000020, 0.039000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 0, -1, 0}, 67.703890, 0.000480, 0.080000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 0, 0, 0}, 71.175810, 0.002540, 1.328000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 0, 1, 0}, 70.987760, 0.000410, 0.071000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 0, 2, 0}, 71.355580, 0.000030, 0.037000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 1, -2, 1}, 245.668850, 0.000140, 30.407000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 1, -1, -1}, 66.453290, 0.000020, 0.074000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 1, -1, 0}, 69.383980, 0.000030, 7.470000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 1, -1, 1}, 66.575580, 0.000290, 0.075000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 1, 0, -1}, 66.755490, 0.000210, 3.326000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 1, 0, 0}, 70.051050, 0.000150, 0.076000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 1, 0, 1}, 247.825660, 0.000010, 0.038000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 1, 1, -1}, 66.453290, 0.000020, 0.077000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 2, -2, 0}, 267.405400, 0.000060, 0.980000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 2, -1, 0}, 265.568280, 0.000630, 0.082000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 2, 0, 0}, 266.191410, 0.000090, 0.039000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 4, -3, 0}, 269.337460, 0.000040, 0.096000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 4, -2, 0}, 270.100340, 0.000100, 0.042000},
	{{0, 6, -9, 0, 0, 0, 0, 0, 4, -1, 0}, 270.179600, 0.000050, 0.027000},
	{{0, 6, -9, 0, 2, 0, 0, 0, 0, 0, 0}, 271.000060, 0.000030, 1.085000},
	{{0, 6, -9, 0, 2, 0, 0, 0, 2, -2, 0}, 271.045710, 0.000010, 1.173000},
	{{0, 6, -9, 0, 2, 0, 0, 0, 2, 0, 0}, 271.045710, 0.000010, 0.039000},
	{{0, 6, -9, 0, 3, 0, 0, 0, -2, 0, 0}, 177.705770, 0.000020, 0.042000},
	{{0, 6, -9, 0, 3, 0, 0, 0, 0, 0, 0}, 21.772410, 0.000020, 0.994000},
	{{0, 6, -9, 2, 0, 0, 0, 0, 0, 0, 0}, 92.787830, 0.000010, 0.551000},
	{{0, 6, -9, 2, 0, 0, 0, 0, 2, -3, 0}, 1.524690, 0.000030, 0.076000},
	{{0, 6, -9, 2, 0, 0, 0, 0, 2, -1, 0}, 1.524690, 0.000030, 0.075000},
	{{0, 6, -8, 0, -2, 0, 0, 0, -2, 0, 0}, 51.826510, 0.000010, 0.043000},
	{{0, 6, -8, 0, -2, 0, 0, 0, -2, 2, 0}, 231.826510, 0.000010, 0.298000},
	{{0, 6, -8, 0, 0, 0, 0, 0, -2, -2, 0}, 347.139080, 0.000020, 0.020000},
	{{0, 6, -8, 0, 0, 0, 0, 0, -2, -1, 0}, 350.841060, 0.000100, 0.028000},
	{{0, 6, -8, 0, 0, 0, 0, 0, -2, 0, 0}, 308.485920, 0.000050, 0.044000},
	{{0, 6, -8, 0, 0, 0, 0, 0, -2, 1, 0}, 172.680910, 0.000250, 0.103000},
	{{0, 6, -8, 0, 0, 0, 0, 0, -2, 2, 0}, 335.370270, 0.000010, 0.284000},
	{{0, 6, -8, 0, 0, 0, 0, 0, -1, 0, -1}, 168.612940, 0.000020, 0.042000},
	{{0, 6, -8, 0, 0, 0, 0, 0, -1, 0, 1}, 357.491920, 0.000060, 0.356000},
	{{0, 6, -8, 0, 0, 0, 0, 0, -1, 1, -1}, 211.021060, 0.000020, 0.093000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 0, -2, 0}, 348.506960, 0.000320, 0.040000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 0, -1, -2}, 167.245850, 0.000030, 0.026000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 0, -1, 0}, 347.129790, 0.014410, 0.087000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 0, 0, 0}, 342.872890, 0.002620, 0.570000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 0, 1, 0}, 344.328610, 0.000350, 0.067000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 0, 2, 0}, 343.351920, 0.000010, 0.035000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 1, -2, 0}, 347.208520, 0.000020, 0.081000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 1, -1, -1}, 168.986090, 0.000100, 0.080000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 1, -1, 0}, 166.189680, 0.000080, 1.155000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 1, -1, 1}, 348.988290, 0.000060, 0.070000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 1, 0, -1}, 169.089770, 0.000820, 1.430000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 1, 0, 0}, 162.222730, 0.000090, 0.071000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 1, 0, 1}, 349.167700, 0.000080, 0.036000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 1, 1, -1}, 168.986090, 0.000100, 0.072000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -4, 0}, 168.880150, 0.000130, 0.038000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -3, -2}, 347.546480, 0.000020, 0.025000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -3, 0}, 168.894830, 0.001920, 0.075000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -2, -2}, 347.101750, 0.000040, 0.037000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -2, 0}, 167.126860, 0.059460, 47.898000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -2, 2}, 347.134170, 0.000070, 0.037000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -1, -2}, 167.246270, 0.000130, 0.073000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -1, -1}, 173.174520, 0.000010, 5.330000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -1, 0}, 167.222620, 0.076670, 0.076000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, -1, 2}, 347.232070, 0.000170, 0.025000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, 0, -2}, 169.189060, 0.000060, 2.822000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, 0, -1}, 169.101730, 0.000070, 0.077000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, 0, 0}, 167.440830, 0.005580, 0.038000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, 0, 2}, 347.246200, 0.000030, 0.019000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, 1, 0}, 167.217700, 0.000410, 0.025000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 2, 2, 0}, 167.289680, 0.000030, 0.019000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 3, -2, 0}, 347.111300, 0.000020, 0.081000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 3, -1, 0}, 347.245570, 0.000030, 0.039000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 4, -4, 0}, 164.601910, 0.000030, 0.557000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 4, -3, 0}, 166.385620, 0.000930, 0.087000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 4, -2, 0}, 167.165160, 0.001660, 0.040000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 4, -1, -2}, 347.246040, 0.000020, 0.090000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 4, -1, 0}, 167.266940, 0.000800, 0.026000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 4, 0, 0}, 167.525630, 0.000130, 0.020000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 6, -4, 0}, 166.396820, 0.000010, 0.044000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 6, -3, 0}, 166.978110, 0.000030, 0.028000},
	{{0, 6, -8, 0, 0, 0, 0, 0, 6, -2, 0}, 167.198400, 0.000020, 0.020000},
	{{0, 6, -8, 0, 0, 0, 1, 0, 2, -2, 0}, 95.812770, 0.000010, 111.409000},
	{{0, 6, -8, 0, 0, 1, 0, 0, 2, -3, 0}, 302.599670, 0.000010, 0.076000},
	{{0, 6, -8, 0, 0, 1, 0, 0, 2, -2, 0}, 85.798650, 0.000070, 76.512000},
	{{0, 6, -8, 0, 0, 1, 0, 0, 2, -1, 0}, 122.599670, 0.000010, 0.075000},
	{{0, 6, -8, 0, 1, 0, 0, 0, 0, -1, 0}, 64.227040, 0.000030, 0.088000},
	{{0, 6, -8, 0, 1, 0, 0, 0, 0, 0, 0}, 91.254110, 0.000010, 0.544000},
	{{0, 6, -8, 0, 1, 0, 0, 0, 2, -2, 0}, 21.554750, 0.000030, 15.767000},
	{{0, 6, -8, 0, 1, 0, 0, 0, 2, -1, 0}, 244.226970, 0.000130, 0.075000},
	{{0, 6, -8, 0, 2, 0, 0, 0, -2, 0, 0}, 179.891950, 0.000010, 0.044000},
	{{0, 6, -8, 0, 2, 0, 0, 0, -2, 2, 0}, 359.891950, 0.000010, 0.271000},
	{{0, 6, -8, 0, 2, 0, 0, 0, 0, -1, 0}, 359.871740, 0.000010, 0.088000},
	{{0, 6, -8, 0, 2, 0, 0, 0, 0, 1, 0}, 359.871740, 0.000010, 0.066000},
	{{0, 6, -8, 0, 2, 0, 0, 0, 2, -1, 0}, 357.288020, 0.000030, 0.075000},
	{{0, 6, -8, 0, 2, 0, 0, 0, 2, 0, 0}, 359.860720, 0.000020, 0.038000},
	{{0, 6, -8, 0, 3, 0, 0, 0, 2, -3, 0}, 169.171140, 0.000010, 0.077000},
	{{0, 6, -8, 0, 3, 0, 0, 0, 2, -2, 0}, 350.120460, 0.000010, 4.310000},
	{{0, 6, -8, 0, 3, 0, 0, 0, 2, -1, 0}, 169.171140, 0.000010, 0.074000},
	{{0, 6, -8, 0, 3, 0, 0, 0, 2, 0, 0}, 170.120460, 0.000010, 0.037000},
	{{0, 6, -8, 0, 4, 0, 0, 0, 2, 0, -2}, 341.341120, 0.000020, 58.227000},
	{{0, 6, -8, 2, 0, 0, 0, 0, 2, -2, 0}, 0.171230, 0.000020, 0.959000},
	{{0, 6, -8, 2, 0, 0, 0, 0, 2, 0, 0}, 353.933990, 0.000020, 0.036000},
	{{0, 6, -7, -2, 0, 0, 0, 0, 2, -3, 0}, 309.222470, 0.000010, 0.075000},
	{{0, 6, -7, -2, 0, 0, 0, 0, 2, -1, 0}, 309.222470, 0.000010, 0.076000},
	{{0, 6, -7, 0, -1, 0, 0, 0, -2, 0, 0}, 104.449980, 0.000020, 0.045000},
	{{0, 6, -7, 0, -1, 0, 0, 0, 0, 0, 0}, 263.984090, 0.000010, 0.375000},
	{{0, 6, -7, 0, -1, 0, 0, 0, 2, -3, 0}, 99.885810, 0.000010, 0.081000},
	{{0, 6, -7, 0, -1, 0, 0, 0, 2, -1, 0}, 279.885810, 0.000010, 0.071000},
	{{0, 6, -7, 0, 0, 0, 0, 0, -2, -1, 0}, 99.198360, 0.000020, 0.028000},
	{{0, 6, -7, 0, 0, 0, 0, 0, -2, 0, 0}, 89.947890, 0.000200, 0.045000},
	{{0, 6, -7, 0, 0, 0, 0, 0, -2, 1, 0}, 88.468780, 0.000320, 0.115000},
	{{0, 6, -7, 0, 0, 0, 0, 0, -2, 2, 0}, 283.820870, 0.000020, 0.221000},
	{{0, 6, -7, 0, 0, 0, 0, 0, -1, 0, -1}, 76.253860, 0.000110, 0.043000},
	{{0, 6, -7, 0, 0, 0, 0, 0, -1, 0, 0}, 91.702300, 0.000010, 0.104000},
	{{0, 6, -7, 0, 0, 0, 0, 0, -1, 0, 1}, 256.237400, 0.000230, 0.263000},
	{{0, 6, -7, 0, 0, 0, 0, 0, -1, 1, -1}, 76.279900, 0.000060, 0.102000},
	{{0, 6, -7, 0, 0, 0, 0, 0, -1, 1, 1}, 256.266940, 0.000030, 0.059000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 0, -2, 0}, 93.696520, 0.000040, 0.042000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 0, -1, 0}, 144.869160, 0.000370, 0.095000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 0, 0, 0}, 271.207030, 0.003950, 0.363000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 0, 1, 0}, 271.547810, 0.000770, 0.062000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 0, 2, 0}, 270.762200, 0.000040, 0.034000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, -2, 0}, 273.164740, 0.000030, 0.088000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, -1, -1}, 76.254200, 0.000370, 0.087000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, -1, 0}, 91.861020, 0.000020, 0.536000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, -1, 1}, 258.794420, 0.000160, 0.065000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, 0, -1}, 76.262260, 0.001600, 0.588000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, 0, 0}, 91.457190, 0.000110, 0.066000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, 0, 1}, 255.619040, 0.000100, 0.035000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, 1, -1}, 76.274340, 0.000180, 0.067000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, 1, 1}, 255.995440, 0.000010, 0.024000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 1, 2, 1}, 76.296130, 0.000020, 0.018000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, -2, 0}, 260.739630, 0.000200, 1.021000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, -1, -2}, 272.265960, 0.000020, 0.079000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, -1, -1}, 262.586580, 0.000010, 1.231000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, -1, 0}, 270.786300, 0.008110, 0.070000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, 0, -2}, 272.265960, 0.000310, 1.549000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, 0, -1}, 256.249240, 0.000080, 0.071000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, 0, 0}, 94.322260, 0.001400, 0.036000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, 1, -2}, 272.265960, 0.000020, 0.072000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, 1, 0}, 94.215700, 0.000190, 0.025000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 2, 2, 0}, 90.147630, 0.000040, 0.019000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 3, -3, -1}, 256.284220, 0.000010, 0.075000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 3, -3, 0}, 272.443110, 0.000020, 10.856000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 3, -2, -1}, 256.217800, 0.000210, 13.397000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 3, -2, 0}, 92.341340, 0.000120, 0.075000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 3, -1, -1}, 256.218730, 0.000760, 0.076000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 3, 0, -1}, 256.131730, 0.000020, 0.038000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 3, 0, 0}, 92.442290, 0.000020, 0.025000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 4, -2, 0}, 272.142280, 0.000130, 0.039000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 4, -1, 0}, 274.547300, 0.000070, 0.026000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 4, 0, 0}, 98.952620, 0.000070, 0.019000},
	{{0, 6, -7, 0, 0, 0, 0, 0, 4, 1, 0}, 92.585470, 0.000010, 0.015000},
	{{0, 6, -7, 0, 1, 0, 0, 0, -2, 0, 0}, 192.896310, 0.000090, 0.046000},
	{{0, 6, -7, 0, 1, 0, 0, 0, 0, -2, 0}, 358.837090, 0.000020, 0.042000},
	{{0, 6, -7, 0, 1, 0, 0, 0, 0, -1, 0}, 178.698430, 0.000010, 0.096000},
	{{0, 6, -7, 0, 1, 0, 0, 0, 0, 0, 0}, 351.203670, 0.000030, 0.352000},
	{{0, 6, -7, 0, 1, 0, 0, 0, 0, 1, 0}, 178.698430, 0.000010, 0.062000},
	{{0, 6, -7, 0, 1, 0, 0, 0, 2, -2, 0}, 178.615770, 0.000020, 0.940000},
	{{0, 6, -7, 0, 1, 0, 0, 0, 2, 0, 0}, 178.615770, 0.000020, 0.036000},
	{{0, 6, -7, 0, 2, 0, 0, 0, 2, -2, 0}, 60.185760, 0.000020, 0.871000},
	{{0, 6, -7, 0, 2, 0, 0, 0, 2, 0, 0}, 60.185760, 0.000020, 0.036000},
	{{0, 6, -7, 2, 0, 0, 0, 0, 2, -2, 0}, 43.648680, 0.000010, 0.490000},
	{{0, 6, -7, 2, 0, 0, 0, 0, 2, 0, 0}, 43.648680, 0.000010, 0.035000},
	{{0, 6, -7, 3, 0, 0, 0, 0, -2, 0, 0}, 236.412560, 0.000020, 0.049000},
	{{0, 6, -7, 3, 0, 0, 0, 0, -2, 2, 0}, 236.412560, 0.000020, 0.163000},
	{{0, 6, -6, -7, 0, 0, 0, 0, 0, -1, 0}, 144.208470, 0.000010, 0.076000},
	{{0, 6, -6, -7, 0, 0, 0, 0, 0, 1, 0}, 144.208470, 0.000010, 0.075000},
	{{0, 6, -6, 0, -2, 0, 0, 0, -2, 0, 0}, 205.430800, 0.000010, 0.047000},
	{{0, 6, -6, 0, -1, 0, 0, 0, -2, 0, 0}, 188.791980, 0.000080, 0.047000},
	{{0, 6, -6, 0, -1, 0, 0, 0, -2, 2, 0}, 185.547450, 0.000010, 0.184000},
	{{0, 6, -6, 0, -1, 0, 0, 0, 0, -2, 0}, 193.047610, 0.000010, 0.044000},
	{{0, 6, -6, 0, -1, 0, 0, 0, 0, 0, 0}, 13.047610, 0.000010, 0.273000},
	{{0, 6, -6, 0, -1, 0, 0, 0, 2, 0, 0}, 5.649580, 0.000030, 0.035000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -4, 2, 0}, 341.619260, 0.000010, 0.052000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -2, -1, 0}, 178.504580, 0.000010, 0.029000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -2, 0, 0}, 348.464140, 0.000040, 0.048000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -2, 0, 2}, 359.693220, 0.000010, 0.171000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -2, 1, 0}, 351.641090, 0.000050, 0.129000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -2, 2, 0}, 356.914320, 0.000100, 0.181000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -2, 3, 0}, 0.209870, 0.000020, 0.053000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -1, 0, 0}, 359.981810, 0.000180, 0.116000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -1, 0, 1}, 13.324980, 0.000030, 0.208000},
	{{0, 6, -6, 0, 0, 0, 0, 0, -1, 1, 0}, 0.420260, 0.000030, 0.216000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 0, -1, 0}, 359.521720, 0.000080, 0.105000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 0, 0, -2}, 179.816080, 0.000010, 0.043000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 0, 0, 0}, 180.021650, 0.022630, 0.266000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 0, 0, 2}, 0.087370, 0.000020, 0.033000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 0, 1, -2}, 180.014530, 0.000020, 0.102000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 0, 1, 0}, 180.031690, 0.003690, 0.059000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 0, 2, 0}, 179.993800, 0.000310, 0.033000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 0, 3, 0}, 179.980520, 0.000020, 0.023000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 1, -1, -1}, 190.387250, 0.000010, 0.095000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 1, -1, 0}, 0.013320, 0.000050, 0.349000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 1, 0, -1}, 190.387250, 0.000210, 0.370000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 1, 0, 0}, 0.020950, 0.000410, 0.062000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 1, 1, -1}, 190.387250, 0.000010, 0.063000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 1, 1, 0}, 359.960520, 0.000010, 0.034000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, -3, 0}, 1.636860, 0.000040, 0.089000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, -2, 0}, 180.033300, 0.000800, 0.505000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, -1, -2}, 180.014690, 0.000030, 0.086000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, -1, 0}, 180.045740, 0.021160, 0.066000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, -1, 2}, 0.022490, 0.000040, 0.024000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, 0, -2}, 180.035410, 0.000670, 0.608000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, 0, 0}, 0.065810, 0.008480, 0.035000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, 0, 2}, 180.024220, 0.000030, 0.018000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, 1, -2}, 0.029860, 0.000030, 0.067000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, 1, 0}, 0.030270, 0.000700, 0.024000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 2, 2, 0}, 0.014200, 0.000090, 0.018000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 3, -2, 0}, 0.066330, 0.000050, 0.070000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 3, -1, 0}, 180.065870, 0.000010, 0.036000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 3, 0, -2}, 0.068230, 0.000030, 0.071000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 3, 0, 0}, 0.079360, 0.000030, 0.024000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 3, 1, 0}, 0.065690, 0.000010, 0.018000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 4, -3, 0}, 180.576540, 0.000030, 0.074000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 4, -2, 0}, 180.018010, 0.000300, 0.037000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 4, -1, -2}, 180.033090, 0.000010, 0.076000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 4, -1, 0}, 179.995080, 0.000040, 0.025000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 4, 0, 0}, 0.025110, 0.000100, 0.019000},
	{{0, 6, -6, 0, 0, 0, 0, 0, 4, 1, 0}, 0.028310, 0.000040, 0.015000},
	{{0, 6, -6, 0, 1, 0, 0, 0, -2, 0, 0}, 346.059120, 0.000050, 0.048000},
	{{0, 6, -6, 0, 1, 0, 0, 0, 0, 0, 0}, 165.720370, 0.000030, 0.261000},
	{{0, 6, -6, 0, 1, 0, 0, 0, 2, 0, 0}, 166.050790, 0.000020, 0.035000},
	{{0, 6, -5, -2, 0, 0, 0, 0, 2, -2, 0}, 139.466620, 0.000010, 0.522000},
	{{0, 6, -5, -2, 0, 0, 0, 0, 2, 0, 0}, 139.466620, 0.000010, 0.035000},
	{{0, 6, -5, 0, -2, 0, 0, 0, 0, 0, 0}, 276.819500, 0.000020, 0.218000},
	{{0, 6, -5, 0, -1, 0, 0, 0, -2, -2, 0}, 1.270340, 0.000010, 0.021000},
	{{0, 6, -5, 0, -1, 0, 0, 0, -2, 2, 0}, 178.668670, 0.000010, 0.155000},
	{{0, 6, -5, 0, -1, 0, 0, 0, 0, 0, 0}, 181.506690, 0.000080, 0.214000},
	{{0, 6, -5, 0, -1, 0, 0, 0, 2, -3, 0}, 179.107110, 0.000010, 0.097000},
	{{0, 6, -5, 0, -1, 0, 0, 0, 2, -2, 0}, 3.886400, 0.000020, 0.345000},
	{{0, 6, -5, 0, -1, 0, 0, 0, 2, -1, 0}, 359.107110, 0.000010, 0.062000},
	{{0, 6, -5, 0, -1, 0, 0, 0, 2, 0, 0}, 1.973360, 0.000030, 0.034000},
	{{0, 6, -5, 0, 0, -1, 0, 0, -2, 0, 0}, 180.206970, 0.000010, 0.050000},
	{{0, 6, -5, 0, 0, -1, 0, 0, -2, 2, 0}, 0.206970, 0.000010, 0.154000},
	{{0, 6, -5, 0, 0, 0, 0, 0, -3, 0, 1}, 103.279350, 0.000020, 0.053000},
	{{0, 6, -5, 0, 0, 0, 0, 0, -2, 0, 0}, 118.005210, 0.000030, 0.050000},
	{{0, 6, -5, 0, 0, 0, 0, 0, -2, 1, 0}, 252.052670, 0.000080, 0.149000},
	{{0, 6, -5, 0, 0, 0, 0, 0, -2, 2, 0}, 275.801680, 0.000030, 0.153000},
	{{0, 6, -5, 0, 0, 0, 0, 0, -1, -1, -1}, 103.283830, 0.000030, 0.029000},
	{{0, 6, -5, 0, 0, 0, 0, 0, -1, -1, 1}, 283.174990, 0.000010, 0.134000},
	{{0, 6, -5, 0, 0, 0, 0, 0, -1, 0, -1}, 103.281080, 0.000230, 0.048000},
	{{0, 6, -5, 0, 0, 0, 0, 0, -1, 0, 1}, 283.275800, 0.000290, 0.172000},
	{{0, 6, -5, 0, 0, 0, 0, 0, -1, 1, -1}, 103.285190, 0.000090, 0.128000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 0, -2, 0}, 252.106800, 0.000020, 0.046000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 0, -1, 0}, 246.956650, 0.000080, 0.118000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 0, 0, -1}, 283.300820, 0.000010, 0.115000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 0, 0, 0}, 93.483170, 0.000130, 0.210000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 0, 1, 0}, 73.351760, 0.000030, 0.056000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 1, -1, -1}, 103.316290, 0.000170, 0.105000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 1, -1, 1}, 283.321820, 0.000080, 0.058000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 1, 0, -1}, 103.323970, 0.001060, 0.270000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 1, 0, 1}, 283.317250, 0.000040, 0.033000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 1, 1, -1}, 103.319990, 0.000110, 0.059000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 1, 1, 1}, 103.321490, 0.000010, 0.023000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 2, -2, 0}, 251.752080, 0.000030, 0.336000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 2, -1, 0}, 266.765750, 0.000390, 0.062000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 2, 0, -2}, 259.899240, 0.000010, 0.378000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 2, 0, -1}, 283.352270, 0.000020, 0.062000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 2, 0, 0}, 87.282460, 0.000330, 0.034000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 2, 1, 0}, 84.065300, 0.000060, 0.023000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 2, 2, 0}, 77.375230, 0.000010, 0.018000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 3, -1, -1}, 99.938700, 0.000030, 0.066000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 3, 0, -1}, 103.323160, 0.000020, 0.035000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 4, -2, 0}, 72.965100, 0.000010, 0.036000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 4, -1, 0}, 86.126140, 0.000020, 0.024000},
	{{0, 6, -5, 0, 0, 0, 0, 0, 4, 0, 0}, 81.565860, 0.000020, 0.018000},
	{{0, 6, -5, 0, 1, 0, 0, 0, 2, -2, 0}, 151.660280, 0.000010, 0.326000},
	{{0, 6, -5, 0, 1, 0, 0, 0, 2, 0, 0}, 331.660280, 0.000010, 0.034000},
	{{0, 6, -4, -3, 0, 0, 0, 0, -2, 0, 0}, 336.218160, 0.000010, 0.049000},
	{{0, 6, -4, -3, 0, 0, 0, 0, -2, 2, 0}, 336.218160, 0.000010, 0.169000},
	{{0, 6, -4, -2, 0, 0, 0, 0, -2, 0, 0}, 0.592590, 0.000010, 0.050000},
	{{0, 6, -4, -2, 0, 0, 0, 0, -2, 1, 0}, 179.684760, 0.000030, 0.147000},
	{{0, 6, -4, -2, 0, 0, 0, 0, -2, 2, 0}, 0.592590, 0.000010, 0.155000},
	{{0, 6, -4, -2, 0, 0, 0, 0, 0, -2, 0}, 179.885480, 0.000010, 0.046000},
	{{0, 6, -4, 0, -3, 0, 0, 0, -2, 0, 0}, 7.779600, 0.000010, 0.052000},
	{{0, 6, -4, 0, -3, 0, 0, 0, -2, 2, 0}, 7.779600, 0.000010, 0.137000},
	{{0, 6, -4, 0, -2, 0, 0, 0, -2, 0, 0}, 0.304510, 0.000020, 0.052000},
	{{0, 6, -4, 0, -2, 0, 0, 0, -2, 1, 0}, 0.147970, 0.000040, 0.170000},
	{{0, 6, -4, 0, -2, 0, 0, 0, -2, 2, 0}, 0.304510, 0.000020, 0.136000},
	{{0, 6, -4, 0, -2, 0, 0, 0, -2, 3, 0}, 0.097040, 0.000010, 0.049000},
	{{0, 6, -4, 0, -2, 0, 0, 0, 0, -2, 0}, 359.747010, 0.000020, 0.048000},
	{{0, 6, -4, 0, 0, 0, 0, 0, -2, -1, 0}, 27.682590, 0.000030, 0.031000},
	{{0, 6, -4, 0, 0, 0, 0, 0, -2, 0, 0}, 27.927810, 0.000260, 0.053000},
	{{0, 6, -4, 0, 0, 0, 0, 0, -2, 1, 0}, 25.508560, 0.000150, 0.175000},
	{{0, 6, -4, 0, 0, 0, 0, 0, -2, 2, 0}, 24.869780, 0.000020, 0.133000},
	{{0, 6, -4, 0, 0, 0, 0, 0, -1, 0, 0}, 205.797410, 0.000020, 0.151000},
	{{0, 6, -4, 0, 0, 0, 0, 0, -1, 0, 1}, 184.295950, 0.000010, 0.147000},
	{{0, 6, -4, 0, 0, 0, 0, 0, 0, -1, 0}, 25.803790, 0.000170, 0.133000},
	{{0, 6, -4, 0, 0, 0, 0, 0, 0, 0, 0}, 26.147000, 0.000970, 0.174000},
	{{0, 6, -4, 0, 0, 0, 0, 0, 0, 1, 0}, 25.973020, 0.000090, 0.053000},
	{{0, 6, -4, 0, 0, 0, 0, 0, 1, 0, -1}, 3.304710, 0.000010, 0.213000},
	{{0, 6, -4, 0, 0, 0, 0, 0, 1, 0, 0}, 206.390260, 0.000020, 0.055000},
	{{0, 6, -4, 0, 0, 0, 0, 0, 2, -2, 0}, 332.047440, 0.000030, 0.251000},
	{{0, 6, -4, 0, 0, 0, 0, 0, 2, -1, 0}, 28.875620, 0.000150, 0.058000},
	{{0, 6, -4, 0, 0, 0, 0, 0, 2, 0, 0}, 219.073270, 0.000110, 0.033000},
	{{0, 6, -3, 0, -3, 0, 0, 0, -2, 0, 0}, 356.683820, 0.000010, 0.055000},
	{{0, 6, -3, 0, -3, 0, 0, 0, -2, 2, 0}, 176.683820, 0.000010, 0.121000},
	{{0, 6, -3, 0, 0, 0, 0, 0, -2, 1, 0}, 280.412000, 0.000020, 0.211000},
	{{0, 6, -3, 0, 0, 0, 0, 0, 0, 0, 0}, 289.303300, 0.000070, 0.148000},
	{{0, 6, -3, 0, 0, 0, 0, 0, 1, 0, -1}, 308.706080, 0.000010, 0.175000},
	{{0, 6, -3, 0, 1, 0, 0, 0, 2, 0, 0}, 125.383670, 0.000030, 0.032000},
	{{0, 6, -2, 0, -1, 0, 0, 0, 2, -2, 0}, 220.737250, 0.000010, 0.170000},
	{{0, 6, -2, 0, -1, 0, 0, 0, 2, 0, 0}, 151.802260, 0.000080, 0.031000},
	{{0, 6, -2, 0, 0, 0, 0, 0, 0, -1, 0}, 128.239430, 0.000030, 0.182000},
	{{0, 6, -2, 0, 0, 0, 0, 0, 0, 0, 0}, 159.371590, 0.000010, 0.129000},
	{{0, 6, -2, 0, 0, 0, 0, 0, 0, 1, 0}, 128.239430, 0.000030, 0.048000},
	{{0, 6, -2, 0, 0, 0, 0, 0, 0, 2, 0}, 184.039560, 0.000010, 0.029000},
	{{0, 6, -2, 0, 0, 0, 0, 0, 2, -2, 0}, 127.835100, 0.000010, 0.167000},
	{{0, 6, -2, 0, 0, 0, 0, 0, 2, 0, 0}, 128.085970, 0.000030, 0.031000},
	{{0, 6, -2, 0, 0, 0, 0, 0, 2, 2, 0}, 128.212270, 0.000020, 0.017000},
	{{0, 6, -2, 0, 0, 0, 0, 0, 4, 0, 0}, 308.008430, 0.000020, 0.017000},
	{{0, 6, -2, 0, 1, 0, 0, 0, 2, 0, 0}, 298.774120, 0.000080, 0.031000},
	{{0, 6, -1, 0, -1, 0, 0, 0, 0, 0, 0}, 129.522630, 0.000020, 0.115000},
	{{0, 6, -1, 0, 0, 0, 0, 0, -2, 0, 0}, 302.033200, 0.000040, 0.063000},
	{{0, 6, -1, 0, 0, 0, 0, 0, -2, 2, 0}, 302.033200, 0.000040, 0.095000},
	{{0, 6, -1, 0, 0, 0, 0, 0, 0, 0, 0}, 79.421720, 0.000010, 0.114000},
	{{0, 6, -1, 0, 0, 0, 0, 0, 2, -2, 0}, 216.984080, 0.000040, 0.143000},
	{{0, 6, -1, 0, 0, 0, 0, 0, 2, 0, 0}, 216.984080, 0.000040, 0.030000},
	{{0, 6, 0, -15, 0, 0, 0, 0, 2, -3, 0}, 323.850910, 0.000010, 0.075000},
	{{0, 6, 0, -15, 0, 0, 0, 0, 2, -1, 0}, 323.850910, 0.000010, 0.075000},
	{{0, 6, 0, -2, 0, 0, 0, 0, 2, -2, 0}, 308.300590, 0.000020, 0.145000},
	{{0, 6, 0, -2, 0, 0, 0, 0, 2, 0, 0}, 308.241780, 0.000030, 0.030000},
	{{0, 6, 0, 0, 0, 0, 0, 0, 2, 0, 0}, 327.133970, 0.000010, 0.029000},
	{{0, 6, 4, 0, -6, 0, 0, 0, 0, -1, 0}, 124.178000, 0.000020, 118.454000},
	{{0, 7, -42, 36, 0, 0, 0, 0, 2, -1, 0}, 306.337640, 0.000010, 968.212000},
	{{0, 7, -19, 11, 0, 0, 0, 0, -2, 1, 0}, 158.556360, 0.000050, 0.075000},
	{{0, 7, -19, 11, 0, 0, 0, 0, -2, 2, 0}, 132.899840, 0.000070, 828.943000},
	{{0, 7, -19, 11, 0, 0, 0, 0, -2, 3, 0}, 158.556360, 0.000050, 0.075000},
	{{0, 7, -16, 0, 0, 0, 0, 0, -2, 0, 0}, 289.614720, 0.000020, 0.034000},
	{{0, 7, -16, 0, 0, 0, 0, 0, -2, 2, 0}, 289.621400, 0.000010, 0.351000},
	{{0, 7, -14, 0, 0, 0, 0, 0, 0, 0, 0}, 305.272080, 0.000020, 0.381000},
	{{0, 7, -14, 0, 2, 0, 0, 0, -2, 0, 0}, 322.847990, 0.000020, 0.037000},
	{{0, 7, -14, 0, 2, 0, 0, 0, -2, 1, 0}, 322.940010, 0.000020, 0.072000},
	{{0, 7, -14, 0, 2, 0, 0, 0, -2, 3, 0}, 142.940010, 0.000020, 0.080000},
	{{0, 7, -14, 2, 0, 0, 0, 0, -2, 1, 0}, 323.059070, 0.000020, 0.077000},
	{{0, 7, -14, 2, 0, 0, 0, 0, -2, 3, 0}, 323.059070, 0.000020, 0.074000},
	{{0, 7, -14, 5, 0, 0, 0, 0, 0, -1, 0}, 289.176360, 0.000030, 0.076000},
	{{0, 7, -14, 5, 0, 0, 0, 0, 0, 1, 0}, 289.176360, 0.000030, 0.075000},
	{{0, 7, -13, 0, -1, 0, 0, 0, -2, 1, 0}, 51.983800, 0.000070, 0.076000},
	{{0, 7, -13, 0, -1, 0, 0, 0, -2, 2, 0}, 230.983890, 0.000010, 14.625000},
	{{0, 7, -13, 0, -1, 0, 0, 0, 0, 1, 0}, 231.983800, 0.000020, 0.087000},
	{{0, 7, -13, 0, 0, 0, 0, 0, -2, -1, 0}, 221.480240, 0.000020, 0.025000},
	{{0, 7, -13, 0, 0, 0, 0, 0, -2, 0, 0}, 219.962420, 0.000020, 0.038000},
	{{0, 7, -13, 0, 0, 0, 0, 0, -2, 1, 0}, 244.050550, 0.000050, 0.076000},
	{{0, 7, -13, 0, 0, 0, 0, 0, -2, 2, 0}, 219.962420, 0.000020, 6.550000},
	{{0, 7, -13, 0, 0, 0, 0, 0, -2, 3, 0}, 100.899810, 0.000020, 0.075000},
	{{0, 7, -13, 0, 0, 0, 0, 0, 0, 0, 0}, 268.196600, 0.000020, 0.617000},
	{{0, 7, -13, 0, 1, 0, 0, 0, -2, 1, 0}, 142.517470, 0.000010, 0.077000},
	{{0, 7, -13, 0, 1, 0, 0, 0, -2, 2, 0}, 141.674680, 0.000010, 4.220000},
	{{0, 7, -13, 0, 1, 0, 0, 0, -2, 3, 0}, 142.517470, 0.000010, 0.074000},
	{{0, 7, -13, 0, 2, 0, 0, 0, -2, 1, 0}, 203.162160, 0.000010, 0.077000},
	{{0, 7, -13, 0, 2, 0, 0, 0, -2, 2, 0}, 51.577780, 0.000020, 3.113000},
	{{0, 7, -13, 0, 2, 0, 0, 0, -2, 3, 0}, 203.162160, 0.000010, 0.074000},
	{{0, 7, -13, 0, 3, 0, 0, 0, -2, 1, 0}, 41.586680, 0.000030, 0.078000},
	{{0, 7, -13, 0, 3, 0, 0, 0, -2, 3, 0}, 41.586680, 0.000030, 0.073000},
	{{0, 7, -13, 2, 0, 0, 0, 0, -2, 0, 0}, 233.726680, 0.000050, 0.040000},
	{{0, 7, -13, 2, 0, 0, 0, 0, -2, 1, 0}, 231.752980, 0.000030, 0.083000},
	{{0, 7, -13, 2, 0, 0, 0, 0, -2, 2, 0}, 231.752980, 0.000030, 0.822000},
	{{0, 7, -13, 2, 0, 0, 0, 0, -2, 3, 0}, 231.752980, 0.000030, 0.069000},
	{{0, 7, -13, 2, 0, 0, 0, 0, 0, -1, 0}, 51.752980, 0.000030, 0.072000},
	{{0, 7, -13, 2, 0, 0, 0, 0, 0, 1, 0}, 51.752980, 0.000030, 0.079000},
	{{0, 7, -13, 3, 0, 0, 0, 0, 0, -1, 0}, 20.157360, 0.000040, 0.075000},
	{{0, 7, -13, 3, 0, 0, 0, 0, 0, 1, 0}, 20.157360, 0.000040, 0.076000},
	{{0, 7, -12, -2, 0, 0, 0, 0, -2, 1, 0}, 1.738670, 0.000010, 0.076000},
	{{0, 7, -12, -2, 0, 0, 0, 0, -2, 3, 0}, 1.738670, 0.000010, 0.075000},
	{{0, 7, -12, 0, 0, 0, 0, 0, -2, 0, 0}, 346.662270, 0.000010, 0.039000},
	{{0, 7, -12, 0, 0, 0, 0, 0, -2, 1, 0}, 357.991460, 0.000060, 0.083000},
	{{0, 7, -12, 0, 0, 0, 0, 0, -2, 2, 0}, 334.223320, 0.000040, 0.868000},
	{{0, 7, -12, 0, 0, 0, 0, 0, -2, 3, 0}, 346.882520, 0.000060, 0.069000},
	{{0, 7, -12, 0, 0, 0, 0, 0, 0, -2, 0}, 173.178510, 0.000010, 0.037000},
	{{0, 7, -12, 0, 0, 0, 0, 0, 0, -1, 0}, 205.745750, 0.000020, 0.072000},
	{{0, 7, -12, 0, 0, 0, 0, 0, 0, 0, 0}, 77.650850, 0.000030, 1.609000},
	{{0, 7, -12, 0, 0, 0, 0, 0, 0, 1, 0}, 192.771600, 0.000030, 0.079000},
	{{0, 7, -12, 0, 0, 0, 0, 0, 0, 2, 0}, 142.715870, 0.000010, 0.039000},
	{{0, 7, -12, 0, 0, 0, 0, 0, 2, -2, 0}, 237.870600, 0.000020, 0.417000},
	{{0, 7, -12, 0, 0, 0, 0, 0, 2, -1, 0}, 237.878030, 0.000020, 0.092000},
	{{0, 7, -12, 0, 1, 0, 0, 0, 0, 0, 0}, 50.435800, 0.000020, 1.862000},
	{{0, 7, -12, 0, 2, 0, 0, 0, -2, 1, 0}, 111.923130, 0.000030, 0.084000},
	{{0, 7, -12, 0, 2, 0, 0, 0, 0, -1, 0}, 291.923130, 0.000020, 0.073000},
	{{0, 7, -12, 0, 2, 0, 0, 0, 0, 1, 0}, 291.923130, 0.000020, 0.078000},
	{{0, 7, -12, 1, 0, 0, 0, 0, 0, -1, 0}, 231.246120, 0.000010, 0.075000},
	{{0, 7, -12, 1, 0, 0, 0, 0, 0, 1, 0}, 231.246120, 0.000010, 0.076000},
	{{0, 7, -12, 2, 0, 0, 0, 0, -2, 0, 0}, 95.386050, 0.000020, 0.041000},
	{{0, 7, -12, 2, 0, 0, 0, 0, -2, 2, 0}, 95.386050, 0.000020, 0.451000},
	{{0, 7, -12, 2, 0, 0, 0, 0, 0, -1, 0}, 95.386050, 0.000010, 0.078000},
	{{0, 7, -12, 2, 0, 0, 0, 0, 0, 1, 0}, 95.386050, 0.000010, 0.073000},
	{{0, 7, -11, -4, 0, 0, 0, 0, -2, 1, 0}, 344.273830, 0.000010, 0.076000},
	{{0, 7, -11, -4, 0, 0, 0, 0, -2, 3, 0}, 344.273830, 0.000010, 0.075000},
	{{0, 7, -11, 0, -1, 0, 0, 0, -2, 0, 0}, 242.173070, 0.000030, 0.041000},
	{{0, 7, -11, 0, 0, 0, 0, 0, -2, 0, 0}, 145.902830, 0.000060, 0.041000},
	{{0, 7, -11, 0, 0, 0, 0, 0, -2, 0, 2}, 231.777430, 0.000020, 0.402000},
	{{0, 7, -11, 0, 0, 0, 0, 0, -2, 1, 0}, 126.373720, 0.000070, 0.090000},
	{{0, 7, -11, 0, 0, 0, 0, 0, -2, 2, 0}, 61.069120, 0.000010, 0.465000},
	{{0, 7, -11, 0, 0, 0, 0, 0, -2, 3, 0}, 51.941400, 0.000030, 0.065000},
	{{0, 7, -11, 0, 0, 0, 0, 0, 0, -1, 0}, 130.281570, 0.000100, 0.078000},
	{{0, 7, -11, 0, 0, 0, 0, 0, 0, 0, 0}, 153.113450, 0.000580, 2.641000},
	{{0, 7, -11, 0, 0, 0, 0, 0, 0, 1, 0}, 130.281570, 0.000100, 0.073000},
	{{0, 7, -11, 0, 0, 0, 0, 0, 1, 0, -1}, 151.660550, 0.000020, 1.481000},
	{{0, 7, -11, 0, 0, 0, 0, 0, 1, 0, 0}, 58.886940, 0.000020, 0.078000},
	{{0, 7, -11, 0, 0, 0, 0, 0, 2, -1, 0}, 172.452570, 0.000030, 0.084000},
	{{0, 7, -11, 0, 0, 0, 0, 0, 2, 0, 0}, 153.035580, 0.000040, 0.040000},
	{{0, 7, -11, 0, 1, 0, 0, 0, -2, 0, 0}, 39.151620, 0.000050, 0.041000},
	{{0, 7, -11, 0, 1, 0, 0, 0, -2, 1, 0}, 218.658540, 0.000030, 0.091000},
	{{0, 7, -11, 0, 1, 0, 0, 0, -2, 2, 0}, 38.657180, 0.000030, 0.447000},
	{{0, 7, -11, 0, 1, 0, 0, 0, 0, -1, 0}, 38.920480, 0.000030, 0.078000},
	{{0, 7, -11, 0, 1, 0, 0, 0, 0, 1, 0}, 38.920480, 0.000030, 0.073000},
	{{0, 7, -10, -2, 0, 0, 0, 0, 0, -1, 0}, 10.983020, 0.000020, 0.077000},
	{{0, 7, -10, -2, 0, 0, 0, 0, 0, 1, 0}, 10.983020, 0.000020, 0.074000},
	{{0, 7, -10, 0, -1, 0, 0, 0, -2, 0, 0}, 235.596530, 0.000010, 0.043000},
	{{0, 7, -10, 0, 0, 0, 0, 0, -4, 0, 0}, 120.982430, 0.000030, 0.021000},
	{{0, 7, -10, 0, 0, 0, 0, 0, -2, 0, 0}, 261.602730, 0.000080, 0.043000},
	{{0, 7, -10, 0, 0, 0, 0, 0, -2, 1, 0}, 254.318650, 0.000110, 0.099000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 0, -1, 0}, 72.522350, 0.000110, 0.084000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 0, 0, 0}, 49.025180, 0.000100, 0.725000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 0, 1, 0}, 254.347080, 0.000010, 0.068000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 0, 2, 0}, 72.637430, 0.000010, 0.036000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 1, -1, -1}, 248.426460, 0.000020, 0.077000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 1, 0, -1}, 248.426460, 0.000360, 3.078000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 1, 0, 0}, 250.088950, 0.000030, 0.073000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 1, 0, 1}, 68.379610, 0.000040, 0.037000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 1, 1, -1}, 248.426460, 0.000020, 0.074000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 1, 2, -3}, 67.854810, 0.000010, 116.052000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 2, -3, 0}, 42.119260, 0.000010, 0.073000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 2, -2, 0}, 254.512570, 0.000060, 2.529000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 2, -1, 0}, 253.773280, 0.000640, 0.078000},
	{{0, 7, -10, 0, 0, 0, 0, 0, 2, 0, 0}, 253.813950, 0.000140, 0.038000},
	{{0, 7, -10, 0, 1, 0, 0, 0, -2, 0, 0}, 24.477210, 0.000010, 0.043000},
	{{0, 7, -10, 0, 1, 0, 0, 0, -2, 2, 0}, 204.477210, 0.000010, 0.309000},
	{{0, 7, -10, 0, 1, 0, 0, 0, 2, -3, 0}, 169.252740, 0.000020, 0.074000},
	{{0, 7, -10, 0, 1, 0, 0, 0, 2, -1, 0}, 169.252740, 0.000020, 0.077000},
	{{0, 7, -10, 0, 2, 0, 0, 0, 2, -3, 0}, 91.242940, 0.000010, 0.074000},
	{{0, 7, -10, 0, 2, 0, 0, 0, 2, -1, 0}, 91.242940, 0.000010, 0.077000},
	{{0, 7, -10, 0, 2, 0, 0, 0, 2, 0, 0}, 271.100270, 0.000020, 0.038000},
	{{0, 7, -10, 0, 3, 0, 0, 0, 2, -3, 0}, 81.412010, 0.000020, 0.075000},
	{{0, 7, -10, 0, 3, 0, 0, 0, 2, -1, 0}, 214.097140, 0.000060, 0.076000},
	{{0, 7, -10, 0, 4, 0, 0, 0, 2, -1, 0}, 172.193320, 0.000030, 0.076000},
	{{0, 7, -10, 2, 0, 0, 0, 0, 2, -3, 0}, 271.311830, 0.000010, 0.079000},
	{{0, 7, -10, 2, 0, 0, 0, 0, 2, -1, 0}, 271.311830, 0.000010, 0.072000},
	{{0, 7, -9, 0, 0, 0, 0, 0, -2, 0, 0}, 171.552450, 0.000060, 0.045000},
	{{0, 7, -9, 0, 0, 0, 0, 0, -2, 1, 0}, 165.246090, 0.000070, 0.110000},
	{{0, 7, -9, 0, 0, 0, 0, 0, -2, 2, 0}, 225.481960, 0.000010, 0.241000},
	{{0, 7, -9, 0, 0, 0, 0, 0, -2, 3, 0}, 356.805630, 0.000010, 0.057000},
	{{0, 7, -9, 0, 0, 0, 0, 0, -1, 0, -1}, 168.278520, 0.000010, 0.043000},
	{{0, 7, -9, 0, 0, 0, 0, 0, -1, 0, 1}, 6.320910, 0.000040, 0.291000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 0, -1, 0}, 171.278100, 0.000290, 0.092000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 0, 0, 0}, 342.003750, 0.001010, 0.420000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 0, 1, 0}, 345.545390, 0.000180, 0.064000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 1, -3, 1}, 168.938610, 0.000020, 0.086000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 1, -1, -1}, 168.924850, 0.000040, 0.084000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 1, -1, 1}, 348.938610, 0.000020, 0.067000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 1, 0, -1}, 168.984610, 0.000370, 0.755000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 1, 0, 0}, 163.217850, 0.000040, 0.068000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 1, 0, 1}, 348.938580, 0.000030, 0.035000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 1, 1, -1}, 168.924850, 0.000040, 0.069000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 2, -3, 0}, 351.316440, 0.000060, 0.079000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 2, -2, 0}, 331.069740, 0.000010, 1.654000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 2, -1, 0}, 348.849700, 0.002380, 0.072000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 2, 0, -2}, 350.086970, 0.000150, 3.688000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 2, 0, -1}, 349.068060, 0.000030, 0.073000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 2, 0, 0}, 174.398740, 0.000230, 0.037000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 2, 1, 0}, 168.914410, 0.000040, 0.025000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 4, -3, 0}, 171.792390, 0.000010, 0.083000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 4, -2, 0}, 347.897120, 0.000040, 0.039000},
	{{0, 7, -9, 0, 0, 0, 0, 0, 4, 0, 0}, 169.730180, 0.000030, 0.019000},
	{{0, 7, -9, 0, 1, 0, 0, 0, -2, 0, 0}, 270.394750, 0.000010, 0.045000},
	{{0, 7, -9, 0, 1, 0, 0, 0, -2, 2, 0}, 90.394750, 0.000010, 0.236000},
	{{0, 7, -9, 0, 1, 0, 0, 0, 0, 0, 0}, 90.970020, 0.000010, 0.406000},
	{{0, 7, -9, 0, 1, 0, 0, 0, 2, -3, 0}, 269.896320, 0.000020, 0.080000},
	{{0, 7, -9, 0, 1, 0, 0, 0, 2, -2, 0}, 89.926960, 0.000020, 1.452000},
	{{0, 7, -9, 0, 1, 0, 0, 0, 2, -1, 0}, 269.896320, 0.000020, 0.072000},
	{{0, 7, -9, 0, 1, 0, 0, 0, 2, 0, 0}, 89.926960, 0.000020, 0.037000},
	{{0, 7, -9, 0, 2, 0, 0, 0, -2, 0, 0}, 179.883640, 0.000010, 0.045000},
	{{0, 7, -9, 0, 2, 0, 0, 0, -2, 2, 0}, 359.883640, 0.000010, 0.231000},
	{{0, 7, -9, 0, 2, 0, 0, 0, 2, -3, 0}, 179.922690, 0.000010, 0.080000},
	{{0, 7, -9, 0, 2, 0, 0, 0, 2, -1, 0}, 179.922690, 0.000010, 0.071000},
	{{0, 7, -9, 0, 2, 0, 0, 0, 2, 0, 0}, 359.843790, 0.000030, 0.037000},
	{{0, 7, -9, 0, 3, 0, 0, 0, 2, -1, 0}, 350.110120, 0.000030, 0.071000},
	{{0, 7, -9, 2, 0, 0, 0, 0, 2, -2, 0}, 0.159300, 0.000020, 0.600000},
	{{0, 7, -9, 4, 0, 0, 0, 0, 0, 0, 0}, 57.643510, 0.000010, 0.222000},
	{{0, 7, -8, -3, 0, 0, 0, 0, 2, -2, 0}, 350.566940, 0.000030, 104.252000},
	{{0, 7, -8, 0, -1, 0, 0, 0, -2, 0, 0}, 104.472090, 0.000020, 0.047000},
	{{0, 7, -8, 0, 0, 0, 0, 0, -2, 0, 0}, 83.231380, 0.000020, 0.047000},
	{{0, 7, -8, 0, 0, 0, 0, 0, -2, 1, 0}, 90.526990, 0.000160, 0.123000},
	{{0, 7, -8, 0, 0, 0, 0, 0, -2, 2, 0}, 275.096810, 0.000010, 0.194000},
	{{0, 7, -8, 0, 0, 0, 0, 0, -1, -1, 1}, 76.151140, 0.000020, 0.113000},
	{{0, 7, -8, 0, 0, 0, 0, 0, -1, 0, -1}, 76.124000, 0.000070, 0.045000},
	{{0, 7, -8, 0, 0, 0, 0, 0, -1, 0, 0}, 93.674820, 0.000040, 0.111000},
	{{0, 7, -8, 0, 0, 0, 0, 0, -1, 0, 1}, 256.102380, 0.000140, 0.226000},
	{{0, 7, -8, 0, 0, 0, 0, 0, -1, 1, -1}, 76.139680, 0.000050, 0.109000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 0, -1, 0}, 89.652010, 0.000250, 0.101000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 0, 0, 0}, 271.722930, 0.003030, 0.296000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 0, 1, 0}, 271.814230, 0.000590, 0.060000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 0, 2, 0}, 271.634620, 0.000040, 0.033000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 1, -1, -1}, 76.144350, 0.000140, 0.091000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 1, -1, 0}, 91.920880, 0.000010, 0.401000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 1, -1, 1}, 256.148930, 0.000070, 0.063000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 1, 0, -1}, 76.144960, 0.000970, 0.430000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 1, 0, 0}, 90.992290, 0.000070, 0.064000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 1, 0, 1}, 256.120830, 0.000020, 0.034000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 1, 1, -1}, 76.142240, 0.000100, 0.064000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 2, -3, 0}, 270.858100, 0.000030, 0.086000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 2, -2, 0}, 270.957820, 0.000090, 0.623000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 2, -1, 0}, 272.009950, 0.004220, 0.067000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 2, 0, -2}, 272.175870, 0.000140, 0.787000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 2, 0, -1}, 258.113720, 0.000040, 0.068000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 2, 0, 0}, 93.510510, 0.001470, 0.036000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 2, 1, 0}, 93.925940, 0.000200, 0.024000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 2, 2, 0}, 92.344540, 0.000030, 0.018000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 3, -1, -1}, 84.029250, 0.000090, 0.072000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 3, 0, -1}, 76.144410, 0.000010, 0.037000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 3, 0, 0}, 92.436200, 0.000020, 0.025000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 4, -2, 0}, 273.500690, 0.000070, 0.038000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 4, -1, 0}, 261.871960, 0.000010, 0.025000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 4, 0, 0}, 98.443850, 0.000070, 0.019000},
	{{0, 7, -8, 0, 0, 0, 0, 0, 4, 1, 0}, 92.566360, 0.000010, 0.015000},
	{{0, 7, -8, 0, 1, 0, 0, 0, -2, 0, 0}, 194.210630, 0.000080, 0.047000},
	{{0, 7, -8, 0, 1, 0, 0, 0, 0, 0, 0}, 11.584500, 0.000020, 0.289000},
	{{0, 7, -8, 0, 1, 0, 0, 0, 2, -2, 0}, 178.563950, 0.000010, 0.592000},
	{{0, 7, -8, 0, 1, 0, 0, 0, 2, 0, 0}, 178.652680, 0.000040, 0.035000},
	{{0, 7, -8, 0, 2, 0, 0, 0, 2, 0, 0}, 60.175410, 0.000030, 0.035000},
	{{0, 7, -8, 0, 2, 0, 0, 0, 4, -4, 0}, 60.175410, 0.000130, 1309.823000},
	{{0, 7, -8, 2, 0, 0, 0, 0, -2, 1, 0}, 40.495810, 0.000010, 0.142000},
	{{0, 7, -8, 3, 0, 0, 0, 0, -2, 0, 0}, 236.402220, 0.000020, 0.051000},
	{{0, 7, -8, 3, 0, 0, 0, 0, -2, 2, 0}, 236.402220, 0.000020, 0.148000},
	{{0, 7, -7, 0, -2, 0, 0, 0, 0, 0, 0}, 194.589680, 0.000020, 0.238000},
	{{0, 7, -7, 0, -1, 0, 0, 0, -2, 0, 0}, 187.378030, 0.000070, 0.049000},
	{{0, 7, -7, 0, -1, 0, 0, 0, -2, 2, 0}, 185.561290, 0.000010, 0.165000},
	{{0, 7, -7, 0, -1, 0, 0, 0, 0, 0, 0}, 161.148440, 0.000020, 0.233000},
	{{0, 7, -7, 0, -1, 0, 0, 0, 2, -1, 0}, 188.701490, 0.000030, 0.063000},
	{{0, 7, -7, 0, -1, 0, 0, 0, 2, 0, 0}, 5.578360, 0.000020, 0.034000},
	{{0, 7, -7, 0, 0, 0, 0, 0, -4, 2, 0}, 179.801360, 0.000020, 0.054000},
	{{0, 7, -7, 0, 0, 0, 0, 0, -2, -1, 0}, 180.994340, 0.000040, 0.030000},
	{{0, 7, -7, 0, 0, 0, 0, 0, -2, 0, 0}, 181.037950, 0.000310, 0.049000},
	{{0, 7, -7, 0, 0, 0, 0, 0, -2, 1, 0}, 180.851180, 0.000230, 0.141000},
	{{0, 7, -7, 0, 0, 0, 0, 0, -2, 2, 0}, 359.736700, 0.000100, 0.163000},
	{{0, 7, -7, 0, 0, 0, 0, 0, -2, 3, 0}, 359.757320, 0.000020, 0.052000},
	{{0, 7, -7, 0, 0, 0, 0, 0, -1, 0, 0}, 359.956780, 0.000140, 0.125000},
	{{0, 7, -7, 0, 0, 0, 0, 0, -1, 0, 1}, 17.560190, 0.000020, 0.184000},
	{{0, 7, -7, 0, 0, 0, 0, 0, -1, 1, 0}, 0.051430, 0.000020, 0.190000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 0, -2, 0}, 179.292960, 0.000020, 0.045000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 0, -1, 0}, 181.431560, 0.000410, 0.113000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 0, 0, -2}, 179.924210, 0.000010, 0.045000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 0, 0, 0}, 179.985870, 0.015470, 0.228000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 0, 0, 2}, 0.023740, 0.000020, 0.032000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 0, 1, 0}, 180.077800, 0.002580, 0.057000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 0, 2, 0}, 180.182030, 0.000200, 0.032000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 0, 3, 0}, 180.130160, 0.000020, 0.023000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 1, -1, 0}, 0.321240, 0.000050, 0.286000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 1, 0, -1}, 190.864770, 0.000170, 0.301000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 1, 0, 0}, 0.018870, 0.000230, 0.060000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 1, 1, 0}, 359.959620, 0.000010, 0.033000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, -4, 0}, 0.021430, 0.000010, 0.042000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, -3, 0}, 0.078500, 0.000030, 0.094000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, -2, 0}, 180.201080, 0.000550, 0.384000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, -1, -2}, 180.023170, 0.000040, 0.091000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, -1, 0}, 180.051520, 0.012900, 0.063000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, -1, 2}, 0.030960, 0.000030, 0.023000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, 0, -2}, 180.029630, 0.000380, 0.440000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, 0, 0}, 0.047120, 0.007080, 0.034000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, 0, 2}, 180.009810, 0.000030, 0.018000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, 1, 0}, 0.005670, 0.000590, 0.024000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 2, 2, 0}, 359.995260, 0.000080, 0.018000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 3, -2, 0}, 0.011160, 0.000030, 0.067000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 3, -1, 0}, 180.062700, 0.000010, 0.035000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 3, 0, 0}, 0.076450, 0.000030, 0.024000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 3, 1, 0}, 0.062510, 0.000010, 0.018000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 4, -2, 0}, 179.903960, 0.000180, 0.037000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 4, -1, 0}, 359.927590, 0.000010, 0.025000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 4, 0, 0}, 0.004040, 0.000090, 0.019000},
	{{0, 7, -7, 0, 0, 0, 0, 0, 4, 1, 0}, 0.017320, 0.000030, 0.015000},
	{{0, 7, -7, 0, 1, 0, 0, 0, -2, 0, 0}, 346.204780, 0.000050, 0.049000},
	{{0, 7, -7, 0, 1, 0, 0, 0, 0, -2, 0}, 351.250590, 0.000010, 0.045000},
	{{0, 7, -7, 0, 1, 0, 0, 0, 0, 0, 0}, 171.250590, 0.000010, 0.224000},
	{{0, 7, -7, 0, 1, 0, 0, 0, 2, -1, 0}, 346.910820, 0.000020, 0.063000},
	{{0, 7, -6, -2, 0, 0, 0, 0, 2, -2, 0}, 139.456270, 0.000010, 0.394000},
	{{0, 7, -6, -2, 0, 0, 0, 0, 2, 0, 0}, 139.456270, 0.000010, 0.034000},
	{{0, 7, -6, 0, -2, 0, 0, 0, -2, 0, 0}, 108.077590, 0.000010, 0.051000},
	{{0, 7, -6, 0, -2, 0, 0, 0, -2, 2, 0}, 108.077590, 0.000010, 0.143000},
	{{0, 7, -6, 0, -2, 0, 0, 0, 0, 0, 0}, 277.935010, 0.000010, 0.192000},
	{{0, 7, -6, 0, -1, 0, 0, 0, -2, -2, 0}, 181.264010, 0.000010, 0.022000},
	{{0, 7, -6, 0, -1, 0, 0, 0, -2, 2, 0}, 178.636090, 0.000010, 0.141000},
	{{0, 7, -6, 0, -1, 0, 0, 0, 0, 0, 0}, 181.508850, 0.000080, 0.189000},
	{{0, 7, -6, 0, -1, 0, 0, 0, 2, 0, 0}, 3.848810, 0.000030, 0.033000},
	{{0, 7, -6, 0, 0, -1, 0, 0, -2, 0, 0}, 180.200630, 0.000010, 0.052000},
	{{0, 7, -6, 0, 0, -1, 0, 0, -2, 2, 0}, 0.200630, 0.000010, 0.140000},
	{{0, 7, -6, 0, 0, -1, 0, 0, 0, 0, 0}, 180.190920, 0.000010, 0.187000},
	{{0, 7, -6, 0, 0, 0, 0, 0, -3, 0, 1}, 103.266020, 0.000020, 0.055000},
	{{0, 7, -6, 0, 0, 0, 0, 0, -2, 0, 0}, 85.009200, 0.000040, 0.052000},
	{{0, 7, -6, 0, 0, 0, 0, 0, -2, 1, 0}, 257.104780, 0.000030, 0.164000},
	{{0, 7, -6, 0, 0, 0, 0, 0, -2, 2, 0}, 287.654310, 0.000030, 0.140000},
	{{0, 7, -6, 0, 0, 0, 0, 0, -1, -1, -1}, 103.272120, 0.000020, 0.030000},
	{{0, 7, -6, 0, 0, 0, 0, 0, -1, 0, -1}, 103.265060, 0.000180, 0.049000},
	{{0, 7, -6, 0, 0, 0, 0, 0, -1, 0, 1}, 283.163140, 0.000220, 0.155000},
	{{0, 7, -6, 0, 0, 0, 0, 0, -1, 1, -1}, 103.268970, 0.000070, 0.140000},
	{{0, 7, -6, 0, 0, 0, 0, 0, -1, 1, 1}, 103.315140, 0.000010, 0.051000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 0, -1, 0}, 238.177580, 0.000030, 0.127000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 0, 0, -1}, 283.290410, 0.000010, 0.124000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 0, 0, 0}, 103.870080, 0.000030, 0.186000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 0, 1, 0}, 81.953300, 0.000030, 0.054000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 0, 2, 0}, 353.012590, 0.000010, 0.031000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 1, -1, -1}, 103.309320, 0.000140, 0.112000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 1, -1, 1}, 283.315700, 0.000060, 0.056000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 1, 0, -1}, 103.317930, 0.000810, 0.231000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 1, 1, -1}, 103.313590, 0.000090, 0.057000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 1, 1, 1}, 103.315360, 0.000010, 0.023000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 2, -2, 0}, 258.704840, 0.000010, 0.277000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 2, -1, 0}, 269.105040, 0.000290, 0.059000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 2, 0, -2}, 261.801230, 0.000010, 0.306000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 2, 0, -1}, 283.350770, 0.000020, 0.060000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 2, 0, 0}, 89.199620, 0.000330, 0.033000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 2, 1, 0}, 85.488050, 0.000050, 0.023000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 3, 0, -1}, 103.316030, 0.000020, 0.034000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 4, -1, 0}, 87.865490, 0.000020, 0.024000},
	{{0, 7, -6, 0, 0, 0, 0, 0, 4, 0, 0}, 81.801140, 0.000030, 0.018000},
	{{0, 7, -5, -3, 0, 0, 0, 0, -2, 0, 0}, 336.215400, 0.000010, 0.050000},
	{{0, 7, -5, -3, 0, 0, 0, 0, -2, 2, 0}, 336.215400, 0.000010, 0.152000},
	{{0, 7, -5, -2, 0, 0, 0, 0, -2, 0, 0}, 0.566700, 0.000010, 0.051000},
	{{0, 7, -5, -2, 0, 0, 0, 0, -2, 2, 0}, 0.566700, 0.000010, 0.141000},
	{{0, 7, -5, -2, 0, 0, 0, 0, 0, -2, 0}, 179.879140, 0.000010, 0.047000},
	{{0, 7, -5, 0, -3, 0, 0, 0, -2, 0, 0}, 7.675810, 0.000010, 0.054000},
	{{0, 7, -5, 0, -3, 0, 0, 0, -2, 2, 0}, 7.675810, 0.000010, 0.127000},
	{{0, 7, -5, 0, -2, 0, 0, 0, -2, 2, 0}, 0.298930, 0.000020, 0.125000},
	{{0, 7, -5, 0, -2, 0, 0, 0, 0, -2, 0}, 0.116390, 0.000030, 0.049000},
	{{0, 7, -5, 0, -2, 0, 0, 0, 0, 0, 0}, 359.919450, 0.000070, 0.161000},
	{{0, 7, -5, 0, -2, 0, 0, 0, 2, -2, 0}, 179.984220, 0.000010, 0.225000},
	{{0, 7, -5, 0, -1, 0, 0, 0, 0, 0, 0}, 89.241180, 0.000010, 0.159000},
	{{0, 7, -5, 0, 0, 0, 0, 0, -2, -1, 0}, 26.839170, 0.000030, 0.032000},
	{{0, 7, -5, 0, 0, 0, 0, 0, -2, 0, 0}, 27.821190, 0.000220, 0.054000},
	{{0, 7, -5, 0, 0, 0, 0, 0, -2, 1, 0}, 25.619160, 0.000130, 0.196000},
	{{0, 7, -5, 0, 0, 0, 0, 0, -1, 0, 0}, 205.919540, 0.000020, 0.167000},
	{{0, 7, -5, 0, 0, 0, 0, 0, 0, -2, 0}, 342.072310, 0.000010, 0.050000},
	{{0, 7, -5, 0, 0, 0, 0, 0, 0, -1, 0}, 26.392560, 0.000140, 0.145000},
	{{0, 7, -5, 0, 0, 0, 0, 0, 0, 0, 0}, 26.206660, 0.000760, 0.157000},
	{{0, 7, -5, 0, 0, 0, 0, 0, 0, 1, 0}, 25.948630, 0.000070, 0.051000},
	{{0, 7, -5, 0, 0, 0, 0, 0, 2, -2, 0}, 327.220990, 0.000030, 0.217000},
	{{0, 7, -5, 0, 0, 0, 0, 0, 2, -1, 0}, 25.919580, 0.000110, 0.056000},
	{{0, 7, -5, 0, 0, 0, 0, 0, 2, 0, 0}, 218.643060, 0.000110, 0.032000},
	{{0, 7, -4, 0, -3, 0, 0, 0, -2, 0, 0}, 356.677480, 0.000010, 0.057000},
	{{0, 7, -4, 0, -3, 0, 0, 0, -2, 2, 0}, 176.677480, 0.000010, 0.112000},
	{{0, 7, -4, 0, -1, 0, 0, 0, 2, -2, 0}, 156.554440, 0.000010, 0.181000},
	{{0, 7, -4, 0, -1, 0, 0, 0, 2, 0, 0}, 336.554440, 0.000010, 0.031000},
	{{0, 7, -4, 0, 0, 0, 0, 0, -2, 1, 0}, 279.313210, 0.000020, 0.244000},
	{{0, 7, -4, 0, 0, 0, 0, 0, 0, 0, 0}, 286.237380, 0.000050, 0.136000},
	{{0, 7, -4, 0, 0, 0, 0, 0, 1, 0, -1}, 309.008000, 0.000010, 0.158000},
	{{0, 7, -4, 0, 0, 0, 0, 0, 2, -2, 0}, 44.093990, 0.000030, 0.178000},
	{{0, 7, -4, 0, 0, 0, 0, 0, 2, 0, 0}, 29.300850, 0.000010, 0.031000},
	{{0, 7, -4, 0, 1, 0, 0, 0, 2, -2, 0}, 307.294560, 0.000030, 0.176000},
	{{0, 7, -4, 0, 1, 0, 0, 0, 2, 0, 0}, 311.498710, 0.000020, 0.031000},
	{{0, 7, -3, 0, -1, 0, 0, 0, 2, -2, 0}, 294.470010, 0.000020, 0.153000},
	{{0, 7, -3, 0, -1, 0, 0, 0, 2, 0, 0}, 157.406160, 0.000050, 0.030000},
	{{0, 7, -3, 0, 0, 0, 0, 0, 0, -1, 0}, 128.235840, 0.000030, 0.205000},
	{{0, 7, -3, 0, 0, 0, 0, 0, 0, 0, 0}, 184.338820, 0.000010, 0.119000},
	{{0, 7, -3, 0, 0, 0, 0, 0, 0, 1, 0}, 128.235840, 0.000030, 0.046000},
	{{0, 7, -3, 0, 0, 0, 0, 0, 2, -2, 0}, 130.592540, 0.000010, 0.151000},
	{{0, 7, -3, 0, 0, 0, 0, 0, 2, -1, 0}, 128.197120, 0.000020, 0.050000},
	{{0, 7, -3, 0, 0, 0, 0, 0, 2, 0, 0}, 127.789120, 0.000030, 0.030000},
	{{0, 7, -3, 0, 0, 0, 0, 0, 2, 1, 0}, 128.253720, 0.000030, 0.022000},
	{{0, 7, -3, 0, 0, 0, 0, 0, 2, 2, 0}, 128.210980, 0.000020, 0.017000},
	{{0, 7, -3, 0, 1, 0, 0, 0, 2, 0, 0}, 298.782140, 0.000070, 0.030000},
	{{0, 7, -2, -2, 0, 0, 0, 0, 2, -2, 0}, 84.566040, 0.000010, 0.153000},
	{{0, 7, -2, -2, 0, 0, 0, 0, 2, 0, 0}, 84.566040, 0.000010, 0.030000},
	{{0, 7, -2, 0, 0, 0, 0, 0, -4, 1, 2}, 307.568120, 0.000030, 224.432000},
	{{0, 7, -2, 0, 0, 0, 0, 0, -2, 0, 0}, 302.012200, 0.000040, 0.065000},
	{{0, 7, -2, 0, 0, 0, 0, 0, -2, 2, 0}, 302.012200, 0.000040, 0.090000},
	{{0, 7, -2, 0, 0, 0, 0, 0, 0, 0, 0}, 106.152470, 0.000020, 0.107000},
	{{0, 7, -2, 0, 0, 0, 0, 0, 2, -2, 0}, 221.976720, 0.000020, 0.132000},
	{{0, 7, -2, 0, 0, 0, 0, 0, 2, 0, 0}, 221.976720, 0.000020, 0.029000},
	{{0, 7, 0, 0, 0, 0, 0, 0, -2, 0, 0}, 146.633920, 0.000050, 0.075000},
	{{0, 7, 0, 0, 0, 0, 0, 0, -2, 1, 0}, 333.575000, 0.000020, 9.702000},
	{{0, 7, 0, 0, 0, 0, 0, 0, -2, 2, 0}, 146.633920, 0.000050, 0.076000},
	{{0, 7, 1, 0, 0, 0, 0, 0, -1, 0, 0}, 81.623140, 0.000070, 100.849000},
	{{0, 7, 4, -4, 0, 0, 0, 0, 0, -1, 0}, 255.057490, 0.000040, 261.124000},
	{{0, 8, -47, 39, 0, 0, 0, 0, 0, 1, 0}, 14.697840, 0.000010, 228.633000},
	{{0, 8, -21, 19, 0, 0, 0, 0, 2, 0, -2}, 29.966740, 0.000010, 743.522000},
	{{0, 8, -17, 0, 0, 0, 0, 0, -2, 0, 0}, 289.622370, 0.000040, 0.035000},
	{{0, 8, -17, 0, 0, 0, 0, 0, -2, 1, 0}, 109.621090, 0.000010, 0.065000},
	{{0, 8, -17, 0, 0, 0, 0, 0, -2, 2, 0}, 289.622370, 0.000040, 0.450000},
	{{0, 8, -17, 0, 0, 0, 0, 0, -2, 3, 0}, 109.621090, 0.000010, 0.091000},
	{{0, 8, -16, 0, 0, 0, 0, 0, -3, 1, 2}, 189.094270, 0.000090, 784.579000},
	{{0, 8, -15, 0, 0, 0, 0, 0, -2, 1, 0}, 135.623890, 0.000020, 0.074000},
	{{0, 8, -15, 0, 0, 0, 0, 0, -2, 3, 0}, 121.176500, 0.000040, 0.077000},
	{{0, 8, -15, 0, 0, 0, 0, 0, 0, -1, 0}, 103.764680, 0.000020, 0.066000},
	{{0, 8, -15, 0, 0, 0, 0, 0, 0, 0, 0}, 66.929640, 0.000040, 0.501000},
	{{0, 8, -15, 0, 0, 0, 0, 0, 0, 1, 0}, 96.507380, 0.000040, 0.089000},
	{{0, 8, -15, 0, 0, 0, 0, 0, 2, -1, 0}, 84.104400, 0.000120, 0.105000},
	{{0, 8, -15, 0, 0, 0, 0, 0, 2, 0, 0}, 83.492780, 0.000170, 0.044000},
	{{0, 8, -15, 0, 0, 0, 0, 0, 2, 1, 0}, 83.703990, 0.000010, 0.028000},
	{{0, 8, -15, 0, 1, 0, 0, 0, -2, 1, 0}, 220.524100, 0.000010, 0.075000},
	{{0, 8, -15, 0, 1, 0, 0, 0, -2, 3, 0}, 220.524100, 0.000010, 0.076000},
	{{0, 8, -15, 0, 2, 0, 0, 0, -2, 0, 0}, 322.898030, 0.000010, 0.038000},
	{{0, 8, -15, 0, 2, 0, 0, 0, -2, 1, 0}, 144.672370, 0.000030, 0.075000},
	{{0, 8, -15, 0, 3, 0, 0, 0, -2, 1, 0}, 311.040690, 0.000020, 0.076000},
	{{0, 8, -15, 0, 3, 0, 0, 0, -2, 3, 0}, 311.040690, 0.000020, 0.075000},
	{{0, 8, -14, 0, 0, 0, 0, 0, -2, -1, 0}, 196.536200, 0.000020, 0.026000},
	{{0, 8, -14, 0, 0, 0, 0, 0, -2, 0, 0}, 161.916950, 0.000170, 0.039000},
	{{0, 8, -14, 0, 0, 0, 0, 0, -2, 0, 2}, 160.847410, 0.000010, 0.900000},
	{{0, 8, -14, 0, 0, 0, 0, 0, -2, 1, 0}, 168.495610, 0.000110, 0.080000},
	{{0, 8, -14, 0, 0, 0, 0, 0, -1, 0, 0}, 338.724470, 0.000010, 0.075000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 0, -1, 0}, 342.434820, 0.000300, 0.070000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 0, 0, 0}, 339.887250, 0.002970, 1.004000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 0, 1, 0}, 339.140580, 0.000920, 0.082000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 0, 2, 0}, 338.525670, 0.000070, 0.039000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 2, -2, 0}, 342.199560, 0.000100, 0.361000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 2, -1, 0}, 340.867900, 0.002650, 0.095000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 2, 0, -2}, 340.776990, 0.000030, 0.322000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 2, 0, 0}, 340.027780, 0.002420, 0.042000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 2, 1, 0}, 339.650270, 0.000240, 0.027000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 2, 2, 0}, 340.378870, 0.000020, 0.020000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 4, -2, 0}, 341.020690, 0.000050, 0.046000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 4, -1, 0}, 340.123330, 0.000100, 0.028000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 4, 0, 0}, 339.869550, 0.000060, 0.021000},
	{{0, 8, -14, 0, 0, 0, 0, 0, 4, 1, 0}, 160.781000, 0.000010, 0.016000},
	{{0, 8, -14, 0, 1, 0, 0, 0, -2, 0, 0}, 141.724720, 0.000020, 0.039000},
	{{0, 8, -14, 0, 1, 0, 0, 0, -2, 1, 0}, 322.055800, 0.000010, 0.081000},
	{{0, 8, -14, 0, 1, 0, 0, 0, -2, 2, 0}, 141.724720, 0.000020, 1.159000},
	{{0, 8, -14, 0, 1, 0, 0, 0, -2, 3, 0}, 142.055800, 0.000010, 0.071000},
	{{0, 8, -14, 0, 2, 0, 0, 0, -2, 0, 0}, 51.543430, 0.000010, 0.039000},
	{{0, 8, -14, 0, 2, 0, 0, 0, -2, 2, 0}, 51.518620, 0.000010, 1.056000},
	{{0, 8, -14, 0, 2, 0, 0, 0, 0, -1, 0}, 231.624180, 0.000020, 0.071000},
	{{0, 8, -14, 0, 2, 0, 0, 0, 0, 1, 0}, 231.624180, 0.000020, 0.080000},
	{{0, 8, -14, 1, 0, 0, 0, 0, 0, 0, 0}, 57.728550, 0.000010, 2.155000},
	{{0, 8, -14, 2, 0, 0, 0, 0, -2, 0, 0}, 233.911010, 0.000040, 0.041000},
	{{0, 8, -14, 2, 0, 0, 0, 0, -2, 1, 0}, 51.808250, 0.000010, 0.088000},
	{{0, 8, -14, 2, 0, 0, 0, 0, -2, 2, 0}, 231.741570, 0.000030, 0.543000},
	{{0, 8, -14, 2, 0, 0, 0, 0, -2, 3, 0}, 51.808250, 0.000010, 0.066000},
	{{0, 8, -14, 2, 0, 0, 0, 0, 0, -1, 0}, 42.040440, 0.000020, 0.076000},
	{{0, 8, -14, 2, 0, 0, 0, 0, 0, 1, 0}, 42.040440, 0.000020, 0.075000},
	{{0, 8, -13, 0, -1, 0, 0, 0, 0, -1, 0}, 151.610170, 0.000020, 0.075000},
	{{0, 8, -13, 0, -1, 0, 0, 0, 0, 1, 0}, 151.610170, 0.000020, 0.076000},
	{{0, 8, -13, 0, 0, -1, 0, 0, 0, -1, 0}, 55.487360, 0.000010, 0.075000},
	{{0, 8, -13, 0, 0, -1, 0, 0, 0, 1, 0}, 55.487360, 0.000010, 0.076000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -6, 1, 0}, 236.615410, 0.000010, 0.016000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -6, 2, 0}, 236.943220, 0.000020, 0.021000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -4, -1, 0}, 233.073500, 0.000050, 0.016000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -4, 0, 0}, 234.822070, 0.000270, 0.020000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -4, 1, 0}, 237.515510, 0.000720, 0.028000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -4, 1, 2}, 237.887440, 0.000010, 0.107000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -4, 2, 0}, 236.549480, 0.000510, 0.044000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -4, 3, 0}, 234.338140, 0.000010, 0.103000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -3, 1, 0}, 57.301470, 0.000030, 0.042000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, -3, 0}, 280.672750, 0.000020, 0.016000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, -2, 0}, 236.109630, 0.000190, 0.020000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, -1, -2}, 57.560750, 0.000010, 0.015000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, -1, 0}, 237.056280, 0.002270, 0.026000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, -1, 2}, 57.186800, 0.000060, 0.090000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, 0, -2}, 56.922850, 0.000050, 0.019000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, 0, 0}, 237.416760, 0.024180, 0.040000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, 0, 2}, 237.887440, 0.000550, 0.474000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, 1, -2}, 57.958690, 0.000090, 0.026000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, 1, 0}, 238.008170, 0.038780, 0.087000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, 1, 2}, 236.327800, 0.000020, 0.065000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, 2, 0}, 241.536270, 0.001520, 0.562000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -2, 3, 0}, 240.357550, 0.000110, 0.067000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -1, -1, 0}, 55.566030, 0.000070, 0.039000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -1, 0, 0}, 55.681430, 0.000650, 0.081000},
	{{0, 8, -13, 0, 0, 0, 0, 0, -1, 1, 0}, 59.356960, 0.000090, 1.122000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, -3, 0}, 232.112100, 0.000180, 0.025000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, -2, -2}, 54.214200, 0.000020, 0.019000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, -2, 0}, 233.249490, 0.002630, 0.038000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, -1, -2}, 54.361520, 0.000100, 0.025000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, -1, 0}, 233.446250, 0.038780, 0.075000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, -1, 2}, 232.426500, 0.000060, 0.074000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, 0, -2}, 57.937440, 0.000110, 0.037000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, 0, 0}, 235.750130, 0.226530, 238.919000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, 0, 2}, 57.967600, 0.000110, 0.037000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, 1, -2}, 232.816800, 0.000060, 0.074000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, 1, 0}, 233.445270, 0.038740, 0.075000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, 1, 2}, 54.361520, 0.000100, 0.025000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, 2, 0}, 233.465840, 0.002620, 0.038000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, 2, 2}, 54.271330, 0.000020, 0.019000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 0, 3, 0}, 233.313250, 0.000180, 0.025000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 1, -1, 0}, 59.454630, 0.000090, 1.133000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 1, 0, 0}, 58.122770, 0.000640, 0.081000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 1, 1, 0}, 57.060790, 0.000070, 0.039000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, -3, 0}, 239.769010, 0.000110, 0.067000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, -2, 0}, 239.146610, 0.001490, 0.565000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, -1, -2}, 236.577290, 0.000020, 0.065000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, -1, 0}, 237.969350, 0.038500, 0.087000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, -1, 2}, 57.910310, 0.000090, 0.026000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, 0, -2}, 237.856790, 0.000550, 0.475000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, 0, 0}, 237.222800, 0.023930, 0.040000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, 0, 2}, 56.714920, 0.000050, 0.019000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, 1, -2}, 57.131730, 0.000060, 0.090000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, 1, 0}, 236.298150, 0.002260, 0.026000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, 1, 2}, 57.387090, 0.000010, 0.015000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, 2, 0}, 235.815630, 0.000180, 0.020000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 2, 3, 0}, 237.432350, 0.000020, 0.016000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 3, -1, 0}, 57.856820, 0.000030, 0.042000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 3, 0, 0}, 238.678990, 0.000020, 0.027000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 4, -3, 0}, 237.322250, 0.000010, 0.103000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 4, -2, 0}, 237.959090, 0.000510, 0.044000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 4, -1, -2}, 237.854490, 0.000010, 0.107000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 4, -1, 0}, 237.320320, 0.000710, 0.028000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 4, 0, 0}, 234.653690, 0.000270, 0.020000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 4, 1, 0}, 236.481670, 0.000050, 0.016000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 6, -2, 0}, 237.299570, 0.000020, 0.021000},
	{{0, 8, -13, 0, 0, 0, 0, 0, 6, -1, 0}, 236.567640, 0.000010, 0.016000},
	{{0, 8, -13, 0, 0, 1, 0, 0, 0, -1, 0}, 231.331010, 0.000020, 0.076000},
	{{0, 8, -13, 0, 0, 1, 0, 0, 0, 1, 0}, 231.331010, 0.000020, 0.075000},
	{{0, 8, -13, 0, 1, 0, 0, 0, -2, 0, 0}, 50.448880, 0.000030, 0.041000},
	{{0, 8, -13, 0, 1, 0, 0, 0, -2, 3, 0}, 229.989440, 0.000020, 0.066000},
	{{0, 8, -13, 0, 1, 0, 0, 0, 0, -2, 0}, 230.423380, 0.000010, 0.038000},
	{{0, 8, -13, 0, 1, 0, 0, 0, 0, -1, 0}, 160.867950, 0.000030, 0.076000},
	{{0, 8, -13, 0, 1, 0, 0, 0, 0, 0, 0}, 50.423380, 0.000010, 11.301000},
	{{0, 8, -13, 0, 1, 0, 0, 0, 0, 1, 0}, 160.867950, 0.000030, 0.075000},
	{{0, 8, -13, 0, 2, 0, 0, 0, -2, 0, 0}, 111.910720, 0.000030, 0.041000},
	{{0, 8, -13, 0, 2, 0, 0, 0, 0, -1, 0}, 292.349760, 0.000020, 0.076000},
	{{0, 8, -13, 0, 2, 0, 0, 0, 0, 0, 0}, 66.154110, 0.000010, 5.787000},
	{{0, 8, -13, 0, 2, 0, 0, 0, 0, 1, 0}, 292.349760, 0.000020, 0.074000},
	{{0, 8, -13, 2, 0, 0, 0, 0, -2, 0, 0}, 95.373640, 0.000010, 0.042000},
	{{0, 8, -13, 2, 0, 0, 0, 0, -2, 2, 0}, 95.373640, 0.000010, 0.352000},
	{{0, 8, -13, 2, 0, 0, 0, 0, 1, 0, -1}, 248.766530, 0.000030, 72.525000},
	{{0, 8, -12, -2, 0, 0, 0, 0, 0, -1, 0}, 282.329720, 0.000020, 0.075000},
	{{0, 8, -12, -2, 0, 0, 0, 0, 0, 1, 0}, 282.329720, 0.000020, 0.076000},
	{{0, 8, -12, -1, 0, 0, 0, 0, 0, 0, 0}, 81.345000, 0.000010, 2.116000},
	{{0, 8, -12, 0, -1, 0, 0, 0, -2, 0, 0}, 242.396470, 0.000050, 0.042000},
	{{0, 8, -12, 0, -1, 0, 0, 0, -2, 1, 0}, 60.436800, 0.000010, 0.095000},
	{{0, 8, -12, 0, -1, 0, 0, 0, -2, 2, 0}, 240.432010, 0.000030, 0.371000},
	{{0, 8, -12, 0, -1, 0, 0, 0, -2, 3, 0}, 240.436800, 0.000010, 0.063000},
	{{0, 8, -12, 0, -1, 0, 0, 0, 0, -1, 0}, 60.436800, 0.000010, 0.081000},
	{{0, 8, -12, 0, -1, 0, 0, 0, 0, 1, 0}, 60.436800, 0.000010, 0.071000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -4, 0, 0}, 127.992020, 0.000030, 0.021000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -4, 1, 0}, 122.522480, 0.000060, 0.028000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -4, 2, 0}, 146.864810, 0.000010, 0.046000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -2, -1, 0}, 111.721350, 0.000140, 0.027000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -2, -1, 2}, 231.768180, 0.000010, 0.099000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -2, 0, 0}, 99.956900, 0.001080, 0.042000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -2, 0, 2}, 178.216710, 0.000010, 0.321000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -2, 1, 0}, 91.842760, 0.001020, 0.095000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -2, 1, 2}, 231.768180, 0.000010, 0.061000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -2, 2, 0}, 64.735540, 0.000060, 0.360000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -2, 3, 0}, 226.477100, 0.000020, 0.062000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -1, -1, 1}, 333.735920, 0.000020, 0.089000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -1, 0, -1}, 153.726090, 0.000010, 0.040000},
	{{0, 8, -12, 0, 0, 0, 0, 0, -1, 1, -1}, 153.468280, 0.000020, 0.087000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 0, -2, 0}, 4.925850, 0.000040, 0.039000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 0, -1, 0}, 7.206040, 0.000700, 0.082000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 0, 0, 0}, 350.368830, 0.003880, 0.996000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 0, 1, -2}, 231.768180, 0.000030, 0.079000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 0, 1, 0}, 342.840840, 0.000830, 0.070000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 0, 2, 0}, 333.212410, 0.000080, 0.036000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 1, -1, -1}, 333.463250, 0.000050, 0.075000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 1, -1, 0}, 135.380980, 0.000020, 8.534000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 1, -1, 1}, 154.040910, 0.000040, 0.074000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 1, 0, -1}, 153.455950, 0.000280, 20.170000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 1, 0, 0}, 155.822410, 0.000160, 0.075000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 1, 0, 1}, 333.481400, 0.000500, 0.037000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 1, 1, -1}, 333.463250, 0.000050, 0.076000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 1, 1, 1}, 333.476070, 0.000050, 0.025000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 2, -2, 0}, 139.805000, 0.000030, 1.299000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 2, -1, 0}, 328.373780, 0.000400, 0.080000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 2, 0, -2}, 325.293170, 0.000020, 0.906000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 2, 0, 0}, 330.956910, 0.000380, 0.039000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 2, 1, 0}, 324.378030, 0.000020, 0.026000},
	{{0, 8, -12, 0, 0, 0, 0, 0, 3, 0, -1}, 333.476070, 0.000020, 0.041000},
	{{0, 8, -12, 0, 1, 0, 0, 0, -2, 0, 0}, 39.039360, 0.000070, 0.042000},
	{{0, 8, -12, 0, 1, 0, 0, 0, -2, 2, 0}, 38.051580, 0.000010, 0.349000},
	{{0, 8, -11, -4, 0, 0, 0, 0, 0, 0, 0}, 35.344570, 0.000010, 8.160000},
	{{0, 8, -11, -2, 0, 0, 0, 0, -2, 0, 0}, 191.191580, 0.000020, 0.042000},
	{{0, 8, -11, -2, 0, 0, 0, 0, -2, 2, 0}, 191.191580, 0.000020, 0.368000},
	{{0, 8, -11, 0, -1, 0, 0, 0, -2, 2, 0}, 52.491720, 0.000010, 0.271000},
	{{0, 8, -11, 0, 0, 0, 0, 0, -4, 0, 0}, 119.589150, 0.000020, 0.021000},
	{{0, 8, -11, 0, 0, 0, 0, 0, -2, -2, 0}, 64.687320, 0.000010, 0.020000},
	{{0, 8, -11, 0, 0, 0, 0, 0, -2, 0, 0}, 307.996360, 0.000070, 0.044000},
	{{0, 8, -11, 0, 0, 0, 0, 0, -2, 1, 0}, 281.914440, 0.000080, 0.106000},
	{{0, 8, -11, 0, 0, 0, 0, 0, -2, 2, 0}, 322.029830, 0.000020, 0.265000},
	{{0, 8, -11, 0, 0, 0, 0, 0, 0, -1, 0}, 256.016050, 0.000200, 0.089000},
	{{0, 8, -11, 0, 0, 0, 0, 0, 0, 0, 0}, 79.718200, 0.000090, 0.499000},
	{{0, 8, -11, 0, 0, 0, 0, 0, 1, 0, -1}, 247.876390, 0.000100, 1.052000},
	{{0, 8, -11, 0, 0, 0, 0, 0, 2, -3, 0}, 52.748760, 0.000020, 0.077000},
	{{0, 8, -11, 0, 0, 0, 0, 0, 2, -2, 0}, 255.422870, 0.000030, 4.345000},
	{{0, 8, -11, 0, 0, 0, 0, 0, 2, -1, 0}, 76.073810, 0.000970, 0.074000},
	{{0, 8, -11, 0, 0, 0, 0, 0, 2, 0, -1}, 66.357320, 0.000050, 0.075000},
	{{0, 8, -11, 0, 0, 0, 0, 0, 2, 0, 0}, 249.573900, 0.000020, 0.037000},
	{{0, 8, -11, 0, 0, 0, 0, 0, 4, -2, 0}, 76.432120, 0.000010, 0.040000},
	{{0, 8, -11, 0, 0, 3, 0, 0, 2, 0, -2}, 176.953980, 0.000060, 666.222000},
	{{0, 8, -11, 0, 1, 0, 0, 0, 2, -1, 0}, 349.978450, 0.000030, 0.074000},
	{{0, 8, -11, 0, 2, 0, 0, 0, 2, -2, 0}, 91.150310, 0.000010, 2.508000},
	{{0, 8, -11, 0, 2, 0, 0, 0, 2, 0, -2}, 91.150310, 0.000010, 15.323000},
	{{0, 8, -11, 0, 2, 0, 0, 0, 2, 0, 0}, 271.150310, 0.000010, 0.037000},
	{{0, 8, -10, 0, 0, 0, 0, 0, -2, 0, 0}, 162.842950, 0.000020, 0.046000},
	{{0, 8, -10, 0, 0, 0, 0, 0, -2, 1, 0}, 168.443580, 0.000030, 0.118000},
	{{0, 8, -10, 0, 0, 0, 0, 0, -2, 2, 0}, 335.294630, 0.000010, 0.209000},
	{{0, 8, -10, 0, 0, 0, 0, 0, -1, 0, 1}, 3.776790, 0.000020, 0.246000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 0, -1, 0}, 186.144870, 0.000050, 0.098000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 0, 0, 0}, 343.308060, 0.000750, 0.333000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 0, 1, 0}, 346.918040, 0.000120, 0.062000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 0, 2, 0}, 349.673250, 0.000020, 0.034000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 1, -1, -1}, 168.834390, 0.000010, 0.088000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 1, 0, -2}, 349.944550, 0.000020, 0.087000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 1, 0, -1}, 168.834390, 0.000210, 0.513000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 1, 0, 0}, 164.208430, 0.000030, 0.065000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 1, 1, -1}, 168.834390, 0.000010, 0.066000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 2, -3, 0}, 345.348850, 0.000020, 0.083000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 2, -2, 0}, 351.802200, 0.000030, 0.813000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 2, -1, 0}, 351.470670, 0.001040, 0.069000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 2, 0, -2}, 350.965080, 0.000040, 1.115000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 2, 0, 0}, 173.860390, 0.000240, 0.036000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 2, 1, 0}, 170.263250, 0.000030, 0.024000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 3, -2, -2}, 168.656560, 0.000020, 0.075000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 3, -1, -2}, 349.784080, 0.000080, 103.674000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 3, -1, -1}, 168.884980, 0.000050, 0.074000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 3, 0, -2}, 169.944540, 0.000120, 0.075000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 4, -2, 0}, 349.388130, 0.000020, 0.039000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 4, -1, 0}, 171.087300, 0.000010, 0.025000},
	{{0, 8, -10, 0, 0, 0, 0, 0, 4, 0, 0}, 170.844680, 0.000030, 0.019000},
	{{0, 8, -10, 0, 1, 0, 0, 0, -2, 0, 0}, 270.456250, 0.000010, 0.046000},
	{{0, 8, -10, 0, 1, 0, 0, 0, -2, 2, 0}, 90.456250, 0.000010, 0.206000},
	{{0, 8, -10, 0, 1, 0, 0, 0, 2, -1, 0}, 269.977000, 0.000030, 0.069000},
	{{0, 8, -10, 0, 1, 0, 0, 0, 2, 0, 0}, 89.977000, 0.000030, 0.036000},
	{{0, 8, -10, 0, 3, 0, 0, 0, 0, 0, 0}, 168.808550, 0.000010, 0.307000},
	{{0, 8, -10, 2, 0, 0, 0, 0, 2, -3, 0}, 180.060530, 0.000010, 0.091000},
	{{0, 8, -10, 2, 0, 0, 0, 0, 2, -1, 0}, 180.060530, 0.000010, 0.064000},
	{{0, 8, -10, 4, 0, 0, 0, 0, -2, 0, 0}, 57.952330, 0.000010, 0.051000},
	{{0, 8, -10, 4, 0, 0, 0, 0, -2, 2, 0}, 57.952330, 0.000010, 0.145000},
	{{0, 8, -10, 4, 0, 0, 0, 0, 0, 0, 0}, 57.635200, 0.000010, 0.195000},
	{{0, 8, -9, 0, -1, 0, 0, 0, -2, 0, 0}, 104.497480, 0.000020, 0.048000},
	{{0, 8, -9, 0, -1, 0, 0, 0, 0, 0, 0}, 353.072340, 0.000010, 0.255000},
	{{0, 8, -9, 0, 0, 0, 0, 0, -2, 0, 0}, 272.401910, 0.000040, 0.048000},
	{{0, 8, -9, 0, 0, 0, 0, 0, -2, 1, 0}, 87.189220, 0.000080, 0.134000},
	{{0, 8, -9, 0, 0, 0, 0, 0, -2, 2, 0}, 275.434020, 0.000010, 0.173000},
	{{0, 8, -9, 0, 0, 0, 0, 0, -1, 0, -1}, 75.950300, 0.000030, 0.046000},
	{{0, 8, -9, 0, 0, 0, 0, 0, -1, 0, 0}, 91.408410, 0.000010, 0.120000},
	{{0, 8, -9, 0, 0, 0, 0, 0, -1, 0, 1}, 255.943170, 0.000130, 0.198000},
	{{0, 8, -9, 0, 0, 0, 0, 0, -1, 1, -1}, 75.966780, 0.000020, 0.117000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 0, -1, 0}, 95.477700, 0.000070, 0.108000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 0, 0, 0}, 271.865370, 0.002260, 0.250000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 0, 1, 0}, 272.450450, 0.000470, 0.058000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 0, 2, 0}, 289.556450, 0.000020, 0.033000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 0, 3, 0}, 273.195410, 0.000020, 0.023000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 1, -1, -1}, 75.958530, 0.000100, 0.097000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 1, -1, 0}, 91.972630, 0.000010, 0.321000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 1, -1, 1}, 254.283920, 0.000040, 0.060000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 1, 0, -1}, 75.979000, 0.000630, 0.339000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 1, 0, 0}, 90.791870, 0.000050, 0.061000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 1, 1, -1}, 75.948370, 0.000060, 0.062000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 2, -3, 0}, 283.153830, 0.000060, 0.091000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 2, -2, 0}, 245.974260, 0.000080, 0.448000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 2, -1, 0}, 272.640140, 0.002670, 0.065000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 2, 0, -2}, 272.315440, 0.000080, 0.527000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 2, 0, -1}, 255.935740, 0.000020, 0.065000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 2, 0, 0}, 95.075140, 0.001340, 0.035000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 2, 1, 0}, 94.372630, 0.000200, 0.024000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 2, 2, 0}, 92.964440, 0.000020, 0.018000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 3, 0, 0}, 92.433330, 0.000020, 0.024000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 4, -3, 0}, 102.531240, 0.000010, 0.073000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 4, -2, 0}, 283.417090, 0.000030, 0.037000},
	{{0, 8, -9, 0, 0, 0, 0, 0, 4, 0, 0}, 98.430390, 0.000050, 0.019000},
	{{0, 8, -9, 0, 1, 0, 0, 0, -2, 0, 0}, 192.749830, 0.000090, 0.048000},
	{{0, 8, -9, 0, 1, 0, 0, 0, 0, 0, 0}, 7.150180, 0.000030, 0.245000},
	{{0, 8, -9, 0, 1, 0, 0, 0, 2, -1, 0}, 178.698900, 0.000050, 0.064000},
	{{0, 8, -9, 0, 1, 0, 0, 0, 2, 0, 0}, 178.540980, 0.000030, 0.035000},
	{{0, 8, -9, 0, 2, 0, 0, 0, -2, 0, 0}, 232.656720, 0.000010, 0.049000},
	{{0, 8, -9, 0, 2, 0, 0, 0, -2, 2, 0}, 232.656720, 0.000010, 0.168000},
	{{0, 8, -9, 0, 2, 0, 0, 0, 2, -2, 0}, 240.163000, 0.000010, 0.417000},
	{{0, 8, -9, 0, 2, 0, 0, 0, 2, 0, 0}, 60.163000, 0.000010, 0.035000},
	{{0, 8, -9, 3, 0, 0, 0, 0, -2, 0, 0}, 236.389800, 0.000010, 0.052000},
	{{0, 8, -9, 3, 0, 0, 0, 0, -2, 2, 0}, 236.389800, 0.000010, 0.136000},
	{{0, 8, -8, 0, -2, 0, 0, 0, 0, 0, 0}, 201.544030, 0.000010, 0.207000},
	{{0, 8, -8, 0, -1, 0, 0, 0, -2, 0, 0}, 185.871680, 0.000070, 0.050000},
	{{0, 8, -8, 0, -1, 0, 0, 0, -2, 1, 0}, 299.107310, 0.000010, 0.152000},
	{{0, 8, -8, 0, -1, 0, 0, 0, -2, 2, 0}, 185.573180, 0.000010, 0.149000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -4, 0, 0}, 179.699500, 0.000020, 0.022000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -4, 1, 0}, 180.249690, 0.000020, 0.032000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -4, 2, 0}, 179.812670, 0.000020, 0.056000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -4, 3, 0}, 180.442280, 0.000010, 0.213000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -2, -1, 0}, 180.370990, 0.000060, 0.030000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -2, 0, 0}, 180.373770, 0.000420, 0.051000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -2, 1, 0}, 179.865140, 0.000310, 0.154000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -2, 2, 0}, 359.498190, 0.000080, 0.148000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -2, 3, 0}, 0.718260, 0.000010, 0.050000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -1, 0, 0}, 359.914550, 0.000110, 0.136000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -1, 0, 1}, 17.333810, 0.000010, 0.165000},
	{{0, 8, -8, 0, 0, 0, 0, 0, -1, 1, 0}, 0.064640, 0.000010, 0.170000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 0, -3, 0}, 1.425100, 0.000020, 0.029000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 0, -2, 0}, 184.431610, 0.000010, 0.046000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 0, -1, 0}, 180.374180, 0.000520, 0.121000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 0, 0, -2}, 179.113270, 0.000010, 0.046000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 0, 0, 0}, 179.966060, 0.010700, 0.200000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 0, 1, 0}, 180.044750, 0.001800, 0.055000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 0, 2, 0}, 179.315850, 0.000170, 0.032000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 1, -1, 0}, 359.990150, 0.000030, 0.243000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 1, 0, -1}, 190.471850, 0.000150, 0.253000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 1, 0, 0}, 0.016800, 0.000160, 0.058000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 1, 1, 0}, 180.056840, 0.000010, 0.033000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, -3, 0}, 186.379660, 0.000010, 0.100000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, -2, 0}, 180.543450, 0.000370, 0.310000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, -1, -2}, 179.993950, 0.000020, 0.097000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, -1, 0}, 180.037020, 0.008240, 0.061000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, -1, 2}, 0.011320, 0.000020, 0.023000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, 0, -2}, 179.990810, 0.000250, 0.345000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, 0, 0}, 0.044980, 0.005710, 0.034000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, 0, 2}, 180.003590, 0.000020, 0.018000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, 1, 0}, 0.008020, 0.000480, 0.023000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 2, 2, 0}, 0.003490, 0.000070, 0.018000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 3, -4, 0}, 0.056850, 0.000010, 0.092000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 3, -1, 0}, 180.052540, 0.000010, 0.035000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 3, 0, 0}, 0.056860, 0.000040, 0.024000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 3, 1, 0}, 0.056820, 0.000010, 0.018000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 4, -2, 0}, 179.625440, 0.000120, 0.036000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 4, -1, 0}, 0.047730, 0.000050, 0.024000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 4, 0, 0}, 359.999980, 0.000100, 0.018000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 4, 1, 0}, 0.004650, 0.000020, 0.015000},
	{{0, 8, -8, 0, 0, 0, 0, 0, 6, -1, 0}, 0.017310, 0.000010, 0.015000},
	{{0, 8, -8, 0, 1, 0, 0, 0, -2, 0, 0}, 348.213640, 0.000070, 0.051000},
	{{0, 8, -8, 0, 1, 0, 0, 0, -2, 1, 0}, 172.709610, 0.000020, 0.156000},
	{{0, 8, -8, 0, 1, 0, 0, 0, 0, -2, 0}, 351.242280, 0.000010, 0.047000},
	{{0, 8, -8, 0, 1, 0, 0, 0, 0, 0, 0}, 171.242280, 0.000010, 0.197000},
	{{0, 8, -7, 0, -1, 0, 0, 0, -2, 0, 0}, 178.601880, 0.000010, 0.053000},
	{{0, 8, -7, 0, -1, 0, 0, 0, -2, 1, 0}, 181.255690, 0.000010, 0.180000},
	{{0, 8, -7, 0, -1, 0, 0, 0, -2, 2, 0}, 178.601880, 0.000010, 0.130000},
	{{0, 8, -7, 0, -1, 0, 0, 0, -2, 3, 0}, 181.255690, 0.000010, 0.048000},
	{{0, 8, -7, 0, -1, 0, 0, 0, 0, 0, 0}, 181.507980, 0.000070, 0.169000},
	{{0, 8, -7, 0, -1, 0, 0, 0, 2, 0, 0}, 3.812030, 0.000020, 0.033000},
	{{0, 8, -7, 0, 0, 0, 0, 0, -2, 0, 0}, 104.478610, 0.000010, 0.053000},
	{{0, 8, -7, 0, 0, 0, 0, 0, -2, 1, 0}, 257.075490, 0.000020, 0.183000},
	{{0, 8, -7, 0, 0, 0, 0, 0, -2, 2, 0}, 256.708540, 0.000030, 0.129000},
	{{0, 8, -7, 0, 0, 0, 0, 0, -1, -1, -1}, 103.258200, 0.000020, 0.030000},
	{{0, 8, -7, 0, 0, 0, 0, 0, -1, -1, 1}, 283.250370, 0.000020, 0.161000},
	{{0, 8, -7, 0, 0, 0, 0, 0, -1, 0, -1}, 103.213530, 0.000140, 0.051000},
	{{0, 8, -7, 0, 0, 0, 0, 0, -1, 0, 1}, 283.241490, 0.000170, 0.142000},
	{{0, 8, -7, 0, 0, 0, 0, 0, -1, 1, -1}, 103.260480, 0.000070, 0.153000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 0, -2, 0}, 77.404700, 0.000030, 0.049000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 0, -1, 0}, 229.848380, 0.000010, 0.138000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 0, 0, -1}, 283.278180, 0.000010, 0.135000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 0, 1, 0}, 65.420570, 0.000030, 0.052000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 1, -1, -1}, 103.299770, 0.000110, 0.120000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 1, -1, 1}, 283.306930, 0.000030, 0.054000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 1, 0, -1}, 103.355290, 0.000600, 0.202000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 1, 0, 1}, 103.300050, 0.000020, 0.031000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 1, 1, -1}, 103.307030, 0.000050, 0.055000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 1, 1, 1}, 103.306890, 0.000020, 0.022000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 1, 2, 1}, 103.307180, 0.000020, 0.017000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 2, -1, 0}, 270.780340, 0.000210, 0.057000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 2, 0, 0}, 90.310230, 0.000300, 0.033000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 2, 1, 0}, 86.785620, 0.000050, 0.023000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 4, -1, 0}, 89.807430, 0.000010, 0.024000},
	{{0, 8, -7, 0, 0, 0, 0, 0, 4, 0, 0}, 83.368420, 0.000030, 0.018000},
	{{0, 8, -6, -3, 0, 0, 0, 0, -2, 0, 0}, 336.210940, 0.000010, 0.052000},
	{{0, 8, -6, -3, 0, 0, 0, 0, -2, 2, 0}, 336.210940, 0.000010, 0.139000},
	{{0, 8, -6, -2, 0, 0, 0, 0, -2, 0, 0}, 0.542850, 0.000010, 0.053000},
	{{0, 8, -6, -2, 0, 0, 0, 0, -2, 2, 0}, 0.542850, 0.000010, 0.130000},
	{{0, 8, -6, 0, -3, 0, 0, 0, -2, 0, 0}, 7.573960, 0.000010, 0.056000},
	{{0, 8, -6, 0, -3, 0, 0, 0, -2, 2, 0}, 7.573960, 0.000010, 0.117000},
	{{0, 8, -6, 0, -2, 0, 0, 0, -2, 2, 0}, 0.291870, 0.000020, 0.116000},
	{{0, 8, -6, 0, -2, 0, 0, 0, 0, -2, 0}, 0.159840, 0.000010, 0.051000},
	{{0, 8, -6, 0, -2, 0, 0, 0, 0, 0, 0}, 359.819830, 0.000040, 0.146000},
	{{0, 8, -6, 0, 0, 0, 0, 0, -2, -1, 0}, 26.480570, 0.000020, 0.032000},
	{{0, 8, -6, 0, 0, 0, 0, 0, -2, 0, 0}, 26.524420, 0.000210, 0.056000},
	{{0, 8, -6, 0, 0, 0, 0, 0, -2, 1, 0}, 26.069000, 0.000120, 0.223000},
	{{0, 8, -6, 0, 0, 0, 0, 0, -1, 0, 0}, 206.016480, 0.000020, 0.186000},
	{{0, 8, -6, 0, 0, 0, 0, 0, 0, -2, 0}, 342.154010, 0.000010, 0.051000},
	{{0, 8, -6, 0, 0, 0, 0, 0, 0, -1, 0}, 26.884440, 0.000130, 0.160000},
	{{0, 8, -6, 0, 0, 0, 0, 0, 0, 0, 0}, 26.021700, 0.000610, 0.143000},
	{{0, 8, -6, 0, 0, 0, 0, 0, 0, 1, 0}, 25.875800, 0.000050, 0.049000},
	{{0, 8, -6, 0, 0, 0, 0, 0, 2, -2, 0}, 282.566740, 0.000010, 0.191000},
	{{0, 8, -6, 0, 0, 0, 0, 0, 2, -1, 0}, 22.167420, 0.000080, 0.054000},
	{{0, 8, -6, 0, 0, 0, 0, 0, 2, 0, 0}, 214.135980, 0.000100, 0.032000},
	{{0, 8, -5, 0, -3, 0, 0, 0, -2, 0, 0}, 356.669170, 0.000010, 0.059000},
	{{0, 8, -5, 0, -3, 0, 0, 0, -2, 2, 0}, 176.669170, 0.000010, 0.105000},
	{{0, 8, -5, 0, 0, 0, 0, 0, -2, 1, 0}, 277.307940, 0.000020, 0.288000},
	{{0, 8, -5, 0, 0, 0, 0, 0, 0, 0, 0}, 283.917030, 0.000030, 0.125000},
	{{0, 8, -5, 0, 0, 0, 0, 0, 2, -2, 0}, 39.478110, 0.000020, 0.161000},
	{{0, 8, -5, 0, 0, 0, 0, 0, 2, -1, 0}, 38.760000, 0.000020, 0.051000},
	{{0, 8, -5, 0, 0, 0, 0, 0, 2, 0, 0}, 39.478110, 0.000020, 0.031000},
	{{0, 8, -4, 0, -1, 0, 0, 0, 2, -2, 0}, 323.432960, 0.000010, 0.140000},
	{{0, 8, -4, 0, -1, 0, 0, 0, 2, 0, 0}, 220.070580, 0.000020, 0.030000},
	{{0, 8, -4, 0, 0, 0, 0, 0, 0, -1, 0}, 308.194310, 0.000010, 0.235000},
	{{0, 8, -4, 0, 0, 0, 0, 0, 0, 0, 0}, 289.762880, 0.000020, 0.111000},
	{{0, 8, -4, 0, 0, 0, 0, 0, 0, 2, 0}, 308.208210, 0.000020, 0.028000},
	{{0, 8, -4, 0, 0, 0, 0, 0, 2, -1, 0}, 128.069310, 0.000020, 0.049000},
	{{0, 8, -4, 0, 0, 0, 0, 0, 2, 0, 0}, 129.009620, 0.000030, 0.030000},
	{{0, 8, -4, 0, 0, 0, 0, 0, 2, 1, 0}, 128.112400, 0.000030, 0.021000},
	{{0, 8, -4, 0, 0, 0, 0, 0, 2, 2, 0}, 128.208220, 0.000020, 0.017000},
	{{0, 8, -3, 0, -2, 0, 0, 0, 2, -2, 0}, 248.020640, 0.000010, 0.124000},
	{{0, 8, -3, 0, -2, 0, 0, 0, 2, 0, 0}, 68.020640, 0.000010, 0.029000},
	{{0, 8, -3, 0, -1, 0, 0, 0, 2, -3, 0}, 129.507980, 0.000010, 0.196000},
	{{0, 8, -3, 0, -1, 0, 0, 0, 2, -1, 0}, 309.507980, 0.000010, 0.047000},
	{{0, 8, -3, 0, -1, 0, 0, 0, 2, 0, 0}, 129.230580, 0.000020, 0.029000},
	{{0, 8, -3, 0, 0, 0, 0, 0, -2, 2, 0}, 302.104380, 0.000010, 0.085000},
	{{0, 8, -2, 0, -2, 0, 0, 0, 2, -2, 0}, 308.306510, 0.000010, 0.110000},
	{{0, 8, -2, 0, -2, 0, 0, 0, 2, 0, 0}, 308.306510, 0.000010, 0.028000},
	{{0, 8, -2, 0, 0, 0, 0, 0, 2, -2, 0}, 334.031870, 0.000020, 0.108000},
	{{0, 8, -2, 0, 0, 0, 0, 0, 2, 0, 0}, 334.031870, 0.000020, 0.028000},
	{{0, 8, -1, -1, 0, 0, 0, 0, -2, 1, 0}, 63.282390, 0.000020, 108.269000},
	{{0, 8, -1, 0, -6, 0, 0, 0, -2, 1, 0}, 253.392480, 0.000030, 60.164000},
	{{0, 9, -18, 0, 0, 0, 0, 0, -2, 0, 0}, 289.618430, 0.000010, 0.036000},
	{{0, 9, -18, 3, 0, 0, 0, 0, -2, 2, 0}, 234.590250, 0.000200, 818.910000},
	{{0, 9, -17, 0, 0, 0, 0, 0, -2, 1, 0}, 208.117670, 0.000030, 0.072000},
	{{0, 9, -17, 0, 3, 0, 0, 0, -2, 0, 2}, 358.662480, 0.000050, 101.557000},
	{{0, 9, -16, 0, 0, 0, 0, 0, -2, 1, 0}, 302.155000, 0.000030, 0.078000},
	{{0, 9, -16, 0, 0, 0, 0, 0, -2, 3, 0}, 268.300280, 0.000010, 0.073000},
	{{0, 9, -16, 0, 0, 0, 0, 0, 0, 0, 0}, 306.942220, 0.000020, 0.730000},
	{{0, 9, -16, 0, 2, 0, 0, 0, -2, 1, 0}, 143.026140, 0.000020, 0.079000},
	{{0, 9, -16, 0, 2, 0, 0, 0, -2, 3, 0}, 323.026140, 0.000020, 0.072000},
	{{0, 9, -15, 0, 0, 0, 0, 0, -2, -1, 0}, 223.572500, 0.000010, 0.026000},
	{{0, 9, -15, 0, 0, 0, 0, 0, -2, 0, 0}, 224.867480, 0.000010, 0.040000},
	{{0, 9, -15, 0, 0, 0, 0, 0, -2, 1, 0}, 238.531560, 0.000040, 0.084000},
	{{0, 9, -15, 0, 0, 0, 0, 0, -2, 2, 0}, 224.867480, 0.000010, 0.712000},
	{{0, 9, -15, 0, 0, 0, 0, 0, 0, -1, 0}, 59.198340, 0.000030, 0.073000},
	{{0, 9, -15, 0, 0, 0, 0, 0, 0, -1, 2}, 223.572500, 0.000020, 0.076000},
	{{0, 9, -15, 0, 0, 0, 0, 0, 0, 0, 0}, 238.244620, 0.000030, 2.701000},
	{{0, 9, -15, 0, 0, 0, 0, 0, 0, 1, 0}, 59.198340, 0.000030, 0.078000},
	{{0, 9, -15, 0, 1, 0, 0, 0, -2, 0, 0}, 141.770090, 0.000020, 0.040000},
	{{0, 9, -15, 0, 1, 0, 0, 0, -2, 1, 0}, 141.770090, 0.000010, 0.085000},
	{{0, 9, -15, 0, 1, 0, 0, 0, -2, 2, 0}, 141.770090, 0.000020, 0.672000},
	{{0, 9, -15, 0, 1, 0, 0, 0, -2, 3, 0}, 321.770090, 0.000010, 0.068000},
	{{0, 9, -15, 0, 1, 0, 0, 0, 0, -1, 0}, 142.096570, 0.000020, 0.074000},
	{{0, 9, -15, 0, 1, 0, 0, 0, 0, 1, 0}, 142.096570, 0.000020, 0.077000},
	{{0, 9, -15, 0, 2, 0, 0, 0, -2, 0, 0}, 51.571870, 0.000030, 0.040000},
	{{0, 9, -15, 0, 2, 0, 0, 0, -2, 1, 0}, 51.587030, 0.000060, 0.086000},
	{{0, 9, -15, 0, 2, 0, 0, 0, -2, 2, 0}, 51.512450, 0.000010, 0.636000},
	{{0, 9, -15, 0, 2, 0, 0, 0, 0, -1, 0}, 51.589800, 0.000020, 0.074000},
	{{0, 9, -15, 0, 2, 0, 0, 0, 0, 1, 0}, 51.589800, 0.000020, 0.077000},
	{{0, 9, -15, 0, 3, 0, 0, 0, 0, -1, 0}, 221.574630, 0.000030, 0.075000},
	{{0, 9, -15, 0, 3, 0, 0, 0, 0, 1, 0}, 221.574630, 0.000030, 0.076000},
	{{0, 9, -15, 0, 4, 0, 0, 0, 0, -1, 0}, 32.237000, 0.000010, 0.075000},
	{{0, 9, -15, 0, 4, 0, 0, 0, 0, 1, 0}, 32.237000, 0.000010, 0.076000},
	{{0, 9, -15, 2, 0, 0, 0, 0, -2, 0, 0}, 231.747700, 0.000040, 0.042000},
	{{0, 9, -15, 2, 0, 0, 0, 0, -2, 2, 0}, 231.728000, 0.000030, 0.405000},
	{{0, 9, -15, 2, 0, 0, 0, 0, 0, -1, 0}, 51.793760, 0.000010, 0.080000},
	{{0, 9, -15, 2, 0, 0, 0, 0, 0, 1, 0}, 51.793760, 0.000010, 0.072000},
	{{0, 9, -14, 0, 0, 0, 0, 0, -4, 2, 0}, 112.962620, 0.000010, 0.045000},
	{{0, 9, -14, 0, 0, 0, 0, 0, -2, -1, 0}, 324.054640, 0.000040, 0.027000},
	{{0, 9, -14, 0, 0, 0, 0, 0, -2, 0, 0}, 304.688500, 0.000070, 0.041000},
	{{0, 9, -14, 0, 0, 0, 0, 0, -2, 1, 0}, 260.811650, 0.000030, 0.092000},
	{{0, 9, -14, 0, 0, 0, 0, 0, -2, 2, 0}, 292.962620, 0.000070, 0.416000},
	{{0, 9, -14, 0, 0, 0, 0, 0, 0, -1, 0}, 226.875540, 0.000020, 0.079000},
	{{0, 9, -14, 0, 0, 0, 0, 0, 0, 0, 0}, 185.045200, 0.000050, 1.588000},
	{{0, 9, -14, 0, 0, 0, 0, 0, 0, 1, 0}, 226.875540, 0.000020, 0.072000},
	{{0, 9, -14, 0, 0, 0, 0, 0, 2, -3, 0}, 57.879530, 0.000030, 0.069000},
	{{0, 9, -14, 0, 0, 0, 0, 0, 2, -1, 0}, 57.879530, 0.000030, 0.083000},
	{{0, 9, -14, 0, 1, 0, 0, 0, -2, 0, 0}, 50.387250, 0.000030, 0.042000},
	{{0, 9, -14, 0, 1, 0, 0, 0, -2, 1, 0}, 230.370610, 0.000010, 0.093000},
	{{0, 9, -14, 0, 1, 0, 0, 0, -2, 3, 0}, 50.387250, 0.000030, 0.064000},
	{{0, 9, -14, 0, 1, 0, 0, 0, 0, -1, 0}, 230.387250, 0.000040, 0.080000},
	{{0, 9, -14, 0, 1, 0, 0, 0, 0, 1, 0}, 230.387250, 0.000040, 0.072000},
	{{0, 9, -14, 0, 2, 0, 0, 0, -2, 0, 0}, 111.896230, 0.000030, 0.042000},
	{{0, 9, -14, 2, 0, 0, 0, 0, -2, 0, 0}, 95.359140, 0.000010, 0.043000},
	{{0, 9, -14, 2, 0, 0, 0, 0, -2, 2, 0}, 95.359140, 0.000010, 0.288000},
	{{0, 9, -13, -3, 0, 0, 0, 0, 0, -1, 0}, 266.567940, 0.000030, 0.076000},
	{{0, 9, -13, -3, 0, 0, 0, 0, 0, 1, 0}, 266.567940, 0.000030, 0.075000},
	{{0, 9, -13, 0, -1, 0, 0, 0, -2, 0, 0}, 241.439700, 0.000080, 0.043000},
	{{0, 9, -13, 0, 0, 0, 0, 0, -2, 0, 0}, 42.753840, 0.000030, 0.043000},
	{{0, 9, -13, 0, 0, 0, 0, 0, -2, 1, 0}, 229.529650, 0.000020, 0.102000},
	{{0, 9, -13, 0, 0, 0, 0, 0, -2, 2, 0}, 39.898320, 0.000010, 0.294000},
	{{0, 9, -13, 0, 0, 0, 0, 0, -2, 3, 0}, 231.778650, 0.000030, 0.060000},
	{{0, 9, -13, 0, 0, 0, 0, 0, 0, -1, 0}, 184.147430, 0.000060, 0.086000},
	{{0, 9, -13, 0, 0, 0, 0, 0, 0, 0, 0}, 168.805750, 0.000030, 0.614000},
	{{0, 9, -13, 0, 0, 0, 0, 0, 0, 1, 0}, 231.778640, 0.000030, 0.067000},
	{{0, 9, -13, 0, 0, 0, 0, 0, 1, 0, -1}, 333.076920, 0.000020, 1.736000},
	{{0, 9, -13, 0, 0, 0, 0, 0, 2, -3, 0}, 61.457420, 0.000040, 0.075000},
	{{0, 9, -13, 0, 0, 0, 0, 0, 2, -2, 0}, 336.467750, 0.000040, 6.930000},
	{{0, 9, -13, 0, 0, 0, 0, 0, 2, -1, 0}, 340.434040, 0.000240, 0.076000},
	{{0, 9, -13, 0, 0, 0, 0, 0, 2, 0, 0}, 75.106420, 0.000020, 0.038000},
	{{0, 9, -13, 0, 1, 0, 0, 0, -2, 0, 0}, 38.840720, 0.000070, 0.043000},
	{{0, 9, -13, 0, 1, 0, 0, 0, 2, -3, 0}, 36.651350, 0.000010, 0.075000},
	{{0, 9, -13, 0, 1, 0, 0, 0, 2, -1, 0}, 295.978740, 0.000020, 0.076000},
	{{0, 9, -13, 0, 2, 0, 0, 0, 2, -2, 0}, 15.670610, 0.000010, 41.153000},
	{{0, 9, -12, -2, 0, 0, 0, 0, -2, 0, 0}, 191.177080, 0.000010, 0.043000},
	{{0, 9, -12, -2, 0, 0, 0, 0, -2, 2, 0}, 191.177080, 0.000010, 0.299000},
	{{0, 9, -12, 0, -1, 0, 0, 0, -2, 0, 0}, 53.653610, 0.000060, 0.045000},
	{{0, 9, -12, 0, -1, 0, 0, 0, -2, 2, 0}, 50.807740, 0.000020, 0.232000},
	{{0, 9, -12, 0, -1, 0, 0, 0, 0, 0, 0}, 232.525830, 0.000020, 0.393000},
	{{0, 9, -12, 0, 0, 0, 0, 0, -2, 0, 0}, 316.728300, 0.000020, 0.045000},
	{{0, 9, -12, 0, 0, 0, 0, 0, -2, 1, 0}, 113.380050, 0.000020, 0.113000},
	{{0, 9, -12, 0, 0, 0, 0, 0, -2, 2, 0}, 300.044240, 0.000030, 0.227000},
	{{0, 9, -12, 0, 0, 0, 0, 0, 0, -1, 0}, 255.255190, 0.000060, 0.094000},
	{{0, 9, -12, 0, 0, 0, 0, 0, 0, 0, 0}, 66.726450, 0.000160, 0.380000},
	{{0, 9, -12, 0, 0, 0, 0, 0, 1, 0, -1}, 249.695590, 0.000050, 0.635000},
	{{0, 9, -12, 0, 0, 0, 0, 0, 2, -2, 0}, 82.164720, 0.000010, 1.169000},
	{{0, 9, -12, 0, 0, 0, 0, 0, 2, -1, 0}, 75.258440, 0.000240, 0.071000},
	{{0, 9, -12, 0, 0, 0, 0, 0, 2, 0, 0}, 255.214820, 0.000060, 0.037000},
	{{0, 9, -12, 0, 2, 0, 0, 0, 2, -1, 0}, 91.195680, 0.000020, 0.070000},
	{{0, 9, -11, 0, 0, 0, 0, 0, -2, 0, 0}, 197.796850, 0.000020, 0.047000},
	{{0, 9, -11, 0, 0, 0, 0, 0, -2, 2, 0}, 341.844090, 0.000030, 0.185000},
	{{0, 9, -11, 0, 0, 0, 0, 0, -2, 3, 0}, 23.954180, 0.000010, 0.054000},
	{{0, 9, -11, 0, 0, 0, 0, 0, -1, 0, 1}, 8.275020, 0.000020, 0.214000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 0, -1, 0}, 316.655110, 0.000020, 0.104000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 0, 0, 0}, 344.551750, 0.000570, 0.276000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 0, 1, 0}, 348.221450, 0.000090, 0.059000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 1, 0, -1}, 168.782100, 0.000140, 0.388000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 1, 0, 0}, 165.110120, 0.000020, 0.063000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 2, -2, 0}, 351.688860, 0.000020, 0.539000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 2, -1, 0}, 352.734880, 0.000610, 0.066000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 2, 0, -2}, 351.936900, 0.000020, 0.657000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 2, 0, 0}, 173.679340, 0.000240, 0.035000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 2, 1, 0}, 170.945790, 0.000030, 0.024000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 4, -1, 0}, 171.733220, 0.000010, 0.025000},
	{{0, 9, -11, 0, 0, 0, 0, 0, 4, 0, 0}, 171.822750, 0.000020, 0.019000},
	{{0, 9, -11, 0, 1, 0, 0, 0, -2, 0, 0}, 270.514840, 0.000010, 0.048000},
	{{0, 9, -11, 0, 1, 0, 0, 0, -2, 2, 0}, 90.514840, 0.000010, 0.182000},
	{{0, 9, -11, 0, 1, 0, 0, 0, 2, 0, 0}, 90.022370, 0.000020, 0.035000},
	{{0, 9, -11, 0, 2, 0, 0, 0, 2, -2, 0}, 179.827190, 0.000020, 0.494000},
	{{0, 9, -11, 2, 0, 0, 0, 0, -2, 0, 0}, 179.190910, 0.000010, 0.050000},
	{{0, 9, -11, 2, 0, 0, 0, 0, -2, 2, 0}, 179.190910, 0.000010, 0.155000},
	{{0, 9, -11, 2, 0, 0, 0, 0, 0, 0, 0}, 0.794190, 0.000010, 0.213000},
	{{0, 9, -11, 2, 0, 0, 0, 0, 2, -2, 0}, 0.046040, 0.000020, 0.343000},
	{{0, 9, -11, 2, 0, 0, 0, 0, 2, 0, 0}, 0.046040, 0.000020, 0.034000},
	{{0, 9, -11, 4, 0, 0, 0, 0, -2, 0, 0}, 57.941990, 0.000010, 0.053000},
	{{0, 9, -11, 4, 0, 0, 0, 0, -2, 2, 0}, 57.941990, 0.000010, 0.133000},
	{{0, 9, -10, 0, -1, 0, 0, 0, -2, 0, 0}, 104.244950, 0.000020, 0.050000},
	{{0, 9, -10, 0, 0, 0, 0, 0, -2, 0, 0}, 275.889600, 0.000050, 0.050000},
	{{0, 9, -10, 0, 0, 0, 0, 0, -2, 1, 0}, 74.995890, 0.000020, 0.146000},
	{{0, 9, -10, 0, 0, 0, 0, 0, -2, 2, 0}, 94.058250, 0.000020, 0.156000},
	{{0, 9, -10, 0, 0, 0, 0, 0, -2, 3, 0}, 271.080050, 0.000010, 0.051000},
	{{0, 9, -10, 0, 0, 0, 0, 0, -1, 0, -1}, 75.743340, 0.000020, 0.047000},
	{{0, 9, -10, 0, 0, 0, 0, 0, -1, 0, 1}, 254.984840, 0.000090, 0.176000},
	{{0, 9, -10, 0, 0, 0, 0, 0, -1, 1, -1}, 75.724020, 0.000020, 0.126000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 0, -1, 0}, 267.468160, 0.000040, 0.116000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 0, 0, 0}, 271.751130, 0.001680, 0.216000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 0, 1, 0}, 272.201750, 0.000380, 0.056000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 0, 2, 0}, 272.375030, 0.000030, 0.032000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 1, -1, -1}, 81.289610, 0.000070, 0.103000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 1, -1, 1}, 255.781680, 0.000030, 0.058000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 1, 0, -1}, 75.779840, 0.000450, 0.280000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 1, 0, 0}, 90.023000, 0.000030, 0.059000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 1, 0, 1}, 75.802060, 0.000010, 0.033000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 1, 1, -1}, 83.445310, 0.000050, 0.059000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 2, -3, 0}, 278.318790, 0.000020, 0.096000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 2, -2, 0}, 244.340990, 0.000050, 0.350000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 2, -1, 0}, 272.608210, 0.001790, 0.062000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 2, 0, -2}, 272.304470, 0.000060, 0.396000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 2, 0, 0}, 94.082800, 0.001170, 0.034000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 2, 1, 0}, 93.630960, 0.000170, 0.023000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 2, 2, 0}, 92.610050, 0.000020, 0.018000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 3, 0, 0}, 92.431840, 0.000020, 0.024000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 4, -2, 0}, 279.104980, 0.000030, 0.036000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 4, -1, 0}, 266.286760, 0.000010, 0.024000},
	{{0, 9, -10, 0, 0, 0, 0, 0, 4, 0, 0}, 98.020000, 0.000040, 0.018000},
	{{0, 9, -10, 0, 1, 0, 0, 0, -2, 0, 0}, 210.576200, 0.000040, 0.050000},
	{{0, 9, -10, 0, 1, 0, 0, 0, -2, 2, 0}, 358.416900, 0.000020, 0.154000},
	{{0, 9, -10, 0, 2, 0, 0, 0, -2, 0, 0}, 232.648400, 0.000010, 0.050000},
	{{0, 9, -10, 0, 2, 0, 0, 0, -2, 2, 0}, 232.648400, 0.000010, 0.152000},
	{{0, 9, -10, 3, 0, 0, 0, 0, -2, 0, 0}, 236.375310, 0.000010, 0.054000},
	{{0, 9, -10, 3, 0, 0, 0, 0, -2, 2, 0}, 236.375310, 0.000010, 0.125000},
	{{0, 9, -10, 3, 0, 0, 0, 0, 0, 0, 0}, 236.824600, 0.000010, 0.161000},
	{{0, 9, -9, 0, -1, 0, 0, 0, -2, 0, 0}, 184.278060, 0.000070, 0.052000},
	{{0, 9, -9, 0, -1, 0, 0, 0, -2, 2, 0}, 185.582980, 0.000010, 0.137000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -4, 1, 0}, 180.132910, 0.000010, 0.033000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -4, 2, 0}, 178.599300, 0.000020, 0.058000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -4, 3, 0}, 180.182280, 0.000010, 0.245000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -2, -2, 0}, 0.169190, 0.000030, 0.022000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -2, -1, 0}, 181.593440, 0.000020, 0.031000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -2, 0, 0}, 180.531090, 0.000390, 0.052000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -2, 1, 0}, 180.237880, 0.000270, 0.171000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -2, 2, 0}, 358.383540, 0.000090, 0.135000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -2, 3, 0}, 359.557600, 0.000010, 0.048000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -1, 0, -1}, 14.729230, 0.000010, 0.050000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -1, 0, 0}, 359.905680, 0.000090, 0.148000},
	{{0, 9, -9, 0, 0, 0, 0, 0, -1, 0, 1}, 17.205390, 0.000010, 0.150000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 0, -1, 0}, 180.002820, 0.000510, 0.131000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 0, 0, 0}, 179.956370, 0.007480, 0.178000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 0, 0, 2}, 0.006680, 0.000010, 0.031000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 0, 1, 0}, 180.199340, 0.001280, 0.053000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 0, 2, 0}, 180.786360, 0.000130, 0.031000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 1, -1, 0}, 3.433320, 0.000010, 0.211000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 1, 0, -1}, 191.192340, 0.000110, 0.219000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 1, 0, 0}, 0.298760, 0.000070, 0.056000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 1, 0, 1}, 191.191650, 0.000020, 0.032000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 1, 1, 0}, 180.048630, 0.000010, 0.032000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 2, -3, 0}, 177.755650, 0.000020, 0.106000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 2, -2, 0}, 179.932270, 0.000270, 0.259000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 2, -1, 0}, 179.997020, 0.005360, 0.058000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 2, 0, -2}, 180.007630, 0.000170, 0.284000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 2, 0, 0}, 0.062110, 0.004510, 0.033000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 2, 0, 2}, 180.003270, 0.000020, 0.017000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 2, 1, 0}, 359.998510, 0.000380, 0.023000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 2, 2, 0}, 359.966930, 0.000050, 0.018000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 3, -1, 0}, 180.046280, 0.000020, 0.034000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 3, 0, 0}, 0.047020, 0.000040, 0.023000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 4, -2, 0}, 179.999280, 0.000080, 0.035000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 4, -1, 0}, 0.002280, 0.000050, 0.024000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 4, 0, 0}, 0.000260, 0.000080, 0.018000},
	{{0, 9, -9, 0, 0, 0, 0, 0, 4, 1, 0}, 0.001810, 0.000010, 0.015000},
	{{0, 9, -9, 0, 1, 0, 0, 0, -2, 0, 0}, 348.227080, 0.000070, 0.053000},
	{{0, 9, -9, 0, 1, 0, 0, 0, -2, 1, 0}, 172.684120, 0.000020, 0.173000},
	{{0, 9, -9, 0, 1, 0, 0, 0, 0, 0, 0}, 175.712380, 0.000020, 0.175000},
	{{0, 9, -9, 0, 1, 0, 0, 0, 2, -2, 0}, 166.883910, 0.000010, 0.254000},
	{{0, 9, -9, 0, 1, 0, 0, 0, 2, 0, 0}, 166.883910, 0.000010, 0.033000},
	{{0, 9, -9, 11, 0, 0, 0, 0, -2, 1, 0}, 301.336980, 0.000010, 285.530000},
	{{0, 9, -8, -3, 0, 0, 0, 0, -2, 0, 0}, 303.086560, 0.000020, 0.051000},
	{{0, 9, -8, -3, 0, 0, 0, 0, -2, 2, 0}, 303.086560, 0.000020, 0.147000},
	{{0, 9, -8, -2, 0, 0, 0, 0, -2, 1, 0}, 136.295090, 0.000010, 0.169000},
	{{0, 9, -8, -2, 0, 0, 0, 0, 0, 0, 0}, 311.830970, 0.000010, 0.180000},
	{{0, 9, -8, 0, -1, 0, 0, 0, -2, 2, 0}, 178.566710, 0.000010, 0.120000},
	{{0, 9, -8, 0, -1, 0, 0, 0, 0, -2, 0}, 180.961830, 0.000020, 0.050000},
	{{0, 9, -8, 0, -1, 0, 0, 0, 0, 0, 0}, 181.368080, 0.000090, 0.153000},
	{{0, 9, -8, 0, -1, 0, 0, 0, 2, 0, 0}, 181.293790, 0.000020, 0.032000},
	{{0, 9, -8, 0, 0, 0, 0, 0, -2, 0, 0}, 255.056750, 0.000020, 0.055000},
	{{0, 9, -8, 0, 0, 0, 0, 0, -2, 1, 0}, 257.168850, 0.000010, 0.206000},
	{{0, 9, -8, 0, 0, 0, 0, 0, -1, -1, 1}, 283.232430, 0.000020, 0.179000},
	{{0, 9, -8, 0, 0, 0, 0, 0, -1, 0, -1}, 103.233410, 0.000120, 0.052000},
	{{0, 9, -8, 0, 0, 0, 0, 0, -1, 0, 1}, 283.231030, 0.000150, 0.130000},
	{{0, 9, -8, 0, 0, 0, 0, 0, -1, 1, -1}, 103.244780, 0.000060, 0.169000},
	{{0, 9, -8, 0, 0, 0, 0, 0, -1, 2, 1}, 283.295250, 0.000020, 0.029000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 0, -1, 0}, 206.211340, 0.000010, 0.151000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 0, 0, 0}, 346.149700, 0.000010, 0.151000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 1, -1, -1}, 103.287640, 0.000090, 0.130000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 1, -1, 1}, 283.295630, 0.000020, 0.052000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 1, 0, -1}, 103.302280, 0.000450, 0.179000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 1, 0, 1}, 103.292840, 0.000030, 0.031000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 1, 1, -1}, 103.295800, 0.000040, 0.053000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 2, -1, 0}, 269.126710, 0.000200, 0.055000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 2, 0, 0}, 89.780930, 0.000300, 0.032000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 2, 1, 0}, 92.059410, 0.000030, 0.022000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 4, -1, 0}, 90.368360, 0.000010, 0.023000},
	{{0, 9, -8, 0, 0, 0, 0, 0, 4, 0, 0}, 84.679790, 0.000020, 0.018000},
	{{0, 9, -7, -3, 0, 0, 0, 0, -2, 0, 0}, 336.204600, 0.000010, 0.053000},
	{{0, 9, -7, -3, 0, 0, 0, 0, -2, 2, 0}, 336.204600, 0.000010, 0.128000},
	{{0, 9, -7, -2, 0, 0, 0, 0, -2, 0, 0}, 0.520160, 0.000010, 0.055000},
	{{0, 9, -7, -2, 0, 0, 0, 0, -2, 2, 0}, 0.520160, 0.000010, 0.120000},
	{{0, 9, -7, 0, -3, 0, 0, 0, -2, 0, 0}, 7.473930, 0.000010, 0.058000},
	{{0, 9, -7, 0, -3, 0, 0, 0, -2, 2, 0}, 7.473930, 0.000010, 0.109000},
	{{0, 9, -7, 0, -2, 0, 0, 0, -2, 2, 0}, 0.283250, 0.000020, 0.108000},
	{{0, 9, -7, 0, -2, 0, 0, 0, 0, 0, 0}, 359.885460, 0.000050, 0.134000},
	{{0, 9, -7, 0, 0, 0, 0, 0, -2, -1, 0}, 26.007390, 0.000020, 0.033000},
	{{0, 9, -7, 0, 0, 0, 0, 0, -2, 0, 0}, 24.896200, 0.000210, 0.058000},
	{{0, 9, -7, 0, 0, 0, 0, 0, -2, 1, 0}, 26.224590, 0.000120, 0.260000},
	{{0, 9, -7, 0, 0, 0, 0, 0, -1, 0, 0}, 206.093600, 0.000020, 0.211000},
	{{0, 9, -7, 0, 0, 0, 0, 0, 0, -1, 0}, 19.985130, 0.000120, 0.178000},
	{{0, 9, -7, 0, 0, 0, 0, 0, 0, 0, 0}, 26.555640, 0.000470, 0.131000},
	{{0, 9, -7, 0, 0, 0, 0, 0, 0, 1, 0}, 15.931400, 0.000040, 0.048000},
	{{0, 9, -7, 0, 0, 0, 0, 0, 2, -2, 0}, 315.874750, 0.000020, 0.171000},
	{{0, 9, -7, 0, 0, 0, 0, 0, 2, -1, 0}, 25.867810, 0.000070, 0.052000},
	{{0, 9, -7, 0, 0, 0, 0, 0, 2, 0, 0}, 210.596180, 0.000080, 0.031000},
	{{0, 9, -6, -4, 0, 0, 0, 0, 0, 0, 0}, 150.731540, 0.000010, 0.154000},
	{{0, 9, -6, 0, -3, 0, 0, 0, -2, 0, 0}, 356.658820, 0.000020, 0.061000},
	{{0, 9, -6, 0, -1, 0, 0, 0, -2, 0, 0}, 24.104600, 0.000010, 0.062000},
	{{0, 9, -6, 0, -1, 0, 0, 0, -2, 2, 0}, 204.104600, 0.000010, 0.097000},
	{{0, 9, -6, 0, 0, 0, 0, 0, -2, 1, 0}, 275.913380, 0.000020, 0.351000},
	{{0, 9, -6, 0, 0, 0, 0, 0, 0, 0, 0}, 262.412770, 0.000040, 0.116000},
	{{0, 9, -6, 0, 0, 0, 0, 0, 2, -3, 0}, 39.367680, 0.000010, 0.156000},
	{{0, 9, -6, 0, 0, 0, 0, 0, 2, -2, 0}, 39.473680, 0.000010, 0.146000},
	{{0, 9, -6, 0, 0, 0, 0, 0, 2, -1, 0}, 39.367680, 0.000010, 0.050000},
	{{0, 9, -6, 0, 0, 0, 0, 0, 2, 0, 0}, 39.473680, 0.000010, 0.030000},
	{{0, 9, -5, 0, 0, 0, 0, 0, -2, 1, 0}, 217.161630, 0.000010, 0.540000},
	{{0, 9, -5, 0, 0, 0, 0, 0, 0, -1, 0}, 307.957840, 0.000020, 0.276000},
	{{0, 9, -5, 0, 0, 0, 0, 0, 0, 0, 0}, 308.203750, 0.000020, 0.104000},
	{{0, 9, -5, 0, 0, 0, 0, 0, 0, 2, 0}, 308.203750, 0.000020, 0.028000},
	{{0, 9, -5, 0, 0, 0, 0, 0, 2, -2, 0}, 129.264790, 0.000020, 0.127000},
	{{0, 9, -5, 0, 0, 0, 0, 0, 2, -1, 0}, 128.260610, 0.000010, 0.047000},
	{{0, 9, -5, 0, 0, 0, 0, 0, 2, 0, 0}, 127.741750, 0.000060, 0.029000},
	{{0, 9, -5, 0, 0, 0, 0, 0, 2, 1, 0}, 128.260610, 0.000010, 0.021000},
	{{0, 9, -5, 0, 0, 0, 0, 0, 2, 2, 0}, 128.223700, 0.000040, 0.016000},
	{{0, 9, -4, 0, -1, 0, 0, 0, 2, -2, 0}, 309.931610, 0.000010, 0.114000},
	{{0, 9, -4, 0, -1, 0, 0, 0, 2, 0, 0}, 309.931610, 0.000010, 0.028000},
	{{0, 9, -4, 0, 0, -1, 0, 0, -3, 2, 0}, 128.093570, 0.000050, 969.469000},
	{{0, 9, -4, 0, 0, 0, 0, 0, -2, 0, 0}, 302.084920, 0.000020, 0.071000},
	{{0, 9, -4, 0, 0, 0, 0, 0, -2, 2, 0}, 302.084920, 0.000020, 0.081000},
	{{0, 9, -4, 0, 0, 0, 0, 0, 2, -2, 0}, 36.070820, 0.000020, 0.113000},
	{{0, 9, -4, 0, 0, 0, 0, 0, 2, 0, 0}, 36.070820, 0.000020, 0.028000},
	{{0, 9, -3, 0, -2, 0, 0, 0, -2, 0, 0}, 87.068530, 0.000010, 0.075000},
	{{0, 9, -3, 0, -2, 0, 0, 0, -2, 1, 0}, 349.737370, 0.000070, 48.456000},
	{{0, 9, -3, 0, -2, 0, 0, 0, -2, 2, 0}, 267.068530, 0.000010, 0.076000},
	{{0, 9, -3, 0, -2, 0, 0, 0, 2, -2, 0}, 128.380120, 0.000010, 0.103000},
	{{0, 9, -3, 0, -2, 0, 0, 0, 2, 0, 0}, 308.380120, 0.000010, 0.028000},
	{{0, 9, -3, 0, -1, 0, 0, 0, -2, 1, 0}, 354.017180, 0.000020, 15.707000},
	{{0, 9, -3, 0, 0, -4, 0, 0, -2, 1, 0}, 201.985560, 0.000030, 82.115000},
	{{0, 9, -3, 0, 0, 0, 0, 0, -2, 0, 0}, 21.570500, 0.000010, 0.076000},
	{{0, 9, -3, 0, 0, 0, 0, 0, -2, 1, 0}, 62.474790, 0.000040, 6.758000},
	{{0, 9, -3, 0, 0, 0, 0, 0, -2, 2, 0}, 21.570500, 0.000010, 0.075000},
	{{0, 9, -3, 0, 0, 0, 0, 0, 2, 0, 0}, 334.114260, 0.000020, 0.027000},
	{{0, 9, -2, 0, -3, 0, 0, 0, -1, 0, 0}, 80.896530, 0.000010, 124.238000},
	{{0, 9, -1, -4, 0, 0, 0, 0, -2, 1, 0}, 65.971840, 0.000040, 47.095000},
	{{0, 9, -1, 0, -4, 0, 0, 0, 0, -1, 0}, 29.003780, 0.000010, 27.222000},
	{{0, 9, 8, -21, 0, 0, 0, 0, -2, 1, 0}, 243.481360, 0.000030, 57.479000},
	{{0, 10, -19, 0, 0, 0, 0, 0, -2, 0, 0}, 289.615290, 0.000020, 0.036000},
	{{0, 10, -19, 0, 0, 0, 0, 0, -2, 2, 0}, 289.615920, 0.000010, 1.030000},
	{{0, 10, -18, 0, 0, -1, 0, 0, -2, 2, 0}, 122.039490, 0.000010, 212.821000},
	{{0, 10, -18, 0, 0, 0, 0, 0, -2, 1, 0}, 66.910610, 0.000020, 0.076000},
	{{0, 10, -18, 0, 0, 0, 0, 0, -2, 3, 0}, 352.581920, 0.000010, 0.075000},
	{{0, 10, -18, 0, 2, 0, 0, 0, -2, 1, 0}, 44.202600, 0.000010, 0.077000},
	{{0, 10, -18, 0, 2, 0, 0, 0, -2, 3, 0}, 44.202600, 0.000010, 0.074000},
	{{0, 10, -17, 0, 0, 0, 0, 0, 0, 0, 0}, 307.725840, 0.000020, 1.343000},
	{{0, 10, -16, 0, 0, 0, 0, 0, -2, -1, 0}, 224.433860, 0.000010, 0.026000},
	{{0, 10, -16, 0, 0, 0, 0, 0, -2, 1, 0}, 224.433860, 0.000010, 0.089000},
	{{0, 10, -16, 0, 0, 0, 0, 0, 0, 0, 0}, 279.138350, 0.000020, 3.918000},
	{{0, 10, -16, 0, 0, 0, 0, 0, 2, -1, 0}, 44.433860, 0.000020, 0.085000},
	{{0, 10, -16, 0, 1, 0, 0, 0, -2, 0, 0}, 141.810850, 0.000030, 0.041000},
	{{0, 10, -16, 0, 1, 0, 0, 0, 0, 2, -2}, 321.810850, 0.000010, 166.045000},
	{{0, 10, -16, 0, 2, 0, 0, 0, -2, 0, 0}, 51.540530, 0.000020, 0.041000},
	{{0, 10, -16, 0, 2, 0, 0, 0, -2, 1, 0}, 231.565640, 0.000020, 0.090000},
	{{0, 10, -16, 0, 2, 0, 0, 0, -2, 2, 0}, 51.540530, 0.000020, 0.455000},
	{{0, 10, -16, 0, 2, 0, 0, 0, -2, 3, 0}, 231.565640, 0.000020, 0.065000},
	{{0, 10, -16, 0, 2, 0, 0, 0, 0, -1, 0}, 51.540540, 0.000020, 0.078000},
	{{0, 10, -16, 0, 2, 0, 0, 0, 0, 1, 0}, 51.540540, 0.000020, 0.073000},
	{{0, 10, -15, 0, 0, 0, 0, 0, -2, -1, 0}, 324.060360, 0.000030, 0.027000},
	{{0, 10, -15, 0, 0, 0, 0, 0, -2, 0, 0}, 357.637020, 0.000010, 0.043000},
	{{0, 10, -15, 0, 0, 0, 0, 0, -2, 1, 0}, 328.788480, 0.000010, 0.098000},
	{{0, 10, -15, 0, 0, 0, 0, 0, -2, 2, 0}, 339.270370, 0.000030, 0.330000},
	{{0, 10, -15, 0, 0, 0, 0, 0, 0, 0, 0}, 173.319720, 0.000020, 0.797000},
	{{0, 10, -15, 0, 0, 0, 0, 0, 1, 0, -1}, 55.652120, 0.000010, 4.964000},
	{{0, 10, -15, 0, 0, 0, 0, 0, 2, -3, 0}, 237.880510, 0.000040, 0.073000},
	{{0, 10, -15, 0, 0, 0, 0, 0, 2, -1, 0}, 237.880510, 0.000040, 0.079000},
	{{0, 10, -15, 0, 0, 0, 0, 0, 2, 0, 0}, 57.911670, 0.000020, 0.038000},
	{{0, 10, -15, 0, 1, 0, 0, 0, -2, 0, 0}, 50.367420, 0.000030, 0.043000},
	{{0, 10, -15, 0, 1, 0, 0, 0, -2, 1, 0}, 50.392330, 0.000040, 0.099000},
	{{0, 10, -15, 0, 1, 0, 0, 0, 0, -1, 0}, 50.392330, 0.000010, 0.084000},
	{{0, 10, -15, 0, 1, 0, 0, 0, 0, 1, 0}, 50.392330, 0.000010, 0.069000},
	{{0, 10, -15, 0, 2, 0, 0, 0, -2, 0, 0}, 111.879660, 0.000020, 0.043000},
	{{0, 10, -15, 1, 0, 0, 0, 0, 2, -2, 0}, 231.202650, 0.000010, 77.527000},
	{{0, 10, -15, 2, 0, 0, 0, 0, -2, 0, 0}, 95.342580, 0.000010, 0.045000},
	{{0, 10, -15, 2, 0, 0, 0, 0, -2, 2, 0}, 95.342580, 0.000010, 0.244000},
	{{0, 10, -14, 0, -1, 0, 0, 0, -2, 0, 0}, 241.433790, 0.000070, 0.044000},
	{{0, 10, -14, 0, 0, 0, 0, 0, -2, 0, 0}, 69.427950, 0.000030, 0.044000},
	{{0, 10, -14, 0, 0, 0, 0, 0, -2, 1, 0}, 231.649260, 0.000010, 0.108000},
	{{0, 10, -14, 0, 0, 0, 0, 0, -