/*

Copyright (C) 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <libnd.h>
#include "libnd_ta_tasks.h"

/* These are all defined in libnd_ta.c */
extern LND_Protocol    *ta_udp;
extern LND_Protocol    *ta_tcp;
extern LND_Protocol    *ta_ip;

void 
libnd_ta_task_basic(LND_TAnalysis *ta, LND_Packet *packet)
{
  if (!ta || !packet)
    return;

  ta->packet_count++;
  ta->byte_count += packet->ph.len;

  if (!libnd_packet_has_proto(packet, ta_ip))
    {
      ta->non_ip_packet_count++;
      ta->non_ip_byte_count += packet->ph.len;
    }
}


void 
libnd_ta_task_ip_protocols(LND_TAnalysis *ta, LND_Packet *packet)
{
  struct ip     *iphdr = NULL;
  LND_TAData    *data;

  if (!ta || !packet)
    return;

  if (! libnd_packet_has_proto(packet, ta_ip))
    return;
  
  if (! (iphdr = (struct ip*) libnd_packet_get_data(packet, ta_ip, 0)))
    {
      D(("ERROR -- were supposed to find an IP header ...\n"));
      return;
    }

  /* Find structure in IP proto hashtable for this protocol.
   * If necessary, create that entry.
   */
  if (! (data = g_hash_table_lookup(ta->ip_conns, GINT_TO_POINTER((int)iphdr->ip_p))))
    {
      data = g_new0(LND_TAData, 1);      
      g_hash_table_insert(ta->ip_conns,
			  GINT_TO_POINTER((int)iphdr->ip_p),
			  data);
    }
  
  D_ASSERT_PTR(data);

  data->packets_count++;
  data->bytes_count += packet->ph.len;
}


void 
libnd_ta_task_ports(LND_TAnalysis *ta, LND_Packet *packet)
{
  struct udphdr *udphdr = NULL;
  struct tcphdr *tcphdr = NULL;
  struct ip     *iphdr  = NULL;
  LND_TAConn     conn, *conn_ptr;
  LND_TAData    *conn_data;
  LND_ProtoData *pd;
  GList *l;

  if (!ta || !packet)
    return;

  if (! libnd_packet_has_proto(packet, ta_ip))
    {
      D(("No IP ...\n"));
      return;
    }

  if (! libnd_packet_has_proto(packet, ta_tcp) &&
      ! libnd_packet_has_proto(packet, ta_udp))
    {
      D(("Neither TCP nor UDP ...\n"));
      return;
    }
  
  /* We now can be sure that we have IP and either TCP or UDP */

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (LND_ProtoData *) l->data;
      
      if (pd->inst.proto->id == ta_ip->id &&
	  pd->inst.nesting == 0)
	{
	  iphdr = (struct ip *) pd->data;
	  continue;
	}
      
      if (pd->inst.proto->id == ta_tcp->id &&
	  pd->inst.nesting == 0)
	{
	  tcphdr = (struct tcphdr *) pd->data;
	  break;
	}
      
      if (pd->inst.proto->id == ta_udp->id &&
	  pd->inst.nesting == 0)
	{
	  udphdr = (struct udphdr *) pd->data;      
	  break;
	}
    }

  if (!tcphdr && !udphdr)
    {
      D(("Something's wrong -- we were supposed to see either UDP or TCP!\n"));
      return;
    }
  
  if (tcphdr)
    {
      conn.sport  = tcphdr->th_sport;
      conn.dport  = tcphdr->th_dport;
      conn.is_tcp = TRUE;
    }
  else
    {
      conn.sport  = udphdr->uh_sport;
      conn.dport  = udphdr->uh_dport;      
      conn.is_tcp = FALSE;
    }

  conn_ptr = &conn;

  if (! (conn_data = (LND_TAData *) g_hash_table_lookup(ta->port_conns,
							(gconstpointer) &conn)))
    {
      /* This port pair is not yet in the table. Add it. */
      conn_ptr  = g_new0(LND_TAConn, 1);
      conn_data = g_new0(LND_TAData, 1); 

      *conn_ptr = conn;
      g_hash_table_insert(ta->port_conns,
			  (gpointer) conn_ptr,
			  (gpointer) conn_data);
    }

  D_ASSERT_PTR(conn_data);

  if (conn_ptr->is_tcp && (tcphdr->th_flags & TH_SYN))
    {
      if (! (tcphdr->th_flags & TH_ACK))
	{
	  conn_ptr->sport  = tcphdr->th_sport;
	  conn_ptr->dport  = tcphdr->th_dport;
	}
      else
	{
	  conn_ptr->dport  = tcphdr->th_sport;
	  conn_ptr->sport  = tcphdr->th_dport;
	}
    }

  conn_data->packets_count++;
  conn_data->bytes_count += packet->ph.len;
}


static void
ta_ports_dport(LND_TAConn    *conn,
	       LND_TAData    *data,
	       LND_TAnalysis *ta)
{
  LND_TAConn *new_conn_ptr, new_conn;
  LND_TAData *new_data_ptr;

  /* Copy the original connection identifier, but not the source port! */
  memset(&new_conn, 0, sizeof(LND_TAConn));
  new_conn.dport = conn->dport;
  new_conn.is_tcp =conn->is_tcp;

  if (! (new_data_ptr = g_hash_table_lookup(ta->port_conns_dport, &new_conn)))
    {
      new_conn_ptr = g_new0(LND_TAConn, 1);
      new_data_ptr = g_new0(LND_TAData, 1);
      
      *new_conn_ptr = new_conn;
      g_hash_table_insert(ta->port_conns_dport, new_conn_ptr, new_data_ptr);
    }
  
  new_data_ptr->packets_count += data->packets_count;
  new_data_ptr->bytes_count += data->bytes_count;
}


void 
libnd_ta_task_dports(LND_TAnalysis *ta)
{
  if (!ta)
    return;
  
  g_hash_table_foreach(ta->port_conns, (GHFunc) ta_ports_dport, ta);
}


static void
ta_ports_sport(LND_TAConn    *conn,
	       LND_TAData    *data,
	       LND_TAnalysis *ta)
{
  LND_TAConn *new_conn_ptr, new_conn;
  LND_TAData *new_data_ptr;

  /* Copy the original connection identifier, but not the source port! */
  memset(&new_conn, 0, sizeof(LND_TAConn));
  new_conn.sport = conn->sport;
  new_conn.is_tcp =conn->is_tcp;

  if (! (new_data_ptr = g_hash_table_lookup(ta->port_conns_sport, &new_conn)))
    {
      new_conn_ptr = g_new0(LND_TAConn, 1);
      new_data_ptr = g_new0(LND_TAData, 1);
      
      *new_conn_ptr = new_conn;
      g_hash_table_insert(ta->port_conns_sport, new_conn_ptr, new_data_ptr);
    }
  
  new_data_ptr->packets_count += data->packets_count;
  new_data_ptr->bytes_count += data->bytes_count;
}


void 
libnd_ta_task_sports(LND_TAnalysis *ta)
{
  if (!ta)
    return;
  
  g_hash_table_foreach(ta->port_conns, (GHFunc) ta_ports_sport, ta);
}


void 
libnd_ta_task_tcp_flows(LND_TAnalysis *ta, LND_Packet *packet)
{
  LND_TCPConn   *conn;
  LND_TAData    *conn_data;

  if (!ta || !packet)
    return;
  if (! libnd_packet_has_proto(packet, ta_ip) ||
      ! libnd_packet_has_proto(packet, ta_tcp))
    return;
  
  if (! (conn = (LND_TCPConn *) libnd_conn_table_lookup(ta->tcp_conns, packet)))
    {
      conn = libnd_tcpconn_new(packet);
      libnd_conn_table_add(ta->tcp_conns, (LND_ConnID *) conn);
      conn_data = g_new0(LND_TAData, 1);
      libnd_conn_data_set((LND_ConnID *) conn, "data", conn_data);
    }

  D_ASSERT_PTR(conn);

  conn_data = (LND_TAData *) libnd_conn_data_get((LND_ConnID *) conn, "data");
  conn_data->packets_count++;
  conn_data->bytes_count += packet->ph.len;

  /* Update TCP connection state */
  libnd_tcpconn_update(conn, packet);
}


void
libnd_ta_task_basic_print(const LND_TAnalysis *ta, FILE *output)
{
  if (!ta || !output)
    return;

  fprintf(output,
	  "\n"
	  "# BASIC ANALYSIS:\n"
	  "# ===============\n"
	  "#\n"
	  "# Total packets -- non-IP packets -- %%non-IP\n\n"
	  "%llu\t%llu\t%5.2f\n\n"
	  "# Total bytes -- non-IP bytes -- %%non-IP\n\n"
	  "%llu\t%llu\t%5.2f\n\n",
	  (unsigned long long int) ta->packet_count,
	  (unsigned long long int) ta->non_ip_packet_count,
	  ((double) ta->non_ip_packet_count ) / ta->packet_count,
	  (unsigned long long int) ta->byte_count,
	  (unsigned long long int) ta->non_ip_byte_count,
	  ((double) ta->non_ip_byte_count ) / ta->byte_count);
}


typedef struct ta_ip_cb_data
{
  const LND_TAnalysis  *ta;
  FILE                 *output;
} TA_IP_CBData;

static void
ta_write_ip_stats_entry(gpointer      protop,
			LND_TAData   *data,
			TA_IP_CBData *cb_data)
{
  fprintf(cb_data->output, "%u\t%u\t%llu\t%5.2f\t%5.2f\n",
	  GPOINTER_TO_INT(protop), data->packets_count,
	  (unsigned long long int) data->bytes_count,
	  ((double) data->packets_count) / cb_data->ta->packet_count * 100,
	  ((double) data->bytes_count) / cb_data->ta->byte_count * 100);
	  
}

void 
libnd_ta_task_ip_protocols_print(const LND_TAnalysis *ta, FILE *output)
{
  TA_IP_CBData cb_data;

  if (!ta || !output)
    return;
  
  cb_data.ta = ta;
  cb_data.output = output;

  fprintf(output,
	  "\n"
	  "# IP PROTOCOL ANALYSIS:\n"
	  "# =====================\n"
	  "#\n"
	  "# Aggregates packets and bytes per IP protocol payload type.\n"
	  "# proto number -- #packets -- #bytes -- %%packets -- %%bytes\n\n");
  
  g_hash_table_foreach(ta->ip_conns, (GHFunc) ta_write_ip_stats_entry, &cb_data);
}


typedef struct ta_ports_cb_data
{
  const LND_TAnalysis  *ta;
  FILE                 *output;
} TA_PortsCBData;


static void
ta_write_portpair(LND_TAConn     *conn,
		  LND_TAData     *data,
		  TA_PortsCBData *cb_data)
{
  fprintf(cb_data->output, "%u\t%u\t%i\t%u\t%llu\t%5.2f\t%5.2f\n",
	  ntohs(conn->sport), ntohs(conn->dport), conn->is_tcp,
	  data->packets_count, data->bytes_count,
	  ((double) data->packets_count) / cb_data->ta->packet_count * 100,
	  ((double) data->bytes_count) / cb_data->ta->byte_count * 100);
}


void 
libnd_ta_task_ports_print(const LND_TAnalysis *ta, FILE *output)
{
  TA_PortsCBData cb_data;

  if (!ta || !output)
    return;

  cb_data.ta = ta;
  cb_data.output = output;

  fprintf(output,
	  "\n"
	  "# TCP/UDP PORTNUMBER ANALYSIS:\n"
	  "# ============================\n"
	  "#\n"
	  "# src port -- dst port -- TCP(1) or UDP(0) -- #packets -- #bytes -- %%packets -- %%bytes\n\n");
  
  g_hash_table_foreach(ta->port_conns, (GHFunc) ta_write_portpair, &cb_data);
}


static void
ta_write_dport(LND_TAConn     *conn,
	      LND_TAData     *data,
	      TA_PortsCBData *cb_data)
{
  fprintf(cb_data->output, "%u\t%i\t%u\t%llu\t%5.2f\t%5.2f\n",
	  ntohs(conn->dport),
	  conn->is_tcp,
	  data->packets_count, data->bytes_count,
	  ((double) data->packets_count) / cb_data->ta->packet_count * 100,
	  ((double) data->bytes_count) / cb_data->ta->byte_count * 100);
	  
}


void 
libnd_ta_task_dports_print(const LND_TAnalysis *ta, FILE *output)
{
  TA_PortsCBData cb_data;

  if (!ta || !output)
    return;

  cb_data.ta = ta;
  cb_data.output = output;

  fprintf(output,
	  "\n"
	  "# TCP/UDP DESTINATION PORTS:\n"
	  "# =========================\n"
	  "#\n"
	  "# Aggregates packets and bytes per destination port. For TCP, we can\n"
	  "# do this right based on whether or not a SYN we see has an ACK or not.\n"
	  "# For UDP and for the case when we don't see SYNs in a TCP connection,\n"
	  "# whatever is the destination port on the first packet wins.\n"
	  "#\n"
	  "# dest port -- TCP(1)/UDP(0) -- #packets -- #bytes -- %%packets -- %%bytes\n\n");
  
  g_hash_table_foreach(ta->port_conns_dport, (GHFunc) ta_write_dport, &cb_data);
}


static void
ta_write_sport(LND_TAConn     *conn,
	      LND_TAData     *data,
	      TA_PortsCBData *cb_data)
{
  fprintf(cb_data->output, "%u\t%i\t%u\t%llu\t%5.2f\t%5.2f\n",
	  ntohs(conn->sport),
	  conn->is_tcp,
	  data->packets_count, data->bytes_count,
	  ((double) data->packets_count) / cb_data->ta->packet_count * 100,
	  ((double) data->bytes_count) / cb_data->ta->byte_count * 100);
	  
}


void 
libnd_ta_task_sports_print(const LND_TAnalysis *ta, FILE *output)
{
  TA_PortsCBData cb_data;

  if (!ta || !output)
    return;

  cb_data.ta = ta;
  cb_data.output = output;

  fprintf(output,
	  "\n"
	  "# TCP/UDP SOURCE PORTS:\n"
	  "# =====================\n"
	  "#\n"
	  "# Aggregates packets and bytes per souce port. For TCP, we can\n"
	  "# do this right based on whether or not a SYN we see has an ACK or not.\n"
	  "# For UDP and for the case when we don't see SYNs in a TCP connection,\n"
	  "# whatever is the destination port on the first packet wins.\n"
	  "#\n"
	  "# src port -- TCP(1)/UDP(0) -- #packets -- #bytes -- %%packets -- %%bytes\n\n");
  
  g_hash_table_foreach(ta->port_conns_sport, (GHFunc) ta_write_sport, &cb_data);
}
	  


static gboolean
ta_write_flow_stats_entry(LND_ConnID *conn, TA_PortsCBData *cb_data)
{
  struct in_addr ip_src, ip_dst;
  guint16 th_sport, th_dport;

  LND_TAData *data = libnd_conn_data_get(conn, "data");

  libnd_conn_get_src(conn, &ip_src, &th_sport);
  libnd_conn_get_dst(conn, &ip_dst, &th_dport);
  
  fprintf(cb_data->output, "%s\t", inet_ntoa(ip_src));
  fprintf(cb_data->output, "%s\t", inet_ntoa(ip_dst));
  fprintf(cb_data->output, "%5u\t%5u\t", ntohs(th_sport), ntohs(th_dport));
  fprintf(cb_data->output, "%llu\t%llu\t%5.2f\t%5.2f\n",
	  (unsigned long long int) data->packets_count,
	  (unsigned long long int) data->bytes_count,
	  ((double) data->packets_count) / cb_data->ta->packet_count * 100,
	  ((double) data->bytes_count) / cb_data->ta->byte_count * 100);
  
  return TRUE;
}


void 
libnd_ta_task_flows_print(const LND_TAnalysis *ta, FILE *output)
{
  TA_PortsCBData cb_data;

  if (!ta || !output)
    return;

  cb_data.ta = ta;
  cb_data.output = output;

  fprintf(output,
	  "\n"
	  "# TCP FLOW  ANALYSIS:\n"
	  "# ===================\n"
	  "#\n"
	  "# Aggregates packets and bytes per TCP flow, doing full TCP connection state tracking.\n"
	  "# IP src -- IP dst -- src port -- dst port -- #packets -- #bytes -- %%packets -- %%bytes\n\n");

  libnd_conn_table_set_policy(ta->tcp_conns, LND_CONN_TABLE_INCLUDE_DEAD);
  libnd_conn_table_foreach(ta->tcp_conns, (LND_ConnFunc) ta_write_flow_stats_entry, &cb_data);
}
