/*

Copyright (C) 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include "libnd_ta.h"
#include "libnd_ta_tasks.h"

/* We need these protocols: */
LND_Protocol    *ta_udp;
LND_Protocol    *ta_tcp;
LND_Protocol    *ta_ip;

/* GHashFunc to hash our connection structures: */
static guint
ta_hash_conn(const LND_TAConn *conn)
{
  return (conn->sport ^ conn->dport ^ conn->is_tcp);
}


/* GCompareFunc to compare our connection structures: */
static gint 
ta_compare_conn(const LND_TAConn *c1, const LND_TAConn *c2)
{
  return ((c1->sport  == c2->sport &&
	   c1->dport  == c2->dport &&
	   c1->is_tcp == c2->is_tcp)   ||
	  (c1->sport  == c2->dport &&
	   c1->dport  == c2->sport &&
	   c1->is_tcp == c2->is_tcp));
}

static gboolean
ta_clear_port_hashtable_entry(gpointer	key,
			      gpointer	value,
			      gpointer	user_data)
{
  g_free(key);
  g_free(value);

  return TRUE;
  TOUCH(user_data);
}

static gboolean
ta_clear_ip_hashtable_entry(gpointer	key,
			    gpointer	value,
			    gpointer	user_data)
{
  g_free(value); /* Just free a LND_TAData structure */

  return TRUE;
  TOUCH(key);
  TOUCH(user_data);
}

/*------------------------------------------------------------- */

LND_TAnalysis  *
libnd_ta_new(void)
{
  LND_TAnalysis *ta;
  
  if (! (ta = g_new0(LND_TAnalysis, 1)))
    return NULL;

  if (! (ta->traces = libnd_traceset_new()))
    {
      D(("out of memory"));
      g_free(ta);
      return NULL;
    }

  ta->port_conns = g_hash_table_new((GHashFunc) ta_hash_conn,
				    (GCompareFunc) ta_compare_conn);
  ta->port_conns_dport = g_hash_table_new((GHashFunc) ta_hash_conn,
					  (GCompareFunc) ta_compare_conn);
  ta->port_conns_sport = g_hash_table_new((GHashFunc) ta_hash_conn,
					  (GCompareFunc) ta_compare_conn);
  
  ta->ip_conns   = g_hash_table_new((GHashFunc) g_direct_hash,
				    (GCompareFunc) g_direct_equal);
  
  ta->tcp_conns  = libnd_conn_table_new(LND_CONN_TABLE_IGNORE_DEAD);

  libnd_ta_do_all(ta);

  return ta;
}


static gboolean
ta_free_cb(LND_ConnID *conn, void *user_data)
{
  LND_TAData *data = libnd_conn_data_remove(conn, "data");
  g_free(data);

  return TRUE;
  TOUCH(user_data);
}

void            
libnd_ta_free(LND_TAnalysis *ta)
{
  if (!ta)
    return;

  libnd_traceset_free(ta->traces);
  
  g_hash_table_foreach_remove(ta->ip_conns,
			      ta_clear_ip_hashtable_entry, NULL);
  g_hash_table_destroy(ta->ip_conns);
  g_hash_table_foreach_remove(ta->port_conns,
			      ta_clear_port_hashtable_entry, NULL);
  g_hash_table_destroy(ta->port_conns);
  
  g_hash_table_foreach_remove(ta->port_conns_dport,
			      ta_clear_port_hashtable_entry, NULL);
  g_hash_table_destroy(ta->port_conns_dport);
  g_hash_table_foreach_remove(ta->port_conns_sport,
			      ta_clear_port_hashtable_entry, NULL);
  g_hash_table_destroy(ta->port_conns_sport);
  
  libnd_conn_table_foreach(ta->tcp_conns, ta_free_cb, NULL);
  libnd_conn_table_free(ta->tcp_conns);

  g_free(ta);
}


gboolean
libnd_ta_set_traceset(LND_TAnalysis *ta, const LND_TraceSet *set)
{
  LND_TraceSet *set_copy;

  if (!ta || !set)
    return FALSE;

  if (! (set_copy = libnd_traceset_copy(set)))
    {
      D(("Out of memory.\n"));
      return FALSE;
    }

  libnd_traceset_free(ta->traces);
  ta->traces = set_copy;

  return TRUE;
}


gboolean
libnd_ta_add_trace(LND_TAnalysis *ta, LND_Trace *trace)
{
  if (!ta)
    return FALSE;
  
  return libnd_traceset_add_trace(ta->traces, trace);
}


gboolean 
libnd_ta_add_trace_name(LND_TAnalysis *ta, const char *file_name)
{
  if (!ta)
    return FALSE;

  return libnd_traceset_add_trace_name(ta->traces, file_name);
}


gboolean        
libnd_ta_add_trace_name_list(LND_TAnalysis *ta, const char *file_name)
{
  if (!ta)
    return FALSE;

  return libnd_traceset_add_trace_name_list(ta->traces, file_name);
}


static gboolean
ta_get_protocols(void)
{
  if (!ta_ip)
    {
      if (! (ta_ip = libnd_proto_registry_find(LND_PROTO_LAYER_NET,
					       0x0800)))
	return FALSE;
    }

  if (!ta_tcp)
    {
      if (! (ta_tcp = libnd_proto_registry_find(LND_PROTO_LAYER_TRANS,
						IPPROTO_TCP)))
	return FALSE;
    }

  if (!ta_udp)
    {
      if (! (ta_udp = libnd_proto_registry_find(LND_PROTO_LAYER_TRANS,
						IPPROTO_UDP)))
	return FALSE;
    }
  
  return TRUE;
}


void            
libnd_ta_set_progress_callback(LND_TAnalysis *ta,
			       LND_TAProgressCB callback,
			       void *user_data)
{
  if (!ta)
    return;

  ta->cb = callback;
  ta->cb_data = user_data;
}


static gboolean
ta_analyze_cb(LND_TraceSet *set, LND_Trace *trace, LND_TAnalysis *ta)
{
  LND_PacketIterator  pit;
  LND_Packet *packet;

  D(("Handling trace %s\n", trace->filename));
  
  /* Iterate over entire individual trace, read-only. */
  for (libnd_pit_init_mode(&pit, trace, LND_PACKET_IT_AREA_R);
       libnd_pit_get(&pit);
       libnd_pit_next(&pit))
    {
      packet = libnd_pit_get(&pit);
      
      /* We always need to do the basic analysis to get packet and byte counts. */
      libnd_ta_task_basic(ta, packet);	  
      if (ta->do_ip) libnd_ta_task_ip_protocols(ta, packet);	  

      /* Port pair analysis needs to be done in any case, if we want any of
       * the three different port number output options.
       */
      if (ta->do_ports || ta->do_src_ports || ta->do_dst_ports)
	libnd_ta_task_ports(ta, packet);	  

      if (ta->do_flows) libnd_ta_task_tcp_flows(ta, packet);
    }
  
  if (ta->cb)
    ta->cb(ta, ++ta->it_count, ta->cb_data);
  
  return TRUE;
  TOUCH(set);
}


gboolean        
libnd_ta_analyze(LND_TAnalysis *ta)
{
  if (! ta)
    return FALSE;

  /* Make sure we have IP, TCP & UDP */
  if (! ta_get_protocols())
    return FALSE;
  
  ta->it_count = 0;
  libnd_traceset_foreach(ta->traces, (LND_TraceSetCB) ta_analyze_cb, ta);
  
  if (ta->do_src_ports) libnd_ta_task_sports(ta);
  if (ta->do_dst_ports) libnd_ta_task_dports(ta);
  
  return TRUE;
}


void            
libnd_ta_do_all(LND_TAnalysis *ta)
{
  if (!ta)
    return;

  ta->do_basic = TRUE;
  ta->do_ip    = TRUE;
  ta->do_ports = TRUE;
  ta->do_src_ports = TRUE;
  ta->do_dst_ports = TRUE;
  ta->do_flows = TRUE;
}


void            
libnd_ta_do_none(LND_TAnalysis *ta)
{
  if (!ta)
    return;

  ta->do_basic = FALSE;
  ta->do_ip    = FALSE;
  ta->do_ports = FALSE;
  ta->do_src_ports = FALSE;
  ta->do_dst_ports = FALSE;
  ta->do_flows = FALSE;
}


gboolean
libnd_ta_write_results(const LND_TAnalysis *ta, FILE *output)
{
  if (!ta || !output)
    return FALSE;

  fprintf(output,
	  "# Netdude Traffic Analysis Output\n"
	  "# Files handled: %i\n"
	  "#\n", libnd_traceset_get_size(ta->traces));

  if (ta->do_basic) libnd_ta_task_basic_print(ta, output);
  if (ta->do_ip) libnd_ta_task_ip_protocols_print(ta, output);  
  if (ta->do_ports) libnd_ta_task_ports_print(ta, output);  
  if (ta->do_src_ports) libnd_ta_task_sports_print(ta, output);  
  if (ta->do_dst_ports) libnd_ta_task_dports_print(ta, output);  
  if (ta->do_flows) libnd_ta_task_flows_print(ta, output);
  
  return TRUE;
}


const char *
name(void)
{
  return "Traffic-Analyzer";
}

const char *
author(void)
{
  return "Christian Kreibich, <christian.kreibich-AT-cl.cam.ac.uk>";
}

const char *
version(void)
{
  return VERSION;
}

/* We need initialization to check if the TCP Connection tracker
 * plugin is installed.
 */
gboolean
init(void)
{
  if (! libnd_plugin_find("Conntrack"))
    {
      D(("Connection tracker plugin not found\n"));
      return FALSE;
    }
  
  if (! libnd_plugin_find("Trace-Set"))
    {
      D(("Trace-Set plugin not found\n"));
      return FALSE;
    }
  
  return TRUE;
}


static void
ta_test_help(void)
{
  printf("Traffic-Analyzer plugin\n"
	 "USAGE: lndtool -r traffic-analyzer [--names-file|-f FILE]\n"
	 "                                   [--output|-o FILE] <TRACE1>\n"
	 "                                   [<TRACE2> <TRACE3> ...]\n"
	 "\n"
	 "  --help, -h, -?           This message.\n"
	 "  --no-basic, -nb          No basic counters (default: yes).\n"
	 "  --no-ip, -ni             No IP protocol usage counters (default: yes).\n"
	 "  --no-ports, -np          No TCP/UDP port number usage analysis (default: yes).\n"
	 "  --no-sports, -nsp        No TCP/UDP source port aggregation (default: yes).\n"
	 "  --no-dports, -ndp        No TCP/UDP dest port aggregation (default: yes).\n"
	 "  --no-flows, -nf          No TCP/UDP flow analysis (default: yes).\n"
	 "  --names-file|-f FILE     Take filenames of traces from flatfile FILE.\n"
	 "  --output|-o FILE         Name of output file to write results to.\n");
}


gboolean
run(LND_Trace *unused, LND_PluginArgs *args)
{
  int i, counter = 0;
  FILE *output;
  LND_Trace *trace;
  LND_TAnalysis *ta;
  char *output_file = "libnetdude-traffic-analysis.txt";

  if (!args)
    return FALSE;

  if (args->argc == 0)
    {
      printf("Please provide one or more traces to scan.\n");
      ta_test_help();
      return FALSE;
    }
  
  ta = libnd_ta_new();
  
  for (i = 0; i < args->argc; i++)
    {
      if (!strcmp(args->argv[i], "-h")     ||
	  !strcmp(args->argv[i], "--help") ||
	  !strcmp(args->argv[i], "-?"))
	{
	  ta_test_help();
	  return FALSE;
	}
      else if (!strcmp(args->argv[i], "--no-basic") ||
	       !strcmp(args->argv[i], "-nb"))
	{
	  ta->do_basic = FALSE;
	}
      else if (!strcmp(args->argv[i], "--no-ip") ||
	       !strcmp(args->argv[i], "-ni"))
	{
	  ta->do_ip = FALSE;
	}
      else if (!strcmp(args->argv[i], "--no-ports") ||
	       !strcmp(args->argv[i], "-np"))
	{
	  ta->do_ports = FALSE;
	}
      else if (!strcmp(args->argv[i], "--no-sports") ||
	       !strcmp(args->argv[i], "-nsp"))
	{
	  ta->do_src_ports = FALSE;
	}
      else if (!strcmp(args->argv[i], "--no-dports") ||
	       !strcmp(args->argv[i], "-ndp"))
	{
	  ta->do_dst_ports = FALSE;
	}
      else if (!strcmp(args->argv[i], "--no-flows") ||
	       !strcmp(args->argv[i], "-nf"))
	{
	  ta->do_flows = FALSE;
	}
      else if (!strcmp(args->argv[i], "-f") ||
	       !strcmp(args->argv[i], "--names-file"))
	{
	  if (++i == args->argc)
	    {
	      printf("You need to pass the name of a file that contains\n"
		     "a list of trace files, one filename per line, when\n"
		     "using the --names-file|-f option.\n");
	      return FALSE;
	    }
	  
	  if (! libnd_ta_add_trace_name_list(ta, args->argv[i]))
	    printf("Could not add all traces provided in '%s'\n", args->argv[i]);
	  else
	    counter++;
	}
      else if (!strcmp(args->argv[i], "-o") ||
	       !strcmp(args->argv[i], "--output"))
	{
	  if (++i == args->argc)
	    {
	      printf("You need to pass the name of an output file that\n"
		     "the results get written to, when using the --output|-o\n"
		     "option.\n");
	      return FALSE;
	    }

	  output_file = args->argv[i];
	}
      else if (args->argv[i][0] != '-') /* Often these are just mistyped args */
	{
	  if (! libnd_ta_add_trace_name(ta, args->argv[i]))
	    printf("Trace file %s not added.\n", args->argv[i]);
	  else
	    counter++;
	}
    }

  if (counter > 0)
    {
      libnd_ta_analyze(ta);
      
      if (! (output = fopen(output_file, "w")))
	{
	  printf("Could not create output file %s\n", output_file);
	  libnd_ta_free(ta);
	  return FALSE;
	}
      
      libnd_ta_write_results(ta, output);
      fclose(output);
    }

  libnd_ta_free(ta);
  
  return TRUE;
}
