/*

Copyright (C) 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd.h>
#include "libnd_tcpfilter.h"

static LND_Protocol *tcp_proto;


LND_TCPFilter *
libnd_tcpfilter_new(void)
{
  LND_TCPFilter *filter;

  if (! (filter = g_new0(LND_TCPFilter, 1)))
    {
      D(("Out of memory.\n"));
      return NULL;
    }

  if (! (filter->tcp_conns = libnd_conn_table_new(LND_CONN_TABLE_IGNORE_DEAD)))
    {
      D(("Out of memory.\n"));
      g_free(filter);
      return NULL;
    }

  if (! (filter->traces = libnd_traceset_new()))
    {
      D(("Out of memory.\n"));
      libnd_tcpfilter_free(filter);
      return NULL;
    }

  filter->do_output     = TRUE;
  filter->output_suffix = ".lnd_tcpflows";
  
  return filter;
}


void
libnd_tcpfilter_free(LND_TCPFilter *filter)
{
  if (!filter)
    return;

  libnd_traceset_free(filter->traces);
  libnd_conn_table_free(filter->tcp_conns);

  g_free(filter);
}


gboolean       
libnd_tcpfilter_set_traceset(LND_TCPFilter *filter, const LND_TraceSet *set)
{
  LND_TraceSet *set_copy;
  
  if (!filter || !set)
    return FALSE;
  
  if (! (set_copy = libnd_traceset_copy(set)))
    {
      D(("Out of memory.\n"));
      return FALSE;
    }
  
  libnd_traceset_free(filter->traces);
  filter->traces = set_copy;

  return TRUE;
 }


gboolean
libnd_tcpfilter_add_trace(LND_TCPFilter *filter, LND_Trace *trace)
{
  if (!filter)
    return FALSE;
  
  return libnd_traceset_add_trace(filter->traces, trace);
}


gboolean 
libnd_tcpfilter_add_trace_name(LND_TCPFilter *filter, const char *file_name)
{
  if (!filter)
    return FALSE;

  return libnd_traceset_add_trace_name(filter->traces, file_name);
}


gboolean        
libnd_tcpfilter_add_trace_name_list(LND_TCPFilter *filter, const char *file_name)
{
  if (!filter)
    return FALSE;

  return libnd_traceset_add_trace_name_list(filter->traces, file_name);
}


static gboolean
run_scan_cb(LND_TraceSet *set, LND_Trace *trace, LND_TCPFilter *filter)
{
  LND_PacketIterator pit;
  LND_TraceArea area_all, area_orig;
  LND_TCPConn *conn;
  LND_Packet *packet;

  D(("Scanning trace %s\n", libnd_trace_get_name(trace)));

  libnd_trace_area_init_space(& area_all, 0.0, 1.0);
  libnd_trace_get_area(trace, &area_orig);
  libnd_trace_set_area(trace, &area_all);
  
  for (libnd_pit_init_mode(&pit, trace, LND_PACKET_IT_AREA_R); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      packet = libnd_pit_get(&pit);

      /* Skip non-TCP packets. */
      if (! libnd_packet_has_proto(packet, tcp_proto))
	continue;
      
      /* Make connection state for this connection exists */
      if (! (conn = (LND_TCPConn *) libnd_conn_table_lookup(filter->tcp_conns, packet)))
	{
	  conn = libnd_tcpconn_new(packet);
	  libnd_conn_table_add(filter->tcp_conns, (LND_ConnID *) conn);
	}
      
      D_ASSERT_PTR(conn);

      /* Now update the connection's state according to the latest packet */
      libnd_tcpconn_update(conn, packet);
    }
  
  libnd_trace_set_area(trace, &area_orig);
  return TRUE;
  TOUCH(set);
}


static gboolean
run_drop_cb(LND_TraceSet *set, LND_Trace *trace, LND_TCPFilter *filter)
{
  LND_PacketIterator pit;
  LND_TraceArea area_all, area_orig;
  LND_TCPConn *conn;
  LND_Packet *packet;
  pcap_dumper_t *pcap_out;
  char filename[MAXPATHLEN];
  char connstr[MAXPATHLEN];
  
  libnd_trace_area_init_space(& area_all, 0.0, 1.0);
  libnd_trace_get_area(trace, &area_orig);
  libnd_trace_set_area(trace, &area_all);
  
  g_snprintf(filename, MAXPATHLEN, "%s%s",
	     libnd_trace_get_name(trace),
	     filter->output_suffix);
  
  D(("Dropping packets in trace %s, writing to %s\n",
     libnd_trace_get_name(trace), filename));

  if (! (pcap_out = pcap_dump_open(pcapnav_pcap(trace->tpm->base->pcn),
				   filename)))
    {
      D(("Could not obtain pcap handle of input trace %s.\n",
	 libnd_trace_get_name(trace)));
      return TRUE;
    }

  for (libnd_pit_init_mode(&pit, trace, LND_PACKET_IT_AREA_R); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      packet = libnd_pit_get(&pit);

      if (! libnd_packet_has_proto(packet, tcp_proto))
	{
	  pcap_dump((u_char *) pcap_out, &packet->ph, packet->data);
	  continue;
	}
      
      if (! (conn = (LND_TCPConn *) libnd_conn_table_lookup(filter->tcp_conns, packet)))
	{
	  D(("Error -- no unexisting connections should exist in second run.\n"));
	  break;
	}
      
      if (libnd_tcpconn_handshake_seen(conn) &&
	  libnd_tcpconn_teardown_seen(conn))
	{
	  pcap_dump((u_char *) pcap_out, &packet->ph, packet->data);
	}
      else
	{
#ifdef LIBND_DEBUG
	  libnd_conn_to_string((LND_ConnID *) conn, connstr, MAXPATHLEN);
	  D(("Dropping %s: 3ws [%c] tdn [%c]\n",
	     connstr, 
	     libnd_tcpconn_handshake_seen(conn) ? 'y' : ' ',
	     libnd_tcpconn_teardown_seen(conn) ? 'y' : ' '));
#endif
	}
    }
  
  pcap_dump_close(pcap_out);
  libnd_trace_set_area(trace, &area_orig);

  return TRUE;
  TOUCH(set);
}


static gboolean
run_stats(LND_TCPConn *conn, LND_TCPFilter *filter)
{
  if (libnd_tcpconn_handshake_seen(conn) &&
      libnd_tcpconn_teardown_seen(conn))
    filter->stats.flows_complete++;
  else
    filter->stats.flows_incomplete++;
  
  return TRUE;
}

static void
report_stats(LND_TCPFilter *filter)
{
  printf("TCP Filter Statistics\n"
	 "------------------------------------------------------------------------\n"
	 "Total flows tracked:   %6u\n"
	 "Complete TCP flows:    %6i\n"
	 "Incomplete TCP flows:  %6i\n",
	 libnd_conn_table_size(filter->tcp_conns),
	 filter->stats.flows_complete,
	 filter->stats.flows_incomplete);
}

void
libnd_tcpfilter_run(LND_TCPFilter *filter)
{
  /* We do two runs. In the first one, we scan over all traces and build up
   * the connection hashtable, updating connection state as we see packets.
   * In the second run, we drop all packets belonging to connections that
   * are not marked as terminated.
   */
  libnd_traceset_foreach(filter->traces, (LND_TraceSetCB) run_scan_cb, filter);

  if (filter->do_output)
    libnd_traceset_foreach(filter->traces, (LND_TraceSetCB) run_drop_cb, filter);
  
  /* If the user wants a state summary, we also go through the connection
   * table and look at the connections' states to report some more detail.
   */
  if (filter->do_stats)
    {
      libnd_conn_table_foreach(filter->tcp_conns, (LND_ConnFunc) run_stats, filter);
      report_stats(filter);
    }
}  


const char *
name(void)
{
  return "TCP-Filter";
}

const char *
author(void)
{
  return "Christian Kreibich <christian@whoop.org>";
}

const char *
version(void)
{
  return VERSION;
}


gboolean
init(void)
{
  if (! libnd_plugin_find("Conntrack"))
    return FALSE;
  if (! libnd_plugin_find("Trace-Set"))
    return FALSE;

  if (! (tcp_proto = libnd_proto_registry_find(LND_PROTO_LAYER_TRANS,
					       IPPROTO_TCP)))
    {
      D(("TCP protocol plugin not found.\n"));
      return FALSE;
    }

  return TRUE;
}


static void
tcpfilter_help(void)
{
  printf("TCP-Filter plugin\n"
	 "USAGE: lndtool -r tcp-filter [--stats|-s] [--dry-run] [--suffix STRING] [--names-file|-f <text file>] [trace1 trace2 ...]\n"
	 "\n"
	 "  --help, -h, -?           This message.\n"
	 "  --stats|-s               Do report statistics about flows at end of run\n"
	 "  --dry-run                Do not create output traces, just scan input.\n"
	 "  --names-file|-f FILE     Take filenames of traces from flatfile FILE.\n"
	 "  --suffix STRING          Suffix for output files. Default: '.lnd_tcpflows'\n");
}


gboolean
run(LND_Trace *unused, LND_PluginArgs *args)
{
  int i;
  LND_TCPFilter *filter;

  if (!args)
    return FALSE;

  if (args->argc == 0)
    {
      printf("Please provide one or more traces to scan.\n");

      tcpfilter_help();
      return FALSE;
    }

  if (! (filter = libnd_tcpfilter_new()))
    {
      D(("Out of memory.\n"));
      return FALSE;
    }

  for (i = 0; i < args->argc; i++)
    {
      if (!strcmp(args->argv[i], "-h")     ||
	  !strcmp(args->argv[i], "--help") ||
	  !strcmp(args->argv[i], "-?"))
	{
	  tcpfilter_help();
	  return FALSE;
	}
      else if (!strcmp(args->argv[i], "-s") ||
	       !strcmp(args->argv[i], "--stats"))
	{
	  filter->do_stats = TRUE;
	}
      else if (!strcmp(args->argv[i], "--dry-run"))
	{
	  filter->do_output = FALSE;
	}
      else if (!strcmp(args->argv[i], "-f") ||
	       !strcmp(args->argv[i], "--names-file"))
	{
	  if (++i == args->argc)
	    {
	      printf("You need to pass the name of a file that contains\n"
		     "a list of trace files, one filename per line, when\n"
		     "using the --names-file|-f option.\n");
	      return FALSE;
	    }
	  
	  if (! libnd_tcpfilter_add_trace_name_list(filter, args->argv[i]))
	    printf("Could not add all traces provided in '%s'\n", args->argv[i]);	  
	}
      else if (!strcmp(args->argv[i], "--suffix"))
	{
	  if (++i == args->argc)
	    {
	      printf("You need to provide a suffix string to append to\n"
		     "output traces when using --suffix|-s\n");
	      return FALSE;
	    }
	  
	  filter->output_suffix = args->argv[i];
	}
      else
	{
	  if (! libnd_tcpfilter_add_trace_name(filter, args->argv[i]))
	    printf("Could not add trace file '%s'\n", args->argv[i]);
	}
    }

  libnd_tcpfilter_run(filter);
  libnd_tcpfilter_free(filter);

  return TRUE;
  TOUCH(unused);
}
